/*
 * Copyright 2015 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var core_1 = require("@blueprintjs/core");
var classNames = require("classnames");
var React = require("react");
var ReactDayPicker = require("react-day-picker");
var Classes = require("./common/classes");
var DateUtils = require("./common/dateUtils");
var Errors = require("./common/errors");
var datePickerCaption_1 = require("./datePickerCaption");
var datePickerCore_1 = require("./datePickerCore");
var DatePicker = (function (_super) {
    __extends(DatePicker, _super);
    function DatePicker(props, context) {
        var _this = this;
        _super.call(this, props, context);
        this.displayName = "Blueprint.DatePicker";
        this.ignoreNextMonthChange = false;
        this.disabledDays = function (day) { return !DateUtils.isDayInRange(day, [_this.props.minDate, _this.props.maxDate]); };
        this.selectedDays = function (day) { return DateUtils.areSameDay(_this.state.value, day); };
        this.handleDayClick = function (_e, day, modifiers) {
            var newValue = day;
            if (_this.props.canClearSelection && modifiers.selected) {
                newValue = null;
            }
            if (_this.props.value === undefined) {
                // component is uncontrolled
                if (!modifiers.disabled) {
                    var displayMonth = day.getMonth();
                    var displayYear = day.getFullYear();
                    var selectedDay = day.getDate();
                    _this.setState({ displayMonth: displayMonth, displayYear: displayYear, selectedDay: selectedDay, value: newValue });
                }
            }
            if (!modifiers.disabled) {
                core_1.Utils.safeInvoke(_this.props.onChange, newValue, true);
                if (_this.state.value != null && _this.state.value.getMonth() !== day.getMonth()) {
                    _this.ignoreNextMonthChange = true;
                }
            }
            else {
                // rerender base component to get around bug where you can navigate past bounds by clicking days
                _this.forceUpdate();
            }
        };
        this.handleMonthChange = function (newDate) {
            var displayMonth = newDate.getMonth();
            var displayYear = newDate.getFullYear();
            var value = _this.state.value;
            if (value !== null) {
                value = _this.computeValidDateInSpecifiedMonthYear(displayYear, displayMonth);
                if (_this.ignoreNextMonthChange) {
                    _this.ignoreNextMonthChange = false;
                }
                else {
                    // if handleDayClick just got run, it means the user selected a date in a new month,
                    // so don't run onChange again
                    core_1.Utils.safeInvoke(_this.props.onChange, value, false);
                }
            }
            _this.setStateWithValueIfUncontrolled({ displayMonth: displayMonth, displayYear: displayYear }, value);
        };
        this.handleMonthSelectChange = function (displayMonth) {
            var value = _this.state.value;
            if (value !== null) {
                value = _this.computeValidDateInSpecifiedMonthYear(value.getFullYear(), displayMonth);
                core_1.Utils.safeInvoke(_this.props.onChange, value, false);
            }
            _this.setStateWithValueIfUncontrolled({ displayMonth: displayMonth }, value);
        };
        this.handleYearSelectChange = function (displayYear) {
            var _a = _this.state, displayMonth = _a.displayMonth, value = _a.value;
            if (value !== null) {
                value = _this.computeValidDateInSpecifiedMonthYear(displayYear, displayMonth);
                core_1.Utils.safeInvoke(_this.props.onChange, value, false);
                displayMonth = value.getMonth();
            }
            else {
                var _b = _this.props, minDate = _b.minDate, maxDate = _b.maxDate;
                var minYear = minDate.getFullYear();
                var maxYear = maxDate.getFullYear();
                var minMonth = minDate.getMonth();
                var maxMonth = maxDate.getMonth();
                if (displayYear === minYear && displayMonth < minMonth) {
                    displayMonth = minMonth;
                }
                else if (displayYear === maxYear && displayMonth > maxMonth) {
                    displayMonth = maxMonth;
                }
            }
            _this.setStateWithValueIfUncontrolled({ displayMonth: displayMonth, displayYear: displayYear }, value);
        };
        this.handleClearClick = function () {
            if (_this.props.value === undefined) {
                _this.setState({ value: null });
            }
            core_1.Utils.safeInvoke(_this.props.onChange, null, true);
        };
        this.handleTodayClick = function () {
            var value = new Date();
            var displayMonth = value.getMonth();
            var displayYear = value.getFullYear();
            var selectedDay = value.getDate();
            if (_this.props.value === undefined) {
                _this.setState({ displayMonth: displayMonth, displayYear: displayYear, selectedDay: selectedDay, value: value });
            }
            else {
                _this.setState({ displayMonth: displayMonth, displayYear: displayYear, selectedDay: selectedDay });
            }
            core_1.Utils.safeInvoke(_this.props.onChange, value, true);
        };
        var value = null;
        if (props.value !== undefined) {
            value = props.value;
        }
        else if (props.defaultValue != null) {
            value = props.defaultValue;
        }
        var selectedDay;
        if (value !== null) {
            selectedDay = value.getDate();
        }
        var initialMonth;
        var today = new Date();
        if (props.initialMonth != null) {
            initialMonth = props.initialMonth;
        }
        else if (value != null) {
            initialMonth = value;
        }
        else if (DateUtils.isDayInRange(today, [props.minDate, props.maxDate])) {
            initialMonth = today;
        }
        else {
            initialMonth = DateUtils.getDateBetween([props.minDate, props.maxDate]);
        }
        this.state = {
            displayMonth: initialMonth.getMonth(),
            displayYear: initialMonth.getFullYear(),
            selectedDay: selectedDay,
            value: value,
        };
    }
    DatePicker.prototype.render = function () {
        var _a = this.props, className = _a.className, locale = _a.locale, localeUtils = _a.localeUtils, maxDate = _a.maxDate, minDate = _a.minDate, showActionsBar = _a.showActionsBar;
        var _b = this.state, displayMonth = _b.displayMonth, displayYear = _b.displayYear;
        return (React.createElement("div", {className: classNames(Classes.DATEPICKER, className)}, 
            React.createElement(ReactDayPicker, {canChangeMonth: true, captionElement: this.renderCaption(), disabledDays: this.disabledDays, enableOutsideDays: true, fromMonth: minDate, initialMonth: new Date(displayYear, displayMonth), locale: locale, localeUtils: localeUtils, modifiers: this.props.modifiers, onDayClick: this.handleDayClick, onMonthChange: this.handleMonthChange, selectedDays: this.selectedDays, toMonth: maxDate}), 
            showActionsBar ? this.renderOptionsBar() : null));
    };
    DatePicker.prototype.componentWillReceiveProps = function (nextProps) {
        if (nextProps.value !== this.props.value) {
            var _a = this.state, displayMonth = _a.displayMonth, displayYear = _a.displayYear, selectedDay = _a.selectedDay;
            if (nextProps.value != null) {
                displayMonth = nextProps.value.getMonth();
                displayYear = nextProps.value.getFullYear();
                selectedDay = nextProps.value.getDate();
            }
            this.setState({ displayMonth: displayMonth, displayYear: displayYear, selectedDay: selectedDay, value: nextProps.value });
        }
        _super.prototype.componentWillReceiveProps.call(this, nextProps);
    };
    DatePicker.prototype.validateProps = function (props) {
        var defaultValue = props.defaultValue, initialMonth = props.initialMonth, maxDate = props.maxDate, minDate = props.minDate, value = props.value;
        if (defaultValue != null && !DateUtils.isDayInRange(defaultValue, [minDate, maxDate])) {
            throw new Error(Errors.DATEPICKER_DEFAULT_VALUE_INVALID);
        }
        if (initialMonth != null && !DateUtils.isMonthInRange(initialMonth, [minDate, maxDate])) {
            throw new Error(Errors.DATEPICKER_INITIAL_MONTH_INVALID);
        }
        if (maxDate != null
            && minDate != null
            && maxDate < minDate
            && !DateUtils.areSameDay(maxDate, minDate)) {
            throw new Error(Errors.DATEPICKER_MAX_DATE_INVALID);
        }
        if (value != null && !DateUtils.isDayInRange(value, [minDate, maxDate])) {
            throw new Error(Errors.DATEPICKER_VALUE_INVALID);
        }
    };
    DatePicker.prototype.renderCaption = function () {
        var _a = this.props, maxDate = _a.maxDate, minDate = _a.minDate;
        return (React.createElement(datePickerCaption_1.DatePickerCaption, {maxDate: maxDate, minDate: minDate, onMonthChange: this.handleMonthSelectChange, onYearChange: this.handleYearSelectChange}));
    };
    DatePicker.prototype.renderOptionsBar = function () {
        return (React.createElement("div", {className: Classes.DATEPICKER_FOOTER}, 
            React.createElement(core_1.Button, {className: "pt-minimal pt-datepicker-footer-button", onClick: this.handleTodayClick, text: "Today"}), 
            React.createElement(core_1.Button, {className: "pt-minimal pt-datepicker-footer-button", onClick: this.handleClearClick, text: "Clear"})));
    };
    DatePicker.prototype.computeValidDateInSpecifiedMonthYear = function (displayYear, displayMonth) {
        var _a = this.props, minDate = _a.minDate, maxDate = _a.maxDate;
        var maxDaysInMonth = new Date(displayYear, displayMonth + 1, 0).getDate();
        var selectedDay = this.state.selectedDay;
        if (selectedDay > maxDaysInMonth) {
            selectedDay = maxDaysInMonth;
        }
        // matches the underlying react-day-picker timestamp behavior
        var value = new Date(displayYear, displayMonth, selectedDay, 12);
        if (value < minDate) {
            value = minDate;
        }
        else if (value > maxDate) {
            value = maxDate;
        }
        return value;
    };
    DatePicker.prototype.setStateWithValueIfUncontrolled = function (newState, value) {
        if (this.props.value === undefined) {
            // uncontrolled mode means we track value in state
            newState.value = value;
        }
        return this.setState(newState);
    };
    DatePicker.defaultProps = {
        canClearSelection: true,
        maxDate: datePickerCore_1.getDefaultMaxDate(),
        minDate: datePickerCore_1.getDefaultMinDate(),
        showActionsBar: false,
    };
    return DatePicker;
}(core_1.AbstractComponent));
exports.DatePicker = DatePicker;
exports.DatePickerFactory = React.createFactory(DatePicker);

//# sourceMappingURL=data:application/json;base64,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
