/*
 * Copyright 2015 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var core_1 = require("@blueprintjs/core");
var classNames = require("classnames");
var React = require("react");
var Classes = require("./common/classes");
var DateUtils = require("./common/dateUtils");
var Utils = require("./common/utils");
(function (TimePickerPrecision) {
    TimePickerPrecision[TimePickerPrecision["MINUTE"] = 0] = "MINUTE";
    TimePickerPrecision[TimePickerPrecision["SECOND"] = 1] = "SECOND";
    TimePickerPrecision[TimePickerPrecision["MILLISECOND"] = 2] = "MILLISECOND";
})(exports.TimePickerPrecision || (exports.TimePickerPrecision = {}));
var TimePickerPrecision = exports.TimePickerPrecision;
var TimePicker = (function (_super) {
    __extends(TimePicker, _super);
    function TimePicker(props, context) {
        var _this = this;
        _super.call(this, props, context);
        this.displayName = "Blueprint.TimePicker";
        // begin method definitions: event handlers
        this.getInputBlurHandler = function (unit) { return function (e) {
            var text = getStringValueFromInputEvent(e);
            _this.updateTime(parseInt(text, 10), unit);
        }; };
        this.getInputChangeHandler = function (unit) { return function (e) {
            var TWO_DIGITS = /^\d{0,2}$/;
            var THREE_DIGITS = /^\d{0,3}$/;
            var text = getStringValueFromInputEvent(e);
            var isValid = false;
            switch (unit) {
                case TimeUnit.HOUR:
                case TimeUnit.MINUTE:
                case TimeUnit.SECOND:
                    isValid = TWO_DIGITS.test(text);
                    break;
                case TimeUnit.MS:
                    isValid = THREE_DIGITS.test(text);
                    break;
                default:
                    throw Error("Invalid TimeUnit");
            }
            if (isValid) {
                switch (unit) {
                    case TimeUnit.HOUR:
                        _this.updateState({ hourText: text });
                        break;
                    case TimeUnit.MINUTE:
                        _this.updateState({ minuteText: text });
                        break;
                    case TimeUnit.SECOND:
                        _this.updateState({ secondText: text });
                        break;
                    case TimeUnit.MS:
                        _this.updateState({ millisecondText: text });
                        break;
                    default: throw Error("Invalid TimeUnit");
                }
            }
        }; };
        this.getInputKeyDownHandler = function (unit) { return function (e) {
            handleKeyEvent(e, (_a = {},
                _a[core_1.Keys.ARROW_UP] = function () { return _this.incrementTime(unit); },
                _a[core_1.Keys.ARROW_DOWN] = function () { return _this.decrementTime(unit); },
                _a[core_1.Keys.ENTER] = function () {
                    e.currentTarget.blur();
                },
                _a
            ));
            var _a;
        }; };
        if (props.value != null) {
            this.state = this.getFullStateFromValue(props.value);
        }
        else if (props.defaultValue != null) {
            this.state = this.getFullStateFromValue(props.defaultValue);
        }
        else {
            this.state = this.getFullStateFromValue(new Date(0, 0, 0, 0, 0, 0, 0));
        }
    }
    TimePicker.prototype.render = function () {
        var _this = this;
        var shouldRenderSeconds = this.props.precision >= TimePickerPrecision.SECOND;
        var shouldRenderMilliseconds = this.props.precision >= TimePickerPrecision.MILLISECOND;
        /* tslint:disable:max-line-length */
        return (React.createElement("div", {className: classNames(Classes.TIMEPICKER, this.props.className)}, 
            React.createElement("div", {className: Classes.TIMEPICKER_ARROW_ROW}, 
                this.maybeRenderArrowButton(true, Classes.TIMEPICKER_HOUR, function () { return _this.incrementTime(TimeUnit.HOUR); }), 
                this.maybeRenderArrowButton(true, Classes.TIMEPICKER_MINUTE, function () { return _this.incrementTime(TimeUnit.MINUTE); }), 
                shouldRenderSeconds ? this.maybeRenderArrowButton(true, Classes.TIMEPICKER_SECOND, function () { return _this.incrementTime(TimeUnit.SECOND); }) : null, 
                shouldRenderMilliseconds ? this.maybeRenderArrowButton(true, Classes.TIMEPICKER_MILLISECOND, function () { return _this.incrementTime(TimeUnit.MS); }) : null), 
            React.createElement("div", {className: Classes.TIMEPICKER_INPUT_ROW}, 
                this.renderInput(Classes.TIMEPICKER_HOUR, TimeUnit.HOUR, this.state.hourText), 
                this.renderDivider(), 
                this.renderInput(Classes.TIMEPICKER_MINUTE, TimeUnit.MINUTE, this.state.minuteText), 
                shouldRenderSeconds ? this.renderDivider() : null, 
                shouldRenderSeconds ? this.renderInput(Classes.TIMEPICKER_SECOND, TimeUnit.SECOND, this.state.secondText) : null, 
                shouldRenderMilliseconds ? this.renderDivider(".") : null, 
                shouldRenderMilliseconds ? this.renderInput(Classes.TIMEPICKER_MILLISECOND, TimeUnit.MS, this.state.millisecondText) : null), 
            React.createElement("div", {className: Classes.TIMEPICKER_ARROW_ROW}, 
                this.maybeRenderArrowButton(false, Classes.TIMEPICKER_HOUR, function () { return _this.decrementTime(TimeUnit.HOUR); }), 
                this.maybeRenderArrowButton(false, Classes.TIMEPICKER_MINUTE, function () { return _this.decrementTime(TimeUnit.MINUTE); }), 
                shouldRenderSeconds ? this.maybeRenderArrowButton(false, Classes.TIMEPICKER_SECOND, function () { return _this.decrementTime(TimeUnit.SECOND); }) : null, 
                shouldRenderMilliseconds ? this.maybeRenderArrowButton(false, Classes.TIMEPICKER_MILLISECOND, function () { return _this.decrementTime(TimeUnit.MS); }) : null)));
        /* tslint:enable:max-line-length */
    };
    TimePicker.prototype.componentWillReceiveProps = function (nextProps) {
        if (nextProps.value != null && !DateUtils.areSameTime(nextProps.value, this.props.value)) {
            this.setState(this.getFullStateFromValue(nextProps.value));
        }
    };
    // begin method definitions: rendering
    TimePicker.prototype.maybeRenderArrowButton = function (isDirectionUp, className, onClick) {
        var classes = classNames(Classes.TIMEPICKER_ARROW_BUTTON, className, "pt-icon-standard", {
            "pt-icon-chevron-up": isDirectionUp,
            "pt-icon-chevron-down": !isDirectionUp,
        });
        return this.props.showArrowButtons ? React.createElement("span", {className: classes, onClick: onClick}) : null;
    };
    TimePicker.prototype.renderDivider = function (text) {
        if (text === void 0) { text = ":"; }
        return React.createElement("span", {className: Classes.TIMEPICKER_DIVIDER_TEXT}, text);
    };
    TimePicker.prototype.renderInput = function (className, unit, value) {
        return (React.createElement("input", {className: classNames(Classes.TIMEPICKER_INPUT, className), onBlur: this.getInputBlurHandler(unit), onChange: this.getInputChangeHandler(unit), onKeyDown: this.getInputKeyDownHandler(unit), value: value}));
    };
    // begin method definitions: state modification
    /**
     * Generates a full ITimePickerState object with all text fields set to formatted strings based on value
     */
    TimePicker.prototype.getFullStateFromValue = function (value) {
        /* tslint:disable:object-literal-sort-keys */
        return {
            hourText: formatTime(value.getHours(), TimeUnit.HOUR),
            minuteText: formatTime(value.getMinutes(), TimeUnit.MINUTE),
            secondText: formatTime(value.getSeconds(), TimeUnit.SECOND),
            millisecondText: formatTime(value.getMilliseconds(), TimeUnit.MS),
            value: value,
        };
        /* tslint:enable:object-literal-sort-keys */
    };
    TimePicker.prototype.incrementTime = function (unit) {
        var newTime = getTimeUnit(this.state.value, unit) + 1;
        this.updateTime(loopTime(newTime, unit), unit);
    };
    TimePicker.prototype.decrementTime = function (unit) {
        var newTime = getTimeUnit(this.state.value, unit) - 1;
        this.updateTime(loopTime(newTime, unit), unit);
    };
    TimePicker.prototype.updateTime = function (time, unit) {
        if (isTimeValid(time, unit)) {
            var newValue = DateUtils.clone(this.state.value);
            setTimeUnit(time, newValue, unit);
            this.updateState({ value: newValue });
        }
        else {
            // reset to last known good state
            this.updateState(this.getFullStateFromValue(this.state.value));
        }
    };
    TimePicker.prototype.updateState = function (state) {
        var newState = state;
        var hasNewValue = newState.value != null
            && !DateUtils.areSameTime(newState.value, this.state.value);
        if (this.props.value == null) {
            // component is uncontrolled
            if (hasNewValue) {
                newState = this.getFullStateFromValue(newState.value);
            }
            this.setState(newState);
        }
        else {
            // component is controlled, and there's a new value
            // so set inputs' text based off of _old_ value and later fire onChange with new value
            if (hasNewValue) {
                this.setState(this.getFullStateFromValue(this.state.value));
            }
            else {
                // no new value, this means only text has changed (from user typing)
                // we want inputs to change, so update state with new text for the inputs
                // but don't change actual value
                var clonedNewState = core_1.Utils.shallowClone(newState);
                clonedNewState.value = DateUtils.clone(this.state.value);
                this.setState(clonedNewState);
            }
        }
        if (hasNewValue) {
            core_1.Utils.safeInvoke(this.props.onChange, newState.value);
        }
    };
    TimePicker.defaultProps = {
        precision: TimePickerPrecision.MINUTE,
        showArrowButtons: false,
    };
    return TimePicker;
}(React.Component));
exports.TimePicker = TimePicker;
var TimeUnit;
(function (TimeUnit) {
    TimeUnit[TimeUnit["HOUR"] = 0] = "HOUR";
    TimeUnit[TimeUnit["MINUTE"] = 1] = "MINUTE";
    TimeUnit[TimeUnit["SECOND"] = 2] = "SECOND";
    TimeUnit[TimeUnit["MS"] = 3] = "MS";
})(TimeUnit || (TimeUnit = {}));
function formatTime(time, unit) {
    switch (unit) {
        case TimeUnit.HOUR:
            return time.toString();
        case TimeUnit.MINUTE:
        case TimeUnit.SECOND:
            return Utils.padWithZeroes(time.toString(), 2);
        case TimeUnit.MS:
            return Utils.padWithZeroes(time.toString(), 3);
        default:
            throw Error("Invalid TimeUnit");
    }
}
function getStringValueFromInputEvent(e) {
    return e.currentTarget.value;
}
function getTimeUnit(date, unit) {
    switch (unit) {
        case TimeUnit.HOUR: return date.getHours();
        case TimeUnit.MINUTE: return date.getMinutes();
        case TimeUnit.SECOND: return date.getSeconds();
        case TimeUnit.MS: return date.getMilliseconds();
        default: throw Error("Invalid TimeUnit");
    }
}
function handleKeyEvent(e, actions, preventDefault) {
    if (preventDefault === void 0) { preventDefault = true; }
    for (var _i = 0, _a = Object.keys(actions); _i < _a.length; _i++) {
        var k = _a[_i];
        var key = Number(k);
        if (e.which === key) {
            if (preventDefault) {
                e.preventDefault();
            }
            actions[key]();
        }
    }
}
function isTimeValid(time, unit) {
    return time != null && !isNaN(time) && minTime(unit) <= time && time <= maxTime(unit);
}
function loopTime(time, unit) {
    var max = maxTime(unit);
    var min = minTime(unit);
    if (time > max) {
        return min;
    }
    else if (time < min) {
        return max;
    }
    return time;
}
function minTime(unit) {
    var min = (_a = {},
        _a[TimeUnit.HOUR] = 0,
        _a[TimeUnit.MINUTE] = 0,
        _a[TimeUnit.SECOND] = 0,
        _a[TimeUnit.MS] = 0,
        _a
    );
    return min[unit];
    var _a;
}
function maxTime(unit) {
    var max = (_a = {},
        _a[TimeUnit.HOUR] = 23,
        _a[TimeUnit.MINUTE] = 59,
        _a[TimeUnit.SECOND] = 59,
        _a[TimeUnit.MS] = 999,
        _a
    );
    return max[unit];
    var _a;
}
function setTimeUnit(time, date, unit) {
    switch (unit) {
        case TimeUnit.HOUR:
            date.setHours(time);
            break;
        case TimeUnit.MINUTE:
            date.setMinutes(time);
            break;
        case TimeUnit.SECOND:
            date.setSeconds(time);
            break;
        case TimeUnit.MS:
            date.setMilliseconds(time);
            break;
        default: throw Error("Invalid TimeUnit");
    }
}
exports.TimePickerFactory = React.createFactory(TimePicker);

//# sourceMappingURL=data:application/json;base64,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
