/*
 * Copyright 2015 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the BSD-3 License as modified (the “License”); you may obtain a copy
 * of the license at https://github.com/palantir/blueprint/blob/master/LICENSE
 * and https://github.com/palantir/blueprint/blob/master/PATENTS
 */
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var core_1 = require("@blueprintjs/core");
var classNames = require("classnames");
var React = require("react");
var Classes = require("./common/classes");
var DateUtils = require("./common/dateUtils");
var Utils = require("./common/utils");
var TimePickerPrecision;
(function (TimePickerPrecision) {
    TimePickerPrecision[TimePickerPrecision["MINUTE"] = 0] = "MINUTE";
    TimePickerPrecision[TimePickerPrecision["SECOND"] = 1] = "SECOND";
    TimePickerPrecision[TimePickerPrecision["MILLISECOND"] = 2] = "MILLISECOND";
})(TimePickerPrecision = exports.TimePickerPrecision || (exports.TimePickerPrecision = {}));
var DEFAULT_MIN_HOUR = 0;
var DEFAULT_MIN_MINUTE = 0;
var DEFAULT_MIN_SECOND = 0;
var DEFAULT_MIN_MILLISECOND = 0;
var DEFAULT_MAX_HOUR = 23;
var DEFAULT_MAX_MINUTE = 59;
var DEFAULT_MAX_SECOND = 59;
var DEFAULT_MAX_MILLISECOND = 999;
function getDefaultMinTime() {
    return new Date(0, 0, 0, DEFAULT_MIN_HOUR, DEFAULT_MIN_MINUTE, DEFAULT_MIN_SECOND, DEFAULT_MIN_MILLISECOND);
}
exports.getDefaultMinTime = getDefaultMinTime;
function getDefaultMaxTime() {
    return new Date(0, 0, 0, DEFAULT_MAX_HOUR, DEFAULT_MAX_MINUTE, DEFAULT_MAX_SECOND, DEFAULT_MAX_MILLISECOND);
}
exports.getDefaultMaxTime = getDefaultMaxTime;
var TimePicker = (function (_super) {
    tslib_1.__extends(TimePicker, _super);
    function TimePicker(props, context) {
        var _this = _super.call(this, props, context) || this;
        // begin method definitions: event handlers
        _this.getInputBlurHandler = function (unit) { return function (e) {
            var text = getStringValueFromInputEvent(e);
            _this.updateTime(parseInt(text, 10), unit);
        }; };
        _this.getInputChangeHandler = function (unit) { return function (e) {
            var TWO_DIGITS = /^\d{0,2}$/;
            var THREE_DIGITS = /^\d{0,3}$/;
            var text = getStringValueFromInputEvent(e);
            var isValid = false;
            switch (unit) {
                case TimeUnit.HOUR:
                case TimeUnit.MINUTE:
                case TimeUnit.SECOND:
                    isValid = TWO_DIGITS.test(text);
                    break;
                case TimeUnit.MS:
                    isValid = THREE_DIGITS.test(text);
                    break;
                default:
                    throw Error("Invalid TimeUnit");
            }
            if (isValid) {
                switch (unit) {
                    case TimeUnit.HOUR:
                        _this.updateState({ hourText: text });
                        break;
                    case TimeUnit.MINUTE:
                        _this.updateState({ minuteText: text });
                        break;
                    case TimeUnit.SECOND:
                        _this.updateState({ secondText: text });
                        break;
                    case TimeUnit.MS:
                        _this.updateState({ millisecondText: text });
                        break;
                    default: throw Error("Invalid TimeUnit");
                }
            }
        }; };
        _this.getInputKeyDownHandler = function (unit) { return function (e) {
            handleKeyEvent(e, (_a = {},
                _a[core_1.Keys.ARROW_UP] = function () { return _this.incrementTime(unit); },
                _a[core_1.Keys.ARROW_DOWN] = function () { return _this.decrementTime(unit); },
                _a[core_1.Keys.ENTER] = function () {
                    e.currentTarget.blur();
                },
                _a));
            var _a;
        }; };
        _this.handleFocus = function (e) {
            if (_this.props.selectAllOnFocus) {
                e.currentTarget.select();
            }
        };
        if (props.value != null) {
            _this.state = _this.getFullStateFromValue(props.value);
        }
        else if (props.defaultValue != null) {
            _this.state = _this.getFullStateFromValue(props.defaultValue);
        }
        else {
            _this.state = _this.getFullStateFromValue(props.minTime);
        }
        return _this;
    }
    TimePicker.prototype.render = function () {
        var _this = this;
        var shouldRenderSeconds = this.props.precision >= TimePickerPrecision.SECOND;
        var shouldRenderMilliseconds = this.props.precision >= TimePickerPrecision.MILLISECOND;
        var classes = classNames(Classes.TIMEPICKER, this.props.className, (_a = {},
            _a[core_1.Classes.DISABLED] = this.props.disabled,
            _a));
        /* tslint:disable:max-line-length */
        return (React.createElement("div", { className: classes },
            React.createElement("div", { className: Classes.TIMEPICKER_ARROW_ROW },
                this.maybeRenderArrowButton(true, Classes.TIMEPICKER_HOUR, function () { return _this.incrementTime(TimeUnit.HOUR); }),
                this.maybeRenderArrowButton(true, Classes.TIMEPICKER_MINUTE, function () { return _this.incrementTime(TimeUnit.MINUTE); }),
                shouldRenderSeconds ? this.maybeRenderArrowButton(true, Classes.TIMEPICKER_SECOND, function () { return _this.incrementTime(TimeUnit.SECOND); }) : null,
                shouldRenderMilliseconds ? this.maybeRenderArrowButton(true, Classes.TIMEPICKER_MILLISECOND, function () { return _this.incrementTime(TimeUnit.MS); }) : null),
            React.createElement("div", { className: Classes.TIMEPICKER_INPUT_ROW },
                this.renderInput(Classes.TIMEPICKER_HOUR, TimeUnit.HOUR, this.state.hourText),
                this.renderDivider(),
                this.renderInput(Classes.TIMEPICKER_MINUTE, TimeUnit.MINUTE, this.state.minuteText),
                shouldRenderSeconds ? this.renderDivider() : null,
                shouldRenderSeconds ? this.renderInput(Classes.TIMEPICKER_SECOND, TimeUnit.SECOND, this.state.secondText) : null,
                shouldRenderMilliseconds ? this.renderDivider(".") : null,
                shouldRenderMilliseconds ? this.renderInput(Classes.TIMEPICKER_MILLISECOND, TimeUnit.MS, this.state.millisecondText) : null),
            React.createElement("div", { className: Classes.TIMEPICKER_ARROW_ROW },
                this.maybeRenderArrowButton(false, Classes.TIMEPICKER_HOUR, function () { return _this.decrementTime(TimeUnit.HOUR); }),
                this.maybeRenderArrowButton(false, Classes.TIMEPICKER_MINUTE, function () { return _this.decrementTime(TimeUnit.MINUTE); }),
                shouldRenderSeconds ? this.maybeRenderArrowButton(false, Classes.TIMEPICKER_SECOND, function () { return _this.decrementTime(TimeUnit.SECOND); }) : null,
                shouldRenderMilliseconds ? this.maybeRenderArrowButton(false, Classes.TIMEPICKER_MILLISECOND, function () { return _this.decrementTime(TimeUnit.MS); }) : null)));
        var _a;
        /* tslint:enable:max-line-length */
    };
    TimePicker.prototype.componentWillReceiveProps = function (nextProps) {
        var didMinTimeChange = nextProps.minTime !== this.props.minTime;
        var didMaxTimeChange = nextProps.maxTime !== this.props.maxTime;
        var didBoundsChange = didMinTimeChange || didMaxTimeChange;
        if (didBoundsChange) {
            var timeInRange = DateUtils.getTimeInRange(this.state.value, nextProps.minTime, nextProps.maxTime);
            this.setState(this.getFullStateFromValue(timeInRange));
        }
        if (nextProps.value != null && !DateUtils.areSameTime(nextProps.value, this.props.value)) {
            this.setState(this.getFullStateFromValue(nextProps.value));
        }
    };
    // begin method definitions: rendering
    TimePicker.prototype.maybeRenderArrowButton = function (isDirectionUp, className, onClick) {
        var classes = classNames(Classes.TIMEPICKER_ARROW_BUTTON, className, "pt-icon-standard", {
            "pt-icon-chevron-down": !isDirectionUp,
            "pt-icon-chevron-up": isDirectionUp,
        });
        return this.props.showArrowButtons ? React.createElement("span", { className: classes, onClick: onClick }) : null;
    };
    TimePicker.prototype.renderDivider = function (text) {
        if (text === void 0) { text = ":"; }
        return React.createElement("span", { className: Classes.TIMEPICKER_DIVIDER_TEXT }, text);
    };
    TimePicker.prototype.renderInput = function (className, unit, value) {
        return (React.createElement("input", { className: classNames(Classes.TIMEPICKER_INPUT, className), onBlur: this.getInputBlurHandler(unit), onChange: this.getInputChangeHandler(unit), onFocus: this.handleFocus, onKeyDown: this.getInputKeyDownHandler(unit), value: value, disabled: this.props.disabled }));
    };
    // begin method definitions: state modification
    /**
     * Generates a full ITimePickerState object with all text fields set to formatted strings based on value
     */
    TimePicker.prototype.getFullStateFromValue = function (value) {
        var timeInRange = DateUtils.getTimeInRange(value, this.props.minTime, this.props.maxTime);
        /* tslint:disable:object-literal-sort-keys */
        return {
            hourText: formatTime(timeInRange.getHours(), TimeUnit.HOUR),
            minuteText: formatTime(timeInRange.getMinutes(), TimeUnit.MINUTE),
            secondText: formatTime(timeInRange.getSeconds(), TimeUnit.SECOND),
            millisecondText: formatTime(timeInRange.getMilliseconds(), TimeUnit.MS),
            value: timeInRange,
        };
        /* tslint:enable:object-literal-sort-keys */
    };
    TimePicker.prototype.incrementTime = function (unit) {
        if (this.props.disabled) {
            return;
        }
        var newTime = getTimeUnit(this.state.value, unit) + 1;
        this.updateTime(loopTime(newTime, unit), unit);
    };
    TimePicker.prototype.decrementTime = function (unit) {
        if (this.props.disabled) {
            return;
        }
        var newTime = getTimeUnit(this.state.value, unit) - 1;
        this.updateTime(loopTime(newTime, unit), unit);
    };
    TimePicker.prototype.updateTime = function (time, unit) {
        var newValue = DateUtils.clone(this.state.value);
        var _a = this.props, minTime = _a.minTime, maxTime = _a.maxTime;
        if (isTimeValid(time, unit)) {
            setTimeUnit(time, newValue, unit);
            if (DateUtils.isTimeInRange(newValue, minTime, maxTime)) {
                this.updateState({ value: newValue });
            }
            else if (!DateUtils.areSameTime(this.state.value, minTime)) {
                this.updateState(this.getFullStateFromValue(newValue));
            }
        }
        else {
            // reset to last known good state
            this.updateState(this.getFullStateFromValue(this.state.value));
        }
    };
    TimePicker.prototype.updateState = function (state) {
        var newState = state;
        var hasNewValue = newState.value != null
            && !DateUtils.areSameTime(newState.value, this.state.value);
        if (this.props.value == null) {
            // component is uncontrolled
            if (hasNewValue) {
                newState = this.getFullStateFromValue(newState.value);
            }
            this.setState(newState);
        }
        else {
            // component is controlled, and there's a new value
            // so set inputs' text based off of _old_ value and later fire onChange with new value
            if (hasNewValue) {
                this.setState(this.getFullStateFromValue(this.state.value));
            }
            else {
                // no new value, this means only text has changed (from user typing)
                // we want inputs to change, so update state with new text for the inputs
                // but don't change actual value
                this.setState(tslib_1.__assign({}, newState, { value: DateUtils.clone(this.state.value) }));
            }
        }
        if (hasNewValue) {
            core_1.Utils.safeInvoke(this.props.onChange, newState.value);
        }
    };
    return TimePicker;
}(React.Component));
TimePicker.defaultProps = {
    disabled: false,
    maxTime: getDefaultMaxTime(),
    minTime: getDefaultMinTime(),
    precision: TimePickerPrecision.MINUTE,
    selectAllOnFocus: false,
    showArrowButtons: false,
};
TimePicker.displayName = "Blueprint.TimePicker";
exports.TimePicker = TimePicker;
var TimeUnit;
(function (TimeUnit) {
    TimeUnit[TimeUnit["HOUR"] = 0] = "HOUR";
    TimeUnit[TimeUnit["MINUTE"] = 1] = "MINUTE";
    TimeUnit[TimeUnit["SECOND"] = 2] = "SECOND";
    TimeUnit[TimeUnit["MS"] = 3] = "MS";
})(TimeUnit || (TimeUnit = {}));
function formatTime(time, unit) {
    switch (unit) {
        case TimeUnit.HOUR:
            return time.toString();
        case TimeUnit.MINUTE:
        case TimeUnit.SECOND:
            return Utils.padWithZeroes(time.toString(), 2);
        case TimeUnit.MS:
            return Utils.padWithZeroes(time.toString(), 3);
        default:
            throw Error("Invalid TimeUnit");
    }
}
function getStringValueFromInputEvent(e) {
    return e.currentTarget.value;
}
function getTimeUnit(date, unit) {
    switch (unit) {
        case TimeUnit.HOUR: return date.getHours();
        case TimeUnit.MINUTE: return date.getMinutes();
        case TimeUnit.SECOND: return date.getSeconds();
        case TimeUnit.MS: return date.getMilliseconds();
        default: throw Error("Invalid TimeUnit");
    }
}
function handleKeyEvent(e, actions, preventDefault) {
    if (preventDefault === void 0) { preventDefault = true; }
    for (var _i = 0, _a = Object.keys(actions); _i < _a.length; _i++) {
        var k = _a[_i];
        var key = Number(k);
        if (e.which === key) {
            if (preventDefault) {
                e.preventDefault();
            }
            actions[key]();
        }
    }
}
function isTimeValid(time, unit) {
    return time != null && !isNaN(time) && minTime(unit) <= time && time <= maxTime(unit);
}
function loopTime(time, unit) {
    var max = maxTime(unit);
    var min = minTime(unit);
    if (time > max) {
        return min;
    }
    else if (time < min) {
        return max;
    }
    return time;
}
function minTime(unit) {
    var min = (_a = {},
        _a[TimeUnit.HOUR] = DEFAULT_MIN_HOUR,
        _a[TimeUnit.MINUTE] = DEFAULT_MIN_MINUTE,
        _a[TimeUnit.SECOND] = DEFAULT_MIN_SECOND,
        _a[TimeUnit.MS] = DEFAULT_MIN_MILLISECOND,
        _a);
    return min[unit];
    var _a;
}
function maxTime(unit) {
    var max = (_a = {},
        _a[TimeUnit.HOUR] = DEFAULT_MAX_HOUR,
        _a[TimeUnit.MINUTE] = DEFAULT_MAX_MINUTE,
        _a[TimeUnit.SECOND] = DEFAULT_MAX_SECOND,
        _a[TimeUnit.MS] = DEFAULT_MAX_MILLISECOND,
        _a);
    return max[unit];
    var _a;
}
function setTimeUnit(time, date, unit) {
    switch (unit) {
        case TimeUnit.HOUR:
            date.setHours(time);
            break;
        case TimeUnit.MINUTE:
            date.setMinutes(time);
            break;
        case TimeUnit.SECOND:
            date.setSeconds(time);
            break;
        case TimeUnit.MS:
            date.setMilliseconds(time);
            break;
        default: throw Error("Invalid TimeUnit");
    }
}
exports.TimePickerFactory = React.createFactory(TimePicker);

//# sourceMappingURL=timePicker.js.map
