/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var regions_1 = require("../regions");
var rect_1 = require("./rect");
var utils_1 = require("./utils");
var EXTREMA_LAST_IN_ROW = ["bp-table-last-in-row"];
var EXTREMA_LAST_IN_COLUMN = ["bp-table-last-in-column"];
var EXTREMA_LAST_IN_ROW_AND_COLUMN = ["bp-table-last-in-column", "bp-table-last-in-row"];
var EXTREMA_NONE = [];
/**
 * This class manages the sizes of grid cells using arrays of individual row/column sizes.
 */
var Grid = (function () {
    /**
     * This constructor accumulates the heights and widths in `O(n)`, saving
     * time in later calculations.
     *
     * @param bleed - The number of rows/cols that we expand beyond the
     *     viewport (on all sides). This helps avoid displaying an empty
     *     viewport when the user scrolls quickly.
     */
    function Grid(rowHeights, columnWidths, bleed, ghostHeight, ghostWidth) {
        var _this = this;
        if (bleed === void 0) { bleed = Grid.DEFAULT_BLEED; }
        if (ghostHeight === void 0) { ghostHeight = Grid.DEFAULT_GHOST_HEIGHT; }
        if (ghostWidth === void 0) { ghostWidth = Grid.DEFAULT_GHOST_WIDTH; }
        this.getCumulativeWidthAt = function (index) {
            if (index >= _this.numCols) {
                return _this.cumulativeColumnWidths[_this.numCols - 1] + _this.ghostWidth * (index - _this.numCols + 1);
            }
            else {
                return _this.cumulativeColumnWidths[index];
            }
        };
        this.getCumulativeHeightAt = function (index) {
            if (index >= _this.numRows) {
                return _this.cumulativeRowHeights[_this.numRows - 1] + _this.ghostHeight * (index - _this.numRows + 1);
            }
            else {
                return _this.cumulativeRowHeights[index];
            }
        };
        this.columnWidths = columnWidths;
        this.rowHeights = rowHeights;
        this.cumulativeColumnWidths = utils_1.Utils.accumulate(columnWidths);
        this.cumulativeRowHeights = utils_1.Utils.accumulate(rowHeights);
        this.numCols = columnWidths.length;
        this.numRows = rowHeights.length;
        this.bleed = bleed;
        this.ghostHeight = ghostHeight;
        this.ghostWidth = ghostWidth;
    }
    /**
     * Returns the `Rect` bounds of a cell in scrollpane client space.
     *
     * Scrollpane client coordinate space uses the origin of the scrollpane
     * client (the inside part that you're moving around).
     *
     * For example, let's say you're scrolling around a block of 1000 x 1000
     * cells. Regardless where you've scrolled, the first cell is always at
     * 0,0 in scrollpane client space. the cell to the right of it is always
     * at, e.g., 100,0.
     */
    Grid.prototype.getCellRect = function (rowIndex, columnIndex) {
        var height = this.rowHeights[rowIndex];
        var top = this.cumulativeRowHeights[rowIndex] - height;
        var width = this.columnWidths[columnIndex];
        var left = this.cumulativeColumnWidths[columnIndex] - width;
        return new rect_1.Rect(left, top, width, height);
    };
    /**
     * Returns the `Rect` bounds of a cell in scrollpane client space.
     *
     * If the cell is beyond the bounds of the user-defined table cells, it is
     * considered a "ghost" cell. If a width/height is not defined for that
     * row/column, we use the default width/height.
     */
    Grid.prototype.getGhostCellRect = function (rowIndex, columnIndex) {
        var left = 0;
        var top = 0;
        var width = 0;
        var height = 0;
        if (rowIndex >= this.rowHeights.length) {
            height = this.ghostHeight;
            top = this.getHeight() + this.ghostHeight * (rowIndex - this.numRows);
        }
        else {
            height = this.rowHeights[rowIndex];
            top = this.cumulativeRowHeights[rowIndex] - height;
        }
        if (columnIndex >= this.columnWidths.length) {
            width = this.ghostWidth;
            left = this.getWidth() + this.ghostWidth * (columnIndex - this.numCols);
        }
        else {
            width = this.columnWidths[columnIndex];
            left = this.cumulativeColumnWidths[columnIndex] - width;
        }
        return new rect_1.Rect(left, top, width, height);
    };
    /**
     * Returns the `Rect` with the base coordinate and height of the specified row.
     */
    Grid.prototype.getRowRect = function (rowIndex) {
        var height = this.rowHeights[rowIndex];
        var top = this.cumulativeRowHeights[rowIndex] - height;
        return new rect_1.Rect(0, top, this.getWidth(), height);
    };
    /**
     * Returns the `Rect` with the base coordinate and width of the specified column.
     */
    Grid.prototype.getColumnRect = function (columnIndex) {
        var width = this.columnWidths[columnIndex];
        var left = this.cumulativeColumnWidths[columnIndex] - width;
        return new rect_1.Rect(left, 0, width, this.getHeight());
    };
    /**
     * Returns the total width of the entire grid
     */
    Grid.prototype.getWidth = function () {
        return this.cumulativeColumnWidths[this.numCols - 1];
    };
    /**
     * Returns the total width of the entire grid
     */
    Grid.prototype.getHeight = function () {
        return this.cumulativeRowHeights[this.numRows - 1];
    };
    /**
     * Returns the `Rect` bounds of entire grid
     */
    Grid.prototype.getRect = function () {
        return new rect_1.Rect(0, 0, this.getWidth(), this.getHeight());
    };
    /**
     * Maps each cell that intersects with the given `Rect` argument. The
     * indices of iteration are extended in both directions by the integer
     * `bleed` class property, then are clamped between 0 and the number of
     * rows/columns.
     *
     * Uses a binary search for each of the 4 edges of the bounds, resulting
     * in a runtime of `O(log(rows) + log(cols))` plus the `O(irows * icols)`
     * iteration of intersecting cells.
     */
    Grid.prototype.mapCellsInRect = function (rect, callback) {
        var results = [];
        if (rect == null) {
            return results;
        }
        var _a = this.getRowIndicesInRect(rect), rowIndexStart = _a.rowIndexStart, rowIndexEnd = _a.rowIndexEnd;
        var _b = this.getColumnIndicesInRect(rect), columnIndexStart = _b.columnIndexStart, columnIndexEnd = _b.columnIndexEnd;
        for (var rowIndex = rowIndexStart; rowIndex <= rowIndexEnd; rowIndex++) {
            for (var columnIndex = columnIndexStart; columnIndex <= columnIndexEnd; columnIndex++) {
                results.push(callback(rowIndex, columnIndex));
            }
        }
        return results;
    };
    /**
     * Maps each row that intersects with the given `Rect` argument.
     *
     * See Grid.mapCellsInRect for more details.
     */
    Grid.prototype.mapRowsInRect = function (rect, callback) {
        var results = [];
        if (rect == null) {
            return results;
        }
        var _a = this.getRowIndicesInRect(rect), rowIndexStart = _a.rowIndexStart, rowIndexEnd = _a.rowIndexEnd;
        for (var rowIndex = rowIndexStart; rowIndex <= rowIndexEnd; rowIndex++) {
            results.push(callback(rowIndex));
        }
        return results;
    };
    /**
     * Maps each column that intersects with the given `Rect` argument.
     *
     * See Grid.mapCellsInRect for more details.
     */
    Grid.prototype.mapColumnsInRect = function (rect, callback) {
        var results = [];
        if (rect == null) {
            return results;
        }
        var _a = this.getColumnIndicesInRect(rect), columnIndexStart = _a.columnIndexStart, columnIndexEnd = _a.columnIndexEnd;
        for (var columnIndex = columnIndexStart; columnIndex <= columnIndexEnd; columnIndex++) {
            results.push(callback(columnIndex));
        }
        return results;
    };
    /**
     * Returns the start and end indices of rows that intersect with the given
     * `Rect` argument.
     */
    Grid.prototype.getRowIndicesInRect = function (rect, includeGhostCells, limit) {
        if (includeGhostCells === void 0) { includeGhostCells = false; }
        if (limit === void 0) { limit = Grid.DEFAULT_MAX_ROWS; }
        if (rect == null) {
            return { rowIndexEnd: 0, rowIndexStart: 0 };
        }
        var searchEnd = includeGhostCells ? Math.max(this.numRows, Grid.DEFAULT_MAX_ROWS) : this.numRows;
        var _a = this.getIndicesInInterval(rect.top, rect.top + rect.height, searchEnd, !includeGhostCells, this.getCumulativeHeightAt), start = _a.start, end = _a.end;
        if (limit > 0 && end - start > limit) {
            end = start + limit;
        }
        return {
            rowIndexEnd: end,
            rowIndexStart: start,
        };
    };
    /**
     * Returns the start and end indices of columns that intersect with the
     * given `Rect` argument.
     */
    Grid.prototype.getColumnIndicesInRect = function (rect, includeGhostCells, limit) {
        if (includeGhostCells === void 0) { includeGhostCells = false; }
        if (limit === void 0) { limit = Grid.DEFAULT_MAX_COLUMNS; }
        if (rect == null) {
            return { columnIndexEnd: 0, columnIndexStart: 0 };
        }
        var searchEnd = includeGhostCells ? Math.max(this.numCols, Grid.DEFAULT_MAX_COLUMNS) : this.numCols;
        var _a = this.getIndicesInInterval(rect.left, rect.left + rect.width, searchEnd, !includeGhostCells, this.getCumulativeWidthAt), start = _a.start, end = _a.end;
        if (limit > 0 && end - start > limit) {
            end = start + limit;
        }
        return {
            columnIndexEnd: end,
            columnIndexStart: start,
        };
    };
    Grid.prototype.isGhostIndex = function (rowIndex, columnIndex) {
        return (rowIndex >= this.numRows || columnIndex >= this.numCols);
    };
    Grid.prototype.getExtremaClasses = function (rowIndex, columnIndex, rowEnd, columnEnd) {
        if (rowIndex === rowEnd && columnIndex === columnEnd) {
            return EXTREMA_LAST_IN_ROW_AND_COLUMN;
        }
        if (rowIndex === rowEnd) {
            return EXTREMA_LAST_IN_COLUMN;
        }
        if (columnIndex === columnEnd) {
            return EXTREMA_LAST_IN_ROW;
        }
        return EXTREMA_NONE;
    };
    Grid.prototype.getRegionStyle = function (region) {
        var cardinality = regions_1.Regions.getRegionCardinality(region);
        switch (cardinality) {
            case regions_1.RegionCardinality.CELLS: {
                var cellRect0 = this.getCellRect(region.rows[0], region.cols[0]);
                var cellRect1 = this.getCellRect(region.rows[1], region.cols[1]);
                var offsetLeft = region.cols[0] === 0 ? 0 : 1;
                var offsetTop = region.rows[0] === 0 ? 0 : 1;
                var rect = cellRect0.union(cellRect1);
                rect.height += offsetTop;
                rect.left -= offsetLeft;
                rect.width += offsetLeft;
                rect.top -= offsetTop;
                return Object.assign(rect.style(), { display: "block" });
            }
            case regions_1.RegionCardinality.FULL_COLUMNS: {
                var cellRect0 = this.getCellRect(0, region.cols[0]);
                var cellRect1 = this.getCellRect(0, region.cols[1]);
                var rect = cellRect0.union(cellRect1);
                var offsetLeft = region.cols[0] === 0 ? 0 : 1;
                return {
                    bottom: 0,
                    display: "block",
                    left: rect.left - offsetLeft,
                    top: 0,
                    width: rect.width + offsetLeft,
                };
            }
            case regions_1.RegionCardinality.FULL_ROWS: {
                var cellRect0 = this.getCellRect(region.rows[0], 0);
                var cellRect1 = this.getCellRect(region.rows[1], 0);
                var rect = cellRect0.union(cellRect1);
                var offsetTop = region.rows[0] === 0 ? 0 : 1;
                return {
                    display: "block",
                    height: rect.height + offsetTop,
                    left: 0,
                    right: 0,
                    top: rect.top - offsetTop,
                };
            }
            case regions_1.RegionCardinality.FULL_TABLE:
                return {
                    bottom: 0,
                    display: "block",
                    left: 0,
                    right: 0,
                    top: 0,
                };
            default: return { display: "none" };
        }
    };
    Grid.prototype.getIndicesInInterval = function (min, max, count, useEndBleed, lookup) {
        var start = utils_1.Utils.binarySearch(min, count - 1, lookup);
        var end = utils_1.Utils.binarySearch(max, count - 1, lookup);
        // correct exact pixel alignment
        if (start >= 0 && min === lookup(start)) {
            start += 1;
        }
        // apply bounded bleeds
        start = Math.max(0, start - this.bleed);
        if (useEndBleed) {
            end = Math.min(count - 1, end + this.bleed);
        }
        else {
            end = Math.min(count - 1, end);
        }
        return { start: start, end: end };
    };
    Grid.DEFAULT_BLEED = 3;
    Grid.DEFAULT_MAX_COLUMNS = 50;
    Grid.DEFAULT_MAX_ROWS = 200;
    Grid.DEFAULT_GHOST_HEIGHT = 20;
    Grid.DEFAULT_GHOST_WIDTH = 150;
    return Grid;
}());
exports.Grid = Grid;

//# sourceMappingURL=data:application/json;base64,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
