/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
/**
 * A simple object for storing the client bounds of HTMLElements. Since
 * ClientRects are immutable, this object enables editing and some simple
 * manipulation methods.
 */
var Rect = (function () {
    function Rect(left, top, width, height) {
        this.left = left;
        this.top = top;
        this.width = width;
        this.height = height;
    }
    /**
     * Returns the smallest Rect that entirely contains the supplied rects
     */
    Rect.union = function (anyRect0, anyRect1) {
        var rect0 = Rect.wrap(anyRect0);
        var rect1 = Rect.wrap(anyRect1);
        var top = Math.min(rect0.top, rect1.top);
        var left = Math.min(rect0.left, rect1.left);
        var bottom = Math.max(rect0.top + rect0.height, rect1.top + rect1.height);
        var right = Math.max(rect0.left + rect0.width, rect1.left + rect1.width);
        var height = bottom - top;
        var width = right - left;
        return new Rect(left, top, width, height);
    };
    /**
     * Returns a new Rect that subtracts the origin of the second argument
     * from the first.
     */
    Rect.subtractOrigin = function (anyRect0, anyRect1) {
        var rect0 = Rect.wrap(anyRect0);
        var rect1 = Rect.wrap(anyRect1);
        return new Rect(rect0.left - rect1.left, rect0.top - rect1.top, rect0.width, rect0.height);
    };
    /**
     * Returns the CSS properties representing the absolute positioning of
     * this Rect.
     */
    Rect.style = function (rect) {
        return {
            height: rect.height + "px",
            left: rect.left + "px",
            position: "absolute",
            top: rect.top + "px",
            width: rect.width + "px",
        };
    };
    /**
     * Given a ClientRect or Rect object, returns a Rect object.
     */
    Rect.wrap = function (rect) {
        if (rect instanceof Rect) {
            return rect;
        }
        else {
            return new Rect(rect.left, rect.top, rect.width, rect.height);
        }
    };
    Rect.prototype.subtractOrigin = function (anyRect) {
        return Rect.subtractOrigin(this, anyRect);
    };
    Rect.prototype.union = function (anyRect) {
        return Rect.union(this, anyRect);
    };
    Rect.prototype.style = function () {
        return Rect.style(this);
    };
    Rect.prototype.sizeStyle = function () {
        return {
            height: this.height + "px",
            width: this.width + "px",
        };
    };
    Rect.prototype.containsX = function (clientX) {
        return clientX >= this.left && clientX <= this.left + this.width;
    };
    Rect.prototype.containsY = function (clientY) {
        return clientY >= this.top && clientY <= this.top + this.height;
    };
    Rect.ORIGIN = new Rect(0, 0, 0, 0);
    return Rect;
}());
exports.Rect = Rect;

//# sourceMappingURL=data:application/json;base64,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
