/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var classNames = require("classnames");
var React = require("react");
;
/* tslint:enable:interface-name */
/**
 * Since Firefox doesn't provide a computed "font" property, we manually
 * construct it using the ordered properties that can be specifed in CSS.
 */
var CSS_FONT_PROPERTIES = [
    "font-style",
    "font-variant",
    "font-weight",
    "font-size",
    "font-family",
];
exports.Utils = {
    /**
     * Returns a clone of the ReactElement with a className that includes the
     * element's original className and any other classes passed in with variadic
     * arguments matching the `classNames` api.
     */
    assignClasses: function (elem) {
        var extendedClasses = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            extendedClasses[_i - 1] = arguments[_i];
        }
        var classes = classNames.apply(void 0, [elem.props.className].concat(extendedClasses));
        return React.cloneElement(elem, { className: classes });
    },
    /**
     * Invokes the callback `n` times, collecting the results in an array, which
     * is the return value. Similar to _.times
     */
    times: function (n, callback) {
        return Array.apply(null, Array(n)).map(function (_none, index) { return callback(index); });
    },
    /**
     * Takes an array of numbers, returns an array of numbers of the same length in which each
     * value is the sum of current and previous values in the input array.
     *
     * Example input:  [10, 20, 50]
     *         output: [10, 30, 80]
     */
    accumulate: function (numbers) {
        var result = [];
        var sum = 0;
        for (var i = 0; i < numbers.length; i++) {
            sum += numbers[i];
            result.push(sum);
        }
        return result;
    },
    /**
     * Returns traditional spreadsheet-style column names
     * e.g. (A, B, ..., Z, AA, AB, ..., ZZ, AAA, AAB, ...).
     *
     * Note that this isn't technically mathematically equivalent to base 26 since
     * there is no zero element.
     */
    toBase26Alpha: function (num) {
        var str = "";
        while (true) {
            var letter = num % 26;
            str = String.fromCharCode(65 + letter) + str;
            num = num - letter;
            if (num <= 0) {
                return str;
            }
            num = (num / 26) - 1;
        }
    },
    /**
     * Performs the binary search algorithm to find the index of the `value`
     * parameter in a sorted list of numbers. If `value` is not in the list, the
     * index where `value` can be inserted to maintain the sort is returned.
     *
     * Unlike a typical binary search implementation, we use a `lookup`
     * callback to access the sorted list of numbers instead of an array. This
     * avoids additional storage overhead.
     *
     * We use this to, for example, find the index of a row/col given its client
     * coordinate.
     *
     * Adapted from lodash https://github.com/lodash/lodash/blob/4.11.2/lodash.js#L3579
     *
     * @param value - the query value
     * @param high - the length of the sorted list of numbers
     * @param lookup - returns the number from the list at the supplied index
     */
    binarySearch: function (value, high, lookup) {
        var low = 0;
        while (low < high) {
            var mid = Math.floor((low + high) / 2.0);
            var computed = lookup(mid);
            if (computed < value) {
                low = mid + 1;
            }
            else {
                high = mid;
            }
        }
        return high;
    },
    /**
     * Returns a copy of the array that will have a length of the supplied parameter.
     * If the array is too long, it will be truncated. If it is too short, it will be
     * filled with the suppleid `fillValue` argument.
     *
     * @param array - the `Array` to copy and adjust
     * @param length - the target length of the array
     * @param fillValue - the value to add to the array if it is too short
     */
    arrayOfLength: function (array, length, fillValue) {
        if (array.length > length) {
            return array.slice(0, length);
        }
        array = array.slice();
        while (array.length < length) {
            array.push(fillValue);
        }
        return array;
    },
    /**
     * Takes in one full array of values and one sparse array of the same
     * length and type. Returns a copy of the `defaults` array, where each
     * value is replaced with the corresponding non-null value at the same
     * index in `sparseOverrides`.
     *
     * @param defaults - the full array of default values
     * @param sparseOverrides - the sparse array of override values
     */
    assignSparseValues: function (defaults, sparseOverrides) {
        if (sparseOverrides == null || defaults.length !== sparseOverrides.length) {
            return defaults;
        }
        defaults = defaults.slice();
        for (var i = 0; i < defaults.length; i++) {
            var override = sparseOverrides[i];
            if (override != null) {
                defaults[i] = override;
            }
        }
        return defaults;
    },
    /**
     * Measures the bounds of supplied element's textContent.
     *
     * We use the computed font from the supplied element and a non-DOM canvas
     * context to measure the text.
     *
     * Returns a `TextMetrics` object.
     */
    measureElementTextContent: function (element) {
        var context = document.createElement("canvas").getContext("2d");
        var style = getComputedStyle(element, null);
        context.font = CSS_FONT_PROPERTIES.map(function (prop) { return style.getPropertyValue(prop); }).join(" ");
        return context.measureText(element.textContent);
    },
    /**
     * Given a number, returns a value that is clamped within a
     * minimum/maximum bounded range. The minimum and maximum are optional. If
     * either is missing, that extrema limit is not applied.
     *
     * Assumes max >= min.
     */
    clamp: function (value, min, max) {
        if (min != null && value < min) {
            value = min;
        }
        if (max != null && value > max) {
            value = max;
        }
        return value;
    },
};

//# sourceMappingURL=data:application/json;base64,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
