/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var PureRender = require("pure-render-decorator");
var React = require("react");
var ReactDOM = require("react-dom");
/**
 * This component provides a simple interface for combined drag and/or click
 * events.
 *
 * Since the mouse interactions for drag and click are overloaded, here are
 * the events that will fire in these cases:
 *
 * A Click Interaction
 * 1. The user presses down on the render element, triggering the onActivate
 *    callback.
 * 2. The user releases the mouse button without moving it, triggering the
 *    onClick callback.
 *
 * A Drag Interaction
 * 1. The user presses down on the render element, triggering the onActivate
 *    callback.
 * 2. The user moves the mouse, triggering the onDragMove callback.
 * 3. The user moves the mouse, triggering the onDragMove callback.
 * 4. The user moves the mouse, triggering the onDragMove callback.
 * 5. The user releases the mouse button, triggering a final onDragMove
 *    callback as well as an onDragEnd callback.
 *
 * If `false` is returned from the onActivate callback, no further events
 * will be fired until the next activation.
 */
var Draggable = (function (_super) {
    __extends(Draggable, _super);
    function Draggable() {
        _super.apply(this, arguments);
    }
    Draggable.prototype.render = function () {
        return React.Children.only(this.props.children);
    };
    Draggable.prototype.componentDidMount = function () {
        this.events = new DragEvents();
        this.events.attach(ReactDOM.findDOMNode(this), this.props);
    };
    Draggable.prototype.componentWillUnmount = function () {
        this.events.detach();
        delete this.events;
    };
    Draggable = __decorate([
        PureRender
    ], Draggable);
    return Draggable;
}(React.Component));
exports.Draggable = Draggable;
var DragEvents = (function () {
    function DragEvents() {
        var _this = this;
        this.handleMouseDown = function (event) {
            _this.initCoordinateData(event);
            if (_this.handler != null && _this.handler.onActivate != null) {
                var exitCode = _this.handler.onActivate(event);
                if (exitCode === false) {
                    return;
                }
            }
            _this.isActivated = true;
            event.preventDefault();
            _this.attachDocumentEventListeners();
        };
        this.handleMouseMove = function (event) {
            event.preventDefault();
            if (_this.isActivated) {
                _this.isDragging = true;
            }
            if (_this.isDragging) {
                var coords = _this.updateCoordinateData(event);
                if (_this.handler != null && _this.handler.onDragMove != null) {
                    _this.handler.onDragMove(event, coords);
                }
            }
        };
        this.handleMouseUp = function (event) {
            event.preventDefault();
            if (_this.handler != null) {
                if (_this.isDragging) {
                    var coords = _this.updateCoordinateData(event);
                    if (_this.handler.onDragMove != null) {
                        _this.handler.onDragMove(event, coords);
                    }
                    if (_this.handler.onDragEnd != null) {
                        _this.handler.onDragEnd(event, coords);
                    }
                }
                else if (_this.isActivated) {
                    if (_this.handler.onDoubleClick != null) {
                        if (_this.doubleClickTimeoutToken == null) {
                            // if this the first click of a possible double-click,
                            // we delay the firing of the click event by the
                            // timeout.
                            _this.doubleClickTimeoutToken = setTimeout(function () {
                                delete _this.doubleClickTimeoutToken;
                                if (_this.handler.onClick != null) {
                                    _this.handler.onClick(event);
                                }
                            }, DragEvents.DOUBLE_CLICK_TIMEOUT_MSEC);
                        }
                        else {
                            // otherwise, this is the second click in the double-
                            // click so we cancel the single-click timeout and
                            // fire the double-click event.
                            clearTimeout(_this.doubleClickTimeoutToken);
                            delete _this.doubleClickTimeoutToken;
                            _this.handler.onDoubleClick(event);
                        }
                    }
                    else if (_this.handler.onClick != null) {
                        _this.handler.onClick(event);
                    }
                }
            }
            _this.isActivated = false;
            _this.isDragging = false;
            _this.detachDocumentEventListeners();
        };
    }
    DragEvents.isAdditive = function (event) {
        return event.ctrlKey || event.metaKey;
    };
    DragEvents.prototype.attach = function (element, handler) {
        this.detach();
        this.handler = handler;
        this.element = element;
        if (this.isValidDragHandler(handler)) {
            this.element.addEventListener("mousedown", this.handleMouseDown);
        }
        return this;
    };
    DragEvents.prototype.detach = function () {
        if (this.element != null) {
            this.element.removeEventListener("mousedown", this.handleMouseDown);
            this.detachDocumentEventListeners();
        }
    };
    DragEvents.prototype.isValidDragHandler = function (handler) {
        return handler != null && (handler.onActivate != null
            || handler.onDragMove != null
            || handler.onDragEnd != null
            || handler.onClick != null
            || handler.onDoubleClick != null);
    };
    DragEvents.prototype.attachDocumentEventListeners = function () {
        document.addEventListener("mousemove", this.handleMouseMove);
        document.addEventListener("mouseup", this.handleMouseUp);
    };
    DragEvents.prototype.detachDocumentEventListeners = function () {
        document.removeEventListener("mousemove", this.handleMouseMove);
        document.removeEventListener("mouseup", this.handleMouseUp);
    };
    DragEvents.prototype.initCoordinateData = function (event) {
        this.activationCoordinates = [event.clientX, event.clientY];
        this.lastCoordinates = this.activationCoordinates;
    };
    DragEvents.prototype.updateCoordinateData = function (event) {
        var currentCoordinates = [event.clientX, event.clientY];
        var deltaCoordinates = [
            currentCoordinates[0] - this.lastCoordinates[0],
            currentCoordinates[1] - this.lastCoordinates[1],
        ];
        var offsetCoordinates = [
            currentCoordinates[0] - this.activationCoordinates[0],
            currentCoordinates[1] - this.activationCoordinates[1],
        ];
        var data = {
            activation: this.activationCoordinates,
            current: currentCoordinates,
            delta: deltaCoordinates,
            last: this.lastCoordinates,
            offset: offsetCoordinates,
        };
        this.lastCoordinates = [event.clientX, event.clientY];
        return data;
    };
    DragEvents.DOUBLE_CLICK_TIMEOUT_MSEC = 500;
    return DragEvents;
}());
exports.DragEvents = DragEvents;

//# sourceMappingURL=data:application/json;base64,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
