/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
/**
 * Efficiently detect when an HTMLElement is resized.
 *
 * Attaches an invisible "resize-sensor" div to the element. Then it checks
 * the element's offsetWidth and offsetHeight whenever a scroll event is
 * triggered on the "resize-sensor" children. These events are further
 * debounced using requestAnimationFrame.
 *
 * Inspired by: https://github.com/marcj/css-element-queries/blob/master/src/ResizeSensor.js
 */
var ResizeSensor = (function () {
    function ResizeSensor() {
    }
    ResizeSensor.attach = function (element, callback) {
        var lifecycle = ResizeSensor.debounce(callback);
        var resizeSensor = document.createElement("div");
        resizeSensor.className = "bp-table-resize-sensor";
        resizeSensor.style.cssText = ResizeSensor.RESIZE_SENSOR_STYLE;
        resizeSensor.innerHTML = ResizeSensor.RESIZE_SENSOR_HTML;
        element.appendChild(resizeSensor);
        if (getComputedStyle(element, null).getPropertyValue("position") === "static") {
            element.style.position = "relative";
        }
        var expand = resizeSensor.childNodes[0];
        var expandChild = expand.childNodes[0];
        var shrink = resizeSensor.childNodes[1];
        var reset = function () {
            expandChild.style.width = "100000px";
            expandChild.style.height = "100000px";
            expand.scrollLeft = 100000;
            expand.scrollTop = 100000;
            shrink.scrollLeft = 100000;
            shrink.scrollTop = 100000;
        };
        reset();
        var lastWidth;
        var lastHeight;
        var onScroll = function () {
            var currentWidth = element.offsetWidth;
            var currentHeight = element.offsetHeight;
            if (currentWidth !== lastWidth || currentHeight !== lastHeight) {
                lastWidth = currentWidth;
                lastHeight = currentHeight;
                lifecycle.trigger();
            }
            reset();
        };
        expand.addEventListener("scroll", onScroll);
        shrink.addEventListener("scroll", onScroll);
        return function () {
            element.removeChild(resizeSensor);
            lifecycle.cancelled = true;
        };
    };
    ResizeSensor.debounce = function (callback) {
        var scope = {
            cancelled: false,
            trigger: function () {
                if (scope.triggered || scope.cancelled) {
                    return;
                }
                scope.triggered = true;
                requestAnimationFrame(function () {
                    scope.triggered = false;
                    if (!scope.cancelled) {
                        callback();
                    }
                });
            },
            triggered: false,
        };
        return scope;
    };
    ResizeSensor.RESIZE_SENSOR_STYLE = "position: absolute; left: 0; top: 0; right: 0; " +
        "bottom: 0; overflow: hidden; z-index: -1; visibility: hidden;";
    ResizeSensor.RESIZE_SENSOR_HTML = "<div class=\"bp-table-resize-sensor-expand\"\n        style=\"" + ResizeSensor.RESIZE_SENSOR_STYLE + "\"><div style=\"position: absolute; left: 0; top: 0; transition: 0s;\"\n        ></div></div><div class=\"bp-table-resize-sensor-shrink\" style=\"" + ResizeSensor.RESIZE_SENSOR_STYLE + "\"\n        ><div style=\"position: absolute; left: 0; top: 0; transition: 0s; width: 200%; height: 200%;\"></div></div>";
    return ResizeSensor;
}());
exports.ResizeSensor = ResizeSensor;

//# sourceMappingURL=data:application/json;base64,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
