/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var rect_1 = require("./common/rect");
var utils_1 = require("./common/utils");
var Locator = (function () {
    function Locator(tableElement, bodyElement, grid) {
        var _this = this;
        this.tableElement = tableElement;
        this.bodyElement = bodyElement;
        this.grid = grid;
        this.convertCellIndexToClientX = function (index) {
            var bodyRect = _this.getBodyRect();
            return bodyRect.left + _this.grid.getCumulativeWidthAt(index);
        };
        this.convertCellIndexToClientY = function (index) {
            var bodyRect = _this.getBodyRect();
            return bodyRect.top + _this.grid.getCumulativeHeightAt(index);
        };
    }
    Locator.prototype.setGrid = function (grid) {
        this.grid = grid;
    };
    Locator.prototype.getViewportRect = function () {
        return new rect_1.Rect(this.bodyElement.scrollLeft, this.bodyElement.scrollTop, this.bodyElement.clientWidth, this.bodyElement.clientHeight);
    };
    Locator.prototype.getWidestVisibleCellInColumn = function (columnIndex) {
        var cellClasses = [
            (".bp-table-cell-col-" + columnIndex),
            ".bp-table-column-name",
        ];
        var cells = this.tableElement.querySelectorAll(cellClasses.join(", "));
        var max = 0;
        for (var i = 0; i < cells.length; i++) {
            var contentWidth = utils_1.Utils.measureElementTextContent(cells.item(i)).width;
            var cellWidth = Math.ceil(contentWidth) + Locator.CELL_HORIZONTAL_PADDING * 2;
            if (cellWidth > max) {
                max = cellWidth;
            }
        }
        return max;
    };
    Locator.prototype.convertPointToColumn = function (clientX) {
        var tableRect = this.getTableRect();
        if (!tableRect.containsX(clientX)) {
            return -1;
        }
        return utils_1.Utils.binarySearch(clientX, this.grid.numCols - 1, this.convertCellIndexToClientX);
    };
    Locator.prototype.convertPointToRow = function (clientY) {
        var tableRect = this.getTableRect();
        if (!tableRect.containsY(clientY)) {
            return -1;
        }
        return utils_1.Utils.binarySearch(clientY, this.grid.numRows - 1, this.convertCellIndexToClientY);
    };
    Locator.prototype.convertPointToCell = function (clientX, clientY) {
        var col = utils_1.Utils.binarySearch(clientX, this.grid.numCols - 1, this.convertCellIndexToClientX);
        var row = utils_1.Utils.binarySearch(clientY, this.grid.numRows - 1, this.convertCellIndexToClientY);
        return { col: col, row: row };
    };
    Locator.prototype.getTableRect = function () {
        return rect_1.Rect.wrap(this.tableElement.getBoundingClientRect());
    };
    Locator.prototype.getBodyRect = function () {
        return this.unscrollElementRect(this.bodyElement);
    };
    /**
     * Subtracts the scroll offset from the element's bounding client rect.
     */
    Locator.prototype.unscrollElementRect = function (element) {
        var rect = rect_1.Rect.wrap(element.getBoundingClientRect());
        rect.left -= element.scrollLeft;
        rect.top -= element.scrollTop;
        return rect;
    };
    Locator.CELL_HORIZONTAL_PADDING = 10;
    return Locator;
}());
exports.Locator = Locator;

//# sourceMappingURL=data:application/json;base64,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
