/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var utils_1 = require("./common/utils");
/**
 * `Region`s contain sets of cells. Additionally, a distinction is drawn, for
 * example, between all cells within a column and the whole column itself.
 * The `RegionCardinality` enum represents these distinct types of `Region`s.
 */
(function (RegionCardinality) {
    /**
     * A region that contains a finite rectangular group of table cells
     */
    RegionCardinality[RegionCardinality["CELLS"] = 0] = "CELLS";
    /**
     * A region that represents all cells within 1 or more rows.
     */
    RegionCardinality[RegionCardinality["FULL_ROWS"] = 1] = "FULL_ROWS";
    /**
     * A region that represents all cells within 1 or more columns.
     */
    RegionCardinality[RegionCardinality["FULL_COLUMNS"] = 2] = "FULL_COLUMNS";
    /**
     * A region that represents all cells in the table.
     */
    RegionCardinality[RegionCardinality["FULL_TABLE"] = 3] = "FULL_TABLE";
})(exports.RegionCardinality || (exports.RegionCardinality = {}));
var RegionCardinality = exports.RegionCardinality;
/**
 * A convenience object for subsets of `RegionCardinality` that are commonly
 * used as the `selectionMode` prop of the `<Table>`.
 */
exports.SelectionModes = {
    ALL: [
        RegionCardinality.FULL_COLUMNS,
        RegionCardinality.FULL_ROWS,
        RegionCardinality.CELLS,
    ],
    COLUMNS_AND_CELLS: [
        RegionCardinality.FULL_COLUMNS,
        RegionCardinality.CELLS,
    ],
    COLUMNS_ONLY: [
        RegionCardinality.FULL_COLUMNS,
    ],
    NONE: [],
    ROWS_AND_CELLS: [
        RegionCardinality.FULL_ROWS,
        RegionCardinality.CELLS,
    ],
    ROWS_ONLY: [
        RegionCardinality.FULL_ROWS,
    ],
};
var Regions = (function () {
    function Regions() {
    }
    /**
     * Determines the cardinality of a region. We use null values to indicate
     * an unbounded interval. Therefore, an example of a region containing the
     * second and third columns would be:
     *
     *     {
     *         rows: null,
     *         cols: [1, 2]
     *     }
     *
     * In this case, this method would return RegionCardinality.FULL_COLUMNS.
     * An example of a region containing a single cell in the table would be:
     *
     *     {
     *         rows: [5, 5],
     *         cols: [2, 2]
     *     }
     *
     * In this case, this method would return RegionCardinality.CELLS.
     */
    Regions.getRegionCardinality = function (region) {
        if (region.cols != null && region.rows != null) {
            return RegionCardinality.CELLS;
        }
        else if (region.cols != null) {
            return RegionCardinality.FULL_COLUMNS;
        }
        else if (region.rows != null) {
            return RegionCardinality.FULL_ROWS;
        }
        else {
            return RegionCardinality.FULL_TABLE;
        }
    };
    /**
     * Returns a region containing one or more cells.
     */
    Regions.cell = function (row, col, row2, col2) {
        return {
            cols: this.normalizeInterval(col, col2),
            rows: this.normalizeInterval(row, row2),
        };
    };
    /**
     * Returns a region containing one or more full rows.
     */
    Regions.row = function (row, row2) {
        return { rows: this.normalizeInterval(row, row2) };
    };
    /**
     * Returns a region containing one or more full columns.
     */
    Regions.column = function (col, col2) {
        return { cols: this.normalizeInterval(col, col2) };
    };
    /**
     * Adds the region to the end of a cloned copy of the supplied region
     * array.
     */
    Regions.add = function (regions, region) {
        var copy = regions.slice();
        copy.push(region);
        return copy;
    };
    /**
     * Replaces the region at the end of a cloned copy of the supplied region
     * array.
     */
    Regions.update = function (regions, region) {
        var copy = regions.slice();
        copy.pop();
        copy.push(region);
        return copy;
    };
    /**
     * Returns true iff the specified region is equal to the last region in
     * the region list. This allows us to avoid immediate additive re-selection.
     */
    Regions.lastRegionIsEqual = function (regions, region) {
        if (regions == null || regions.length === 0) {
            return false;
        }
        var lastRegion = regions[regions.length - 1];
        return Regions.regionsEqual(lastRegion, region);
    };
    /**
     * Returns the index of the region that is equal to the supplied
     * parameter. Returns -1 if no such region is found.
     */
    Regions.findMatchingRegion = function (regions, region) {
        if (regions == null) {
            return -1;
        }
        for (var i = 0; i < regions.length; i++) {
            if (Regions.regionsEqual(regions[i], region)) {
                return i;
            }
        }
        return -1;
    };
    /**
     * Returns true if the regions contain a region that has FULL_COLUMNS
     * cardinality and contains the specified column index.
     */
    Regions.hasFullColumn = function (regions, col) {
        if (regions == null) {
            return false;
        }
        for (var _i = 0, regions_1 = regions; _i < regions_1.length; _i++) {
            var region = regions_1[_i];
            var cardinality = Regions.getRegionCardinality(region);
            if (cardinality === RegionCardinality.FULL_TABLE) {
                return true;
            }
            if (cardinality === RegionCardinality.FULL_COLUMNS && Regions.intervalContainsIndex(region.cols, col)) {
                return true;
            }
        }
        return false;
    };
    /**
     * Returns true if the regions contain a region that has FULL_ROWS
     * cardinality and contains the specified row index.
     */
    Regions.hasFullRow = function (regions, row) {
        if (regions == null) {
            return false;
        }
        for (var _i = 0, regions_2 = regions; _i < regions_2.length; _i++) {
            var region = regions_2[_i];
            var cardinality = Regions.getRegionCardinality(region);
            if (cardinality === RegionCardinality.FULL_TABLE) {
                return true;
            }
            if (cardinality === RegionCardinality.FULL_ROWS && Regions.intervalContainsIndex(region.rows, row)) {
                return true;
            }
        }
        return false;
    };
    /**
     * Returns true if the regions contain the query region. The query region
     * may be a subset of the `regions` parameter.
     */
    Regions.containsRegion = function (regions, query) {
        if (regions == null || query == null) {
            return false;
        }
        for (var _i = 0, regions_3 = regions; _i < regions_3.length; _i++) {
            var region = regions_3[_i];
            var cardinality = Regions.getRegionCardinality(region);
            switch (cardinality) {
                case RegionCardinality.FULL_TABLE:
                    return true;
                case RegionCardinality.FULL_COLUMNS:
                    if (Regions.intervalOverlaps(region.cols, query.cols)) {
                        return true;
                    }
                    continue;
                case RegionCardinality.FULL_ROWS:
                    if (Regions.intervalOverlaps(region.rows, query.rows)) {
                        return true;
                    }
                    continue;
                case RegionCardinality.CELLS:
                    if (Regions.intervalOverlaps(region.cols, query.cols)
                        && Regions.intervalOverlaps(region.rows, query.rows)) {
                        return true;
                    }
                    continue;
                default:
                    break;
            }
        }
        return false;
    };
    Regions.eachUniqueFullColumn = function (regions, iteratee) {
        if (regions == null || regions.length === 0 || iteratee == null) {
            return;
        }
        var seen = {};
        regions.forEach(function (region) {
            if (Regions.getRegionCardinality(region) === RegionCardinality.FULL_COLUMNS) {
                var _a = region.cols, start = _a[0], end = _a[1];
                for (var col = start; col <= end; col++) {
                    if (!seen[col]) {
                        seen[col] = true;
                        iteratee(col);
                    }
                }
            }
        });
    };
    /**
     * Using the supplied array of non-contiguous `IRegion`s, this method
     * returns an ordered array of every unique cell that exists in those
     * regions.
     */
    Regions.enumerateUniqueCells = function (regions, numRows, numCols) {
        if (regions == null || regions.length === 0) {
            return [];
        }
        var seen = {};
        var list = [];
        for (var _i = 0, regions_4 = regions; _i < regions_4.length; _i++) {
            var region = regions_4[_i];
            Regions.eachCellInRegion(region, numRows, numCols, function (row, col) {
                // add to list if not seen
                var key = row + "-" + col;
                if (seen[key] !== true) {
                    seen[key] = true;
                    list.push([row, col]);
                }
            });
        }
        // sort list by rows then columns
        list.sort(Regions.rowFirstComparator);
        return list;
    };
    /**
     * Maps a dense array of cell coordinates to a sparse 2-dimensional array
     * of cell values.
     *
     * We create a new 2-dimensional array representing the smallest single
     * contiguous `IRegion` that contains all cells in the supplied array. We
     * invoke the mapper callback only on the cells in the supplied coordinate
     * array and store the result. Returns the resulting 2-dimensional array.
     */
    Regions.sparseMapCells = function (cells, mapper) {
        var bounds = Regions.getBoundingRegion(cells);
        if (bounds == null) {
            return null;
        }
        var numRows = bounds.rows[1] + 1 - bounds.rows[0];
        var numCols = bounds.cols[1] + 1 - bounds.cols[0];
        var result = utils_1.Utils.times(numRows, function () { return new Array(numCols); });
        cells.forEach(function (_a) {
            var row = _a[0], col = _a[1];
            result[row - bounds.rows[0]][col - bounds.cols[0]] = mapper(row, col);
        });
        return result;
    };
    /**
     * Returns the smallest single contiguous `IRegion` that contains all cells in the
     * supplied array.
     */
    Regions.getBoundingRegion = function (cells) {
        var minRow;
        var maxRow;
        var minCol;
        var maxCol;
        for (var _i = 0, cells_1 = cells; _i < cells_1.length; _i++) {
            var _a = cells_1[_i], row = _a[0], col = _a[1];
            minRow = (minRow == null || row < minRow) ? row : minRow;
            maxRow = (maxRow == null || row > maxRow) ? row : maxRow;
            minCol = (minCol == null || col < minCol) ? col : minCol;
            maxCol = (maxCol == null || col > maxCol) ? col : maxCol;
        }
        if (minRow == null) {
            return null;
        }
        return {
            cols: [minCol, maxCol],
            rows: [minRow, maxRow],
        };
    };
    Regions.isValid = function (region) {
        if (region == null) {
            return false;
        }
        if ((region.rows != null) && (region.rows[0] < 0 || region.rows[1] < 0)) {
            return false;
        }
        if ((region.cols != null) && (region.cols[0] < 0 || region.cols[1] < 0)) {
            return false;
        }
        return true;
    };
    Regions.joinStyledRegionGroups = function (selectedRegions, otherRegions) {
        var regionGroups = [];
        if (otherRegions != null) {
            regionGroups = regionGroups.concat(otherRegions);
        }
        if (selectedRegions != null && selectedRegions.length > 0) {
            regionGroups.push({
                className: "bp-table-selection-region",
                regions: selectedRegions,
            });
        }
        return regionGroups;
    };
    /**
     * Iterates over the cells within an `IRegion`, invoking the callback with
     * each cell's coordinates.
     */
    Regions.eachCellInRegion = function (region, numRows, numCols, iteratee) {
        var cardinality = Regions.getRegionCardinality(region);
        switch (cardinality) {
            case RegionCardinality.FULL_TABLE:
                for (var row = 0; row < numRows; row++) {
                    for (var col = 0; col < numCols; col++) {
                        iteratee(row, col);
                    }
                }
                break;
            case RegionCardinality.FULL_COLUMNS:
                for (var row = 0; row < numRows; row++) {
                    for (var col = region.cols[0]; col <= region.cols[1]; col++) {
                        iteratee(row, col);
                    }
                }
                break;
            case RegionCardinality.FULL_ROWS:
                for (var row = region.rows[0]; row <= region.rows[1]; row++) {
                    for (var col = 0; col < numCols; col++) {
                        iteratee(row, col);
                    }
                }
                break;
            case RegionCardinality.CELLS:
                for (var row = region.rows[0]; row <= region.rows[1]; row++) {
                    for (var col = region.cols[0]; col <= region.cols[1]; col++) {
                        iteratee(row, col);
                    }
                }
                break;
            default:
                break;
        }
    };
    Regions.regionsEqual = function (regionA, regionB) {
        return Regions.intervalsEqual(regionA.rows, regionB.rows)
            && Regions.intervalsEqual(regionA.cols, regionB.cols);
    };
    Regions.intervalsEqual = function (ivalA, ivalB) {
        if (ivalA == null) {
            return ivalB == null;
        }
        else if (ivalB == null) {
            return false;
        }
        else {
            return ivalA[0] === ivalB[0] && ivalA[1] === ivalB[1];
        }
    };
    Regions.intervalContainsIndex = function (interval, index) {
        if (interval == null) {
            return false;
        }
        return interval[0] <= index && interval[1] >= index;
    };
    Regions.intervalOverlaps = function (ivalA, ivalB) {
        if (ivalA == null || ivalB == null) {
            return false;
        }
        if (ivalA[1] < ivalB[0] || ivalA[0] > ivalB[1]) {
            return false;
        }
        return true;
    };
    Regions.rowFirstComparator = function (a, b) {
        var rowDiff = a[0] - b[0];
        return rowDiff === 0 ? a[1] - b[1] : rowDiff;
    };
    Regions.numericalComparator = function (a, b) {
        return a - b;
    };
    Regions.normalizeInterval = function (coord, coord2) {
        if (coord2 == null) {
            coord2 = coord;
        }
        var interval = [coord, coord2];
        interval.sort(Regions.numericalComparator);
        return interval;
    };
    return Regions;
}());
exports.Regions = Regions;

//# sourceMappingURL=data:application/json;base64,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
