/**
 * Copyright 2016 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 - http://www.apache.org/licenses/LICENSE-2.0
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var __assign = (this && this.__assign) || Object.assign || function(t) {
    for (var s, i = 1, n = arguments.length; i < n; i++) {
        s = arguments[i];
        for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
            t[p] = s[p];
    }
    return t;
};
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var column_1 = require("./column");
var grid_1 = require("./common/grid");
var utils_1 = require("./common/utils");
var columnHeader_1 = require("./headers/columnHeader");
var columnHeaderCell_1 = require("./headers/columnHeaderCell");
var rowHeader_1 = require("./headers/rowHeader");
var resizeSensor_1 = require("./interactions/resizeSensor");
var guides_1 = require("./layers/guides");
var regions_1 = require("./layers/regions");
var locator_1 = require("./locator");
var regions_2 = require("./regions");
var tableBody_1 = require("./tableBody");
var core_1 = require("@blueprintjs/core");
var classNames = require("classnames");
var PureRender = require("pure-render-decorator");
var React = require("react");
var Table = (function (_super) {
    __extends(Table, _super);
    function Table(props, context) {
        var _this = this;
        _super.call(this, props, context);
        this.columnHeaderCellRenderer = function (columnIndex) {
            var props = _this.getColumnProps(columnIndex);
            var renderColumnHeader = props.renderColumnHeader;
            if (renderColumnHeader != null) {
                return renderColumnHeader(columnIndex);
            }
            else if (props.name != null) {
                return React.createElement(columnHeaderCell_1.ColumnHeaderCell, __assign({}, props));
            }
            else {
                return React.createElement(columnHeaderCell_1.ColumnHeaderCell, __assign({}, props, {name: utils_1.Utils.toBase26Alpha(columnIndex)}));
            }
        };
        this.bodyCellRenderer = function (rowIndex, columnIndex) {
            return _this.getColumnProps(columnIndex).renderCell(rowIndex, columnIndex);
        };
        this.handleColumnWidthChanged = function (columnIndex, width) {
            var columnWidths = _this.state.columnWidths.slice();
            columnWidths[columnIndex] = width;
            _this.invalidateGrid();
            _this.setState({ columnWidths: columnWidths });
            var onColumnWidthChanged = _this.props.onColumnWidthChanged;
            if (onColumnWidthChanged != null) {
                onColumnWidthChanged(columnIndex, width);
            }
        };
        this.handleRowHeightChanged = function (rowIndex, height) {
            var rowHeights = _this.state.rowHeights.slice();
            rowHeights[rowIndex] = height;
            _this.invalidateGrid();
            _this.setState({ rowHeights: rowHeights });
            var onRowHeightChanged = _this.props.onRowHeightChanged;
            if (onRowHeightChanged != null) {
                onRowHeightChanged(rowIndex, height);
            }
        };
        this.handleRootScroll = function (_event) {
            // Bug #211 - Native browser text selection events can cause the root
            // element to scroll even though it has a overflow:hidden style. The
            // only viable solution to this is to unscroll the element after the
            // browser scrolls it.
            if (_this.rootTableElement != null) {
                _this.rootTableElement.scrollLeft = 0;
                _this.rootTableElement.scrollTop = 0;
            }
        };
        this.handleBodyScroll = function (event) {
            // Prevent the event from propagating to avoid a resize event on the
            // resize sensor.
            event.stopPropagation();
            var _a = _this.state, locator = _a.locator, isLayoutLocked = _a.isLayoutLocked;
            if (locator != null && !isLayoutLocked) {
                var viewportRect = locator.getViewportRect();
                _this.setState({ viewportRect: viewportRect });
            }
        };
        this.handleColumnResizeGuide = function (verticalGuides) {
            _this.setState({ verticalGuides: verticalGuides });
        };
        this.handleRowResizeGuide = function (horizontalGuides) {
            _this.setState({ horizontalGuides: horizontalGuides });
        };
        this.clearSelection = function (_selectedRegions) {
            _this.handleSelection([]);
        };
        this.handleSelection = function (selectedRegions) {
            _this.setState({ selectedRegions: selectedRegions });
            var onSelection = _this.props.onSelection;
            if (onSelection != null) {
                onSelection(selectedRegions);
            }
        };
        this.handleLayoutLock = function (isLayoutLocked) {
            if (isLayoutLocked === void 0) { isLayoutLocked = false; }
            _this.setState({ isLayoutLocked: isLayoutLocked });
        };
        this.setBodyRef = function (ref) { return _this.bodyElement = ref; };
        this.setMenuRef = function (ref) { return _this.menuElement = ref; };
        this.setRootTableRef = function (ref) { return _this.rootTableElement = ref; };
        this.setRowHeaderRef = function (ref) { return _this.rowHeaderElement = ref; };
        var _a = this.props, defaultRowHeight = _a.defaultRowHeight, defaultColumnWidth = _a.defaultColumnWidth, numRows = _a.numRows, columnWidths = _a.columnWidths, rowHeights = _a.rowHeights, children = _a.children;
        this.childrenArray = React.Children.toArray(children);
        this.columnIdToIndex = Table.createColumnIdIndex(this.childrenArray);
        // Create height/width arrays using the lengths from props and
        // children, the default values from props, and finally any sparse
        // arrays passed into props.
        var newColumnWidths = this.childrenArray.map(function () { return defaultColumnWidth; });
        newColumnWidths = utils_1.Utils.assignSparseValues(newColumnWidths, columnWidths);
        var newRowHeights = utils_1.Utils.times(numRows, function () { return defaultRowHeight; });
        newRowHeights = utils_1.Utils.assignSparseValues(newRowHeights, rowHeights);
        this.state = {
            columnWidths: newColumnWidths,
            isLayoutLocked: false,
            rowHeights: newRowHeights,
            selectedRegions: [],
        };
    }
    Table.createColumnIdIndex = function (children) {
        var columnIdToIndex = {};
        for (var i = 0; i < children.length; i++) {
            var key = children[i].props.id;
            if (key != null) {
                columnIdToIndex[String(key)] = i;
            }
        }
        return columnIdToIndex;
    };
    Table.prototype.componentWillReceiveProps = function (nextProps) {
        var _this = this;
        var defaultRowHeight = nextProps.defaultRowHeight, defaultColumnWidth = nextProps.defaultColumnWidth, columnWidths = nextProps.columnWidths, rowHeights = nextProps.rowHeights, children = nextProps.children, numRows = nextProps.numRows;
        var newChildArray = React.Children.toArray(children);
        // Try to maintain widths of columns by looking up the width of the
        // column that had the same `ID` prop. If none is found, use the
        // previous width at the same index.
        var previousColumnWidths = newChildArray.map(function (child, index) {
            var mappedIndex = _this.columnIdToIndex[child.props.id];
            return _this.state.columnWidths[mappedIndex != null ? mappedIndex : index];
        });
        // Make sure the width/height arrays have the correct length, but keep
        // as many existing widths/heights when possible. Also, apply the
        // sparse width/heights from props.
        var newColumnWidths = this.state.columnWidths;
        newColumnWidths = utils_1.Utils.arrayOfLength(newColumnWidths, newChildArray.length, defaultColumnWidth);
        newColumnWidths = utils_1.Utils.assignSparseValues(newColumnWidths, previousColumnWidths);
        newColumnWidths = utils_1.Utils.assignSparseValues(newColumnWidths, columnWidths);
        var newRowHeights = this.state.rowHeights;
        newRowHeights = utils_1.Utils.arrayOfLength(newRowHeights, numRows, defaultRowHeight);
        newRowHeights = utils_1.Utils.assignSparseValues(newRowHeights, rowHeights);
        this.childrenArray = newChildArray;
        this.columnIdToIndex = Table.createColumnIdIndex(this.childrenArray);
        this.invalidateGrid();
        this.setState({
            columnWidths: newColumnWidths,
            rowHeights: newRowHeights,
        });
    };
    Table.prototype.render = function () {
        var isRowHeaderShown = this.props.isRowHeaderShown;
        this.validateGrid();
        return (React.createElement("div", {className: "bp-table-container", ref: this.setRootTableRef, onScroll: this.handleRootScroll}, 
            React.createElement("div", {className: "bp-table-top-container"}, 
                isRowHeaderShown ? this.renderMenu() : undefined, 
                this.renderColumnHeader()), 
            React.createElement("div", {className: "bp-table-bottom-container"}, 
                isRowHeaderShown ? this.renderRowHeader() : undefined, 
                this.renderBody())));
    };
    /**
     * When the component mounts, the HTML Element refs will be available, so
     * we constructor the Locator, which queries the elements' bounding
     * ClientRects.
     */
    Table.prototype.componentDidMount = function () {
        var _this = this;
        this.validateGrid();
        var locator = new locator_1.Locator(this.rootTableElement, this.bodyElement, this.grid);
        var viewportRect = locator.getViewportRect();
        this.setState({ locator: locator, viewportRect: viewportRect });
        this.resizeSensorDetach = resizeSensor_1.ResizeSensor.attach(this.rootTableElement, function () {
            if (!_this.state.isLayoutLocked) {
                _this.setState({ viewportRect: locator.getViewportRect() });
            }
        });
        this.syncMenuWidth();
    };
    Table.prototype.componentWillUnmount = function () {
        if (this.resizeSensorDetach != null) {
            this.resizeSensorDetach();
            delete this.resizeSensorDetach;
        }
    };
    Table.prototype.componentDidUpdate = function () {
        var locator = this.state.locator;
        if (locator != null) {
            this.validateGrid();
            locator.setGrid(this.grid);
        }
        this.syncMenuWidth();
    };
    Table.prototype.validateProps = function (props) {
        var ERROR_MESSAGE = "Children of Table must be Columns";
        React.Children.forEach(props.children, function (child) {
            // save as a variable so that union type narrowing works
            var cType = child.type;
            if (typeof cType === "string") {
                throw new Error(ERROR_MESSAGE);
            }
            else {
                var isColumn = cType.prototype === column_1.Column.prototype || column_1.Column.prototype.isPrototypeOf(cType);
                if (!isColumn) {
                    throw new Error(ERROR_MESSAGE);
                }
            }
        });
    };
    Table.prototype.renderMenu = function () {
        return (React.createElement("div", {className: "bp-table-menu", ref: this.setMenuRef}));
    };
    Table.prototype.syncMenuWidth = function () {
        var _a = this, menuElement = _a.menuElement, rowHeaderElement = _a.rowHeaderElement;
        if (menuElement != null && rowHeaderElement != null) {
            var width = rowHeaderElement.getBoundingClientRect().width;
            menuElement.style.width = width + "px";
        }
    };
    Table.prototype.getColumnProps = function (columnIndex) {
        var column = this.childrenArray[columnIndex];
        return column.props;
    };
    Table.prototype.renderColumnHeader = function () {
        var grid = this.grid;
        var _a = this.state, locator = _a.locator, selectedRegions = _a.selectedRegions, viewportRect = _a.viewportRect;
        var _b = this.props, allowMultipleSelection = _b.allowMultipleSelection, fillBodyWithGhostCells = _b.fillBodyWithGhostCells, isColumnResizable = _b.isColumnResizable, maxColumnWidth = _b.maxColumnWidth, minColumnWidth = _b.minColumnWidth;
        var classes = classNames("bp-table-column-headers", {
            "bp-table-selection-enabled": this.isSelectionModeEnabled(regions_2.RegionCardinality.FULL_COLUMNS),
        });
        var columnIndices = grid.getColumnIndicesInRect(viewportRect, fillBodyWithGhostCells);
        return (React.createElement("div", {className: classes}, 
            React.createElement(columnHeader_1.ColumnHeader, __assign({allowMultipleSelection: allowMultipleSelection, cellRenderer: this.columnHeaderCellRenderer, grid: grid, isResizable: isColumnResizable, locator: locator, maxColumnWidth: maxColumnWidth, minColumnWidth: minColumnWidth, onColumnWidthChanged: this.handleColumnWidthChanged, onLayoutLock: this.handleLayoutLock, onResizeGuide: this.handleColumnResizeGuide, onSelection: this.getEnabledSelectionHandler(regions_2.RegionCardinality.FULL_COLUMNS), selectedRegions: selectedRegions, viewportRect: viewportRect}, columnIndices), this.props.children), 
            this.maybeRenderColumnHeaderRegions()));
    };
    Table.prototype.renderRowHeader = function () {
        var grid = this.grid;
        var _a = this.state, locator = _a.locator, selectedRegions = _a.selectedRegions, viewportRect = _a.viewportRect;
        var _b = this.props, allowMultipleSelection = _b.allowMultipleSelection, fillBodyWithGhostCells = _b.fillBodyWithGhostCells, isRowResizable = _b.isRowResizable, maxRowHeight = _b.maxRowHeight, minRowHeight = _b.minRowHeight, renderRowHeader = _b.renderRowHeader;
        var classes = classNames("bp-table-row-headers", {
            "bp-table-selection-enabled": this.isSelectionModeEnabled(regions_2.RegionCardinality.FULL_ROWS),
        });
        var rowIndices = grid.getRowIndicesInRect(viewportRect, fillBodyWithGhostCells);
        return (React.createElement("div", {className: classes, ref: this.setRowHeaderRef}, 
            React.createElement(rowHeader_1.RowHeader, __assign({allowMultipleSelection: allowMultipleSelection, grid: grid, locator: locator, isResizable: isRowResizable, maxRowHeight: maxRowHeight, minRowHeight: minRowHeight, onLayoutLock: this.handleLayoutLock, onResizeGuide: this.handleRowResizeGuide, onRowHeightChanged: this.handleRowHeightChanged, onSelection: this.getEnabledSelectionHandler(regions_2.RegionCardinality.FULL_ROWS), renderRowHeader: renderRowHeader, selectedRegions: selectedRegions, viewportRect: viewportRect}, rowIndices)), 
            this.maybeRenderRowHeaderRegions()));
    };
    Table.prototype.renderBody = function () {
        var grid = this.grid;
        var _a = this.props, allowMultipleSelection = _a.allowMultipleSelection, fillBodyWithGhostCells = _a.fillBodyWithGhostCells, renderBodyContextMenu = _a.renderBodyContextMenu;
        var _b = this.state, locator = _b.locator, selectedRegions = _b.selectedRegions, viewportRect = _b.viewportRect, verticalGuides = _b.verticalGuides, horizontalGuides = _b.horizontalGuides;
        var style = grid.getRect().sizeStyle();
        var rowIndices = grid.getRowIndicesInRect(viewportRect, fillBodyWithGhostCells);
        var columnIndices = grid.getColumnIndicesInRect(viewportRect, fillBodyWithGhostCells);
        var noVerticalScroll = fillBodyWithGhostCells &&
            grid.isGhostIndex(rowIndices.rowIndexEnd, 0) &&
            viewportRect != null && viewportRect.top === 0;
        var noHorizontalScroll = fillBodyWithGhostCells &&
            grid.isGhostIndex(0, columnIndices.columnIndexEnd) &&
            viewportRect != null && viewportRect.left === 0;
        // disable scroll for ghost cells
        var classes = classNames("bp-table-body", {
            "bp-table-selection-enabled": this.isSelectionModeEnabled(regions_2.RegionCardinality.CELLS),
            "bp-table-no-vertical-scroll": noVerticalScroll,
            "bp-table-no-horizontal-scroll": noHorizontalScroll,
        });
        return (React.createElement("div", {className: classes, onScroll: this.handleBodyScroll, ref: this.setBodyRef}, 
            React.createElement("div", {className: "bp-table-body-scroll-client", style: style}, 
                React.createElement(tableBody_1.TableBody, __assign({allowMultipleSelection: allowMultipleSelection, cellRenderer: this.bodyCellRenderer, grid: grid, locator: locator, onSelection: this.getEnabledSelectionHandler(regions_2.RegionCardinality.CELLS), renderBodyContextMenu: renderBodyContextMenu, selectedRegions: selectedRegions, viewportRect: viewportRect}, rowIndices, columnIndices)), 
                this.maybeRenderBodyRegions(), 
                React.createElement(guides_1.GuideLayer, {className: "bp-table-resize-guides", verticalGuides: verticalGuides, horizontalGuides: horizontalGuides}))
        ));
    };
    Table.prototype.isGuidesShowing = function () {
        return this.state.verticalGuides != null || this.state.horizontalGuides != null;
    };
    Table.prototype.isSelectionModeEnabled = function (selectionMode) {
        return this.props.selectionModes.indexOf(selectionMode) !== -1;
    };
    Table.prototype.getEnabledSelectionHandler = function (selectionMode) {
        if (!this.isSelectionModeEnabled(selectionMode)) {
            // If the selection mode isn't enabled, return a callback that
            // will clear the selection. For example, if row selection is
            // disabled, clicking on the row header will clear the table's
            // selection. If all selection modes are enabled, clicking on the
            // same region twice will clear the selection.
            return this.clearSelection;
        }
        else {
            return this.handleSelection;
        }
    };
    Table.prototype.invalidateGrid = function () {
        this.grid = null;
    };
    Table.prototype.validateGrid = function () {
        if (this.grid == null) {
            var _a = this.props, defaultRowHeight = _a.defaultRowHeight, defaultColumnWidth = _a.defaultColumnWidth;
            var _b = this.state, rowHeights = _b.rowHeights, columnWidths = _b.columnWidths;
            this.grid = new grid_1.Grid(rowHeights, columnWidths, grid_1.Grid.DEFAULT_BLEED, defaultRowHeight, defaultColumnWidth);
        }
    };
    /**
     * Renders a `RegionLayer`, applying styles to the regions using the
     * supplied `IRegionStyler`. `RegionLayer` is a `PureRender` component, so
     * the `IRegionStyler` should be a new instance on every render if we
     * intend to redraw the region layer.
     */
    Table.prototype.maybeRenderRegions = function (getRegionStyle) {
        if (this.isGuidesShowing()) {
            return undefined;
        }
        var regionGroups = regions_2.Regions.joinStyledRegionGroups(this.state.selectedRegions, this.props.styledRegionGroups);
        return regionGroups.map(function (regionGroup, index) {
            return (React.createElement(regions_1.RegionLayer, {className: classNames(regionGroup.className), key: index, regions: regionGroup.regions, getRegionStyle: getRegionStyle}));
        });
    };
    Table.prototype.maybeRenderBodyRegions = function () {
        var _this = this;
        var styler = function (region) {
            var cardinality = regions_2.Regions.getRegionCardinality(region);
            var style = _this.grid.getRegionStyle(region);
            switch (cardinality) {
                case regions_2.RegionCardinality.CELLS:
                    return style;
                case regions_2.RegionCardinality.FULL_COLUMNS:
                    style.top = "-1px";
                    return style;
                case regions_2.RegionCardinality.FULL_ROWS:
                    style.left = "-1px";
                    return style;
                default:
                    return { display: "none" };
            }
        };
        return this.maybeRenderRegions(styler);
    };
    Table.prototype.maybeRenderColumnHeaderRegions = function () {
        var _this = this;
        var styler = function (region) {
            var grid = _this.grid;
            var viewportRect = _this.state.viewportRect;
            if (viewportRect == null) {
                return {};
            }
            var cardinality = regions_2.Regions.getRegionCardinality(region);
            var style = grid.getRegionStyle(region);
            switch (cardinality) {
                case regions_2.RegionCardinality.FULL_COLUMNS:
                    style.bottom = "-1px";
                    style.transform = "translate3d(" + -viewportRect.left + "px, 0, 0)";
                    return style;
                default:
                    return { display: "none" };
            }
        };
        return this.maybeRenderRegions(styler);
    };
    Table.prototype.maybeRenderRowHeaderRegions = function () {
        var _this = this;
        var styler = function (region) {
            var grid = _this.grid;
            var viewportRect = _this.state.viewportRect;
            if (viewportRect == null) {
                return {};
            }
            var cardinality = regions_2.Regions.getRegionCardinality(region);
            var style = grid.getRegionStyle(region);
            switch (cardinality) {
                case regions_2.RegionCardinality.FULL_ROWS:
                    style.right = "-1px";
                    style.transform = "translate3d(0, " + -viewportRect.top + "px, 0)";
                    return style;
                default:
                    return { display: "none" };
            }
        };
        return this.maybeRenderRegions(styler);
    };
    Table.defaultProps = {
        allowMultipleSelection: true,
        defaultColumnWidth: 150,
        defaultRowHeight: 20,
        fillBodyWithGhostCells: false,
        isRowHeaderShown: true,
        minColumnWidth: 50,
        minRowHeight: 20,
        numRows: 0,
        renderRowHeader: rowHeader_1.renderDefaultRowHeader,
        selectionModes: regions_2.SelectionModes.ALL,
    };
    Table = __decorate([
        PureRender
    ], Table);
    return Table;
}(core_1.AbstractComponent));
exports.Table = Table;

//# sourceMappingURL=data:application/json;base64,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
