var Sidjs = (function () {
    'use strict';

    var NodeTypes;
    (function (NodeTypes) {
        NodeTypes[NodeTypes["adapter"] = 0] = "adapter";
        NodeTypes[NodeTypes["bridge"] = 1] = "bridge";
        NodeTypes[NodeTypes["chain"] = 2] = "chain";
        NodeTypes[NodeTypes["channel"] = 3] = "channel";
        NodeTypes[NodeTypes["gateway"] = 4] = "gateway";
        NodeTypes[NodeTypes["activator"] = 5] = "activator";
    })(NodeTypes || (NodeTypes = {}));
    class NodeTypeMapper {
        mapNodeType(componentType) {
            var _a;
            return (_a = Object.values(NodeTypes)
                .find(nodeType => componentType === null || componentType === void 0 ? void 0 : componentType.toLowerCase().endsWith(nodeType))) !== null && _a !== void 0 ? _a : "other";
        }
    }

    class NameFormatter {
        formatNode(node) {
            var _a, _b;
            let nodeName = this.formatName(node.name);
            let handlers = (_b = (_a = node.handlers) === null || _a === void 0 ? void 0 : _a.map(h => `{ ${h.type} | ${this.formatName(h.name)} }`)) !== null && _b !== void 0 ? _b : [];
            return handlers.length > 0 ? `{ ${[nodeName, ...handlers].join(" | ")} }` : nodeName;
        }
        formatName(name) {
            return name === null || name === void 0 ? void 0 : name.replace(/\(.+/, "").replace(/^.+\./, "");
        }
    }

    class GraphGenerator {
        constructor(nameFormatter = new NameFormatter()) {
            this.nameFormatter = nameFormatter;
        }
        generateLinks(links, config) {
            return links.map(link => `${config.indent}${link.from} -> ${link.to}`);
        }
        generateNodes(groupType, nodes, config) {
            var _a;
            let attributes = (_a = config.nodeAttributes.get(groupType)) !== null && _a !== void 0 ? _a : {};
            if (groupType === "gateway") {
                let gateways = {};
                nodes.forEach(node => {
                    var _a, _b, _c;
                    let gatewayName = (_b = (_a = node.name) === null || _a === void 0 ? void 0 : _a.replace(/\..*/, "")) !== null && _b !== void 0 ? _b : node.nodeId;
                    let gateway = gateways[gatewayName] = (_c = gateways[gatewayName]) !== null && _c !== void 0 ? _c : [];
                    gateway.push(node);
                });
                return Object.entries(gateways).flatMap(([name, routes]) => this.generateCluster(name, routes, attributes, config.indent));
            }
            return this.generateGraphNodes(config.indent, nodes, attributes);
        }
        generateCluster(name, nodes, attributes, indent = "") {
            return [`${indent}subgraph cluster_${name} {`, `${indent + indent}label="${name}"`, ...this.generateGraphNodes(indent + indent, nodes, attributes), `${indent}}`];
        }
        generateGraphNodes(indent, nodes, attributes) {
            return nodes.map(node => this.generateNode(indent, node.nodeId, Object.assign(Object.assign({}, attributes), { label: this.nameFormatter.formatNode(node) })));
        }
        generateNode(indent, nodeId, attributes) {
            return `${indent}${nodeId} [${this.formatAttributes(attributes)}]`;
        }
        formatAttributes(attributes) {
            return Object.entries(attributes).map(([name, value]) => `${name}="${value}"`).join(" ");
        }
    }

    class IntegrationParser {
        constructor(config = {}, nodeTypeMapper = new NodeTypeMapper(), graphGenerator = new GraphGenerator()) {
            this.config = {
                indent: "  ",
                newline: "\n",
                nodesColorScheme: "greys9",
                linksColorScheme: "greys9",
                nodeAttributes: new Map([["channel", { shape: "box" }], ["chain", { shape: "record" }], ["bridge", { shape: "plain" }], ["adapter", { shape: "rarrow" }]]),
            };
            this.config = Object.assign(Object.assign({}, this.config), config);
            this.nodeTypeMapper = new NodeTypeMapper();
            this.graphGenerator = new GraphGenerator();
        }
        convert(graph) {
            return this.convertGraph(typeof graph === "string" ? JSON.parse(graph) : graph);
        }
        convertGraph(graph) {
            var _a, _b, _c;
            return [`digraph "${(_a = graph.contentDescriptor) === null || _a === void 0 ? void 0 : _a.name}" {`,
                ...this.generateNodes((_b = graph.nodes) !== null && _b !== void 0 ? _b : []),
                ...this.generateLinks((_c = graph.links) !== null && _c !== void 0 ? _c : []),
                "}"].join(this.config.newline);
        }
        generateNodes(nodes) {
            return Object.entries(this.groupNodes(nodes)).flatMap(([group, nodes]) => this.generateNodeGroup(group, nodes));
        }
        generateNodeGroup(group, nodes) {
            return this.graphGenerator.generateNodes(group, nodes, this.config);
        }
        generateLinks(links) {
            return this.graphGenerator.generateLinks(links, this.config);
        }
        groupNodes(nodes) {
            let groups = {};
            nodes.forEach(node => {
                var _a;
                let nodeType = this.nodeTypeMapper.mapNodeType(node.componentType);
                let group = groups[nodeType] = (_a = groups[nodeType]) !== null && _a !== void 0 ? _a : [];
                group.push(node);
            });
            return groups;
        }
    }

    return IntegrationParser;

})();
//# sourceMappingURL=sidjs.js.map
