'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _chalk = require('chalk');

var _chalk2 = _interopRequireDefault(_chalk);

var _figures = require('figures');

var _figures2 = _interopRequireDefault(_figures);

var _logUpdate = require('log-update');

var _logUpdate2 = _interopRequireDefault(_logUpdate);

var _TaskResult = require('./TaskResult');

var _TaskResult2 = _interopRequireDefault(_TaskResult);

var _constants = require('./constants');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class Renderer {

  constructor(loader) {
    this.instance = 0;

    this.chalk = _chalk2.default;
    this.load = loader;
    this.log = _logUpdate2.default;
  }

  indent(length) {
    return '    '.repeat(length);
  }

  render() {
    const output = [];
    const results = this.load();

    results.forEach(result => {
      output.push(...this.renderResult(result, 0));
    });

    return output.join('\n');
  }

  renderResult(result, level = 0, suffix = '') {
    const output = [];

    let message = `${this.indent(level)}${this.renderStatus(result)} ${result.title}`;

    if (result.isSkipped()) {
      message += ` ${_chalk2.default.yellow('[skipped]')}`;
    } else if (result.hasFailed()) {
      message += ` ${_chalk2.default.red('[failed]')}`;
    } else if (suffix) {
      message += ` ${suffix}`;
    }

    output.push(message);

    let pendingTask;
    let runningTask;
    let failedTask;
    let passed = 0;

    result.tasks.forEach(task => {
      if (task.isPending() && !pendingTask) {
        pendingTask = task;
      } else if (task.isRunning() && !runningTask) {
        runningTask = task;
      } else if (task.hasFailed() && !failedTask) {
        failedTask = task;
      } else if (task.hasPassed()) {
        passed += 1;
      }
    });

    const activeTask = failedTask || runningTask || pendingTask;
    const taskSuffix = _chalk2.default.gray(`[${passed}/${result.tasks.length}]`);

    if (activeTask) {
      output.push(...this.renderResult(activeTask, level + 1, taskSuffix));
    }

    result.routines.forEach(routine => {
      output.push(...this.renderResult(routine, level + 1));
    });

    return output;
  }

  renderStatus(result) {
    switch (result.status) {
      case _constants.PENDING:
        return _chalk2.default.gray(_figures2.default.circle);
      case _constants.RUNNING:
        return _chalk2.default.gray(result.spinner());
      case _constants.SKIPPED:
        return _chalk2.default.yellow(_figures2.default.circleDotted);
      case _constants.PASSED:
        return _chalk2.default.green(_figures2.default.tick);
      case _constants.FAILED:
        return _chalk2.default.red(_figures2.default.cross);
      default:
        return '';
    }
  }

  reset() {
    this.log.clear();
  }

  start() {
    if (!this.instance) {
      this.instance = setInterval(() => this.update(), 100);
    }
  }

  stop() {
    this.update();

    if (this.instance) {
      clearInterval(this.instance);
      this.instance = 0;
    }

    this.log.done();
  }

  update() {
    if (this.instance) {
      this.log(this.render());
    } else {
      this.start();
    }
  }
}
exports.default = Renderer; /**
                             * @copyright   2017, Miles Johnson
                             * @license     https://opensource.org/licenses/MIT
                             * 
                             */