import type {Slot} from 'vue'
import {DOCUMENT, HAS_ELEMENT_SUPPORT} from '../constants/env'
import type {AnimationFrame} from '../types/safeTypes'
import {HAS_WINDOW_SUPPORT} from './env'
import {toString} from './stringUtils'

const ELEMENT_PROTO = HAS_ELEMENT_SUPPORT ? Element.prototype : undefined

/**
 * @link https://developer.mozilla.org/en-US/docs/Web/API/Element/matches#Polyfill
 */
export const matchesEl =
  ELEMENT_PROTO?.matches ||
  (ELEMENT_PROTO as any)?.msMatchesSelector ||
  ELEMENT_PROTO?.webkitMatchesSelector

export const isElement = (el: HTMLElement | Element): boolean =>
  !!(el && el.nodeType === Node.ELEMENT_NODE)

export const getBCR = (el: HTMLElement) => (isElement(el) ? el.getBoundingClientRect() : null)

export const getActiveElement = (excludes = []): Element | null => {
  const {activeElement} = document
  return activeElement && !excludes.some((el: HTMLElement) => el === activeElement)
    ? activeElement
    : null
}

export const isActiveElement = (el: HTMLElement): boolean =>
  isElement(el) && el === getActiveElement()

export const attemptFocus = (el: HTMLElement, options = {}): boolean => {
  try {
    el.focus(options)
  } catch (e) {
    // eslint-disable-next-line no-console
    console.error(e)
  }
  return isActiveElement(el)
}

/**
 * Attempt to blur an element, and return `true` if successful
 *
 * @param el
 * @returns
 */
export const attemptBlur = (el: HTMLElement): boolean => {
  try {
    el.blur()
  } catch (e) {
    // eslint-disable-next-line no-console
    console.error(e)
  }
  return !isActiveElement(el)
}

export const getStyle = (el: HTMLElement, prop: string) =>
  prop && isElement(el) ? el.getAttribute(prop) || null : null

export const contains = (parent: Node, child: Node): boolean => parent.contains(child)

export const isVisible = (el: HTMLElement): boolean => {
  //if (!isElement(el) || !el.parentNode || !contains(DOCUMENT.body, el)) {
  // Note this can fail for shadow dom elements since they
  // are not a direct descendant of document.body
  //return false
  //}
  if (getStyle(el, 'display') === 'none') {
    // We do this check to help with vue-test-utils when using v-show
    return false
  }
  // All browsers support getBoundingClientRect(), except JSDOM as it returns all 0's for values :(
  // So any tests that need isVisible will fail in JSDOM
  // Except when we override the getBCR prototype in some tests
  const bcr = getBCR(el)
  return !!(bcr && bcr.height > 0 && bcr.width > 0)
}

export const isEmptySlot = (el: Slot | undefined): boolean => (el?.() ?? []).length === 0

/**
 * Select a single element, returns `null` if not found
 *
 * @param selector
 * @param root
 * @returns
 */
export const select = (selector: any, root: any) =>
  (isElement(root) ? root : DOCUMENT).querySelector(selector) || null

export const selectAll = (selector: any, root: any) =>
  Array.from([(isElement(root) ? root : DOCUMENT).querySelectorAll(selector)])

export const getAttr = (el: HTMLElement | Element, attr: string): string | null =>
  attr && isElement(el) ? el.getAttribute(attr) : null

/**
 * Get an element given an ID
 */
export const getById = (id: string) =>
  DOCUMENT.getElementById(/^#/.test(id) ? id.slice(1) : id) || null

export const setAttr = (el: HTMLElement, attr: string, value: string): void => {
  if (attr && isElement(el)) {
    el.setAttribute(attr, value)
  }
}

/**
 * Remove an attribute from an element
 *
 * @param el
 * @param attr
 */
export const removeAttr = (el: HTMLElement, attr: string): void => {
  if (attr && isElement(el)) {
    el.removeAttribute(attr)
  }
}

export const isTag = (tag: any, name: any): boolean =>
  toString(tag).toLowerCase() === toString(name).toLowerCase()

export const requestAF: AnimationFrame = HAS_WINDOW_SUPPORT
  ? window.requestAnimationFrame ||
    window.webkitRequestAnimationFrame ||
    window.mozRequestAnimationFrame ||
    window.msRequestAnimationFrame ||
    window.oRequestAnimationFrame ||
    // Fallback, but not a true polyfill
    // Only needed for Opera Mini
    ((cb) => setTimeout(cb, 16))
  : (cb) => setTimeout(cb, 0)

export const matches = (el: Element, selector: string) =>
  isElement(el) ? matchesEl.call(el, selector) : false

/**
 * @link https://developer.mozilla.org/en-US/docs/Web/API/Element/closest
 */
/* eslint-disable @typescript-eslint/no-this-alias */
export const closestEl =
  ELEMENT_PROTO?.closest ||
  function (this: Element, sel: string) {
    let el = this
    if (!el) return null
    do {
      // Use our "patched" matches function
      if (matches(el, sel)) {
        return el
      }
      el = el.parentElement || (el.parentNode as any)
    } while (el !== null && el.nodeType === Node.ELEMENT_NODE)
    return null
  }

/**
 * Finds closest element matching selector. Returns `null` if not found
 *
 * @param selector
 * @param root
 * @param includeRoot
 */
export const closest = (selector: string, root: Element, includeRoot = false) => {
  if (!isElement(root)) {
    return null
  }
  const el = closestEl.call(root, selector)
  // Native closest behaviour when `includeRoot` is truthy,
  // else emulate jQuery closest and return `null` if match is
  // the passed in root element when `includeRoot` is falsey
  return includeRoot ? el : el === root ? null : el
}

export const getTransitionDelay = (element: HTMLElement) => {
  const style = window.getComputedStyle(element)
  // if multiple durations are defined, we take the first
  const transitionDelay = style.transitionDelay.split(',')[0] || ''
  const transitionDuration = style.transitionDuration.split(',')[0] || ''
  const transitionDelayMs = Number(transitionDelay.slice(0, -1)) * 1000
  const transitionDurationMs = Number(transitionDuration.slice(0, -1)) * 1000
  return transitionDelayMs + transitionDurationMs
}
