# Form Radio Inputs

> For cross browser consistency, `<b-form-radio-group>` and `<b-form-radio>` uses
Bootstrap's custom radio input to replace the browser default radio input. It
is built on top of semantic and accessible markup, so it is a solid replacement
for the default radio input.

The individual radio inputs in radio input group can be specified via the `options`
prop of `<b-form-radio-group>`, or via manual placement of the `<b-form-radio>` sub component.

```html
<template>
  <div>
    <b-form-group label="Radios using <code>options</code>">
      <b-form-radio-group id="radios1" v-model="selected" :options="options" name="radioOpenions">
      </b-form-radio-group>
    </b-form-group>

    <b-form-group label="Radios using sub-components">
      <b-form-radio-group id="radios2" v-model="selected" name="radioSubComponent">
        <b-form-radio value="first">Toggle this custom radio</b-form-radio>
        <b-form-radio value="second">Or toggle this other custom radio</b-form-radio>
        <b-form-radio value="third" disabled>This one is Disabled</b-form-radio>
        <b-form-radio :value="{fourth: 4}">This is the 4th radio</b-form-radio>
      </b-form-radio-group>
    </b-form-group>

    <div class="mt-3">
      Selected: <strong>{{ selected }}</strong>
    </div>
  </div>
</template>

<script>
export default {
  data () {
    return {
      selected: 'first',
      options: [
        { text: 'Toggle this custom radio', value: 'first' },
        { text: 'Or toggle this other custom radio', value: 'second' },
        { text: 'This one is Disabled', value: 'third', disabled: true },
        { text: 'This is the 4th radio', value: {fourth: 4} }
      ]
    }
  }
}
</script>

<!-- form-radio-1.vue -->
```
Feel free to mix and match `options` prop and `<b-form-radio>` in `<b-form-radio-group>`.
Manually placed `<b-form-radio>` inputs will appear _below_ any radio inputs generated by
the `options` prop. To have them apper _above_ the inputs generated by `options`, place them
in the named slot `first`.


## Options
Please see options in [`<b-form-select>`](/docs/components/form-select) docs for details on passing
options (value array) to `<b-form-radio-group>`.

## Inline or stacked
By default `<b-form-radio>` generates inline radio inputs. Set the prop `stacked` to make
the radios appear one over the other.

```html
<template>
  <div>
    <b-form-group label="Inline radios (default)">
      <b-form-radio-group v-model="selected"
                          :options="options"
                          name="radioInline">
      </b-form-radio-group>
    </b-form-group>

    <b-form-group label="Stacked radios">
      <b-form-radio-group v-model="selected"
                          :options="options"
                          stacked
                          name="radiosStacked">
      </b-form-radio-group>
    </b-form-group>

    <div class="mt-3">
      Selected: <strong>{{ selected }}</strong>
    </div>
  </div>
</template>

<script>
export default {
  data () {
    return {
      selected: 'first',
      options: [
        { text: 'First radio', value: 'first' },
        { text: 'Second radio', value: 'second' },
        { text: 'Third radio', value: 'third' }
      ]
    }
  }
}
</script>

<!-- form-radio-stacked-1.vue -->
```

## Button style radios
Render radios with the look of buttons by setting the prop `buttons`. Set the button variant by
setting the `button-variant` prop to one of the standard Bootstrap button variants (see
[`<b-button>`](./button) for supported variants). The default `button-variant` is `secondary`.

The `buttons` prop has precedence over `plain`, and `button-variant` has no effect if
`buttons` is not set.

Button style radios will have the class `.active` automatically applied to their label
when they are in the checked state.

```html
<template>
  <div>
    <b-form-group label="Button style radios">
      <b-form-radio-group id="btnradios1"
                          buttons
                          v-model="selected"
                          :options="options"
                          name="radiosBtnDefault" />
    </b-form-group>

    <b-form-group label="Button style radios with <code>outline-primary</code> variant and size <code>lg</code>">
      <b-form-radio-group id="btnradios2"
                          buttons
                          button-variant="outline-primary"
                          size="lg"
                          v-model="selected"
                          :options="options"
                          name="radioBtnOutline" />
    </b-form-group>

    <b-form-group label="Stacked button style radios">
      <b-form-radio-group id="btnradios3"
                          buttons
                          stacked
                          v-model="selected"
                          :options="options"
                          name="radioBtnStacked" />
    </b-form-group>
  </div>
</template>

<script>
export default {
  data () {
    return {
      selected: 'radio1',
      options: [
        { text: 'Radio 1', value: 'radio1' },
        { text: 'Radio 3', value: 'radio2' },
        { text: 'Radio 3 (disabled)', value: 'radio3', disabled: true },
        { text: 'Radio 4', value: 'radio4' }
      ]
    }
  }
}
</script>

<!-- form-radio-buttons.vue -->
```


**Note:** `<b-form-radio-group>` uses the HTML attribute `data-toggle="buttons"` to
apply the button styling to the radios. This can cause a potential conflict if you
are including Bootstrap V4's jQuery code in your project for other purposes.
To get around this, you will need to exclude the Bootstrap V4 jQuery buttons plugin, and
include only the other Bootstrap V4 jQuery plugins you reqwuire.


## Non custom style radio inputs (plain)
You can have `b-form-radio` render a browser native radio input by setting the `plain` prop.


```html
<template>
  <div>
    <b-form-group label="Plain inline radios">
      <b-form-radio-group v-model="selected"
                          :options="options"
                          plain
                          name="plainInline" />
    </b-form-group>

    <b-form-group label="Plain stacked radios">
      <b-form-radio-group v-model="selected"
                          :options="options"
                          plain
                          stacked
                          name="plainStacked" />
    </b-form-group>
  </div>
</template>

<script>
export default {
  data () {
    return {
      selected: 'first',
      options: [
        { text: 'First radio', value: 'first' },
        { text: 'Second radio', value: 'second' },
        { text: 'Third radio', value: 'third' }
      ]
    }
  }
}
</script>

<!-- form-radio-plain-1.vue -->
```

**Note:** `plain` will have no effect if `buttons` is set.


## Contextual States
Bootstrap includes validation styles for `valid` and `invalid` states
on most form controls.

Generally speaking, you’ll want to use a particular state for specific types of feedback:
- `'invalid'` is great for when there’s a blocking or required field. A user must fill in
this field properly to submit the form.
- `'valid'` is ideal for situations when you have per-field validation throughout a form
and want to encourage a user through the rest of the fields.
- `null` Displays no validation state

To apply one of the contextual state icons on `<b-form-radio>`, set the `state` prop
to `'invalid'` (or `false`), `'valid'` (or `true`), or `null`.

**Note:** contextual state is not supported for radios rendered in buttons mode.

### Conveying contextual validation state to assistive technologies and colorblind users:
Using these contextual states to denote the state of a form control only provides
a visual, color-based indication, which will not be conveyed to users of assistive
technologies - such as screen readers - or to colorblind users.

Ensure that an alternative indication of state is also provided. For instance, you
could include a hint about state in the form control's `<label>` text itself, or by
providing an additional help text block (i.e. `<b-form-feedbck>`). Specifically
for assistive technologies, invalid form controls can also be assigned an
`aria-invalid="true"` attribute (see below).


### ARIA `aria-invalid` attribute
When `<b-form-radio>` has an invalid contextual state (i.e. `invalid`) you may also
want to set the `<b-form-radio>` prop `aria-invalid` to `true`.

Supported `invalid` values are:
- `false` (default) No errors detected
- `true` The value has failed validation.

`aria-invalid` is automatically set if `state` is `invalid`.


## Radio component aliases
- `<b-form-radio-group>` can be used by the shorter alias `<b-radio-group>`.
- `<b-form-radio>` can be used by the shorter alias of `<b-radio>`.

## Component Reference
