# Form Radio Inputs

> For cross browser consistency, `<b-form-radio-group>` and `<b-form-radio>` uses Bootstrap's custom
> radio input to replace the browser default radio input. It is built on top of semantic and
> accessible markup, so it is a solid replacement for the default radio input.

## Individual radios

```html
<template>
  <div>
    <b-form-group label="Individual radios">
      <b-form-radio value="A" v-model="selected" name="some-radios">Option A</b-form-radio>
      <b-form-radio value="B" v-model="selected" name="some-radios">Option B</b-form-radio>
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: ''
      }
    }
  }
</script>

<!-- b-form-radio.vue -->
```

## Grouped radios

The individual radio inputs in `<b-form-radio-group>` can be specified via the `options` prop, or
via manual placement of the `<b-form-radio>` sub component. When using manually placed
`<b-form-radio>` components within a `<b-form-radio-group>`, they will inherit most props and the
v-model from the `<b-form-radio-group>`.

```html
<template>
  <div>
    <b-form-group label="Radios using options">
      <b-form-radio-group id="radios1" v-model="selected" :options="options" name="radioOptions" />
    </b-form-group>

    <b-form-group label="Radios using sub-components">
      <b-form-radio-group id="radios2" v-model="selected" name="radioSubComponent">
        <b-form-radio value="first">Toggle this custom radio</b-form-radio>
        <b-form-radio value="second">Or toggle this other custom radio</b-form-radio>
        <b-form-radio value="third" disabled>This one is Disabled</b-form-radio>
        <b-form-radio :value="{ fourth: 4 }">This is the 4th radio</b-form-radio>
      </b-form-radio-group>
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'first',
        options: [
          { text: 'Toggle this custom radio', value: 'first' },
          { text: 'Or toggle this other custom radio', value: 'second' },
          { text: 'This one is Disabled', value: 'third', disabled: true },
          { text: 'This is the 4th radio', value: { fourth: 4 } }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-group.vue -->
```

Feel free to mix and match `options` prop and `<b-form-radio>` in `<b-form-radio-group>`. Manually
placed `<b-form-radio>` inputs will appear _below_ any radio inputs generated by the `options` prop.
To have them appear _above_ the inputs generated by `options`, place them in the named slot `first`.

```html
<template>
  <div>
    <b-form-group label="Radios using options and slots">
      <b-form-radio-group
        id="radio-slots"
        v-model="selected"
        :options="options"
        name="radioOptionsSlots"
      >
        <!-- Radios in this slot will appear first -->
        <template slot="first">
          <b-form-radio value="first">Toggle this custom radio from slot first</b-form-radio>
        </template>

        <!-- Radios in the default slot will appear after any option generated radios -->
        <b-form-radio :value="{ fourth: 4 }">This is the 4th radio</b-form-radio>
        <b-form-radio value="fifth">This is the 5th radio</b-form-radio>
      </b-form-radio-group>
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: '',
        options: [
          { text: 'Or toggle this other custom radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-group-slot.vue -->
```

## Radio group options array

Please see options in [`<b-form-select>`](/docs/components/form-select) docs for details on passing
options (value array) to `<b-form-radio-group>` via the `options` prop.

## Radio value and v-model

`<b-form-radio>` and `<b-form-radio-group>` do not have a value by default. You must explicitly
supply a value (to which the `v-model` is set to when the radio is checked) via the `value` prop.

The `v-model` of both `<b-form-radio>` and `<b-form-radio-group>` binds to the `checked` prop. To
pre-check a radio, you must set the `v-model` value to the radio's value. Do not use the `checked`
prop directly.

Radio supports values of many types, such as a `string`, `boolean`, `number`, or an `object`.

## Inline or stacked radios

By default `<b-form-radio-group>` generates inline radio inputs, while `<b-form-radio>` generates
stacked radios. Set the prop `stacked` on `<b-form-radio-group>` to make the radios appear one over
the other, or when using radios not in a group, set the `inline` prop on `b-form-radio` to true to
render them inline.

```html
<template>
  <div>
    <b-form-group label="Inline radios (default)">
      <b-form-radio-group v-model="selected" :options="options" name="radioInline" />
    </b-form-group>

    <b-form-group label="Stacked radios">
      <b-form-radio-group v-model="selected" :options="options" stacked name="radiosStacked" />
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'first',
        options: [
          { text: 'First radio', value: 'first' },
          { text: 'Second radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-stacked.vue -->
```

## Button style radios

Render radios with the look of buttons by setting the prop `buttons` to `true` on
`<b-form-radio-group>`. Set the button variant by setting the `button-variant` prop to one of the
standard Bootstrap button variants (see [`<b-button>`](/docs/components/button) for supported
variants). The default `button-variant` is `secondary`.

The `buttons` prop has precedence over `plain`, and `button-variant` has no effect if `buttons` is
not set.

Button style radios will have the class `.active` automatically applied to their label when they are
in the checked state.

```html
<template>
  <div>
    <b-form-group label="Button style radios">
      <b-form-radio-group
        id="btnradios1"
        buttons
        v-model="selected"
        :options="options"
        name="radiosBtnDefault"
      />
    </b-form-group>

    <b-form-group label="Button style radios with outline-primary variant and size lg">
      <b-form-radio-group
        id="btnradios2"
        buttons
        button-variant="outline-primary"
        size="lg"
        v-model="selected"
        :options="options"
        name="radioBtnOutline"
      />
    </b-form-group>

    <b-form-group label="Stacked button style radios">
      <b-form-radio-group
        id="btnradios3"
        buttons
        stacked
        v-model="selected"
        :options="options"
        name="radioBtnStacked"
      />
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'radio1',
        options: [
          { text: 'Radio 1', value: 'radio1' },
          { text: 'Radio 3', value: 'radio2' },
          { text: 'Radio 3 (disabled)', value: 'radio3', disabled: true },
          { text: 'Radio 4', value: 'radio4' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-buttons.vue -->
```

## Non-custom style radio inputs (plain)

You can have `<b-form-radio>` and `<b-form-radio-group>` render a browser native-styled radio input
by setting the `plain` prop.

```html
<template>
  <div>
    <b-form-group label="Plain inline radios">
      <b-form-radio-group v-model="selected" :options="options" plain name="plainInline" />
    </b-form-group>

    <b-form-group label="Plain stacked radios">
      <b-form-radio-group v-model="selected" :options="options" plain stacked name="plainStacked" />
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'first',
        options: [
          { text: 'First radio', value: 'first' },
          { text: 'Second radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-plain.vue -->
```

**Note:** `plain` will have no effect if `buttons`/`button` is set.

## Contextual States

Bootstrap includes validation styles for `valid` and `invalid` states on most form controls.

Generally speaking, you’ll want to use a particular state for specific types of feedback:

- `'invalid'` is great for when there’s a blocking or required field. A user must fill in this field
  properly to submit the form.
- `'valid'` is ideal for situations when you have per-field validation throughout a form and want to
  encourage a user through the rest of the fields.
- `null` Displays no validation state

To apply one of the contextual state icons on `<b-form-radio>`, set the `state` prop to `'invalid'`
(or `false`), `'valid'` (or `true`), or `null`.

**Note:** Contextual state is not supported for radios rendered in buttons mode.

### Contextual state with feedback example

```html
<template>
  <div>
    <b-form-radio-group v-model="value" :options="options" :state="state" name="radioValidation">
      <b-form-invalid-feedback :state="state">Please select one</b-form-invalid-feedback>
      <b-form-valid-feedback :state="state">Thank you</b-form-valid-feeback>
    </b-form-radio-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        value: null,
        options: [
          { text: 'First radio', value: 'first' },
          { text: 'Second radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    },
    computed: {
      state() {
        return Boolean(this.value)
      }
    }
  }
</script>

<!-- b-form-radio-validation.vue -->
```

### Conveying contextual validation state to assistive technologies and colorblind users:

Using these contextual states to denote the state of a form control only provides a visual,
color-based indication, which will not be conveyed to users of assistive technologies - such as
screen readers - or to colorblind users.

Ensure that an alternative indication of state is also provided. For instance, you could include a
hint about state in the form control's `<label>` text itself, or by providing an additional help
text block (i.e. `<b-form-invalid-feedback>`). Specifically for assistive technologies, invalid form
controls can also be assigned an `aria-invalid="true"` attribute (see below).

### ARIA `aria-invalid` attribute

When `<b-form-radio-group>` has an invalid contextual state (i.e. `invalid`) you may also want to
set the `<b-form-radio-group>` prop `aria-invalid` to `true`.

Supported `aria-invalid` values are:

- `false` (default) No errors detected
- `true` The value has failed validation.

`aria-invalid` is automatically set to `true` if the `state` prop is `'invalid'` (or `false`).

<!-- Component reference added automatically from component package.json -->
