# Form Checkbox Inputs

> For cross browser consistency, `<b-form-checkbox-group>` and `<b-form-checkbox>` use Bootstrap's
> custom checkbox input to replace the browser default checkbox input. It is built on top of
> semantic and accessible markup, so it is a solid replacement for the default checkbox input.

**Example 1:** Single checkbox

```html
<template>
  <div>
    <b-form-checkbox
      id="checkbox-1"
      v-model="status"
      name="checkbox-1"
      value="accepted"
      unchecked-value="not_accepted"
    >
      I accept the terms and use
    </b-form-checkbox>

    <div>State: <strong>{{ status }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        status: 'not_accepted'
      }
    }
  }
</script>

<!-- b-form-checkbox.vue -->
```

**Example 2:** Multiple choice checkboxes

```html
<template>
  <div>
    <b-form-group label="Using options array:">
      <b-form-checkbox-group
        id="checkbox-group-1"
        v-model="selected"
        :options="options"
        name="flavour-1"
      ></b-form-checkbox-group>
    </b-form-group>

    <b-form-group label="Using sub-components:">
      <b-form-checkbox-group id="checkbox-group-2" v-model="selected" name="flavour-2">
        <b-form-checkbox value="orange">Orange</b-form-checkbox>
        <b-form-checkbox value="apple">Apple</b-form-checkbox>
        <b-form-checkbox value="pineapple">Pineapple</b-form-checkbox>
        <b-form-checkbox value="grape">Grape</b-form-checkbox>
      </b-form-checkbox-group>
    </b-form-group>

    <div>Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: [], // Must be an array reference!
        options: [
          { text: 'Orange', value: 'orange' },
          { text: 'Apple', value: 'apple' },
          { text: 'Pineapple', value: 'pineapple' },
          { text: 'Grape', value: 'grape' }
        ]
      }
    }
  }
</script>

<!-- b-form-checkbox-multiple.vue -->
```

Feel free to mix and match `options` prop and `<b-form-checkbox>` in `<b-form-checkbox-group>`.
Manually placed `<b-form-checkbox>` inputs will appear _below_ any checkbox inputs generated by the
`options` prop. To have them appear _above_ the inputs generated by `options`, place them in the
named slot `first`.

## Checkbox group options array

Please see options in [`<b-form-select>`](/docs/components/form-select) docs for details on passing
`options` (value array) to `<b-form-checkbox-group>`.

## Inline and Stacked checkboxes

`<b-form-checkbox-group>` components render inline checkboxes by default, while `<b-form-checkbox>`
renders block-level (stacked) checkboxes.

Set the prop `stacked` on `<b-form-checkbox-group>` to place each form control one over the other,
or if using individual checkboxes not inside a `<b-form-checkbox-group>`, set the `inline` prop on
`<b-form-checkbox>`.

```html
<template>
  <div>
    <b-form-group label="Form-checkbox-group inline checkboxes (default)">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        name="flavour-1a"
      ></b-form-checkbox-group>
    </b-form-group>

    <b-form-group label="Form-checkbox-group stacked checkboxes">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        name="flavour-2a"
        stacked
      ></b-form-checkbox-group>
    </b-form-group>

    <b-form-group label="Individual stacked checkboxes (default)">
      <b-form-checkbox
        v-for="option in options"
        v-model="selected"
        :key="option.value"
        :value="option.value"
        name="flavour-3a"
      >
        {{ option.text }}
      </b-form-checkbox>
    </b-form-group>

    <b-form-group label="Individual inline checkboxes">
      <b-form-checkbox
        v-for="option in options"
        v-model="selected"
        :key="option.value"
        :value="option.value"
        name="flavour-4a"
        inline
      >
        {{ option.text }}
      </b-form-checkbox>
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: [], // Must be an array reference!
        options: [
          { text: 'Orange', value: 'orange' },
          { text: 'Apple', value: 'apple' },
          { text: 'Pineapple', value: 'pineapple' },
          { text: 'Grape', value: 'grape' }
        ]
      }
    }
  }
</script>

<!-- b-form-checkbox-stacked.vue -->
```

## Checkbox values and `v-model`

By default, `<b-form-checkbox>` value will be `true` when checked and `false` when unchecked. You
can customize the checked and unchecked values by specifying the `value` and `unchecked-value`
properties, respectively.

The `v-model` binds to the `checked` prop. When you have multiple checkboxes that bind to a single
data state variable, you **must** provide an array reference (`[]`) to your `v-model`. Do not use
the `checked` prop directly.

Note that when `v-model` is bound to multiple checkboxes (i.e an array ref), the `unchecked-value`
is **not used**. Only the value(s) of the checked checkboxes will be returned in the `v-model` bound
array. You should provide a unique value for each checkbox's `value` prop (the default of `true`
will not work when bound to an array)

To pre-check any radios, set the `v-model` to the value(s) of the checks that you would like
pre-selected.

When placing individual `<b-form-checkbox>` components within a `<b-form-checkbox-group>`, most
props and the `v-model` are inherited from the `<b-form-checkbox-group>`.

### Multiple checkboxes and accessibility

When binding multiple checkboxes together, you must set the `name` prop to the same value for all
`<b-form-checkbox>`s in the group individually or via the `name` prop of `<b-form-checkbox-group>`.
This will inform users of assistive technologies that the checkboxes are related and enables native
browser keyboard navigation.

Whenever using multiple checkboxes, it is recommended that the checkboxes be placed in a
[`<b-form-group>`](/docs/components/form-group) component to associate a label with the entire group
of checkboxes. See examples above.

## Button style checkboxes

You can optionally render checkboxes to appear as buttons, either individually, or in a group.

Button style checkboxes will have the class `.active` automatically applied to the label when they
are in the _checked_ state.

### Individual checkbox button style

A single checkbox can be rendered with a button appearance by setting the prop `button` to `true`

Change the button variant by setting the `button-variant` prop to one of the standard Bootstrap
button variants (see [`<b-button>`](/docs/components/button) for supported variants). The default
variant is `secondary`.

```html
<template>
  <div>
    <b-form-checkbox v-model="checked1" name="check-button" button>
      Button Checkbox <b>(Checked: {{ checked1 }})</b>
    </b-form-checkbox>
    <b-form-checkbox v-model="checked2" name="check-button" button button-variant="info">
      Button Checkbox <b>(Checked: {{ checked2 }})</b>
    </b-form-checkbox>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        checked1: false,
        checked2: false
      }
    }
  }
</script>

<!-- b-form-checkbox-button.vue -->
```

The `inline` prop has no effect on individual button-style checkboxes.

### Grouped button style checkboxes

Render groups of checkboxes with the look of a button-group by setting the prop `buttons` on
`<b-form-checkbox-group>`. Change the button variant by setting the `button-variant` prop to one of
the standard Bootstrap button variants (see [`<b-button>`](/docs/components/button) for supported
variants). The default `button-variant` is `secondary`.

```html
<template>
  <div>
    <b-form-group label="Button-group style checkboxes">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        name="buttons-1"
        buttons
      ></b-form-checkbox-group>
    </b-form-group>

    <b-form-group label="Button-group style checkboxes with variant primary and large buttons">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        buttons
        button-variant="primary"
        size="lg"
        name="buttons-2"
      ></b-form-checkbox-group>
    </b-form-group>

    <b-form-group label="Stacked (vertical) button-group style checkboxes">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        stacked
        buttons
      ></b-form-checkbox-group>
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: [], // Must be an array reference!
        options: [
          { text: 'Orange', value: 'orange' },
          { text: 'Apple', value: 'apple' },
          { text: 'Pineapple', value: 'pineapple' },
          { text: 'Grape', value: 'grape' }
        ]
      }
    }
  }
</script>

<!-- b-form-checkbox-button-group.vue -->
```

## Switch style checkboxes

Switch styling is supported on `<b-form-checkbox>` and `<b-form-checkbox-group>` components.

**Note:** If the checkbox is in [button mode](#button-style-checkboxes), switch mode will have no
effect.

### Individual checkbox switch style

A single checkbox can be rendered with a switch appearance by setting the prop `switch` to `true`

```html
<template>
  <div>
    <b-form-checkbox v-model="checked" name="check-button" switch>
      Switch Checkbox <b>(Checked: {{ checked }})</b>
    </b-form-checkbox>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        checked: false
      }
    }
  }
</script>

<!-- b-form-checkbox-switch.vue -->
```

### Grouped switch style checkboxes

Render groups of checkboxes with the look of a switches by setting the prop `switches` on
`<b-form-checkbox-group>`.

```html
<template>
  <div>
    <b-form-group label="Inline switch style checkboxes">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        switches
      ></b-form-checkbox-group>
    </b-form-group>

    <b-form-group label="Stacked (vertical) switch style checkboxes">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        switches
        stacked
      ></b-form-checkbox-group>
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: [], // Must be an array reference!
        options: [
          { text: 'Red', value: 'red' },
          { text: 'Green', value: 'green' },
          { text: 'Yellow (disabled)', value: 'yellow', disabled: true },
          { text: 'Blue', value: 'blue' }
        ]
      }
    }
  }
</script>

<!-- b-form-checkboxes-switch-group.vue -->
```

## Non custom check inputs (plain)

You can have `<b-form-checkbox-group>` or `<b-form-checkbox>` render a browser native checkbox input
by setting the `plain` prop.

```html
<template>
  <div>
    <b-form-group label="Plain inline checkboxes">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        plain
      ></b-form-checkbox-group>
    </b-form-group>

    <b-form-group label="Plain stacked checkboxes">
      <b-form-checkbox-group
        v-model="selected"
        :options="options"
        plain
        stacked
      ></b-form-checkbox-group>
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: [], // Must be an array reference!
        options: [
          { text: 'Orange', value: 'orange' },
          { text: 'Apple', value: 'apple' },
          { text: 'Pineapple', value: 'pineapple' },
          { text: 'Grape', value: 'grape' }
        ]
      }
    }
  }
</script>

<!-- b-form-checkbox-plain.vue -->
```

**Note:** The `plain` prop has no effect when `button` or `buttons` is set.

## Contextual states

Bootstrap includes validation styles for `valid` and `invalid` states on most form controls.

Generally speaking, you’ll want to use a particular state for specific types of feedback:

- `'invalid'` (or `false`) is great for when there’s a blocking or required field. A user must fill
  in this field properly to submit the form.
- `'valid'` (or `true`) is ideal for situations when you have per-field validation throughout a form
  and want to encourage a user through the rest of the fields.
- `null` Displays no validation state

To apply one of the contextual state icons on `<b-form-checkbox>`, set the `state` prop to
`'invalid'` (or `false`), `'valid'` (or `true`), or `null`.

**Note:** Contextual states are **not** supported when in button mode.

### Contextual state and validation example

```html
<template>
  <div>
    <b-form-checkbox-group
      v-model="value"
      :options="options"
      :state="state"
      name="checkbox-validation"
    >
      <b-form-invalid-feedback :state="state">Please select two</b-form-invalid-feedback>
      <b-form-valid-feedback :state="state">Thank you</b-form-valid-feedback>
    </b-form-checkbox-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        value: [],
        options: [
          { text: 'First Check', value: 'first' },
          { text: 'Second Check', value: 'second' },
          { text: 'Third Check', value: 'third' }
        ]
      }
    },
    computed: {
      state() {
        return this.value.length === 2
      }
    }
  }
</script>

<!-- form-checkbox-validation.vue -->
```

### Required constraint

When using individual `<b-form-checkbox>` components (not in a `<b-form-checkbox-group>`), and you
want the checkbox(es) to be `required` in your form, you **must** provide a `name` on each
`<b-form-checkbox>` in order for the required constraint to work. All `<b-form-checkbox>` components
tied to the same `v-model` **must** have the same `name`.

The `name` is required in order for Assistive Technologies (such as screen readers, and keyboard
only users) to know which checkboxes belong to the same form variable (the name also automatically
enables native browser keyboard navigation), hence `required` will only work if `name` is set.
`<b-form-checkbox-group>` will automatically generate a unique input name if one is not provided on
the group.

## Indeterminate (tri-state) support

Normally a checkbox input can only have two states: _checked_ or _unchecked_. They can have any
value, but they either submit that value (_checked_) or don't (_unchecked_) with a form submission
(although BootstrapVue allows a value for the _unchecked_ state on a single checkbox)

_Visually_, there are actually three states a checkbox can be in: _checked_, _unchecked_, or
**_indeterminate_**.

The _indeterminate_ state is **visual only**. The checkbox is still either checked or unchecked as a
value. That means the visual indeterminate state masks the real value of the checkbox, so that
better make sense in your UI!

`<b-form-checkbox>` supports setting this visual indeterminate state via the `indeterminate` prop
(defaults to `false`). Clicking the checkbox will clear its indeterminate state. The `indeterminate`
prop can be synced to the checkbox's state by v-binding the `indeterminate` prop with the `.sync`
modifier.

**Note:** indeterminate styling is not supported in button or switch mode, nor is it supported in
`<b-form-checkbox-group>` (multiple checkboxes).

**Single Indeterminate checkbox:**

```html
<template>
  <div>
    <b-form-checkbox v-model="checked" :indeterminate.sync="indeterminate">
      Click me to see what happens
    </b-form-checkbox>

    <div class="mt-3">
      Checked: <strong>{{ checked }}</strong><br>
      Indeterminate: <strong>{{ indeterminate }}</strong>
    </div>

    <b-button @click="toggleIndeterminate">Toggle Indeterminate State</b-button>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        checked: true,
        indeterminate: true
      }
    },
    methods: {
      toggleIndeterminate() {
        this.indeterminate = !this.indeterminate
      }
    }
  }
</script>

<!-- b-form-checkbox-indeterminate.vue -->
```

**Indeterminate checkbox use-case example:**

```html
<template>
  <div>
    <b-form-group>
      <template slot="label">
        <b>Choose your flavours:</b><br>
        <b-form-checkbox
          v-model="allSelected"
          :indeterminate="indeterminate"
          aria-describedby="flavours"
          aria-controls="flavours"
          @change="toggleAll"
        >
          {{ allSelected ? 'Un-select All' : 'Select All' }}
        </b-form-checkbox>
      </template>

      <b-form-checkbox-group
        id="flavors"
        v-model="selected"
        :options="flavours"
        name="flavors"
        class="ml-4"
        aria-label="Individual flavours"
        stacked
      ></b-form-checkbox-group>
    </b-form-group>

    <div>
      Selected: <strong>{{ selected }}</strong><br>
      All Selected: <strong>{{ allSelected }}</strong><br>
      Indeterminate: <strong>{{ indeterminate }}</strong>
    </div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        flavours: ['Orange', 'Grape', 'Apple', 'Lime', 'Very Berry'],
        selected: [],
        allSelected: false,
        indeterminate: false
      }
    },
    methods: {
      toggleAll(checked) {
        this.selected = checked ? this.flavours.slice() : []
      }
    },
    watch: {
      selected(newVal, oldVal) {
        // Handle changes in individual flavour checkboxes
        if (newVal.length === 0) {
          this.indeterminate = false
          this.allSelected = false
        } else if (newVal.length === this.flavours.length) {
          this.indeterminate = false
          this.allSelected = true
        } else {
          this.indeterminate = true
          this.allSelected = false
        }
      }
    }
  }
</script>

<!-- b-form-checkbox-indeterminate-multiple.vue -->
```

**Note:** indeterminate is not supported in `button` mode, nor in multiple checkbox mode. Also pay
attention that plain checkbox (i.e. with prop `plain`) also supports indeterminate state on
Windows/Linux/Mac/Android, but not on iOS.

### Indeterminate state and accessibility

Not all screen readers will convey the indeterminate state to screen reader users. So it is
recommended to provide some form of textual feedback to the user (possibly by via the `.sr-only`
class) if the indeterminate state has special contextual meaning in your application.

<!-- Component reference added automatically from component package.json -->
