"use strict";

exports.__esModule = true;
exports.default = void 0;

var _keyCodes = _interopRequireDefault(require("../../../utils/key-codes"));

var _get = _interopRequireDefault(require("../../../utils/get"));

var _toString = _interopRequireDefault(require("../../../utils/to-string"));

var _array = require("../../../utils/array");

var _inspect = require("../../../utils/inspect");

var _filterEvent = _interopRequireDefault(require("./filter-event"));

var _textSelectionActive = _interopRequireDefault(require("./text-selection-active"));

var _tr = require("../tr");

var _td = require("../td");

var _th = require("../th");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(source, true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(source).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

var detailsSlotName = 'row-details';
var _default = {
  props: {
    tbodyTrClass: {
      type: [String, Array, Function],
      default: null
    }
  },
  methods: {
    // Methods for computing classes, attributes and styles for table cells
    getTdValues: function getTdValues(item, key, tdValue, defValue) {
      var parent = this.$parent;

      if (tdValue) {
        var value = (0, _get.default)(item, key, '');

        if ((0, _inspect.isFunction)(tdValue)) {
          return tdValue(value, key, item);
        } else if ((0, _inspect.isString)(tdValue) && (0, _inspect.isFunction)(parent[tdValue])) {
          return parent[tdValue](value, key, item);
        }

        return tdValue;
      }

      return defValue;
    },
    // Method to get the value for a field
    getFormattedValue: function getFormattedValue(item, field) {
      var key = field.key;
      var formatter = this.getFieldFormatter(key);
      var value = (0, _get.default)(item, key, null);

      if ((0, _inspect.isFunction)(formatter)) {
        value = formatter(value, key, item);
      }

      return (0, _inspect.isUndefinedOrNull)(value) ? '' : value;
    },
    // Factory function methods
    toggleDetailsFactory: function toggleDetailsFactory(hasDetailsSlot, item) {
      var _this = this;

      // Returns a function to toggle a row's details slot
      return function () {
        if (hasDetailsSlot) {
          _this.$set(item, '_showDetails', !item._showDetails);
        }
      };
    },
    rowEvtFactory: function rowEvtFactory(handler, item, rowIndex) {
      var _this2 = this;

      // Return a row event handler
      return function (evt) {
        // If table is busy (via provider) then don't propagate
        if (_this2.stopIfBusy && _this2.stopIfBusy(evt)) {
          return;
        } // Otherwise call the handler


        handler(evt, item, rowIndex);
      };
    },
    // Row event handlers (will be wrapped by the above rowEvtFactory function)
    tbodyRowKeydown: function tbodyRowKeydown(evt, item, rowIndex) {
      // Keypress handler
      var keyCode = evt.keyCode;
      var target = evt.target; // `this.$refs.itemRow`s is most likely an array of `BTr` components, but it
      // could be regular `tr` elements, so we map to the `tr` elements just in case

      var trs = (this.$refs.itemRows || []).map(function (tr) {
        return tr.$el || tr;
      });

      if (!(target && target.tagName === 'TR' && target === document.activeElement)) {
        // Ignore if not the active tr element
        return;
      } else if (target.tabIndex !== 0) {
        // Ignore if not focusable

        /* istanbul ignore next */
        return;
      } else if (trs.length === 0) {
        // No item rows

        /* istanbul ignore next */
        return;
      }

      var index = trs.indexOf(target);

      if (keyCode === _keyCodes.default.ENTER || keyCode === _keyCodes.default.SPACE) {
        // We also allow enter/space to trigger a click (when row is focused)
        evt.stopPropagation();
        evt.preventDefault(); // We translate to a row-clicked event

        this.rowClicked(evt, item, rowIndex);
      } else if ((0, _array.arrayIncludes)([_keyCodes.default.UP, _keyCodes.default.DOWN, _keyCodes.default.HOME, _keyCodes.default.END], keyCode)) {
        // Keyboard navigation of rows
        evt.stopPropagation();
        evt.preventDefault();
        var shift = evt.shiftKey;

        if (keyCode === _keyCodes.default.HOME || shift && keyCode === _keyCodes.default.UP) {
          // Focus first row
          trs[0].focus();
        } else if (keyCode === _keyCodes.default.END || shift && keyCode === _keyCodes.default.DOWN) {
          // Focus last row
          trs[trs.length - 1].focus();
        } else if (keyCode === _keyCodes.default.UP && index > 0) {
          // Focus previous row
          trs[index - 1].focus();
        } else if (keyCode === _keyCodes.default.DOWN && index < trs.length - 1) {
          // Focus next row
          trs[index + 1].focus();
        }
      }
    },
    rowClicked: function rowClicked(evt, item, index) {
      if ((0, _filterEvent.default)(evt)) {
        // clicked on a non-disabled control so ignore
        return;
      } else if ((0, _textSelectionActive.default)(this.$el)) {
        // User is selecting text, so ignore

        /* istanbul ignore next: JSDOM doesn't support getSelection() */
        return;
      }

      this.$emit('row-clicked', item, index, evt);
    },
    middleMouseRowClicked: function middleMouseRowClicked(evt, item, index) {
      if (evt.which === 2) {
        this.$emit('row-middle-clicked', item, index, evt);
      }
    },
    rowDblClicked: function rowDblClicked(evt, item, index) {
      if ((0, _filterEvent.default)(evt)) {
        // clicked on a non-disabled control so ignore

        /* istanbul ignore next: event filtering already tested via click handler */
        return;
      }

      this.$emit('row-dblclicked', item, index, evt);
    },
    rowHovered: function rowHovered(evt, item, index) {
      this.$emit('row-hovered', item, index, evt);
    },
    rowUnhovered: function rowUnhovered(evt, item, index) {
      this.$emit('row-unhovered', item, index, evt);
    },
    rowContextmenu: function rowContextmenu(evt, item, index) {
      this.$emit('row-contextmenu', item, index, evt);
    },
    // Render helpers
    renderTbodyRowCell: function renderTbodyRowCell(field, colIndex, item, rowIndex) {
      // Renders a TD or TH for a row's field
      var h = this.$createElement;
      var hasDetailsSlot = this.hasNormalizedSlot(detailsSlotName);
      var formatted = this.getFormattedValue(item, field);
      var key = field.key;
      var data = {
        // For the Vue key, we concatenate the column index and
        // field key (as field keys could be duplicated)
        // TODO: Although we do prevent duplicate field keys...
        //   So we could change this to: `row-${rowIndex}-cell-${key}`
        key: "row-".concat(rowIndex, "-cell-").concat(colIndex, "-").concat(key),
        class: [field.class ? field.class : '', this.getTdValues(item, key, field.tdClass, '')],
        props: {
          stackedHeading: this.isStacked ? field.label : null,
          stickyColumn: field.stickyColumn,
          variant: item._cellVariants && item._cellVariants[key] ? item._cellVariants[key] : field.variant || null
        },
        attrs: _objectSpread({
          'aria-colindex': String(colIndex + 1)
        }, this.getTdValues(item, key, field.tdAttr, {}))
      };
      var slotScope = {
        item: item,
        index: rowIndex,
        field: field,
        unformatted: (0, _get.default)(item, key, ''),
        value: formatted,
        toggleDetails: this.toggleDetailsFactory(hasDetailsSlot, item),
        detailsShowing: Boolean(item._showDetails)
      };

      if (this.selectedRows) {
        // Add in rowSelected scope property if selectable rows supported
        slotScope.rowSelected = this.isRowSelected(rowIndex);
      } // TODO:
      //   Using `field.key` as scoped slot name is deprecated, to be removed in future release
      //   New format uses the square bracketed naming convention


      var $childNodes = this.normalizeSlot(["[".concat(key, "]"), '[]', key], slotScope) || (0, _toString.default)(formatted);

      if (this.isStacked) {
        // We wrap in a DIV to ensure rendered as a single cell when visually stacked!
        $childNodes = [h('div', {}, [$childNodes])];
      } // Render either a td or th cell


      return h(field.isRowHeader ? _th.BTh : _td.BTd, data, [$childNodes]);
    },
    renderTbodyRow: function renderTbodyRow(item, rowIndex) {
      var _this3 = this;

      // Renders an item's row (or rows if details supported)
      var h = this.$createElement;
      var fields = this.computedFields;
      var tableStriped = this.striped;
      var hasDetailsSlot = this.hasNormalizedSlot(detailsSlotName);
      var rowShowDetails = Boolean(item._showDetails && hasDetailsSlot);
      var hasRowClickHandler = this.$listeners['row-clicked'] || this.isSelectable; // We can return more than one TR if rowDetails enabled

      var $rows = []; // Details ID needed for aria-describedby when details showing

      var detailsId = rowShowDetails ? this.safeId("_details_".concat(rowIndex, "_")) : null; // For each item data field in row

      var $tds = fields.map(function (field, colIndex) {
        return _this3.renderTbodyRowCell(field, colIndex, item, rowIndex);
      }); // Calculate the row number in the dataset (indexed from 1)

      var ariaRowIndex = null;

      if (this.currentPage && this.perPage && this.perPage > 0) {
        ariaRowIndex = String((this.currentPage - 1) * this.perPage + rowIndex + 1);
      } // Create a unique :key to help ensure that sub components are re-rendered rather than
      // re-used, which can cause issues. If a primary key is not provided we use the rendered
      // rows index within the tbody.
      // See: https://github.com/bootstrap-vue/bootstrap-vue/issues/2410


      var primaryKey = this.primaryKey;
      var hasPkValue = primaryKey && !(0, _inspect.isUndefinedOrNull)(item[primaryKey]);
      var rowKey = hasPkValue ? (0, _toString.default)(item[primaryKey]) : String(rowIndex); // If primary key is provided, use it to generate a unique ID on each tbody > tr
      // In the format of '{tableId}__row_{primaryKeyValue}'

      var rowId = hasPkValue ? this.safeId("_row_".concat(item[primaryKey])) : null;
      var evtFactory = this.rowEvtFactory;
      var handlers = {};

      if (hasRowClickHandler) {
        handlers.click = evtFactory(this.rowClicked, item, rowIndex);
        handlers.keydown = evtFactory(this.tbodyRowKeydown, item, rowIndex);
      } // Selectable classes and attributes


      var selectableClasses = this.selectableRowClasses ? this.selectableRowClasses(rowIndex) : {};
      var selectableAttrs = this.selectableRowAttrs ? this.selectableRowAttrs(rowIndex) : {}; // Add the item row

      $rows.push(h(_tr.BTr, {
        key: "__b-table-row-".concat(rowKey, "__"),
        ref: 'itemRows',
        refInFor: true,
        class: [(0, _inspect.isFunction)(this.tbodyTrClass) ? this.tbodyTrClass(item, 'row') : this.tbodyTrClass, selectableClasses, rowShowDetails ? 'b-table-has-details' : ''],
        props: {
          variant: item._rowVariant || null
        },
        attrs: _objectSpread({
          id: rowId,
          tabindex: hasRowClickHandler ? '0' : null,
          'data-pk': rowId ? String(item[primaryKey]) : null,
          // Should this be `aria-details` instead?
          'aria-describedby': detailsId,
          'aria-owns': detailsId,
          'aria-rowindex': ariaRowIndex
        }, selectableAttrs),
        on: _objectSpread({}, handlers, {
          // TODO:
          //   Instantiate the following handlers only if we have registered
          //   listeners i.e. `this.$listeners['row-middle-clicked']`, etc.
          //
          //   Could make all of this (including the above click/key handlers)
          //   the result of a factory function and/or make it a delegated event
          //   handler on the tbody (if we store the row index as a data-attribute
          //   on the TR as we can lookup the item data from the computedItems array
          //   or it could be a hidden prop (via attrs) on BTr instance)
          auxclick: evtFactory(this.middleMouseRowClicked, item, rowIndex),
          contextmenu: evtFactory(this.rowContextmenu, item, rowIndex),
          // Note: These events are not accessibility friendly!
          dblclick: evtFactory(this.rowDblClicked, item, rowIndex),
          mouseenter: evtFactory(this.rowHovered, item, rowIndex),
          mouseleave: evtFactory(this.rowUnhovered, item, rowIndex)
        })
      }, $tds)); // Row Details slot

      if (rowShowDetails) {
        var detailsScope = {
          item: item,
          index: rowIndex,
          fields: fields,
          toggleDetails: this.toggleDetailsFactory(hasDetailsSlot, item) // Render the details slot in a TD

        };
        var $details = h(_td.BTd, {
          props: {
            colspan: fields.length
          },
          attrs: {
            id: detailsId
          }
        }, [this.normalizeSlot(detailsSlotName, detailsScope)]); // Add a hidden row to keep table row striping consistent when details showing

        if (tableStriped) {
          $rows.push( // We don't use `BTr` here as we dont need the extra functionality
          h('tr', {
            key: "__b-table-details-".concat(rowIndex, "-stripe__"),
            staticClass: 'd-none',
            attrs: {
              'aria-hidden': 'true',
              role: 'presentation'
            }
          }));
        } // Add the actual details row


        $rows.push(h(_tr.BTr, {
          key: "__b-table-details-".concat(rowIndex, "__"),
          staticClass: 'b-table-details',
          class: [(0, _inspect.isFunction)(this.tbodyTrClass) ? this.tbodyTrClass(item, detailsSlotName) : this.tbodyTrClass],
          props: {
            variant: item._rowVariant || null
          },
          attrs: {
            id: detailsId
          }
        }, [$details]));
      } else if (hasDetailsSlot) {
        // Only add the placeholder if a the table has a row-details slot defined (but not shown)
        $rows.push(h());

        if (tableStriped) {
          // Add extra placeholder if table is striped
          $rows.push(h());
        }
      } // Return the row(s)


      return $rows;
    }
  }
};
exports.default = _default;