import bImg from './img';
import { isVisible, getBCR, eventOn, eventOff } from '../../utils/dom';
var THROTTLE = 100;

export default {
  components: { bImg: bImg },
  render: function render(h) {
    return h('b-img', {
      props: {
        src: this.computedSrc,
        alt: this.alt,
        blank: this.computedBlank,
        blankColor: this.blankColor,
        width: this.computedWidth,
        height: this.computedHeight,
        fluid: this.fluid,
        fluidGrow: this.fluidGrow,
        block: this.block,
        thumbnail: this.thumbnail,
        rounded: this.rounded,
        left: this.left,
        right: this.right,
        center: this.center
      }
    });
  },
  data: function data() {
    return {
      isShown: false,
      scrollTimeout: null
    };
  },

  props: {
    src: {
      type: String,
      default: null,
      required: true
    },
    alt: {
      type: String,
      default: null
    },
    width: {
      type: [Number, String],
      default: null
    },
    height: {
      type: [Number, String],
      default: null
    },
    blankSrc: {
      // If null, a blank image is generated
      type: String,
      default: null
    },
    blankColor: {
      type: String,
      default: 'transparent'
    },
    blankWidth: {
      type: [Number, String],
      default: null
    },
    blankHeight: {
      type: [Number, String],
      default: null
    },
    fluid: {
      type: Boolean,
      default: false
    },
    fluidGrow: {
      type: Boolean,
      default: false
    },
    block: {
      type: Boolean,
      default: false
    },
    thumbnail: {
      type: Boolean,
      default: false
    },
    rounded: {
      type: [Boolean, String],
      default: false
    },
    left: {
      type: Boolean,
      default: false
    },
    right: {
      type: Boolean,
      default: false
    },
    center: {
      type: Boolean,
      default: false
    },
    offset: {
      type: [Number, String],
      default: 360
    },
    throttle: {
      type: [Number, String],
      default: THROTTLE
    }
  },
  computed: {
    computedSrc: function computedSrc() {
      return !this.blankSrc || this.isShown ? this.src : this.blankSrc;
    },
    computedBlank: function computedBlank() {
      return !(this.isShown || this.blankSrc);
    },
    computedWidth: function computedWidth() {
      return this.isShown ? this.width : this.blankWidth || this.width;
    },
    computedHeight: function computedHeight() {
      return this.isShown ? this.height : this.blankHeight || this.height;
    }
  },
  mounted: function mounted() {
    this.setListeners(true);
    this.checkView();
  },
  activated: function activated() {
    this.setListeners(true);
    this.checkView();
  },
  deactivated: function deactivated() {
    this.setListeners(false);
  },
  beforeDdestroy: function beforeDdestroy() {
    this.setListeners(false);
  },

  methods: {
    setListeners: function setListeners(on) {
      clearTimeout(this.scrollTimer);
      this.scrollTimeout = null;
      var root = window;
      if (on) {
        eventOn(root, 'scroll', this.onScroll);
        eventOn(root, 'resize', this.onScroll);
        eventOn(root, 'orientationchange', this.onScroll);
      } else {
        eventOff(root, 'scroll', this.onScroll);
        eventOff(root, 'resize', this.onScroll);
        eventOff(root, 'orientationchange', this.onScroll);
      }
    },
    checkView: function checkView() {
      // check bounding box + offset to see if we should show
      if (!isVisible(this.$el)) {
        // Element is hidden, so skip for now
        return;
      }
      var offset = parseInt(this.offset, 10) || 0;
      var docElement = document.documentElement;
      var view = {
        l: 0 - offset,
        t: 0 - offset,
        b: docElement.clientHeight + offset,
        r: docElement.clientWidth + offset
      };
      var box = getBCR(this.$el);
      if (box.right >= view.l && box.bottom >= view.t && box.left <= view.r && box.top <= view.b) {
        // image is in view (or about to be in view)
        this.isShown = true;
        this.setListeners(false);
      }
    },
    onScroll: function onScroll() {
      if (this.isShown) {
        this.setListeners(false);
      } else {
        clearTimeout(this.scrollTimeout);
        this.scrollTimeout = setTimeout(this.checkView, parseInt(this.throttle, 10) || THROTTLE);
      }
    }
  }
};