# Form Radio Inputs

> For cross browser consistency, `<b-form-radio-group>` and `<b-form-radio>` uses Bootstrap's custom
> radio input to replace the browser default radio input. It is built on top of semantic and
> accessible markup, so it is a solid replacement for the default radio input.

## Individual radios

```html
<template>
  <div>
    <b-form-group label="Individual radios">
      <b-form-radio v-model="selected" name="some-radios" value="A">Option A</b-form-radio>
      <b-form-radio v-model="selected" name="some-radios" value="B">Option B</b-form-radio>
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: ''
      }
    }
  }
</script>

<!-- b-form-radio.vue -->
```

## Grouped radios

The individual radio inputs in `<b-form-radio-group>` can be specified via the `options` prop, or
via manual placement of the `<b-form-radio>` sub component. When using manually placed
`<b-form-radio>` components within a `<b-form-radio-group>`, they will inherit most props and the
v-model from the `<b-form-radio-group>`.

```html
<template>
  <div>
    <b-form-group label="Radios using options">
      <b-form-radio-group
        id="radio-group-1"
        v-model="selected"
        :options="options"
        name="radio-options"
      ></b-form-radio-group>
    </b-form-group>

    <b-form-group label="Radios using sub-components">
      <b-form-radio-group id="radio-group-2" v-model="selected" name="radio-sub-component">
        <b-form-radio value="first">Toggle this custom radio</b-form-radio>
        <b-form-radio value="second">Or toggle this other custom radio</b-form-radio>
        <b-form-radio value="third" disabled>This one is Disabled</b-form-radio>
        <b-form-radio :value="{ fourth: 4 }">This is the 4th radio</b-form-radio>
      </b-form-radio-group>
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'first',
        options: [
          { text: 'Toggle this custom radio', value: 'first' },
          { text: 'Or toggle this other custom radio', value: 'second' },
          { text: 'This one is Disabled', value: 'third', disabled: true },
          { text: 'This is the 4th radio', value: { fourth: 4 } }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-group.vue -->
```

Feel free to mix and match `options` prop and `<b-form-radio>` in `<b-form-radio-group>`. Manually
placed `<b-form-radio>` inputs will appear _below_ any radio inputs generated by the `options` prop.
To have them appear _above_ the inputs generated by `options`, place them in the named slot `first`.

```html
<template>
  <div>
    <b-form-group label="Radios using options and slots">
      <b-form-radio-group
        id="radio-slots"
        v-model="selected"
        :options="options"
        name="radio-options-slots"
      >
        <!-- Radios in this slot will appear first -->
        <template v-slot:first>
          <b-form-radio value="first">Toggle this custom radio from slot first</b-form-radio>
        </template>

        <!-- Radios in the default slot will appear after any option generated radios -->
        <b-form-radio :value="{ fourth: 4 }">This is the 4th radio</b-form-radio>
        <b-form-radio value="fifth">This is the 5th radio</b-form-radio>
      </b-form-radio-group>
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: '',
        options: [
          { text: 'Or toggle this other custom radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-group-slot.vue -->
```

## Radio group options array

`options` can be an array of strings or objects. Available fields:

- **`value`** The selected value which will be set on `v-model`
- **`disabled`** Disables item for selection
- **`text`** Display text, or **`html`** Display basic inline html

`value` can be a string, number, or simple object. Avoid using complex types in values.

If both `html` and `text` are provided, `html` will take precedence. Only basic/native HTML is
supported in the `html` field (components will not work). Note that not all browsers will render
inline html (i.e. `<i>`, `<strong>`, etc) inside `<option>` elements of a `<select>`.

<p class="alert alert-danger">
  <strong>Be cautious</strong> of placing user supplied content in the <code>html</code> field,
  as it may make you vulnerable to
  <a class="alert-link" href="https://en.wikipedia.org/wiki/Cross-site_scripting">
  <abbr title="Cross Site Scripting Attacks">XSS attacks</abbr></a>, if you do not first
  <a class="alert-link" href="https://en.wikipedia.org/wiki/HTML_sanitization">sanitize</a> the
  user supplied string.
</p>

<!-- eslint-disable no-unused-vars -->

```js
const options = ['A', 'B', 'C', { text: 'D', value: { d: 1 }, disabled: true }, 'E', 'F']
```

If an array entry is a string, it will be used for both the generated `value` and `text` fields.

You can mix using strings and [objects](#options-as-an-array-of-objects) in the array.

Internally, BootstrapVue will convert the above array to the following array (the
[array of objects](#options-as-an-array-of-objects)) format:

<!-- eslint-disable no-unused-vars -->

```js
const options = [
  { text: 'A', value: 'A', disabled: false },
  { text: 'B', value: 'B', disabled: false },
  { text: 'C', value: 'C', disabled: false },
  { text: 'D', value: { d: 1 }, disabled: true },
  { text: 'E', value: 'E', disabled: false },
  { text: 'F', value: 'F', disabled: false }
]
```

### Options as an array of objects

<!-- eslint-disable no-unused-vars -->

```js
const options = [
  { text: 'Item 1', value: 'first' },
  { text: 'Item 2', value: 'second' },
  { html: '<b>Item</b> 3', value: 'third', disabled: true },
  { text: 'Item 4' },
  { text: 'Item 5', value: { foo: 'bar', baz: true } }
]
```

If `value` is missing, then `text` will be used as both the `value` and `text` fields. If you use
the `html` property, you **must** supply a `value` property.

Internally, BootstrapVue will convert the above array to the following array (the
[array of objects](#options-as-an-array-of-objects)) format:

<!-- eslint-disable no-unused-vars -->

```js
const options = [
  { text: 'Item 1', value: 'first', disabled: false },
  { text: 'Item 2', value: 'second', disabled: false },
  { html: '<b>Item</b> 3', value: 'third', disabled: true },
  { text: 'Item 4', value: 'Item 4', disabled: false },
  { text: 'Item 5', value: 'E', disabled: false }
]
```

### Changing the option field names

If you want to customize the field property names (for example using `name` field for display
`text`) you can easily change them by setting the `text-field`, `html-field`, `value-field`, and
`disabled-field` props to a string that contains the property name you would like to use:

```html
<template>
  <div>
    <b-form-radio-group
      v-model="selected"
      :options="options"
      class="mb-3"
      value-field="item"
      text-field="name"
      disabled-field="notEnabled"
    ></b-form-radio-group>
    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'A',
        options: [
          { item: 'A', name: 'Option A' },
          { item: 'B', name: 'Option B' },
          { item: 'D', name: 'Option C', notEnabled: true },
          { item: { d: 1 }, name: 'Option D' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-group-options-fields.vue -->
```

## Radio value and v-model

`<b-form-radio>` components do not have a value by default. You must explicitly supply a value via
the `value` prop on `<b-form-radio>`. This value will be sent to the `v-model` when the radio is
checked.

The `v-model` of both `<b-form-radio>` and `<b-form-radio-group>` binds to the `checked` prop. To
pre-check a radio, you must set the `v-model` value to the one of the radio's value (i.e. must match
the value of specified on one of the the radio's `value` prop). Do not use the `checked` prop
directly. Each radio in a radio group _must_ have a unique value.

Radios support values of many types, such as a `string`, `boolean`, `number`, or a plain `object`.

## Inline or stacked radios

By default `<b-form-radio-group>` generates inline radio inputs, while `<b-form-radio>` generates
stacked radios. Set the prop `stacked` on `<b-form-radio-group>` to make the radios appear one over
the other, or when using radios not in a group, set the `inline` prop on `b-form-radio` to true to
render them inline.

```html
<template>
  <div>
    <b-form-group label="Inline radios (default)">
      <b-form-radio-group
        v-model="selected"
        :options="options"
        name="radio-inline"
      ></b-form-radio-group>
    </b-form-group>

    <b-form-group label="Stacked radios">
      <b-form-radio-group
        v-model="selected"
        :options="options"
        name="radios-stacked"
        stacked
      ></b-form-radio-group>
    </b-form-group>

    <div class="mt-3">Selected: <strong>{{ selected }}</strong></div>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'first',
        options: [
          { text: 'First radio', value: 'first' },
          { text: 'Second radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-stacked.vue -->
```

## Control sizing

Use the `size` prop to control the size of the radio. The default size is medium. Supported size
values are `sm` (small) and `lg` (large).

```html
<div>
  <b-form-radio name="radio-size" size="sm">Small</b-form-radio>
  <b-form-radio name="radio-size">Default</b-form-radio>
  <b-form-radio name="radio-size" size="lg">Large</b-form-radio>
</div>

<!-- form-radio-sizes.vue -->
```

Sizes can be set on individual `<b-form-radio>` components, or inherited from the `size` setting of
`<b-form-radio-group>`.

**Note:** Bootstrap v4.x does not natively support sizes for the custom radio control. However,
BootstrapVue includes custom SCSS/CSS that adds support for sizing the custom radios.

## Button style radios

Render radios with the look of buttons by setting the prop `buttons` to `true` on
`<b-form-radio-group>`. Set the button variant by setting the `button-variant` prop to one of the
standard Bootstrap button variants (see [`<b-button>`](/docs/components/button) for supported
variants). The default `button-variant` is `secondary`.

The `buttons` prop has precedence over `plain`, and `button-variant` has no effect if `buttons` is
not set.

Button style radios will have the class `.active` automatically applied to their label when they are
in the checked state.

```html
<template>
  <div>
    <b-form-group label="Button style radios">
      <b-form-radio-group
        id="btn-radios-1"
        v-model="selected"
        :options="options"
        buttons
        name="radios-btn-default"
      ></b-form-radio-group>
    </b-form-group>

    <b-form-group label="Button style radios with outline-primary variant and size lg">
      <b-form-radio-group
        id="btn-radios-2"
        v-model="selected"
        :options="options"
        buttons
        button-variant="outline-primary"
        size="lg"
        name="radio-btn-outline"
      ></b-form-radio-group>
    </b-form-group>

    <b-form-group label="Stacked button style radios">
      <b-form-radio-group
        id="btn-radios-3"
        v-model="selected"
        :options="options"
        buttons
        stacked
        name="radio-btn-stacked"
      ></b-form-radio-group>
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'radio1',
        options: [
          { text: 'Radio 1', value: 'radio1' },
          { text: 'Radio 3', value: 'radio2' },
          { text: 'Radio 3 (disabled)', value: 'radio3', disabled: true },
          { text: 'Radio 4', value: 'radio4' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-buttons.vue -->
```

## Non-custom style radio inputs (plain)

You can have `<b-form-radio>` and `<b-form-radio-group>` render a browser native-styled radio input
by setting the `plain` prop.

```html
<template>
  <div>
    <b-form-group label="Plain inline radios">
      <b-form-radio-group
        v-model="selected"
        :options="options"
        plain
        name="plain-inline"
      ></b-form-radio-group>
    </b-form-group>

    <b-form-group label="Plain stacked radios">
      <b-form-radio-group
        v-model="selected"
        :options="options"
        plain
        stacked
        name="plain-stacked"
      ></b-form-radio-group>
    </b-form-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        selected: 'first',
        options: [
          { text: 'First radio', value: 'first' },
          { text: 'Second radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    }
  }
</script>

<!-- b-form-radio-plain.vue -->
```

**Note:** `plain` will have no effect if `buttons`/`button` is set.

## Required constraint

When using individual `<b-form-radio>` components (not in a `<b-form-radio-group>`), and you want
the radio(s) to be `required` in your form, you **must** provide a `name` on each `<b-form-radio>`
in order for the required constraint to work. All `<b-form-radio>` components tied to the same
`v-model` **must** have the same `name`.

The `name` is required in order for Assistive Technologies (such as screen readers, and keyboard
only users) to know which radios belong to the same form variable (the name also automatically
enables native browser keyboard navigation), hence `required` will only work if `name` is set.
`<b-form-radio-group>` will automatically generate a unique input name if one is not provided on the
group.

## Autofocus

When the `autofocus` prop is set on `<b-form-radio>`, the input will be auto-focused when it is
inserted (i.e. **mounted**) into the document or re-activated when inside a Vue `<keep-alive>`
component. Note that this prop **does not** set the `autofocus` attribute on the input, nor can it
tell when the input becomes visible.

## Contextual states

Bootstrap includes validation styles for `valid` and `invalid` states on most form controls.

Generally speaking, you'll want to use a particular state for specific types of feedback:

- `false` (denotes invalid state) is great for when there's a blocking or required field. A user
  must fill in this field properly to submit the form.
- `true` (denotes valid state) is ideal for situations when you have per-field validation throughout
  a form and want to encourage a user through the rest of the fields.
- `null` Displays no validation state (neither valid nor invalid)

To apply one of the contextual state icons on `<b-form-radio>`, set the `state` prop to `false` (for
invalid), `true` (for valid), or `null` (no validation state).

**Note:** Contextual state is not supported for radios rendered in buttons mode.

### Contextual state with feedback example

```html
<template>
  <div>
    <b-form-radio-group v-model="value" :options="options" :state="state" name="radio-validation">
      <b-form-invalid-feedback :state="state">Please select one</b-form-invalid-feedback>
      <b-form-valid-feedback :state="state">Thank you</b-form-valid-feedback>
    </b-form-radio-group>
  </div>
</template>

<script>
  export default {
    data() {
      return {
        value: null,
        options: [
          { text: 'First radio', value: 'first' },
          { text: 'Second radio', value: 'second' },
          { text: 'Third radio', value: 'third' }
        ]
      }
    },
    computed: {
      state() {
        return Boolean(this.value)
      }
    }
  }
</script>

<!-- b-form-radio-validation.vue -->
```

### Conveying contextual validation state to assistive technologies and colorblind users

Using these contextual states to denote the state of a form control only provides a visual,
color-based indication, which will not be conveyed to users of assistive technologies - such as
screen readers - or to colorblind users.

Ensure that an alternative indication of state is also provided. For instance, you could include a
hint about state in the form control's `<label>` text itself, or by providing an additional help
text block (i.e. `<b-form-invalid-feedback>`). Specifically for assistive technologies, invalid form
controls can also be assigned an `aria-invalid="true"` attribute (see below).

### ARIA `aria-invalid` attribute

When `<b-form-radio-group>` has an invalid contextual state (i.e. state = `false`) you may also want
to set the `<b-form-radio-group>` prop `aria-invalid` to `true`.

Supported `aria-invalid` values are:

- `false` (default) No errors detected
- `true` The value has failed validation.

`aria-invalid` is automatically set to `true` if the `state` prop is `false`.

<!-- Component reference added automatically from component package.json -->
