/*!
 * Bowser - a browser detector
 * https://github.com/lancedikson/bowser
 * MIT License | (c) Dustin Diaz 2012-2015
 * MIT License | (c) Denis Demchenko 2015-2017
 */
import Parser from './parser';

/**
 * Bowser class.
 * Keep it simple as much as it can be.
 * It's supposed to work with collections of {@link Parser} instances
 * rather then solve one-instance problems.
 * All the one-instance stuff is located in Parser class.
 */
class Bowser {
  /**
   * Creates a {@link module:parser:Parser} instance
   *
   * @param {String} UA UserAgent string
   * @param {Boolean} [skipParsing=false] same as skipParsing for {@link Parser}
   * @returns {Parser}
   * @throws {Error} when UA is not a String
   *
   * @example
   * const bowser = new Bowser(window.navigator.userAgent);
   * bowser.getResult()
   */
  static getParser(UA, skipParsing = false) {
    if (typeof UA !== 'string') {
      throw new Error('UserAgent should be a string');
    }
    return new Parser(UA, skipParsing);
  }

  /**
   * Creates a {@link Parser} instance and runs {@link Parser.getResult} immediately
   *
   * @param UA
   * @return {ParsedResult}
   */
  static parse(UA) {
    return (new Parser(UA)).getResult();
  }
}

export default Bowser;
