'use strict';

var hooks = require('../preact/hooks');
var minDash = require('min-dash');
var compat = require('../preact/compat');
var jsxRuntime = require('../preact/jsx-runtime');
var preact = require('../preact');
var classnames = require('classnames');
var minDom = require('min-dom');
var feelers = require('feelers');
var Editor = require('@bpmn-io/feel-editor');
var view = require('@codemirror/view');
var focusTrap = require('focus-trap');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var focusTrap__namespace = /*#__PURE__*/_interopNamespaceDefault(focusTrap);

var ArrowIcon = function ArrowIcon(props) {
  return jsxRuntime.jsx("svg", {
    ...props,
    children: jsxRuntime.jsx("path", {
      fillRule: "evenodd",
      d: "m11.657 8-4.95 4.95a1 1 0 0 1-1.414-1.414L8.828 8 5.293 4.464A1 1 0 1 1 6.707 3.05L11.657 8Z"
    })
  });
};
ArrowIcon.defaultProps = {
  xmlns: "http://www.w3.org/2000/svg",
  width: "16",
  height: "16"
};
var CloseIcon = function CloseIcon(props) {
  return jsxRuntime.jsx("svg", {
    ...props,
    children: jsxRuntime.jsx("path", {
      fillRule: "evenodd",
      d: "m12 4.7-.7-.7L8 7.3 4.7 4l-.7.7L7.3 8 4 11.3l.7.7L8 8.7l3.3 3.3.7-.7L8.7 8 12 4.7Z",
      fill: "currentColor"
    })
  });
};
CloseIcon.defaultProps = {
  xmlns: "http://www.w3.org/2000/svg",
  width: "16",
  height: "16"
};
var CreateIcon = function CreateIcon(props) {
  return jsxRuntime.jsx("svg", {
    ...props,
    children: jsxRuntime.jsx("path", {
      fillRule: "evenodd",
      d: "M9 13V9h4a1 1 0 0 0 0-2H9V3a1 1 0 1 0-2 0v4H3a1 1 0 1 0 0 2h4v4a1 1 0 0 0 2 0Z"
    })
  });
};
CreateIcon.defaultProps = {
  xmlns: "http://www.w3.org/2000/svg",
  width: "16",
  height: "16"
};
var DeleteIcon = function DeleteIcon(props) {
  return jsxRuntime.jsx("svg", {
    ...props,
    children: jsxRuntime.jsx("path", {
      fillRule: "evenodd",
      d: "M12 6v7c0 1.1-.4 1.55-1.5 1.55h-5C4.4 14.55 4 14.1 4 13V6h8Zm-1.5 1.5h-5v4.3c0 .66.5 1.2 1.111 1.2H9.39c.611 0 1.111-.54 1.111-1.2V7.5ZM13 3h-2l-1-1H6L5 3H3v1.5h10V3Z"
    })
  });
};
DeleteIcon.defaultProps = {
  xmlns: "http://www.w3.org/2000/svg",
  width: "16",
  height: "16"
};
var DragIcon = function DragIcon(props) {
  return jsxRuntime.jsxs("svg", {
    ...props,
    children: [jsxRuntime.jsx("path", {
      fill: "#fff",
      style: {
        mixBlendMode: "multiply"
      },
      d: "M0 0h16v16H0z"
    }), jsxRuntime.jsx("path", {
      fill: "#fff",
      style: {
        mixBlendMode: "multiply"
      },
      d: "M0 0h16v16H0z"
    }), jsxRuntime.jsx("path", {
      d: "M7 3H5v2h2V3zm4 0H9v2h2V3zM7 7H5v2h2V7zm4 0H9v2h2V7zm-4 4H5v2h2v-2zm4 0H9v2h2v-2z",
      fill: "#161616"
    })]
  });
};
DragIcon.defaultProps = {
  width: "16",
  height: "16",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
var ExternalLinkIcon = function ExternalLinkIcon(props) {
  return jsxRuntime.jsx("svg", {
    ...props,
    children: jsxRuntime.jsx("path", {
      fillRule: "evenodd",
      clipRule: "evenodd",
      d: "M12.637 12.637v-4.72h1.362v4.721c0 .36-.137.676-.411.95-.275.275-.591.412-.95.412H3.362c-.38 0-.703-.132-.967-.396A1.315 1.315 0 0 1 2 12.638V3.362c0-.38.132-.703.396-.967S2.982 2 3.363 2h4.553v1.363H3.363v9.274h9.274ZM14 2H9.28l-.001 1.362h2.408L5.065 9.984l.95.95 6.622-6.622v2.409H14V2Z",
      fill: "currentcolor"
    })
  });
};
ExternalLinkIcon.defaultProps = {
  width: "16",
  height: "16",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
var FeelIcon$1 = function FeelIcon(props) {
  return jsxRuntime.jsx("svg", {
    ...props,
    children: jsxRuntime.jsx("path", {
      d: "M3.617 11.99c-.137.684-.392 1.19-.765 1.518-.362.328-.882.492-1.558.492H0l.309-1.579h1.264l1.515-7.64h-.912l.309-1.579h.911l.236-1.191c.137-.685.387-1.192.75-1.52C4.753.164 5.277 0 5.953 0h1.294L6.94 1.579H5.675l-.323 1.623h1.264l-.309 1.579H5.043l-1.426 7.208ZM5.605 11.021l3.029-4.155L7.28 3.202h2.073l.706 2.547h.176l1.691-2.547H14l-3.014 4.051 1.338 3.768H10.25l-.706-2.606H9.37L7.678 11.02H5.605Z",
      fill: "currentcolor"
    })
  });
};
FeelIcon$1.defaultProps = {
  width: "14",
  height: "14",
  fill: "none",
  xmlns: "http://www.w3.org/2000/svg"
};
var LaunchIcon = function LaunchIcon(props) {
  return jsxRuntime.jsxs("svg", {
    ...props,
    children: [jsxRuntime.jsx("path", {
      d: "M26 28H6a2.003 2.003 0 0 1-2-2V6a2.003 2.003 0 0 1 2-2h10v2H6v20h20V16h2v10a2.003 2.003 0 0 1-2 2Z"
    }), jsxRuntime.jsx("path", {
      d: "M20 2v2h6.586L18 12.586 19.414 14 28 5.414V12h2V2H20z"
    })]
  });
};
LaunchIcon.defaultProps = {
  xmlns: "http://www.w3.org/2000/svg",
  viewBox: "0 0 32 32"
};
var OpenPopupIcon = function OpenPopupIcon(props) {
  return jsxRuntime.jsxs("svg", {
    ...props,
    children: [jsxRuntime.jsx("path", {
      fill: "currentColor",
      d: "M28 4H10a2.006 2.006 0 0 0-2 2v14a2.006 2.006 0 0 0 2 2h18a2.006 2.006 0 0 0 2-2V6a2.006 2.006 0 0 0-2-2Zm0 16H10V6h18Z"
    }), jsxRuntime.jsx("path", {
      fill: "currentColor",
      d: "M18 26H4V16h2v-2H4a2.006 2.006 0 0 0-2 2v10a2.006 2.006 0 0 0 2 2h14a2.006 2.006 0 0 0 2-2v-2h-2Z"
    })]
  });
};
OpenPopupIcon.defaultProps = {
  xmlns: "http://www.w3.org/2000/svg",
  width: "16",
  height: "16",
  viewBox: "0 0 32 32"
};

function Header(props) {
  const {
    element,
    headerProvider
  } = props;
  const {
    getElementIcon,
    getDocumentationRef,
    getElementLabel,
    getTypeLabel
  } = headerProvider;
  const label = getElementLabel(element);
  const type = getTypeLabel(element);
  const documentationRef = getDocumentationRef && getDocumentationRef(element);
  const ElementIcon = getElementIcon(element);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-header",
    children: [jsxRuntime.jsx("div", {
      class: "bio-properties-panel-header-icon",
      children: ElementIcon && jsxRuntime.jsx(ElementIcon, {
        width: "32",
        height: "32",
        viewBox: "0 0 32 32"
      })
    }), jsxRuntime.jsxs("div", {
      class: "bio-properties-panel-header-labels",
      children: [jsxRuntime.jsx("div", {
        title: type,
        class: "bio-properties-panel-header-type",
        children: type
      }), label ? jsxRuntime.jsx("div", {
        title: label,
        class: "bio-properties-panel-header-label",
        children: label
      }) : null]
    }), jsxRuntime.jsx("div", {
      class: "bio-properties-panel-header-actions",
      children: documentationRef ? jsxRuntime.jsx("a", {
        rel: "noreferrer",
        class: "bio-properties-panel-header-link",
        href: documentationRef,
        title: "Open documentation",
        target: "_blank",
        children: jsxRuntime.jsx(ExternalLinkIcon, {})
      }) : null
    })]
  });
}

const DescriptionContext = preact.createContext({
  description: {},
  getDescriptionForId: () => {}
});

const ErrorsContext = preact.createContext({
  errors: {}
});

/**
 * @typedef {Function} <propertiesPanel.showEntry> callback
 *
 * @example
 *
 * useEvent('propertiesPanel.showEntry', ({ focus = false, ...rest }) => {
 *   // ...
 * });
 *
 * @param {Object} context
 * @param {boolean} [context.focus]
 *
 * @returns void
 */

const EventContext = preact.createContext({
  eventBus: null
});

/**
 * @typedef { {
 *   parserDialect?: import('@bpmn-io/feel-editor').ParserDialect,
 *   builtins?: import('@bpmn-io/feel-editor').Variable[],
 *   dialect?: import('@bpmn-io/feel-editor').Dialect
 * } } FeelLanguageContextType
 */

/**
 * @type {import('preact').Context<FeelLanguageContextType>}
 */
const FeelLanguageContext = preact.createContext({});

const LayoutContext = preact.createContext({
  layout: {},
  setLayout: () => {},
  getLayoutForKey: () => {},
  setLayoutForKey: () => {}
});

const TooltipContext = preact.createContext({
  tooltip: {},
  getTooltipForId: () => {}
});

/**
 * Accesses the global TooltipContext and returns a tooltip for a given id and element.
 *
 * @example
 * ```jsx
 * function TextField(props) {
 *   const tooltip = useTooltipContext('input1', element);
 * }
 * ```
 *
 * @param {string} id
 * @param {object} element
 *
 * @returns {string}
 */
function useTooltipContext(id, element) {
  const {
    getTooltipForId
  } = hooks.useContext(TooltipContext);
  return getTooltipForId(id, element);
}

function TooltipWrapper(props) {
  const {
    forId,
    element
  } = props;
  const contextDescription = useTooltipContext(forId, element);
  const value = props.value || contextDescription;
  if (!value) {
    return props.children;
  }
  return jsxRuntime.jsx(Tooltip, {
    ...props,
    value: value,
    forId: `bio-properties-panel-${forId}`
  });
}
function Tooltip(props) {
  const {
    forId,
    value,
    parent,
    direction = 'right',
    position
  } = props;
  const [visible, setVisible] = hooks.useState(false);

  // Tooltip will be shown after SHOW_DELAY ms from hovering over the source element.
  const SHOW_DELAY = 200;
  let timeout = null;
  const wrapperRef = hooks.useRef(null);
  const tooltipRef = hooks.useRef(null);
  const show = (_, delay) => {
    if (visible) return;
    if (delay) {
      timeout = setTimeout(() => {
        setVisible(true);
      }, SHOW_DELAY);
    } else {
      setVisible(true);
    }
  };
  const hide = () => {
    clearTimeout(timeout);
    setVisible(false);
  };
  const handleMouseLeave = ({
    relatedTarget
  }) => {
    // Don't hide the tooltip when moving mouse between the wrapper and the tooltip.
    if (relatedTarget === wrapperRef.current || relatedTarget === tooltipRef.current || relatedTarget?.parentElement === tooltipRef.current) {
      return;
    }
    hide();
  };
  const handleFocusOut = e => {
    const {
      target
    } = e;

    // Don't hide the tooltip if the wrapper or the tooltip itself is clicked.
    const isHovered = target.matches(':hover') || tooltipRef.current?.matches(':hover');
    if (target === wrapperRef.current && isHovered) {
      e.stopPropagation();
      return;
    }
    hide();
  };
  const hideTooltipViaEscape = e => {
    e.code === 'Escape' && hide();
  };
  const renderTooltip = () => {
    return jsxRuntime.jsxs("div", {
      class: `bio-properties-panel-tooltip ${direction}`,
      role: "tooltip",
      id: "bio-properties-panel-tooltip",
      "aria-labelledby": forId,
      style: position || getTooltipPosition(wrapperRef.current),
      ref: tooltipRef,
      onClick: e => e.stopPropagation(),
      onMouseLeave: handleMouseLeave,
      children: [jsxRuntime.jsx("div", {
        class: "bio-properties-panel-tooltip-content",
        children: value
      }), jsxRuntime.jsx("div", {
        class: "bio-properties-panel-tooltip-arrow"
      })]
    });
  };
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-tooltip-wrapper",
    tabIndex: "0",
    ref: wrapperRef,
    onMouseEnter: e => show(e, true),
    onMouseLeave: handleMouseLeave,
    onFocus: show,
    onBlur: handleFocusOut,
    onKeyDown: hideTooltipViaEscape,
    children: [props.children, visible ? parent ? compat.createPortal(renderTooltip(), parent.current) : renderTooltip() : null]
  });
}

// helper

function getTooltipPosition(refElement) {
  const refPosition = refElement.getBoundingClientRect();
  const right = `calc(100% - ${refPosition.x}px)`;
  const top = `${refPosition.top - 10}px`;
  return `right: ${right}; top: ${top};`;
}

/**
 * Accesses the global DescriptionContext and returns a description for a given id and element.
 *
 * @example
 * ```jsx
 * function TextField(props) {
 *   const description = useDescriptionContext('input1', element);
 * }
 * ```
 *
 * @param {string} id
 * @param {object} element
 *
 * @returns {string}
 */
function useDescriptionContext(id, element) {
  const {
    getDescriptionForId
  } = hooks.useContext(DescriptionContext);
  return getDescriptionForId(id, element);
}

function useDebounce(callback, debounceFn) {
  const debouncedCallback = hooks.useCallback(debounceFn(callback), [callback, debounceFn]);

  // make sure previous call is not stalled
  hooks.useEffect(() => {
    return () => {
      debouncedCallback.cancel?.();
    };
  }, [debouncedCallback]);
  return debouncedCallback;
}

function useError(id) {
  const {
    errors
  } = hooks.useContext(ErrorsContext);
  return errors[id];
}
function useErrors() {
  const {
    errors
  } = hooks.useContext(ErrorsContext);
  return errors;
}

/**
 * Subscribe to an event immediately. Update subscription after inputs changed.
 *
 * @param {string} event
 * @param {Function} callback
 */
function useEvent(event, callback, eventBus) {
  const eventContext = hooks.useContext(EventContext);
  if (!eventBus) {
    ({
      eventBus
    } = eventContext);
  }
  const didMount = hooks.useRef(false);

  // (1) subscribe immediately
  if (eventBus && !didMount.current) {
    eventBus.on(event, callback);
  }

  // (2) update subscription after inputs changed
  hooks.useEffect(() => {
    if (eventBus && didMount.current) {
      eventBus.on(event, callback);
    }
    didMount.current = true;
    return () => {
      if (eventBus) {
        eventBus.off(event, callback);
      }
    };
  }, [callback, event, eventBus]);
}

const KEY_LENGTH = 6;

/**
 * Create a persistent key factory for plain objects without id.
 *
 * @example
 * ```jsx
 * function List({ objects }) {
 *   const getKey = useKeyFactory();
 *   return (<ol>{
 *     objects.map(obj => {
 *       const key = getKey(obj);
 *       return <li key={key}>obj.name</li>
 *     })
 *   }</ol>);
 * }
 * ```
 *
 * @param {any[]} dependencies
 * @returns {(element: object) => string}
 */
function useKeyFactory(dependencies = []) {
  const map = hooks.useMemo(() => new Map(), dependencies);
  const getKey = el => {
    let key = map.get(el);
    if (!key) {
      key = Math.random().toString().slice(-KEY_LENGTH);
      map.set(el, key);
    }
    return key;
  };
  return getKey;
}

/**
 * Creates a state that persists in the global LayoutContext.
 *
 * @example
 * ```jsx
 * function Group(props) {
 *   const [ open, setOpen ] = useLayoutState([ 'groups', 'foo', 'open' ], false);
 * }
 * ```
 *
 * @param {(string|number)[]} path
 * @param {any} [defaultValue]
 *
 * @returns {[ any, Function ]}
 */
function useLayoutState(path, defaultValue) {
  const {
    getLayoutForKey,
    setLayoutForKey
  } = hooks.useContext(LayoutContext);
  const layoutForKey = getLayoutForKey(path, defaultValue);
  const setState = hooks.useCallback(newValue => {
    setLayoutForKey(path, newValue);
  }, [setLayoutForKey]);
  return [layoutForKey, setState];
}

/**
 * @pinussilvestrus: we need to introduce our own hook to persist the previous
 * state on updates.
 *
 * cf. https://reactjs.org/docs/hooks-faq.html#how-to-get-the-previous-props-or-state
 */

function usePrevious(value) {
  const ref = hooks.useRef();
  hooks.useEffect(() => {
    ref.current = value;
  });
  return ref.current;
}

/**
 * Subscribe to `propertiesPanel.showEntry`.
 *
 * @param {string} id
 *
 * @returns {import('preact').Ref}
 */
function useShowEntryEvent(id) {
  const {
    onShow
  } = hooks.useContext(LayoutContext);
  const ref = hooks.useRef();
  const focus = hooks.useRef(false);
  const onShowEntry = hooks.useCallback(event => {
    if (event.id === id) {
      onShow();
      if (!focus.current) {
        focus.current = true;
      }
    }
  }, [id]);
  hooks.useEffect(() => {
    if (focus.current && ref.current) {
      if (minDash.isFunction(ref.current.focus)) {
        ref.current.focus();
      }
      if (minDash.isFunction(ref.current.select)) {
        ref.current.select();
      }
      focus.current = false;
    }
  });
  useEvent('propertiesPanel.showEntry', onShowEntry);
  return ref;
}

/**
 * @callback setSticky
 * @param {boolean} value
 */

/**
 * Use IntersectionObserver to identify when DOM element is in sticky mode.
 * If sticky is observered setSticky(true) will be called.
 * If sticky mode is left, setSticky(false) will be called.
 *
 *
 * @param {Object} ref
 * @param {string} scrollContainerSelector
 * @param {setSticky} setSticky
 */
function useStickyIntersectionObserver(ref, scrollContainerSelector, setSticky) {
  const [scrollContainer, setScrollContainer] = hooks.useState(minDom.query(scrollContainerSelector));
  const updateScrollContainer = hooks.useCallback(() => {
    const newScrollContainer = minDom.query(scrollContainerSelector);
    if (newScrollContainer !== scrollContainer) {
      setScrollContainer(newScrollContainer);
    }
  }, [scrollContainerSelector, scrollContainer]);
  hooks.useEffect(() => {
    updateScrollContainer();
  }, [updateScrollContainer]);
  useEvent('propertiesPanel.attach', updateScrollContainer);
  useEvent('propertiesPanel.detach', updateScrollContainer);
  hooks.useEffect(() => {
    const Observer = IntersectionObserver;

    // return early if IntersectionObserver is not available
    if (!Observer) {
      return;
    }

    // TODO(@barmac): test this
    if (!ref.current || !scrollContainer) {
      return;
    }
    const observer = new Observer(entries => {
      // scroll container is unmounted, do not update sticky state
      if (scrollContainer.scrollHeight === 0) {
        return;
      }
      entries.forEach(entry => {
        if (entry.intersectionRatio < 1) {
          setSticky(true);
        } else if (entry.intersectionRatio === 1) {
          setSticky(false);
        }
      });
    }, {
      root: scrollContainer,
      rootMargin: '0px 0px 999999% 0px',
      // Use bottom margin to avoid stickyness when scrolling out to bottom
      threshold: [1]
    });
    observer.observe(ref.current);

    // Unobserve if unmounted
    return () => {
      observer.unobserve(ref.current);
    };
  }, [ref.current, scrollContainer, setSticky]);
}

/**
 * Creates a static function reference with changing body.
 * This is necessary when external libraries require a callback function
 * that has references to state variables.
 *
 * Usage:
 * const callback = useStaticCallback((val) => {val === currentState});
 *
 * The `callback` reference is static and can be safely used in external
 * libraries or as a prop that does not cause rerendering of children.
 *
 * @param {Function} callback function with changing reference
 * @returns {Function} static function reference
 */
function useStaticCallback(callback) {
  const callbackRef = hooks.useRef(callback);
  callbackRef.current = callback;
  return hooks.useCallback((...args) => callbackRef.current(...args), []);
}

function useElementVisible(element) {
  const [visible, setVisible] = hooks.useState(!!element && !!element.clientHeight);
  hooks.useLayoutEffect(() => {
    if (!element) return;
    const resizeObserver = new ResizeObserver(([entry]) => {
      requestAnimationFrame(() => {
        const newVisible = !!entry.contentRect.height;
        if (newVisible !== visible) {
          setVisible(newVisible);
        }
      });
    });
    resizeObserver.observe(element);
    return () => resizeObserver.disconnect();
  }, [element, visible]);
  return visible;
}

function Group(props) {
  const {
    element,
    entries = [],
    id,
    label,
    shouldOpen = false
  } = props;
  const groupRef = hooks.useRef(null);
  const [open, setOpen] = useLayoutState(['groups', id, 'open'], shouldOpen);
  const onShow = hooks.useCallback(() => setOpen(true), [setOpen]);
  const toggleOpen = () => setOpen(!open);
  const [edited, setEdited] = hooks.useState(false);
  const [sticky, setSticky] = hooks.useState(false);

  // set edited state depending on all entries
  hooks.useEffect(() => {
    // TODO(@barmac): replace with CSS when `:has()` is supported in all major browsers, or rewrite as in https://github.com/camunda/camunda-modeler/issues/3815#issuecomment-1733038161
    const scheduled = requestAnimationFrame(() => {
      const hasOneEditedEntry = entries.find(entry => {
        const {
          id,
          isEdited
        } = entry;
        const entryNode = minDom.query(`[data-entry-id="${id}"]`);
        if (!minDash.isFunction(isEdited) || !entryNode) {
          return false;
        }
        const inputNode = minDom.query('.bio-properties-panel-input', entryNode);
        return isEdited(inputNode);
      });
      setEdited(hasOneEditedEntry);
    });
    return () => cancelAnimationFrame(scheduled);
  }, [entries, setEdited]);

  // set error state depending on all entries
  const allErrors = useErrors();
  const hasErrors = entries.some(entry => allErrors[entry.id]);

  // set css class when group is sticky to top
  useStickyIntersectionObserver(groupRef, 'div.bio-properties-panel-scroll-container', setSticky);
  const propertiesPanelContext = {
    ...hooks.useContext(LayoutContext),
    onShow
  };
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-group",
    "data-group-id": 'group-' + id,
    ref: groupRef,
    children: [jsxRuntime.jsxs("div", {
      class: classnames('bio-properties-panel-group-header', edited ? '' : 'empty', open ? 'open' : '', sticky && open ? 'sticky' : ''),
      onClick: toggleOpen,
      children: [jsxRuntime.jsx("div", {
        title: props.tooltip ? null : label,
        "data-title": label,
        class: "bio-properties-panel-group-header-title",
        children: jsxRuntime.jsx(TooltipWrapper, {
          value: props.tooltip,
          forId: 'group-' + id,
          element: element,
          parent: groupRef,
          children: label
        })
      }), jsxRuntime.jsxs("div", {
        class: "bio-properties-panel-group-header-buttons",
        children: [jsxRuntime.jsx(DataMarker, {
          edited: edited,
          hasErrors: hasErrors
        }), jsxRuntime.jsx("button", {
          type: "button",
          title: "Toggle section",
          class: "bio-properties-panel-group-header-button bio-properties-panel-arrow",
          children: jsxRuntime.jsx(ArrowIcon, {
            class: open ? 'bio-properties-panel-arrow-down' : 'bio-properties-panel-arrow-right'
          })
        })]
      })]
    }), jsxRuntime.jsx("div", {
      class: classnames('bio-properties-panel-group-entries', open ? 'open' : ''),
      children: jsxRuntime.jsx(LayoutContext.Provider, {
        value: propertiesPanelContext,
        children: entries.map(entry => {
          const {
            component: Component,
            id
          } = entry;
          return preact.createElement(Component, {
            ...entry,
            element: element,
            key: id
          });
        })
      })
    })]
  });
}
function DataMarker(props) {
  const {
    edited,
    hasErrors
  } = props;
  if (hasErrors) {
    return jsxRuntime.jsx("div", {
      title: "Section contains an error",
      class: "bio-properties-panel-dot bio-properties-panel-dot--error"
    });
  }
  if (edited) {
    return jsxRuntime.jsx("div", {
      title: "Section contains edits",
      class: "bio-properties-panel-dot"
    });
  }
  return null;
}

/**
 * @typedef { {
 *  text: (element: object) => string,
 *  icon?: (element: Object) => import('preact').Component
 * } } PlaceholderDefinition
 *
 * @param { PlaceholderDefinition } props
 */
function Placeholder(props) {
  const {
    text,
    icon: Icon
  } = props;
  return jsxRuntime.jsx("div", {
    class: "bio-properties-panel open",
    children: jsxRuntime.jsxs("section", {
      class: "bio-properties-panel-placeholder",
      children: [Icon && jsxRuntime.jsx(Icon, {
        class: "bio-properties-panel-placeholder-icon"
      }), jsxRuntime.jsx("p", {
        class: "bio-properties-panel-placeholder-text",
        children: text
      })]
    })
  });
}

const DEFAULT_LAYOUT = {};
const DEFAULT_DESCRIPTION = {};
const DEFAULT_TOOLTIP = {};

/**
 * @typedef { {
 *    component: import('preact').Component,
 *    id: String,
 *    isEdited?: Function
 * } } EntryDefinition
 *
 * @typedef { {
 *    autoFocusEntry: String,
 *    autoOpen?: Boolean,
 *    entries: Array<EntryDefinition>,
 *    id: String,
 *    label: String,
 *    remove: (event: MouseEvent) => void
 * } } ListItemDefinition
 *
 * @typedef { {
 *    add: (event: MouseEvent) => void,
 *    component: import('preact').Component,
 *    element: Object,
 *    id: String,
 *    items: Array<ListItemDefinition>,
 *    label: String,
 *    shouldOpen?: Boolean
 * } } ListGroupDefinition
 *
 * @typedef { {
 *    component?: import('preact').Component,
 *    entries: Array<EntryDefinition>,
 *    id: String,
 *    label: String,
 *    shouldOpen?: Boolean
 * } } GroupDefinition
 *
 *  @typedef { {
 *    [id: String]: GetDescriptionFunction
 * } } DescriptionConfig
 *
 *  @typedef { {
 *    [id: String]: GetTooltipFunction
 * } } TooltipConfig
 *
 * @callback { {
 * @param {string} id
 * @param {Object} element
 * @returns {string}
 * } } GetDescriptionFunction
 *
 * @callback { {
 * @param {string} id
 * @param {Object} element
 * @returns {string}
 * } } GetTooltipFunction
 *
 * @typedef { {
 *  getEmpty: (element: object) => import('./components/Placeholder').PlaceholderDefinition,
 *  getMultiple: (element: Object) => import('./components/Placeholder').PlaceholderDefinition
 * } } PlaceholderProvider
 *
 */

/**
 * A basic properties panel component. Describes *how* content will be rendered, accepts
 * data from implementor to describe *what* will be rendered.
 *
 * @param {Object} props
 * @param {Object|Array} props.element
 * @param {import('./components/Header').HeaderProvider} props.headerProvider
 * @param {PlaceholderProvider} [props.placeholderProvider]
 * @param {Array<GroupDefinition|ListGroupDefinition>} props.groups
 * @param {Object} [props.layoutConfig]
 * @param {Function} [props.layoutChanged]
 * @param {DescriptionConfig} [props.descriptionConfig]
 * @param {Function} [props.descriptionLoaded]
 * @param {TooltipConfig} [props.tooltipConfig]
 * @param {Function} [props.tooltipLoaded]
 * @param {HTMLElement} [props.feelPopupContainer]
 * @param {Function} [props.getFeelPopupLinks]
 * @param {Object} [props.eventBus]
 */
function PropertiesPanel(props) {
  const {
    element,
    headerProvider,
    placeholderProvider,
    groups,
    layoutConfig,
    layoutChanged,
    descriptionConfig,
    descriptionLoaded,
    tooltipConfig,
    tooltipLoaded,
    eventBus
  } = props;

  // set-up layout context
  const [layout, setLayout] = hooks.useState(createLayout(layoutConfig));

  // react to external changes in the layout config
  useUpdateLayoutEffect(() => {
    const newLayout = createLayout(layoutConfig);
    setLayout(newLayout);
  }, [layoutConfig]);
  hooks.useEffect(() => {
    if (typeof layoutChanged === 'function') {
      layoutChanged(layout);
    }
  }, [layout, layoutChanged]);
  const getLayoutForKey = (key, defaultValue) => {
    return minDash.get(layout, key, defaultValue);
  };
  const setLayoutForKey = (key, config) => {
    const newLayout = minDash.assign({}, layout);
    minDash.set(newLayout, key, config);
    setLayout(newLayout);
  };
  const layoutContext = {
    layout,
    setLayout,
    getLayoutForKey,
    setLayoutForKey
  };

  // set-up description context
  const description = hooks.useMemo(() => createDescriptionContext(descriptionConfig), [descriptionConfig]);
  hooks.useEffect(() => {
    if (typeof descriptionLoaded === 'function') {
      descriptionLoaded(description);
    }
  }, [description, descriptionLoaded]);
  const getDescriptionForId = (id, element) => {
    return description[id] && description[id](element);
  };
  const descriptionContext = {
    description,
    getDescriptionForId
  };

  // set-up tooltip context
  const tooltip = hooks.useMemo(() => createTooltipContext(tooltipConfig), [tooltipConfig]);
  hooks.useEffect(() => {
    if (typeof tooltipLoaded === 'function') {
      tooltipLoaded(tooltip);
    }
  }, [tooltip, tooltipLoaded]);
  const getTooltipForId = (id, element) => {
    return tooltip[id] && tooltip[id](element);
  };
  const tooltipContext = {
    tooltip,
    getTooltipForId
  };
  const [errors, setErrors] = hooks.useState({});
  const onSetErrors = ({
    errors
  }) => setErrors(errors);
  useEvent('propertiesPanel.setErrors', onSetErrors, eventBus);
  const errorsContext = {
    errors
  };
  const eventContext = {
    eventBus
  };
  const propertiesPanelContext = {
    element
  };

  // empty state
  if (placeholderProvider && !element) {
    return jsxRuntime.jsx(Placeholder, {
      ...placeholderProvider.getEmpty()
    });
  }

  // multiple state
  if (placeholderProvider && minDash.isArray(element)) {
    return jsxRuntime.jsx(Placeholder, {
      ...placeholderProvider.getMultiple()
    });
  }
  return jsxRuntime.jsx(LayoutContext.Provider, {
    value: propertiesPanelContext,
    children: jsxRuntime.jsx(ErrorsContext.Provider, {
      value: errorsContext,
      children: jsxRuntime.jsx(DescriptionContext.Provider, {
        value: descriptionContext,
        children: jsxRuntime.jsx(TooltipContext.Provider, {
          value: tooltipContext,
          children: jsxRuntime.jsx(LayoutContext.Provider, {
            value: layoutContext,
            children: jsxRuntime.jsx(EventContext.Provider, {
              value: eventContext,
              children: jsxRuntime.jsxs("div", {
                class: "bio-properties-panel",
                children: [jsxRuntime.jsx(Header, {
                  element: element,
                  headerProvider: headerProvider
                }), jsxRuntime.jsx("div", {
                  class: "bio-properties-panel-scroll-container",
                  children: groups.map(group => {
                    const {
                      component: Component = Group,
                      id
                    } = group;
                    return preact.createElement(Component, {
                      ...group,
                      key: id,
                      element: element
                    });
                  })
                })]
              })
            })
          })
        })
      })
    })
  });
}

// helpers //////////////////

function createLayout(overrides = {}, defaults = DEFAULT_LAYOUT) {
  return {
    ...defaults,
    ...overrides
  };
}
function createDescriptionContext(overrides = {}) {
  return {
    ...DEFAULT_DESCRIPTION,
    ...overrides
  };
}
function createTooltipContext(overrides = {}) {
  return {
    ...DEFAULT_TOOLTIP,
    ...overrides
  };
}

// hooks //////////////////

/**
 * This hook behaves like useLayoutEffect, but does not trigger on the first render.
 *
 * @param {Function} effect
 * @param {Array} deps
 */
function useUpdateLayoutEffect(effect, deps) {
  const isMounted = hooks.useRef(false);
  hooks.useLayoutEffect(() => {
    if (isMounted.current) {
      return effect();
    } else {
      isMounted.current = true;
    }
  }, deps);
}

function DropdownButton(props) {
  const {
    class: className,
    children,
    menuItems = []
  } = props;
  const dropdownRef = hooks.useRef(null);
  const menuRef = hooks.useRef(null);
  const [open, setOpen] = hooks.useState(false);
  const close = () => setOpen(false);
  function onDropdownToggle(event) {
    if (menuRef.current && menuRef.current.contains(event.target)) {
      return;
    }
    event.stopPropagation();
    setOpen(open => !open);
  }
  function onActionClick(event, action) {
    event.stopPropagation();
    close();
    action();
  }
  useGlobalClick([dropdownRef.current], () => close());
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-dropdown-button', {
      open
    }, className),
    onClick: onDropdownToggle,
    ref: dropdownRef,
    children: [children, jsxRuntime.jsx("div", {
      class: "bio-properties-panel-dropdown-button__menu",
      ref: menuRef,
      children: menuItems.map((item, index) => jsxRuntime.jsx(MenuItem, {
        onClick: onActionClick,
        item: item
      }, index))
    })]
  });
}
function MenuItem({
  item,
  onClick
}) {
  if (item.separator) {
    return jsxRuntime.jsx("div", {
      class: "bio-properties-panel-dropdown-button__menu-item bio-properties-panel-dropdown-button__menu-item--separator"
    });
  }
  if (item.action) {
    return jsxRuntime.jsx("button", {
      type: "button",
      class: "bio-properties-panel-dropdown-button__menu-item bio-properties-panel-dropdown-button__menu-item--actionable",
      onClick: event => onClick(event, item.action),
      children: item.entry
    });
  }
  return jsxRuntime.jsx("div", {
    class: "bio-properties-panel-dropdown-button__menu-item",
    children: item.entry
  });
}

/**
 *
 * @param {Array<null | Element>} ignoredElements
 * @param {Function} callback
 */
function useGlobalClick(ignoredElements, callback) {
  hooks.useEffect(() => {
    /**
     * @param {MouseEvent} event
     */
    function listener(event) {
      if (ignoredElements.some(element => element && element.contains(event.target))) {
        return;
      }
      callback();
    }
    document.addEventListener('click', listener, {
      capture: true
    });
    return () => document.removeEventListener('click', listener, {
      capture: true
    });
  }, [...ignoredElements, callback]);
}

function HeaderButton(props) {
  const {
    children = null,
    class: classname,
    onClick = () => {},
    ...otherProps
  } = props;
  return jsxRuntime.jsx("button", {
    type: "button",
    ...otherProps,
    onClick: onClick,
    class: classnames('bio-properties-panel-group-header-button', classname),
    children: children
  });
}

/**
 * @typedef { {
 *   [key: string]: string;
 * } } TranslateReplacements
 */

/**
 * A simple translation stub to be used for multi-language support.
 * Can be easily replaced with a more sophisticated solution.
 *
 * @param {string} template to interpolate
 * @param {TranslateReplacements} [replacements] a map with substitutes
 *
 * @return {string} the translated string
 */
function translateFallback(template, replacements) {
  replacements = replacements || {};
  return template.replace(/{([^}]+)}/g, function (_, key) {
    return replacements[key] || '{' + key + '}';
  });
}

function CollapsibleEntry(props) {
  const {
    element,
    entries = [],
    id,
    label,
    open: shouldOpen,
    remove,
    translate = translateFallback
  } = props;
  const [open, setOpen] = hooks.useState(shouldOpen);
  const toggleOpen = () => setOpen(!open);
  const {
    onShow
  } = hooks.useContext(LayoutContext);
  const propertiesPanelContext = {
    ...hooks.useContext(LayoutContext),
    onShow: hooks.useCallback(() => {
      setOpen(true);
      if (minDash.isFunction(onShow)) {
        onShow();
      }
    }, [onShow, setOpen])
  };
  const placeholderLabel = translate('<empty>');
  return jsxRuntime.jsxs("div", {
    "data-entry-id": id,
    class: classnames('bio-properties-panel-collapsible-entry', open ? 'open' : ''),
    children: [jsxRuntime.jsxs("div", {
      class: "bio-properties-panel-collapsible-entry-header",
      onClick: toggleOpen,
      children: [jsxRuntime.jsx("div", {
        title: label || placeholderLabel,
        class: classnames('bio-properties-panel-collapsible-entry-header-title', !label && 'empty'),
        children: label || placeholderLabel
      }), jsxRuntime.jsx("button", {
        type: "button",
        title: translate('Toggle list item'),
        class: "bio-properties-panel-arrow  bio-properties-panel-collapsible-entry-arrow",
        children: jsxRuntime.jsx(ArrowIcon, {
          class: open ? 'bio-properties-panel-arrow-down' : 'bio-properties-panel-arrow-right'
        })
      }), remove ? jsxRuntime.jsx("button", {
        type: "button",
        title: translate('Delete item'),
        class: "bio-properties-panel-remove-entry",
        onClick: remove,
        children: jsxRuntime.jsx(DeleteIcon, {})
      }) : null]
    }), jsxRuntime.jsx("div", {
      class: classnames('bio-properties-panel-collapsible-entry-entries', open ? 'open' : ''),
      children: jsxRuntime.jsx(LayoutContext.Provider, {
        value: propertiesPanelContext,
        children: entries.map(entry => {
          const {
            component: Component,
            id
          } = entry;
          return preact.createElement(Component, {
            ...entry,
            element: element,
            key: id
          });
        })
      })
    })]
  });
}

function ListItem(props) {
  const {
    autoFocusEntry,
    autoOpen,
    translate = translateFallback
  } = props;

  // focus specified entry on auto open
  hooks.useEffect(() => {
    if (autoOpen && autoFocusEntry) {
      const entry = minDom.query(`[data-entry-id="${autoFocusEntry}"]`);
      const focusableInput = minDom.query('.bio-properties-panel-input', entry);
      if (focusableInput) {
        if (minDash.isFunction(focusableInput.select)) {
          focusableInput.select();
        } else if (minDash.isFunction(focusableInput.focus)) {
          focusableInput.focus();
        }
        focusableInput.scrollIntoView();
      }
    }
  }, [autoOpen, autoFocusEntry]);
  return jsxRuntime.jsx("div", {
    class: "bio-properties-panel-list-item",
    children: jsxRuntime.jsx(CollapsibleEntry, {
      ...props,
      open: autoOpen,
      translate: translate
    })
  });
}

const noop$6 = () => {};

/**
 * @param {import('../PropertiesPanel').ListGroupDefinition} props
 */
function ListGroup(props) {
  const {
    add,
    element,
    id,
    items,
    label,
    shouldOpen = false,
    translate = translateFallback
  } = props;
  hooks.useEffect(() => {
    if (props.shouldSort != undefined) {
      console.warn('the property \'shouldSort\' is no longer supported');
    }
  }, [props.shouldSort]);
  const groupRef = hooks.useRef(null);
  const [open, setOpen] = useLayoutState(['groups', id, 'open'], shouldOpen);
  const [sticky, setSticky] = hooks.useState(false);
  const onShow = hooks.useCallback(() => setOpen(true), [setOpen]);
  const [localItems, setLocalItems] = hooks.useState([]);

  // Flag to mark that add button was clicked in the last render cycle
  const [addTriggered, setAddTriggered] = hooks.useState(false);
  const prevElement = usePrevious(element);
  const toggleOpen = hooks.useCallback(() => setOpen(!open), [open]);
  const openItemIds = element === prevElement && open && addTriggered ? getNewItemIds(items, localItems) : [];

  // reset local state after items changed
  hooks.useEffect(() => {
    setLocalItems(items);
    setAddTriggered(false);
  }, [items]);

  // set css class when group is sticky to top
  useStickyIntersectionObserver(groupRef, 'div.bio-properties-panel-scroll-container', setSticky);
  const hasItems = !!items.length;
  const propertiesPanelContext = {
    ...hooks.useContext(LayoutContext),
    onShow
  };
  const handleAddClick = e => {
    setAddTriggered(true);
    setOpen(true);
    add(e);
  };
  const allErrors = useErrors();
  const hasError = items.some(item => {
    if (allErrors[item.id]) {
      return true;
    }
    if (!item.entries) {
      return;
    }

    // also check if the error is nested, e.g. for name-value entries
    return item.entries.some(entry => allErrors[entry.id]);
  });
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-group",
    "data-group-id": 'group-' + id,
    ref: groupRef,
    children: [jsxRuntime.jsxs("div", {
      class: classnames('bio-properties-panel-group-header', hasItems ? '' : 'empty', hasItems && open ? 'open' : '', sticky && open ? 'sticky' : ''),
      onClick: hasItems ? toggleOpen : noop$6,
      children: [jsxRuntime.jsx("div", {
        title: props.tooltip ? null : label,
        "data-title": label,
        class: "bio-properties-panel-group-header-title",
        children: jsxRuntime.jsx(TooltipWrapper, {
          value: props.tooltip,
          forId: 'group-' + id,
          element: element,
          parent: groupRef,
          children: label
        })
      }), jsxRuntime.jsxs("div", {
        class: "bio-properties-panel-group-header-buttons",
        children: [add ? jsxRuntime.jsxs("button", {
          type: "button",
          title: translate('Create new list item'),
          class: "bio-properties-panel-group-header-button bio-properties-panel-add-entry",
          onClick: handleAddClick,
          children: [jsxRuntime.jsx(CreateIcon, {}), !hasItems ? jsxRuntime.jsx("span", {
            class: "bio-properties-panel-add-entry-label",
            children: translate('Create')
          }) : null]
        }) : null, hasItems ? jsxRuntime.jsx("div", {
          title: translate(`List contains {numOfItems} item${items.length != 1 ? 's' : ''}`, {
            numOfItems: items.length
          }),
          class: classnames('bio-properties-panel-list-badge', hasError ? 'bio-properties-panel-list-badge--error' : ''),
          children: items.length
        }) : null, hasItems ? jsxRuntime.jsx("button", {
          type: "button",
          title: translate('Toggle section'),
          class: "bio-properties-panel-group-header-button bio-properties-panel-arrow",
          children: jsxRuntime.jsx(ArrowIcon, {
            class: open ? 'bio-properties-panel-arrow-down' : 'bio-properties-panel-arrow-right'
          })
        }) : null]
      })]
    }), jsxRuntime.jsx("div", {
      class: classnames('bio-properties-panel-list', open && hasItems ? 'open' : ''),
      children: jsxRuntime.jsx(LayoutContext.Provider, {
        value: propertiesPanelContext,
        children: items.map((item, index) => {
          if (!item) {
            return;
          }
          const {
            id
          } = item;

          // if item was added, open it
          // existing items will not be affected as autoOpen
          // is only applied on first render
          const autoOpen = openItemIds.includes(item.id);
          return preact.createElement(ListItem, {
            ...item,
            autoOpen: autoOpen,
            element: element,
            index: index,
            key: id,
            translate: translate
          });
        })
      })
    })]
  });
}
function getNewItemIds(newItems, oldItems) {
  const newIds = newItems.map(item => item.id);
  const oldIds = oldItems.map(item => item.id);
  return newIds.filter(itemId => !oldIds.includes(itemId));
}

function Description(props) {
  const {
    element,
    forId,
    value
  } = props;
  const contextDescription = useDescriptionContext(forId, element);
  const description = value || contextDescription;
  if (description) {
    return jsxRuntime.jsx("div", {
      class: "bio-properties-panel-description",
      children: description
    });
  }
}

function Checkbox(props) {
  const {
    id,
    label,
    onChange,
    disabled,
    value = false,
    onFocus,
    onBlur,
    tooltip
  } = props;
  const [localValue, setLocalValue] = hooks.useState(value);
  const handleChangeCallback = ({
    target
  }) => {
    onChange(target.checked);
  };
  const handleChange = e => {
    handleChangeCallback(e);
    setLocalValue(e.target.value);
  };
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value);
  }, [value]);
  const ref = useShowEntryEvent(id);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-checkbox",
    children: [jsxRuntime.jsx("input", {
      ref: ref,
      id: prefixId$8(id),
      name: id,
      onFocus: onFocus,
      onBlur: onBlur,
      type: "checkbox",
      class: "bio-properties-panel-input",
      onChange: handleChange,
      checked: localValue,
      disabled: disabled
    }), jsxRuntime.jsx("label", {
      for: prefixId$8(id),
      class: "bio-properties-panel-label",
      children: jsxRuntime.jsx(TooltipWrapper, {
        value: tooltip,
        forId: id,
        element: props.element,
        children: label
      })
    })]
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {string|import('preact').Component} props.tooltip
 * @param {boolean} [props.disabled]
 */
function CheckboxEntry(props) {
  const {
    element,
    id,
    description,
    label,
    getValue,
    setValue,
    disabled,
    onFocus,
    onBlur,
    tooltip
  } = props;
  const value = getValue(element);
  const error = useError(id);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-entry bio-properties-panel-checkbox-entry",
    "data-entry-id": id,
    children: [jsxRuntime.jsx(Checkbox, {
      disabled: disabled,
      id: id,
      label: label,
      onChange: setValue,
      onFocus: onFocus,
      onBlur: onBlur,
      value: value,
      tooltip: tooltip,
      element: element
    }, element), error && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-error",
      children: error
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}
function isEdited$8(node) {
  return node && !!node.checked;
}

// helpers /////////////////

function prefixId$8(id) {
  return `bio-properties-panel-${id}`;
}

function OpenPopupButton({
  onClick,
  title = 'Open pop-up editor'
}) {
  return jsxRuntime.jsx("button", {
    type: "button",
    title: title,
    class: 'bio-properties-panel-open-feel-popup',
    onClick: onClick,
    children: jsxRuntime.jsx(OpenPopupIcon, {})
  });
}

const noop$5 = () => {};

/**
 * Buffer `.focus()` calls while the editor is not initialized.
 * Set Focus inside when the editor is ready.
 */
const useBufferedFocus$1 = function (editor, ref) {
  const [buffer, setBuffer] = hooks.useState(undefined);
  ref.current = hooks.useMemo(() => ({
    focus: offset => {
      if (editor) {
        editor.focus(offset);
      } else {
        if (typeof offset === 'undefined') {
          offset = Infinity;
        }
        setBuffer(offset);
      }
    }
  }), [editor]);
  hooks.useEffect(() => {
    if (typeof buffer !== 'undefined' && editor) {
      editor.focus(buffer);
      setBuffer(false);
    }
  }, [editor, buffer]);
};
const TemplatingEditor = compat.forwardRef((props, ref) => {
  const {
    onInput,
    disabled,
    tooltipContainer,
    enableGutters,
    value,
    onLint = noop$5,
    onOpenPopup = noop$5,
    popupOpen,
    contentAttributes = {},
    hostLanguage = null,
    singleLine = false
  } = props;
  const inputRef = hooks.useRef();
  const [editor, setEditor] = hooks.useState();
  const [localValue, setLocalValue] = hooks.useState(value || '');
  useBufferedFocus$1(editor, ref);
  const handleInput = useStaticCallback(newValue => {
    onInput(newValue);
    setLocalValue(newValue);
  });
  hooks.useEffect(() => {
    let editor;
    editor = new feelers.FeelersEditor({
      container: inputRef.current,
      onChange: handleInput,
      value: localValue,
      onLint,
      contentAttributes,
      tooltipContainer,
      enableGutters,
      hostLanguage,
      singleLine,
      lineWrap: true
    });
    setEditor(editor);
    return () => {
      onLint([]);
      inputRef.current.innerHTML = '';
      setEditor(null);
    };
  }, []);
  hooks.useEffect(() => {
    if (!editor) {
      return;
    }
    if (value === localValue) {
      return;
    }
    editor.setValue(value);
    setLocalValue(value);
  }, [value]);
  const handleClick = () => {
    ref.current.focus();
  };
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-feelers-editor-container', popupOpen ? 'popupOpen' : null),
    children: [popupOpen && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-feelers-editor__popup-placeholder",
      children: "Opened in popup"
    }), jsxRuntime.jsx("div", {
      name: props.name,
      class: classnames('bio-properties-panel-feelers-editor bio-properties-panel-input', localValue ? 'edited' : null, disabled ? 'disabled' : null),
      ref: inputRef,
      onClick: handleClick
    }), !disabled && jsxRuntime.jsx(OpenPopupButton, {
      onClick: () => onOpenPopup('feelers')
    })]
  });
});

const noop$4 = () => {};

/**
 * Buffer `.focus()` calls while the editor is not initialized.
 * Set Focus inside when the editor is ready.
 */
const useBufferedFocus = function (editor, ref) {
  const [buffer, setBuffer] = hooks.useState(undefined);
  ref.current = hooks.useMemo(() => ({
    focus: offset => {
      if (editor) {
        editor.focus(offset);
      } else {
        if (typeof offset === 'undefined') {
          offset = Infinity;
        }
        setBuffer(offset);
      }
    }
  }), [editor]);
  hooks.useEffect(() => {
    if (typeof buffer !== 'undefined' && editor) {
      editor.focus(buffer);
      setBuffer(false);
    }
  }, [editor, buffer]);
};
const FeelEditor = compat.forwardRef((props, ref) => {
  const {
    contentAttributes,
    enableGutters,
    value,
    onInput,
    onFeelToggle = noop$4,
    onLint = noop$4,
    onOpenPopup = noop$4,
    placeholder,
    popupOpen,
    disabled,
    tooltipContainer,
    variables,
    feelLanguageContext
  } = props;
  const inputRef = hooks.useRef();
  const [editor, setEditor] = hooks.useState();
  const [localValue, setLocalValue] = hooks.useState(value || '');
  const {
    builtins,
    dialect,
    parserDialect
  } = feelLanguageContext || {};
  useBufferedFocus(editor, ref);
  const handleInput = useStaticCallback(newValue => {
    onInput(newValue);
    setLocalValue(newValue);
  });
  hooks.useEffect(() => {
    let editor;

    /* Trigger FEEL toggle when
     *
     * - `backspace` is pressed
     * - AND the cursor is at the beginning of the input
     */
    const onKeyDown = e => {
      if (e.key !== 'Backspace' || !editor) {
        return;
      }
      const selection = editor.getSelection();
      const range = selection.ranges[selection.mainIndex];
      if (range.from === 0 && range.to === 0) {
        onFeelToggle();
      }
    };
    editor = new Editor({
      container: inputRef.current,
      onChange: handleInput,
      onKeyDown: onKeyDown,
      onLint: onLint,
      placeholder: placeholder,
      tooltipContainer: tooltipContainer,
      value: localValue,
      variables,
      builtins,
      dialect,
      parserDialect,
      extensions: [...(enableGutters ? [view.lineNumbers()] : []), view.EditorView.lineWrapping],
      contentAttributes
    });
    setEditor(editor);
    return () => {
      onLint([]);
      inputRef.current.innerHTML = '';
      setEditor(null);
    };
  }, []);
  hooks.useEffect(() => {
    if (!editor) {
      return;
    }
    if (value === localValue) {
      return;
    }
    editor.setValue(value);
    setLocalValue(value);
  }, [value]);
  hooks.useEffect(() => {
    if (!editor) {
      return;
    }
    editor.setVariables(variables);
  }, [variables]);
  hooks.useEffect(() => {
    if (!editor) {
      return;
    }
    editor.setPlaceholder(placeholder);
  }, [placeholder]);
  const handleClick = () => {
    ref.current.focus();
  };
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-feel-editor-container', disabled ? 'disabled' : null, popupOpen ? 'popupOpen' : null),
    children: [popupOpen && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-feel-editor__open-popup-placeholder",
      children: "Opened in editor"
    }), jsxRuntime.jsx("div", {
      name: props.name,
      class: classnames('bio-properties-panel-input', localValue ? 'edited' : null),
      ref: inputRef,
      onClick: handleClick
    }), !disabled && jsxRuntime.jsx(OpenPopupButton, {
      onClick: () => onOpenPopup('feel')
    })]
  });
});

function FeelIndicator(props) {
  const {
    active
  } = props;
  if (!active) {
    return null;
  }
  return jsxRuntime.jsx("span", {
    class: "bio-properties-panel-feel-indicator",
    children: "="
  });
}

const noop$3 = () => {};

/**
 * @param {Object} props
 * @param {Object} props.label
 * @param {String} props.feel
 */
function FeelIcon(props) {
  const {
    feel = false,
    active,
    disabled = false,
    onClick = noop$3
  } = props;
  const feelRequiredLabel = 'FEEL expression is mandatory';
  const feelOptionalLabel = `Click to ${active ? 'remove' : 'set a'} dynamic value with FEEL expression`;
  const handleClick = e => {
    onClick(e);

    // when pointer event was created from keyboard, keep focus on button
    if (!e.pointerType) {
      e.stopPropagation();
    }
  };
  return jsxRuntime.jsx("button", {
    type: "button",
    class: classnames('bio-properties-panel-feel-icon', active ? 'active' : null, feel === 'required' ? 'required' : 'optional'),
    onClick: handleClick,
    disabled: feel === 'required' || disabled,
    title: feel === 'required' ? feelRequiredLabel : feelOptionalLabel,
    children: jsxRuntime.jsx(FeelIcon$1, {})
  });
}

function ToggleSwitch(props) {
  const {
    id,
    label,
    onInput,
    value,
    switcherLabel,
    inline,
    onFocus,
    onBlur,
    inputRef,
    tooltip
  } = props;
  const [localValue, setLocalValue] = hooks.useState(value);
  const handleInputCallback = async () => {
    onInput(!value);
  };
  const handleInput = e => {
    handleInputCallback();
    setLocalValue(e.target.value);
  };
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value);
  }, [value]);
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-toggle-switch', {
      inline
    }),
    children: [jsxRuntime.jsx("label", {
      class: "bio-properties-panel-label",
      for: prefixId$7(id),
      children: jsxRuntime.jsx(TooltipWrapper, {
        value: tooltip,
        forId: id,
        element: props.element,
        children: label
      })
    }), jsxRuntime.jsxs("div", {
      class: "bio-properties-panel-field-wrapper",
      children: [jsxRuntime.jsxs("label", {
        class: "bio-properties-panel-toggle-switch__switcher",
        children: [jsxRuntime.jsx("input", {
          ref: inputRef,
          id: prefixId$7(id),
          class: "bio-properties-panel-input",
          type: "checkbox",
          onFocus: onFocus,
          onBlur: onBlur,
          name: id,
          onInput: handleInput,
          checked: !!localValue
        }), jsxRuntime.jsx("span", {
          class: "bio-properties-panel-toggle-switch__slider"
        })]
      }), switcherLabel && jsxRuntime.jsx("p", {
        class: "bio-properties-panel-toggle-switch__label",
        children: switcherLabel
      })]
    })]
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {String} props.label
 * @param {String} props.switcherLabel
 * @param {Boolean} props.inline
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {string|import('preact').Component} props.tooltip
 */
function ToggleSwitchEntry(props) {
  const {
    element,
    id,
    description,
    label,
    switcherLabel,
    inline,
    getValue,
    setValue,
    onFocus,
    onBlur,
    tooltip
  } = props;
  const value = getValue(element);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-entry bio-properties-panel-toggle-switch-entry",
    "data-entry-id": id,
    children: [jsxRuntime.jsx(ToggleSwitch, {
      id: id,
      label: label,
      value: value,
      onInput: setValue,
      onFocus: onFocus,
      onBlur: onBlur,
      switcherLabel: switcherLabel,
      inline: inline,
      tooltip: tooltip,
      element: element
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}
function isEdited$7(node) {
  return node && !!node.checked;
}

// helpers /////////////////

function prefixId$7(id) {
  return `bio-properties-panel-${id}`;
}

function NumberField(props) {
  const {
    debounce,
    disabled,
    displayLabel = true,
    id,
    inputRef,
    label,
    max,
    min,
    onInput,
    step,
    value = '',
    onFocus,
    onBlur,
    tooltip
  } = props;
  const [localValue, setLocalValue] = hooks.useState(value);
  const handleInputCallback = hooks.useMemo(() => {
    return debounce(target => {
      if (target.validity.valid) {
        onInput(target.value ? parseFloat(target.value) : undefined);
      }
    });
  }, [onInput, debounce]);
  const handleInput = e => {
    handleInputCallback(e.target);
    setLocalValue(e.target.value);
  };
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value);
  }, [value]);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-numberfield",
    children: [displayLabel && jsxRuntime.jsx("label", {
      for: prefixId$6(id),
      class: "bio-properties-panel-label",
      children: jsxRuntime.jsx(TooltipWrapper, {
        value: tooltip,
        forId: id,
        element: props.element,
        children: label
      })
    }), jsxRuntime.jsx("input", {
      id: prefixId$6(id),
      ref: inputRef,
      type: "number",
      name: id,
      spellCheck: "false",
      autoComplete: "off",
      disabled: disabled,
      class: "bio-properties-panel-input",
      max: max,
      min: min,
      onInput: handleInput,
      onFocus: onFocus,
      onBlur: onBlur,
      step: step,
      value: localValue
    })]
  });
}

/**
 * @param {Object} props
 * @param {Boolean} props.debounce
 * @param {String} props.description
 * @param {Boolean} props.disabled
 * @param {Object} props.element
 * @param {Function} props.getValue
 * @param {String} props.id
 * @param {String} props.label
 * @param {String} props.max
 * @param {String} props.min
 * @param {Function} props.setValue
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {String} props.step
 * @param {Function} props.validate
 */
function NumberFieldEntry(props) {
  const {
    debounce,
    description,
    disabled,
    element,
    getValue,
    id,
    label,
    max,
    min,
    setValue,
    step,
    onFocus,
    onBlur,
    validate,
    tooltip
  } = props;
  const globalError = useError(id);
  const [localError, setLocalError] = hooks.useState(null);
  let value = getValue(element);
  hooks.useEffect(() => {
    if (minDash.isFunction(validate)) {
      const newValidationError = validate(value) || null;
      setLocalError(newValidationError);
    }
  }, [value, validate]);
  const onInput = newValue => {
    let newValidationError = null;
    if (minDash.isFunction(validate)) {
      newValidationError = validate(newValue) || null;
    }
    setValue(newValue, newValidationError);
    setLocalError(newValidationError);
  };
  const error = globalError || localError;
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-entry', error ? 'has-error' : ''),
    "data-entry-id": id,
    children: [jsxRuntime.jsx(NumberField, {
      debounce: debounce,
      disabled: disabled,
      id: id,
      label: label,
      onFocus: onFocus,
      onBlur: onBlur,
      onInput: onInput,
      max: max,
      min: min,
      step: step,
      value: value,
      tooltip: tooltip
    }, element), error && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-error",
      children: error
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}
function isEdited$6(node) {
  return node && !!node.value;
}

// helpers /////////////////

function prefixId$6(id) {
  return `bio-properties-panel-${id}`;
}

const noop$2 = () => {};

/**
 * @typedef {'required'|'optional'|'static'} FeelType
 */

/**
 * @param {Object} props
 * @param {Boolean} props.debounce
 * @param {String} props.id
 * @param {Object} props.element
 * @param {String} props.label
 * @param {String} props.hostLanguage
 * @param {Function} props.onInput
 * @param {Function} props.onBlur
 * @param {Function} props.onError
 * @param {FeelType} [props.feel]
 * @param {String} props.value
 * @param {Boolean} [props.singleLine]
 * @param {Function} props.tooltipContainer
 * @param {Function | import('preact').Component} props.OptionalComponent
 * @param {Boolean} props.disabled
 * @param {Array} props.variables
 * @param {string} [props.placeholder]
 * @param {string | import('preact').Component} props.tooltip
 */
function FeelTextfield(props) {
  const {
    debounce,
    id,
    element,
    label,
    hostLanguage,
    onInput,
    onBlur,
    onError,
    placeholder,
    feel,
    value = '',
    disabled = false,
    variables,
    singleLine,
    tooltipContainer,
    OptionalComponent = OptionalFeelInput,
    tooltip
  } = props;
  const [localValue, setLocalValue] = hooks.useState(value);
  const editorRef = useShowEntryEvent(id);
  const containerRef = hooks.useRef();
  const feelActive = minDash.isString(localValue) && localValue.startsWith('=') || feel === 'required';
  const feelOnlyValue = minDash.isString(localValue) && localValue.startsWith('=') ? localValue.substring(1) : localValue;
  const feelLanguageContext = hooks.useContext(FeelLanguageContext);
  const [focus, _setFocus] = hooks.useState(undefined);
  const {
    eventBus
  } = hooks.useContext(EventContext);
  const [isPopupOpen, setIsPopupOpen] = hooks.useState(false);
  const setFocus = (offset = 0) => {
    const hasFocus = containerRef.current.contains(document.activeElement);

    // Keep caret position if it is already focused, otherwise focus at the end
    const position = hasFocus ? document.activeElement.selectionStart : Infinity;
    _setFocus(position + offset);
  };

  /**
   * @type { import('min-dash').DebouncedFunction }
   */
  const handleInputCallback = useDebounce(onInput, debounce);
  const handleInput = newValue => {
    // we don't commit empty FEEL expressions,
    // but instead serialize them as <undefined>
    const newModelValue = newValue === '' || newValue === '=' ? undefined : newValue;
    handleInputCallback(newModelValue);
  };
  const handleFeelToggle = useStaticCallback(() => {
    if (feel === 'required') {
      return;
    }
    if (!feelActive) {
      setLocalValue('=' + localValue);
      handleInput('=' + localValue);
    } else {
      setLocalValue(feelOnlyValue);
      handleInput(feelOnlyValue);
    }
  });
  const handleLocalInput = newValue => {
    if (feelActive) {
      newValue = '=' + newValue;
    }
    if (newValue === localValue) {
      return;
    }
    setLocalValue(newValue);
    handleInput(newValue);
    if (!feelActive && minDash.isString(newValue) && newValue.startsWith('=')) {
      // focus is behind `=` sign that will be removed
      setFocus(-1);
    }
  };
  const handleOnBlur = e => {
    if (e.target.type === 'checkbox') {
      onInput(e.target.checked);
    } else {
      const trimmedValue = e.target.value.trim();
      onInput(trimmedValue);
    }
    if (onBlur) {
      onBlur(e);
    }
  };
  const handleLint = useStaticCallback((lint = []) => {
    const syntaxError = lint.some(report => report.type === 'Syntax Error');
    if (syntaxError) {
      onError('Unparsable FEEL expression.');
    } else {
      onError(undefined);
    }
  });
  const handleOpenPopup = (type = 'feel') => {
    const isOpen = eventBus.fire('propertiesPanel.openPopup', {
      element,
      entryId: id,
      hostLanguage,
      label,
      onInput: handleLocalInput,
      singleLine,
      sourceElement: editorRef.current,
      tooltipContainer,
      type,
      value: feelOnlyValue,
      variables,
      feelLanguageContext
    });
    if (isOpen) {
      eventBus.once('propertiesPanelPopup.close', () => {
        setIsPopupOpen(false);
      });
    }
    setIsPopupOpen(isOpen === true);
  };
  hooks.useEffect(() => {
    if (typeof focus !== 'undefined') {
      editorRef.current.focus(focus);
      _setFocus(undefined);
    }
  }, [focus]);
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }

    // External value change removed content => keep FEEL configuration
    if (!value) {
      setLocalValue(feelActive ? '=' : '');
      return;
    }
    setLocalValue(value);
  }, [value]);
  hooks.useEffect(() => {
    return () => {
      eventBus.fire('propertiesPanel.closePopup');
    };
  }, []);

  // copy-paste integration
  hooks.useEffect(() => {
    const copyHandler = event => {
      if (!feelActive) {
        return;
      }
      event.clipboardData.setData('application/FEEL', event.clipboardData.getData('text'));
    };
    const pasteHandler = event => {
      if (feelActive || isPopupOpen) {
        return;
      }
      const data = event.clipboardData.getData('application/FEEL');
      if (data) {
        setTimeout(() => {
          handleFeelToggle();
          setFocus();
        });
      }
    };
    containerRef.current.addEventListener('copy', copyHandler);
    containerRef.current.addEventListener('cut', copyHandler);
    containerRef.current.addEventListener('paste', pasteHandler);
    return () => {
      containerRef.current.removeEventListener('copy', copyHandler);
      containerRef.current.removeEventListener('cut', copyHandler);
      containerRef.current.removeEventListener('paste', pasteHandler);
    };
  }, [containerRef, feelActive, handleFeelToggle, setFocus]);
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-feel-entry', {
      'feel-active': feelActive
    }),
    children: [jsxRuntime.jsxs("label", {
      for: prefixId$5(id),
      class: "bio-properties-panel-label",
      onClick: () => setFocus(),
      children: [jsxRuntime.jsx(TooltipWrapper, {
        value: tooltip,
        forId: id,
        element: props.element,
        children: label
      }), jsxRuntime.jsx(FeelIcon, {
        label: label,
        feel: feel,
        onClick: handleFeelToggle,
        active: feelActive
      })]
    }), jsxRuntime.jsxs("div", {
      class: "bio-properties-panel-feel-container",
      ref: containerRef,
      children: [jsxRuntime.jsx(FeelIndicator, {
        active: feelActive,
        disabled: feel !== 'optional' || disabled,
        onClick: handleFeelToggle
      }), feelActive ? jsxRuntime.jsx(FeelEditor, {
        name: id,
        onInput: handleLocalInput,
        contentAttributes: {
          'id': prefixId$5(id),
          'aria-label': label
        },
        disabled: disabled,
        popupOpen: isPopupOpen,
        onFeelToggle: () => {
          handleFeelToggle();
          setFocus(true);
        },
        onLint: handleLint,
        onOpenPopup: handleOpenPopup,
        placeholder: placeholder,
        value: feelOnlyValue,
        variables: variables,
        feelLanguageContext: feelLanguageContext,
        ref: editorRef,
        tooltipContainer: tooltipContainer
      }) : jsxRuntime.jsx(OptionalComponent, {
        ...props,
        popupOpen: isPopupOpen,
        onInput: handleLocalInput,
        onBlur: handleOnBlur,
        contentAttributes: {
          'id': prefixId$5(id),
          'aria-label': label
        },
        value: localValue,
        ref: editorRef,
        onOpenPopup: handleOpenPopup,
        containerRef: containerRef
      })]
    })]
  });
}
const OptionalFeelInput = compat.forwardRef((props, ref) => {
  const {
    id,
    disabled,
    onInput,
    value,
    onFocus,
    onBlur,
    placeholder
  } = props;
  const inputRef = hooks.useRef();

  // To be consistent with the FEEL editor, set focus at start of input
  // this ensures clean editing experience when switching with the keyboard
  ref.current = {
    focus: position => {
      const input = inputRef.current;
      if (!input) {
        return;
      }
      input.focus();
      if (typeof position === 'number') {
        if (position > value.length) {
          position = value.length;
        }
        input.setSelectionRange(position, position);
      }
    }
  };
  return jsxRuntime.jsx("input", {
    id: prefixId$5(id),
    type: "text",
    ref: inputRef,
    name: id,
    spellCheck: "false",
    autoComplete: "off",
    disabled: disabled,
    class: "bio-properties-panel-input",
    onInput: e => onInput(e.target.value),
    onFocus: onFocus,
    onBlur: onBlur,
    placeholder: placeholder,
    value: value || ''
  });
});
const OptionalFeelNumberField = compat.forwardRef((props, ref) => {
  const {
    id,
    debounce,
    disabled,
    onInput,
    value,
    min,
    max,
    step,
    onFocus,
    onBlur
  } = props;
  const inputRef = hooks.useRef();

  // To be consistent with the FEEL editor, set focus at start of input
  // this ensures clean editing experience when switching with the keyboard
  ref.current = {
    focus: position => {
      const input = inputRef.current;
      if (!input) {
        return;
      }
      input.focus();
      if (typeof position === 'number' && position !== Infinity) {
        if (position > value.length) {
          position = value.length;
        }
        input.setSelectionRange(position, position);
      }
    }
  };
  return jsxRuntime.jsx(NumberField, {
    id: id,
    debounce: debounce,
    disabled: disabled,
    displayLabel: false,
    inputRef: inputRef,
    max: max,
    min: min,
    onInput: onInput,
    step: step,
    value: value,
    onFocus: onFocus,
    onBlur: onBlur
  });
});
const OptionalFeelTextArea = compat.forwardRef((props, ref) => {
  const {
    id,
    disabled,
    onInput,
    value,
    onFocus,
    onBlur,
    placeholder
  } = props;
  const inputRef = hooks.useRef();

  // To be consistent with the FEEL editor, set focus at start of input
  // this ensures clean editing experience when switching with the keyboard
  ref.current = {
    focus: () => {
      const input = inputRef.current;
      if (!input) {
        return;
      }
      input.focus();
      input.setSelectionRange(0, 0);
    }
  };
  return jsxRuntime.jsx("textarea", {
    id: prefixId$5(id),
    type: "text",
    ref: inputRef,
    name: id,
    spellCheck: "false",
    autoComplete: "off",
    disabled: disabled,
    class: "bio-properties-panel-input",
    onInput: e => onInput(e.target.value),
    onFocus: onFocus,
    onBlur: onBlur,
    placeholder: placeholder,
    value: value || '',
    "data-gramm": "false"
  });
});
const OptionalFeelToggleSwitch = compat.forwardRef((props, ref) => {
  const {
    id,
    onInput,
    value,
    onFocus,
    onBlur,
    switcherLabel
  } = props;
  const inputRef = hooks.useRef();

  // To be consistent with the FEEL editor, set focus at start of input
  // this ensures clean editing experience when switching with the keyboard
  ref.current = {
    focus: () => {
      const input = inputRef.current;
      if (!input) {
        return;
      }
      input.focus();
    }
  };
  return jsxRuntime.jsx(ToggleSwitch, {
    id: id,
    value: value,
    inputRef: inputRef,
    onInput: onInput,
    onFocus: onFocus,
    onBlur: onBlur,
    switcherLabel: switcherLabel
  });
});
const OptionalFeelCheckbox = compat.forwardRef((props, ref) => {
  const {
    id,
    disabled,
    onInput,
    value,
    onFocus,
    onBlur
  } = props;
  const inputRef = hooks.useRef();
  const handleChange = ({
    target
  }) => {
    onInput(target.checked);
  };

  // To be consistent with the FEEL editor, set focus at start of input
  // this ensures clean editing experience when switching with the keyboard
  ref.current = {
    focus: () => {
      const input = inputRef.current;
      if (!input) {
        return;
      }
      input.focus();
    }
  };
  return jsxRuntime.jsx("input", {
    ref: inputRef,
    id: prefixId$5(id),
    name: id,
    onFocus: onFocus,
    onBlur: onBlur,
    type: "checkbox",
    class: "bio-properties-panel-input",
    onChange: handleChange,
    checked: value,
    disabled: disabled
  });
});

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {FeelType} [props.feel]
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.tooltipContainer
 * @param {Function} props.validate
 * @param {Function} props.show
 * @param {Function} props.example
 * @param {Array} props.variables
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {string} [props.placeholder]
 * @param {string|import('preact').Component} props.tooltip
 */
function FeelEntry(props) {
  const {
    element,
    id,
    description,
    debounce,
    disabled,
    feel,
    label,
    getValue,
    setValue,
    tooltipContainer,
    hostLanguage,
    singleLine,
    validate,
    show = noop$2,
    example,
    variables,
    onFocus,
    onBlur,
    placeholder,
    tooltip
  } = props;
  const [validationError, setValidationError] = hooks.useState(null);
  const [localError, setLocalError] = hooks.useState(null);
  let value = getValue(element);
  hooks.useEffect(() => {
    if (minDash.isFunction(validate)) {
      const newValidationError = validate(value) || null;
      setValidationError(newValidationError);
    }
  }, [value, validate]);
  const onInput = useStaticCallback(newValue => {
    const value = getValue(element);
    let newValidationError = null;
    if (minDash.isFunction(validate)) {
      newValidationError = validate(newValue) || null;
    }

    // don't create multiple commandStack entries for the same value
    if (newValue !== value) {
      setValue(newValue, newValidationError);
    }
    setValidationError(newValidationError);
  });
  const onError = hooks.useCallback(err => {
    setLocalError(err);
  }, []);
  const temporaryError = useError(id);
  const error = temporaryError || localError || validationError;
  return jsxRuntime.jsxs("div", {
    class: classnames(props.class, 'bio-properties-panel-entry', error ? 'has-error' : ''),
    "data-entry-id": id,
    children: [preact.createElement(FeelTextfield, {
      ...props,
      debounce: debounce,
      disabled: disabled,
      feel: feel,
      id: id,
      key: element,
      label: label,
      onInput: onInput,
      onError: onError,
      onFocus: onFocus,
      onBlur: onBlur,
      placeholder: placeholder,
      example: example,
      hostLanguage: hostLanguage,
      singleLine: singleLine,
      show: show,
      value: value,
      variables: variables,
      tooltipContainer: tooltipContainer,
      OptionalComponent: props.OptionalComponent,
      tooltip: tooltip
    }), error && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-error",
      children: error
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {String} props.max
 * @param {String} props.min
 * @param {String} props.step
 * @param {FeelType} [props.feel]
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.tooltipContainer
 * @param {Function} props.validate
 * @param {Function} props.show
 * @param {Function} props.example
 * @param {Array} props.variables
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 */
function FeelNumberEntry(props) {
  return jsxRuntime.jsx(FeelEntry, {
    class: "bio-properties-panel-feel-number",
    OptionalComponent: OptionalFeelNumberField,
    ...props
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {FeelType} [props.feel]
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.tooltipContainer
 * @param {Function} props.validate
 * @param {Function} props.show
 * @param {Function} props.example
 * @param {Array} props.variables
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {string} [props.placeholder]
 */
function FeelTextAreaEntry(props) {
  return jsxRuntime.jsx(FeelEntry, {
    class: "bio-properties-panel-feel-textarea",
    OptionalComponent: OptionalFeelTextArea,
    ...props
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {FeelType} [props.feel]
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.tooltipContainer
 * @param {Function} props.validate
 * @param {Function} props.show
 * @param {Function} props.example
 * @param {Array} props.variables
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 */
function FeelToggleSwitchEntry(props) {
  return jsxRuntime.jsx(FeelEntry, {
    class: "bio-properties-panel-feel-toggle-switch",
    OptionalComponent: OptionalFeelToggleSwitch,
    ...props
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {FeelType} [props.feel]
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.tooltipContainer
 * @param {Function} props.validate
 * @param {Function} props.show
 * @param {Function} props.example
 * @param {Array} props.variables
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 */
function FeelCheckboxEntry(props) {
  return jsxRuntime.jsx(FeelEntry, {
    class: "bio-properties-panel-feel-checkbox",
    OptionalComponent: OptionalFeelCheckbox,
    ...props
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {String} props.hostLanguage
 * @param {Boolean} props.singleLine
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {FeelType} [props.feel]
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.tooltipContainer
 * @param {Function} props.validate
 * @param {Function} props.show
 * @param {Function} props.example
 * @param {Array} props.variables
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 */
function FeelTemplatingEntry(props) {
  return jsxRuntime.jsx(FeelEntry, {
    class: "bio-properties-panel-feel-templating",
    OptionalComponent: TemplatingEditor,
    ...props
  });
}
function isEdited$5(node) {
  if (!node) {
    return false;
  }
  if (node.type === 'checkbox') {
    return !!node.checked || node.classList.contains('edited');
  }
  return !!node.value || node.classList.contains('edited');
}

// helpers /////////////////

function prefixId$5(id) {
  return `bio-properties-panel-${id}`;
}

const noop$1 = () => {};

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.tooltipContainer
 * @param {Function} props.validate
 * @param {Function} props.show
 */
function TemplatingEntry(props) {
  const {
    element,
    id,
    description,
    debounce,
    disabled,
    label,
    getValue,
    setValue,
    tooltipContainer,
    validate,
    show = noop$1
  } = props;
  const [validationError, setValidationError] = hooks.useState(null);
  const [localError, setLocalError] = hooks.useState(null);
  let value = getValue(element);
  hooks.useEffect(() => {
    if (minDash.isFunction(validate)) {
      const newValidationError = validate(value) || null;
      setValidationError(newValidationError);
    }
  }, [value, validate]);
  const onInput = useStaticCallback(newValue => {
    let newValidationError = null;
    if (minDash.isFunction(validate)) {
      newValidationError = validate(newValue) || null;
    }

    // don't create multiple commandStack entries for the same value
    if (newValue !== value) {
      setValue(newValue, newValidationError);
    }
    setValidationError(newValidationError);
  });
  const onError = hooks.useCallback(err => {
    setLocalError(err);
  }, []);
  const temporaryError = useError(id);
  const error = localError || temporaryError || validationError;
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-entry', error ? 'has-error' : ''),
    "data-entry-id": id,
    children: [jsxRuntime.jsx(Templating, {
      debounce: debounce,
      disabled: disabled,
      id: id,
      label: label,
      onInput: onInput,
      onError: onError,
      show: show,
      value: value,
      tooltipContainer: tooltipContainer
    }, element), error && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-error",
      children: error
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}
function Templating(props) {
  const {
    debounce,
    id,
    label,
    onInput,
    onError,
    value = '',
    disabled = false,
    tooltipContainer
  } = props;
  const [localValue, setLocalValue] = hooks.useState(value);
  const editorRef = useShowEntryEvent(id);
  const containerRef = hooks.useRef();
  const [focus, _setFocus] = hooks.useState(undefined);
  const setFocus = (offset = 0) => {
    const hasFocus = containerRef.current.contains(document.activeElement);

    // Keep caret position if it is already focused, otherwise focus at the end
    const position = hasFocus ? document.activeElement.selectionStart : Infinity;
    _setFocus(position + offset);
  };
  const handleInputCallback = hooks.useMemo(() => {
    return debounce(newValue => onInput(newValue.length ? newValue : undefined));
  }, [onInput, debounce]);
  const handleInput = newValue => {
    handleInputCallback(newValue);
    setLocalValue(newValue);
  };
  const handleLint = useStaticCallback(lint => {
    const errors = lint && lint.length && lint.filter(e => e.severity === 'error') || [];
    if (!errors.length) {
      onError(undefined);
      return;
    }
    const error = lint[0];
    const message = `${error.source}: ${error.message}`;
    onError(message);
  });
  hooks.useEffect(() => {
    if (typeof focus !== 'undefined') {
      editorRef.current.focus(focus);
      _setFocus(undefined);
    }
  }, [focus]);
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value ? value : '');
  }, [value]);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-feelers",
    children: [jsxRuntime.jsx("label", {
      id: prefixIdLabel(id),
      class: "bio-properties-panel-label",
      onClick: () => setFocus(),
      children: label
    }), jsxRuntime.jsx("div", {
      class: "bio-properties-panel-feelers-input",
      ref: containerRef,
      children: jsxRuntime.jsx(TemplatingEditor, {
        name: id,
        onInput: handleInput,
        contentAttributes: {
          'aria-labelledby': prefixIdLabel(id)
        },
        disabled: disabled,
        onLint: handleLint,
        value: localValue,
        ref: editorRef,
        tooltipContainer: tooltipContainer
      })
    })]
  });
}
function isEdited$4(node) {
  return node && (!!node.value || node.classList.contains('edited'));
}

// helpers /////////////////

function prefixIdLabel(id) {
  return `bio-properties-panel-feelers-${id}-label`;
}

function List(props) {
  const {
    id,
    element,
    items = [],
    component,
    label = '<empty>',
    open: shouldOpen,
    onAdd,
    onRemove,
    autoFocusEntry,
    ...restProps
  } = props;
  const entryRef = hooks.useRef(null);
  const [open, setOpen] = hooks.useState(!!shouldOpen);
  const [sticky, setSticky] = hooks.useState(false);
  const hasItems = !!items.length;
  const toggleOpen = () => hasItems && setOpen(!open);
  const elementChanged = usePrevious(element) !== element;
  const newItems = useNewItems(items, elementChanged);
  hooks.useEffect(() => {
    if (open && !hasItems) {
      setOpen(false);
    }
  }, [open, hasItems]);

  /**
   * @param {MouseEvent} event
   */
  function addItem(event) {
    event.stopPropagation();
    onAdd();
    if (!open) {
      setOpen(true);
    }
  }

  // set css class when entry is sticky to top
  useStickyIntersectionObserver(entryRef, 'div.bio-properties-panel-scroll-container', setSticky);
  return jsxRuntime.jsxs("div", {
    "data-entry-id": id,
    class: classnames('bio-properties-panel-entry', 'bio-properties-panel-list-entry', hasItems ? '' : 'empty', open ? 'open' : ''),
    ref: entryRef,
    children: [jsxRuntime.jsxs("div", {
      class: classnames('bio-properties-panel-list-entry-header', sticky && open ? 'sticky' : ''),
      onClick: toggleOpen,
      children: [jsxRuntime.jsx("div", {
        title: label,
        class: classnames('bio-properties-panel-list-entry-header-title', open && 'open'),
        children: label
      }), jsxRuntime.jsxs("div", {
        class: "bio-properties-panel-list-entry-header-buttons",
        children: [jsxRuntime.jsxs("button", {
          type: "button",
          title: "Create new list item",
          onClick: addItem,
          class: "bio-properties-panel-add-entry",
          children: [jsxRuntime.jsx(CreateIcon, {}), !hasItems ? jsxRuntime.jsx("span", {
            class: "bio-properties-panel-add-entry-label",
            children: "Create"
          }) : null]
        }), hasItems && jsxRuntime.jsx("div", {
          title: `List contains ${items.length} item${items.length != 1 ? 's' : ''}`,
          class: "bio-properties-panel-list-badge",
          children: items.length
        }), hasItems && jsxRuntime.jsx("button", {
          type: "button",
          title: "Toggle list item",
          class: "bio-properties-panel-arrow",
          children: jsxRuntime.jsx(ArrowIcon, {
            class: open ? 'bio-properties-panel-arrow-down' : 'bio-properties-panel-arrow-right'
          })
        })]
      })]
    }), hasItems && jsxRuntime.jsx(ItemsList, {
      ...restProps,
      autoFocusEntry: autoFocusEntry,
      component: component,
      element: element,
      id: id,
      items: items,
      newItems: newItems,
      onRemove: onRemove,
      open: open
    })]
  });
}
function ItemsList(props) {
  const {
    autoFocusEntry,
    component: Component,
    element,
    id,
    items,
    newItems,
    onRemove,
    open,
    ...restProps
  } = props;
  const getKey = useKeyFactory();
  const newItem = newItems[0];
  hooks.useEffect(() => {
    if (newItem && autoFocusEntry) {
      // (0) select the parent entry (containing all list items)
      const entry = minDom.query(`[data-entry-id="${id}"]`);

      // (1) select the first input or a custom element to be focussed
      const selector = typeof autoFocusEntry === 'boolean' ? '.bio-properties-panel-input' : autoFocusEntry;
      const focusableInput = minDom.query(selector, entry);

      // (2) set focus
      if (focusableInput) {
        if (minDash.isFunction(focusableInput.select)) {
          focusableInput.select();
        } else if (minDash.isFunction(focusableInput.focus)) {
          focusableInput.focus();
        }
      }
    }
  }, [newItem, autoFocusEntry, id]);
  return jsxRuntime.jsx("ol", {
    class: classnames('bio-properties-panel-list-entry-items', open ? 'open' : ''),
    children: items.map((item, index) => {
      const key = getKey(item);
      return jsxRuntime.jsxs("li", {
        class: "bio-properties-panel-list-entry-item",
        children: [jsxRuntime.jsx(Component, {
          ...restProps,
          element: element,
          id: id,
          index: index,
          item: item,
          open: item === newItem
        }), onRemove && jsxRuntime.jsx("button", {
          type: "button",
          title: "Delete item",
          class: "bio-properties-panel-remove-entry bio-properties-panel-remove-list-entry",
          onClick: () => onRemove && onRemove(item),
          children: jsxRuntime.jsx(DeleteIcon, {})
        })]
      }, key);
    })
  });
}
function useNewItems(items = [], shouldReset) {
  const previousItems = usePrevious(items.slice()) || [];
  if (shouldReset) {
    return [];
  }
  return previousItems ? items.filter(item => !previousItems.includes(item)) : [];
}

function Select(props) {
  const {
    id,
    label,
    onChange,
    options = [],
    value = '',
    disabled,
    onFocus,
    onBlur,
    tooltip
  } = props;
  const ref = useShowEntryEvent(id);
  const [localValue, setLocalValue] = hooks.useState(value);
  const handleChangeCallback = ({
    target
  }) => {
    onChange(target.value);
  };
  const handleChange = e => {
    handleChangeCallback(e);
    setLocalValue(e.target.value);
  };
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value);
  }, [value]);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-select",
    children: [jsxRuntime.jsx("label", {
      for: prefixId$4(id),
      class: "bio-properties-panel-label",
      children: jsxRuntime.jsx(TooltipWrapper, {
        value: tooltip,
        forId: id,
        element: props.element,
        children: label
      })
    }), jsxRuntime.jsx("select", {
      ref: ref,
      id: prefixId$4(id),
      name: id,
      class: "bio-properties-panel-input",
      onInput: handleChange,
      onFocus: onFocus,
      onBlur: onBlur,
      value: localValue,
      disabled: disabled,
      children: options.map((option, idx) => {
        if (option.children) {
          return jsxRuntime.jsx("optgroup", {
            label: option.label,
            children: option.children.map((child, idx) => jsxRuntime.jsx("option", {
              value: child.value,
              disabled: child.disabled,
              children: child.label
            }, idx))
          }, idx);
        }
        return jsxRuntime.jsx("option", {
          value: option.value,
          disabled: option.disabled,
          children: option.label
        }, idx);
      })
    })]
  });
}

/**
 * @param {object} props
 * @param {object} props.element
 * @param {string} props.id
 * @param {string} [props.description]
 * @param {string} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {Function} props.getOptions
 * @param {boolean} [props.disabled]
 * @param {Function} [props.validate]
 * @param {string|import('preact').Component} props.tooltip
 */
function SelectEntry(props) {
  const {
    element,
    id,
    description,
    label,
    getValue,
    setValue,
    getOptions,
    disabled,
    onFocus,
    onBlur,
    validate,
    tooltip
  } = props;
  const options = getOptions(element);
  const globalError = useError(id);
  const [localError, setLocalError] = hooks.useState(null);
  let value = getValue(element);
  hooks.useEffect(() => {
    if (minDash.isFunction(validate)) {
      const newValidationError = validate(value) || null;
      setLocalError(newValidationError);
    }
  }, [value, validate]);
  const onChange = newValue => {
    let newValidationError = null;
    if (minDash.isFunction(validate)) {
      newValidationError = validate(newValue) || null;
    }
    setValue(newValue, newValidationError);
    setLocalError(newValidationError);
  };
  const error = globalError || localError;
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-entry', error ? 'has-error' : ''),
    "data-entry-id": id,
    children: [jsxRuntime.jsx(Select, {
      id: id,
      label: label,
      value: value,
      onChange: onChange,
      onFocus: onFocus,
      onBlur: onBlur,
      options: options,
      disabled: disabled,
      tooltip: tooltip,
      element: element
    }, element), error && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-error",
      children: error
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}
function isEdited$3(node) {
  return node && !!node.value;
}

// helpers /////////////////

function prefixId$4(id) {
  return `bio-properties-panel-${id}`;
}

function Simple(props) {
  const {
    debounce,
    disabled,
    element,
    getValue,
    id,
    onBlur,
    onFocus,
    setValue
  } = props;
  const value = getValue(element);
  const [localValue, setLocalValue] = hooks.useState(value);
  const handleInputCallback = hooks.useMemo(() => {
    return debounce(target => setValue(target.value.length ? target.value : undefined));
  }, [setValue, debounce]);
  const handleInput = e => {
    handleInputCallback(e.target);
    setLocalValue(e.target.value);
  };
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value);
  }, [value]);
  return jsxRuntime.jsx("div", {
    class: "bio-properties-panel-simple",
    children: jsxRuntime.jsx("input", {
      id: prefixId$3(id),
      type: "text",
      name: id,
      spellCheck: "false",
      autoComplete: "off",
      disabled: disabled,
      class: "bio-properties-panel-input",
      onInput: handleInput,
      "aria-label": localValue || '<empty>',
      onFocus: onFocus,
      onBlur: onBlur,
      value: localValue
    }, element)
  });
}
function isEdited$2(node) {
  return node && !!node.value;
}

// helpers /////////////////

function prefixId$3(id) {
  return `bio-properties-panel-${id}`;
}

function resizeToContents(element) {
  element.style.height = 'auto';

  // a 2px pixel offset is required to prevent scrollbar from
  // appearing on OS with a full length scroll bar (Windows/Linux)
  element.style.height = `${element.scrollHeight + 2}px`;
}
function TextArea(props) {
  const {
    id,
    label,
    debounce,
    onInput,
    value = '',
    disabled,
    monospace,
    onFocus,
    onBlur,
    autoResize = true,
    placeholder,
    rows = autoResize ? 1 : 2,
    tooltip
  } = props;
  const [localValue, setLocalValue] = hooks.useState(value);
  const ref = useShowEntryEvent(id);
  const visible = useElementVisible(ref.current);

  /**
   * @type { import('min-dash').DebouncedFunction }
   */
  const handleInputCallback = useDebounce(onInput, debounce);
  const handleInput = newValue => {
    const newModelValue = newValue === '' ? undefined : newValue;
    handleInputCallback(newModelValue);
  };
  const handleLocalInput = e => {
    autoResize && resizeToContents(e.target);
    if (e.target.value === localValue) {
      return;
    }
    setLocalValue(e.target.value);
    handleInput(e.target.value);
  };
  const handleOnBlur = e => {
    const trimmedValue = e.target.value.trim();

    // trim and commit on blur
    onInput(trimmedValue);
    if (onBlur) {
      onBlur(e);
    }
  };
  hooks.useLayoutEffect(() => {
    autoResize && resizeToContents(ref.current);
  }, []);
  hooks.useLayoutEffect(() => {
    visible && autoResize && resizeToContents(ref.current);
  }, [visible]);
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value);
  }, [value]);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-textarea",
    children: [jsxRuntime.jsx("label", {
      for: prefixId$2(id),
      class: "bio-properties-panel-label",
      children: jsxRuntime.jsx(TooltipWrapper, {
        value: tooltip,
        forId: id,
        element: props.element,
        children: label
      })
    }), jsxRuntime.jsx("textarea", {
      ref: ref,
      id: prefixId$2(id),
      name: id,
      spellCheck: "false",
      class: classnames('bio-properties-panel-input', monospace ? 'bio-properties-panel-input-monospace' : '', autoResize ? 'auto-resize' : ''),
      onInput: handleLocalInput,
      onFocus: onFocus,
      onBlur: handleOnBlur,
      placeholder: placeholder,
      rows: rows,
      value: localValue,
      disabled: disabled,
      "data-gramm": "false"
    })]
  });
}

/**
 * @param {object} props
 * @param {object} props.element
 * @param {string} props.id
 * @param {string} props.description
 * @param {boolean} props.debounce
 * @param {string} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {number} props.rows
 * @param {boolean} props.monospace
 * @param {Function} [props.validate]
 * @param {boolean} [props.disabled]
 */
function TextAreaEntry(props) {
  const {
    element,
    id,
    description,
    debounce,
    label,
    getValue,
    setValue,
    rows,
    monospace,
    disabled,
    validate,
    onFocus,
    onBlur,
    placeholder,
    autoResize,
    tooltip
  } = props;
  const globalError = useError(id);
  const [localError, setLocalError] = hooks.useState(null);
  let value = getValue(element);
  hooks.useEffect(() => {
    if (minDash.isFunction(validate)) {
      const newValidationError = validate(value) || null;
      setLocalError(newValidationError);
    }
  }, [value, validate]);
  const onInput = useStaticCallback(newValue => {
    const value = getValue(element);
    let newValidationError = null;
    if (minDash.isFunction(validate)) {
      newValidationError = validate(newValue) || null;
    }
    if (newValue !== value) {
      setValue(newValue, newValidationError);
    }
    setLocalError(newValidationError);
  });
  const error = globalError || localError;
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-entry', error ? 'has-error' : ''),
    "data-entry-id": id,
    children: [jsxRuntime.jsx(TextArea, {
      id: id,
      label: label,
      value: value,
      onInput: onInput,
      onFocus: onFocus,
      onBlur: onBlur,
      rows: rows,
      debounce: debounce,
      monospace: monospace,
      disabled: disabled,
      placeholder: placeholder,
      autoResize: autoResize,
      tooltip: tooltip,
      element: element
    }, element), error && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-error",
      children: error
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}
function isEdited$1(node) {
  return node && !!node.value;
}

// helpers /////////////////

function prefixId$2(id) {
  return `bio-properties-panel-${id}`;
}

function Textfield(props) {
  const {
    debounce,
    disabled = false,
    id,
    label,
    onInput,
    onFocus,
    onBlur,
    placeholder,
    value = '',
    tooltip
  } = props;
  const [localValue, setLocalValue] = hooks.useState(value || '');
  const ref = useShowEntryEvent(id);

  /**
   * @type { import('min-dash').DebouncedFunction }
   */
  const handleInputCallback = useDebounce(onInput, debounce);
  const handleOnBlur = e => {
    const trimmedValue = e.target.value.trim();

    // trim and commit on blur
    onInput(trimmedValue);
    if (onBlur) {
      onBlur(e);
    }
  };
  const handleInput = newValue => {
    const newModelValue = newValue === '' ? undefined : newValue;
    handleInputCallback(newModelValue);
  };
  const handleLocalInput = e => {
    if (e.target.value === localValue) {
      return;
    }
    setLocalValue(e.target.value);
    handleInput(e.target.value);
  };
  hooks.useEffect(() => {
    if (value === localValue) {
      return;
    }
    setLocalValue(value);
  }, [value]);
  return jsxRuntime.jsxs("div", {
    class: "bio-properties-panel-textfield",
    children: [jsxRuntime.jsx("label", {
      for: prefixId$1(id),
      class: "bio-properties-panel-label",
      children: jsxRuntime.jsx(TooltipWrapper, {
        value: tooltip,
        forId: id,
        element: props.element,
        children: label
      })
    }), jsxRuntime.jsx("input", {
      ref: ref,
      id: prefixId$1(id),
      type: "text",
      name: id,
      spellCheck: "false",
      autoComplete: "off",
      disabled: disabled,
      class: "bio-properties-panel-input",
      onInput: handleLocalInput,
      onFocus: onFocus,
      onBlur: handleOnBlur,
      placeholder: placeholder,
      value: localValue
    })]
  });
}

/**
 * @param {Object} props
 * @param {Object} props.element
 * @param {String} props.id
 * @param {String} props.description
 * @param {Boolean} props.debounce
 * @param {Boolean} props.disabled
 * @param {String} props.label
 * @param {Function} props.getValue
 * @param {Function} props.setValue
 * @param {Function} props.onFocus
 * @param {Function} props.onBlur
 * @param {string|import('preact').Component} props.tooltip
 * @param {Function} props.validate
 */
function TextfieldEntry(props) {
  const {
    element,
    id,
    description,
    debounce,
    disabled,
    label,
    getValue,
    setValue,
    validate,
    onFocus,
    onBlur,
    placeholder,
    tooltip
  } = props;
  const globalError = useError(id);
  const [localError, setLocalError] = hooks.useState(null);
  let value = getValue(element);
  hooks.useEffect(() => {
    if (minDash.isFunction(validate)) {
      const newValidationError = validate(value) || null;
      setLocalError(newValidationError);
    }
  }, [value, validate]);
  const onInput = useStaticCallback(newValue => {
    const value = getValue(element);
    let newValidationError = null;
    if (minDash.isFunction(validate)) {
      newValidationError = validate(newValue) || null;
    }
    if (newValue !== value) {
      setValue(newValue, newValidationError);
    }
    setLocalError(newValidationError);
  });
  const error = globalError || localError;
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-entry', error ? 'has-error' : ''),
    "data-entry-id": id,
    children: [jsxRuntime.jsx(Textfield, {
      debounce: debounce,
      disabled: disabled,
      id: id,
      label: label,
      onInput: onInput,
      onFocus: onFocus,
      onBlur: onBlur,
      placeholder: placeholder,
      value: value,
      tooltip: tooltip,
      element: element
    }, element), error && jsxRuntime.jsx("div", {
      class: "bio-properties-panel-error",
      children: error
    }), jsxRuntime.jsx(Description, {
      forId: id,
      element: element,
      value: description
    })]
  });
}
function isEdited(node) {
  return node && !!node.value;
}

// helpers /////////////////

function prefixId$1(id) {
  return `bio-properties-panel-${id}`;
}

const DEFAULT_DEBOUNCE_TIME = 600;

/**
 * Creates a debounced version of a function, delaying its execution based on `debounceDelay`.
 *
 * - If `debounceDelay` is `false`, the function executes immediately without debouncing.
 * - If a number is provided, the function execution is delayed by the given time in milliseconds.
 *
 * @param { Boolean | Number } [debounceDelay=300]
 *
 * @example
 * const debounce = debounceInput();
 * const debouncedFn = debounce(fn);
 *
 * debouncedFn(); // Executes after 300ms (default) if no further calls occur.
 */
function debounceInput(debounceDelay) {
  /**
   * Applies debounce to the provided function, with a previously setup delay.
   *
   * @template { (...args: any[]) => any } T
   *
   * @param {T} fn
   *
   * @return { (...p: Parameters<T>) => any }
   */
  return function debounce(fn) {
    if (debounceDelay === false) {
      return fn;
    }
    var debounceTime = minDash.isNumber(debounceDelay) ? debounceDelay : DEFAULT_DEBOUNCE_TIME;
    return minDash.debounce(fn, debounceTime);
  };
}
debounceInput.$inject = ['config.debounceInput'];

var index$1 = {
  debounceInput: ['factory', debounceInput]
};

/**
 * Add a dragger that calls back the passed function with
 * { event, delta } on drag.
 *
 * @example
 *
 * function dragMove(event, delta) {
 *   // we are dragging (!!)
 * }
 *
 * domElement.addEventListener('dragstart', dragger(dragMove));
 *
 * @param {Function} fn
 * @param {Element} [dragPreview]
 *
 * @return {Function} drag start callback function
 */
function createDragger(fn, dragPreview) {
  let self;
  let startX, startY;

  /** drag start */
  function onDragStart(event) {
    self = this;
    startX = event.clientX;
    startY = event.clientY;

    // (1) prevent preview image
    if (event.dataTransfer) {
      event.dataTransfer.setDragImage(dragPreview || emptyCanvas(), 0, 0);
    }

    // (2) setup drag listeners

    // attach drag + cleanup event
    // we need to do this to make sure we track cursor
    // movements before we reach other drag event handlers,
    // e.g. in child containers.
    document.addEventListener('dragover', onDrag, true);
    document.addEventListener('dragenter', preventDefault, true);
    document.addEventListener('dragend', onEnd);
    document.addEventListener('drop', preventDefault);
  }
  function onDrag(event) {
    const delta = {
      x: event.clientX - startX,
      y: event.clientY - startY
    };

    // call provided fn with event, delta
    return fn.call(self, event, delta);
  }
  function onEnd() {
    document.removeEventListener('dragover', onDrag, true);
    document.removeEventListener('dragenter', preventDefault, true);
    document.removeEventListener('dragend', onEnd);
    document.removeEventListener('drop', preventDefault);
  }
  return onDragStart;
}
function preventDefault(event) {
  event.preventDefault();
  event.stopPropagation();
}
function emptyCanvas() {
  return minDom.domify('<canvas width="0" height="0" />');
}

const noop = () => {};

/**
 * A generic popup component.
 *
 * @param {Object} props
 * @param {string} [props.className]
 * @param {boolean} [props.delayInitialFocus]
 * @param {{top: number, left: number}} [props.position]
 * @param {number} [props.width]
 * @param {number} [props.height]
 * @param {Function} props.onClose
 * @param {Function} [props.onPostActivate]
 * @param {Function} [props.onPostDeactivate]
 * @param {boolean} [props.returnFocus]
 * @param {boolean} [props.closeOnEscape]
 * @param {string} props.title
 * @param {Ref} [ref]
 */
function PopupComponent(props, globalRef) {
  const {
    className,
    delayInitialFocus,
    position,
    width,
    height,
    onClose,
    onPostActivate = noop,
    onPostDeactivate = noop,
    returnFocus = true,
    closeOnEscape = true,
    title
  } = props;
  const focusTrapRef = hooks.useRef(null);
  const localRef = hooks.useRef(null);
  const popupRef = globalRef || localRef;
  const handleKeydown = event => {
    // do not allow keyboard events to bubble
    event.stopPropagation();
    if (closeOnEscape && event.key === 'Escape') {
      onClose();
    }
  };

  // re-activate focus trap on focus
  const handleFocus = () => {
    if (focusTrapRef.current) {
      focusTrapRef.current.activate();
    }
  };
  let style = {};
  if (position) {
    style = {
      ...style,
      top: position.top + 'px',
      left: position.left + 'px'
    };
  }
  if (width) {
    style.width = width + 'px';
  }
  if (height) {
    style.height = height + 'px';
  }
  hooks.useEffect(() => {
    if (popupRef.current) {
      popupRef.current.addEventListener('focusin', handleFocus);
    }
    return () => {
      if (popupRef.current) {
        popupRef.current.removeEventListener('focusin', handleFocus);
      }
    };
  }, [popupRef]);
  hooks.useEffect(() => {
    if (popupRef.current) {
      focusTrapRef.current = focusTrap__namespace.createFocusTrap(popupRef.current, {
        clickOutsideDeactivates: true,
        delayInitialFocus,
        fallbackFocus: popupRef.current,
        onPostActivate,
        onPostDeactivate,
        returnFocusOnDeactivate: returnFocus
      });
      focusTrapRef.current.activate();
    }
    return () => focusTrapRef.current && focusTrapRef.current.deactivate();
  }, [popupRef]);
  return jsxRuntime.jsx("div", {
    "aria-label": title,
    tabIndex: -1,
    ref: popupRef,
    onKeyDown: handleKeydown,
    role: "dialog",
    class: classnames('bio-properties-panel-popup', className),
    style: style,
    children: props.children
  });
}
const Popup = compat.forwardRef(PopupComponent);
Popup.Title = Title;
Popup.Body = Body;
Popup.Footer = Footer;
function Title(props) {
  const {
    children,
    className,
    draggable,
    eventBus,
    title,
    showCloseButton = false,
    closeButtonTooltip = 'Close popup',
    onClose,
    ...rest
  } = props;

  // we can't use state as we need to
  // manipulate this inside dragging events
  const context = hooks.useRef({
    startPosition: null,
    newPosition: null
  });
  const dragPreviewRef = hooks.useRef();
  const titleRef = hooks.useRef();
  const onMove = (event, delta) => {
    cancel(event);
    const {
      x: dx,
      y: dy
    } = delta;
    const newPosition = {
      x: context.current.startPosition.x + dx,
      y: context.current.startPosition.y + dy
    };
    const popupParent = getPopupParent(titleRef.current);
    popupParent.style.top = newPosition.y + 'px';
    popupParent.style.left = newPosition.x + 'px';
    eventBus?.fire('feelPopup.dragover', {
      newPosition,
      delta
    });
  };
  const onMoveStart = event => {
    // initialize drag handler
    const onDragStart = createDragger(onMove, dragPreviewRef.current);
    onDragStart(event);
    event.stopPropagation();
    const popupParent = getPopupParent(titleRef.current);
    const bounds = popupParent.getBoundingClientRect();
    context.current.startPosition = {
      x: bounds.left,
      y: bounds.top
    };
    eventBus?.fire('feelPopup.dragstart');
  };
  const onMoveEnd = () => {
    context.current.newPosition = null;
    eventBus?.fire('feelPopup.dragend');
  };
  return jsxRuntime.jsxs("div", {
    class: classnames('bio-properties-panel-popup__header', draggable && 'draggable', className),
    ref: titleRef,
    draggable: draggable,
    onDragStart: onMoveStart,
    onDragEnd: onMoveEnd,
    ...rest,
    children: [draggable && jsxRuntime.jsxs(jsxRuntime.Fragment, {
      children: [jsxRuntime.jsx("div", {
        ref: dragPreviewRef,
        class: "bio-properties-panel-popup__drag-preview"
      }), jsxRuntime.jsx("div", {
        class: "bio-properties-panel-popup__drag-handle",
        children: jsxRuntime.jsx(DragIcon, {})
      })]
    }), jsxRuntime.jsx("div", {
      class: "bio-properties-panel-popup__title",
      children: title
    }), children, showCloseButton && jsxRuntime.jsx("button", {
      title: closeButtonTooltip,
      class: "bio-properties-panel-popup__close",
      onClick: onClose,
      children: jsxRuntime.jsx(CloseIcon, {})
    })]
  });
}
function Body(props) {
  const {
    children,
    className,
    ...rest
  } = props;
  return jsxRuntime.jsx("div", {
    class: classnames('bio-properties-panel-popup__body', className),
    ...rest,
    children: children
  });
}
function Footer(props) {
  const {
    children,
    className,
    ...rest
  } = props;
  return jsxRuntime.jsx("div", {
    class: classnames('bio-properties-panel-popup__footer', className),
    ...rest,
    children: props.children
  });
}

// helpers //////////////////////

function getPopupParent(node) {
  return node.closest('.bio-properties-panel-popup');
}
function cancel(event) {
  event.preventDefault();
  event.stopPropagation();
}

const FEEL_POPUP_WIDTH = 700;
const FEEL_POPUP_HEIGHT = 250;

/**
 * FEEL expression editor popup component
 * @param {FeelPopupProps} props
 */
function FeelPopup$1(props) {
  const {
    entryId,
    onInput,
    onClose,
    title,
    type,
    value,
    links,
    variables,
    position,
    hostLanguage,
    singleLine,
    sourceElement,
    tooltipContainer,
    eventBus,
    feelLanguageContext
  } = props;
  const editorRef = hooks.useRef();
  const popupRef = hooks.useRef();
  const isAutoCompletionOpen = hooks.useRef(false);
  const handleSetReturnFocus = () => {
    sourceElement && sourceElement.focus();
  };
  const onKeyDownCapture = event => {
    // we use capture here to make sure we handle the event before the editor does
    if (event.key === 'Escape') {
      isAutoCompletionOpen.current = autoCompletionOpen(event.target);
    }
  };
  const onKeyDown = event => {
    if (event.key === 'Escape') {
      // close popup only if auto completion is not open
      if (!isAutoCompletionOpen.current) {
        onClose();
        isAutoCompletionOpen.current = false;
      }
    }
  };
  hooks.useEffect(() => {
    // set focus on editor when popup is opened
    if (editorRef.current) {
      editorRef.current.focus();
    }
  }, [editorRef]);
  return jsxRuntime.jsxs(Popup, {
    className: "bio-properties-panel-feel-popup",
    position: position,
    title: title,
    returnFocus: false,
    closeOnEscape: false,
    delayInitialFocus: false,
    onPostDeactivate: handleSetReturnFocus,
    height: FEEL_POPUP_HEIGHT,
    width: FEEL_POPUP_WIDTH,
    ref: popupRef,
    children: [jsxRuntime.jsx(Popup.Title, {
      title: title,
      eventBus: eventBus,
      showCloseButton: true,
      closeButtonTooltip: "Save and close",
      onClose: onClose,
      draggable: true,
      children: jsxRuntime.jsx(jsxRuntime.Fragment, {
        children: (links || []).map((link, index) => {
          return jsxRuntime.jsxs("a", {
            rel: "noreferrer",
            href: link.href,
            target: "_blank",
            class: "bio-properties-panel-feel-popup__title-link",
            children: [link.title, jsxRuntime.jsx(LaunchIcon, {})]
          }, index);
        })
      })
    }), jsxRuntime.jsx(Popup.Body, {
      children: jsxRuntime.jsxs("div", {
        onKeyDownCapture: onKeyDownCapture,
        onKeyDown: onKeyDown,
        class: "bio-properties-panel-feel-popup__body",
        children: [type === 'feel' && jsxRuntime.jsx(FeelEditor, {
          enableGutters: true,
          id: prefixId(entryId),
          name: entryId,
          onInput: onInput,
          value: value,
          variables: variables,
          feelLanguageContext: feelLanguageContext,
          ref: editorRef,
          tooltipContainer: tooltipContainer
        }), type === 'feelers' && jsxRuntime.jsx(TemplatingEditor, {
          id: prefixId(entryId),
          contentAttributes: {
            'aria-label': title
          },
          enableGutters: true,
          hostLanguage: hostLanguage,
          name: entryId,
          onInput: onInput,
          value: value,
          ref: editorRef,
          singleLine: singleLine,
          tooltipContainer: tooltipContainer
        })]
      })
    })]
  });
}
function prefixId(id) {
  return `bio-properties-panel-${id}`;
}
function autoCompletionOpen(element) {
  const editor = element.closest('.cm-editor');
  return editor ? editor.querySelector('.cm-tooltip-autocomplete') : null;
}

function getPopupTitle({
  element,
  label
}) {
  let popupTitle = '';
  if (element && element.type) {
    popupTitle = `${element.type} / `;
  }
  return `${popupTitle}${label}`;
}
function getPopupPosition() {
  const viewportWidth = window.innerWidth || document.documentElement.clientWidth;
  const viewportHeight = window.innerHeight || document.documentElement.clientHeight;
  return {
    left: Math.max(0, (viewportWidth - FEEL_POPUP_WIDTH) / 2),
    top: Math.max(0, (viewportHeight - FEEL_POPUP_HEIGHT) / 2)
  };
}

/**
 * FEEL popup component, built as a singleton.
 *
 * In order to implement a custom replacement, handle the following events:
 * - `propertiesPanel.openPopup`
 * - `propertiesPanel.closePopup`
 * - `propertiesPanel.detach`
 *
 * Within the custom renderer, make sure to emit the following events:
 *  - `feelPopup.open` - fired before the popup is mounted
 *  - `feelPopup.opened` - fired after the popup is mounted. Event context contains the DOM node of the popup as `domNode`
 *  - `feelPopup.close` - fired before the popup is unmounted. Event context contains the DOM node of the popup as `domNode`
 *  - `feelPopup.closed` - fired after the popup is unmounted
 */
class FeelPopup {
  constructor(eventBus, config = {}) {
    this._eventBus = eventBus;
    this._config = config;
    this._isOpen = false;
    eventBus.on('propertiesPanel.openPopup', (_, context) => {
      this.open(context.entryId, context, context.sourceElement);

      // return true to indicate that popup was opened
      return true;
    });
    eventBus.on(['propertiesPanel.closePopup', 'propertiesPanel.detach'], () => {
      this.close();
    });
  }

  /**
   * Check if the FEEL popup is open.
   * @return {Boolean}
   */
  isOpen() {
    return this._isOpen;
  }

  /**
   * Open the FEEL popup.
   *
   * @param {String} entryId
   * @param {Object} popupConfig
   * @param {HTMLElement} sourceElement
   */
  open(entryId, popupConfig, sourceElement) {
    // close before opening a new one
    this.close();
    this._openPopup({
      ...popupConfig,
      entryId,
      sourceElement
    });
  }

  /**
   * Close the FEEL popup.
   */
  close() {
    this._closePopup();
  }
  _openPopup(context) {
    const {
      element,
      label,
      sourceElement,
      type
    } = context;
    this._isOpen = true;
    this._eventBus.fire('propertiesPanelPopup.open', {
      container: this._config.feelPopupContainer,
      config: {
        ...context,
        links: this._config.getFeelPopupLinks?.(type) || [],
        onClose: () => {
          this._closePopup();

          // setTimeout to ensure the focus happens after the DOM updates make it focusable
          setTimeout(() => {
            sourceElement && sourceElement.focus();
          }, 0);
        },
        position: getPopupPosition(),
        title: getPopupTitle({
          element,
          label
        })
      }
    });
  }
  _closePopup() {
    if (this._isOpen) {
      this._isOpen = false;
      this._eventBus.fire('propertiesPanelPopup.close');
    }
  }
}
FeelPopup.$inject = ['eventBus', 'config.propertiesPanel'];

class FeelPopupRenderer {
  constructor(eventBus) {
    this._eventBus = eventBus;
    this._container = null;
    this._element = null;
    eventBus.on('propertiesPanelPopup.open', context => {
      this._renderPopup(context);
    });
    eventBus.on('propertiesPanelPopup.close', () => {
      this._removePopup();
    });
  }
  _renderPopup(context) {
    let {
      container,
      config
    } = context;
    container = this._container = getContainer(container) || document.body;
    const element = this._element = createElement();
    container.appendChild(element);

    // TODO(philippfromme): there is no useService in this context, so we need
    // to pass the event bus as a prop or create a context provider, however,
    // a custom renderer would have to use that context provider as well to have
    // access to the event bus and other services
    this._emit('feelPopup.open');
    preact.render(jsxRuntime.jsx(FeelPopup$1, {
      ...config,
      eventBus: this._eventBus
    }), element);
    this._emit('feelPopup.opened', {
      domNode: element
    });
  }
  _removePopup() {
    if (!this._container) {
      return;
    }
    this._emit('feelPopup.close', {
      domNode: this._element
    });
    preact.render(null, this._element);
    this._container.removeChild(this._element);
    this._container = null;
    this._emit('feelPopup.closed');
  }
  _emit(event, context) {
    this._eventBus.fire(event, context);
  }
}
FeelPopupRenderer.$inject = ['eventBus'];

// helpers /////////////////

function createElement() {
  const element = document.createElement('div');
  element.classList.add('bio-properties-panel-popup-container');
  return element;
}
function getContainer(container) {
  if (minDash.isString(container)) {
    return minDom.query(container);
  }
  return container;
}

var index = {
  __init__: ['feelPopup', 'feelPopupRenderer'],
  feelPopup: ['type', FeelPopup],
  feelPopupRenderer: ['type', FeelPopupRenderer]
};

exports.ArrowIcon = ArrowIcon;
exports.CheckboxEntry = CheckboxEntry;
exports.CloseIcon = CloseIcon;
exports.CollapsibleEntry = CollapsibleEntry;
exports.CreateIcon = CreateIcon;
exports.DebounceInputModule = index$1;
exports.DeleteIcon = DeleteIcon;
exports.DescriptionContext = DescriptionContext;
exports.DescriptionEntry = Description;
exports.DragIcon = DragIcon;
exports.DropdownButton = DropdownButton;
exports.ErrorsContext = ErrorsContext;
exports.EventContext = EventContext;
exports.ExternalLinkIcon = ExternalLinkIcon;
exports.FeelCheckboxEntry = FeelCheckboxEntry;
exports.FeelEntry = FeelEntry;
exports.FeelIcon = FeelIcon$1;
exports.FeelLanguageContext = FeelLanguageContext;
exports.FeelNumberEntry = FeelNumberEntry;
exports.FeelPopupModule = index;
exports.FeelTemplatingEntry = FeelTemplatingEntry;
exports.FeelTextAreaEntry = FeelTextAreaEntry;
exports.FeelToggleSwitchEntry = FeelToggleSwitchEntry;
exports.Group = Group;
exports.Header = Header;
exports.HeaderButton = HeaderButton;
exports.LaunchIcon = LaunchIcon;
exports.LayoutContext = LayoutContext;
exports.ListEntry = List;
exports.ListGroup = ListGroup;
exports.ListItem = ListItem;
exports.NumberFieldEntry = NumberFieldEntry;
exports.OpenPopupIcon = OpenPopupIcon;
exports.Placeholder = Placeholder;
exports.PropertiesPanel = PropertiesPanel;
exports.PropertiesPanelContext = LayoutContext;
exports.SelectEntry = SelectEntry;
exports.SimpleEntry = Simple;
exports.TemplatingEntry = TemplatingEntry;
exports.TextAreaEntry = TextAreaEntry;
exports.TextFieldEntry = TextfieldEntry;
exports.ToggleSwitchEntry = ToggleSwitchEntry;
exports.TooltipContext = TooltipContext;
exports.TooltipEntry = TooltipWrapper;
exports.isCheckboxEntryEdited = isEdited$8;
exports.isFeelEntryEdited = isEdited$5;
exports.isNumberFieldEntryEdited = isEdited$6;
exports.isSelectEntryEdited = isEdited$3;
exports.isSimpleEntryEdited = isEdited$2;
exports.isTemplatingEntryEdited = isEdited$4;
exports.isTextAreaEntryEdited = isEdited$1;
exports.isTextFieldEntryEdited = isEdited;
exports.isToggleSwitchEntryEdited = isEdited$7;
exports.useDebounce = useDebounce;
exports.useDescriptionContext = useDescriptionContext;
exports.useElementVisible = useElementVisible;
exports.useError = useError;
exports.useErrors = useErrors;
exports.useEvent = useEvent;
exports.useKeyFactory = useKeyFactory;
exports.useLayoutState = useLayoutState;
exports.usePrevious = usePrevious;
exports.useShowEntryEvent = useShowEntryEvent;
exports.useStaticCallback = useStaticCallback;
exports.useStickyIntersectionObserver = useStickyIntersectionObserver;
exports.useTooltipContext = useTooltipContext;
//# sourceMappingURL=index.js.map
