/*
Copyright 2020-2025 Bonitasoft S.A.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/// <reference types="@typed-mxgraph/typed-mxgraph" />

import type { mxAbstractCanvas2D } from 'mxgraph';
import type { mxCellRenderer } from 'mxgraph';
import { mxGraphExportObject } from 'mxgraph';

/**
 * Enum values are using the real name of the field in the BPMN XSD.
 * @category BPMN
 */
export declare enum AssociationDirectionKind {
    NONE = "None",
    ONE = "One",
    BOTH = "Both"
}

/**
 * @category Custom Behavior
 */
export declare interface BaseBpmnSemantic {
    id: string;
    /** `true` when relates to a BPMN Shape, `false` when relates to a BPMN Edge. */
    isShape: boolean;
    kind: BpmnElementKind;
    name: string;
}

/**
 * Wrapper of `mxAbstractCanvas2D` to simplify method calls when painting icons/markers of BPMN shapes.
 *
 * It can scale dimensions passed to the method of the original `mxAbstractCanvas2D`.
 *
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @example
 * The vanilla canvas calls when a scale factor must be applied to position
 * ```javascript
 * const scaleX = 0.26;
 * const scaleY = 0.35;
 * c.moveTo(8 * scaleX, 39 * scaleY);
 * c.lineTo(12 * scaleX, 25 * scaleY);
 * ```
 *
 * @example
 * With `BpmnCanvas`
 * ```javascript
 * const canvas = new BpmnCanvas(c, 0.26, 0.35);
 * canvas.moveTo(8, 39);
 * canvas.lineTo(12, 25);
 * ```
 *
 * @category BPMN Theme
 * @experimental
 */
export declare class BpmnCanvas {
    private readonly canvas;
    private readonly iconOriginalSize;
    private readonly scaleX;
    private readonly scaleY;
    private iconPaintingOriginX;
    private iconPaintingOriginY;
    private readonly shapeConfiguration;
    constructor({ canvas, shapeConfig, iconConfig }: BpmnCanvasConfiguration);
    /**
     * Set the icon origin to the top left corner of the shape.
     */
    setIconOriginToShapeTopLeft(topMargin?: number, leftMargin?: number): void;
    /**
     * Set the icon origin to ensure that the icon is centered on the shape.
     */
    setIconOriginForIconCentered(): void;
    /**
     * Set the icon origin to ensure that, on the shape, the icon is horizontally centered and vertically aligned to the bottom.
     */
    setIconOriginForIconBottomCentered(bottomMargin?: number): void;
    /**
     * Translate the icon origin using the scale factor associated to the horizontal and vertical directions.
     *
     * The values should be given with using the icon original size (as translated values will be scaled as other values passed to method of this class).
     *
     * @param dx the horizontal translation
     * @param dy the vertical translation
     */
    translateIconOrigin(dx: number, dy: number): void;
    private computeScaleFromOriginX;
    private computeScaleFromOriginY;
    private updateCanvasStyle;
    arcTo(rx: number, ry: number, angle: number, largeArcFlag: number, sweepFlag: number, x: number, y: number): void;
    begin(): void;
    close(): void;
    curveTo(x1: number, y1: number, x2: number, y2: number, x3: number, y3: number): void;
    fill(): void;
    fillAndStroke(): void;
    setFillColor(fillColor: string): void;
    stroke(): void;
    setStrokeColor(color: string): void;
    setRoundLineJoin(): void;
    lineTo(x: number, y: number): void;
    moveTo(x: number, y: number): void;
    rect(x: number, y: number, w: number, h: number): void;
    roundrect(x: number, y: number, w: number, h: number, dx: number, dy: number): void;
    ellipse(x: number, y: number, w: number, h: number): void;
    rotateOnIconCenter(theta: number): void;
}

/**
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare interface BpmnCanvasConfiguration {
    canvas: mxAbstractCanvas2D;
    shapeConfig: ShapeConfiguration;
    iconConfig: IconConfiguration;
}

/**
 * @category Custom Behavior
 */
export declare interface BpmnElement {
    bpmnSemantic: BpmnSemantic;
    htmlElement: HTMLElement;
}

/**
 * @category BPMN
 */
export declare type BpmnElementKind = FlowKind | ShapeBpmnElementKind;

/**
 * `BpmnElementRegistry` is a public API that permits to find the BpmnElements present in the diagram.
 * How to access it:
 *
 * ```javascript
 * // 1. Initialize the BpmnVisualization.
 * const bpmnVisualization = new BpmnVisualization({ container: 'bpmn-container' });
 * // 2. Get diagram and load it.
 * const bpmn = 'BPMN diagram string - whether coming from bpmn.xml file or some API call';
 * bpmnVisualization.load(bpmn);
 * // 3. Access registry directly from bpmnVisualization.
 * bpmnVisualization.bpmnElementsRegistry
 * ```
 *
 * **WARN**: subject to change, feedback welcome.
 *
 *  @category Custom Behavior
 *  @experimental
 */
export declare class BpmnElementsRegistry implements CssClassesRegistry, ElementsRegistry, OverlaysRegistry, StyleRegistry {
    private readonly bpmnModelRegistry;
    private readonly htmlElementRegistry;
    private readonly cssClassesRegistry;
    private readonly overlaysRegistry;
    private readonly styleRegistry;
    getModelElementsByIds(bpmnElementIds: string | string[]): BpmnSemantic[];
    getElementsByIds(bpmnElementIds: string | string[]): BpmnElement[];
    getModelElementsByKinds(bpmnKinds: BpmnElementKind | BpmnElementKind[]): BpmnSemantic[];
    getElementsByKinds(bpmnKinds: BpmnElementKind | BpmnElementKind[]): BpmnElement[];
    private getRelatedBpmnSemantic;
    addCssClasses(bpmnElementIds: string | string[], classNames: string | string[]): void;
    removeCssClasses(bpmnElementIds: string | string[], classNames: string | string[]): void;
    removeAllCssClasses(bpmnElementIds?: string | string[]): void;
    toggleCssClasses(bpmnElementIds: string | string[], classNames: string | string[]): void;
    addOverlays(bpmnElementId: string, overlays: Overlay | Overlay[]): void;
    removeAllOverlays(bpmnElementId: string): void;
    updateStyle(bpmnElementIds: string | string[], styleUpdate: StyleUpdate): void;
    resetStyle(bpmnElementIds?: string | string[]): void;
}

/**
 * {@link ShapeBpmnElementKind} related to BPMN Events.
 * @category BPMN
 */
export declare type BpmnEventKind = ShapeBpmnElementKind.EVENT_BOUNDARY | ShapeBpmnElementKind.EVENT_START | ShapeBpmnElementKind.EVENT_END | ShapeBpmnElementKind.EVENT_INTERMEDIATE_THROW | ShapeBpmnElementKind.EVENT_INTERMEDIATE_CATCH;

declare class BpmnGraph extends mxgraph.mxGraph {
    private currentZoomLevel;
    createCellRenderer(): mxCellRenderer;
    /**
     * Shortcut for an update of the model within a transaction.
     *
     * This method is inspired from {@link https://github.com/maxGraph/maxGraph/blob/v0.1.0/packages/core/src/view/Graph.ts#L487-L494 maxGraph}.
     *
     * @param callbackFunction the update to be made in the transaction.
     *
     * @experimental subject to change, may move to a subclass of `mxGraphModel`
     * @alpha
     */
    batchUpdate(callbackFunction: () => void): void;
    private setCurrentZoomLevel;
    private manageMouseWheelZoomEvent;
    private createMouseWheelZoomListener;
    private getEventRelativeCoordinates;
    private getScaleAndTranslationDeltas;
    private calculateTranslationDeltas;
    private calculateFactorAndScale;
}

/**
 * @category Custom Behavior
 */
export declare type BpmnSemantic = EdgeBpmnSemantic | ShapeBpmnSemantic;

/**
 * Define BPMN specific keys used in mxGraph styles. Use constants defined in this class instead of hard coded string values.
 *
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare const BpmnStyleIdentifier: {
    EDGE: string;
    EDGE_START_FILL_COLOR: string;
    EDGE_END_FILL_COLOR: string;
    EVENT_BASED_GATEWAY_KIND: string;
    EVENT_DEFINITION_KIND: string;
    GLOBAL_TASK_KIND: string;
    SUB_PROCESS_KIND: string;
    IS_INITIATING: string;
    IS_INSTANTIATING: string;
    IS_INTERRUPTING: string;
    EXTRA_CSS_CLASSES: string;
    MARKERS: string;
    MESSAGE_FLOW_ICON: string;
};

/**
 * Let initialize `bpmn-visualization`. It requires at minimum to pass the HTMLElement in the page where the BPMN diagram is rendered.
 * ```javascript
 * const bpmnVisualization = new BpmnVisualization({ container: 'bpmn-container' });
 * ```
 * For more options, see {@link GlobalOptions}
 *
 * @category Initialization & Configuration
 */
export declare class BpmnVisualization {
    /**
     * Direct access to the `mxGraph` instance that powers `bpmn-visualization`.
     * It is for **advanced users**, so please use the lib API first and access to the `mxGraph` instance only when there is no alternative.
     *
     * **WARN**: subject to change, could be removed or made available in another way.
     *
     * @experimental
     */
    readonly graph: BpmnGraph;
    /**
     * Perform BPMN diagram navigation.
     *
     * **WARN**: subject to change, feedback welcome.
     *
     * @experimental
     * @since 0.24.0
     */
    readonly navigation: Navigation;
    /**
     * Interact with BPMN diagram elements rendered in the page.
     *
     * **WARN**: subject to change, feedback welcome.
     *
     * @experimental
     */
    readonly bpmnElementsRegistry: BpmnElementsRegistry;
    private readonly bpmnModelRegistry;
    private readonly parserOptions;
    private readonly rendererOptions;
    constructor(options: GlobalOptions);
    /**
     * Load and render the BPMN diagram.
     * @param xml The BPMN content as xml string
     * @param options Let decide how to load the model and render the diagram
     * @throws `Error` when loading fails. This is generally due to a parsing error caused by a malformed BPMN content
     */
    load(xml: string, options?: LoadOptions): void;
}

/**
 * @category Element Style
 */
export declare interface CssClassesRegistry {
    /**
     * Add one or more CSS classes to one or more BPMN elements.
     *
     * **Notes**:
     *
     * - If an ID is passed that does not reference an existing BPMN element, its reference is retained in the registry, but no rendering changes are made.
     * - This method is intended to set CSS classes on specific elements, e.g. to hide or highlight them. During BPMN diagram rendering, `bpmn-visualization` sets specific CSS classes to all elements based on their types.
     * - To style all elements of a given type, use the default classes instead of adding new ones. The classes allow identification of elements of the same `family' and of the same specific type.
     * - For example, a BPMN Service Task is an `Activity` and a `Task`. So it has the `bpmn-type-activity` and the `bpmn-type-task` classes. It shares these classes with all types of `Tasks`.
     * - It also has the specific `bpmn-service-task` to distinguish it from a BPMN User Task which has a `bpmn-user-task`.
     * - In addition, labels also have the `bpmn-label` classes.
     *
     * See the repository providing the [examples of the `bpmn-visualization` TypeScript library](https://github.com/process-analytics/bpmn-visualization-examples/) for more details.
     *
     * @example
     * ```javascript
     * // Add 'success-path' to BPMN elements with id: flow_1 and flow_5
     * bpmnVisualization.bpmnElementsRegistry.addCssClasses(['flow_1', 'flow_5'], 'success-path');
     *
     * // Add 'suspicious-path' and 'additional-info' to BPMN element with id: task_3
     * bpmnVisualization.bpmnElementsRegistry.addCssClasses('task_3', ['suspicious-path', 'additional-info']);
     * ```
     *
     * @param bpmnElementIds The BPMN ID of the element(s) to add the CSS classes to. Passing a nullish parameter or an empty array has no effect.
     * @param classNames The name of the class(es) to add to the BPMN element(s).
     *
     * @see {@link removeCssClasses} to remove specific CSS classes from a BPMN element.
     * @see {@link removeAllCssClasses} to remove all CSS classes from a BPMN element.
     * @see {@link toggleCssClasses} to toggle CSS classes on a BPMN element.
     * @see {@link StyleRegistry#updateStyle} to directly update the style of BPMN elements.
     */
    addCssClasses(bpmnElementIds: string | string[], classNames: string | string[]): void;
    /**
     * Remove one or more CSS classes that were previously added to one or more BPMN elements using the {@link addCssClasses} or the {@link toggleCssClasses} methods.
     *
     * **Note**: If you pass IDs that are not related to existing BPMN elements, they will be ignored and no changes will be made to the rendering.
     *
     * @example
     * ```javascript
     * // Remove 'highlight' from BPMN elements with ID: activity_1 and activity_2
     * bpmnVisualization.bpmnElementsRegistry.removeCssClasses(['activity_1', 'activity_2'], 'highlight');
     *
     * // Remove 'running' and 'additional-info' from BPMN element with ID: task_3
     * bpmnVisualization.bpmnElementsRegistry.removeCssClasses('task_3', ['running', 'additional-info']);
     * ```
     *
     * @param bpmnElementIds The BPMN ID of the element(s) from which to remove the CSS classes. Passing a nullish parameter or an empty array has no effect.
     * @param classNames The name of the class(es) to remove from the BPMN element(s).
     *
     * @see {@link removeAllCssClasses} to remove all CSS classes from a BPMN element.
     */
    removeCssClasses(bpmnElementIds: string | string[], classNames: string | string[]): void;
    /**
     * Remove any CSS classes that were previously added to one or more BPMN elements using the {@link addCssClasses} or the {@link toggleCssClasses} methods.
     *
     * **Note**: If you pass IDs that are not related to existing BPMN elements, they will be ignored and no changes will be made to the rendering.
     *
     * @example
     * ```javascript
     * // Remove all CSS classes from all BPMN elements
     * bpmnVisualization.bpmnElementsRegistry.removeAllCssClasses();
     *
     * // Remove all CSS classes from BPMN elements with ID: activity_1 and activity_2
     * bpmnVisualization.bpmnElementsRegistry.removeAllCssClasses(['activity_1', 'activity_2']);
     *
     * // Remove all CSS classes from BPMN element with ID: task_3
     * bpmnVisualization.bpmnElementsRegistry.removeAllCssClasses('task_3');
     * ```
     *
     * @param bpmnElementIds The BPMN ID of the element(s) from which to remove all CSS classes.
     * When passing a nullish parameter, all CSS classes associated with all BPMN elements will be removed. Passing an empty array has no effect.
     *
     * @see {@link removeCssClasses} to remove specific classes from a BPMN element.
     * @since 0.34.0
     */
    removeAllCssClasses(bpmnElementIds?: string | string[]): void;
    /**
     * Toggle one or more CSS classes on one or more BPMN elements.
     *
     * **Note**: If an ID is passed that does not reference an existing BPMN element, its reference is retained in the registry, but no rendering changes are made.
     *
     * @example
     * ```javascript
     * // Toggle 'highlight' for BPMN elements with ID: activity_1 and activity_2
     * bpmnVisualization.bpmnElementsRegistry.toggleCssClasses(['activity_1', 'activity_2'], 'highlight');
     *
     * // Toggle 'running' and 'additional-info' for BPMN element with ID: task_3
     * bpmnVisualization.bpmnElementsRegistry.toggleCssClasses('task_3', ['running', 'additional-info']);
     * ```
     *
     * @param bpmnElementIds The BPMN ID of the element(s) on which to toggle the CSS classes. Passing a nullish parameter or an empty array has no effect.
     * @param classNames The name of the class(es) to toggle on the BPMN element(s).
     *
     * @see {@link removeCssClasses} to remove specific CSS classes from a BPMN element.
     * @see {@link removeAllCssClasses} to remove all CSS classes from a BPMN element.
     * @see {@link addCssClasses} to add CSS classes to a BPMN element.
     */
    toggleCssClasses(bpmnElementIds: string | string[], classNames: string | string[]): void;
}

/**
 * Extended properties available when {@link BaseBpmnSemantic.isShape} is `false`.
 * @category Custom Behavior
 */
export declare interface EdgeBpmnSemantic extends BaseBpmnSemantic {
    sourceRefId: string;
    targetRefId: string;
}

/**
 * @category Element Style
 */
export declare type EdgeStyleUpdate = {
    font?: Font;
    /**
     * The value must be between 0 and 100:
     * - If the set value is less than 0, the used value is 0.
     * - If the set value is greater than 100, the used value is 100.
     *
     * **NOTE**: `opacity` does not apply to the font style.
     *
     * **Notes about the `default` special keyword**:
     * - It is used to apply the opacity defined in the default style of the BPMN element.
     * - It can be used when the style is first updated and then needs to be reset to its initial value.
     */
    opacity?: Opacity;
    stroke?: Stroke;
};

/**
 * @category Custom Behavior
 */
export declare interface ElementsRegistry {
    /**
     * Get all model elements in the form of {@link BpmnSemantic} objects corresponding to the identifiers supplied. The returned array contains elements in the order of the `bpmnElementIds` parameter.
     *
     * Not found elements are not returned as undefined in the array, so the returned array contains at most as many elements as the `bpmnElementIds` parameter.
     *
     * ```javascript
     * ...
     * // Find all elements by specified id or ids
     * const bpmnElements1 = bpmnVisualization.bpmnElementsRegistry.getModelElementsByIds('userTask_1');
     * const bpmnElements2 = bpmnVisualization.bpmnElementsRegistry.getModelElementsByIds(['startEvent_3', 'userTask_2']);
     * // now you can do whatever you want with the elements
     * ...
     * ```
     *
     * If you also need to retrieve the related DOM elements and more information, use {@link getElementsByIds} instead.
     *
     * @param bpmnElementIds The BPMN ID of the element(s) to retrieve.
     * @since 0.39.0
     */
    getModelElementsByIds(bpmnElementIds: string | string[]): BpmnSemantic[];
    /**
     * Get all elements by ids. The returned array contains elements in the order of the `bpmnElementIds` parameter.
     *
     * Not found elements are not returned as undefined in the array, so the returned array contains at most as many elements as the `bpmnElementIds` parameter.
     *
     * ```javascript
     * ...
     * // Find all elements by specified id or ids
     * const bpmnElements1 = bpmnVisualization.bpmnElementsRegistry.getElementsByIds('userTask_1');
     * const bpmnElements2 = bpmnVisualization.bpmnElementsRegistry.getElementsByIds(['startEvent_3', 'userTask_2']);
     * // now you can do whatever you want with the elements
     * ...
     * ```
     *
     * **WARNING**: this method is not designed to accept a large amount of ids. It does DOM lookup to retrieve the HTML elements relative to the BPMN elements.
     * Attempts to retrieve too many elements, especially on large BPMN diagram, may lead to performance issues.
     *
     * If you only need to retrieve the BPMN model data, use {@link getModelElementsByIds} instead.
     *
     * @param bpmnElementIds The BPMN ID of the element(s) to retrieve.
     */
    getElementsByIds(bpmnElementIds: string | string[]): BpmnElement[];
    /**
     * Get all model elements in the form of {@link BpmnSemantic} objects corresponding to the BPMN kinds supplied
     *
     * ```javascript
     * ...
     * // Find all elements by desired kind or kinds
     * const bpmnElements1 = bpmnVisualization.bpmnElementsRegistry.getModelElementsByKinds(ShapeBpmnElementKind.TASK);
     * const bpmnElements2 = bpmnVisualization.bpmnElementsRegistry.getModelElementsByKinds([ShapeBpmnElementKind.EVENT_END, ShapeBpmnElementKind.POOL]);
     * // now you can do whatever you want with the elements
     * ...
     * ```
     *
     * If you also need to retrieve the related DOM elements and more information, use {@link getElementsByKinds} instead.
     *
     * @param bpmnKinds The BPMN kind of the element(s) to retrieve.
     * @since 0.39.0
     */
    getModelElementsByKinds(bpmnKinds: BpmnElementKind | BpmnElementKind[]): BpmnSemantic[];
    /**
     * Get all elements by kinds.
     *
     * ```javascript
     * ...
     * // Find all elements by desired type or types
     * const bpmnTaskElements = bpmnVisualization.bpmnElementsRegistry.getElementsByKinds(ShapeBpmnElementKind.TASK);
     * const bpmnEndEventAndPoolElements = bpmnVisualization.bpmnElementsRegistry.getElementsByKinds([ShapeBpmnElementKind.EVENT_END, ShapeBpmnElementKind.POOL]);
     * // now you can do whatever you want with the elements
     * ...
     * ```
     *
     * If you only need to retrieve the BPMN model data, use {@link getModelElementsByKinds} instead.
     *
     * **WARNING**: this method is not designed to accept a large amount of types. It does DOM lookup to retrieve the HTML elements relative to the BPMN elements.
     * Attempts to retrieve too many elements, especially on large BPMN diagrams, may lead to performance issues.
     *
     * @param bpmnKinds The BPMN kind of the element(s) to retrieve.
     */
    getElementsByKinds(bpmnKinds: BpmnElementKind | BpmnElementKind[]): BpmnElement[];
}

/**
 * @category Element Style
 */
export declare type Fill = StyleWithOpacity & {
    /**
     * Possible values are all HTML color names, HEX codes, {@link FillColorGradient}, as well as special keywords such as:
     * - `default` to use the color defined in the BPMN element default style.
     * - `inherit` to apply the fill color of the direct parent element.
     * - `none` for no color.
     * - `swimlane` to apply the fill color of the nearest parent element with the type {@link ShapeBpmnElementKind.LANE} or {@link ShapeBpmnElementKind.POOL}.
     *
     * **Notes about the `default` special keyword**:
     * - It can be used when the style is first updated and then needs to be reset to its initial value.
     * - It doesn't use the color set in the BPMN source when the "BPMN in Color" support is enabled. It uses the color defined in the BPMN element default style.
     * - If a gradient was set, it will be completely reverted.
     */
    color?: FillColorGradient | 'default' | 'inherit' | 'none' | 'swimlane' | string;
};

/**
 * It is a linear gradient managed by `mxGraph`.
 * For more information about mxGraph, refer to the documentation at:
 * {@link https://jgraph.github.io/mxgraph/docs/js-api/files/util/mxConstants-js.html#mxConstants.STYLE_GRADIENTCOLOR}
 *
 * **Notes**:
 * Using the same color for the start color and end color will have the same effect as setting only the fill color with an HTML color name, HEX code or special keyword.
 *
 * @category Element Style
 */
export declare type FillColorGradient = {
    /**
     * It can be any HTML color name or HEX code, as well as special keywords such as:
     * - `inherit` to apply the fill color of the direct parent element.
     * - `none` for no color.
     * - `swimlane` to apply the fill color of the nearest parent element with the type {@link ShapeBpmnElementKind.LANE} or {@link ShapeBpmnElementKind.POOL}.
     */
    startColor: 'inherit' | 'none' | 'swimlane' | string;
    /**
     * It can be any HTML color name or HEX code, as well as special keywords such as:
     * - `inherit` to apply the fill color of the direct parent element.
     * - `none` for no color.
     * - `swimlane` to apply the fill color of the nearest parent element with the type {@link ShapeBpmnElementKind.LANE} or {@link ShapeBpmnElementKind.POOL}.
     */
    endColor: 'inherit' | 'none' | 'swimlane' | string;
    /**
     * Specifies how the colors transition within the gradient.
     *
     * Taking the example of `bottom-to-top`, this means that the start color is at the bottom of the paint pattern and the end color is at the top, with a gradient between them.
     *
     * @see {@link GradientDirection}
     */
    direction: GradientDirection;
};

/**
 * @category Initialization & Configuration
 */
export declare interface FitOptions {
    /** @default {@link FitType.None} */
    type?: FitType;
    /**
     * Negative values fallback to default.
     * @default 0
     */
    margin?: number;
}

/**
 * @category Navigation
 */
export declare enum FitType {
    /** No fit, use dimensions and coordinates from the BPMN diagram. */
    None = "None",
    /** Fit the whole html container available to render the BPMN diagram. */
    HorizontalVertical = "HorizontalVertical",
    /** Fit only horizontally. */
    Horizontal = "Horizontal",
    /** Fit only vertically. */
    Vertical = "Vertical",
    /** Fit and center the BPMN Diagram. */
    Center = "Center"
}

/**
 * Enum values are using the real name of the field in the BPMN XSD.
 * @category BPMN
 */
export declare enum FlowKind {
    SEQUENCE_FLOW = "sequenceFlow",
    MESSAGE_FLOW = "messageFlow",
    ASSOCIATION_FLOW = "association"
}

/**
 * Note about properties that can be reset to default values.
 *
 * Except for color (when the "BPMN in Color" support is disabled), all style properties can be set in the BPMN diagram via LabelStyle and can then override the default values.
 * Currently, there is no way to know if they are overridden. So it is not possible to reset each property with the "Update Style" API.
 *
 * @category Element Style
 */
export declare type Font = StyleWithOpacity & {
    /**
     * Possible values are all HTML color names or HEX codes, as well as special keywords such as:
     * - `default` to use the color defined in the BPMN element default style.
     * - `inherit` to apply the font color of the direct parent element.
     * - `swimlane` to apply the font color of the nearest parent element with the type {@link ShapeBpmnElementKind.LANE} or {@link ShapeBpmnElementKind.POOL}.
     *
     * **Notes about the `default` special keyword**:
     * - It can be used when the style is first updated and then needs to be reset to its initial value.
     * - It doesn't use the color set in the BPMN source when the "BPMN in Color" support is enabled. It uses the color defined in the BPMN element default style.
     */
    color?: 'default' | 'inherit' | 'swimlane' | string;
    /**
     *  The type of the value is int (in px).
     */
    size?: number;
    family?: string;
    /**
     *  @default false
     */
    isBold?: boolean;
    /**
     *  @default false
     */
    isItalic?: boolean;
    /**
     *  @default false
     */
    isUnderline?: boolean;
    /**
     *  @default false
     */
    isStrikeThrough?: boolean;
};

/**
 * Returns the version of `bpmn-visualization` and the version of its dependencies.
 * @since 0.43.0
 */
export declare const getVersion: () => Version;

/**
 * Options to configure the `bpmn-visualization` initialization.
 * @category Initialization & Configuration
 */
export declare interface GlobalOptions {
    /** The id of a DOM element or an HTML node where the BPMN diagram is rendered. */
    container: string | HTMLElement;
    /** Configure the BPMN diagram navigation (panning and zoom). */
    navigation?: NavigationConfiguration;
    /** Configure the BPMN parser. */
    parser?: ParserOptions;
    /** Configure how the BPMN diagram and its elements are rendered. */
    renderer?: RendererOptions;
}

/**
 * {@link ShapeBpmnElementKind} related to BPMN Global Tasks.
 * @category BPMN
 */
export declare type GlobalTaskKind = ShapeBpmnElementKind.GLOBAL_TASK | ShapeBpmnElementKind.GLOBAL_TASK_MANUAL | ShapeBpmnElementKind.GLOBAL_TASK_SCRIPT | ShapeBpmnElementKind.GLOBAL_TASK_USER | ShapeBpmnElementKind.GLOBAL_TASK_BUSINESS_RULE;

/**
 * @category Element Style
 */
export declare type GradientDirection = 'left-to-right' | 'right-to-left' | 'bottom-to-top' | 'top-to-bottom';

/**
 * @category BPMN Theme
 */
export declare interface IconConfiguration {
    originalSize: Size;
    /** If `undefined`, no scaling will be done in {@link BpmnCanvas}. */
    ratioFromParent?: number;
    styleConfig: IconStyleConfiguration;
    setIconOriginFunct: (canvas: BpmnCanvas) => void;
}

/**
 * Default implementation for the icons.
 *
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare class IconPainter {
    /**
     * Utility paint icon methods to easily instantiate a {@link BpmnCanvas} from a {@link PaintParameter}.
     *
     * @param canvas                  mxgraph `mxAbstractCanvas2D` in charge of performing the paint operations.
     * @param ratioFromParent         the actual size of the icon will be computed from the shape dimensions using this ratio.
     * @param setIconOriginFunct      called function to set the origin of the icon. Generally, it calls a method of {@link BpmnCanvas}.
     * @param shapeConfig             dimension and style of the shape where the icon is painted.
     * @param iconStyleConfig         style of the icon.
     * @param originalIconSize        original size of the icon used to compute the scaling/ratio in {@link BpmnCanvas}.
     * @protected
     */
    protected newBpmnCanvas({ canvas, ratioFromParent, setIconOriginFunct, shapeConfig, iconStyleConfig }: PaintParameter, originalIconSize: Size): BpmnCanvas;
    /**
     * This icon is used by `message event`, `receive task`, `send task`.
     */
    paintEnvelopeIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `inclusive gateway` and `terminate event`.
     */
    paintCircleIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `timer event`.
     */
    paintClockIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `signal event`.
     */
    paintTriangleIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `escalation event`.
     */
    paintUpArrowheadIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `compensation event`.
     */
    paintDoubleLeftArrowheadsIcon(paintParameter: PaintParameter): void;
    private drawCrossIcon;
    /**
     * This icon is used by `conditional event`.
     */
    paintListIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `exclusive gateway`.
     */
    paintXCrossIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `parallel gateway` and 'event-based gateway'.
     */
    paintPlusCrossIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `complex gateway`.
     */
    paintAsteriskIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `user task`.
     */
    paintPersonIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `service tasks`.
     */
    paintGearIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used to render the `expand marker` on `activities`.
     */
    paintExpandIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used to render the `loop marker` on `activities`.
     */
    paintLoopIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used to render the `sequential multi-instance marker` on `activities`.
     */
    paintSequentialMultiInstanceIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used to render the `parallel multi-instance marker` on `activities`.
     */
    paintParallelMultiInstanceIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `link event`.
     */
    paintRightArrowIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `error event`.
     */
    paintErrorIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `manual task`.
     */
    paintHandIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `script task`.
     */
    paintScriptIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `business rule task`.
     */
    paintTableIcon(paintParameter: PaintParameter): void;
    /**
     * This icon is used by `event-based gateway`.
     */
    paintPentagon(paintParameter: PaintParameter): void;
}

/**
 * Hold the instance of {@link IconPainter} used by the BPMN Theme.
 *
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare class IconPainterProvider {
    private static instance;
    static get(): IconPainter;
    static set(painter: IconPainter): void;
}

/**
 * @category BPMN Theme
 */
export declare interface IconStyleConfiguration {
    isFilled: boolean;
    fillColor: string;
    strokeColor: string;
    strokeWidth: number;
    margin: number;
}

/**
 * Options when loading a BPMN Diagram.
 * @category Initialization & Configuration
 */
export declare interface LoadOptions {
    fit?: FitOptions;
    modelFilter?: ModelFilter;
}

/**
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare const MarkerIdentifier: {
    ARROW_DASH: string;
};

/**
 * Enum values are using the real name of the `visible message` field in the BPMN XSD, except for `none` that is not present in the specification.
 * @category BPMN
 */
export declare enum MessageVisibleKind {
    NONE = "none",
    INITIATING = "initiating",
    NON_INITIATING = "non_initiating"
}

/**
 * Model filtering configuration.
 *
 * Here is an example of how to perform model filtering when loading a BPMN diagram:
 * ```
 * bpmnVisualization.load(diagram, {
 *   modelFilter: {
 *     pools: [
 *       {
 *         // id of the Participant related to the Pool to display
 *         id: 'id1'
 *       },
 *       {
 *         // Name of the Participant, or name of the Process referenced by the Participant
 *         // when the Participant doesn't have a name.
 *         // This is how `bpmn-visualization` builds the name into its internal model.
 *         name: 'name2'
 *       },
 *       {
 *       id: 'id3',
 *       // in this case, we only use the id, and ignore the name
 *       name: 'name3'
 *       }
 *     ]},
 * });
 * ```
 *
 * @category Initialization & Configuration
 */
export declare interface ModelFilter {
    pools?: PoolFilter | PoolFilter[];
}

/**
 * The `mxgraph` context that allows access to the mxGraph objects.
 *
 * **WARNING**: this is for advanced users.
 *
 * Here are some examples where calling the mxGraph API directly can be useful:
 * ```javascript
 * // Get the mxGraph version
 * const mxGraphVersion = mxgraph.mxClient.VERSION;
 * // Call mxUtils in custom BPMN Shapes
 * c.setFillColor(mxgraph.mxUtils.getValue(this.style, BpmnStyleIdentifier.EDGE_START_FILL_COLOR, this.stroke));
 * ```
 *
 * @since 0.30.0
 */
export declare const mxgraph: mxGraphExportObject;

/**
 * Perform BPMN diagram navigation.
 *
 * **WARN**: subject to change, feedback welcome.
 *
 * @category Navigation
 * @experimental
 * @since 0.24.0
 */
export declare class Navigation {
    private readonly graph;
    constructor(graph: BpmnGraph);
    fit(options?: FitOptions): void;
    zoom(type: ZoomType): void;
}

/**
 * Configure the BPMN diagram navigation (panning and zoom).
 * @category Initialization & Configuration
 */
export declare interface NavigationConfiguration {
    /**
     * Enable the navigation with the mouse wheel or with gesture/pinch on touch devices.
     *
     * **IMPORTANT**: the navigation API is not affected by this value. Navigation actions performed with the API always have an effect.
     * @default false
     */
    enabled: boolean;
    /** Tune how the zoom behaves when using the mouse wheel or with gesture/pinch on touch devices. */
    zoom?: ZoomConfiguration;
}

/**
 *  @category Element Style
 */
export declare type Opacity = 'default' | number;

/**
 * @category Overlays
 */
export declare type Overlay = {
    position: OverlayPosition;
    label?: string;
    style?: OverlayStyle;
};

/**
 * @category Overlays
 */
export declare type OverlayEdgePosition = 'start' | 'middle' | 'end';

/**
 * @category Overlays
 */
export declare type OverlayFill = {
    /** @default {@link StyleDefault.DEFAULT_OVERLAY_FILL_COLOR} */
    color?: string;
    /**
     * A number between `0` (transparent) and `100` (opaque).
     *
     * **IMPORTANT**: this property is currently not taken into account by the default Overlay. See https://github.com/process-analytics/bpmn-visualization-js/issues/1234.
     *
     * To solve this problem, define a color that contains opacity information. For example: `color: `rgba(0, 0, 255, 0.8)`.
     * @default {@link StyleDefault.DEFAULT_OVERLAY_FILL_OPACITY}
     */
    opacity?: number;
};

/**
 * The font family is {@link StyleDefault.DEFAULT_FONT_FAMILY}.
 * @category Overlays
 */
export declare type OverlayFont = {
    /** @default {@link StyleDefault.DEFAULT_OVERLAY_FONT_COLOR} */
    color?: string;
    /** @default {@link StyleDefault.DEFAULT_OVERLAY_FONT_SIZE} */
    size?: number;
};

/**
 * @category Overlays
 */
export declare type OverlayPosition = OverlayShapePosition | OverlayEdgePosition;

/**
 * @category Overlays
 */
export declare type OverlayShapePosition = 'top-left' | 'top-right' | 'top-center' | 'bottom-left' | 'bottom-right' | 'bottom-center' | 'middle-left' | 'middle-right';

/**
 * @category Overlays
 */
export declare interface OverlaysRegistry {
    /**
     * Add one/several overlays to a BPMN element.
     *
     * Notice that if you pass an id that is not related to an existing BPMN element, nothing happens on the rendering side.
     *
     * @example
     * ```javascript
     * // Add an overlay to BPMN elements with id 'task_1'
     * bpmnVisualization.bpmnElementsRegistry.addOverlays('task_1', {
     *    position: 'top-left',
     *    label: '40',
     *    style: {
     *      font: { color: 'Chartreuse', size: 8 },
     *      fill: { color: 'Pink', opacity: 50 },
     *      stroke: { color: 'DarkSeaGreen', width: 2 }
     *    }
     * });
     *
     * // Add several overlays to BPMN element with id 'task_3'
     * bpmnVisualization.bpmnElementsRegistry.addOverlays('task_3', [
     *    {
     *      position: 'bottom-right',
     *      label: '110',
     *      style: {
     *        font: { color: '#663399', size: 8 },
     *        fill: { color: '#FFDAB9', opacity: 50 },
     *        stroke: { color: 'DarkSeaGreen', width: 2 }
     *      }
     *    },
     *    {
     *      position: 'top-left',
     *      label: '40',
     *      style: {
     *        font: { color: 'MidnightBlue', size: 30 },
     *        fill: { color: 'Aquamarine', opacity: 70 },
     *        stroke: { color: '#4B0082', width: 1 }
     *      }
     *    }
     * ]);
     * ```
     *
     * @param bpmnElementId The BPMN id of the element where to add the overlays
     * @param overlays The overlays to add to the BPMN element
     */
    addOverlays(bpmnElementId: string, overlays: Overlay | Overlay[]): void;
    /**
     * Remove all overlays of a BPMN element.
     *
     * Notice that if you pass an id that is not related to an existing BPMN element, nothing happens on the rendering side.
     *
     * <b>WARNING</b>: could be renamed when adding support for removal of one or several specific overlays.
     *
     * @example
     * ```javascript
     * //  all overlays of the BPMN element with id: activity_1
     * bpmnVisualization.bpmnElementsRegistry.removeAllOverlays('activity_1');
     * ```
     *
     * @param bpmnElementId The BPMN id of the element where to remove the overlays
     */
    removeAllOverlays(bpmnElementId: string): void;
}

/**
 * @category Overlays
 */
export declare type OverlayStroke = {
    /**
     * If you don't want to display a stroke, you can set the color to
     *   * `transparent`
     *   * the same value as for the fill color. This increases the padding/margin.
     *
     * @default {@link StyleDefault.DEFAULT_OVERLAY_STROKE_COLOR}
     * */
    color?: string;
    /**
     * **IMPORTANT**: this property is currently not taken into account by the default Overlay. See https://github.com/process-analytics/bpmn-visualization-js/issues/1234
     * @default {@link StyleDefault.DEFAULT_OVERLAY_STROKE_WIDTH}
     */
    width?: number;
};

/**
 * @category Overlays
 */
export declare type OverlayStyle = {
    font?: OverlayFont;
    fill?: OverlayFill;
    stroke?: OverlayStroke;
};

/**
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare interface PaintParameter {
    /** Underlying 'Canvas' in charge of performing the paint operations. */
    canvas: mxAbstractCanvas2D;
    /** The dimension and the style of the shape where the icon is painted. */
    shapeConfig: ShapeConfiguration;
    /** The style configuration of the icon. */
    iconStyleConfig: IconStyleConfiguration;
    /**
     * The actual size of the icon will be computed from the shape dimensions using this ratio. The ratio between the original dimensions of the icon are kept.
     * If not set, use the dimensions defined in the icon implementation.
     */
    ratioFromParent?: number;
    /** The called function to set the origin of the icon. Generally, it calls a method of {@link BpmnCanvas}. */
    setIconOriginFunct: (canvas: BpmnCanvas) => void;
}

/**
 * Configure the BPMN parser.
 * @category Initialization & Configuration
 */
export declare type ParserOptions = {
    /**
     * Apply additional processing to the XML attributes in the BPMN source.
     *
     * When defined, this function is called after the `bpmn-visualization` attribute processing.
     * You can use it to perform extra entities decoding. This can be done by using libraries like {@link https://www.npmjs.com/package/entities}.
     * ```ts
     * import { decodeXML } from 'entities';
     * const parserOptions: ParserOptions = {
     *   parser: {
     *     additionalXmlAttributeProcessor: (val: string) => { return decodeXML(val) }
     *   }
     * }
     * ```
     * @param val the value of the 'name' attribute to be processed.
     */
    additionalXmlAttributeProcessor?: (value: string) => string;
    /**
     * If `true`, disable the console logs produced by the parser.
     * @default false
     */
    disableConsoleLog?: boolean;
};

/**
 * Pool filtering configuration.
 *
 * A Pool is the graphical representation of a Participant in a Collaboration.
 * @category Initialization & Configuration
 */
export declare interface PoolFilter {
    /** id of the Participant related to the Pool to display */
    id?: string;
    /**
     * Name of the Participant, or name of the Process referenced by the Participant when the Participant doesn't have a name.
     * This is how `bpmn-visualization` builds the name into its internal model.
     * If `id` is set, this property is ignored.
     */
    name?: string;
}

/**
 * Global configuration for the rendering of the BPMN elements.
 *
 * @category Initialization & Configuration
 * @since 0.35.0
 */
export declare type RendererOptions = {
    /**
     * If set to `false`, support the "BPMN in Color" specification with a fallback with bpmn.io colors. For more details about the support, see
     * {@link https://github.com/process-analytics/bpmn-visualization-js/pull/2614}.
     *
     * Otherwise, disable the support.
     *
     * @default true
     */
    ignoreBpmnColors?: boolean;
};

/**
 * Enum values are used internally to identify sequence the flow markers and to manage their related style.
 * @category BPMN
 */
export declare enum SequenceFlowKind {
    NORMAL = "normal",
    DEFAULT = "default",
    CONDITIONAL_FROM_ACTIVITY = "conditional_from_activity",
    CONDITIONAL_FROM_GATEWAY = "conditional_from_gateway"
}

/**
 * @category BPMN
 */
export declare enum ShapeBpmnCallActivityKind {
    CALLING_PROCESS = "process",
    CALLING_GLOBAL_TASK = "global task"
}

/**
 * The real name of the field in the BPMN XSD.
 * @category BPMN
 */
export declare enum ShapeBpmnElementKind {
    LANE = "lane",
    POOL = "pool",
    CALL_ACTIVITY = "callActivity",
    SUB_PROCESS = "subProcess",
    TASK = "task",
    TASK_USER = "userTask",
    TASK_SERVICE = "serviceTask",
    TASK_RECEIVE = "receiveTask",
    TASK_SEND = "sendTask",
    TASK_MANUAL = "manualTask",
    TASK_SCRIPT = "scriptTask",
    TASK_BUSINESS_RULE = "businessRuleTask",
    GLOBAL_TASK = "globalTask",
    GLOBAL_TASK_USER = "globalUserTask",
    GLOBAL_TASK_MANUAL = "globalManualTask",
    GLOBAL_TASK_SCRIPT = "globalScriptTask",
    GLOBAL_TASK_BUSINESS_RULE = "globalBusinessRuleTask",
    GROUP = "group",
    TEXT_ANNOTATION = "textAnnotation",
    GATEWAY_PARALLEL = "parallelGateway",
    GATEWAY_EXCLUSIVE = "exclusiveGateway",
    GATEWAY_INCLUSIVE = "inclusiveGateway",
    GATEWAY_EVENT_BASED = "eventBasedGateway",
    GATEWAY_COMPLEX = "complexGateway",
    EVENT_START = "startEvent",
    EVENT_END = "endEvent",
    EVENT_INTERMEDIATE_CATCH = "intermediateCatchEvent",
    EVENT_INTERMEDIATE_THROW = "intermediateThrowEvent",
    EVENT_BOUNDARY = "boundaryEvent"
}

/**
 * Values available for the `eventGatewayType` property in the BPMN specification.
 * @category BPMN
 */
export declare enum ShapeBpmnEventBasedGatewayKind {
    Exclusive = "Exclusive",
    /** When no type is provided in the BPMN source. */
    None = "None",
    Parallel = "Parallel"
}

/**
 * Base name of the EventDefinition fields in the BPMN XSD for event kinds. In the xsd, the value is <enum_value>EventDefinition.
 *
 * For instance, TERMINATE --> terminateEventDefinition
 * @category BPMN
 */
export declare enum ShapeBpmnEventDefinitionKind {
    NONE = "none",
    TERMINATE = "terminate",
    CANCEL = "cancel",
    COMPENSATION = "compensate",
    CONDITIONAL = "conditional",
    ERROR = "error",
    ESCALATION = "escalation",
    LINK = "link",
    MESSAGE = "message",
    SIGNAL = "signal",
    TIMER = "timer"
}

/**
 * @category BPMN
 */
export declare enum ShapeBpmnMarkerKind {
    ADHOC = "adhoc",
    COMPENSATION = "compensation",
    EXPAND = "expand",
    LOOP = "loop",
    MULTI_INSTANCE_PARALLEL = "multi-parallel",
    MULTI_INSTANCE_SEQUENTIAL = "multi-sequential"
}

/**
 * Extended properties available when {@link BaseBpmnSemantic.isShape} is `true`.
 * @category Custom Behavior
 */
export declare interface ShapeBpmnSemantic extends BaseBpmnSemantic {
    /** Set when the {@link BaseBpmnSemantic.kind} relates to a BPMN Call Activity calling a global task. */
    callActivityGlobalTaskKind?: GlobalTaskKind;
    /** Set when the {@link BaseBpmnSemantic.kind} relates to a BPMN Call Activity. */
    callActivityKind?: ShapeBpmnCallActivityKind;
    /** Set when the {@link BaseBpmnSemantic.kind} relates to a BPMN event. */
    eventDefinitionKind?: ShapeBpmnEventDefinitionKind;
    /** IDs of the incoming flows/edges. */
    incomingIds: string[];
    /** Set when the {@link BaseBpmnSemantic.kind} relates to a BPMN intermediate catch event with {@link ShapeBpmnSemantic.eventDefinitionKind} set to {@link ShapeBpmnEventDefinitionKind.LINK}. */
    linkEventSourceIds?: string[];
    /** Set when the {@link BaseBpmnSemantic.kind} relates to a BPMN intermediate throw event with {@link ShapeBpmnSemantic.eventDefinitionKind} set to {@link ShapeBpmnEventDefinitionKind.LINK}. */
    linkEventTargetId?: string;
    /** IDs of the outgoing flows/edges. */
    outgoingIds: string[];
    /**
     * This is the ID of the direct parent of the current element, which can be a:
     * - process/participant
     * - lane
     * - sub-process
     * - call activity
     *
     * For the Boundary events, it is the activity to which the item belongs.
     *
     * **Special case**: it can be `undefined` when there is a single process without Participant/Pool. In this case, the direct children of the process have no parent.
     */
    parentId: string;
    /** Set when the {@link BaseBpmnSemantic.kind} relates to a BPMN sub-process. */
    subProcessKind?: ShapeBpmnSubProcessKind;
}

/**
 * @category BPMN
 */
export declare enum ShapeBpmnSubProcessKind {
    AD_HOC = "adhoc",
    EMBEDDED = "embedded",
    EVENT = "event",
    TRANSACTION = "transaction"
}

/**
 * @category BPMN Theme
 */
export declare interface ShapeConfiguration extends Size {
    x: number;
    y: number;
    strokeWidth?: number;
}

/**
 * @category Element Style
 */
export declare type ShapeStyleUpdate = EdgeStyleUpdate & {
    fill?: Fill;
};

/**
 * Utils to simplify the management of {@link ShapeBpmnElementKind}.
 *
 * This class is mainly used for internal purpose. You may use it to customize the BPMN theme as proposed in the examples but be aware it is subject to change.
 *
 * @category BPMN
 * @experimental
 */
export declare class ShapeUtil {
    static isEvent(kind: ShapeBpmnElementKind | string): boolean;
    static eventKinds(): ShapeBpmnElementKind[];
    static isBoundaryEvent(kind: ShapeBpmnElementKind): boolean;
    static isStartEvent(kind: ShapeBpmnElementKind): boolean;
    static isCatchEvent(kind: ShapeBpmnElementKind): boolean;
    static isIntermediateCatchEvent(kind: ShapeBpmnElementKind): boolean;
    static isIntermediateThrowEvent(kind: ShapeBpmnElementKind): boolean;
    static isCallActivity(kind: ShapeBpmnElementKind): boolean;
    static isSubProcess(kind: ShapeBpmnElementKind): boolean;
    static canHaveNoneEvent(kind: ShapeBpmnElementKind): boolean;
    static isActivity(kind: ShapeBpmnElementKind | string): boolean;
    static activityKinds(): ShapeBpmnElementKind[];
    static isWithDefaultSequenceFlow(kind: ShapeBpmnElementKind): boolean;
    /**
     * Returns `true` if `kind` is related to a task, for instance {@link ShapeBpmnElementKind.TASK}, {@link ShapeBpmnElementKind.TASK_SERVICE}, but not a {@link ShapeBpmnElementKind.GLOBAL_TASK}.
     */
    static isTask(kind: ShapeBpmnElementKind | string): boolean;
    /**
     * Returns all kinds related to a task, for instance {@link ShapeBpmnElementKind.TASK}, {@link ShapeBpmnElementKind.TASK_SEND}, but not a {@link ShapeBpmnElementKind.GLOBAL_TASK}.
     */
    static taskKinds(): ShapeBpmnElementKind[];
    static gatewayKinds(): ShapeBpmnElementKind[];
    static isGateway(kind: ShapeBpmnElementKind | string): boolean;
    static flowNodeKinds(): ShapeBpmnElementKind[];
    static isPoolOrLane(kind: ShapeBpmnElementKind | string): boolean;
}

/**
 * @category BPMN Theme
 */
export declare interface Size {
    width: number;
    height: number;
}

/**
 * @category Element Style
 */
export declare type Stroke = StyleWithOpacity & {
    /**
     * Possible values are all HTML color names or HEX codes, as well as special keywords such as:
     * - `default` to use the color defined in the BPMN element default style.
     * - `inherit` to apply the stroke color of the direct parent element.
     * - `none` for no color.
     * - `swimlane` to apply the stroke color of the nearest parent element with the type {@link ShapeBpmnElementKind.LANE} or {@link ShapeBpmnElementKind.POOL}.
     *
     * **Notes about the `default` special keyword**:
     * - It can be used when the style is first updated and then needs to be reset to its initial value.
     * - It doesn't use the color set in the BPMN source when the "BPMN in Color" support is enabled. It uses the color defined in the BPMN element default style.
     */
    color?: 'default' | 'inherit' | 'none' | 'swimlane' | string;
    /**
     * Defines the stroke width in pixels.
     *
     * The value must be between 1 and 50.
     *
     * If the set value is less than 1, the used value is 1.
     * If the set value is greater than 50, the used value is 50.
     *
     * To hide the stroke, set the `color` property to `none`.
     *
     * The `default` value resets the width to the value defined in the default style of the BPMN element. It can be used when the style is first updated and then needs to be reset to its initial value.
     *
     * **WARNING**: Changing the stroke width of Activities may be misleading as the default stroke widths have a meaning according to the BPMN Specification.
     * For example, updating the stroke width of a task using the same value as the default stroke width of a Call Activity can be confusing. In this case, you should also change another property such as the stroke color to allow the user to differentiate between them.
     */
    width?: 'default' | number;
};

/**
 * Configure the styles used for BPMN rendering.
 *
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare class StyleConfigurator {
    private readonly graph;
    constructor(graph: BpmnGraph);
    configureStyles(): void;
    private getStylesheet;
    private putCellStyle;
    private configureDefaultVertexStyle;
    private configurePoolStyle;
    private configureLaneStyle;
    private configureEventStyles;
    private configureTextAnnotationStyle;
    private configureGroupStyle;
    private configureActivityStyles;
    private configureGatewayStyles;
    private configureDefaultEdgeStyle;
    private configureEdgeStyles;
    private configureFlowStyles;
}

/**
 * Store all rendering defaults used by `bpmn-visualization`.
 *
 * **WARN**: You may use it to customize the BPMN Theme as suggested in the examples. But be aware that the way the default BPMN theme can be modified is subject to change.
 *
 * @category BPMN Theme
 * @experimental
 */
export declare const StyleDefault: {
    STROKE_WIDTH_THIN: number;
    STROKE_WIDTH_THICK: number;
    /** Margin between the bottom of markers and the bottom of the activity. */
    SHAPE_ACTIVITY_BOTTOM_MARGIN: number;
    /** Margin between the top of icon and the top of the activity. */
    SHAPE_ACTIVITY_TOP_MARGIN: number;
    /** Margin between the left of icon and the left of the activity. */
    SHAPE_ACTIVITY_LEFT_MARGIN: number;
    /** Spacing between 2 markers. */
    SHAPE_ACTIVITY_MARKER_ICON_MARGIN: number;
    /** Theoretical width of a marker. Used to position the markers when there are several markers. */
    SHAPE_ACTIVITY_MARKER_ICON_SIZE: number;
    POOL_LABEL_SIZE: number;
    POOL_LABEL_FILL_COLOR: string;
    LANE_LABEL_SIZE: number;
    LANE_LABEL_FILL_COLOR: string;
    SUB_PROCESS_TRANSACTION_INNER_RECT_OFFSET: number;
    SUB_PROCESS_TRANSACTION_INNER_RECT_ARC_SIZE: number;
    TEXT_ANNOTATION_BORDER_LENGTH: number;
    TEXT_ANNOTATION_FILL_COLOR: string;
    GROUP_FILL_COLOR: string;
    DEFAULT_FILL_COLOR: string;
    DEFAULT_STROKE_COLOR: string;
    DEFAULT_FONT_FAMILY: string;
    DEFAULT_FONT_SIZE: number;
    DEFAULT_FONT_COLOR: string;
    DEFAULT_MARGIN: number;
    SHAPE_ARC_SIZE: number;
    DEFAULT_OVERLAY_FILL_COLOR: string;
    DEFAULT_OVERLAY_FILL_OPACITY: number;
    DEFAULT_OVERLAY_STROKE_COLOR: string;
    DEFAULT_OVERLAY_STROKE_WIDTH: number;
    DEFAULT_OVERLAY_FONT_SIZE: number;
    DEFAULT_OVERLAY_FONT_COLOR: string;
    SEQUENCE_FLOW_CONDITIONAL_FROM_ACTIVITY_MARKER_FILL_COLOR: string;
    MESSAGE_FLOW_MARKER_START_FILL_COLOR: string;
    MESSAGE_FLOW_MARKER_END_FILL_COLOR: string;
};

/**
 * @category Element Style
 */
export declare interface StyleRegistry {
    /**
     * Update the style of one or several BPMN elements.
     *
     * @example
     * ```javascript
     * bpmnVisualization.bpmnElementsRegistry.updateStyle('activity_1', {
     *   stroke: {
     *     color: 'red',
     *   },
     * });
     * ```
     *
     * **Notes**:
     *
     * - This method is intended to update the style of specific elements, e.g. to update their colors. When rendering a BPMN diagram, `bpmn-visualization` applies style properties
     * to all elements according to their types.
     * So if you want to style all elements of a certain type, change the default configuration of the styles instead of updating the element afterwards. See the repository providing the
     * [examples of the `bpmn-visualization` TypeScript library](https://github.com/process-analytics/bpmn-visualization-examples/) for more details.
     * - If you pass IDs that are not related to existing BPMN elements, they will be ignored and no changes will be made to the rendering.
     *
     * @param bpmnElementIds The BPMN ID of the element(s) whose style must be updated.
     * @param styleUpdate The style properties to update.
     *
     * @see {@link resetStyle} to reset the style of one or several BPMN elements.
     * @see {@link CssClassesRegistry#addCssClasses} to add CSS classes to a BPMN element.
     * @see {@link CssClassesRegistry#removeCssClasses} to remove specific classes from a BPMN element.
     * @see {@link CssClassesRegistry#removeAllCssClasses} to remove all CSS classes from a BPMN element.
     * @see {@link CssClassesRegistry#toggleCssClasses} to toggle CSS classes on a BPMN element.
     * @since 0.33.0
     */
    updateStyle(bpmnElementIds: string | string[], styleUpdate: StyleUpdate): void;
    /**
     * Reset the style that were previously updated to one or more BPMN elements using the {@link updateStyle} method.
     *
     * @example
     * ```javascript
     * bpmnVisualization.bpmnElementsRegistry.resetStyle('activity_1');
     * ```
     *
     * **Notes**:
     *
     * - This method is intended to update the style of specific elements, e.g. to update their colors. When rendering a BPMN diagram, `bpmn-visualization` applies style properties
     * to all elements according to their types.
     * So if you want to style all elements of a certain type, change the default configuration of the styles instead of updating the element afterward. See the repository providing the
     * [examples of the `bpmn-visualization` TypeScript library](https://github.com/process-analytics/bpmn-visualization-examples/) for more details.
     * - If you pass IDs that are not related to existing BPMN elements, they will be ignored and no changes will be made to the rendering.
     *
     * @param bpmnElementIds The BPMN ID of the element(s) whose style must be reset.
     * When passing a nullish parameter, the style of all BPMN elements will be reset. Passing an empty array has no effect.
     *
     * @see {@link updateStyle} to update the style of one or several BPMN elements.
     * @see {@link CssClassesRegistry#addCssClasses} to add CSS classes to a BPMN element.
     * @see {@link CssClassesRegistry#removeCssClasses} to remove specific classes from a BPMN element.
     * @see {@link CssClassesRegistry#removeAllCssClasses} to remove all CSS classes from a BPMN element.
     * @see {@link CssClassesRegistry#toggleCssClasses} to toggle CSS classes on a BPMN element.
     * @since 0.37.0
     */
    resetStyle(bpmnElementIds?: string | string[]): void;
}

/**
 * @category Element Style
 */
export declare type StyleUpdate = EdgeStyleUpdate | ShapeStyleUpdate;

/**
 * @category Element Style
 */
export declare type StyleWithOpacity = {
    /**
     * The value must be between 0 and 100:
     * - If the set value is less than 0, the used value is 0.
     * - If the set value is greater than 100, the used value is 100.
     *
     * **Notes about the `default` special keyword**:
     * - It is used to apply the opacity defined in the default style of the BPMN element.
     * - It can be used when the style is first updated and then needs to be reset to its initial value.
     */
    opacity?: Opacity;
};

/**
 * The version of `bpmn-visualization` and the version of its dependencies.
 */
export declare interface Version {
    /** The `bpmn-visualization` version. */
    lib: string;
    /** The version of the `bpmn-visualization` dependencies. This may **not** list all dependencies. */
    dependencies: Map<string, string>;
}

/**
 * Zoom specific options.
 * @category Initialization & Configuration
 */
export declare interface ZoomConfiguration {
    /**
     * Value in `milliseconds` responsible for throttling the mouse scroll event (not every event is firing the function handler, only limited number can lunch handler). A smaller value
     * results in more events fired, bigger gain in zoom factor.
     * Values must be in the [0, 100] interval, values outside this interval are set to the interval bounds.
     * @default 50
     */
    throttleDelay?: number;
    /**
     * Value in `milliseconds` responsible for debouncing the zoom function - the actual scaling. A bigger value results in bigger gain in zoom factor before actual scaling takes place.
     * Values must be in the [0, 100] interval, values outside this interval are set to the interval bounds.
     * @default 50
     */
    debounceDelay?: number;
}

/**
 * @category Navigation
 * @since 0.24.0
 */
export declare enum ZoomType {
    In = "in",
    Out = "out"
}

export { }
