/*
Copyright 2020-2025 Bonitasoft S.A.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

import factory from 'mxgraph';
import { debounce, throttle } from 'es-toolkit';
import { XMLParser } from 'fast-xml-parser';

var FitType;
(function (FitType) {
    FitType["None"] = "None";
    FitType["HorizontalVertical"] = "HorizontalVertical";
    FitType["Horizontal"] = "Horizontal";
    FitType["Vertical"] = "Vertical";
    FitType["Center"] = "Center";
})(FitType || (FitType = {}));
var ZoomType;
(function (ZoomType) {
    ZoomType["In"] = "in";
    ZoomType["Out"] = "out";
})(ZoomType || (ZoomType = {}));

function htmlElement(element) {
    if (element instanceof HTMLElement) {
        return element;
    }
    return document.querySelector(`#${element}`);
}

var ShapeBpmnElementKind;
(function (ShapeBpmnElementKind) {
    ShapeBpmnElementKind["LANE"] = "lane";
    ShapeBpmnElementKind["POOL"] = "pool";
    ShapeBpmnElementKind["CALL_ACTIVITY"] = "callActivity";
    ShapeBpmnElementKind["SUB_PROCESS"] = "subProcess";
    ShapeBpmnElementKind["TASK"] = "task";
    ShapeBpmnElementKind["TASK_USER"] = "userTask";
    ShapeBpmnElementKind["TASK_SERVICE"] = "serviceTask";
    ShapeBpmnElementKind["TASK_RECEIVE"] = "receiveTask";
    ShapeBpmnElementKind["TASK_SEND"] = "sendTask";
    ShapeBpmnElementKind["TASK_MANUAL"] = "manualTask";
    ShapeBpmnElementKind["TASK_SCRIPT"] = "scriptTask";
    ShapeBpmnElementKind["TASK_BUSINESS_RULE"] = "businessRuleTask";
    ShapeBpmnElementKind["GLOBAL_TASK"] = "globalTask";
    ShapeBpmnElementKind["GLOBAL_TASK_USER"] = "globalUserTask";
    ShapeBpmnElementKind["GLOBAL_TASK_MANUAL"] = "globalManualTask";
    ShapeBpmnElementKind["GLOBAL_TASK_SCRIPT"] = "globalScriptTask";
    ShapeBpmnElementKind["GLOBAL_TASK_BUSINESS_RULE"] = "globalBusinessRuleTask";
    ShapeBpmnElementKind["GROUP"] = "group";
    ShapeBpmnElementKind["TEXT_ANNOTATION"] = "textAnnotation";
    ShapeBpmnElementKind["GATEWAY_PARALLEL"] = "parallelGateway";
    ShapeBpmnElementKind["GATEWAY_EXCLUSIVE"] = "exclusiveGateway";
    ShapeBpmnElementKind["GATEWAY_INCLUSIVE"] = "inclusiveGateway";
    ShapeBpmnElementKind["GATEWAY_EVENT_BASED"] = "eventBasedGateway";
    ShapeBpmnElementKind["GATEWAY_COMPLEX"] = "complexGateway";
    ShapeBpmnElementKind["EVENT_START"] = "startEvent";
    ShapeBpmnElementKind["EVENT_END"] = "endEvent";
    ShapeBpmnElementKind["EVENT_INTERMEDIATE_CATCH"] = "intermediateCatchEvent";
    ShapeBpmnElementKind["EVENT_INTERMEDIATE_THROW"] = "intermediateThrowEvent";
    ShapeBpmnElementKind["EVENT_BOUNDARY"] = "boundaryEvent";
})(ShapeBpmnElementKind || (ShapeBpmnElementKind = {}));
var ShapeBpmnCallActivityKind;
(function (ShapeBpmnCallActivityKind) {
    ShapeBpmnCallActivityKind["CALLING_PROCESS"] = "process";
    ShapeBpmnCallActivityKind["CALLING_GLOBAL_TASK"] = "global task";
})(ShapeBpmnCallActivityKind || (ShapeBpmnCallActivityKind = {}));
var ShapeBpmnEventBasedGatewayKind;
(function (ShapeBpmnEventBasedGatewayKind) {
    ShapeBpmnEventBasedGatewayKind["Exclusive"] = "Exclusive";
    ShapeBpmnEventBasedGatewayKind["None"] = "None";
    ShapeBpmnEventBasedGatewayKind["Parallel"] = "Parallel";
})(ShapeBpmnEventBasedGatewayKind || (ShapeBpmnEventBasedGatewayKind = {}));
var ShapeBpmnEventDefinitionKind;
(function (ShapeBpmnEventDefinitionKind) {
    ShapeBpmnEventDefinitionKind["NONE"] = "none";
    ShapeBpmnEventDefinitionKind["TERMINATE"] = "terminate";
    ShapeBpmnEventDefinitionKind["CANCEL"] = "cancel";
    ShapeBpmnEventDefinitionKind["COMPENSATION"] = "compensate";
    ShapeBpmnEventDefinitionKind["CONDITIONAL"] = "conditional";
    ShapeBpmnEventDefinitionKind["ERROR"] = "error";
    ShapeBpmnEventDefinitionKind["ESCALATION"] = "escalation";
    ShapeBpmnEventDefinitionKind["LINK"] = "link";
    ShapeBpmnEventDefinitionKind["MESSAGE"] = "message";
    ShapeBpmnEventDefinitionKind["SIGNAL"] = "signal";
    ShapeBpmnEventDefinitionKind["TIMER"] = "timer";
})(ShapeBpmnEventDefinitionKind || (ShapeBpmnEventDefinitionKind = {}));
var ShapeBpmnMarkerKind;
(function (ShapeBpmnMarkerKind) {
    ShapeBpmnMarkerKind["ADHOC"] = "adhoc";
    ShapeBpmnMarkerKind["COMPENSATION"] = "compensation";
    ShapeBpmnMarkerKind["EXPAND"] = "expand";
    ShapeBpmnMarkerKind["LOOP"] = "loop";
    ShapeBpmnMarkerKind["MULTI_INSTANCE_PARALLEL"] = "multi-parallel";
    ShapeBpmnMarkerKind["MULTI_INSTANCE_SEQUENTIAL"] = "multi-sequential";
})(ShapeBpmnMarkerKind || (ShapeBpmnMarkerKind = {}));
var ShapeBpmnSubProcessKind;
(function (ShapeBpmnSubProcessKind) {
    ShapeBpmnSubProcessKind["AD_HOC"] = "adhoc";
    ShapeBpmnSubProcessKind["EMBEDDED"] = "embedded";
    ShapeBpmnSubProcessKind["EVENT"] = "event";
    ShapeBpmnSubProcessKind["TRANSACTION"] = "transaction";
})(ShapeBpmnSubProcessKind || (ShapeBpmnSubProcessKind = {}));

function convertEmptyStringAndObject(element, acceptEmptyString) {
    if (element === '') {
        return acceptEmptyString ? {} : undefined;
    }
    return element;
}
function ensureIsArray(elements, acceptEmptyString = false) {
    if (elements === undefined || elements === null) {
        return [];
    }
    return ((Array.isArray(elements) ? elements : [elements])
        .map(element => convertEmptyStringAndObject(element, acceptEmptyString))
        .filter(Boolean));
}
function filter(arrayToFilter, suffix, options) {
    const patterns = [];
    if (options === null || options === void 0 ? void 0 : options.startingWith) {
        patterns.push(`^(${options.startingWith}).*`);
    }
    else if (options === null || options === void 0 ? void 0 : options.notStartingWith) {
        patterns.push(`^(?!(${options.notStartingWith})).*`);
    }
    patterns.push(`${suffix}$`);
    const pattern = patterns.join('');
    return arrayToFilter.filter(element => ((options === null || options === void 0 ? void 0 : options.ignoreCase) ? new RegExp(pattern, 'i').test(element) : new RegExp(pattern).test(element)));
}

class ShapeUtil {
    static isEvent(kind) {
        return isKindOf(EVENT_KINDS, kind);
    }
    static eventKinds() {
        return [...EVENT_KINDS];
    }
    static isBoundaryEvent(kind) {
        return ShapeBpmnElementKind.EVENT_BOUNDARY === kind;
    }
    static isStartEvent(kind) {
        return ShapeBpmnElementKind.EVENT_START === kind;
    }
    static isCatchEvent(kind) {
        return ShapeBpmnElementKind.EVENT_INTERMEDIATE_CATCH === kind || ShapeBpmnElementKind.EVENT_BOUNDARY === kind || ShapeBpmnElementKind.EVENT_START === kind;
    }
    static isIntermediateCatchEvent(kind) {
        return ShapeBpmnElementKind.EVENT_INTERMEDIATE_CATCH === kind;
    }
    static isIntermediateThrowEvent(kind) {
        return ShapeBpmnElementKind.EVENT_INTERMEDIATE_THROW === kind;
    }
    static isCallActivity(kind) {
        return ShapeBpmnElementKind.CALL_ACTIVITY === kind;
    }
    static isSubProcess(kind) {
        return ShapeBpmnElementKind.SUB_PROCESS === kind;
    }
    static canHaveNoneEvent(kind) {
        return ShapeBpmnElementKind.EVENT_INTERMEDIATE_THROW === kind || ShapeBpmnElementKind.EVENT_END === kind || ShapeBpmnElementKind.EVENT_START === kind;
    }
    static isActivity(kind) {
        return isKindOf(ACTIVITY_KINDS, kind);
    }
    static activityKinds() {
        return [...ACTIVITY_KINDS];
    }
    static isWithDefaultSequenceFlow(kind) {
        return FLOW_NODE_WITH_DEFAULT_SEQUENCE_FLOW_KINDS.has(kind);
    }
    static isTask(kind) {
        return isKindOf(TASK_KINDS, kind);
    }
    static taskKinds() {
        return [...TASK_KINDS];
    }
    static gatewayKinds() {
        return [...GATEWAY_KINDS];
    }
    static isGateway(kind) {
        return isKindOf(GATEWAY_KINDS, kind);
    }
    static flowNodeKinds() {
        return Object.values(ShapeBpmnElementKind).filter(kind => !ShapeUtil.isPoolOrLane(kind));
    }
    static isPoolOrLane(kind) {
        return kind == ShapeBpmnElementKind.POOL || kind == ShapeBpmnElementKind.LANE;
    }
}
function filterKind(suffix, options) {
    return filter(Object.values(ShapeBpmnElementKind), suffix, options);
}
function isKindOf(referenceKinds, kind) {
    return Object.values(referenceKinds)
        .map(value => value)
        .includes(kind);
}
const EVENT_KINDS = filterKind('Event');
const GATEWAY_KINDS = filterKind('Gateway');
const TASK_KINDS = filterKind('Task', { ignoreCase: true, notStartingWith: 'global' });
const ACTIVITY_KINDS = [...TASK_KINDS, ShapeBpmnElementKind.CALL_ACTIVITY, ShapeBpmnElementKind.SUB_PROCESS];
const FLOW_NODE_WITH_DEFAULT_SEQUENCE_FLOW_KINDS = new Set([
    ...ACTIVITY_KINDS,
    ShapeBpmnElementKind.GATEWAY_EXCLUSIVE,
    ShapeBpmnElementKind.GATEWAY_INCLUSIVE,
    ShapeBpmnElementKind.GATEWAY_COMPLEX,
]);
const eventDefinitionKinds = Object.values(ShapeBpmnEventDefinitionKind).filter(kind => kind != ShapeBpmnEventDefinitionKind.NONE);

var AssociationDirectionKind;
(function (AssociationDirectionKind) {
    AssociationDirectionKind["NONE"] = "None";
    AssociationDirectionKind["ONE"] = "One";
    AssociationDirectionKind["BOTH"] = "Both";
})(AssociationDirectionKind || (AssociationDirectionKind = {}));
var FlowKind;
(function (FlowKind) {
    FlowKind["SEQUENCE_FLOW"] = "sequenceFlow";
    FlowKind["MESSAGE_FLOW"] = "messageFlow";
    FlowKind["ASSOCIATION_FLOW"] = "association";
})(FlowKind || (FlowKind = {}));
var MessageVisibleKind;
(function (MessageVisibleKind) {
    MessageVisibleKind["NONE"] = "none";
    MessageVisibleKind["INITIATING"] = "initiating";
    MessageVisibleKind["NON_INITIATING"] = "non_initiating";
})(MessageVisibleKind || (MessageVisibleKind = {}));
var SequenceFlowKind;
(function (SequenceFlowKind) {
    SequenceFlowKind["NORMAL"] = "normal";
    SequenceFlowKind["DEFAULT"] = "default";
    SequenceFlowKind["CONDITIONAL_FROM_ACTIVITY"] = "conditional_from_activity";
    SequenceFlowKind["CONDITIONAL_FROM_GATEWAY"] = "conditional_from_gateway";
})(SequenceFlowKind || (SequenceFlowKind = {}));

class Flow {
    constructor(id, name, kind, sourceReferenceId, targetReferenceId) {
        Object.defineProperty(this, "id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: id
        });
        Object.defineProperty(this, "name", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: name
        });
        Object.defineProperty(this, "kind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: kind
        });
        Object.defineProperty(this, "sourceReferenceId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: sourceReferenceId
        });
        Object.defineProperty(this, "targetReferenceId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: targetReferenceId
        });
    }
}
class SequenceFlow extends Flow {
    constructor(id, name, sourceReferenceId, targetReferenceId, sequenceFlowKind = SequenceFlowKind.NORMAL) {
        super(id, name, FlowKind.SEQUENCE_FLOW, sourceReferenceId, targetReferenceId);
        Object.defineProperty(this, "sequenceFlowKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: sequenceFlowKind
        });
    }
}
class MessageFlow extends Flow {
    constructor(id, name, sourceReferenceId, targetReferenceId) {
        super(id, name, FlowKind.MESSAGE_FLOW, sourceReferenceId, targetReferenceId);
    }
}
class AssociationFlow extends Flow {
    constructor(id, name, sourceReferenceId, targetReferenceId, associationDirectionKind = AssociationDirectionKind.NONE) {
        super(id, name, FlowKind.ASSOCIATION_FLOW, sourceReferenceId, targetReferenceId);
        Object.defineProperty(this, "associationDirectionKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: associationDirectionKind
        });
    }
}

const mxgraph = initialize();
const mxCellRenderer = mxgraph.mxCellRenderer;
const mxClient = mxgraph.mxClient;
const mxConstants = mxgraph.mxConstants;
const mxEvent = mxgraph.mxEvent;
const mxPerimeter = mxgraph.mxPerimeter;
const mxPoint = mxgraph.mxPoint;
const mxRectangle = mxgraph.mxRectangle;
const mxRectangleShape = mxgraph.mxRectangleShape;
const mxSvgCanvas2D = mxgraph.mxSvgCanvas2D;
const mxUtils = mxgraph.mxUtils;
function initialize() {
    window.mxForceIncludes = false;
    window.mxLoadResources = false;
    window.mxLoadStylesheets = false;
    window.mxResourceExtension = '.txt';
    return factory({});
}

class CoordinatesTranslator {
    constructor(graph) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: graph
        });
    }
    computeRelativeCoordinates(parent, absoluteCoordinate) {
        const translateForRoot = this.getTranslateForRoot(parent);
        const relativeX = absoluteCoordinate.x + translateForRoot.x;
        const relativeY = absoluteCoordinate.y + translateForRoot.y;
        return new mxPoint(relativeX, relativeY);
    }
    getTranslateForRoot(cell) {
        const model = this.graph.getModel();
        const offset = new mxPoint(0, 0);
        while (cell != null) {
            const geo = model.getGeometry(cell);
            if (geo != null) {
                offset.x -= geo.x;
                offset.y -= geo.y;
            }
            cell = model.getParent(cell);
        }
        return offset;
    }
    computeEdgeCenter(edge) {
        const points = edge.geometry.points;
        const p0 = points[0];
        const pe = points.at(-1);
        const dx = pe.x - p0.x;
        const dy = pe.y - p0.y;
        return new mxPoint(p0.x + dx / 2, p0.y + dy / 2);
    }
}

class Shape {
    constructor(id, bpmnElement, bounds, label, isHorizontal) {
        Object.defineProperty(this, "id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: id
        });
        Object.defineProperty(this, "bpmnElement", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bpmnElement
        });
        Object.defineProperty(this, "bounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bounds
        });
        Object.defineProperty(this, "label", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: label
        });
        Object.defineProperty(this, "isHorizontal", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: isHorizontal
        });
        Object.defineProperty(this, "extensions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
    }
}

class ShapeBpmnElement {
    constructor(id, name, kind, parentId, instantiate = false) {
        Object.defineProperty(this, "id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: id
        });
        Object.defineProperty(this, "name", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: name
        });
        Object.defineProperty(this, "kind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: kind
        });
        Object.defineProperty(this, "parentId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: parentId
        });
        Object.defineProperty(this, "instantiate", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: instantiate
        });
        Object.defineProperty(this, "incomingIds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "outgoingIds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
    }
}
class ShapeBpmnActivity extends ShapeBpmnElement {
    constructor(id, name, kind, parentId, instantiate, markers = []) {
        super(id, name, kind, parentId, instantiate);
        Object.defineProperty(this, "markers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: markers
        });
    }
}
class ShapeBpmnCallActivity extends ShapeBpmnActivity {
    constructor(id, name, callActivityKind, parentId, markers, globalTaskKind) {
        super(id, name, ShapeBpmnElementKind.CALL_ACTIVITY, parentId, undefined, markers);
        Object.defineProperty(this, "callActivityKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: callActivityKind
        });
        Object.defineProperty(this, "globalTaskKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: globalTaskKind
        });
    }
}
class ShapeBpmnSubProcess extends ShapeBpmnActivity {
    constructor(id, name, subProcessKind, parentId, markers) {
        subProcessKind == ShapeBpmnSubProcessKind.AD_HOC && !markers.includes(ShapeBpmnMarkerKind.ADHOC) && markers.push(ShapeBpmnMarkerKind.ADHOC);
        super(id, name, ShapeBpmnElementKind.SUB_PROCESS, parentId, undefined, markers);
        Object.defineProperty(this, "subProcessKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: subProcessKind
        });
    }
}
class ShapeBpmnEvent extends ShapeBpmnElement {
    constructor(id, name, elementKind, eventDefinitionKind, parentId) {
        super(id, name, elementKind, parentId);
        Object.defineProperty(this, "eventDefinitionKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: eventDefinitionKind
        });
    }
}
class ShapeBpmnIntermediateCatchEvent extends ShapeBpmnEvent {
    constructor(id, name, eventDefinitionKind, parentId) {
        super(id, name, ShapeBpmnElementKind.EVENT_INTERMEDIATE_CATCH, eventDefinitionKind, parentId);
        Object.defineProperty(this, "sourceIds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
    }
}
class ShapeBpmnIntermediateThrowEvent extends ShapeBpmnEvent {
    constructor(id, name, eventDefinitionKind, parentId) {
        super(id, name, ShapeBpmnElementKind.EVENT_INTERMEDIATE_THROW, eventDefinitionKind, parentId);
        Object.defineProperty(this, "targetId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
    }
}
class ShapeBpmnStartEvent extends ShapeBpmnEvent {
    constructor(id, name, eventDefinitionKind, parentId, isInterrupting) {
        super(id, name, ShapeBpmnElementKind.EVENT_START, eventDefinitionKind, parentId);
        Object.defineProperty(this, "isInterrupting", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: isInterrupting
        });
    }
}
class ShapeBpmnBoundaryEvent extends ShapeBpmnEvent {
    constructor(id, name, eventDefinitionKind, parentId, isInterrupting = true) {
        super(id, name, ShapeBpmnElementKind.EVENT_BOUNDARY, eventDefinitionKind, parentId);
        Object.defineProperty(this, "isInterrupting", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: isInterrupting
        });
    }
}
class ShapeBpmnEventBasedGateway extends ShapeBpmnElement {
    constructor(id, name, parentId, instantiate, gatewayKind = ShapeBpmnEventBasedGatewayKind.None) {
        super(id, name, ShapeBpmnElementKind.GATEWAY_EVENT_BASED, parentId, instantiate);
        Object.defineProperty(this, "gatewayKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: gatewayKind
        });
    }
}

function ensureInRange(value, min, max, defaultValue) {
    const inRangeValue = value !== null && value !== void 0 ? value : defaultValue;
    return Math.min(Math.max(inRangeValue, min), max);
}
function ensurePositiveValue(input) {
    return Math.max(input !== null && input !== void 0 ? input : 0, 0);
}
function ensureValidZoomConfiguration(config) {
    const validatedConfig = config !== null && config !== void 0 ? config : {};
    validatedConfig.debounceDelay = ensureInRange(validatedConfig.debounceDelay, 0, 100, 50);
    validatedConfig.throttleDelay = ensureInRange(validatedConfig.throttleDelay, 0, 100, 50);
    return validatedConfig;
}
function ensureOpacityValue(opacity) {
    return opacity == 'default' ? undefined : ensureInRange(opacity, 0, 100, 100);
}
function ensureStrokeWidthValue(strokeWidth) {
    return strokeWidth == 'default' ? undefined : ensureInRange(strokeWidth, 1, 50, 1);
}

const BpmnStyleIdentifier = {
    EDGE: 'bpmn.edge',
    EDGE_START_FILL_COLOR: 'bpmn.edge.startFillColor',
    EDGE_END_FILL_COLOR: 'bpmn.edge.endFillColor',
    EVENT_BASED_GATEWAY_KIND: 'bpmn.gatewayKind',
    EVENT_DEFINITION_KIND: 'bpmn.eventDefinitionKind',
    GLOBAL_TASK_KIND: 'bpmn.globalTaskKind',
    SUB_PROCESS_KIND: 'bpmn.subProcessKind',
    IS_INITIATING: 'bpmn.isInitiating',
    IS_INSTANTIATING: 'bpmn.isInstantiating',
    IS_INTERRUPTING: 'bpmn.isInterrupting',
    EXTRA_CSS_CLASSES: 'bpmn.extra.css.classes',
    MARKERS: 'bpmn.markers',
    MESSAGE_FLOW_ICON: 'bpmn.messageFlowIcon',
};
const MarkerIdentifier = {
    ARROW_DASH: 'bpmn.dash',
};

const StyleDefault = {
    STROKE_WIDTH_THIN: 2,
    STROKE_WIDTH_THICK: 5,
    SHAPE_ACTIVITY_BOTTOM_MARGIN: 7,
    SHAPE_ACTIVITY_TOP_MARGIN: 7,
    SHAPE_ACTIVITY_LEFT_MARGIN: 7,
    SHAPE_ACTIVITY_MARKER_ICON_MARGIN: 5,
    SHAPE_ACTIVITY_MARKER_ICON_SIZE: 20,
    POOL_LABEL_SIZE: 30,
    POOL_LABEL_FILL_COLOR: 'none',
    LANE_LABEL_SIZE: 30,
    LANE_LABEL_FILL_COLOR: 'none',
    SUB_PROCESS_TRANSACTION_INNER_RECT_OFFSET: 4,
    SUB_PROCESS_TRANSACTION_INNER_RECT_ARC_SIZE: 6,
    TEXT_ANNOTATION_BORDER_LENGTH: 10,
    TEXT_ANNOTATION_FILL_COLOR: 'none',
    GROUP_FILL_COLOR: 'none',
    DEFAULT_FILL_COLOR: 'White',
    DEFAULT_STROKE_COLOR: 'Black',
    DEFAULT_FONT_FAMILY: 'Arial, Helvetica, sans-serif',
    DEFAULT_FONT_SIZE: 11,
    DEFAULT_FONT_COLOR: 'Black',
    DEFAULT_MARGIN: 0,
    SHAPE_ARC_SIZE: 20,
    DEFAULT_OVERLAY_FILL_COLOR: 'White',
    DEFAULT_OVERLAY_FILL_OPACITY: 100,
    DEFAULT_OVERLAY_STROKE_COLOR: 'Black',
    DEFAULT_OVERLAY_STROKE_WIDTH: 1,
    DEFAULT_OVERLAY_FONT_SIZE: 11,
    DEFAULT_OVERLAY_FONT_COLOR: 'Black',
    SEQUENCE_FLOW_CONDITIONAL_FROM_ACTIVITY_MARKER_FILL_COLOR: 'White',
    MESSAGE_FLOW_MARKER_START_FILL_COLOR: 'White',
    MESSAGE_FLOW_MARKER_END_FILL_COLOR: 'White',
};
const getBpmnIsInstantiating = (style) => mxUtils.getValue(style, BpmnStyleIdentifier.IS_INSTANTIATING, 'false') == 'true';
const convertDefaultValue = (value) => (value == 'default' ? undefined : value);
const updateStroke = (cellStyle, stroke) => {
    if (stroke) {
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_STROKECOLOR, stroke.color, convertDefaultValue);
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_STROKE_OPACITY, stroke.opacity, ensureOpacityValue);
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_STROKEWIDTH, stroke.width, ensureStrokeWidthValue);
    }
    return cellStyle;
};
const setStyle = (cellStyle, key, value, converter = (value) => value) => {
    return value == undefined ? cellStyle : mxUtils.setStyle(cellStyle, key, converter(value));
};
const setStyleFlag = (cellStyle, key, flag, value) => value == undefined ? cellStyle : mxUtils.setStyleFlag(cellStyle, key, flag, value);
const updateFont = (cellStyle, font) => {
    if (font) {
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_FONTCOLOR, font.color, convertDefaultValue);
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_FONTSIZE, font.size);
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_FONTFAMILY, font.family);
        cellStyle = setStyleFlag(cellStyle, mxConstants.STYLE_FONTSTYLE, mxConstants.FONT_BOLD, font.isBold);
        cellStyle = setStyleFlag(cellStyle, mxConstants.STYLE_FONTSTYLE, mxConstants.FONT_ITALIC, font.isItalic);
        cellStyle = setStyleFlag(cellStyle, mxConstants.STYLE_FONTSTYLE, mxConstants.FONT_UNDERLINE, font.isUnderline);
        cellStyle = setStyleFlag(cellStyle, mxConstants.STYLE_FONTSTYLE, mxConstants.FONT_STRIKETHROUGH, font.isStrikeThrough);
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_TEXT_OPACITY, font.opacity, ensureOpacityValue);
    }
    return cellStyle;
};
const getStyleValue = (cellStyle, key, defaultValue) => {
    var _a;
    return (_a = cellStyle === null || cellStyle === void 0 ? void 0 : cellStyle.split(';').map(entry => entry.split('=')).filter(([k]) => k === key).map(([, v]) => v)[0]) !== null && _a !== void 0 ? _a : defaultValue;
};
const convertDirection = (direction) => {
    switch (direction) {
        case 'right-to-left': {
            return mxConstants.DIRECTION_WEST;
        }
        case 'bottom-to-top': {
            return mxConstants.DIRECTION_NORTH;
        }
        case 'top-to-bottom': {
            return mxConstants.DIRECTION_SOUTH;
        }
        default: {
            return mxConstants.DIRECTION_EAST;
        }
    }
};
const updateFill = (cellStyle, fill) => {
    const color = fill.color;
    if (color) {
        const isGradient = isFillColorGradient(color);
        const fillColor = isGradient ? color.startColor : color;
        cellStyle = setStyle(cellStyle, mxConstants.STYLE_FILLCOLOR, fillColor, convertDefaultValue);
        if (isGradient) {
            cellStyle = mxUtils.setStyle(cellStyle, mxConstants.STYLE_GRADIENTCOLOR, color.endColor);
            cellStyle = mxUtils.setStyle(cellStyle, mxConstants.STYLE_GRADIENT_DIRECTION, convertDirection(color.direction));
        }
        else if (color === 'default') {
            cellStyle = mxUtils.setStyle(cellStyle, mxConstants.STYLE_GRADIENTCOLOR, undefined);
            cellStyle = mxUtils.setStyle(cellStyle, mxConstants.STYLE_GRADIENT_DIRECTION, undefined);
        }
        if (cellStyle.includes(ShapeBpmnElementKind.POOL) || cellStyle.includes(ShapeBpmnElementKind.LANE)) {
            cellStyle = setStyle(cellStyle, mxConstants.STYLE_SWIMLANE_FILLCOLOR, fillColor, convertDefaultValue);
        }
    }
    cellStyle = setStyle(cellStyle, mxConstants.STYLE_FILL_OPACITY, fill.opacity, ensureOpacityValue);
    return cellStyle;
};
const isShapeStyleUpdate = (style) => {
    return style && typeof style === 'object' && 'fill' in style;
};
const isFillColorGradient = (color) => {
    return color && typeof color === 'object';
};

class StyleComputer {
    constructor(options) {
        var _a;
        Object.defineProperty(this, "ignoreBpmnColors", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.ignoreBpmnColors = (_a = options === null || options === void 0 ? void 0 : options.ignoreBpmnColors) !== null && _a !== void 0 ? _a : true;
    }
    computeStyle(bpmnCell, labelBounds) {
        const styles = [bpmnCell.bpmnElement.kind];
        let mainStyleValues;
        if (bpmnCell instanceof Shape) {
            mainStyleValues = this.computeShapeStyleValues(bpmnCell);
        }
        else {
            styles.push(...computeEdgeBaseStyles(bpmnCell));
            mainStyleValues = this.computeEdgeStyleValues(bpmnCell);
        }
        const fontStyleValues = this.computeFontStyleValues(bpmnCell);
        const labelStyleValues = computeLabelStyleValues(bpmnCell, labelBounds);
        styles.push(...toArrayOfMxGraphStyleEntries([...mainStyleValues, ...fontStyleValues, ...labelStyleValues]));
        return styles.join(';');
    }
    computeShapeStyleValues(shape) {
        const styleValues = new Map();
        const bpmnElement = shape.bpmnElement;
        if (bpmnElement instanceof ShapeBpmnEvent) {
            computeEventShapeStyle(bpmnElement, styleValues);
        }
        else if (bpmnElement instanceof ShapeBpmnActivity) {
            computeActivityShapeStyle(bpmnElement, styleValues);
        }
        else if (ShapeUtil.isPoolOrLane(bpmnElement.kind)) {
            styleValues.set(mxConstants.STYLE_HORIZONTAL, shape.isHorizontal ? '0' : '1');
        }
        else if (bpmnElement instanceof ShapeBpmnEventBasedGateway) {
            styleValues.set(BpmnStyleIdentifier.IS_INSTANTIATING, String(bpmnElement.instantiate));
            styleValues.set(BpmnStyleIdentifier.EVENT_BASED_GATEWAY_KIND, String(bpmnElement.gatewayKind));
        }
        if (!this.ignoreBpmnColors) {
            const extensions = shape.extensions;
            const fillColor = extensions.fillColor;
            if (fillColor) {
                styleValues.set(mxConstants.STYLE_FILLCOLOR, fillColor);
                if (ShapeUtil.isPoolOrLane(bpmnElement.kind)) {
                    styleValues.set(mxConstants.STYLE_SWIMLANE_FILLCOLOR, fillColor);
                }
            }
            extensions.strokeColor && styleValues.set(mxConstants.STYLE_STROKECOLOR, extensions.strokeColor);
        }
        return styleValues;
    }
    computeEdgeStyleValues(edge) {
        const styleValues = new Map();
        if (!this.ignoreBpmnColors) {
            const extensions = edge.extensions;
            extensions.strokeColor && styleValues.set(mxConstants.STYLE_STROKECOLOR, extensions.strokeColor);
        }
        return styleValues;
    }
    computeFontStyleValues(bpmnCell) {
        var _a, _b;
        const styleValues = new Map();
        const font = (_a = bpmnCell.label) === null || _a === void 0 ? void 0 : _a.font;
        if (font) {
            styleValues.set(mxConstants.STYLE_FONTFAMILY, font.name);
            styleValues.set(mxConstants.STYLE_FONTSIZE, font.size);
            styleValues.set(mxConstants.STYLE_FONTSTYLE, getFontStyleValue(font));
        }
        if (!this.ignoreBpmnColors) {
            const extensions = (_b = bpmnCell.label) === null || _b === void 0 ? void 0 : _b.extensions;
            (extensions === null || extensions === void 0 ? void 0 : extensions.color) && styleValues.set(mxConstants.STYLE_FONTCOLOR, extensions.color);
        }
        return styleValues;
    }
    computeMessageFlowIconStyle(edge) {
        const styleValues = [];
        styleValues.push(['shape', BpmnStyleIdentifier.MESSAGE_FLOW_ICON], [BpmnStyleIdentifier.IS_INITIATING, String(edge.messageVisibleKind === MessageVisibleKind.INITIATING)]);
        if (!this.ignoreBpmnColors) {
            edge.extensions.strokeColor && styleValues.push([mxConstants.STYLE_STROKECOLOR, edge.extensions.strokeColor]);
        }
        return toArrayOfMxGraphStyleEntries(styleValues).join(';');
    }
}
function computeEventShapeStyle(bpmnElement, styleValues) {
    styleValues.set(BpmnStyleIdentifier.EVENT_DEFINITION_KIND, bpmnElement.eventDefinitionKind);
    if (bpmnElement instanceof ShapeBpmnBoundaryEvent || (bpmnElement instanceof ShapeBpmnStartEvent && bpmnElement.isInterrupting !== undefined)) {
        styleValues.set(BpmnStyleIdentifier.IS_INTERRUPTING, String(bpmnElement.isInterrupting));
    }
}
function computeActivityShapeStyle(bpmnElement, styleValues) {
    if (bpmnElement instanceof ShapeBpmnSubProcess) {
        styleValues.set(BpmnStyleIdentifier.SUB_PROCESS_KIND, bpmnElement.subProcessKind);
    }
    else if (bpmnElement.kind === ShapeBpmnElementKind.TASK_RECEIVE) {
        styleValues.set(BpmnStyleIdentifier.IS_INSTANTIATING, String(bpmnElement.instantiate));
    }
    else if (bpmnElement instanceof ShapeBpmnCallActivity) {
        styleValues.set(BpmnStyleIdentifier.GLOBAL_TASK_KIND, bpmnElement.globalTaskKind);
    }
    const markers = bpmnElement.markers;
    if (markers.length > 0) {
        styleValues.set(BpmnStyleIdentifier.MARKERS, markers.join(','));
    }
}
function computeEdgeBaseStyles(edge) {
    const styles = [];
    const bpmnElement = edge.bpmnElement;
    if (bpmnElement instanceof SequenceFlow) {
        styles.push(bpmnElement.sequenceFlowKind);
    }
    if (bpmnElement instanceof AssociationFlow) {
        styles.push(bpmnElement.associationDirectionKind);
    }
    return styles;
}
function computeLabelStyleValues(bpmnCell, labelBounds) {
    const styleValues = new Map();
    const bpmnElement = bpmnCell.bpmnElement;
    if (labelBounds) {
        styleValues.set(mxConstants.STYLE_VERTICAL_ALIGN, mxConstants.ALIGN_TOP);
        if (bpmnCell.bpmnElement.kind != ShapeBpmnElementKind.TEXT_ANNOTATION) {
            styleValues.set(mxConstants.STYLE_ALIGN, mxConstants.ALIGN_CENTER);
        }
        if (bpmnCell instanceof Shape) {
            styleValues.set(mxConstants.STYLE_LABEL_WIDTH, labelBounds.width + 1);
            styleValues.set(mxConstants.STYLE_LABEL_POSITION, 'ignore');
            styleValues.set(mxConstants.STYLE_VERTICAL_LABEL_POSITION, mxConstants.ALIGN_MIDDLE);
        }
    }
    else if (bpmnCell instanceof Shape &&
        (bpmnElement instanceof ShapeBpmnSubProcess || (bpmnElement instanceof ShapeBpmnCallActivity && bpmnElement.callActivityKind === ShapeBpmnCallActivityKind.CALLING_PROCESS)) &&
        !bpmnElement.markers.includes(ShapeBpmnMarkerKind.EXPAND)) {
        styleValues.set(mxConstants.STYLE_VERTICAL_ALIGN, mxConstants.ALIGN_TOP);
    }
    return styleValues;
}
function getFontStyleValue(font) {
    let value = 0;
    if (font.isBold) {
        value += mxConstants.FONT_BOLD;
    }
    if (font.isItalic) {
        value += mxConstants.FONT_ITALIC;
    }
    if (font.isStrikeThrough) {
        value += mxConstants.FONT_STRIKETHROUGH;
    }
    if (font.isUnderline) {
        value += mxConstants.FONT_UNDERLINE;
    }
    return value;
}
function toArrayOfMxGraphStyleEntries(styleValues) {
    return styleValues.filter(([, v]) => v && v != 'undefined').map(([key, value]) => `${key}=${value}`);
}

class BpmnRenderer {
    constructor(graph, coordinatesTranslator, styleComputer) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: graph
        });
        Object.defineProperty(this, "coordinatesTranslator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: coordinatesTranslator
        });
        Object.defineProperty(this, "styleComputer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: styleComputer
        });
    }
    render(renderedModel, fitOptions) {
        this.insertShapesAndEdges(renderedModel);
        this.graph.customFit(fitOptions);
    }
    insertShapesAndEdges({ pools, lanes, subprocesses, otherFlowNodes, boundaryEvents, edges }) {
        this.graph.batchUpdate(() => {
            this.graph.getModel().clear();
            this.insertShapes(pools);
            this.insertShapes(lanes);
            this.insertShapes(subprocesses);
            this.insertShapes(otherFlowNodes);
            this.insertShapes(boundaryEvents);
            this.insertEdges(edges);
        });
    }
    insertShapes(shapes) {
        for (const shape of shapes)
            this.insertShape(shape);
    }
    getParent(bpmnElement) {
        const bpmnElementParent = this.getCell(bpmnElement.parentId);
        return bpmnElementParent !== null && bpmnElementParent !== void 0 ? bpmnElementParent : this.graph.getDefaultParent();
    }
    insertShape(shape) {
        var _a;
        const bpmnElement = shape.bpmnElement;
        const parent = this.getParent(bpmnElement);
        const bounds = shape.bounds;
        let labelBounds = (_a = shape.label) === null || _a === void 0 ? void 0 : _a.bounds;
        labelBounds = ShapeUtil.isPoolOrLane(bpmnElement.kind) ? undefined : labelBounds;
        const style = this.styleComputer.computeStyle(shape, labelBounds);
        this.insertVertex(parent, bpmnElement.id, bpmnElement.name, bounds, labelBounds, style);
    }
    insertEdges(edges) {
        var _a;
        for (const internalEdge of edges) {
            const bpmnElement = internalEdge.bpmnElement;
            const parent = this.graph.getDefaultParent();
            const source = this.getCell(bpmnElement.sourceReferenceId);
            const target = this.getCell(bpmnElement.targetReferenceId);
            const labelBounds = (_a = internalEdge.label) === null || _a === void 0 ? void 0 : _a.bounds;
            const style = this.styleComputer.computeStyle(internalEdge, labelBounds);
            const edge = this.graph.insertEdge(parent, bpmnElement.id, bpmnElement.name, source, target, style);
            this.insertWaypoints(internalEdge.waypoints, edge);
            if (labelBounds) {
                edge.geometry.width = labelBounds.width;
                edge.geometry.height = labelBounds.height;
                const edgeCenterCoordinate = this.coordinatesTranslator.computeEdgeCenter(edge);
                edge.geometry.relative = false;
                const labelBoundsRelativeCoordinateFromParent = this.coordinatesTranslator.computeRelativeCoordinates(edge.parent, new mxPoint(labelBounds.x, labelBounds.y));
                const relativeLabelX = labelBoundsRelativeCoordinateFromParent.x + labelBounds.width / 2 - edgeCenterCoordinate.x;
                const relativeLabelY = labelBoundsRelativeCoordinateFromParent.y - edgeCenterCoordinate.y;
                edge.geometry.offset = new mxPoint(relativeLabelX, relativeLabelY);
            }
            this.insertMessageFlowIconIfNeeded(internalEdge, edge);
        }
    }
    insertMessageFlowIconIfNeeded(internalEdge, edge) {
        if (internalEdge.bpmnElement instanceof MessageFlow && internalEdge.messageVisibleKind !== MessageVisibleKind.NONE) {
            const cell = this.graph.insertVertex(edge, messageFlowIconId(edge.id), undefined, 0, 0, 20, 14, this.styleComputer.computeMessageFlowIconStyle(internalEdge));
            cell.geometry.relative = true;
            cell.geometry.offset = new mxPoint(-10, -7);
        }
    }
    insertWaypoints(waypoints, edge) {
        if (waypoints) {
            edge.geometry.points = waypoints.map(waypoint => this.coordinatesTranslator.computeRelativeCoordinates(edge.parent, new mxPoint(waypoint.x, waypoint.y)));
        }
    }
    getCell(id) {
        return this.graph.getModel().getCell(id);
    }
    insertVertex(parent, id, value, bounds, labelBounds, style) {
        const vertexCoordinates = this.coordinatesTranslator.computeRelativeCoordinates(parent, new mxPoint(bounds.x, bounds.y));
        const cell = this.graph.insertVertex(parent, id, value, vertexCoordinates.x, vertexCoordinates.y, bounds.width, bounds.height, style);
        if (labelBounds) {
            const relativeLabelX = labelBounds.x - bounds.x;
            const relativeLabelY = labelBounds.y - bounds.y;
            cell.geometry.offset = new mxPoint(relativeLabelX, relativeLabelY);
        }
        return cell;
    }
}
function newBpmnRenderer(graph, options) {
    return new BpmnRenderer(graph, new CoordinatesTranslator(graph), new StyleComputer(options));
}
function messageFlowIconId(messageFlowId) {
    return `messageFlowIcon_of_${messageFlowId}`;
}

class MxGraphCustomOverlay extends mxgraph.mxCellOverlay {
    constructor(label, options) {
        super(null, '', options.position.horizontalAlign, options.position.verticalAlign, null, 'default');
        Object.defineProperty(this, "label", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: label
        });
        Object.defineProperty(this, "style", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.style = options.style;
    }
    getBounds(state) {
        const isEdge = state.view.graph.getModel().isEdge(state.cell);
        const s = state.view.scale;
        let pt;
        const w = 0;
        const h = 0;
        if (isEdge) {
            pt = this.computeEdgeBounds(state);
        }
        else {
            pt = new mxPoint();
            if (this.align == mxConstants.ALIGN_LEFT) {
                pt.x = state.x;
            }
            else if (this.align == mxConstants.ALIGN_CENTER) {
                pt.x = state.x + state.width / 2;
            }
            else {
                pt.x = state.x + state.width;
            }
            if (this.verticalAlign == mxConstants.ALIGN_TOP) {
                pt.y = state.y;
            }
            else if (this.verticalAlign == mxConstants.ALIGN_MIDDLE) {
                pt.y = state.y + state.height / 2;
            }
            else {
                pt.y = state.y + state.height;
            }
        }
        return new mxRectangle(Math.round(pt.x - (w * this.defaultOverlap - this.offset.x) * s), Math.round(pt.y - (h * this.defaultOverlap - this.offset.y) * s), w * s, h * s);
    }
    computeEdgeBounds(state) {
        const pts = state.absolutePoints;
        if (this.align == mxConstants.ALIGN_LEFT) {
            return pts[0];
        }
        else if (this.align == mxConstants.ALIGN_CENTER) {
            if (pts.length % 2 == 1) {
                return pts[Math.floor(pts.length / 2)];
            }
            else {
                const index = pts.length / 2;
                const p0 = pts[index - 1];
                const p1 = pts[index];
                return new mxPoint(p0.x + (p1.x - p0.x) / 2, p0.y + (p1.y - p0.y) / 2);
            }
        }
        else {
            return pts.at(-1);
        }
    }
}

class OverlayBadgeShape extends mxgraph.mxText {
    constructor(value, bounds, style) {
        super(value, bounds, undefined, undefined, style.font.color, undefined, style.font.size, undefined, undefined, undefined, undefined, undefined, undefined, undefined, style.fill.color, style.stroke.color);
        this.fillOpacity = style.fill.opacity;
        this.strokewidth = style.stroke.width;
    }
}

function isFlowKind(kind) {
    return Object.values(FlowKind)
        .map(value => value)
        .includes(kind);
}

function computeAllBpmnClassNamesOfCell(cell, isLabel) {
    return computeAllBpmnClassNames(cell.style, isLabel);
}
function computeAllBpmnClassNames(style, isLabel) {
    const classes = [];
    const styleElements = style.split(';');
    const pseudoBpmnElementKind = styleElements[0];
    const bpmnElementKind = pseudoBpmnElementKind.replace(/shape=bpmn./g, '');
    const typeClasses = new Map();
    typeClasses.set('bpmn-type-activity', ShapeUtil.isActivity(bpmnElementKind));
    typeClasses.set('bpmn-type-container', ShapeUtil.isPoolOrLane(bpmnElementKind));
    typeClasses.set('bpmn-type-event', ShapeUtil.isEvent(bpmnElementKind));
    typeClasses.set('bpmn-type-flow', isFlowKind(bpmnElementKind));
    typeClasses.set('bpmn-type-gateway', ShapeUtil.isGateway(bpmnElementKind));
    typeClasses.set('bpmn-type-task', ShapeUtil.isTask(bpmnElementKind));
    for (const [className] of [...typeClasses].filter(([, isType]) => isType))
        classes.push(className);
    classes.push(computeBpmnBaseClassName(bpmnElementKind));
    for (const [key, value] of styleElements.map(entry => {
        const elements = entry.split('=');
        return [elements[0], elements[1]];
    })) {
        switch (key) {
            case BpmnStyleIdentifier.EVENT_DEFINITION_KIND: {
                classes.push(`bpmn-event-def-${value}`);
                break;
            }
            case BpmnStyleIdentifier.EVENT_BASED_GATEWAY_KIND: {
                classes.push(`bpmn-gateway-kind-${value.toLowerCase()}`);
                break;
            }
            case BpmnStyleIdentifier.IS_INITIATING: {
                classes.push(value == 'true' ? 'bpmn-icon-initiating' : 'bpmn-icon-non-initiating');
                break;
            }
            case BpmnStyleIdentifier.SUB_PROCESS_KIND: {
                classes.push(`bpmn-sub-process-${value.toLowerCase()}`);
                break;
            }
            case BpmnStyleIdentifier.GLOBAL_TASK_KIND: {
                classes.push(computeBpmnBaseClassName(value));
                break;
            }
        }
    }
    if (isLabel) {
        classes.push('bpmn-label');
    }
    return classes;
}
function computeBpmnBaseClassName(bpmnElementKind) {
    return bpmnElementKind ? 'bpmn-' + bpmnElementKind.replace(/([A-Z])/g, g => '-' + g[0].toLowerCase()) : '';
}

const overrideCreateSvgCanvas = function (shape) {
    const originalShapeCreateSvgCanvas = shape.createSvgCanvas;
    shape.createSvgCanvas = function () {
        var _a;
        const canvas = originalShapeCreateSvgCanvas.bind(this)();
        const originalCanvasGetTextCss = canvas.getTextCss;
        canvas.getTextCss = function () {
            const originalPointerEvents = this.pointerEvents;
            this.pointerEvents = false;
            const textCss = originalCanvasGetTextCss.bind(this)();
            this.pointerEvents = originalPointerEvents;
            return textCss;
        };
        if ((_a = this.state) === null || _a === void 0 ? void 0 : _a.cell) {
            const cell = this.state.cell;
            const allBpmnClassNames = computeAllBpmnClassNamesOfCell(cell, this.dialect === mxConstants.DIALECT_STRICTHTML);
            const extraCssClasses = this.state.style[BpmnStyleIdentifier.EXTRA_CSS_CLASSES];
            if (typeof extraCssClasses == 'string') {
                allBpmnClassNames.push(...extraCssClasses.split(','));
            }
            this.node.setAttribute('class', allBpmnClassNames.join(' '));
            this.node.dataset.bpmnId = this.state.cell.id;
        }
        return canvas;
    };
};

class BpmnCellRenderer extends mxgraph.mxCellRenderer {
    constructor(iconPainter) {
        super();
        Object.defineProperty(this, "iconPainter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: iconPainter
        });
    }
    createCellOverlays(state) {
        const graph = state.view.graph;
        const overlays = graph.getCellOverlays(state.cell);
        let dict = null;
        if (overlays != null) {
            dict = new mxgraph.mxDictionary();
            for (const currentOverlay of overlays) {
                const shape = state.overlays == null ? null : state.overlays.remove(currentOverlay);
                if (shape != null) {
                    dict.put(currentOverlay, shape);
                    continue;
                }
                let overlayShape;
                if (currentOverlay instanceof MxGraphCustomOverlay) {
                    overlayShape = new OverlayBadgeShape(currentOverlay.label, new mxRectangle(0, 0, 0, 0), currentOverlay.style);
                }
                else {
                    overlayShape = new mxgraph.mxImageShape(new mxRectangle(0, 0, 0, 0), currentOverlay.image.src);
                    overlayShape.preserveImageAspect = false;
                }
                overlayShape.dialect = state.view.graph.dialect;
                overlayShape.overlay = currentOverlay;
                this.initializeOverlay(state, overlayShape);
                this.installCellOverlayListeners(state, currentOverlay, overlayShape);
                if (currentOverlay.cursor != null) {
                    overlayShape.node.style.cursor = currentOverlay.cursor;
                }
                if (overlayShape instanceof OverlayBadgeShape) {
                    overlayShape.node.classList.add('overlay-badge');
                    overlayShape.node.dataset.bpmnId = state.cell.id;
                }
                dict.put(currentOverlay, overlayShape);
            }
        }
        if (state.overlays != null) {
            state.overlays.visit(function (_id, shape) {
                shape.destroy();
            });
        }
        state.overlays = dict;
    }
    createShape(state) {
        const shape = super.createShape(state);
        if ('iconPainter' in shape) {
            shape.iconPainter = this.iconPainter;
        }
        overrideCreateSvgCanvas(shape);
        return shape;
    }
    createLabel(state, value) {
        super.createLabel(state, value);
        overrideCreateSvgCanvas(state.text);
    }
}

function computeScaledIconSize(initialIconSize, iconStyleConfiguration, shapeConfiguration, ratioFromShape) {
    let iconWidthProportionalToShape;
    let iconHeightProportionalToShape;
    if (initialIconSize.height < initialIconSize.width || (initialIconSize.height == initialIconSize.width && shapeConfiguration.width <= shapeConfiguration.height)) {
        iconWidthProportionalToShape = shapeConfiguration.width;
        iconHeightProportionalToShape = (shapeConfiguration.width * initialIconSize.height) / initialIconSize.width;
    }
    else {
        iconWidthProportionalToShape = (shapeConfiguration.height * initialIconSize.width) / initialIconSize.height;
        iconHeightProportionalToShape = shapeConfiguration.height;
    }
    const inset = iconStyleConfiguration.strokeWidth ? (iconStyleConfiguration.strokeWidth - 1) * 2 : 0;
    const paintIconWidth = iconWidthProportionalToShape * ratioFromShape - inset;
    const paintIconHeight = iconHeightProportionalToShape * ratioFromShape - inset;
    return { width: paintIconWidth, height: paintIconHeight };
}
class BpmnCanvas {
    constructor({ canvas, shapeConfig, iconConfig }) {
        Object.defineProperty(this, "canvas", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "iconOriginalSize", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "scaleX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "scaleY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "iconPaintingOriginX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(this, "iconPaintingOriginY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(this, "shapeConfiguration", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.canvas = canvas;
        this.shapeConfiguration = shapeConfig;
        this.iconOriginalSize = iconConfig.originalSize;
        const ratioFromShape = iconConfig.ratioFromParent;
        if (ratioFromShape) {
            const scaledIconSize = computeScaledIconSize(this.iconOriginalSize, iconConfig.styleConfig, this.shapeConfiguration, ratioFromShape);
            this.scaleX = scaledIconSize.width / this.iconOriginalSize.width;
            this.scaleY = scaledIconSize.height / this.iconOriginalSize.height;
        }
        else {
            this.scaleX = 1;
            this.scaleY = 1;
        }
        this.updateCanvasStyle(iconConfig.styleConfig);
        iconConfig.setIconOriginFunct(this);
    }
    setIconOriginToShapeTopLeftProportionally(shapeDimensionProportion) {
        const shape = this.shapeConfiguration;
        this.iconPaintingOriginX = shape.x + shape.width / shapeDimensionProportion;
        this.iconPaintingOriginY = shape.y + shape.height / shapeDimensionProportion;
    }
    setIconOriginToShapeTopLeft(topMargin = StyleDefault.SHAPE_ACTIVITY_TOP_MARGIN, leftMargin = StyleDefault.SHAPE_ACTIVITY_LEFT_MARGIN) {
        const shape = this.shapeConfiguration;
        this.iconPaintingOriginX = shape.x + leftMargin;
        this.iconPaintingOriginY = shape.y + topMargin;
    }
    setIconOriginForIconCentered() {
        const shape = this.shapeConfiguration;
        this.iconPaintingOriginX = shape.x + (shape.width - this.iconOriginalSize.width * this.scaleX) / 2;
        this.iconPaintingOriginY = shape.y + (shape.height - this.iconOriginalSize.height * this.scaleY) / 2;
    }
    setIconOriginForIconBottomCentered(bottomMargin = StyleDefault.SHAPE_ACTIVITY_BOTTOM_MARGIN) {
        const shape = this.shapeConfiguration;
        this.iconPaintingOriginX = shape.x + (shape.width - this.iconOriginalSize.width * this.scaleX) / 2;
        this.iconPaintingOriginY = shape.y + (shape.height - this.iconOriginalSize.height * this.scaleY - bottomMargin);
    }
    translateIconOrigin(dx, dy) {
        this.iconPaintingOriginX += this.scaleX * dx;
        this.iconPaintingOriginY += this.scaleY * dy;
    }
    computeScaleFromOriginX(x) {
        return this.iconPaintingOriginX + x * this.scaleX;
    }
    computeScaleFromOriginY(y) {
        return this.iconPaintingOriginY + y * this.scaleY;
    }
    updateCanvasStyle({ isFilled, strokeColor, fillColor, strokeWidth }) {
        if (isFilled) {
            this.canvas.setFillColor(strokeColor);
        }
        else {
            this.canvas.setFillColor(fillColor);
        }
        this.canvas.setStrokeWidth(strokeWidth);
    }
    arcTo(rx, ry, angle, largeArcFlag, sweepFlag, x, y) {
        this.canvas.arcTo(rx * this.scaleX, ry * this.scaleY, angle, largeArcFlag, sweepFlag, this.computeScaleFromOriginX(x), this.computeScaleFromOriginY(y));
    }
    begin() {
        this.canvas.begin();
    }
    close() {
        this.canvas.close();
    }
    curveTo(x1, y1, x2, y2, x3, y3) {
        this.canvas.curveTo(this.computeScaleFromOriginX(x1), this.computeScaleFromOriginY(y1), this.computeScaleFromOriginX(x2), this.computeScaleFromOriginY(y2), this.computeScaleFromOriginX(x3), this.computeScaleFromOriginY(y3));
    }
    fill() {
        this.canvas.fill();
    }
    fillAndStroke() {
        this.canvas.fillAndStroke();
    }
    setFillColor(fillColor) {
        this.canvas.setFillColor(fillColor);
    }
    stroke() {
        this.canvas.stroke();
    }
    setStrokeColor(color) {
        this.canvas.setStrokeColor(color);
    }
    setRoundLineJoin() {
        this.canvas.setLineJoin('round');
    }
    lineTo(x, y) {
        this.canvas.lineTo(this.computeScaleFromOriginX(x), this.computeScaleFromOriginY(y));
    }
    moveTo(x, y) {
        this.canvas.moveTo(this.computeScaleFromOriginX(x), this.computeScaleFromOriginY(y));
    }
    rect(x, y, w, h) {
        this.canvas.rect(this.computeScaleFromOriginX(x), this.computeScaleFromOriginY(y), w * this.scaleX, h * this.scaleY);
    }
    roundrect(x, y, w, h, dx, dy) {
        this.canvas.roundrect(this.computeScaleFromOriginX(x), this.computeScaleFromOriginY(y), w * this.scaleX, h * this.scaleY, dx, dy);
    }
    ellipse(x, y, w, h) {
        this.canvas.ellipse(this.computeScaleFromOriginX(x), this.computeScaleFromOriginY(y), w * this.scaleX, h * this.scaleY);
    }
    rotateOnIconCenter(theta) {
        const rotationCenterX = this.iconPaintingOriginX + (this.iconOriginalSize.width / 2) * this.scaleX;
        const rotationCenterY = this.iconPaintingOriginY + (this.iconOriginalSize.height / 2) * this.scaleY;
        this.canvas.rotate(theta, false, false, rotationCenterX, rotationCenterY);
    }
}

function buildPaintParameter({ canvas, x, y, width, height, shape, ratioFromParent, isFilled, iconStrokeWidth, }) {
    const shapeStrokeWidth = mxUtils.getValue(shape.style, mxConstants.STYLE_STROKEWIDTH, StyleDefault.STROKE_WIDTH_THIN);
    const fillColor = mxUtils.getValue(shape.style, mxConstants.STYLE_FILLCOLOR, StyleDefault.DEFAULT_FILL_COLOR);
    const strokeColor = mxUtils.getValue(shape.style, mxConstants.STYLE_STROKECOLOR, StyleDefault.DEFAULT_STROKE_COLOR);
    const margin = mxUtils.getValue(shape.style, mxConstants.STYLE_MARGIN, StyleDefault.DEFAULT_MARGIN);
    ratioFromParent !== null && ratioFromParent !== void 0 ? ratioFromParent : (ratioFromParent = 0.25);
    isFilled !== null && isFilled !== void 0 ? isFilled : (isFilled = false);
    iconStrokeWidth !== null && iconStrokeWidth !== void 0 ? iconStrokeWidth : (iconStrokeWidth = 0);
    return {
        canvas,
        ratioFromParent,
        setIconOriginFunct: (internalCanvas) => internalCanvas.setIconOriginForIconCentered(),
        shapeConfig: { x, y, width, height, strokeWidth: shapeStrokeWidth },
        iconStyleConfig: { isFilled, fillColor, strokeColor, strokeWidth: iconStrokeWidth, margin },
    };
}
class IconPainter {
    newBpmnCanvas({ canvas, ratioFromParent, setIconOriginFunct, shapeConfig, iconStyleConfig }, originalIconSize) {
        return new BpmnCanvas({
            canvas,
            shapeConfig,
            iconConfig: {
                originalSize: originalIconSize,
                styleConfig: iconStyleConfig,
                ratioFromParent,
                setIconOriginFunct,
            },
        });
    }
    paintEnvelopeIcon(paintParameter) {
        const originalIconSize = { width: 485.41, height: 321.76 };
        const canvas = this.newBpmnCanvas(paintParameter, originalIconSize);
        const w = originalIconSize.width;
        const h = originalIconSize.height;
        canvas.rect(0, 0, w, h);
        canvas.fillAndStroke();
        const { iconStyleConfig } = paintParameter;
        if (iconStyleConfig.isFilled) {
            canvas.setStrokeColor(iconStyleConfig.fillColor);
        }
        canvas.begin();
        canvas.moveTo(0, 0);
        canvas.lineTo(w * 0.5, h * 0.6);
        canvas.lineTo(w, 0);
        canvas.moveTo(0, h);
        canvas.lineTo(w / 3, h * 0.45);
        canvas.moveTo(w, h);
        canvas.lineTo((w * 2) / 3, h * 0.45);
        canvas.stroke();
    }
    paintCircleIcon(paintParameter) {
        const originalIconSize = { width: paintParameter.shapeConfig.width, height: paintParameter.shapeConfig.height };
        const canvas = this.newBpmnCanvas(paintParameter, originalIconSize);
        const w = originalIconSize.width;
        const h = originalIconSize.height;
        if (w > 0 && h > 0) {
            canvas.ellipse(0, 0, w, h);
        }
        if (paintParameter.iconStyleConfig.isFilled) {
            canvas.fillAndStroke();
        }
        else {
            canvas.stroke();
        }
    }
    paintClockIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 152, width: 152 });
        canvas.begin();
        canvas.moveTo(184, 60);
        canvas.curveTo(188.4, 60, 192, 56.4, 192, 52);
        canvas.lineTo(192, 48);
        canvas.curveTo(192, 40, 188.4, 40, 184, 40);
        canvas.curveTo(179.6, 40, 176, 43.6, 176, 48);
        canvas.lineTo(176, 52);
        canvas.curveTo(176, 56.4, 179.6, 60, 184, 60);
        canvas.close();
        canvas.moveTo(184, 308);
        canvas.curveTo(179.6, 308, 176, 311.6, 176, 316);
        canvas.lineTo(176, 320);
        canvas.curveTo(176, 324.4, 179.6, 328, 184, 328);
        canvas.curveTo(188.4, 328, 192, 324.4, 192, 320);
        canvas.lineTo(192, 316);
        canvas.curveTo(192, 311.6, 188.4, 308, 184, 308);
        canvas.close();
        canvas.moveTo(52, 176);
        canvas.lineTo(48, 176);
        canvas.curveTo(43.6, 176, 40, 179.6, 40, 184);
        canvas.curveTo(40, 188.4, 43.6, 192, 48, 192);
        canvas.lineTo(52, 192);
        canvas.curveTo(56.4, 192, 69, 188.4, 60, 184);
        canvas.curveTo(60, 179.6, 56.4, 176, 52, 176);
        canvas.close();
        canvas.moveTo(320, 176);
        canvas.lineTo(316, 176);
        canvas.curveTo(311.6, 176, 308, 179.6, 308, 184);
        canvas.curveTo(308, 188.4, 311.6, 192, 316, 192);
        canvas.lineTo(320, 192);
        canvas.curveTo(324.4, 192, 328, 188.4, 328, 184);
        canvas.curveTo(328, 179.6, 324.4, 176, 320, 176);
        canvas.moveTo(93.6, 82.4);
        canvas.curveTo(90.4, 79.2, 85.6, 79.2, 82.4, 82.4);
        canvas.curveTo(79.2, 85.6, 79.2, 90.4, 82.4, 93.6);
        canvas.lineTo(85.2, 96.4);
        canvas.curveTo(86.8, 98, 88.8, 98.8, 90.8, 98.8);
        canvas.curveTo(92.8, 98.8, 94.4, 98, 96.4, 96.4);
        canvas.curveTo(99.6, 93.2, 99.6, 88.4, 96.4, 85.2);
        canvas.lineTo(93.6, 82.4);
        canvas.moveTo(85.2, 271.6);
        canvas.lineTo(82.4, 274.4);
        canvas.curveTo(79.2, 277.6, 79.2, 282.4, 82.4, 285.6);
        canvas.curveTo(84, 287.2, 86, 288, 88, 288);
        canvas.curveTo(90, 288, 92, 287.2, 93.6, 285.6);
        canvas.lineTo(96.4, 282.8);
        canvas.curveTo(99.6, 279.6, 99.6, 274.8, 96.4, 271.6);
        canvas.curveTo(93.2, 268.4, 88.4, 268.4, 85.2, 271.6);
        canvas.moveTo(274.4, 82.4);
        canvas.lineTo(271.6, 85.2);
        canvas.curveTo(268.4, 88.4, 268.4, 93.2, 271.6, 96.4);
        canvas.curveTo(273.298, 98, 275.2, 98.8, 277.2, 98.8);
        canvas.curveTo(279.2, 98.8, 281.2, 98, 282.8, 96.4);
        canvas.lineTo(285.6, 93.6);
        canvas.curveTo(288.8, 90.4, 288.8, 85.6, 285.6, 82.4);
        canvas.curveTo(282.4, 79.2, 277.6, 79.2, 274.4, 82.4);
        canvas.moveTo(192, 180.8);
        canvas.lineTo(192, 108);
        canvas.curveTo(192, 103.6, 188.4, 100, 184, 100);
        canvas.curveTo(179.6, 100, 176, 103.6, 176, 108);
        canvas.lineTo(176, 184);
        canvas.curveTo(176, 186, 176.8, 188, 178.4, 189.6);
        canvas.lineTo(266, 277.2);
        canvas.curveTo(267.6, 278.8, 269.6, 279.6, 271.6, 279.6);
        canvas.curveTo(273.6, 279.6, 275.6, 278.8, 277.2, 277.2);
        canvas.curveTo(280.4, 274, 280.4, 269.2, 277.2, 266);
        canvas.lineTo(192, 180.8);
        canvas.moveTo(184, 0);
        canvas.curveTo(82.4, 0, 0, 82.4, 0, 184);
        canvas.curveTo(0, 285.6, 82.4, 368, 184, 368);
        canvas.curveTo(285.6, 368, 368, 285.6, 368, 184);
        canvas.curveTo(368, 82.4, 285.6, 0, 184, 0);
        canvas.moveTo(184, 352);
        canvas.curveTo(91.2, 352, 16, 276.8, 16, 184);
        canvas.curveTo(16, 91.2, 91.2, 16, 184, 16);
        canvas.curveTo(276.8, 16, 352, 91.2, 352, 184);
        canvas.curveTo(352, 276.8, 276.8, 352, 184, 352);
        canvas.fillAndStroke();
    }
    paintTriangleIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 735, width: 849 });
        canvas.begin();
        canvas.moveTo(497, 55);
        canvas.lineTo(817, 609);
        canvas.curveTo(849, 665, 808, 735, 744, 735);
        canvas.lineTo(105, 735);
        canvas.curveTo(40, 735, 0, 665, 32, 609);
        canvas.lineTo(352, 55);
        canvas.curveTo(384, 0, 465, 0, 497, 55);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintUpArrowheadIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 50, width: 40 });
        canvas.begin();
        canvas.moveTo(0, 49.5);
        canvas.lineTo(19.5, 1);
        canvas.curveTo(19.75, 0.25, 20, 0, 20.25, 0.25);
        canvas.lineTo(40, 49.5);
        canvas.curveTo(40, 49.5, 39.75, 50, 39.6, 49.75);
        canvas.lineTo(20, 30);
        canvas.lineTo(0.4, 49.75);
        canvas.curveTo(0.4, 49.75, 0.25, 50, 0, 49.5);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintDoubleLeftArrowheadsIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 53.5, width: 105 });
        canvas.begin();
        canvas.moveTo(91.4, 0);
        canvas.curveTo(91.4, 0, 91.2, 0, 91, 0.2);
        canvas.lineTo(50, 25);
        canvas.curveTo(47.9, 25.8, 46.7, 26.6, 46.4, 27.3);
        canvas.lineTo(46.4, 0);
        canvas.curveTo(46.4, 0, 46.2, 0, 46, 0.2);
        canvas.lineTo(4.9, 25);
        canvas.curveTo(2, 26.2, 0, 27.3, 4.9, 28.5);
        canvas.lineTo(45.8, 53);
        canvas.curveTo(46, 53.3, 46.2, 53.5, 46.4, 53.5);
        canvas.lineTo(46.4, 27);
        canvas.curveTo(46.6, 27.3, 47.8, 28.1, 49.9, 29.9);
        canvas.lineTo(90.8, 53.3);
        canvas.curveTo(91, 53.3, 91.2, 53.5, 91.4, 53.5);
        canvas.lineTo(91.4, 0);
        canvas.close();
        canvas.fillAndStroke();
    }
    drawCrossIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 1, width: 1 });
        canvas.begin();
        canvas.moveTo(0.38, 0);
        canvas.lineTo(0.62, 0);
        canvas.lineTo(0.62, 0.38);
        canvas.lineTo(1, 0.38);
        canvas.lineTo(1, 0.62);
        canvas.lineTo(0.62, 0.62);
        canvas.lineTo(0.62, 1);
        canvas.lineTo(0.38, 1);
        canvas.lineTo(0.38, 0.62);
        canvas.lineTo(0, 0.62);
        canvas.lineTo(0, 0.38);
        canvas.lineTo(0.38, 0.38);
        canvas.close();
        return canvas;
    }
    paintListIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 60, width: 60 });
        canvas.begin();
        canvas.moveTo(0, 0);
        canvas.lineTo(60, 0);
        canvas.lineTo(60, 60);
        canvas.lineTo(0, 60);
        canvas.lineTo(0, 0);
        canvas.close();
        canvas.moveTo(5, 5);
        canvas.lineTo(55, 5);
        canvas.close();
        canvas.moveTo(5, 21.6);
        canvas.lineTo(55, 21.6);
        canvas.close();
        canvas.moveTo(5, 38.3);
        canvas.lineTo(55, 38.3);
        canvas.close();
        canvas.moveTo(5, 55);
        canvas.lineTo(55, 55);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintXCrossIcon(paintParameter) {
        const canvas = this.drawCrossIcon(paintParameter);
        canvas.rotateOnIconCenter(45);
        canvas.fillAndStroke();
    }
    paintPlusCrossIcon(paintParameter) {
        this.drawCrossIcon(paintParameter).fillAndStroke();
    }
    paintAsteriskIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 1, width: 1 });
        drawVerticalLine(canvas);
        canvas.fillAndStroke();
        drawVerticalLine(canvas);
        canvas.rotateOnIconCenter(60);
        canvas.fillAndStroke();
        drawVerticalLine(canvas);
        canvas.rotateOnIconCenter(240);
        canvas.fillAndStroke();
    }
    paintPersonIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(Object.assign(Object.assign({}, paintParameter), { iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { isFilled: true }) }), { height: 239.68, width: 143.61 });
        canvas.begin();
        canvas.moveTo(124.31, 150.29);
        canvas.lineTo(99.66, 141.03);
        canvas.arcTo(6.43, 6.43, 0, 0, 1, 95.51, 135.03);
        canvas.lineTo(95.51, 130.66);
        canvas.arcTo(47.75, 47.75, 0, 0, 0, 119.51, 89.25);
        canvas.lineTo(119.51, 71.25);
        canvas.arcTo(47.62, 47.62, 0, 0, 0, 101.18, 33.64);
        canvas.arcTo(29.35, 29.35, 0, 0, 0, 101.52, 29.14);
        canvas.arcTo(29.68, 29.68, 0, 0, 0, 42.17, 29.14);
        canvas.arcTo(29.24, 29.24, 0, 0, 0, 42.53, 33.63);
        canvas.arcTo(47.65, 47.65, 0, 0, 0, 24.14, 71.23);
        canvas.lineTo(24.14, 89.23);
        canvas.arcTo(47.7, 47.7, 0, 0, 0, 48.19, 130.63);
        canvas.lineTo(48.19, 135.03);
        canvas.arcTo(6.43, 6.43, 0, 0, 1, 44.03, 141.03);
        canvas.lineTo(19.31, 150.29);
        canvas.arcTo(29.81, 29.81, 0, 0, 0, 0.09, 178.03);
        canvas.lineTo(0.09, 233.51);
        canvas.arcTo(5.63, 5.63, 0, 1, 0, 11.34, 233.51);
        canvas.lineTo(11.34, 178.03);
        canvas.arcTo(18.19, 18.19, 0, 0, 1, 11.57, 175.17);
        canvas.lineTo(20.5, 184.11);
        canvas.arcTo(12.32, 12.32, 0, 0, 1, 24.14, 192.89);
        canvas.lineTo(24.14, 233.51);
        canvas.arcTo(5.63, 5.63, 0, 1, 0, 35.39, 233.51);
        canvas.lineTo(35.39, 192.93);
        canvas.arcTo(23.5, 23.5, 0, 0, 0, 28.46, 176.2);
        canvas.lineTo(17.04, 164.78);
        canvas.arcTo(18.34, 18.34, 0, 0, 1, 23.29, 160.78);
        canvas.lineTo(43.65, 153.15);
        canvas.lineTo(66.22, 175.72);
        canvas.lineTo(66.22, 233.51);
        canvas.arcTo(5.63, 5.63, 0, 1, 0, 77.47, 233.51);
        canvas.lineTo(77.47, 175.76);
        canvas.lineTo(100.04, 153.19);
        canvas.lineTo(120.4, 160.82);
        canvas.arcTo(18.39, 18.39, 0, 0, 1, 126.65, 164.82);
        canvas.lineTo(115.24, 176.24);
        canvas.arcTo(23.5, 23.5, 0, 0, 0, 108.31, 192.93);
        canvas.lineTo(108.31, 233.55);
        canvas.arcTo(5.63, 5.63, 0, 1, 0, 119.56, 233.55);
        canvas.lineTo(119.56, 192.93);
        canvas.arcTo(12.35, 12.35, 0, 0, 1, 123.19, 184.15);
        canvas.lineTo(132.13, 175.22);
        canvas.arcTo(18, 18, 0, 0, 1, 132.36, 178.08);
        canvas.lineTo(132.36, 233.56);
        canvas.arcTo(5.63, 5.63, 0, 0, 0, 143.61, 233.56);
        canvas.lineTo(143.61, 178.03);
        canvas.arcTo(29.81, 29.81, 0, 0, 0, 124.31, 150.29);
        canvas.close();
        canvas.moveTo(71.85, 10.72);
        canvas.arcTo(18.46, 18.46, 0, 0, 1, 90.17, 27.18);
        canvas.arcTo(47.68, 47.68, 0, 0, 0, 53.53, 27.18);
        canvas.arcTo(18.44, 18.44, 0, 0, 1, 71.85, 10.72);
        canvas.close();
        canvas.moveTo(35.39, 71.23);
        canvas.arcTo(36.46, 36.46, 0, 0, 1, 108.31, 71.23);
        canvas.lineTo(108.31, 77.4);
        canvas.curveTo(82.12, 75.4, 56.97, 60.55, 56.71, 60.4);
        canvas.arcTo(5.62, 5.62, 0, 0, 0, 48.78, 62.71);
        canvas.curveTo(46.24, 67.79, 40.45, 71.89, 35.39, 74.62);
        canvas.close();
        canvas.moveTo(35.39, 89.23);
        canvas.lineTo(35.39, 87.08);
        canvas.curveTo(40.55, 84.85, 49.73, 80.08, 55.67, 72.66);
        canvas.curveTo(64.83, 77.46, 85.92, 87.21, 108.31, 88.66);
        canvas.lineTo(108.31, 89.24);
        canvas.arcTo(36.46, 36.46, 0, 1, 1, 35.39, 89.24);
        canvas.close();
        canvas.moveTo(71.85, 165.45);
        canvas.lineTo(54.06, 147.69);
        canvas.arcTo(17.7, 17.7, 0, 0, 0, 59.43, 135.32);
        canvas.arcTo(47.57, 47.57, 0, 0, 0, 84.27, 135.32);
        canvas.arcTo(17.7, 17.7, 0, 0, 0, 89.64, 147.69);
        canvas.close();
        canvas.fill();
    }
    paintGearIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { height: 100, width: 100 });
        paintGearIconBackground(canvas);
        canvas.translateIconOrigin(14, 14);
        paintGearIconForeground(canvas);
    }
    paintExpandIcon(paintParameter) {
        const originalIconSize = { width: 16, height: 16 };
        const canvas = this.newBpmnCanvas(paintParameter, originalIconSize);
        const w = originalIconSize.width;
        const h = originalIconSize.height;
        canvas.roundrect(0, 0, w, h, 2, 2);
        canvas.stroke();
        canvas.begin();
        canvas.moveTo(w / 2, h / 4);
        canvas.lineTo(w / 2, (h * 3) / 4);
        canvas.close();
        canvas.moveTo(w / 4, h / 2);
        canvas.lineTo((w * 3) / 4, h / 2);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintLoopIcon(paintParameter) {
        const { iconStyleConfig } = paintParameter;
        iconStyleConfig.fillColor = iconStyleConfig.strokeColor;
        const canvas = this.newBpmnCanvas(paintParameter, { width: 22.49, height: 21.62 });
        canvas.begin();
        canvas.moveTo(5.5, 19.08);
        canvas.arcTo(8, 8, 0, 1, 1, 10.5, 21.08);
        canvas.stroke();
        canvas.begin();
        canvas.moveTo(7.5, 14.08);
        canvas.lineTo(5.75, 19.08);
        canvas.lineTo(0, 17.08);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintSequentialMultiInstanceIcon(paintParameter) {
        const originalIconSize = { width: 16, height: 16 };
        const bpmnCanvas = this.newBpmnCanvas(paintParameter, originalIconSize);
        const { canvas, iconStyleConfig } = paintParameter;
        canvas.setFillColor(iconStyleConfig.strokeColor);
        const barWidth = originalIconSize.width;
        const barHeight = originalIconSize.height / 5;
        bpmnCanvas.rect(0, 0, barWidth, barHeight);
        bpmnCanvas.fill();
        bpmnCanvas.rect(0, 2 * barHeight, barWidth, barHeight);
        bpmnCanvas.fill();
        bpmnCanvas.rect(0, 4 * barHeight, barWidth, barHeight);
        bpmnCanvas.fill();
    }
    paintParallelMultiInstanceIcon(paintParameter) {
        const originalIconSize = { width: 16, height: 16 };
        const bpmnCanvas = this.newBpmnCanvas(paintParameter, originalIconSize);
        const { canvas, iconStyleConfig } = paintParameter;
        canvas.setFillColor(iconStyleConfig.strokeColor);
        const barWidth = originalIconSize.width / 5;
        const barHeight = originalIconSize.height;
        bpmnCanvas.begin();
        bpmnCanvas.rect(0, 0, barWidth, barHeight);
        bpmnCanvas.fill();
        bpmnCanvas.rect(2 * barWidth, 0, barWidth, barHeight);
        bpmnCanvas.fill();
        bpmnCanvas.rect(4 * barWidth, 0, barWidth, barHeight);
        bpmnCanvas.fill();
    }
    paintRightArrowIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { width: 512, height: 415.23 });
        canvas.setRoundLineJoin();
        canvas.begin();
        canvas.moveTo(512, 207.61);
        canvas.lineTo(304.38, 0);
        canvas.lineTo(304.38, 135.39);
        canvas.lineTo(0, 135.39);
        canvas.lineTo(0, 279.84);
        canvas.lineTo(304.38, 279.84);
        canvas.lineTo(304.38, 415.23);
        canvas.lineTo(512, 207.61);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintErrorIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { width: 72.44, height: 71.82 });
        canvas.begin();
        canvas.moveTo(0, 53.32);
        canvas.lineTo(19.48, 0);
        canvas.lineTo(19.48, 0);
        canvas.lineTo(50.85, 40.07);
        canvas.lineTo(72.44, 18.21);
        canvas.lineTo(53.12, 71.82);
        canvas.lineTo(22.5, 31.37);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintHandIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { width: 343.65, height: 354.12 });
        canvas.begin();
        canvas.moveTo(231.66, 336.39);
        canvas.curveTo(240.84, 316.9, 220.53, 306.92, 220.53, 306.92);
        canvas.curveTo(215.2, 303.67, 188.58, 287.43, 140.67, 258.19);
        canvas.lineTo(146.33, 248.39);
        canvas.curveTo(223.98, 269.38, 267.12, 281.04, 275.75, 283.38);
        canvas.curveTo(275.75, 283.38, 297.25, 288, 301.42, 267.77);
        canvas.curveTo(306.34, 245.29, 288.32, 238.63, 288.32, 238.63);
        canvas.curveTo(279.91, 236.44, 237.86, 225.48, 162.18, 205.75);
        canvas.lineTo(165.2, 194.8);
        canvas.curveTo(255.88, 204.4, 306.27, 209.73, 316.34, 210.8);
        canvas.curveTo(316.34, 210.8, 339.89, 212.16, 341.76, 189.55);
        canvas.curveTo(343.65, 166.93, 320.5, 164.13, 320.5, 164.13);
        canvas.curveTo(310.43, 163.1, 260.04, 157.99, 169.35, 148.77);
        canvas.lineTo(169.35, 138.97);
        canvas.curveTo(253.41, 132.12, 300.11, 128.32, 309.45, 127.56);
        canvas.curveTo(309.45, 127.56, 332.27, 122.38, 332.27, 102.61);
        canvas.curveTo(332.27, 82.85, 305.48, 81.87, 305.48, 81.87);
        canvas.curveTo(293.99, 82.2, 236.54, 83.88, 133.13, 86.9);
        canvas.lineTo(127.61, 81.87);
        canvas.curveTo(145.3, 59.39, 155.12, 46.9, 157.09, 44.41);
        canvas.curveTo(157.09, 44.41, 171.12, 26.8, 156.78, 12.72);
        canvas.curveTo(143.83, 0, 124.08, 14.49, 124.08, 14.49);
        canvas.curveTo(116.45, 19.41, 78.35, 44.06, 9.77, 88.43);
        canvas.lineTo(0, 251.94);
        canvas.curveTo(122.84, 308.79, 191.09, 340.37, 204.74, 346.69);
        canvas.curveTo(204.74, 346.69, 222.91, 354.12, 231.66, 336.39);
        canvas.fillAndStroke();
    }
    paintScriptIcon(paintParameter) {
        paintParameter.iconStyleConfig.fillColor = paintParameter.iconStyleConfig.strokeColor;
        const canvas = this.newBpmnCanvas(paintParameter, { width: 458.75, height: 461.64 });
        canvas.begin();
        canvas.moveTo(67.85, 0.57);
        canvas.curveTo(50.73, 0, 33.26, 8.86, 22.35, 18.84);
        canvas.curveTo(8.11, 32.15, 0, 50.77, 0, 70.26);
        canvas.curveTo(0, 73.15, 0, 87.59, 0, 113.6);
        canvas.curveTo(55.4, 113.6, 86.18, 113.6, 92.33, 113.6);
        canvas.curveTo(94.92, 150.46, 85.64, 180.4, 74.22, 211.27);
        canvas.curveTo(40.16, 298.07, 30.77, 339.83, 55.56, 410.87);
        canvas.curveTo(63.72, 438.26, 87.59, 457.85, 114.91, 461.09);
        canvas.curveTo(216.96, 460.85, 294.9, 461.64, 388.41, 461.2);
        canvas.curveTo(407.2, 461.09, 425.14, 453.55, 438.3, 440.13);
        canvas.curveTo(451.46, 426.71, 458.75, 403.06, 458.46, 384.26);
        canvas.curveTo(458.43, 382.23, 458.18, 365.93, 458.15, 363.89);
        canvas.curveTo(432.12, 364.24, 406.09, 364.04, 380.06, 364.04);
        canvas.curveTo(377.61, 347.52, 377.24, 337.58, 378.28, 324.48);
        canvas.curveTo(380.5, 296.47, 389.08, 273.36, 398.59, 247.1);
        canvas.curveTo(408.11, 220.83, 418.41, 191.47, 420.86, 154.24);
        canvas.curveTo(422.11, 135.34, 421.4, 110.24, 417.77, 86.75);
        canvas.curveTo(417.76, 86.71, 417.73, 86.54, 417.69, 86.22);
        canvas.curveTo(417.64, 85.95, 417.61, 85.79, 417.6, 85.76);
        canvas.curveTo(414.03, 68.13, 410.49, 48.84, 399.79, 31.47);
        canvas.curveTo(389.09, 14.11, 366.95, 0.59, 341.75, 0.59);
        canvas.curveTo(286.97, 0.59, 122.63, 0.57, 67.85, 0.57);
        canvas.close();
        canvas.moveTo(85.04, 72.68);
        canvas.curveTo(80.63, 72.68, 45.33, 72.68, 40.92, 72.68);
        canvas.curveTo(40.46, 58.4, 47.15, 51.87, 50.27, 48.75);
        canvas.curveTo(55.8, 44.28, 59.84, 41, 73.82, 41);
        canvas.curveTo(78.45, 52.13, 82.23, 62.71, 85.04, 72.68);
        canvas.close();
        canvas.moveTo(364.94, 52.9);
        canvas.curveTo(370, 61.11, 373.9, 76.44, 377.38, 93.51);
        canvas.curveTo(380.35, 113.1, 381.01, 136.42, 380.02, 151.57);
        canvas.curveTo(377.97, 182.76, 369.51, 207.12, 360.1, 233.1);
        canvas.curveTo(350.69, 259.09, 340.27, 286.77, 337.53, 321.27);
        canvas.curveTo(336.38, 335.86, 336.72, 346.69, 338.87, 364.01);
        canvas.curveTo(326.35, 364.01, 263.75, 364.01, 151.06, 364.01);
        canvas.curveTo(151.06, 382.2, 151.06, 392.31, 151.06, 394.33);
        canvas.curveTo(147.77, 404.8, 138.9, 418.2, 127.43, 419.94);
        canvas.curveTo(111.49, 422.35, 97.86, 411.8, 94.75, 399.19);
        canvas.curveTo(65.14, 321.99, 94.93, 275.54, 112.57, 225.47);
        canvas.curveTo(130.14, 177.95, 137.92, 117.41, 112.71, 42.09);
        canvas.curveTo(192.88, 41.9, 274.33, 42.21, 342.89, 41.98);
        canvas.curveTo(357.15, 42.03, 359.83, 44.61, 364.94, 52.9);
        canvas.close();
        canvas.moveTo(409.96, 399.48);
        canvas.curveTo(409.96, 408.42, 398.54, 425.67, 392.02, 425.67);
        canvas.curveTo(325.19, 425.79, 252.29, 425.67, 185.23, 425.67);
        canvas.curveTo(189.88, 424.43, 194.66, 405.64, 194.66, 399.48);
        canvas.curveTo(237.72, 399.48, 388.43, 399.48, 409.96, 399.48);
        canvas.close();
        canvas.fill();
        canvas.begin();
        canvas.moveTo(182.1, 131.2);
        canvas.lineTo(182.1, 151.68);
        canvas.lineTo(321.89, 151.68);
        canvas.lineTo(321.89, 131.2);
        canvas.lineTo(182.1, 131.2);
        canvas.close();
        canvas.moveTo(162.25, 251.09);
        canvas.lineTo(162.25, 271.49);
        canvas.lineTo(301.96, 271.49);
        canvas.lineTo(301.96, 251.09);
        canvas.lineTo(162.25, 251.09);
        canvas.close();
        canvas.fill();
    }
    paintTableIcon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { width: 640, height: 640 });
        canvas.begin();
        canvas.moveTo(0.19, 0.1);
        canvas.lineTo(298.78, 0.1);
        canvas.lineTo(298.78, 198.88);
        canvas.lineTo(0.19, 198.88);
        canvas.lineTo(0.19, 0.1);
        canvas.close();
        canvas.fillAndStroke();
        canvas.begin();
        canvas.moveTo(0, 0);
        canvas.lineTo(298.78, 0);
        canvas.lineTo(298.78, 48.88);
        canvas.lineTo(0, 48.88);
        canvas.lineTo(0, 0);
        canvas.close();
        canvas.fillAndStroke();
        canvas.begin();
        canvas.moveTo(0, 48.88);
        canvas.lineTo(98.78, 48.88);
        canvas.lineTo(98.78, 198.88);
        canvas.lineTo(0, 198.88);
        canvas.lineTo(0, 48.88);
        canvas.close();
        canvas.fillAndStroke();
        canvas.begin();
        canvas.moveTo(1.09, 122.69);
        canvas.lineTo(298.78, 122.69);
        canvas.close();
        canvas.fillAndStroke();
        canvas.setFillColor(paintParameter.iconStyleConfig.strokeColor);
        canvas.begin();
        canvas.moveTo(0, 0);
        canvas.lineTo(298.78, 0);
        canvas.lineTo(298.78, 48.88);
        canvas.lineTo(0, 48.88);
        canvas.lineTo(0, 0);
        canvas.close();
        canvas.fillAndStroke();
    }
    paintPentagon(paintParameter) {
        const canvas = this.newBpmnCanvas(paintParameter, { width: 16, height: 16 });
        canvas.begin();
        canvas.moveTo(16, 6.5);
        canvas.lineTo(8, 0);
        canvas.lineTo(0, 6.5);
        canvas.lineTo(3, 16);
        canvas.lineTo(13, 16);
        canvas.lineTo(16, 6.5);
        canvas.lineTo(8, 0);
        canvas.stroke();
    }
}
function drawVerticalLine(canvas) {
    canvas.begin();
    canvas.moveTo(0.38, 0);
    canvas.lineTo(0.62, 0);
    canvas.lineTo(0.62, 1);
    canvas.lineTo(0.38, 1);
    canvas.close();
}
function paintGearIconBackground(canvas) {
    canvas.begin();
    canvas.moveTo(2.06, 24.62);
    canvas.lineTo(10.17, 30.95);
    canvas.lineTo(9.29, 37.73);
    canvas.lineTo(0, 41.42);
    canvas.lineTo(2.95, 54.24);
    canvas.lineTo(13.41, 52.92);
    canvas.lineTo(17.39, 58.52);
    canvas.lineTo(13.56, 67.66);
    canvas.lineTo(24.47, 74.44);
    canvas.lineTo(30.81, 66.33);
    canvas.lineTo(37.88, 67.21);
    canvas.lineTo(41.57, 76.5);
    canvas.lineTo(54.24, 73.55);
    canvas.lineTo(53.06, 62.94);
    canvas.lineTo(58.52, 58.52);
    canvas.lineTo(67.21, 63.09);
    canvas.lineTo(74.58, 51.88);
    canvas.lineTo(66.03, 45.25);
    canvas.lineTo(66.92, 38.62);
    canvas.lineTo(76.5, 34.93);
    canvas.lineTo(73.7, 22.26);
    canvas.lineTo(62.64, 23.44);
    canvas.lineTo(58.81, 18.42);
    canvas.lineTo(62.79, 8.7);
    canvas.lineTo(51.74, 2.21);
    canvas.lineTo(44.81, 10.47);
    canvas.lineTo(38.03, 9.43);
    canvas.lineTo(33.75, 0);
    canvas.lineTo(21.52, 3.24);
    canvas.lineTo(22.7, 13.56);
    canvas.lineTo(18.13, 17.54);
    canvas.lineTo(8.7, 13.56);
    canvas.close();
    const arcStartX = 24.8;
    const arcStartY = 39;
    paintGearInnerCircle(canvas, arcStartX, arcStartY);
}
function paintGearIconForeground(canvas) {
    canvas.begin();
    canvas.moveTo(16.46, 41.42);
    canvas.lineTo(24.57, 47.75);
    canvas.lineTo(23.69, 54.53);
    canvas.lineTo(14.4, 58.22);
    canvas.lineTo(17.35, 71.04);
    canvas.lineTo(27.81, 69.72);
    canvas.lineTo(31.79, 75.32);
    canvas.lineTo(27.96, 84.46);
    canvas.lineTo(38.87, 91.24);
    canvas.lineTo(45.21, 83.13);
    canvas.lineTo(52.28, 84.01);
    canvas.lineTo(55.97, 93.3);
    canvas.lineTo(68.64, 90.35);
    canvas.lineTo(67.46, 79.74);
    canvas.lineTo(72.92, 75.32);
    canvas.lineTo(81.61, 79.89);
    canvas.lineTo(88.98, 68.68);
    canvas.lineTo(80.43, 62.05);
    canvas.lineTo(81.32, 55.42);
    canvas.lineTo(90.9, 51.73);
    canvas.lineTo(88.1, 39.06);
    canvas.lineTo(77.04, 40.24);
    canvas.lineTo(73.21, 35.22);
    canvas.lineTo(77.19, 25.5);
    canvas.lineTo(66.14, 19.01);
    canvas.lineTo(59.21, 27.27);
    canvas.lineTo(52.43, 26.23);
    canvas.lineTo(48.15, 16.8);
    canvas.lineTo(35.92, 20.04);
    canvas.lineTo(37.1, 30.36);
    canvas.lineTo(32.53, 34.34);
    canvas.lineTo(23.1, 30.36);
    canvas.close();
    const arcStartX = 39.2;
    const arcStartY = 55.8;
    paintGearInnerCircle(canvas, arcStartX, arcStartY);
    canvas.begin();
    paintGearInnerCircle(canvas, arcStartX, arcStartY);
}
function paintGearInnerCircle(canvas, arcStartX, arcStartY) {
    const arcRay = 13.5;
    canvas.moveTo(arcStartX, arcStartY);
    canvas.arcTo(arcRay, arcRay, 0, 1, 1, arcStartX + 2 * arcRay, arcStartY);
    canvas.arcTo(arcRay, arcRay, 0, 0, 1, arcStartX, arcStartY);
    canvas.close();
    canvas.fillAndStroke();
}
class IconPainterProvider {
    static get() {
        return this.instance;
    }
    static set(painter) {
        this.instance = painter;
    }
}
Object.defineProperty(IconPainterProvider, "instance", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: new IconPainter()
});

const zoomFactorIn = 1.25;
const zoomFactorOut = 1 / zoomFactorIn;
class BpmnGraph extends mxgraph.mxGraph {
    constructor(container) {
        super(container);
        Object.defineProperty(this, "currentZoomLevel", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        this.zoomFactor = zoomFactorIn;
        if (this.container) {
            this.container.style.cursor = 'default';
        }
    }
    createGraphView() {
        return new BpmnGraphView(this);
    }
    createCellRenderer() {
        return new BpmnCellRenderer(IconPainterProvider.get());
    }
    batchUpdate(callbackFunction) {
        this.model.beginUpdate();
        try {
            callbackFunction();
        }
        finally {
            this.model.endUpdate();
        }
    }
    fit(border, keepOrigin, margin, enabled, ignoreWidth, ignoreHeight, maxHeight) {
        const scale = super.fit(border, keepOrigin, margin, enabled, ignoreWidth, ignoreHeight, maxHeight);
        this.setCurrentZoomLevel(scale);
        return scale;
    }
    setCurrentZoomLevel(scale) {
        this.currentZoomLevel = scale !== null && scale !== void 0 ? scale : this.view.scale;
    }
    zoomActual() {
        super.zoomActual();
        this.setCurrentZoomLevel();
    }
    zoomIn() {
        super.zoomIn();
        this.setCurrentZoomLevel();
    }
    zoomOut() {
        super.zoomOut();
        this.setCurrentZoomLevel();
    }
    customFit(fitOptions) {
        this.zoomActual();
        const type = fitOptions === null || fitOptions === void 0 ? void 0 : fitOptions.type;
        if (type == undefined || type == FitType.None) {
            return;
        }
        const margin = ensurePositiveValue(fitOptions === null || fitOptions === void 0 ? void 0 : fitOptions.margin);
        if (type == FitType.Center) {
            const maxScale = 3;
            const bounds = this.getGraphBounds();
            const clientWidth = this.container.clientWidth - margin;
            const clientHeight = this.container.clientHeight - margin;
            const width = bounds.width / this.view.scale;
            const height = bounds.height / this.view.scale;
            const scale = Math.min(maxScale, Math.min(clientWidth / width, clientHeight / height));
            this.setCurrentZoomLevel(scale);
            this.view.scaleAndTranslate(scale, (margin + clientWidth - width * scale) / (2 * scale) - bounds.x / this.view.scale, (margin + clientHeight - height * scale) / (2 * scale) - bounds.y / this.view.scale);
        }
        else {
            let ignoreWidth = false;
            let ignoreHeight = false;
            switch (type) {
                case FitType.Horizontal: {
                    ignoreHeight = true;
                    break;
                }
                case FitType.Vertical: {
                    ignoreWidth = true;
                    break;
                }
            }
            this.fit(this.border, false, margin, true, ignoreWidth, ignoreHeight);
        }
    }
    registerMouseWheelZoomListeners(config) {
        config = ensureValidZoomConfiguration(config);
        mxEvent.addMouseWheelListener(debounce(this.createMouseWheelZoomListener(true), config.debounceDelay), this.container);
        mxEvent.addMouseWheelListener(throttle(this.createMouseWheelZoomListener(false), config.throttleDelay), this.container);
    }
    manageMouseWheelZoomEvent(up, event, performScaling) {
        if (performScaling) {
            const [offsetX, offsetY] = this.getEventRelativeCoordinates(event);
            const [newScale, dx, dy] = this.getScaleAndTranslationDeltas(offsetX, offsetY);
            this.view.scaleAndTranslate(newScale, this.view.translate.x + dx, this.view.translate.y + dy);
            mxEvent.consume(event);
        }
        else {
            this.currentZoomLevel *= up ? zoomFactorIn : zoomFactorOut;
        }
    }
    createMouseWheelZoomListener(performScaling) {
        return (event, up) => {
            if (mxEvent.isConsumed(event) || !(event instanceof MouseEvent)) {
                return;
            }
            const isZoomWheelEvent = event.ctrlKey && !event.altKey && !event.shiftKey && !event.metaKey;
            if (isZoomWheelEvent) {
                this.manageMouseWheelZoomEvent(up, event, performScaling);
            }
        };
    }
    getEventRelativeCoordinates(event) {
        const rect = this.container.getBoundingClientRect();
        const x = event.clientX - rect.left;
        const y = event.clientY - rect.top;
        return [x, y];
    }
    getScaleAndTranslationDeltas(offsetX, offsetY) {
        const [factor, scale] = this.calculateFactorAndScale();
        const [dx, dy] = this.calculateTranslationDeltas(factor, scale, offsetX * 2, offsetY * 2);
        return [scale, dx, dy];
    }
    calculateTranslationDeltas(factor, scale, dx, dy) {
        if (factor > 1) {
            const f = (factor - 1) / (scale * 2);
            dx *= -f;
            dy *= -f;
        }
        else {
            const f = (1 / factor - 1) / (this.view.scale * 2);
            dx *= f;
            dy *= f;
        }
        return [dx, dy];
    }
    calculateFactorAndScale() {
        const scale = Math.round(this.currentZoomLevel * 100) / 100;
        const factor = scale / this.view.scale;
        return [factor, scale];
    }
}
class BpmnGraphView extends mxgraph.mxGraphView {
    getFloatingTerminalPoint(edge, start, end, source) {
        const edgePoints = edge.absolutePoints.filter(Boolean);
        const needsFloatingTerminalPoint = edgePoints.length < 2;
        if (needsFloatingTerminalPoint) {
            return super.getFloatingTerminalPoint(edge, start, end, source);
        }
        const pts = edge.absolutePoints;
        return source ? pts[1] : pts.at(-2);
    }
}

class MarkerConfigurator {
    configureMarkers() {
        this.registerArrowDashMarker();
    }
    registerArrowDashMarker() {
        const createMarker = (c, _shape, _type, pe, unitX, unitY, size, _source, strokewidth) => {
            const nx = unitX * (size + strokewidth + 4);
            const ny = unitY * (size + strokewidth + 4);
            return function () {
                c.begin();
                c.moveTo(pe.x - nx / 2 - ny / 2, pe.y - ny / 2 + nx / 2);
                c.lineTo(pe.x + ny / 2 - (3 * nx) / 2, pe.y - (3 * ny) / 2 - nx / 2);
                c.stroke();
            };
        };
        mxgraph.mxMarker.addMarker(MarkerIdentifier.ARROW_DASH, createMarker);
    }
}

const referenceOrderedMarkers = [
    ShapeBpmnMarkerKind.LOOP,
    ShapeBpmnMarkerKind.MULTI_INSTANCE_PARALLEL,
    ShapeBpmnMarkerKind.MULTI_INSTANCE_SEQUENTIAL,
    ShapeBpmnMarkerKind.COMPENSATION,
    ShapeBpmnMarkerKind.EXPAND,
    ShapeBpmnMarkerKind.ADHOC,
];
function orderActivityMarkers(markers) {
    const orderedMarkers = referenceOrderedMarkers.filter(marker => markers.includes(marker));
    for (const marker of markers.filter(marker => !orderedMarkers.includes(marker)))
        orderedMarkers.push(marker);
    return orderedMarkers;
}

function getMarkerIconOriginFunction(numberOfMarkers, markerPosition) {
    return numberOfMarkers === 1
        ? (canvas) => canvas.setIconOriginForIconBottomCentered()
        : (canvas) => {
            canvas.setIconOriginForIconBottomCentered();
            const xTranslation = (Math.pow(-1, markerPosition) * (StyleDefault.SHAPE_ACTIVITY_MARKER_ICON_SIZE + StyleDefault.SHAPE_ACTIVITY_MARKER_ICON_MARGIN)) / 2;
            canvas.translateIconOrigin(xTranslation, 0);
        };
}
class BaseActivityShape extends mxRectangleShape {
    constructor() {
        super(undefined, undefined, undefined);
        Object.defineProperty(this, "iconPainter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: undefined
        });
        Object.defineProperty(this, "markerPainterFunctions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map([
                [ShapeBpmnMarkerKind.EXPAND, (paintParameter) => this.iconPainter.paintExpandIcon(paintParameter)],
                [ShapeBpmnMarkerKind.LOOP, (paintParameter) => this.iconPainter.paintLoopIcon(paintParameter)],
                [ShapeBpmnMarkerKind.MULTI_INSTANCE_PARALLEL, (paintParameter) => this.iconPainter.paintParallelMultiInstanceIcon(paintParameter)],
                [ShapeBpmnMarkerKind.MULTI_INSTANCE_SEQUENTIAL, (paintParameter) => this.iconPainter.paintSequentialMultiInstanceIcon(paintParameter)],
            ])
        });
    }
    paintForeground(c, x, y, w, h) {
        super.paintForeground(c, x, y, w, h);
        this.paintMarkerIcons(buildPaintParameter({ canvas: c, x, y, width: w, height: h, shape: this, ratioFromParent: 0, iconStrokeWidth: 1.5 }));
    }
    paintMarkerIcons(paintParameter) {
        var _a;
        const markers = mxUtils.getValue(this.style, BpmnStyleIdentifier.MARKERS, undefined);
        if (markers) {
            const orderedMarkers = orderActivityMarkers(markers.split(','));
            for (const [index, marker] of orderedMarkers.entries()) {
                paintParameter = Object.assign(Object.assign({}, paintParameter), { setIconOriginFunct: getMarkerIconOriginFunction(orderedMarkers.length, index + 1) });
                paintParameter.canvas.save();
                (_a = this.markerPainterFunctions.get(marker)) === null || _a === void 0 ? void 0 : _a(paintParameter);
                paintParameter.canvas.restore();
            }
        }
    }
    paintEnvelopeIcon(paintParameter, isFilled) {
        this.iconPainter.paintEnvelopeIcon(Object.assign(Object.assign({}, paintParameter), { setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeft(), ratioFromParent: 0.2, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { isFilled: isFilled }) }));
    }
}
class BaseTaskShape extends BaseActivityShape {
    paintForeground(c, x, y, w, h) {
        super.paintForeground(c, x, y, w, h);
        this.paintTaskIcon(buildPaintParameter({ canvas: c, x, y, width: w, height: h, shape: this }));
    }
}
class TaskShape extends BaseTaskShape {
    paintTaskIcon(_paintParameter) {
    }
}
class ServiceTaskShape extends BaseTaskShape {
    paintTaskIcon(paintParameter) {
        this.iconPainter.paintGearIcon(Object.assign(Object.assign({}, paintParameter), { setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(20) }));
    }
}
class UserTaskShape extends BaseTaskShape {
    paintTaskIcon(paintParameter) {
        this.iconPainter.paintPersonIcon(Object.assign(Object.assign({}, paintParameter), { setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(20) }));
    }
}
class ReceiveTaskShape extends BaseTaskShape {
    paintTaskIcon(paintParameter) {
        if (!getBpmnIsInstantiating(this.style)) {
            this.paintEnvelopeIcon(paintParameter, false);
            return;
        }
        const leftMargin = 4;
        const topMargin = 4;
        const circleShapeConfig = Object.assign(Object.assign({}, paintParameter.shapeConfig), { width: 20, height: 20 });
        this.iconPainter.paintCircleIcon({
            canvas: paintParameter.canvas,
            shapeConfig: circleShapeConfig,
            iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { isFilled: false }),
            ratioFromParent: undefined,
            setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeft(topMargin, leftMargin),
        });
        circleShapeConfig.x += leftMargin;
        circleShapeConfig.y += topMargin;
        this.iconPainter.paintEnvelopeIcon(Object.assign(Object.assign({}, paintParameter), { shapeConfig: circleShapeConfig, ratioFromParent: 0.65, setIconOriginFunct: (canvas) => canvas.setIconOriginForIconCentered() }));
    }
}
class SendTaskShape extends BaseTaskShape {
    paintTaskIcon(paintParameter) {
        this.paintEnvelopeIcon(paintParameter, true);
    }
}
class ManualTaskShape extends BaseTaskShape {
    paintTaskIcon(paintParameter) {
        this.iconPainter.paintHandIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.18, setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(20) }));
    }
}
class ScriptTaskShape extends BaseTaskShape {
    paintTaskIcon(paintParameter) {
        this.iconPainter.paintScriptIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.22, setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(20) }));
    }
}
class CallActivityShape extends BaseActivityShape {
    paintForeground(c, x, y, w, h) {
        super.paintForeground(c, x, y, w, h);
        const paintParameter = buildPaintParameter({ canvas: c, x, y, width: w, height: h, shape: this });
        switch (mxUtils.getValue(this.style, BpmnStyleIdentifier.GLOBAL_TASK_KIND, undefined)) {
            case ShapeBpmnElementKind.GLOBAL_TASK_MANUAL: {
                this.iconPainter.paintHandIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.18, setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(20) }));
                break;
            }
            case ShapeBpmnElementKind.GLOBAL_TASK_SCRIPT: {
                this.iconPainter.paintScriptIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.22, setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(20) }));
                break;
            }
            case ShapeBpmnElementKind.GLOBAL_TASK_USER: {
                this.iconPainter.paintPersonIcon(Object.assign(Object.assign({}, paintParameter), { setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(20) }));
                break;
            }
            case ShapeBpmnElementKind.GLOBAL_TASK_BUSINESS_RULE: {
                this.iconPainter.paintTableIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.6, setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(15) }));
                break;
            }
        }
    }
}
class SubProcessShape extends BaseActivityShape {
    paintBackground(c, x, y, w, h) {
        const subProcessKind = mxUtils.getValue(this.style, BpmnStyleIdentifier.SUB_PROCESS_KIND, undefined);
        c.save();
        if (subProcessKind === ShapeBpmnSubProcessKind.EVENT) {
            c.setDashed(true, false);
            c.setDashPattern('1 2');
        }
        super.paintBackground(c, x, y, w, h);
        if (subProcessKind === ShapeBpmnSubProcessKind.TRANSACTION) {
            const innerOffset = StyleDefault.SUB_PROCESS_TRANSACTION_INNER_RECT_OFFSET;
            const innerArcSize = StyleDefault.SUB_PROCESS_TRANSACTION_INNER_RECT_ARC_SIZE;
            c.roundrect(x + innerOffset, y + innerOffset, w - 2 * innerOffset, h - 2 * innerOffset, innerArcSize, innerArcSize);
            c.stroke();
        }
        c.restore();
    }
}
class BusinessRuleTaskShape extends BaseTaskShape {
    paintTaskIcon(paintParameter) {
        this.iconPainter.paintTableIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.6, setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(15) }));
    }
}

class BpmnConnector extends mxgraph.mxConnector {
    paintEdgeShape(c, pts) {
        const sourceMarker = this.createMarker(c, pts, true);
        const targetMarker = this.createMarker(c, pts, false);
        this.paintEdgeLine(c, pts);
        c.setShadow(false);
        c.setDashed(false, false);
        if (sourceMarker != null) {
            c.setFillColor(mxUtils.getValue(this.style, BpmnStyleIdentifier.EDGE_START_FILL_COLOR, this.stroke));
            sourceMarker();
        }
        if (targetMarker != null) {
            c.setFillColor(mxUtils.getValue(this.style, BpmnStyleIdentifier.EDGE_END_FILL_COLOR, this.stroke));
            targetMarker();
        }
    }
    paintEdgeLine(c, pts) {
        const previous = getPointerEventsValue(c);
        setPointerEventsValue(c, 'stroke');
        this.paintLine(c, pts, this.isRounded);
        setPointerEventsValue(c, previous);
    }
}
function getPointerEventsValue(c) {
    return c instanceof mxSvgCanvas2D ? c.pointerEventsValue : null;
}
function setPointerEventsValue(c, value) {
    if (c instanceof mxSvgCanvas2D) {
        c.pointerEventsValue = value;
    }
}

class EventShape extends mxgraph.mxEllipse {
    constructor() {
        super(undefined, undefined, undefined);
        Object.defineProperty(this, "iconPainter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: undefined
        });
        Object.defineProperty(this, "iconPainters", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map([
                [ShapeBpmnEventDefinitionKind.MESSAGE, (paintParameter) => this.iconPainter.paintEnvelopeIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.5 }))],
                [ShapeBpmnEventDefinitionKind.TERMINATE, (paintParameter) => this.iconPainter.paintCircleIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.6 }))],
                [
                    ShapeBpmnEventDefinitionKind.TIMER,
                    (paintParameter) => this.iconPainter.paintClockIcon(Object.assign(Object.assign({}, paintParameter), { setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(5) })),
                ],
                [
                    ShapeBpmnEventDefinitionKind.SIGNAL,
                    (paintParameter) => this.iconPainter.paintTriangleIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.55, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { strokeWidth: StyleDefault.STROKE_WIDTH_THIN.valueOf() }), setIconOriginFunct: (canvas) => canvas.setIconOriginToShapeTopLeftProportionally(4) })),
                ],
                [
                    ShapeBpmnEventDefinitionKind.LINK,
                    (paintParameter) => this.iconPainter.paintRightArrowIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.55, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { strokeWidth: 1.5 }) })),
                ],
                [
                    ShapeBpmnEventDefinitionKind.ERROR,
                    (paintParameter) => this.iconPainter.paintErrorIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.55, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { strokeWidth: 1.5 }) })),
                ],
                [
                    ShapeBpmnEventDefinitionKind.COMPENSATION,
                    (paintParameter) => this.iconPainter.paintDoubleLeftArrowheadsIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.7, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { strokeWidth: 1.5 }) })),
                ],
                [ShapeBpmnEventDefinitionKind.CANCEL, (paintParameter) => this.iconPainter.paintXCrossIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.78 }))],
                [
                    ShapeBpmnEventDefinitionKind.ESCALATION,
                    (paintParameter) => this.iconPainter.paintUpArrowheadIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.55, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { strokeWidth: StyleDefault.STROKE_WIDTH_THIN.valueOf() }) })),
                ],
                [
                    ShapeBpmnEventDefinitionKind.CONDITIONAL,
                    (paintParameter) => this.iconPainter.paintListIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.6, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { strokeWidth: 1.5 }) })),
                ],
            ])
        });
        Object.defineProperty(this, "withFilledIcon", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
    }
    paintVertexShape(c, x, y, w, h) {
        const paintParameter = buildPaintParameter({ canvas: c, x, y, width: w, height: h, shape: this, isFilled: this.withFilledIcon });
        setDashedOuterShapePattern(paintParameter, mxUtils.getValue(this.style, BpmnStyleIdentifier.IS_INTERRUPTING, undefined));
        this.paintOuterShape(paintParameter);
        restoreOriginalOuterShapePattern(paintParameter);
        this.paintInnerShape(paintParameter);
    }
    paintOuterShape({ canvas, shapeConfig: { x, y, width, height } }) {
        super.paintVertexShape(canvas, x, y, width, height);
    }
    paintInnerShape(paintParameter) {
        const paintIcon = this.iconPainters.get(mxUtils.getValue(this.style, BpmnStyleIdentifier.EVENT_DEFINITION_KIND, ShapeBpmnEventDefinitionKind.NONE));
        paintIcon === null || paintIcon === void 0 ? void 0 : paintIcon(paintParameter);
    }
}
function setDashedOuterShapePattern(paintParameter, isInterrupting) {
    paintParameter.canvas.save();
    if (isInterrupting === 'false') {
        paintParameter.canvas.setDashed(true, false);
        paintParameter.canvas.setDashPattern('3 2');
    }
}
function restoreOriginalOuterShapePattern(paintParameter) {
    paintParameter.canvas.restore();
}
class EndEventShape extends EventShape {
    constructor() {
        super();
        this.withFilledIcon = true;
    }
}
class IntermediateEventShape extends EventShape {
    paintOuterShape({ canvas, shapeConfig: { x, y, width, height, strokeWidth } }) {
        canvas.ellipse(x, y, width, height);
        canvas.fillAndStroke();
        const inset = strokeWidth * 1.5;
        canvas.ellipse(width * 0.02 + inset + x, height * 0.02 + inset + y, width * 0.96 - 2 * inset, height * 0.96 - 2 * inset);
        canvas.stroke();
    }
}
class ThrowIntermediateEventShape extends IntermediateEventShape {
    constructor() {
        super();
        this.withFilledIcon = true;
    }
}

class MessageFlowIconShape extends mxRectangleShape {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "iconPainter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: undefined
        });
    }
    paintVertexShape(c, x, y, w, h) {
        const paintParameter = buildPaintParameter({
            canvas: c,
            x,
            y,
            width: w,
            height: h,
            shape: this,
            ratioFromParent: 1,
            isFilled: mxUtils.getValue(this.style, BpmnStyleIdentifier.IS_INITIATING, 'true') == 'false',
        });
        this.iconPainter.paintEnvelopeIcon(paintParameter);
    }
}

class GatewayShape extends mxgraph.mxRhombus {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "iconPainter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: undefined
        });
    }
    paintVertexShape(c, x, y, w, h) {
        const paintParameter = buildPaintParameter({ canvas: c, x, y, width: w, height: h, shape: this });
        this.paintOuterShape(paintParameter);
        this.paintInnerShape(paintParameter);
    }
    paintOuterShape({ canvas, shapeConfig: { x, y, width, height } }) {
        super.paintVertexShape(canvas, x, y, width, height);
    }
}
class ExclusiveGatewayShape extends GatewayShape {
    paintInnerShape(paintParameter) {
        this.iconPainter.paintXCrossIcon(Object.assign(Object.assign({}, paintParameter), { iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { isFilled: true }), ratioFromParent: 0.5 }));
    }
}
class ParallelGatewayShape extends GatewayShape {
    paintInnerShape(paintParameter) {
        this.iconPainter.paintPlusCrossIcon(Object.assign(Object.assign({}, paintParameter), { iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { isFilled: true }), ratioFromParent: 0.5 }));
    }
}
class InclusiveGatewayShape extends GatewayShape {
    paintInnerShape(paintParameter) {
        this.iconPainter.paintCircleIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.62, iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { isFilled: false, strokeWidth: StyleDefault.STROKE_WIDTH_THICK.valueOf() }) }));
    }
}
class ComplexGatewayShape extends GatewayShape {
    paintInnerShape(paintParameter) {
        this.iconPainter.paintAsteriskIcon(Object.assign(Object.assign({}, paintParameter), { iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { isFilled: true }), ratioFromParent: 0.5 }));
    }
}
class EventBasedGatewayShape extends GatewayShape {
    paintInnerShape(paintParameter) {
        paintParameter = Object.assign(Object.assign({}, paintParameter), { iconStyleConfig: Object.assign(Object.assign({}, paintParameter.iconStyleConfig), { strokeWidth: 1 }) });
        this.iconPainter.paintCircleIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.55 }));
        if (!getBpmnIsInstantiating(this.style)) {
            this.iconPainter.paintCircleIcon(Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.45 }));
        }
        const innerIconPaintParameter = Object.assign(Object.assign({}, paintParameter), { ratioFromParent: 0.3 });
        if (mxUtils.getValue(this.style, BpmnStyleIdentifier.EVENT_BASED_GATEWAY_KIND, ShapeBpmnEventBasedGatewayKind.Exclusive) == ShapeBpmnEventBasedGatewayKind.Parallel) {
            this.iconPainter.paintPlusCrossIcon(innerIconPaintParameter);
        }
        else {
            this.iconPainter.paintPentagon(innerIconPaintParameter);
        }
    }
}

class TextAnnotationShape extends mxRectangleShape {
    paintForeground(c, x, y, _w, h) {
        c.begin();
        c.moveTo(x + StyleDefault.TEXT_ANNOTATION_BORDER_LENGTH, y);
        c.lineTo(x, y);
        c.lineTo(x, y + h);
        c.lineTo(x + StyleDefault.TEXT_ANNOTATION_BORDER_LENGTH, y + h);
        c.stroke();
    }
    paintBackground(c, x, y, w, h) {
        c.save();
        c.setStrokeColor('none');
        super.paintBackground(c, x, y, w, h);
        c.restore();
    }
}

const registerShapes = () => {
    const shapesToRegister = [
        [ShapeBpmnElementKind.EVENT_END, EndEventShape],
        [ShapeBpmnElementKind.EVENT_START, EventShape],
        [ShapeBpmnElementKind.EVENT_INTERMEDIATE_THROW, ThrowIntermediateEventShape],
        [ShapeBpmnElementKind.EVENT_INTERMEDIATE_CATCH, IntermediateEventShape],
        [ShapeBpmnElementKind.EVENT_BOUNDARY, IntermediateEventShape],
        [ShapeBpmnElementKind.GATEWAY_COMPLEX, ComplexGatewayShape],
        [ShapeBpmnElementKind.GATEWAY_EVENT_BASED, EventBasedGatewayShape],
        [ShapeBpmnElementKind.GATEWAY_EXCLUSIVE, ExclusiveGatewayShape],
        [ShapeBpmnElementKind.GATEWAY_INCLUSIVE, InclusiveGatewayShape],
        [ShapeBpmnElementKind.GATEWAY_PARALLEL, ParallelGatewayShape],
        [ShapeBpmnElementKind.SUB_PROCESS, SubProcessShape],
        [ShapeBpmnElementKind.CALL_ACTIVITY, CallActivityShape],
        [ShapeBpmnElementKind.TASK, TaskShape],
        [ShapeBpmnElementKind.TASK_SERVICE, ServiceTaskShape],
        [ShapeBpmnElementKind.TASK_USER, UserTaskShape],
        [ShapeBpmnElementKind.TASK_RECEIVE, ReceiveTaskShape],
        [ShapeBpmnElementKind.TASK_SEND, SendTaskShape],
        [ShapeBpmnElementKind.TASK_MANUAL, ManualTaskShape],
        [ShapeBpmnElementKind.TASK_SCRIPT, ScriptTaskShape],
        [ShapeBpmnElementKind.TASK_BUSINESS_RULE, BusinessRuleTaskShape],
        [ShapeBpmnElementKind.TEXT_ANNOTATION, TextAnnotationShape],
        [BpmnStyleIdentifier.EDGE, BpmnConnector],
        [BpmnStyleIdentifier.MESSAGE_FLOW_ICON, MessageFlowIconShape],
    ];
    for (const [shapeName, shapeClass] of shapesToRegister) {
        mxCellRenderer.registerShape(shapeName, shapeClass);
    }
};
class ShapeConfigurator {
    configureShapes() {
        registerShapes();
    }
}

const arrowDefaultSize = 12;
class MapWithDefault extends Map {
    get(key) {
        var _a;
        return ((_a = super.get(key)) !== null && _a !== void 0 ? _a : (() => {
        }));
    }
}
const specificFlowStyles = new MapWithDefault([
    [
        FlowKind.SEQUENCE_FLOW,
        (style) => {
            style[mxConstants.STYLE_ENDARROW] = mxConstants.ARROW_BLOCK_THIN;
        },
    ],
    [
        FlowKind.MESSAGE_FLOW,
        (style) => {
            style[mxConstants.STYLE_DASHED] = true;
            style[mxConstants.STYLE_DASH_PATTERN] = '8 5';
            style[mxConstants.STYLE_STARTARROW] = mxConstants.ARROW_OVAL;
            style[mxConstants.STYLE_STARTSIZE] = 8;
            style[mxConstants.STYLE_STARTFILL] = true;
            style[BpmnStyleIdentifier.EDGE_START_FILL_COLOR] = StyleDefault.MESSAGE_FLOW_MARKER_START_FILL_COLOR;
            style[mxConstants.STYLE_ENDARROW] = mxConstants.ARROW_BLOCK_THIN;
            style[mxConstants.STYLE_ENDFILL] = true;
            style[BpmnStyleIdentifier.EDGE_END_FILL_COLOR] = StyleDefault.MESSAGE_FLOW_MARKER_END_FILL_COLOR;
        },
    ],
    [
        FlowKind.ASSOCIATION_FLOW,
        (style) => {
            style[mxConstants.STYLE_DASHED] = true;
            style[mxConstants.STYLE_DASH_PATTERN] = '1 2';
            style[mxConstants.STYLE_STARTSIZE] = arrowDefaultSize;
        },
    ],
]);
const specificSequenceFlowStyles = new MapWithDefault([
    [
        SequenceFlowKind.DEFAULT,
        (style) => {
            style[mxConstants.STYLE_STARTARROW] = MarkerIdentifier.ARROW_DASH;
        },
    ],
    [
        SequenceFlowKind.CONDITIONAL_FROM_ACTIVITY,
        (style) => {
            style[mxConstants.STYLE_STARTARROW] = mxConstants.ARROW_DIAMOND_THIN;
            style[mxConstants.STYLE_STARTSIZE] = 18;
            style[mxConstants.STYLE_STARTFILL] = true;
            style[BpmnStyleIdentifier.EDGE_START_FILL_COLOR] = StyleDefault.SEQUENCE_FLOW_CONDITIONAL_FROM_ACTIVITY_MARKER_FILL_COLOR;
        },
    ],
]);
const specificAssociationFlowStyles = new MapWithDefault([
    [
        AssociationDirectionKind.NONE,
        (_style) => {
        },
    ],
    [
        AssociationDirectionKind.ONE,
        (style) => {
            style[mxConstants.STYLE_ENDARROW] = mxConstants.ARROW_OPEN_THIN;
        },
    ],
    [
        AssociationDirectionKind.BOTH,
        (style) => {
            style[mxConstants.STYLE_STARTARROW] = mxConstants.ARROW_OPEN_THIN;
            style[mxConstants.STYLE_ENDARROW] = mxConstants.ARROW_OPEN_THIN;
        },
    ],
]);
class StyleConfigurator {
    constructor(graph) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: graph
        });
    }
    configureStyles() {
        this.configureDefaultVertexStyle();
        this.configurePoolStyle();
        this.configureLaneStyle();
        this.configureTextAnnotationStyle();
        this.configureGroupStyle();
        this.configureActivityStyles();
        this.configureEventStyles();
        this.configureGatewayStyles();
        this.configureDefaultEdgeStyle();
        this.configureFlowStyles();
    }
    getStylesheet() {
        return this.graph.getStylesheet();
    }
    putCellStyle(name, style) {
        this.getStylesheet().putCellStyle(name, style);
    }
    configureDefaultVertexStyle() {
        const style = this.getStylesheet().getDefaultVertexStyle();
        configureCommonDefaultStyle(style);
        style[mxConstants.STYLE_ABSOLUTE_ARCSIZE] = true;
        style[mxConstants.STYLE_ARCSIZE] = StyleDefault.SHAPE_ARC_SIZE;
    }
    configurePoolStyle() {
        const style = {};
        style[mxConstants.STYLE_SHAPE] = mxConstants.SHAPE_SWIMLANE;
        style[mxConstants.STYLE_VERTICAL_ALIGN] = mxConstants.ALIGN_MIDDLE;
        style[mxConstants.STYLE_ALIGN] = mxConstants.ALIGN_CENTER;
        style[mxConstants.STYLE_STARTSIZE] = StyleDefault.POOL_LABEL_SIZE;
        style[mxConstants.STYLE_FILLCOLOR] = StyleDefault.POOL_LABEL_FILL_COLOR;
        this.graph.getStylesheet().putCellStyle(ShapeBpmnElementKind.POOL, style);
    }
    configureLaneStyle() {
        const style = {};
        style[mxConstants.STYLE_SHAPE] = mxConstants.SHAPE_SWIMLANE;
        style[mxConstants.STYLE_VERTICAL_ALIGN] = mxConstants.ALIGN_MIDDLE;
        style[mxConstants.STYLE_ALIGN] = mxConstants.ALIGN_CENTER;
        style[mxConstants.STYLE_SWIMLANE_LINE] = 0;
        style[mxConstants.STYLE_STARTSIZE] = StyleDefault.LANE_LABEL_SIZE;
        style[mxConstants.STYLE_FILLCOLOR] = StyleDefault.LANE_LABEL_FILL_COLOR;
        this.graph.getStylesheet().putCellStyle(ShapeBpmnElementKind.LANE, style);
    }
    configureEventStyles() {
        for (const kind of ShapeUtil.eventKinds()) {
            const style = {};
            style[mxConstants.STYLE_SHAPE] = kind;
            style[mxConstants.STYLE_PERIMETER] = mxPerimeter.EllipsePerimeter;
            style[mxConstants.STYLE_STROKEWIDTH] = kind == ShapeBpmnElementKind.EVENT_END ? StyleDefault.STROKE_WIDTH_THICK : StyleDefault.STROKE_WIDTH_THIN;
            style[mxConstants.STYLE_VERTICAL_LABEL_POSITION] = mxConstants.ALIGN_BOTTOM;
            this.putCellStyle(kind, style);
        }
    }
    configureTextAnnotationStyle() {
        const style = {};
        style[mxConstants.STYLE_SHAPE] = ShapeBpmnElementKind.TEXT_ANNOTATION;
        style[mxConstants.STYLE_VERTICAL_ALIGN] = mxConstants.ALIGN_MIDDLE;
        style[mxConstants.STYLE_ALIGN] = mxConstants.ALIGN_LEFT;
        style[mxConstants.STYLE_SPACING_LEFT] = 5;
        style[mxConstants.STYLE_FILLCOLOR] = StyleDefault.TEXT_ANNOTATION_FILL_COLOR;
        style[mxConstants.STYLE_STROKEWIDTH] = StyleDefault.STROKE_WIDTH_THIN;
        this.putCellStyle(ShapeBpmnElementKind.TEXT_ANNOTATION, style);
    }
    configureGroupStyle() {
        const style = {};
        style[mxConstants.STYLE_ROUNDED] = true;
        style[mxConstants.STYLE_DASHED] = true;
        style[mxConstants.STYLE_DASH_PATTERN] = '7 4 1 4';
        style[mxConstants.STYLE_STROKEWIDTH] = StyleDefault.STROKE_WIDTH_THIN;
        style[mxConstants.STYLE_FILLCOLOR] = StyleDefault.GROUP_FILL_COLOR;
        style[mxConstants.STYLE_ALIGN] = mxConstants.ALIGN_CENTER;
        style[mxConstants.STYLE_VERTICAL_ALIGN] = mxConstants.ALIGN_TOP;
        this.putCellStyle(ShapeBpmnElementKind.GROUP, style);
    }
    configureActivityStyles() {
        for (const kind of ShapeUtil.activityKinds()) {
            const style = {};
            style[mxConstants.STYLE_SHAPE] = kind;
            style[mxConstants.STYLE_ROUNDED] = true;
            style[mxConstants.STYLE_VERTICAL_ALIGN] = mxConstants.ALIGN_MIDDLE;
            style[mxConstants.STYLE_STROKEWIDTH] = kind == ShapeBpmnElementKind.CALL_ACTIVITY ? StyleDefault.STROKE_WIDTH_THICK : StyleDefault.STROKE_WIDTH_THIN;
            this.putCellStyle(kind, style);
        }
    }
    configureGatewayStyles() {
        for (const kind of ShapeUtil.gatewayKinds()) {
            const style = {};
            style[mxConstants.STYLE_SHAPE] = kind;
            style[mxConstants.STYLE_PERIMETER] = mxPerimeter.RhombusPerimeter;
            style[mxConstants.STYLE_STROKEWIDTH] = StyleDefault.STROKE_WIDTH_THIN;
            style[mxConstants.STYLE_VERTICAL_ALIGN] = mxConstants.ALIGN_TOP;
            style[mxConstants.STYLE_LABEL_POSITION] = mxConstants.ALIGN_LEFT;
            style[mxConstants.STYLE_VERTICAL_LABEL_POSITION] = mxConstants.ALIGN_TOP;
            this.putCellStyle(kind, style);
        }
    }
    configureDefaultEdgeStyle() {
        const style = this.getStylesheet().getDefaultEdgeStyle();
        configureCommonDefaultStyle(style);
        style[mxConstants.STYLE_SHAPE] = BpmnStyleIdentifier.EDGE;
        style[mxConstants.STYLE_ENDSIZE] = arrowDefaultSize;
        style[mxConstants.STYLE_STROKEWIDTH] = 1.5;
        style[mxConstants.STYLE_ROUNDED] = true;
        style[mxConstants.STYLE_ARCSIZE] = 5;
        style[mxConstants.STYLE_VERTICAL_ALIGN] = mxConstants.ALIGN_BOTTOM;
        delete style[mxConstants.STYLE_ENDARROW];
    }
    configureEdgeStyles(styleKinds, specificStyles) {
        for (const kind of styleKinds) {
            const style = {};
            specificStyles.get(kind)(style);
            this.graph.getStylesheet().putCellStyle(kind.toString(), style);
        }
    }
    configureFlowStyles() {
        this.configureEdgeStyles(Object.values(FlowKind), specificFlowStyles);
        this.configureEdgeStyles(Object.values(SequenceFlowKind), specificSequenceFlowStyles);
        this.configureEdgeStyles(Object.values(AssociationDirectionKind), specificAssociationFlowStyles);
    }
}
function configureCommonDefaultStyle(style) {
    style[mxConstants.STYLE_FONTFAMILY] = StyleDefault.DEFAULT_FONT_FAMILY;
    style[mxConstants.STYLE_FONTSIZE] = StyleDefault.DEFAULT_FONT_SIZE;
    style[mxConstants.STYLE_FONTCOLOR] = StyleDefault.DEFAULT_FONT_COLOR;
    style[mxConstants.STYLE_FILLCOLOR] = StyleDefault.DEFAULT_FILL_COLOR;
    style[mxConstants.STYLE_STROKECOLOR] = StyleDefault.DEFAULT_STROKE_COLOR;
    style[mxConstants.STYLE_LABEL_BACKGROUNDCOLOR] = mxConstants.NONE;
    style[mxConstants.STYLE_WHITE_SPACE] = 'wrap';
}

class GraphConfigurator {
    constructor(container) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.graph = new BpmnGraph(container);
    }
    configure(options) {
        this.configureGraph();
        this.configureNavigationSupport(options);
        new StyleConfigurator(this.graph).configureStyles();
        new ShapeConfigurator().configureShapes();
        new MarkerConfigurator().configureMarkers();
        return this.graph;
    }
    configureGraph() {
        this.graph.setCellsLocked(true);
        this.graph.setCellsSelectable(false);
        this.graph.setEdgeLabelsMovable(false);
        this.graph.setHtmlLabels(true);
        this.graph.setConstrainChildren(false);
        this.graph.setExtendParents(false);
        this.graph.foldingEnabled = false;
    }
    configureNavigationSupport(options) {
        var _a;
        const panningHandler = this.graph.panningHandler;
        if ((_a = options === null || options === void 0 ? void 0 : options.navigation) === null || _a === void 0 ? void 0 : _a.enabled) {
            panningHandler.addListener(mxEvent.PAN_START, setContainerCursor(this.graph, 'grab'));
            panningHandler.addListener(mxEvent.PAN_END, setContainerCursor(this.graph, 'default'));
            panningHandler.usePopupTrigger = false;
            panningHandler.isForcePanningEvent = (me) => mxEvent.isLeftMouseButton(me.getEvent()) || mxEvent.isMultiTouchEvent(me.getEvent());
            this.graph.setPanning(true);
            this.graph.registerMouseWheelZoomListeners(options.navigation.zoom);
        }
        else {
            this.graph.setPanning(false);
            panningHandler.setPinchEnabled(false);
            panningHandler.isForcePanningEvent = (_me) => false;
        }
    }
}
function setContainerCursor(graph, cursor) {
    return () => {
        graph.container.style.cursor = cursor;
    };
}

class Navigation {
    constructor(graph) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: graph
        });
    }
    fit(options) {
        this.graph.customFit(options);
    }
    zoom(type) {
        type == 'in' ? this.graph.zoomIn() : this.graph.zoomOut();
    }
}

class CategoryConverter {
    constructor(convertedElements) {
        Object.defineProperty(this, "convertedElements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: convertedElements
        });
    }
    deserialize(definitions) {
        const categoryValues = ensureIsArray(definitions.category).flatMap(category => ensureIsArray(category.categoryValue));
        for (const categoryValue of categoryValues) {
            this.convertedElements.registerCategoryValue(categoryValue.id, categoryValue.value);
        }
    }
}

class JsonParsingWarning {
}
class ParsingMessageCollector {
    constructor(options) {
        Object.defineProperty(this, "options", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: options
        });
    }
    warning(warning) {
        var _a;
        if ((_a = this.options) === null || _a === void 0 ? void 0 : _a.disableConsoleLog) {
            return;
        }
        const message = warning.getMessage();
        console.warn(`[bv-parser] ${message.template}`, ...message.arguments);
    }
}

class GroupUnknownCategoryValueWarning extends JsonParsingWarning {
    constructor(groupBpmnElementId, categoryValueReference) {
        super();
        Object.defineProperty(this, "groupBpmnElementId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: groupBpmnElementId
        });
        Object.defineProperty(this, "categoryValueReference", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: categoryValueReference
        });
    }
    getMessage() {
        return {
            arguments: [this.categoryValueReference, this.groupBpmnElementId],
            template: 'Group json deserialization: unable to find category value ref %s for bpmn element %s',
        };
    }
}
class ShapeUnknownBpmnElementWarning extends JsonParsingWarning {
    constructor(bpmnElementId) {
        super();
        Object.defineProperty(this, "bpmnElementId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bpmnElementId
        });
    }
    getMessage() {
        return {
            arguments: [this.bpmnElementId],
            template: 'Shape json deserialization: unable to find bpmn element with id %s',
        };
    }
}
class EdgeUnknownBpmnElementWarning extends JsonParsingWarning {
    constructor(bpmnElementId) {
        super();
        Object.defineProperty(this, "bpmnElementId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bpmnElementId
        });
    }
    getMessage() {
        return {
            arguments: [this.bpmnElementId],
            template: 'Edge json deserialization: unable to find bpmn element with id %s',
        };
    }
}
class LabelStyleMissingFontWarning extends JsonParsingWarning {
    constructor(shapeOrEdgeId, labelStyleId) {
        super();
        Object.defineProperty(this, "shapeOrEdgeId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: shapeOrEdgeId
        });
        Object.defineProperty(this, "labelStyleId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: labelStyleId
        });
    }
    getMessage() {
        return {
            arguments: [this.labelStyleId, this.shapeOrEdgeId],
            template: 'Unable to assign font from style %s to shape/edge %s',
        };
    }
}
class LaneUnknownFlowNodeReferenceWarning extends JsonParsingWarning {
    constructor(laneId, flowNodeReference) {
        super();
        Object.defineProperty(this, "laneId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: laneId
        });
        Object.defineProperty(this, "flowNodeReference", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: flowNodeReference
        });
    }
    getMessage() {
        return {
            arguments: [this.flowNodeReference, this.laneId],
            template: 'Unable to assign lane %s as parent: flow node %s is not found',
        };
    }
}
class BoundaryEventNotAttachedToActivityWarning extends JsonParsingWarning {
    constructor(bpmnElementId, attachedToReference, attachedToKind) {
        super();
        Object.defineProperty(this, "bpmnElementId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bpmnElementId
        });
        Object.defineProperty(this, "attachedToReference", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: attachedToReference
        });
        Object.defineProperty(this, "attachedToKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: attachedToKind
        });
    }
    getMessage() {
        return {
            arguments: [this.bpmnElementId, this.attachedToReference, this.attachedToKind],
            template: 'The boundary event %s must be attached to an activity, and not to %s of kind %s',
        };
    }
}

class ConvertedElements {
    constructor() {
        Object.defineProperty(this, "poolsById", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "poolsByProcessRef", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "messageFlows", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "flowNodes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "lanes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "sequenceFlows", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "associationFlows", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "eventDefinitionsOfDefinitions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "globalTasks", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "categoryValues", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
    }
    getFlows() {
        return [...this.messageFlows.values(), ...this.sequenceFlows.values(), ...this.associationFlows.values()];
    }
    findPoolById(id) {
        return this.poolsById.get(id);
    }
    findPoolByProcessRef(processReference) {
        return this.poolsByProcessRef.get(processReference);
    }
    registerPool(pool, processReference) {
        this.poolsById.set(pool.id, pool);
        if (processReference) {
            this.poolsByProcessRef.set(processReference, pool);
        }
    }
    findMessageFlow(id) {
        return this.messageFlows.get(id);
    }
    registerMessageFlow(messageFlow) {
        this.messageFlows.set(messageFlow.id, messageFlow);
    }
    findFlowNode(id) {
        return this.flowNodes.get(id);
    }
    registerFlowNode(flowNode) {
        this.flowNodes.set(flowNode.id, flowNode);
    }
    findLane(id) {
        return this.lanes.get(id);
    }
    registerLane(lane) {
        this.lanes.set(lane.id, lane);
    }
    findSequenceFlow(id) {
        return this.sequenceFlows.get(id);
    }
    registerSequenceFlow(sequenceFlow) {
        this.sequenceFlows.set(sequenceFlow.id, sequenceFlow);
    }
    findAssociationFlow(id) {
        return this.associationFlows.get(id);
    }
    registerAssociationFlow(associationFlow) {
        this.associationFlows.set(associationFlow.id, associationFlow);
    }
    findEventDefinitionOfDefinition(id) {
        return this.eventDefinitionsOfDefinitions.get(id);
    }
    registerEventDefinitionsOfDefinition(id, eventDefinition) {
        this.eventDefinitionsOfDefinitions.set(id, eventDefinition);
    }
    findGlobalTask(id) {
        return this.globalTasks.get(id);
    }
    registerGlobalTask(id, kind) {
        this.globalTasks.set(id, kind);
    }
    findCategoryValue(categoryValue) {
        return this.categoryValues.get(categoryValue);
    }
    registerCategoryValue(id, value) {
        this.categoryValues.set(id, { value });
    }
}
const buildShapeBpmnGroup = (convertedElements, parsingMessageCollector, groupBpmnElement, processId) => {
    const categoryValueData = convertedElements.findCategoryValue(groupBpmnElement.categoryValueRef);
    if (categoryValueData) {
        return new ShapeBpmnElement(groupBpmnElement.id, categoryValueData.value, ShapeBpmnElementKind.GROUP, processId);
    }
    parsingMessageCollector.warning(new GroupUnknownCategoryValueWarning(groupBpmnElement.id, groupBpmnElement.categoryValueRef));
    return undefined;
};
const convertAndRegisterAssociationFlows = (convertedElements, bpmnElements) => {
    for (const association of ensureIsArray(bpmnElements)) {
        const direction = association.associationDirection;
        convertedElements.registerAssociationFlow(new AssociationFlow(association.id, undefined, association.sourceRef, association.targetRef, direction));
    }
};

class CollaborationConverter {
    constructor(convertedElements, parsingMessageCollector) {
        Object.defineProperty(this, "convertedElements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: convertedElements
        });
        Object.defineProperty(this, "parsingMessageCollector", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: parsingMessageCollector
        });
    }
    deserialize(collaborations) {
        for (const collaboration of ensureIsArray(collaborations))
            this.parseCollaboration(collaboration);
    }
    parseCollaboration(collaboration) {
        this.buildParticipant(collaboration.participant);
        this.buildMessageFlows(collaboration.messageFlow);
        convertAndRegisterAssociationFlows(this.convertedElements, collaboration.association);
        this.buildGroups(collaboration.group);
        this.buildTextAnnotation(collaboration.textAnnotation);
    }
    buildParticipant(bpmnElements) {
        for (const participant of ensureIsArray(bpmnElements))
            this.convertedElements.registerPool(new ShapeBpmnElement(participant.id, participant.name, ShapeBpmnElementKind.POOL), participant.processRef);
    }
    buildMessageFlows(bpmnElements) {
        for (const messageFlow of ensureIsArray(bpmnElements))
            this.convertedElements.registerMessageFlow(new MessageFlow(messageFlow.id, messageFlow.name, messageFlow.sourceRef, messageFlow.targetRef));
    }
    buildGroups(bpmnElements) {
        for (const groupBpmnElement of ensureIsArray(bpmnElements)) {
            const shapeBpmnElement = buildShapeBpmnGroup(this.convertedElements, this.parsingMessageCollector, groupBpmnElement);
            shapeBpmnElement && this.convertedElements.registerFlowNode(shapeBpmnElement);
        }
    }
    buildTextAnnotation(bpmnElements) {
        for (const textAnnotation of ensureIsArray(bpmnElements)) {
            this.convertedElements.registerFlowNode(new ShapeBpmnElement(textAnnotation.id, textAnnotation.text, ShapeBpmnElementKind.TEXT_ANNOTATION));
        }
    }
}

class Bounds {
    constructor(x, y, width, height) {
        Object.defineProperty(this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x
        });
        Object.defineProperty(this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y
        });
        Object.defineProperty(this, "width", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: width
        });
        Object.defineProperty(this, "height", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: height
        });
    }
}

class Edge {
    constructor(id, bpmnElement, waypoints, label, messageVisibleKind = MessageVisibleKind.NONE) {
        Object.defineProperty(this, "id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: id
        });
        Object.defineProperty(this, "bpmnElement", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bpmnElement
        });
        Object.defineProperty(this, "waypoints", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: waypoints
        });
        Object.defineProperty(this, "label", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: label
        });
        Object.defineProperty(this, "messageVisibleKind", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: messageVisibleKind
        });
        Object.defineProperty(this, "extensions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
    }
}
class Waypoint {
    constructor(x, y) {
        Object.defineProperty(this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x
        });
        Object.defineProperty(this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y
        });
    }
}

class Label {
    constructor(font, bounds) {
        Object.defineProperty(this, "font", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: font
        });
        Object.defineProperty(this, "bounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bounds
        });
        Object.defineProperty(this, "extensions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
    }
}
class Font {
    constructor(name, size, isBold, isItalic, isUnderline, isStrikeThrough) {
        Object.defineProperty(this, "name", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: name
        });
        Object.defineProperty(this, "size", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: size
        });
        Object.defineProperty(this, "isBold", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: isBold
        });
        Object.defineProperty(this, "isItalic", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: isItalic
        });
        Object.defineProperty(this, "isUnderline", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: isUnderline
        });
        Object.defineProperty(this, "isStrikeThrough", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: isStrikeThrough
        });
    }
}

class DiagramConverter {
    constructor(convertedElements, parsingMessageCollector) {
        Object.defineProperty(this, "convertedElements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: convertedElements
        });
        Object.defineProperty(this, "parsingMessageCollector", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: parsingMessageCollector
        });
        Object.defineProperty(this, "convertedFonts", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
    }
    deserialize(bpmnDiagrams) {
        const flowNodes = [];
        const lanes = [];
        const pools = [];
        const edges = [];
        const bpmnDiagram = ensureIsArray(bpmnDiagrams)[0];
        if (bpmnDiagram) {
            this.deserializeFonts(bpmnDiagram.BPMNLabelStyle);
            const plane = bpmnDiagram.BPMNPlane;
            const convertedEdges = this.deserializeEdges(plane.BPMNEdge);
            const convertedShapes = this.deserializeShapes(plane.BPMNShape);
            flowNodes.push(...convertedShapes.flowNodes);
            lanes.push(...convertedShapes.lanes);
            pools.push(...convertedShapes.pools);
            edges.push(...convertedEdges);
        }
        return { flowNodes, lanes, pools, edges };
    }
    deserializeFonts(bpmnLabelStyle) {
        for (const labelStyle of ensureIsArray(bpmnLabelStyle))
            for (const font of ensureIsArray(labelStyle.Font))
                this.convertedFonts.set(labelStyle.id, new Font(font.name, font.size, font.isBold, font.isItalic, font.isUnderline, font.isStrikeThrough));
    }
    deserializeShapes(shapes) {
        const convertedShapes = { flowNodes: [], lanes: [], pools: [] };
        for (const shape of ensureIsArray(shapes)) {
            if (this.deserializeShapeAndStoreIfFound(shape, convertedShapes.flowNodes, (bpmnElementId) => this.convertedElements.findFlowNode(bpmnElementId))) {
                continue;
            }
            if (this.deserializeShapeAndStoreIfFound(shape, convertedShapes.lanes, (bpmnElementId) => this.convertedElements.findLane(bpmnElementId))) {
                continue;
            }
            if (this.deserializeShapeAndStoreIfFound(shape, convertedShapes.pools, (bpmnElementId) => this.convertedElements.findPoolById(bpmnElementId))) {
                continue;
            }
            this.parsingMessageCollector.warning(new ShapeUnknownBpmnElementWarning(shape.bpmnElement));
        }
        return convertedShapes;
    }
    deserializeShapeAndStoreIfFound(shape, storage, findShapeElement) {
        const element = this.deserializeShape(shape, findShapeElement);
        if (element) {
            storage.push(element);
            return true;
        }
        return false;
    }
    deserializeShape(bpmnShape, findShapeElement) {
        var _a;
        const bpmnElement = findShapeElement(bpmnShape.bpmnElement);
        if (bpmnElement) {
            const bounds = deserializeBounds(bpmnShape);
            if ((bpmnElement instanceof ShapeBpmnSubProcess ||
                (bpmnElement instanceof ShapeBpmnCallActivity && bpmnElement.callActivityKind === ShapeBpmnCallActivityKind.CALLING_PROCESS)) &&
                !bpmnShape.isExpanded) {
                bpmnElement.markers.push(ShapeBpmnMarkerKind.EXPAND);
            }
            let isHorizontal;
            if (ShapeUtil.isPoolOrLane(bpmnElement.kind)) {
                isHorizontal = (_a = bpmnShape.isHorizontal) !== null && _a !== void 0 ? _a : true;
            }
            const bpmnLabel = bpmnShape.BPMNLabel;
            const label = this.deserializeLabel(bpmnLabel, bpmnShape.id);
            const shape = new Shape(bpmnShape.id, bpmnElement, bounds, label, isHorizontal);
            setColorExtensionsOnShape(shape, bpmnShape);
            return shape;
        }
    }
    deserializeEdges(edges) {
        return ensureIsArray(edges)
            .map(bpmnEdge => {
            const flow = this.convertedElements.findSequenceFlow(bpmnEdge.bpmnElement) ||
                this.convertedElements.findMessageFlow(bpmnEdge.bpmnElement) ||
                this.convertedElements.findAssociationFlow(bpmnEdge.bpmnElement);
            if (!flow) {
                this.parsingMessageCollector.warning(new EdgeUnknownBpmnElementWarning(bpmnEdge.bpmnElement));
                return;
            }
            const waypoints = this.deserializeWaypoints(bpmnEdge.waypoint);
            const label = this.deserializeLabel(bpmnEdge.BPMNLabel, bpmnEdge.id);
            const messageVisibleKind = bpmnEdge.messageVisibleKind ? bpmnEdge.messageVisibleKind : MessageVisibleKind.NONE;
            const edge = new Edge(bpmnEdge.id, flow, waypoints, label, messageVisibleKind);
            setColorExtensionsOnEdge(edge, bpmnEdge);
            return edge;
        })
            .filter(Boolean);
    }
    deserializeWaypoints(waypoints) {
        return ensureIsArray(waypoints).map(waypoint => new Waypoint(waypoint.x, waypoint.y));
    }
    deserializeLabel(bpmnLabel, id) {
        if (bpmnLabel && typeof bpmnLabel === 'object') {
            const font = this.findFont(bpmnLabel.labelStyle, id);
            const bounds = deserializeBounds(bpmnLabel);
            const label = new Label(font, bounds);
            if ('color' in bpmnLabel) {
                label.extensions.color = bpmnLabel.color;
                return label;
            }
            if (font || bounds) {
                return label;
            }
        }
    }
    findFont(labelStyle, id) {
        let font;
        if (labelStyle) {
            font = this.convertedFonts.get(labelStyle);
            if (!font) {
                this.parsingMessageCollector.warning(new LabelStyleMissingFontWarning(id, labelStyle));
            }
        }
        return font;
    }
}
function setColorExtensionsOnShape(shape, bpmnShape) {
    if ('background-color' in bpmnShape) {
        shape.extensions.fillColor = bpmnShape['background-color'];
    }
    else if ('fill' in bpmnShape) {
        shape.extensions.fillColor = bpmnShape.fill;
    }
    if ('border-color' in bpmnShape) {
        shape.extensions.strokeColor = bpmnShape['border-color'];
    }
    else if ('stroke' in bpmnShape) {
        shape.extensions.strokeColor = bpmnShape.stroke;
    }
}
function deserializeBounds(boundedElement) {
    const bounds = boundedElement.Bounds;
    if (bounds) {
        return new Bounds(bounds.x, bounds.y, bounds.width, bounds.height);
    }
}
function setColorExtensionsOnEdge(edge, bpmnEdge) {
    if ('border-color' in bpmnEdge) {
        edge.extensions.strokeColor = bpmnEdge['border-color'];
    }
    else if ('stroke' in bpmnEdge) {
        edge.extensions.strokeColor = bpmnEdge.stroke;
    }
}

const isTLinkEventDefinition = (eventDefinition) => 'source' in eventDefinition || 'target' in eventDefinition;

class EventDefinitionConverter {
    constructor(convertedElements) {
        Object.defineProperty(this, "convertedElements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: convertedElements
        });
    }
    deserialize(definitions) {
        for (const eventDefinitionKind of eventDefinitionKinds) {
            const eventDefinitions = definitions[(eventDefinitionKind + 'EventDefinition')];
            for (const eventDefinition of ensureIsArray(eventDefinitions, true)) {
                this.convertedElements.registerEventDefinitionsOfDefinition(eventDefinition.id, Object.assign({ id: eventDefinition.id, kind: eventDefinitionKind }, (isTLinkEventDefinition(eventDefinition)
                    ? {
                        source: eventDefinition.source,
                        target: eventDefinition.target,
                    }
                    : {})));
            }
        }
    }
}

class GlobalTaskConverter {
    constructor(convertedElements) {
        Object.defineProperty(this, "convertedElements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: convertedElements
        });
    }
    deserialize(definitions) {
        this.parseGlobalTasks(definitions.globalTask, ShapeBpmnElementKind.GLOBAL_TASK);
        this.parseGlobalTasks(definitions.globalBusinessRuleTask, ShapeBpmnElementKind.GLOBAL_TASK_BUSINESS_RULE);
        this.parseGlobalTasks(definitions.globalManualTask, ShapeBpmnElementKind.GLOBAL_TASK_MANUAL);
        this.parseGlobalTasks(definitions.globalScriptTask, ShapeBpmnElementKind.GLOBAL_TASK_SCRIPT);
        this.parseGlobalTasks(definitions.globalUserTask, ShapeBpmnElementKind.GLOBAL_TASK_USER);
    }
    parseGlobalTasks(globalTasks, kind) {
        for (const globalTask of ensureIsArray(globalTasks))
            this.convertedElements.registerGlobalTask(globalTask.id, kind);
    }
}

const computeSubProcessKind = (processedSemanticType, bpmnElement) => {
    switch (processedSemanticType) {
        case 'adHocSubProcess': {
            return ShapeBpmnSubProcessKind.AD_HOC;
        }
        case 'transaction': {
            return ShapeBpmnSubProcessKind.TRANSACTION;
        }
        default: {
            return bpmnElement.triggeredByEvent ? ShapeBpmnSubProcessKind.EVENT : ShapeBpmnSubProcessKind.EMBEDDED;
        }
    }
};
const orderedFlowNodeBpmnTypes = [
    'adHocSubProcess',
    'transaction',
    ...ShapeUtil.flowNodeKinds().filter(kind => kind !== ShapeBpmnElementKind.EVENT_BOUNDARY),
    ShapeBpmnElementKind.EVENT_BOUNDARY,
];
function getShapeBpmnElementKind(bpmnSemanticType) {
    return ['adHocSubProcess', 'transaction'].includes(bpmnSemanticType) ? ShapeBpmnElementKind.SUB_PROCESS : bpmnSemanticType;
}
class ProcessConverter {
    constructor(convertedElements, parsingMessageCollector) {
        Object.defineProperty(this, "convertedElements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: convertedElements
        });
        Object.defineProperty(this, "parsingMessageCollector", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: parsingMessageCollector
        });
        Object.defineProperty(this, "defaultSequenceFlowIds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "elementsWithoutParentByProcessId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "callActivitiesCallingProcess", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "eventsByLinkEventDefinition", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
    }
    deserialize(processes) {
        for (const process of ensureIsArray(processes))
            this.parseProcess(process);
        for (const process of ensureIsArray(processes))
            this.assignParentOfProcessElementsCalledByCallActivity(process.id);
        this.assignIncomingAndOutgoingIdsFromFlows();
        this.assignSourceAndTargetIdsToLinkEvents();
    }
    assignParentOfProcessElementsCalledByCallActivity(processId) {
        const callActivity = this.callActivitiesCallingProcess.get(processId);
        if (callActivity) {
            const pool = this.convertedElements.findPoolByProcessRef(processId);
            if (pool) {
                pool.parentId = callActivity.id;
            }
            for (const element of this.elementsWithoutParentByProcessId.get(processId)) {
                element.parentId = callActivity.id;
            }
        }
    }
    assignIncomingAndOutgoingIdsFromFlows() {
        const fillShapeBpmnElementAttribute = (shapeBpmnElementId, shapeBpmnElementAttributeName, valueToAdd) => {
            var _a, _b;
            const shapeBpmnElement = (_b = (_a = this.convertedElements.findFlowNode(shapeBpmnElementId)) !== null && _a !== void 0 ? _a : this.convertedElements.findLane(shapeBpmnElementId)) !== null && _b !== void 0 ? _b : this.convertedElements.findPoolById(shapeBpmnElementId);
            if (shapeBpmnElement && !shapeBpmnElement[shapeBpmnElementAttributeName].includes(valueToAdd)) {
                shapeBpmnElement[shapeBpmnElementAttributeName].push(valueToAdd);
            }
        };
        for (const flow of this.convertedElements.getFlows()) {
            fillShapeBpmnElementAttribute(flow.sourceReferenceId, 'outgoingIds', flow.id);
            fillShapeBpmnElementAttribute(flow.targetReferenceId, 'incomingIds', flow.id);
        }
    }
    assignSourceAndTargetIdsToLinkEvents() {
        var _a;
        const linkEventDefinitions = [...this.eventsByLinkEventDefinition.entries()].filter(([targetEventDefinition]) => targetEventDefinition.id);
        for (const [eventDefinition, bpmnEvent] of this.eventsByLinkEventDefinition) {
            if (bpmnEvent instanceof ShapeBpmnIntermediateThrowEvent) {
                const target = linkEventDefinitions.find(([targetEventDefinition]) => eventDefinition.target === targetEventDefinition.id);
                bpmnEvent.targetId = (_a = target === null || target === void 0 ? void 0 : target[1]) === null || _a === void 0 ? void 0 : _a.id;
            }
            else if (bpmnEvent instanceof ShapeBpmnIntermediateCatchEvent) {
                bpmnEvent.sourceIds = linkEventDefinitions
                    .filter(([sourceEventDefinition]) => Array.isArray(eventDefinition.source) ? eventDefinition.source.includes(sourceEventDefinition.id) : eventDefinition.source === sourceEventDefinition.id)
                    .map(([, sourceEvent]) => sourceEvent.id);
            }
        }
    }
    parseProcess(process) {
        const processReference = process.id;
        const pool = this.convertedElements.findPoolByProcessRef(processReference);
        if (pool && !pool.name) {
            this.convertedElements.registerPool(new ShapeBpmnElement(pool.id, process.name, ShapeBpmnElementKind.POOL), processReference);
        }
        this.buildProcessInnerElements(process, pool === null || pool === void 0 ? void 0 : pool.id);
    }
    buildProcessInnerElements(process, parentId) {
        this.elementsWithoutParentByProcessId.set(process.id, []);
        for (const bpmnType of orderedFlowNodeBpmnTypes)
            this.buildFlowNodeBpmnElements(process[bpmnType], getShapeBpmnElementKind(bpmnType), parentId, process.id, bpmnType);
        this.buildLaneSetBpmnElements(process.laneSet, parentId, process.id);
        this.buildSequenceFlows(process.sequenceFlow);
        convertAndRegisterAssociationFlows(this.convertedElements, process.association);
    }
    buildFlowNodeBpmnElements(bpmnElements, kind, parentId, processId, processedSemanticType) {
        for (const bpmnElement of ensureIsArray(bpmnElements)) {
            const shapeBpmnElement = this.buildFlowNodeBpmnElement(kind, bpmnElement, parentId, processedSemanticType);
            if ('default' in bpmnElement && ShapeUtil.isWithDefaultSequenceFlow(kind)) {
                this.defaultSequenceFlowIds.push(bpmnElement.default);
            }
            if (shapeBpmnElement) {
                this.convertedElements.registerFlowNode(shapeBpmnElement);
                if (!parentId) {
                    this.elementsWithoutParentByProcessId.get(processId).push(shapeBpmnElement);
                }
            }
        }
    }
    buildFlowNodeBpmnElement(kind, bpmnElement, parentId, processedSemanticType) {
        if (ShapeUtil.isEvent(kind)) {
            return this.buildShapeBpmnEvent(bpmnElement, kind, parentId);
        }
        else if (ShapeUtil.isActivity(kind)) {
            return this.buildShapeBpmnActivity(bpmnElement, kind, parentId, processedSemanticType);
        }
        else if (kind == ShapeBpmnElementKind.GATEWAY_EVENT_BASED) {
            const eventBasedGatewayBpmnElement = bpmnElement;
            return new ShapeBpmnEventBasedGateway(eventBasedGatewayBpmnElement.id, eventBasedGatewayBpmnElement.name, parentId, eventBasedGatewayBpmnElement.instantiate, ShapeBpmnEventBasedGatewayKind[eventBasedGatewayBpmnElement.eventGatewayType]);
        }
        else if (kind == ShapeBpmnElementKind.GROUP) {
            return buildShapeBpmnGroup(this.convertedElements, this.parsingMessageCollector, bpmnElement, parentId);
        }
        else {
            const name = kind === ShapeBpmnElementKind.TEXT_ANNOTATION ? bpmnElement.text : bpmnElement.name;
            return new ShapeBpmnElement(bpmnElement.id, name, kind, parentId, bpmnElement.instantiate);
        }
    }
    buildShapeBpmnActivity(bpmnElement, kind, parentId, processedSemanticType) {
        const markers = buildMarkers(bpmnElement);
        if (ShapeUtil.isSubProcess(kind)) {
            return this.buildShapeBpmnSubProcess(bpmnElement, parentId, computeSubProcessKind(processedSemanticType, bpmnElement), markers);
        }
        if (!ShapeUtil.isCallActivity(kind)) {
            return new ShapeBpmnActivity(bpmnElement.id, bpmnElement.name, kind, parentId, bpmnElement.instantiate, markers);
        }
        return this.buildShapeBpmnCallActivity(bpmnElement, parentId, markers);
    }
    buildShapeBpmnCallActivity(bpmnElement, parentId, markers) {
        const globalTaskKind = this.convertedElements.findGlobalTask(bpmnElement.calledElement);
        if (!globalTaskKind) {
            const shapeBpmnCallActivity = new ShapeBpmnCallActivity(bpmnElement.id, bpmnElement.name, ShapeBpmnCallActivityKind.CALLING_PROCESS, parentId, markers);
            this.callActivitiesCallingProcess.set(bpmnElement.calledElement, shapeBpmnCallActivity);
            return shapeBpmnCallActivity;
        }
        return new ShapeBpmnCallActivity(bpmnElement.id, bpmnElement.name, ShapeBpmnCallActivityKind.CALLING_GLOBAL_TASK, parentId, markers, globalTaskKind);
    }
    buildShapeBpmnEvent(bpmnElement, elementKind, parentId) {
        const eventDefinitionsByKind = this.getEventDefinitions(bpmnElement);
        const numberOfEventDefinitions = [...eventDefinitionsByKind.entries()]
            .map(([, registeredEventDefinitions]) => registeredEventDefinitions.length)
            .reduce((counter, it) => counter + it, 0);
        if (numberOfEventDefinitions == 0 && ShapeUtil.canHaveNoneEvent(elementKind)) {
            return new ShapeBpmnEvent(bpmnElement.id, bpmnElement.name, elementKind, ShapeBpmnEventDefinitionKind.NONE, parentId);
        }
        if (numberOfEventDefinitions == 1) {
            const [eventDefinitionKind, eventDefinitions] = [...eventDefinitionsByKind.entries()][0];
            const bpmnEvent = ShapeUtil.isCatchEvent(elementKind)
                ? this.buildShapeBpmnCatchEvent(bpmnElement, elementKind, eventDefinitionKind, parentId)
                : this.buildShapeBpmnThrowEvent(bpmnElement, elementKind, eventDefinitionKind, parentId);
            if (eventDefinitionKind === ShapeBpmnEventDefinitionKind.LINK && (eventDefinitions[0].id || eventDefinitions[0].target || eventDefinitions[0].source)) {
                this.eventsByLinkEventDefinition.set(eventDefinitions[0], bpmnEvent);
            }
            return bpmnEvent;
        }
    }
    buildShapeBpmnCatchEvent(bpmnElement, elementKind, eventDefinitionKind, parentId) {
        if (ShapeUtil.isBoundaryEvent(elementKind)) {
            return this.buildShapeBpmnBoundaryEvent(bpmnElement, eventDefinitionKind);
        }
        if (ShapeUtil.isStartEvent(elementKind)) {
            return new ShapeBpmnStartEvent(bpmnElement.id, bpmnElement.name, eventDefinitionKind, parentId, bpmnElement.isInterrupting);
        }
        return new ShapeBpmnIntermediateCatchEvent(bpmnElement.id, bpmnElement.name, eventDefinitionKind, parentId);
    }
    buildShapeBpmnThrowEvent(bpmnElement, elementKind, eventDefinitionKind, parentId) {
        if (ShapeUtil.isIntermediateThrowEvent(elementKind)) {
            return new ShapeBpmnIntermediateThrowEvent(bpmnElement.id, bpmnElement.name, eventDefinitionKind, parentId);
        }
        return new ShapeBpmnEvent(bpmnElement.id, bpmnElement.name, elementKind, eventDefinitionKind, parentId);
    }
    buildShapeBpmnBoundaryEvent(bpmnElement, eventDefinitionKind) {
        const parent = this.convertedElements.findFlowNode(bpmnElement.attachedToRef);
        if (ShapeUtil.isActivity(parent === null || parent === void 0 ? void 0 : parent.kind)) {
            return new ShapeBpmnBoundaryEvent(bpmnElement.id, bpmnElement.name, eventDefinitionKind, bpmnElement.attachedToRef, bpmnElement.cancelActivity);
        }
        else {
            this.parsingMessageCollector.warning(new BoundaryEventNotAttachedToActivityWarning(bpmnElement.id, bpmnElement.attachedToRef, parent === null || parent === void 0 ? void 0 : parent.kind));
        }
    }
    getEventDefinitions(bpmnElement) {
        const eventDefinitionsByKind = new Map();
        for (const eventDefinitionKind of eventDefinitionKinds) {
            const eventDefinition = bpmnElement[`${eventDefinitionKind}EventDefinition`];
            eventDefinitionsByKind.set(eventDefinitionKind, ensureIsArray(eventDefinition, true));
        }
        for (const eventDefinitionReference of ensureIsArray(bpmnElement.eventDefinitionRef)) {
            const eventDefinition = this.convertedElements.findEventDefinitionOfDefinition(eventDefinitionReference);
            eventDefinition && eventDefinitionsByKind.get(eventDefinition.kind).push(eventDefinition);
        }
        for (const [kind] of [...eventDefinitionsByKind.entries()].filter(([, registeredEventDefinitions]) => registeredEventDefinitions.length === 0)) {
            eventDefinitionsByKind.delete(kind);
        }
        return eventDefinitionsByKind;
    }
    buildShapeBpmnSubProcess(bpmnElement, parentId, subProcessKind, markers) {
        const convertedSubProcess = new ShapeBpmnSubProcess(bpmnElement.id, bpmnElement.name, subProcessKind, parentId, markers);
        this.buildProcessInnerElements(bpmnElement, bpmnElement.id);
        return convertedSubProcess;
    }
    buildLaneSetBpmnElements(laneSets, parentId, processId) {
        for (const laneSet of ensureIsArray(laneSets))
            this.buildLaneBpmnElements(laneSet.lane, parentId, processId);
    }
    buildLaneBpmnElements(lanes, parentId, processId) {
        var _a;
        for (const lane of ensureIsArray(lanes)) {
            const shapeBpmnElement = new ShapeBpmnElement(lane.id, lane.name, ShapeBpmnElementKind.LANE, parentId);
            this.convertedElements.registerLane(shapeBpmnElement);
            if (!parentId) {
                this.elementsWithoutParentByProcessId.get(processId).push(shapeBpmnElement);
            }
            this.assignParentOfLaneFlowNodes(lane);
            if ((_a = lane.childLaneSet) === null || _a === void 0 ? void 0 : _a.lane) {
                this.buildLaneBpmnElements(lane.childLaneSet.lane, lane.id, processId);
            }
        }
    }
    assignParentOfLaneFlowNodes(lane) {
        for (const flowNodeReference of ensureIsArray(lane.flowNodeRef)) {
            const shapeBpmnElement = this.convertedElements.findFlowNode(flowNodeReference);
            const laneId = lane.id;
            if (shapeBpmnElement) {
                if (!ShapeUtil.isBoundaryEvent(shapeBpmnElement.kind)) {
                    shapeBpmnElement.parentId = laneId;
                }
            }
            else {
                this.parsingMessageCollector.warning(new LaneUnknownFlowNodeReferenceWarning(laneId, flowNodeReference));
            }
        }
    }
    buildSequenceFlows(bpmnElements) {
        for (const sequenceFlow of ensureIsArray(bpmnElements)) {
            const kind = this.getSequenceFlowKind(sequenceFlow);
            this.convertedElements.registerSequenceFlow(new SequenceFlow(sequenceFlow.id, sequenceFlow.name, sequenceFlow.sourceRef, sequenceFlow.targetRef, kind));
        }
    }
    getSequenceFlowKind(sequenceFlow) {
        if (this.defaultSequenceFlowIds.includes(sequenceFlow.id)) {
            return SequenceFlowKind.DEFAULT;
        }
        else {
            const sourceShapeBpmnElement = this.convertedElements.findFlowNode(sequenceFlow.sourceRef);
            if (sourceShapeBpmnElement && ShapeUtil.isWithDefaultSequenceFlow(sourceShapeBpmnElement.kind) && sequenceFlow.conditionExpression) {
                return ShapeUtil.isActivity(sourceShapeBpmnElement.kind) ? SequenceFlowKind.CONDITIONAL_FROM_ACTIVITY : SequenceFlowKind.CONDITIONAL_FROM_GATEWAY;
            }
        }
        return SequenceFlowKind.NORMAL;
    }
}
const buildMarkers = (bpmnElement) => {
    const markers = [];
    const standardLoopCharacteristics = bpmnElement.standardLoopCharacteristics;
    const multiInstanceLoopCharacteristics = ensureIsArray(bpmnElement.multiInstanceLoopCharacteristics, true)[0];
    if (standardLoopCharacteristics !== undefined) {
        markers.push(ShapeBpmnMarkerKind.LOOP);
    }
    else if (multiInstanceLoopCharacteristics) {
        markers.push(multiInstanceLoopCharacteristics.isSequential ? ShapeBpmnMarkerKind.MULTI_INSTANCE_SEQUENTIAL : ShapeBpmnMarkerKind.MULTI_INSTANCE_PARALLEL);
    }
    return markers;
};

class BpmnJsonParser {
    constructor(categoryConverter, collaborationConverter, eventDefinitionConverter, globalTaskConverter, processConverter, diagramConverter) {
        Object.defineProperty(this, "categoryConverter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: categoryConverter
        });
        Object.defineProperty(this, "collaborationConverter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: collaborationConverter
        });
        Object.defineProperty(this, "eventDefinitionConverter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: eventDefinitionConverter
        });
        Object.defineProperty(this, "globalTaskConverter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: globalTaskConverter
        });
        Object.defineProperty(this, "processConverter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: processConverter
        });
        Object.defineProperty(this, "diagramConverter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: diagramConverter
        });
    }
    parse(json) {
        const definitions = json.definitions;
        this.categoryConverter.deserialize(definitions);
        this.collaborationConverter.deserialize(definitions.collaboration);
        this.eventDefinitionConverter.deserialize(definitions);
        this.globalTaskConverter.deserialize(definitions);
        this.processConverter.deserialize(definitions.process);
        return this.diagramConverter.deserialize(definitions.BPMNDiagram);
    }
}
function newBpmnJsonParser(parsingMessageCollector) {
    const convertedElements = new ConvertedElements();
    return new BpmnJsonParser(new CategoryConverter(convertedElements), new CollaborationConverter(convertedElements, parsingMessageCollector), new EventDefinitionConverter(convertedElements), new GlobalTaskConverter(convertedElements), new ProcessConverter(convertedElements, parsingMessageCollector), new DiagramConverter(convertedElements, parsingMessageCollector));
}

const entitiesReplacements = [
    { regex: /&(amp|#38|#x26);/g, val: '&' },
    { regex: /&(apos|#39|#x27);/g, val: "'" },
    { regex: /&#(xa|xA|10);/g, val: '\n' },
    { regex: /&(gt|#62|#x3e|#x3E);/g, val: '>' },
    { regex: /&(lt|#60|#x3c|#x3C);/g, val: '<' },
    { regex: /&(quot|#34|#x22);/g, val: '"' },
];
const nodesWithNumericAttributes = new Set(['BPMNShape.Bounds', 'BPMNShape.BPMNLabel.Bounds', 'BPMNEdge.BPMNLabel.Bounds', 'BPMNEdge.waypoint'].map(element => `definitions.BPMNDiagram.BPMNPlane.${element}`));
const numericAttributes = new Set(['x', 'y', 'width', 'height']);
const isNumeric = (attributeName, nodePath) => {
    return nodesWithNumericAttributes.has(nodePath) && numericAttributes.has(attributeName);
};
class BpmnXmlParser {
    constructor(options) {
        Object.defineProperty(this, "options", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: options
        });
        Object.defineProperty(this, "x2jOptions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {
                attributeNamePrefix: '',
                removeNSPrefix: true,
                ignoreAttributes: false,
                parseAttributeValue: true,
                processEntities: false,
                attributeValueProcessor: (name, value, nodePath) => {
                    if (isNumeric(name, nodePath)) {
                        return Number(value);
                    }
                    return this.processAttribute(value);
                },
            }
        });
        Object.defineProperty(this, "xmlParser", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new XMLParser(this.x2jOptions)
        });
    }
    parse(xml) {
        let model;
        try {
            model = this.xmlParser.parse(xml);
        }
        catch (_a) {
            throw new Error('XML parsing failed. Invalid BPMN source.');
        }
        if (!model.definitions) {
            throw new Error(`XML parsing failed. Unable to retrieve 'definitions' from the BPMN source.`);
        }
        return model;
    }
    processAttribute(value) {
        var _a;
        for (const replacement of entitiesReplacements) {
            value = value.replace(replacement.regex, replacement.val);
        }
        if ((_a = this.options) === null || _a === void 0 ? void 0 : _a.additionalXmlAttributeProcessor) {
            value = this.options.additionalXmlAttributeProcessor(value);
        }
        return value;
    }
}

class BpmnParser {
    constructor(jsonParser, xmlParser) {
        Object.defineProperty(this, "jsonParser", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: jsonParser
        });
        Object.defineProperty(this, "xmlParser", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: xmlParser
        });
    }
    parse(bpmnAsXml) {
        const json = this.xmlParser.parse(bpmnAsXml);
        return this.jsonParser.parse(json);
    }
}
function newBpmnParser(options) {
    return new BpmnParser(newBpmnJsonParser(new ParsingMessageCollector(options)), new BpmnXmlParser(options));
}

function createNewCssClassesUpdater(graph) {
    return new CssClassesUpdater(graph);
}
class CssClassesUpdater {
    constructor(graph) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: graph
        });
    }
    updateAndRefreshCssClassesOfCell(bpmnElementId, cssClasses) {
        this.updateAndRefreshCssClassesOfElement(bpmnElementId, cssClasses);
        this.updateAndRefreshCssClassesOfElement(messageFlowIconId(bpmnElementId), cssClasses);
    }
    updateAndRefreshCssClassesOfElement(elementId, cssClasses) {
        const model = this.graph.getModel();
        const cell = model.getCell(elementId);
        if (!cell) {
            return;
        }
        let cellStyle = cell.getStyle();
        cellStyle = setStyle(cellStyle, BpmnStyleIdentifier.EXTRA_CSS_CLASSES, cssClasses.join(','));
        model.setStyle(cell, cellStyle);
    }
}

function createNewCssRegistry(graph) {
    return new CssClassesRegistryImpl(createNewCssClassesUpdater(graph), new CssClassesCache());
}
class CssClassesRegistryImpl {
    constructor(cssClassesUpdater, cssClassesCache) {
        Object.defineProperty(this, "cssClassesUpdater", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cssClassesUpdater
        });
        Object.defineProperty(this, "cssClassesCache", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cssClassesCache
        });
    }
    clearCache() {
        this.cssClassesCache.clear();
    }
    addCssClasses(bpmnElementIds, classNames) {
        this.updateCssClasses(bpmnElementIds, classNames, this.cssClassesCache.addClassNames);
    }
    removeCssClasses(bpmnElementIds, classNames) {
        this.updateCssClasses(bpmnElementIds, classNames, this.cssClassesCache.removeClassNames);
    }
    removeAllCssClasses(bpmnElementIds) {
        if (bpmnElementIds || bpmnElementIds == '') {
            for (const bpmnElementId of ensureIsArray(bpmnElementIds)) {
                const isChanged = this.cssClassesCache.removeAllClassNames(bpmnElementId);
                this.updateCellIfChanged(isChanged, bpmnElementId);
            }
        }
        else {
            const bpmnIds = this.cssClassesCache.getBpmnIds();
            this.cssClassesCache.clear();
            for (const bpmnElementId of bpmnIds)
                this.updateCellIfChanged(true, bpmnElementId);
        }
    }
    toggleCssClasses(bpmnElementIds, classNames) {
        this.updateCssClasses(bpmnElementIds, classNames, this.cssClassesCache.toggleClassNames);
    }
    updateCssClasses(bpmnElementIds, classNames, updateClassNames) {
        const arrayClassNames = ensureIsArray(classNames);
        for (const bpmnElementId of ensureIsArray(bpmnElementIds))
            this.updateCellIfChanged(updateClassNames(bpmnElementId, arrayClassNames), bpmnElementId);
    }
    updateCellIfChanged(updateCell, bpmnElementId) {
        if (updateCell) {
            const allClassNames = this.cssClassesCache.getClassNames(bpmnElementId);
            this.cssClassesUpdater.updateAndRefreshCssClassesOfCell(bpmnElementId, allClassNames);
        }
    }
}
class CssClassesCache {
    constructor() {
        Object.defineProperty(this, "classNamesByBpmnId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        Object.defineProperty(this, "clear", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: () => {
                this.classNamesByBpmnId.clear();
            }
        });
        Object.defineProperty(this, "addClassNames", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (bpmnElementId, classNames) => this.updateClassNames(bpmnElementId, classNames, (currentClassNames, className) => currentClassNames.add(className))
        });
        Object.defineProperty(this, "removeClassNames", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (bpmnElementId, classNames) => this.updateClassNames(bpmnElementId, classNames, (currentClassNames, className) => currentClassNames.delete(className))
        });
        Object.defineProperty(this, "toggleClassNames", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (bpmnElementId, classNames) => {
                this.updateClassNames(bpmnElementId, classNames, (currentClassNames, className) => currentClassNames.has(className) ? currentClassNames.delete(className) : currentClassNames.add(className));
                return classNames && classNames.length > 0;
            }
        });
    }
    getClassNames(bpmnElementId) {
        return this.classNamesByBpmnId.has(bpmnElementId) ? [...this.classNamesByBpmnId.get(bpmnElementId)] : [];
    }
    getBpmnIds() {
        return [...this.classNamesByBpmnId.keys()];
    }
    removeAllClassNames(bpmnElementId) {
        const currentClassNames = this.getOrInitializeClassNames(bpmnElementId);
        const initialClassNamesNumber = currentClassNames.size;
        currentClassNames.clear();
        return currentClassNames.size < initialClassNamesNumber;
    }
    updateClassNames(bpmnElementId, classNames, manageClassNames) {
        const currentClassNames = this.getOrInitializeClassNames(bpmnElementId);
        const initialClassNamesNumber = currentClassNames.size;
        for (const className of ensureIsArray(classNames))
            manageClassNames(currentClassNames, className);
        return currentClassNames.size != initialClassNamesNumber;
    }
    getOrInitializeClassNames(bpmnElementId) {
        let classNames = this.classNamesByBpmnId.get(bpmnElementId);
        if (classNames == null) {
            classNames = new Set();
            this.classNamesByBpmnId.set(bpmnElementId, classNames);
        }
        return classNames;
    }
}

const overlayPositions = new Map([
    ['start', { horizontalAlign: 'left', verticalAlign: 'top' }],
    ['middle', { horizontalAlign: 'center', verticalAlign: 'top' }],
    ['end', { horizontalAlign: 'right', verticalAlign: 'top' }],
    ['top-left', { horizontalAlign: 'left', verticalAlign: 'top' }],
    ['top-right', { horizontalAlign: 'right', verticalAlign: 'top' }],
    ['top-center', { horizontalAlign: 'center', verticalAlign: 'top' }],
    ['bottom-left', { horizontalAlign: 'left', verticalAlign: 'bottom' }],
    ['bottom-right', { horizontalAlign: 'right', verticalAlign: 'bottom' }],
    ['bottom-center', { horizontalAlign: 'center', verticalAlign: 'bottom' }],
    ['middle-left', { horizontalAlign: 'left', verticalAlign: 'middle' }],
    ['middle-right', { horizontalAlign: 'right', verticalAlign: 'middle' }],
]);
const convertPosition = (overlay) => overlayPositions.get(overlay.position);
const convertFill = (convertedStyle, apiFill) => {
    var _a, _b;
    if (apiFill) {
        convertedStyle.fill.color = (_a = apiFill.color) !== null && _a !== void 0 ? _a : convertedStyle.fill.color;
        convertedStyle.fill.opacity = (_b = apiFill.opacity) !== null && _b !== void 0 ? _b : convertedStyle.fill.opacity;
    }
};
const convertStroke = (convertedStyle, apiStroke) => {
    var _a, _b;
    if (apiStroke) {
        convertedStyle.stroke.color = (_a = apiStroke.color) !== null && _a !== void 0 ? _a : convertedStyle.stroke.color;
        convertedStyle.stroke.width = (_b = apiStroke.width) !== null && _b !== void 0 ? _b : convertedStyle.stroke.width;
    }
};
const convertFont = (convertedStyle, apiFont) => {
    var _a, _b;
    if (apiFont) {
        convertedStyle.font.color = (_a = apiFont.color) !== null && _a !== void 0 ? _a : convertedStyle.font.color;
        convertedStyle.font.size = (_b = apiFont.size) !== null && _b !== void 0 ? _b : convertedStyle.font.size;
    }
};
const convertStyle = (overlay) => {
    const defaultStyle = {
        fill: { color: StyleDefault.DEFAULT_OVERLAY_FILL_COLOR.valueOf(), opacity: StyleDefault.DEFAULT_OVERLAY_FILL_OPACITY.valueOf() },
        stroke: { color: StyleDefault.DEFAULT_OVERLAY_STROKE_COLOR.valueOf(), width: StyleDefault.DEFAULT_OVERLAY_STROKE_WIDTH.valueOf() },
        font: { color: StyleDefault.DEFAULT_OVERLAY_FONT_COLOR.valueOf(), size: StyleDefault.DEFAULT_OVERLAY_FONT_SIZE.valueOf() },
    };
    const style = overlay.style;
    const convertedStyle = Object.assign({}, defaultStyle);
    if (!style) {
        return convertedStyle;
    }
    convertFill(convertedStyle, style.fill);
    convertStroke(convertedStyle, style.stroke);
    convertFont(convertedStyle, style.font);
    return convertedStyle;
};
class OverlayConverter {
    convert(overlay) {
        const position = convertPosition(overlay);
        const style = convertStyle(overlay);
        return { position, style };
    }
}

function createNewOverlaysUpdater(graph) {
    return new OverlaysUpdater(graph, new OverlayConverter());
}
class OverlaysUpdater {
    constructor(graph, overlayConverter) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: graph
        });
        Object.defineProperty(this, "overlayConverter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: overlayConverter
        });
    }
    addOverlays(bpmnElementId, overlays) {
        const cell = this.graph.getModel().getCell(bpmnElementId);
        if (!cell) {
            return;
        }
        for (const overlay of ensureIsArray(overlays)) {
            const bpmnOverlay = new MxGraphCustomOverlay(overlay.label, this.overlayConverter.convert(overlay));
            this.graph.addCellOverlay(cell, bpmnOverlay);
        }
    }
    removeAllOverlays(bpmnElementId) {
        const cell = this.graph.getModel().getCell(bpmnElementId);
        if (!cell) {
            return;
        }
        this.graph.removeCellOverlays(cell);
    }
}

function createNewOverlaysRegistry(graph) {
    return new OverlaysRegistryImpl(createNewOverlaysUpdater(graph));
}
class OverlaysRegistryImpl {
    constructor(overlaysUpdater) {
        Object.defineProperty(this, "overlaysUpdater", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: overlaysUpdater
        });
    }
    addOverlays(bpmnElementId, overlays) {
        this.overlaysUpdater.addOverlays(bpmnElementId, overlays);
    }
    removeAllOverlays(bpmnElementId) {
        this.overlaysUpdater.removeAllOverlays(bpmnElementId);
    }
}

class BpmnQuerySelectors {
    element(bpmnElementId) {
        return `svg > g > g > g[data-bpmn-id="${bpmnElementId}"]`;
    }
    elementsOfKind(bpmnKindCssClassname) {
        return `svg > g > g > g.${bpmnKindCssClassname}:not(.bpmn-label)`;
    }
}

function createNewStyleUpdater(graph) {
    return new StyleUpdater(graph, new StyleManager(graph.getModel()));
}
const withCellIdsOfMessageFlowIcons = (bpmnElementIds) => {
    const cellIds = ensureIsArray(bpmnElementIds);
    cellIds.push(...cellIds.map(id => messageFlowIconId(id)));
    return cellIds;
};
class StyleUpdater {
    constructor(graph, styleManager) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: graph
        });
        Object.defineProperty(this, "styleManager", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: styleManager
        });
    }
    clear() {
        this.styleManager.clear();
    }
    updateStyle(bpmnElementIds, styleUpdate) {
        if (!styleUpdate) {
            return;
        }
        const model = this.graph.getModel();
        const cells = withCellIdsOfMessageFlowIcons(bpmnElementIds)
            .map(id => model.getCell(id))
            .filter(Boolean);
        if (cells.length === 0) {
            return;
        }
        this.graph.batchUpdate(() => {
            for (const cell of cells) {
                this.styleManager.ensureStyleIsStored(cell);
                let cellStyle = cell.getStyle();
                cellStyle = setStyle(cellStyle, mxConstants.STYLE_OPACITY, styleUpdate.opacity, ensureOpacityValue);
                cellStyle = updateStroke(cellStyle, styleUpdate.stroke);
                cellStyle = updateFont(cellStyle, styleUpdate.font);
                if (isShapeStyleUpdate(styleUpdate)) {
                    cellStyle = updateFill(cellStyle, styleUpdate.fill);
                }
                model.setStyle(cell, cellStyle);
            }
        });
    }
    resetStyle(bpmnElementIds) {
        this.graph.batchUpdate(() => {
            if (bpmnElementIds || bpmnElementIds == '') {
                for (const id of withCellIdsOfMessageFlowIcons(bpmnElementIds)) {
                    this.styleManager.resetStyleIfIsStored(id);
                }
            }
            else {
                this.styleManager.resetAllStyles();
            }
        });
    }
}
const cssClassesStyleIdentifier = BpmnStyleIdentifier.EXTRA_CSS_CLASSES;
class StyleManager {
    constructor(model) {
        Object.defineProperty(this, "model", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: model
        });
        Object.defineProperty(this, "stylesCache", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
    }
    clear() {
        this.stylesCache.clear();
    }
    resetAllStyles() {
        for (const cellId of this.stylesCache.keys()) {
            this.resetStyle(cellId, this.stylesCache.get(cellId));
        }
    }
    resetStyleIfIsStored(cellId) {
        const style = this.stylesCache.get(cellId);
        if (style) {
            this.resetStyle(cellId, style);
        }
    }
    resetStyle(cellId, initialStyle) {
        const cell = this.model.getCell(cellId);
        const cssClasses = getStyleValue(cell.getStyle(), cssClassesStyleIdentifier, '');
        const styleWithCssClasses = setStyle(initialStyle, cssClassesStyleIdentifier, cssClasses);
        this.model.setStyle(cell, styleWithCssClasses);
        this.stylesCache.delete(cellId);
    }
    ensureStyleIsStored(cell) {
        const cellId = cell.getId();
        if (!this.stylesCache.has(cellId)) {
            this.stylesCache.set(cellId, cell.getStyle());
        }
    }
}

function createNewStyleRegistry(graph) {
    return new StyleRegistryImpl(createNewStyleUpdater(graph));
}
class StyleRegistryImpl {
    constructor(styleUpdater) {
        Object.defineProperty(this, "styleUpdater", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: styleUpdater
        });
    }
    clearCache() {
        this.styleUpdater.clear();
    }
    updateStyle(bpmnElementIds, styleUpdate) {
        this.styleUpdater.updateStyle(bpmnElementIds, styleUpdate);
    }
    resetStyle(bpmnElementIds) {
        this.styleUpdater.resetStyle(bpmnElementIds);
    }
}

function createNewBpmnElementsRegistry(bpmnModelRegistry, graph) {
    return new BpmnElementsRegistry(bpmnModelRegistry, new HtmlElementRegistry(graph.container, new BpmnQuerySelectors()), createNewCssRegistry(graph), createNewOverlaysRegistry(graph), createNewStyleRegistry(graph));
}
class BpmnElementsRegistry {
    constructor(bpmnModelRegistry, htmlElementRegistry, cssClassesRegistry, overlaysRegistry, styleRegistry) {
        Object.defineProperty(this, "bpmnModelRegistry", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bpmnModelRegistry
        });
        Object.defineProperty(this, "htmlElementRegistry", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: htmlElementRegistry
        });
        Object.defineProperty(this, "cssClassesRegistry", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cssClassesRegistry
        });
        Object.defineProperty(this, "overlaysRegistry", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: overlaysRegistry
        });
        Object.defineProperty(this, "styleRegistry", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: styleRegistry
        });
        this.bpmnModelRegistry.registerOnLoadCallback(() => {
            this.cssClassesRegistry.clearCache();
            this.styleRegistry.clearCache();
        });
    }
    getModelElementsByIds(bpmnElementIds) {
        return filterDuplicates(bpmnElementIds)
            .map(id => this.bpmnModelRegistry.getBpmnSemantic(id))
            .filter(Boolean);
    }
    getElementsByIds(bpmnElementIds) {
        return this.getModelElementsByIds(bpmnElementIds).map(bpmnSemantic => ({
            bpmnSemantic: bpmnSemantic,
            htmlElement: this.htmlElementRegistry.getBpmnHtmlElement(bpmnSemantic.id),
        }));
    }
    getModelElementsByKinds(bpmnKinds) {
        return filterDuplicates(bpmnKinds)
            .flatMap(kind => this.htmlElementRegistry.getBpmnHtmlElements(kind))
            .map(htmlElement => this.getRelatedBpmnSemantic(htmlElement));
    }
    getElementsByKinds(bpmnKinds) {
        return filterDuplicates(bpmnKinds)
            .flatMap(kind => this.htmlElementRegistry.getBpmnHtmlElements(kind))
            .map(htmlElement => ({ htmlElement, bpmnSemantic: this.getRelatedBpmnSemantic(htmlElement) }));
    }
    getRelatedBpmnSemantic(htmlElement) {
        return this.bpmnModelRegistry.getBpmnSemantic(htmlElement.dataset.bpmnId);
    }
    addCssClasses(bpmnElementIds, classNames) {
        this.cssClassesRegistry.addCssClasses(bpmnElementIds, classNames);
    }
    removeCssClasses(bpmnElementIds, classNames) {
        this.cssClassesRegistry.removeCssClasses(bpmnElementIds, classNames);
    }
    removeAllCssClasses(bpmnElementIds) {
        this.cssClassesRegistry.removeAllCssClasses(bpmnElementIds);
    }
    toggleCssClasses(bpmnElementIds, classNames) {
        this.cssClassesRegistry.toggleCssClasses(bpmnElementIds, classNames);
    }
    addOverlays(bpmnElementId, overlays) {
        this.overlaysRegistry.addOverlays(bpmnElementId, overlays);
    }
    removeAllOverlays(bpmnElementId) {
        this.overlaysRegistry.removeAllOverlays(bpmnElementId);
    }
    updateStyle(bpmnElementIds, styleUpdate) {
        this.styleRegistry.updateStyle(bpmnElementIds, styleUpdate);
    }
    resetStyle(bpmnElementIds) {
        this.styleRegistry.resetStyle(bpmnElementIds);
    }
}
class HtmlElementRegistry {
    constructor(container, querySelectors) {
        Object.defineProperty(this, "container", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: container
        });
        Object.defineProperty(this, "querySelectors", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: querySelectors
        });
    }
    getBpmnHtmlElement(bpmnElementId) {
        return this.container.querySelector(this.querySelectors.element(bpmnElementId));
    }
    getBpmnHtmlElements(bpmnElementKind) {
        const selectors = this.querySelectors.elementsOfKind(computeBpmnBaseClassName(bpmnElementKind));
        return [...this.container.querySelectorAll(selectors)];
    }
}
const filterDuplicates = (ids) => [...new Set(ensureIsArray(ids))];

class ModelFiltering {
    filter(bpmnModel, modelFilter) {
        const poolIdsFilter = [];
        const poolNamesFilter = [];
        for (const filter of ensureIsArray(modelFilter === null || modelFilter === void 0 ? void 0 : modelFilter.pools).filter(p => p && Object.keys(p).length > 0))
            filter.id ? poolIdsFilter.push(filter.id) : filter.name && poolNamesFilter.push(filter.name);
        if (poolIdsFilter.length === 0 && poolNamesFilter.length === 0) {
            return bpmnModel;
        }
        const { filteredPools, filteredPoolIds } = filterPools(bpmnModel, poolIdsFilter, poolNamesFilter);
        const filteredPoolAndBlackPoolIds = [...poolIdsFilter, ...filteredPoolIds];
        const { filteredLanes, filteredLaneIds, filteredFlowNodes, filteredFlowNodeIds } = filterLanesAndFlowNodes(bpmnModel.lanes, bpmnModel.flowNodes, filteredPoolAndBlackPoolIds);
        const filteredEdges = filterEdges(bpmnModel.edges, [...filteredPoolAndBlackPoolIds, ...filteredLaneIds, ...filteredFlowNodeIds]);
        if (filteredPools.length === 0 && filteredLanes.length === 0 && filteredFlowNodes.length === 0 && filteredEdges.length === 0) {
            let errorMessageSuffix = poolIdsFilter.length > 0 ? ` for ids [${poolIdsFilter}]` : '';
            const messageSeparator = errorMessageSuffix ? ' and' : '';
            errorMessageSuffix += poolNamesFilter.length > 0 ? `${messageSeparator} for names [${poolNamesFilter}]` : '';
            throw new Error('No matching pools' + errorMessageSuffix);
        }
        return { lanes: filteredLanes, flowNodes: filteredFlowNodes, pools: filteredPools, edges: filteredEdges };
    }
}
function filterPools(bpmnModel, poolIdsFilter, poolNamesFilter) {
    const filteredPools = bpmnModel.pools.filter(pool => poolIdsFilter.includes(pool.bpmnElement.id) || poolNamesFilter.includes(pool.bpmnElement.name));
    const filteredPoolIds = filteredPools.map(shape => shape.bpmnElement.id);
    return { filteredPools, filteredPoolIds };
}
function filterLanesAndFlowNodes(lanes, flowNodes, parentIdsToFilter) {
    const { filteredLanes, filteredLaneIds } = filterLanes(lanes, parentIdsToFilter);
    const { filteredLanes: filteredSubLanes, filteredLaneIds: filteredSubLanesIds, filteredFlowNodes, filteredFlowNodeIds, } = filterFlowNodes(flowNodes, [...parentIdsToFilter, ...filteredLaneIds], lanes);
    filteredLanes.push(...filteredSubLanes);
    filteredLaneIds.push(...filteredSubLanesIds);
    return { filteredLanes, filteredLaneIds, filteredFlowNodes, filteredFlowNodeIds };
}
function filterLanes(lanes, parentIdsToFilter) {
    const filteredLanes = lanes.filter(shape => parentIdsToFilter.includes(shape.bpmnElement.parentId));
    const filteredLaneIds = filteredLanes.map(shape => shape.bpmnElement.id);
    if (filteredLanes.length > 0) {
        const { filteredLanes: filteredSubLanes, filteredLaneIds: filteredSubLaneIds } = filterLanes(lanes, filteredLaneIds);
        filteredLanes.push(...filteredSubLanes);
        filteredLaneIds.push(...filteredSubLaneIds);
    }
    return { filteredLanes, filteredLaneIds };
}
function filterFlowNodes(flowNodes, parentIdsToFilter, lanes) {
    const filteredFlowNodes = flowNodes.filter(shape => parentIdsToFilter.includes(shape.bpmnElement.parentId));
    if (filteredFlowNodes.length === 0) {
        return { filteredLanes: [], filteredLaneIds: [], filteredFlowNodes: [], filteredFlowNodeIds: [] };
    }
    const filteredFlowNodeIds = filteredFlowNodes.map(shape => shape.bpmnElement.id);
    const { filteredLanes, filteredLaneIds, filteredFlowNodes: filteredChildFlowNodes, filteredFlowNodeIds: filteredChildFlowNodeIds, } = filterLanesAndFlowNodes(lanes, flowNodes, filteredFlowNodeIds);
    filteredFlowNodes.push(...filteredChildFlowNodes);
    filteredFlowNodeIds.push(...filteredChildFlowNodeIds);
    return { filteredLanes, filteredLaneIds, filteredFlowNodes, filteredFlowNodeIds };
}
function filterEdges(edges, filteredElementIds) {
    return edges.filter(edge => filteredElementIds.includes(edge.bpmnElement.sourceReferenceId) && filteredElementIds.includes(edge.bpmnElement.targetReferenceId));
}

class BpmnModelRegistry {
    constructor() {
        Object.defineProperty(this, "searchableModel", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "onLoadCallback", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
    }
    load(bpmnModel, modelFilter) {
        var _a;
        const filteredModel = new ModelFiltering().filter(bpmnModel, modelFilter);
        this.searchableModel = new SearchableModel(filteredModel);
        (_a = this.onLoadCallback) === null || _a === void 0 ? void 0 : _a.call(this);
        return toRenderedModel(filteredModel);
    }
    registerOnLoadCallback(callback) {
        this.onLoadCallback = callback;
    }
    getBpmnSemantic(bpmnElementId) {
        var _a;
        const element = (_a = this.searchableModel) === null || _a === void 0 ? void 0 : _a.elementById(bpmnElementId);
        if (!element) {
            return undefined;
        }
        const bpmnElement = element.bpmnElement;
        const isShape = bpmnElement instanceof ShapeBpmnElement;
        return Object.assign({ id: bpmnElementId, isShape: isShape, kind: bpmnElement.kind, name: bpmnElement.name }, (bpmnElement instanceof Flow
            ? {
                sourceRefId: bpmnElement.sourceReferenceId,
                targetRefId: bpmnElement.targetReferenceId,
            }
            : {
                callActivityGlobalTaskKind: bpmnElement instanceof ShapeBpmnCallActivity ? bpmnElement.globalTaskKind : undefined,
                callActivityKind: bpmnElement instanceof ShapeBpmnCallActivity ? bpmnElement.callActivityKind : undefined,
                eventDefinitionKind: bpmnElement instanceof ShapeBpmnEvent ? bpmnElement.eventDefinitionKind : undefined,
                linkEventSourceIds: bpmnElement instanceof ShapeBpmnIntermediateCatchEvent && bpmnElement.eventDefinitionKind == ShapeBpmnEventDefinitionKind.LINK ? bpmnElement.sourceIds : undefined,
                linkEventTargetId: bpmnElement instanceof ShapeBpmnIntermediateThrowEvent ? bpmnElement.targetId : undefined,
                incomingIds: bpmnElement.incomingIds,
                outgoingIds: bpmnElement.outgoingIds,
                parentId: bpmnElement.parentId,
                subProcessKind: bpmnElement instanceof ShapeBpmnSubProcess ? bpmnElement.subProcessKind : undefined,
            }));
    }
}
function toRenderedModel(bpmnModel) {
    const collapsedSubProcessIds = new Set(bpmnModel.flowNodes
        .filter(shape => {
        const bpmnElement = shape.bpmnElement;
        return ShapeUtil.isSubProcess(bpmnElement.kind) && bpmnElement.markers.includes(ShapeBpmnMarkerKind.EXPAND);
    })
        .map(shape => shape.bpmnElement.id));
    const subprocesses = [];
    const boundaryEvents = [];
    const otherFlowNodes = [];
    for (const shape of bpmnModel.flowNodes) {
        const kind = shape.bpmnElement.kind;
        if (ShapeUtil.isSubProcess(kind)) {
            subprocesses.push(shape);
        }
        else if (ShapeUtil.isBoundaryEvent(kind)) {
            boundaryEvents.push(shape);
        }
        else if (!collapsedSubProcessIds.has(shape.bpmnElement.parentId)) {
            otherFlowNodes.push(shape);
        }
    }
    return { boundaryEvents: boundaryEvents, edges: bpmnModel.edges, lanes: bpmnModel.lanes, otherFlowNodes: otherFlowNodes, pools: bpmnModel.pools, subprocesses: subprocesses };
}
class SearchableModel {
    constructor(bpmnModel) {
        Object.defineProperty(this, "elements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Map()
        });
        for (const shapeOrEdge of [...bpmnModel.pools, ...bpmnModel.lanes, ...bpmnModel.flowNodes, ...bpmnModel.edges]) {
            this.elements.set(shapeOrEdge.bpmnElement.id, shapeOrEdge);
        }
    }
    elementById(id) {
        return this.elements.get(id);
    }
}

class BpmnVisualization {
    constructor(options) {
        Object.defineProperty(this, "graph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "navigation", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "bpmnElementsRegistry", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "bpmnModelRegistry", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "parserOptions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "rendererOptions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.rendererOptions = options === null || options === void 0 ? void 0 : options.renderer;
        const configurator = new GraphConfigurator(htmlElement(options === null || options === void 0 ? void 0 : options.container));
        this.graph = configurator.configure(options);
        this.navigation = new Navigation(this.graph);
        this.bpmnModelRegistry = new BpmnModelRegistry();
        this.bpmnElementsRegistry = createNewBpmnElementsRegistry(this.bpmnModelRegistry, this.graph);
        this.parserOptions = options === null || options === void 0 ? void 0 : options.parser;
    }
    load(xml, options) {
        const bpmnModel = newBpmnParser(this.parserOptions).parse(xml);
        const renderedModel = this.bpmnModelRegistry.load(bpmnModel, options === null || options === void 0 ? void 0 : options.modelFilter);
        newBpmnRenderer(this.graph, this.rendererOptions).render(renderedModel, options === null || options === void 0 ? void 0 : options.fit);
    }
}

const libraryVersion = '0.46.0';
const getVersion = () => {
    return { lib: libraryVersion, dependencies: new Map([['mxGraph', mxClient.VERSION]]) };
};

export { AssociationDirectionKind, BpmnCanvas, BpmnElementsRegistry, BpmnStyleIdentifier, BpmnVisualization, FitType, FlowKind, IconPainter, IconPainterProvider, MarkerIdentifier, MessageVisibleKind, SequenceFlowKind, ShapeBpmnCallActivityKind, ShapeBpmnElementKind, ShapeBpmnEventBasedGatewayKind, ShapeBpmnEventDefinitionKind, ShapeBpmnMarkerKind, ShapeBpmnSubProcessKind, ShapeUtil, StyleConfigurator, StyleDefault, ZoomType, getVersion, mxgraph };
