/// <reference types="node" />
import { BaseFileSystem, FileSystem, BFSCallback, FileSystemOptions } from '../core/file_system';
import { FileFlag } from '../core/file_flag';
import { File } from '../core/file';
import Stats from '../core/node_fs_stats';
/**
 * Configuration options for an XmlHttpRequest file system.
 */
export interface XmlHttpRequestOptions {
    index?: string | object;
    baseUrl?: string;
}
/**
 * A simple filesystem backed by XMLHttpRequests. You must create a directory listing using the
 * `make_xhrfs_index` tool provided by BrowserFS.
 *
 * If you install BrowserFS globally with `npm i -g browserfs`, you can generate a listing by
 * running `make_xhrfs_index` in your terminal in the directory you would like to index:
 *
 * ```
 * make_xhrfs_index > index.json
 * ```
 *
 * Listings objects look like the following:
 *
 * ```json
 * {
 *   "home": {
 *     "jvilk": {
 *       "someFile.txt": null,
 *       "someDir": {
 *         // Empty directory
 *       }
 *     }
 *   }
 * }
 * ```
 *
 * *This example has the folder `/home/jvilk` with subfile `someFile.txt` and subfolder `someDir`.*
 */
export default class XmlHttpRequest extends BaseFileSystem implements FileSystem {
    static readonly Name: string;
    static readonly Options: FileSystemOptions;
    /**
     * Construct an XmlHttpRequest file system backend with the given options.
     */
    static Create(opts: XmlHttpRequestOptions, cb: BFSCallback<XmlHttpRequest>): void;
    static isAvailable(): boolean;
    /**
     * **Deprecated. Please use XmlHttpRequest.Create() method instead to construct XmlHttpRequest objects.**
     *
     * Constructs an XmlHttpRequest object using the directory listing at the given URL.
     * Uses the base URL as the URL prefix for fetched files.
     * @param cb Called when the file system has been instantiated, or if an error occurs.
     */
    static FromURL(url: string, cb: BFSCallback<XmlHttpRequest>, baseUrl?: string, deprecateMsg?: boolean): void;
    readonly prefixUrl: string;
    private _index;
    /**
     * **Deprecated. Please use XmlHttpRequest.Create() method instead to construct XmlHttpRequest objects.**
     *
     * Constructs the file system. You must provide the directory listing as a JSON object
     * produced by the `make_xhrfs_index` script.
     *
     * **DEPRECATED:** You may pass a URL to the file index to the constructor, which will fetch the file index
     * *synchronously* and may freeze up the web page. This behavior will be removed in the next major version
     * of BrowserFS.
     *
     * @param listingUrlOrObj index object or the path to the JSON file index generated by
     *   `make_xhrfs_index`.
     * @param prefixUrl URL that is prepended to any file locations in the file index. e.g. if `prefixUrl = 'data/`, and the user wants to open the file `/foo.txt`,
     * the file system will fetch file `data/foo.txt`. The browser will access the file relative to the currrent webpage
     * URL.
     */
    constructor(listingUrlOrObj: string | object, prefixUrl?: string, deprecateMsg?: boolean);
    empty(): void;
    getName(): string;
    diskSpace(path: string, cb: (total: number, free: number) => void): void;
    isReadOnly(): boolean;
    supportsLinks(): boolean;
    supportsProps(): boolean;
    supportsSynch(): boolean;
    /**
     * Special XHR function: Preload the given file into the index.
     * @param [String] path
     * @param [BrowserFS.Buffer] buffer
     */
    preloadFile(path: string, buffer: Buffer): void;
    stat(path: string, isLstat: boolean, cb: BFSCallback<Stats>): void;
    statSync(path: string, isLstat: boolean): Stats;
    open(path: string, flags: FileFlag, mode: number, cb: BFSCallback<File>): void;
    openSync(path: string, flags: FileFlag, mode: number): File;
    readdir(path: string, cb: BFSCallback<string[]>): void;
    readdirSync(path: string): string[];
    /**
     * We have the entire file as a buffer; optimize readFile.
     */
    readFile(fname: string, encoding: string, flag: FileFlag, cb: BFSCallback<string | Buffer>): void;
    /**
     * Specially-optimized readfile.
     */
    readFileSync(fname: string, encoding: string, flag: FileFlag): any;
    private getXhrPath(filePath);
    /**
     * Asynchronously download the given file.
     */
    private _requestFileAsync(p, type, cb);
    private _requestFileAsync(p, type, cb);
    private _requestFileAsync(p, type, cb);
    /**
     * Synchronously download the given file.
     */
    private _requestFileSync(p, type);
    private _requestFileSync(p, type);
    private _requestFileSync(p, type);
    /**
     * Only requests the HEAD content, for the file size.
     */
    private _requestFileSizeAsync(path, cb);
    private _requestFileSizeSync(path);
}
