describe('functions are available', function() {
    before(function() {});
    it('is jquery available', function() {
        expect($).to.be.a('function');
    });
    it('$.fn.busyLoad is a function', function() {
        expect($.fn.busyLoad).to.be.a('function');
    });
    it('$.busyLoadFull is a function', function() {
        expect($.busyLoadFull).to.be.a('function');
    });
    it('$.busyLoadSetup is a function', function() {
        expect($.busyLoadSetup).to.be.a('function');
    });
    it("jQuery is working", function() {
        expect($('body').length).to.be.equal(1);
    });
});
describe('busy-load core', function() {
    it('$.fn.busyLoad.defaults. is same as from $.busyLoadSetup', function() {
        $.busyLoadSetup({
            color: "green"
        });
        expect($.fn.busyLoad.defaults.color).to.be.equal('green');
    });
    it('$.busyLoadFull("show") without options', function() {
        $.busyLoadFull('show');
        expect($('body>.busy-load-container')).to.exist;
    });
    it('$.busyLoadFull("hide")', function() {
        $.busyLoadFull('hide');
        expect($('body>.busy-load-container')).not.to.exist;
    });
    it('busyLoad("show") without options', function() {
        $('#maskme').busyLoad('show');
        expect($('#maskme>.busy-load-container')).to.exist;
    });
    it('busyLoad("hide")', function() {
        $('#maskme').busyLoad('hide');
        expect($('#maskme>.busy-load-container')).not.to.exist;
    });
    it('caller gets active class', function() {
        $('#maskme').busyLoad('show');
        expect($('#maskme')).to.have.class('busy-load-active');
    });
    it('caller removes active class', function() {
        $('#maskme').busyLoad('hide');
        expect($('#maskme')).not.to.have.class('busy-load-active');
    });
    it('caller turns position of static to relative', function() {
        $('#maskme').css('postion', 'static');
        $('#maskme').busyLoad('show'); 
        expect($('#maskme')).to.have.css('position', 'relative');
        $('#maskme').busyLoad('hide');
    });


    it('busyLoad("show") on multiple elements', function() {
        $('.maskme').busyLoad('show');
        expect($('.maskme>.busy-load-container').length).to.be.equal(4);
    });
    it('busyLoad("hide") on multiple elements', function() {
        $('.maskme').busyLoad('hide');
        expect($('#maskme>.busy-load-container')).not.to.exist;
    });
});
describe('busy-load components', function() {
    before(function() {
        $('#maskme').busyLoad('show', {
            text: 'Loading'
        });
    });
    after(function() {
        $('#maskme').busyLoad('hide');
    });
    it('container exists', function() {
        expect($('#maskme>.busy-load-container')).to.exist;
    });
    it('container-item exists', function() {
        expect($('#maskme>.busy-load-container>.busy-load-container-item')).to.exist;
    });
    it('spinner exists', function() {
        expect($('#maskme>.busy-load-container>.busy-load-container-item>.busy-load-spinner')).to.exist;
    });
    it('text exists', function() {
        expect($('#maskme>.busy-load-container>.busy-load-container-item>.busy-load-text')).to.exist;
    });
});
describe('busy-load settings', function() {
    afterEach(function() {
        $('#maskme').busyLoad('hide');
    });
    it('use a custom spinner', function() {
        $('#maskme').busyLoad('show', {
            spinner: 'cube-grid'
        });
        expect($('#maskme .busy-load-spinner')).to.have.class('spinner-cube-grid');
    });
    it('use an image', function() {
        $('#maskme').busyLoad('show', {
            image: "data:image/gif;base64,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"
    });
        expect($('#maskme>.busy-load-container img')).to.exist;
    });
    it('use fontawesome', function() {
        $('#maskme').busyLoad('show', {
            fontawesome: 'fa fa-refresh fa-spin fa-2x fa-fw'
        });
        expect($('#maskme>.busy-load-container .fa-refresh')).to.exist;
    });
    it('use a custom element', function() {
        $('#maskme').busyLoad('show', {
            custom: $("<div>", {
                id: "iamcustom"
            })
        });
        expect($('#maskme>.busy-load-container #iamcustom')).to.exist;
    });
    it('set color', function() {
        $('#maskme').busyLoad('show', {
            color: "rgb(0, 0, 255)"
        });
        expect($('#maskme  .busy-load-spinner')).to.have.css('color', 'rgb(0, 0, 255)');
    });
    it('set background', function() {
        $('#maskme').busyLoad('show', {
            "background": "rgb(0, 0, 255)"
        });
        expect($('#maskme .busy-load-container')).to.have.css('background-color', 'rgb(0, 0, 255)');
    });
    it('set min/maxSize', function() {
        $('#maskme').busyLoad('show', {
            "minSize": "21px",
            "maxSize": "21px",
        });
        expect($('#maskme .busy-load-spinner')).to.have.css('min-width', '21px');
        expect($('#maskme .busy-load-spinner')).to.have.css('min-height', '21px');
        expect($('#maskme .busy-load-spinner')).to.have.css('max-width', '21px');
        expect($('#maskme .busy-load-spinner')).to.have.css('max-height', '21px');
    });
    it('text matches', function() {
        $('#maskme').busyLoad('show', {
            "text": "Loading"
        });
        expect($('#maskme>.busy-load-container>.busy-load-container-item>.busy-load-text').text()).to.contain('Loading');
    });
    it('text matches also text with blanks', function() {
        $('#maskme').busyLoad('show', {
            "text": "Whatever you want"
        });
        expect($('#maskme .busy-load-text').text()).to.contain('Whatever you want');
    });
    it('set textColor', function() {
        $('#maskme').busyLoad('show', {
            "text": "Loading",
            "textColor": "rgb(0, 21, 255)",
        });
        expect($('#maskme .busy-load-text')).to.have.css('color', 'rgb(0, 21, 255)');
    });
    it('textColor default is color', function() {
        $.fn.busyLoad.defaults.color = 'rgb(0, 128, 128)';
        $('#maskme').busyLoad('show', {
            "text": "Loading"
        });
        expect($('#maskme .busy-load-text')).to.have.css('color', 'rgb(0, 128, 128)');
    });
    it('set textMargin & position', function() {
        $('#maskme').busyLoad('show', {
            "text": "Loading",
            "textPosition": "top",
            "textMargin": "10px",
        });
        expect($('#maskme .busy-load-text')).to.have.css('margin-bottom', '10px');
    });
    it('set containerClass', function() {
        $('#maskme').busyLoad('show', {
            "containerClass": "something-special",
        });
        expect($('#maskme .something-special')).to.exist;
    });
    it('set containerItemClass', function() {
        $('#maskme').busyLoad('show', {
            "containerItemClass": "something-very-special",
        });
        expect($('#maskme .something-very-special')).to.exist;
    });
    it('set spinnerClass', function() {
        $('#maskme').busyLoad('show', {
            "spinnerClass": "something-very-very-special",
        });
        expect($('#maskme .something-very-very-special')).to.exist;
    });
    it('set textClass', function() {
        $('#maskme').busyLoad('show', {
            "text": "Loading",
            "textClass": "something-very-very-very-special",
        });
        expect($('#maskme .something-very-very-very-special')).to.exist;
    });
});
describe('busy-load settings with wrong parameters', function() {
    var fnAction = function() {
        $('#maskme').busyLoad('action-wrong')
    };
    it('wrong action', function() {
        expect(fnAction).to.throw("don't know action 'action-wrong'");
    });
    var fnAnimations = function() {
        $('#maskme').busyLoad('show', {
            animation: "animation-wrong"
        })
    };
    it('wrong animation', function() {
        expect(fnAnimations).to.throw("don\'t know animation: animation-wrong");
    });
    var fnTextPos = function() {
        $('#maskme').busyLoad('show', {
            textPosition: "textPosition-wrong"
        })
    };
    it('wrong textPosition', function() {
        expect(fnTextPos).to.throw("don\'t know textPosition: textPosition-wrong");
    });
    var fnCustomTag = function() {
        $('#maskme').busyLoad('show', {
            custom: "customEl-wrong"
        })
    };
    it('wrong customTag', function() {
        expect(fnCustomTag).to.throw("wrong type for creating a tag");
    });
    var fnSpinner = function() {
        $('#maskme').busyLoad('show', {
            spinner: "spinner-wrong"
        })
    };
    it('wrong spinner', function() {
        expect(fnSpinner).to.throw("don\'t know spinner: spinner-wrong");
    });
});
describe('busy-load events', function() {    
    it('Event: bl.show should be fired', function() { 
        var eventSpy = sinon.spy(); 
        $(document).on('bl.show', eventSpy); 
        $('#maskme').busyLoad('show');
        expect(eventSpy).to.have.been.calledOnce;
    });    
    it('Event: bl.hide should be fired', function() { 
        var eventSpy = sinon.spy(); 
        $(document).on('bl.hide', eventSpy); 
        $('#maskme').busyLoad('hide');
        expect(eventSpy).to.have.been.calledOnce;
    });
    it('Event: bl.shown should be fired', function() { 
        var eventSpy = sinon.spy(); 
        $(document).on('bl.shown', eventSpy); 
        $('#maskme').busyLoad('show');
        expect(eventSpy).to.have.been.calledOnce;
    });
    it('Event: bl.hidden should be fired', function() { 
        var eventSpy = sinon.spy(); 
        $(document).on('bl.hidden', eventSpy); 
        $('#maskme').busyLoad('hide');
        expect(eventSpy).to.have.been.calledOnce;
    });
});