// This file is part of the bwip-js project available at:
//
// 	  http://metafloor.github.io/bwip-js
//
// Copyright (c) 2011-2021 Mark Warren
//
// This file contains code automatically generated from:
// Barcode Writer in Pure PostScript - Version 2021-02-06
// Copyright (c) 2004-2021 Terry Burton
//
// The MIT License
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        define([], factory);
    } else if (typeof module === 'object' && module.exports) {
        module.exports = factory();
    } else {
        root.bwipjs = factory();
	}
}(typeof self !== 'undefined' ? self : this, function () {
"use strict";
// exports.js

//@@BEGIN-NODE-JS-ONLY@@
var url = require('url');

// bwipjs.request(req, res [, overrides])
//
// Returns a PNG image from the query args of a node.js http request object.
//
// This function is asynchronous.
//
// Node.js usage only.
function Request(req, res, extra) {
	var opts = url.parse(req.url, true).query;

	// Convert boolean empty parameters to true
	for (var id in opts) {
		if (opts[id] === '') {
			opts[id] = true;
		}
	}

	// Add in server options/overrides
	if (extra) {
		for (var id in extra) {
			opts[id] = extra[id];
		}
	}

	ToBuffer(opts, function(err, png) {
		if (err) {
			res.writeHead(400, { 'Content-Type':'text/plain' });
			res.end('' + (err.stack || err), 'utf-8');
		} else {
			res.writeHead(200, { 'Content-Type':'image/png' });
			res.end(png, 'binary');
		}
	});
}

// bwipjs.toBuffer(options[, callback])
//
// Uses the built-in graphics drawing and zlib PNG encoding to return a
// barcode image in a node.js Buffer.
//
// `options` are a bwip-js/BWIPP options object.
// `callback` is an optional callback handler with prototype:
//
// 		function callback(err, png)
//
// 		`err` is an Error object or string.  If `err` is set, `png` is null.
// 		`png` is a node Buffer containing the PNG image.
//
// If `callback` is not provided, a Promise is returned.
//
// Node.js usage only.
function ToBuffer(opts, callback) {
	try {
		FixupOptions(opts);
		return Render(opts, DrawingZlibPng(opts, callback));
	} catch (e) {
		if (callback) {
			callback(e);
		} else {
			return new Promise(function(resolve, reject) {
				reject(e);
			});
		}
	}
}
//@@ENDOF-NODE-JS-ONLY@@


// bwipjs.fixupOptions(options)
//
// Call this before passing your options object to a drawing constructor.
function FixupOptions(opts) {
	var scale	= opts.scale || 2;
	var scaleX	= +opts.scaleX || scale;
	var scaleY	= +opts.scaleY || scaleX;

	// Fix up padding.
	opts.paddingleft = padding(opts.paddingleft, opts.paddingwidth, opts.padding, scaleX);
	opts.paddingright = padding(opts.paddingright, opts.paddingwidth, opts.padding, scaleX);
	opts.paddingtop = padding(opts.paddingtop, opts.paddingheight, opts.padding, scaleY);
	opts.paddingbottom = padding(opts.paddingbottom, opts.paddingheight, opts.padding, scaleY);

	// We override BWIPP's background color functionality.  If in CMYK, convert to RGB so
	// the drawing interface is consistent.
	if (/^[0-9a-fA-F]{8}$/.test(''+opts.backgroundcolor)) {
		var cmyk = opts.backgroundcolor;
		var c = parseInt(cmyk.substr(0,2), 16) / 255;
		var m = parseInt(cmyk.substr(2,2), 16) / 255;
		var y = parseInt(cmyk.substr(4,2), 16) / 255;
		var k = parseInt(cmyk.substr(6,2), 16) / 255;
		var r = Math.floor((1-c) * (1-k) * 255).toString(16);
		var g = Math.floor((1-m) * (1-k) * 255).toString(16);
		var b = Math.floor((1-y) * (1-k) * 255).toString(16);
		opts.backgroundcolor = (r.length == 1 ? '0' : '') + r +
							   (g.length == 1 ? '0' : '') + g +
							   (b.length == 1 ? '0' : '') + b;
	}

	return opts;

	function padding(a, b, c, s) {
		if (a != null) {
			return a*s;
		}
		if (b != null) {
			return b*s;
		}
		return c*s || 0;
	}
}

var BWIPJS_OPTIONS = {
	bcid:1,
	text:1,
	scale:1,
	scaleX:1,
	scaleY:1,
	rotate:1,
	padding:1,
	paddingwidth:1,
	paddingheight:1,
	paddingtop:1,
	paddingleft:1,
	paddingright:1,
	paddingbottom:1,
	backgroundcolor:1,
};

// bwipjs.render(options, drawing)
//
// Renders a barcode using the provided drawing object.
//
// This function is synchronous and throws on error.
function Render(params, drawing) {
	// Set the bwip-js defaults
	var scale	= params.scale || 2;
	var scaleX	= +params.scaleX || scale;
	var scaleY	= +params.scaleY || scaleX;
	var rotate	= params.rotate || 'N';

	// The required parameters
	var bcid = params.bcid;
	var text = params.text;

	if (!text) {
		throw new ReferenceError('bwip-js: bar code text not specified.');
	}
	if (!bcid) {
		throw new ReferenceError('bwip-js: bar code type not specified.');
	}

	// Create a barcode writer object.  This is the interface between
	// the low-level BWIPP code, the bwip-js graphics context, and the
	// drawing interface.
	var bw = new BWIPJS(drawing);

	// Set the BWIPP options
	var opts = {};
	for (var id in params) {
		if (!BWIPJS_OPTIONS[id]) {
			opts[id] = params[id];
		}
	}

	// Fix a disconnect in the BWIPP rendering logic
	if (opts.alttext) {
		opts.includetext = true;
	}
	// We use mm rather than inches for height - except pharmacode2 height
	// which is already in mm.
	if (+opts.height && bcid != 'pharmacode2') {
		opts.height = opts.height / 25.4 || 0.5;
	}
	// Likewise, width
	if (+opts.width) {
		opts.width = opts.width / 25.4 || 0;
	}

	// Scale the image
	bw.scale(scaleX, scaleY);

	// Call into the BWIPP cross-compiled code and render the image.
	BWIPP()(bw, bcid, text, opts);
	return bw.render();		// Return whatever drawing.end() returns
}

// bwipjs.raw(options)
// bwipjs.raw(encoder, text, opts-string)
//
// Invokes the low level BWIPP code and returns the raw encoding data.
//
// This function is synchronous and throws on error.
function Raw(encoder, text, options) {
	if (arguments.length == 1) {
		options = encoder;
		encoder = options.bcid;
		text = options.text;
	}

	// The drawing interface is just needed for the pre-init() calls.
	var bw = new BWIPJS(DrawingBuiltin({}));
	var stack = BWIPP()(bw, encoder, text, options, true);

	// bwip-js uses Maps to emulate PostScript dictionary objects; but Maps
	// are not a typical/expected return value.  Convert to plain-old-objects.
	var ids = { pixs:1, pixx:1, pixy:1, sbs:1, bbs:1, bhs:1, width:1, height:1 };
	for (var i = 0; i < stack.length; i++) {
		var elt = stack[i];
		if (elt instanceof Map) {
			var obj = {};
			// Could they make Maps any harder to iterate over???
			for (var keys = elt.keys(), size = elt.size, k = 0; k < size; k++) {
				var id = keys.next().value;
				if (ids[id]) {
					var val = elt.get(id);
					if (val instanceof Array) {
						// The postscript arrays have extra named properties
						// to emulate array views.  Return cleaned up arrays.
						obj[id] = val.b.slice(val.o, val.o + val.length);
					} else {
						obj[id] = val;
					}
				}
			}
			stack[i] = obj;
		} else {
			// This should never exec...
			stack.splice(i--, 1);
		}
	}
	return stack;
}
// bwip-js // Barcode Writer in Pure JavaScript
// https://github.com/metafloor/bwip-js
//
// This code was automatically generated from:
// Barcode Writer in Pure PostScript - Version 2021-02-06
//
// Copyright (c) 2011-2021 Mark Warren
// Copyright (c) 2004-2021 Terry Burton
//
// Licensed MIT. See the LICENSE file in the bwip-js root directory.
function BWIPP() {
    "use strict";
    // bwip-js/barcode-hdr.js
    //
    // This code is injected above the cross-compiled barcode.ps.

    // The BWIPJS object (graphics interface)
    var $$ = null;

    // The global dictionary.  Each renderer and encoder declare a
    // $1 local dict.
    var $0 = {
        $error: new Map // the postscript error object
    };

    var $j = 0; // stack pointer
    var $k = []; // operand stack

    // Array ctor
    //	$a()	: Build a new array up to the Infinity-marker on the stack.
    //	$a(arr)	: Convert native array to a "view" of the array.
    //	$a(len)	: Create a new array of length `len`
    function $a(a) {
        if (!arguments.length) {
            for (var i = $j - 1; i >= 0 && $k[i] !== Infinity; i--);
            if (i < 0) {
                throw new Error('array-marker-not-found');
            }
            a = $k.splice(i + 1, $j - 1 - i);
            $j = i;
        } else if (!(a instanceof Array)) {
            a = new Array(+arguments[0]);
            for (var i = 0, l = a.length; i < l; i++) {
                a[i] = null;
            }
        }
        a.b = a; // base array
        a.o = 0; // offset into base
        return a;
    }

    // dict ctor
    //	$d() : look for the Infinity marker on the stack
    function $d() {
        // Build the dictionary in the order the keys/values were pushed so enumeration
        // occurs in the correct sequence.
        for (var mark = $j - 1; mark >= 0 && $k[mark] !== Infinity; mark -= 2) {
            if ($k[mark - 1] === Infinity) {
                throw new Error('dict-malformed-stack');
            }
        }
        if (mark < 0) {
            throw 'dict-marker-not-found';
        }
        var d = new Map;
        for (var i = mark + 1; i < $j; i += 2) {
            // Unlike javascript, postscript dict keys differentiate between
            // numbers and the string representation of a number.
            var k = $k[i]; // "key" into the dict entry
            var t = typeof k;
            if (t == 'number' || t == 'string') {
                d.set(k, $k[i + 1]);
            } else if (k instanceof Uint8Array) {
                d.set($z(k), $k[i + 1]);
            } else {
                throw 'dict-not-a-valid-key(' + k + ')';
            }
        }
        $j = mark;
        return d;
    }

    // string ctor
    //	s(number)	: create zero-filled string of number-length
    //	s(string)	: make a copy of the string
    //	s(uint8[])	: make a copy of the string
    //
    // Returns a Uint8Array-string.
    function $s(v) {
        var t = typeof v;
        if (t === 'number') {
            return new Uint8Array(v);
        }
        if (t !== 'string') {
            v = '' + v;
        }
        var s = new Uint8Array(v.length);
        for (var i = 0; i < v.length; i++) {
            s[i] = v.charCodeAt(i);
        }
        return s;
    }

    // ... n c roll
    function $r(n, c) {
        if ($j < n) {
            throw 'roll: --stack-underflow--';
        }
        if (!c) {
            return;
        }
        if (c < 0) {
            var t = $k.splice($j - n, -c);
        } else {
            var t = $k.splice($j - n, n - c);
        }
        $k.splice.apply($k, [$j - t.length, 0].concat(t));
    }

    // Primarily designed to convert uint8-string to string, but will call the
    // the toString() method on any value.
    function $z(s) {
        if (s instanceof Uint8Array) {
            // Postscript treats nul-char as end of string, even if string is
            // longer.
            for (var i = 0, l = s.length; i < l && s[i]; i++);
            if (i < l) {
                return String.fromCharCode.apply(null, s.subarray(0, i));
            }
            return String.fromCharCode.apply(null, s)
        }
        return '' + s;
    }

    // Copies source to dest and returns a view of just the copied characters
    function $strcpy(dst, src) {
        if (typeof dst === 'string') {
            dst = $s(dst);
        }
        if (src instanceof Uint8Array) {
            for (var i = 0, l = src.length; i < l; i++) {
                dst[i] = src[i];
            }
        } else {
            for (var i = 0, l = src.length; i < l; i++) {
                dst[i] = src.charCodeAt(i);
            }
        }
        return src.length < dst.length ? dst.subarray(0, src.length) : dst;
    }

    // Copies source to dest and should (but doesn't) return a view of just the copied elements
    function $arrcpy(dst, src) {
        for (var i = 0, l = src.length; i < l; i++) {
            dst[i] = src[i];
        }
        dst.length = src.length;
        return dst;
    }

    // cvs operator - convert a value to its string representation
    //	s : string to store into
    //	v : any value
    function $cvs(s, v) {
        var t = typeof v;
        if (t == 'number' || t == 'boolean' || v === null) {
            v = '' + v;
        } else if (t !== 'string') {
            v = '--nostringval--';
        }
        for (var i = 0, l = v.length; i < l; i++) {
            s[i] = v.charCodeAt(i);
        }
        $k[$j++] = i < s.length ? s.subarray(0, i) : s;
    }
    // cvrs operator - convert a number to a radix string
    //	s : string to store into
    //	n : number
    //	r : radix
    function $cvrs(s, n, r) {
        return $strcpy(s, (~~n).toString(r).toUpperCase());
    }

    // get operator
    //	s : source
    //	k : key
    function $get(s, k) {
        if (s instanceof Uint8Array) {
            return s[k];
        }
        if (typeof s === 'string') {
            return s.charCodeAt(k);
        }
        if (s instanceof Array) {
            return s.b[s.o + k];
        }
        if (k instanceof Uint8Array) {
            return s.get($z(k));
        }
        return s.get(k);
    }

    // put operator
    //	d : dest
    //	k : key
    //	v : value
    function $put(d, k, v) {
        if (d instanceof Uint8Array) {
            d[k] = v;
        } else if (d instanceof Array) {
            d.b[d.o + k] = v;
        } else if (typeof d == 'object') {
            if (k instanceof Uint8Array) {
                d.set($z(k), v);
            } else {
                d.set(k, v);
            }
        } else {
            throw 'put-not-writable-' + (typeof d);
        }
    }

    // getinterval operator
    //	s : src
    //	o : offset
    //	l : length
    function $geti(s, o, l) {
        if (s instanceof Uint8Array) {
            return s.subarray(o, o + l);
        }
        if (s instanceof Array) {
            var a = new Array(l);
            a.b = s.b; // base array
            a.o = s.o + o; // offset into base
            return a;
        }
        // Must be a string
        return s.substr(o, l);
    }

    // putinterval operator
    //	d : dst
    //	o : offset
    //	s : src
    function $puti(d, o, s) {
        if (d instanceof Uint8Array) {
            if (typeof s == 'string') {
                for (var i = 0, l = s.length; i < l; i++) {
                    d[o + i] = s.charCodeAt(i);
                }
            } else {
                // When both d and s are the same, we want to copy
                // backwards, which works for the general case as well.
                for (var i = s.length - 1; i >= 0; i--) {
                    d[o + i] = s[i];
                }
            }
        } else if (d instanceof Array) {
            // Operate on the base arrays
            var darr = d.b;
            var doff = o + d.o;
            var sarr = s.b;
            var soff = s.o;

            for (var i = 0, l = s.length; i < l; i++) {
                darr[doff + i] = sarr[soff + i];
            }
        } else {
            throw 'putinterval-not-writable-' + (typeof d);
        }
    }

    // type operator
    function $type(v) {
        // null can be mis-typed - get it out of the way
        if (v === null || v === undefined) {
            return 'nulltype';
        }
        var t = typeof v;
        if (t == 'number') {
            return v % 1 ? 'realtype' : 'integertype';
        }
        if (t == 'boolean') {
            return 'booleantype';
        }
        if (t == 'string' || v instanceof Uint8Array) {
            return 'stringtype';
        }
        if (t == 'function') {
            return 'operatortype';
        }
        if (v instanceof Array) {
            return 'arraytype';
        }
        return 'dicttype';
        // filetype
        // fonttype
        // gstatetype
        // marktype	(v === Infinity)
        // nametype
        // savetype
    }

    // search operator
    //		string seek search suffix match prefix true %if-found
    //						   string false				%if-not-found
    function $search(str, seek) {
        if (!(str instanceof Uint8Array)) {
            str = $s(str);
        }
        var ls = str.length;

        // Virtually all uses of search in BWIPP are for single-characters.
        // Optimize for that case.
        if (seek.length == 1) {
            var lk = 1;
            var cd = seek instanceof Uint8Array ? seek[0] : seek.charCodeAt(0);
            for (var i = 0; i < ls && str[i] != cd; i++);
        } else {
            // Slow path, 
            if (!(seek instanceof Uint8Array)) {
                seek = $(seek);
            }
            var lk = seek.length;
            var cd = seek[0];
            for (var i = 0; i < ls && str[i] != cd; i++);
            while (i < ls) {
                for (var j = 1; j < lk && str[i + j] === seek[j]; j++);
                if (j === lk) {
                    break;
                }
                for (i++; i < ls && str[i] != cd; i++);
            }
        }
        if (i < ls) {
            $k[$j++] = str.subarray(i + lk);
            $k[$j++] = str.subarray(i, i + lk);
            $k[$j++] = str.subarray(0, i);
            $k[$j++] = true;
        } else {
            $k[$j++] = str;
            $k[$j++] = false;
        }
    }

    // The callback is omitted when forall is being used just to push onto the
    // stack.  The callback normally returns undefined.  A return of true means break.
    function $forall(o, cb) {
        if (o instanceof Uint8Array) {
            for (var i = 0, l = o.length; i < l; i++) {
                $k[$j++] = o[i];
                if (cb && cb()) break;
            }
        } else if (o instanceof Array) {
            // The array may be a view.
            for (var a = o.b, i = o.o, l = o.o + o.length; i < l; i++) {
                $k[$j++] = a[i];
                if (cb && cb()) break;
            }
        } else if (typeof o === 'string') {
            for (var i = 0, l = o.length; i < l; i++) {
                $k[$j++] = o.charCodeAt(i);
                if (cb && cb()) break;
            }
        } else if (o instanceof Map) {
            for (var keys = o.keys(), i = 0, l = o.size; i < l; i++) {
                var id = keys.next().value;
                $k[$j++] = id;
                $k[$j++] = o.get(id);
                if (cb && cb()) break;
            }
        } else {
            for (var id in o) {
                $k[$j++] = id;
                $k[$j++] = o[id];
                if (cb && cb()) break;
            }
        }
    }

    function $cleartomark() {
        while ($j > 0 && $k[--$j] !== Infinity);
    }

    function $counttomark() {
        for (var i = $j - 1; i >= 0 && $k[i] !== Infinity; i--);
        return $j - i - 1;
    }

    function $aload(a) {
        for (var i = 0, l = a.length, b = a.b, o = a.o; i < l; i++) {
            $k[$j++] = b[o + i];
        }
        // This push has been optimized out.  See $.aload() in psc.js.
        //$k[$j++] = a;
    }

    function $astore(a) {
        for (var i = 0, l = a.length, b = a.b, o = a.o + l - 1; i < l; i++) {
            b[o - i] = $k[--$j];
        }
        $k[$j++] = a;
    }

    function $eq(a, b) {
        if (typeof a === 'string' && typeof b === 'string') {
            return a == b;
        }
        if (a instanceof Uint8Array && b instanceof Uint8Array) {
            if (a.length != b.length) {
                return false;
            }
            for (var i = 0, l = a.length; i < l; i++) {
                if (a[i] != b[i]) {
                    return false;
                }
            }
            return true;
        }
        if (a instanceof Uint8Array && typeof b === 'string' ||
            b instanceof Uint8Array && typeof a === 'string') {
            if (a instanceof Uint8Array) {
                a = $z(a);
            } else {
                b = $z(b);
            }
            return a == b;
        }
        return a == b;
    }

    function $ne(a, b) {
        return !$eq(a, b);
    }

    function $lt(a, b) {
        if (a instanceof Uint8Array) {
            a = $z(a);
        }
        if (b instanceof Uint8Array) {
            b = $z(b);
        }
        return a < b;
    }

    function $le(a, b) {
        if (a instanceof Uint8Array) {
            a = $z(a);
        }
        if (b instanceof Uint8Array) {
            b = $z(b);
        }
        return a <= b;
    }

    function $gt(a, b) {
        if (a instanceof Uint8Array) {
            a = $z(a);
        }
        if (b instanceof Uint8Array) {
            b = $z(b);
        }
        return a > b;
    }

    function $ge(a, b) {
        if (a instanceof Uint8Array) {
            a = $z(a);
        }
        if (b instanceof Uint8Array) {
            b = $z(b);
        }
        return a >= b;
    }

    function $an(a, b) { // and
        return (typeof a === 'boolean') ? a && b : a & b;
    }

    function $or(a, b) { // or
        return (typeof a === 'boolean') ? a || b : a | b;
    }

    function $xo(a, b) { // xor
        return (typeof a === 'boolean') ? !a && b || a && !b : a ^ b;
    }

    function $nt(a) {
        return typeof a == 'boolean' ? !a : ~a;
    }
    // emulate single-precision floating-point (pseudo-polyfill for Math.fround)
    var $f = (function(fa) {
        return function(v) {
            return Number.isInteger(v) ? v : (fa[0] = v, fa[0]);
        };
    })(new Float32Array(1));

    // DEBUG-BEGIN
    function $stack() {
        console.log('[[[');
        for (var i = $j - 1; i >= 0; i--) {
            console.log(tostring($k[i]));
        }
        console.log(']]]');

        function tostring(v) {
            // null can be mis-typed - get it out of the way
            if (v === null) {
                return 'null';
            } else if (v === undefined) {
                return '<undefined>';
            } else if (v instanceof Array) {
                var s = '<array,' + v.o + ',' + v.length + '>[';
                for (var j = v.o, a = v.b, l = v.length + v.o; j < l; j++) {
                    s += (j == v.o ? '' : ',') + tostring(a[j]);
                }
                return s + ']';
            } else if (v instanceof Uint8Array) {
                return '(' + $z[v] + ')';
            } else if (typeof v === 'object') {
                var s = '<<';
                for (var id in v) {
                    s += (s.length == 7 ? '' : ',') + id + ':' + tostring(v[id]);
                }
                return s + '>>';
            } else if (typeof v === 'string') {
                return '"' + v + '"';
            } else {
                return '' + v;
            }
        }
    }
    // DEBUG-END
    $0.raiseerror = function() {
        $put($0.$error, 'errorinfo', $k[--$j]); //#55
        $put($0.$error, 'errorname', $k[--$j]); //#56
        $put($0.$error, 'command', null); //#57
        $put($0.$error, 'newerror', true); //#58
        throw new Error($z($0.$error.get("errorname")) + ": " + $z($0.$error.get("errorinfo"))); //#59
    };
    $0.parseinput = function() {
        var $1 = {}; //#80
        $1.fncvals = $k[--$j]; //#82
        $1.barcode = $k[--$j]; //#83
        var _2 = 'parse'; //#85
        $1[_2] = $get($1.fncvals, _2); //#85
        delete $1.fncvals[_2]; //#85
        var _6 = 'parsefnc'; //#86
        $1[_6] = $get($1.fncvals, _6); //#86
        delete $1.fncvals[_6]; //#86
        var _A = 'parseonly'; //#87
        var _C = $get($1.fncvals, _A) !== undefined; //#87
        $1[_A] = _C; //#87
        delete $1.fncvals[_A]; //#87
        var _E = 'eci'; //#88
        var _G = $get($1.fncvals, _E) !== undefined; //#88
        $1[_E] = _G; //#88
        delete $1.fncvals[_E]; //#88
        $1.msg = $a($1.barcode.length); //#90
        $1.j = 0; //#91
        $k[$j++] = $1.barcode; //#186
        for (;;) { //#186
            $search($k[--$j], "^"); //#93
            var _M = $k[--$j]; //#93
            var _N = $k[--$j]; //#93
            $k[$j++] = _M; //#96
            $k[$j++] = _N.length; //#96
            $k[$j++] = $1.msg; //#96
            $k[$j++] = $1.j; //#96
            $k[$j++] = _N; //#96
            $k[$j++] = Infinity; //#96
            var _Q = $k[--$j]; //#96
            var _R = $k[--$j]; //#96
            $k[$j++] = _Q; //#96
            $forall(_R); //#96
            var _S = $a(); //#96
            var _T = $k[--$j]; //#96
            $puti($k[--$j], _T, _S); //#96
            $1.j = $f($k[--$j] + $1.j); //#97
            if ($k[--$j]) { //#184
                $j--; //#99
                for (var _Y = 0, _Z = 1; _Y < _Z; _Y++) { //#182
                    if ($an($nt($1.parse), $nt($1.parsefnc))) { //#106
                        $put($1.msg, $1.j, 94); //#103
                        $1.j = $f($1.j + 1); //#104
                        break; //#105
                    } //#105
                    $put($1.msg, $1.j, 94); //#109
                    $1.j = $f($1.j + 1); //#110
                    if ($1.parse) { //#129
                        var _j = $k[--$j]; //#114
                        $k[$j++] = _j; //#128
                        if (_j.length >= 3) { //#128
                            var _k = $k[--$j]; //#115
                            var _l = $geti(_k, 0, 3); //#115
                            $k[$j++] = _k; //#117
                            $k[$j++] = true; //#117
                            for (var _m = 0, _n = _l.length; _m < _n; _m++) { //#117
                                var _o = $get(_l, _m); //#117
                                if ((_o < 48) || (_o > 57)) { //#116
                                    $j--; //#116
                                    $k[$j++] = false; //#116
                                } //#116
                            } //#116
                            if ($k[--$j]) { //#127
                                var _q = $k[--$j]; //#119
                                var _s = ~~$z($geti(_q, 0, 3)); //#119
                                $k[$j++] = _q; //#122
                                $k[$j++] = _s; //#122
                                if (_s > 255) { //#122
                                    $j -= 2; //#120
                                    $k[$j++] = 'bwipp.invalidOrdinal'; //#121
                                    $k[$j++] = "Ordinal must be 000 to 255"; //#121
                                    $0.raiseerror(); //#121
                                } //#121
                                $1.j = $f($1.j - 1); //#123
                                $put($1.msg, $1.j, $k[--$j]); //#124
                                $1.j = $f($1.j + 1); //#125
                                var _y = $k[--$j]; //#126
                                $k[$j++] = $geti(_y, 3, _y.length - 3); //#126
                            } //#126
                        } //#126
                    } //#126
                    if (($or($1.parseonly, $nt($1.parsefnc))) || ($get($1.msg, $f($1.j - 1)) != 94)) { //#134
                        break; //#134
                    } //#134
                    $1.j = $f($1.j - 1); //#137
                    var _16 = $k[--$j]; //#138
                    $k[$j++] = _16; //#141
                    if (_16.length < 3) { //#141
                        $j--; //#139
                        $k[$j++] = 'bwipp.truncatedFNC'; //#140
                        $k[$j++] = "Function character truncated"; //#140
                        $0.raiseerror(); //#140
                    } //#140
                    var _17 = $k[--$j]; //#142
                    $k[$j++] = _17; //#147
                    if ($get(_17, 0) == 94) { //#147
                        $put($1.msg, $1.j, 94); //#143
                        $1.j = $f($1.j + 1); //#144
                        var _1C = $k[--$j]; //#145
                        $k[$j++] = $geti(_1C, 1, _1C.length - 1); //#146
                        break; //#146
                    } //#146
                    var _1E = $k[--$j]; //#148
                    $k[$j++] = _1E; //#165
                    if ($eq($geti(_1E, 0, 3), "ECI") && $1.eci) { //#165
                        var _1H = $k[--$j]; //#149
                        $k[$j++] = _1H; //#152
                        if (_1H.length < 9) { //#152
                            $j--; //#150
                            $k[$j++] = 'bwipp.truncatedECI'; //#151
                            $k[$j++] = "ECI truncated"; //#151
                            $0.raiseerror(); //#151
                        } //#151
                        var _1I = $k[--$j]; //#153
                        var _1J = $geti(_1I, 3, 6); //#153
                        $k[$j++] = _1I; //#159
                        $k[$j++] = _1J; //#159
                        for (var _1K = 0, _1L = _1J.length; _1K < _1L; _1K++) { //#159
                            var _1M = $get(_1J, _1K); //#159
                            if ((_1M < 48) || (_1M > 57)) { //#158
                                $j -= 2; //#156
                                $k[$j++] = 'bwipp.invalidECI'; //#157
                                $k[$j++] = "ECI must be 000000 to 999999"; //#157
                                $0.raiseerror(); //#157
                            } //#157
                        } //#157
                        var _1N = $k[--$j]; //#160
                        $k[$j++] = 0; //#160
                        $forall(_1N, function() { //#160
                            var _1O = $k[--$j]; //#160
                            var _1P = $k[--$j]; //#160
                            $k[$j++] = ($f(_1P - $f(_1O - 48))) * 10; //#160
                        }); //#160
                        $put($1.msg, $1.j, (~~($k[--$j] / 10)) - 1000000); //#161
                        $1.j = $f($1.j + 1); //#162
                        var _1U = $k[--$j]; //#163
                        $k[$j++] = $geti(_1U, 9, _1U.length - 9); //#164
                        break; //#164
                    } //#164
                    var _1W = $k[--$j]; //#166
                    $k[$j++] = _1W; //#169
                    if (_1W.length < 4) { //#169
                        $j--; //#167
                        $k[$j++] = 'bwipp.truncatedFNC'; //#168
                        $k[$j++] = "Function character truncated"; //#168
                        $0.raiseerror(); //#168
                    } //#168
                    var _1X = $k[--$j]; //#170
                    var _1Y = $geti(_1X, 0, 4); //#170
                    var _1a = $get($1.fncvals, _1Y) !== undefined; //#170
                    $k[$j++] = _1X; //#175
                    $k[$j++] = _1Y; //#175
                    if (!_1a) { //#175
                        var _1b = $k[--$j]; //#171
                        var _1c = $s(_1b.length + 28); //#171
                        $puti(_1c, 28, _1b); //#171
                        $puti(_1c, 0, "Unknown function character: "); //#172
                        var _1d = $k[--$j]; //#173
                        $k[$j++] = _1c; //#173
                        $k[$j++] = _1d; //#173
                        $j--; //#173
                        var _1e = $k[--$j]; //#174
                        $k[$j++] = 'bwipp.unknownFNC'; //#174
                        $k[$j++] = _1e; //#174
                        $0.raiseerror(); //#174
                    } //#174
                    $put($1.msg, $1.j, $get($1.fncvals, $k[--$j])); //#177
                    $1.j = $f($1.j + 1); //#178
                    var _1l = $k[--$j]; //#179
                    $k[$j++] = $geti(_1l, 4, _1l.length - 4); //#180
                    break; //#180
                } //#180
            } else { //#184
                break; //#184
            } //#184
        } //#184
        if ($nt($1.parseonly)) { //#192
            $k[$j++] = $geti($1.msg, 0, $1.j); //#189
        } else { //#192
            $k[$j++] = $s($1.j); //#192
            for (var _1v = 0, _1u = $f($1.j - 1); _1v <= _1u; _1v += 1) { //#192
                var _1w = $k[--$j]; //#192
                $put(_1w, _1v, $get($1.msg, _1v)); //#192
                $k[$j++] = _1w; //#192
            } //#192
        } //#192
    };
    $0.gs1lint = function() {
        var $1 = {}; //#217
        $1.vals = $k[--$j]; //#219
        $1.ais = $k[--$j]; //#220
        $1.lintnumeric = function() {
            var _2 = $k[--$j]; //#223
            $k[$j++] = true; //#225
            $forall(_2, function() { //#225
                var _3 = $k[--$j]; //#224
                if ((_3 < 48) || (_3 > 57)) { //#224
                    $j--; //#224
                    $k[$j++] = false; //#224
                    return true; //#224
                } //#224
            }); //#224
            if ($nt($k[--$j])) { //#226
                $j--; //#226
                $k[$j++] = 'bwipp.GS1notNumeric'; //#226
                $k[$j++] = "Not numeric"; //#226
                $k[$j++] = false; //#226
                return true; //#226
            } //#226
        }; //#227
        $1.lintcset82 = function() {
            var _5 = $k[--$j]; //#230
            $k[$j++] = true; //#235
            $forall(_5, function() { //#235
                var _6 = $s(1); //#231
                $put(_6, 0, $k[--$j]); //#231
                $search("!\"%&'()*+,-./0123456789:;<=>?ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz", _6); //#233
                if ($nt($k[--$j])) { //#233
                    $j -= 2; //#233
                    $k[$j++] = false; //#233
                    return true; //#233
                } //#233
                $j -= 3; //#234
            }); //#234
            if ($nt($k[--$j])) { //#236
                $j--; //#236
                $k[$j++] = 'bwipp.GS1badCSET82character'; //#236
                $k[$j++] = "Invalid CSET 82 character"; //#236
                $k[$j++] = false; //#236
                return true; //#236
            } //#236
        }; //#237
        $1.lintcset39 = function() {
            var _A = $k[--$j]; //#240
            $k[$j++] = true; //#245
            $forall(_A, function() { //#245
                var _B = $s(1); //#241
                $put(_B, 0, $k[--$j]); //#241
                $search("#-/0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", _B); //#243
                if ($nt($k[--$j])) { //#243
                    $j -= 2; //#243
                    $k[$j++] = false; //#243
                    return true; //#243
                } //#243
                $j -= 3; //#244
            }); //#244
            if ($nt($k[--$j])) { //#246
                $j--; //#246
                $k[$j++] = 'bwipp.GS1badCSET39character'; //#246
                $k[$j++] = "Invalid CSET 39 character"; //#246
                $k[$j++] = false; //#246
                return true; //#246
            } //#246
        }; //#247
        $1.lintkey = function() {
            var _F = $k[--$j]; //#250
            $k[$j++] = _F; //#250
            if (_F.length < 2) { //#250
                $j -= 2; //#250
                $k[$j++] = 'bwipp.GS1keyTooShort'; //#250
                $k[$j++] = "Key is too short"; //#250
                $k[$j++] = false; //#250
                return true; //#250
            } //#250
            var _G = $k[--$j]; //#251
            var _H = $get(_G, 0); //#251
            var _I = $get(_G, 1); //#252
            if (((_H < 48) || (_H > 57)) || ((_I < 48) || (_I > 57))) { //#254
                $j--; //#253
                $k[$j++] = 'bwipp.GS1badGCP'; //#253
                $k[$j++] = "Non-numeric company prefix"; //#253
                $k[$j++] = false; //#253
                return true; //#253
            } //#253
        }; //#255
        $1.lintimporteridx = function() {
            $search("-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz", $k[--$j]); //#259
            if ($nt($k[--$j])) { //#259
                $j -= 2; //#259
                $k[$j++] = 'bwipp.GS1badImporterIndex'; //#259
                $k[$j++] = "Invalid importer index"; //#259
                $k[$j++] = false; //#259
                return true; //#259
            } //#259
            $j -= 3; //#260
        }; //#261
        $1.lintcsum = function() {
            $k[$j++] = Infinity; //#264
            var _L = $k[--$j]; //#264
            var _M = $k[--$j]; //#264
            var _N = ((_M.length % 2) == 0) ? 3 : 1; //#265
            $k[$j++] = _L; //#266
            $k[$j++] = _N; //#266
            $forall(_M, function() { //#266
                var _O = $k[--$j]; //#266
                var _P = $k[--$j]; //#266
                $k[$j++] = $f(_O - 48) * _P; //#266
                $k[$j++] = $f(4 - _P); //#266
            }); //#266
            $j--; //#266
            var _Q = $counttomark() + 1; //#267
            $k[$j++] = 0; //#267
            for (var _R = 0, _S = _Q - 1; _R < _S; _R++) { //#267
                var _T = $k[--$j]; //#267
                var _U = $k[--$j]; //#267
                $k[$j++] = $f(_U + _T); //#267
            } //#267
            var _V = $k[--$j]; //#267
            var _W = $k[--$j]; //#267
            $k[$j++] = _V; //#267
            $k[$j++] = _W; //#267
            $j--; //#267
            if (($k[--$j] % 10) != 0) { //#268
                $j--; //#268
                $k[$j++] = 'bwipp.GS1badChecksum'; //#268
                $k[$j++] = "Bad checksum"; //#268
                $k[$j++] = false; //#268
                return true; //#268
            } //#268
        }; //#269
        $k[$j++] = Infinity; //#273
        $k[$j++] = 0; //#273
        for (var _Y = 0, _Z = "!\"%&'()*+,-./0123456789:;<=>?ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz".length; _Y < _Z; _Y++) { //#273
            var _b = $k[--$j]; //#273
            $k[$j++] = $get("!\"%&'()*+,-./0123456789:;<=>?ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz", _Y); //#273
            $k[$j++] = _b; //#273
            $k[$j++] = $f(_b + 1); //#273
        } //#273
        $j--; //#273
        $1.cset82 = $d(); //#274
        $k[$j++] = Infinity; //#278
        $k[$j++] = 0; //#278
        for (var _d = 0, _e = "23456789ABCDEFGHJKLMNPQRSTUVWXYZ".length; _d < _e; _d++) { //#278
            var _g = $k[--$j]; //#278
            $k[$j++] = $get("23456789ABCDEFGHJKLMNPQRSTUVWXYZ", _d); //#278
            $k[$j++] = _g; //#278
            $k[$j++] = $f(_g + 1); //#278
        } //#278
        $j--; //#278
        $1.cset32 = $d(); //#279
        $1.lintcsumalpha = function() {
            var _i = $k[--$j]; //#282
            $k[$j++] = _i; //#282
            if (_i.length < 2) { //#282
                $j -= 2; //#282
                $k[$j++] = 'bwipp.GS1alphaTooShort'; //#282
                $k[$j++] = "Alphanumeric string is too short to check"; //#282
                $k[$j++] = false; //#282
                return true; //#282
            } //#282
            var _j = $k[--$j]; //#283
            $k[$j++] = _j; //#285
            $k[$j++] = _j.length - 2; //#285
            $k[$j++] = $geti(_j, 0, _j.length - 2); //#285
            $k[$j++] = Infinity; //#285
            var _l = $k[--$j]; //#285
            var _m = $k[--$j]; //#285
            $k[$j++] = _l; //#287
            $forall(_m, function() { //#287
                var _n = $k[--$j]; //#286
                var _p = $get($1.cset82, _n) !== undefined; //#286
                $k[$j++] = _n; //#286
                if (_p) { //#286
                    var _s = $get($1.cset82, $k[--$j]); //#286
                    $k[$j++] = _s; //#286
                } else { //#286
                    $k[$j++] = -1; //#286
                    return true; //#286
                } //#286
            }); //#286
            var _t = $k[--$j]; //#288
            $k[$j++] = _t; //#288
            if (_t == -1) { //#288
                $cleartomark(); //#288
                $j -= 3; //#288
                $k[$j++] = 'bwipp.GS1UnknownCSET82Character'; //#288
                $k[$j++] = "Unknown CSET 82 character"; //#288
                $k[$j++] = false; //#288
                return true; //#288
            } //#288
            $astore($a($counttomark())); //#289
            var _w = $k[--$j]; //#289
            var _x = $k[--$j]; //#289
            $k[$j++] = _w; //#289
            $k[$j++] = _x; //#289
            $j--; //#289
            var _y = $k[--$j]; //#290
            var _z = $k[--$j]; //#290
            var _11 = $geti($k[--$j], _z, 2); //#291
            $k[$j++] = _y; //#291
            $k[$j++] = _11; //#291
            $k[$j++] = Infinity; //#291
            var _12 = $k[--$j]; //#291
            var _13 = $k[--$j]; //#291
            $k[$j++] = _12; //#293
            $forall(_13, function() { //#293
                var _14 = $k[--$j]; //#292
                var _16 = $get($1.cset32, _14) !== undefined; //#292
                $k[$j++] = _14; //#292
                if (_16) { //#292
                    var _19 = $get($1.cset32, $k[--$j]); //#292
                    $k[$j++] = _19; //#292
                } else { //#292
                    $k[$j++] = -1; //#292
                    return true; //#292
                } //#292
            }); //#292
            var _1A = $k[--$j]; //#294
            $k[$j++] = _1A; //#294
            if (_1A == -1) { //#294
                $cleartomark(); //#294
                $j -= 2; //#294
                $k[$j++] = 'bwipp.GS1UnknownCSET32Character'; //#294
                $k[$j++] = "Unknown CSET 32 character"; //#294
                $k[$j++] = false; //#294
                return true; //#294
            } //#294
            $astore($a($counttomark())); //#295
            var _1D = $k[--$j]; //#295
            var _1E = $k[--$j]; //#295
            $k[$j++] = _1D; //#295
            $k[$j++] = _1E; //#295
            $j--; //#295
            var _1F = $k[--$j]; //#296
            var _1I = $k[--$j]; //#296
            var _1J = $a([2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83]); //#297
            var _1K = _1I.length; //#299
            $k[$j++] = $f(($get(_1F, 0) << 5) + $get(_1F, 1)); //#299
            $k[$j++] = _1I; //#299
            $k[$j++] = _1J; //#299
            $k[$j++] = _1K; //#299
            if (_1K > _1J.length) { //#299
                $j -= 5; //#299
                $k[$j++] = 'bwipp.GS1alphaTooLong'; //#299
                $k[$j++] = "Alphanumeric string is too long to check"; //#299
                $k[$j++] = false; //#299
                return true; //#299
            } //#299
            var _1L = $k[--$j]; //#300
            var _1N = $geti($k[--$j], 0, _1L); //#300
            for (var _1O = 0, _1P = _1N.length; _1O < _1P; _1O++) { //#300
                var _1R = $k[--$j]; //#300
                $k[$j++] = $get(_1N, _1O); //#300
                $k[$j++] = _1R; //#300
            } //#300
            var _1S = $k[--$j]; //#301
            $k[$j++] = 0; //#301
            $forall(_1S, function() { //#301
                var _1T = $k[--$j]; //#301
                var _1U = $k[--$j]; //#301
                var _1V = $k[--$j]; //#301
                $k[$j++] = $f(_1U + (_1V * _1T)); //#301
            }); //#301
            var _1W = $k[--$j]; //#301
            if ($k[--$j] != (_1W % 1021)) { //#302
                $j--; //#302
                $k[$j++] = 'bwipp.GS1badAlphaCheckCharacters'; //#302
                $k[$j++] = "Bad alphanumeric check characters"; //#302
                $k[$j++] = false; //#302
                return true; //#302
            } //#302
        }; //#303
        $k[$j++] = Infinity; //#324
        var _1Y = $a(['004', '008', '010', '012', '016', '020', '024', '028', '031', '032', '036', '040', '044', '048', '050', '051', '052', '056', '060', '064', '068', '070', '072', '074', '076', '084', '086', '090', '092', '096', '100', '104', '108', '112', '116', '120', '124', '132', '136', '140', '144', '148', '152', '156', '158', '162', '166', '170', '174', '175', '178', '180', '184', '188', '191', '192', '196', '203', '204', '208', '212', '214', '218', '222', '226', '231', '232', '233', '234', '238', '239', '242', '246', '248', '250', '254', '258', '260', '262', '266', '268', '270', '275', '276', '288', '292', '296', '300', '304', '308', '312', '316', '320', '324', '328', '332', '334', '336', '340', '344', '348', '352', '356', '360', '364', '368', '372', '376', '380', '384', '388', '392', '398', '400', '404', '408', '410', '414', '417', '418', '422', '426', '428', '430', '434', '438', '440', '442', '446', '450', '454', '458', '462', '466', '470', '474', '478', '480', '484', '492', '496', '498', '499', '500', '504', '508', '512', '516', '520', '524', '528', '531', '533', '534', '535', '540', '548', '554', '558', '562', '566', '570', '574', '578', '580', '581', '583', '584', '585', '586', '591', '598', '600', '604', '608', '612', '616', '620', '624', '626', '630', '634', '638', '642', '643', '646', '652', '654', '659', '660', '662', '663', '666', '670', '674', '678', '682', '686', '688', '690', '694', '702', '703', '704', '705', '706', '710', '716', '724', '728', '729', '732', '740', '744', '748', '752', '756', '760', '762', '764', '768', '772', '776', '780', '784', '788', '792', '795', '796', '798', '800', '804', '807', '818', '826', '831', '832', '833', '834', '840', '850', '854', '858', '860', '862', '876', '882', '887', '894']); //#323
        for (var _1Z = 0, _1a = _1Y.length; _1Z < _1a; _1Z++) { //#324
            var _1b = $get(_1Y, _1Z); //#324
            $k[$j++] = _1b; //#324
            $k[$j++] = _1b; //#324
        } //#324
        $1.iso3166 = $d(); //#324
        $1.lintiso3166 = function() {
            var _1f = $get($1.iso3166, $k[--$j]) !== undefined; //#327
            if (!_1f) { //#327
                $j--; //#327
                $k[$j++] = 'bwipp.GS1UnknownCountry'; //#327
                $k[$j++] = "Unknown country code"; //#327
                $k[$j++] = false; //#327
                return true; //#327
            } //#327
        }; //#328
        $1.lintiso3166999 = function() {
            var _1g = $k[--$j]; //#331
            $k[$j++] = _1g; //#335
            if ($ne(_1g, '999')) { //#334
                var _1j = $get($1.iso3166, $k[--$j]) !== undefined; //#332
                if (!_1j) { //#332
                    $j--; //#332
                    $k[$j++] = 'bwipp.GS1UnknownCountryOr999'; //#332
                    $k[$j++] = "Unknown country code or not 999"; //#332
                    $k[$j++] = false; //#332
                    return true; //#332
                } //#332
            } else { //#334
                $j--; //#334
            } //#334
        }; //#336
        $1.lintiso3166list = function() {
            var _1k = $k[--$j]; //#339
            $k[$j++] = _1k; //#341
            if ((_1k.length % 3) != 0) { //#341
                $j -= 2; //#340
                $k[$j++] = 'bwipp.GS1BadCountryListLength'; //#340
                $k[$j++] = "Not a group of three-digit country codes"; //#340
                $k[$j++] = false; //#340
                return true; //#340
            } //#340
            var _1l = $k[--$j]; //#343
            $k[$j++] = _1l; //#346
            $k[$j++] = true; //#346
            for (var _1n = 0, _1m = _1l.length - 1; _1n <= _1m; _1n += 3) { //#346
                var _1o = $k[--$j]; //#344
                var _1p = $k[--$j]; //#344
                var _1s = $get($1.iso3166, $geti(_1p, _1n, 3)) !== undefined; //#345
                $k[$j++] = _1p; //#345
                $k[$j++] = _1o; //#345
                if (!_1s) { //#345
                    $j -= 2; //#345
                    $k[$j++] = false; //#345
                    break; //#345
                } //#345
            } //#345
            if ($nt($k[--$j])) { //#347
                $j--; //#347
                $k[$j++] = 'bwipp.GS1UnknownCountry'; //#347
                $k[$j++] = "Unknown country code"; //#347
                $k[$j++] = false; //#347
                return true; //#347
            } //#347
            $j--; //#348
        }; //#349
        $k[$j++] = Infinity; //#376
        var _1u = $a(['AD', 'AE', 'AF', 'AG', 'AI', 'AL', 'AM', 'AO', 'AQ', 'AR', 'AS', 'AT', 'AU', 'AW', 'AX', 'AZ', 'BA', 'BB', 'BD', 'BE', 'BF', 'BG', 'BH', 'BI', 'BJ', 'BL', 'BM', 'BN', 'BO', 'BQ', 'BR', 'BS', 'BT', 'BV', 'BW', 'BY', 'BZ', 'CA', 'CC', 'CD', 'CF', 'CG', 'CH', 'CI', 'CK', 'CL', 'CM', 'CN', 'CO', 'CR', 'CU', 'CV', 'CW', 'CX', 'CY', 'CZ', 'DE', 'DJ', 'DK', 'DM', 'DO', 'DZ', 'EC', 'EE', 'EG', 'EH', 'ER', 'ES', 'ET', 'FI', 'FJ', 'FK', 'FM', 'FO', 'FR', 'GA', 'GB', 'GD', 'GE', 'GF', 'GG', 'GH', 'GI', 'GL', 'GM', 'GN', 'GP', 'GQ', 'GR', 'GS', 'GT', 'GU', 'GW', 'GY', 'HK', 'HM', 'HN', 'HR', 'HT', 'HU', 'ID', 'IE', 'IL', 'IM', 'IN', 'IO', 'IQ', 'IR', 'IS', 'IT', 'JE', 'JM', 'JO', 'JP', 'KE', 'KG', 'KH', 'KI', 'KM', 'KN', 'KP', 'KR', 'KW', 'KY', 'KZ', 'LA', 'LB', 'LC', 'LI', 'LK', 'LR', 'LS', 'LT', 'LU', 'LV', 'LY', 'MA', 'MC', 'MD', 'ME', 'MF', 'MG', 'MH', 'MK', 'ML', 'MM', 'MN', 'MO', 'MP', 'MQ', 'MR', 'MS', 'MT', 'MU', 'MV', 'MW', 'MX', 'MY', 'MZ', 'NA', 'NC', 'NE', 'NF', 'NG', 'NI', 'NL', 'NO', 'NP', 'NR', 'NU', 'NZ', 'OM', 'PA', 'PE', 'PF', 'PG', 'PH', 'PK', 'PL', 'PM', 'PN', 'PR', 'PS', 'PT', 'PW', 'PY', 'QA', 'RE', 'RO', 'RS', 'RU', 'RW', 'SA', 'SB', 'SC', 'SD', 'SE', 'SG', 'SH', 'SI', 'SJ', 'SK', 'SL', 'SM', 'SN', 'SO', 'SR', 'SS', 'ST', 'SV', 'SX', 'SY', 'SZ', 'TC', 'TD', 'TF', 'TG', 'TH', 'TJ', 'TK', 'TL', 'TM', 'TN', 'TO', 'TR', 'TT', 'TV', 'TW', 'TZ', 'UA', 'UG', 'UM', 'US', 'UY', 'UZ', 'VA', 'VC', 'VE', 'VG', 'VI', 'VN', 'VU', 'WF', 'WS', 'YE', 'YT', 'ZA', 'ZM', 'ZW']); //#375
        for (var _1v = 0, _1w = _1u.length; _1v < _1w; _1v++) { //#376
            var _1x = $get(_1u, _1v); //#376
            $k[$j++] = _1x; //#376
            $k[$j++] = _1x; //#376
        } //#376
        $1.iso3166alpha2 = $d(); //#376
        $1.lintiso3166alpha2 = function() {
            var _21 = $get($1.iso3166alpha2, $k[--$j]) !== undefined; //#379
            if (!_21) { //#379
                $j--; //#379
                $k[$j++] = 'bwipp.GS1UnknownCountryAlpha'; //#379
                $k[$j++] = "Unknown country alpha code"; //#379
                $k[$j++] = false; //#379
                return true; //#379
            } //#379
        }; //#380
        $k[$j++] = Infinity; //#405
        var _22 = $a(['008', '012', '032', '036', '044', '048', '050', '051', '052', '060', '064', '068', '072', '084', '090', '096', '104', '108', '116', '124', '132', '136', '144', '152', '156', '170', '174', '188', '191', '192', '203', '208', '214', '222', '230', '232', '238', '242', '262', '270', '292', '320', '324', '328', '332', '340', '344', '348', '352', '356', '360', '364', '368', '376', '388', '392', '398', '400', '404', '408', '410', '414', '417', '418', '422', '426', '430', '434', '446', '454', '458', '462', '480', '484', '496', '498', '504', '512', '516', '524', '532', '533', '548', '554', '558', '566', '578', '586', '590', '598', '600', '604', '608', '634', '643', '646', '654', '682', '690', '694', '702', '704', '706', '710', '728', '748', '752', '756', '760', '764', '776', '780', '784', '788', '800', '807', '818', '826', '834', '840', '858', '860', '882', '886', '901', '927', '928', '929', '930', '931', '932', '933', '934', '936', '938', '940', '941', '943', '944', '946', '947', '948', '949', '950', '951', '952', '953', '955', '956', '957', '958', '959', '960', '961', '962', '963', '964', '965', '967', '968', '969', '970', '971', '972', '973', '975', '976', '977', '978', '979', '980', '981', '984', '985', '986', '990', '994', '997', '999']); //#404
        for (var _23 = 0, _24 = _22.length; _23 < _24; _23++) { //#405
            var _25 = $get(_22, _23); //#405
            $k[$j++] = _25; //#405
            $k[$j++] = _25; //#405
        } //#405
        $1.iso4217 = $d(); //#405
        $1.lintiso4217 = function() {
            var _29 = $get($1.iso4217, $k[--$j]) !== undefined; //#408
            if (!_29) { //#408
                $j--; //#408
                $k[$j++] = 'bwipp.GS1UnknownCurrency'; //#408
                $k[$j++] = "Unknown currency code"; //#408
                $k[$j++] = false; //#408
                return true; //#408
            } //#408
        }; //#409
        $1.lintiban = function() {
            var _2A = $k[--$j]; //#412
            $k[$j++] = _2A; //#412
            if (_2A.length < 4) { //#412
                $j -= 2; //#412
                $k[$j++] = 'bwipp.GS1tooShort'; //#412
                $k[$j++] = "IBAN too short"; //#412
                $k[$j++] = false; //#412
                return true; //#412
            } //#412
            var _2B = $k[--$j]; //#413
            $k[$j++] = _2B; //#418
            $k[$j++] = true; //#418
            $forall(_2B, function() { //#418
                var _2C = $s(1); //#414
                $put(_2C, 0, $k[--$j]); //#414
                $search("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", _2C); //#416
                if ($nt($k[--$j])) { //#416
                    $j -= 2; //#416
                    $k[$j++] = false; //#416
                    return true; //#416
                } //#416
                $j -= 3; //#417
            }); //#417
            if ($nt($k[--$j])) { //#419
                $j--; //#419
                $k[$j++] = 'bwipp.GS1badIBANcharacter'; //#419
                $k[$j++] = "Invalid IBAN character"; //#419
                $k[$j++] = false; //#419
                return true; //#419
            } //#419
            var _2G = $k[--$j]; //#420
            $k[$j++] = _2G; //#420
            $k[$j++] = _2G; //#420
            $k[$j++] = Infinity; //#420
            var _2H = $k[--$j]; //#420
            var _2I = $k[--$j]; //#420
            $k[$j++] = _2H; //#420
            $k[$j++] = _2I; //#420
            $k[$j++] = Infinity; //#420
            var _2J = $k[--$j]; //#420
            var _2K = $k[--$j]; //#420
            $k[$j++] = _2J; //#420
            $forall(_2K); //#420
            $r($counttomark(), -4); //#420
            $astore($a($counttomark())); //#420
            var _2O = $k[--$j]; //#420
            var _2P = $k[--$j]; //#420
            $k[$j++] = _2O; //#420
            $k[$j++] = _2P; //#420
            $j--; //#420
            $forall($k[--$j], function() { //#422
                var _2S = $f($k[--$j] - 48); //#421
                $k[$j++] = _2S; //#421
                if (_2S > 9) { //#421
                    var _2U = $f($k[--$j] - 7); //#421
                    $k[$j++] = ~~(_2U / 10); //#421
                    $k[$j++] = _2U % 10; //#421
                } //#421
            }); //#421
            $astore($a($counttomark())); //#422
            var _2X = $k[--$j]; //#422
            var _2Y = $k[--$j]; //#422
            $k[$j++] = _2X; //#422
            $k[$j++] = _2Y; //#422
            $j--; //#422
            var _2Z = $k[--$j]; //#423
            $k[$j++] = 0; //#423
            $forall(_2Z, function() { //#423
                var _2a = $k[--$j]; //#423
                var _2b = $k[--$j]; //#423
                $k[$j++] = ($f(_2a + (_2b * 10))) % 97; //#423
            }); //#423
            if ($k[--$j] != 1) { //#424
                $j -= 2; //#424
                $k[$j++] = 'bwipp.GS1badIBANchecksum'; //#424
                $k[$j++] = "IBAN checksum incorrect"; //#424
                $k[$j++] = false; //#424
                return true; //#424
            } //#424
            var _2e = $geti($k[--$j], 0, 2); //#425
            $k[$j++] = _2e; //#425
            $1.lintiso3166alpha2(); //#425
        }; //#426
        $1.lintzero = function() {
            if ($ne($k[--$j], "0")) { //#429
                $j--; //#429
                $k[$j++] = 'bwipp.GS1zeroRequired'; //#429
                $k[$j++] = "Zero is required"; //#429
                $k[$j++] = false; //#429
                return true; //#429
            } //#429
        }; //#430
        $1.lintnonzero = function() {
            var _2g = $k[--$j]; //#433
            $k[$j++] = false; //#433
            $forall(_2g, function() { //#433
                if ($k[--$j] != 48) { //#433
                    $j--; //#433
                    $k[$j++] = true; //#433
                } //#433
            }); //#433
            if ($nt($k[--$j])) { //#434
                $j--; //#434
                $k[$j++] = 'bwipp.GS1zeroNotPermitted'; //#434
                $k[$j++] = "Zero not permitted"; //#434
                $k[$j++] = false; //#434
                return true; //#434
            } //#434
        }; //#435
        $1.lintnozeroprefix = function() {
            var _2j = $k[--$j]; //#438
            if ((_2j.length > 1) && ($get(_2j, 0) == 48)) { //#440
                $j--; //#439
                $k[$j++] = 'bwipp.GS1badZeroPrefix'; //#439
                $k[$j++] = "Zero prefix is not permitted"; //#439
                $k[$j++] = false; //#439
                return true; //#439
            } //#439
        }; //#441
        $1.lintyymmd0 = function() {
            var _2l = $k[--$j]; //#444
            var _2n = ~~$z($geti(_2l, 2, 2)); //#444
            $k[$j++] = _2l; //#444
            if ((_2n < 1) || (_2n > 12)) { //#444
                $j--; //#444
                $k[$j++] = 'bwipp.GS1badMonth'; //#444
                $k[$j++] = "Invalid month"; //#444
                $k[$j++] = false; //#444
                return true; //#444
            } //#444
            var _2o = $k[--$j]; //#445
            var _2q = ~~$z($geti(_2o, 0, 2)); //#445
            var _2r = _2q - 21; //#446
            $k[$j++] = _2o; //#446
            $k[$j++] = _2q; //#446
            $k[$j++] = _2r; //#446
            if (_2r >= 51) { //#446
                $j--; //#446
                var _2s = $k[--$j]; //#446
                $k[$j++] = $f(_2s + 1900); //#446
            } else { //#446
                if ($k[--$j] <= -50) { //#446
                    var _2u = $k[--$j]; //#446
                    $k[$j++] = $f(_2u + 2100); //#446
                } else { //#446
                    var _2v = $k[--$j]; //#446
                    $k[$j++] = $f(_2v + 2000); //#446
                } //#446
            } //#446
            var _2w = $k[--$j]; //#447
            $k[$j++] = ((_2w % 400) == 0) || (((_2w % 4) == 0) && ((_2w % 100) != 0)); //#448
            $k[$j++] = Infinity; //#448
            var _2x = $k[--$j]; //#448
            var _2z = $k[--$j] ? 29 : 28; //#448
            $k[$j++] = _2x; //#448
            $k[$j++] = 31; //#448
            $k[$j++] = _2z; //#448
            $k[$j++] = 31; //#448
            $k[$j++] = 30; //#448
            $k[$j++] = 31; //#448
            $k[$j++] = 30; //#448
            $k[$j++] = 31; //#448
            $k[$j++] = 31; //#448
            $k[$j++] = 30; //#448
            $k[$j++] = 31; //#448
            $k[$j++] = 30; //#448
            $k[$j++] = 31; //#448
            var _30 = $a(); //#448
            var _31 = $k[--$j]; //#449
            if ($get(_30, (~~$z($geti(_31, 2, 2))) - 1) < (~~$z($geti(_31, 4, 2)))) { //#450
                $j--; //#450
                $k[$j++] = 'bwipp.GS1badDay'; //#450
                $k[$j++] = "Invalid day of month"; //#450
                $k[$j++] = false; //#450
                return true; //#450
            } //#450
        }; //#451
        $1.lintyymmdd = function() {
            var _35 = $k[--$j]; //#454
            $k[$j++] = _35; //#454
            if (_35.length != 6) { //#454
                $j--; //#454
                $k[$j++] = 'bwipp.GS1badDateLength'; //#454
                $k[$j++] = "Invalid length for date"; //#454
                $k[$j++] = false; //#454
                return true; //#454
            } //#454
            var _36 = $k[--$j]; //#455
            $k[$j++] = _36; //#455
            if ((~~$z($geti(_36, 4, 2))) < 1) { //#455
                $j--; //#455
                $k[$j++] = 'bwipp.GS1badDay'; //#455
                $k[$j++] = "Invalid day of month"; //#455
                $k[$j++] = false; //#455
                return true; //#455
            } //#455
            $1.lintyymmd0(); //#456
        }; //#457
        $1.lintyymmddhh = function() {
            var _38 = $k[--$j]; //#460
            $k[$j++] = _38; //#460
            if (_38.length != 8) { //#460
                $j--; //#460
                $k[$j++] = 'bwipp.GS1badYYMMDDHHLength'; //#460
                $k[$j++] = "Invalid length for date with hour"; //#460
                $k[$j++] = false; //#460
                return true; //#460
            } //#460
            var _39 = $k[--$j]; //#461
            $k[$j++] = _39; //#461
            if ((~~$z($geti(_39, 6, 2))) > 23) { //#461
                $j -= 2; //#461
                $k[$j++] = 'bwipp.GS1badHour'; //#461
                $k[$j++] = "Invalid hour of day"; //#461
                $k[$j++] = false; //#461
                return true; //#461
            } //#461
            var _3C = $geti($k[--$j], 0, 6); //#462
            $k[$j++] = _3C; //#462
            $1.lintyymmdd(); //#462
        }; //#463
        $1.linthhmm = function() {
            var _3D = $k[--$j]; //#466
            $k[$j++] = _3D; //#466
            if ((~~$z($geti(_3D, 0, 2))) > 23) { //#466
                $j -= 2; //#466
                $k[$j++] = 'bwipp.GS1badHour'; //#466
                $k[$j++] = "Invalid hour of day"; //#466
                $k[$j++] = false; //#466
                return true; //#466
            } //#466
            if ((~~$z($geti($k[--$j], 2, 2))) > 59) { //#467
                $j--; //#467
                $k[$j++] = 'bwipp.GS1badMinute'; //#467
                $k[$j++] = "Invalid minute in the hour"; //#467
                $k[$j++] = false; //#467
                return true; //#467
            } //#467
        }; //#468
        $1.lintmmoptss = function() {
            var _3H = $k[--$j]; //#471
            var _3I = _3H.length; //#471
            $k[$j++] = _3H; //#473
            if ((_3I != 2) && (_3I != 4)) { //#473
                $j--; //#472
                $k[$j++] = 'bwipp.GS1badTimeLength'; //#472
                $k[$j++] = "Invalid length for optional minutes and seconds"; //#472
                $k[$j++] = false; //#472
                return true; //#472
            } //#472
            var _3J = $k[--$j]; //#474
            $k[$j++] = _3J; //#474
            if ((~~$z($geti(_3J, 0, 2))) > 59) { //#474
                $j -= 2; //#474
                $k[$j++] = 'bwipp.GS1badMinute'; //#474
                $k[$j++] = "Invalid minute in the hour"; //#474
                $k[$j++] = false; //#474
                return true; //#474
            } //#474
            var _3L = $k[--$j]; //#475
            $k[$j++] = _3L; //#477
            if (_3L.length >= 4) { //#477
                var _3M = $k[--$j]; //#476
                $k[$j++] = _3M; //#476
                if ((~~$z($geti(_3M, 2, 2))) > 59) { //#476
                    $j -= 2; //#476
                    $k[$j++] = 'bwipp.GS1badSecond'; //#476
                    $k[$j++] = "Invalid second in the minute"; //#476
                    $k[$j++] = false; //#476
                    return true; //#476
                } //#476
            } //#476
            $j--; //#478
        }; //#479
        $1.lintyesno = function() {
            var _3O = $k[--$j]; //#482
            if ($ne(_3O, "0") && $ne(_3O, "1")) { //#484
                $j--; //#483
                $k[$j++] = 'bwipp.GS1badBoolean'; //#483
                $k[$j++] = "Neither 0 nor 1 for yes or no"; //#483
                $k[$j++] = false; //#483
                return true; //#483
            } //#483
        }; //#485
        $1.lintwinding = function() {
            var _3P = $k[--$j]; //#488
            if ($ne(_3P, "0") && ($ne(_3P, "1") && $ne(_3P, "9"))) { //#490
                $j--; //#489
                $k[$j++] = 'bwipp.GS1badWinding'; //#489
                $k[$j++] = "Invalid winding direction"; //#489
                $k[$j++] = false; //#489
                return true; //#489
            } //#489
        }; //#491
        $1.lintpieceoftotal = function() {
            var _3Q = $k[--$j]; //#494
            $k[$j++] = _3Q; //#494
            if ((_3Q.length % 2) != 0) { //#494
                $j -= 2; //#494
                $k[$j++] = 'bwipp.GS1badPieceTotalLength'; //#494
                $k[$j++] = "Invalid piece/total length"; //#494
                $k[$j++] = false; //#494
                return true; //#494
            } //#494
            var _3R = $k[--$j]; //#495
            var _3T = ~~$z($geti(_3R, 0, ~~(_3R.length / 2))); //#496
            $k[$j++] = _3R; //#496
            $k[$j++] = _3T; //#496
            if (_3T == 0) { //#496
                $j -= 3; //#496
                $k[$j++] = 'bwipp.GS1badPieceNumber'; //#496
                $k[$j++] = "Invalid piece number"; //#496
                $k[$j++] = false; //#496
                return true; //#496
            } //#496
            var _3U = $k[--$j]; //#497
            var _3V = $k[--$j]; //#497
            var _3W = ~~(_3V.length / 2); //#497
            var _3Y = ~~$z($geti(_3V, _3W, _3W)); //#498
            $k[$j++] = _3U; //#498
            $k[$j++] = _3Y; //#498
            if (_3Y == 0) { //#498
                $j -= 3; //#498
                $k[$j++] = 'bwipp.GS1badPieceTotal'; //#498
                $k[$j++] = "Invalid total number"; //#498
                $k[$j++] = false; //#498
                return true; //#498
            } //#498
            var _3Z = $k[--$j]; //#499
            if ($gt($k[--$j], _3Z)) { //#499
                $j--; //#499
                $k[$j++] = 'bwipp.GS1pieceExceedsTotal'; //#499
                $k[$j++] = "Piece number exceeds total"; //#499
                $k[$j++] = false; //#499
                return true; //#499
            } //#499
        }; //#500
        $1.lintpcenc = function() {
            for (;;) { //#513
                $search($k[--$j], "%"); //#504
                if ($nt($k[--$j])) { //#504
                    $j--; //#504
                    break; //#504
                } //#504
                $j -= 2; //#505
                var _3d = $k[--$j]; //#505
                $k[$j++] = _3d; //#505
                if (_3d.length < 2) { //#505
                    $j -= 2; //#505
                    $k[$j++] = 'bwipp.GS1badPercentEscape'; //#505
                    $k[$j++] = "Invalid % escape"; //#505
                    $k[$j++] = false; //#505
                    break; //#505
                } //#505
                var _3e = $k[--$j]; //#506
                var _3f = $geti(_3e, 0, 2); //#506
                $k[$j++] = _3e; //#511
                $k[$j++] = true; //#511
                for (var _3g = 0, _3h = _3f.length; _3g < _3h; _3g++) { //#511
                    var _3j = $s(1); //#507
                    $put(_3j, 0, $get(_3f, _3g)); //#507
                    $search("0123456789ABCDEFabcdef", _3j); //#509
                    if ($nt($k[--$j])) { //#509
                        $j -= 2; //#509
                        $k[$j++] = false; //#509
                        return true; //#509
                    } //#509
                    $j -= 3; //#510
                } //#510
                if ($nt($k[--$j])) { //#512
                    $j -= 2; //#512
                    $k[$j++] = 'bwipp.GS1badPercentChars'; //#512
                    $k[$j++] = "Invalid characters for percent encoding"; //#512
                    $k[$j++] = false; //#512
                    break; //#512
                } //#512
            } //#512
        }; //#514
        $1.lintcouponcode = function() {
            var _3m = $k[--$j]; //#517
            $k[$j++] = _3m; //#519
            $k[$j++] = true; //#519
            $forall(_3m, function() { //#519
                var _3n = $k[--$j]; //#518
                if ((_3n < 48) || (_3n > 57)) { //#518
                    $j--; //#518
                    $k[$j++] = false; //#518
                    return true; //#518
                } //#518
            }); //#518
            if ($nt($k[--$j])) { //#520
                $j -= 2; //#520
                $k[$j++] = 'bwipp.GS1couponNotNumeric'; //#520
                $k[$j++] = "Coupon not numeric"; //#520
                $k[$j++] = false; //#520
                return true; //#520
            } //#520
            var _3p = $k[--$j]; //#523
            $k[$j++] = _3p; //#525
            if (_3p.length < 1) { //#525
                $j -= 2; //#524
                $k[$j++] = 'bwipp.GS1couponTooShortGCPVLI'; //#524
                $k[$j++] = "Coupon too short: Missing GCP VLI"; //#524
                $k[$j++] = false; //#524
                return true; //#524
            } //#524
            var _3q = $k[--$j]; //#526
            var _3s = ~~$z($geti(_3q, 0, 1)); //#526
            $k[$j++] = _3q; //#528
            $k[$j++] = _3s; //#528
            if (_3s > 6) { //#528
                $j -= 2; //#527
                $k[$j++] = 'bwipp.GS1couponBadGCPVLI'; //#527
                $k[$j++] = "Coupon GCP length indicator must be 0-6"; //#527
                $k[$j++] = false; //#527
                return true; //#527
            } //#527
            var _3t = $k[--$j]; //#529
            var _3u = $k[--$j]; //#530
            $k[$j++] = _3u; //#532
            $k[$j++] = $f($f(_3t + 6) + 1); //#532
            if (($f($f(_3t + 6) + 1)) > _3u.length) { //#532
                $j -= 3; //#531
                $k[$j++] = 'bwipp.GS1couponTooShortGCP'; //#531
                $k[$j++] = "Coupon too short: GCP truncated"; //#531
                $k[$j++] = false; //#531
                return true; //#531
            } //#531
            var _3v = $k[--$j]; //#533
            var _3w = $k[--$j]; //#533
            var _3x = $geti(_3w, _3v, $f(_3w.length - _3v)); //#533
            $k[$j++] = _3x; //#538
            if (_3x.length < 6) { //#538
                $j -= 2; //#537
                $k[$j++] = 'bwipp.GS1couponTooShortOfferCode'; //#537
                $k[$j++] = "Coupon too short: Offer Code truncated"; //#537
                $k[$j++] = false; //#537
                return true; //#537
            } //#537
            var _3y = $k[--$j]; //#539
            var _3z = $geti(_3y, 6, _3y.length - 6); //#539
            $k[$j++] = _3z; //#544
            if (_3z.length < 1) { //#544
                $j -= 2; //#543
                $k[$j++] = 'bwipp.GS1couponTooShortSaveValueVLI'; //#543
                $k[$j++] = "Coupon too short: Missing Save Value VLI"; //#543
                $k[$j++] = false; //#543
                return true; //#543
            } //#543
            var _40 = $k[--$j]; //#545
            var _42 = ~~$z($geti(_40, 0, 1)); //#545
            $k[$j++] = _40; //#547
            $k[$j++] = _42; //#547
            if ((_42 < 1) || (_42 > 5)) { //#547
                $j -= 2; //#546
                $k[$j++] = 'bwipp.GS1couponBadSaveValueVLI'; //#546
                $k[$j++] = "Coupon Save Value length indicator must be 1-5"; //#546
                $k[$j++] = false; //#546
                return true; //#546
            } //#546
            var _43 = $k[--$j]; //#548
            var _44 = $k[--$j]; //#549
            $k[$j++] = _44; //#551
            $k[$j++] = $f(_43 + 1); //#551
            if ($f(_43 + 1) > _44.length) { //#551
                $j -= 3; //#550
                $k[$j++] = 'bwipp.GS1couponTooShortSaveValue'; //#550
                $k[$j++] = "Coupon too short: Save Value truncated"; //#550
                $k[$j++] = false; //#550
                return true; //#550
            } //#550
            var _45 = $k[--$j]; //#552
            var _46 = $k[--$j]; //#552
            var _47 = $geti(_46, _45, $f(_46.length - _45)); //#552
            $k[$j++] = _47; //#557
            if (_47.length < 1) { //#557
                $j -= 2; //#556
                $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseRequirementVLI'; //#556
                $k[$j++] = "Coupon too short: Missing 1st Purchase Requirement VLI"; //#556
                $k[$j++] = false; //#556
                return true; //#556
            } //#556
            var _48 = $k[--$j]; //#558
            var _4A = ~~$z($geti(_48, 0, 1)); //#558
            $k[$j++] = _48; //#560
            $k[$j++] = _4A; //#560
            if ((_4A < 1) || (_4A > 5)) { //#560
                $j -= 2; //#559
                $k[$j++] = 'bwipp.GS1couponBad1stPurchaseRequirementVLI'; //#559
                $k[$j++] = "Coupon 1st Purchase Requirement length indicator must be 1-5"; //#559
                $k[$j++] = false; //#559
                return true; //#559
            } //#559
            var _4B = $k[--$j]; //#561
            var _4C = $k[--$j]; //#562
            $k[$j++] = _4C; //#564
            $k[$j++] = $f(_4B + 1); //#564
            if ($f(_4B + 1) > _4C.length) { //#564
                $j -= 3; //#563
                $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseRequirement'; //#563
                $k[$j++] = "Coupon too short: 1st Purchase Requirement truncated"; //#563
                $k[$j++] = false; //#563
                return true; //#563
            } //#563
            var _4D = $k[--$j]; //#565
            var _4E = $k[--$j]; //#565
            var _4F = $geti(_4E, _4D, $f(_4E.length - _4D)); //#565
            $k[$j++] = _4F; //#570
            if (_4F.length < 1) { //#570
                $j -= 2; //#569
                $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseRequirementCode'; //#569
                $k[$j++] = "Coupon too short: Missing 1st Purchase Requirement Code"; //#569
                $k[$j++] = false; //#569
                return true; //#569
            } //#569
            var _4G = $k[--$j]; //#571
            var _4I = ~~$z($geti(_4G, 0, 1)); //#571
            $k[$j++] = _4G; //#573
            if ((_4I > 4) && (_4I != 9)) { //#573
                $j -= 2; //#572
                $k[$j++] = 'bwipp.GS1couponBad1stPurchaseRequirementCode'; //#572
                $k[$j++] = "Coupon 1st Purchase Requirement Code must be 0-4 or 9"; //#572
                $k[$j++] = false; //#572
                return true; //#572
            } //#572
            var _4J = $k[--$j]; //#574
            var _4K = $geti(_4J, 1, _4J.length - 1); //#574
            $k[$j++] = _4K; //#579
            if (_4K.length < 3) { //#579
                $j -= 2; //#578
                $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseFamilyCode'; //#578
                $k[$j++] = "Coupon too short: 1st Purchase Family Code truncated"; //#578
                $k[$j++] = false; //#578
                return true; //#578
            } //#578
            var _4L = $k[--$j]; //#580
            var _4M = $geti(_4L, 3, _4L.length - 3); //#580
            $k[$j++] = _4M; //#636
            if (_4M.length >= 1) { //#636
                var _4N = $k[--$j]; //#583
                $k[$j++] = _4N; //#636
                if ((~~$z($geti(_4N, 0, 1))) == 1) { //#636
                    var _4P = $k[--$j]; //#584
                    var _4Q = $geti(_4P, 1, _4P.length - 1); //#584
                    $k[$j++] = _4Q; //#589
                    if (_4Q.length < 1) { //#589
                        $j -= 2; //#588
                        $k[$j++] = 'bwipp.GS1couponTooShortAdditionalPurchaseRulesCode'; //#588
                        $k[$j++] = "Coupon too short: Missing Additional Purchase Rules Code"; //#588
                        $k[$j++] = false; //#588
                        return true; //#588
                    } //#588
                    var _4R = $k[--$j]; //#590
                    $k[$j++] = _4R; //#592
                    if ((~~$z($geti(_4R, 0, 1))) > 3) { //#592
                        $j -= 2; //#591
                        $k[$j++] = 'bwipp.GS1couponBadAdditionalPurchaseRulesCode'; //#591
                        $k[$j++] = "Coupon Additional Purchase Rules Code must be 0-3"; //#591
                        $k[$j++] = false; //#591
                        return true; //#591
                    } //#591
                    var _4T = $k[--$j]; //#593
                    var _4U = $geti(_4T, 1, _4T.length - 1); //#593
                    $k[$j++] = _4U; //#598
                    if (_4U.length < 1) { //#598
                        $j -= 2; //#597
                        $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseRequirementVLI'; //#597
                        $k[$j++] = "Coupon too short: Missing 2nd Purchase Requirement VLI"; //#597
                        $k[$j++] = false; //#597
                        return true; //#597
                    } //#597
                    var _4V = $k[--$j]; //#599
                    var _4X = ~~$z($geti(_4V, 0, 1)); //#599
                    $k[$j++] = _4V; //#601
                    $k[$j++] = _4X; //#601
                    if ((_4X < 1) || (_4X > 5)) { //#601
                        $j -= 2; //#600
                        $k[$j++] = 'bwipp.GS1couponBad2ndPurchaseRequirementVLI'; //#600
                        $k[$j++] = "Coupon 2nd Purchase Requirement length indicator must be 1-5"; //#600
                        $k[$j++] = false; //#600
                        return true; //#600
                    } //#600
                    var _4Y = $k[--$j]; //#602
                    var _4Z = $k[--$j]; //#603
                    $k[$j++] = _4Z; //#605
                    $k[$j++] = $f(_4Y + 1); //#605
                    if ($f(_4Y + 1) > _4Z.length) { //#605
                        $j -= 3; //#604
                        $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseRequirement'; //#604
                        $k[$j++] = "Coupon too short: 2nd Purchase Requirement truncated"; //#604
                        $k[$j++] = false; //#604
                        return true; //#604
                    } //#604
                    var _4a = $k[--$j]; //#606
                    var _4b = $k[--$j]; //#606
                    var _4c = $geti(_4b, _4a, $f(_4b.length - _4a)); //#606
                    $k[$j++] = _4c; //#611
                    if (_4c.length < 1) { //#611
                        $j -= 2; //#610
                        $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseRequirementCode'; //#610
                        $k[$j++] = "Coupon too short: Missing 2nd Purchase Requirement Code"; //#610
                        $k[$j++] = false; //#610
                        return true; //#610
                    } //#610
                    var _4d = $k[--$j]; //#612
                    var _4f = ~~$z($geti(_4d, 0, 1)); //#612
                    $k[$j++] = _4d; //#614
                    if ((_4f > 4) && (_4f != 9)) { //#614
                        $j -= 2; //#613
                        $k[$j++] = 'bwipp.GS1couponBad2ndPurchaseRequirementCode'; //#613
                        $k[$j++] = "Coupon 2nd Purchase Requirement Code must be 0-4 or 9"; //#613
                        $k[$j++] = false; //#613
                        return true; //#613
                    } //#613
                    var _4g = $k[--$j]; //#615
                    var _4h = $geti(_4g, 1, _4g.length - 1); //#615
                    $k[$j++] = _4h; //#620
                    if (_4h.length < 3) { //#620
                        $j -= 2; //#619
                        $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseFamilyCode'; //#619
                        $k[$j++] = "Coupon too short: 2nd Purchase Family Code truncated"; //#619
                        $k[$j++] = false; //#619
                        return true; //#619
                    } //#619
                    var _4i = $k[--$j]; //#621
                    var _4j = $geti(_4i, 3, _4i.length - 3); //#621
                    $k[$j++] = _4j; //#626
                    if (_4j.length < 1) { //#626
                        $j -= 2; //#625
                        $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseGCPVLI'; //#625
                        $k[$j++] = "Coupon too short: Missing 2nd Purchase GCP VLI"; //#625
                        $k[$j++] = false; //#625
                        return true; //#625
                    } //#625
                    var _4k = $k[--$j]; //#627
                    var _4m = ~~$z($geti(_4k, 0, 1)); //#627
                    $k[$j++] = _4k; //#629
                    $k[$j++] = _4m; //#629
                    if ((_4m > 6) && (_4m != 9)) { //#629
                        $j -= 2; //#628
                        $k[$j++] = 'bwipp.GS1couponBad2ndPurchaseGCPVLI'; //#628
                        $k[$j++] = "Coupon 2nd Purchase GCP length indicator must be 0-6 or 9"; //#628
                        $k[$j++] = false; //#628
                        return true; //#628
                    } //#628
                    var _4n = $k[--$j]; //#630
                    $k[$j++] = _4n; //#630
                    if (_4n != 9) { //#630
                        var _4o = $k[--$j]; //#630
                        $k[$j++] = $f(_4o + 6); //#630
                    } else { //#630
                        $j--; //#630
                        $k[$j++] = 0; //#630
                    } //#630
                    var _4p = $k[--$j]; //#630
                    var _4q = $k[--$j]; //#631
                    $k[$j++] = _4q; //#633
                    $k[$j++] = $f(_4p + 1); //#633
                    if ($f(_4p + 1) > _4q.length) { //#633
                        $j -= 3; //#632
                        $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseGCP'; //#632
                        $k[$j++] = "Coupon too short: 2nd Purchase GCP truncated"; //#632
                        $k[$j++] = false; //#632
                        return true; //#632
                    } //#632
                    var _4r = $k[--$j]; //#634
                    var _4s = $k[--$j]; //#634
                    $k[$j++] = $geti(_4s, _4r, $f(_4s.length - _4r)); //#634
                } //#634
            } //#634
            var _4u = $k[--$j]; //#639
            $k[$j++] = _4u; //#683
            if (_4u.length >= 1) { //#683
                var _4v = $k[--$j]; //#639
                $k[$j++] = _4v; //#683
                if ((~~$z($geti(_4v, 0, 1))) == 2) { //#683
                    var _4x = $k[--$j]; //#640
                    var _4y = $geti(_4x, 1, _4x.length - 1); //#640
                    $k[$j++] = _4y; //#645
                    if (_4y.length < 1) { //#645
                        $j -= 2; //#644
                        $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseRequirementVLI'; //#644
                        $k[$j++] = "Coupon too short: Missing 3rd Purchase Requirement VLI"; //#644
                        $k[$j++] = false; //#644
                        return true; //#644
                    } //#644
                    var _4z = $k[--$j]; //#646
                    var _51 = ~~$z($geti(_4z, 0, 1)); //#646
                    $k[$j++] = _4z; //#648
                    $k[$j++] = _51; //#648
                    if ((_51 < 1) || (_51 > 5)) { //#648
                        $j -= 2; //#647
                        $k[$j++] = 'bwipp.GS1couponBad3rdPurchaseRequirementVLI'; //#647
                        $k[$j++] = "Coupon 3rd Purchase Requirement length indicator must be 1-5"; //#647
                        $k[$j++] = false; //#647
                        return true; //#647
                    } //#647
                    var _52 = $k[--$j]; //#649
                    var _53 = $k[--$j]; //#650
                    $k[$j++] = _53; //#652
                    $k[$j++] = $f(_52 + 1); //#652
                    if ($f(_52 + 1) > _53.length) { //#652
                        $j -= 3; //#651
                        $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseRequirement'; //#651
                        $k[$j++] = "Coupon too short: 3rd Purchase Requirement truncated"; //#651
                        $k[$j++] = false; //#651
                        return true; //#651
                    } //#651
                    var _54 = $k[--$j]; //#653
                    var _55 = $k[--$j]; //#653
                    var _56 = $geti(_55, _54, $f(_55.length - _54)); //#653
                    $k[$j++] = _56; //#658
                    if (_56.length < 1) { //#658
                        $j -= 2; //#657
                        $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseRequirementCode'; //#657
                        $k[$j++] = "Coupon too short: Missing 3rd Purchase Requirement Code"; //#657
                        $k[$j++] = false; //#657
                        return true; //#657
                    } //#657
                    var _57 = $k[--$j]; //#659
                    var _59 = ~~$z($geti(_57, 0, 1)); //#659
                    $k[$j++] = _57; //#661
                    if ((_59 > 4) && (_59 != 9)) { //#661
                        $j -= 2; //#660
                        $k[$j++] = 'bwipp.GS1couponBad3rdPurchaseRequirementCode'; //#660
                        $k[$j++] = "Coupon 3rd Purchase Requirement Code must be 0-4 or 9"; //#660
                        $k[$j++] = false; //#660
                        return true; //#660
                    } //#660
                    var _5A = $k[--$j]; //#662
                    var _5B = $geti(_5A, 1, _5A.length - 1); //#662
                    $k[$j++] = _5B; //#667
                    if (_5B.length < 3) { //#667
                        $j -= 2; //#666
                        $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseFamilyCode'; //#666
                        $k[$j++] = "Coupon too short: 3rd Purchase Family Code truncated"; //#666
                        $k[$j++] = false; //#666
                        return true; //#666
                    } //#666
                    var _5C = $k[--$j]; //#668
                    var _5D = $geti(_5C, 3, _5C.length - 3); //#668
                    $k[$j++] = _5D; //#673
                    if (_5D.length < 1) { //#673
                        $j -= 2; //#672
                        $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseGCPVLI'; //#672
                        $k[$j++] = "Coupon too short: Missing 3rd Purchase GCP VLI"; //#672
                        $k[$j++] = false; //#672
                        return true; //#672
                    } //#672
                    var _5E = $k[--$j]; //#674
                    var _5G = ~~$z($geti(_5E, 0, 1)); //#674
                    $k[$j++] = _5E; //#676
                    $k[$j++] = _5G; //#676
                    if ((_5G > 6) && (_5G != 9)) { //#676
                        $j -= 2; //#675
                        $k[$j++] = 'bwipp.GS1couponBad3rdPurchaseGCPVLI'; //#675
                        $k[$j++] = "Coupon 3rd Purchase GCP length indicator must be 0-6 or 9"; //#675
                        $k[$j++] = false; //#675
                        return true; //#675
                    } //#675
                    var _5H = $k[--$j]; //#677
                    $k[$j++] = _5H; //#677
                    if (_5H != 9) { //#677
                        var _5I = $k[--$j]; //#677
                        $k[$j++] = $f(_5I + 6); //#677
                    } else { //#677
                        $j--; //#677
                        $k[$j++] = 0; //#677
                    } //#677
                    var _5J = $k[--$j]; //#677
                    var _5K = $k[--$j]; //#678
                    $k[$j++] = _5K; //#680
                    $k[$j++] = $f(_5J + 1); //#680
                    if ($f(_5J + 1) > _5K.length) { //#680
                        $j -= 3; //#679
                        $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseGCP'; //#679
                        $k[$j++] = "Coupon too short: 3rd Purchase GCP truncated"; //#679
                        $k[$j++] = false; //#679
                        return true; //#679
                    } //#679
                    var _5L = $k[--$j]; //#681
                    var _5M = $k[--$j]; //#681
                    $k[$j++] = $geti(_5M, _5L, $f(_5M.length - _5L)); //#681
                } //#681
            } //#681
            $1.couponexpire = -1; //#686
            var _5O = $k[--$j]; //#687
            $k[$j++] = _5O; //#705
            if (_5O.length >= 1) { //#705
                var _5P = $k[--$j]; //#687
                $k[$j++] = _5P; //#705
                if ((~~$z($geti(_5P, 0, 1))) == 3) { //#705
                    var _5R = $k[--$j]; //#688
                    var _5S = $geti(_5R, 1, _5R.length - 1); //#688
                    $k[$j++] = _5S; //#693
                    if (_5S.length < 6) { //#693
                        $j -= 2; //#692
                        $k[$j++] = 'bwipp.GS1couponTooShortExpirationDate'; //#692
                        $k[$j++] = "Coupon too short: Expiration date"; //#692
                        $k[$j++] = false; //#692
                        return true; //#692
                    } //#692
                    var _5T = $k[--$j]; //#694
                    var _5V = ~~$z($geti(_5T, 2, 2)); //#694
                    $k[$j++] = _5T; //#694
                    if ((_5V < 1) || (_5V > 12)) { //#694
                        $j -= 2; //#694
                        $k[$j++] = 'bwipp.GS1couponExpirationDateBadMonth'; //#694
                        $k[$j++] = "Invalid month in expiration date"; //#694
                        $k[$j++] = false; //#694
                        return true; //#694
                    } //#694
                    var _5W = $k[--$j]; //#695
                    var _5Y = ~~$z($geti(_5W, 0, 2)); //#695
                    var _5Z = _5Y - 21; //#696
                    $k[$j++] = _5W; //#696
                    $k[$j++] = _5Y; //#696
                    $k[$j++] = _5Z; //#696
                    if (_5Z >= 51) { //#696
                        $j--; //#696
                        var _5a = $k[--$j]; //#696
                        $k[$j++] = $f(_5a + 1900); //#696
                    } else { //#696
                        if ($k[--$j] <= -50) { //#696
                            var _5c = $k[--$j]; //#696
                            $k[$j++] = $f(_5c + 2100); //#696
                        } else { //#696
                            var _5d = $k[--$j]; //#696
                            $k[$j++] = $f(_5d + 2000); //#696
                        } //#696
                    } //#696
                    var _5e = $k[--$j]; //#697
                    $k[$j++] = ((_5e % 400) == 0) || (((_5e % 4) == 0) && ((_5e % 100) != 0)); //#698
                    $k[$j++] = Infinity; //#698
                    var _5f = $k[--$j]; //#698
                    var _5h = $k[--$j] ? 29 : 28; //#698
                    $k[$j++] = _5f; //#698
                    $k[$j++] = 31; //#698
                    $k[$j++] = _5h; //#698
                    $k[$j++] = 31; //#698
                    $k[$j++] = 30; //#698
                    $k[$j++] = 31; //#698
                    $k[$j++] = 30; //#698
                    $k[$j++] = 31; //#698
                    $k[$j++] = 31; //#698
                    $k[$j++] = 30; //#698
                    $k[$j++] = 31; //#698
                    $k[$j++] = 30; //#698
                    $k[$j++] = 31; //#698
                    var _5i = $a(); //#698
                    var _5j = $k[--$j]; //#699
                    var _5n = ~~$z($geti(_5j, 4, 2)); //#700
                    $k[$j++] = _5j; //#701
                    if (($get(_5i, (~~$z($geti(_5j, 2, 2))) - 1) < _5n) || (_5n < 1)) { //#701
                        $j -= 2; //#701
                        $k[$j++] = 'bwipp.GS1couponExpirationDateBadDay'; //#701
                        $k[$j++] = "Invalid day of month in expiration date"; //#701
                        $k[$j++] = false; //#701
                        return true; //#701
                    } //#701
                    var _5o = $k[--$j]; //#702
                    $1.couponexpire = ~~$z($geti(_5o, 0, 6)); //#702
                    $k[$j++] = $geti(_5o, 6, _5o.length - 6); //#703
                } //#703
            } //#703
            var _5r = $k[--$j]; //#708
            $k[$j++] = _5r; //#729
            if (_5r.length >= 1) { //#729
                var _5s = $k[--$j]; //#708
                $k[$j++] = _5s; //#729
                if ((~~$z($geti(_5s, 0, 1))) == 4) { //#729
                    var _5u = $k[--$j]; //#709
                    var _5v = $geti(_5u, 1, _5u.length - 1); //#709
                    $k[$j++] = _5v; //#714
                    if (_5v.length < 6) { //#714
                        $j -= 2; //#713
                        $k[$j++] = 'bwipp.GS1couponTooShortStartDate'; //#713
                        $k[$j++] = "Coupon too short: Start date"; //#713
                        $k[$j++] = false; //#713
                        return true; //#713
                    } //#713
                    var _5w = $k[--$j]; //#715
                    var _5y = ~~$z($geti(_5w, 2, 2)); //#715
                    $k[$j++] = _5w; //#715
                    if ((_5y < 1) || (_5y > 12)) { //#715
                        $j -= 2; //#715
                        $k[$j++] = 'bwipp.GS1couponStartDateBadMonth'; //#715
                        $k[$j++] = "Invalid month in start date"; //#715
                        $k[$j++] = false; //#715
                        return true; //#715
                    } //#715
                    var _5z = $k[--$j]; //#716
                    var _61 = ~~$z($geti(_5z, 0, 2)); //#716
                    var _62 = _61 - 21; //#717
                    $k[$j++] = _5z; //#717
                    $k[$j++] = _61; //#717
                    $k[$j++] = _62; //#717
                    if (_62 >= 51) { //#717
                        $j--; //#717
                        var _63 = $k[--$j]; //#717
                        $k[$j++] = $f(_63 + 1900); //#717
                    } else { //#717
                        if ($k[--$j] <= -50) { //#717
                            var _65 = $k[--$j]; //#717
                            $k[$j++] = $f(_65 + 2100); //#717
                        } else { //#717
                            var _66 = $k[--$j]; //#717
                            $k[$j++] = $f(_66 + 2000); //#717
                        } //#717
                    } //#717
                    var _67 = $k[--$j]; //#718
                    $k[$j++] = ((_67 % 400) == 0) || (((_67 % 4) == 0) && ((_67 % 100) != 0)); //#719
                    $k[$j++] = Infinity; //#719
                    var _68 = $k[--$j]; //#719
                    var _6A = $k[--$j] ? 29 : 28; //#719
                    $k[$j++] = _68; //#719
                    $k[$j++] = 31; //#719
                    $k[$j++] = _6A; //#719
                    $k[$j++] = 31; //#719
                    $k[$j++] = 30; //#719
                    $k[$j++] = 31; //#719
                    $k[$j++] = 30; //#719
                    $k[$j++] = 31; //#719
                    $k[$j++] = 31; //#719
                    $k[$j++] = 30; //#719
                    $k[$j++] = 31; //#719
                    $k[$j++] = 30; //#719
                    $k[$j++] = 31; //#719
                    var _6B = $a(); //#719
                    var _6C = $k[--$j]; //#720
                    var _6G = ~~$z($geti(_6C, 4, 2)); //#721
                    $k[$j++] = _6C; //#722
                    if (($get(_6B, (~~$z($geti(_6C, 2, 2))) - 1) < _6G) || (_6G < 1)) { //#722
                        $j -= 2; //#722
                        $k[$j++] = 'bwipp.GS1couponStartDateBadDay'; //#722
                        $k[$j++] = "Invalid day of month in start date"; //#722
                        $k[$j++] = false; //#722
                        return true; //#722
                    } //#722
                    var _6H = $k[--$j]; //#723
                    $1.couponstart = ~~$z($geti(_6H, 0, 6)); //#723
                    $k[$j++] = _6H; //#726
                    if (($1.couponexpire != -1) && ($1.couponexpire < $1.couponstart)) { //#726
                        $j -= 2; //#725
                        $k[$j++] = 'bwipp.GS1couponExpireDateBeforeStartDate'; //#725
                        $k[$j++] = "Coupon expires before it starts"; //#725
                        $k[$j++] = false; //#725
                        return true; //#725
                    } //#725
                    var _6M = $k[--$j]; //#727
                    $k[$j++] = $geti(_6M, 6, _6M.length - 6); //#727
                } //#727
            } //#727
            var _6O = $k[--$j]; //#732
            $k[$j++] = _6O; //#745
            if (_6O.length >= 1) { //#745
                var _6P = $k[--$j]; //#732
                $k[$j++] = _6P; //#745
                if ((~~$z($geti(_6P, 0, 1))) == 5) { //#745
                    var _6R = $k[--$j]; //#733
                    var _6S = $geti(_6R, 1, _6R.length - 1); //#733
                    $k[$j++] = _6S; //#738
                    if (_6S.length < 1) { //#738
                        $j -= 2; //#737
                        $k[$j++] = 'bwipp.GS1couponTooShortSerialNumberVLI'; //#737
                        $k[$j++] = "Coupon too short: Missing Serial Number VLI"; //#737
                        $k[$j++] = false; //#737
                        return true; //#737
                    } //#737
                    var _6T = $k[--$j]; //#739
                    var _6U = $geti(_6T, 0, 1); //#739
                    $k[$j++] = _6T; //#742
                    $k[$j++] = ((~~$z(_6U)) + 6) + 1; //#742
                    if ((((~~$z(_6U)) + 6) + 1) > _6T.length) { //#742
                        $j -= 3; //#741
                        $k[$j++] = 'bwipp.GS1couponTooShortSerialNumber'; //#741
                        $k[$j++] = "Coupon too short: Serial Number truncated"; //#741
                        $k[$j++] = false; //#741
                        return true; //#741
                    } //#741
                    var _6V = $k[--$j]; //#743
                    var _6W = $k[--$j]; //#743
                    $k[$j++] = $geti(_6W, _6V, $f(_6W.length - _6V)); //#743
                } //#743
            } //#743
            var _6Y = $k[--$j]; //#748
            $k[$j++] = _6Y; //#764
            if (_6Y.length >= 1) { //#764
                var _6Z = $k[--$j]; //#748
                $k[$j++] = _6Z; //#764
                if ((~~$z($geti(_6Z, 0, 1))) == 6) { //#764
                    var _6b = $k[--$j]; //#749
                    var _6c = $geti(_6b, 1, _6b.length - 1); //#749
                    $k[$j++] = _6c; //#754
                    if (_6c.length < 1) { //#754
                        $j -= 2; //#753
                        $k[$j++] = 'bwipp.GS1couponTooShortRetailerGCPGLNVLI'; //#753
                        $k[$j++] = "Coupon too short: Missing Retailer GCP/GLN VLI"; //#753
                        $k[$j++] = false; //#753
                        return true; //#753
                    } //#753
                    var _6d = $k[--$j]; //#755
                    var _6f = ~~$z($geti(_6d, 0, 1)); //#755
                    $k[$j++] = _6d; //#757
                    $k[$j++] = _6f; //#757
                    if ((_6f < 1) || (_6f > 7)) { //#757
                        $j -= 2; //#756
                        $k[$j++] = 'bwipp.GS1couponBadRetailerGCPGLNVLI'; //#756
                        $k[$j++] = "Coupon Retailer GCP/GLN length indicator must be 1-7"; //#756
                        $k[$j++] = false; //#756
                        return true; //#756
                    } //#756
                    var _6g = $k[--$j]; //#758
                    var _6h = $k[--$j]; //#759
                    $k[$j++] = _6h; //#761
                    $k[$j++] = $f($f(_6g + 6) + 1); //#761
                    if (($f($f(_6g + 6) + 1)) > _6h.length) { //#761
                        $j -= 3; //#760
                        $k[$j++] = 'bwipp.GS1couponTooShortRetailerGCPGLN'; //#760
                        $k[$j++] = "Coupon too short: Retailer GCP/GLN truncated"; //#760
                        $k[$j++] = false; //#760
                        return true; //#760
                    } //#760
                    var _6i = $k[--$j]; //#762
                    var _6j = $k[--$j]; //#762
                    $k[$j++] = $geti(_6j, _6i, $f(_6j.length - _6i)); //#762
                } //#762
            } //#762
            var _6l = $k[--$j]; //#767
            $k[$j++] = _6l; //#803
            if (_6l.length >= 1) { //#803
                var _6m = $k[--$j]; //#767
                $k[$j++] = _6m; //#803
                if ((~~$z($geti(_6m, 0, 1))) == 9) { //#803
                    var _6o = $k[--$j]; //#768
                    var _6p = $geti(_6o, 1, _6o.length - 1); //#768
                    $k[$j++] = _6p; //#773
                    if (_6p.length < 1) { //#773
                        $j -= 2; //#772
                        $k[$j++] = 'bwipp.GS1couponTooShortSaveValueCode'; //#772
                        $k[$j++] = "Coupon too short: Missing Save Value Code"; //#772
                        $k[$j++] = false; //#772
                        return true; //#772
                    } //#772
                    var _6q = $k[--$j]; //#774
                    var _6s = ~~$z($geti(_6q, 0, 1)); //#774
                    $k[$j++] = _6q; //#776
                    if ((_6s > 6) || ((_6s == 3) || (_6s == 4))) { //#776
                        $j -= 2; //#775
                        $k[$j++] = 'bwipp.GS1couponBadSaveValueCode'; //#775
                        $k[$j++] = "Coupon Save Value Code must be 0,1,2,5 or 6"; //#775
                        $k[$j++] = false; //#775
                        return true; //#775
                    } //#775
                    var _6t = $k[--$j]; //#777
                    var _6u = $geti(_6t, 1, _6t.length - 1); //#777
                    $k[$j++] = _6u; //#782
                    if (_6u.length < 1) { //#782
                        $j -= 2; //#781
                        $k[$j++] = 'bwipp.GS1couponTooShortSaveValueAppliesToItem'; //#781
                        $k[$j++] = "Coupon too short: Missing Save Value Applies to Item"; //#781
                        $k[$j++] = false; //#781
                        return true; //#781
                    } //#781
                    var _6v = $k[--$j]; //#783
                    $k[$j++] = _6v; //#785
                    if ((~~$z($geti(_6v, 0, 1))) > 2) { //#785
                        $j -= 2; //#784
                        $k[$j++] = 'bwipp.GS1couponBadSaveValueAppliesToItem'; //#784
                        $k[$j++] = "Coupon Save Value Applies to Item must be 0-2"; //#784
                        $k[$j++] = false; //#784
                        return true; //#784
                    } //#784
                    var _6x = $k[--$j]; //#786
                    var _6y = $geti(_6x, 1, _6x.length - 1); //#786
                    $k[$j++] = _6y; //#791
                    if (_6y.length < 1) { //#791
                        $j -= 2; //#790
                        $k[$j++] = 'bwipp.GS1couponTooShortStoreCouponFlag'; //#790
                        $k[$j++] = "Coupon too short: Missing Store Coupon Flag"; //#790
                        $k[$j++] = false; //#790
                        return true; //#790
                    } //#790
                    var _6z = $k[--$j]; //#792
                    var _70 = $geti(_6z, 1, _6z.length - 1); //#792
                    $k[$j++] = _70; //#797
                    if (_70.length < 1) { //#797
                        $j -= 2; //#796
                        $k[$j++] = 'bwipp.GS1couponTooShortDontMultiplyFlag'; //#796
                        $k[$j++] = "Coupon too short: Missing Don't Multiply Flag"; //#796
                        $k[$j++] = false; //#796
                        return true; //#796
                    } //#796
                    var _71 = $k[--$j]; //#798
                    $k[$j++] = _71; //#800
                    if ((~~$z($geti(_71, 0, 1))) > 1) { //#800
                        $j -= 2; //#799
                        $k[$j++] = 'bwipp.GS1couponBadDontMultiplyFlag'; //#799
                        $k[$j++] = "Don't Multiply Flag must be 0 or 1"; //#799
                        $k[$j++] = false; //#799
                        return true; //#799
                    } //#799
                    var _73 = $k[--$j]; //#801
                    $k[$j++] = $geti(_73, 1, _73.length - 1); //#801
                } //#801
            } //#801
            var _75 = $k[--$j]; //#805
            $k[$j++] = _75; //#807
            if (_75.length != 0) { //#807
                $j -= 2; //#806
                $k[$j++] = 'bwipp.GS1couponUnrecognisedOptionalField'; //#806
                $k[$j++] = "Coupon fields must be 1,2,3,4,5,6 or 9, increasing order"; //#806
                $k[$j++] = false; //#806
                return true; //#806
            } //#806
            $j--; //#808
        }; //#809
        $1.lintcouponposoffer = function() {
            var _76 = $k[--$j]; //#812
            $k[$j++] = _76; //#814
            $k[$j++] = true; //#814
            $forall(_76, function() { //#814
                var _77 = $k[--$j]; //#813
                if ((_77 < 48) || (_77 > 57)) { //#813
                    $j--; //#813
                    $k[$j++] = false; //#813
                    return true; //#813
                } //#813
            }); //#813
            if ($nt($k[--$j])) { //#815
                $j -= 2; //#815
                $k[$j++] = 'bwipp.GS1couponNotNumeric'; //#815
                $k[$j++] = "Coupon not numeric"; //#815
                $k[$j++] = false; //#815
                return true; //#815
            } //#815
            var _79 = $k[--$j]; //#817
            $k[$j++] = _79; //#819
            if (_79.length < 1) { //#819
                $j -= 2; //#818
                $k[$j++] = 'bwipp.GS1couponTooShortFormatCode'; //#818
                $k[$j++] = "Coupon too short: Missing Format Code"; //#818
                $k[$j++] = false; //#818
                return true; //#818
            } //#818
            var _7A = $k[--$j]; //#820
            var _7B = $geti(_7A, 0, 1); //#820
            $k[$j++] = _7A; //#822
            if ($ne(_7B, "0") && $ne(_7B, "1")) { //#822
                $j -= 2; //#821
                $k[$j++] = 'bwipp.GS1couponBadFormatCode'; //#821
                $k[$j++] = "Coupon format must be 0 or 1"; //#821
                $k[$j++] = false; //#821
                return true; //#821
            } //#821
            var _7C = $k[--$j]; //#823
            var _7D = $geti(_7C, 1, _7C.length - 1); //#823
            $k[$j++] = _7D; //#827
            if (_7D.length < 1) { //#827
                $j -= 2; //#826
                $k[$j++] = 'bwipp.GS1couponTooShortFunderVLI'; //#826
                $k[$j++] = "Coupon too short: Missing Funder VLI"; //#826
                $k[$j++] = false; //#826
                return true; //#826
            } //#826
            var _7E = $k[--$j]; //#828
            var _7G = ~~$z($geti(_7E, 0, 1)); //#828
            $k[$j++] = _7E; //#830
            $k[$j++] = _7G; //#830
            if (_7G > 6) { //#830
                $j -= 3; //#829
                $k[$j++] = 'bwipp.GS1couponBadFunderVLI'; //#829
                $k[$j++] = "Coupon Funder length indicator must be 0-6"; //#829
                $k[$j++] = false; //#829
                return true; //#829
            } //#829
            var _7H = $k[--$j]; //#831
            var _7I = $k[--$j]; //#832
            $k[$j++] = _7I; //#834
            $k[$j++] = $f($f(_7H + 6) + 1); //#834
            if (($f($f(_7H + 6) + 1)) > _7I.length) { //#834
                $j -= 3; //#833
                $k[$j++] = 'bwipp.GS1couponTooShortFunder'; //#833
                $k[$j++] = "Coupon too short: Truncated Funder ID"; //#833
                $k[$j++] = false; //#833
                return true; //#833
            } //#833
            var _7J = $k[--$j]; //#835
            var _7K = $k[--$j]; //#835
            var _7L = $geti(_7K, _7J, $f(_7K.length - _7J)); //#835
            $k[$j++] = _7L; //#839
            if (_7L.length < 6) { //#839
                $j -= 2; //#838
                $k[$j++] = 'bwipp.GS1couponTooShortOfferCode'; //#838
                $k[$j++] = "Coupon too short: Truncated Offer Code"; //#838
                $k[$j++] = false; //#838
                return true; //#838
            } //#838
            var _7M = $k[--$j]; //#840
            var _7N = $geti(_7M, 6, _7M.length - 6); //#840
            $k[$j++] = _7N; //#844
            if (_7N.length < 1) { //#844
                $j -= 2; //#843
                $k[$j++] = 'bwipp.GS1couponTooShortSnVLI'; //#843
                $k[$j++] = "Coupon too short: Missing SN VLI"; //#843
                $k[$j++] = false; //#843
                return true; //#843
            } //#843
            var _7O = $k[--$j]; //#845
            var _7P = $geti(_7O, 0, 1); //#845
            $k[$j++] = _7O; //#849
            $k[$j++] = ((~~$z(_7P)) + 6) + 1; //#849
            if ((((~~$z(_7P)) + 6) + 1) > _7O.length) { //#849
                $j -= 3; //#848
                $k[$j++] = 'bwipp.GS1couponTooShortSn'; //#848
                $k[$j++] = "Coupon too short: Truncated SN"; //#848
                $k[$j++] = false; //#848
                return true; //#848
            } //#848
            var _7Q = $k[--$j]; //#850
            var _7R = $k[--$j]; //#850
            var _7S = $geti(_7R, _7Q, $f(_7R.length - _7Q)); //#850
            $k[$j++] = _7S; //#853
            if (_7S.length != 0) { //#853
                $j -= 2; //#852
                $k[$j++] = 'bwipp.GS1couponTooLong'; //#852
                $k[$j++] = "Coupon too long"; //#852
                $k[$j++] = false; //#852
                return true; //#852
            } //#852
            $j--; //#854
        }; //#855
        var _7U = new Map([
            ["cset", 'N'],
            ["min", 18],
            ["max", 18],
            ["check", $a(['lintcsum', 'lintkey'])]
        ]); //#860
        var _7X = new Map([
            ["cset", 'N'],
            ["min", 14],
            ["max", 14],
            ["check", $a(['lintcsum', 'lintkey'])]
        ]); //#866
        var _7Y = $a([_7X]); //#866
        var _7a = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#873
        var _7d = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a(['lintyymmd0'])]
        ]); //#879
        var _7e = $a([_7d]); //#879
        var _7g = new Map([
            ["cset", 'N'],
            ["min", 2],
            ["max", 2],
            ["check", $a([])]
        ]); //#891
        var _7j = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#897
        var _7k = $a([_7j]); //#897
        var _7m = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 28],
            ["check", $a([])]
        ]); //#904
        var _7p = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#910
        var _7q = $a([_7p]); //#910
        var _7s = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 6],
            ["check", $a([])]
        ]); //#917
        var _7v = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#923
        var _7y = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#929
        var _7z = $a([_7y]); //#929
        var _81 = new Map([
            ["cset", 'N'],
            ["min", 13],
            ["max", 13],
            ["check", $a(['lintcsum', 'lintkey'])]
        ]); //#936
        var _83 = new Map([
            ["cset", 'X'],
            ["min", 0],
            ["max", 17],
            ["check", $a([])]
        ]); //#937
        var _86 = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#943
        var _89 = new Map([
            ["cset", 'N'],
            ["min", 13],
            ["max", 13],
            ["check", $a(['lintcsum', 'lintkey'])]
        ]); //#949
        var _8B = new Map([
            ["cset", 'N'],
            ["min", 0],
            ["max", 12],
            ["check", $a([])]
        ]); //#950
        var _8E = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 8],
            ["check", $a([])]
        ]); //#956
        var _8H = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a([])]
        ]); //#962
        var _8I = $a([_8H]); //#962
        var _8K = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 8],
            ["check", $a([])]
        ]); //#1285
        var _8N = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 15],
            ["check", $a([])]
        ]); //#1291
        var _8O = $a([_8N]); //#1291
        var _8Q = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintiso4217'])]
        ]); //#1306
        var _8S = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 15],
            ["check", $a([])]
        ]); //#1307
        var _8T = $a([_8Q, _8S]); //#1307
        var _8V = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 15],
            ["check", $a([])]
        ]); //#1322
        var _8W = $a([_8V]); //#1322
        var _8Y = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintiso4217'])]
        ]); //#1337
        var _8a = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 15],
            ["check", $a([])]
        ]); //#1338
        var _8b = $a([_8Y, _8a]); //#1338
        var _8d = new Map([
            ["cset", 'N'],
            ["min", 4],
            ["max", 4],
            ["check", $a([])]
        ]); //#1353
        var _8e = $a([_8d]); //#1353
        var _8g = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a([])]
        ]); //#1362
        var _8h = $a([_8g]); //#1362
        var _8j = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#1373
        var _8m = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a(['lintkey'])]
        ]); //#1379
        var _8p = new Map([
            ["cset", 'N'],
            ["min", 17],
            ["max", 17],
            ["check", $a(['lintcsum', 'lintkey'])]
        ]); //#1385
        var _8s = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#1391
        var _8v = new Map([
            ["cset", 'N'],
            ["min", 13],
            ["max", 13],
            ["check", $a(['lintcsum', 'lintkey'])]
        ]); //#1397
        var _8w = $a([_8v]); //#1397
        var _8y = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#1410
        var _91 = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintiso3166'])]
        ]); //#1416
        var _93 = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 9],
            ["check", $a([])]
        ]); //#1417
        var _96 = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintiso3166'])]
        ]); //#1423
        var _99 = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 15],
            ["check", $a(['lintiso3166list'])]
        ]); //#1429
        var _9C = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintiso3166'])]
        ]); //#1435
        var _9F = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 15],
            ["check", $a(['lintiso3166list'])]
        ]); //#1441
        var _9I = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintiso3166'])]
        ]); //#1447
        var _9L = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 3],
            ["check", $a([])]
        ]); //#1453
        var _9O = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 35],
            ["check", $a(['lintpcenc'])]
        ]); //#1459
        var _9P = $a([_9O]); //#1459
        var _9R = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 70],
            ["check", $a(['lintpcenc'])]
        ]); //#1466
        var _9S = $a([_9R]); //#1466
        var _9U = new Map([
            ["cset", 'X'],
            ["min", 2],
            ["max", 2],
            ["check", $a(['lintiso3166alpha2'])]
        ]); //#1476
        var _9X = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#1482
        var _9a = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 35],
            ["check", $a(['lintpcenc'])]
        ]); //#1488
        var _9b = $a([_9a]); //#1488
        var _9d = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 70],
            ["check", $a(['lintpcenc'])]
        ]); //#1495
        var _9e = $a([_9d]); //#1495
        var _9g = new Map([
            ["cset", 'X'],
            ["min", 2],
            ["max", 2],
            ["check", $a(['lintiso3166alpha2'])]
        ]); //#1505
        var _9j = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#1511
        var _9m = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#1517
        var _9p = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 35],
            ["check", $a(['lintpcenc'])]
        ]); //#1523
        var _9s = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 1],
            ["check", $a(['lintyesno'])]
        ]); //#1529
        var _9t = $a([_9s]); //#1529
        var _9v = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a(['lintyymmd0'])]
        ]); //#1537
        var _9x = new Map([
            ["cset", 'N'],
            ["min", 4],
            ["max", 4],
            ["check", $a(['linthhmm'])]
        ]); //#1538
        var _9y = $a([_9v, _9x]); //#1538
        var _A0 = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a(['lintyymmdd'])]
        ]); //#1545
        var _A3 = new Map([
            ["cset", 'N'],
            ["min", 13],
            ["max", 13],
            ["check", $a([])]
        ]); //#1551
        var _A6 = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#1557
        var _A9 = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a(['lintyymmdd'])]
        ]); //#1563
        var _AB = new Map([
            ["cset", 'N'],
            ["min", 4],
            ["max", 4],
            ["check", $a(['linthhmm'])]
        ]); //#1564
        var _AE = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 4],
            ["check", $a([])]
        ]); //#1570
        var _AH = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 12],
            ["check", $a([])]
        ]); //#1576
        var _AK = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a(['lintyymmdd'])]
        ]); //#1582
        var _AN = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a(['lintyymmdd'])]
        ]); //#1588
        var _AP = new Map([
            ["cset", 'N'],
            ["min", 0],
            ["max", 6],
            ["check", $a(['lintyymmdd'])]
        ]); //#1589
        var _AS = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 3],
            ["check", $a([])]
        ]); //#1595
        var _AV = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 10],
            ["check", $a([])]
        ]); //#1601
        var _AY = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 2],
            ["check", $a([])]
        ]); //#1607
        var _Ab = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#1613
        var _Ac = $a([_Ab]); //#1613
        var _Ae = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a(['lintkey'])]
        ]); //#1621
        var _Ah = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintiso3166999'])]
        ]); //#1627
        var _Aj = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 27],
            ["check", $a([])]
        ]); //#1628
        var _Ak = $a([_Ah, _Aj]); //#1628
        var _Am = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 1],
            ["check", $a([])]
        ]); //#1643
        var _Ao = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 1],
            ["check", $a([])]
        ]); //#1644
        var _Aq = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 1],
            ["check", $a([])]
        ]); //#1645
        var _As = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 1],
            ["check", $a(['lintimporteridx'])]
        ]); //#1646
        var _Av = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#1652
        var _Aw = $a([_Av]); //#1652
        var _Ay = new Map([
            ["cset", 'X'],
            ["min", 2],
            ["max", 2],
            ["check", $a([])]
        ]); //#1662
        var _B0 = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 28],
            ["check", $a([])]
        ]); //#1663
        var _B1 = $a([_Ay, _B0]); //#1663
        var _B3 = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#1678
        var _B6 = new Map([
            ["cset", 'N'],
            ["min", 4],
            ["max", 4],
            ["check", $a(['lintnonzero'])]
        ]); //#1684
        var _B8 = new Map([
            ["cset", 'N'],
            ["min", 5],
            ["max", 5],
            ["check", $a(['lintnonzero'])]
        ]); //#1685
        var _BA = new Map([
            ["cset", 'N'],
            ["min", 3],
            ["max", 3],
            ["check", $a(['lintnonzero'])]
        ]); //#1686
        var _BC = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 1],
            ["check", $a(['lintwinding'])]
        ]); //#1687
        var _BE = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 1],
            ["check", $a([])]
        ]); //#1688
        var _BH = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#1694
        var _BK = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 1],
            ["check", $a(['lintzero'])]
        ]); //#1700
        var _BM = new Map([
            ["cset", 'N'],
            ["min", 13],
            ["max", 13],
            ["check", $a(['lintcsum', 'lintkey'])]
        ]); //#1701
        var _BO = new Map([
            ["cset", 'X'],
            ["min", 0],
            ["max", 16],
            ["check", $a([])]
        ]); //#1702
        var _BR = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a(['lintkey'])]
        ]); //#1708
        var _BU = new Map([
            ["cset", 'N'],
            ["min", 6],
            ["max", 6],
            ["check", $a([])]
        ]); //#1714
        var _BX = new Map([
            ["cset", 'N'],
            ["min", 14],
            ["max", 14],
            ["check", $a(['lintcsum'])]
        ]); //#1720
        var _BZ = new Map([
            ["cset", 'N'],
            ["min", 4],
            ["max", 4],
            ["check", $a(['lintpieceoftotal'])]
        ]); //#1721
        var _Bc = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 34],
            ["check", $a(['lintiban'])]
        ]); //#1727
        var _Bf = new Map([
            ["cset", 'N'],
            ["min", 8],
            ["max", 8],
            ["check", $a(['lintyymmddhh'])]
        ]); //#1733
        var _Bh = new Map([
            ["cset", 'N'],
            ["min", 0],
            ["max", 4],
            ["check", $a(['lintmmoptss'])]
        ]); //#1734
        var _Bk = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 50],
            ["check", $a([])]
        ]); //#1740
        var _Bn = new Map([
            ["cset", 'C'],
            ["min", 1],
            ["max", 30],
            ["check", $a(['lintkey'])]
        ]); //#1746
        var _Bq = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 12],
            ["check", $a(['lintnozeroprefix'])]
        ]); //#1752
        var _Bt = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 20],
            ["check", $a([])]
        ]); //#1758
        var _Bw = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 25],
            ["check", $a(['lintcsumalpha', 'lintkey'])]
        ]); //#1764
        var _Bz = new Map([
            ["cset", 'N'],
            ["min", 18],
            ["max", 18],
            ["check", $a(['lintcsum'])]
        ]); //#1770
        var _C0 = $a([_Bz]); //#1770
        var _C2 = new Map([
            ["cset", 'N'],
            ["min", 1],
            ["max", 10],
            ["check", $a([])]
        ]); //#1777
        var _C5 = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 25],
            ["check", $a([])]
        ]); //#1783
        var _C8 = new Map([
            ["cset", 'N'],
            ["min", 14],
            ["max", 14],
            ["check", $a(['lintcsum'])]
        ]); //#1789
        var _CA = new Map([
            ["cset", 'N'],
            ["min", 4],
            ["max", 4],
            ["check", $a(['lintpieceoftotal'])]
        ]); //#1790
        var _CD = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 70],
            ["check", $a(['lintcouponcode'])]
        ]); //#1796
        var _CG = new Map([
            ["cset", 'N'],
            ["min", 4],
            ["max", 4],
            ["check", $a([])]
        ]); //#1802
        var _CJ = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 70],
            ["check", $a(['lintcouponposoffer'])]
        ]); //#1808
        var _CM = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 70],
            ["check", $a([])]
        ]); //#1814
        var _CP = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 30],
            ["check", $a([])]
        ]); //#1820
        var _CS = new Map([
            ["cset", 'X'],
            ["min", 1],
            ["max", 90],
            ["check", $a([])]
        ]); //#1826
        var _CT = $a([_CS]); //#1826
        var _CU = new Map([
            ["00", $a([_7U])],
            ["01", _7Y],
            ["02", _7Y],
            ["10", $a([_7a])],
            ["11", _7e],
            ["12", _7e],
            ["13", _7e],
            ["14", _7e],
            ["15", _7e],
            ["16", _7e],
            ["17", _7e],
            ["20", $a([_7g])],
            ["21", _7k],
            ["22", _7k],
            ["235", $a([_7m])],
            ["240", _7q],
            ["241", _7q],
            ["242", $a([_7s])],
            ["243", $a([_7v])],
            ["250", _7z],
            ["251", _7z],
            ["253", $a([_81, _83])],
            ["254", $a([_86])],
            ["255", $a([_89, _8B])],
            ["30", $a([_8E])],
            ["3100", _8I],
            ["3101", _8I],
            ["3102", _8I],
            ["3103", _8I],
            ["3104", _8I],
            ["3105", _8I],
            ["3110", _8I],
            ["3111", _8I],
            ["3112", _8I],
            ["3113", _8I],
            ["3114", _8I],
            ["3115", _8I],
            ["3120", _8I],
            ["3121", _8I],
            ["3122", _8I],
            ["3123", _8I],
            ["3124", _8I],
            ["3125", _8I],
            ["3130", _8I],
            ["3131", _8I],
            ["3132", _8I],
            ["3133", _8I],
            ["3134", _8I],
            ["3135", _8I],
            ["3140", _8I],
            ["3141", _8I],
            ["3142", _8I],
            ["3143", _8I],
            ["3144", _8I],
            ["3145", _8I],
            ["3150", _8I],
            ["3151", _8I],
            ["3152", _8I],
            ["3153", _8I],
            ["3154", _8I],
            ["3155", _8I],
            ["3160", _8I],
            ["3161", _8I],
            ["3162", _8I],
            ["3163", _8I],
            ["3164", _8I],
            ["3165", _8I],
            ["3200", _8I],
            ["3201", _8I],
            ["3202", _8I],
            ["3203", _8I],
            ["3204", _8I],
            ["3205", _8I],
            ["3210", _8I],
            ["3211", _8I],
            ["3212", _8I],
            ["3213", _8I],
            ["3214", _8I],
            ["3215", _8I],
            ["3220", _8I],
            ["3221", _8I],
            ["3222", _8I],
            ["3223", _8I],
            ["3224", _8I],
            ["3225", _8I],
            ["3230", _8I],
            ["3231", _8I],
            ["3232", _8I],
            ["3233", _8I],
            ["3234", _8I],
            ["3235", _8I],
            ["3240", _8I],
            ["3241", _8I],
            ["3242", _8I],
            ["3243", _8I],
            ["3244", _8I],
            ["3245", _8I],
            ["3250", _8I],
            ["3251", _8I],
            ["3252", _8I],
            ["3253", _8I],
            ["3254", _8I],
            ["3255", _8I],
            ["3260", _8I],
            ["3261", _8I],
            ["3262", _8I],
            ["3263", _8I],
            ["3264", _8I],
            ["3265", _8I],
            ["3270", _8I],
            ["3271", _8I],
            ["3272", _8I],
            ["3273", _8I],
            ["3274", _8I],
            ["3275", _8I],
            ["3280", _8I],
            ["3281", _8I],
            ["3282", _8I],
            ["3283", _8I],
            ["3284", _8I],
            ["3285", _8I],
            ["3290", _8I],
            ["3291", _8I],
            ["3292", _8I],
            ["3293", _8I],
            ["3294", _8I],
            ["3295", _8I],
            ["3300", _8I],
            ["3301", _8I],
            ["3302", _8I],
            ["3303", _8I],
            ["3304", _8I],
            ["3305", _8I],
            ["3310", _8I],
            ["3311", _8I],
            ["3312", _8I],
            ["3313", _8I],
            ["3314", _8I],
            ["3315", _8I],
            ["3320", _8I],
            ["3321", _8I],
            ["3322", _8I],
            ["3323", _8I],
            ["3324", _8I],
            ["3325", _8I],
            ["3330", _8I],
            ["3331", _8I],
            ["3332", _8I],
            ["3333", _8I],
            ["3334", _8I],
            ["3335", _8I],
            ["3340", _8I],
            ["3341", _8I],
            ["3342", _8I],
            ["3343", _8I],
            ["3344", _8I],
            ["3345", _8I],
            ["3350", _8I],
            ["3351", _8I],
            ["3352", _8I],
            ["3353", _8I],
            ["3354", _8I],
            ["3355", _8I],
            ["3360", _8I],
            ["3361", _8I],
            ["3362", _8I],
            ["3363", _8I],
            ["3364", _8I],
            ["3365", _8I],
            ["3370", _8I],
            ["3371", _8I],
            ["3372", _8I],
            ["3373", _8I],
            ["3374", _8I],
            ["3375", _8I],
            ["3400", _8I],
            ["3401", _8I],
            ["3402", _8I],
            ["3403", _8I],
            ["3404", _8I],
            ["3405", _8I],
            ["3410", _8I],
            ["3411", _8I],
            ["3412", _8I],
            ["3413", _8I],
            ["3414", _8I],
            ["3415", _8I],
            ["3420", _8I],
            ["3421", _8I],
            ["3422", _8I],
            ["3423", _8I],
            ["3424", _8I],
            ["3425", _8I],
            ["3430", _8I],
            ["3431", _8I],
            ["3432", _8I],
            ["3433", _8I],
            ["3434", _8I],
            ["3435", _8I],
            ["3440", _8I],
            ["3441", _8I],
            ["3442", _8I],
            ["3443", _8I],
            ["3444", _8I],
            ["3445", _8I],
            ["3450", _8I],
            ["3451", _8I],
            ["3452", _8I],
            ["3453", _8I],
            ["3454", _8I],
            ["3455", _8I],
            ["3460", _8I],
            ["3461", _8I],
            ["3462", _8I],
            ["3463", _8I],
            ["3464", _8I],
            ["3465", _8I],
            ["3470", _8I],
            ["3471", _8I],
            ["3472", _8I],
            ["3473", _8I],
            ["3474", _8I],
            ["3475", _8I],
            ["3480", _8I],
            ["3481", _8I],
            ["3482", _8I],
            ["3483", _8I],
            ["3484", _8I],
            ["3485", _8I],
            ["3490", _8I],
            ["3491", _8I],
            ["3492", _8I],
            ["3493", _8I],
            ["3494", _8I],
            ["3495", _8I],
            ["3500", _8I],
            ["3501", _8I],
            ["3502", _8I],
            ["3503", _8I],
            ["3504", _8I],
            ["3505", _8I],
            ["3510", _8I],
            ["3511", _8I],
            ["3512", _8I],
            ["3513", _8I],
            ["3514", _8I],
            ["3515", _8I],
            ["3520", _8I],
            ["3521", _8I],
            ["3522", _8I],
            ["3523", _8I],
            ["3524", _8I],
            ["3525", _8I],
            ["3530", _8I],
            ["3531", _8I],
            ["3532", _8I],
            ["3533", _8I],
            ["3534", _8I],
            ["3535", _8I],
            ["3540", _8I],
            ["3541", _8I],
            ["3542", _8I],
            ["3543", _8I],
            ["3544", _8I],
            ["3545", _8I],
            ["3550", _8I],
            ["3551", _8I],
            ["3552", _8I],
            ["3553", _8I],
            ["3554", _8I],
            ["3555", _8I],
            ["3560", _8I],
            ["3561", _8I],
            ["3562", _8I],
            ["3563", _8I],
            ["3564", _8I],
            ["3565", _8I],
            ["3570", _8I],
            ["3571", _8I],
            ["3572", _8I],
            ["3573", _8I],
            ["3574", _8I],
            ["3575", _8I],
            ["3600", _8I],
            ["3601", _8I],
            ["3602", _8I],
            ["3603", _8I],
            ["3604", _8I],
            ["3605", _8I],
            ["3610", _8I],
            ["3611", _8I],
            ["3612", _8I],
            ["3613", _8I],
            ["3614", _8I],
            ["3615", _8I],
            ["3620", _8I],
            ["3621", _8I],
            ["3622", _8I],
            ["3623", _8I],
            ["3624", _8I],
            ["3625", _8I],
            ["3630", _8I],
            ["3631", _8I],
            ["3632", _8I],
            ["3633", _8I],
            ["3634", _8I],
            ["3635", _8I],
            ["3640", _8I],
            ["3641", _8I],
            ["3642", _8I],
            ["3643", _8I],
            ["3644", _8I],
            ["3645", _8I],
            ["3650", _8I],
            ["3651", _8I],
            ["3652", _8I],
            ["3653", _8I],
            ["3654", _8I],
            ["3655", _8I],
            ["3660", _8I],
            ["3661", _8I],
            ["3662", _8I],
            ["3663", _8I],
            ["3664", _8I],
            ["3665", _8I],
            ["3670", _8I],
            ["3671", _8I],
            ["3672", _8I],
            ["3673", _8I],
            ["3674", _8I],
            ["3675", _8I],
            ["3680", _8I],
            ["3681", _8I],
            ["3682", _8I],
            ["3683", _8I],
            ["3684", _8I],
            ["3685", _8I],
            ["3690", _8I],
            ["3691", _8I],
            ["3692", _8I],
            ["3693", _8I],
            ["3694", _8I],
            ["3695", _8I],
            ["37", $a([_8K])],
            ["3900", _8O],
            ["3901", _8O],
            ["3902", _8O],
            ["3903", _8O],
            ["3904", _8O],
            ["3905", _8O],
            ["3906", _8O],
            ["3907", _8O],
            ["3908", _8O],
            ["3909", _8O],
            ["3910", _8T],
            ["3911", _8T],
            ["3912", _8T],
            ["3913", _8T],
            ["3914", _8T],
            ["3915", _8T],
            ["3916", _8T],
            ["3917", _8T],
            ["3918", _8T],
            ["3919", _8T],
            ["3920", _8W],
            ["3921", _8W],
            ["3922", _8W],
            ["3923", _8W],
            ["3924", _8W],
            ["3925", _8W],
            ["3926", _8W],
            ["3927", _8W],
            ["3928", _8W],
            ["3929", _8W],
            ["3930", _8b],
            ["3931", _8b],
            ["3932", _8b],
            ["3933", _8b],
            ["3934", _8b],
            ["3935", _8b],
            ["3936", _8b],
            ["3937", _8b],
            ["3938", _8b],
            ["3939", _8b],
            ["3940", _8e],
            ["3941", _8e],
            ["3942", _8e],
            ["3943", _8e],
            ["3950", _8h],
            ["3951", _8h],
            ["3952", _8h],
            ["3953", _8h],
            ["3954", _8h],
            ["3955", _8h],
            ["400", $a([_8j])],
            ["401", $a([_8m])],
            ["402", $a([_8p])],
            ["403", $a([_8s])],
            ["410", _8w],
            ["411", _8w],
            ["412", _8w],
            ["413", _8w],
            ["414", _8w],
            ["415", _8w],
            ["416", _8w],
            ["417", _8w],
            ["420", $a([_8y])],
            ["421", $a([_91, _93])],
            ["422", $a([_96])],
            ["423", $a([_99])],
            ["424", $a([_9C])],
            ["425", $a([_9F])],
            ["426", $a([_9I])],
            ["427", $a([_9L])],
            ["4300", _9P],
            ["4301", _9P],
            ["4302", _9S],
            ["4303", _9S],
            ["4304", _9S],
            ["4305", _9S],
            ["4306", _9S],
            ["4307", $a([_9U])],
            ["4308", $a([_9X])],
            ["4310", _9b],
            ["4311", _9b],
            ["4312", _9e],
            ["4313", _9e],
            ["4314", _9e],
            ["4315", _9e],
            ["4316", _9e],
            ["4317", $a([_9g])],
            ["4318", $a([_9j])],
            ["4319", $a([_9m])],
            ["4320", $a([_9p])],
            ["4321", _9t],
            ["4322", _9t],
            ["4323", _9t],
            ["4324", _9y],
            ["4325", _9y],
            ["4326", $a([_A0])],
            ["7001", $a([_A3])],
            ["7002", $a([_A6])],
            ["7003", $a([_A9, _AB])],
            ["7004", $a([_AE])],
            ["7005", $a([_AH])],
            ["7006", $a([_AK])],
            ["7007", $a([_AN, _AP])],
            ["7008", $a([_AS])],
            ["7009", $a([_AV])],
            ["7010", $a([_AY])],
            ["7020", _Ac],
            ["7021", _Ac],
            ["7022", _Ac],
            ["7023", $a([_Ae])],
            ["7030", _Ak],
            ["7031", _Ak],
            ["7032", _Ak],
            ["7033", _Ak],
            ["7034", _Ak],
            ["7035", _Ak],
            ["7036", _Ak],
            ["7037", _Ak],
            ["7038", _Ak],
            ["7039", _Ak],
            ["7040", $a([_Am, _Ao, _Aq, _As])],
            ["710", _Aw],
            ["711", _Aw],
            ["712", _Aw],
            ["713", _Aw],
            ["714", _Aw],
            ["7230", _B1],
            ["7231", _B1],
            ["7232", _B1],
            ["7233", _B1],
            ["7234", _B1],
            ["7235", _B1],
            ["7236", _B1],
            ["7237", _B1],
            ["7238", _B1],
            ["7239", _B1],
            ["7240", $a([_B3])],
            ["8001", $a([_B6, _B8, _BA, _BC, _BE])],
            ["8002", $a([_BH])],
            ["8003", $a([_BK, _BM, _BO])],
            ["8004", $a([_BR])],
            ["8005", $a([_BU])],
            ["8006", $a([_BX, _BZ])],
            ["8007", $a([_Bc])],
            ["8008", $a([_Bf, _Bh])],
            ["8009", $a([_Bk])],
            ["8010", $a([_Bn])],
            ["8011", $a([_Bq])],
            ["8012", $a([_Bt])],
            ["8013", $a([_Bw])],
            ["8017", _C0],
            ["8018", _C0],
            ["8019", $a([_C2])],
            ["8020", $a([_C5])],
            ["8026", $a([_C8, _CA])],
            ["8110", $a([_CD])],
            ["8111", $a([_CG])],
            ["8112", $a([_CJ])],
            ["8200", $a([_CM])],
            ["90", $a([_CP])],
            ["91", _CT],
            ["92", _CT],
            ["93", _CT],
            ["94", _CT],
            ["95", _CT],
            ["96", _CT],
            ["97", _CT],
            ["98", _CT],
            ["99", _CT]
        ]); //#1837
        $1.gs1syntax = _CU; //#1839
        $k[$j++] = true; //#1866
        for (var _CX = 0, _CW = $1.vals.length - 1; _CX <= _CW; _CX += 1) { //#1866
            $1.ai = $get($1.ais, _CX); //#1843
            $1.val = $get($1.vals, _CX); //#1844
            var _Ce = $get($1.gs1syntax, $1.ai) !== undefined; //#1845
            if (_Ce) { //#1864
                $forall($get($1.gs1syntax, $1.ai), function() { //#1858
                    $1.props = $k[--$j]; //#1847
                    var _Ck = $get($1.props, 'max'); //#1848
                    var _Cl = $1.val; //#1848
                    var _Cm = _Cl.length; //#1848
                    if (_Ck > _Cl.length) { //#1848
                        var _ = _Cm; //#1848
                        _Cm = _Ck; //#1848
                        _Ck = _; //#1848
                    } //#1848
                    $1.eval = $geti($1.val, 0, _Ck); //#1849
                    var _Cr = $1.eval.length; //#1850
                    $1.val = $geti($1.val, _Cr, $1.val.length - _Cr); //#1850
                    if ($1.eval.length < $get($1.props, 'min')) { //#1853
                        $j--; //#1852
                        $k[$j++] = 'bwipp.GS1valueTooShort'; //#1852
                        $k[$j++] = "Too short"; //#1852
                        $k[$j++] = false; //#1852
                        return true; //#1852
                    } //#1852
                    var _Cx = new Map([
                        ["N", 'lintnumeric'],
                        ["X", 'lintcset82'],
                        ["C", 'lintcset39']
                    ]); //#1854
                    $k[$j++] = $1.eval; //#1854
                    if ($1[$get(_Cx, $get($1.props, 'cset'))]() === true) {
                        return true;
                    } //#1854
                    if ($1.eval.length > 0) { //#1857
                        $forall($get($1.props, 'check'), function() { //#1856
                            var _D8 = $1[$k[--$j]]; //#1856
                            $k[$j++] = $1.eval; //#1856
                            if (_D8() === true) {
                                return true;
                            } //#1856
                        }); //#1856
                    } //#1856
                }); //#1856
                var _D9 = $k[--$j]; //#1859
                $k[$j++] = _D9; //#1859
                if ($nt(_D9)) { //#1859
                    break; //#1859
                } //#1859
                if ($1.val.length != 0) { //#1862
                    $j--; //#1861
                    $k[$j++] = 'bwipp.GS1valueTooLong'; //#1861
                    $k[$j++] = "Too long"; //#1861
                    $k[$j++] = false; //#1861
                    break; //#1861
                } //#1861
            } else { //#1864
                $j--; //#1864
                $k[$j++] = 'bwipp.GS1unknownAI'; //#1864
                $k[$j++] = "Unrecognised AI"; //#1864
                $k[$j++] = false; //#1864
                break; //#1864
            } //#1864
        } //#1864
        if ($nt($k[--$j])) { //#1874
            var _DC = $k[--$j]; //#1868
            var _DE = $s((_DC.length + $1.ai.length) + 5); //#1868
            $puti(_DE, 0, "AI "); //#1869
            $puti(_DE, 3, $1.ai); //#1870
            $puti(_DE, 3 + $1.ai.length, ": "); //#1871
            $puti(_DE, 5 + $1.ai.length, _DC); //#1872
            $k[$j++] = _DE; //#1873
            $0.raiseerror(); //#1873
        } //#1873
        $k[$j++] = true; //#1877
    };
    $0.renmatrix = function() {
        if ($0.bwipjs_dontdraw) { //#2145
            return; //#2145
        } //#2145
        var $1 = {}; //#2147
        $1.args = $k[--$j]; //#2149
        $1.width = 1; //#2152
        $1.height = 1; //#2153
        $1.barcolor = "unset"; //#2154
        $1.backgroundcolor = "unset"; //#2155
        $1.colormap = "unset"; //#2156
        $1.dotty = false; //#2157
        $1.inkspread = 0; //#2158
        $1.inkspreadh = 0; //#2159
        $1.inkspreadv = 0; //#2160
        $1.includetext = false; //#2161
        $1.txt = $a([]); //#2162
        $1.textcolor = "unset"; //#2163
        $1.textxalign = "unset"; //#2164
        $1.textyalign = "unset"; //#2165
        $1.textfont = "OCR-B"; //#2166
        $1.textsize = 10; //#2167
        $1.textxoffset = 0; //#2168
        $1.textyoffset = 0; //#2169
        $1.textgaps = 0; //#2170
        $1.alttext = ""; //#2171
        $forall($1.args, function() { //#2174
            var _4 = $k[--$j]; //#2174
            $1[$k[--$j]] = _4; //#2174
        }); //#2174
        var _6 = $1.opt; //#2175
        for (var _B = _6.size, _A = _6.keys(), _9 = 0; _9 < _B; _9++) { //#2175
            var _7 = _A.next().value; //#2175
            $1[_7] = _6.get(_7); //#2175
        } //#2175
        $1.width = +$1.width; //#2177
        $1.height = +$1.height; //#2178
        $1.barcolor = "" + $1.barcolor; //#2179
        $1.backgroundcolor = "" + $1.backgroundcolor; //#2180
        $1.inkspread = +$1.inkspread; //#2181
        $1.inkspreadh = +$1.inkspreadh; //#2182
        $1.inkspreadv = +$1.inkspreadv; //#2183
        $1.textcolor = "" + $1.textcolor; //#2184
        $1.textxalign = "" + $1.textxalign; //#2185
        $1.textyalign = "" + $1.textyalign; //#2186
        $1.textfont = "" + $1.textfont; //#2187
        $1.textsize = +$1.textsize; //#2188
        $1.textxoffset = +$1.textxoffset; //#2189
        $1.textyoffset = +$1.textyoffset; //#2190
        $1.textgaps = +$1.textgaps; //#2191
        $1.alttext = "" + $1.alttext; //#2192
        if ($1.inkspread != 0) { //#2194
            $1.inkspreadh = $1.inkspread; //#2194
        } //#2194
        if ($1.inkspread != 0) { //#2195
            $1.inkspreadv = $1.inkspread; //#2195
        } //#2195
        $1.xyget = function() {
            var _X = $k[--$j]; //#2197
            var _a = $get($1.pixs, $f($k[--$j] + (_X * $1.pixx))); //#2197
            $k[$j++] = _a; //#2197
        }; //#2197
        $1.cget = function() {
            var _c = $k[--$j]; //#2198
            var _f = $get($1.cache, $f($k[--$j] + (_c * $1.pixx))); //#2198
            var _g = $k[--$j]; //#2198
            $k[$j++] = $an(_g, _f); //#2198
        }; //#2198
        $1.cput = function() {
            var _h = $k[--$j]; //#2200
            $k[$j++] = _h; //#2204
            if ((_h % 4) == 0) { //#2203
                var _i = $k[--$j]; //#2201
                var _j = $k[--$j]; //#2201
                var _k = $k[--$j]; //#2201
                var _l = $1.pixx; //#2201
                var _m = $1.cache; //#2201
                $put(_m, $f(_k + (_j * _l)), $or($get(_m, $f(_k + (_j * _l))), _i)); //#2201
            } else { //#2203
                $j -= 3; //#2203
            } //#2203
        }; //#2205
        $1.abcd = function() {
            $k[$j++] = $s(4); //#2212
            $k[$j++] = 0; //#2212
            $k[$j++] = Infinity; //#2212
            var _p = $k[--$j]; //#2209
            var _q = $k[--$j]; //#2209
            var _r = $k[--$j]; //#2209
            var _s = $k[--$j]; //#2209
            var _v = $f($k[--$j] + (_s * $1.pixx)); //#2210
            $k[$j++] = _r; //#2211
            $k[$j++] = _q; //#2211
            $k[$j++] = _p; //#2211
            $k[$j++] = _v; //#2211
            $aload($geti($1.pixs, _v, 2)); //#2211
            var _y = $k[--$j]; //#2211
            var _z = $k[--$j]; //#2211
            var _13 = $geti($1.pixs, $f($k[--$j] + $1.pixx), 2); //#2212
            $k[$j++] = _z; //#2212
            $k[$j++] = _y; //#2212
            $aload(_13); //#2212
            var _14 = $a(); //#2212
            for (var _15 = 0, _16 = _14.length; _15 < _16; _15++) { //#2213
                var _18 = $k[--$j]; //#2213
                var _19 = $k[--$j]; //#2213
                $put(_19, _18, $f($get(_14, _15) + 48)); //#2213
                $k[$j++] = _19; //#2213
                $k[$j++] = $f(_18 + 1); //#2213
            } //#2213
            $j--; //#2213
        }; //#2214
        $1.right = function() {
            if ($1.dir != 1) { //#2216
                $k[$j++] = $1.x; //#2216
                $k[$j++] = $1.y; //#2216
                $k[$j++] = $1.dir; //#2216
                $1.cput(); //#2216
                $k[$j++] = $a([$1.x, $1.y]); //#2216
            } //#2216
            $1.x = $1.x + 1; //#2216
            $1.dir = 1; //#2216
        }; //#2216
        $1.down = function() {
            if ($1.dir != 2) { //#2217
                $k[$j++] = $1.x; //#2217
                $k[$j++] = $1.y; //#2217
                $k[$j++] = $1.dir; //#2217
                $1.cput(); //#2217
                $k[$j++] = $a([$1.x, $1.y]); //#2217
            } //#2217
            $1.y = $1.y + 1; //#2217
            $1.dir = 2; //#2217
        }; //#2217
        $1.left = function() {
            if ($1.dir != 4) { //#2218
                $k[$j++] = $1.x; //#2218
                $k[$j++] = $1.y; //#2218
                $k[$j++] = $1.dir; //#2218
                $1.cput(); //#2218
                $k[$j++] = $a([$1.x, $1.y]); //#2218
            } //#2218
            $1.x = $1.x - 1; //#2218
            $1.dir = 4; //#2218
        }; //#2218
        $1.up = function() {
            if ($1.dir != 8) { //#2219
                $k[$j++] = $1.x; //#2219
                $k[$j++] = $1.y; //#2219
                $k[$j++] = $1.dir; //#2219
                $1.cput(); //#2219
                $k[$j++] = $a([$1.x, $1.y]); //#2219
            } //#2219
            $1.y = $1.y - 1; //#2219
            $1.dir = 8; //#2219
        }; //#2219
        $1.trace = function() {
            $1.y = $k[--$j]; //#2223
            $1.x = $k[--$j]; //#2223
            $k[$j++] = 'dir'; //#2225
            $k[$j++] = $f($1.x + 1); //#2225
            $k[$j++] = $f($1.y + 1); //#2225
            $1.xyget(); //#2225
            var _1l = ($k[--$j] == 1) ? 8 : 4; //#2225
            $1[$k[--$j]] = _1l; //#2225
            $1.sx = $1.x; //#2226
            $1.sy = $1.y; //#2226
            $1.sdir = $1.dir; //#2226
            $k[$j++] = Infinity; //#2230
            for (;;) { //#2244
                $k[$j++] = $1.x; //#2231
                $k[$j++] = $1.y; //#2231
                $1.abcd(); //#2231
                for (var _1s = 0, _1t = 1; _1s < _1t; _1s++) { //#2242
                    var _1u = $k[--$j]; //#2233
                    $k[$j++] = _1u; //#2233
                    if ($eq(_1u, "0001") || ($eq(_1u, "0011") || $eq(_1u, "1011"))) { //#2233
                        $j--; //#2233
                        $1.right(); //#2233
                        break; //#2233
                    } //#2233
                    var _1v = $k[--$j]; //#2234
                    $k[$j++] = _1v; //#2234
                    if ($eq(_1v, "0010") || ($eq(_1v, "1010") || $eq(_1v, "1110"))) { //#2234
                        $j--; //#2234
                        $1.down(); //#2234
                        break; //#2234
                    } //#2234
                    var _1w = $k[--$j]; //#2235
                    $k[$j++] = _1w; //#2235
                    if ($eq(_1w, "1000") || ($eq(_1w, "1100") || $eq(_1w, "1101"))) { //#2235
                        $j--; //#2235
                        $1.left(); //#2235
                        break; //#2235
                    } //#2235
                    var _1x = $k[--$j]; //#2236
                    $k[$j++] = _1x; //#2236
                    if ($eq(_1x, "0100") || ($eq(_1x, "0101") || $eq(_1x, "0111"))) { //#2236
                        $j--; //#2236
                        $1.up(); //#2236
                        break; //#2236
                    } //#2236
                    var _1y = $k[--$j]; //#2237
                    $k[$j++] = _1y; //#2241
                    if ($eq(_1y, "1001")) { //#2240
                        if ($1.dir == 2) { //#2238
                            $j--; //#2238
                            $1.left(); //#2238
                            break; //#2238
                        } else { //#2238
                            $j--; //#2238
                            $1.right(); //#2238
                            break; //#2238
                        } //#2238
                    } else { //#2240
                        if ($1.dir == 1) { //#2240
                            $j--; //#2240
                            $1.down(); //#2240
                            break; //#2240
                        } else { //#2240
                            $j--; //#2240
                            $1.up(); //#2240
                            break; //#2240
                        } //#2240
                    } //#2240
                } //#2240
                if (($eq($1.x, $1.sx) && $eq($1.y, $1.sy)) && ($1.dir == $1.sdir)) { //#2243
                    break; //#2243
                } //#2243
            } //#2243
            $astore($a($counttomark())); //#2245
            var _29 = $k[--$j]; //#2245
            var _2A = $k[--$j]; //#2245
            $k[$j++] = _29; //#2245
            $k[$j++] = _2A; //#2245
            $j--; //#2245
        }; //#2247
        $1.drawlayer = function() {
            $1.pixsorig = $1.pixs; //#2263
            $1.pixs = $k[--$j]; //#2264
            $k[$j++] = Infinity; //#2274
            for (var _2E = 0, _2F = $1.pixx + 2; _2E < _2F; _2E++) { //#2268
                $k[$j++] = 0; //#2268
            } //#2268
            for (var _2J = 0, _2K = $1.pixx, _2I = $1.pixs.length - 1; _2K < 0 ? _2J >= _2I : _2J <= _2I; _2J += _2K) { //#2273
                $k[$j++] = 0; //#2271
                $aload($geti($1.pixs, _2J, $1.pixx)); //#2271
                $k[$j++] = 0; //#2272
            } //#2272
            for (var _2P = 0, _2Q = $1.pixx + 2; _2P < _2Q; _2P++) { //#2274
                $k[$j++] = 0; //#2274
            } //#2274
            $1.pixs = $a(); //#2274
            $1.pixx = $1.pixx + 2; //#2276
            $1.pixy = $1.pixy + 2; //#2277
            $k[$j++] = Infinity; //#2280
            for (var _2V = 0, _2W = $1.pixs.length; _2V < _2W; _2V++) { //#2280
                $k[$j++] = 0; //#2280
            } //#2280
            $1.cache = $a(); //#2280
            $k[$j++] = Infinity; //#2296
            for (var _2a = 0, _2Z = $1.pixy - 2; _2a <= _2Z; _2a += 1) { //#2300
                $1.j = _2a; //#2285
                for (var _2d = 0, _2c = $1.pixx - 2; _2d <= _2c; _2d += 1) { //#2299
                    $1.i = _2d; //#2287
                    $k[$j++] = 'k'; //#2288
                    $k[$j++] = $1.i; //#2288
                    $k[$j++] = $1.j; //#2288
                    $1.abcd(); //#2288
                    var _2g = $k[--$j]; //#2288
                    $1[$k[--$j]] = _2g; //#2288
                    if ($eq($1.k, "0001") || $eq($1.k, "1001")) { //#2293
                        $k[$j++] = 8; //#2290
                        $k[$j++] = $1.i; //#2290
                        $k[$j++] = $1.j; //#2290
                        $1.cget(); //#2290
                        if ($k[--$j] == 0) { //#2292
                            $k[$j++] = $1.i; //#2291
                            $k[$j++] = $1.j; //#2291
                            $1.trace(); //#2291
                        } //#2291
                    } //#2291
                    if ($eq($1.k, "1110")) { //#2298
                        $k[$j++] = 4; //#2295
                        $k[$j++] = $1.i; //#2295
                        $k[$j++] = $1.j; //#2295
                        $1.cget(); //#2295
                        if ($k[--$j] == 0) { //#2297
                            $k[$j++] = $1.i; //#2296
                            $k[$j++] = $1.j; //#2296
                            $1.trace(); //#2296
                        } //#2296
                    } //#2296
                } //#2296
            } //#2296
            $1.paths = $a(); //#2296
            $1.pixx = $1.pixx - 2; //#2304
            $1.pixy = $1.pixy - 2; //#2305
            $$.newpath(); //#2308
            var _2y = $1.paths; //#2309
            for (var _2z = 0, _30 = _2y.length; _2z < _30; _2z++) { //#2327
                $1.p = $get(_2y, _2z); //#2310
                $1.len = $1.p.length; //#2311
                $aload($get($1.p, $1.len - 1)); //#2312
                $aload($get($1.p, 0)); //#2313
                for (var _3A = 0, _39 = $1.len - 1; _3A <= _39; _3A += 1) { //#2324
                    $1.i = _3A; //#2315
                    $aload($get($1.p, ($1.i + 1) % $1.len)); //#2316
                    var _3F = $k[--$j]; //#2316
                    var _3G = $k[--$j]; //#2316
                    var _3H = $k[--$j]; //#2316
                    var _3I = $k[--$j]; //#2316
                    var _3J = $k[--$j]; //#2316
                    var _3K = $k[--$j]; //#2316
                    $k[$j++] = _3I; //#2318
                    $k[$j++] = _3H; //#2318
                    $k[$j++] = _3G; //#2318
                    $k[$j++] = _3F; //#2318
                    $k[$j++] = _3K; //#2318
                    $k[$j++] = _3I; //#2318
                    $k[$j++] = $1.inkspreadh; //#2318
                    if ($lt(_3F, _3J)) { //#2318
                        var _3M = $k[--$j]; //#2318
                        var _3N = $k[--$j]; //#2318
                        $k[$j++] = $f(_3N + _3M); //#2318
                    } else { //#2318
                        var _3O = $k[--$j]; //#2318
                        var _3P = $k[--$j]; //#2318
                        $k[$j++] = $f(_3P - _3O); //#2318
                    } //#2318
                    var _3Q = $k[--$j]; //#2319
                    var _3R = $k[--$j]; //#2319
                    var _3S = $k[--$j]; //#2319
                    var _3T = $k[--$j]; //#2319
                    var _3U = $k[--$j]; //#2320
                    $k[$j++] = _3U; //#2321
                    $k[$j++] = _3Q; //#2321
                    $k[$j++] = _3T; //#2321
                    $k[$j++] = _3S; //#2321
                    $k[$j++] = _3U; //#2321
                    $k[$j++] = $1.inkspreadv; //#2321
                    if ($gt(_3T, _3R)) { //#2321
                        var _3W = $k[--$j]; //#2321
                        var _3X = $k[--$j]; //#2321
                        $k[$j++] = $f(_3X + _3W); //#2321
                    } else { //#2321
                        var _3Y = $k[--$j]; //#2321
                        var _3Z = $k[--$j]; //#2321
                        $k[$j++] = $f(_3Z - _3Y); //#2321
                    } //#2321
                    var _3a = $k[--$j]; //#2322
                    var _3b = $k[--$j]; //#2322
                    var _3c = $k[--$j]; //#2322
                    var _3d = $k[--$j]; //#2322
                    $k[$j++] = _3c; //#2323
                    $k[$j++] = _3b; //#2323
                    $k[$j++] = _3d; //#2323
                    $k[$j++] = $f($1.pixy - _3a); //#2323
                    if ($1.i == 0) { //#2323
                        var _3g = $k[--$j]; //#2323
                        $$.moveto($k[--$j], _3g); //#2323
                    } else { //#2323
                        var _3i = $k[--$j]; //#2323
                        $$.lineto($k[--$j], _3i); //#2323
                    } //#2323
                } //#2323
                $$.closepath(); //#2325
                $j -= 4; //#2326
            } //#2326
            $$.fill(); //#2328
            $1.pixs = $1.pixsorig; //#2330
        }; //#2332
        $1.drawlayerdots = function() {
            $1.pixsorig = $1.pixs; //#2336
            $1.pixs = $k[--$j]; //#2337
            $$.newpath(); //#2339
            for (var _3p = 0, _3o = $1.pixs.length - 1; _3p <= _3o; _3p += 1) { //#2347
                $1.x = _3p % $1.pixx; //#2341
                $1.y = ~~(_3p / $1.pixx); //#2342
                $k[$j++] = $1.x; //#2343
                $k[$j++] = $1.y; //#2343
                $1.xyget(); //#2343
                if ($k[--$j] == 1) { //#2346
                    $$.moveto($f($1.x + 0.5), $f(($1.pixy - $1.y) - 0.5)); //#2344
                    $$.arc($f($1.x + 0.5), $f(($1.pixy - $1.y) - 0.5), $f(0.5 - $1.inkspread), 0, 360, 1); //#2345
                } //#2345
            } //#2345
            $$.fill(); //#2348
            $1.pixs = $1.pixsorig; //#2350
        }; //#2352
        $$.save(); //#2354
        $1.inkspread = $1.inkspread / 2; //#2357
        $1.inkspreadh = $1.inkspreadh / 2; //#2358
        $1.inkspreadv = $1.inkspreadv / 2; //#2359
        var _46 = $$.currpos(); //#2360
        $$.translate(_46.x, _46.y); //#2360
        $$.scale(($1.width / $1.pixx) * 72, ($1.height / $1.pixy) * 72); //#2361
        $$.moveto(0, 0); //#2362
        $$.lineto($1.pixx, 0); //#2362
        $$.lineto($1.pixx, $1.pixy); //#2362
        $$.lineto(0, $1.pixy); //#2362
        $$.closepath(); //#2362
        if ($eq($1.colormap, "unset")) { //#2367
            var _4H = new Map([
                [1, $1.barcolor]
            ]); //#2366
            $1.colormap = _4H; //#2366
        } //#2366
        var _4I = $1.colormap; //#2369
        for (var _4N = _4I.size, _4M = _4I.keys(), _4L = 0; _4L < _4N; _4L++) { //#2375
            var _4J = _4M.next().value; //#2375
            $$.setcolor(_4I.get(_4J)); //#2370
            $1.key = _4J; //#2371
            $k[$j++] = Infinity; //#2373
            var _4O = $1.pixs; //#2373
            for (var _4P = 0, _4Q = _4O.length; _4P < _4Q; _4P++) { //#2373
                var _4T = $eq($get(_4O, _4P), $1.key) ? 1 : 0; //#2373
                $k[$j++] = _4T; //#2373
            } //#2373
            var _4U = $a(); //#2373
            $k[$j++] = _4U; //#2374
            if ($1.dotty) { //#2374
                $1.drawlayerdots(); //#2374
            } else { //#2374
                $1.drawlayer(); //#2374
            } //#2374
        } //#2374
        if ($ne($1.textcolor, "unset")) { //#2378
            $$.setcolor($1.textcolor); //#2378
        } //#2378
        if ($1.includetext) { //#2435
            if (($eq($1.textxalign, "unset") && $eq($1.textyalign, "unset")) && $eq($1.alttext, "")) { //#2433
                $1.s = 0; //#2381
                $1.fn = ""; //#2381
                var _4c = $1.txt; //#2382
                for (var _4d = 0, _4e = _4c.length; _4d < _4e; _4d++) { //#2391
                    $forall($get(_4c, _4d)); //#2383
                    var _4g = $k[--$j]; //#2384
                    var _4h = $k[--$j]; //#2384
                    $k[$j++] = _4h; //#2389
                    $k[$j++] = _4g; //#2389
                    if ((_4g != $1.s) || $ne(_4h, $1.fn)) { //#2388
                        var _4k = $k[--$j]; //#2385
                        var _4l = $k[--$j]; //#2385
                        $1.s = _4k; //#2385
                        $1.fn = _4l; //#2385
                        $$.selectfont(_4l, _4k); //#2386
                    } else { //#2388
                        $j -= 2; //#2388
                    } //#2388
                    var _4m = $k[--$j]; //#2390
                    $$.moveto($k[--$j], _4m); //#2390
                    $$.show($k[--$j], 0, 0); //#2390
                } //#2390
            } else { //#2433
                $$.selectfont($1.textfont, $1.textsize); //#2393
                if ($eq($1.alttext, "")) { //#2399
                    $k[$j++] = Infinity; //#2395
                    var _4s = $1.txt; //#2395
                    for (var _4t = 0, _4u = _4s.length; _4t < _4u; _4t++) { //#2395
                        $forall($get($get(_4s, _4t), 0)); //#2395
                    } //#2395
                    $1.txt = $a(); //#2395
                    $1.tstr = $s($1.txt.length); //#2396
                    for (var _52 = 0, _51 = $1.txt.length - 1; _52 <= _51; _52 += 1) { //#2397
                        $put($1.tstr, _52, $get($1.txt, _52)); //#2397
                    } //#2397
                } else { //#2399
                    $1.tstr = $1.alttext; //#2399
                } //#2399
                if ($1.tstr.length == 0) { //#2409
                    $k[$j++] = 0; //#2404
                } else { //#2409
                    $$.save(); //#2406
                    $$.newpath(); //#2407
                    $$.moveto(0, 0); //#2407
                    $$.charpath("0", false); //#2407
                    var _58 = $$.pathbbox(); //#2407
                    $$.restore(); //#2409
                    $k[$j++] = _58.ury; //#2409
                } //#2409
                $1.textascent = $k[--$j]; //#2418
                var _5B = $$.stringwidth($1.tstr); //#2419
                $1.textwidth = $f(_5B.w + (($1.tstr.length - 1) * $1.textgaps)); //#2419
                $1.textxpos = $f($1.textxoffset + ($f($1.x - $1.textwidth) / 2)); //#2421
                if ($eq($1.textxalign, "left")) { //#2422
                    $1.textxpos = $1.textxoffset; //#2422
                } //#2422
                if ($eq($1.textxalign, "right")) { //#2423
                    $1.textxpos = $f($f($1.x - $1.textxoffset) - $1.textwidth); //#2423
                } //#2423
                if ($eq($1.textxalign, "offleft")) { //#2424
                    $1.textxpos = -$f($1.textwidth + $1.textxoffset); //#2424
                } //#2424
                if ($eq($1.textxalign, "offright")) { //#2425
                    $1.textxpos = $f($1.x + $1.textxoffset); //#2425
                } //#2425
                if ($eq($1.textxalign, "justify") && ($1.textwidth < $1.x)) { //#2429
                    $1.textxpos = 0; //#2427
                    $1.textgaps = $f($1.x - $1.textwidth) / ($1.tstr.length - 1); //#2428
                } //#2428
                $1.textypos = -($f($f($1.textyoffset + $1.textascent) + 1)); //#2430
                if ($eq($1.textyalign, "above")) { //#2431
                    $1.textypos = $f($f($1.textyoffset + $1.pixy) + 1); //#2431
                } //#2431
                if ($eq($1.textyalign, "center")) { //#2432
                    $1.textypos = $f($1.textyoffset + ($f($1.pixy - $1.textascent) / 2)); //#2432
                } //#2432
                $$.moveto($1.textxpos, $1.textypos); //#2433
                $$.show($1.tstr, $1.textgaps, 0); //#2433
            } //#2433
        } //#2433
        $$.restore(); //#2437
    };
    $0.ean5 = function() {
        var $1 = {}; //#2555
        $1.options = $k[--$j]; //#2557
        $1.barcode = $k[--$j]; //#2558
        $1.dontdraw = false; //#2560
        $1.includetext = false; //#2561
        $1.textfont = "OCR-B"; //#2562
        $1.textsize = 12; //#2563
        $1.textxoffset = 0; //#2564
        $1.textyoffset = "unset"; //#2565
        $1.height = 0.7; //#2566
        $forall($1.options, function() { //#2577
            var _3 = $k[--$j]; //#2577
            $1[$k[--$j]] = _3; //#2577
        }); //#2577
        $1.textfont = "" + $1.textfont; //#2579
        $1.textsize = +$1.textsize; //#2580
        $1.height = +$1.height; //#2581
        $1.textxoffset = +$1.textxoffset; //#2582
        if ($eq($1.textyoffset, "unset")) { //#2586
            $1.textyoffset = $f(($1.height * 72) + 1); //#2584
        } else { //#2586
            $1.textyoffset = +$1.textyoffset; //#2586
        } //#2586
        if ($1.barcode.length != 5) { //#2592
            $k[$j++] = 'bwipp.ean5badLength'; //#2591
            $k[$j++] = "EAN-5 add-on must be 5 digits"; //#2591
            $0.raiseerror(); //#2591
        } //#2591
        $forall($1.barcode, function() { //#2597
            var _E = $k[--$j]; //#2594
            if ((_E < 48) || (_E > 57)) { //#2596
                $k[$j++] = 'bwipp.ean5badCharacter'; //#2595
                $k[$j++] = "EAN-5 add-on must contain only digits"; //#2595
                $0.raiseerror(); //#2595
            } //#2595
        }); //#2595
        $1.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "112", "11"]); //#2604
        $1.barchars = "0123456789"; //#2607
        $1.mirrormaps = $a(["11000", "10100", "10010", "10001", "01100", "00110", "00011", "01010", "01001", "00101"]); //#2613
        $1.checksum = 0; //#2614
        for (var _H = 0; _H <= 4; _H += 1) { //#2623
            $1.i = _H; //#2616
            $1.barchar = $f($get($1.barcode, $1.i) - 48); //#2617
            if (($1.i % 2) == 0) { //#2621
                $1.checksum = $f(($1.barchar * 3) + $1.checksum); //#2619
            } else { //#2621
                $1.checksum = $f(($1.barchar * 9) + $1.checksum); //#2621
            } //#2621
        } //#2621
        $1.checksum = $1.checksum % 10; //#2624
        $1.mirrormap = $get($1.mirrormaps, $1.checksum); //#2625
        $1.sbs = $s(31); //#2627
        $1.txt = $a(5); //#2628
        for (var _W = 0; _W <= 4; _W += 1) { //#2658
            $1.i = _W; //#2631
            if ($1.i == 0) { //#2637
                $puti($1.sbs, 0, $get($1.encs, 10)); //#2635
            } else { //#2637
                $puti($1.sbs, (($1.i - 1) * 6) + 7, $get($1.encs, 11)); //#2637
            } //#2637
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#2641
            $j--; //#2642
            $1.indx = $k[--$j].length; //#2643
            $j -= 2; //#2644
            $1.enc = $get($1.encs, $1.indx); //#2645
            if ($get($1.mirrormap, $1.i) == 49) { //#2655
                $1.enclen = $1.enc.length; //#2647
                $1.revenc = $s($1.enclen); //#2648
                for (var _v = 0, _u = $1.enclen - 1; _v <= _u; _v += 1) { //#2653
                    $1.j = _v; //#2650
                    $1.char = $get($1.enc, $1.j); //#2651
                    $put($1.revenc, ($1.enclen - $1.j) - 1, $1.char); //#2652
                } //#2652
                $1.enc = $1.revenc; //#2654
            } //#2654
            $puti($1.sbs, ($1.i * 6) + 3, $1.enc); //#2656
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(((($1.i - 1) * 9) + 13) + $1.textxoffset), $1.textyoffset, $1.textfont, $1.textsize])); //#2657
        } //#2657
        $k[$j++] = Infinity; //#2672
        $k[$j++] = Infinity; //#2663
        var _1I = $1.sbs; //#2663
        for (var _1J = 0, _1K = _1I.length; _1J < _1K; _1J++) { //#2663
            $k[$j++] = $get(_1I, _1J) - 48; //#2663
        } //#2663
        var _1M = $a(); //#2663
        $k[$j++] = Infinity; //#2664
        for (var _1N = 0, _1O = 16; _1N < _1O; _1N++) { //#2664
            $k[$j++] = $1.height; //#2664
        } //#2664
        var _1Q = $a(); //#2664
        $k[$j++] = Infinity; //#2665
        for (var _1R = 0, _1S = 16; _1R < _1S; _1R++) { //#2665
            $k[$j++] = 0; //#2665
        } //#2665
        var _1T = $a(); //#2665
        $k[$j++] = 'ren'; //#2668
        $k[$j++] = $0.renlinear; //#2668
        $k[$j++] = 'sbs'; //#2668
        $k[$j++] = _1M; //#2668
        $k[$j++] = 'bhs'; //#2668
        $k[$j++] = _1Q; //#2668
        $k[$j++] = 'bbs'; //#2668
        $k[$j++] = _1T; //#2668
        if ($1.includetext) { //#2668
            $k[$j++] = 'txt'; //#2667
            $k[$j++] = $1.txt; //#2667
        } //#2667
        $k[$j++] = 'opt'; //#2672
        $k[$j++] = $1.options; //#2672
        $k[$j++] = 'guardrightpos'; //#2672
        $k[$j++] = 10; //#2672
        $k[$j++] = 'guardrightypos'; //#2672
        $k[$j++] = $f($1.textyoffset + 4); //#2672
        $k[$j++] = 'bordertop'; //#2672
        $k[$j++] = 10; //#2672
        var _1Y = $d(); //#2672
        $k[$j++] = _1Y; //#2675
        if (!$1.dontdraw) { //#2675
            $0.renlinear(); //#2675
        } //#2675
    };
    $0.ean2 = function() {
        var $1 = {}; //#2704
        $1.options = $k[--$j]; //#2706
        $1.barcode = $k[--$j]; //#2707
        $1.dontdraw = false; //#2709
        $1.includetext = false; //#2710
        $1.textfont = "OCR-B"; //#2711
        $1.textsize = 12; //#2712
        $1.textxoffset = 0; //#2713
        $1.textyoffset = "unset"; //#2714
        $1.height = 0.7; //#2715
        $forall($1.options, function() { //#2726
            var _3 = $k[--$j]; //#2726
            $1[$k[--$j]] = _3; //#2726
        }); //#2726
        $1.textfont = "" + $1.textfont; //#2728
        $1.textsize = +$1.textsize; //#2729
        $1.height = +$1.height; //#2730
        $1.textxoffset = +$1.textxoffset; //#2731
        if ($eq($1.textyoffset, "unset")) { //#2735
            $1.textyoffset = $f(($1.height * 72) + 1); //#2733
        } else { //#2735
            $1.textyoffset = +$1.textyoffset; //#2735
        } //#2735
        if ($1.barcode.length != 2) { //#2741
            $k[$j++] = 'bwipp.ean2badLength'; //#2740
            $k[$j++] = "EAN-2 add-on must be 2 digits"; //#2740
            $0.raiseerror(); //#2740
        } //#2740
        $forall($1.barcode, function() { //#2746
            var _E = $k[--$j]; //#2743
            if ((_E < 48) || (_E > 57)) { //#2745
                $k[$j++] = 'bwipp.ean2badCharacter'; //#2744
                $k[$j++] = "EAN-2 add-on must contain only digits"; //#2744
                $0.raiseerror(); //#2744
            } //#2744
        }); //#2744
        $1.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "112", "11"]); //#2753
        $1.barchars = "0123456789"; //#2756
        $1.mirrormap = $get($a(["00", "01", "10", "11"]), (~~$z($geti($1.barcode, 0, 2))) % 4); //#2759
        $1.sbs = $s(13); //#2761
        $1.txt = $a(2); //#2762
        for (var _M = 0; _M <= 1; _M += 1) { //#2792
            $1.i = _M; //#2765
            if ($1.i == 0) { //#2771
                $puti($1.sbs, 0, $get($1.encs, 10)); //#2769
            } else { //#2771
                $puti($1.sbs, (($1.i - 1) * 6) + 7, $get($1.encs, 11)); //#2771
            } //#2771
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#2775
            $j--; //#2776
            $1.indx = $k[--$j].length; //#2777
            $j -= 2; //#2778
            $1.enc = $get($1.encs, $1.indx); //#2779
            if ($get($1.mirrormap, $1.i) == 49) { //#2789
                $1.enclen = $1.enc.length; //#2781
                $1.revenc = $s($1.enclen); //#2782
                for (var _l = 0, _k = $1.enclen - 1; _l <= _k; _l += 1) { //#2787
                    $1.j = _l; //#2784
                    $1.char = $get($1.enc, $1.j); //#2785
                    $put($1.revenc, ($1.enclen - $1.j) - 1, $1.char); //#2786
                } //#2786
                $1.enc = $1.revenc; //#2788
            } //#2788
            $puti($1.sbs, ($1.i * 6) + 3, $1.enc); //#2790
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(((($1.i - 1) * 9) + 13) + $1.textxoffset), $1.textyoffset, $1.textfont, $1.textsize])); //#2791
        } //#2791
        $k[$j++] = Infinity; //#2806
        $k[$j++] = Infinity; //#2797
        var _18 = $1.sbs; //#2797
        for (var _19 = 0, _1A = _18.length; _19 < _1A; _19++) { //#2797
            $k[$j++] = $get(_18, _19) - 48; //#2797
        } //#2797
        var _1C = $a(); //#2797
        $k[$j++] = Infinity; //#2798
        for (var _1D = 0, _1E = 12; _1D < _1E; _1D++) { //#2798
            $k[$j++] = $1.height; //#2798
        } //#2798
        var _1G = $a(); //#2798
        $k[$j++] = Infinity; //#2799
        for (var _1H = 0, _1I = 12; _1H < _1I; _1H++) { //#2799
            $k[$j++] = 0; //#2799
        } //#2799
        var _1J = $a(); //#2799
        $k[$j++] = 'ren'; //#2802
        $k[$j++] = $0.renlinear; //#2802
        $k[$j++] = 'sbs'; //#2802
        $k[$j++] = _1C; //#2802
        $k[$j++] = 'bhs'; //#2802
        $k[$j++] = _1G; //#2802
        $k[$j++] = 'bbs'; //#2802
        $k[$j++] = _1J; //#2802
        if ($1.includetext) { //#2802
            $k[$j++] = 'txt'; //#2801
            $k[$j++] = $1.txt; //#2801
        } //#2801
        $k[$j++] = 'opt'; //#2806
        $k[$j++] = $1.options; //#2806
        $k[$j++] = 'guardrightpos'; //#2806
        $k[$j++] = 10; //#2806
        $k[$j++] = 'guardrightypos'; //#2806
        $k[$j++] = $f($1.textyoffset + 4); //#2806
        $k[$j++] = 'bordertop'; //#2806
        $k[$j++] = 10; //#2806
        var _1O = $d(); //#2806
        $k[$j++] = _1O; //#2809
        if (!$1.dontdraw) { //#2809
            $0.renlinear(); //#2809
        } //#2809
    };
    $0.ean13 = function() {
        var $1 = {}; //#2840
        $1.options = $k[--$j]; //#2842
        $1.barcode = $k[--$j]; //#2843
        $1.dontdraw = false; //#2845
        $1.includetext = false; //#2846
        $1.textfont = "OCR-B"; //#2847
        $1.textsize = 12; //#2848
        $1.textxoffset = -10; //#2849
        $1.textyoffset = -4; //#2850
        $1.height = 1; //#2851
        $1.addongap = 12; //#2852
        $1.addontextfont = "unset"; //#2853
        $1.addontextsize = "unset"; //#2854
        $1.addontextxoffset = "unset"; //#2855
        $1.addontextyoffset = "unset"; //#2856
        $forall($1.options, function() { //#2867
            var _3 = $k[--$j]; //#2867
            $1[$k[--$j]] = _3; //#2867
        }); //#2867
        $1.textfont = "" + $1.textfont; //#2869
        $1.textsize = +$1.textsize; //#2870
        $1.textxoffset = +$1.textxoffset; //#2871
        $1.textyoffset = +$1.textyoffset; //#2872
        $1.height = +$1.height; //#2873
        $1.addongap = +$1.addongap; //#2874
        if ($ne($1.addontextfont, "unset")) { //#2875
            $1.addontextfont = "" + $1.addontextfont; //#2875
        } //#2875
        if ($ne($1.addontextsize, "unset")) { //#2876
            $1.addontextsize = +$1.addontextsize; //#2876
        } //#2876
        if ($ne($1.addontextxoffset, "unset")) { //#2877
            $1.addontextxoffset = +$1.addontextxoffset; //#2877
        } //#2877
        if ($ne($1.addontextyoffset, "unset")) { //#2878
            $1.addontextyoffset = +$1.addontextyoffset; //#2878
        } //#2878
        $search($1.barcode, " "); //#2881
        if ($k[--$j]) { //#2887
            $1.barcode = $k[--$j]; //#2882
            $j--; //#2883
            $1.addon = $k[--$j]; //#2884
        } else { //#2887
            $j--; //#2886
            $1.addon = ""; //#2887
        } //#2887
        if (($1.barcode.length != 12) && ($1.barcode.length != 13)) { //#2893
            $k[$j++] = 'bwipp.ean13badLength'; //#2892
            $k[$j++] = "EAN-13 must be 12 or 13 digits"; //#2892
            $0.raiseerror(); //#2892
        } //#2892
        $forall($1.barcode, function() { //#2898
            var _Q = $k[--$j]; //#2895
            if ((_Q < 48) || (_Q > 57)) { //#2897
                $k[$j++] = 'bwipp.ean13badCharacter'; //#2896
                $k[$j++] = "EAN-13 must contain only digits"; //#2896
                $0.raiseerror(); //#2896
            } //#2896
        }); //#2896
        if ((($1.addon.length != 0) && ($1.addon.length != 2)) && ($1.addon.length != 5)) { //#2901
            $k[$j++] = 'bwipp.ean13badAddOnLength'; //#2900
            $k[$j++] = "Add-on for EAN-13 must be 2 or 5 digits"; //#2900
            $0.raiseerror(); //#2900
        } //#2900
        $1.pad = $s(13); //#2904
        $1.checksum = 0; //#2905
        for (var _V = 0; _V <= 11; _V += 1) { //#2914
            $1.i = _V; //#2907
            $1.barchar = $f($get($1.barcode, $1.i) - 48); //#2908
            if (($1.i % 2) == 0) { //#2912
                $1.checksum = $f($1.barchar + $1.checksum); //#2910
            } else { //#2912
                $1.checksum = $f(($1.barchar * 3) + $1.checksum); //#2912
            } //#2912
        } //#2912
        $1.checksum = ($f(10 - ($1.checksum % 10))) % 10; //#2915
        if ($1.barcode.length == 13) { //#2920
            if ($get($1.barcode, 12) != $f($1.checksum + 48)) { //#2919
                $k[$j++] = 'bwipp.ean13badCheckDigit'; //#2918
                $k[$j++] = "Incorrect EAN-13 check digit provided"; //#2918
                $0.raiseerror(); //#2918
            } //#2918
        } //#2918
        $puti($1.pad, 0, $1.barcode); //#2921
        $put($1.pad, 12, $f($1.checksum + 48)); //#2922
        $1.barcode = $1.pad; //#2923
        $1.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "11111", "111"]); //#2930
        $1.barchars = "0123456789"; //#2933
        $1.mirrormaps = $a(["000000", "001011", "001101", "001110", "010011", "011001", "011100", "010101", "010110", "011010"]); //#2939
        $1.sbs = $s(59); //#2941
        $1.txt = $a(13); //#2942
        $puti($1.sbs, 0, $get($1.encs, 10)); //#2945
        $1.mirrormap = $get($1.mirrormaps, $get($1.barcode, 0) - 48); //#2948
        $put($1.txt, 0, $a([$geti($1.barcode, 0, 1), $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])); //#2949
        for (var _17 = 1; _17 <= 6; _17 += 1) { //#2972
            $1.i = _17; //#2954
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#2955
            $j--; //#2956
            $1.indx = $k[--$j].length; //#2957
            $j -= 2; //#2958
            $1.enc = $get($1.encs, $1.indx); //#2959
            if ($get($1.mirrormap, $1.i - 1) == 49) { //#2969
                $1.enclen = $1.enc.length; //#2961
                $1.revenc = $s($1.enclen); //#2962
                for (var _1O = 0, _1N = $1.enclen - 1; _1O <= _1N; _1O += 1) { //#2967
                    $1.j = _1O; //#2964
                    $1.char = $get($1.enc, $1.j); //#2965
                    $put($1.revenc, ($1.enclen - $1.j) - 1, $1.char); //#2966
                } //#2966
                $1.enc = $1.revenc; //#2968
            } //#2968
            $puti($1.sbs, (($1.i - 1) * 4) + 3, $1.enc); //#2970
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(((($1.i - 1) * 7) + 14) + $1.textxoffset), $1.textyoffset, $1.textfont, $1.textsize])); //#2971
        } //#2971
        $puti($1.sbs, ((7 - 1) * 4) + 3, $get($1.encs, 11)); //#2975
        for (var _1o = 7; _1o <= 12; _1o += 1) { //#2988
            $1.i = _1o; //#2980
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#2981
            $j--; //#2982
            $1.indx = $k[--$j].length; //#2983
            $j -= 2; //#2984
            $1.enc = $get($1.encs, $1.indx); //#2985
            $puti($1.sbs, (($1.i - 1) * 4) + 8, $1.enc); //#2986
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(((($1.i - 1) * 7) + 18) + $1.textxoffset), $1.textyoffset, $1.textfont, $1.textsize])); //#2987
        } //#2987
        $puti($1.sbs, 56, $get($1.encs, 12)); //#2991
        $k[$j++] = Infinity; //#2993
        var _2E = $1.sbs; //#2993
        for (var _2F = 0, _2G = _2E.length; _2F < _2G; _2F++) { //#2993
            $k[$j++] = $get(_2E, _2F) - 48; //#2993
        } //#2993
        $1.sbs = $a(); //#2993
        if ($1.includetext) { //#3000
            $k[$j++] = Infinity; //#2995
            $k[$j++] = $1.height; //#2995
            $k[$j++] = $1.height; //#2995
            for (var _2M = 0, _2N = 12; _2M < _2N; _2M++) { //#2995
                $k[$j++] = $f($1.height - 0.075); //#2995
            } //#2995
            $k[$j++] = $1.height; //#2995
            $k[$j++] = $1.height; //#2995
            for (var _2R = 0, _2S = 12; _2R < _2S; _2R++) { //#2995
                $k[$j++] = $f($1.height - 0.075); //#2995
            } //#2995
            $k[$j++] = $1.height; //#2995
            $k[$j++] = $1.height; //#2995
            $1.bhs = $a(); //#2995
            $k[$j++] = Infinity; //#2996
            $k[$j++] = 0; //#2996
            $k[$j++] = 0; //#2996
            for (var _2X = 0, _2Y = 12; _2X < _2Y; _2X++) { //#2996
                $k[$j++] = 0.075; //#2996
            } //#2996
            $k[$j++] = 0; //#2996
            $k[$j++] = 0; //#2996
            for (var _2Z = 0, _2a = 12; _2Z < _2a; _2Z++) { //#2996
                $k[$j++] = 0.075; //#2996
            } //#2996
            $k[$j++] = 0; //#2996
            $k[$j++] = 0; //#2996
            $1.bbs = $a(); //#2996
        } else { //#3000
            $k[$j++] = Infinity; //#2998
            for (var _2c = 0, _2d = 30; _2c < _2d; _2c++) { //#2998
                $k[$j++] = $1.height; //#2998
            } //#2998
            $1.bhs = $a(); //#2998
            $k[$j++] = Infinity; //#2999
            for (var _2g = 0, _2h = 30; _2g < _2h; _2g++) { //#2999
                $k[$j++] = 0; //#2999
            } //#2999
            $1.bbs = $a(); //#2999
            $1.txt = $a([]); //#3000
        } //#3000
        $1.guardrightypos = 0; //#3002
        if ($ne($1.addon, "")) { //#3023
            $k[$j++] = Infinity; //#3013
            $k[$j++] = 'dontdraw'; //#3010
            $k[$j++] = true; //#3010
            $k[$j++] = 'includetext'; //#3010
            $k[$j++] = true; //#3010
            $k[$j++] = 'height'; //#3010
            $k[$j++] = $f($1.height - 0.15); //#3010
            $k[$j++] = 'textxoffset'; //#3010
            if ($1.addontextxoffset != "unset") { //#3010
                $k[$j++] = $1.addontextxoffset; //#3010
            } else { //#3010
                $k[$j++] = $f(95 + $1.addongap); //#3010
            } //#3010
            if ($1.addontextyoffset != "unset") { //#3011
                $k[$j++] = 'textyoffset'; //#3011
                $k[$j++] = $1.addontextyoffset; //#3011
            } //#3011
            var _2s = ($1.addontextsize != "unset") ? $1.addontextsize : $1.textsize; //#3012
            var _2u = $ne($1.addontextfont, "unset") ? $1.addontextfont : $1.textfont; //#3013
            $k[$j++] = 'textsize'; //#3013
            $k[$j++] = _2s; //#3013
            $k[$j++] = 'textfont'; //#3013
            $k[$j++] = _2u; //#3013
            $1.addopts = $d(); //#3014
            if ($1.addon.length == 2) { //#3015
                $k[$j++] = $1.addon; //#3015
                $k[$j++] = $1.addopts; //#3015
                $0.ean2(); //#3015
            } //#3015
            if ($1.addon.length == 5) { //#3016
                $k[$j++] = $1.addon; //#3016
                $k[$j++] = $1.addopts; //#3016
                $0.ean5(); //#3016
            } //#3016
            $1.addcode = $k[--$j]; //#3017
            $k[$j++] = Infinity; //#3018
            $aload($1.sbs); //#3018
            $k[$j++] = $1.addongap; //#3018
            $aload($get($1.addcode, "sbs")); //#3018
            $1.sbs = $a(); //#3018
            $k[$j++] = Infinity; //#3019
            $aload($1.bhs); //#3019
            $forall($get($1.addcode, "bhs"), function() { //#3019
                var _3B = $k[--$j]; //#3019
                $k[$j++] = $f(_3B - 0.075); //#3019
            }); //#3019
            $1.bhs = $a(); //#3019
            $k[$j++] = Infinity; //#3020
            $aload($1.bbs); //#3020
            $forall($get($1.addcode, "bbs"), function() { //#3020
                var _3G = $k[--$j]; //#3020
                $k[$j++] = $f(_3G + 0.075); //#3020
            }); //#3020
            $1.bbs = $a(); //#3020
            $k[$j++] = Infinity; //#3021
            $aload($1.txt); //#3021
            $aload($get($1.addcode, "txt")); //#3021
            $1.txt = $a(); //#3021
            $1.guardrightypos = $f(($1.height * 72) - 6); //#3022
        } //#3022
        var _3T = new Map([
            ["ren", $0.renlinear],
            ["sbs", $1.sbs],
            ["bhs", $1.bhs],
            ["bbs", $1.bbs],
            ["txt", $1.txt],
            ["opt", $1.options],
            ["guardrightpos", 10],
            ["guardrightypos", $1.guardrightypos],
            ["borderbottom", 5]
        ]); //#3035
        $k[$j++] = _3T; //#3038
        if (!$1.dontdraw) { //#3038
            $0.renlinear(); //#3038
        } //#3038
    };
    $0.ean8 = function() {
        var $1 = {}; //#3069
        $1.options = $k[--$j]; //#3071
        $1.barcode = $k[--$j]; //#3072
        $1.dontdraw = false; //#3074
        $1.includetext = false; //#3075
        $1.textfont = "OCR-B"; //#3076
        $1.textsize = 12; //#3077
        $1.textxoffset = 4; //#3078
        $1.textyoffset = -4; //#3079
        $1.height = 1; //#3080
        $1.addongap = 12; //#3081
        $1.addontextfont = "unset"; //#3082
        $1.addontextsize = "unset"; //#3083
        $1.addontextxoffset = "unset"; //#3084
        $1.addontextyoffset = "unset"; //#3085
        $forall($1.options, function() { //#3096
            var _3 = $k[--$j]; //#3096
            $1[$k[--$j]] = _3; //#3096
        }); //#3096
        $1.textfont = "" + $1.textfont; //#3098
        $1.textsize = +$1.textsize; //#3099
        $1.textxoffset = +$1.textxoffset; //#3100
        $1.textyoffset = +$1.textyoffset; //#3101
        $1.height = +$1.height; //#3102
        $1.addongap = +$1.addongap; //#3103
        if ($ne($1.addontextfont, "unset")) { //#3104
            $1.addontextfont = "" + $1.addontextfont; //#3104
        } //#3104
        if ($ne($1.addontextsize, "unset")) { //#3105
            $1.addontextsize = +$1.addontextsize; //#3105
        } //#3105
        if ($ne($1.addontextxoffset, "unset")) { //#3106
            $1.addontextxoffset = +$1.addontextxoffset; //#3106
        } //#3106
        if ($ne($1.addontextyoffset, "unset")) { //#3107
            $1.addontextyoffset = +$1.addontextyoffset; //#3107
        } //#3107
        $search($1.barcode, " "); //#3110
        if ($k[--$j]) { //#3116
            $1.barcode = $k[--$j]; //#3111
            $j--; //#3112
            $1.addon = $k[--$j]; //#3113
        } else { //#3116
            $j--; //#3115
            $1.addon = ""; //#3116
        } //#3116
        if (($1.barcode.length != 7) && ($1.barcode.length != 8)) { //#3122
            $k[$j++] = 'bwipp.ean8badLength'; //#3121
            $k[$j++] = "EAN-8 must be 7 or 8 digits"; //#3121
            $0.raiseerror(); //#3121
        } //#3121
        $forall($1.barcode, function() { //#3127
            var _Q = $k[--$j]; //#3124
            if ((_Q < 48) || (_Q > 57)) { //#3126
                $k[$j++] = 'bwipp.ean8badCharacter'; //#3125
                $k[$j++] = "EAN-8 must contain only digits"; //#3125
                $0.raiseerror(); //#3125
            } //#3125
        }); //#3125
        if ((($1.addon.length != 0) && ($1.addon.length != 2)) && ($1.addon.length != 5)) { //#3130
            $k[$j++] = 'bwipp.ean8badAddOnLength'; //#3129
            $k[$j++] = "Add-on for EAN-8 must be 2 or 5 digits"; //#3129
            $0.raiseerror(); //#3129
        } //#3129
        $1.pad = $s(8); //#3133
        $1.checksum = 0; //#3134
        for (var _V = 0; _V <= 6; _V += 1) { //#3143
            $1.i = _V; //#3136
            $1.barchar = $f($get($1.barcode, $1.i) - 48); //#3137
            if (($1.i % 2) != 0) { //#3141
                $1.checksum = $f($1.barchar + $1.checksum); //#3139
            } else { //#3141
                $1.checksum = $f(($1.barchar * 3) + $1.checksum); //#3141
            } //#3141
        } //#3141
        $1.checksum = ($f(10 - ($1.checksum % 10))) % 10; //#3144
        if ($1.barcode.length == 8) { //#3149
            if ($get($1.barcode, 7) != $f($1.checksum + 48)) { //#3148
                $k[$j++] = 'bwipp.ean8badCheckDigit'; //#3147
                $k[$j++] = "Incorrect EAN-8 check digit provided"; //#3147
                $0.raiseerror(); //#3147
            } //#3147
        } //#3147
        $puti($1.pad, 0, $1.barcode); //#3150
        $put($1.pad, 7, $f($1.checksum + 48)); //#3151
        $1.barcode = $1.pad; //#3152
        $1.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "11111", "111"]); //#3159
        $1.barchars = "0123456789"; //#3162
        $1.sbs = $s(43); //#3164
        $1.txt = $a(8); //#3165
        $puti($1.sbs, 0, $get($1.encs, 10)); //#3168
        for (var _u = 0; _u <= 3; _u += 1) { //#3181
            $1.i = _u; //#3173
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#3174
            $j--; //#3175
            $1.indx = $k[--$j].length; //#3176
            $j -= 2; //#3177
            $1.enc = $get($1.encs, $1.indx); //#3178
            $puti($1.sbs, ($1.i * 4) + 3, $1.enc); //#3179
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(($1.i * 7) + $1.textxoffset), $1.textyoffset, $1.textfont, $1.textsize])); //#3180
        } //#3180
        $puti($1.sbs, (4 * 4) + 3, $get($1.encs, 11)); //#3184
        for (var _1K = 4; _1K <= 7; _1K += 1) { //#3197
            $1.i = _1K; //#3189
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#3190
            $j--; //#3191
            $1.indx = $k[--$j].length; //#3192
            $j -= 2; //#3193
            $1.enc = $get($1.encs, $1.indx); //#3194
            $puti($1.sbs, ($1.i * 4) + 8, $1.enc); //#3195
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(($f(($1.i * 7) + $1.textxoffset)) + 4), $1.textyoffset, $1.textfont, $1.textsize])); //#3196
        } //#3196
        $puti($1.sbs, 40, $get($1.encs, 12)); //#3200
        $k[$j++] = Infinity; //#3202
        var _1k = $1.sbs; //#3202
        for (var _1l = 0, _1m = _1k.length; _1l < _1m; _1l++) { //#3202
            $k[$j++] = $get(_1k, _1l) - 48; //#3202
        } //#3202
        $1.sbs = $a(); //#3202
        if ($1.includetext) { //#3209
            $k[$j++] = Infinity; //#3204
            $k[$j++] = $1.height; //#3204
            $k[$j++] = $1.height; //#3204
            for (var _1s = 0, _1t = 8; _1s < _1t; _1s++) { //#3204
                $k[$j++] = $f($1.height - 0.075); //#3204
            } //#3204
            $k[$j++] = $1.height; //#3204
            $k[$j++] = $1.height; //#3204
            for (var _1x = 0, _1y = 8; _1x < _1y; _1x++) { //#3204
                $k[$j++] = $f($1.height - 0.075); //#3204
            } //#3204
            $k[$j++] = $1.height; //#3204
            $k[$j++] = $1.height; //#3204
            $1.bhs = $a(); //#3204
            $k[$j++] = Infinity; //#3205
            $k[$j++] = 0; //#3205
            $k[$j++] = 0; //#3205
            for (var _23 = 0, _24 = 8; _23 < _24; _23++) { //#3205
                $k[$j++] = 0.075; //#3205
            } //#3205
            $k[$j++] = 0; //#3205
            $k[$j++] = 0; //#3205
            for (var _25 = 0, _26 = 8; _25 < _26; _25++) { //#3205
                $k[$j++] = 0.075; //#3205
            } //#3205
            $k[$j++] = 0; //#3205
            $k[$j++] = 0; //#3205
            $1.bbs = $a(); //#3205
        } else { //#3209
            $k[$j++] = Infinity; //#3207
            for (var _28 = 0, _29 = 22; _28 < _29; _28++) { //#3207
                $k[$j++] = $1.height; //#3207
            } //#3207
            $1.bhs = $a(); //#3207
            $k[$j++] = Infinity; //#3208
            for (var _2C = 0, _2D = 22; _2C < _2D; _2C++) { //#3208
                $k[$j++] = 0; //#3208
            } //#3208
            $1.bbs = $a(); //#3208
            $1.txt = $a([]); //#3209
        } //#3209
        $1.guardrightypos = 0; //#3211
        if ($ne($1.addon, "")) { //#3232
            $k[$j++] = Infinity; //#3222
            $k[$j++] = 'dontdraw'; //#3219
            $k[$j++] = true; //#3219
            $k[$j++] = 'includetext'; //#3219
            $k[$j++] = true; //#3219
            $k[$j++] = 'height'; //#3219
            $k[$j++] = $f($1.height - 0.15); //#3219
            $k[$j++] = 'textxoffset'; //#3219
            if ($1.addontextxoffset != "unset") { //#3219
                $k[$j++] = $1.addontextxoffset; //#3219
            } else { //#3219
                $k[$j++] = $f(67 + $1.addongap); //#3219
            } //#3219
            if ($1.addontextyoffset != "unset") { //#3220
                $k[$j++] = 'textyoffset'; //#3220
                $k[$j++] = $1.addontextyoffset; //#3220
            } //#3220
            var _2O = ($1.addontextsize != "unset") ? $1.addontextsize : $1.textsize; //#3221
            var _2Q = $ne($1.addontextfont, "unset") ? $1.addontextfont : $1.textfont; //#3222
            $k[$j++] = 'textsize'; //#3222
            $k[$j++] = _2O; //#3222
            $k[$j++] = 'textfont'; //#3222
            $k[$j++] = _2Q; //#3222
            $1.addopts = $d(); //#3223
            if ($1.addon.length == 2) { //#3224
                $k[$j++] = $1.addon; //#3224
                $k[$j++] = $1.addopts; //#3224
                $0.ean2(); //#3224
            } //#3224
            if ($1.addon.length == 5) { //#3225
                $k[$j++] = $1.addon; //#3225
                $k[$j++] = $1.addopts; //#3225
                $0.ean5(); //#3225
            } //#3225
            $1.addcode = $k[--$j]; //#3226
            $k[$j++] = Infinity; //#3227
            $aload($1.sbs); //#3227
            $k[$j++] = $1.addongap; //#3227
            $aload($get($1.addcode, "sbs")); //#3227
            $1.sbs = $a(); //#3227
            $k[$j++] = Infinity; //#3228
            $aload($1.bhs); //#3228
            $forall($get($1.addcode, "bhs"), function() { //#3228
                var _2h = $k[--$j]; //#3228
                $k[$j++] = $f(_2h - 0.075); //#3228
            }); //#3228
            $1.bhs = $a(); //#3228
            $k[$j++] = Infinity; //#3229
            $aload($1.bbs); //#3229
            $forall($get($1.addcode, "bbs"), function() { //#3229
                var _2m = $k[--$j]; //#3229
                $k[$j++] = $f(_2m + 0.075); //#3229
            }); //#3229
            $1.bbs = $a(); //#3229
            $k[$j++] = Infinity; //#3230
            $aload($1.txt); //#3230
            $aload($get($1.addcode, "txt")); //#3230
            $1.txt = $a(); //#3230
            $1.guardrightypos = $f(($1.height * 72) - 6); //#3231
        } //#3231
        var _2z = new Map([
            ["ren", $0.renlinear],
            ["sbs", $1.sbs],
            ["bhs", $1.bhs],
            ["bbs", $1.bbs],
            ["txt", $1.txt],
            ["opt", $1.options],
            ["guardleftpos", 10],
            ["guardrightpos", 10],
            ["guardrightypos", $1.guardrightypos],
            ["borderbottom", 5]
        ]); //#3245
        $k[$j++] = _2z; //#3248
        if (!$1.dontdraw) { //#3248
            $0.renlinear(); //#3248
        } //#3248
    };
    $0.upca = function() {
        var $1 = {}; //#3279
        $1.options = $k[--$j]; //#3281
        $1.barcode = $k[--$j]; //#3282
        $1.dontdraw = false; //#3284
        $1.includetext = false; //#3285
        $1.textfont = "OCR-B"; //#3286
        $1.textsize = 12; //#3287
        $1.textxoffset = -7; //#3288
        $1.textyoffset = -4; //#3289
        $1.height = 1; //#3290
        $1.addongap = 12; //#3291
        $1.addontextfont = "unset"; //#3292
        $1.addontextsize = "unset"; //#3293
        $1.addontextxoffset = "unset"; //#3294
        $1.addontextyoffset = "unset"; //#3295
        $forall($1.options, function() { //#3306
            var _3 = $k[--$j]; //#3306
            $1[$k[--$j]] = _3; //#3306
        }); //#3306
        $1.textfont = "" + $1.textfont; //#3308
        $1.textsize = +$1.textsize; //#3309
        $1.textxoffset = +$1.textxoffset; //#3310
        $1.textyoffset = +$1.textyoffset; //#3311
        $1.height = +$1.height; //#3312
        $1.addongap = +$1.addongap; //#3313
        if ($ne($1.addontextfont, "unset")) { //#3314
            $1.addontextfont = "" + $1.addontextfont; //#3314
        } //#3314
        if ($ne($1.addontextsize, "unset")) { //#3315
            $1.addontextsize = +$1.addontextsize; //#3315
        } //#3315
        if ($ne($1.addontextxoffset, "unset")) { //#3316
            $1.addontextxoffset = +$1.addontextxoffset; //#3316
        } //#3316
        if ($ne($1.addontextyoffset, "unset")) { //#3317
            $1.addontextyoffset = +$1.addontextyoffset; //#3317
        } //#3317
        $search($1.barcode, " "); //#3320
        if ($k[--$j]) { //#3326
            $1.barcode = $k[--$j]; //#3321
            $j--; //#3322
            $1.addon = $k[--$j]; //#3323
        } else { //#3326
            $j--; //#3325
            $1.addon = ""; //#3326
        } //#3326
        if (($1.barcode.length == 7) || ($1.barcode.length == 8)) { //#3372
            $forall($1.barcode, function() { //#3336
                var _Q = $k[--$j]; //#3333
                if ((_Q < 48) || (_Q > 57)) { //#3335
                    $k[$j++] = 'bwipp.upcAupcEbadCharacter'; //#3334
                    $k[$j++] = "UPC-E must contain only digits"; //#3334
                    $0.raiseerror(); //#3334
                } //#3334
            }); //#3334
            var _S = $get($1.barcode, 0); //#3337
            if ((_S != 48) && (_S != 49)) { //#3339
                $k[$j++] = 'bwipp.upcAupcEbadNumberSystem'; //#3338
                $k[$j++] = "UPC-E must have number system 0 or 1"; //#3338
                $0.raiseerror(); //#3338
            } //#3338
            for (var _T = 0, _U = 1; _T < _U; _T++) { //#3367
                var _W = ($1.barcode.length == 8) ? 12 : 11; //#3341
                $1.upcacode = $s(_W); //#3341
                if ($f($get($1.barcode, 6) - 48) <= 2) { //#3348
                    $puti($1.upcacode, 0, $geti($1.barcode, 0, 3)); //#3343
                    $puti($1.upcacode, 3, $geti($1.barcode, 6, 1)); //#3344
                    $puti($1.upcacode, 4, "0000"); //#3345
                    $puti($1.upcacode, 8, $geti($1.barcode, 3, 3)); //#3346
                    break; //#3347
                } //#3347
                if ($f($get($1.barcode, 6) - 48) == 3) { //#3354
                    $puti($1.upcacode, 0, $geti($1.barcode, 0, 4)); //#3350
                    $puti($1.upcacode, 4, "00000"); //#3351
                    $puti($1.upcacode, 9, $geti($1.barcode, 4, 2)); //#3352
                    break; //#3353
                } //#3353
                if ($f($get($1.barcode, 6) - 48) == 4) { //#3360
                    $puti($1.upcacode, 0, $geti($1.barcode, 0, 5)); //#3356
                    $puti($1.upcacode, 5, "00000"); //#3357
                    $puti($1.upcacode, 10, $geti($1.barcode, 5, 1)); //#3358
                    break; //#3359
                } //#3359
                if ($f($get($1.barcode, 6) - 48) >= 5) { //#3366
                    $puti($1.upcacode, 0, $geti($1.barcode, 0, 6)); //#3362
                    $puti($1.upcacode, 6, "0000"); //#3363
                    $puti($1.upcacode, 10, $geti($1.barcode, 6, 1)); //#3364
                    break; //#3365
                } //#3365
            } //#3365
            if ($1.barcode.length == 8) { //#3370
                $puti($1.upcacode, 11, $geti($1.barcode, 7, 1)); //#3369
            } //#3369
            $1.barcode = $1.upcacode; //#3371
        } //#3371
        if (($1.barcode.length != 11) && ($1.barcode.length != 12)) { //#3377
            $k[$j++] = 'bwipp.upcAbadLength'; //#3376
            $k[$j++] = "UPC-A must be 11 or 12 digits"; //#3376
            $0.raiseerror(); //#3376
        } //#3376
        var _1I = $1.barcode; //#3378
        for (var _1J = 0, _1K = _1I.length; _1J < _1K; _1J++) { //#3382
            var _1L = $get(_1I, _1J); //#3382
            if ((_1L < 48) || (_1L > 57)) { //#3381
                $k[$j++] = 'bwipp.upcAbadCharacter'; //#3380
                $k[$j++] = "UPC-A must contain only digits"; //#3380
                $0.raiseerror(); //#3380
            } //#3380
        } //#3380
        if ((($1.addon.length != 0) && ($1.addon.length != 2)) && ($1.addon.length != 5)) { //#3385
            $k[$j++] = 'bwipp.upcAbadAddOnLength'; //#3384
            $k[$j++] = "Add-on for UPC-A must be 2 or 5 digits"; //#3384
            $0.raiseerror(); //#3384
        } //#3384
        $1.pad = $s(12); //#3388
        $1.checksum = 0; //#3389
        for (var _1Q = 0; _1Q <= 10; _1Q += 1) { //#3398
            $1.i = _1Q; //#3391
            $1.barchar = $get($1.barcode, $1.i) - 48; //#3392
            if (($1.i % 2) != 0) { //#3396
                $1.checksum = $1.checksum + $1.barchar; //#3394
            } else { //#3396
                $1.checksum = $1.checksum + ($1.barchar * 3); //#3396
            } //#3396
        } //#3396
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#3399
        if ($1.barcode.length == 12) { //#3404
            if ($get($1.barcode, 11) != ($1.checksum + 48)) { //#3403
                $k[$j++] = 'bwipp.upcAbadCheckDigit'; //#3402
                $k[$j++] = "Incorrect UPC check digit provided"; //#3402
                $0.raiseerror(); //#3402
            } //#3402
        } //#3402
        $puti($1.pad, 0, $1.barcode); //#3405
        $put($1.pad, 11, $1.checksum + 48); //#3406
        $1.barcode = $1.pad; //#3407
        $1.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "11111", "111"]); //#3414
        $1.barchars = "0123456789"; //#3417
        $1.sbs = $s(59); //#3419
        $1.txt = $a(12); //#3420
        $puti($1.sbs, 0, $get($1.encs, 10)); //#3423
        for (var _1p = 0; _1p <= 5; _1p += 1) { //#3440
            $1.i = _1p; //#3428
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#3429
            $j--; //#3430
            $1.indx = $k[--$j].length; //#3431
            $j -= 2; //#3432
            $1.enc = $get($1.encs, $1.indx); //#3433
            $puti($1.sbs, ($1.i * 4) + 3, $1.enc); //#3434
            if ($1.i == 0) { //#3438
                $put($1.txt, 0, $a([$geti($1.barcode, 0, 1), $1.textxoffset, $1.textyoffset, $1.textfont, $f($1.textsize - 2)])); //#3436
            } else { //#3438
                $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f((($1.i * 7) + 11) + $1.textxoffset), $1.textyoffset, $1.textfont, $1.textsize])); //#3438
            } //#3438
        } //#3438
        $puti($1.sbs, (6 * 4) + 3, $get($1.encs, 11)); //#3443
        for (var _2O = 6; _2O <= 11; _2O += 1) { //#3460
            $1.i = _2O; //#3448
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#3449
            $j--; //#3450
            $1.indx = $k[--$j].length; //#3451
            $j -= 2; //#3452
            $1.enc = $get($1.encs, $1.indx); //#3453
            $puti($1.sbs, ($1.i * 4) + 8, $1.enc); //#3454
            if ($1.i == 11) { //#3458
                $put($1.txt, 11, $a([$geti($1.barcode, 11, 1), $f($1.textxoffset + 104), $1.textyoffset, $1.textfont, $f($1.textsize - 2)])); //#3456
            } else { //#3458
                $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(($f(($1.i * 7) + $1.textxoffset)) + 15), $1.textyoffset, $1.textfont, $1.textsize])); //#3458
            } //#3458
        } //#3458
        $puti($1.sbs, 56, $get($1.encs, 12)); //#3463
        $k[$j++] = Infinity; //#3465
        var _2x = $1.sbs; //#3465
        for (var _2y = 0, _2z = _2x.length; _2y < _2z; _2y++) { //#3465
            $k[$j++] = $get(_2x, _2y) - 48; //#3465
        } //#3465
        $1.sbs = $a(); //#3465
        if ($1.includetext) { //#3472
            $k[$j++] = Infinity; //#3467
            for (var _33 = 0, _34 = 4; _33 < _34; _33++) { //#3467
                $k[$j++] = $1.height; //#3467
            } //#3467
            for (var _36 = 0, _37 = 10; _36 < _37; _36++) { //#3467
                $k[$j++] = $f($1.height - 0.075); //#3467
            } //#3467
            $k[$j++] = $1.height; //#3467
            $k[$j++] = $1.height; //#3467
            for (var _3B = 0, _3C = 10; _3B < _3C; _3B++) { //#3467
                $k[$j++] = $f($1.height - 0.075); //#3467
            } //#3467
            for (var _3E = 0, _3F = 4; _3E < _3F; _3E++) { //#3467
                $k[$j++] = $1.height; //#3467
            } //#3467
            $1.bhs = $a(); //#3467
            $k[$j++] = Infinity; //#3468
            $k[$j++] = 0; //#3468
            $k[$j++] = 0; //#3468
            $k[$j++] = 0; //#3468
            $k[$j++] = 0; //#3468
            for (var _3I = 0, _3J = 10; _3I < _3J; _3I++) { //#3468
                $k[$j++] = 0.075; //#3468
            } //#3468
            $k[$j++] = 0; //#3468
            $k[$j++] = 0; //#3468
            for (var _3K = 0, _3L = 10; _3K < _3L; _3K++) { //#3468
                $k[$j++] = 0.075; //#3468
            } //#3468
            $k[$j++] = 0; //#3468
            $k[$j++] = 0; //#3468
            $k[$j++] = 0; //#3468
            $k[$j++] = 0; //#3468
            $1.bbs = $a(); //#3468
        } else { //#3472
            $k[$j++] = Infinity; //#3470
            for (var _3N = 0, _3O = 30; _3N < _3O; _3N++) { //#3470
                $k[$j++] = $1.height; //#3470
            } //#3470
            $1.bhs = $a(); //#3470
            $k[$j++] = Infinity; //#3471
            for (var _3R = 0, _3S = 30; _3R < _3S; _3R++) { //#3471
                $k[$j++] = 0; //#3471
            } //#3471
            $1.bbs = $a(); //#3471
            $1.txt = $a([]); //#3472
        } //#3472
        $1.guardrightypos = 0; //#3474
        if ($ne($1.addon, "")) { //#3495
            $k[$j++] = Infinity; //#3485
            $k[$j++] = 'dontdraw'; //#3482
            $k[$j++] = true; //#3482
            $k[$j++] = 'includetext'; //#3482
            $k[$j++] = true; //#3482
            $k[$j++] = 'height'; //#3482
            $k[$j++] = $f($1.height - 0.15); //#3482
            $k[$j++] = 'textxoffset'; //#3482
            if ($1.addontextxoffset != "unset") { //#3482
                $k[$j++] = $1.addontextxoffset; //#3482
            } else { //#3482
                $k[$j++] = $f(95 + $1.addongap); //#3482
            } //#3482
            if ($1.addontextyoffset != "unset") { //#3483
                $k[$j++] = 'textyoffset'; //#3483
                $k[$j++] = $1.addontextyoffset; //#3483
            } //#3483
            var _3d = ($1.addontextsize != "unset") ? $1.addontextsize : $1.textsize; //#3484
            var _3f = $ne($1.addontextfont, "unset") ? $1.addontextfont : $1.textfont; //#3485
            $k[$j++] = 'textsize'; //#3485
            $k[$j++] = _3d; //#3485
            $k[$j++] = 'textfont'; //#3485
            $k[$j++] = _3f; //#3485
            $1.addopts = $d(); //#3486
            if ($1.addon.length == 2) { //#3487
                $k[$j++] = $1.addon; //#3487
                $k[$j++] = $1.addopts; //#3487
                $0.ean2(); //#3487
            } //#3487
            if ($1.addon.length == 5) { //#3488
                $k[$j++] = $1.addon; //#3488
                $k[$j++] = $1.addopts; //#3488
                $0.ean5(); //#3488
            } //#3488
            $1.addcode = $k[--$j]; //#3489
            $k[$j++] = Infinity; //#3490
            $aload($1.sbs); //#3490
            $k[$j++] = $1.addongap; //#3490
            $aload($get($1.addcode, "sbs")); //#3490
            $1.sbs = $a(); //#3490
            $k[$j++] = Infinity; //#3491
            $aload($1.bhs); //#3491
            $forall($get($1.addcode, "bhs"), function() { //#3491
                var _3w = $k[--$j]; //#3491
                $k[$j++] = $f(_3w - 0.075); //#3491
            }); //#3491
            $1.bhs = $a(); //#3491
            $k[$j++] = Infinity; //#3492
            $aload($1.bbs); //#3492
            $forall($get($1.addcode, "bbs"), function() { //#3492
                var _41 = $k[--$j]; //#3492
                $k[$j++] = $f(_41 + 0.075); //#3492
            }); //#3492
            $1.bbs = $a(); //#3492
            $k[$j++] = Infinity; //#3493
            $aload($1.txt); //#3493
            $aload($get($1.addcode, "txt")); //#3493
            $1.txt = $a(); //#3493
            $1.guardrightypos = $f(($1.height * 72) - 6); //#3494
        } //#3494
        var _4E = new Map([
            ["ren", $0.renlinear],
            ["sbs", $1.sbs],
            ["bhs", $1.bhs],
            ["bbs", $1.bbs],
            ["txt", $1.txt],
            ["opt", $1.options],
            ["guardrightpos", 10],
            ["guardrightypos", $1.guardrightypos],
            ["borderbottom", 5]
        ]); //#3507
        $k[$j++] = _4E; //#3510
        if (!$1.dontdraw) { //#3510
            $0.renlinear(); //#3510
        } //#3510
    };
    $0.upce = function() {
        var $1 = {}; //#3541
        $1.options = $k[--$j]; //#3543
        $1.barcode = $k[--$j]; //#3544
        $1.dontdraw = false; //#3546
        $1.includetext = false; //#3547
        $1.textfont = "OCR-B"; //#3548
        $1.textsize = 12; //#3549
        $1.textxoffset = -7; //#3550
        $1.textyoffset = -4; //#3551
        $1.height = 1; //#3552
        $1.addongap = 12; //#3553
        $1.addontextfont = "unset"; //#3554
        $1.addontextsize = "unset"; //#3555
        $1.addontextxoffset = "unset"; //#3556
        $1.addontextyoffset = "unset"; //#3557
        $forall($1.options, function() { //#3568
            var _3 = $k[--$j]; //#3568
            $1[$k[--$j]] = _3; //#3568
        }); //#3568
        $1.textfont = "" + $1.textfont; //#3570
        $1.textsize = +$1.textsize; //#3571
        $1.textxoffset = +$1.textxoffset; //#3572
        $1.textyoffset = +$1.textyoffset; //#3573
        $1.height = +$1.height; //#3574
        $1.addongap = +$1.addongap; //#3575
        if ($ne($1.addontextfont, "unset")) { //#3576
            $1.addontextfont = "" + $1.addontextfont; //#3576
        } //#3576
        if ($ne($1.addontextsize, "unset")) { //#3577
            $1.addontextsize = +$1.addontextsize; //#3577
        } //#3577
        if ($ne($1.addontextxoffset, "unset")) { //#3578
            $1.addontextxoffset = +$1.addontextxoffset; //#3578
        } //#3578
        if ($ne($1.addontextyoffset, "unset")) { //#3579
            $1.addontextyoffset = +$1.addontextyoffset; //#3579
        } //#3579
        $search($1.barcode, " "); //#3582
        if ($k[--$j]) { //#3588
            $1.barcode = $k[--$j]; //#3583
            $j--; //#3584
            $1.addon = $k[--$j]; //#3585
        } else { //#3588
            $j--; //#3587
            $1.addon = ""; //#3588
        } //#3588
        if (($1.barcode.length == 11) || ($1.barcode.length == 12)) { //#3630
            $forall($1.barcode, function() { //#3598
                var _Q = $k[--$j]; //#3595
                if ((_Q < 48) || (_Q > 57)) { //#3597
                    $k[$j++] = 'bwipp.upcEupcAbadCharacter'; //#3596
                    $k[$j++] = "UPC-A must contain only digits"; //#3596
                    $0.raiseerror(); //#3596
                } //#3596
            }); //#3596
            for (var _R = 0, _S = 1; _R < _S; _R++) { //#3625
                var _U = ($1.barcode.length == 12) ? 8 : 7; //#3600
                $1.upcecode = $s(_U); //#3600
                if (($f($get($1.barcode, 3) - 48) <= 2) && $eq($geti($1.barcode, 4, 4), "0000")) { //#3606
                    $puti($1.upcecode, 0, $geti($1.barcode, 0, 3)); //#3602
                    $puti($1.upcecode, 3, $geti($1.barcode, 8, 3)); //#3603
                    $puti($1.upcecode, 6, $geti($1.barcode, 3, 1)); //#3604
                    break; //#3605
                } //#3605
                if ($eq($geti($1.barcode, 4, 5), "00000")) { //#3612
                    $puti($1.upcecode, 0, $geti($1.barcode, 0, 4)); //#3608
                    $puti($1.upcecode, 4, $geti($1.barcode, 9, 2)); //#3609
                    $puti($1.upcecode, 6, "3"); //#3610
                    break; //#3611
                } //#3611
                if ($eq($geti($1.barcode, 5, 5), "00000")) { //#3618
                    $puti($1.upcecode, 0, $geti($1.barcode, 0, 5)); //#3614
                    $puti($1.upcecode, 5, $geti($1.barcode, 10, 1)); //#3615
                    $puti($1.upcecode, 6, "4"); //#3616
                    break; //#3617
                } //#3617
                if (($f($get($1.barcode, 10) - 48) >= 5) && $eq($geti($1.barcode, 6, 4), "0000")) { //#3623
                    $puti($1.upcecode, 0, $geti($1.barcode, 0, 6)); //#3620
                    $puti($1.upcecode, 6, $geti($1.barcode, 10, 1)); //#3621
                    break; //#3622
                } //#3622
                $k[$j++] = 'bwipp.upcEupcAnotCompressible'; //#3624
                $k[$j++] = "UPC-A cannot be converted to a UPC-E"; //#3624
                $0.raiseerror(); //#3624
            } //#3624
            if ($1.barcode.length == 12) { //#3628
                $puti($1.upcecode, 7, $geti($1.barcode, 11, 1)); //#3627
            } //#3627
            $1.barcode = $1.upcecode; //#3629
        } //#3629
        if (($1.barcode.length != 7) && ($1.barcode.length != 8)) { //#3635
            $k[$j++] = 'bwipp.upcEbadLength'; //#3634
            $k[$j++] = "UPC-E must be 7 or 8 digits"; //#3634
            $0.raiseerror(); //#3634
        } //#3634
        var _1I = $1.barcode; //#3636
        for (var _1J = 0, _1K = _1I.length; _1J < _1K; _1J++) { //#3640
            var _1L = $get(_1I, _1J); //#3640
            if ((_1L < 48) || (_1L > 57)) { //#3639
                $k[$j++] = 'bwipp.upcEbadCharacter'; //#3638
                $k[$j++] = "UPC-E must contain only digits"; //#3638
                $0.raiseerror(); //#3638
            } //#3638
        } //#3638
        if ((($1.addon.length != 0) && ($1.addon.length != 2)) && ($1.addon.length != 5)) { //#3643
            $k[$j++] = 'bwipp.upcEbadAddOnLength'; //#3642
            $k[$j++] = "Add-on for UPC-E must be 2 or 5 digits"; //#3642
            $0.raiseerror(); //#3642
        } //#3642
        var _1Q = $get($1.barcode, 0); //#3646
        if ((_1Q != 48) && (_1Q != 49)) { //#3648
            $k[$j++] = 'bwipp.upcEbadNumberSystem'; //#3647
            $k[$j++] = "UPC-E must have number system 0 or 1"; //#3647
            $0.raiseerror(); //#3647
        } //#3647
        $1.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "111111"]); //#3655
        $1.barchars = "0123456789"; //#3658
        $1.mirrormaps = $a(["000111", "001011", "001101", "001110", "010011", "011001", "011100", "010101", "010110", "011010"]); //#3663
        for (var _1T = 0, _1U = 1; _1T < _1U; _1T++) { //#3693
            $1.upcacode = $s(11); //#3667
            if (($get($1.barcode, 6) - 48) <= 2) { //#3674
                $puti($1.upcacode, 0, $geti($1.barcode, 0, 3)); //#3669
                $puti($1.upcacode, 3, $geti($1.barcode, 6, 1)); //#3670
                $puti($1.upcacode, 4, "0000"); //#3671
                $puti($1.upcacode, 8, $geti($1.barcode, 3, 3)); //#3672
                break; //#3673
            } //#3673
            if (($get($1.barcode, 6) - 48) == 3) { //#3680
                $puti($1.upcacode, 0, $geti($1.barcode, 0, 4)); //#3676
                $puti($1.upcacode, 4, "00000"); //#3677
                $puti($1.upcacode, 9, $geti($1.barcode, 4, 2)); //#3678
                break; //#3679
            } //#3679
            if (($get($1.barcode, 6) - 48) == 4) { //#3686
                $puti($1.upcacode, 0, $geti($1.barcode, 0, 5)); //#3682
                $puti($1.upcacode, 5, "00000"); //#3683
                $puti($1.upcacode, 10, $geti($1.barcode, 5, 1)); //#3684
                break; //#3685
            } //#3685
            if (($get($1.barcode, 6) - 48) >= 5) { //#3692
                $puti($1.upcacode, 0, $geti($1.barcode, 0, 6)); //#3688
                $puti($1.upcacode, 6, "0000"); //#3689
                $puti($1.upcacode, 10, $geti($1.barcode, 6, 1)); //#3690
                break; //#3691
            } //#3691
        } //#3691
        $1.checksum = 0; //#3694
        for (var _29 = 0; _29 <= 10; _29 += 1) { //#3703
            $1.i = _29; //#3696
            $1.barchar = $get($1.upcacode, $1.i) - 48; //#3697
            if (($1.i % 2) != 0) { //#3701
                $1.checksum = $1.checksum + $1.barchar; //#3699
            } else { //#3701
                $1.checksum = $1.checksum + ($1.barchar * 3); //#3701
            } //#3701
        } //#3701
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#3704
        if ($1.barcode.length == 8) { //#3709
            if ($get($1.barcode, 7) != ($1.checksum + 48)) { //#3708
                $k[$j++] = 'bwipp.upcEbadCheckDigit'; //#3707
                $k[$j++] = "Incorrect UPC check digit provided"; //#3707
                $0.raiseerror(); //#3707
            } //#3707
        } //#3707
        $1.pad = $s(8); //#3710
        $puti($1.pad, 0, $1.barcode); //#3711
        $put($1.pad, 7, $1.checksum + 48); //#3712
        $1.barcode = $1.pad; //#3713
        $1.txt = $a(8); //#3714
        $put($1.txt, 0, $a([$geti($1.barcode, 0, 1), $1.textxoffset, $1.textyoffset, $1.textfont, $f($1.textsize - 2)])); //#3715
        $1.mirrormap = $get($1.mirrormaps, $1.checksum); //#3718
        if ($get($1.barcode, 0) == 48) { //#3732
            $1.invt = $s($1.mirrormap.length); //#3722
            for (var _2l = 0, _2k = $1.mirrormap.length - 1; _2l <= _2k; _2l += 1) { //#3730
                $1.i = _2l; //#3724
                if ($get($1.mirrormap, $1.i) == 48) { //#3728
                    $put($1.invt, $1.i, 49); //#3726
                } else { //#3728
                    $put($1.invt, $1.i, 48); //#3728
                } //#3728
            } //#3728
            $1.mirrormap = $1.invt; //#3731
        } //#3731
        $1.sbs = $s(33); //#3734
        $puti($1.sbs, 0, $get($1.encs, 10)); //#3737
        for (var _2y = 1; _2y <= 6; _2y += 1) { //#3759
            $1.i = _2y; //#3740
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#3742
            $j--; //#3743
            $1.indx = $k[--$j].length; //#3744
            $j -= 2; //#3745
            $1.enc = $get($1.encs, $1.indx); //#3746
            if ($get($1.mirrormap, $1.i - 1) == 49) { //#3756
                $1.enclen = $1.enc.length; //#3748
                $1.revenc = $s($1.enclen); //#3749
                for (var _3F = 0, _3E = $1.enclen - 1; _3F <= _3E; _3F += 1) { //#3754
                    $1.j = _3F; //#3751
                    $1.char = $get($1.enc, $1.j); //#3752
                    $put($1.revenc, ($1.enclen - $1.j) - 1, $1.char); //#3753
                } //#3753
                $1.enc = $1.revenc; //#3755
            } //#3755
            $puti($1.sbs, (($1.i - 1) * 4) + 3, $1.enc); //#3757
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(((($1.i - 1) * 7) + 11) + $1.textxoffset), $1.textyoffset, $1.textfont, $1.textsize])); //#3758
        } //#3758
        $put($1.txt, 7, $a([$geti($1.barcode, 7, 1), $f(((6 * 7) + 18) + $1.textxoffset), $1.textyoffset, $1.textfont, $f($1.textsize - 2)])); //#3761
        $puti($1.sbs, 27, $get($1.encs, 11)); //#3764
        $k[$j++] = Infinity; //#3766
        var _3n = $1.sbs; //#3766
        for (var _3o = 0, _3p = _3n.length; _3o < _3p; _3o++) { //#3766
            $k[$j++] = $get(_3n, _3o) - 48; //#3766
        } //#3766
        $1.sbs = $a(); //#3766
        if ($1.includetext) { //#3773
            $k[$j++] = Infinity; //#3768
            $k[$j++] = $1.height; //#3768
            $k[$j++] = $1.height; //#3768
            for (var _3v = 0, _3w = 12; _3v < _3w; _3v++) { //#3768
                $k[$j++] = $f($1.height - 0.075); //#3768
            } //#3768
            $k[$j++] = $1.height; //#3768
            $k[$j++] = $1.height; //#3768
            $k[$j++] = $1.height; //#3768
            $1.bhs = $a(); //#3768
            $k[$j++] = Infinity; //#3769
            $k[$j++] = 0; //#3769
            $k[$j++] = 0; //#3769
            for (var _42 = 0, _43 = 12; _42 < _43; _42++) { //#3769
                $k[$j++] = 0.075; //#3769
            } //#3769
            $k[$j++] = 0; //#3769
            $k[$j++] = 0; //#3769
            $k[$j++] = 0; //#3769
            $1.bbs = $a(); //#3769
        } else { //#3773
            $k[$j++] = Infinity; //#3771
            for (var _45 = 0, _46 = 17; _45 < _46; _45++) { //#3771
                $k[$j++] = $1.height; //#3771
            } //#3771
            $1.bhs = $a(); //#3771
            $k[$j++] = Infinity; //#3772
            for (var _49 = 0, _4A = 17; _49 < _4A; _49++) { //#3772
                $k[$j++] = 0; //#3772
            } //#3772
            $1.bbs = $a(); //#3772
            $1.txt = $a([]); //#3773
        } //#3773
        $1.guardrightypos = 0; //#3775
        if ($ne($1.addon, "")) { //#3796
            $k[$j++] = Infinity; //#3786
            $k[$j++] = 'dontdraw'; //#3783
            $k[$j++] = true; //#3783
            $k[$j++] = 'includetext'; //#3783
            $k[$j++] = true; //#3783
            $k[$j++] = 'height'; //#3783
            $k[$j++] = $f($1.height - 0.15); //#3783
            $k[$j++] = 'textxoffset'; //#3783
            if ($1.addontextxoffset != "unset") { //#3783
                $k[$j++] = $1.addontextxoffset; //#3783
            } else { //#3783
                $k[$j++] = $f(51 + $1.addongap); //#3783
            } //#3783
            if ($1.addontextyoffset != "unset") { //#3784
                $k[$j++] = 'textyoffset'; //#3784
                $k[$j++] = $1.addontextyoffset; //#3784
            } //#3784
            var _4L = ($1.addontextsize != "unset") ? $1.addontextsize : $1.textsize; //#3785
            var _4N = $ne($1.addontextfont, "unset") ? $1.addontextfont : $1.textfont; //#3786
            $k[$j++] = 'textsize'; //#3786
            $k[$j++] = _4L; //#3786
            $k[$j++] = 'textfont'; //#3786
            $k[$j++] = _4N; //#3786
            $1.addopts = $d(); //#3787
            if ($1.addon.length == 2) { //#3788
                $k[$j++] = $1.addon; //#3788
                $k[$j++] = $1.addopts; //#3788
                $0.ean2(); //#3788
            } //#3788
            if ($1.addon.length == 5) { //#3789
                $k[$j++] = $1.addon; //#3789
                $k[$j++] = $1.addopts; //#3789
                $0.ean5(); //#3789
            } //#3789
            $1.addcode = $k[--$j]; //#3790
            $k[$j++] = Infinity; //#3791
            $aload($1.sbs); //#3791
            $k[$j++] = $1.addongap; //#3791
            $aload($get($1.addcode, "sbs")); //#3791
            $1.sbs = $a(); //#3791
            $k[$j++] = Infinity; //#3792
            $aload($1.bhs); //#3792
            $forall($get($1.addcode, "bhs"), function() { //#3792
                var _4e = $k[--$j]; //#3792
                $k[$j++] = $f(_4e - 0.075); //#3792
            }); //#3792
            $1.bhs = $a(); //#3792
            $k[$j++] = Infinity; //#3793
            $aload($1.bbs); //#3793
            $forall($get($1.addcode, "bbs"), function() { //#3793
                var _4j = $k[--$j]; //#3793
                $k[$j++] = $f(_4j + 0.075); //#3793
            }); //#3793
            $1.bbs = $a(); //#3793
            $k[$j++] = Infinity; //#3794
            $aload($1.txt); //#3794
            $aload($get($1.addcode, "txt")); //#3794
            $1.txt = $a(); //#3794
            $1.guardrightypos = $f(($1.height * 72) - 6); //#3795
        } //#3795
        var _4w = new Map([
            ["ren", $0.renlinear],
            ["sbs", $1.sbs],
            ["bhs", $1.bhs],
            ["bbs", $1.bbs],
            ["txt", $1.txt],
            ["opt", $1.options],
            ["guardrightpos", 10],
            ["guardrightypos", $1.guardrightypos],
            ["borderbottom", 5]
        ]); //#3808
        $k[$j++] = _4w; //#3811
        if (!$1.dontdraw) { //#3811
            $0.renlinear(); //#3811
        } //#3811
    };
    $0.isbn = function() {
        var $1 = {}; //#3841
        $1.options = $k[--$j]; //#3843
        $1.barcode = $k[--$j]; //#3844
        $1.dontdraw = false; //#3846
        $1.includetext = false; //#3847
        $1.isbntextfont = 'OCR-A'; //#3848
        $1.isbntextsize = 8; //#3849
        $1.isbntextxoffset = "unset"; //#3850
        $1.isbntextyoffset = "unset"; //#3851
        $1.height = 1; //#3852
        $1.addongap = 12; //#3853
        $1.legacy = false; //#3854
        $forall($1.options, function() { //#3865
            var _3 = $k[--$j]; //#3865
            $1[$k[--$j]] = _3; //#3865
        }); //#3865
        $1.isbntextfont = "" + $1.isbntextfont; //#3867
        $1.isbntextsize = +$1.isbntextsize; //#3868
        if ($ne($1.isbntextxoffset, "unset")) { //#3869
            $1.isbntextxoffset = +$1.isbntextxoffset; //#3869
        } //#3869
        if ($ne($1.isbntextyoffset, "unset")) { //#3870
            $1.isbntextyoffset = +$1.isbntextyoffset; //#3870
        } //#3870
        $1.height = +$1.height; //#3871
        $1.addongap = +$1.addongap; //#3872
        $search($1.barcode, " "); //#3875
        if ($k[--$j]) { //#3881
            $1.barcode = $k[--$j]; //#3876
            $j--; //#3877
            $1.addon = $k[--$j]; //#3878
        } else { //#3881
            $j--; //#3880
            $1.addon = ""; //#3881
        } //#3881
        if ((($1.barcode.length != 15) && ($1.barcode.length != 17)) && (($1.barcode.length != 11) && ($1.barcode.length != 13))) { //#3888
            $k[$j++] = 'bwipp.isbnBadLength'; //#3887
            $k[$j++] = "ISBN-13 must be 15 or 17 characters including dashes. ISBN-10 must be 11 or 13 characters including dashes"; //#3887
            $0.raiseerror(); //#3887
        } //#3887
        if ((($1.addon.length != 0) && ($1.addon.length != 2)) && ($1.addon.length != 5)) { //#3891
            $k[$j++] = 'bwipp.isbnBadAddOnLength'; //#3890
            $k[$j++] = "Add-on for ISBN must be 2 or 5 digits"; //#3890
            $0.raiseerror(); //#3890
        } //#3890
        if ($1.barcode.length >= 15) { //#3955
            var _Q = $geti($1.barcode, 0, 4); //#3893
            if ($ne(_Q, "978-") && $ne(_Q, "979-")) { //#3895
                $k[$j++] = 'bwipp.isbn13badPrefix'; //#3894
                $k[$j++] = "ISBN-13 prefix must be 978- or 979-"; //#3894
                $0.raiseerror(); //#3894
            } //#3894
            $1.wasdash = false; //#3896
            $1.numdash = 0; //#3896
            $1.numdigit = 0; //#3896
            var _S = $geti($1.barcode, 5, 9); //#3897
            for (var _T = 0, _U = _S.length; _T < _U; _T++) { //#3910
                var _V = $get(_S, _T); //#3910
                $k[$j++] = _V; //#3905
                if (_V == 45) { //#3905
                    if ($1.wasdash) { //#3902
                        $k[$j++] = 'bwipp.isbn13adjacentDashes'; //#3901
                        $k[$j++] = "ISBN-13 does not permit adjacent dashes"; //#3901
                        $0.raiseerror(); //#3901
                    } //#3901
                    $1.wasdash = true; //#3903
                    $1.numdash = $1.numdash + 1; //#3904
                } //#3904
                var _Y = $k[--$j]; //#3906
                if ((_Y >= 48) && (_Y <= 57)) { //#3909
                    $1.wasdash = false; //#3907
                    $1.numdigit = $1.numdigit + 1; //#3908
                } //#3908
            } //#3908
            if (($1.numdash != 2) || ($1.numdigit != 7)) { //#3913
                $k[$j++] = 'bwipp.isbn13numDashesDigits'; //#3912
                $k[$j++] = "Incorrect number of dashes and digits for ISBN-13"; //#3912
                $0.raiseerror(); //#3912
            } //#3912
            var _d = $get($1.barcode, 14); //#3914
            if ((_d < 48) || (_d > 57)) { //#3916
                $k[$j++] = 'bwipp.isbn13character15'; //#3915
                $k[$j++] = "ISBN-13 character 15 must be a digit"; //#3915
                $0.raiseerror(); //#3915
            } //#3915
            if ($1.barcode.length == 17) { //#3924
                if ($ne($geti($1.barcode, 15, 1), "-")) { //#3920
                    $k[$j++] = 'bwipp.isbn13character16'; //#3919
                    $k[$j++] = "ISBN-13 penultimate character must be a dash"; //#3919
                    $0.raiseerror(); //#3919
                } //#3919
                var _i = $get($1.barcode, 16); //#3921
                if ((_i < 48) || (_i > 57)) { //#3923
                    $k[$j++] = 'bwipp.isbn13character17'; //#3922
                    $k[$j++] = "ISBN-13 final character must be a digit"; //#3922
                    $0.raiseerror(); //#3922
                } //#3922
            } //#3922
        } else { //#3955
            var _k = $get($1.barcode, 0); //#3926
            if ((_k < 48) || (_k > 57)) { //#3928
                $k[$j++] = 'bwipp.isbn10FirstDigit'; //#3927
                $k[$j++] = "ISBN-10 first character must be a digit"; //#3927
                $0.raiseerror(); //#3927
            } //#3927
            $1.wasdash = false; //#3929
            $1.numdash = 0; //#3929
            $1.numdigit = 0; //#3929
            var _m = $geti($1.barcode, 1, 9); //#3930
            for (var _n = 0, _o = _m.length; _n < _o; _n++) { //#3943
                var _p = $get(_m, _n); //#3943
                $k[$j++] = _p; //#3938
                if (_p == 45) { //#3938
                    if ($1.wasdash) { //#3935
                        $k[$j++] = 'bwipp.isbn10adjacentDashes'; //#3934
                        $k[$j++] = "ISBN-10 does not permit adjacent dashes"; //#3934
                        $0.raiseerror(); //#3934
                    } //#3934
                    $1.wasdash = true; //#3936
                    $1.numdash = $1.numdash + 1; //#3937
                } //#3937
                var _s = $k[--$j]; //#3939
                if ((_s >= 48) && (_s <= 57)) { //#3942
                    $1.wasdash = false; //#3940
                    $1.numdigit = $1.numdigit + 1; //#3941
                } //#3941
            } //#3941
            if (($1.numdash != 2) || ($1.numdigit != 7)) { //#3946
                $k[$j++] = 'bwipp.isbn10numDashesDigits'; //#3945
                $k[$j++] = "Incorrect number of dashes and digits for ISBN-10"; //#3945
                $0.raiseerror(); //#3945
            } //#3945
            var _x = $get($1.barcode, 10); //#3947
            if ((_x < 48) || (_x > 57)) { //#3949
                $k[$j++] = 'bwipp.isbn10character11'; //#3948
                $k[$j++] = "ISBN-10 character 11 must be a digit"; //#3948
                $0.raiseerror(); //#3948
            } //#3948
            if ($1.barcode.length == 13) { //#3957
                if ($ne($geti($1.barcode, 11, 1), "-")) { //#3953
                    $k[$j++] = 'bwipp.isbn10character12'; //#3952
                    $k[$j++] = "ISBN-10 penultimate character must be a dash"; //#3952
                    $0.raiseerror(); //#3952
                } //#3952
                var _12 = $get($1.barcode, 12); //#3954
                if (((_12 < 48) || (_12 > 57)) && (_12 != 88)) { //#3956
                    $k[$j++] = 'bwipp.isbn10character13'; //#3955
                    $k[$j++] = "ISBN-10 final character must be a digit or X"; //#3955
                    $0.raiseerror(); //#3955
                } //#3955
            } //#3955
        } //#3955
        $1.isbntxt = $1.barcode; //#3960
        if ($1.isbntxt.length <= 13) { //#3986
            $1.isbn = $s(10); //#3964
            $1.checksum = 0; //#3965
            $1.i = 0; //#3966
            $1.n = 0; //#3966
            for (;;) { //#3978
                $1.isbnchar = $f($get($1.isbntxt, $1.i) - 48); //#3968
                if ($1.isbnchar != -3) { //#3975
                    $put($1.isbn, $1.n, $f($1.isbnchar + 48)); //#3970
                    if ($1.n < 9) { //#3973
                        $1.checksum = $f($1.checksum + ((10 - $1.n) * $1.isbnchar)); //#3972
                    } //#3972
                    $1.n = $1.n + 1; //#3974
                } //#3974
                $1.i = $1.i + 1; //#3976
                if ($1.i == $1.isbntxt.length) { //#3977
                    break; //#3977
                } //#3977
            } //#3977
            var _1M = ($f(11 - ($1.checksum % 11))) % 11; //#3979
            $k[$j++] = 'checksum'; //#3979
            $k[$j++] = _1M; //#3979
            if (_1M == 10) { //#3979
                $j--; //#3979
                $k[$j++] = 40; //#3979
            } //#3979
            var _1N = $k[--$j]; //#3979
            $1[$k[--$j]] = $f(_1N + 48); //#3979
            $1.isbn = $geti($1.isbn, 0, 9); //#3980
            if ($1.isbntxt.length == 13) { //#3985
                if ($get($1.isbntxt, 12) != $1.checksum) { //#3984
                    $k[$j++] = 'bwipp.isbn10badCheckDigit'; //#3983
                    $k[$j++] = "Incorrect ISBN-10 check digit provided"; //#3983
                    $0.raiseerror(); //#3983
                } //#3983
            } //#3983
        } //#3983
        if ((!$1.legacy) || ($1.isbntxt.length >= 15)) { //#4022
            if ($1.isbntxt.length <= 13) { //#3996
                $1.pad = $s(15); //#3992
                $puti($1.pad, 0, "978-"); //#3993
                $puti($1.pad, 4, $geti($1.isbntxt, 0, 11)); //#3994
                $1.isbntxt = $1.pad; //#3995
            } //#3995
            $1.isbn = $s(13); //#3999
            $1.checksum = 0; //#4000
            $1.i = 0; //#4001
            $1.n = 0; //#4001
            for (;;) { //#4013
                $1.isbnchar = $get($1.isbntxt, $1.i) - 48; //#4003
                if ($1.isbnchar != -3) { //#4010
                    $put($1.isbn, $1.n, $1.isbnchar + 48); //#4005
                    if ($1.n < 12) { //#4008
                        $k[$j++] = 'checksum'; //#4007
                        $k[$j++] = $1.isbnchar; //#4007
                        if (($1.n % 2) != 0) { //#4007
                            var _1p = $k[--$j]; //#4007
                            $k[$j++] = _1p * 3; //#4007
                        } //#4007
                        var _1r = $k[--$j]; //#4007
                        $1[$k[--$j]] = $f(_1r + $1.checksum); //#4007
                    } //#4007
                    $1.n = $1.n + 1; //#4009
                } //#4009
                $1.i = $1.i + 1; //#4011
                if ($1.i == $1.isbntxt.length) { //#4012
                    break; //#4012
                } //#4012
            } //#4012
            $1.checksum = ((10 - ($1.checksum % 10)) % 10) + 48; //#4014
            $1.isbn = $geti($1.isbn, 0, 12); //#4015
            if ($1.isbntxt.length == 17) { //#4020
                if ($get($1.isbntxt, 16) != $1.checksum) { //#4019
                    $k[$j++] = 'bwipp.isbn13badCheckDigit'; //#4018
                    $k[$j++] = "Incorrect ISBN-13 check digit provided"; //#4018
                    $0.raiseerror(); //#4018
                } //#4018
            } //#4018
        } //#4018
        var _25 = ($1.isbn.length == 12) ? 22 : 18; //#4025
        $1.pad = $s(_25); //#4025
        $puti($1.pad, 0, "ISBN "); //#4026
        $puti($1.pad, 5, $1.isbntxt); //#4027
        $put($1.pad, $1.pad.length - 2, 45); //#4028
        $put($1.pad, $1.pad.length - 1, $1.checksum); //#4029
        $1.isbntxt = $1.pad; //#4030
        $1.barcode = $s(12); //#4033
        if ($1.isbn.length == 9) { //#4038
            $puti($1.barcode, 0, "978"); //#4035
            $puti($1.barcode, 3, $1.isbn); //#4036
        } else { //#4038
            $puti($1.barcode, 0, $1.isbn); //#4038
        } //#4038
        if ($ne($1.addon, "")) { //#4048
            var _2P = $s((12 + $1.addon.length) + 1); //#4043
            $puti(_2P, 0, $1.barcode); //#4044
            $puti(_2P, 12, " "); //#4045
            $puti(_2P, 13, $1.addon); //#4046
            $1.barcode = _2P; //#4047
        } //#4047
        $put($1.options, "dontdraw", true); //#4051
        $put($1.options, "addongap", $1.addongap); //#4052
        $k[$j++] = 'args'; //#4053
        $k[$j++] = $1.barcode; //#4053
        $k[$j++] = $1.options; //#4053
        $0.ean13(); //#4053
        var _2X = $k[--$j]; //#4053
        $1[$k[--$j]] = _2X; //#4053
        if ($1.includetext) { //#4072
            if ($1.isbntextxoffset == "unset") { //#4059
                $k[$j++] = 'isbntextxoffset'; //#4058
                if ($1.isbn.length == 9) { //#4058
                    $k[$j++] = -1; //#4058
                } else { //#4058
                    $k[$j++] = -12; //#4058
                } //#4058
                var _2c = $k[--$j]; //#4058
                $1[$k[--$j]] = _2c; //#4058
            } //#4058
            if ($1.isbntextyoffset == "unset") { //#4062
                $1.isbntextyoffset = $f(($1.height * 72) + 3); //#4061
            } //#4061
            var _2h = $get($1.args, "txt") !== undefined; //#4063
            if (_2h) { //#4070
                $1.txt = $get($1.args, "txt"); //#4064
                $1.newtxt = $a($1.txt.length + 1); //#4065
                $puti($1.newtxt, 0, $1.txt); //#4066
                $put($1.newtxt, $1.newtxt.length - 1, $a([$1.isbntxt, $1.isbntextxoffset, $1.isbntextyoffset, $1.isbntextfont, $1.isbntextsize])); //#4067
                $put($1.args, "txt", $1.newtxt); //#4068
            } else { //#4070
                $put($1.args, "txt", $a([$a([$1.isbntxt, $1.isbntextxoffset, $1.isbntextyoffset, $1.isbntextfont, $1.isbntextsize])])); //#4070
            } //#4070
        } //#4070
        $put($1.args, "opt", $1.options); //#4074
        $k[$j++] = $1.args; //#4077
        if (!$1.dontdraw) { //#4077
            $0.renlinear(); //#4077
        } //#4077
    };
    $0.ismn = function() {
        var $1 = {}; //#4107
        $1.options = $k[--$j]; //#4109
        $1.barcode = $k[--$j]; //#4110
        $1.dontdraw = false; //#4112
        $1.includetext = false; //#4113
        $1.ismntextfont = 'OCR-A'; //#4114
        $1.ismntextsize = 8; //#4115
        $1.ismntextxoffset = "unset"; //#4116
        $1.ismntextyoffset = "unset"; //#4117
        $1.height = 1; //#4118
        $1.addongap = 12; //#4119
        $1.legacy = false; //#4120
        $forall($1.options, function() { //#4131
            var _3 = $k[--$j]; //#4131
            $1[$k[--$j]] = _3; //#4131
        }); //#4131
        $1.ismntextfont = "" + $1.ismntextfont; //#4133
        $1.ismntextsize = +$1.ismntextsize; //#4134
        if ($ne($1.ismntextxoffset, "unset")) { //#4135
            $1.ismntextxoffset = +$1.ismntextxoffset; //#4135
        } //#4135
        if ($ne($1.ismntextyoffset, "unset")) { //#4136
            $1.ismntextyoffset = +$1.ismntextyoffset; //#4136
        } //#4136
        $1.height = +$1.height; //#4137
        $1.addongap = +$1.addongap; //#4138
        $search($1.barcode, " "); //#4141
        if ($k[--$j]) { //#4147
            $1.barcode = $k[--$j]; //#4142
            $j--; //#4143
            $1.addon = $k[--$j]; //#4144
        } else { //#4147
            $j--; //#4146
            $1.addon = ""; //#4147
        } //#4147
        if ((($1.barcode.length != 15) && ($1.barcode.length != 17)) && (($1.barcode.length != 11) && ($1.barcode.length != 13))) { //#4154
            $k[$j++] = 'bwipp.ismnBadLength'; //#4153
            $k[$j++] = "ISMN-13 must be 15 or 17 characters including dashes. ISMN-10 must be 11 or 13 characters including dashes"; //#4153
            $0.raiseerror(); //#4153
        } //#4153
        if ((($1.addon.length != 0) && ($1.addon.length != 2)) && ($1.addon.length != 5)) { //#4157
            $k[$j++] = 'bwipp.ismnBadAddOnLength'; //#4156
            $k[$j++] = "Add-on for ISMN must be 2 or 5 digits"; //#4156
            $0.raiseerror(); //#4156
        } //#4156
        if ($1.barcode.length >= 15) { //#4224
            if ($ne($geti($1.barcode, 0, 4), "979-")) { //#4161
                $k[$j++] = 'bwipp.ismn13badPrefix'; //#4160
                $k[$j++] = "ISMN-13 prefix must be 979-"; //#4160
                $0.raiseerror(); //#4160
            } //#4160
            $1.wasdash = false; //#4162
            $1.numdash = 0; //#4162
            $1.numdigit = 0; //#4162
            var _S = $geti($1.barcode, 5, 9); //#4163
            for (var _T = 0, _U = _S.length; _T < _U; _T++) { //#4176
                var _V = $get(_S, _T); //#4176
                $k[$j++] = _V; //#4171
                if (_V == 45) { //#4171
                    if ($1.wasdash) { //#4168
                        $k[$j++] = 'bwipp.ismn13adjacentDashes'; //#4167
                        $k[$j++] = "ISMN-13 does not permit adjacent dashes"; //#4167
                        $0.raiseerror(); //#4167
                    } //#4167
                    $1.wasdash = true; //#4169
                    $1.numdash = $1.numdash + 1; //#4170
                } //#4170
                var _Y = $k[--$j]; //#4172
                if ((_Y >= 48) && (_Y <= 57)) { //#4175
                    $1.wasdash = false; //#4173
                    $1.numdigit = $1.numdigit + 1; //#4174
                } //#4174
            } //#4174
            if (($1.numdash != 2) || ($1.numdigit != 7)) { //#4179
                $k[$j++] = 'bwipp.ismn13numDashesDigits'; //#4178
                $k[$j++] = "Incorrect number of dashes and digits for ISMN-13"; //#4178
                $0.raiseerror(); //#4178
            } //#4178
            var _d = $get($1.barcode, 14); //#4180
            if ((_d < 48) || (_d > 57)) { //#4182
                $k[$j++] = 'bwipp.ismn13character15'; //#4181
                $k[$j++] = "ISMN-13 character 15 must be a digit"; //#4181
                $0.raiseerror(); //#4181
            } //#4181
            if ($1.barcode.length == 17) { //#4190
                if ($ne($geti($1.barcode, 15, 1), "-")) { //#4186
                    $k[$j++] = 'bwipp.ismn13character16'; //#4185
                    $k[$j++] = "ISMN-13 penultimate character must be a dash"; //#4185
                    $0.raiseerror(); //#4185
                } //#4185
                var _i = $get($1.barcode, 16); //#4187
                if ((_i < 48) || (_i > 57)) { //#4189
                    $k[$j++] = 'bwipp.ismn13character17'; //#4188
                    $k[$j++] = "ISMN-13 final character must be a digit"; //#4188
                    $0.raiseerror(); //#4188
                } //#4188
            } //#4188
        } else { //#4224
            if ($ne($geti($1.barcode, 0, 2), "M-")) { //#4194
                $k[$j++] = 'bwipp.ismn10badPrefix'; //#4193
                $k[$j++] = "ISMN-10 prefix must be M-"; //#4193
                $0.raiseerror(); //#4193
            } //#4193
            var _m = $get($1.barcode, 2); //#4195
            if ((_m < 48) || (_m > 57)) { //#4197
                $k[$j++] = 'bwipp.ismn10character3'; //#4196
                $k[$j++] = "ISMN-10 character 3 must be a digit"; //#4196
                $0.raiseerror(); //#4196
            } //#4196
            $1.wasdash = false; //#4198
            $1.numdash = 0; //#4198
            $1.numdigit = 0; //#4198
            var _o = $geti($1.barcode, 3, 7); //#4199
            for (var _p = 0, _q = _o.length; _p < _q; _p++) { //#4212
                var _r = $get(_o, _p); //#4212
                $k[$j++] = _r; //#4207
                if (_r == 45) { //#4207
                    if ($1.wasdash) { //#4204
                        $k[$j++] = 'bwipp.ismn10adjacentDashes'; //#4203
                        $k[$j++] = "ISMN-10 does not permit adjacent dashes"; //#4203
                        $0.raiseerror(); //#4203
                    } //#4203
                    $1.wasdash = true; //#4205
                    $1.numdash = $1.numdash + 1; //#4206
                } //#4206
                var _u = $k[--$j]; //#4208
                if ((_u >= 48) && (_u <= 57)) { //#4211
                    $1.wasdash = false; //#4209
                    $1.numdigit = $1.numdigit + 1; //#4210
                } //#4210
            } //#4210
            if (($1.numdash != 1) || ($1.numdigit != 6)) { //#4215
                $k[$j++] = 'bwipp.ismn10numDashesDigits'; //#4214
                $k[$j++] = "Incorrect number of dashes and digits for ISMN-10"; //#4214
                $0.raiseerror(); //#4214
            } //#4214
            var _z = $get($1.barcode, 10); //#4216
            if ((_z < 48) || (_z > 57)) { //#4218
                $k[$j++] = 'bwipp.ismn10character11'; //#4217
                $k[$j++] = "ISMN-10 character 11 must be a digit"; //#4217
                $0.raiseerror(); //#4217
            } //#4217
            if ($1.barcode.length == 13) { //#4226
                if ($ne($geti($1.barcode, 11, 1), "-")) { //#4222
                    $k[$j++] = 'bwipp.ismn10character12'; //#4221
                    $k[$j++] = "ISMN-10 penultimate character must be a dash"; //#4221
                    $0.raiseerror(); //#4221
                } //#4221
                var _14 = $get($1.barcode, 12); //#4223
                if (((_14 < 48) || (_14 > 57)) && (_14 != 88)) { //#4225
                    $k[$j++] = 'bwipp.ismn10character13'; //#4224
                    $k[$j++] = "ISMN-10 final character must be a digit or X"; //#4224
                    $0.raiseerror(); //#4224
                } //#4224
            } //#4224
        } //#4224
        $1.ismntxt = $1.barcode; //#4229
        $1.legacytxt = ""; //#4232
        if ($1.ismntxt.length <= 13) { //#4239
            $1.legacytxt = $1.ismntxt; //#4234
            $1.pad = $s($1.ismntxt.length + 4); //#4235
            $puti($1.pad, 0, "979-0-"); //#4236
            $puti($1.pad, 6, $geti($1.ismntxt, 2, $1.ismntxt.length - 2)); //#4237
            $1.ismntxt = $1.pad; //#4238
        } //#4238
        $1.ismn = $s(13); //#4242
        $1.checksum = 0; //#4243
        $1.i = 0; //#4244
        $1.n = 0; //#4244
        for (;;) { //#4260
            $1.ismnchar = $get($1.ismntxt, $1.i) - 48; //#4246
            if ($1.ismnchar != -3) { //#4257
                $put($1.ismn, $1.n, $1.ismnchar + 48); //#4248
                if ($1.n < 12) { //#4255
                    if (($1.n % 2) == 0) { //#4253
                        $1.checksum = $1.ismnchar + $1.checksum; //#4251
                    } else { //#4253
                        $1.checksum = ($1.ismnchar * 3) + $1.checksum; //#4253
                    } //#4253
                } //#4253
                $1.n = $1.n + 1; //#4256
            } //#4256
            $1.i = $1.i + 1; //#4258
            if ($1.i == $1.ismntxt.length) { //#4259
                break; //#4259
            } //#4259
        } //#4259
        $1.checksum = ((10 - ($1.checksum % 10)) % 10) + 48; //#4261
        if (($1.barcode.length == 13) || ($1.barcode.length == 17)) { //#4268
            var _1b = $1.barcode; //#4265
            if ($get(_1b, _1b.length - 1) != $1.checksum) { //#4267
                $k[$j++] = 'bwipp.ismnBadCheckDigit'; //#4266
                $k[$j++] = "Incorrect ISMN check digit provided"; //#4266
                $0.raiseerror(); //#4266
            } //#4266
        } //#4266
        if ($1.legacy && $ne($1.legacytxt, "")) { //#4275
            $1.ismntxt = $1.legacytxt; //#4272
            $1.pad = $s(18); //#4273
        } else { //#4275
            $1.pad = $s(22); //#4275
        } //#4275
        $puti($1.pad, 0, "ISMN "); //#4277
        $puti($1.pad, 5, $1.ismntxt); //#4278
        $put($1.pad, $1.pad.length - 2, 45); //#4279
        $put($1.pad, $1.pad.length - 1, $1.checksum); //#4280
        $1.ismntxt = $1.pad; //#4281
        $1.barcode = $geti($1.ismn, 0, 12); //#4284
        if ($ne($1.addon, "")) { //#4293
            var _1w = $s((12 + $1.addon.length) + 1); //#4288
            $puti(_1w, 0, $1.barcode); //#4289
            $puti(_1w, 12, " "); //#4290
            $puti(_1w, 13, $1.addon); //#4291
            $1.barcode = _1w; //#4292
        } //#4292
        $put($1.options, "dontdraw", true); //#4296
        $put($1.options, "addongap", $1.addongap); //#4297
        $k[$j++] = 'args'; //#4298
        $k[$j++] = $1.barcode; //#4298
        $k[$j++] = $1.options; //#4298
        $0.ean13(); //#4298
        var _24 = $k[--$j]; //#4298
        $1[$k[--$j]] = _24; //#4298
        if ($1.includetext) { //#4317
            if ($1.ismntextxoffset == "unset") { //#4304
                $k[$j++] = 'ismntextxoffset'; //#4303
                if ($1.ismntxt.length == 18) { //#4303
                    $k[$j++] = -1; //#4303
                } else { //#4303
                    $k[$j++] = -12; //#4303
                } //#4303
                var _29 = $k[--$j]; //#4303
                $1[$k[--$j]] = _29; //#4303
            } //#4303
            if ($1.ismntextyoffset == "unset") { //#4307
                $1.ismntextyoffset = $f(($1.height * 72) + 3); //#4306
            } //#4306
            var _2E = $get($1.args, "txt") !== undefined; //#4308
            if (_2E) { //#4315
                $1.txt = $get($1.args, "txt"); //#4309
                $1.newtxt = $a($1.txt.length + 1); //#4310
                $puti($1.newtxt, 0, $1.txt); //#4311
                $put($1.newtxt, $1.newtxt.length - 1, $a([$1.ismntxt, $1.ismntextxoffset, $1.ismntextyoffset, $1.ismntextfont, $1.ismntextsize])); //#4312
                $put($1.args, "txt", $1.newtxt); //#4313
            } else { //#4315
                $put($1.args, "txt", $a([$a([$1.ismntxt, $1.ismntextxoffset, $1.ismntextyoffset, $1.ismntextfont, $1.ismntextsize])])); //#4315
            } //#4315
        } //#4315
        $put($1.args, "opt", $1.options); //#4319
        $k[$j++] = $1.args; //#4322
        if (!$1.dontdraw) { //#4322
            $0.renlinear(); //#4322
        } //#4322
    };
    $0.issn = function() {
        var $1 = {}; //#4352
        $1.options = $k[--$j]; //#4354
        $1.issntxt = $k[--$j]; //#4355
        $1.dontdraw = false; //#4357
        $1.includetext = false; //#4358
        $1.issntextfont = 'OCR-A'; //#4359
        $1.issntextsize = 8; //#4360
        $1.issntextxoffset = "unset"; //#4361
        $1.issntextyoffset = "unset"; //#4362
        $1.height = 1; //#4363
        $1.addongap = 12; //#4364
        $forall($1.options, function() { //#4375
            var _3 = $k[--$j]; //#4375
            $1[$k[--$j]] = _3; //#4375
        }); //#4375
        $1.issntextfont = "" + $1.issntextfont; //#4377
        $1.issntextsize = +$1.issntextsize; //#4378
        if ($ne($1.issntextxoffset, "unset")) { //#4379
            $1.issntextxoffset = +$1.issntextxoffset; //#4379
        } //#4379
        if ($ne($1.issntextyoffset, "unset")) { //#4380
            $1.issntextyoffset = +$1.issntextyoffset; //#4380
        } //#4380
        $1.height = +$1.height; //#4381
        $1.addongap = +$1.addongap; //#4382
        $search($1.issntxt, " "); //#4385
        if ($k[--$j]) { //#4391
            $1.issntxt = $k[--$j]; //#4386
            $j--; //#4387
            $1.seqvar = $k[--$j]; //#4388
        } else { //#4391
            $j--; //#4390
            $1.seqvar = "00"; //#4391
        } //#4391
        $search($1.seqvar, " "); //#4395
        if ($k[--$j]) { //#4401
            $1.seqvar = $k[--$j]; //#4396
            $j--; //#4397
            $1.addon = $k[--$j]; //#4398
        } else { //#4401
            $j--; //#4400
            $1.addon = ""; //#4401
        } //#4401
        if (($1.issntxt.length != 8) && ($1.issntxt.length != 9)) { //#4407
            $k[$j++] = 'bwipp.issnBadLength'; //#4406
            $k[$j++] = "ISSN must be 8 or 9 characters including dash, in the format XXXX-XXXX"; //#4406
            $0.raiseerror(); //#4406
        } //#4406
        var _O = $geti($1.issntxt, 0, 4); //#4408
        for (var _P = 0, _Q = _O.length; _P < _Q; _P++) { //#4412
            var _R = $get(_O, _P); //#4412
            if ((_R < 48) || (_R > 57)) { //#4411
                $k[$j++] = 'bwipp.issnFirstThroughFourthNotNumeric'; //#4410
                $k[$j++] = "ISSN first four characters must be numeral characters"; //#4410
                $0.raiseerror(); //#4410
            } //#4410
        } //#4410
        if ($ne($geti($1.issntxt, 4, 1), "-")) { //#4415
            $k[$j++] = 'bwipp.issnNeedsDash'; //#4414
            $k[$j++] = "ISSN fifth character must be a dash"; //#4414
            $0.raiseerror(); //#4414
        } //#4414
        var _V = $geti($1.issntxt, 5, 3); //#4416
        for (var _W = 0, _X = _V.length; _W < _X; _W++) { //#4420
            var _Y = $get(_V, _W); //#4420
            if ((_Y < 48) || (_Y > 57)) { //#4419
                $k[$j++] = 'bwipp.issnSixthThroughEighthNotNumeric'; //#4418
                $k[$j++] = "ISSN sixth through eighth characters must be numerals"; //#4418
                $0.raiseerror(); //#4418
            } //#4418
        } //#4418
        if ($1.issntxt.length == 9) { //#4425
            var _b = $get($1.issntxt, 8); //#4422
            if (((_b < 48) || (_b > 57)) && (_b != 88)) { //#4424
                $k[$j++] = 'bwipp.issnNinthCharacterBadFormat'; //#4423
                $k[$j++] = "ISSN ninth character must be a number or the character X"; //#4423
                $0.raiseerror(); //#4423
            } //#4423
        } //#4423
        if ($1.seqvar.length != 2) { //#4428
            $k[$j++] = 'bwipp.issnBadSequenceVariantLength'; //#4427
            $k[$j++] = "Sequence variant for ISSN must be 2 digits"; //#4427
            $0.raiseerror(); //#4427
        } //#4427
        $forall($1.seqvar, function() { //#4433
            var _e = $k[--$j]; //#4430
            if ((_e < 48) || (_e > 57)) { //#4432
                $k[$j++] = 'bwipp.issnSequenceVariantBadCharacter'; //#4431
                $k[$j++] = "Sequence variant for ISSN must contain only digits"; //#4431
                $0.raiseerror(); //#4431
            } //#4431
        }); //#4431
        if ((($1.addon.length != 0) && ($1.addon.length != 2)) && ($1.addon.length != 5)) { //#4436
            $k[$j++] = 'bwipp.issnBadAddOnLength'; //#4435
            $k[$j++] = "Add-on for ISSN must be 2 or 5 digits"; //#4435
            $0.raiseerror(); //#4435
        } //#4435
        $1.issn = $s(8); //#4439
        $1.checksum = 0; //#4440
        $1.i = 0; //#4441
        $1.n = 0; //#4441
        for (;;) { //#4453
            $1.issnchar = $f($get($1.issntxt, $1.i) - 48); //#4443
            if ($1.issnchar != -3) { //#4450
                $put($1.issn, $1.n, $f($1.issnchar + 48)); //#4445
                if ($1.n < 7) { //#4448
                    $1.checksum = $f($1.checksum + ($1.issnchar * (8 - $1.n))); //#4447
                } //#4447
                $1.n = $1.n + 1; //#4449
            } //#4449
            $1.i = $1.i + 1; //#4451
            if ($1.i == $1.issntxt.length) { //#4452
                break; //#4452
            } //#4452
        } //#4452
        $1.checksum = ($f(11 - ($1.checksum % 11))) % 11; //#4454
        var _10 = $f($1.checksum + 48); //#4455
        $k[$j++] = 'checksum'; //#4455
        $k[$j++] = _10; //#4455
        if (_10 == 58) { //#4455
            $j--; //#4455
            $k[$j++] = 88; //#4455
        } //#4455
        var _11 = $k[--$j]; //#4455
        $1[$k[--$j]] = _11; //#4455
        if ($1.issntxt.length == 9) { //#4460
            if ($get($1.issntxt, 8) != $1.checksum) { //#4459
                $k[$j++] = 'bwipp.issnBadCheckDigit'; //#4458
                $k[$j++] = "Incorrect ISSN check digit provided"; //#4458
                $0.raiseerror(); //#4458
            } //#4458
        } //#4458
        $1.pad = $s(14); //#4463
        $puti($1.pad, 0, "ISSN "); //#4464
        $puti($1.pad, 5, $1.issntxt); //#4465
        $put($1.pad, 13, $1.checksum); //#4466
        $1.issntxt = $1.pad; //#4467
        $1.barcode = $geti($1.issn, 0, 7); //#4470
        $1.barcode = $s(12); //#4473
        $puti($1.barcode, 0, "977"); //#4474
        $puti($1.barcode, 3, $1.issn); //#4475
        $puti($1.barcode, 10, $1.seqvar); //#4476
        if ($ne($1.addon, "")) { //#4485
            var _1O = $s((12 + $1.addon.length) + 1); //#4480
            $puti(_1O, 0, $1.barcode); //#4481
            $puti(_1O, 12, " "); //#4482
            $puti(_1O, 13, $1.addon); //#4483
            $1.barcode = _1O; //#4484
        } //#4484
        $put($1.options, "dontdraw", true); //#4488
        $put($1.options, "addongap", $1.addongap); //#4489
        $k[$j++] = 'args'; //#4490
        $k[$j++] = $1.barcode; //#4490
        $k[$j++] = $1.options; //#4490
        $0.ean13(); //#4490
        var _1W = $k[--$j]; //#4490
        $1[$k[--$j]] = _1W; //#4490
        if ($1.includetext) { //#4507
            if ($1.issntextxoffset == "unset") { //#4494
                $1.issntextxoffset = 10; //#4494
            } //#4494
            if ($1.issntextyoffset == "unset") { //#4497
                $1.issntextyoffset = $f(($1.height * 72) + 3); //#4496
            } //#4496
            var _1d = $get($1.args, "txt") !== undefined; //#4498
            if (_1d) { //#4505
                $1.txt = $get($1.args, "txt"); //#4499
                $1.newtxt = $a($1.txt.length + 1); //#4500
                $puti($1.newtxt, 0, $1.txt); //#4501
                $put($1.newtxt, $1.newtxt.length - 1, $a([$1.issntxt, $1.issntextxoffset, $1.issntextyoffset, $1.issntextfont, $1.issntextsize])); //#4502
                $put($1.args, "txt", $1.newtxt); //#4503
            } else { //#4505
                $put($1.args, "txt", $a([$a([$1.issntxt, $1.issntextxoffset, $1.issntextyoffset, $1.issntextfont, $1.issntextsize])])); //#4505
            } //#4505
        } //#4505
        $put($1.args, "opt", $1.options); //#4509
        $k[$j++] = $1.args; //#4512
        if (!$1.dontdraw) { //#4512
            $0.renlinear(); //#4512
        } //#4512
    };
    $0.code128 = function() {
        var $1 = {}; //#4542
        $1.options = $k[--$j]; //#4544
        $1.barcode = $k[--$j]; //#4545
        $1.dontdraw = false; //#4547
        $1.includetext = false; //#4548
        $1.textfont = "OCR-B"; //#4549
        $1.textsize = 10; //#4550
        $1.textxoffset = 0; //#4551
        $1.textyoffset = -8; //#4552
        $1.height = 1; //#4553
        $1.encoding = "auto"; //#4554
        $1.raw = false; //#4555
        $1.parse = false; //#4556
        $1.parsefnc = false; //#4557
        $forall($1.options, function() { //#4568
            var _3 = $k[--$j]; //#4568
            $1[$k[--$j]] = _3; //#4568
        }); //#4568
        $1.textfont = "" + $1.textfont; //#4570
        $1.textsize = +$1.textsize; //#4571
        $1.textxoffset = +$1.textxoffset; //#4572
        $1.textyoffset = +$1.textyoffset; //#4573
        $1.height = +$1.height; //#4574
        $1.sta = -1; //#4577
        $1.stb = -2; //#4577
        $1.stc = -3; //#4577
        $1.swa = -4; //#4578
        $1.swb = -5; //#4578
        $1.swc = -6; //#4578
        $1.fn1 = -7; //#4579
        $1.fn2 = -8; //#4579
        $1.fn3 = -9; //#4579
        $1.fn4 = -10; //#4580
        $1.sft = -11; //#4580
        $1.stp = -12; //#4580
        $1.lka = -13; //#4581
        $1.lkc = -14; //#4581
        var _I = new Map([
            ["parse", $1.parse],
            ["parsefnc", $1.parsefnc],
            ["FNC1", $1.fn1],
            ["FNC1", $1.fn1],
            ["FNC2", $1.fn2],
            ["FNC3", $1.fn3],
            ["LNKA", $1.lka],
            ["LNKC", $1.lkc]
        ]); //#4593
        $1.fncvals = _I; //#4594
        $k[$j++] = 'msg'; //#4595
        $k[$j++] = $1.barcode; //#4595
        $k[$j++] = $1.fncvals; //#4595
        $0.parseinput(); //#4595
        var _L = $k[--$j]; //#4595
        $1[$k[--$j]] = _L; //#4595
        $1.msglen = $1.msg.length; //#4596
        $1.charmaps = $a([$a([32, 32, "00"]), $a(["!", "!", "01"]), $a(["\"", "\"", "02"]), $a(["#", "#", "03"]), $a(["$", "$", "04"]), $a(["%", "%", "05"]), $a(["&", "&", "06"]), $a(["'", "'", "07"]), $a([40, 40, "08"]), $a([41, 41, "09"]), $a(["*", "*", "10"]), $a(["+", "+", "11"]), $a([",", ",", "12"]), $a(["-", "-", "13"]), $a([".", ".", "14"]), $a(["/", "/", "15"]), $a(["0", "0", "16"]), $a(["1", "1", "17"]), $a(["2", "2", "18"]), $a(["3", "3", "19"]), $a(["4", "4", "20"]), $a(["5", "5", "21"]), $a(["6", "6", "22"]), $a(["7", "7", "23"]), $a(["8", "8", "24"]), $a(["9", "9", "25"]), $a([":", ":", "26"]), $a([";", ";", "27"]), $a(["<", "<", "28"]), $a(["=", "=", "29"]), $a([">", ">", "30"]), $a(["?", "?", "31"]), $a(["@", "@", "32"]), $a(["A", "A", "33"]), $a(["B", "B", "34"]), $a(["C", "C", "35"]), $a(["D", "D", "36"]), $a(["E", "E", "37"]), $a(["F", "F", "38"]), $a(["G", "G", "39"]), $a(["H", "H", "40"]), $a(["I", "I", "41"]), $a(["J", "J", "42"]), $a(["K", "K", "43"]), $a(["L", "L", "44"]), $a(["M", "M", "45"]), $a(["N", "N", "46"]), $a(["O", "O", "47"]), $a(["P", "P", "48"]), $a(["Q", "Q", "49"]), $a(["R", "R", "50"]), $a(["S", "S", "51"]), $a(["T", "T", "52"]), $a(["U", "U", "53"]), $a(["V", "V", "54"]), $a(["W", "W", "55"]), $a(["X", "X", "56"]), $a(["Y", "Y", "57"]), $a(["Z", "Z", "58"]), $a(["[", "[", "59"]), $a([92, 92, "60"]), $a(["]", "]", "61"]), $a(["^", "^", "62"]), $a(["_", "_", "63"]), $a([0, "`", "64"]), $a([1, "a", "65"]), $a([2, "b", "66"]), $a([3, "c", "67"]), $a([4, "d", "68"]), $a([5, "e", "69"]), $a([6, "f", "70"]), $a([7, "g", "71"]), $a([8, "h", "72"]), $a([9, "i", "73"]), $a([10, "j", "74"]), $a([11, "k", "75"]), $a([12, "l", "76"]), $a([13, "m", "77"]), $a([14, "n", "78"]), $a([15, "o", "79"]), $a([16, "p", "80"]), $a([17, "q", "81"]), $a([18, "r", "82"]), $a([19, "s", "83"]), $a([20, "t", "84"]), $a([21, "u", "85"]), $a([22, "v", "86"]), $a([23, "w", "87"]), $a([24, "x", "88"]), $a([25, "y", "89"]), $a([26, "z", "90"]), $a([27, "{", "91"]), $a([28, "|", "92"]), $a([29, "}", "93"]), $a([30, "~", "94"]), $a([31, 127, "95"]), $a([$1.fn3, $1.fn3, "96"]), $a([$1.fn2, $1.fn2, "97"]), $a([$1.sft, $1.sft, "98"]), $a([$1.swc, $1.swc, "99"]), $a([$1.swb, $1.fn4, $1.swb]), $a([$1.fn4, $1.swa, $1.swa]), $a([$1.fn1, $1.fn1, $1.fn1]), $a([$1.sta, $1.sta, $1.sta]), $a([$1.stb, $1.stb, $1.stb]), $a([$1.stc, $1.stc, $1.stc]), $a([$1.stp, $1.stp, $1.stp])]); //#4637
        $1.charvals = $a([new Map, new Map, new Map]); //#4640
        for (var _2e = 0, _2d = $1.charmaps.length - 1; _2e <= _2d; _2e += 1) { //#4649
            $1.i = _2e; //#4642
            $1.encs = $get($1.charmaps, $1.i); //#4643
            for (var _2i = 0; _2i <= 2; _2i += 1) { //#4648
                $1.j = _2i; //#4645
                var _2l = $get($1.encs, $1.j); //#4646
                $k[$j++] = _2l; //#4646
                if ($eq($type(_2l), 'stringtype')) { //#4646
                    var _2o = $get($k[--$j], 0); //#4646
                    $k[$j++] = _2o; //#4646
                } //#4646
                $put($get($1.charvals, $1.j), $k[--$j], $1.i); //#4647
            } //#4647
        } //#4647
        $1.seta = $get($1.charvals, 0); //#4650
        $1.setb = $get($1.charvals, 1); //#4651
        $1.setc = $get($1.charvals, 2); //#4652
        $put($1.seta, $1.lka, $get($1.seta, $1.swb)); //#4655
        $put($1.seta, $1.lkc, $get($1.seta, $1.swc)); //#4655
        $put($1.setb, $1.lka, $get($1.setb, $1.swc)); //#4656
        $put($1.setb, $1.lkc, $get($1.setb, $1.swa)); //#4656
        $put($1.setc, $1.lka, $get($1.setc, $1.swa)); //#4657
        $put($1.setc, $1.lkc, $get($1.setc, $1.swb)); //#4657
        if ($1.raw) { //#4659
            $1.encoding = "raw"; //#4659
        } //#4659
        if ($eq($1.encoding, "raw")) { //#4673
            $1.cws = $a($1.barcode.length); //#4662
            $1.i = 0; //#4663
            $1.j = 0; //#4663
            for (;;) { //#4670
                if ($1.i == $1.barcode.length) { //#4665
                    break; //#4665
                } //#4665
                $1.cw = ~~$z($geti($1.barcode, $1.i + 1, 3)); //#4666
                $put($1.cws, $1.j, $1.cw); //#4667
                $1.i = $1.i + 4; //#4668
                $1.j = $1.j + 1; //#4669
            } //#4669
            $1.cws = $geti($1.cws, 0, $1.j); //#4671
            $1.text = ""; //#4672
        } //#4672
        if ($eq($1.encoding, "auto")) { //#4898
            $1.text = $s($1.msglen); //#4677
            for (var _3q = 0, _3p = $1.msglen - 1; _3q <= _3p; _3q += 1) { //#4681
                $1.i = _3q; //#4679
                var _3v = $get($1.msg, $1.i); //#4680
                $k[$j++] = $1.text; //#4680
                $k[$j++] = $1.i; //#4680
                $k[$j++] = _3v; //#4680
                if (_3v < 0) { //#4680
                    $j--; //#4680
                    $k[$j++] = 32; //#4680
                } //#4680
                var _3w = $k[--$j]; //#4680
                var _3x = $k[--$j]; //#4680
                $put($k[--$j], _3x, _3w); //#4680
            } //#4680
            $k[$j++] = Infinity; //#4684
            for (var _40 = 0, _41 = $1.msglen; _40 < _41; _40++) { //#4684
                $k[$j++] = 0; //#4684
            } //#4684
            $k[$j++] = 0; //#4684
            $1.numSA = $a(); //#4684
            $k[$j++] = Infinity; //#4685
            for (var _44 = 0, _45 = $1.msglen; _44 < _45; _44++) { //#4685
                $k[$j++] = 0; //#4685
            } //#4685
            $k[$j++] = 0; //#4685
            $1.numEA = $a(); //#4685
            for (var _48 = $1.msglen - 1; _48 >= 0; _48 -= 1) { //#4695
                $1.i = _48; //#4687
                if ($get($1.msg, $1.i) >= 0) { //#4694
                    if ($get($1.msg, $1.i) >= 128) { //#4692
                        $put($1.numEA, $1.i, $f($get($1.numEA, $1.i + 1) + 1)); //#4690
                    } else { //#4692
                        $put($1.numSA, $1.i, $f($get($1.numSA, $1.i + 1) + 1)); //#4692
                    } //#4692
                } //#4692
            } //#4692
            $1.ea = false; //#4698
            $1.msgtmp = $a([]); //#4698
            for (var _4S = 0, _4R = $1.msglen - 1; _4S <= _4R; _4S += 1) { //#4713
                $1.i = _4S; //#4700
                $1.c = $get($1.msg, $1.i); //#4701
                if ((!$xo($1.ea, $1.c < 128)) && ($1.c >= 0)) { //#4711
                    if ($1.ea) { //#4703
                        $k[$j++] = $1.numSA; //#4703
                    } else { //#4703
                        $k[$j++] = $1.numEA; //#4703
                    } //#4703
                    var _4e = $get($k[--$j], $1.i); //#4703
                    var _4h = ($f(_4e + $1.i) == $1.msglen) ? 3 : 5; //#4704
                    if (_4e < _4h) { //#4709
                        $k[$j++] = Infinity; //#4706
                        $aload($1.msgtmp); //#4706
                        $k[$j++] = $1.fn4; //#4706
                        $1.msgtmp = $a(); //#4706
                    } else { //#4709
                        $k[$j++] = Infinity; //#4708
                        $aload($1.msgtmp); //#4708
                        $k[$j++] = $1.fn4; //#4708
                        $k[$j++] = $1.fn4; //#4708
                        $1.msgtmp = $a(); //#4708
                        $1.ea = !$1.ea; //#4709
                    } //#4709
                } //#4709
                $k[$j++] = Infinity; //#4712
                $aload($1.msgtmp); //#4712
                if ($1.c >= 0) { //#4712
                    $k[$j++] = $1.c & 127; //#4712
                } else { //#4712
                    $k[$j++] = $1.c; //#4712
                } //#4712
                $1.msgtmp = $a(); //#4712
            } //#4712
            $1.msg = $1.msgtmp; //#4714
            $1.msglen = $1.msg.length; //#4715
            $1.numsscr = function() {
                $1.n = 0; //#4719
                $1.s = 0; //#4719
                $1.p = $k[--$j]; //#4720
                for (;;) { //#4733
                    if ($1.p >= $1.msglen) { //#4721
                        break; //#4721
                    } //#4721
                    var _52 = $get($1.msg, $1.p); //#4722
                    var _54 = $get($1.setc, _52) !== undefined; //#4723
                    $k[$j++] = _52; //#4723
                    if (!_54) { //#4723
                        $j--; //#4723
                        break; //#4723
                    } //#4723
                    var _55 = $k[--$j]; //#4724
                    $k[$j++] = _55; //#4729
                    if (_55 <= -1) { //#4728
                        var _57 = $k[--$j]; //#4726
                        if ((_57 == $1.fn1) && (($1.s % 2) == 0)) { //#4726
                            $1.s = $1.s + 1; //#4726
                        } else { //#4726
                            break; //#4726
                        } //#4726
                    } else { //#4728
                        $j--; //#4728
                    } //#4728
                    $1.n = $1.n + 1; //#4730
                    $1.s = $1.s + 1; //#4731
                    $1.p = $f($1.p + 1); //#4732
                } //#4732
                $k[$j++] = $1.n; //#4734
                $k[$j++] = $1.s; //#4734
            }; //#4735
            $1.enca = function() {
                $put($1.cws, $1.j, $get($1.seta, $k[--$j])); //#4739
                $1.j = $1.j + 1; //#4740
            }; //#4741
            $1.encb = function() {
                $put($1.cws, $1.j, $get($1.setb, $k[--$j])); //#4743
                $1.j = $1.j + 1; //#4744
            }; //#4745
            $1.encc = function() {
                var _5R = $k[--$j]; //#4747
                $k[$j++] = _5R; //#4751
                if ($ne($type(_5R), 'arraytype')) { //#4750
                    var _5V = $get($1.setc, $k[--$j]); //#4748
                    $k[$j++] = _5V; //#4748
                } else { //#4750
                    $aload($k[--$j]); //#4750
                    var _5X = $k[--$j]; //#4750
                    var _5Y = $k[--$j]; //#4750
                    $k[$j++] = $f($f(_5X - 48) + ($f(_5Y - 48) * 10)); //#4750
                } //#4750
                $put($1.cws, $1.j, $k[--$j]); //#4752
                $1.j = $1.j + 1; //#4753
            }; //#4754
            $1.anotb = function() {
                var _5d = $k[--$j]; //#4757
                var _5f = $get($1.seta, _5d) !== undefined; //#4757
                var _5h = $get($1.setb, _5d) !== undefined; //#4757
                $k[$j++] = _5f && (!_5h); //#4757
            }; //#4757
            $1.bnota = function() {
                var _5i = $k[--$j]; //#4758
                var _5k = $get($1.setb, _5i) !== undefined; //#4758
                var _5m = $get($1.seta, _5i) !== undefined; //#4758
                $k[$j++] = _5k && (!_5m); //#4758
            }; //#4758
            $k[$j++] = Infinity; //#4761
            for (var _5o = 0, _5p = $1.msg.length; _5o < _5p; _5o++) { //#4761
                $k[$j++] = 0; //#4761
            } //#4761
            $k[$j++] = 9999; //#4761
            $1.nextanotb = $a(); //#4761
            $k[$j++] = Infinity; //#4762
            for (var _5s = 0, _5t = $1.msg.length; _5s < _5t; _5s++) { //#4762
                $k[$j++] = 0; //#4762
            } //#4762
            $k[$j++] = 9999; //#4762
            $1.nextbnota = $a(); //#4762
            for (var _5w = $1.msg.length - 1; _5w >= 0; _5w -= 1) { //#4775
                $1.i = _5w; //#4764
                $k[$j++] = $get($1.msg, $1.i); //#4765
                $1.anotb(); //#4765
                if ($k[--$j]) { //#4768
                    $put($1.nextanotb, $1.i, 0); //#4766
                } else { //#4768
                    $put($1.nextanotb, $1.i, $f($get($1.nextanotb, $1.i + 1) + 1)); //#4768
                } //#4768
                $k[$j++] = $get($1.msg, $1.i); //#4770
                $1.bnota(); //#4770
                if ($k[--$j]) { //#4773
                    $put($1.nextbnota, $1.i, 0); //#4771
                } else { //#4773
                    $put($1.nextbnota, $1.i, $f($get($1.nextbnota, $1.i + 1) + 1)); //#4773
                } //#4773
            } //#4773
            $1.abeforeb = function() {
                var _6J = $k[--$j]; //#4778
                $k[$j++] = $lt($get($1.nextanotb, _6J), $get($1.nextbnota, _6J)); //#4778
            }; //#4778
            $1.bbeforea = function() {
                var _6O = $k[--$j]; //#4779
                $k[$j++] = $lt($get($1.nextbnota, _6O), $get($1.nextanotb, _6O)); //#4779
            }; //#4779
            $1.cws = $a(($1.barcode.length * 2) + 3); //#4781
            $1.j = 0; //#4784
            if ($1.msglen > 0) { //#4785
                $k[$j++] = 0; //#4785
                $1.numsscr(); //#4785
            } else { //#4785
                $k[$j++] = -1; //#4785
                $k[$j++] = -1; //#4785
            } //#4785
            $1.nums = $k[--$j]; //#4785
            $1.nchars = $k[--$j]; //#4785
            for (;;) { //#4810
                if ($1.msglen == 0) { //#4791
                    $k[$j++] = $1.stb; //#4788
                    $1.enca(); //#4788
                    $1.cset = "setb"; //#4789
                    break; //#4790
                } //#4790
                if (($1.msglen == 2) && ($1.nums == 2)) { //#4796
                    $k[$j++] = $1.stc; //#4793
                    $1.enca(); //#4793
                    $1.cset = "setc"; //#4794
                    break; //#4795
                } //#4795
                if ($1.nums >= 4) { //#4801
                    $k[$j++] = $1.stc; //#4798
                    $1.enca(); //#4798
                    $1.cset = "setc"; //#4799
                    break; //#4800
                } //#4800
                $k[$j++] = 0; //#4802
                $1.abeforeb(); //#4802
                if ($k[--$j]) { //#4806
                    $k[$j++] = $1.sta; //#4803
                    $1.enca(); //#4803
                    $1.cset = "seta"; //#4804
                    break; //#4805
                } //#4805
                $k[$j++] = $1.stb; //#4807
                $1.enca(); //#4807
                $1.cset = "setb"; //#4808
                break; //#4809
            } //#4809
            $1.i = 0; //#4813
            for (;;) { //#4896
                if ($1.i == $1.msglen) { //#4814
                    break; //#4814
                } //#4814
                $k[$j++] = $1.i; //#4816
                $1.numsscr(); //#4816
                $1.nums = $k[--$j]; //#4816
                $1.nchars = $k[--$j]; //#4816
                for (;;) { //#4894
                    if ((($eq($1.cset, "seta") || $eq($1.cset, "setb")) && ($1.nums >= 4)) && ($get($1.msg, $1.i) != $1.fn1)) { //#4833
                        if (($1.nums % 2) == 0) { //#4831
                            $k[$j++] = $1.swc; //#4823
                            if ($eq($1.cset, "seta")) { //#4823
                                $1.enca(); //#4823
                            } else { //#4823
                                $1.encb(); //#4823
                            } //#4823
                            $1.cset = "setc"; //#4824
                            break; //#4825
                        } else { //#4831
                            $k[$j++] = $get($1.msg, $1.i); //#4827
                            if ($eq($1.cset, "seta")) { //#4827
                                $1.enca(); //#4827
                            } else { //#4827
                                $1.encb(); //#4827
                            } //#4827
                            $1.i = $1.i + 1; //#4828
                            $k[$j++] = $1.swc; //#4829
                            if ($eq($1.cset, "seta")) { //#4829
                                $1.enca(); //#4829
                            } else { //#4829
                                $1.encb(); //#4829
                            } //#4829
                            $1.cset = "setc"; //#4830
                            break; //#4831
                        } //#4831
                    } //#4831
                    $k[$j++] = $eq($1.cset, "setb"); //#4834
                    $k[$j++] = $get($1.msg, $1.i); //#4834
                    $1.anotb(); //#4834
                    var _78 = $k[--$j]; //#4834
                    var _79 = $k[--$j]; //#4834
                    if ((_79 && _78)) { //#4846
                        if ($1.i < ($1.msglen - 1)) { //#4842
                            $k[$j++] = $1.i + 1; //#4836
                            $1.bbeforea(); //#4836
                            if ($k[--$j]) { //#4841
                                $k[$j++] = $1.sft; //#4837
                                $1.encb(); //#4837
                                $k[$j++] = $get($1.msg, $1.i); //#4838
                                $1.enca(); //#4838
                                $1.i = $1.i + 1; //#4839
                                break; //#4840
                            } //#4840
                        } //#4840
                        $k[$j++] = $1.swa; //#4843
                        $1.encb(); //#4843
                        $1.cset = "seta"; //#4844
                        break; //#4845
                    } //#4845
                    $k[$j++] = $eq($1.cset, "seta"); //#4847
                    $k[$j++] = $get($1.msg, $1.i); //#4847
                    $1.bnota(); //#4847
                    var _7O = $k[--$j]; //#4847
                    var _7P = $k[--$j]; //#4847
                    if ((_7P && _7O)) { //#4859
                        if ($1.i < ($1.msglen - 1)) { //#4855
                            $k[$j++] = $1.i + 1; //#4849
                            $1.abeforeb(); //#4849
                            if ($k[--$j]) { //#4854
                                $k[$j++] = $1.sft; //#4850
                                $1.enca(); //#4850
                                $k[$j++] = $get($1.msg, $1.i); //#4851
                                $1.encb(); //#4851
                                $1.i = $1.i + 1; //#4852
                                break; //#4853
                            } //#4853
                        } //#4853
                        $k[$j++] = $1.swb; //#4856
                        $1.enca(); //#4856
                        $1.cset = "setb"; //#4857
                        break; //#4858
                    } //#4858
                    if (($eq($1.cset, "setc") && ($1.nums < 2)) && ($get($1.msg, $1.i) > -1)) { //#4869
                        $k[$j++] = $1.i; //#4861
                        $1.abeforeb(); //#4861
                        if ($k[--$j]) { //#4865
                            $k[$j++] = $1.swa; //#4862
                            $1.encc(); //#4862
                            $1.cset = "seta"; //#4863
                            break; //#4864
                        } //#4864
                        $k[$j++] = $1.swb; //#4866
                        $1.encc(); //#4866
                        $1.cset = "setb"; //#4867
                        break; //#4868
                    } //#4868
                    if ($eq($1.cset, "seta")) { //#4876
                        $k[$j++] = $get($1.msg, $1.i); //#4873
                        $1.enca(); //#4873
                        $1.i = $1.i + 1; //#4874
                        break; //#4875
                    } //#4875
                    if ($eq($1.cset, "setb")) { //#4881
                        $k[$j++] = $get($1.msg, $1.i); //#4878
                        $1.encb(); //#4878
                        $1.i = $1.i + 1; //#4879
                        break; //#4880
                    } //#4880
                    if ($eq($1.cset, "setc")) { //#4891
                        if ($get($1.msg, $1.i) <= -1) { //#4888
                            $k[$j++] = $get($1.msg, $1.i); //#4884
                            $1.encc(); //#4884
                            $1.i = $1.i + 1; //#4885
                        } else { //#4888
                            $k[$j++] = $geti($1.msg, $1.i, 2); //#4887
                            $1.encc(); //#4887
                            $1.i = $1.i + 2; //#4888
                        } //#4888
                        break; //#4890
                    } //#4890
                    break; //#4893
                } //#4893
            } //#4893
            $1.cws = $geti($1.cws, 0, $1.j); //#4897
        } //#4897
        var _89 = $a($1.j + 2); //#4901
        $puti(_89, 0, $1.cws); //#4901
        $1.cws = _89; //#4901
        $1.csum = $get($1.cws, 0); //#4902
        for (var _8F = 1, _8E = $1.j - 1; _8F <= _8E; _8F += 1) { //#4906
            $1.i = _8F; //#4904
            $1.csum = $f($1.csum + ($get($1.cws, $1.i) * $1.i)); //#4905
        } //#4905
        $1.csum = $1.csum % 103; //#4907
        $put($1.cws, $1.j, $1.csum); //#4908
        $put($1.cws, $1.j + 1, $get($1.seta, $1.stp)); //#4909
        $1.encs = $a(["212222", "222122", "222221", "121223", "121322", "131222", "122213", "122312", "132212", "221213", "221312", "231212", "112232", "122132", "122231", "113222", "123122", "123221", "223211", "221132", "221231", "213212", "223112", "312131", "311222", "321122", "321221", "312212", "322112", "322211", "212123", "212321", "232121", "111323", "131123", "131321", "112313", "132113", "132311", "211313", "231113", "231311", "112133", "112331", "132131", "113123", "113321", "133121", "313121", "211331", "231131", "213113", "213311", "213131", "311123", "311321", "331121", "312113", "312311", "332111", "314111", "221411", "431111", "111224", "111422", "121124", "121421", "141122", "141221", "112214", "112412", "122114", "122411", "142112", "142211", "241211", "221114", "413111", "241112", "134111", "111242", "121142", "121241", "114212", "124112", "124211", "411212", "421112", "421211", "212141", "214121", "412121", "111143", "111341", "131141", "114113", "114311", "411113", "411311", "113141", "114131", "311141", "411131", "211412", "211214", "211232", "2331112"]); //#4929
        $1.sbs = $s(($1.cws.length * 6) + 1); //#4932
        for (var _8Z = 0, _8Y = $1.cws.length - 1; _8Z <= _8Y; _8Z += 1) { //#4936
            $1.i = _8Z; //#4934
            $puti($1.sbs, $1.i * 6, $get($1.encs, $get($1.cws, $1.i))); //#4935
        } //#4935
        $k[$j++] = Infinity; //#4946
        $k[$j++] = Infinity; //#4941
        var _8h = $1.sbs; //#4941
        for (var _8i = 0, _8j = _8h.length; _8i < _8j; _8i++) { //#4941
            $k[$j++] = $get(_8h, _8i) - 48; //#4941
        } //#4941
        var _8l = $a(); //#4941
        $k[$j++] = Infinity; //#4942
        for (var _8n = 0, _8o = ~~(($1.sbs.length + 1) / 2); _8n < _8o; _8n++) { //#4942
            $k[$j++] = $1.height; //#4942
        } //#4942
        var _8q = $a(); //#4942
        $k[$j++] = Infinity; //#4943
        for (var _8s = 0, _8t = ~~(($1.sbs.length + 1) / 2); _8s < _8t; _8s++) { //#4943
            $k[$j++] = 0; //#4943
        } //#4943
        var _8u = $a(); //#4943
        $k[$j++] = 'ren'; //#4946
        $k[$j++] = $0.renlinear; //#4946
        $k[$j++] = 'sbs'; //#4946
        $k[$j++] = _8l; //#4946
        $k[$j++] = 'bhs'; //#4946
        $k[$j++] = _8q; //#4946
        $k[$j++] = 'bbs'; //#4946
        $k[$j++] = _8u; //#4946
        $k[$j++] = 'txt'; //#4946
        $k[$j++] = $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])]); //#4946
        $k[$j++] = 'textxalign'; //#4946
        $k[$j++] = "center"; //#4946
        $k[$j++] = 'opt'; //#4946
        $k[$j++] = $1.options; //#4946
        var _93 = $d(); //#4946
        $k[$j++] = _93; //#4949
        if (!$1.dontdraw) { //#4949
            $0.renlinear(); //#4949
        } //#4949
    };
    $0['gs1-128'] = function() {
        var $1 = {}; //#4981
        $1.options = $k[--$j]; //#4983
        $1.barcode = $k[--$j]; //#4984
        $1.dontdraw = false; //#4986
        $1.includetext = false; //#4987
        $1.textfont = "OCR-B"; //#4988
        $1.textsize = 10; //#4989
        $1.textxoffset = 0; //#4990
        $1.textyoffset = -8; //#4991
        $1.height = 0.5; //#4992
        $1.linkagea = false; //#4993
        $1.linkagec = false; //#4994
        $1.parse = false; //#4995
        $1.dontlint = false; //#4996
        $forall($1.options, function() { //#5007
            var _3 = $k[--$j]; //#5007
            $1[$k[--$j]] = _3; //#5007
        }); //#5007
        $1.textfont = "" + $1.textfont; //#5009
        $1.textsize = +$1.textsize; //#5010
        $1.textxoffset = +$1.textxoffset; //#5011
        $1.textyoffset = +$1.textyoffset; //#5012
        $1.height = +$1.height; //#5013
        $1.text = $1.barcode; //#5015
        $1.expand = function() {
            var _C = new Map([
                ["parse", $1.parse],
                ["parseonly", true],
                ["parsefnc", false]
            ]); //#5022
            $1.fncvals = _C; //#5023
            $k[$j++] = $1.fncvals; //#5024
            $0.parseinput(); //#5024
        }; //#5025
        $1.ais = $a([]); //#5028
        $1.vals = $a([]); //#5029
        var _G = $1.barcode; //#5030
        $k[$j++] = $geti(_G, 1, _G.length - 1); //#5043
        for (;;) { //#5043
            var _I = $k[--$j]; //#5032
            $k[$j++] = _I; //#5032
            if ($eq(_I, "")) { //#5032
                break; //#5032
            } //#5032
            $search($k[--$j], ")"); //#5033
            $j--; //#5033
            var _K = $k[--$j]; //#5034
            var _L = $k[--$j]; //#5034
            $k[$j++] = _K; //#5034
            $k[$j++] = _L; //#5034
            $j--; //#5034
            var _M = $k[--$j]; //#5035
            var _N = $k[--$j]; //#5035
            $k[$j++] = _M; //#5035
            $search(_N, "("); //#5035
            if ($k[--$j]) { //#5038
                var _P = $k[--$j]; //#5036
                var _Q = $k[--$j]; //#5036
                $k[$j++] = _P; //#5036
                $k[$j++] = _Q; //#5036
                $j--; //#5036
                var _R = $k[--$j]; //#5036
                var _S = $k[--$j]; //#5036
                var _T = $k[--$j]; //#5036
                $k[$j++] = _S; //#5036
                $k[$j++] = _T; //#5036
                $k[$j++] = _R; //#5036
            } else { //#5038
                var _U = $k[--$j]; //#5038
                var _V = $k[--$j]; //#5038
                $k[$j++] = ""; //#5038
                $k[$j++] = _V; //#5038
                $k[$j++] = _U; //#5038
            } //#5038
            $k[$j++] = Infinity; //#5040
            $aload($1.ais); //#5040
            var _Y = $k[$j - 1 - ($counttomark() + 2)]; //#5040
            $k[$j++] = _Y; //#5040
            $1.ais = $a(); //#5040
            $k[$j++] = Infinity; //#5041
            $aload($1.vals); //#5041
            var _c = $k[$j - 1 - ($counttomark() + 1)]; //#5041
            $k[$j++] = _c; //#5041
            $1.expand(); //#5041
            $1.vals = $a(); //#5041
            $j -= 2; //#5042
        } //#5042
        $j--; //#5044
        if (!$1.dontlint) { //#5046
            $k[$j++] = $1.ais; //#5046
            $k[$j++] = $1.vals; //#5046
            $0.gs1lint(); //#5046
            $j--; //#5046
        } //#5046
        $1.aifixed = new Map; //#5051
        $k[$j++] = Infinity; //#5053
        for (var _h = 0; _h <= 4; _h += 1) { //#5053
            $k[$j++] = _h; //#5053
        } //#5053
        var _i = $a(); //#5053
        for (var _j = 0, _k = _i.length; _j < _k; _j++) { //#5056
            var _n = $strcpy($s(2), "00"); //#5055
            $put(_n, 1, $f($get(_i, _j) + 48)); //#5055
            $put($1.aifixed, _n, _n); //#5055
        } //#5055
        $k[$j++] = Infinity; //#5061
        for (var _p = 11; _p <= 20; _p += 1) { //#5058
            $k[$j++] = _p; //#5058
        } //#5058
        $k[$j++] = 23; //#5060
        for (var _q = 31; _q <= 36; _q += 1) { //#5060
            $k[$j++] = _q; //#5060
        } //#5060
        $k[$j++] = 41; //#5061
        var _r = $a(); //#5061
        for (var _s = 0, _t = _r.length; _s < _t; _s++) { //#5064
            var _w = $cvrs($s(2), $get(_r, _s), 10); //#5063
            $put($1.aifixed, _w, _w); //#5063
        } //#5063
        $1.fnc1 = -1; //#5067
        $1.c128 = $a([$1.fnc1]); //#5068
        for (var _12 = 0, _11 = $1.ais.length - 1; _12 <= _11; _12 += 1) { //#5084
            $1.i = _12; //#5070
            $1.ai = $get($1.ais, $1.i); //#5071
            $1.val = $get($1.vals, $1.i); //#5072
            var _1C = $a(($1.c128.length + $1.ai.length) + $1.val.length); //#5073
            $puti(_1C, 0, $1.c128); //#5074
            $k[$j++] = _1C; //#5075
            $k[$j++] = _1C; //#5075
            $k[$j++] = $1.c128.length; //#5075
            $k[$j++] = $1.ai; //#5075
            $k[$j++] = Infinity; //#5075
            var _1G = $k[--$j]; //#5075
            var _1H = $k[--$j]; //#5075
            $k[$j++] = _1G; //#5075
            $forall(_1H); //#5075
            var _1I = $a(); //#5075
            var _1J = $k[--$j]; //#5075
            $puti($k[--$j], _1J, _1I); //#5075
            var _1L = $k[--$j]; //#5076
            $k[$j++] = _1L; //#5076
            $k[$j++] = _1L; //#5076
            $k[$j++] = $1.c128.length + $1.ai.length; //#5076
            $k[$j++] = $1.val; //#5076
            $k[$j++] = Infinity; //#5076
            var _1P = $k[--$j]; //#5076
            var _1Q = $k[--$j]; //#5076
            $k[$j++] = _1P; //#5076
            $forall(_1Q); //#5076
            var _1R = $a(); //#5076
            var _1S = $k[--$j]; //#5076
            $puti($k[--$j], _1S, _1R); //#5076
            $1.c128 = $k[--$j]; //#5077
            var _1a = $get($1.aifixed, $geti($1.ai, 0, 2)) !== undefined; //#5078
            if (($1.i != ($1.ais.length - 1)) && (!_1a)) { //#5083
                var _1c = $a($1.c128.length + 1); //#5079
                $puti(_1c, 0, $1.c128); //#5080
                $put(_1c, $1.c128.length, $1.fnc1); //#5081
                $1.c128 = _1c; //#5082
            } //#5082
        } //#5082
        $1.barcode = $s(($1.c128.length + 1) * 5); //#5087
        $1.i = 0; //#5088
        $1.j = 0; //#5088
        for (;;) { //#5098
            if ($1.i == $1.c128.length) { //#5089
                break; //#5089
            } //#5089
            var _1m = $get($1.c128, $1.i); //#5090
            $k[$j++] = _1m; //#5095
            if (_1m == $1.fnc1) { //#5094
                $j--; //#5091
                $puti($1.barcode, $1.j, "^FNC1"); //#5091
                $1.j = $1.j + 4; //#5092
            } else { //#5094
                $put($1.barcode, $1.j, $k[--$j]); //#5094
            } //#5094
            $1.i = $1.i + 1; //#5096
            $1.j = $1.j + 1; //#5097
        } //#5097
        if ($1.linkagea || $1.linkagec) { //#5102
            $k[$j++] = $1.barcode; //#5100
            $k[$j++] = $1.j; //#5100
            if ($1.linkagea) { //#5100
                $k[$j++] = "^LNKA"; //#5100
            } else { //#5100
                $k[$j++] = "^LNKC"; //#5100
            } //#5100
            var _21 = $k[--$j]; //#5100
            var _22 = $k[--$j]; //#5100
            $puti($k[--$j], _22, _21); //#5100
            $1.j = $1.j + 5; //#5101
        } //#5101
        $1.barcode = $geti($1.barcode, 0, $1.j); //#5103
        delete $1.options["parse"]; //#5106
        $put($1.options, "height", $1.height); //#5107
        $put($1.options, "dontdraw", true); //#5108
        $put($1.options, "parsefnc", true); //#5109
        $k[$j++] = 'args'; //#5110
        $k[$j++] = $1.barcode; //#5110
        $k[$j++] = $1.options; //#5110
        $0.code128(); //#5110
        var _2F = $k[--$j]; //#5110
        $1[$k[--$j]] = _2F; //#5110
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#5112
        $put($1.args, "textxalign", "center"); //#5113
        $put($1.args, "opt", $1.options); //#5114
        $k[$j++] = $1.args; //#5117
        if (!$1.dontdraw) { //#5117
            $0.renlinear(); //#5117
        } //#5117
    };
    $0.ean14 = function() {
        var $1 = {}; //#5147
        $1.options = $k[--$j]; //#5149
        $1.barcode = $k[--$j]; //#5150
        $1.dontdraw = false; //#5152
        $1.includetext = false; //#5153
        $1.textfont = "OCR-B"; //#5154
        $1.textsize = 10; //#5155
        $1.textxoffset = 0; //#5156
        $1.textyoffset = -8; //#5157
        $1.height = 1; //#5158
        $forall($1.options, function() { //#5169
            var _3 = $k[--$j]; //#5169
            $1[$k[--$j]] = _3; //#5169
        }); //#5169
        $1.textfont = "" + $1.textfont; //#5171
        $1.textsize = +$1.textsize; //#5172
        $1.textxoffset = +$1.textxoffset; //#5173
        $1.textyoffset = +$1.textyoffset; //#5174
        $1.height = +$1.height; //#5175
        $1.text = $1.barcode; //#5178
        var _B = $1.barcode; //#5179
        $k[$j++] = $s(_B.length); //#5180
        $k[$j++] = 0; //#5180
        $forall(_B, function() { //#5180
            var _D = $k[--$j]; //#5180
            $k[$j++] = _D; //#5180
            if (_D != 32) { //#5180
                var _E = $k[--$j]; //#5180
                var _F = $k[--$j]; //#5180
                var _G = $k[--$j]; //#5180
                $put(_G, _F, _E); //#5180
                $k[$j++] = _G; //#5180
                $k[$j++] = $f(_F + 1); //#5180
            } else { //#5180
                $j--; //#5180
            } //#5180
        }); //#5180
        var _H = $k[--$j]; //#5181
        $1.barcode = $geti($k[--$j], 0, _H); //#5181
        $1.hasspace = $1.text.length != $1.barcode.length; //#5182
        if ($ne($geti($1.barcode, 0, 4), "(01)")) { //#5187
            $k[$j++] = 'bwipp.ean14badAI'; //#5186
            $k[$j++] = "GS1-14 must begin with (01) application identifier"; //#5186
            $0.raiseerror(); //#5186
        } //#5186
        if (($1.barcode.length != 17) && ($1.barcode.length != 18)) { //#5190
            $k[$j++] = 'bwipp.ean14badLength'; //#5189
            $k[$j++] = "GS1-14 must be 13 or 14 digits"; //#5189
            $0.raiseerror(); //#5189
        } //#5189
        var _S = $geti($1.barcode, 4, $1.barcode.length - 4); //#5191
        for (var _T = 0, _U = _S.length; _T < _U; _T++) { //#5195
            var _V = $get(_S, _T); //#5195
            if ((_V < 48) || (_V > 57)) { //#5194
                $k[$j++] = 'bwipp.ean14badCharacter'; //#5193
                $k[$j++] = "GS1-14 must contain only digits"; //#5193
                $0.raiseerror(); //#5193
            } //#5193
        } //#5193
        $1.checksum = 0; //#5198
        for (var _W = 0; _W <= 12; _W += 1) { //#5202
            $1.i = _W; //#5200
            $k[$j++] = 'checksum'; //#5201
            $k[$j++] = $1.checksum; //#5201
            $k[$j++] = $f($get($1.barcode, $1.i + 4) - 48); //#5201
            if (($1.i % 2) == 0) { //#5201
                var _c = $k[--$j]; //#5201
                $k[$j++] = _c * 3; //#5201
            } //#5201
            var _d = $k[--$j]; //#5201
            var _e = $k[--$j]; //#5201
            $1[$k[--$j]] = $f(_e + _d); //#5201
        } //#5201
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#5203
        if ($1.barcode.length == 18) { //#5217
            if ($get($1.barcode, 17) != ($1.checksum + 48)) { //#5207
                $k[$j++] = 'bwipp.ean14badCheckDigit'; //#5206
                $k[$j++] = "Incorrect GS1-14 check digit provided"; //#5206
                $0.raiseerror(); //#5206
            } //#5206
        } else { //#5217
            var _l = $s(18); //#5209
            $puti(_l, 0, $1.barcode); //#5210
            $put(_l, 17, $1.checksum + 48); //#5211
            $1.barcode = _l; //#5212
            var _q = $1.hasspace ? 2 : 1; //#5213
            var _r = $s($1.text.length + _q); //#5213
            $puti(_r, _r.length - 2, " "); //#5214
            $put(_r, _r.length - 1, $1.checksum + 48); //#5215
            $puti(_r, 0, $1.text); //#5216
            $1.text = _r; //#5217
        } //#5217
        $1.gtin = $geti($1.barcode, 4, 14); //#5219
        $1.barcode = $s(21); //#5222
        $puti($1.barcode, 0, "^FNC101"); //#5223
        $puti($1.barcode, 7, $1.gtin); //#5224
        $put($1.options, "dontdraw", true); //#5227
        $put($1.options, "parsefnc", true); //#5228
        $k[$j++] = 'args'; //#5229
        $k[$j++] = $1.barcode; //#5229
        $k[$j++] = $1.options; //#5229
        $0.code128(); //#5229
        var _14 = $k[--$j]; //#5229
        $1[$k[--$j]] = _14; //#5229
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#5231
        $put($1.args, "textxalign", "center"); //#5232
        $put($1.args, "opt", $1.options); //#5233
        $k[$j++] = $1.args; //#5236
        if (!$1.dontdraw) { //#5236
            $0.renlinear(); //#5236
        } //#5236
    };
    $0.sscc18 = function() {
        var $1 = {}; //#5266
        $1.options = $k[--$j]; //#5268
        $1.barcode = $k[--$j]; //#5269
        $1.dontdraw = false; //#5271
        $1.includetext = false; //#5272
        $1.textfont = "OCR-B"; //#5273
        $1.textsize = 10; //#5274
        $1.textxoffset = 0; //#5275
        $1.textyoffset = -8; //#5276
        $1.height = 1; //#5277
        $forall($1.options, function() { //#5288
            var _3 = $k[--$j]; //#5288
            $1[$k[--$j]] = _3; //#5288
        }); //#5288
        $1.textfont = "" + $1.textfont; //#5290
        $1.textsize = +$1.textsize; //#5291
        $1.textxoffset = +$1.textxoffset; //#5292
        $1.textyoffset = +$1.textyoffset; //#5293
        $1.height = +$1.height; //#5294
        $1.text = $1.barcode; //#5297
        var _B = $1.barcode; //#5298
        $k[$j++] = $s(_B.length); //#5299
        $k[$j++] = 0; //#5299
        $forall(_B, function() { //#5299
            var _D = $k[--$j]; //#5299
            $k[$j++] = _D; //#5299
            if (_D != 32) { //#5299
                var _E = $k[--$j]; //#5299
                var _F = $k[--$j]; //#5299
                var _G = $k[--$j]; //#5299
                $put(_G, _F, _E); //#5299
                $k[$j++] = _G; //#5299
                $k[$j++] = $f(_F + 1); //#5299
            } else { //#5299
                $j--; //#5299
            } //#5299
        }); //#5299
        var _H = $k[--$j]; //#5300
        $1.barcode = $geti($k[--$j], 0, _H); //#5300
        $1.hasspace = $1.text.length != $1.barcode.length; //#5301
        if ($ne($geti($1.barcode, 0, 4), "(00)")) { //#5306
            $k[$j++] = 'bwipp.sscc18badAI'; //#5305
            $k[$j++] = "SSCC-18 must begin with (00) application identifier"; //#5305
            $0.raiseerror(); //#5305
        } //#5305
        if (($1.barcode.length != 21) && ($1.barcode.length != 22)) { //#5309
            $k[$j++] = 'bwipp.sscc18badLength'; //#5308
            $k[$j++] = "SSCC-18 must be 17 or 18 digits"; //#5308
            $0.raiseerror(); //#5308
        } //#5308
        var _S = $geti($1.barcode, 4, $1.barcode.length - 4); //#5310
        for (var _T = 0, _U = _S.length; _T < _U; _T++) { //#5314
            var _V = $get(_S, _T); //#5314
            if ((_V < 48) || (_V > 57)) { //#5313
                $k[$j++] = 'bwipp.sscc18badCharacter'; //#5312
                $k[$j++] = "SSCC-18 must contain only digits"; //#5312
                $0.raiseerror(); //#5312
            } //#5312
        } //#5312
        $1.checksum = 0; //#5317
        for (var _W = 0; _W <= 16; _W += 1) { //#5321
            $1.i = _W; //#5319
            $k[$j++] = 'checksum'; //#5320
            $k[$j++] = $1.checksum; //#5320
            $k[$j++] = $f($get($1.barcode, $1.i + 4) - 48); //#5320
            if (($1.i % 2) == 0) { //#5320
                var _c = $k[--$j]; //#5320
                $k[$j++] = _c * 3; //#5320
            } //#5320
            var _d = $k[--$j]; //#5320
            var _e = $k[--$j]; //#5320
            $1[$k[--$j]] = $f(_e + _d); //#5320
        } //#5320
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#5322
        if ($1.barcode.length == 22) { //#5336
            if ($get($1.barcode, 21) != ($1.checksum + 48)) { //#5326
                $k[$j++] = 'bwipp.sscc18badCheckDigit'; //#5325
                $k[$j++] = "Incorrect SSCC-18 check digit provided"; //#5325
                $0.raiseerror(); //#5325
            } //#5325
        } else { //#5336
            var _l = $s(22); //#5328
            $puti(_l, 0, $1.barcode); //#5329
            $put(_l, 21, $1.checksum + 48); //#5330
            $1.barcode = _l; //#5331
            var _q = $1.hasspace ? 2 : 1; //#5332
            var _r = $s($1.text.length + _q); //#5332
            $puti(_r, _r.length - 2, " "); //#5333
            $put(_r, _r.length - 1, $1.checksum + 48); //#5334
            $puti(_r, 0, $1.text); //#5335
            $1.text = _r; //#5336
        } //#5336
        $1.sscc = $geti($1.barcode, 4, 18); //#5338
        $1.barcode = $s(25); //#5341
        $puti($1.barcode, 0, "^FNC100"); //#5342
        $puti($1.barcode, 7, $1.sscc); //#5343
        $put($1.options, "dontdraw", true); //#5346
        $put($1.options, "parsefnc", true); //#5347
        $k[$j++] = 'args'; //#5348
        $k[$j++] = $1.barcode; //#5348
        $k[$j++] = $1.options; //#5348
        $0.code128(); //#5348
        var _14 = $k[--$j]; //#5348
        $1[$k[--$j]] = _14; //#5348
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#5350
        $put($1.args, "textxalign", "center"); //#5351
        $put($1.args, "opt", $1.options); //#5352
        $k[$j++] = $1.args; //#5355
        if (!$1.dontdraw) { //#5355
            $0.renlinear(); //#5355
        } //#5355
    };
    $0.code39 = function() {
        var $1 = {}; //#5384
        $1.options = $k[--$j]; //#5386
        $1.barcode = $k[--$j]; //#5387
        $1.dontdraw = false; //#5389
        $1.includecheck = false; //#5390
        $1.validatecheck = false; //#5391
        $1.includetext = false; //#5392
        $1.includecheckintext = false; //#5393
        $1.hidestars = false; //#5394
        $1.textfont = "OCR-B"; //#5395
        $1.textsize = 10; //#5396
        $1.textyoffset = -8; //#5397
        $1.height = 1; //#5398
        $forall($1.options, function() { //#5409
            var _3 = $k[--$j]; //#5409
            $1[$k[--$j]] = _3; //#5409
        }); //#5409
        $1.textfont = "" + $1.textfont; //#5411
        $1.textsize = +$1.textsize; //#5412
        $1.textyoffset = +$1.textyoffset; //#5413
        $1.height = +$1.height; //#5414
        $1.barchars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%"; //#5417
        $1.charvals = new Map; //#5418
        for (var _9 = 0; _9 <= 42; _9 += 1) { //#5419
            $put($1.charvals, $geti($1.barchars, _9, 1), _9); //#5419
        } //#5419
        for (var _F = 0, _E = $1.barcode.length - 1; _F <= _E; _F += 1) { //#5426
            var _J = $get($1.charvals, $geti($1.barcode, _F, 1)) !== undefined; //#5423
            if (!_J) { //#5425
                $k[$j++] = 'bwipp.code39badCharacter'; //#5424
                $k[$j++] = "Code 39 must contain only digits, capital letters, spaces and the symbols -.$/+%"; //#5424
                $0.raiseerror(); //#5424
            } //#5424
        } //#5424
        $k[$j++] = 'barlen'; //#5428
        $k[$j++] = $1.barcode.length; //#5428
        if ($1.validatecheck) { //#5428
            var _M = $k[--$j]; //#5428
            $k[$j++] = $f(_M - 1); //#5428
        } //#5428
        var _N = $k[--$j]; //#5428
        $1[$k[--$j]] = _N; //#5428
        $1.checksum = 0; //#5430
        for (var _R = 0, _Q = $f($1.barlen - 1); _R <= _Q; _R += 1) { //#5434
            $1.checksum = $f($get($1.charvals, $geti($1.barcode, _R, 1)) + $1.checksum); //#5433
        } //#5433
        $1.checksum = $1.checksum % 43; //#5435
        if ($1.validatecheck) { //#5442
            if ($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum)) { //#5439
                $k[$j++] = 'bwipp.code39badCheckDigit'; //#5438
                $k[$j++] = "Incorrect Code 39 check digit provided"; //#5438
                $0.raiseerror(); //#5438
            } //#5438
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#5440
            $1.includecheck = true; //#5441
        } //#5441
        $1.encs = $a(["1113313111", "3113111131", "1133111131", "3133111111", "1113311131", "3113311111", "1133311111", "1113113131", "3113113111", "1133113111", "3111131131", "1131131131", "3131131111", "1111331131", "3111331111", "1131331111", "1111133131", "3111133111", "1131133111", "1111333111", "3111111331", "1131111331", "3131111311", "1111311331", "3111311311", "1131311311", "1111113331", "3111113311", "1131113311", "1111313311", "3311111131", "1331111131", "3331111111", "1311311131", "3311311111", "1331311111", "1311113131", "3311113111", "1331113111", "1313131111", "1313111311", "1311131311", "1113131311", "1311313111"]); //#5455
        var _l = $1.includecheck ? 3 : 2; //#5457
        $1.sbs = $s($f($1.barlen + _l) * 10); //#5457
        var _p = $1.includecheck ? 3 : 2; //#5458
        $1.txt = $a($f($1.barlen + _p)); //#5458
        $puti($1.sbs, 0, $get($1.encs, 43)); //#5461
        if (!$1.hidestars) { //#5465
            $put($1.txt, 0, $a(["*", 0, $1.textyoffset, $1.textfont, $1.textsize])); //#5463
        } else { //#5465
            $put($1.txt, 0, $a(["", 0, $1.textyoffset, $1.textfont, $1.textsize])); //#5465
        } //#5465
        for (var _17 = 0, _16 = $f($1.barlen - 1); _17 <= _16; _17 += 1) { //#5473
            $1.i = _17; //#5469
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#5470
            $puti($1.sbs, ($1.i * 10) + 10, $get($1.encs, $1.indx)); //#5471
            $put($1.txt, $1.i + 1, $a([$geti($1.barcode, $1.i, 1), ($1.i + 1) * 16, $1.textyoffset, $1.textfont, $1.textsize])); //#5472
        } //#5472
        if ($1.includecheck) { //#5494
            $puti($1.sbs, $f(($1.barlen * 10) + 10), $get($1.encs, $1.checksum)); //#5477
            if ($1.includecheckintext) { //#5481
                $put($1.txt, $f($1.barlen + 1), $a([$geti($1.barchars, $1.checksum, 1), $f($1.barlen + 1) * 16, $1.textyoffset, $1.textfont, $1.textsize])); //#5479
            } else { //#5481
                $put($1.txt, $f($1.barlen + 1), $a(["", $f($1.barlen + 1) * 16, $1.textyoffset, $1.textfont, $1.textsize])); //#5481
            } //#5481
            $puti($1.sbs, $f(($1.barlen * 10) + 20), $get($1.encs, 43)); //#5483
            if (!$1.hidestars) { //#5487
                $put($1.txt, $f($1.barlen + 2), $a(["*", $f($1.barlen + 2) * 16, $1.textyoffset, $1.textfont, $1.textsize])); //#5485
            } else { //#5487
                $put($1.txt, $f($1.barlen + 2), $a(["", $f($1.barlen + 2) * 16, $1.textyoffset, $1.textfont, $1.textsize])); //#5487
            } //#5487
        } else { //#5494
            $puti($1.sbs, $f(($1.barlen * 10) + 10), $get($1.encs, 43)); //#5490
            if (!$1.hidestars) { //#5494
                $put($1.txt, $f($1.barlen + 1), $a(["*", $f($1.barlen + 1) * 16, $1.textyoffset, $1.textfont, $1.textsize])); //#5492
            } else { //#5494
                $put($1.txt, $f($1.barlen + 1), $a(["", $f($1.barlen + 1) * 16, $1.textyoffset, $1.textfont, $1.textsize])); //#5494
            } //#5494
        } //#5494
        $k[$j++] = Infinity; //#5507
        $k[$j++] = Infinity; //#5501
        var _2S = $1.sbs; //#5501
        for (var _2T = 0, _2U = _2S.length; _2T < _2U; _2T++) { //#5501
            $k[$j++] = $get(_2S, _2T) - 48; //#5501
        } //#5501
        var _2W = $a(); //#5501
        $k[$j++] = Infinity; //#5502
        for (var _2Y = 0, _2Z = ~~(($1.sbs.length + 1) / 2); _2Y < _2Z; _2Y++) { //#5502
            $k[$j++] = $1.height; //#5502
        } //#5502
        var _2b = $a(); //#5502
        $k[$j++] = Infinity; //#5503
        for (var _2d = 0, _2e = ~~(($1.sbs.length + 1) / 2); _2d < _2e; _2d++) { //#5503
            $k[$j++] = 0; //#5503
        } //#5503
        var _2f = $a(); //#5503
        $k[$j++] = 'ren'; //#5506
        $k[$j++] = $0.renlinear; //#5506
        $k[$j++] = 'sbs'; //#5506
        $k[$j++] = _2W; //#5506
        $k[$j++] = 'bhs'; //#5506
        $k[$j++] = _2b; //#5506
        $k[$j++] = 'bbs'; //#5506
        $k[$j++] = _2f; //#5506
        if ($1.includetext) { //#5506
            $k[$j++] = 'txt'; //#5505
            $k[$j++] = $1.txt; //#5505
        } //#5505
        $k[$j++] = 'opt'; //#5507
        $k[$j++] = $1.options; //#5507
        var _2j = $d(); //#5507
        $k[$j++] = _2j; //#5510
        if (!$1.dontdraw) { //#5510
            $0.renlinear(); //#5510
        } //#5510
    };
    $0.code39ext = function() {
        var $1 = {}; //#5541
        $1.options = $k[--$j]; //#5543
        $1.barcode = $k[--$j]; //#5544
        $1.dontdraw = false; //#5546
        $1.includetext = false; //#5547
        $1.parse = false; //#5548
        $forall($1.options, function() { //#5559
            var _3 = $k[--$j]; //#5559
            $1[$k[--$j]] = _3; //#5559
        }); //#5559
        var _6 = new Map([
            ["parse", $1.parse],
            ["parseonly", true],
            ["parsefnc", false]
        ]); //#5565
        $1.fncvals = _6; //#5566
        $k[$j++] = 'barcode'; //#5567
        $k[$j++] = $1.barcode; //#5567
        $k[$j++] = $1.fncvals; //#5567
        $0.parseinput(); //#5567
        var _9 = $k[--$j]; //#5567
        $1[$k[--$j]] = _9; //#5567
        $1.barlen = $1.barcode.length; //#5568
        delete $1.options["parse"]; //#5569
        $forall($1.barcode, function() { //#5576
            if ($k[--$j] >= 128) { //#5575
                $k[$j++] = 'bwipp.code39extBadCharacter'; //#5574
                $k[$j++] = "Code 39 Extended must contain only ASCII characters"; //#5574
                $0.raiseerror(); //#5574
            } //#5574
        }); //#5574
        $1.extencs = $a(["%U", "$A", "$B", "$C", "$D", "$E", "$F", "$G", "$H", "$I", "$J", "$K", "$L", "$M", "$N", "$O", "$P", "$Q", "$R", "$S", "$T", "$U", "$V", "$W", "$X", "$Y", "$Z", "%A", "%B", "%C", "%D", "%E", " ", "/A", "/B", "/C", "/D", "/E", "/F", "/G", "/H", "/I", "/J", "/K", "/L", "-", ".", "/O", "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "/Z", "%F", "%G", "%H", "%I", "%J", "%V", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "%K", "%L", "%M", "%N", "%O", "%W", "+A", "+B", "+C", "+D", "+E", "+F", "+G", "+H", "+I", "+J", "+K", "+L", "+M", "+N", "+O", "+P", "+Q", "+R", "+S", "+T", "+U", "+V", "+W", "+X", "+Y", "+Z", "%P", "%Q", "%R", "%S", "%T"]); //#5588
        $1.newcode = $s($1.barlen * 2); //#5590
        $1.newtext = $s($1.barlen * 2); //#5591
        $1.j = 0; //#5592
        for (var _M = 0, _L = $1.barlen - 1; _M <= _L; _M += 1) { //#5600
            $1.i = _M; //#5594
            $1.extchar = $get($1.extencs, $get($1.barcode, $1.i)); //#5595
            $puti($1.newcode, $1.j, $1.extchar); //#5596
            $put($1.newtext, $1.j, $get($1.barcode, $1.i)); //#5597
            if ($1.extchar.length != 1) { //#5598
                $puti($1.newtext, $1.j + 1, " "); //#5598
            } //#5598
            $1.j = $1.j + $1.extchar.length; //#5599
        } //#5599
        $1.newcode = $geti($1.newcode, 0, $1.j); //#5601
        $1.newtext = $geti($1.newtext, 0, $1.j); //#5602
        $put($1.options, "dontdraw", true); //#5605
        $k[$j++] = 'args'; //#5606
        $k[$j++] = $1.newcode; //#5606
        $k[$j++] = $1.options; //#5606
        $0.code39(); //#5606
        var _o = $k[--$j]; //#5606
        $1[$k[--$j]] = _o; //#5606
        if ($1.includetext) { //#5618
            $1.txt = $get($1.args, "txt"); //#5610
            for (var _v = 0, _u = $1.newtext.length - 1; _v <= _u; _v += 1) { //#5616
                $1.i = _v; //#5612
                $1.txtentry = $get($1.txt, $1.i + 1); //#5613
                $put($1.txtentry, 0, $geti($1.newtext, $1.i, 1)); //#5614
                $put($1.txt, $1.i + 1, $1.txtentry); //#5615
            } //#5615
            $put($1.args, "txt", $1.txt); //#5617
        } //#5617
        $put($1.args, "opt", $1.options); //#5619
        $k[$j++] = $1.args; //#5622
        if (!$1.dontdraw) { //#5622
            $0.renlinear(); //#5622
        } //#5622
    };
    $0.code32 = function() {
        var $1 = {}; //#5652
        $1.options = $k[--$j]; //#5654
        $1.barcode = $k[--$j]; //#5655
        $1.dontdraw = false; //#5657
        $1.includetext = false; //#5658
        $1.textfont = "OCR-B"; //#5659
        $1.textsize = 10; //#5660
        $1.textyoffset = -8; //#5661
        $1.textxoffset = 0; //#5662
        $1.height = 1; //#5663
        $forall($1.options, function() { //#5674
            var _3 = $k[--$j]; //#5674
            $1[$k[--$j]] = _3; //#5674
        }); //#5674
        $1.textfont = "" + $1.textfont; //#5676
        $1.textsize = +$1.textsize; //#5677
        $1.textyoffset = +$1.textyoffset; //#5678
        $1.height = +$1.height; //#5679
        if (($1.barcode.length != 8) && ($1.barcode.length != 9)) { //#5684
            $k[$j++] = 'bwipp.code32badLength'; //#5683
            $k[$j++] = "Italian Pharmacode must be 8 or 9 digits"; //#5683
            $0.raiseerror(); //#5683
        } //#5683
        $forall($1.barcode, function() { //#5689
            var _C = $k[--$j]; //#5686
            if ((_C < 48) || (_C > 57)) { //#5688
                $k[$j++] = 'bwipp.code32badCharacter'; //#5687
                $k[$j++] = "Italian Pharmacode must contain only digits"; //#5687
                $0.raiseerror(); //#5687
            } //#5687
        }); //#5687
        $1.checksum = 0; //#5692
        for (var _D = 0; _D <= 7; _D += 1) { //#5697
            $1.i = _D; //#5694
            $k[$j++] = $f($get($1.barcode, $1.i) - 48); //#5695
            if (($1.i % 2) != 0) { //#5695
                var _I = $k[--$j]; //#5695
                $k[$j++] = _I * 2; //#5695
            } //#5695
            var _J = $k[--$j]; //#5695
            $k[$j++] = _J; //#5695
            if (_J > 9) { //#5695
                var _K = $k[--$j]; //#5695
                $k[$j++] = $f(_K - 9); //#5695
            } //#5695
            $1.checksum = $f($k[--$j] + $1.checksum); //#5696
        } //#5696
        $1.checksum = $1.checksum % 10; //#5698
        if ($1.barcode.length == 9) { //#5703
            if ($get($1.barcode, 8) != $f($1.checksum + 48)) { //#5702
                $k[$j++] = 'bwipp.code32badCheckDigit'; //#5701
                $k[$j++] = "Incorrect Italian Pharmacode check digit provided"; //#5701
                $0.raiseerror(); //#5701
            } //#5701
        } //#5701
        var _S = $s(10); //#5704
        $puti(_S, 0, $1.barcode); //#5705
        $put(_S, 8, $f($1.checksum + 48)); //#5706
        $1.text = _S; //#5707
        $1.val = $cvrs($s(6), ~~$z($1.text), 32); //#5710
        $1.barcode = $s(6); //#5711
        for (var _Z = 0; _Z <= 5; _Z += 1) { //#5712
            $put($1.barcode, _Z, 48); //#5712
        } //#5712
        $puti($1.barcode, 6 - $1.val.length, $1.val); //#5713
        for (var _e = 0; _e <= 5; _e += 1) { //#5720
            var _g = $get($1.barcode, _e); //#5717
            $k[$j++] = _e; //#5718
            $k[$j++] = _g; //#5718
            $k[$j++] = _g; //#5718
            for (var _h = 0, _i = "AEIO".length; _h < _i; _h++) { //#5718
                if ($k[--$j] >= $get("AEIO", _h)) { //#5718
                    var _l = $k[--$j]; //#5718
                    $k[$j++] = $f(_l + 1); //#5718
                } //#5718
                var _m = $k[--$j]; //#5718
                $k[$j++] = _m; //#5718
                $k[$j++] = _m; //#5718
            } //#5718
            $j--; //#5718
            var _o = $k[--$j]; //#5719
            $put($1.barcode, $k[--$j], _o); //#5719
        } //#5719
        $puti($1.text, 1, $geti($1.text, 0, 9)); //#5723
        $put($1.text, 0, 65); //#5724
        $put($1.options, "dontdraw", true); //#5727
        $k[$j++] = 'args'; //#5728
        $k[$j++] = $1.barcode; //#5728
        $k[$j++] = $1.options; //#5728
        $0.code39(); //#5728
        var _x = $k[--$j]; //#5728
        $1[$k[--$j]] = _x; //#5728
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#5730
        $put($1.args, "textxalign", "center"); //#5731
        $put($1.args, "opt", $1.options); //#5732
        $k[$j++] = $1.args; //#5735
        if (!$1.dontdraw) { //#5735
            $0.renlinear(); //#5735
        } //#5735
    };
    $0.pzn = function() {
        var $1 = {}; //#5765
        $1.options = $k[--$j]; //#5767
        $1.barcode = $k[--$j]; //#5768
        $1.dontdraw = false; //#5770
        $1.includetext = false; //#5771
        $1.textfont = "OCR-B"; //#5772
        $1.textsize = 10; //#5773
        $1.textyoffset = -8; //#5774
        $1.textxoffset = 0; //#5775
        $1.height = 1; //#5776
        $1.pzn8 = false; //#5777
        $forall($1.options, function() { //#5788
            var _3 = $k[--$j]; //#5788
            $1[$k[--$j]] = _3; //#5788
        }); //#5788
        $1.textfont = "" + $1.textfont; //#5790
        $1.textsize = +$1.textsize; //#5791
        $1.textyoffset = +$1.textyoffset; //#5792
        $1.textxoffset = +$1.textxoffset; //#5793
        $1.height = +$1.height; //#5794
        if ($1.pzn8) { //#5803
            if (($1.barcode.length != 7) && ($1.barcode.length != 8)) { //#5800
                $k[$j++] = 'bwipp.pzn8badLength'; //#5799
                $k[$j++] = "PZN8 must be 7 or 8 digits"; //#5799
                $0.raiseerror(); //#5799
            } //#5799
        } else { //#5803
            if (($1.barcode.length != 6) && ($1.barcode.length != 7)) { //#5804
                $k[$j++] = 'bwipp.pzn7badLength'; //#5803
                $k[$j++] = "PZN7 must be 6 or 7 digits"; //#5803
                $0.raiseerror(); //#5803
            } //#5803
        } //#5803
        $forall($1.barcode, function() { //#5810
            var _G = $k[--$j]; //#5807
            if ((_G < 48) || (_G > 57)) { //#5809
                $k[$j++] = 'bwipp.pznBadCharacter'; //#5808
                $k[$j++] = "PZN must contain only digits"; //#5808
                $0.raiseerror(); //#5808
            } //#5808
        }); //#5808
        var _I = $1.pzn8 ? 9 : 8; //#5813
        $1.msglen = _I; //#5813
        $1.checksum = 0; //#5814
        for (var _L = 0, _K = $1.msglen - 3; _L <= _K; _L += 1) { //#5819
            $1.i = _L; //#5816
            var _R = $1.pzn8 ? 1 : 2; //#5817
            $1.checksum = $f(($f($get($1.barcode, $1.i) - 48) * ($1.i + _R)) + $1.checksum); //#5818
        } //#5818
        $1.checksum = $1.checksum % 11; //#5820
        if ($1.checksum == 10) { //#5823
            $k[$j++] = 'bwipp.pznBadInputSequence'; //#5822
            $k[$j++] = "Incorrect PZN input sequence provided"; //#5822
            $0.raiseerror(); //#5822
        } //#5822
        if ($1.barcode.length == ($1.msglen - 1)) { //#5828
            if ($get($1.barcode, $1.msglen - 2) != $f($1.checksum + 48)) { //#5827
                $k[$j++] = 'bwipp.pznBadCheckDigit'; //#5826
                $k[$j++] = "Incorrect PZN check digit provided"; //#5826
                $0.raiseerror(); //#5826
            } //#5826
        } //#5826
        var _c = $s($1.msglen); //#5829
        $put(_c, 0, 45); //#5830
        $puti(_c, 1, $1.barcode); //#5831
        $put(_c, $1.msglen - 1, $f($1.checksum + 48)); //#5832
        $1.msg = _c; //#5833
        $put($1.options, "dontdraw", true); //#5836
        $k[$j++] = 'args'; //#5837
        $k[$j++] = $1.msg; //#5837
        $k[$j++] = $1.options; //#5837
        $0.code39(); //#5837
        var _j = $k[--$j]; //#5837
        $1[$k[--$j]] = _j; //#5837
        $1.text = $s($1.msglen + 5); //#5840
        $puti($1.text, 0, "PZN - "); //#5841
        $puti($1.text, 6, $geti($1.msg, 1, $1.msglen - 1)); //#5842
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#5844
        $put($1.args, "textxalign", "center"); //#5845
        $put($1.args, "opt", $1.options); //#5846
        $k[$j++] = $1.args; //#5849
        if (!$1.dontdraw) { //#5849
            $0.renlinear(); //#5849
        } //#5849
    };
    $0.code93 = function() {
        var $1 = {}; //#5879
        $1.options = $k[--$j]; //#5881
        $1.barcode = $k[--$j]; //#5882
        $1.dontdraw = false; //#5884
        $1.includecheck = false; //#5885
        $1.includetext = false; //#5886
        $1.textfont = "OCR-B"; //#5887
        $1.textsize = 10; //#5888
        $1.textyoffset = -8; //#5889
        $1.height = 1; //#5890
        $1.parse = false; //#5891
        $1.parsefnc = false; //#5892
        $forall($1.options, function() { //#5903
            var _3 = $k[--$j]; //#5903
            $1[$k[--$j]] = _3; //#5903
        }); //#5903
        $1.textfont = "" + $1.textfont; //#5905
        $1.textsize = +$1.textsize; //#5906
        $1.textyoffset = +$1.textyoffset; //#5907
        $1.height = +$1.height; //#5908
        $1.encs = $a(["131112", "111213", "111312", "111411", "121113", "121212", "121311", "111114", "131211", "141111", "211113", "211212", "211311", "221112", "221211", "231111", "112113", "112212", "112311", "122112", "132111", "111123", "111222", "111321", "121122", "131121", "212112", "212211", "211122", "211221", "221121", "222111", "112122", "112221", "122121", "123111", "121131", "311112", "311211", "321111", "112131", "113121", "211131", "121221", "312111", "311121", "122211", "111141", "1111411"]); //#5921
        $1.barchars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%"; //#5924
        $1.sft1 = -1; //#5927
        $1.sft2 = -2; //#5927
        $1.sft3 = -3; //#5927
        $1.sft4 = -4; //#5927
        var _G = new Map([
            ["parse", $1.parse],
            ["parsefnc", $1.parsefnc],
            ["SFT$", $1.sft1],
            ["SFT%", $1.sft2],
            ["SFT/", $1.sft3],
            ["SFT+", $1.sft4]
        ]); //#5934
        $1.fncvals = _G; //#5935
        $k[$j++] = 'msg'; //#5936
        $k[$j++] = $1.barcode; //#5936
        $k[$j++] = $1.fncvals; //#5936
        $0.parseinput(); //#5936
        var _J = $k[--$j]; //#5936
        $1[$k[--$j]] = _J; //#5936
        $1.msglen = $1.msg.length; //#5937
        if ($1.includecheck) { //#5942
            $1.sbs = $s(($1.msglen * 6) + 25); //#5940
        } else { //#5942
            $1.sbs = $s(($1.msglen * 6) + 13); //#5942
        } //#5942
        $1.txt = $a($1.msglen); //#5944
        $puti($1.sbs, 0, $get($1.encs, 47)); //#5947
        $1.checksum1 = 0; //#5949
        $1.checksum2 = 0; //#5949
        for (var _Y = 0, _X = $1.msglen - 1; _Y <= _X; _Y += 1) { //#5968
            $1.i = _Y; //#5951
            var _b = $get($1.msg, $1.i); //#5953
            $k[$j++] = _b; //#5962
            if (_b < 0) { //#5961
                $1.indx = $f(42 - $k[--$j]); //#5954
                $1.char = " "; //#5955
            } else { //#5961
                var _d = $s(1); //#5957
                $put(_d, 0, $k[--$j]); //#5957
                $1.char = _d; //#5957
                $search($1.barchars, $1.char); //#5958
                $j--; //#5959
                $1.indx = $k[--$j].length; //#5960
                $j -= 2; //#5961
            } //#5961
            $1.enc = $get($1.encs, $1.indx); //#5963
            $puti($1.sbs, ($1.i * 6) + 6, $1.enc); //#5964
            $put($1.txt, $1.i, $a([$1.char, ($1.i * 9) + 9, $1.textyoffset, $1.textfont, $1.textsize])); //#5965
            $1.checksum1 = $1.checksum1 + ((((($1.msglen - $1.i) - 1) % 20) + 1) * $1.indx); //#5966
            $1.checksum2 = $1.checksum2 + (((($1.msglen - $1.i) % 15) + 1) * $1.indx); //#5967
        } //#5967
        if ($1.includecheck) { //#5980
            $1.checksum1 = $1.checksum1 % 47; //#5972
            $1.checksum2 = ($1.checksum2 + $1.checksum1) % 47; //#5973
            $puti($1.sbs, ($1.msglen * 6) + 6, $get($1.encs, $1.checksum1)); //#5974
            $puti($1.sbs, ($1.msglen * 6) + 12, $get($1.encs, $1.checksum2)); //#5975
            $puti($1.sbs, ($1.msglen * 6) + 18, $get($1.encs, 48)); //#5977
        } else { //#5980
            $puti($1.sbs, ($1.msglen * 6) + 6, $get($1.encs, 48)); //#5980
        } //#5980
        $k[$j++] = Infinity; //#5992
        $k[$j++] = Infinity; //#5986
        var _1Q = $1.sbs; //#5986
        for (var _1R = 0, _1S = _1Q.length; _1R < _1S; _1R++) { //#5986
            $k[$j++] = $get(_1Q, _1R) - 48; //#5986
        } //#5986
        var _1U = $a(); //#5986
        $k[$j++] = Infinity; //#5987
        for (var _1W = 0, _1X = ~~(($1.sbs.length + 1) / 2); _1W < _1X; _1W++) { //#5987
            $k[$j++] = $1.height; //#5987
        } //#5987
        var _1Z = $a(); //#5987
        $k[$j++] = Infinity; //#5988
        for (var _1b = 0, _1c = ~~(($1.sbs.length + 1) / 2); _1b < _1c; _1b++) { //#5988
            $k[$j++] = 0; //#5988
        } //#5988
        var _1d = $a(); //#5988
        $k[$j++] = 'ren'; //#5991
        $k[$j++] = $0.renlinear; //#5991
        $k[$j++] = 'sbs'; //#5991
        $k[$j++] = _1U; //#5991
        $k[$j++] = 'bhs'; //#5991
        $k[$j++] = _1Z; //#5991
        $k[$j++] = 'bbs'; //#5991
        $k[$j++] = _1d; //#5991
        if ($1.includetext) { //#5991
            $k[$j++] = 'txt'; //#5990
            $k[$j++] = $1.txt; //#5990
        } //#5990
        $k[$j++] = 'opt'; //#5992
        $k[$j++] = $1.options; //#5992
        var _1h = $d(); //#5992
        $k[$j++] = _1h; //#5995
        if (!$1.dontdraw) { //#5995
            $0.renlinear(); //#5995
        } //#5995
    };
    $0.code93ext = function() {
        var $1 = {}; //#6026
        $1.options = $k[--$j]; //#6028
        $1.barcode = $k[--$j]; //#6029
        $1.dontdraw = false; //#6031
        $1.includetext = false; //#6032
        $1.parse = false; //#6033
        $forall($1.options, function() { //#6044
            var _3 = $k[--$j]; //#6044
            $1[$k[--$j]] = _3; //#6044
        }); //#6044
        var _6 = new Map([
            ["parse", $1.parse],
            ["parseonly", true],
            ["parsefnc", false]
        ]); //#6050
        $1.fncvals = _6; //#6051
        $k[$j++] = 'barcode'; //#6052
        $k[$j++] = $1.barcode; //#6052
        $k[$j++] = $1.fncvals; //#6052
        $0.parseinput(); //#6052
        var _9 = $k[--$j]; //#6052
        $1[$k[--$j]] = _9; //#6052
        $1.barlen = $1.barcode.length; //#6053
        delete $1.options["parse"]; //#6054
        $1.extencs = $a(["^SFT%U", "^SFT$A", "^SFT$B", "^SFT$C", "^SFT$D", "^SFT$E", "^SFT$F", "^SFT$G", "^SFT$H", "^SFT$I", "^SFT$J", "^SFT$K", "^SFT$L", "^SFT$M", "^SFT$N", "^SFT$O", "^SFT$P", "^SFT$Q", "^SFT$R", "^SFT$S", "^SFT$T", "^SFT$U", "^SFT$V", "^SFT$W", "^SFT$X", "^SFT$Y", "^SFT$Z", "^SFT%A", "^SFT%B", "^SFT%C", "^SFT%D", "^SFT%E", " ", "^SFT/A", "^SFT/B", "^SFT/C", "^SFT/D", "^SFT/E", "^SFT/F", "^SFT/G", "^SFT/H", "^SFT/I", "^SFT/J", "^SFT/K", "^SFT/L", "-", ".", "^SFT/O", "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "^SFT/Z", "^SFT%F", "^SFT%G", "^SFT%H", "^SFT%I", "^SFT%J", "^SFT%V", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "^SFT%K", "^SFT%L", "^SFT%M", "^SFT%N", "^SFT%O", "^SFT%W", "^SFT+A", "^SFT+B", "^SFT+C", "^SFT+D", "^SFT+E", "^SFT+F", "^SFT+G", "^SFT+H", "^SFT+I", "^SFT+J", "^SFT+K", "^SFT+L", "^SFT+M", "^SFT+N", "^SFT+O", "^SFT+P", "^SFT+Q", "^SFT+R", "^SFT+S", "^SFT+T", "^SFT+U", "^SFT+V", "^SFT+W", "^SFT+X", "^SFT+Y", "^SFT+Z", "^SFT%P", "^SFT%Q", "^SFT%R", "^SFT%S", "^SFT%T"]); //#6074
        $1.newcode = $s($1.barlen * 6); //#6076
        $1.newtext = $s($1.barlen * 6); //#6077
        $1.j = 0; //#6078
        $1.k = 0; //#6078
        for (var _K = 0, _J = $1.barlen - 1; _K <= _J; _K += 1) { //#6088
            $1.i = _K; //#6080
            $1.extchar = $get($1.extencs, $get($1.barcode, $1.i)); //#6081
            $1.extlen = $1.extchar.length; //#6082
            $puti($1.newcode, $1.j, $1.extchar); //#6083
            $put($1.newtext, $1.k, $get($1.barcode, $1.i)); //#6084
            if ($1.extlen != 1) { //#6085
                $puti($1.newtext, $1.k + 1, " "); //#6085
            } //#6085
            $1.j = $1.j + $1.extlen; //#6086
            var _g = ($1.extlen == 1) ? 1 : 2; //#6087
            $1.k = $1.k + _g; //#6087
        } //#6087
        $1.newcode = $geti($1.newcode, 0, $1.j); //#6089
        $1.newtext = $geti($1.newtext, 0, $1.k); //#6090
        $put($1.options, "dontdraw", true); //#6093
        $put($1.options, "parsefnc", true); //#6094
        $k[$j++] = 'args'; //#6095
        $k[$j++] = $1.newcode; //#6095
        $k[$j++] = $1.options; //#6095
        $0.code93(); //#6095
        var _r = $k[--$j]; //#6095
        $1[$k[--$j]] = _r; //#6095
        if ($1.includetext) { //#6107
            $1.txt = $get($1.args, "txt"); //#6099
            for (var _y = 0, _x = $1.newtext.length - 1; _y <= _x; _y += 1) { //#6105
                $1.i = _y; //#6101
                $1.txtentry = $get($1.txt, $1.i); //#6102
                $put($1.txtentry, 0, $geti($1.newtext, $1.i, 1)); //#6103
                $put($1.txt, $1.i, $1.txtentry); //#6104
            } //#6104
            $put($1.args, "txt", $1.txt); //#6106
        } //#6106
        $put($1.args, "opt", $1.options); //#6108
        $k[$j++] = $1.args; //#6111
        if (!$1.dontdraw) { //#6111
            $0.renlinear(); //#6111
        } //#6111
    };
    $0.interleaved2of5 = function() {
        var $1 = {}; //#6140
        $1.options = $k[--$j]; //#6142
        $1.barcode = $k[--$j]; //#6143
        $1.dontdraw = false; //#6145
        $1.includecheck = false; //#6146
        $1.includetext = false; //#6147
        $1.includecheckintext = false; //#6148
        $1.textfont = "OCR-B"; //#6149
        $1.textsize = 10; //#6150
        $1.textyoffset = -8; //#6151
        $1.height = 1; //#6152
        $forall($1.options, function() { //#6163
            var _3 = $k[--$j]; //#6163
            $1[$k[--$j]] = _3; //#6163
        }); //#6163
        $1.textfont = "" + $1.textfont; //#6165
        $1.textsize = +$1.textsize; //#6166
        $1.textyoffset = +$1.textyoffset; //#6167
        $1.height = +$1.height; //#6168
        $forall($1.barcode, function() { //#6175
            var _A = $k[--$j]; //#6172
            if ((_A < 48) || (_A > 57)) { //#6174
                $k[$j++] = 'bwipp.interleaved2of5badCharacter'; //#6173
                $k[$j++] = "Interleaved 2 of 5 must contain only digits"; //#6173
                $0.raiseerror(); //#6173
            } //#6173
        }); //#6173
        $1.barlen = $1.barcode.length; //#6177
        if (((($1.barlen % 2) == 0) && $1.includecheck) || ((($1.barlen % 2) != 0) && (!$1.includecheck))) { //#6188
            $1.pad = $s($1.barlen + 1); //#6183
            $put($1.pad, 0, 48); //#6184
            $puti($1.pad, 1, $1.barcode); //#6185
            $1.barcode = $1.pad; //#6186
            $1.barlen = $1.barlen + 1; //#6187
        } //#6187
        if ($1.includecheck) { //#6207
            $1.checksum = 0; //#6192
            for (var _Q = 0, _P = $1.barlen - 1; _Q <= _P; _Q += 1) { //#6200
                $1.i = _Q; //#6194
                if (($1.i % 2) == 0) { //#6198
                    $1.checksum = $1.checksum + (($get($1.barcode, $1.i) - 48) * 3); //#6196
                } else { //#6198
                    $1.checksum = $1.checksum + ($get($1.barcode, $1.i) - 48); //#6198
                } //#6198
            } //#6198
            $1.checksum = (10 - ($1.checksum % 10)) % 10; //#6201
            $1.pad = $s($1.barlen + 1); //#6202
            $puti($1.pad, 0, $1.barcode); //#6203
            $put($1.pad, $1.barlen, $1.checksum + 48); //#6204
            $1.barcode = $1.pad; //#6205
            $1.barlen = $1.barlen + 1; //#6206
        } //#6206
        $1.encs = $a(["11221", "21112", "12112", "22111", "11212", "21211", "12211", "11122", "21121", "12121", "1111", "2111"]); //#6214
        $1.barchars = "0123456789"; //#6217
        $1.sbs = $s(($1.barlen * 5) + 8); //#6218
        $1.txt = $a($1.barlen); //#6219
        $puti($1.sbs, 0, $get($1.encs, 10)); //#6222
        for (var _u = 0, _t = $1.barlen - 1; _u <= _t; _u += 2) { //#6256
            $1.i = _u; //#6225
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#6227
            $j--; //#6228
            $1.indx = $k[--$j].length; //#6229
            $j -= 2; //#6230
            $1.enca = $get($1.encs, $1.indx); //#6231
            $search($1.barchars, $geti($1.barcode, $1.i + 1, 1)); //#6233
            $j--; //#6234
            $1.indx = $k[--$j].length; //#6235
            $j -= 2; //#6236
            $1.encb = $get($1.encs, $1.indx); //#6237
            $1.intl = $s($1.enca.length * 2); //#6240
            for (var _1F = 0, _1E = $1.enca.length - 1; _1F <= _1E; _1F += 1) { //#6247
                $1.j = _1F; //#6242
                $1.achar = $get($1.enca, $1.j); //#6243
                $1.bchar = $get($1.encb, $1.j); //#6244
                $put($1.intl, $1.j * 2, $1.achar); //#6245
                $put($1.intl, ($1.j * 2) + 1, $1.bchar); //#6246
            } //#6246
            $puti($1.sbs, ($1.i * 5) + 4, $1.intl); //#6249
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), ($1.i * 9) + 4, $1.textyoffset, $1.textfont, $1.textsize])); //#6250
            if (($1.includecheck && (!$1.includecheckintext)) && (($1.barlen - 2) == $1.i)) { //#6254
                $put($1.txt, $1.i + 1, $a([" ", (($1.i + 1) * 9) + 4, $1.textyoffset, $1.textfont, $1.textsize])); //#6252
            } else { //#6254
                $put($1.txt, $1.i + 1, $a([$geti($1.barcode, $1.i + 1, 1), (($1.i + 1) * 9) + 4, $1.textyoffset, $1.textfont, $1.textsize])); //#6254
            } //#6254
        } //#6254
        $puti($1.sbs, ($1.barlen * 5) + 4, $get($1.encs, 11)); //#6259
        $k[$j++] = Infinity; //#6272
        $k[$j++] = Infinity; //#6264
        var _24 = $1.sbs; //#6264
        for (var _25 = 0, _26 = _24.length; _25 < _26; _25++) { //#6264
            $k[$j++] = $get(_24, _25) - 48; //#6264
        } //#6264
        var _28 = $a(); //#6264
        $k[$j++] = Infinity; //#6265
        for (var _2A = 0, _2B = ~~(($1.sbs.length + 1) / 2); _2A < _2B; _2A++) { //#6265
            $k[$j++] = $1.height; //#6265
        } //#6265
        var _2D = $a(); //#6265
        $k[$j++] = Infinity; //#6266
        for (var _2F = 0, _2G = ~~(($1.sbs.length + 1) / 2); _2F < _2G; _2F++) { //#6266
            $k[$j++] = 0; //#6266
        } //#6266
        var _2H = $a(); //#6266
        $k[$j++] = 'ren'; //#6269
        $k[$j++] = $0.renlinear; //#6269
        $k[$j++] = 'sbs'; //#6269
        $k[$j++] = _28; //#6269
        $k[$j++] = 'bhs'; //#6269
        $k[$j++] = _2D; //#6269
        $k[$j++] = 'bbs'; //#6269
        $k[$j++] = _2H; //#6269
        if ($1.includetext) { //#6269
            $k[$j++] = 'txt'; //#6268
            $k[$j++] = $1.txt; //#6268
        } //#6268
        $k[$j++] = 'barratio'; //#6272
        $k[$j++] = 2; //#6272
        $k[$j++] = 'spaceratio'; //#6272
        $k[$j++] = 2; //#6272
        $k[$j++] = 'opt'; //#6272
        $k[$j++] = $1.options; //#6272
        var _2L = $d(); //#6272
        $k[$j++] = _2L; //#6275
        if (!$1.dontdraw) { //#6275
            $0.renlinear(); //#6275
        } //#6275
    };
    $0.itf14 = function() {
        var $1 = {}; //#6305
        $1.options = $k[--$j]; //#6307
        $1.barcode = $k[--$j]; //#6308
        $1.dontdraw = false; //#6310
        $1.includetext = false; //#6311
        $1.includecheckintext = true; //#6312
        $1.textfont = "OCR-B"; //#6313
        $1.textsize = 10; //#6314
        $1.textxoffset = 0; //#6315
        $1.textyoffset = 3; //#6316
        $1.height = 0.5; //#6317
        $1.showborder = true; //#6318
        $1.borderwidth = 4; //#6319
        $1.borderleft = 15; //#6320
        $1.borderright = 15; //#6321
        $forall($1.options, function() { //#6332
            var _3 = $k[--$j]; //#6332
            $1[$k[--$j]] = _3; //#6332
        }); //#6332
        $1.textfont = "" + $1.textfont; //#6334
        $1.textsize = +$1.textsize; //#6335
        $1.textxoffset = +$1.textxoffset; //#6336
        $1.textyoffset = +$1.textyoffset; //#6337
        $1.height = +$1.height; //#6338
        $1.borderwidth = +$1.borderwidth; //#6339
        $1.borderleft = +$1.borderleft; //#6340
        $1.borderright = +$1.borderright; //#6341
        $1.text = $1.barcode; //#6344
        var _E = $1.barcode; //#6345
        $k[$j++] = $s(_E.length); //#6346
        $k[$j++] = 0; //#6346
        $forall(_E, function() { //#6346
            var _G = $k[--$j]; //#6346
            $k[$j++] = _G; //#6346
            if (_G != 32) { //#6346
                var _H = $k[--$j]; //#6346
                var _I = $k[--$j]; //#6346
                var _J = $k[--$j]; //#6346
                $put(_J, _I, _H); //#6346
                $k[$j++] = _J; //#6346
                $k[$j++] = $f(_I + 1); //#6346
            } else { //#6346
                $j--; //#6346
            } //#6346
        }); //#6346
        var _K = $k[--$j]; //#6347
        $1.barcode = $geti($k[--$j], 0, _K); //#6347
        $1.hasspace = $1.text.length != $1.barcode.length; //#6348
        if (($1.barcode.length != 13) && ($1.barcode.length != 14)) { //#6353
            $k[$j++] = 'bwipp.itf14badLength'; //#6352
            $k[$j++] = "ITF-14 must be 13 or 14 digits"; //#6352
            $0.raiseerror(); //#6352
        } //#6352
        var _R = $1.barcode; //#6354
        for (var _S = 0, _T = _R.length; _S < _T; _S++) { //#6358
            var _U = $get(_R, _S); //#6358
            if ((_U < 48) || (_U > 57)) { //#6357
                $k[$j++] = 'bwipp.itf14badCharacter'; //#6356
                $k[$j++] = "ITF-14 must contain only digits"; //#6356
                $0.raiseerror(); //#6356
            } //#6356
        } //#6356
        $1.checksum = 0; //#6361
        for (var _V = 0; _V <= 12; _V += 1) { //#6365
            $1.i = _V; //#6363
            $k[$j++] = 'checksum'; //#6364
            $k[$j++] = $1.checksum; //#6364
            $k[$j++] = $f($get($1.barcode, $1.i) - 48); //#6364
            if (($1.i % 2) == 0) { //#6364
                var _b = $k[--$j]; //#6364
                $k[$j++] = _b * 3; //#6364
            } //#6364
            var _c = $k[--$j]; //#6364
            var _d = $k[--$j]; //#6364
            $1[$k[--$j]] = $f(_d + _c); //#6364
        } //#6364
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#6366
        if ($1.barcode.length == 14) { //#6380
            if ($get($1.barcode, 13) != ($1.checksum + 48)) { //#6370
                $k[$j++] = 'bwipp.itf14badCheckDigit'; //#6369
                $k[$j++] = "Incorrect ITF-14 check digit provided"; //#6369
                $0.raiseerror(); //#6369
            } //#6369
        } else { //#6380
            var _k = $s(14); //#6372
            $puti(_k, 0, $1.barcode); //#6373
            $put(_k, 13, $1.checksum + 48); //#6374
            $1.barcode = _k; //#6375
            var _p = $1.hasspace ? 2 : 1; //#6376
            var _q = $s($1.text.length + _p); //#6376
            $puti(_q, _q.length - 2, " "); //#6377
            $put(_q, _q.length - 1, $1.checksum + 48); //#6378
            $puti(_q, 0, $1.text); //#6379
            $1.text = _q; //#6380
        } //#6380
        $put($1.options, "dontdraw", true); //#6384
        $put($1.options, "showborder", $1.showborder); //#6385
        $put($1.options, "borderwidth", $1.borderwidth); //#6386
        $put($1.options, "borderleft", $1.borderleft); //#6387
        $put($1.options, "borderright", $1.borderright); //#6388
        $put($1.options, "height", $1.height); //#6389
        $put($1.options, "textyoffset", $1.textyoffset); //#6390
        $k[$j++] = 'args'; //#6391
        $k[$j++] = $1.barcode; //#6391
        $k[$j++] = $1.options; //#6391
        $0.interleaved2of5(); //#6391
        var _18 = $k[--$j]; //#6391
        $1[$k[--$j]] = _18; //#6391
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#6393
        $put($1.args, "textxalign", "center"); //#6394
        $put($1.args, "opt", $1.options); //#6395
        $k[$j++] = $1.args; //#6398
        if (!$1.dontdraw) { //#6398
            $0.renlinear(); //#6398
        } //#6398
    };
    $0.identcode = function() {
        var $1 = {}; //#6428
        $1.options = $k[--$j]; //#6430
        $1.barcode = $k[--$j]; //#6431
        $1.dontdraw = false; //#6433
        $1.includetext = false; //#6434
        $1.textfont = "OCR-B"; //#6435
        $1.textsize = 10; //#6436
        $1.textxoffset = 0; //#6437
        $1.textyoffset = -8; //#6438
        $1.height = 1; //#6439
        $forall($1.options, function() { //#6450
            var _3 = $k[--$j]; //#6450
            $1[$k[--$j]] = _3; //#6450
        }); //#6450
        $1.textfont = "" + $1.textfont; //#6452
        $1.textsize = +$1.textsize; //#6453
        $1.textxoffset = +$1.textxoffset; //#6454
        $1.textyoffset = +$1.textyoffset; //#6455
        $1.height = +$1.height; //#6456
        if (($1.barcode.length != 11) && ($1.barcode.length != 12)) { //#6461
            $k[$j++] = 'bwipp.identcodeBadLength'; //#6460
            $k[$j++] = "Deutsche Post Identcode must be 11 or 12 digits"; //#6460
            $0.raiseerror(); //#6460
        } //#6460
        $forall($1.barcode, function() { //#6466
            var _D = $k[--$j]; //#6463
            if ((_D < 48) || (_D > 57)) { //#6465
                $k[$j++] = 'bwipp.identcodeBadCharacter'; //#6464
                $k[$j++] = "Deutsche Post Identcode must contain only digits"; //#6464
                $0.raiseerror(); //#6464
            } //#6464
        }); //#6464
        $1.checksum = 0; //#6469
        for (var _E = 0; _E <= 10; _E += 1) { //#6473
            $1.i = _E; //#6471
            $k[$j++] = 'checksum'; //#6472
            $k[$j++] = $1.checksum; //#6472
            $k[$j++] = $f($get($1.barcode, $1.i) - 48); //#6472
            if (($1.i % 2) == 0) { //#6472
                var _K = $k[--$j]; //#6472
                $k[$j++] = _K * 4; //#6472
            } else { //#6472
                var _L = $k[--$j]; //#6472
                $k[$j++] = _L * 9; //#6472
            } //#6472
            var _M = $k[--$j]; //#6472
            var _N = $k[--$j]; //#6472
            $1[$k[--$j]] = $f(_N + _M); //#6472
        } //#6472
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#6474
        if ($1.barcode.length == 12) { //#6479
            if ($get($1.barcode, 11) != ($1.checksum + 48)) { //#6478
                $k[$j++] = 'bwipp.identcodeBadCheckDigit'; //#6477
                $k[$j++] = "Incorrect Deutsche Post Identcode check digit provided"; //#6477
                $0.raiseerror(); //#6477
            } //#6477
        } //#6477
        var _U = $s(12); //#6480
        $puti(_U, 0, $1.barcode); //#6481
        $put(_U, 11, $1.checksum + 48); //#6482
        $1.barcode = _U; //#6483
        $1.text = $strcpy($s(16), "  .       .     "); //#6486
        $puti($1.text, 0, $geti($1.barcode, 0, 2)); //#6487
        $puti($1.text, 3, $geti($1.barcode, 2, 3)); //#6488
        $puti($1.text, 7, $geti($1.barcode, 5, 3)); //#6489
        $puti($1.text, 11, $geti($1.barcode, 8, 3)); //#6490
        $puti($1.text, 15, $geti($1.barcode, 11, 1)); //#6491
        $put($1.options, "dontdraw", true); //#6494
        $put($1.options, "includecheck", false); //#6495
        $k[$j++] = 'args'; //#6496
        $k[$j++] = $1.barcode; //#6496
        $k[$j++] = $1.options; //#6496
        $0.interleaved2of5(); //#6496
        var _s = $k[--$j]; //#6496
        $1[$k[--$j]] = _s; //#6496
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#6498
        $put($1.args, "textxalign", "center"); //#6499
        $put($1.args, "opt", $1.options); //#6500
        $k[$j++] = $1.args; //#6503
        if (!$1.dontdraw) { //#6503
            $0.renlinear(); //#6503
        } //#6503
    };
    $0.leitcode = function() {
        var $1 = {}; //#6533
        $1.options = $k[--$j]; //#6535
        $1.barcode = $k[--$j]; //#6536
        $1.dontdraw = false; //#6538
        $1.includetext = false; //#6539
        $1.textfont = "OCR-B"; //#6540
        $1.textsize = 10; //#6541
        $1.textxoffset = 0; //#6542
        $1.textyoffset = -8; //#6543
        $1.height = 1; //#6544
        $forall($1.options, function() { //#6555
            var _3 = $k[--$j]; //#6555
            $1[$k[--$j]] = _3; //#6555
        }); //#6555
        $1.textfont = "" + $1.textfont; //#6557
        $1.textsize = +$1.textsize; //#6558
        $1.textxoffset = +$1.textxoffset; //#6559
        $1.textyoffset = +$1.textyoffset; //#6560
        $1.height = +$1.height; //#6561
        if (($1.barcode.length != 13) && ($1.barcode.length != 14)) { //#6566
            $k[$j++] = 'bwipp.leitcodeBadLength'; //#6565
            $k[$j++] = "Deutsche Post Leitcode must be 13 or 14 digits"; //#6565
            $0.raiseerror(); //#6565
        } //#6565
        $forall($1.barcode, function() { //#6571
            var _D = $k[--$j]; //#6568
            if ((_D < 48) || (_D > 57)) { //#6570
                $k[$j++] = 'bwipp.leitcodeBadCharacter'; //#6569
                $k[$j++] = "Deutsche Post Leitcode must contain only digits"; //#6569
                $0.raiseerror(); //#6569
            } //#6569
        }); //#6569
        $1.checksum = 0; //#6574
        for (var _E = 0; _E <= 12; _E += 1) { //#6578
            $1.i = _E; //#6576
            $k[$j++] = 'checksum'; //#6577
            $k[$j++] = $1.checksum; //#6577
            $k[$j++] = $f($get($1.barcode, $1.i) - 48); //#6577
            if (($1.i % 2) == 0) { //#6577
                var _K = $k[--$j]; //#6577
                $k[$j++] = _K * 4; //#6577
            } else { //#6577
                var _L = $k[--$j]; //#6577
                $k[$j++] = _L * 9; //#6577
            } //#6577
            var _M = $k[--$j]; //#6577
            var _N = $k[--$j]; //#6577
            $1[$k[--$j]] = $f(_N + _M); //#6577
        } //#6577
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#6579
        if ($1.barcode.length == 14) { //#6584
            if ($get($1.barcode, 13) != ($1.checksum + 48)) { //#6583
                $k[$j++] = 'bwipp.leitcodeBadCheckDigit'; //#6582
                $k[$j++] = "Incorrect Deutsche Post Leitcode check digit provided"; //#6582
                $0.raiseerror(); //#6582
            } //#6582
        } //#6582
        var _U = $s(14); //#6585
        $puti(_U, 0, $1.barcode); //#6586
        $put(_U, 13, $1.checksum + 48); //#6587
        $1.barcode = _U; //#6588
        $1.text = $strcpy($s(18), "     .   .   .    "); //#6591
        $puti($1.text, 0, $geti($1.barcode, 0, 5)); //#6592
        $puti($1.text, 6, $geti($1.barcode, 5, 3)); //#6593
        $puti($1.text, 10, $geti($1.barcode, 8, 3)); //#6594
        $puti($1.text, 14, $geti($1.barcode, 11, 2)); //#6595
        $puti($1.text, 17, $geti($1.barcode, 13, 1)); //#6596
        $put($1.options, "dontdraw", true); //#6599
        $put($1.options, "includecheck", false); //#6600
        $k[$j++] = 'args'; //#6601
        $k[$j++] = $1.barcode; //#6601
        $k[$j++] = $1.options; //#6601
        $0.interleaved2of5(); //#6601
        var _s = $k[--$j]; //#6601
        $1[$k[--$j]] = _s; //#6601
        $put($1.args, "txt", $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])])); //#6603
        $put($1.args, "textxalign", "center"); //#6604
        $put($1.args, "opt", $1.options); //#6605
        $k[$j++] = $1.args; //#6608
        if (!$1.dontdraw) { //#6608
            $0.renlinear(); //#6608
        } //#6608
    };
    $0.databaromni = function() {
        var $1 = {}; //#6638
        $1.options = $k[--$j]; //#6640
        $1.barcode = $k[--$j]; //#6641
        $1.dontdraw = false; //#6643
        $1.height = 33 / 72; //#6644
        $1.linkage = false; //#6645
        $1.format = "omni"; //#6646
        $1.barxmult = 33; //#6647
        $forall($1.options, function() { //#6658
            var _3 = $k[--$j]; //#6658
            $1[$k[--$j]] = _3; //#6658
        }); //#6658
        if ($eq($1.format, "truncated")) { //#6660
            $1.height = 13 / 72; //#6660
        } //#6660
        $1.height = +$1.height; //#6661
        $1.barxmult = ~~$1.barxmult; //#6662
        if ($ne($geti($1.barcode, 0, 4), "(01)")) { //#6667
            $k[$j++] = 'bwipp.databaromniBadAI'; //#6666
            $k[$j++] = "GS1 DataBar Omnidirectional must begin with (01) application identifier"; //#6666
            $0.raiseerror(); //#6666
        } //#6666
        if (($1.barcode.length != 17) && ($1.barcode.length != 18)) { //#6670
            $k[$j++] = 'bwipp.databaromniBadLength'; //#6669
            $k[$j++] = "GS1 DataBar Omnidirectional must be 13 or 14 digits"; //#6669
            $0.raiseerror(); //#6669
        } //#6669
        var _E = $geti($1.barcode, 4, $1.barcode.length - 4); //#6671
        for (var _F = 0, _G = _E.length; _F < _G; _F++) { //#6675
            var _H = $get(_E, _F); //#6675
            if ((_H < 48) || (_H > 57)) { //#6674
                $k[$j++] = 'bwipp.databaromniBadCharacter'; //#6673
                $k[$j++] = "GS1 DataBar Omnidirectional must contain only digits"; //#6673
                $0.raiseerror(); //#6673
            } //#6673
        } //#6673
        $1.checksum = 0; //#6678
        for (var _I = 0; _I <= 12; _I += 1) { //#6682
            $1.i = _I; //#6680
            $k[$j++] = 'checksum'; //#6681
            $k[$j++] = $1.checksum; //#6681
            $k[$j++] = $f($get($1.barcode, $1.i + 4) - 48); //#6681
            if (($1.i % 2) == 0) { //#6681
                var _O = $k[--$j]; //#6681
                $k[$j++] = _O * 3; //#6681
            } //#6681
            var _P = $k[--$j]; //#6681
            var _Q = $k[--$j]; //#6681
            $1[$k[--$j]] = $f(_Q + _P); //#6681
        } //#6681
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#6683
        if ($1.barcode.length == 18) { //#6688
            if ($get($1.barcode, 17) != ($1.checksum + 48)) { //#6687
                $k[$j++] = 'bwipp.databaromniBadCheckDigit'; //#6686
                $k[$j++] = "Incorrect GS1 DataBar Omnidirectional check digit provided"; //#6686
                $0.raiseerror(); //#6686
            } //#6686
        } //#6686
        var _X = $s(18); //#6689
        $puti(_X, 0, $1.barcode); //#6690
        $put(_X, 17, $1.checksum + 48); //#6691
        $1.barcode = _X; //#6692
        $1.txt = $a($1.barcode.length); //#6695
        for (var _e = 0, _d = $1.barcode.length - 1; _e <= _d; _e += 1) { //#6699
            $1.i = _e; //#6697
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), 0, 0, "", 0])); //#6698
        } //#6698
        $1.ncr = function() {
            var _l = $k[--$j]; //#6702
            var _m = $k[--$j]; //#6702
            var _n = $f(_m - _l); //#6702
            if (_l < $f(_m - _l)) { //#6702
                var _ = _n; //#6702
                _n = _l; //#6702
                _l = _; //#6702
            } //#6702
            $k[$j++] = _n; //#6709
            $k[$j++] = 1; //#6709
            $k[$j++] = 1; //#6709
            for (var _p = _m, _o = $f(_l + 1); _p >= _o; _p -= 1) { //#6709
                var _q = $k[--$j]; //#6705
                var _r = $k[--$j]; //#6706
                var _s = $k[--$j]; //#6706
                $k[$j++] = _s; //#6708
                $k[$j++] = _r; //#6708
                $k[$j++] = _q * _p; //#6708
                if ($le(_r, _s)) { //#6708
                    var _t = $k[--$j]; //#6707
                    var _u = $k[--$j]; //#6707
                    $k[$j++] = $f(_u + 1); //#6707
                    $k[$j++] = ~~(_t / _u); //#6707
                } //#6707
            } //#6707
            for (;;) { //#6713
                var _v = $k[--$j]; //#6711
                var _w = $k[--$j]; //#6711
                var _x = $k[--$j]; //#6711
                $k[$j++] = _x; //#6711
                $k[$j++] = _w; //#6711
                $k[$j++] = _v; //#6711
                if ($gt(_w, _x)) { //#6711
                    break; //#6711
                } //#6711
                var _y = $k[--$j]; //#6712
                var _z = $k[--$j]; //#6712
                $k[$j++] = $f(_z + 1); //#6712
                $k[$j++] = ~~(_y / _z); //#6712
            } //#6712
            var _10 = $k[--$j]; //#6714
            var _11 = $k[--$j]; //#6714
            $k[$j++] = _10; //#6714
            $k[$j++] = _11; //#6714
            $j--; //#6714
            var _12 = $k[--$j]; //#6714
            var _13 = $k[--$j]; //#6714
            $k[$j++] = _12; //#6714
            $k[$j++] = _13; //#6714
            $j--; //#6714
        }; //#6715
        $1.getRSSwidths = function() {
            $1.oe = $k[--$j]; //#6718
            $1.el = $k[--$j]; //#6719
            $1.mw = $k[--$j]; //#6720
            $1.nm = $k[--$j]; //#6721
            $1.val = $k[--$j]; //#6722
            $1.out = $a($1.el); //#6723
            $1.mask = 0; //#6724
            for (var _1D = 0, _1C = $f($1.el - 2); _1D <= _1C; _1D += 1) { //#6751
                $1.bar = _1D; //#6726
                $1.ew = 1; //#6727
                var _1F = $1.bar; //#6728
                $1.mask = $1.mask | ((_1F < 0 ? 1 >>> -_1F : 1 << _1F)); //#6728
                for (;;) { //#6747
                    $k[$j++] = 'sval'; //#6729
                    $k[$j++] = $f($f($1.nm - $1.ew) - 1); //#6729
                    $k[$j++] = $f($f($1.el - $1.bar) - 2); //#6729
                    $1.ncr(); //#6729
                    var _1K = $k[--$j]; //#6729
                    $1[$k[--$j]] = _1K; //#6729
                    if (($1.oe && ($1.mask == 0)) && (($f(($f($f($1.nm - $1.ew) - ($1.el * 2))) + ($1.bar * 2))) >= -2)) { //#6732
                        $k[$j++] = 'sval'; //#6731
                        $k[$j++] = $1.sval; //#6731
                        $k[$j++] = $f(($f($f($1.nm - $1.ew) - $1.el)) + $1.bar); //#6731
                        $k[$j++] = $f($f($1.el - $1.bar) - 2); //#6731
                        $1.ncr(); //#6731
                        var _1Z = $k[--$j]; //#6731
                        var _1a = $k[--$j]; //#6731
                        $1[$k[--$j]] = $f(_1a - _1Z); //#6731
                    } //#6731
                    if ($f($1.el - $1.bar) > 2) { //#6741
                        $1.lval = 0; //#6734
                        for (var _1k = $f(($f(($f($f($1.nm - $1.ew) - $1.el)) + $1.bar)) + 2), _1j = $f($1.mw + 1); _1k >= _1j; _1k -= 1) { //#6738
                            $k[$j++] = $f(($f($f($1.nm - _1k) - $1.ew)) - 1); //#6736
                            $k[$j++] = $f($f($1.el - $1.bar) - 3); //#6736
                            $1.ncr(); //#6736
                            $1.lval = $f($k[--$j] + $1.lval); //#6737
                        } //#6737
                        $1.sval = $f($1.sval - ($1.lval * ($f($f($1.el - $1.bar) - 1)))); //#6739
                    } else { //#6741
                        if ($f($1.nm - $1.ew) > $1.mw) { //#6741
                            $1.sval = $f($1.sval - 1); //#6741
                        } //#6741
                    } //#6741
                    $1.val = $f($1.val - $1.sval); //#6743
                    if ($1.val < 0) { //#6744
                        break; //#6744
                    } //#6744
                    $1.ew = $1.ew + 1; //#6745
                    var _24 = $1.bar; //#6746
                    $1.mask = $1.mask & (~((_24 < 0 ? 1 >>> -_24 : 1 << _24))); //#6746
                } //#6746
                $1.val = $f($1.val + $1.sval); //#6748
                $1.nm = $f($1.nm - $1.ew); //#6749
                $put($1.out, $1.bar, $1.ew); //#6750
            } //#6750
            $put($1.out, $f($1.el - 1), $1.nm); //#6752
            $k[$j++] = $1.out; //#6753
        }; //#6754
        $k[$j++] = Infinity; //#6758
        var _2H = $1.linkage ? 1 : 0; //#6757
        var _2J = $geti($1.barcode, 4, 13); //#6758
        $k[$j++] = _2H; //#6758
        for (var _2K = 0, _2L = _2J.length; _2K < _2L; _2K++) { //#6758
            $k[$j++] = $f($get(_2J, _2K) - 48); //#6758
        } //#6758
        $1.binval = $a(); //#6758
        for (var _2O = 0; _2O <= 12; _2O += 1) { //#6765
            $1.i = _2O; //#6762
            var _2P = $1.binval; //#6763
            var _2Q = $1.i; //#6763
            $put(_2P, _2Q + 1, $f($get(_2P, _2Q + 1) + (($get($1.binval, $1.i) % 4537077) * 10))); //#6763
            $put($1.binval, $1.i, ~~($get($1.binval, $1.i) / 4537077)); //#6764
        } //#6764
        $1.right = $get($1.binval, 13) % 4537077; //#6766
        var _2c = $1.binval; //#6767
        $put(_2c, 13, ~~($get(_2c, 13) / 4537077)); //#6767
        $1.left = 0; //#6769
        $1.i = true; //#6770
        for (var _2e = 0; _2e <= 13; _2e += 1) { //#6780
            $1.j = _2e; //#6772
            var _2h = $get($1.binval, $1.j); //#6773
            $k[$j++] = _2h; //#6779
            if ((_2h == 0) && $1.i) { //#6778
                $j--; //#6775
            } else { //#6778
                $1.i = false; //#6777
                $1.left = $f($1.left + ($k[--$j] * (~~Math.pow(10, 13 - $1.j)))); //#6778
            } //#6778
        } //#6778
        $1.d1 = ~~($1.left / 1597); //#6782
        $1.d2 = $1.left % 1597; //#6783
        $1.d3 = ~~($1.right / 1597); //#6784
        $1.d4 = $1.right % 1597; //#6785
        $1.tab164 = $a([160, 0, 12, 4, 8, 1, 161, 1, 960, 161, 10, 6, 6, 3, 80, 10, 2014, 961, 8, 8, 4, 5, 31, 34, 2714, 2015, 6, 10, 3, 6, 10, 70, 2840, 2715, 4, 12, 1, 8, 1, 126]); //#6793
        $1.tab154 = $a([335, 0, 5, 10, 2, 7, 4, 84, 1035, 336, 7, 8, 4, 5, 20, 35, 1515, 1036, 9, 6, 6, 3, 48, 10, 1596, 1516, 11, 4, 8, 1, 81, 1]); //#6800
        $1.i = 0; //#6802
        for (;;) { //#6812
            if ($1.d1 <= $get($1.tab164, $1.i)) { //#6810
                var _2y = $geti($1.tab164, $1.i + 1, 7); //#6804
                for (var _2z = 0, _30 = _2y.length; _2z < _30; _2z++) { //#6804
                    $k[$j++] = $get(_2y, _2z); //#6804
                } //#6804
                $1.d1te = $k[--$j]; //#6805
                $1.d1to = $k[--$j]; //#6805
                $1.d1mwe = $k[--$j]; //#6806
                $1.d1mwo = $k[--$j]; //#6806
                $1.d1ele = $k[--$j]; //#6807
                $1.d1elo = $k[--$j]; //#6807
                $1.d1gs = $k[--$j]; //#6808
                break; //#6809
            } //#6809
            $1.i = $1.i + 8; //#6811
        } //#6811
        $1.i = 0; //#6814
        for (;;) { //#6824
            if ($1.d2 <= $get($1.tab154, $1.i)) { //#6822
                var _3G = $geti($1.tab154, $1.i + 1, 7); //#6816
                for (var _3H = 0, _3I = _3G.length; _3H < _3I; _3H++) { //#6816
                    $k[$j++] = $get(_3G, _3H); //#6816
                } //#6816
                $1.d2te = $k[--$j]; //#6817
                $1.d2to = $k[--$j]; //#6817
                $1.d2mwe = $k[--$j]; //#6818
                $1.d2mwo = $k[--$j]; //#6818
                $1.d2ele = $k[--$j]; //#6819
                $1.d2elo = $k[--$j]; //#6819
                $1.d2gs = $k[--$j]; //#6820
                break; //#6821
            } //#6821
            $1.i = $1.i + 8; //#6823
        } //#6823
        $1.i = 0; //#6826
        for (;;) { //#6836
            if ($1.d3 <= $get($1.tab164, $1.i)) { //#6834
                var _3Y = $geti($1.tab164, $1.i + 1, 7); //#6828
                for (var _3Z = 0, _3a = _3Y.length; _3Z < _3a; _3Z++) { //#6828
                    $k[$j++] = $get(_3Y, _3Z); //#6828
                } //#6828
                $1.d3te = $k[--$j]; //#6829
                $1.d3to = $k[--$j]; //#6829
                $1.d3mwe = $k[--$j]; //#6830
                $1.d3mwo = $k[--$j]; //#6830
                $1.d3ele = $k[--$j]; //#6831
                $1.d3elo = $k[--$j]; //#6831
                $1.d3gs = $k[--$j]; //#6832
                break; //#6833
            } //#6833
            $1.i = $1.i + 8; //#6835
        } //#6835
        $1.i = 0; //#6838
        for (;;) { //#6848
            if ($1.d4 <= $get($1.tab154, $1.i)) { //#6846
                var _3q = $geti($1.tab154, $1.i + 1, 7); //#6840
                for (var _3r = 0, _3s = _3q.length; _3r < _3s; _3r++) { //#6840
                    $k[$j++] = $get(_3q, _3r); //#6840
                } //#6840
                $1.d4te = $k[--$j]; //#6841
                $1.d4to = $k[--$j]; //#6841
                $1.d4mwe = $k[--$j]; //#6842
                $1.d4mwo = $k[--$j]; //#6842
                $1.d4ele = $k[--$j]; //#6843
                $1.d4elo = $k[--$j]; //#6843
                $1.d4gs = $k[--$j]; //#6844
                break; //#6845
            } //#6845
            $1.i = $1.i + 8; //#6847
        } //#6847
        $k[$j++] = 'd1wo'; //#6850
        $k[$j++] = ~~($f($1.d1 - $1.d1gs) / $1.d1te); //#6850
        $k[$j++] = $1.d1elo; //#6850
        $k[$j++] = $1.d1mwo; //#6850
        $k[$j++] = 4; //#6850
        $k[$j++] = false; //#6850
        $1.getRSSwidths(); //#6850
        var _47 = $k[--$j]; //#6850
        $1[$k[--$j]] = _47; //#6850
        $k[$j++] = 'd1we'; //#6851
        $k[$j++] = $f($1.d1 - $1.d1gs) % $1.d1te; //#6851
        $k[$j++] = $1.d1ele; //#6851
        $k[$j++] = $1.d1mwe; //#6851
        $k[$j++] = 4; //#6851
        $k[$j++] = true; //#6851
        $1.getRSSwidths(); //#6851
        var _4E = $k[--$j]; //#6851
        $1[$k[--$j]] = _4E; //#6851
        $k[$j++] = 'd2wo'; //#6852
        $k[$j++] = $f($1.d2 - $1.d2gs) % $1.d2to; //#6852
        $k[$j++] = $1.d2elo; //#6852
        $k[$j++] = $1.d2mwo; //#6852
        $k[$j++] = 4; //#6852
        $k[$j++] = true; //#6852
        $1.getRSSwidths(); //#6852
        var _4L = $k[--$j]; //#6852
        $1[$k[--$j]] = _4L; //#6852
        $k[$j++] = 'd2we'; //#6853
        $k[$j++] = ~~($f($1.d2 - $1.d2gs) / $1.d2to); //#6853
        $k[$j++] = $1.d2ele; //#6853
        $k[$j++] = $1.d2mwe; //#6853
        $k[$j++] = 4; //#6853
        $k[$j++] = false; //#6853
        $1.getRSSwidths(); //#6853
        var _4S = $k[--$j]; //#6853
        $1[$k[--$j]] = _4S; //#6853
        $k[$j++] = 'd3wo'; //#6854
        $k[$j++] = ~~($f($1.d3 - $1.d3gs) / $1.d3te); //#6854
        $k[$j++] = $1.d3elo; //#6854
        $k[$j++] = $1.d3mwo; //#6854
        $k[$j++] = 4; //#6854
        $k[$j++] = false; //#6854
        $1.getRSSwidths(); //#6854
        var _4Z = $k[--$j]; //#6854
        $1[$k[--$j]] = _4Z; //#6854
        $k[$j++] = 'd3we'; //#6855
        $k[$j++] = $f($1.d3 - $1.d3gs) % $1.d3te; //#6855
        $k[$j++] = $1.d3ele; //#6855
        $k[$j++] = $1.d3mwe; //#6855
        $k[$j++] = 4; //#6855
        $k[$j++] = true; //#6855
        $1.getRSSwidths(); //#6855
        var _4g = $k[--$j]; //#6855
        $1[$k[--$j]] = _4g; //#6855
        $k[$j++] = 'd4wo'; //#6856
        $k[$j++] = $f($1.d4 - $1.d4gs) % $1.d4to; //#6856
        $k[$j++] = $1.d4elo; //#6856
        $k[$j++] = $1.d4mwo; //#6856
        $k[$j++] = 4; //#6856
        $k[$j++] = true; //#6856
        $1.getRSSwidths(); //#6856
        var _4n = $k[--$j]; //#6856
        $1[$k[--$j]] = _4n; //#6856
        $k[$j++] = 'd4we'; //#6857
        $k[$j++] = ~~($f($1.d4 - $1.d4gs) / $1.d4to); //#6857
        $k[$j++] = $1.d4ele; //#6857
        $k[$j++] = $1.d4mwe; //#6857
        $k[$j++] = 4; //#6857
        $k[$j++] = false; //#6857
        $1.getRSSwidths(); //#6857
        var _4u = $k[--$j]; //#6857
        $1[$k[--$j]] = _4u; //#6857
        $1.d1w = $a(8); //#6859
        for (var _4x = 0; _4x <= 3; _4x += 1) { //#6864
            $1.i = _4x; //#6861
            $put($1.d1w, $1.i * 2, $get($1.d1wo, $1.i)); //#6862
            $put($1.d1w, ($1.i * 2) + 1, $get($1.d1we, $1.i)); //#6863
        } //#6863
        $1.d2w = $a(8); //#6866
        for (var _59 = 0; _59 <= 3; _59 += 1) { //#6871
            $1.i = _59; //#6868
            $put($1.d2w, 7 - ($1.i * 2), $get($1.d2wo, $1.i)); //#6869
            $put($1.d2w, 6 - ($1.i * 2), $get($1.d2we, $1.i)); //#6870
        } //#6870
        $1.d3w = $a(8); //#6873
        for (var _5L = 0; _5L <= 3; _5L += 1) { //#6878
            $1.i = _5L; //#6875
            $put($1.d3w, 7 - ($1.i * 2), $get($1.d3wo, $1.i)); //#6876
            $put($1.d3w, 6 - ($1.i * 2), $get($1.d3we, $1.i)); //#6877
        } //#6877
        $1.d4w = $a(8); //#6880
        for (var _5X = 0; _5X <= 3; _5X += 1) { //#6885
            $1.i = _5X; //#6882
            $put($1.d4w, $1.i * 2, $get($1.d4wo, $1.i)); //#6883
            $put($1.d4w, ($1.i * 2) + 1, $get($1.d4we, $1.i)); //#6884
        } //#6884
        $k[$j++] = Infinity; //#6891
        var _5i = $1.d1w; //#6888
        for (var _5j = 0, _5k = _5i.length; _5j < _5k; _5j++) { //#6888
            $k[$j++] = $get(_5i, _5j); //#6888
        } //#6888
        var _5m = $1.d2w; //#6889
        for (var _5n = 0, _5o = _5m.length; _5n < _5o; _5n++) { //#6889
            $k[$j++] = $get(_5m, _5n); //#6889
        } //#6889
        var _5q = $1.d3w; //#6890
        for (var _5r = 0, _5s = _5q.length; _5r < _5s; _5r++) { //#6890
            $k[$j++] = $get(_5q, _5r); //#6890
        } //#6890
        var _5u = $1.d4w; //#6891
        for (var _5v = 0, _5w = _5u.length; _5v < _5w; _5v++) { //#6891
            $k[$j++] = $get(_5u, _5v); //#6891
        } //#6891
        $1.widths = $a(); //#6891
        $1.checkweights = $a([1, 3, 9, 27, 2, 6, 18, 54, 58, 72, 24, 8, 29, 36, 12, 4, 74, 51, 17, 32, 37, 65, 48, 16, 64, 34, 23, 69, 49, 68, 46, 59]); //#6899
        $1.checkwidths = $a([3, 8, 2, 1, 1, 3, 5, 5, 1, 1, 3, 3, 7, 1, 1, 3, 1, 9, 1, 1, 2, 7, 4, 1, 1, 2, 5, 6, 1, 1, 2, 3, 8, 1, 1, 1, 5, 7, 1, 1, 1, 3, 9, 1, 1]); //#6905
        $1.checksum = 0; //#6907
        for (var _61 = 0; _61 <= 31; _61 += 1) { //#6911
            $1.i = _61; //#6909
            $1.checksum = $f($1.checksum + ($get($1.widths, $1.i) * $get($1.checkweights, $1.i))); //#6910
        } //#6910
        $1.checksum = $1.checksum % 79; //#6912
        if ($1.checksum >= 8) { //#6913
            $1.checksum = $f($1.checksum + 1); //#6913
        } //#6913
        if ($1.checksum >= 72) { //#6914
            $1.checksum = $f($1.checksum + 1); //#6914
        } //#6914
        $1.checklt = $geti($1.checkwidths, (~~($1.checksum / 9)) * 5, 5); //#6915
        $1.checkrtrev = $geti($1.checkwidths, ($1.checksum % 9) * 5, 5); //#6916
        $1.checkrt = $a(5); //#6917
        for (var _6L = 0; _6L <= 4; _6L += 1) { //#6921
            $1.i = _6L; //#6919
            $put($1.checkrt, $1.i, $get($1.checkrtrev, 4 - $1.i)); //#6920
        } //#6920
        if ($eq($1.format, "omni") || $eq($1.format, "truncated")) { //#7043
            $k[$j++] = Infinity; //#6928
            var _6T = $1.d1w; //#6927
            $k[$j++] = 1; //#6927
            for (var _6U = 0, _6V = _6T.length; _6U < _6V; _6U++) { //#6927
                $k[$j++] = $get(_6T, _6U); //#6927
            } //#6927
            var _6X = $1.checklt; //#6927
            for (var _6Y = 0, _6Z = _6X.length; _6Y < _6Z; _6Y++) { //#6927
                $k[$j++] = $get(_6X, _6Y); //#6927
            } //#6927
            var _6b = $1.d2w; //#6927
            for (var _6c = 0, _6d = _6b.length; _6c < _6d; _6c++) { //#6927
                $k[$j++] = $get(_6b, _6c); //#6927
            } //#6927
            var _6f = $1.d4w; //#6928
            for (var _6g = 0, _6h = _6f.length; _6g < _6h; _6g++) { //#6928
                $k[$j++] = $get(_6f, _6g); //#6928
            } //#6928
            var _6j = $1.checkrt; //#6928
            for (var _6k = 0, _6l = _6j.length; _6k < _6l; _6k++) { //#6928
                $k[$j++] = $get(_6j, _6k); //#6928
            } //#6928
            var _6n = $1.d3w; //#6928
            for (var _6o = 0, _6p = _6n.length; _6o < _6p; _6o++) { //#6928
                $k[$j++] = $get(_6n, _6o); //#6928
            } //#6928
            $k[$j++] = 1; //#6928
            $k[$j++] = 1; //#6928
            $1.sbs = $a(); //#6928
            $k[$j++] = Infinity; //#6938
            var _6s = $1.sbs; //#6933
            $k[$j++] = Infinity; //#6934
            for (var _6u = 0, _6v = ~~(($1.sbs.length + 1) / 2); _6u < _6v; _6u++) { //#6934
                $k[$j++] = $1.height; //#6934
            } //#6934
            var _6x = $a(); //#6934
            $k[$j++] = Infinity; //#6935
            for (var _6z = 0, _70 = ~~(($1.sbs.length + 1) / 2); _6z < _70; _6z++) { //#6935
                $k[$j++] = 0; //#6935
            } //#6935
            var _71 = $a(); //#6935
            $k[$j++] = 'ren'; //#6938
            $k[$j++] = $0.renlinear; //#6938
            $k[$j++] = 'sbs'; //#6938
            $k[$j++] = _6s; //#6938
            $k[$j++] = 'bhs'; //#6938
            $k[$j++] = _6x; //#6938
            $k[$j++] = 'bbs'; //#6938
            $k[$j++] = _71; //#6938
            $k[$j++] = 'txt'; //#6938
            $k[$j++] = $1.txt; //#6938
            $k[$j++] = 'textxalign'; //#6938
            $k[$j++] = "center"; //#6938
            $k[$j++] = 'opt'; //#6938
            $k[$j++] = $1.options; //#6938
            var _74 = $d(); //#6938
            $k[$j++] = _74; //#6941
            if (!$1.dontdraw) { //#6941
                $0.renlinear(); //#6941
            } //#6941
        } else { //#7043
            $k[$j++] = Infinity; //#6945
            var _76 = $1.d1w; //#6945
            $k[$j++] = 1; //#6945
            $k[$j++] = 1; //#6945
            for (var _77 = 0, _78 = _76.length; _77 < _78; _77++) { //#6945
                $k[$j++] = $get(_76, _77); //#6945
            } //#6945
            var _7A = $1.checklt; //#6945
            for (var _7B = 0, _7C = _7A.length; _7B < _7C; _7B++) { //#6945
                $k[$j++] = $get(_7A, _7B); //#6945
            } //#6945
            var _7E = $1.d2w; //#6945
            for (var _7F = 0, _7G = _7E.length; _7F < _7G; _7F++) { //#6945
                $k[$j++] = $get(_7E, _7F); //#6945
            } //#6945
            $k[$j++] = 1; //#6945
            $k[$j++] = 1; //#6945
            $k[$j++] = 0; //#6945
            $1.top = $a(); //#6945
            $k[$j++] = Infinity; //#6946
            var _7J = $1.d4w; //#6946
            $k[$j++] = 1; //#6946
            $k[$j++] = 1; //#6946
            for (var _7K = 0, _7L = _7J.length; _7K < _7L; _7K++) { //#6946
                $k[$j++] = $get(_7J, _7K); //#6946
            } //#6946
            var _7N = $1.checkrt; //#6946
            for (var _7O = 0, _7P = _7N.length; _7O < _7P; _7O++) { //#6946
                $k[$j++] = $get(_7N, _7O); //#6946
            } //#6946
            var _7R = $1.d3w; //#6946
            for (var _7S = 0, _7T = _7R.length; _7S < _7T; _7S++) { //#6946
                $k[$j++] = $get(_7R, _7S); //#6946
            } //#6946
            $k[$j++] = 1; //#6946
            $k[$j++] = 1; //#6946
            $k[$j++] = 0; //#6946
            $1.bot = $a(); //#6946
            for (var _7W = 0; _7W <= 24; _7W += 2) { //#6951
                $1.i = _7W; //#6948
                for (var _7a = 0, _7b = $get($1.top, $1.i); _7a < _7b; _7a++) { //#6949
                    $k[$j++] = 0; //#6949
                } //#6949
                for (var _7f = 0, _7g = $get($1.top, $1.i + 1); _7f < _7g; _7f++) { //#6950
                    $k[$j++] = 1; //#6950
                } //#6950
            } //#6950
            $astore($a(50)); //#6952
            $1.top = $k[--$j]; //#6952
            for (var _7j = 0; _7j <= 24; _7j += 2) { //#6957
                $1.i = _7j; //#6954
                for (var _7n = 0, _7o = $get($1.bot, $1.i); _7n < _7o; _7n++) { //#6955
                    $k[$j++] = 1; //#6955
                } //#6955
                for (var _7s = 0, _7t = $get($1.bot, $1.i + 1); _7s < _7t; _7s++) { //#6956
                    $k[$j++] = 0; //#6956
                } //#6956
            } //#6956
            $astore($a(50)); //#6958
            $1.bot = $k[--$j]; //#6958
            if ($eq($1.format, "stacked")) { //#6980
                $1.sep = $a(50); //#6962
                $put($1.sep, 0, 0); //#6963
                for (var _7z = 1; _7z <= 49; _7z += 1) { //#6971
                    $1.i = _7z; //#6965
                    if ($eq($get($1.top, $1.i), $get($1.bot, $1.i))) { //#6969
                        $put($1.sep, $1.i, $f(1 - $get($1.top, $1.i))); //#6967
                    } else { //#6969
                        $put($1.sep, $1.i, $f(1 - $get($1.sep, $1.i - 1))); //#6969
                    } //#6969
                } //#6969
                $puti($1.sep, 0, $a([0, 0, 0, 0])); //#6972
                $puti($1.sep, 46, $a([0, 0, 0, 0])); //#6973
                $k[$j++] = Infinity; //#6977
                for (var _8K = 0, _8L = 5; _8K < _8L; _8K++) { //#6975
                    $aload($1.top); //#6975
                } //#6975
                $aload($1.sep); //#6976
                for (var _8O = 0, _8P = 7; _8O < _8P; _8O++) { //#6977
                    $aload($1.bot); //#6977
                } //#6977
                $1.pixs = $a(); //#6977
                $1.pixy = ~~($1.pixs.length / 50); //#6979
            } //#6979
            if ($eq($1.format, "stackedomni")) { //#7030
                $k[$j++] = Infinity; //#6984
                $forall($1.top, function() { //#6984
                    var _8V = $k[--$j]; //#6984
                    $k[$j++] = $f(1 - _8V); //#6984
                }); //#6984
                $1.sep1 = $a(); //#6984
                $puti($1.sep1, 0, $a([0, 0, 0, 0])); //#6985
                $puti($1.sep1, 46, $a([0, 0, 0, 0])); //#6986
                for (var _8b = 18; _8b <= 30; _8b += 1) { //#6999
                    $1.i = _8b; //#6988
                    if ($get($1.top, $1.i) == 0) { //#6996
                        if ($get($1.top, $1.i - 1) == 1) { //#6993
                            $k[$j++] = 1; //#6991
                        } else { //#6993
                            var _8l = ($get($1.sep1, $1.i - 1) == 0) ? 1 : 0; //#6993
                            $k[$j++] = _8l; //#6993
                        } //#6993
                    } else { //#6996
                        $k[$j++] = 0; //#6996
                    } //#6996
                    $put($1.sep1, $1.i, $k[--$j]); //#6998
                } //#6998
                $k[$j++] = Infinity; //#7000
                $k[$j++] = 0; //#7000
                $k[$j++] = 0; //#7000
                $k[$j++] = 0; //#7000
                $k[$j++] = 0; //#7000
                for (var _8p = 0, _8q = 21; _8p < _8q; _8p++) { //#7000
                    $k[$j++] = 0; //#7000
                    $k[$j++] = 1; //#7000
                } //#7000
                $k[$j++] = 0; //#7000
                $k[$j++] = 0; //#7000
                $k[$j++] = 0; //#7000
                $k[$j++] = 0; //#7000
                $1.sep2 = $a(); //#7000
                $k[$j++] = Infinity; //#7001
                $forall($1.bot, function() { //#7001
                    var _8t = $k[--$j]; //#7001
                    $k[$j++] = $f(1 - _8t); //#7001
                }); //#7001
                $1.sep3 = $a(); //#7001
                $puti($1.sep3, 0, $a([0, 0, 0, 0])); //#7002
                $puti($1.sep3, 46, $a([0, 0, 0, 0])); //#7003
                for (var _8z = 19; _8z <= 31; _8z += 1) { //#7016
                    $1.i = _8z; //#7005
                    if ($get($1.bot, $1.i) == 0) { //#7013
                        if ($get($1.bot, $1.i - 1) == 1) { //#7010
                            $k[$j++] = 1; //#7008
                        } else { //#7010
                            var _99 = ($get($1.sep3, $1.i - 1) == 0) ? 1 : 0; //#7010
                            $k[$j++] = _99; //#7010
                        } //#7010
                    } else { //#7013
                        $k[$j++] = 0; //#7013
                    } //#7013
                    $put($1.sep3, $1.i, $k[--$j]); //#7015
                } //#7015
                $1.f3 = $a([1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1]); //#7017
                $k[$j++] = true; //#7018
                for (var _9E = 0; _9E <= 12; _9E += 1) { //#7018
                    var _9J = $k[--$j]; //#7018
                    $k[$j++] = _9J && $eq($get($1.bot, _9E + 19), $get($1.f3, _9E)); //#7018
                } //#7018
                if ($k[--$j]) { //#7021
                    $puti($1.sep3, 19, $a([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0])); //#7020
                } //#7020
                $k[$j++] = Infinity; //#7027
                for (var _9O = 0, _9P = $1.barxmult; _9O < _9P; _9O++) { //#7023
                    $aload($1.top); //#7023
                } //#7023
                $aload($1.sep1); //#7024
                $aload($1.sep2); //#7025
                $aload($1.sep3); //#7026
                for (var _9V = 0, _9W = $1.barxmult; _9V < _9W; _9V++) { //#7027
                    $aload($1.bot); //#7027
                } //#7027
                $1.pixs = $a(); //#7027
                $1.pixy = ~~($1.pixs.length / 50); //#7029
            } //#7029
            var _9e = new Map([
                ["ren", $0.renmatrix],
                ["pixs", $1.pixs],
                ["pixx", 50],
                ["pixy", $1.pixy],
                ["height", $1.pixy / 72],
                ["width", 50 / 72],
                ["opt", $1.options]
            ]); //#7040
            $k[$j++] = _9e; //#7043
            if (!$1.dontdraw) { //#7043
                $0.renmatrix(); //#7043
            } //#7043
        } //#7043
    };
    $0.databarstacked = function() {
        var $1 = {}; //#7075
        $1.options = $k[--$j]; //#7077
        $1.barcode = $k[--$j]; //#7078
        $1.dontdraw = false; //#7080
        $forall($1.options, function() { //#7091
            var _3 = $k[--$j]; //#7091
            $1[$k[--$j]] = _3; //#7091
        }); //#7091
        if ($ne($geti($1.barcode, 0, 4), "(01)")) { //#7096
            $k[$j++] = 'bwipp.databarstackedBadAI'; //#7095
            $k[$j++] = "GS1 DataBar Stacked must begin with (01) application identifier"; //#7095
            $0.raiseerror(); //#7095
        } //#7095
        if (($1.barcode.length != 17) && ($1.barcode.length != 18)) { //#7099
            $k[$j++] = 'bwipp.databarstackedBadLength'; //#7098
            $k[$j++] = "GS1 DataBar Stacked must be 13 or 14 digits"; //#7098
            $0.raiseerror(); //#7098
        } //#7098
        var _B = $geti($1.barcode, 4, $1.barcode.length - 4); //#7100
        for (var _C = 0, _D = _B.length; _C < _D; _C++) { //#7104
            var _E = $get(_B, _C); //#7104
            if ((_E < 48) || (_E > 57)) { //#7103
                $k[$j++] = 'bwipp.databarstackedBadCharacter'; //#7102
                $k[$j++] = "GS1 DataBar Stacked must contain only digits"; //#7102
                $0.raiseerror(); //#7102
            } //#7102
        } //#7102
        $1.checksum = 0; //#7107
        for (var _F = 0; _F <= 12; _F += 1) { //#7111
            $1.i = _F; //#7109
            $k[$j++] = 'checksum'; //#7110
            $k[$j++] = $1.checksum; //#7110
            $k[$j++] = $f($get($1.barcode, $1.i + 4) - 48); //#7110
            if (($1.i % 2) == 0) { //#7110
                var _L = $k[--$j]; //#7110
                $k[$j++] = _L * 3; //#7110
            } //#7110
            var _M = $k[--$j]; //#7110
            var _N = $k[--$j]; //#7110
            $1[$k[--$j]] = $f(_N + _M); //#7110
        } //#7110
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#7112
        if ($1.barcode.length == 18) { //#7117
            if ($get($1.barcode, 17) != ($1.checksum + 48)) { //#7116
                $k[$j++] = 'bwipp.databarstackedBadCheckDigit'; //#7115
                $k[$j++] = "Incorrect GS1 DataBar Stacked check digit provided"; //#7115
                $0.raiseerror(); //#7115
            } //#7115
        } //#7115
        var _U = $s(18); //#7118
        $puti(_U, 0, $1.barcode); //#7119
        $put(_U, 17, $1.checksum + 48); //#7120
        $1.barcode = _U; //#7121
        $put($1.options, "dontdraw", true); //#7124
        $put($1.options, "format", "stacked"); //#7125
        $k[$j++] = 'args'; //#7127
        $k[$j++] = $1.barcode; //#7127
        $k[$j++] = $1.options; //#7127
        $0.databaromni(); //#7127
        var _b = $k[--$j]; //#7127
        $1[$k[--$j]] = _b; //#7127
        $put($1.args, "opt", $1.options); //#7129
        $k[$j++] = $1.args; //#7132
        if (!$1.dontdraw) { //#7132
            $0.renmatrix(); //#7132
        } //#7132
    };
    $0.databarstackedomni = function() {
        var $1 = {}; //#7162
        $1.options = $k[--$j]; //#7164
        $1.barcode = $k[--$j]; //#7165
        $1.dontdraw = false; //#7167
        $forall($1.options, function() { //#7178
            var _3 = $k[--$j]; //#7178
            $1[$k[--$j]] = _3; //#7178
        }); //#7178
        if ($ne($geti($1.barcode, 0, 4), "(01)")) { //#7183
            $k[$j++] = 'bwipp.databarstackedomniBadAI'; //#7182
            $k[$j++] = "GS1 DataBar Stacked Omnidirectional must begin with (01) application identifier"; //#7182
            $0.raiseerror(); //#7182
        } //#7182
        if (($1.barcode.length != 17) && ($1.barcode.length != 18)) { //#7186
            $k[$j++] = 'bwipp.databarstackedomniBadLength'; //#7185
            $k[$j++] = "GS1 DataBar Stacked Omnidirectional must be 13 or 14 digits"; //#7185
            $0.raiseerror(); //#7185
        } //#7185
        var _B = $geti($1.barcode, 4, $1.barcode.length - 4); //#7187
        for (var _C = 0, _D = _B.length; _C < _D; _C++) { //#7191
            var _E = $get(_B, _C); //#7191
            if ((_E < 48) || (_E > 57)) { //#7190
                $k[$j++] = 'bwipp.databarstackedomniBadCharacter'; //#7189
                $k[$j++] = "GS1 DataBar Stacked Omnidirectional must contain only digits"; //#7189
                $0.raiseerror(); //#7189
            } //#7189
        } //#7189
        $1.checksum = 0; //#7194
        for (var _F = 0; _F <= 12; _F += 1) { //#7198
            $1.i = _F; //#7196
            $k[$j++] = 'checksum'; //#7197
            $k[$j++] = $1.checksum; //#7197
            $k[$j++] = $f($get($1.barcode, $1.i + 4) - 48); //#7197
            if (($1.i % 2) == 0) { //#7197
                var _L = $k[--$j]; //#7197
                $k[$j++] = _L * 3; //#7197
            } //#7197
            var _M = $k[--$j]; //#7197
            var _N = $k[--$j]; //#7197
            $1[$k[--$j]] = $f(_N + _M); //#7197
        } //#7197
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#7199
        if ($1.barcode.length == 18) { //#7204
            if ($get($1.barcode, 17) != ($1.checksum + 48)) { //#7203
                $k[$j++] = 'bwipp.databarstackedomniBadCheckDigit'; //#7202
                $k[$j++] = "Incorrect GS1 DataBar Stacked Omnidirectional check digit provided"; //#7202
                $0.raiseerror(); //#7202
            } //#7202
        } //#7202
        var _U = $s(18); //#7205
        $puti(_U, 0, $1.barcode); //#7206
        $put(_U, 17, $1.checksum + 48); //#7207
        $1.barcode = _U; //#7208
        $put($1.options, "dontdraw", true); //#7211
        $put($1.options, "format", "stackedomni"); //#7212
        $k[$j++] = 'args'; //#7214
        $k[$j++] = $1.barcode; //#7214
        $k[$j++] = $1.options; //#7214
        $0.databaromni(); //#7214
        var _b = $k[--$j]; //#7214
        $1[$k[--$j]] = _b; //#7214
        $put($1.args, "opt", $1.options); //#7216
        $k[$j++] = $1.args; //#7219
        if (!$1.dontdraw) { //#7219
            $0.renmatrix(); //#7219
        } //#7219
    };
    $0.databartruncated = function() {
        var $1 = {}; //#7249
        $1.options = $k[--$j]; //#7251
        $1.barcode = $k[--$j]; //#7252
        $1.dontdraw = false; //#7254
        $forall($1.options, function() { //#7265
            var _3 = $k[--$j]; //#7265
            $1[$k[--$j]] = _3; //#7265
        }); //#7265
        if ($ne($geti($1.barcode, 0, 4), "(01)")) { //#7270
            $k[$j++] = 'bwipp.databartruncatedBadAI'; //#7269
            $k[$j++] = "GS1 DataBar Truncated must begin with (01) application identifier"; //#7269
            $0.raiseerror(); //#7269
        } //#7269
        if (($1.barcode.length != 17) && ($1.barcode.length != 18)) { //#7273
            $k[$j++] = 'bwipp.databartruncatedBadLength'; //#7272
            $k[$j++] = "GS1 DataBar Truncated must be 13 or 14 digits"; //#7272
            $0.raiseerror(); //#7272
        } //#7272
        var _B = $geti($1.barcode, 4, $1.barcode.length - 4); //#7274
        for (var _C = 0, _D = _B.length; _C < _D; _C++) { //#7278
            var _E = $get(_B, _C); //#7278
            if ((_E < 48) || (_E > 57)) { //#7277
                $k[$j++] = 'bwipp.databartruncatedBadCharacter'; //#7276
                $k[$j++] = "GS1 DataBar Truncated must contain only digits"; //#7276
                $0.raiseerror(); //#7276
            } //#7276
        } //#7276
        $1.checksum = 0; //#7281
        for (var _F = 0; _F <= 12; _F += 1) { //#7285
            $1.i = _F; //#7283
            $k[$j++] = 'checksum'; //#7284
            $k[$j++] = $1.checksum; //#7284
            $k[$j++] = $f($get($1.barcode, $1.i + 4) - 48); //#7284
            if (($1.i % 2) == 0) { //#7284
                var _L = $k[--$j]; //#7284
                $k[$j++] = _L * 3; //#7284
            } //#7284
            var _M = $k[--$j]; //#7284
            var _N = $k[--$j]; //#7284
            $1[$k[--$j]] = $f(_N + _M); //#7284
        } //#7284
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#7286
        if ($1.barcode.length == 18) { //#7291
            if ($get($1.barcode, 17) != ($1.checksum + 48)) { //#7290
                $k[$j++] = 'bwipp.databartruncatedBadCheckDigit'; //#7289
                $k[$j++] = "Incorrect GS1 DataBar Truncated check digit provided"; //#7289
                $0.raiseerror(); //#7289
            } //#7289
        } //#7289
        var _U = $s(18); //#7292
        $puti(_U, 0, $1.barcode); //#7293
        $put(_U, 17, $1.checksum + 48); //#7294
        $1.barcode = _U; //#7295
        $put($1.options, "dontdraw", true); //#7298
        $put($1.options, "format", "truncated"); //#7299
        $k[$j++] = 'args'; //#7301
        $k[$j++] = $1.barcode; //#7301
        $k[$j++] = $1.options; //#7301
        $0.databaromni(); //#7301
        var _b = $k[--$j]; //#7301
        $1[$k[--$j]] = _b; //#7301
        $put($1.args, "opt", $1.options); //#7303
        $k[$j++] = $1.args; //#7306
        if (!$1.dontdraw) { //#7306
            $0.renlinear(); //#7306
        } //#7306
    };
    $0.databarlimited = function() {
        var $1 = {}; //#7335
        $1.options = $k[--$j]; //#7337
        $1.barcode = $k[--$j]; //#7338
        $1.dontdraw = false; //#7340
        $1.height = 10 / 72; //#7341
        $1.linkage = false; //#7342
        $forall($1.options, function() { //#7353
            var _3 = $k[--$j]; //#7353
            $1[$k[--$j]] = _3; //#7353
        }); //#7353
        $1.height = +$1.height; //#7355
        if ($ne($geti($1.barcode, 0, 4), "(01)")) { //#7360
            $k[$j++] = 'bwipp.databarlimitedBadAI'; //#7359
            $k[$j++] = "GS1 DataBar Limited must begin with (01) application identifier"; //#7359
            $0.raiseerror(); //#7359
        } //#7359
        if (($1.barcode.length != 17) && ($1.barcode.length != 18)) { //#7363
            $k[$j++] = 'bwipp.databarlimitedBadLength'; //#7362
            $k[$j++] = "GS1 DataBar Limited must be 13 or 14 digits"; //#7362
            $0.raiseerror(); //#7362
        } //#7362
        var _B = $get($1.barcode, 4); //#7364
        if ((_B < 48) || (_B > 49)) { //#7366
            $k[$j++] = 'bwipp.databarlimitedBadStartDigit'; //#7365
            $k[$j++] = "GS1 DataBar Limited must begin with 0 or 1"; //#7365
            $0.raiseerror(); //#7365
        } //#7365
        var _E = $geti($1.barcode, 5, $1.barcode.length - 5); //#7367
        for (var _F = 0, _G = _E.length; _F < _G; _F++) { //#7371
            var _H = $get(_E, _F); //#7371
            if ((_H < 48) || (_H > 57)) { //#7370
                $k[$j++] = 'bwipp.databarlimitedBadCharacter'; //#7369
                $k[$j++] = "GS1 DataBar Limited must contain only digits"; //#7369
                $0.raiseerror(); //#7369
            } //#7369
        } //#7369
        $1.checksum = 0; //#7374
        for (var _I = 0; _I <= 12; _I += 1) { //#7378
            $1.i = _I; //#7376
            $k[$j++] = 'checksum'; //#7377
            $k[$j++] = $1.checksum; //#7377
            $k[$j++] = $f($get($1.barcode, $1.i + 4) - 48); //#7377
            if (($1.i % 2) == 0) { //#7377
                var _O = $k[--$j]; //#7377
                $k[$j++] = _O * 3; //#7377
            } //#7377
            var _P = $k[--$j]; //#7377
            var _Q = $k[--$j]; //#7377
            $1[$k[--$j]] = $f(_Q + _P); //#7377
        } //#7377
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#7379
        if ($1.barcode.length == 18) { //#7384
            if ($get($1.barcode, 17) != ($1.checksum + 48)) { //#7383
                $k[$j++] = 'bwipp.databarlimitedBadCheckDigit'; //#7382
                $k[$j++] = "Incorrect GS1 DataBar Limited check digit provided"; //#7382
                $0.raiseerror(); //#7382
            } //#7382
        } //#7382
        var _X = $s(18); //#7385
        $puti(_X, 0, $1.barcode); //#7386
        $put(_X, 17, $1.checksum + 48); //#7387
        $1.barcode = _X; //#7388
        $k[$j++] = Infinity; //#7392
        var _b = $geti($1.barcode, 4, 13); //#7392
        for (var _c = 0, _d = _b.length; _c < _d; _c++) { //#7392
            $k[$j++] = $f($get(_b, _c) - 48); //#7392
        } //#7392
        $1.binval = $a(); //#7392
        if ($1.linkage) { //#7400
            $1.linkval = $a([2, 0, 1, 5, 1, 3, 3, 5, 3, 1, 0, 9, 6]); //#7395
            for (var _k = 0, _j = $1.binval.length - 1; _k <= _j; _k += 1) { //#7399
                $1.i = _k; //#7397
                $put($1.binval, $1.i, $f($get($1.binval, $1.i) + $get($1.linkval, $1.i))); //#7398
            } //#7398
        } //#7398
        $1.txt = $a($1.barcode.length); //#7403
        for (var _x = 0, _w = $1.barcode.length - 1; _x <= _w; _x += 1) { //#7407
            $1.i = _x; //#7405
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), 0, 0, "", 0])); //#7406
        } //#7406
        $1.ncr = function() {
            var _14 = $k[--$j]; //#7410
            var _15 = $k[--$j]; //#7410
            var _16 = $f(_15 - _14); //#7410
            if (_14 < $f(_15 - _14)) { //#7410
                var _ = _16; //#7410
                _16 = _14; //#7410
                _14 = _; //#7410
            } //#7410
            $k[$j++] = _16; //#7417
            $k[$j++] = 1; //#7417
            $k[$j++] = 1; //#7417
            for (var _18 = _15, _17 = $f(_14 + 1); _18 >= _17; _18 -= 1) { //#7417
                var _19 = $k[--$j]; //#7413
                var _1A = $k[--$j]; //#7414
                var _1B = $k[--$j]; //#7414
                $k[$j++] = _1B; //#7416
                $k[$j++] = _1A; //#7416
                $k[$j++] = _19 * _18; //#7416
                if ($le(_1A, _1B)) { //#7416
                    var _1C = $k[--$j]; //#7415
                    var _1D = $k[--$j]; //#7415
                    $k[$j++] = $f(_1D + 1); //#7415
                    $k[$j++] = ~~(_1C / _1D); //#7415
                } //#7415
            } //#7415
            for (;;) { //#7421
                var _1E = $k[--$j]; //#7419
                var _1F = $k[--$j]; //#7419
                var _1G = $k[--$j]; //#7419
                $k[$j++] = _1G; //#7419
                $k[$j++] = _1F; //#7419
                $k[$j++] = _1E; //#7419
                if ($gt(_1F, _1G)) { //#7419
                    break; //#7419
                } //#7419
                var _1H = $k[--$j]; //#7420
                var _1I = $k[--$j]; //#7420
                $k[$j++] = $f(_1I + 1); //#7420
                $k[$j++] = ~~(_1H / _1I); //#7420
            } //#7420
            var _1J = $k[--$j]; //#7422
            var _1K = $k[--$j]; //#7422
            $k[$j++] = _1J; //#7422
            $k[$j++] = _1K; //#7422
            $j--; //#7422
            var _1L = $k[--$j]; //#7422
            var _1M = $k[--$j]; //#7422
            $k[$j++] = _1L; //#7422
            $k[$j++] = _1M; //#7422
            $j--; //#7422
        }; //#7423
        $1.getRSSwidths = function() {
            $1.oe = $k[--$j]; //#7426
            $1.el = $k[--$j]; //#7427
            $1.mw = $k[--$j]; //#7428
            $1.nm = $k[--$j]; //#7429
            $1.val = $k[--$j]; //#7430
            $1.out = $a($1.el); //#7431
            $1.mask = 0; //#7432
            for (var _1W = 0, _1V = $f($1.el - 2); _1W <= _1V; _1W += 1) { //#7459
                $1.bar = _1W; //#7434
                $1.ew = 1; //#7435
                var _1Y = $1.bar; //#7436
                $1.mask = $1.mask | ((_1Y < 0 ? 1 >>> -_1Y : 1 << _1Y)); //#7436
                for (;;) { //#7455
                    $k[$j++] = 'sval'; //#7437
                    $k[$j++] = $f($f($1.nm - $1.ew) - 1); //#7437
                    $k[$j++] = $f($f($1.el - $1.bar) - 2); //#7437
                    $1.ncr(); //#7437
                    var _1d = $k[--$j]; //#7437
                    $1[$k[--$j]] = _1d; //#7437
                    if (($1.oe && ($1.mask == 0)) && (($f(($f($f($1.nm - $1.ew) - ($1.el * 2))) + ($1.bar * 2))) >= -2)) { //#7440
                        $k[$j++] = 'sval'; //#7439
                        $k[$j++] = $1.sval; //#7439
                        $k[$j++] = $f(($f($f($1.nm - $1.ew) - $1.el)) + $1.bar); //#7439
                        $k[$j++] = $f($f($1.el - $1.bar) - 2); //#7439
                        $1.ncr(); //#7439
                        var _1s = $k[--$j]; //#7439
                        var _1t = $k[--$j]; //#7439
                        $1[$k[--$j]] = $f(_1t - _1s); //#7439
                    } //#7439
                    if ($f($1.el - $1.bar) > 2) { //#7449
                        $1.lval = 0; //#7442
                        for (var _23 = $f(($f(($f($f($1.nm - $1.ew) - $1.el)) + $1.bar)) + 2), _22 = $f($1.mw + 1); _23 >= _22; _23 -= 1) { //#7446
                            $k[$j++] = $f(($f($f($1.nm - _23) - $1.ew)) - 1); //#7444
                            $k[$j++] = $f($f($1.el - $1.bar) - 3); //#7444
                            $1.ncr(); //#7444
                            $1.lval = $f($k[--$j] + $1.lval); //#7445
                        } //#7445
                        $1.sval = $f($1.sval - ($1.lval * ($f($f($1.el - $1.bar) - 1)))); //#7447
                    } else { //#7449
                        if ($f($1.nm - $1.ew) > $1.mw) { //#7449
                            $1.sval = $f($1.sval - 1); //#7449
                        } //#7449
                    } //#7449
                    $1.val = $f($1.val - $1.sval); //#7451
                    if ($1.val < 0) { //#7452
                        break; //#7452
                    } //#7452
                    $1.ew = $1.ew + 1; //#7453
                    var _2N = $1.bar; //#7454
                    $1.mask = $1.mask & (~((_2N < 0 ? 1 >>> -_2N : 1 << _2N))); //#7454
                } //#7454
                $1.val = $f($1.val + $1.sval); //#7456
                $1.nm = $f($1.nm - $1.ew); //#7457
                $put($1.out, $1.bar, $1.ew); //#7458
            } //#7458
            $put($1.out, $f($1.el - 1), $1.nm); //#7460
            $k[$j++] = $1.out; //#7461
        }; //#7462
        for (var _2Z = 0; _2Z <= 11; _2Z += 1) { //#7468
            $1.i = _2Z; //#7465
            var _2a = $1.binval; //#7466
            var _2b = $1.i; //#7466
            $put(_2a, _2b + 1, $f($get(_2a, _2b + 1) + (($get($1.binval, $1.i) % 2013571) * 10))); //#7466
            $put($1.binval, $1.i, ~~($get($1.binval, $1.i) / 2013571)); //#7467
        } //#7467
        $1.d2 = $get($1.binval, 12) % 2013571; //#7469
        var _2n = $1.binval; //#7470
        $put(_2n, 12, ~~($get(_2n, 12) / 2013571)); //#7470
        $1.d1 = 0; //#7472
        $1.i = true; //#7473
        for (var _2p = 0; _2p <= 12; _2p += 1) { //#7483
            $1.j = _2p; //#7475
            var _2s = $get($1.binval, $1.j); //#7476
            $k[$j++] = _2s; //#7482
            if ((_2s == 0) && $1.i) { //#7481
                $j--; //#7478
            } else { //#7481
                $1.i = false; //#7480
                $1.d1 = $f($1.d1 + ($k[--$j] * (~~Math.pow(10, 12 - $1.j)))); //#7481
            } //#7481
        } //#7481
        $1.tab267 = $a([183063, 0, 17, 9, 6, 3, 6538, 28, 820063, 183064, 13, 13, 5, 4, 875, 728, 1000775, 820064, 9, 17, 3, 6, 28, 6454, 1491020, 1000776, 15, 11, 5, 4, 2415, 203, 1979844, 1491021, 11, 15, 4, 5, 203, 2408, 1996938, 1979845, 19, 7, 8, 1, 17094, 1, 2013570, 1996939, 7, 19, 1, 8, 1, 16632]); //#7493
        $1.i = 0; //#7495
        for (;;) { //#7505
            if ($1.d1 <= $get($1.tab267, $1.i)) { //#7503
                var _34 = $geti($1.tab267, $1.i + 1, 7); //#7497
                for (var _35 = 0, _36 = _34.length; _35 < _36; _35++) { //#7497
                    $k[$j++] = $get(_34, _35); //#7497
                } //#7497
                $1.d1te = $k[--$j]; //#7498
                $1.d1to = $k[--$j]; //#7498
                $1.d1mwe = $k[--$j]; //#7499
                $1.d1mwo = $k[--$j]; //#7499
                $1.d1ele = $k[--$j]; //#7500
                $1.d1elo = $k[--$j]; //#7500
                $1.d1gs = $k[--$j]; //#7501
                break; //#7502
            } //#7502
            $1.i = $1.i + 8; //#7504
        } //#7504
        $1.i = 0; //#7507
        for (;;) { //#7517
            if ($1.d2 <= $get($1.tab267, $1.i)) { //#7515
                var _3M = $geti($1.tab267, $1.i + 1, 7); //#7509
                for (var _3N = 0, _3O = _3M.length; _3N < _3O; _3N++) { //#7509
                    $k[$j++] = $get(_3M, _3N); //#7509
                } //#7509
                $1.d2te = $k[--$j]; //#7510
                $1.d2to = $k[--$j]; //#7510
                $1.d2mwe = $k[--$j]; //#7511
                $1.d2mwo = $k[--$j]; //#7511
                $1.d2ele = $k[--$j]; //#7512
                $1.d2elo = $k[--$j]; //#7512
                $1.d2gs = $k[--$j]; //#7513
                break; //#7514
            } //#7514
            $1.i = $1.i + 8; //#7516
        } //#7516
        $k[$j++] = 'd1wo'; //#7519
        $k[$j++] = ~~($f($1.d1 - $1.d1gs) / $1.d1te); //#7519
        $k[$j++] = $1.d1elo; //#7519
        $k[$j++] = $1.d1mwo; //#7519
        $k[$j++] = 7; //#7519
        $k[$j++] = false; //#7519
        $1.getRSSwidths(); //#7519
        var _3d = $k[--$j]; //#7519
        $1[$k[--$j]] = _3d; //#7519
        $k[$j++] = 'd1we'; //#7520
        $k[$j++] = $f($1.d1 - $1.d1gs) % $1.d1te; //#7520
        $k[$j++] = $1.d1ele; //#7520
        $k[$j++] = $1.d1mwe; //#7520
        $k[$j++] = 7; //#7520
        $k[$j++] = true; //#7520
        $1.getRSSwidths(); //#7520
        var _3k = $k[--$j]; //#7520
        $1[$k[--$j]] = _3k; //#7520
        $k[$j++] = 'd2wo'; //#7521
        $k[$j++] = ~~($f($1.d2 - $1.d2gs) / $1.d2te); //#7521
        $k[$j++] = $1.d2elo; //#7521
        $k[$j++] = $1.d2mwo; //#7521
        $k[$j++] = 7; //#7521
        $k[$j++] = false; //#7521
        $1.getRSSwidths(); //#7521
        var _3r = $k[--$j]; //#7521
        $1[$k[--$j]] = _3r; //#7521
        $k[$j++] = 'd2we'; //#7522
        $k[$j++] = $f($1.d2 - $1.d2gs) % $1.d2te; //#7522
        $k[$j++] = $1.d2ele; //#7522
        $k[$j++] = $1.d2mwe; //#7522
        $k[$j++] = 7; //#7522
        $k[$j++] = true; //#7522
        $1.getRSSwidths(); //#7522
        var _3y = $k[--$j]; //#7522
        $1[$k[--$j]] = _3y; //#7522
        $1.d1w = $a(14); //#7524
        for (var _41 = 0; _41 <= 6; _41 += 1) { //#7529
            $1.i = _41; //#7526
            $put($1.d1w, $1.i * 2, $get($1.d1wo, $1.i)); //#7527
            $put($1.d1w, ($1.i * 2) + 1, $get($1.d1we, $1.i)); //#7528
        } //#7528
        $1.d2w = $a(14); //#7531
        for (var _4D = 0; _4D <= 6; _4D += 1) { //#7536
            $1.i = _4D; //#7533
            $put($1.d2w, $1.i * 2, $get($1.d2wo, $1.i)); //#7534
            $put($1.d2w, ($1.i * 2) + 1, $get($1.d2we, $1.i)); //#7535
        } //#7535
        $k[$j++] = Infinity; //#7540
        var _4O = $1.d1w; //#7539
        for (var _4P = 0, _4Q = _4O.length; _4P < _4Q; _4P++) { //#7539
            $k[$j++] = $get(_4O, _4P); //#7539
        } //#7539
        var _4S = $1.d2w; //#7540
        for (var _4T = 0, _4U = _4S.length; _4T < _4U; _4T++) { //#7540
            $k[$j++] = $get(_4S, _4T); //#7540
        } //#7540
        $1.widths = $a(); //#7540
        $1.checkweights = $a([1, 3, 9, 27, 81, 65, 17, 51, 64, 14, 42, 37, 22, 66, 20, 60, 2, 6, 18, 54, 73, 41, 34, 13, 39, 28, 84, 74]); //#7546
        $k[$j++] = Infinity; //#7561
        for (var _4Y = 0; _4Y <= 43; _4Y += 1) { //#7549
            $k[$j++] = _4Y; //#7549
        } //#7549
        $k[$j++] = 45; //#7551
        $k[$j++] = 52; //#7551
        $k[$j++] = 57; //#7551
        for (var _4Z = 63; _4Z <= 66; _4Z += 1) { //#7551
            $k[$j++] = _4Z; //#7551
        } //#7551
        for (var _4a = 73; _4a <= 79; _4a += 1) { //#7552
            $k[$j++] = _4a; //#7552
        } //#7552
        $k[$j++] = 82; //#7554
        for (var _4b = 126; _4b <= 130; _4b += 1) { //#7554
            $k[$j++] = _4b; //#7554
        } //#7554
        $k[$j++] = 132; //#7556
        for (var _4c = 141; _4c <= 146; _4c += 1) { //#7556
            $k[$j++] = _4c; //#7556
        } //#7556
        for (var _4d = 210; _4d <= 217; _4d += 1) { //#7557
            $k[$j++] = _4d; //#7557
        } //#7557
        $k[$j++] = 220; //#7559
        for (var _4e = 316; _4e <= 320; _4e += 1) { //#7559
            $k[$j++] = _4e; //#7559
        } //#7559
        $k[$j++] = 322; //#7561
        $k[$j++] = 323; //#7561
        $k[$j++] = 326; //#7561
        $k[$j++] = 337; //#7561
        $1.checkseq = $a(); //#7561
        $1.checksum = 0; //#7564
        for (var _4g = 0; _4g <= 27; _4g += 1) { //#7568
            $1.i = _4g; //#7566
            $1.checksum = $f($1.checksum + ($get($1.widths, $1.i) * $get($1.checkweights, $1.i))); //#7567
        } //#7567
        $1.checksum = $1.checksum % 89; //#7569
        $1.seq = $get($1.checkseq, $1.checksum); //#7570
        $k[$j++] = 'swidths'; //#7571
        $k[$j++] = ~~($1.seq / 21); //#7571
        $k[$j++] = 8; //#7571
        $k[$j++] = 3; //#7571
        $k[$j++] = 6; //#7571
        $k[$j++] = false; //#7571
        $1.getRSSwidths(); //#7571
        var _4t = $k[--$j]; //#7571
        $1[$k[--$j]] = _4t; //#7571
        $k[$j++] = 'bwidths'; //#7572
        $k[$j++] = $1.seq % 21; //#7572
        $k[$j++] = 8; //#7572
        $k[$j++] = 3; //#7572
        $k[$j++] = 6; //#7572
        $k[$j++] = false; //#7572
        $1.getRSSwidths(); //#7572
        var _4w = $k[--$j]; //#7572
        $1[$k[--$j]] = _4w; //#7572
        $1.checkwidths = $a([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1]); //#7574
        for (var _4z = 0; _4z <= 5; _4z += 1) { //#7579
            $1.i = _4z; //#7576
            $put($1.checkwidths, $1.i * 2, $get($1.swidths, $1.i)); //#7577
            $put($1.checkwidths, ($1.i * 2) + 1, $get($1.bwidths, $1.i)); //#7578
        } //#7578
        $k[$j++] = Infinity; //#7582
        var _5A = $1.d1w; //#7582
        $k[$j++] = 1; //#7582
        for (var _5B = 0, _5C = _5A.length; _5B < _5C; _5B++) { //#7582
            $k[$j++] = $get(_5A, _5B); //#7582
        } //#7582
        var _5E = $1.checkwidths; //#7582
        for (var _5F = 0, _5G = _5E.length; _5F < _5G; _5F++) { //#7582
            $k[$j++] = $get(_5E, _5F); //#7582
        } //#7582
        var _5I = $1.d2w; //#7582
        for (var _5J = 0, _5K = _5I.length; _5J < _5K; _5J++) { //#7582
            $k[$j++] = $get(_5I, _5J); //#7582
        } //#7582
        $k[$j++] = 1; //#7582
        $k[$j++] = 1; //#7582
        $k[$j++] = 5; //#7582
        $1.sbs = $a(); //#7582
        $k[$j++] = Infinity; //#7593
        var _5N = $1.sbs; //#7588
        $k[$j++] = Infinity; //#7589
        for (var _5P = 0, _5Q = ~~(($1.sbs.length + 1) / 2); _5P < _5Q; _5P++) { //#7589
            $k[$j++] = $1.height; //#7589
        } //#7589
        var _5S = $a(); //#7589
        $k[$j++] = Infinity; //#7590
        for (var _5U = 0, _5V = ~~(($1.sbs.length + 1) / 2); _5U < _5V; _5U++) { //#7590
            $k[$j++] = 0; //#7590
        } //#7590
        var _5W = $a(); //#7590
        $k[$j++] = 'ren'; //#7593
        $k[$j++] = $0.renlinear; //#7593
        $k[$j++] = 'sbs'; //#7593
        $k[$j++] = _5N; //#7593
        $k[$j++] = 'bhs'; //#7593
        $k[$j++] = _5S; //#7593
        $k[$j++] = 'bbs'; //#7593
        $k[$j++] = _5W; //#7593
        $k[$j++] = 'txt'; //#7593
        $k[$j++] = $1.txt; //#7593
        $k[$j++] = 'textxalign'; //#7593
        $k[$j++] = "center"; //#7593
        $k[$j++] = 'opt'; //#7593
        $k[$j++] = $1.options; //#7593
        var _5Z = $d(); //#7593
        $k[$j++] = _5Z; //#7596
        if (!$1.dontdraw) { //#7596
            $0.renlinear(); //#7596
        } //#7596
    };
    $0.databarexpanded = function() {
        var $1 = {}; //#7628
        $1.options = $k[--$j]; //#7630
        $1.barcode = $k[--$j]; //#7631
        $1.parse = false; //#7633
        $1.dontlint = false; //#7634
        $1.dontdraw = false; //#7635
        $1.height = 34 / 72; //#7636
        $1.format = "expanded"; //#7637
        $1.segments = -1; //#7638
        $1.linkage = false; //#7639
        $1.barxmult = 34; //#7640
        $forall($1.options, function() { //#7651
            var _3 = $k[--$j]; //#7651
            $1[$k[--$j]] = _3; //#7651
        }); //#7651
        $1.height = +$1.height; //#7653
        $1.segments = ~~$1.segments; //#7654
        $1.barxmult = ~~$1.barxmult; //#7655
        if ($1.segments == -1) { //#7661
            var _A = $eq($1.format, "expandedstacked") ? 4 : 22; //#7658
            $1.segments = _A; //#7658
        } else { //#7661
            if ((($1.segments < 2) || ($1.segments > 22)) || (($1.segments % 2) != 0)) { //#7662
                $k[$j++] = 'bwipp.gs1databarexpandedBadSegments'; //#7661
                $k[$j++] = "The number of segments must be even from 2 to 22"; //#7661
                $0.raiseerror(); //#7661
            } //#7661
        } //#7661
        $1.expand = function() {
            var _F = new Map([
                ["parse", $1.parse],
                ["parseonly", true],
                ["parsefnc", false]
            ]); //#7670
            $1.fncvals = _F; //#7671
            $k[$j++] = $1.fncvals; //#7672
            $0.parseinput(); //#7672
        }; //#7673
        $1.ais = $a([]); //#7676
        $1.vals = $a([]); //#7677
        var _J = $1.barcode; //#7678
        $k[$j++] = $geti(_J, 1, _J.length - 1); //#7691
        for (;;) { //#7691
            var _L = $k[--$j]; //#7680
            $k[$j++] = _L; //#7680
            if ($eq(_L, "")) { //#7680
                break; //#7680
            } //#7680
            $search($k[--$j], ")"); //#7681
            $j--; //#7681
            var _N = $k[--$j]; //#7682
            var _O = $k[--$j]; //#7682
            $k[$j++] = _N; //#7682
            $k[$j++] = _O; //#7682
            $j--; //#7682
            var _P = $k[--$j]; //#7683
            var _Q = $k[--$j]; //#7683
            $k[$j++] = _P; //#7683
            $search(_Q, "("); //#7683
            if ($k[--$j]) { //#7686
                var _S = $k[--$j]; //#7684
                var _T = $k[--$j]; //#7684
                $k[$j++] = _S; //#7684
                $k[$j++] = _T; //#7684
                $j--; //#7684
                var _U = $k[--$j]; //#7684
                var _V = $k[--$j]; //#7684
                var _W = $k[--$j]; //#7684
                $k[$j++] = _V; //#7684
                $k[$j++] = _W; //#7684
                $k[$j++] = _U; //#7684
            } else { //#7686
                var _X = $k[--$j]; //#7686
                var _Y = $k[--$j]; //#7686
                $k[$j++] = ""; //#7686
                $k[$j++] = _Y; //#7686
                $k[$j++] = _X; //#7686
            } //#7686
            $k[$j++] = Infinity; //#7688
            $aload($1.ais); //#7688
            var _b = $k[$j - 1 - ($counttomark() + 2)]; //#7688
            $k[$j++] = _b; //#7688
            $1.ais = $a(); //#7688
            $k[$j++] = Infinity; //#7689
            $aload($1.vals); //#7689
            var _f = $k[$j - 1 - ($counttomark() + 1)]; //#7689
            $k[$j++] = _f; //#7689
            $1.expand(); //#7689
            $1.vals = $a(); //#7689
            $j -= 2; //#7690
        } //#7690
        $j--; //#7692
        if (!$1.dontlint) { //#7694
            $k[$j++] = $1.ais; //#7694
            $k[$j++] = $1.vals; //#7694
            $0.gs1lint(); //#7694
            $j--; //#7694
        } //#7694
        for (;;) { //#7782
            if ($1.ais.length == 2) { //#7706
                if ($eq($get($1.ais, 0), "01") && $eq($get($1.ais, 1), "3103")) { //#7705
                    if ($eq($geti($get($1.vals, 0), 0, 1), "9") && ((~~$z($get($1.vals, 1))) <= 32767)) { //#7704
                        $k[$j++] = "0100"; //#7703
                        $k[$j++] = false; //#7703
                        break; //#7703
                    } //#7703
                } //#7703
            } //#7703
            if ($1.ais.length == 2) { //#7715
                if ($eq($get($1.ais, 0), "01") && $eq($get($1.ais, 1), "3202")) { //#7714
                    if ($eq($geti($get($1.vals, 0), 0, 1), "9") && ((~~$z($get($1.vals, 1))) <= 9999)) { //#7713
                        $k[$j++] = "0101"; //#7712
                        $k[$j++] = false; //#7712
                        break; //#7712
                    } //#7712
                } //#7712
            } //#7712
            if ($1.ais.length == 2) { //#7724
                if ($eq($get($1.ais, 0), "01") && $eq($get($1.ais, 1), "3203")) { //#7723
                    if ($eq($geti($get($1.vals, 0), 0, 1), "9") && ((~~$z($get($1.vals, 1))) <= 22767)) { //#7722
                        $k[$j++] = "0101"; //#7721
                        $k[$j++] = false; //#7721
                        break; //#7721
                    } //#7721
                } //#7721
            } //#7721
            var _1F = $1.ais.length; //#7727
            if ((_1F == 2) || (_1F == 3)) { //#7758
                $k[$j++] = 'ai310x'; //#7728
                $k[$j++] = false; //#7728
                for (var _1G = 3100; _1G <= 3109; _1G += 1) { //#7728
                    var _1L = $k[--$j]; //#7728
                    $k[$j++] = _1L || $eq($cvrs($s(4), _1G, 10), $get($1.ais, 1)); //#7728
                } //#7728
                var _1M = $k[--$j]; //#7728
                $1[$k[--$j]] = _1M; //#7728
                $k[$j++] = 'ai320x'; //#7729
                $k[$j++] = false; //#7729
                for (var _1O = 3200; _1O <= 3209; _1O += 1) { //#7729
                    var _1T = $k[--$j]; //#7729
                    $k[$j++] = _1T || $eq($cvrs($s(4), _1O, 10), $get($1.ais, 1)); //#7729
                } //#7729
                var _1U = $k[--$j]; //#7729
                $1[$k[--$j]] = _1U; //#7729
                if ($1.ais.length == 3) { //#7733
                    var _1X = $a(["11", "13", "15", "17"]); //#7731
                    $k[$j++] = 'aibad'; //#7731
                    $k[$j++] = true; //#7731
                    for (var _1Y = 0, _1Z = _1X.length; _1Y < _1Z; _1Y++) { //#7731
                        var _1d = $k[--$j]; //#7731
                        $k[$j++] = _1d && $ne($get(_1X, _1Y), $get($1.ais, 2)); //#7731
                    } //#7731
                    var _1e = $k[--$j]; //#7731
                    $1[$k[--$j]] = _1e; //#7731
                } else { //#7733
                    $1.aibad = false; //#7733
                } //#7733
                if (($eq($get($1.ais, 0), "01") && ($1.ai310x || $1.ai320x)) && (!$1.aibad)) { //#7757
                    if ($1.ais.length == 3) { //#7754
                        var _1u = ~~$z($geti($get($1.vals, 2), 2, 2)); //#7739
                        var _1y = ~~$z($geti($get($1.vals, 2), 4, 2)); //#7740
                        if ((($eq($geti($get($1.vals, 0), 0, 1), "9") && ((~~$z($get($1.vals, 1))) <= 99999)) && ((_1u >= 1) && (_1u <= 12))) && ((_1y >= 0) && (_1y <= 31))) { //#7749
                            if ($1.ai310x && $eq($get($1.ais, 2), "11")) { //#7741
                                $k[$j++] = "0111000"; //#7741
                                $k[$j++] = false; //#7741
                                break; //#7741
                            } //#7741
                            if ($1.ai320x && $eq($get($1.ais, 2), "11")) { //#7742
                                $k[$j++] = "0111001"; //#7742
                                $k[$j++] = false; //#7742
                                break; //#7742
                            } //#7742
                            if ($1.ai310x && $eq($get($1.ais, 2), "13")) { //#7743
                                $k[$j++] = "0111010"; //#7743
                                $k[$j++] = false; //#7743
                                break; //#7743
                            } //#7743
                            if ($1.ai320x && $eq($get($1.ais, 2), "13")) { //#7744
                                $k[$j++] = "0111011"; //#7744
                                $k[$j++] = false; //#7744
                                break; //#7744
                            } //#7744
                            if ($1.ai310x && $eq($get($1.ais, 2), "15")) { //#7745
                                $k[$j++] = "0111100"; //#7745
                                $k[$j++] = false; //#7745
                                break; //#7745
                            } //#7745
                            if ($1.ai320x && $eq($get($1.ais, 2), "15")) { //#7746
                                $k[$j++] = "0111101"; //#7746
                                $k[$j++] = false; //#7746
                                break; //#7746
                            } //#7746
                            if ($1.ai310x && $eq($get($1.ais, 2), "17")) { //#7747
                                $k[$j++] = "0111110"; //#7747
                                $k[$j++] = false; //#7747
                                break; //#7747
                            } //#7747
                            if ($1.ai320x && $eq($get($1.ais, 2), "17")) { //#7748
                                $k[$j++] = "0111111"; //#7748
                                $k[$j++] = false; //#7748
                                break; //#7748
                            } //#7748
                        } //#7748
                    } else { //#7754
                        if ($eq($geti($get($1.vals, 0), 0, 1), "9") && ((~~$z($get($1.vals, 1))) <= 99999)) { //#7755
                            if ($1.ai310x) { //#7753
                                $k[$j++] = "0111000"; //#7753
                                $k[$j++] = false; //#7753
                                break; //#7753
                            } //#7753
                            if ($1.ai320x) { //#7754
                                $k[$j++] = "0111001"; //#7754
                                $k[$j++] = false; //#7754
                                break; //#7754
                            } //#7754
                        } //#7754
                    } //#7754
                } //#7754
            } //#7754
            if ($1.ais.length >= 2) { //#7766
                $k[$j++] = 'ai392x'; //#7762
                $k[$j++] = false; //#7762
                for (var _2V = 3920; _2V <= 3923; _2V += 1) { //#7762
                    var _2a = $k[--$j]; //#7762
                    $k[$j++] = _2a || $eq($cvrs($s(4), _2V, 10), $get($1.ais, 1)); //#7762
                } //#7762
                var _2b = $k[--$j]; //#7762
                $1[$k[--$j]] = _2b; //#7762
                if ($eq($get($1.ais, 0), "01") && $1.ai392x) { //#7765
                    if ($eq($geti($get($1.vals, 0), 0, 1), "9")) { //#7764
                        $k[$j++] = "01100"; //#7764
                        $k[$j++] = true; //#7764
                        break; //#7764
                    } //#7764
                } //#7764
            } //#7764
            if ($1.ais.length >= 2) { //#7774
                $k[$j++] = 'ai393x'; //#7770
                $k[$j++] = false; //#7770
                for (var _2k = 3930; _2k <= 3933; _2k += 1) { //#7770
                    var _2p = $k[--$j]; //#7770
                    $k[$j++] = _2p || $eq($cvrs($s(4), _2k, 10), $get($1.ais, 1)); //#7770
                } //#7770
                var _2q = $k[--$j]; //#7770
                $1[$k[--$j]] = _2q; //#7770
                if ($eq($get($1.ais, 0), "01") && $1.ai393x) { //#7773
                    if ($eq($geti($get($1.vals, 0), 0, 1), "9")) { //#7772
                        $k[$j++] = "01101"; //#7772
                        $k[$j++] = true; //#7772
                        break; //#7772
                    } //#7772
                } //#7772
            } //#7772
            if ($eq($get($1.ais, 0), "01")) { //#7777
                $k[$j++] = "1"; //#7777
                $k[$j++] = true; //#7777
                break; //#7777
            } //#7777
            $k[$j++] = "00"; //#7780
            $k[$j++] = true; //#7780
            break; //#7780
        } //#7780
        $1.gpfallow = $k[--$j]; //#7783
        $1.method = $k[--$j]; //#7784
        $1.conv12to40 = function() {
            var _33 = $strcpy($s(40), "0000000000000000000000000000000000000000"); //#7787
            var _34 = $k[--$j]; //#7788
            var _38 = $cvrs($s(10), ~~$z($geti(_34, 0, 3)), 2); //#7788
            $puti($geti(_33, 0, 10), 10 - _38.length, _38); //#7788
            var _3C = $cvrs($s(10), ~~$z($geti(_34, 3, 3)), 2); //#7789
            $puti($geti(_33, 10, 10), 10 - _3C.length, _3C); //#7789
            var _3G = $cvrs($s(10), ~~$z($geti(_34, 6, 3)), 2); //#7790
            $puti($geti(_33, 20, 10), 10 - _3G.length, _3G); //#7790
            var _3K = $cvrs($s(10), ~~$z($geti(_34, 9, 3)), 2); //#7791
            $puti($geti(_33, 30, 10), 10 - _3K.length, _3K); //#7791
            $k[$j++] = _33; //#7792
            $k[$j++] = _34; //#7792
            $j--; //#7792
        }; //#7793
        $1.conv13to44 = function() {
            var _3M = $strcpy($s(44), "00000000000000000000000000000000000000000000"); //#7796
            var _3N = $k[--$j]; //#7797
            var _3R = $cvrs($s(4), ~~$z($geti(_3N, 0, 1)), 2); //#7797
            $puti($geti(_3M, 0, 4), 4 - _3R.length, _3R); //#7797
            $k[$j++] = _3M; //#7798
            $k[$j++] = _3M; //#7798
            $k[$j++] = $geti(_3N, 1, 12); //#7798
            $1.conv12to40(); //#7798
            var _3T = $k[--$j]; //#7798
            $puti($k[--$j], 4, _3T); //#7798
        }; //#7799
        $1.tobin = function() {
            var _3W = $s($k[--$j]); //#7802
            $k[$j++] = _3W; //#7802
            for (var _3Y = 0, _3X = _3W.length - 1; _3Y <= _3X; _3Y += 1) { //#7802
                var _3Z = $k[--$j]; //#7802
                $put(_3Z, _3Y, 48); //#7802
                $k[$j++] = _3Z; //#7802
            } //#7802
            var _3a = $k[--$j]; //#7803
            var _3d = $cvrs($s(_3a.length), $k[--$j], 2); //#7803
            $puti(_3a, _3a.length - _3d.length, _3d); //#7803
            $k[$j++] = _3a; //#7803
        }; //#7804
        $1.fnc1 = -1; //#7806
        $1.lnumeric = -2; //#7806
        $1.lalphanumeric = -3; //#7806
        $1.liso646 = -4; //#7806
        if ($eq($1.method, "00")) { //#7811
            $1.cdf = $a([]); //#7809
            $1.gpf = $a([]); //#7810
        } //#7810
        if ($eq($1.method, "1")) { //#7819
            $k[$j++] = 'cdf'; //#7814
            $k[$j++] = $geti($get($1.vals, 0), 0, 13); //#7814
            $1.conv13to44(); //#7814
            var _3l = $k[--$j]; //#7814
            $1[$k[--$j]] = _3l; //#7814
            $k[$j++] = Infinity; //#7815
            var _3n = $1.cdf; //#7815
            for (var _3o = 0, _3p = _3n.length; _3o < _3p; _3o++) { //#7815
                $k[$j++] = $f($get(_3n, _3o) - 48); //#7815
            } //#7815
            $1.cdf = $a(); //#7815
            $1.gpf = $a([]); //#7816
            $1.ais = $geti($1.ais, 1, $1.ais.length - 1); //#7817
            $1.vals = $geti($1.vals, 1, $1.vals.length - 1); //#7818
        } //#7818
        if ($eq($1.method, "0100")) { //#7829
            $1.cdf = $s(55); //#7822
            $k[$j++] = $1.cdf; //#7823
            $k[$j++] = 0; //#7823
            $k[$j++] = $geti($get($1.vals, 0), 1, 12); //#7823
            $1.conv12to40(); //#7823
            var _45 = $k[--$j]; //#7823
            var _46 = $k[--$j]; //#7823
            $puti($k[--$j], _46, _45); //#7823
            $k[$j++] = $1.cdf; //#7824
            $k[$j++] = 40; //#7824
            $k[$j++] = ~~$z($get($1.vals, 1)); //#7824
            $k[$j++] = 15; //#7824
            $1.tobin(); //#7824
            var _4B = $k[--$j]; //#7824
            var _4C = $k[--$j]; //#7824
            $puti($k[--$j], _4C, _4B); //#7824
            $k[$j++] = Infinity; //#7825
            var _4E = $1.cdf; //#7825
            for (var _4F = 0, _4G = _4E.length; _4F < _4G; _4F++) { //#7825
                $k[$j++] = $get(_4E, _4F) - 48; //#7825
            } //#7825
            $1.cdf = $a(); //#7825
            $1.gpf = $a([]); //#7826
            $1.ais = $a([]); //#7827
            $1.vals = $a([]); //#7828
        } //#7828
        if ($eq($1.method, "0101")) { //#7844
            $1.cdf = $s(55); //#7832
            $k[$j++] = $1.cdf; //#7833
            $k[$j++] = 0; //#7833
            $k[$j++] = $geti($get($1.vals, 0), 1, 12); //#7833
            $1.conv12to40(); //#7833
            var _4S = $k[--$j]; //#7833
            var _4T = $k[--$j]; //#7833
            $puti($k[--$j], _4T, _4S); //#7833
            if ($eq($get($1.ais, 1), "3202")) { //#7837
                $k[$j++] = ~~$z($get($1.vals, 1)); //#7835
                $k[$j++] = 15; //#7835
                $1.tobin(); //#7835
            } else { //#7837
                $k[$j++] = (~~$z($get($1.vals, 1))) + 10000; //#7837
                $k[$j++] = 15; //#7837
                $1.tobin(); //#7837
            } //#7837
            $puti($1.cdf, 40, $k[--$j]); //#7839
            $k[$j++] = Infinity; //#7840
            var _4d = $1.cdf; //#7840
            for (var _4e = 0, _4f = _4d.length; _4e < _4f; _4e++) { //#7840
                $k[$j++] = $get(_4d, _4e) - 48; //#7840
            } //#7840
            $1.cdf = $a(); //#7840
            $1.gpf = $a([]); //#7841
            $1.ais = $a([]); //#7842
            $1.vals = $a([]); //#7843
        } //#7843
        if ($1.method.length == 7) { //#7865
            $1.cdf = $s(76); //#7847
            $k[$j++] = $1.cdf; //#7848
            $k[$j++] = 0; //#7848
            $k[$j++] = $geti($get($1.vals, 0), 1, 12); //#7848
            $1.conv12to40(); //#7848
            var _4r = $k[--$j]; //#7848
            var _4s = $k[--$j]; //#7848
            $puti($k[--$j], _4s, _4r); //#7848
            var _4u = $s(6); //#7849
            $puti(_4u, 0, $geti($get($1.ais, 1), 3, 1)); //#7849
            $puti(_4u, 1, $geti($get($1.vals, 1), 1, 5)); //#7850
            $k[$j++] = ~~$z(_4u); //#7850
            $k[$j++] = 20; //#7850
            $1.tobin(); //#7850
            $puti($1.cdf, 40, $k[--$j]); //#7851
            if ($1.ais.length == 3) { //#7858
                var _55 = $get($1.vals, 2); //#7853
                $k[$j++] = ((~~$z($geti(_55, 0, 2))) * 384) + ((((~~$z($geti(_55, 2, 2))) - 1) * 32) + (~~$z($geti(_55, 4, 2)))); //#7856
            } else { //#7858
                $k[$j++] = 38400; //#7858
            } //#7858
            $k[$j++] = 16; //#7860
            $1.tobin(); //#7860
            $puti($1.cdf, 60, $k[--$j]); //#7860
            $k[$j++] = Infinity; //#7861
            var _5B = $1.cdf; //#7861
            for (var _5C = 0, _5D = _5B.length; _5C < _5D; _5C++) { //#7861
                $k[$j++] = $get(_5B, _5C) - 48; //#7861
            } //#7861
            $1.cdf = $a(); //#7861
            $1.gpf = $a([]); //#7862
            $1.ais = $a([]); //#7863
            $1.vals = $a([]); //#7864
        } //#7864
        if ($eq($1.method, "01100")) { //#7875
            $1.cdf = $s(42); //#7868
            $k[$j++] = $1.cdf; //#7869
            $k[$j++] = 0; //#7869
            $k[$j++] = $geti($get($1.vals, 0), 1, 12); //#7869
            $1.conv12to40(); //#7869
            var _5P = $k[--$j]; //#7869
            var _5Q = $k[--$j]; //#7869
            $puti($k[--$j], _5Q, _5P); //#7869
            $k[$j++] = $1.cdf; //#7870
            $k[$j++] = 40; //#7870
            $k[$j++] = ~~$z($geti($get($1.ais, 1), 3, 1)); //#7870
            $k[$j++] = 2; //#7870
            $1.tobin(); //#7870
            var _5W = $k[--$j]; //#7870
            var _5X = $k[--$j]; //#7870
            $puti($k[--$j], _5X, _5W); //#7870
            $k[$j++] = Infinity; //#7871
            var _5Z = $1.cdf; //#7871
            for (var _5a = 0, _5b = _5Z.length; _5a < _5b; _5a++) { //#7871
                $k[$j++] = $get(_5Z, _5a) - 48; //#7871
            } //#7871
            $1.cdf = $a(); //#7871
            $k[$j++] = Infinity; //#7872
            $forall($get($1.vals, 1)); //#7872
            if ($1.ais.length > 2) { //#7872
                $k[$j++] = $1.fnc1; //#7872
            } //#7872
            $1.gpf = $a(); //#7872
            $1.ais = $geti($1.ais, 2, $1.ais.length - 2); //#7873
            $1.vals = $geti($1.vals, 2, $1.vals.length - 2); //#7874
        } //#7874
        if ($eq($1.method, "01101")) { //#7886
            $1.cdf = $s(52); //#7878
            $k[$j++] = $1.cdf; //#7879
            $k[$j++] = 0; //#7879
            $k[$j++] = $geti($get($1.vals, 0), 1, 12); //#7879
            $1.conv12to40(); //#7879
            var _5v = $k[--$j]; //#7879
            var _5w = $k[--$j]; //#7879
            $puti($k[--$j], _5w, _5v); //#7879
            $k[$j++] = $1.cdf; //#7880
            $k[$j++] = 40; //#7880
            $k[$j++] = ~~$z($geti($get($1.ais, 1), 3, 1)); //#7880
            $k[$j++] = 2; //#7880
            $1.tobin(); //#7880
            var _62 = $k[--$j]; //#7880
            var _63 = $k[--$j]; //#7880
            $puti($k[--$j], _63, _62); //#7880
            $k[$j++] = $1.cdf; //#7881
            $k[$j++] = 42; //#7881
            $k[$j++] = ~~$z($geti($get($1.vals, 1), 0, 3)); //#7881
            $k[$j++] = 10; //#7881
            $1.tobin(); //#7881
            var _69 = $k[--$j]; //#7881
            var _6A = $k[--$j]; //#7881
            $puti($k[--$j], _6A, _69); //#7881
            $k[$j++] = Infinity; //#7882
            var _6C = $1.cdf; //#7882
            for (var _6D = 0, _6E = _6C.length; _6D < _6E; _6D++) { //#7882
                $k[$j++] = $get(_6C, _6D) - 48; //#7882
            } //#7882
            $1.cdf = $a(); //#7882
            $k[$j++] = Infinity; //#7883
            var _6I = $get($1.vals, 1); //#7883
            var _6J = $geti(_6I, 3, _6I.length - 3); //#7883
            for (var _6K = 0, _6L = _6J.length; _6K < _6L; _6K++) { //#7883
                $k[$j++] = $get(_6J, _6K); //#7883
            } //#7883
            if ($1.ais.length > 2) { //#7883
                $k[$j++] = $1.fnc1; //#7883
            } //#7883
            $1.gpf = $a(); //#7883
            $1.ais = $geti($1.ais, 2, $1.ais.length - 2); //#7884
            $1.vals = $geti($1.vals, 2, $1.vals.length - 2); //#7885
        } //#7885
        if ($1.gpfallow) { //#7892
            $1.vlf = $a(2); //#7890
        } else { //#7892
            $1.vlf = $a([]); //#7892
        } //#7892
        $1.aifixed = new Map; //#7898
        $k[$j++] = Infinity; //#7900
        for (var _6Z = 0; _6Z <= 4; _6Z += 1) { //#7900
            $k[$j++] = _6Z; //#7900
        } //#7900
        var _6a = $a(); //#7900
        for (var _6b = 0, _6c = _6a.length; _6b < _6c; _6b++) { //#7903
            var _6f = $strcpy($s(2), "00"); //#7902
            $put(_6f, 1, $f($get(_6a, _6b) + 48)); //#7902
            $put($1.aifixed, _6f, _6f); //#7902
        } //#7902
        $k[$j++] = Infinity; //#7908
        for (var _6h = 11; _6h <= 20; _6h += 1) { //#7905
            $k[$j++] = _6h; //#7905
        } //#7905
        $k[$j++] = 23; //#7907
        for (var _6i = 31; _6i <= 36; _6i += 1) { //#7907
            $k[$j++] = _6i; //#7907
        } //#7907
        $k[$j++] = 41; //#7908
        var _6j = $a(); //#7908
        for (var _6k = 0, _6l = _6j.length; _6k < _6l; _6k++) { //#7911
            var _6o = $cvrs($s(2), $get(_6j, _6k), 10); //#7910
            $put($1.aifixed, _6o, _6o); //#7910
        } //#7910
        $k[$j++] = Infinity; //#7923
        for (var _6q = 0; _6q <= 119; _6q += 1) { //#7922
            var _6s = $strcpy($s(2), "00"); //#7915
            var _6u = $cvrs($s(2), _6q, 11); //#7915
            $puti(_6s, 2 - _6u.length, _6u); //#7916
            $k[$j++] = _6q; //#7917
            $k[$j++] = _6s; //#7917
            if ($get(_6s, 0) == 65) { //#7917
                var _6w = $k[--$j]; //#7917
                $put(_6w, 0, 94); //#7917
                $k[$j++] = _6w; //#7917
            } //#7917
            var _6x = $k[--$j]; //#7918
            $k[$j++] = _6x; //#7918
            if ($get(_6x, 1) == 65) { //#7918
                var _6z = $k[--$j]; //#7918
                $put(_6z, 1, 94); //#7918
                $k[$j++] = _6z; //#7918
            } //#7918
            var _70 = $k[--$j]; //#7919
            var _73 = $strcpy($s(7), "0000000"); //#7920
            var _75 = $cvrs($s(7), $f($k[--$j] + 8), 2); //#7920
            $puti(_73, 7 - _75.length, _75); //#7921
            $k[$j++] = _70; //#7921
            $k[$j++] = _73; //#7921
        } //#7921
        $k[$j++] = $1.lalphanumeric; //#7923
        $k[$j++] = "0000"; //#7923
        $1.numeric = $d(); //#7924
        $k[$j++] = Infinity; //#7933
        for (var _78 = 48; _78 <= 57; _78 += 1) { //#7927
            $k[$j++] = _78; //#7927
            $k[$j++] = _78 - 43; //#7927
            $k[$j++] = 5; //#7927
            $1.tobin(); //#7927
        } //#7927
        $k[$j++] = $1.fnc1; //#7929
        $k[$j++] = "01111"; //#7929
        for (var _7A = 65; _7A <= 90; _7A += 1) { //#7929
            $k[$j++] = _7A; //#7929
            $k[$j++] = _7A - 33; //#7929
            $k[$j++] = 6; //#7929
            $1.tobin(); //#7929
        } //#7929
        $k[$j++] = 42; //#7931
        $k[$j++] = "111010"; //#7931
        for (var _7B = 44; _7B <= 47; _7B += 1) { //#7931
            $k[$j++] = _7B; //#7931
            $k[$j++] = _7B + 15; //#7931
            $k[$j++] = 6; //#7931
            $1.tobin(); //#7931
        } //#7931
        $k[$j++] = $1.lnumeric; //#7933
        $k[$j++] = "000"; //#7933
        $k[$j++] = $1.liso646; //#7933
        $k[$j++] = "00100"; //#7933
        $1.alphanumeric = $d(); //#7934
        $k[$j++] = Infinity; //#7948
        for (var _7F = 48; _7F <= 57; _7F += 1) { //#7937
            $k[$j++] = _7F; //#7937
            $k[$j++] = _7F - 43; //#7937
            $k[$j++] = 5; //#7937
            $1.tobin(); //#7937
        } //#7937
        $k[$j++] = $1.fnc1; //#7939
        $k[$j++] = "01111"; //#7939
        for (var _7H = 65; _7H <= 90; _7H += 1) { //#7939
            $k[$j++] = _7H; //#7939
            $k[$j++] = _7H - 1; //#7939
            $k[$j++] = 7; //#7939
            $1.tobin(); //#7939
        } //#7939
        for (var _7I = 97; _7I <= 122; _7I += 1) { //#7940
            $k[$j++] = _7I; //#7940
            $k[$j++] = _7I - 7; //#7940
            $k[$j++] = 7; //#7940
            $1.tobin(); //#7940
        } //#7940
        $k[$j++] = 33; //#7943
        $k[$j++] = "11101000"; //#7943
        $k[$j++] = 34; //#7943
        $k[$j++] = "11101001"; //#7943
        for (var _7J = 37; _7J <= 47; _7J += 1) { //#7943
            $k[$j++] = _7J; //#7943
            $k[$j++] = _7J + 197; //#7943
            $k[$j++] = 8; //#7943
            $1.tobin(); //#7943
        } //#7943
        for (var _7K = 58; _7K <= 63; _7K += 1) { //#7944
            $k[$j++] = _7K; //#7944
            $k[$j++] = _7K + 187; //#7944
            $k[$j++] = 8; //#7944
            $1.tobin(); //#7944
        } //#7944
        $k[$j++] = 95; //#7948
        $k[$j++] = "11111011"; //#7948
        $k[$j++] = 32; //#7948
        $k[$j++] = "11111100"; //#7948
        $k[$j++] = $1.lnumeric; //#7948
        $k[$j++] = "000"; //#7948
        $k[$j++] = $1.lalphanumeric; //#7948
        $k[$j++] = "00100"; //#7948
        $1.iso646 = $d(); //#7949
        for (var _7Q = 0, _7P = $1.ais.length - 1; _7Q <= _7P; _7Q += 1) { //#7967
            $1.i = _7Q; //#7953
            $1.ai = $get($1.ais, $1.i); //#7954
            $1.val = $get($1.vals, $1.i); //#7955
            var _7a = $a(($1.gpf.length + $1.ai.length) + $1.val.length); //#7956
            $puti(_7a, 0, $1.gpf); //#7957
            $k[$j++] = _7a; //#7958
            $k[$j++] = _7a; //#7958
            $k[$j++] = $1.gpf.length; //#7958
            $k[$j++] = $1.ai; //#7958
            $k[$j++] = Infinity; //#7958
            var _7e = $k[--$j]; //#7958
            var _7f = $k[--$j]; //#7958
            $k[$j++] = _7e; //#7958
            $forall(_7f); //#7958
            var _7g = $a(); //#7958
            var _7h = $k[--$j]; //#7958
            $puti($k[--$j], _7h, _7g); //#7958
            var _7j = $k[--$j]; //#7959
            $k[$j++] = _7j; //#7959
            $k[$j++] = _7j; //#7959
            $k[$j++] = $1.gpf.length + $1.ai.length; //#7959
            $k[$j++] = $1.val; //#7959
            $k[$j++] = Infinity; //#7959
            var _7n = $k[--$j]; //#7959
            var _7o = $k[--$j]; //#7959
            $k[$j++] = _7n; //#7959
            $forall(_7o); //#7959
            var _7p = $a(); //#7959
            var _7q = $k[--$j]; //#7959
            $puti($k[--$j], _7q, _7p); //#7959
            $1.gpf = $k[--$j]; //#7960
            var _7y = $get($1.aifixed, $geti($1.ai, 0, 2)) !== undefined; //#7961
            if (($1.i != ($1.ais.length - 1)) && (!_7y)) { //#7966
                var _80 = $a($1.gpf.length + 1); //#7962
                $puti(_80, 0, $1.gpf); //#7963
                $put(_80, $1.gpf.length, $1.fnc1); //#7964
                $1.gpf = _80; //#7965
            } //#7965
        } //#7965
        $1.rembits = function() {
            var _84 = $k[--$j]; //#7971
            var _85 = 48; //#7973
            var _86 = (~~Math.ceil(_84 / 12)) * 12; //#7973
            if (((~~Math.ceil(_84 / 12)) * 12) < 48) { //#7973
                var _ = _85; //#7973
                _85 = _86; //#7973
                _86 = _; //#7973
            } //#7973
            var _87 = ~~(_86 / 12); //#7974
            $k[$j++] = _84; //#7978
            $k[$j++] = _86; //#7978
            $k[$j++] = _87; //#7978
            if ((_87 % $1.segments) == 1) { //#7977
                var _89 = $k[--$j]; //#7975
                var _8A = $k[--$j]; //#7975
                $k[$j++] = $f(_89 + 1) * 12; //#7975
                $k[$j++] = _8A; //#7975
                $j--; //#7975
            } else { //#7977
                $j--; //#7977
            } //#7977
            var _8B = $k[--$j]; //#7979
            var _8C = $k[--$j]; //#7979
            $k[$j++] = $f(_8B - _8C); //#7979
        }; //#7980
        $1.encode = function() {
            var _8D = $k[--$j]; //#7983
            $k[$j++] = _8D; //#7983
            if ($ne(_8D, 'raw')) { //#7983
                var _8E = $k[--$j]; //#7983
                var _8G = $get(_8E, $k[--$j]); //#7983
                $k[$j++] = _8G; //#7983
            } else { //#7983
                $j--; //#7983
            } //#7983
            $k[$j++] = Infinity; //#7984
            var _8H = $k[--$j]; //#7984
            var _8I = $k[--$j]; //#7984
            $k[$j++] = _8H; //#7984
            $forall(_8I, function() { //#7984
                var _8J = $k[--$j]; //#7984
                $k[$j++] = $f(_8J - 48); //#7984
            }); //#7984
            var _8K = $a(); //#7984
            $puti($1.gpfenc, $1.j, _8K); //#7986
            $1.j = _8K.length + $1.j; //#7987
        }; //#7988
        $k[$j++] = Infinity; //#7991
        for (var _8P = 0, _8Q = $1.gpf.length; _8P < _8Q; _8P++) { //#7991
            $k[$j++] = 0; //#7991
        } //#7991
        $k[$j++] = 0; //#7991
        $k[$j++] = -1; //#7991
        $1.numericruns = $a(); //#7991
        $k[$j++] = Infinity; //#7992
        for (var _8T = 0, _8U = $1.gpf.length; _8T < _8U; _8T++) { //#7992
            $k[$j++] = 0; //#7992
        } //#7992
        $k[$j++] = 0; //#7992
        $1.alphanumericruns = $a(); //#7992
        $k[$j++] = Infinity; //#7993
        for (var _8X = 0, _8Y = $1.gpf.length; _8X < _8Y; _8X++) { //#7993
            $k[$j++] = 0; //#7993
        } //#7993
        $k[$j++] = 9999; //#7993
        $1.nextiso646only = $a(); //#7993
        for (var _8b = $1.gpf.length - 1; _8b >= 0; _8b -= 1) { //#8015
            $1.i = _8b; //#7995
            var _8g = $strcpy($s(2), "00"); //#7997
            var _8j = $get($1.gpf, $1.i); //#7998
            $k[$j++] = $get($1.gpf, $1.i); //#7998
            $k[$j++] = _8g; //#7998
            $k[$j++] = _8g; //#7998
            $k[$j++] = 0; //#7998
            $k[$j++] = _8j; //#7998
            if (_8j == $1.fnc1) { //#7998
                $j--; //#7998
                $k[$j++] = 94; //#7998
            } //#7998
            var _8l = $k[--$j]; //#7998
            var _8m = $k[--$j]; //#7998
            $put($k[--$j], _8m, _8l); //#7998
            if ($1.i < ($1.gpf.length - 1)) { //#7999
                var _8q = $k[--$j]; //#7999
                var _8t = $get($1.gpf, $1.i + 1); //#7999
                $k[$j++] = _8q; //#7999
                $k[$j++] = _8q; //#7999
                $k[$j++] = 1; //#7999
                $k[$j++] = _8t; //#7999
                if (_8t == $1.fnc1) { //#7999
                    $j--; //#7999
                    $k[$j++] = 94; //#7999
                } //#7999
                var _8v = $k[--$j]; //#7999
                var _8w = $k[--$j]; //#7999
                $put($k[--$j], _8w, _8v); //#7999
            } //#7999
            var _90 = $get($1.numeric, $k[--$j]) !== undefined; //#8000
            if (_90) { //#8003
                $put($1.numericruns, $1.i, $f($get($1.numericruns, $1.i + 2) + 2)); //#8001
            } else { //#8003
                $put($1.numericruns, $1.i, 0); //#8003
            } //#8003
            var _98 = $k[--$j]; //#8005
            var _9A = $get($1.alphanumeric, _98) !== undefined; //#8005
            $k[$j++] = _98; //#8009
            if (_9A) { //#8008
                $put($1.alphanumericruns, $1.i, $f($get($1.alphanumericruns, $1.i + 1) + 1)); //#8006
            } else { //#8008
                $put($1.alphanumericruns, $1.i, 0); //#8008
            } //#8008
            var _9I = $k[--$j]; //#8010
            var _9K = $get($1.iso646, _9I) !== undefined; //#8010
            var _9M = $get($1.alphanumeric, _9I) !== undefined; //#8010
            if (_9K && (!_9M)) { //#8013
                $put($1.nextiso646only, $1.i, 0); //#8011
            } else { //#8013
                $put($1.nextiso646only, $1.i, $f($get($1.nextiso646only, $1.i + 1) + 1)); //#8013
            } //#8013
        } //#8013
        $1.gpfenc = $a(252); //#8018
        $1.i = 0; //#8019
        $1.j = 0; //#8019
        $1.mode = "numeric"; //#8019
        for (;;) { //#8109
            if ($1.i == $1.gpf.length) { //#8021
                break; //#8021
            } //#8021
            for (;;) { //#8108
                if ($eq($1.mode, "numeric")) { //#8057
                    if ($1.i <= ($1.gpf.length - 2)) { //#8054
                        var _9a = $s(2); //#8026
                        var _9d = $get($1.gpf, $1.i); //#8027
                        $k[$j++] = _9a; //#8027
                        $k[$j++] = _9a; //#8027
                        $k[$j++] = 0; //#8027
                        $k[$j++] = _9d; //#8027
                        if (_9d == $1.fnc1) { //#8027
                            $j--; //#8027
                            $k[$j++] = 94; //#8027
                        } //#8027
                        var _9f = $k[--$j]; //#8027
                        var _9g = $k[--$j]; //#8027
                        $put($k[--$j], _9g, _9f); //#8027
                        var _9i = $k[--$j]; //#8028
                        var _9l = $get($1.gpf, $1.i + 1); //#8028
                        $k[$j++] = _9i; //#8028
                        $k[$j++] = _9i; //#8028
                        $k[$j++] = 1; //#8028
                        $k[$j++] = _9l; //#8028
                        if (_9l == $1.fnc1) { //#8028
                            $j--; //#8028
                            $k[$j++] = 94; //#8028
                        } //#8028
                        var _9n = $k[--$j]; //#8028
                        var _9o = $k[--$j]; //#8028
                        $put($k[--$j], _9o, _9n); //#8028
                        var _9q = $k[--$j]; //#8029
                        var _9s = $get($1.numeric, _9q) !== undefined; //#8029
                        $k[$j++] = _9q; //#8033
                        if (_9s) { //#8033
                            $k[$j++] = $1.numeric; //#8030
                            $1.encode(); //#8030
                            $1.i = $1.i + 2; //#8031
                            break; //#8032
                        } //#8032
                        $j--; //#8034
                        $k[$j++] = $1.lalphanumeric; //#8035
                        $k[$j++] = $1.numeric; //#8035
                        $1.encode(); //#8035
                        $1.mode = "alphanumeric"; //#8036
                        break; //#8037
                    } else { //#8054
                        var _9z = $get($1.gpf, $1.i); //#8039
                        if ((_9z < 48) || (_9z > 57)) { //#8043
                            $k[$j++] = $1.lalphanumeric; //#8040
                            $k[$j++] = $1.numeric; //#8040
                            $1.encode(); //#8040
                            $1.mode = "alphanumeric"; //#8041
                            break; //#8042
                        } //#8042
                        $k[$j++] = 'rem'; //#8044
                        $k[$j++] = ((((12 + 1) + $1.method.length) + $1.vlf.length) + $1.cdf.length) + $1.j; //#8044
                        $1.rembits(); //#8044
                        var _A6 = $k[--$j]; //#8044
                        $1[$k[--$j]] = _A6; //#8044
                        if (($1.rem >= 4) && ($1.rem <= 6)) { //#8054
                            var _AD = $geti($strcpy($s(6), "000000"), 0, $1.rem); //#8046
                            var _AI = $cvrs($s(4), $f($get($1.gpf, $1.i) - 47), 2); //#8047
                            $puti(_AD, 4 - _AI.length, _AI); //#8048
                            $k[$j++] = _AD; //#8048
                            $k[$j++] = 'raw'; //#8048
                            $1.encode(); //#8048
                            $1.i = $1.i + 1; //#8049
                            break; //#8050
                        } else { //#8054
                            var _AK = $s(2); //#8052
                            $put(_AK, 0, $get($1.gpf, $1.i)); //#8052
                            $put(_AK, 1, 94); //#8052
                            $k[$j++] = _AK; //#8052
                            $k[$j++] = $1.numeric; //#8052
                            $1.encode(); //#8052
                            $1.i = $1.i + 1; //#8053
                            break; //#8054
                        } //#8054
                    } //#8054
                } //#8054
                if ($eq($1.mode, "alphanumeric")) { //#8084
                    if ($get($1.gpf, $1.i) == $1.fnc1) { //#8065
                        $k[$j++] = $1.fnc1; //#8061
                        $k[$j++] = $1.alphanumeric; //#8061
                        $1.encode(); //#8061
                        $1.mode = "numeric"; //#8062
                        $1.i = $1.i + 1; //#8063
                        break; //#8064
                    } //#8064
                    var _Aa = $get($1.gpf, $1.i); //#8066
                    var _Ac = $get($1.iso646, _Aa) !== undefined; //#8066
                    var _Ae = $get($1.alphanumeric, _Aa) !== undefined; //#8066
                    if (_Ac && (!_Ae)) { //#8070
                        $k[$j++] = $1.liso646; //#8067
                        $k[$j++] = $1.alphanumeric; //#8067
                        $1.encode(); //#8067
                        $1.mode = "iso646"; //#8068
                        break; //#8069
                    } //#8069
                    if ($get($1.numericruns, $1.i) >= 6) { //#8075
                        $k[$j++] = $1.lnumeric; //#8072
                        $k[$j++] = $1.alphanumeric; //#8072
                        $1.encode(); //#8072
                        $1.mode = "numeric"; //#8073
                        break; //#8074
                    } //#8074
                    var _Ao = $get($1.numericruns, $1.i); //#8076
                    if ((_Ao >= 4) && ($f(_Ao + $1.i) == $1.gpf.length)) { //#8080
                        $k[$j++] = $1.lnumeric; //#8077
                        $k[$j++] = $1.alphanumeric; //#8077
                        $1.encode(); //#8077
                        $1.mode = "numeric"; //#8078
                        break; //#8079
                    } //#8079
                    $k[$j++] = $get($1.gpf, $1.i); //#8081
                    $k[$j++] = $1.alphanumeric; //#8081
                    $1.encode(); //#8081
                    $1.i = $1.i + 1; //#8082
                    break; //#8083
                } //#8083
                if ($eq($1.mode, "iso646")) { //#8106
                    if ($get($1.gpf, $1.i) == $1.fnc1) { //#8092
                        $k[$j++] = $1.fnc1; //#8088
                        $k[$j++] = $1.iso646; //#8088
                        $1.encode(); //#8088
                        $1.mode = "numeric"; //#8089
                        $1.i = $1.i + 1; //#8090
                        break; //#8091
                    } //#8091
                    if (($get($1.numericruns, $1.i) >= 4) && ($get($1.nextiso646only, $1.i) >= 10)) { //#8097
                        $k[$j++] = $1.lnumeric; //#8094
                        $k[$j++] = $1.iso646; //#8094
                        $1.encode(); //#8094
                        $1.mode = "numeric"; //#8095
                        break; //#8096
                    } //#8096
                    if (($get($1.alphanumericruns, $1.i) >= 5) && ($get($1.nextiso646only, $1.i) >= 10)) { //#8102
                        $k[$j++] = $1.lalphanumeric; //#8099
                        $k[$j++] = $1.iso646; //#8099
                        $1.encode(); //#8099
                        $1.mode = "alphanumeric"; //#8100
                        break; //#8101
                    } //#8101
                    $k[$j++] = $get($1.gpf, $1.i); //#8103
                    $k[$j++] = $1.iso646; //#8103
                    $1.encode(); //#8103
                    $1.i = $1.i + 1; //#8104
                    break; //#8105
                } //#8105
            } //#8105
        } //#8105
        $1.gpf = $geti($1.gpfenc, 0, $1.j); //#8110
        var _BY = ((((1 + 12) + $1.method.length) + $1.vlf.length) + $1.cdf.length) + $1.gpf.length; //#8114
        $k[$j++] = _BY; //#8114
        $k[$j++] = _BY; //#8114
        $1.rembits(); //#8114
        var _BZ = $k[--$j]; //#8114
        $1.pad = $a(_BZ); //#8114
        $k[$j++] = _BZ; //#8121
        if ($1.vlf.length != 0) { //#8120
            var _Bc = $k[--$j]; //#8116
            var _Be = ~~($f($k[--$j] + _Bc) / 12); //#8117
            $put($1.vlf, 0, _Be % 2); //#8117
            var _Bg = (_Be <= 14) ? 0 : 1; //#8118
            $put($1.vlf, 1, _Bg); //#8118
        } else { //#8120
            $j -= 2; //#8120
        } //#8120
        if ($1.pad.length > 0) { //#8130
            for (var _Bl = 0, _Bk = $1.pad.length - 1; _Bl <= _Bk; _Bl += 5) { //#8126
                $1.i = _Bl; //#8124
                var _Bm = $1.pad; //#8125
                var _Bn = $1.i; //#8125
                var _Bo = $a([0, 0, 1, 0, 0]); //#8125
                var _Bp = $1.pad; //#8125
                var _Bq = $1.i; //#8125
                var _Br = 5; //#8125
                var _Bs = _Bp.length - _Bq; //#8125
                if ((_Bp.length - _Bq) > 5) { //#8125
                    var _ = _Br; //#8125
                    _Br = _Bs; //#8125
                    _Bs = _; //#8125
                } //#8125
                $puti(_Bm, _Bn, $geti(_Bo, 0, _Bs)); //#8125
            } //#8125
            if ($eq($1.mode, "numeric")) { //#8129
                $k[$j++] = Infinity; //#8128
                $k[$j++] = 0; //#8128
                $k[$j++] = 0; //#8128
                $k[$j++] = 0; //#8128
                $k[$j++] = 0; //#8128
                $aload($1.pad); //#8128
                $1.pad = $geti($a(), 0, $1.pad.length); //#8128
            } //#8128
        } //#8128
        $k[$j++] = Infinity; //#8139
        var _C0 = $1.linkage ? 1 : 0; //#8134
        $k[$j++] = _C0; //#8135
        $forall($1.method, function() { //#8135
            var _C2 = $k[--$j]; //#8135
            $k[$j++] = $f(_C2 - 48); //#8135
        }); //#8135
        $aload($1.vlf); //#8136
        $aload($1.cdf); //#8137
        $aload($1.gpf); //#8138
        $aload($1.pad); //#8139
        $1.binval = $a(); //#8139
        $1.datalen = ~~($1.binval.length / 12); //#8141
        $1.ncr = function() {
            var _C9 = $k[--$j]; //#8144
            var _CA = $k[--$j]; //#8144
            var _CB = $f(_CA - _C9); //#8144
            if (_C9 < $f(_CA - _C9)) { //#8144
                var _ = _CB; //#8144
                _CB = _C9; //#8144
                _C9 = _; //#8144
            } //#8144
            $k[$j++] = _CB; //#8151
            $k[$j++] = 1; //#8151
            $k[$j++] = 1; //#8151
            for (var _CD = _CA, _CC = $f(_C9 + 1); _CD >= _CC; _CD -= 1) { //#8151
                var _CE = $k[--$j]; //#8147
                var _CF = $k[--$j]; //#8148
                var _CG = $k[--$j]; //#8148
                $k[$j++] = _CG; //#8150
                $k[$j++] = _CF; //#8150
                $k[$j++] = _CE * _CD; //#8150
                if ($le(_CF, _CG)) { //#8150
                    var _CH = $k[--$j]; //#8149
                    var _CI = $k[--$j]; //#8149
                    $k[$j++] = $f(_CI + 1); //#8149
                    $k[$j++] = ~~(_CH / _CI); //#8149
                } //#8149
            } //#8149
            for (;;) { //#8155
                var _CJ = $k[--$j]; //#8153
                var _CK = $k[--$j]; //#8153
                var _CL = $k[--$j]; //#8153
                $k[$j++] = _CL; //#8153
                $k[$j++] = _CK; //#8153
                $k[$j++] = _CJ; //#8153
                if ($gt(_CK, _CL)) { //#8153
                    break; //#8153
                } //#8153
                var _CM = $k[--$j]; //#8154
                var _CN = $k[--$j]; //#8154
                $k[$j++] = $f(_CN + 1); //#8154
                $k[$j++] = ~~(_CM / _CN); //#8154
            } //#8154
            var _CO = $k[--$j]; //#8156
            var _CP = $k[--$j]; //#8156
            $k[$j++] = _CO; //#8156
            $k[$j++] = _CP; //#8156
            $j--; //#8156
            var _CQ = $k[--$j]; //#8156
            var _CR = $k[--$j]; //#8156
            $k[$j++] = _CQ; //#8156
            $k[$j++] = _CR; //#8156
            $j--; //#8156
        }; //#8157
        $1.getRSSwidths = function() {
            $1.oe = $k[--$j]; //#8160
            $1.el = $k[--$j]; //#8161
            $1.mw = $k[--$j]; //#8162
            $1.nm = $k[--$j]; //#8163
            $1.val = $k[--$j]; //#8164
            $1.out = $a($1.el); //#8165
            $1.mask = 0; //#8166
            for (var _Cb = 0, _Ca = $f($1.el - 2); _Cb <= _Ca; _Cb += 1) { //#8193
                $1.bar = _Cb; //#8168
                $1.ew = 1; //#8169
                var _Cd = $1.bar; //#8170
                $1.mask = $1.mask | ((_Cd < 0 ? 1 >>> -_Cd : 1 << _Cd)); //#8170
                for (;;) { //#8189
                    $k[$j++] = 'sval'; //#8171
                    $k[$j++] = $f($f($1.nm - $1.ew) - 1); //#8171
                    $k[$j++] = $f($f($1.el - $1.bar) - 2); //#8171
                    $1.ncr(); //#8171
                    var _Ci = $k[--$j]; //#8171
                    $1[$k[--$j]] = _Ci; //#8171
                    if (($1.oe && ($1.mask == 0)) && (($f(($f($f($1.nm - $1.ew) - ($1.el * 2))) + ($1.bar * 2))) >= -2)) { //#8174
                        $k[$j++] = 'sval'; //#8173
                        $k[$j++] = $1.sval; //#8173
                        $k[$j++] = $f(($f($f($1.nm - $1.ew) - $1.el)) + $1.bar); //#8173
                        $k[$j++] = $f($f($1.el - $1.bar) - 2); //#8173
                        $1.ncr(); //#8173
                        var _Cx = $k[--$j]; //#8173
                        var _Cy = $k[--$j]; //#8173
                        $1[$k[--$j]] = $f(_Cy - _Cx); //#8173
                    } //#8173
                    if ($f($1.el - $1.bar) > 2) { //#8183
                        $1.lval = 0; //#8176
                        for (var _D8 = $f(($f(($f($f($1.nm - $1.ew) - $1.el)) + $1.bar)) + 2), _D7 = $f($1.mw + 1); _D8 >= _D7; _D8 -= 1) { //#8180
                            $k[$j++] = $f(($f($f($1.nm - _D8) - $1.ew)) - 1); //#8178
                            $k[$j++] = $f($f($1.el - $1.bar) - 3); //#8178
                            $1.ncr(); //#8178
                            $1.lval = $f($k[--$j] + $1.lval); //#8179
                        } //#8179
                        $1.sval = $f($1.sval - ($1.lval * ($f($f($1.el - $1.bar) - 1)))); //#8181
                    } else { //#8183
                        if ($f($1.nm - $1.ew) > $1.mw) { //#8183
                            $1.sval = $f($1.sval - 1); //#8183
                        } //#8183
                    } //#8183
                    $1.val = $f($1.val - $1.sval); //#8185
                    if ($1.val < 0) { //#8186
                        break; //#8186
                    } //#8186
                    $1.ew = $1.ew + 1; //#8187
                    var _DS = $1.bar; //#8188
                    $1.mask = $1.mask & (~((_DS < 0 ? 1 >>> -_DS : 1 << _DS))); //#8188
                } //#8188
                $1.val = $f($1.val + $1.sval); //#8190
                $1.nm = $f($1.nm - $1.ew); //#8191
                $put($1.out, $1.bar, $1.ew); //#8192
            } //#8192
            $put($1.out, $f($1.el - 1), $1.nm); //#8194
            $k[$j++] = $1.out; //#8195
        }; //#8196
        $1.tab174 = $a([347, 0, 12, 5, 7, 2, 87, 4, 1387, 348, 10, 7, 5, 4, 52, 20, 2947, 1388, 8, 9, 4, 5, 30, 52, 3987, 2948, 6, 11, 3, 6, 10, 104, 4191, 3988, 4, 13, 1, 8, 1, 204]); //#8204
        $1.dxw = $a($1.datalen); //#8206
        for (var _Dj = 0, _Di = $1.datalen - 1; _Dj <= _Di; _Dj += 1) { //#8247
            $1.x = _Dj; //#8210
            $1.d = $geti($1.binval, $1.x * 12, 12); //#8212
            $k[$j++] = 'd'; //#8213
            $k[$j++] = 0; //#8213
            for (var _Dn = 0; _Dn <= 11; _Dn += 1) { //#8213
                $1.j = _Dn; //#8213
                var _Ds = $k[--$j]; //#8213
                $k[$j++] = $f(_Ds + ((~~Math.pow(2, 11 - $1.j)) * $get($1.d, $1.j))); //#8213
            } //#8213
            var _Dt = $k[--$j]; //#8213
            $1[$k[--$j]] = _Dt; //#8213
            $1.j = 0; //#8215
            for (;;) { //#8225
                if ($le($1.d, $get($1.tab174, $1.j))) { //#8223
                    var _E1 = $geti($1.tab174, $1.j + 1, 7); //#8217
                    for (var _E2 = 0, _E3 = _E1.length; _E2 < _E3; _E2++) { //#8217
                        $k[$j++] = $get(_E1, _E2); //#8217
                    } //#8217
                    $1.dte = $k[--$j]; //#8218
                    $1.dto = $k[--$j]; //#8218
                    $1.dmwe = $k[--$j]; //#8219
                    $1.dmwo = $k[--$j]; //#8219
                    $1.dele = $k[--$j]; //#8220
                    $1.delo = $k[--$j]; //#8220
                    $1.dgs = $k[--$j]; //#8221
                    break; //#8222
                } //#8222
                $1.j = $1.j + 8; //#8224
            } //#8224
            $k[$j++] = 'dwo'; //#8227
            $k[$j++] = ~~($f($1.d - $1.dgs) / $1.dte); //#8227
            $k[$j++] = $1.delo; //#8227
            $k[$j++] = $1.dmwo; //#8227
            $k[$j++] = 4; //#8227
            $k[$j++] = true; //#8227
            $1.getRSSwidths(); //#8227
            var _EI = $k[--$j]; //#8227
            $1[$k[--$j]] = _EI; //#8227
            $k[$j++] = 'dwe'; //#8228
            $k[$j++] = $f($1.d - $1.dgs) % $1.dte; //#8228
            $k[$j++] = $1.dele; //#8228
            $k[$j++] = $1.dmwe; //#8228
            $k[$j++] = 4; //#8228
            $k[$j++] = false; //#8228
            $1.getRSSwidths(); //#8228
            var _EP = $k[--$j]; //#8228
            $1[$k[--$j]] = _EP; //#8228
            $1.dw = $a(8); //#8230
            if (($1.x % 2) == 0) { //#8241
                for (var _ET = 0; _ET <= 3; _ET += 1) { //#8236
                    $1.j = _ET; //#8233
                    $put($1.dw, 7 - ($1.j * 2), $get($1.dwo, $1.j)); //#8234
                    $put($1.dw, 6 - ($1.j * 2), $get($1.dwe, $1.j)); //#8235
                } //#8235
            } else { //#8241
                for (var _Ee = 0; _Ee <= 3; _Ee += 1) { //#8242
                    $1.j = _Ee; //#8239
                    $put($1.dw, $1.j * 2, $get($1.dwo, $1.j)); //#8240
                    $put($1.dw, ($1.j * 2) + 1, $get($1.dwe, $1.j)); //#8241
                } //#8241
            } //#8241
            $put($1.dxw, $1.x, $1.dw); //#8245
        } //#8245
        $1.finderwidths = $a([1, 8, 4, 1, 1, 1, 1, 4, 8, 1, 3, 6, 4, 1, 1, 1, 1, 4, 6, 3, 3, 4, 6, 1, 1, 1, 1, 6, 4, 3, 3, 2, 8, 1, 1, 1, 1, 8, 2, 3, 2, 6, 5, 1, 1, 1, 1, 5, 6, 2, 2, 2, 9, 1, 1, 1, 1, 9, 2, 2]); //#8256
        $1.finderseq = $a([$a([0, 1]), $a([0, 3, 2]), $a([0, 5, 2, 7]), $a([0, 9, 2, 7, 4]), $a([0, 9, 2, 7, 6, 11]), $a([0, 9, 2, 7, 8, 11, 10]), $a([0, 1, 2, 3, 4, 5, 6, 7]), $a([0, 1, 2, 3, 4, 5, 6, 9, 8]), $a([0, 1, 2, 3, 4, 5, 6, 9, 10, 11]), $a([0, 1, 2, 3, 4, 7, 6, 9, 8, 11, 10])]); //#8269
        $1.seq = $get($1.finderseq, ~~(($1.datalen - 2) / 2)); //#8271
        $1.fxw = $a($1.seq.length); //#8272
        for (var _FB = 0, _FA = $1.seq.length - 1; _FB <= _FA; _FB += 1) { //#8276
            $1.x = _FB; //#8274
            $put($1.fxw, $1.x, $geti($1.finderwidths, $get($1.seq, $1.x) * 5, 5)); //#8275
        } //#8275
        $1.checkweights = $a([-1, -1, -1, -1, -1, -1, -1, -1, 77, 96, 32, 81, 27, 9, 3, 1, 20, 60, 180, 118, 143, 7, 21, 63, 205, 209, 140, 117, 39, 13, 145, 189, 193, 157, 49, 147, 19, 57, 171, 91, 132, 44, 85, 169, 197, 136, 186, 62, 185, 133, 188, 142, 4, 12, 36, 108, 50, 87, 29, 80, 97, 173, 128, 113, 150, 28, 84, 41, 123, 158, 52, 156, 166, 196, 206, 139, 187, 203, 138, 46, 76, 17, 51, 153, 37, 111, 122, 155, 146, 119, 110, 107, 106, 176, 129, 43, 16, 48, 144, 10, 30, 90, 59, 177, 164, 125, 112, 178, 200, 137, 116, 109, 70, 210, 208, 202, 184, 130, 179, 115, 190, 204, 68, 93, 31, 151, 191, 134, 148, 22, 66, 198, 172, 94, 71, 2, 40, 154, 192, 64, 162, 54, 18, 6, 120, 149, 25, 75, 14, 42, 126, 167, 175, 199, 207, 69, 23, 78, 26, 79, 103, 98, 83, 38, 114, 131, 182, 124, 159, 53, 88, 170, 127, 183, 61, 161, 55, 165, 73, 8, 24, 72, 5, 15, 89, 100, 174, 58, 160, 194, 135, 45]); //#8303
        $k[$j++] = Infinity; //#8306
        $forall($1.seq, function() { //#8306
            $aload($geti($1.checkweights, $k[--$j] * 16, 16)); //#8306
        }); //#8306
        var _FO = $a(); //#8306
        $1.checkweightseq = $geti(_FO, 8, _FO.length - 8); //#8307
        $k[$j++] = Infinity; //#8310
        var _FQ = $1.dxw; //#8310
        for (var _FR = 0, _FS = _FQ.length; _FR < _FS; _FR++) { //#8310
            $forall($get(_FQ, _FR)); //#8310
        } //#8310
        $1.widths = $a(); //#8310
        $1.checksum = 0; //#8313
        for (var _FX = 0, _FW = $1.widths.length - 1; _FX <= _FW; _FX += 1) { //#8317
            $1.i = _FX; //#8315
            $1.checksum = $f($1.checksum + ($get($1.widths, $1.i) * $get($1.checkweightseq, $1.i))); //#8316
        } //#8316
        $1.checksum = $f(($1.checksum % 211) + (($1.datalen - 3) * 211)); //#8318
        $1.i = 0; //#8320
        for (;;) { //#8330
            if ($1.checksum <= $get($1.tab174, $1.i)) { //#8328
                var _Fn = $geti($1.tab174, $1.i + 1, 7); //#8322
                for (var _Fo = 0, _Fp = _Fn.length; _Fo < _Fp; _Fo++) { //#8322
                    $k[$j++] = $get(_Fn, _Fo); //#8322
                } //#8322
                $1.cte = $k[--$j]; //#8323
                $1.cto = $k[--$j]; //#8323
                $1.cmwe = $k[--$j]; //#8324
                $1.cmwo = $k[--$j]; //#8324
                $1.cele = $k[--$j]; //#8325
                $1.celo = $k[--$j]; //#8325
                $1.cgs = $k[--$j]; //#8326
                break; //#8327
            } //#8327
            $1.i = $1.i + 8; //#8329
        } //#8329
        $k[$j++] = 'cwo'; //#8332
        $k[$j++] = ~~($f($1.checksum - $1.cgs) / $1.cte); //#8332
        $k[$j++] = $1.celo; //#8332
        $k[$j++] = $1.cmwo; //#8332
        $k[$j++] = 4; //#8332
        $k[$j++] = true; //#8332
        $1.getRSSwidths(); //#8332
        var _G4 = $k[--$j]; //#8332
        $1[$k[--$j]] = _G4; //#8332
        $k[$j++] = 'cwe'; //#8333
        $k[$j++] = $f($1.checksum - $1.cgs) % $1.cte; //#8333
        $k[$j++] = $1.cele; //#8333
        $k[$j++] = $1.cmwe; //#8333
        $k[$j++] = 4; //#8333
        $k[$j++] = false; //#8333
        $1.getRSSwidths(); //#8333
        var _GB = $k[--$j]; //#8333
        $1[$k[--$j]] = _GB; //#8333
        $1.cw = $a(8); //#8335
        for (var _GE = 0; _GE <= 3; _GE += 1) { //#8340
            $1.i = _GE; //#8337
            $put($1.cw, $1.i * 2, $get($1.cwo, $1.i)); //#8338
            $put($1.cw, ($1.i * 2) + 1, $get($1.cwe, $1.i)); //#8339
        } //#8339
        var _GP = $a(22); //#8342
        $put(_GP, 0, $1.cw); //#8343
        $puti(_GP, 1, $1.dxw); //#8343
        $1.dxw = $geti(_GP, 0, $1.datalen + 1); //#8344
        $1.datalen = $1.dxw.length; //#8345
        $1.rows = $a(~~Math.ceil($1.datalen / $1.segments)); //#8348
        $1.numrows = $1.rows.length; //#8349
        for (var _Gb = 0, _Ga = $1.numrows - 1; _Gb <= _Ga; _Gb += 1) { //#8364
            $1.r = _Gb; //#8351
            $k[$j++] = Infinity; //#8352
            if ((($1.segments % 4) != 0) && (($1.r % 2) == 1)) { //#8353
                $k[$j++] = 0; //#8353
            } //#8353
            $k[$j++] = 1; //#8361
            $k[$j++] = 1; //#8361
            for (var _Gg = 0, _Gf = $1.segments - 1; _Gg <= _Gf; _Gg += 1) { //#8361
                $1.pos = _Gg + ($1.r * $1.segments); //#8356
                if ($1.pos < $1.datalen) { //#8360
                    $forall($get($1.dxw, $1.pos)); //#8358
                    if (($1.pos % 2) == 0) { //#8359
                        $forall($get($1.fxw, ~~($1.pos / 2))); //#8359
                    } //#8359
                } //#8359
            } //#8359
            var _Gs = $counttomark() + 2; //#8363
            $k[$j++] = 1; //#8363
            $k[$j++] = 1; //#8363
            $astore($a(_Gs)); //#8363
            $put($1.rows, $1.r, $k[--$j]); //#8363
            $j--; //#8363
        } //#8363
        if ($ne($1.format, "expandedstacked")) { //#8492
            var _Gz = $get($1.rows, 0); //#8369
            $1.sbs = $geti(_Gz, 1, _Gz.length - 1); //#8369
            $k[$j++] = Infinity; //#8378
            $k[$j++] = 1; //#8377
            for (var _H3 = 0, _H2 = $1.datalen - 1; _H3 <= _H2; _H3 += 1) { //#8377
                $1.i = _H3; //#8374
                $forall($get($1.dxw, $1.i)); //#8375
                if (($1.i % 2) == 0) { //#8376
                    $forall($get($1.fxw, ~~($1.i / 2))); //#8376
                } //#8376
            } //#8376
            $k[$j++] = 1; //#8378
            $k[$j++] = 1; //#8378
            $1.sbs = $a(); //#8378
            delete $1.options["parse"]; //#8381
            $k[$j++] = Infinity; //#8388
            var _HD = $1.sbs; //#8385
            $k[$j++] = Infinity; //#8386
            for (var _HF = 0, _HG = ~~(($1.sbs.length + 1) / 2); _HF < _HG; _HF++) { //#8386
                $k[$j++] = $1.height; //#8386
            } //#8386
            var _HI = $a(); //#8386
            $k[$j++] = Infinity; //#8387
            for (var _HK = 0, _HL = ~~(($1.sbs.length + 1) / 2); _HK < _HL; _HK++) { //#8387
                $k[$j++] = 0; //#8387
            } //#8387
            var _HM = $a(); //#8387
            $k[$j++] = 'ren'; //#8388
            $k[$j++] = $0.renlinear; //#8388
            $k[$j++] = 'sbs'; //#8388
            $k[$j++] = _HD; //#8388
            $k[$j++] = 'bhs'; //#8388
            $k[$j++] = _HI; //#8388
            $k[$j++] = 'bbs'; //#8388
            $k[$j++] = _HM; //#8388
            $k[$j++] = 'opt'; //#8388
            $k[$j++] = $1.options; //#8388
            var _HO = $d(); //#8388
            $k[$j++] = _HO; //#8391
            if (!$1.dontdraw) { //#8391
                $0.renlinear(); //#8391
            } //#8391
        } else { //#8492
            $1.seps = $a($1.numrows); //#8395
            for (var _HU = 0, _HT = $1.numrows - 1; _HU <= _HT; _HU += 1) { //#8451
                $1.r = _HU; //#8398
                $1.row = $get($1.rows, $1.r); //#8399
                $k[$j++] = Infinity; //#8402
                for (var _Ha = 0, _HZ = $1.row.length - 1; _Ha <= _HZ; _Ha += 2) { //#8407
                    $1.i = _Ha; //#8404
                    for (var _He = 0, _Hf = $get($1.row, $1.i); _He < _Hf; _He++) { //#8405
                        $k[$j++] = 0; //#8405
                    } //#8405
                    if ($1.i < ($1.row.length - 1)) { //#8406
                        for (var _Hl = 0, _Hm = $get($1.row, $1.i + 1); _Hl < _Hm; _Hl++) { //#8406
                            $k[$j++] = 1; //#8406
                        } //#8406
                    } //#8406
                } //#8406
                $astore($a($counttomark())); //#8408
                $1.row = $k[--$j]; //#8408
                $j--; //#8408
                $k[$j++] = Infinity; //#8411
                $forall($1.row, function() { //#8411
                    var _Hr = $k[--$j]; //#8411
                    $k[$j++] = $f(1 - _Hr); //#8411
                }); //#8411
                $1.sep = $a(); //#8411
                $k[$j++] = Infinity; //#8414
                for (var _Hv = 19, _Hu = $1.row.length - 13; _Hv <= _Hu; _Hv += 98) { //#8413
                    $k[$j++] = _Hv; //#8413
                } //#8413
                for (var _Hy = 68, _Hx = $1.row.length - 13; _Hy <= _Hx; _Hy += 98) { //#8414
                    $k[$j++] = _Hy; //#8414
                } //#8414
                $1.finderpos = $a(); //#8414
                var _I0 = $1.finderpos; //#8416
                for (var _I1 = 0, _I2 = _I0.length; _I1 < _I2; _I1++) { //#8430
                    var _I3 = $get(_I0, _I1); //#8430
                    for (var _I5 = _I3, _I4 = $f(_I3 + 14); _I5 <= _I4; _I5 += 1) { //#8429
                        $1.i = _I5; //#8418
                        if ($get($1.row, $1.i) == 0) { //#8426
                            if ($get($1.row, $1.i - 1) == 1) { //#8423
                                $k[$j++] = 1; //#8421
                            } else { //#8423
                                var _IF = ($get($1.sep, $1.i - 1) == 0) ? 1 : 0; //#8423
                                $k[$j++] = _IF; //#8423
                            } //#8423
                        } else { //#8426
                            $k[$j++] = 0; //#8426
                        } //#8426
                        $put($1.sep, $1.i, $k[--$j]); //#8428
                    } //#8428
                } //#8428
                $puti($1.sep, 0, $a([0, 0, 0, 0])); //#8431
                $puti($1.sep, $1.row.length - 4, $a([0, 0, 0, 0])); //#8432
                if ((($1.segments % 4) == 0) && (($1.r % 2) == 1)) { //#8446
                    if (($get($1.rows, $1.r).length != $get($1.rows, 0).length) && (($1.finderpos.length % 2) == 1)) { //#8444
                        $k[$j++] = Infinity; //#8438
                        $k[$j++] = 0; //#8438
                        $aload($1.row); //#8438
                        $1.row = $a(); //#8438
                        $k[$j++] = Infinity; //#8439
                        $k[$j++] = 0; //#8439
                        $aload($1.sep); //#8439
                        $1.sep = $a(); //#8439
                    } else { //#8444
                        for (var _Ib = $1.row.length - 1; _Ib >= 0; _Ib -= 1) { //#8441
                            $k[$j++] = $get($1.row, _Ib); //#8441
                        } //#8441
                        $astore($1.row); //#8442
                        $j--; //#8442
                        for (var _Ig = $1.sep.length - 1; _Ig >= 0; _Ig -= 1) { //#8443
                            $k[$j++] = $get($1.sep, _Ig); //#8443
                        } //#8443
                        $astore($1.sep); //#8444
                        $j--; //#8444
                    } //#8444
                } //#8444
                $put($1.rows, $1.r, $1.row); //#8448
                $put($1.seps, $1.r, $1.sep); //#8449
            } //#8449
            $1.pixx = $get($1.rows, 0).length; //#8454
            $k[$j++] = Infinity; //#8455
            for (var _It = 0, _Iu = $1.pixx; _It < _Iu; _It++) { //#8455
                $k[$j++] = 0; //#8455
            } //#8455
            var _Iv = $a(); //#8455
            $puti(_Iv, 0, $get($1.rows, $1.numrows - 1)); //#8455
            $put($1.rows, $1.numrows - 1, _Iv); //#8456
            $k[$j++] = Infinity; //#8457
            for (var _J2 = 0, _J3 = $1.pixx; _J2 < _J3; _J2++) { //#8457
                $k[$j++] = 0; //#8457
            } //#8457
            var _J4 = $a(); //#8457
            $puti(_J4, 0, $get($1.seps, $1.numrows - 1)); //#8457
            $put($1.seps, $1.numrows - 1, _J4); //#8458
            $k[$j++] = Infinity; //#8461
            for (var _JB = 0, _JC = (~~($1.pixx / 2)) + 1; _JB < _JC; _JB++) { //#8461
                $k[$j++] = 0; //#8461
                $k[$j++] = 1; //#8461
            } //#8461
            $1.sep = $geti($a(), 0, $1.pixx); //#8461
            $puti($1.sep, 0, $a([0, 0, 0, 0])); //#8462
            $puti($1.sep, $1.pixx - 4, $a([0, 0, 0, 0])); //#8463
            $k[$j++] = Infinity; //#8475
            for (var _JN = 0, _JM = $1.numrows - 1; _JN <= _JM; _JN += 1) { //#8477
                $1.r = _JN; //#8468
                if ($1.r != 0) { //#8471
                    $aload($get($1.seps, $1.r)); //#8470
                } //#8470
                for (var _JT = 0, _JU = $1.barxmult; _JT < _JU; _JT++) { //#8472
                    $aload($get($1.rows, $1.r)); //#8472
                } //#8472
                if ($1.r != ($1.numrows - 1)) { //#8476
                    $aload($get($1.seps, $1.r)); //#8474
                    $aload($1.sep); //#8475
                } //#8475
            } //#8475
            $1.pixs = $a(); //#8475
            delete $1.options["parse"]; //#8480
            var _Jo = new Map([
                ["ren", $0.renmatrix],
                ["pixs", $1.pixs],
                ["pixx", $1.pixx],
                ["pixy", ~~($1.pixs.length / $1.pixx)],
                ["height", (~~($1.pixs.length / $1.pixx)) / 72],
                ["width", $1.pixx / 72],
                ["opt", $1.options]
            ]); //#8489
            $k[$j++] = _Jo; //#8492
            if (!$1.dontdraw) { //#8492
                $0.renmatrix(); //#8492
            } //#8492
        } //#8492
    };
    $0.databarexpandedstacked = function() {
        var $1 = {}; //#8524
        $1.options = $k[--$j]; //#8526
        $1.barcode = $k[--$j]; //#8527
        $1.dontdraw = false; //#8529
        $forall($1.options, function() { //#8540
            var _3 = $k[--$j]; //#8540
            $1[$k[--$j]] = _3; //#8540
        }); //#8540
        $put($1.options, "dontdraw", true); //#8543
        $put($1.options, "format", "expandedstacked"); //#8544
        $k[$j++] = 'args'; //#8546
        $k[$j++] = $1.barcode; //#8546
        $k[$j++] = $1.options; //#8546
        $0.databarexpanded(); //#8546
        var _9 = $k[--$j]; //#8546
        $1[$k[--$j]] = _9; //#8546
        $put($1.args, "opt", $1.options); //#8548
        $k[$j++] = $1.args; //#8551
        if (!$1.dontdraw) { //#8551
            $0.renmatrix(); //#8551
        } //#8551
    };
    $0.gs1northamericancoupon = function() {
        var $1 = {}; //#8583
        $1.options = $k[--$j]; //#8585
        $1.barcode = $k[--$j]; //#8586
        $1.dontdraw = false; //#8588
        $1.includetext = false; //#8590
        $1.coupontextfont = "OCR-B"; //#8591
        $1.coupontextsize = 9; //#8592
        $1.coupontextxoffset = "unset"; //#8593
        $1.coupontextyoffset = "unset"; //#8594
        $1.parse = false; //#8595
        $1.dontlint = false; //#8596
        $forall($1.options, function() { //#8607
            var _3 = $k[--$j]; //#8607
            $1[$k[--$j]] = _3; //#8607
        }); //#8607
        $1.coupontextfont = "" + $1.coupontextfont; //#8609
        $1.coupontextsize = +$1.coupontextsize; //#8610
        if ($ne($1.coupontextxoffset, "unset")) { //#8611
            $1.coupontextxoffset = +$1.coupontextxoffset; //#8611
        } //#8611
        if ($ne($1.coupontextyoffset, "unset")) { //#8612
            $1.coupontextyoffset = +$1.coupontextyoffset; //#8612
        } //#8612
        $1.expand = function() {
            var _C = new Map([
                ["parse", $1.parse],
                ["parseonly", true],
                ["parsefnc", false]
            ]); //#8619
            $1.fncvals = _C; //#8620
            $k[$j++] = $1.fncvals; //#8621
            $0.parseinput(); //#8621
        }; //#8622
        $1.ais = $a([]); //#8625
        $1.vals = $a([]); //#8626
        var _G = $1.barcode; //#8627
        $k[$j++] = $geti(_G, 1, _G.length - 1); //#8640
        for (;;) { //#8640
            var _I = $k[--$j]; //#8629
            $k[$j++] = _I; //#8629
            if ($eq(_I, "")) { //#8629
                break; //#8629
            } //#8629
            $search($k[--$j], ")"); //#8630
            $j--; //#8630
            var _K = $k[--$j]; //#8631
            var _L = $k[--$j]; //#8631
            $k[$j++] = _K; //#8631
            $k[$j++] = _L; //#8631
            $j--; //#8631
            var _M = $k[--$j]; //#8632
            var _N = $k[--$j]; //#8632
            $k[$j++] = _M; //#8632
            $search(_N, "("); //#8632
            if ($k[--$j]) { //#8635
                var _P = $k[--$j]; //#8633
                var _Q = $k[--$j]; //#8633
                $k[$j++] = _P; //#8633
                $k[$j++] = _Q; //#8633
                $j--; //#8633
                var _R = $k[--$j]; //#8633
                var _S = $k[--$j]; //#8633
                var _T = $k[--$j]; //#8633
                $k[$j++] = _S; //#8633
                $k[$j++] = _T; //#8633
                $k[$j++] = _R; //#8633
            } else { //#8635
                var _U = $k[--$j]; //#8635
                var _V = $k[--$j]; //#8635
                $k[$j++] = ""; //#8635
                $k[$j++] = _V; //#8635
                $k[$j++] = _U; //#8635
            } //#8635
            $k[$j++] = Infinity; //#8637
            $aload($1.ais); //#8637
            var _Y = $k[$j - 1 - ($counttomark() + 2)]; //#8637
            $k[$j++] = _Y; //#8637
            $1.ais = $a(); //#8637
            $k[$j++] = Infinity; //#8638
            $aload($1.vals); //#8638
            var _c = $k[$j - 1 - ($counttomark() + 1)]; //#8638
            $k[$j++] = _c; //#8638
            $1.expand(); //#8638
            $1.vals = $a(); //#8638
            $j -= 2; //#8639
        } //#8639
        $j--; //#8641
        if ($1.ais.length != 1) { //#8646
            $k[$j++] = 'bwipp.gs1northamericancouponBadAIStructure'; //#8645
            $k[$j++] = "A GS1 North American Coupon should consist of a single AI (8110)"; //#8645
            $0.raiseerror(); //#8645
        } //#8645
        if ($ne($get($1.ais, 0), "8110")) { //#8649
            $k[$j++] = 'bwipp.gs1northamericancouponBadAIStructure'; //#8648
            $k[$j++] = "A GS1 North American Coupon should consist of a single AI (8110)"; //#8648
            $0.raiseerror(); //#8648
        } //#8648
        if (!$1.dontlint) { //#8651
            $k[$j++] = $1.ais; //#8651
            $k[$j++] = $1.vals; //#8651
            $0.gs1lint(); //#8651
            $j--; //#8651
        } //#8651
        $1.val = $get($1.vals, 0); //#8654
        $1.vli = $f($get($1.val, 0) - 48); //#8655
        if (($1.vli < 0) || ($1.vli > 6)) { //#8658
            $k[$j++] = 'bwipp.gs1northamericancouponBadVLI'; //#8657
            $k[$j++] = "The AI (8110) data should start with a Company Prefix length indicator in the range 0 to 6"; //#8657
            $0.raiseerror(); //#8657
        } //#8657
        $1.gcp = $geti($1.val, 1, $f($1.vli + 6)); //#8659
        $1.cod = $geti($1.val, $f($1.vli + 7), 6); //#8660
        $1.coupontext = $s($1.gcp.length + 7); //#8661
        $puti($1.coupontext, 0, $1.gcp); //#8662
        $puti($1.coupontext, $1.gcp.length, "-"); //#8663
        $puti($1.coupontext, $1.gcp.length + 1, $1.cod); //#8664
        delete $1.options["parse"]; //#8667
        $put($1.options, "dontdraw", true); //#8668
        $k[$j++] = 'args'; //#8670
        $k[$j++] = $1.barcode; //#8670
        $k[$j++] = $1.options; //#8670
        $0.databarexpandedstacked(); //#8670
        var _19 = $k[--$j]; //#8670
        $1[$k[--$j]] = _19; //#8670
        if ($1.includetext) { //#8689
            if ($1.coupontextxoffset == "unset") { //#8676
                $1.coupontextxoffset = 0; //#8675
            } //#8675
            if ($1.coupontextyoffset == "unset") { //#8679
                $1.coupontextyoffset = $f(($get($1.args, "height") * 72) + 3); //#8678
            } //#8678
            var _1H = $get($1.args, "txt") !== undefined; //#8680
            if (_1H) { //#8687
                $1.txt = $get($1.args, "txt"); //#8681
                $1.newtxt = $a($1.txt.length + 1); //#8682
                $puti($1.newtxt, 0, $1.txt); //#8683
                $put($1.newtxt, $1.newtxt.length - 1, $a([$1.coupontext, $1.coupontextxoffset, $1.coupontextyoffset, $1.coupontextfont, $1.coupontextsize])); //#8684
                $put($1.args, "txt", $1.newtxt); //#8685
            } else { //#8687
                $put($1.args, "txt", $a([$a([$1.coupontext, $1.coupontextxoffset, $1.coupontextyoffset, $1.coupontextfont, $1.coupontextsize])])); //#8687
            } //#8687
        } //#8687
        $put($1.args, "opt", $1.options); //#8691
        $k[$j++] = $1.args; //#8694
        if (!$1.dontdraw) { //#8694
            $0.renmatrix(); //#8694
        } //#8694
    };
    $0.pharmacode = function() {
        var $1 = {}; //#8723
        $1.options = $k[--$j]; //#8725
        $1.barcode = $k[--$j]; //#8726
        $1.dontdraw = false; //#8728
        $1.height = (8 * 2.835) / 72; //#8729
        $1.nwidth = 0.5 * 2.835; //#8730
        $1.wwidth = 1.5 * 2.835; //#8731
        $1.swidth = 1 * 2.835; //#8732
        $forall($1.options, function() { //#8743
            var _3 = $k[--$j]; //#8743
            $1[$k[--$j]] = _3; //#8743
        }); //#8743
        $1.height = +$1.height; //#8745
        $1.nwidth = +$1.nwidth; //#8746
        $1.wwidth = +$1.wwidth; //#8747
        $1.swidth = +$1.swidth; //#8748
        if (($1.barcode.length < 1) || ($1.barcode.length > 6)) { //#8753
            $k[$j++] = 'bwipp.pharmacodeBadLength'; //#8752
            $k[$j++] = "Pharmacode must be 1 to 6 digits"; //#8752
            $0.raiseerror(); //#8752
        } //#8752
        $forall($1.barcode, function() { //#8758
            var _C = $k[--$j]; //#8755
            if ((_C < 48) || (_C > 57)) { //#8757
                $k[$j++] = 'bwipp.pharmacodeBadCharacter'; //#8756
                $k[$j++] = "Pharmacode must contain only digits"; //#8756
                $0.raiseerror(); //#8756
            } //#8756
        }); //#8756
        var _E = ~~$z($1.barcode); //#8759
        if ((_E < 3) || (_E > 131070)) { //#8761
            $k[$j++] = 'bwipp.pharmacodeBadValue'; //#8760
            $k[$j++] = "Pharmacode value must be between 3 and 131070"; //#8760
            $0.raiseerror(); //#8760
        } //#8760
        $1.txt = $a($1.barcode.length); //#8764
        for (var _J = 0, _I = $1.barcode.length - 1; _J <= _I; _J += 1) { //#8768
            $1.i = _J; //#8766
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), 0, 0, "", 0])); //#8767
        } //#8767
        $1.barcode = $cvrs($s(17), (~~$z($1.barcode)) + 1, 2); //#8771
        $1.barcode = $geti($1.barcode, 1, $1.barcode.length - 1); //#8772
        $1.barlen = $1.barcode.length; //#8774
        $1.sbs = $a($1.barlen * 2); //#8775
        for (var _b = 0, _a = $1.barlen - 1; _b <= _a; _b += 1) { //#8786
            $1.i = _b; //#8778
            $1.enc = $geti($1.barcode, $1.i, 1); //#8779
            if ($eq($1.enc, "0")) { //#8783
                $put($1.sbs, $1.i * 2, $1.nwidth); //#8781
            } else { //#8783
                $put($1.sbs, $1.i * 2, $1.wwidth); //#8783
            } //#8783
            $put($1.sbs, ($1.i * 2) + 1, $1.swidth); //#8785
        } //#8785
        $k[$j++] = Infinity; //#8796
        var _p = $1.sbs; //#8791
        $k[$j++] = Infinity; //#8792
        for (var _r = 0, _s = ~~(($1.sbs.length + 1) / 2); _r < _s; _r++) { //#8792
            $k[$j++] = $1.height; //#8792
        } //#8792
        var _u = $a(); //#8792
        $k[$j++] = Infinity; //#8793
        for (var _w = 0, _x = ~~(($1.sbs.length + 1) / 2); _w < _x; _w++) { //#8793
            $k[$j++] = 0; //#8793
        } //#8793
        var _y = $a(); //#8793
        $k[$j++] = 'ren'; //#8796
        $k[$j++] = $0.renlinear; //#8796
        $k[$j++] = 'sbs'; //#8796
        $k[$j++] = _p; //#8796
        $k[$j++] = 'bhs'; //#8796
        $k[$j++] = _u; //#8796
        $k[$j++] = 'bbs'; //#8796
        $k[$j++] = _y; //#8796
        $k[$j++] = 'txt'; //#8796
        $k[$j++] = $1.txt; //#8796
        $k[$j++] = 'textxalign'; //#8796
        $k[$j++] = "center"; //#8796
        $k[$j++] = 'opt'; //#8796
        $k[$j++] = $1.options; //#8796
        var _11 = $d(); //#8796
        $k[$j++] = _11; //#8799
        if (!$1.dontdraw) { //#8799
            $0.renlinear(); //#8799
        } //#8799
    };
    $0.pharmacode2 = function() {
        var $1 = {}; //#8828
        $1.options = $k[--$j]; //#8830
        $1.barcode = $k[--$j]; //#8831
        $1.dontdraw = false; //#8833
        $1.includetext = false; //#8834
        $1.height = 4; //#8835
        $forall($1.options, function() { //#8846
            var _3 = $k[--$j]; //#8846
            $1[$k[--$j]] = _3; //#8846
        }); //#8846
        $1.height = +$1.height; //#8848
        if (($1.barcode.length < 1) || ($1.barcode.length > 8)) { //#8853
            $k[$j++] = 'bwipp.pharmacode2BadLength'; //#8852
            $k[$j++] = "Two-track Pharmacode must be 1 to 6 digits"; //#8852
            $0.raiseerror(); //#8852
        } //#8852
        $forall($1.barcode, function() { //#8858
            var _9 = $k[--$j]; //#8855
            if ((_9 < 48) || (_9 > 57)) { //#8857
                $k[$j++] = 'bwipp.pharmacode2badCharacter'; //#8856
                $k[$j++] = "Two-track Pharmacode must contain only digits"; //#8856
                $0.raiseerror(); //#8856
            } //#8856
        }); //#8856
        var _B = ~~$z($1.barcode); //#8859
        if ((_B < 4) || (_B > 64570080)) { //#8861
            $k[$j++] = 'bwipp.pharmacode2badValue'; //#8860
            $k[$j++] = "Two-track Pharmacode value must be between 4 and 64570080"; //#8860
            $0.raiseerror(); //#8860
        } //#8860
        $1.txt = $a($1.barcode.length); //#8864
        for (var _G = 0, _F = $1.barcode.length - 1; _G <= _F; _G += 1) { //#8868
            $1.i = _G; //#8866
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), 0, 0, "", 0])); //#8867
        } //#8867
        $1.encstr = $s(16); //#8870
        $1.sbs = $a(32); //#8871
        $1.bar = (1 / 25.4) * $1.height; //#8872
        $1.spc = (1 / 25.4) * 72; //#8873
        $1.i = 15; //#8876
        $1.val = ~~$z($1.barcode); //#8877
        for (;;) { //#8884
            if ($1.val == 0) { //#8879
                break; //#8879
            } //#8879
            var _T = $1.val % 3; //#8880
            $1.val = ~~($f($1.val - $get($a([3, 1, 2]), _T)) / 3); //#8881
            $put($1.encstr, $1.i, $get($a([2, 0, 1]), _T)); //#8882
            $1.i = $1.i - 1; //#8883
        } //#8883
        $1.encstr = $geti($1.encstr, $1.i + 1, 15 - $1.i); //#8885
        $1.bhs = $a($1.encstr.length); //#8888
        $1.bbs = $a($1.encstr.length); //#8889
        for (var _m = 0, _l = $1.encstr.length - 1; _m <= _l; _m += 1) { //#8895
            $1.i = _m; //#8891
            var _p = $get($1.encstr, $1.i); //#8892
            $put($1.bhs, $1.i, $1.bar * $get($a([1, 1, 2]), _p)); //#8893
            $put($1.bbs, $1.i, $get($a([0, $1.bar, 0]), _p)); //#8894
        } //#8894
        $k[$j++] = Infinity; //#8906
        var _10 = $1.bhs; //#8900
        var _11 = $1.bbs; //#8901
        $k[$j++] = Infinity; //#8902
        for (var _13 = 0, _14 = $1.encstr.length * 2; _13 < _14; _13++) { //#8902
            $k[$j++] = $1.spc; //#8902
        } //#8902
        var _16 = $a(); //#8902
        $k[$j++] = 'ren'; //#8906
        $k[$j++] = $0.renlinear; //#8906
        $k[$j++] = 'bhs'; //#8906
        $k[$j++] = _10; //#8906
        $k[$j++] = 'bbs'; //#8906
        $k[$j++] = _11; //#8906
        $k[$j++] = 'sbs'; //#8906
        $k[$j++] = _16; //#8906
        $k[$j++] = 'txt'; //#8906
        $k[$j++] = $1.txt; //#8906
        $k[$j++] = 'textxalign'; //#8906
        $k[$j++] = "center"; //#8906
        $k[$j++] = 'textyoffset'; //#8906
        $k[$j++] = 4; //#8906
        $k[$j++] = 'opt'; //#8906
        $k[$j++] = $1.options; //#8906
        var _19 = $d(); //#8906
        $k[$j++] = _19; //#8909
        if (!$1.dontdraw) { //#8909
            $0.renlinear(); //#8909
        } //#8909
    };
    $0.code2of5 = function() {
        var $1 = {}; //#8938
        $1.options = $k[--$j]; //#8940
        $1.barcode = $k[--$j]; //#8941
        $1.dontdraw = false; //#8943
        $1.includecheck = false; //#8944
        $1.validatecheck = false; //#8945
        $1.includetext = false; //#8946
        $1.includecheckintext = false; //#8947
        $1.textfont = "OCR-B"; //#8948
        $1.textsize = 10; //#8949
        $1.textyoffset = -8; //#8950
        $1.height = 1; //#8951
        $1.version = "industrial"; //#8952
        $forall($1.options, function() { //#8963
            var _3 = $k[--$j]; //#8963
            $1[$k[--$j]] = _3; //#8963
        }); //#8963
        $1.textfont = "" + $1.textfont; //#8965
        $1.textsize = +$1.textsize; //#8966
        $1.textyoffset = +$1.textyoffset; //#8967
        $1.height = +$1.height; //#8968
        $forall($1.barcode, function() { //#8975
            var _A = $k[--$j]; //#8972
            if ((_A < 48) || (_A > 57)) { //#8974
                $k[$j++] = 'bwipp.code2of5badCharacter'; //#8973
                $k[$j++] = "Code 25 must contain only digits"; //#8973
                $0.raiseerror(); //#8973
            } //#8973
        }); //#8973
        $k[$j++] = 'barlen'; //#8977
        $k[$j++] = $1.barcode.length; //#8977
        if ($1.validatecheck) { //#8977
            var _D = $k[--$j]; //#8977
            $k[$j++] = $f(_D - 1); //#8977
        } //#8977
        var _E = $k[--$j]; //#8977
        $1[$k[--$j]] = _E; //#8977
        $1.checksum = 0; //#8979
        for (var _I = 0, _H = $f($1.barlen - 1); _I <= _H; _I += 1) { //#8983
            $1.i = _I; //#8981
            $k[$j++] = 'checksum'; //#8982
            $k[$j++] = $1.checksum; //#8982
            $k[$j++] = $f($get($1.barcode, $1.i) - 48); //#8982
            if (($f($1.barlen - $1.i) % 2) != 0) { //#8982
                var _P = $k[--$j]; //#8982
                $k[$j++] = _P * 3; //#8982
            } //#8982
            var _Q = $k[--$j]; //#8982
            var _R = $k[--$j]; //#8982
            $1[$k[--$j]] = $f(_R + _Q); //#8982
        } //#8982
        $1.checksum = (10 - ($1.checksum % 10)) % 10; //#8984
        if ($1.validatecheck) { //#8991
            if ($get($1.barcode, $1.barlen) != ($1.checksum + 48)) { //#8988
                $k[$j++] = 'bwipp.code2of5badCheckDigit'; //#8987
                $k[$j++] = "Incorrect Code 25 check digit provided"; //#8987
                $0.raiseerror(); //#8987
            } //#8987
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#8989
            $1.includecheck = true; //#8990
        } //#8990
        var _h = new Map([
            ["industrial", $a(["1111313111", "3111111131", "1131111131", "3131111111", "1111311131", "3111311111", "1131311111", "1111113131", "3111113111", "1131113111", "313111", "31113"])],
            ["iata", $a(["1111313111", "3111111131", "1131111131", "3131111111", "1111311131", "3111311111", "1131311111", "1111113131", "3111113111", "1131113111", "1111", "311"])],
            ["matrix", $a(["113311", "311131", "131131", "331111", "113131", "313111", "133111", "111331", "311311", "131311", "311111", "31111"])],
            ["coop", $a(["331111", "111331", "113131", "113311", "131131", "131311", "133111", "311131", "311311", "313111", "3131", "133"])],
            ["datalogic", $a(["113311", "311131", "131131", "331111", "113131", "313111", "133111", "111331", "311311", "131311", "1111", "311"])]
        ]); //#9015
        $1.versions = _h; //#9017
        var _k = $get($1.versions, $1.version) !== undefined; //#9018
        if (!_k) { //#9020
            $k[$j++] = 'bwipp.code2of5badVersion'; //#9019
            $k[$j++] = "Unrecognised Code 25 version"; //#9019
            $0.raiseerror(); //#9019
        } //#9019
        $1.encs = $get($1.versions, $1.version); //#9021
        $1.cs = $get($1.encs, 0).length; //#9023
        $k[$j++] = 'cw'; //#9024
        $k[$j++] = 0; //#9024
        $forall($get($1.encs, 0), function() { //#9024
            var _s = $k[--$j]; //#9024
            var _t = $k[--$j]; //#9024
            $k[$j++] = $f(_t + $f(_s - 48)); //#9024
        }); //#9024
        var _u = $k[--$j]; //#9024
        $1[$k[--$j]] = _u; //#9024
        $1.ss = $get($1.encs, 10).length; //#9025
        $k[$j++] = 'sw'; //#9026
        $k[$j++] = 0; //#9026
        $forall($get($1.encs, 10), function() { //#9026
            var _10 = $k[--$j]; //#9026
            var _11 = $k[--$j]; //#9026
            $k[$j++] = $f(_11 + $f(_10 - 48)); //#9026
        }); //#9026
        var _12 = $k[--$j]; //#9026
        $1[$k[--$j]] = _12; //#9026
        $1.es = $get($1.encs, 11).length; //#9027
        $1.barchars = "0123456789"; //#9030
        $k[$j++] = 'sbs'; //#9032
        $k[$j++] = $1.barlen; //#9032
        if ($1.includecheck) { //#9032
            var _18 = $k[--$j]; //#9032
            $k[$j++] = $f(_18 + 1); //#9032
        } //#9032
        var _1D = $s($f(($f(($k[--$j] * $1.cs) + $1.ss)) + $1.es)); //#9032
        $1[$k[--$j]] = _1D; //#9032
        $k[$j++] = 'txt'; //#9033
        $k[$j++] = $1.barlen; //#9033
        if ($1.includecheck) { //#9033
            var _1H = $k[--$j]; //#9033
            $k[$j++] = $f(_1H + 1); //#9033
        } //#9033
        var _1J = $a($k[--$j]); //#9033
        $1[$k[--$j]] = _1J; //#9033
        $puti($1.sbs, 0, $get($1.encs, 10)); //#9036
        for (var _1Q = 0, _1P = $f($1.barlen - 1); _1Q <= _1P; _1Q += 1) { //#9047
            $1.i = _1Q; //#9038
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#9040
            $j--; //#9041
            $1.indx = $k[--$j].length; //#9042
            $j -= 2; //#9043
            $1.enc = $get($1.encs, $1.indx); //#9044
            $puti($1.sbs, ($1.i * $1.cs) + $1.ss, $1.enc); //#9045
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $f(($1.i * $1.cw) + $1.sw), $1.textyoffset, $1.textfont, $1.textsize])); //#9046
        } //#9046
        if ($1.includecheck) { //#9059
            $puti($1.sbs, $f(($1.barlen * $1.cs) + $1.ss), $get($1.encs, $1.checksum)); //#9051
            $puti($1.sbs, $f(($f(($1.barlen * $1.cs) + $1.cs)) + $1.ss), $get($1.encs, 11)); //#9052
            if ($1.includecheckintext) { //#9056
                $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum, 1), $f(($1.barlen * $1.cw) + $1.sw), $1.textyoffset, $1.textfont, $1.textsize])); //#9054
            } else { //#9056
                $put($1.txt, $1.barlen, $a([" ", $f(($1.barlen * $1.cw) + $1.sw), $1.textyoffset, $1.textfont, $1.textsize])); //#9056
            } //#9056
        } else { //#9059
            $puti($1.sbs, $f(($1.barlen * $1.cs) + $1.ss), $get($1.encs, 11)); //#9059
        } //#9059
        $k[$j++] = Infinity; //#9071
        $k[$j++] = Infinity; //#9065
        $forall($1.sbs, function() { //#9065
            var _2Y = $k[--$j]; //#9065
            $k[$j++] = $f(_2Y - 48); //#9065
        }); //#9065
        var _2Z = $a(); //#9065
        $k[$j++] = Infinity; //#9066
        for (var _2b = 0, _2c = ~~(($1.sbs.length + 1) / 2); _2b < _2c; _2b++) { //#9066
            $k[$j++] = $1.height; //#9066
        } //#9066
        var _2e = $a(); //#9066
        $k[$j++] = Infinity; //#9067
        for (var _2g = 0, _2h = ~~(($1.sbs.length + 1) / 2); _2g < _2h; _2g++) { //#9067
            $k[$j++] = 0; //#9067
        } //#9067
        var _2i = $a(); //#9067
        $k[$j++] = 'ren'; //#9070
        $k[$j++] = $0.renlinear; //#9070
        $k[$j++] = 'sbs'; //#9070
        $k[$j++] = _2Z; //#9070
        $k[$j++] = 'bhs'; //#9070
        $k[$j++] = _2e; //#9070
        $k[$j++] = 'bbs'; //#9070
        $k[$j++] = _2i; //#9070
        if ($1.includetext) { //#9070
            $k[$j++] = 'txt'; //#9069
            $k[$j++] = $1.txt; //#9069
        } //#9069
        $k[$j++] = 'opt'; //#9071
        $k[$j++] = $1.options; //#9071
        var _2m = $d(); //#9071
        $k[$j++] = _2m; //#9074
        if (!$1.dontdraw) { //#9074
            $0.renlinear(); //#9074
        } //#9074
    };
    $0.industrial2of5 = function() {
        var $1 = {}; //#9104
        $1.options = $k[--$j]; //#9106
        $1.barcode = $k[--$j]; //#9107
        $1.dontdraw = false; //#9109
        $1.includetext = false; //#9110
        $1.textfont = "OCR-B"; //#9111
        $1.textsize = 10; //#9112
        $1.textxoffset = 0; //#9113
        $1.textyoffset = -8; //#9114
        $1.height = 1; //#9115
        $forall($1.options, function() { //#9126
            var _3 = $k[--$j]; //#9126
            $1[$k[--$j]] = _3; //#9126
        }); //#9126
        $1.textfont = "" + $1.textfont; //#9128
        $1.textsize = +$1.textsize; //#9129
        $1.textxoffset = +$1.textxoffset; //#9130
        $1.textyoffset = +$1.textyoffset; //#9131
        $1.height = +$1.height; //#9132
        $put($1.options, "dontdraw", true); //#9135
        $put($1.options, "version", "industrial"); //#9136
        $k[$j++] = 'args'; //#9138
        $k[$j++] = $1.barcode; //#9138
        $k[$j++] = $1.options; //#9138
        $0.code2of5(); //#9138
        var _E = $k[--$j]; //#9138
        $1[$k[--$j]] = _E; //#9138
        $put($1.args, "opt", $1.options); //#9140
        $k[$j++] = $1.args; //#9143
        if (!$1.dontdraw) { //#9143
            $0.renlinear(); //#9143
        } //#9143
    };
    $0.iata2of5 = function() {
        var $1 = {}; //#9173
        $1.options = $k[--$j]; //#9175
        $1.barcode = $k[--$j]; //#9176
        $1.dontdraw = false; //#9178
        $1.includetext = false; //#9179
        $1.textfont = "OCR-B"; //#9180
        $1.textsize = 10; //#9181
        $1.textxoffset = 0; //#9182
        $1.textyoffset = -8; //#9183
        $1.height = 1; //#9184
        $forall($1.options, function() { //#9195
            var _3 = $k[--$j]; //#9195
            $1[$k[--$j]] = _3; //#9195
        }); //#9195
        $1.textfont = "" + $1.textfont; //#9197
        $1.textsize = +$1.textsize; //#9198
        $1.textxoffset = +$1.textxoffset; //#9199
        $1.textyoffset = +$1.textyoffset; //#9200
        $1.height = +$1.height; //#9201
        $put($1.options, "dontdraw", true); //#9204
        $put($1.options, "version", "iata"); //#9205
        $k[$j++] = 'args'; //#9207
        $k[$j++] = $1.barcode; //#9207
        $k[$j++] = $1.options; //#9207
        $0.code2of5(); //#9207
        var _E = $k[--$j]; //#9207
        $1[$k[--$j]] = _E; //#9207
        $put($1.args, "opt", $1.options); //#9209
        $k[$j++] = $1.args; //#9212
        if (!$1.dontdraw) { //#9212
            $0.renlinear(); //#9212
        } //#9212
    };
    $0.matrix2of5 = function() {
        var $1 = {}; //#9242
        $1.options = $k[--$j]; //#9244
        $1.barcode = $k[--$j]; //#9245
        $1.dontdraw = false; //#9247
        $1.includetext = false; //#9248
        $1.textfont = "OCR-B"; //#9249
        $1.textsize = 10; //#9250
        $1.textxoffset = 0; //#9251
        $1.textyoffset = -8; //#9252
        $1.height = 1; //#9253
        $forall($1.options, function() { //#9264
            var _3 = $k[--$j]; //#9264
            $1[$k[--$j]] = _3; //#9264
        }); //#9264
        $1.textfont = "" + $1.textfont; //#9266
        $1.textsize = +$1.textsize; //#9267
        $1.textxoffset = +$1.textxoffset; //#9268
        $1.textyoffset = +$1.textyoffset; //#9269
        $1.height = +$1.height; //#9270
        $put($1.options, "dontdraw", true); //#9273
        $put($1.options, "version", "matrix"); //#9274
        $k[$j++] = 'args'; //#9276
        $k[$j++] = $1.barcode; //#9276
        $k[$j++] = $1.options; //#9276
        $0.code2of5(); //#9276
        var _E = $k[--$j]; //#9276
        $1[$k[--$j]] = _E; //#9276
        $put($1.args, "opt", $1.options); //#9278
        $k[$j++] = $1.args; //#9281
        if (!$1.dontdraw) { //#9281
            $0.renlinear(); //#9281
        } //#9281
    };
    $0.coop2of5 = function() {
        var $1 = {}; //#9311
        $1.options = $k[--$j]; //#9313
        $1.barcode = $k[--$j]; //#9314
        $1.dontdraw = false; //#9316
        $1.includetext = false; //#9317
        $1.textfont = "OCR-B"; //#9318
        $1.textsize = 10; //#9319
        $1.textxoffset = 0; //#9320
        $1.textyoffset = -8; //#9321
        $1.height = 1; //#9322
        $forall($1.options, function() { //#9333
            var _3 = $k[--$j]; //#9333
            $1[$k[--$j]] = _3; //#9333
        }); //#9333
        $1.textfont = "" + $1.textfont; //#9335
        $1.textsize = +$1.textsize; //#9336
        $1.textxoffset = +$1.textxoffset; //#9337
        $1.textyoffset = +$1.textyoffset; //#9338
        $1.height = +$1.height; //#9339
        $put($1.options, "dontdraw", true); //#9342
        $put($1.options, "version", "coop"); //#9343
        $k[$j++] = 'args'; //#9345
        $k[$j++] = $1.barcode; //#9345
        $k[$j++] = $1.options; //#9345
        $0.code2of5(); //#9345
        var _E = $k[--$j]; //#9345
        $1[$k[--$j]] = _E; //#9345
        $put($1.args, "opt", $1.options); //#9347
        $k[$j++] = $1.args; //#9350
        if (!$1.dontdraw) { //#9350
            $0.renlinear(); //#9350
        } //#9350
    };
    $0.datalogic2of5 = function() {
        var $1 = {}; //#9380
        $1.options = $k[--$j]; //#9382
        $1.barcode = $k[--$j]; //#9383
        $1.dontdraw = false; //#9385
        $1.includetext = false; //#9386
        $1.textfont = "OCR-B"; //#9387
        $1.textsize = 10; //#9388
        $1.textxoffset = 0; //#9389
        $1.textyoffset = -8; //#9390
        $1.height = 1; //#9391
        $forall($1.options, function() { //#9402
            var _3 = $k[--$j]; //#9402
            $1[$k[--$j]] = _3; //#9402
        }); //#9402
        $1.textfont = "" + $1.textfont; //#9404
        $1.textsize = +$1.textsize; //#9405
        $1.textxoffset = +$1.textxoffset; //#9406
        $1.textyoffset = +$1.textyoffset; //#9407
        $1.height = +$1.height; //#9408
        $put($1.options, "dontdraw", true); //#9411
        $put($1.options, "version", "datalogic"); //#9412
        $k[$j++] = 'args'; //#9414
        $k[$j++] = $1.barcode; //#9414
        $k[$j++] = $1.options; //#9414
        $0.code2of5(); //#9414
        var _E = $k[--$j]; //#9414
        $1[$k[--$j]] = _E; //#9414
        $put($1.args, "opt", $1.options); //#9416
        $k[$j++] = $1.args; //#9419
        if (!$1.dontdraw) { //#9419
            $0.renlinear(); //#9419
        } //#9419
    };
    $0.code11 = function() {
        var $1 = {}; //#9448
        $1.options = $k[--$j]; //#9450
        $1.barcode = $k[--$j]; //#9451
        $1.dontdraw = false; //#9453
        $1.includecheck = false; //#9454
        $1.validatecheck = false; //#9455
        $1.includetext = false; //#9456
        $1.includecheckintext = false; //#9457
        $1.textfont = "OCR-B"; //#9458
        $1.textsize = 10; //#9459
        $1.textyoffset = -8; //#9460
        $1.height = 1; //#9461
        $forall($1.options, function() { //#9472
            var _3 = $k[--$j]; //#9472
            $1[$k[--$j]] = _3; //#9472
        }); //#9472
        $1.textfont = "" + $1.textfont; //#9474
        $1.textsize = +$1.textsize; //#9475
        $1.textyoffset = +$1.textyoffset; //#9476
        $1.height = +$1.height; //#9477
        $1.encs = $a(["111131", "311131", "131131", "331111", "113131", "313111", "133111", "111331", "311311", "311111", "113111", "113311"]); //#9484
        $1.barchars = "0123456789-"; //#9487
        $1.charvals = new Map; //#9488
        for (var _A = 0; _A <= 10; _A += 1) { //#9489
            $put($1.charvals, $geti($1.barchars, _A, 1), _A); //#9489
        } //#9489
        for (var _G = 0, _F = $1.barcode.length - 1; _G <= _F; _G += 1) { //#9496
            var _K = $get($1.charvals, $geti($1.barcode, _G, 1)) !== undefined; //#9493
            if (!_K) { //#9495
                $k[$j++] = 'bwipp.code11badCharacter'; //#9494
                $k[$j++] = "Code 11 must contain only digits and dashes"; //#9494
                $0.raiseerror(); //#9494
            } //#9494
        } //#9494
        $1.barlen = $1.barcode.length; //#9498
        if ($1.validatecheck) { //#9504
            if ($1.barlen == 11) { //#9502
                $k[$j++] = 'bwipp.code11badLength'; //#9501
                $k[$j++] = "Code 11 cannot be 11 characters using check digits"; //#9501
                $0.raiseerror(); //#9501
            } //#9501
            var _Q = ($1.barlen <= 10) ? 1 : 2; //#9503
            $1.barlen = $1.barlen - _Q; //#9503
        } //#9503
        $k[$j++] = 'numchecks'; //#9506
        if ($1.includecheck || $1.validatecheck) { //#9506
            var _U = ($1.barlen >= 10) ? 2 : 1; //#9506
            $k[$j++] = _U; //#9506
        } else { //#9506
            $k[$j++] = 0; //#9506
        } //#9506
        var _V = $k[--$j]; //#9506
        $1[$k[--$j]] = _V; //#9506
        $1.checksum1 = 0; //#9507
        $1.checksum2 = 0; //#9507
        for (var _Z = 0, _Y = $1.barlen - 1; _Z <= _Y; _Z += 1) { //#9513
            $1.i = _Z; //#9509
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#9510
            $1.checksum1 = $f($1.checksum1 + ((((($1.barlen - $1.i) - 1) % 10) + 1) * $1.indx)); //#9511
            $1.checksum2 = $f($1.checksum2 + (((($1.barlen - $1.i) % 9) + 1) * $1.indx)); //#9512
        } //#9512
        $1.checksum1 = $1.checksum1 % 11; //#9514
        $1.checksum2 = $f($1.checksum2 + $1.checksum1) % 11; //#9515
        if ($1.validatecheck) { //#9529
            if ($1.numchecks == 1) { //#9524
                if ($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum1)) { //#9520
                    $k[$j++] = 'bwipp.code11badCheckDigit'; //#9519
                    $k[$j++] = "Incorrect Code 11 check digit provided"; //#9519
                    $0.raiseerror(); //#9519
                } //#9519
            } else { //#9524
                if (($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum1)) || ($get($1.barcode, $1.barlen + 1) != $get($1.barchars, $1.checksum2))) { //#9525
                    $k[$j++] = 'bwipp.code11badCheckDigits'; //#9524
                    $k[$j++] = "Incorrect Code 11 check digits provided"; //#9524
                    $0.raiseerror(); //#9524
                } //#9524
            } //#9524
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#9527
            $1.includecheck = true; //#9528
        } //#9528
        $1.sbs = $s($f(($f($1.barlen + $1.numchecks) * 6) + 12)); //#9531
        $1.txt = $a($f($1.barlen + $1.numchecks)); //#9532
        $puti($1.sbs, 0, $get($1.encs, 11)); //#9535
        $1.xpos = 8; //#9537
        for (var _1O = 0, _1N = $1.barlen - 1; _1O <= _1N; _1O += 1) { //#9547
            $1.i = _1O; //#9539
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#9540
            $1.enc = $get($1.encs, $1.indx); //#9541
            $puti($1.sbs, ($1.i * 6) + 6, $1.enc); //#9542
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9543
            for (var _1k = 0; _1k <= 5; _1k += 1) { //#9546
                $1.xpos = $f($f($get($1.enc, _1k) - 48) + $1.xpos); //#9545
            } //#9545
        } //#9545
        if ($1.includecheck) { //#9576
            if ($1.barlen >= 10) { //#9573
                $puti($1.sbs, ($1.barlen * 6) + 6, $get($1.encs, $1.checksum1)); //#9552
                $puti($1.sbs, ($1.barlen * 6) + 12, $get($1.encs, $1.checksum2)); //#9553
                if ($1.includecheckintext) { //#9563
                    $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum1, 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9555
                    $1.enc = $get($1.encs, $1.checksum1); //#9556
                    for (var _2E = 0; _2E <= 5; _2E += 1) { //#9559
                        $1.xpos = $f($f($get($1.enc, _2E) - 48) + $1.xpos); //#9558
                    } //#9558
                    $put($1.txt, $1.barlen + 1, $a([$geti($1.barchars, $1.checksum2, 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9560
                } else { //#9563
                    $put($1.txt, $1.barlen, $a(["", $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9562
                    $put($1.txt, $1.barlen + 1, $a(["", $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9563
                } //#9563
                $puti($1.sbs, ($1.barlen * 6) + 18, $get($1.encs, 11)); //#9565
            } else { //#9573
                $puti($1.sbs, ($1.barlen * 6) + 6, $get($1.encs, $1.checksum1)); //#9567
                if ($1.includecheckintext) { //#9571
                    $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum1, 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9569
                } else { //#9571
                    $put($1.txt, $1.barlen, $a(["", $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9571
                } //#9571
                $puti($1.sbs, ($1.barlen * 6) + 12, $get($1.encs, 11)); //#9573
            } //#9573
        } else { //#9576
            $puti($1.sbs, ($1.barlen * 6) + 6, $get($1.encs, 11)); //#9576
        } //#9576
        $k[$j++] = Infinity; //#9588
        $k[$j++] = Infinity; //#9582
        var _3F = $1.sbs; //#9582
        for (var _3G = 0, _3H = _3F.length; _3G < _3H; _3G++) { //#9582
            $k[$j++] = $get(_3F, _3G) - 48; //#9582
        } //#9582
        var _3J = $a(); //#9582
        $k[$j++] = Infinity; //#9583
        for (var _3L = 0, _3M = ~~(($1.sbs.length + 1) / 2); _3L < _3M; _3L++) { //#9583
            $k[$j++] = $1.height; //#9583
        } //#9583
        var _3O = $a(); //#9583
        $k[$j++] = Infinity; //#9584
        for (var _3Q = 0, _3R = ~~(($1.sbs.length + 1) / 2); _3Q < _3R; _3Q++) { //#9584
            $k[$j++] = 0; //#9584
        } //#9584
        var _3S = $a(); //#9584
        $k[$j++] = 'ren'; //#9587
        $k[$j++] = $0.renlinear; //#9587
        $k[$j++] = 'sbs'; //#9587
        $k[$j++] = _3J; //#9587
        $k[$j++] = 'bhs'; //#9587
        $k[$j++] = _3O; //#9587
        $k[$j++] = 'bbs'; //#9587
        $k[$j++] = _3S; //#9587
        if ($1.includetext) { //#9587
            $k[$j++] = 'txt'; //#9586
            $k[$j++] = $1.txt; //#9586
        } //#9586
        $k[$j++] = 'opt'; //#9588
        $k[$j++] = $1.options; //#9588
        var _3W = $d(); //#9588
        $k[$j++] = _3W; //#9591
        if (!$1.dontdraw) { //#9591
            $0.renlinear(); //#9591
        } //#9591
    };
    $0.bc412 = function() {
        var $1 = {}; //#9620
        $1.options = $k[--$j]; //#9622
        $1.barcode = $k[--$j]; //#9623
        $1.dontdraw = false; //#9625
        $1.includecheck = false; //#9626
        $1.validatecheck = false; //#9627
        $1.includetext = false; //#9628
        $1.includecheckintext = false; //#9629
        $1.includestartstop = false; //#9630
        $1.semi = false; //#9631
        $1.textfont = "OCR-B"; //#9632
        $1.textsize = 10; //#9633
        $1.textyoffset = -8; //#9634
        $1.height = 1; //#9635
        $forall($1.options, function() { //#9646
            var _3 = $k[--$j]; //#9646
            $1[$k[--$j]] = _3; //#9646
        }); //#9646
        $1.textfont = "" + $1.textfont; //#9648
        $1.textsize = +$1.textsize; //#9649
        $1.textyoffset = +$1.textyoffset; //#9650
        $1.height = +$1.height; //#9651
        if ($1.semi) { //#9656
            $1.includecheck = true; //#9654
            $1.includestartstop = true; //#9655
        } //#9655
        $1.barchars = "0R9GLVHA8EZ4NTS1J2Q6C7DYKBUIX3FWP5M"; //#9659
        $1.charvals = new Map; //#9660
        for (var _A = 0; _A <= 34; _A += 1) { //#9661
            $put($1.charvals, $geti($1.barchars, _A, 1), _A); //#9661
        } //#9661
        for (var _G = 0, _F = $1.barcode.length - 1; _G <= _F; _G += 1) { //#9668
            var _K = $get($1.charvals, $geti($1.barcode, _G, 1)) !== undefined; //#9665
            if (!_K) { //#9667
                $k[$j++] = 'bwipp.bc412badCharacter'; //#9666
                $k[$j++] = "BC412 must contain only digits and capital letters except O"; //#9666
                $0.raiseerror(); //#9666
            } //#9666
        } //#9666
        $k[$j++] = 'barlen'; //#9670
        $k[$j++] = $1.barcode.length; //#9670
        if ($1.validatecheck) { //#9670
            var _N = $k[--$j]; //#9670
            $k[$j++] = $f(_N - 1); //#9670
        } //#9670
        var _O = $k[--$j]; //#9670
        $1[$k[--$j]] = _O; //#9670
        $1.checksum = 0; //#9672
        for (var _S = 0, _R = $f($1.barlen - 1); _S <= _R; _S += 1) { //#9676
            $1.checksum = $f($get($1.charvals, $geti($1.barcode, _S, 1)) + $1.checksum); //#9675
        } //#9675
        $1.checksum = $1.checksum % 35; //#9677
        if ($1.validatecheck) { //#9684
            if ($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum)) { //#9681
                $k[$j++] = 'bwipp.bc412badCheckDigit'; //#9680
                $k[$j++] = "Incorrect BC412 check digit provided"; //#9680
                $0.raiseerror(); //#9680
            } //#9680
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#9682
            $1.includecheck = true; //#9683
        } //#9683
        $1.encs = $a(["11111115", "13111212", "11131113", "12111213", "12121311", "13131111", "12111312", "11131212", "11121411", "11151111", "15111111", "11111511", "12131211", "13121112", "13111212", "11111214", "12121113", "11111313", "13111113", "11121213", "11141112", "11121312", "11141211", "14121111", "12121212", "11131311", "13121211", "12111411", "14111211", "11111412", "12111114", "14111112", "12141111", "11121114", "12131112", "12", "111"]); //#9696
        $1.sbs = $s($f(($f($1.barlen + 1) * 8) + 5)); //#9698
        if ($1.includecheck) { //#9702
            $1.txt = $a($f($1.barlen + 1)); //#9700
        } else { //#9702
            $1.txt = $a($1.barlen); //#9702
        } //#9702
        if ($1.includestartstop) { //#9712
            $puti($1.sbs, 0, $get($1.encs, 35)); //#9707
            $1.pos = 2; //#9708
            $1.txtpos = 3; //#9709
        } else { //#9712
            $1.pos = 0; //#9711
            $1.txtpos = 0; //#9712
        } //#9712
        for (var _x = 0, _w = $f($1.barlen - 1); _x <= _w; _x += 1) { //#9721
            $1.i = _x; //#9716
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#9717
            $puti($1.sbs, $1.pos, $get($1.encs, $1.indx)); //#9718
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), ($1.i * 12) + $1.txtpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9719
            $1.pos = $1.pos + 8; //#9720
        } //#9720
        if ($1.includecheck) { //#9732
            $puti($1.sbs, $1.pos, $get($1.encs, $1.checksum)); //#9725
            if ($1.includecheckintext) { //#9729
                $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum, 1), $f(($1.barlen * 12) + $1.txtpos), $1.textyoffset, $1.textfont, $1.textsize])); //#9727
            } else { //#9729
                $put($1.txt, $1.barlen, $a(["", $f(($1.barlen * 12) + $1.txtpos), $1.textyoffset, $1.textfont, $1.textsize])); //#9729
            } //#9729
            $1.pos = $1.pos + 8; //#9731
        } //#9731
        if ($1.includestartstop) { //#9738
            $puti($1.sbs, $1.pos, $get($1.encs, 36)); //#9736
            $1.pos = $1.pos + 2; //#9737
        } //#9737
        $1.sbs = $geti($1.sbs, 0, $1.pos); //#9741
        $k[$j++] = Infinity; //#9750
        $k[$j++] = Infinity; //#9744
        var _1u = $1.sbs; //#9744
        for (var _1v = 0, _1w = _1u.length; _1v < _1w; _1v++) { //#9744
            $k[$j++] = $f($get(_1u, _1v) - 48); //#9744
        } //#9744
        var _1y = $a(); //#9744
        $k[$j++] = Infinity; //#9745
        for (var _20 = 0, _21 = ~~(($1.sbs.length + 1) / 2); _20 < _21; _20++) { //#9745
            $k[$j++] = $1.height; //#9745
        } //#9745
        var _23 = $a(); //#9745
        $k[$j++] = Infinity; //#9746
        for (var _25 = 0, _26 = ~~(($1.sbs.length + 1) / 2); _25 < _26; _25++) { //#9746
            $k[$j++] = 0; //#9746
        } //#9746
        var _27 = $a(); //#9746
        $k[$j++] = 'ren'; //#9749
        $k[$j++] = $0.renlinear; //#9749
        $k[$j++] = 'sbs'; //#9749
        $k[$j++] = _1y; //#9749
        $k[$j++] = 'bhs'; //#9749
        $k[$j++] = _23; //#9749
        $k[$j++] = 'bbs'; //#9749
        $k[$j++] = _27; //#9749
        if ($1.includetext) { //#9749
            $k[$j++] = 'txt'; //#9748
            $k[$j++] = $1.txt; //#9748
        } //#9748
        $k[$j++] = 'opt'; //#9750
        $k[$j++] = $1.options; //#9750
        var _2B = $d(); //#9750
        $k[$j++] = _2B; //#9753
        if (!$1.dontdraw) { //#9753
            $0.renlinear(); //#9753
        } //#9753
    };
    $0.rationalizedCodabar = function() {
        var $1 = {}; //#9782
        $1.options = $k[--$j]; //#9784
        $1.barcode = $k[--$j]; //#9785
        $1.dontdraw = false; //#9787
        $1.altstartstop = false; //#9788
        $1.includecheck = false; //#9789
        $1.validatecheck = false; //#9790
        $1.includetext = false; //#9791
        $1.includecheckintext = false; //#9792
        $1.textfont = "OCR-B"; //#9793
        $1.textsize = 10; //#9794
        $1.textyoffset = -8; //#9795
        $1.height = 1; //#9796
        $forall($1.options, function() { //#9807
            var _3 = $k[--$j]; //#9807
            $1[$k[--$j]] = _3; //#9807
        }); //#9807
        $1.textfont = "" + $1.textfont; //#9809
        $1.textsize = +$1.textsize; //#9810
        $1.textyoffset = +$1.textyoffset; //#9811
        $1.height = +$1.height; //#9812
        $1.encs = $a(["11111331", "11113311", "11131131", "33111111", "11311311", "31111311", "13111131", "13113111", "13311111", "31131111", "11133111", "11331111", "31113131", "31311131", "31313111", "11313131", "11331311", "13131131", "11131331", "11133311"]); //#9820
        if ($1.altstartstop) { //#9826
            $1.barchars = "0123456789-$:/.+TN*E"; //#9824
        } else { //#9826
            $1.barchars = "0123456789-$:/.+ABCD"; //#9826
        } //#9826
        $1.charvals = new Map; //#9828
        for (var _B = 0; _B <= 19; _B += 1) { //#9829
            $put($1.charvals, $geti($1.barchars, _B, 1), _B); //#9829
        } //#9829
        $1.bodyvals = new Map; //#9830
        for (var _F = 0; _F <= 15; _F += 1) { //#9831
            $put($1.bodyvals, $geti($1.barchars, _F, 1), _F); //#9831
        } //#9831
        $1.ssvals = new Map; //#9832
        for (var _J = 16; _J <= 19; _J += 1) { //#9833
            $put($1.ssvals, $geti($1.barchars, _J, 1), _J); //#9833
        } //#9833
        var _Q = $get($1.ssvals, $geti($1.barcode, 0, 1)) !== undefined; //#9836
        var _V = $get($1.ssvals, $geti($1.barcode, $1.barcode.length - 1, 1)) !== undefined; //#9837
        if ((!_Q) || (!_V)) { //#9843
            if ($1.altstartstop) { //#9841
                $k[$j++] = 'bwipp.rationalizedCodabarBadAltStartStop'; //#9839
                $k[$j++] = "Codabar start and stop characters must be one of E N T or *"; //#9839
                $0.raiseerror(); //#9839
            } else { //#9841
                $k[$j++] = 'bwipp.rationalizedCodabarBadStartStop'; //#9841
                $k[$j++] = "Codabar start and stop characters must be one of A B C or D"; //#9841
                $0.raiseerror(); //#9841
            } //#9841
        } //#9841
        for (var _Z = 1, _Y = $1.barcode.length - 2; _Z <= _Y; _Z += 1) { //#9848
            var _d = $get($1.bodyvals, $geti($1.barcode, _Z, 1)) !== undefined; //#9845
            if (!_d) { //#9847
                $k[$j++] = 'bwipp.rationalizedCodabarBadCharacter'; //#9846
                $k[$j++] = "Codabar body must contain only digits and symbols - $ : / . +"; //#9846
                $0.raiseerror(); //#9846
            } //#9846
        } //#9846
        $k[$j++] = 'barlen'; //#9850
        $k[$j++] = $1.barcode.length; //#9850
        if ($1.validatecheck) { //#9850
            var _g = $k[--$j]; //#9850
            $k[$j++] = $f(_g - 1); //#9850
        } //#9850
        var _h = $k[--$j]; //#9850
        $1[$k[--$j]] = _h; //#9850
        $1.checksum = 0; //#9852
        for (var _l = 0, _k = $f($1.barlen - 2); _l <= _k; _l += 1) { //#9856
            $1.checksum = $f($get($1.charvals, $geti($1.barcode, _l, 1)) + $1.checksum); //#9855
        } //#9855
        $1.checksum = $f($get($1.charvals, $geti($1.barcode, $1.barcode.length - 1, 1)) + $1.checksum); //#9858
        $1.checksum = ($f(16 - ($1.checksum % 16))) % 16; //#9859
        if ($1.validatecheck) { //#9869
            if ($get($1.barcode, $f($1.barlen - 1)) != $get($1.barchars, $1.checksum)) { //#9863
                $k[$j++] = 'bwipp.rationalizedCodabarBadCheckDigit'; //#9862
                $k[$j++] = "Incorrect Codabar check digit provided"; //#9862
                $0.raiseerror(); //#9862
            } //#9862
            var _16 = $s($1.barlen); //#9864
            $puti(_16, 0, $geti($1.barcode, 0, $f($1.barlen - 1))); //#9865
            $puti(_16, $f($1.barlen - 1), $geti($1.barcode, $1.barlen, 1)); //#9866
            $1.barcode = _16; //#9867
            $1.includecheck = true; //#9868
        } //#9868
        $k[$j++] = 'sbs'; //#9871
        $k[$j++] = $1.barlen; //#9871
        if ($1.includecheck) { //#9871
            var _1G = $k[--$j]; //#9871
            $k[$j++] = $f(_1G + 1); //#9871
        } //#9871
        var _1I = $s($k[--$j] * 8); //#9871
        $1[$k[--$j]] = _1I; //#9871
        $k[$j++] = 'txt'; //#9872
        $k[$j++] = $1.barlen; //#9872
        if ($1.includecheck) { //#9872
            var _1M = $k[--$j]; //#9872
            $k[$j++] = $f(_1M + 1); //#9872
        } //#9872
        var _1O = $a($k[--$j]); //#9872
        $1[$k[--$j]] = _1O; //#9872
        $1.xpos = 0; //#9874
        for (var _1S = 0, _1R = $f($1.barlen - 2); _1S <= _1R; _1S += 1) { //#9884
            $1.i = _1S; //#9876
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#9877
            $1.enc = $get($1.encs, $1.indx); //#9878
            $puti($1.sbs, $1.i * 8, $1.enc); //#9879
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9880
            for (var _1o = 0; _1o <= 7; _1o += 1) { //#9883
                $1.xpos = $f($f($get($1.enc, _1o) - 48) + $1.xpos); //#9882
            } //#9882
        } //#9882
        if ($1.includecheck) { //#9908
            $puti($1.sbs, $f(($1.barlen * 8) - 8), $get($1.encs, $1.checksum)); //#9889
            if ($1.includecheckintext) { //#9893
                $put($1.txt, $f($1.barlen - 1), $a([$geti($1.barchars, $1.checksum, 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9891
            } else { //#9893
                $put($1.txt, $f($1.barlen - 1), $a([" ", $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9893
            } //#9893
            for (var _2G = 0; _2G <= 7; _2G += 1) { //#9897
                $1.xpos = $f($f($get($get($1.encs, $1.checksum), _2G) - 48) + $1.xpos); //#9896
            } //#9896
            $1.indx = $get($1.charvals, $geti($1.barcode, $f($1.barlen - 1), 1)); //#9899
            $1.enc = $get($1.encs, $1.indx); //#9900
            $puti($1.sbs, $1.barlen * 8, $1.enc); //#9901
            $put($1.txt, $1.barlen, $a([$geti($1.barcode, $f($1.barlen - 1), 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9902
        } else { //#9908
            $1.indx = $get($1.charvals, $geti($1.barcode, $f($1.barlen - 1), 1)); //#9905
            $1.enc = $get($1.encs, $1.indx); //#9906
            $puti($1.sbs, $f(($1.barlen * 8) - 8), $1.enc); //#9907
            $put($1.txt, $f($1.barlen - 1), $a([$geti($1.barcode, $f($1.barlen - 1), 1), $1.xpos, $1.textyoffset, $1.textfont, $1.textsize])); //#9908
        } //#9908
        $k[$j++] = Infinity; //#9920
        $k[$j++] = Infinity; //#9914
        $forall($1.sbs, function() { //#9914
            var _33 = $k[--$j]; //#9914
            $k[$j++] = $f(_33 - 48); //#9914
        }); //#9914
        var _34 = $a(); //#9914
        $k[$j++] = Infinity; //#9915
        for (var _36 = 0, _37 = ~~(($1.sbs.length + 1) / 2); _36 < _37; _36++) { //#9915
            $k[$j++] = $1.height; //#9915
        } //#9915
        var _39 = $a(); //#9915
        $k[$j++] = Infinity; //#9916
        for (var _3B = 0, _3C = ~~(($1.sbs.length + 1) / 2); _3B < _3C; _3B++) { //#9916
            $k[$j++] = 0; //#9916
        } //#9916
        var _3D = $a(); //#9916
        $k[$j++] = 'ren'; //#9919
        $k[$j++] = $0.renlinear; //#9919
        $k[$j++] = 'sbs'; //#9919
        $k[$j++] = _34; //#9919
        $k[$j++] = 'bhs'; //#9919
        $k[$j++] = _39; //#9919
        $k[$j++] = 'bbs'; //#9919
        $k[$j++] = _3D; //#9919
        if ($1.includetext) { //#9919
            $k[$j++] = 'txt'; //#9918
            $k[$j++] = $1.txt; //#9918
        } //#9918
        $k[$j++] = 'opt'; //#9920
        $k[$j++] = $1.options; //#9920
        var _3H = $d(); //#9920
        $k[$j++] = _3H; //#9923
        if (!$1.dontdraw) { //#9923
            $0.renlinear(); //#9923
        } //#9923
    };
    $0.onecode = function() {
        var $1 = {}; //#9952
        $1.options = $k[--$j]; //#9954
        $1.barcode = $k[--$j]; //#9955
        $1.dontdraw = false; //#9957
        $1.height = 0.15; //#9958
        $forall($1.options, function() { //#9969
            var _3 = $k[--$j]; //#9969
            $1[$k[--$j]] = _3; //#9969
        }); //#9969
        $1.height = +$1.height; //#9971
        $1.barlen = $1.barcode.length; //#9973
        $k[$j++] = Infinity; //#9982
        var _9 = $eq($geti($1.barcode, 5, 1), "9") ? 14 : 11; //#9979
        $k[$j++] = 2; //#9982
        $k[$j++] = 2; //#9982
        $k[$j++] = 5; //#9982
        $k[$j++] = 5; //#9982
        $k[$j++] = _9; //#9982
        $k[$j++] = _9; //#9982
        $k[$j++] = 20; //#9982
        $k[$j++] = 20; //#9982
        $k[$j++] = 25; //#9982
        $k[$j++] = 25; //#9982
        $k[$j++] = 29; //#9982
        $k[$j++] = 29; //#9982
        $1.txtdict = $d(); //#9983
        $1.txt = $a($1.barlen + 6); //#9984
        $1.spacecnt = 0; //#9985
        for (var _F = 0, _E = $1.barlen - 1; _F <= _E; _F += 1) { //#9993
            $1.i = _F; //#9987
            var _I = $get($1.txtdict, $1.i) !== undefined; //#9988
            if (_I) { //#9991
                $put($1.txt, $1.i + $1.spacecnt, $a([" ", 0, 0, "", 0])); //#9989
                $1.spacecnt = $1.spacecnt + 1; //#9990
            } //#9990
            $put($1.txt, $1.i + $1.spacecnt, $a([$geti($1.barcode, $1.i, 1), 0, 0, "", 0])); //#9992
        } //#9992
        $1.txt = $geti($1.txt, 0, $1.barcode.length + $1.spacecnt); //#9994
        $1.normalize = function() {
            $1.base = $k[--$j]; //#9997
            $1.num = $k[--$j]; //#9998
            for (var _c = $1.num.length - 1; _c >= 1; _c -= 1) { //#10003
                $1.i = _c; //#10000
                var _d = $1.num; //#10001
                var _e = $1.i; //#10001
                $put(_d, _e - 1, $f($get(_d, _e - 1) + (~~($get($1.num, $1.i) / $1.base)))); //#10001
                $put($1.num, $1.i, $get($1.num, $1.i) % $1.base); //#10002
            } //#10002
            for (;;) { //#10009
                if ($lt($get($1.num, 0), $1.base)) { //#10005
                    break; //#10005
                } //#10005
                $k[$j++] = Infinity; //#10006
                $k[$j++] = 0; //#10006
                $forall($1.num); //#10006
                $1.num = $a(); //#10006
                $put($1.num, 0, $f($get($1.num, 0) + (~~($get($1.num, 1) / $1.base)))); //#10007
                $put($1.num, 1, $get($1.num, 1) % $1.base); //#10008
            } //#10008
            $k[$j++] = Infinity; //#10011
            $1.i = true; //#10011
            var _15 = $1.num; //#10011
            for (var _16 = 0, _17 = _15.length; _16 < _17; _16++) { //#10011
                var _18 = $get(_15, _16); //#10011
                $k[$j++] = _18; //#10011
                if ((_18 == 0) && $1.i) { //#10011
                    $j--; //#10011
                } else { //#10011
                    $1.i = false; //#10011
                } //#10011
            } //#10011
            $1.num = $a(); //#10011
            if ($1.num.length == 0) { //#10012
                $1.num = $a([0]); //#10012
            } //#10012
            $k[$j++] = $1.num; //#10013
        }; //#10014
        $1.bigadd = function() {
            var _1E = $k[--$j]; //#10017
            var _1F = $k[--$j]; //#10017
            $1.offset = Math.abs(_1E.length - _1F.length); //#10018
            if (_1E.length < _1F.length) { //#10019
                var _ = _1E; //#10019
                _1E = _1F; //#10019
                _1F = _; //#10019
            } //#10019
            $1.a = _1E; //#10020
            $1.b = _1F; //#10020
            for (var _1I = 0, _1H = $1.b.length - 1; _1I <= _1H; _1I += 1) { //#10023
                var _1J = $1.a; //#10022
                var _1K = $1.offset; //#10022
                $put(_1J, _1I + _1K, $f($get(_1J, _1I + _1K) + $get($1.b, _1I))); //#10022
            } //#10022
            $k[$j++] = $1.a; //#10024
        }; //#10025
        if ($1.barlen == 20) { //#10028
            $k[$j++] = $a([0]); //#10028
        } //#10028
        if ($1.barlen == 25) { //#10029
            $k[$j++] = $a([1]); //#10029
        } //#10029
        if ($1.barlen == 29) { //#10030
            $k[$j++] = $a([1, 0, 0, 0, 0, 1]); //#10030
        } //#10030
        if ($1.barlen == 31) { //#10031
            $k[$j++] = $a([1, 0, 0, 0, 1, 0, 0, 0, 0, 1]); //#10031
        } //#10031
        var _1X = $k[--$j]; //#10032
        $k[$j++] = 'binval'; //#10032
        $k[$j++] = _1X; //#10032
        $k[$j++] = Infinity; //#10032
        var _1a = $geti($1.barcode, 20, $1.barlen - 20); //#10032
        for (var _1b = 0, _1c = _1a.length; _1b < _1c; _1b++) { //#10032
            $k[$j++] = $f($get(_1a, _1b) - 48); //#10032
        } //#10032
        var _1e = $a(); //#10032
        $k[$j++] = _1e; //#10032
        $1.bigadd(); //#10032
        var _1f = $k[--$j]; //#10032
        $1[$k[--$j]] = _1f; //#10032
        $k[$j++] = Infinity; //#10033
        var _1h = $1.binval; //#10033
        for (var _1i = 0, _1j = _1h.length; _1i < _1j; _1i++) { //#10033
            $k[$j++] = $get(_1h, _1i); //#10033
        } //#10033
        $k[$j++] = $f($get($1.barcode, 0) - 48); //#10033
        $1.binval = $a(); //#10033
        $k[$j++] = Infinity; //#10034
        var _1o = $1.binval; //#10034
        for (var _1p = 0, _1q = _1o.length; _1p < _1q; _1p++) { //#10034
            $k[$j++] = $get(_1o, _1p) * 5; //#10034
        } //#10034
        var _1s = $a(); //#10034
        $k[$j++] = 'binval'; //#10034
        $k[$j++] = _1s; //#10034
        $k[$j++] = $a([$f($get($1.barcode, 1) - 48)]); //#10034
        $1.bigadd(); //#10034
        $k[$j++] = 10; //#10034
        $1.normalize(); //#10034
        var _1w = $k[--$j]; //#10034
        $1[$k[--$j]] = _1w; //#10034
        $k[$j++] = Infinity; //#10035
        var _1y = $1.binval; //#10035
        for (var _1z = 0, _20 = _1y.length; _1z < _20; _1z++) { //#10035
            $k[$j++] = $get(_1y, _1z); //#10035
        } //#10035
        var _23 = $geti($1.barcode, 2, 18); //#10035
        for (var _24 = 0, _25 = _23.length; _24 < _25; _24++) { //#10035
            $k[$j++] = $f($get(_23, _24) - 48); //#10035
        } //#10035
        $1.binval = $a(); //#10035
        $1.bytes = $a(13); //#10038
        $k[$j++] = Infinity; //#10039
        var _29 = $1.binval; //#10039
        for (var _2A = 0, _2B = _29.length; _2A < _2B; _2A++) { //#10039
            $k[$j++] = $get(_29, _2A); //#10039
        } //#10039
        $1.bintmp = $a(); //#10039
        for (var _2E = 12; _2E >= 0; _2E -= 1) { //#10049
            $1.i = _2E; //#10041
            for (var _2H = 0, _2G = $1.bintmp.length - 2; _2H <= _2G; _2H += 1) { //#10046
                $1.j = _2H; //#10043
                var _2I = $1.bintmp; //#10044
                var _2J = $1.j; //#10044
                $put(_2I, _2J + 1, $f($get(_2I, _2J + 1) + (($get($1.bintmp, $1.j) % 256) * 10))); //#10044
                $put($1.bintmp, $1.j, ~~($get($1.bintmp, $1.j) / 256)); //#10045
            } //#10045
            $put($1.bytes, $1.i, $get($1.bintmp, $1.bintmp.length - 1) % 256); //#10047
            var _2Y = $1.bintmp; //#10048
            var _2Z = $1.bintmp; //#10048
            $put(_2Y, _2Z.length - 1, ~~($get(_2Y, _2Z.length - 1) / 256)); //#10048
        } //#10048
        $1.fcs = 2047; //#10052
        $1.dat = $get($1.bytes, 0) << 5; //#10053
        for (var _2d = 0, _2e = 6; _2d < _2e; _2d++) { //#10062
            if ((($1.fcs ^ $1.dat) & 1024) != 0) { //#10058
                $1.fcs = ($1.fcs << 1) ^ 3893; //#10056
            } else { //#10058
                $1.fcs = $1.fcs << 1; //#10058
            } //#10058
            $1.fcs = $1.fcs & 2047; //#10060
            $1.dat = $1.dat << 1; //#10061
        } //#10061
        for (var _2l = 1; _2l <= 12; _2l += 1) { //#10074
            $1.dat = $get($1.bytes, _2l) << 3; //#10064
            for (var _2o = 0, _2p = 8; _2o < _2p; _2o++) { //#10073
                if ((($1.fcs ^ $1.dat) & 1024) != 0) { //#10069
                    $1.fcs = ($1.fcs << 1) ^ 3893; //#10067
                } else { //#10069
                    $1.fcs = $1.fcs << 1; //#10069
                } //#10069
                $1.fcs = $1.fcs & 2047; //#10071
                $1.dat = $1.dat << 1; //#10072
            } //#10072
        } //#10072
        $1.codewords = $a(10); //#10077
        for (var _2x = 9; _2x >= 0; _2x -= 1) { //#10092
            $1.i = _2x; //#10079
            if ($1.i == 9) { //#10083
                $1.b = 636; //#10081
            } else { //#10083
                $1.b = 1365; //#10083
            } //#10083
            for (var _31 = 0, _30 = $1.binval.length - 2; _31 <= _30; _31 += 1) { //#10089
                $1.j = _31; //#10086
                var _32 = $1.binval; //#10087
                var _33 = $1.j; //#10087
                $put(_32, _33 + 1, $f($get(_32, _33 + 1) + (($get($1.binval, $1.j) % $1.b) * 10))); //#10087
                $put($1.binval, $1.j, ~~($get($1.binval, $1.j) / $1.b)); //#10088
            } //#10088
            $put($1.codewords, $1.i, $get($1.binval, $1.binval.length - 1) % $1.b); //#10090
            var _3L = $1.binval; //#10091
            var _3M = $1.binval; //#10091
            $put(_3L, _3M.length - 1, ~~($get(_3L, _3M.length - 1) / $1.b)); //#10091
        } //#10091
        $put($1.codewords, 9, $get($1.codewords, 9) * 2); //#10095
        if (($1.fcs & 1024) != 0) { //#10098
            $put($1.codewords, 0, $f($get($1.codewords, 0) + 659)); //#10097
        } //#10097
        $1.tab513 = $a([31, 7936, 47, 7808, 55, 7552, 59, 7040, 61, 6016, 62, 3968, 79, 7744, 87, 7488, 91, 6976, 93, 5952, 94, 3904, 103, 7360, 107, 6848, 109, 5824, 110, 3776, 115, 6592, 117, 5568, 118, 3520, 121, 5056, 122, 3008, 124, 1984, 143, 7712, 151, 7456, 155, 6944, 157, 5920, 158, 3872, 167, 7328, 171, 6816, 173, 5792, 174, 3744, 179, 6560, 181, 5536, 182, 3488, 185, 5024, 186, 2976, 188, 1952, 199, 7264, 203, 6752, 205, 5728, 206, 3680, 211, 6496, 213, 5472, 214, 3424, 217, 4960, 218, 2912, 220, 1888, 227, 6368, 229, 5344, 230, 3296, 233, 4832, 234, 2784, 236, 1760, 241, 4576, 242, 2528, 244, 1504, 248, 992, 271, 7696, 279, 7440, 283, 6928, 285, 5904, 286, 3856, 295, 7312, 299, 6800, 301, 5776, 302, 3728, 307, 6544, 309, 5520, 310, 3472, 313, 5008, 314, 2960, 316, 1936, 327, 7248, 331, 6736, 333, 5712, 334, 3664, 339, 6480, 341, 5456, 342, 3408, 345, 4944, 346, 2896, 348, 1872, 355, 6352, 357, 5328, 358, 3280, 361, 4816, 362, 2768, 364, 1744, 369, 4560, 370, 2512, 372, 1488, 376, 976, 391, 7216, 395, 6704, 397, 5680, 398, 3632, 403, 6448, 405, 5424, 406, 3376, 409, 4912, 410, 2864, 412, 1840, 419, 6320, 421, 5296, 422, 3248, 425, 4784, 426, 2736, 428, 1712, 433, 4528, 434, 2480, 436, 1456, 440, 944, 451, 6256, 453, 5232, 454, 3184, 457, 4720, 458, 2672, 460, 1648, 465, 4464, 466, 2416, 468, 1392, 472, 880, 481, 4336, 482, 2288, 484, 1264, 488, 752, 527, 7688, 535, 7432, 539, 6920, 541, 5896, 542, 3848, 551, 7304, 555, 6792, 557, 5768, 558, 3720, 563, 6536, 565, 5512, 566, 3464, 569, 5000, 570, 2952, 572, 1928, 583, 7240, 587, 6728, 589, 5704, 590, 3656, 595, 6472, 597, 5448, 598, 3400, 601, 4936, 602, 2888, 604, 1864, 611, 6344, 613, 5320, 614, 3272, 617, 4808, 618, 2760, 620, 1736, 625, 4552, 626, 2504, 628, 1480, 632, 968, 647, 7208, 651, 6696, 653, 5672, 654, 3624, 659, 6440, 661, 5416, 662, 3368, 665, 4904, 666, 2856, 668, 1832, 675, 6312, 677, 5288, 678, 3240, 681, 4776, 682, 2728, 684, 1704, 689, 4520, 690, 2472, 692, 1448, 696, 936, 707, 6248, 709, 5224, 710, 3176, 713, 4712, 714, 2664, 716, 1640, 721, 4456, 722, 2408, 724, 1384, 728, 872, 737, 4328, 738, 2280, 740, 1256, 775, 7192, 779, 6680, 781, 5656, 782, 3608, 787, 6424, 789, 5400, 790, 3352, 793, 4888, 794, 2840, 796, 1816, 803, 6296, 805, 5272, 806, 3224, 809, 4760, 810, 2712, 812, 1688, 817, 4504, 818, 2456, 820, 1432, 824, 920, 835, 6232, 837, 5208, 838, 3160, 841, 4696, 842, 2648, 844, 1624, 849, 4440, 850, 2392, 852, 1368, 865, 4312, 866, 2264, 868, 1240, 899, 6200, 901, 5176, 902, 3128, 905, 4664, 906, 2616, 908, 1592, 913, 4408, 914, 2360, 916, 1336, 929, 4280, 930, 2232, 932, 1208, 961, 4216, 962, 2168, 964, 1144, 1039, 7684, 1047, 7428, 1051, 6916, 1053, 5892, 1054, 3844, 1063, 7300, 1067, 6788, 1069, 5764, 1070, 3716, 1075, 6532, 1077, 5508, 1078, 3460, 1081, 4996, 1082, 2948, 1084, 1924, 1095, 7236, 1099, 6724, 1101, 5700, 1102, 3652, 1107, 6468, 1109, 5444, 1110, 3396, 1113, 4932, 1114, 2884, 1116, 1860, 1123, 6340, 1125, 5316, 1126, 3268, 1129, 4804, 1130, 2756, 1132, 1732, 1137, 4548, 1138, 2500, 1140, 1476, 1159, 7204, 1163, 6692, 1165, 5668, 1166, 3620, 1171, 6436, 1173, 5412, 1174, 3364, 1177, 4900, 1178, 2852, 1180, 1828, 1187, 6308, 1189, 5284, 1190, 3236, 1193, 4772, 1194, 2724, 1196, 1700, 1201, 4516, 1202, 2468, 1204, 1444, 1219, 6244, 1221, 5220, 1222, 3172, 1225, 4708, 1226, 2660, 1228, 1636, 1233, 4452, 1234, 2404, 1236, 1380, 1249, 4324, 1250, 2276, 1287, 7188, 1291, 6676, 1293, 5652, 1294, 3604, 1299, 6420, 1301, 5396, 1302, 3348, 1305, 4884, 1306, 2836, 1308, 1812, 1315, 6292, 1317, 5268, 1318, 3220, 1321, 4756, 1322, 2708, 1324, 1684, 1329, 4500, 1330, 2452, 1332, 1428, 1347, 6228, 1349, 5204, 1350, 3156, 1353, 4692, 1354, 2644, 1356, 1620, 1361, 4436, 1362, 2388, 1377, 4308, 1378, 2260, 1411, 6196, 1413, 5172, 1414, 3124, 1417, 4660, 1418, 2612, 1420, 1588, 1425, 4404, 1426, 2356, 1441, 4276, 1442, 2228, 1473, 4212, 1474, 2164, 1543, 7180, 1547, 6668, 1549, 5644, 1550, 3596, 1555, 6412, 1557, 5388, 1558, 3340, 1561, 4876, 1562, 2828, 1564, 1804, 1571, 6284, 1573, 5260, 1574, 3212, 1577, 4748, 1578, 2700, 1580, 1676, 1585, 4492, 1586, 2444, 1603, 6220, 1605, 5196, 1606, 3148, 1609, 4684, 1610, 2636, 1617, 4428, 1618, 2380, 1633, 4300, 1634, 2252, 1667, 6188, 1669, 5164, 1670, 3116, 1673, 4652, 1674, 2604, 1681, 4396, 1682, 2348, 1697, 4268, 1698, 2220, 1729, 4204, 1730, 2156, 1795, 6172, 1797, 5148, 1798, 3100, 1801, 4636, 1802, 2588, 1809, 4380, 1810, 2332, 1825, 4252, 1826, 2204, 1857, 4188, 1858, 2140, 1921, 4156, 1922, 2108, 2063, 7682, 2071, 7426, 2075, 6914, 2077, 5890, 2078, 3842, 2087, 7298, 2091, 6786, 2093, 5762, 2094, 3714, 2099, 6530, 2101, 5506, 2102, 3458, 2105, 4994, 2106, 2946, 2119, 7234, 2123, 6722, 2125, 5698, 2126, 3650, 2131, 6466, 2133, 5442, 2134, 3394, 2137, 4930, 2138, 2882, 2147, 6338, 2149, 5314, 2150, 3266, 2153, 4802, 2154, 2754, 2161, 4546, 2162, 2498, 2183, 7202, 2187, 6690, 2189, 5666, 2190, 3618, 2195, 6434, 2197, 5410, 2198, 3362, 2201, 4898, 2202, 2850, 2211, 6306, 2213, 5282, 2214, 3234, 2217, 4770, 2218, 2722, 2225, 4514, 2226, 2466, 2243, 6242, 2245, 5218, 2246, 3170, 2249, 4706, 2250, 2658, 2257, 4450, 2258, 2402, 2273, 4322, 2311, 7186, 2315, 6674, 2317, 5650, 2318, 3602, 2323, 6418, 2325, 5394, 2326, 3346, 2329, 4882, 2330, 2834, 2339, 6290, 2341, 5266, 2342, 3218, 2345, 4754, 2346, 2706, 2353, 4498, 2354, 2450, 2371, 6226, 2373, 5202, 2374, 3154, 2377, 4690, 2378, 2642, 2385, 4434, 2401, 4306, 2435, 6194, 2437, 5170, 2438, 3122, 2441, 4658, 2442, 2610, 2449, 4402, 2465, 4274, 2497, 4210, 2567, 7178, 2571, 6666, 2573, 5642, 2574, 3594, 2579, 6410, 2581, 5386, 2582, 3338, 2585, 4874, 2586, 2826, 2595, 6282, 2597, 5258, 2598, 3210, 2601, 4746, 2602, 2698, 2609, 4490, 2627, 6218, 2629, 5194, 2630, 3146, 2633, 4682, 2641, 4426, 2657, 4298, 2691, 6186, 2693, 5162, 2694, 3114, 2697, 4650, 2705, 4394, 2721, 4266, 2753, 4202, 2819, 6170, 2821, 5146, 2822, 3098, 2825, 4634, 2833, 4378, 2849, 4250, 2881, 4186, 2945, 4154, 3079, 7174, 3083, 6662, 3085, 5638, 3086, 3590, 3091, 6406, 3093, 5382, 3094, 3334, 3097, 4870, 3107, 6278, 3109, 5254, 3110, 3206, 3113, 4742, 3121, 4486, 3139, 6214, 3141, 5190, 3145, 4678, 3153, 4422, 3169, 4294, 3203, 6182, 3205, 5158, 3209, 4646, 3217, 4390, 3233, 4262, 3265, 4198, 3331, 6166, 3333, 5142, 3337, 4630, 3345, 4374, 3361, 4246, 3393, 4182, 3457, 4150, 3587, 6158, 3589, 5134, 3593, 4622, 3601, 4366, 3617, 4238, 3649, 4174, 3713, 4142, 3841, 4126, 4111, 7681, 4119, 7425, 4123, 6913, 4125, 5889, 4135, 7297, 4139, 6785, 4141, 5761, 4147, 6529, 4149, 5505, 4153, 4993, 4167, 7233, 4171, 6721, 4173, 5697, 4179, 6465, 4181, 5441, 4185, 4929, 4195, 6337, 4197, 5313, 4201, 4801, 4209, 4545, 4231, 7201, 4235, 6689, 4237, 5665, 4243, 6433, 4245, 5409, 4249, 4897, 4259, 6305, 4261, 5281, 4265, 4769, 4273, 4513, 4291, 6241, 4293, 5217, 4297, 4705, 4305, 4449, 4359, 7185, 4363, 6673, 4365, 5649, 4371, 6417, 4373, 5393, 4377, 4881, 4387, 6289, 4389, 5265, 4393, 4753, 4401, 4497, 4419, 6225, 4421, 5201, 4425, 4689, 4483, 6193, 4485, 5169, 4489, 4657, 4615, 7177, 4619, 6665, 4621, 5641, 4627, 6409, 4629, 5385, 4633, 4873, 4643, 6281, 4645, 5257, 4649, 4745, 4675, 6217, 4677, 5193, 4739, 6185, 4741, 5161, 4867, 6169, 4869, 5145, 5127, 7173, 5131, 6661, 5133, 5637, 5139, 6405, 5141, 5381, 5155, 6277, 5157, 5253, 5187, 6213, 5251, 6181, 5379, 6165, 5635, 6157, 6151, 7171, 6155, 6659, 6163, 6403, 6179, 6275, 6211, 5189, 4681, 4433, 4321, 3142, 2634, 2386, 2274, 1612, 1364, 1252, 856, 744, 496]); //#10188
        $1.tab213 = $a([3, 6144, 5, 5120, 6, 3072, 9, 4608, 10, 2560, 12, 1536, 17, 4352, 18, 2304, 20, 1280, 24, 768, 33, 4224, 34, 2176, 36, 1152, 40, 640, 48, 384, 65, 4160, 66, 2112, 68, 1088, 72, 576, 80, 320, 96, 192, 129, 4128, 130, 2080, 132, 1056, 136, 544, 144, 288, 257, 4112, 258, 2064, 260, 1040, 264, 528, 513, 4104, 514, 2056, 516, 1032, 1025, 4100, 1026, 2052, 2049, 4098, 4097, 2050, 1028, 520, 272, 160]); //#10197
        $1.chars = $a(10); //#10199
        for (var _3Z = 0; _3Z <= 9; _3Z += 1) { //#10208
            $1.i = _3Z; //#10201
            var _3c = $get($1.codewords, $1.i); //#10202
            $k[$j++] = _3c; //#10206
            if (_3c <= 1286) { //#10205
                var _3f = $get($1.tab513, $k[--$j]); //#10203
                $k[$j++] = _3f; //#10203
            } else { //#10205
                var _3i = $get($1.tab213, $f($k[--$j] - 1287)); //#10205
                $k[$j++] = _3i; //#10205
            } //#10205
            $put($1.chars, $1.i, $k[--$j]); //#10207
        } //#10207
        for (var _3m = 9; _3m >= 0; _3m -= 1) { //#10215
            $1.i = _3m; //#10211
            if (((~~Math.pow(2, $1.i)) & $1.fcs) != 0) { //#10214
                $put($1.chars, $1.i, $get($1.chars, $1.i) ^ 8191); //#10213
            } //#10213
        } //#10213
        $1.barmap = $a([7, 2, 4, 3, 1, 10, 0, 0, 9, 12, 2, 8, 5, 5, 6, 11, 8, 9, 3, 1, 0, 1, 5, 12, 2, 5, 1, 8, 4, 4, 9, 11, 6, 3, 8, 10, 3, 9, 7, 6, 5, 11, 1, 4, 8, 5, 2, 12, 9, 10, 0, 2, 7, 1, 6, 7, 3, 6, 4, 9, 0, 3, 8, 6, 6, 4, 2, 7, 1, 1, 9, 9, 7, 10, 5, 2, 4, 0, 3, 8, 6, 2, 0, 4, 8, 11, 1, 0, 9, 8, 3, 12, 2, 6, 7, 7, 5, 1, 4, 10, 1, 12, 6, 9, 7, 3, 8, 0, 5, 8, 9, 7, 4, 6, 2, 10, 3, 4, 0, 5, 8, 4, 5, 7, 7, 11, 1, 9, 6, 0, 9, 6, 0, 6, 4, 8, 2, 1, 3, 2, 5, 9, 8, 12, 4, 11, 6, 1, 9, 5, 7, 4, 3, 3, 1, 2, 0, 7, 2, 0, 1, 3, 4, 1, 6, 10, 3, 5, 8, 7, 9, 4, 2, 11, 5, 6, 0, 8, 7, 12, 4, 2, 8, 1, 5, 10, 3, 0, 9, 3, 0, 9, 6, 5, 2, 4, 7, 8, 1, 7, 5, 0, 4, 5, 2, 3, 0, 10, 6, 12, 9, 2, 3, 11, 1, 6, 8, 8, 7, 9, 5, 4, 0, 11, 1, 5, 2, 2, 9, 1, 4, 12, 8, 3, 6, 6, 7, 0, 3, 7, 4, 7, 7, 5, 0, 12, 1, 11, 2, 9, 9, 0, 6, 8, 5, 3, 3, 10, 8, 2]); //#10232
        $1.bbs = $a(65); //#10234
        $1.bhs = $a(65); //#10235
        for (var _3x = 0; _3x <= 64; _3x += 1) { //#10256
            $1.i = _3x; //#10237
            $1.dec = ($get($1.chars, $get($1.barmap, $1.i * 4)) & (~~Math.pow(2, $get($1.barmap, ($1.i * 4) + 1)))) != 0; //#10238
            $1.asc = ($get($1.chars, $get($1.barmap, ($1.i * 4) + 2)) & (~~Math.pow(2, $get($1.barmap, ($1.i * 4) + 3)))) != 0; //#10239
            if ((!$1.dec) && (!$1.asc)) { //#10243
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#10241
                $put($1.bhs, $1.i, (2 * $1.height) / 8); //#10242
            } //#10242
            if ((!$1.dec) && $1.asc) { //#10247
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#10245
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#10246
            } //#10246
            if ($1.dec && (!$1.asc)) { //#10251
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#10249
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#10250
            } //#10250
            if ($1.dec && $1.asc) { //#10255
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#10253
                $put($1.bhs, $1.i, (8 * $1.height) / 8); //#10254
            } //#10254
        } //#10254
        $k[$j++] = Infinity; //#10269
        var _4k = $1.bbs; //#10261
        var _4l = $1.bhs; //#10262
        $k[$j++] = Infinity; //#10263
        for (var _4n = 0, _4o = $1.bhs.length - 1; _4n < _4o; _4n++) { //#10263
            $k[$j++] = 1.44; //#10263
            $k[$j++] = 1.872; //#10263
        } //#10263
        $k[$j++] = 1.44; //#10263
        var _4p = $a(); //#10263
        $k[$j++] = 'ren'; //#10269
        $k[$j++] = $0.renlinear; //#10269
        $k[$j++] = 'bbs'; //#10269
        $k[$j++] = _4k; //#10269
        $k[$j++] = 'bhs'; //#10269
        $k[$j++] = _4l; //#10269
        $k[$j++] = 'sbs'; //#10269
        $k[$j++] = _4p; //#10269
        $k[$j++] = 'txt'; //#10269
        $k[$j++] = $1.txt; //#10269
        $k[$j++] = 'textxalign'; //#10269
        $k[$j++] = "left"; //#10269
        $k[$j++] = 'textfont'; //#10269
        $k[$j++] = "OCR-B"; //#10269
        $k[$j++] = 'textyoffset'; //#10269
        $k[$j++] = 1; //#10269
        $k[$j++] = 'textxoffset'; //#10269
        $k[$j++] = -0.3; //#10269
        $k[$j++] = 'opt'; //#10269
        $k[$j++] = $1.options; //#10269
        var _4s = $d(); //#10269
        $k[$j++] = _4s; //#10272
        if (!$1.dontdraw) { //#10272
            $0.renlinear(); //#10272
        } //#10272
    };
    $0.postnet = function() {
        var $1 = {}; //#10301
        $1.options = $k[--$j]; //#10303
        $1.barcode = $k[--$j]; //#10304
        $1.dontdraw = false; //#10306
        $1.includetext = false; //#10307
        $1.validatecheck = false; //#10308
        $1.includecheckintext = false; //#10309
        $1.textfont = "OCR-B"; //#10310
        $1.textsize = 10; //#10311
        $1.textyoffset = -8; //#10312
        $1.height = 0.125; //#10313
        $forall($1.options, function() { //#10324
            var _3 = $k[--$j]; //#10324
            $1[$k[--$j]] = _3; //#10324
        }); //#10324
        $1.textfont = "" + $1.textfont; //#10326
        $1.textsize = +$1.textsize; //#10327
        $1.textyoffset = +$1.textyoffset; //#10328
        $1.height = +$1.height; //#10329
        $k[$j++] = 'barlen'; //#10331
        $k[$j++] = $1.barcode.length; //#10331
        if ($1.validatecheck) { //#10331
            var _B = $k[--$j]; //#10331
            $k[$j++] = $f(_B - 1); //#10331
        } //#10331
        var _C = $k[--$j]; //#10331
        $1[$k[--$j]] = _C; //#10331
        if ((($1.barlen != 5) && ($1.barlen != 9)) && ($1.barlen != 11)) { //#10336
            $k[$j++] = 'bwipp.postnetBadLength'; //#10335
            $k[$j++] = "USPS POSTNET must be 5, 9 or 11 digits excluding check digit"; //#10335
            $0.raiseerror(); //#10335
        } //#10335
        $forall($1.barcode, function() { //#10341
            var _I = $k[--$j]; //#10338
            if ((_I < 48) || (_I > 57)) { //#10340
                $k[$j++] = 'bwipp.postnetBadCharacter'; //#10339
                $k[$j++] = "USPS POSTNET must contain only digits"; //#10339
                $0.raiseerror(); //#10339
            } //#10339
        }); //#10339
        $1.barchars = "0123456789"; //#10344
        $1.checksum = 0; //#10346
        for (var _L = 0, _K = $f($1.barlen - 1); _L <= _K; _L += 1) { //#10350
            $1.i = _L; //#10348
            $1.checksum = $f($1.checksum + $f($get($1.barcode, $1.i) - 48)); //#10349
        } //#10349
        $1.checksum = ($f(10 - ($1.checksum % 10))) % 10; //#10351
        if ($1.validatecheck) { //#10357
            if ($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum)) { //#10355
                $k[$j++] = 'bwipp.postnetBadCheckDigit'; //#10354
                $k[$j++] = "Incorrect USPS POSTNET check digit provided"; //#10354
                $0.raiseerror(); //#10354
            } //#10354
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#10356
        } //#10356
        $1.encs = $a(["55222", "22255", "22525", "22552", "25225", "25252", "25522", "52225", "52252", "52522", "5", "5"]); //#10364
        $1.bhs = $a($f(($1.barlen * 5) + 7)); //#10366
        $1.txt = $a($f($1.barlen + 1)); //#10367
        $1.enc = $get($1.encs, 10); //#10370
        $1.heights = $a($1.enc.length); //#10371
        for (var _m = 0, _l = $1.enc.length - 1; _m <= _l; _m += 1) { //#10375
            $1.j = _m; //#10373
            $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10374
        } //#10374
        $puti($1.bhs, 0, $1.heights); //#10376
        for (var _x = 0, _w = $f($1.barlen - 1); _x <= _w; _x += 1) { //#10388
            $1.i = _x; //#10379
            $1.enc = $get($1.encs, $f($get($1.barcode, $1.i) - 48)); //#10380
            $1.heights = $a($1.enc.length); //#10381
            for (var _17 = 0, _16 = $1.enc.length - 1; _17 <= _16; _17 += 1) { //#10385
                $1.j = _17; //#10383
                $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10384
            } //#10384
            $puti($1.bhs, ($1.i * 5) + 1, $1.heights); //#10386
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), (($1.i * 5) + 1) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10387
        } //#10387
        $1.enc = $get($1.encs, $1.checksum); //#10391
        $1.heights = $a($1.enc.length); //#10392
        for (var _1Y = 0, _1X = $1.enc.length - 1; _1Y <= _1X; _1Y += 1) { //#10396
            $1.j = _1Y; //#10394
            $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10395
        } //#10395
        $puti($1.bhs, $f(($1.barlen * 5) + 1), $1.heights); //#10397
        if ($1.includecheckintext) { //#10402
            $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum, 1), ($f(($1.barlen * 5) + 1)) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10400
        } else { //#10402
            $put($1.txt, $1.barlen, $a([" ", (($f(($1.barlen * 5) + 1)) * 72) / 25, $1.textyoffset, $1.textfont, $1.textsize])); //#10402
        } //#10402
        $1.enc = $get($1.encs, 11); //#10406
        $1.heights = $a($1.enc.length); //#10407
        for (var _26 = 0, _25 = $1.enc.length - 1; _26 <= _25; _26 += 1) { //#10411
            $1.j = _26; //#10409
            $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10410
        } //#10410
        $puti($1.bhs, $f(($1.barlen * 5) + 6), $1.heights); //#10412
        $k[$j++] = Infinity; //#10423
        var _2G = $1.bhs; //#10417
        $k[$j++] = Infinity; //#10418
        for (var _2I = 0, _2J = $1.bhs.length; _2I < _2J; _2I++) { //#10418
            $k[$j++] = 0; //#10418
        } //#10418
        var _2K = $a(); //#10418
        $k[$j++] = Infinity; //#10419
        for (var _2M = 0, _2N = $1.bhs.length - 1; _2M < _2N; _2M++) { //#10419
            $k[$j++] = 1.44; //#10419
            $k[$j++] = 1.872; //#10419
        } //#10419
        $k[$j++] = 1.44; //#10419
        var _2O = $a(); //#10419
        $k[$j++] = 'ren'; //#10422
        $k[$j++] = $0.renlinear; //#10422
        $k[$j++] = 'bhs'; //#10422
        $k[$j++] = _2G; //#10422
        $k[$j++] = 'bbs'; //#10422
        $k[$j++] = _2K; //#10422
        $k[$j++] = 'sbs'; //#10422
        $k[$j++] = _2O; //#10422
        if ($1.includetext) { //#10422
            $k[$j++] = 'txt'; //#10421
            $k[$j++] = $1.txt; //#10421
        } //#10421
        $k[$j++] = 'opt'; //#10423
        $k[$j++] = $1.options; //#10423
        var _2S = $d(); //#10423
        $k[$j++] = _2S; //#10426
        if (!$1.dontdraw) { //#10426
            $0.renlinear(); //#10426
        } //#10426
    };
    $0.planet = function() {
        var $1 = {}; //#10455
        $1.options = $k[--$j]; //#10457
        $1.barcode = $k[--$j]; //#10458
        $1.dontdraw = false; //#10460
        $1.includetext = false; //#10461
        $1.validatecheck = false; //#10462
        $1.includecheckintext = false; //#10463
        $1.textfont = "OCR-B"; //#10464
        $1.textsize = 10; //#10465
        $1.textyoffset = -8; //#10466
        $1.height = 0.125; //#10467
        $forall($1.options, function() { //#10478
            var _3 = $k[--$j]; //#10478
            $1[$k[--$j]] = _3; //#10478
        }); //#10478
        $1.textfont = "" + $1.textfont; //#10480
        $1.textsize = +$1.textsize; //#10481
        $1.textyoffset = +$1.textyoffset; //#10482
        $1.height = +$1.height; //#10483
        $k[$j++] = 'barlen'; //#10485
        $k[$j++] = $1.barcode.length; //#10485
        if ($1.validatecheck) { //#10485
            var _B = $k[--$j]; //#10485
            $k[$j++] = $f(_B - 1); //#10485
        } //#10485
        var _C = $k[--$j]; //#10485
        $1[$k[--$j]] = _C; //#10485
        if (($1.barlen != 11) && ($1.barlen != 13)) { //#10490
            $k[$j++] = 'bwipp.planetBadLength'; //#10489
            $k[$j++] = "USPS PLANET must be 11 or 13 digits excluding check digit"; //#10489
            $0.raiseerror(); //#10489
        } //#10489
        $forall($1.barcode, function() { //#10495
            var _H = $k[--$j]; //#10492
            if ((_H < 48) || (_H > 57)) { //#10494
                $k[$j++] = 'bwipp.planetBadCharacter'; //#10493
                $k[$j++] = "USPS PLANET must contain only digits"; //#10493
                $0.raiseerror(); //#10493
            } //#10493
        }); //#10493
        $1.barchars = "0123456789"; //#10498
        $1.checksum = 0; //#10501
        for (var _K = 0, _J = $f($1.barlen - 1); _K <= _J; _K += 1) { //#10505
            $1.i = _K; //#10503
            $1.checksum = $f($1.checksum + $f($get($1.barcode, $1.i) - 48)); //#10504
        } //#10504
        $1.checksum = ($f(10 - ($1.checksum % 10))) % 10; //#10506
        if ($1.validatecheck) { //#10512
            if ($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum)) { //#10510
                $k[$j++] = 'bwipp.planetBadCheckDigit'; //#10509
                $k[$j++] = "Incorrect USPS PLANET check digit provided"; //#10509
                $0.raiseerror(); //#10509
            } //#10509
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#10511
        } //#10511
        $1.encs = $a(["22555", "55522", "55252", "55225", "52552", "52525", "52255", "25552", "25525", "25255", "5", "5"]); //#10519
        $1.bhs = $a($f(($1.barlen * 5) + 7)); //#10521
        $1.txt = $a($f($1.barlen + 1)); //#10522
        $1.enc = $get($1.encs, 10); //#10525
        $1.heights = $a($1.enc.length); //#10526
        for (var _l = 0, _k = $1.enc.length - 1; _l <= _k; _l += 1) { //#10530
            $1.j = _l; //#10528
            $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10529
        } //#10529
        $puti($1.bhs, 0, $1.heights); //#10531
        for (var _w = 0, _v = $f($1.barlen - 1); _w <= _v; _w += 1) { //#10543
            $1.i = _w; //#10534
            $1.enc = $get($1.encs, $f($get($1.barcode, $1.i) - 48)); //#10535
            $1.heights = $a($1.enc.length); //#10536
            for (var _16 = 0, _15 = $1.enc.length - 1; _16 <= _15; _16 += 1) { //#10540
                $1.j = _16; //#10538
                $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10539
            } //#10539
            $puti($1.bhs, ($1.i * 5) + 1, $1.heights); //#10541
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), (($1.i * 5) + 1) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10542
        } //#10542
        $1.enc = $get($1.encs, $1.checksum); //#10546
        $1.heights = $a($1.enc.length); //#10547
        for (var _1X = 0, _1W = $1.enc.length - 1; _1X <= _1W; _1X += 1) { //#10551
            $1.j = _1X; //#10549
            $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10550
        } //#10550
        $puti($1.bhs, $f(($1.barlen * 5) + 1), $1.heights); //#10552
        if ($1.includecheckintext) { //#10557
            $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum, 1), ($f(($1.barlen * 5) + 1)) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10555
        } else { //#10557
            $put($1.txt, $1.barlen, $a([" ", (($f(($1.barlen * 5) + 1)) * 72) / 25, $1.textyoffset, $1.textfont, $1.textsize])); //#10557
        } //#10557
        $1.enc = $get($1.encs, 11); //#10561
        $1.heights = $a($1.enc.length); //#10562
        for (var _25 = 0, _24 = $1.enc.length - 1; _25 <= _24; _25 += 1) { //#10566
            $1.j = _25; //#10564
            $put($1.heights, $1.j, ((~~$z($geti($1.enc, $1.j, 1))) * $1.height) / 5); //#10565
        } //#10565
        $puti($1.bhs, $f(($1.barlen * 5) + 6), $1.heights); //#10567
        $k[$j++] = Infinity; //#10578
        var _2F = $1.bhs; //#10572
        $k[$j++] = Infinity; //#10573
        for (var _2H = 0, _2I = $1.bhs.length; _2H < _2I; _2H++) { //#10573
            $k[$j++] = 0; //#10573
        } //#10573
        var _2J = $a(); //#10573
        $k[$j++] = Infinity; //#10574
        for (var _2L = 0, _2M = $1.bhs.length - 1; _2L < _2M; _2L++) { //#10574
            $k[$j++] = 1.44; //#10574
            $k[$j++] = 1.872; //#10574
        } //#10574
        $k[$j++] = 1.44; //#10574
        var _2N = $a(); //#10574
        $k[$j++] = 'ren'; //#10577
        $k[$j++] = $0.renlinear; //#10577
        $k[$j++] = 'bhs'; //#10577
        $k[$j++] = _2F; //#10577
        $k[$j++] = 'bbs'; //#10577
        $k[$j++] = _2J; //#10577
        $k[$j++] = 'sbs'; //#10577
        $k[$j++] = _2N; //#10577
        if ($1.includetext) { //#10577
            $k[$j++] = 'txt'; //#10576
            $k[$j++] = $1.txt; //#10576
        } //#10576
        $k[$j++] = 'opt'; //#10578
        $k[$j++] = $1.options; //#10578
        var _2R = $d(); //#10578
        $k[$j++] = _2R; //#10581
        if (!$1.dontdraw) { //#10581
            $0.renlinear(); //#10581
        } //#10581
    };
    $0.royalmail = function() {
        var $1 = {}; //#10610
        $1.options = $k[--$j]; //#10612
        $1.barcode = $k[--$j]; //#10613
        $1.dontdraw = false; //#10615
        $1.includetext = false; //#10616
        $1.validatecheck = false; //#10617
        $1.includecheckintext = false; //#10618
        $1.textfont = "OCR-B"; //#10619
        $1.textsize = 10; //#10620
        $1.textyoffset = -8; //#10621
        $1.height = 0.175; //#10622
        $forall($1.options, function() { //#10633
            var _3 = $k[--$j]; //#10633
            $1[$k[--$j]] = _3; //#10633
        }); //#10633
        $1.textfont = "" + $1.textfont; //#10635
        $1.textsize = +$1.textsize; //#10636
        $1.textyoffset = +$1.textyoffset; //#10637
        $1.height = +$1.height; //#10638
        $1.barchars = "ZUVWXY501234B6789AHCDEFGNIJKLMTOPQRS"; //#10641
        $1.charvals = new Map; //#10642
        for (var _9 = 0; _9 <= 35; _9 += 1) { //#10643
            $put($1.charvals, $geti($1.barchars, _9, 1), _9); //#10643
        } //#10643
        for (var _F = 0, _E = $1.barcode.length - 1; _F <= _E; _F += 1) { //#10650
            var _J = $get($1.charvals, $geti($1.barcode, _F, 1)) !== undefined; //#10647
            if (!_J) { //#10649
                $k[$j++] = 'bwipp.royalmailBadCharacter'; //#10648
                $k[$j++] = "RM4SCC must contain only capital letters and digits"; //#10648
                $0.raiseerror(); //#10648
            } //#10648
        } //#10648
        $k[$j++] = 'barlen'; //#10652
        $k[$j++] = $1.barcode.length; //#10652
        if ($1.validatecheck) { //#10652
            var _M = $k[--$j]; //#10652
            $k[$j++] = $f(_M - 1); //#10652
        } //#10652
        var _N = $k[--$j]; //#10652
        $1[$k[--$j]] = _N; //#10652
        $1.checksumrow = 0; //#10654
        $1.checksumcol = 0; //#10655
        for (var _R = 0, _Q = $f($1.barlen - 1); _R <= _Q; _R += 1) { //#10661
            $1.i = _R; //#10657
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#10658
            $1.checksumrow = $1.checksumrow + (~~($1.indx / 6)); //#10659
            $1.checksumcol = $f($1.checksumcol + ($1.indx % 6)); //#10660
        } //#10660
        $1.checksum = $f((($1.checksumrow % 6) * 6) + ($1.checksumcol % 6)); //#10662
        if ($1.validatecheck) { //#10668
            if ($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum)) { //#10666
                $k[$j++] = 'bwipp.royalmailBadCheckDigit'; //#10665
                $k[$j++] = "Incorrect RM4SCC check digit provided"; //#10665
                $0.raiseerror(); //#10665
            } //#10665
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#10667
        } //#10667
        $1.encs = $a(["3300", "2211", "2301", "2310", "3201", "3210", "1122", "0033", "0123", "0132", "1023", "1032", "1302", "0213", "0303", "0312", "1203", "1212", "1320", "0231", "0321", "0330", "1221", "1230", "3102", "2013", "2103", "2112", "3003", "3012", "3120", "2031", "2121", "2130", "3021", "3030", "2", "3"]); //#10679
        $1.encstr = $s($f(($1.barlen * 4) + 6)); //#10681
        $1.txt = $a($f($1.barlen + 1)); //#10682
        $puti($1.encstr, 0, $get($1.encs, 36)); //#10685
        for (var _x = 0, _w = $f($1.barlen - 1); _x <= _w; _x += 1) { //#10692
            $1.i = _x; //#10688
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#10689
            $puti($1.encstr, ($1.i * 4) + 1, $get($1.encs, $1.indx)); //#10690
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), (($1.i * 4) + 1) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10691
        } //#10691
        $puti($1.encstr, $f(($1.barlen * 4) + 1), $get($1.encs, $1.checksum)); //#10695
        if ($1.includecheckintext) { //#10699
            $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum, 1), ($f(($1.barlen * 4) + 1)) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10697
        } else { //#10699
            $put($1.txt, $1.barlen, $a([" ", ($f(($1.barlen * 4) + 1)) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10699
        } //#10699
        $puti($1.encstr, $f(($1.barlen * 4) + 5), $get($1.encs, 37)); //#10703
        $1.bbs = $a($1.encstr.length); //#10705
        $1.bhs = $a($1.encstr.length); //#10706
        for (var _1p = 0, _1o = $1.encstr.length - 1; _1p <= _1o; _1p += 1) { //#10726
            $1.i = _1p; //#10708
            $1.enc = $geti($1.encstr, $1.i, 1); //#10709
            if ($eq($1.enc, "0")) { //#10713
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#10711
                $put($1.bhs, $1.i, (2 * $1.height) / 8); //#10712
            } //#10712
            if ($eq($1.enc, "1")) { //#10717
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#10715
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#10716
            } //#10716
            if ($eq($1.enc, "2")) { //#10721
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#10719
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#10720
            } //#10720
            if ($eq($1.enc, "3")) { //#10725
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#10723
                $put($1.bhs, $1.i, (8 * $1.height) / 8); //#10724
            } //#10724
        } //#10724
        $k[$j++] = Infinity; //#10737
        var _2L = $1.bbs; //#10731
        var _2M = $1.bhs; //#10732
        $k[$j++] = Infinity; //#10733
        for (var _2O = 0, _2P = $1.bhs.length - 1; _2O < _2P; _2O++) { //#10733
            $k[$j++] = 1.44; //#10733
            $k[$j++] = 1.872; //#10733
        } //#10733
        $k[$j++] = 1.44; //#10733
        var _2Q = $a(); //#10733
        $k[$j++] = 'ren'; //#10736
        $k[$j++] = $0.renlinear; //#10736
        $k[$j++] = 'bbs'; //#10736
        $k[$j++] = _2L; //#10736
        $k[$j++] = 'bhs'; //#10736
        $k[$j++] = _2M; //#10736
        $k[$j++] = 'sbs'; //#10736
        $k[$j++] = _2Q; //#10736
        if ($1.includetext) { //#10736
            $k[$j++] = 'txt'; //#10735
            $k[$j++] = $1.txt; //#10735
        } //#10735
        $k[$j++] = 'opt'; //#10737
        $k[$j++] = $1.options; //#10737
        var _2U = $d(); //#10737
        $k[$j++] = _2U; //#10740
        if (!$1.dontdraw) { //#10740
            $0.renlinear(); //#10740
        } //#10740
    };
    $0.auspost = function() {
        var $1 = {}; //#10769
        $1.options = $k[--$j]; //#10771
        $1.barcode = $k[--$j]; //#10772
        $1.dontdraw = false; //#10774
        $1.includetext = false; //#10775
        $1.textfont = "OCR-B"; //#10776
        $1.textsize = 10; //#10777
        $1.textyoffset = -8; //#10778
        $1.height = 0.175; //#10779
        $1.custinfoenc = "character"; //#10780
        $forall($1.options, function() { //#10791
            var _3 = $k[--$j]; //#10791
            $1[$k[--$j]] = _3; //#10791
        }); //#10791
        $1.textfont = "" + $1.textfont; //#10793
        $1.textsize = +$1.textsize; //#10794
        $1.textyoffset = +$1.textyoffset; //#10795
        $1.height = +$1.height; //#10796
        $1.encs = $a(["000", "001", "002", "010", "011", "012", "020", "021", "022", "100", "101", "102", "110", "111", "112", "120", "121", "122", "200", "201", "202", "210", "211", "212", "220", "221", "222", "300", "301", "302", "310", "311", "312", "320", "321", "322", "023", "030", "031", "032", "033", "103", "113", "123", "130", "131", "132", "133", "203", "213", "223", "230", "231", "232", "233", "303", "313", "323", "330", "331", "332", "333", "003", "013", "00", "01", "02", "10", "11", "12", "20", "21", "22", "30", "13", "3"]); //#10810
        $1.barchars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789abcdefghijklmnopqrstuvwxyz #"; //#10813
        $1.barlen = $1.barcode.length; //#10815
        if ($eq($geti($1.barcode, 0, 2), "11")) { //#10816
            $k[$j++] = 37; //#10816
        } //#10816
        if ($eq($geti($1.barcode, 0, 2), "45")) { //#10817
            $k[$j++] = 37; //#10817
        } //#10817
        if ($eq($geti($1.barcode, 0, 2), "59")) { //#10818
            $k[$j++] = 52; //#10818
        } //#10818
        if ($eq($geti($1.barcode, 0, 2), "62")) { //#10819
            $k[$j++] = 67; //#10819
        } //#10819
        $1.encstr = $s($k[--$j]); //#10820
        $1.txt = $a($1.barlen - 2); //#10821
        $puti($1.encstr, 0, $get($1.encs, 74)); //#10824
        for (var _Q = 0; _Q <= 1; _Q += 1) { //#10831
            $1.i = _Q; //#10828
            $puti($1.encstr, ($1.i * 2) + 2, $get($1.encs, (~~$z($geti($1.barcode, $1.i, 1))) + 64)); //#10830
        } //#10830
        for (var _Y = 2; _Y <= 9; _Y += 1) { //#10839
            $1.i = _Y; //#10835
            $puti($1.encstr, ($1.i * 2) + 2, $get($1.encs, (~~$z($geti($1.barcode, $1.i, 1))) + 64)); //#10837
            $put($1.txt, $1.i - 2, $a([$geti($1.barcode, $1.i, 1), ((($1.i - 2) * 2) + 6) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10838
        } //#10838
        if ($eq($1.custinfoenc, "numeric")) { //#10861
            for (var _t = 0, _s = $1.barlen - 11; _t <= _s; _t += 1) { //#10848
                $1.i = _t; //#10844
                $puti($1.encstr, ($1.i * 2) + 22, $get($1.encs, (~~$z($geti($1.barcode, $1.i + 10, 1))) + 64)); //#10846
                $put($1.txt, $1.i + 8, $a([$geti($1.barcode, $1.i + 10, 1), (($1.i * 2) + 22) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10847
            } //#10847
            $1.ciflen = ($1.barlen - 10) * 2; //#10849
        } else { //#10861
            for (var _1E = 0, _1D = $1.barlen - 11; _1E <= _1D; _1E += 1) { //#10860
                $1.i = _1E; //#10852
                $search($1.barchars, $geti($1.barcode, $1.i + 10, 1)); //#10853
                $j--; //#10854
                $1.indx = $k[--$j].length; //#10855
                $j -= 2; //#10856
                $1.enc = $get($1.encs, $1.indx); //#10857
                $puti($1.encstr, ($1.i * 3) + 22, $1.enc); //#10858
                $put($1.txt, $1.i + 8, $a([$geti($1.barcode, $1.i + 10, 1), (($1.i * 3) + 22) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#10859
            } //#10859
            $1.ciflen = ($1.barlen - 10) * 3; //#10861
        } //#10861
        for (var _1e = 22 + $1.ciflen, _1d = $1.encstr.length - 14; _1e <= _1d; _1e += 1) { //#10867
            $puti($1.encstr, _1e, $get($1.encs, 75)); //#10866
        } //#10866
        $1.rstable = $a(64 * 64); //#10870
        $k[$j++] = $1.rstable; //#10871
        $k[$j++] = 0; //#10871
        $k[$j++] = Infinity; //#10871
        for (var _1k = 0, _1l = 64; _1k < _1l; _1k++) { //#10871
            $k[$j++] = 0; //#10871
        } //#10871
        var _1m = $a(); //#10871
        var _1n = $k[--$j]; //#10871
        $puti($k[--$j], _1n, _1m); //#10871
        $k[$j++] = $1.rstable; //#10872
        $k[$j++] = 64; //#10872
        $k[$j++] = Infinity; //#10872
        for (var _1q = 0; _1q <= 63; _1q += 1) { //#10872
            $k[$j++] = _1q; //#10872
        } //#10872
        var _1r = $a(); //#10872
        var _1s = $k[--$j]; //#10872
        $puti($k[--$j], _1s, _1r); //#10872
        $1.prev = 1; //#10873
        for (var _1u = 0, _1v = 64; _1u < _1v; _1u++) { //#10888
            $1.next = $1.prev << 1; //#10875
            if (($1.next & 64) != 0) { //#10878
                $1.next = $1.next ^ 67; //#10877
            } //#10877
            for (var _1z = 0; _1z <= 63; _1z += 1) { //#10886
                $1.j = _1z; //#10880
                $1.nextcell = function() {
                    $k[$j++] = $1.rstable; //#10881
                    $k[$j++] = (64 * $1.next) + $1.j; //#10881
                }; //#10881
                $1.nextcell(); //#10882
                var _27 = $k[--$j]; //#10882
                $put($k[--$j], _27, $get($1.rstable, (64 * $1.prev) + $1.j) << 1); //#10882
                $1.nextcell(); //#10883
                var _29 = $k[--$j]; //#10883
                if (($get($k[--$j], _29) & 64) != 0) { //#10885
                    $1.nextcell(); //#10884
                    $1.nextcell(); //#10884
                    var _2C = $k[--$j]; //#10884
                    var _2E = $get($k[--$j], _2C); //#10884
                    var _2F = $k[--$j]; //#10884
                    $put($k[--$j], _2F, _2E ^ 67); //#10884
                } //#10884
            } //#10884
            $1.prev = $1.next; //#10887
        } //#10887
        $1.rscodes = $a((~~(($1.encstr.length - 16) / 3)) + 4); //#10891
        $k[$j++] = $1.rscodes; //#10892
        $k[$j++] = 0; //#10892
        $k[$j++] = Infinity; //#10892
        for (var _2L = 0, _2M = 4; _2L < _2M; _2L++) { //#10892
            $k[$j++] = 0; //#10892
        } //#10892
        var _2N = $a(); //#10892
        var _2O = $k[--$j]; //#10892
        $puti($k[--$j], _2O, _2N); //#10892
        for (var _2S = 2, _2R = $1.encstr.length - 16; _2S <= _2R; _2S += 3) { //#10900
            $1.i = _2S; //#10894
            $put($1.rscodes, ($1.rscodes.length - (~~(($1.i - 2) / 3))) - 1, (((~~$z($geti($1.encstr, $1.i, 1))) * 16) + ((~~$z($geti($1.encstr, $1.i + 1, 1))) * 4)) + (~~$z($geti($1.encstr, $1.i + 2, 1)))); //#10899
        } //#10899
        for (var _2g = $1.rscodes.length - 5; _2g >= 0; _2g -= 1) { //#10909
            $1.i = _2g; //#10902
            for (var _2h = 0; _2h <= 4; _2h += 1) { //#10908
                $1.j = _2h; //#10904
                $put($1.rscodes, $1.i + $1.j, $xo($get($1.rscodes, $1.i + $1.j), $get($1.rstable, $f((64 * $get($a([48, 17, 29, 30, 1]), $1.j)) + $get($1.rscodes, $1.i + 4))))); //#10907
            } //#10907
        } //#10907
        $1.checkcode = $strcpy($s(12), "000000000000"); //#10910
        for (var _2z = 0; _2z <= 3; _2z += 1) { //#10915
            $1.i = _2z; //#10912
            $1.enc = $cvrs($s(3), $get($1.rscodes, 3 - $1.i), 4); //#10913
            $puti($1.checkcode, ($1.i * 3) + (3 - $1.enc.length), $1.enc); //#10914
        } //#10914
        $puti($1.encstr, $1.encstr.length - 14, $1.checkcode); //#10918
        $puti($1.encstr, $1.encstr.length - 2, $get($1.encs, 74)); //#10919
        $1.bbs = $a($1.encstr.length); //#10921
        $1.bhs = $a($1.encstr.length); //#10922
        for (var _3M = 0, _3L = $1.encstr.length - 1; _3M <= _3L; _3M += 1) { //#10942
            $1.i = _3M; //#10924
            $1.enc = $geti($1.encstr, $1.i, 1); //#10925
            if ($eq($1.enc, "0")) { //#10929
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#10927
                $put($1.bhs, $1.i, (8 * $1.height) / 8); //#10928
            } //#10928
            if ($eq($1.enc, "1")) { //#10933
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#10931
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#10932
            } //#10932
            if ($eq($1.enc, "2")) { //#10937
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#10935
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#10936
            } //#10936
            if ($eq($1.enc, "3")) { //#10941
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#10939
                $put($1.bhs, $1.i, (2 * $1.height) / 8); //#10940
            } //#10940
        } //#10940
        $k[$j++] = Infinity; //#10953
        var _3s = $1.bbs; //#10947
        var _3t = $1.bhs; //#10948
        $k[$j++] = Infinity; //#10949
        for (var _3v = 0, _3w = $1.bhs.length - 1; _3v < _3w; _3v++) { //#10949
            $k[$j++] = 1.44; //#10949
            $k[$j++] = 1.872; //#10949
        } //#10949
        $k[$j++] = 1.44; //#10949
        var _3x = $a(); //#10949
        $k[$j++] = 'ren'; //#10952
        $k[$j++] = $0.renlinear; //#10952
        $k[$j++] = 'bbs'; //#10952
        $k[$j++] = _3s; //#10952
        $k[$j++] = 'bhs'; //#10952
        $k[$j++] = _3t; //#10952
        $k[$j++] = 'sbs'; //#10952
        $k[$j++] = _3x; //#10952
        if ($1.includetext) { //#10952
            $k[$j++] = 'txt'; //#10951
            $k[$j++] = $1.txt; //#10951
        } //#10951
        $k[$j++] = 'opt'; //#10953
        $k[$j++] = $1.options; //#10953
        var _41 = $d(); //#10953
        $k[$j++] = _41; //#10956
        if (!$1.dontdraw) { //#10956
            $0.renlinear(); //#10956
        } //#10956
    };
    $0.kix = function() {
        var $1 = {}; //#10985
        $1.options = $k[--$j]; //#10987
        $1.barcode = $k[--$j]; //#10988
        $1.dontdraw = false; //#10990
        $1.includetext = false; //#10991
        $1.textfont = "OCR-B"; //#10992
        $1.textsize = 10; //#10993
        $1.textyoffset = -8; //#10994
        $1.height = 0.175; //#10995
        $forall($1.options, function() { //#11006
            var _3 = $k[--$j]; //#11006
            $1[$k[--$j]] = _3; //#11006
        }); //#11006
        $1.textfont = "" + $1.textfont; //#11008
        $1.textsize = +$1.textsize; //#11009
        $1.textyoffset = +$1.textyoffset; //#11010
        $1.height = +$1.height; //#11011
        $1.encs = $a(["0033", "0123", "0132", "1023", "1032", "1122", "0213", "0303", "0312", "1203", "1212", "1302", "0231", "0321", "0330", "1221", "1230", "1320", "2013", "2103", "2112", "3003", "3012", "3102", "2031", "2121", "2130", "3021", "3030", "3120", "2211", "2301", "2310", "3201", "3210", "3300"]); //#11021
        $1.barchars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"; //#11024
        $1.charvals = new Map; //#11025
        for (var _A = 0; _A <= 35; _A += 1) { //#11026
            $put($1.charvals, $geti($1.barchars, _A, 1), _A); //#11026
        } //#11026
        for (var _G = 0, _F = $1.barcode.length - 1; _G <= _F; _G += 1) { //#11033
            var _K = $get($1.charvals, $geti($1.barcode, _G, 1)) !== undefined; //#11030
            if (!_K) { //#11032
                $k[$j++] = 'bwipp.kixBadCharacter'; //#11031
                $k[$j++] = "KIX must contain only capital letters and digits"; //#11031
                $0.raiseerror(); //#11031
            } //#11031
        } //#11031
        $1.barlen = $1.barcode.length; //#11035
        $1.encstr = $s($1.barlen * 4); //#11036
        $1.txt = $a($1.barlen); //#11037
        for (var _S = 0, _R = $1.barlen - 1; _S <= _R; _S += 1) { //#11044
            $1.i = _S; //#11040
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#11041
            $puti($1.encstr, $1.i * 4, $get($1.encs, $1.indx)); //#11042
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), ($1.i * 4) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#11043
        } //#11043
        $1.bbs = $a($1.encstr.length); //#11046
        $1.bhs = $a($1.encstr.length); //#11047
        for (var _t = 0, _s = $1.encstr.length - 1; _t <= _s; _t += 1) { //#11067
            $1.i = _t; //#11049
            $1.enc = $geti($1.encstr, $1.i, 1); //#11050
            if ($eq($1.enc, "0")) { //#11054
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#11052
                $put($1.bhs, $1.i, (2 * $1.height) / 8); //#11053
            } //#11053
            if ($eq($1.enc, "1")) { //#11058
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#11056
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#11057
            } //#11057
            if ($eq($1.enc, "2")) { //#11062
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#11060
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#11061
            } //#11061
            if ($eq($1.enc, "3")) { //#11066
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#11064
                $put($1.bhs, $1.i, (8 * $1.height) / 8); //#11065
            } //#11065
        } //#11065
        $k[$j++] = Infinity; //#11078
        var _1P = $1.bbs; //#11072
        var _1Q = $1.bhs; //#11073
        $k[$j++] = Infinity; //#11074
        for (var _1S = 0, _1T = $1.bhs.length - 1; _1S < _1T; _1S++) { //#11074
            $k[$j++] = 1.44; //#11074
            $k[$j++] = 1.872; //#11074
        } //#11074
        $k[$j++] = 1.44; //#11074
        var _1U = $a(); //#11074
        $k[$j++] = 'ren'; //#11077
        $k[$j++] = $0.renlinear; //#11077
        $k[$j++] = 'bbs'; //#11077
        $k[$j++] = _1P; //#11077
        $k[$j++] = 'bhs'; //#11077
        $k[$j++] = _1Q; //#11077
        $k[$j++] = 'sbs'; //#11077
        $k[$j++] = _1U; //#11077
        if ($1.includetext) { //#11077
            $k[$j++] = 'txt'; //#11076
            $k[$j++] = $1.txt; //#11076
        } //#11076
        $k[$j++] = 'opt'; //#11078
        $k[$j++] = $1.options; //#11078
        var _1Y = $d(); //#11078
        $k[$j++] = _1Y; //#11081
        if (!$1.dontdraw) { //#11081
            $0.renlinear(); //#11081
        } //#11081
    };
    $0.japanpost = function() {
        var $1 = {}; //#11110
        $1.options = $k[--$j]; //#11112
        $1.barcode = $k[--$j]; //#11113
        $1.dontdraw = false; //#11115
        $1.includetext = false; //#11116
        $1.includecheckintext = false; //#11117
        $1.textfont = "OCR-B"; //#11118
        $1.textsize = 10; //#11119
        $1.textyoffset = -8; //#11120
        $1.height = 0.175; //#11121
        $forall($1.options, function() { //#11132
            var _3 = $k[--$j]; //#11132
            $1[$k[--$j]] = _3; //#11132
        }); //#11132
        $1.textfont = "" + $1.textfont; //#11134
        $1.textsize = +$1.textsize; //#11135
        $1.textyoffset = +$1.textyoffset; //#11136
        $1.height = +$1.height; //#11137
        $1.encs = $a(["300", "330", "312", "132", "321", "303", "123", "231", "213", "033", "030", "120", "102", "210", "012", "201", "021", "003", "333", "31", "13"]); //#11145
        $1.barchars = "0123456789-ABCDEFGHIJKLMNOPQRSTUVWXYZ"; //#11148
        $1.barlen = $1.barcode.length; //#11149
        $1.encstr = $s(((20 + 1) * 3) + 4); //#11151
        $1.digits = $a(20); //#11152
        $1.txt = $a($1.barlen + 1); //#11153
        $puti($1.encstr, 0, $get($1.encs, 19)); //#11156
        $1.checksum = 0; //#11158
        $1.j = 0; //#11159
        $1.i = 0; //#11159
        for (var _K = 0, _J = $1.barlen - 1; _K <= _J; _K += 1) { //#11193
            $1.i = _K; //#11161
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#11162
            $j--; //#11163
            $1.indx = $k[--$j].length; //#11164
            $j -= 2; //#11165
            if (($1.indx >= 11) && ($1.indx < 37)) { //#11176
                if ($1.j > 18) { //#11167
                    break; //#11167
                } //#11167
                $1.digit = (~~(($1.indx - 1) / 10)) + 10; //#11171
                $puti($1.encstr, ($1.j * 3) + 2, $get($1.encs, $1.digit)); //#11172
                $1.checksum = $1.checksum + $1.digit; //#11173
                $put($1.digits, $1.j, $1.digit); //#11174
                $1.j = $1.j + 1; //#11175
            } //#11175
            if ($1.j > 19) { //#11177
                break; //#11177
            } //#11177
            $1.digit = 0; //#11178
            if ($1.indx == 37) { //#11181
                $1.digit = 14; //#11180
            } //#11180
            if (($1.indx >= 11) && ($1.indx < 37)) { //#11184
                $1.digit = ($1.indx - 1) % 10; //#11183
            } //#11183
            if ($1.indx < 11) { //#11187
                $1.digit = $1.indx; //#11186
            } //#11186
            $puti($1.encstr, ($1.j * 3) + 2, $get($1.encs, $1.digit)); //#11188
            $1.checksum = $1.checksum + $1.digit; //#11189
            $put($1.digits, $1.j, $1.digit); //#11190
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), (($1.j * 3) + 2) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#11191
            $1.j = $1.j + 1; //#11192
        } //#11192
        for (var _19 = $1.j, _18 = 20 - 1; _19 <= _18; _19 += 1) { //#11201
            $1.k = _19; //#11197
            $puti($1.encstr, ($1.k * 3) + 2, $get($1.encs, 14)); //#11198
            $1.checksum = $1.checksum + 14; //#11199
            $put($1.digits, $1.k, 14); //#11200
        } //#11200
        $1.checksum = 19 - ($1.checksum % 19); //#11204
        $puti($1.encstr, 2 + (20 * 3), $get($1.encs, $1.checksum)); //#11205
        $puti($1.encstr, 2 + (21 * 3), $get($1.encs, 20)); //#11208
        $1.checkdigit = " "; //#11209
        if ($1.includecheckintext) { //#11212
            $1.checkdigit = $geti($1.barchars, $1.checksum, 1); //#11211
        } //#11211
        $put($1.txt, $1.i + 1, $a([$1.checkdigit, ((20 * 3) + 2) * 3.312, $1.textyoffset, $1.textfont, $1.textsize])); //#11213
        $puti($1.encstr, ((20 + 1) * 3) + 2, $get($1.encs, 20)); //#11216
        $1.bbs = $a((21 * 3) + 4); //#11218
        $1.bhs = $a($1.bbs.length); //#11219
        for (var _1i = 0, _1h = $1.bbs.length - 1; _1i <= _1h; _1i += 1) { //#11243
            $1.i = _1i; //#11221
            $1.enc = $geti($1.encstr, $1.i, 1); //#11222
            $1.bunit = 0; //#11223
            $1.hunit = 0; //#11224
            if ($eq($1.enc, "0")) { //#11228
                $1.bunit = 3; //#11226
                $1.hunit = 2; //#11227
            } //#11227
            if ($eq($1.enc, "1")) { //#11232
                $1.bunit = 0; //#11230
                $1.hunit = 5; //#11231
            } //#11231
            if ($eq($1.enc, "2")) { //#11236
                $1.bunit = 3; //#11234
                $1.hunit = 5; //#11235
            } //#11235
            if ($eq($1.enc, "3")) { //#11240
                $1.bunit = 0; //#11238
                $1.hunit = 8; //#11239
            } //#11239
            $put($1.bbs, $1.i, ($1.bunit * $1.height) / 8); //#11241
            $put($1.bhs, $1.i, ($1.hunit * $1.height) / 8); //#11242
        } //#11242
        $k[$j++] = Infinity; //#11254
        var _1y = $1.bbs; //#11248
        var _1z = $1.bhs; //#11249
        $k[$j++] = Infinity; //#11250
        for (var _21 = 0, _22 = $1.bhs.length - 1; _21 < _22; _21++) { //#11250
            $k[$j++] = 1.44; //#11250
            $k[$j++] = 1.872; //#11250
        } //#11250
        $k[$j++] = 1.44; //#11250
        var _23 = $a(); //#11250
        $k[$j++] = 'ren'; //#11253
        $k[$j++] = $0.renlinear; //#11253
        $k[$j++] = 'bbs'; //#11253
        $k[$j++] = _1y; //#11253
        $k[$j++] = 'bhs'; //#11253
        $k[$j++] = _1z; //#11253
        $k[$j++] = 'sbs'; //#11253
        $k[$j++] = _23; //#11253
        if ($1.includetext) { //#11253
            $k[$j++] = 'txt'; //#11252
            $k[$j++] = $1.txt; //#11252
        } //#11252
        $k[$j++] = 'opt'; //#11254
        $k[$j++] = $1.options; //#11254
        var _27 = $d(); //#11254
        $k[$j++] = _27; //#11257
        if (!$1.dontdraw) { //#11257
            $0.renlinear(); //#11257
        } //#11257
    };
    $0.msi = function() {
        var $1 = {}; //#11286
        $1.options = $k[--$j]; //#11288
        $1.barcode = $k[--$j]; //#11289
        $1.dontdraw = false; //#11291
        $1.includecheck = false; //#11292
        $1.includetext = false; //#11293
        $1.includecheckintext = false; //#11294
        $1.checktype = "mod10"; //#11295
        $1.badmod11 = false; //#11296
        $1.textfont = "OCR-B"; //#11297
        $1.textsize = 10; //#11298
        $1.textyoffset = -8; //#11299
        $1.height = 1; //#11300
        $forall($1.options, function() { //#11311
            var _3 = $k[--$j]; //#11311
            $1[$k[--$j]] = _3; //#11311
        }); //#11311
        $1.textfont = "" + $1.textfont; //#11313
        $1.textsize = +$1.textsize; //#11314
        $1.textyoffset = +$1.textyoffset; //#11315
        $1.height = +$1.height; //#11316
        $1.encs = $a(["12121212", "12121221", "12122112", "12122121", "12211212", "12211221", "12212112", "12212121", "21121212", "21121221", "21", "121"]); //#11323
        $1.barchars = "0123456789"; //#11326
        $1.barlen = $1.barcode.length; //#11328
        $1.txtlen = $1.barlen; //#11329
        $1.mod10 = function() {
            $1.code = $k[--$j]; //#11332
            $k[$j++] = 0; //#11333
            $k[$j++] = 0; //#11333
            $forall($1.code, function() { //#11333
                var _E = $k[--$j]; //#11333
                var _F = $k[--$j]; //#11333
                var _G = $k[--$j]; //#11333
                $k[$j++] = $f($f(_E - 48) + (_F * 10)); //#11333
                $k[$j++] = _G; //#11333
            }); //#11333
            $j--; //#11333
            var _H = $k[--$j]; //#11333
            $k[$j++] = _H * 2; //#11334
            $k[$j++] = 0; //#11334
            for (;;) { //#11334
                var _I = $k[--$j]; //#11334
                var _J = $k[--$j]; //#11334
                var _K = ~~(_J / 10); //#11334
                $k[$j++] = $f(_I + (_J % 10)); //#11334
                $k[$j++] = _K; //#11334
                if (_K == 0) { //#11334
                    $j--; //#11334
                    break; //#11334
                } //#11334
                var _L = $k[--$j]; //#11334
                var _M = $k[--$j]; //#11334
                $k[$j++] = _L; //#11334
                $k[$j++] = _M; //#11334
            } //#11334
            $k[$j++] = 0; //#11335
            $k[$j++] = 0; //#11335
            $forall($1.code, function() { //#11335
                var _O = $k[--$j]; //#11335
                var _P = $k[--$j]; //#11335
                var _Q = $k[--$j]; //#11335
                $k[$j++] = $f(_P + $f(_O - 48)); //#11335
                $k[$j++] = _Q; //#11335
            }); //#11335
            var _R = $k[--$j]; //#11335
            var _S = $k[--$j]; //#11335
            $k[$j++] = _R; //#11335
            $k[$j++] = _S; //#11335
            $j--; //#11335
            var _T = $k[--$j]; //#11335
            var _W = $s($1.code.length + 1); //#11337
            $puti(_W, 0, $1.code); //#11337
            $put(_W, $1.code.length, $f((($f(10 - ($f($k[--$j] + _T) % 10))) % 10) + 48)); //#11337
            $k[$j++] = _W; //#11337
        }; //#11338
        $1.mod11 = function() {
            $1.code = $k[--$j]; //#11341
            $k[$j++] = $1.code.length - 1; //#11342
            $forall($1.code, function() { //#11342
                var _c = $k[--$j]; //#11342
                var _d = $k[--$j]; //#11342
                $k[$j++] = $f(_c - 48); //#11342
                $k[$j++] = $f((_d % 6) + 2); //#11342
                $k[$j++] = $f(_d - 1); //#11342
            }); //#11342
            $j--; //#11342
            $k[$j++] = 0; //#11343
            for (var _f = 0, _g = $1.code.length; _f < _g; _f++) { //#11343
                var _h = $k[--$j]; //#11343
                var _i = $k[--$j]; //#11343
                var _j = $k[--$j]; //#11343
                $k[$j++] = $f(_h + (_j * _i)); //#11343
            } //#11343
            var _l = ($f(11 - ($k[--$j] % 11))) % 11; //#11345
            $k[$j++] = _l; //#11349
            if ((_l == 10) && $1.badmod11) { //#11348
                $j--; //#11346
                var _o = $s($1.code.length + 2); //#11346
                $puti(_o, 0, $1.code); //#11346
                $puti(_o, $1.code.length, "10"); //#11346
                $k[$j++] = _o; //#11346
            } else { //#11348
                var _s = $s($1.code.length + 1); //#11348
                $puti(_s, 0, $1.code); //#11348
                $put(_s, $1.code.length, $f($k[--$j] + 48)); //#11348
                $k[$j++] = _s; //#11348
            } //#11348
        }; //#11350
        $1.ncrmod11 = function() {
            $1.code = $k[--$j]; //#11353
            $k[$j++] = $1.code.length - 1; //#11354
            $forall($1.code, function() { //#11354
                var _z = $k[--$j]; //#11354
                var _10 = $k[--$j]; //#11354
                $k[$j++] = $f(_z - 48); //#11354
                $k[$j++] = $f((_10 % 8) + 2); //#11354
                $k[$j++] = $f(_10 - 1); //#11354
            }); //#11354
            $j--; //#11354
            $k[$j++] = 0; //#11355
            for (var _12 = 0, _13 = $1.code.length; _12 < _13; _12++) { //#11355
                var _14 = $k[--$j]; //#11355
                var _15 = $k[--$j]; //#11355
                var _16 = $k[--$j]; //#11355
                $k[$j++] = $f(_14 + (_16 * _15)); //#11355
            } //#11355
            var _18 = ($f(11 - ($k[--$j] % 11))) % 11; //#11357
            $k[$j++] = _18; //#11361
            if ((_18 == 10) && $1.badmod11) { //#11360
                $j--; //#11358
                var _1B = $s($1.code.length + 2); //#11358
                $puti(_1B, 0, $1.code); //#11358
                $puti(_1B, $1.code.length, "10"); //#11358
                $k[$j++] = _1B; //#11358
            } else { //#11360
                var _1F = $s($1.code.length + 1); //#11360
                $puti(_1F, 0, $1.code); //#11360
                $put(_1F, $1.code.length, $f($k[--$j] + 48)); //#11360
                $k[$j++] = _1F; //#11360
            } //#11360
        }; //#11362
        if ($1.includecheck) { //#11375
            if ($eq($1.checktype, "mod10")) { //#11366
                $k[$j++] = $1.barcode; //#11366
                $1.mod10(); //#11366
            } //#11366
            if ($eq($1.checktype, "mod1010")) { //#11367
                $k[$j++] = $1.barcode; //#11367
                $1.mod10(); //#11367
                $1.mod10(); //#11367
            } //#11367
            if ($eq($1.checktype, "mod11")) { //#11368
                $k[$j++] = $1.barcode; //#11368
                $1.mod11(); //#11368
            } //#11368
            if ($eq($1.checktype, "ncrmod11")) { //#11369
                $k[$j++] = $1.barcode; //#11369
                $1.ncrmod11(); //#11369
            } //#11369
            if ($eq($1.checktype, "mod1110")) { //#11370
                $k[$j++] = $1.barcode; //#11370
                $1.mod11(); //#11370
                $1.mod10(); //#11370
            } //#11370
            if ($eq($1.checktype, "ncrmod1110")) { //#11371
                $k[$j++] = $1.barcode; //#11371
                $1.ncrmod11(); //#11371
                $1.mod10(); //#11371
            } //#11371
            $1.barcode = $k[--$j]; //#11372
            $1.barlen = $1.barcode.length; //#11373
            if ($1.includecheckintext) { //#11374
                $1.txtlen = $1.barlen; //#11374
            } //#11374
        } //#11374
        $1.sbs = $s(($1.barlen * 8) + 5); //#11377
        $1.txt = $a($1.barlen); //#11378
        $puti($1.sbs, 0, $get($1.encs, 10)); //#11381
        for (var _1j = 0, _1i = $1.barlen - 1; _1j <= _1i; _1j += 1) { //#11393
            $1.i = _1j; //#11384
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#11386
            $j--; //#11387
            $1.indx = $k[--$j].length; //#11388
            $j -= 2; //#11389
            $1.enc = $get($1.encs, $1.indx); //#11390
            $puti($1.sbs, ($1.i * 8) + 2, $1.enc); //#11391
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), ($1.i * 12) + 3, $1.textyoffset, $1.textfont, $1.textsize])); //#11392
        } //#11392
        $puti($1.sbs, ($1.barlen * 8) + 2, $get($1.encs, 11)); //#11396
        $k[$j++] = Infinity; //#11407
        $k[$j++] = Infinity; //#11401
        var _29 = $1.sbs; //#11401
        for (var _2A = 0, _2B = _29.length; _2A < _2B; _2A++) { //#11401
            $k[$j++] = $get(_29, _2A) - 48; //#11401
        } //#11401
        var _2D = $a(); //#11401
        $k[$j++] = Infinity; //#11402
        for (var _2F = 0, _2G = ~~(($1.sbs.length + 1) / 2); _2F < _2G; _2F++) { //#11402
            $k[$j++] = $1.height; //#11402
        } //#11402
        var _2I = $a(); //#11402
        $k[$j++] = Infinity; //#11403
        for (var _2K = 0, _2L = ~~(($1.sbs.length + 1) / 2); _2K < _2L; _2K++) { //#11403
            $k[$j++] = 0; //#11403
        } //#11403
        var _2M = $a(); //#11403
        $k[$j++] = 'ren'; //#11406
        $k[$j++] = $0.renlinear; //#11406
        $k[$j++] = 'sbs'; //#11406
        $k[$j++] = _2D; //#11406
        $k[$j++] = 'bhs'; //#11406
        $k[$j++] = _2I; //#11406
        $k[$j++] = 'bbs'; //#11406
        $k[$j++] = _2M; //#11406
        if ($1.includetext) { //#11406
            $k[$j++] = 'txt'; //#11405
            $k[$j++] = $geti($1.txt, 0, $1.txtlen); //#11405
        } //#11405
        $k[$j++] = 'opt'; //#11407
        $k[$j++] = $1.options; //#11407
        var _2S = $d(); //#11407
        $k[$j++] = _2S; //#11410
        if (!$1.dontdraw) { //#11410
            $0.renlinear(); //#11410
        } //#11410
    };
    $0.plessey = function() {
        var $1 = {}; //#11439
        $1.options = $k[--$j]; //#11441
        $1.barcode = $k[--$j]; //#11442
        $1.dontdraw = false; //#11444
        $1.includetext = false; //#11445
        $1.validatecheck = false; //#11446
        $1.includecheckintext = false; //#11447
        $1.unidirectional = false; //#11448
        $1.textfont = "OCR-B"; //#11449
        $1.textsize = 10; //#11450
        $1.textyoffset = -8; //#11451
        $1.height = 1; //#11452
        $forall($1.options, function() { //#11463
            var _3 = $k[--$j]; //#11463
            $1[$k[--$j]] = _3; //#11463
        }); //#11463
        $1.textfont = "" + $1.textfont; //#11465
        $1.textsize = +$1.textsize; //#11466
        $1.textyoffset = +$1.textyoffset; //#11467
        $1.height = +$1.height; //#11468
        $1.barchars = "0123456789ABCDEF"; //#11471
        $1.charvals = new Map; //#11472
        for (var _9 = 0; _9 <= 15; _9 += 1) { //#11473
            $put($1.charvals, $geti($1.barchars, _9, 1), _9); //#11473
        } //#11473
        for (var _F = 0, _E = $1.barcode.length - 1; _F <= _E; _F += 1) { //#11480
            var _J = $get($1.charvals, $geti($1.barcode, _F, 1)) !== undefined; //#11477
            if (!_J) { //#11479
                $k[$j++] = 'bwipp.plesseyBadCharacter'; //#11478
                $k[$j++] = "Plessey must contain only digits and letters A B C D E F"; //#11478
                $0.raiseerror(); //#11478
            } //#11478
        } //#11478
        $k[$j++] = 'barlen'; //#11482
        $k[$j++] = $1.barcode.length; //#11482
        if ($1.validatecheck) { //#11482
            var _M = $k[--$j]; //#11482
            $k[$j++] = $f(_M - 2); //#11482
        } //#11482
        var _N = $k[--$j]; //#11482
        $1[$k[--$j]] = _N; //#11482
        $1.checkbits = $a($f(($1.barlen * 4) + 8)); //#11485
        $puti($1.checkbits, $1.barlen * 4, $a([0, 0, 0, 0, 0, 0, 0, 0])); //#11486
        for (var _W = 0, _V = $f($1.barlen - 1); _W <= _V; _W += 1) { //#11496
            $1.i = _W; //#11488
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#11489
            $puti($1.checkbits, $1.i * 4, $a([$1.indx & 1, ($1.indx >>> 1) & 1, ($1.indx >>> 2) & 1, $1.indx >>> 3])); //#11495
        } //#11495
        $1.checksalt = $a([1, 1, 1, 1, 0, 1, 0, 0, 1]); //#11497
        for (var _m = 0, _l = $f(($1.barlen * 4) - 1); _m <= _l; _m += 1) { //#11506
            $1.i = _m; //#11499
            if ($get($1.checkbits, $1.i) == 1) { //#11505
                for (var _q = 0; _q <= 8; _q += 1) { //#11504
                    $1.j = _q; //#11502
                    $put($1.checkbits, $1.i + $1.j, $xo($get($1.checkbits, $1.i + $1.j), $get($1.checksalt, $1.j))); //#11503
                } //#11503
            } //#11503
        } //#11503
        $1.checkval = 0; //#11507
        for (var _11 = 0; _11 <= 7; _11 += 1) { //#11511
            $1.i = _11; //#11509
            $1.checkval = $f($1.checkval + ((~~Math.pow(2, $1.i)) * $get($1.checkbits, $f(($1.barlen * 4) + $1.i)))); //#11510
        } //#11510
        $1.checksum1 = $1.checkval & 15; //#11512
        $1.checksum2 = $1.checkval >>> 4; //#11513
        if ($1.validatecheck) { //#11520
            if (($get($1.barcode, $1.barlen) != $get($1.barchars, $1.checksum1)) || ($get($1.barcode, $f($1.barlen + 1)) != $get($1.barchars, $1.checksum2))) { //#11518
                $k[$j++] = 'bwipp.plesseyBadCheckDigits'; //#11517
                $k[$j++] = "Incorrect Plessey check digits provided"; //#11517
                $0.raiseerror(); //#11517
            } //#11517
            $1.barcode = $geti($1.barcode, 0, $1.barlen); //#11519
        } //#11519
        $1.encs = $a(["14141414", "32141414", "14321414", "32321414", "14143214", "32143214", "14323214", "32323214", "14141432", "32141432", "14321432", "32321432", "14143232", "32143232", "14323232", "32323232", "32321432", "541412323", "323"]); //#11529
        var _1T = $1.unidirectional ? 27 : 33; //#11531
        $1.sbs = $s($f(($1.barlen * 8) + _1T)); //#11531
        $1.txt = $a($f($1.barlen + 2)); //#11532
        $puti($1.sbs, 0, $get($1.encs, 16)); //#11535
        for (var _1c = 0, _1b = $f($1.barlen - 1); _1c <= _1b; _1c += 1) { //#11542
            $1.i = _1c; //#11538
            $1.indx = $get($1.charvals, $geti($1.barcode, $1.i, 1)); //#11539
            $puti($1.sbs, ($1.i * 8) + 8, $get($1.encs, $1.indx)); //#11540
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), ($1.i * 20) + 20, $1.textyoffset, $1.textfont, $1.textsize])); //#11541
        } //#11541
        $puti($1.sbs, $f(($1.barlen * 8) + 8), $get($1.encs, $1.checksum1)); //#11545
        $puti($1.sbs, $f(($1.barlen * 8) + 16), $get($1.encs, $1.checksum2)); //#11546
        if ($1.includecheckintext) { //#11552
            $put($1.txt, $1.barlen, $a([$geti($1.barchars, $1.checksum1, 1), $f(($1.barlen * 20) + 20), $1.textyoffset, $1.textfont, $1.textsize])); //#11548
            $put($1.txt, $f($1.barlen + 1), $a([$geti($1.barchars, $1.checksum2, 1), $f(($f($1.barlen + 1) * 20) + 20), $1.textyoffset, $1.textfont, $1.textsize])); //#11549
        } else { //#11552
            $put($1.txt, $1.barlen, $a([" ", $f(($1.barlen * 20) + 20), $1.textyoffset, $1.textfont, $1.textsize])); //#11551
            $put($1.txt, $f($1.barlen + 1), $a([" ", $f(($f($1.barlen + 1) * 20) + 20), $1.textyoffset, $1.textfont, $1.textsize])); //#11552
        } //#11552
        var _2k = $1.unidirectional ? 18 : 17; //#11556
        $puti($1.sbs, $f(($1.barlen * 8) + 24), $get($1.encs, _2k)); //#11556
        $k[$j++] = Infinity; //#11567
        $k[$j++] = Infinity; //#11561
        var _2m = $1.sbs; //#11561
        for (var _2n = 0, _2o = _2m.length; _2n < _2o; _2n++) { //#11561
            $k[$j++] = $get(_2m, _2n) - 48; //#11561
        } //#11561
        var _2q = $a(); //#11561
        $k[$j++] = Infinity; //#11562
        for (var _2s = 0, _2t = ~~(($1.sbs.length + 1) / 2); _2s < _2t; _2s++) { //#11562
            $k[$j++] = $1.height; //#11562
        } //#11562
        var _2v = $a(); //#11562
        $k[$j++] = Infinity; //#11563
        for (var _2x = 0, _2y = ~~(($1.sbs.length + 1) / 2); _2x < _2y; _2x++) { //#11563
            $k[$j++] = 0; //#11563
        } //#11563
        var _2z = $a(); //#11563
        $k[$j++] = 'ren'; //#11566
        $k[$j++] = $0.renlinear; //#11566
        $k[$j++] = 'sbs'; //#11566
        $k[$j++] = _2q; //#11566
        $k[$j++] = 'bhs'; //#11566
        $k[$j++] = _2v; //#11566
        $k[$j++] = 'bbs'; //#11566
        $k[$j++] = _2z; //#11566
        if ($1.includetext) { //#11566
            $k[$j++] = 'txt'; //#11565
            $k[$j++] = $1.txt; //#11565
        } //#11565
        $k[$j++] = 'opt'; //#11567
        $k[$j++] = $1.options; //#11567
        var _33 = $d(); //#11567
        $k[$j++] = _33; //#11570
        if (!$1.dontdraw) { //#11570
            $0.renlinear(); //#11570
        } //#11570
    };
    $0.telepen = function() {
        var $1 = {}; //#11600
        $1.options = $k[--$j]; //#11602
        $1.barcode = $k[--$j]; //#11603
        $1.numeric = false; //#11605
        $1.dontdraw = false; //#11606
        $1.includetext = false; //#11607
        $1.textfont = "OCR-B"; //#11608
        $1.textsize = 10; //#11609
        $1.textyoffset = -8; //#11610
        $1.height = 1; //#11611
        $1.parse = false; //#11612
        $forall($1.options, function() { //#11623
            var _3 = $k[--$j]; //#11623
            $1[$k[--$j]] = _3; //#11623
        }); //#11623
        $1.textfont = "" + $1.textfont; //#11625
        $1.textsize = +$1.textsize; //#11626
        $1.textyoffset = +$1.textyoffset; //#11627
        $1.height = +$1.height; //#11628
        var _A = new Map([
            ["parse", $1.parse],
            ["parseonly", true],
            ["parsefnc", false]
        ]); //#11634
        $1.fncvals = _A; //#11635
        $k[$j++] = 'barcode'; //#11636
        $k[$j++] = $1.barcode; //#11636
        $k[$j++] = $1.fncvals; //#11636
        $0.parseinput(); //#11636
        var _D = $k[--$j]; //#11636
        $1[$k[--$j]] = _D; //#11636
        $1.barlen = $1.barcode.length; //#11637
        delete $1.options["parse"]; //#11638
        $1.encs = $a(["31313131", "1131313111", "33313111", "1111313131", "3111313111", "11333131", "13133131", "111111313111", "31333111", "1131113131", "33113131", "1111333111", "3111113131", "1113133111", "1311133111", "111111113131", "3131113111", "11313331", "333331", "111131113111", "31113331", "1133113111", "1313113111", "1111113331", "31131331", "113111113111", "3311113111", "1111131331", "311111113111", "1113111331", "1311111331", "11111111113111", "31313311", "1131311131", "33311131", "1111313311", "3111311131", "11333311", "13133311", "111111311131", "31331131", "1131113311", "33113311", "1111331131", "3111113311", "1113131131", "1311131131", "111111113311", "3131111131", "1131131311", "33131311", "111131111131", "3111131311", "1133111131", "1313111131", "111111131311", "3113111311", "113111111131", "3311111131", "111113111311", "311111111131", "111311111311", "131111111311", "11111111111131", "3131311111", "11313133", "333133", "111131311111", "31113133", "1133311111", "1313311111", "1111113133", "313333", "113111311111", "3311311111", "11113333", "311111311111", "11131333", "13111333", "11111111311111", "31311133", "1131331111", "33331111", "1111311133", "3111331111", "11331133", "13131133", "111111331111", "3113131111", "1131111133", "33111133", "111113131111", "3111111133", "111311131111", "131111131111", "111111111133", "31311313", "113131111111", "3331111111", "1111311313", "311131111111", "11331313", "13131313", "11111131111111", "3133111111", "1131111313", "33111313", "111133111111", "3111111313", "111313111111", "131113111111", "111111111313", "313111111111", "1131131113", "33131113", "11113111111111", "3111131113", "113311111111", "131311111111", "111111131113", "3113111113", "11311111111111", "331111111111", "111113111113", "31111111111111", "111311111113", "131111111113", "1111111111111111"]); //#11674
        $1.barlen = $1.barcode.length; //#11676
        $1.sbs = $s(($1.barlen * 16) + 48); //#11677
        $1.txt = $a($1.barlen); //#11678
        $1.enc = $get($1.encs, 95); //#11681
        $puti($1.sbs, 0, $1.enc); //#11682
        $1.l = $1.enc.length; //#11683
        $1.checksum = 0; //#11685
        $1.i = 0; //#11686
        $1.j = 0; //#11686
        for (;;) { //#11718
            if ($1.i == $1.barlen) { //#11688
                break; //#11688
            } //#11688
            if ($1.numeric) { //#11711
                if ($get($1.barcode, $1.i) > 16) { //#11702
                    $1.np = $geti($1.barcode, $1.i, 2); //#11691
                    if ($eq($geti($1.np, 1, 1), "X")) { //#11695
                        $1.indx = (~~$z($geti($1.np, 0, 1))) + 17; //#11693
                    } else { //#11695
                        $1.indx = (~~$z($1.np)) + 27; //#11695
                    } //#11695
                    $put($1.txt, $1.j, $a([$1.np, ($1.j * 16) + 16, $1.textyoffset, $1.textfont, $1.textsize])); //#11697
                    $1.i = $1.i + 2; //#11698
                } else { //#11702
                    $1.indx = $get($1.barcode, $1.i); //#11700
                    $put($1.txt, $1.j, $a([" ", ($1.j * 16) + 16, $1.textyoffset, $1.textfont, $1.textsize])); //#11701
                    $1.i = $1.i + 1; //#11702
                } //#11702
            } else { //#11711
                $1.indx = $get($1.barcode, $1.i); //#11705
                if (($1.indx >= 32) && ($1.indx <= 126)) { //#11709
                    $put($1.txt, $1.j, $a([$geti($1.barcode, $1.i, 1), ($1.j * 16) + 16, $1.textyoffset, $1.textfont, $1.textsize])); //#11707
                } else { //#11709
                    $put($1.txt, $1.j, $a([" ", ($1.j * 16) + 16, $1.textyoffset, $1.textfont, $1.textsize])); //#11709
                } //#11709
                $1.i = $1.i + 1; //#11711
            } //#11711
            $1.checksum = $f($1.checksum + $1.indx); //#11713
            $1.enc = $get($1.encs, $1.indx); //#11714
            $puti($1.sbs, $1.l, $1.enc); //#11715
            $1.l = $1.l + $1.enc.length; //#11716
            $1.j = $1.j + 1; //#11717
        } //#11717
        $1.checksum = ($f(127 - ($1.checksum % 127))) % 127; //#11721
        $1.enc = $get($1.encs, $1.checksum); //#11722
        $puti($1.sbs, $1.l, $1.enc); //#11723
        $1.l = $1.l + $1.enc.length; //#11724
        $1.enc = $get($1.encs, 122); //#11727
        $puti($1.sbs, $1.l, $1.enc); //#11728
        $1.l = $1.l + $1.enc.length; //#11729
        $1.sbs = $geti($1.sbs, 0, $1.l); //#11732
        $1.txt = $geti($1.txt, 0, $1.j); //#11733
        $k[$j++] = Infinity; //#11744
        $k[$j++] = Infinity; //#11738
        var _1u = $1.sbs; //#11738
        for (var _1v = 0, _1w = _1u.length; _1v < _1w; _1v++) { //#11738
            $k[$j++] = $f($get(_1u, _1v) - 48); //#11738
        } //#11738
        var _1y = $a(); //#11738
        $k[$j++] = Infinity; //#11739
        for (var _20 = 0, _21 = ~~(($1.sbs.length + 1) / 2); _20 < _21; _20++) { //#11739
            $k[$j++] = $1.height; //#11739
        } //#11739
        var _23 = $a(); //#11739
        $k[$j++] = Infinity; //#11740
        for (var _25 = 0, _26 = ~~(($1.sbs.length + 1) / 2); _25 < _26; _25++) { //#11740
            $k[$j++] = 0; //#11740
        } //#11740
        var _27 = $a(); //#11740
        $k[$j++] = 'ren'; //#11743
        $k[$j++] = $0.renlinear; //#11743
        $k[$j++] = 'sbs'; //#11743
        $k[$j++] = _1y; //#11743
        $k[$j++] = 'bhs'; //#11743
        $k[$j++] = _23; //#11743
        $k[$j++] = 'bbs'; //#11743
        $k[$j++] = _27; //#11743
        if ($1.includetext) { //#11743
            $k[$j++] = 'txt'; //#11742
            $k[$j++] = $1.txt; //#11742
        } //#11742
        $k[$j++] = 'opt'; //#11744
        $k[$j++] = $1.options; //#11744
        var _2B = $d(); //#11744
        $k[$j++] = _2B; //#11747
        if (!$1.dontdraw) { //#11747
            $0.renlinear(); //#11747
        } //#11747
    };
    $0.telepennumeric = function() {
        var $1 = {}; //#11777
        $1.options = $k[--$j]; //#11779
        $1.barcode = $k[--$j]; //#11780
        $1.dontdraw = false; //#11782
        $1.includetext = false; //#11783
        $1.textfont = "OCR-B"; //#11784
        $1.textsize = 10; //#11785
        $1.textxoffset = 0; //#11786
        $1.textyoffset = -8; //#11787
        $1.height = 1; //#11788
        $forall($1.options, function() { //#11799
            var _3 = $k[--$j]; //#11799
            $1[$k[--$j]] = _3; //#11799
        }); //#11799
        $1.textfont = "" + $1.textfont; //#11801
        $1.textsize = +$1.textsize; //#11802
        $1.textxoffset = +$1.textxoffset; //#11803
        $1.textyoffset = +$1.textyoffset; //#11804
        $1.height = +$1.height; //#11805
        $put($1.options, "dontdraw", true); //#11808
        $put($1.options, "numeric", true); //#11809
        $k[$j++] = 'args'; //#11811
        $k[$j++] = $1.barcode; //#11811
        $k[$j++] = $1.options; //#11811
        $0.telepen(); //#11811
        var _E = $k[--$j]; //#11811
        $1[$k[--$j]] = _E; //#11811
        $put($1.args, "opt", $1.options); //#11813
        $k[$j++] = $1.args; //#11816
        if (!$1.dontdraw) { //#11816
            $0.renlinear(); //#11816
        } //#11816
    };
    $0.posicode = function() {
        var $1 = {}; //#11846
        $1.options = $k[--$j]; //#11848
        $1.barcode = $k[--$j]; //#11849
        $1.dontdraw = false; //#11851
        $1.includetext = false; //#11852
        $1.textfont = "OCR-B"; //#11853
        $1.textsize = 10; //#11854
        $1.textxoffset = 0; //#11855
        $1.textyoffset = -8; //#11856
        $1.height = 1; //#11857
        $1.encoding = "auto"; //#11858
        $1.version = "a"; //#11859
        $1.checkoffset = 0; //#11860
        $1.raw = false; //#11861
        $1.parse = false; //#11862
        $1.parsefnc = false; //#11863
        $forall($1.options, function() { //#11874
            var _3 = $k[--$j]; //#11874
            $1[$k[--$j]] = _3; //#11874
        }); //#11874
        $1.textfont = "" + $1.textfont; //#11876
        $1.textsize = +$1.textsize; //#11877
        $1.textxoffset = +$1.textxoffset; //#11878
        $1.textyoffset = +$1.textyoffset; //#11879
        $1.encoding = "" + $1.encoding; //#11880
        $1.version = "" + $1.version; //#11881
        $1.checkoffset = ~~$1.checkoffset; //#11882
        $1.height = +$1.height; //#11883
        $1.la0 = -1; //#11886
        $1.la1 = -2; //#11886
        $1.la2 = -3; //#11886
        $1.sf0 = -4; //#11887
        $1.sf1 = -5; //#11887
        $1.sf2 = -6; //#11887
        $1.fn1 = -7; //#11888
        $1.fn2 = -8; //#11888
        $1.fn3 = -9; //#11888
        $1.fn4 = -10; //#11888
        var _1l = new Map([
            ["normal", $a([$a(["0", "^", "'"]), $a(["1", ";", 27]), $a(["2", "<", 28]), $a(["3", "=", 29]), $a(["4", ">", 30]), $a(["5", "?", 31]), $a(["6", "@", "!"]), $a(["7", "[", "\""]), $a(["8", 92, "#"]), $a(["9", "]", "&"]), $a(["A", "a", 1]), $a(["B", "b", 2]), $a(["C", "c", 3]), $a(["D", "d", 4]), $a(["E", "e", 5]), $a(["F", "f", 6]), $a(["G", "g", 7]), $a(["H", "h", 8]), $a(["I", "i", 9]), $a(["J", "j", 10]), $a(["K", "k", 11]), $a(["L", "l", 12]), $a(["M", "m", 13]), $a(["N", "n", 14]), $a(["O", "o", 15]), $a(["P", "p", 16]), $a(["Q", "q", 17]), $a(["R", "r", 18]), $a(["S", "s", 19]), $a(["T", "t", 20]), $a(["U", "u", 21]), $a(["V", "v", 22]), $a(["W", "w", 23]), $a(["X", "x", 24]), $a(["Y", "y", 25]), $a(["Z", "z", 26]), $a(["-", "_", 40]), $a([".", "`", 41]), $a([" ", 127, 0]), $a(["$", "{", "*"]), $a(["/", "|", ","]), $a(["+", "}", ":"]), $a(["%", "~", $1.fn1]), $a([$1.la1, $1.la0, $1.fn2]), $a([$1.sf1, $1.sf0, $1.fn3]), $a([$1.sf2, $1.sf2, $1.fn4])])],
            ["limited", $a([$a(["0", -98, -98]), $a(["1", -98, -98]), $a(["2", -98, -98]), $a(["3", -98, -98]), $a(["4", -98, -98]), $a(["5", -98, -98]), $a(["6", -98, -98]), $a(["7", -98, -98]), $a(["8", -98, -98]), $a(["9", -98, -98]), $a(["A", -98, -98]), $a(["B", -98, -98]), $a(["C", -98, -98]), $a(["D", -98, -98]), $a(["E", -98, -98]), $a(["F", -98, -98]), $a(["G", -98, -98]), $a(["H", -98, -98]), $a(["I", -98, -98]), $a(["J", -98, -98]), $a(["K", -98, -98]), $a(["L", -98, -98]), $a(["M", -98, -98]), $a(["N", -98, -98]), $a(["O", -98, -98]), $a(["P", -98, -98]), $a(["Q", -98, -98]), $a(["R", -98, -98]), $a(["S", -98, -98]), $a(["T", -98, -98]), $a(["U", -98, -98]), $a(["V", -98, -98]), $a(["W", -98, -98]), $a(["X", -98, -98]), $a(["Y", -98, -98]), $a(["Z", -98, -98]), $a(["-", -98, -98]), $a([".", -98, -98])])]
        ]); //#11926
        $k[$j++] = _1l; //#11929
        if ($eq($1.version, "a") || $eq($1.version, "b")) { //#11929
            $k[$j++] = 'normal'; //#11929
        } else { //#11929
            $k[$j++] = 'limited'; //#11929
        } //#11929
        var _1o = $k[--$j]; //#11929
        $1.charmaps = $get($k[--$j], _1o); //#11930
        var _1r = $1.charmaps; //#11934
        var _1s = $1.charmaps; //#11934
        var _1t = $1.charmaps; //#11934
        $1.charvals = $a([new Map, new Map, new Map]); //#11935
        for (var _1x = 0, _1w = $1.charmaps.length - 1; _1x <= _1w; _1x += 1) { //#11944
            $1.i = _1x; //#11937
            $1.encs = $get($1.charmaps, $1.i); //#11938
            for (var _21 = 0; _21 <= 2; _21 += 1) { //#11943
                $1.j = _21; //#11940
                var _24 = $get($1.encs, $1.j); //#11941
                $k[$j++] = _24; //#11941
                if ($eq($type(_24), 'stringtype')) { //#11941
                    var _27 = $get($k[--$j], 0); //#11941
                    $k[$j++] = _27; //#11941
                } //#11941
                $put($get($1.charvals, $1.j), $k[--$j], $1.i); //#11942
            } //#11942
        } //#11942
        $1.set0 = $get($1.charvals, 0); //#11945
        $1.set1 = $get($1.charvals, 1); //#11946
        $1.set2 = $get($1.charvals, 2); //#11947
        if ($1.raw) { //#11949
            $1.encoding = "raw"; //#11949
        } //#11949
        if ($eq($1.encoding, "raw")) { //#11963
            $1.cws = $a($1.barcode.length); //#11952
            $1.i = 0; //#11953
            $1.j = 0; //#11953
            for (;;) { //#11960
                if ($1.i == $1.barcode.length) { //#11955
                    break; //#11955
                } //#11955
                $1.cw = ~~$z($geti($1.barcode, $1.i + 1, 3)); //#11956
                $put($1.cws, $1.j, $1.cw); //#11957
                $1.i = $1.i + 4; //#11958
                $1.j = $1.j + 1; //#11959
            } //#11959
            $1.cws = $geti($1.cws, 0, $1.j); //#11961
            $1.text = ""; //#11962
        } //#11962
        if ($eq($1.encoding, "auto")) { //#12078
            var _2g = new Map([
                ["parse", $1.parse],
                ["parsefnc", $1.parsefnc],
                ["FNC1", $1.fn1],
                ["FNC2", $1.fn2],
                ["FNC3", $1.fn3]
            ]); //#11973
            $1.fncvals = _2g; //#11975
            $k[$j++] = 'msg'; //#11976
            $k[$j++] = $1.barcode; //#11976
            $k[$j++] = $1.fncvals; //#11976
            $0.parseinput(); //#11976
            var _2j = $k[--$j]; //#11976
            $1[$k[--$j]] = _2j; //#11976
            $1.msglen = $1.msg.length; //#11977
            $1.text = $s($1.msglen); //#11979
            for (var _2q = 0, _2p = $1.msglen - 1; _2q <= _2p; _2q += 1) { //#11983
                $1.i = _2q; //#11981
                var _2v = $get($1.msg, $1.i); //#11982
                $k[$j++] = $1.text; //#11982
                $k[$j++] = $1.i; //#11982
                $k[$j++] = _2v; //#11982
                if (_2v < 0) { //#11982
                    $j--; //#11982
                    $k[$j++] = 32; //#11982
                } //#11982
                var _2w = $k[--$j]; //#11982
                var _2x = $k[--$j]; //#11982
                $put($k[--$j], _2x, _2w); //#11982
            } //#11982
            $k[$j++] = Infinity; //#11986
            for (var _30 = 0, _31 = $1.msglen; _30 < _31; _30++) { //#11986
                $k[$j++] = 0; //#11986
            } //#11986
            $k[$j++] = 0; //#11986
            $1.numSA = $a(); //#11986
            $k[$j++] = Infinity; //#11987
            for (var _34 = 0, _35 = $1.msglen; _34 < _35; _34++) { //#11987
                $k[$j++] = 0; //#11987
            } //#11987
            $k[$j++] = 0; //#11987
            $1.numEA = $a(); //#11987
            for (var _38 = $1.msglen - 1; _38 >= 0; _38 -= 1) { //#11997
                $1.i = _38; //#11989
                if ($get($1.msg, $1.i) >= 0) { //#11996
                    if ($get($1.msg, $1.i) >= 128) { //#11994
                        $put($1.numEA, $1.i, $f($get($1.numEA, $1.i + 1) + 1)); //#11992
                    } else { //#11994
                        $put($1.numSA, $1.i, $f($get($1.numSA, $1.i + 1) + 1)); //#11994
                    } //#11994
                } //#11994
            } //#11994
            $1.ea = false; //#12000
            $1.msgtmp = $a([]); //#12000
            for (var _3S = 0, _3R = $1.msglen - 1; _3S <= _3R; _3S += 1) { //#12015
                $1.i = _3S; //#12002
                $1.c = $get($1.msg, $1.i); //#12003
                if ((!$xo($1.ea, $1.c < 128)) && ($1.c >= 0)) { //#12013
                    if ($1.ea) { //#12005
                        $k[$j++] = $1.numSA; //#12005
                    } else { //#12005
                        $k[$j++] = $1.numEA; //#12005
                    } //#12005
                    var _3e = $get($k[--$j], $1.i); //#12005
                    var _3h = ($f(_3e + $1.i) == $1.msglen) ? 3 : 5; //#12006
                    if (_3e < _3h) { //#12011
                        $k[$j++] = Infinity; //#12008
                        $aload($1.msgtmp); //#12008
                        $k[$j++] = $1.fn4; //#12008
                        $1.msgtmp = $a(); //#12008
                    } else { //#12011
                        $k[$j++] = Infinity; //#12010
                        $aload($1.msgtmp); //#12010
                        $k[$j++] = $1.fn4; //#12010
                        $k[$j++] = $1.fn4; //#12010
                        $1.msgtmp = $a(); //#12010
                        $1.ea = !$1.ea; //#12011
                    } //#12011
                } //#12011
                $k[$j++] = Infinity; //#12014
                $aload($1.msgtmp); //#12014
                if ($1.c >= 0) { //#12014
                    $k[$j++] = $1.c & 127; //#12014
                } else { //#12014
                    $k[$j++] = $1.c; //#12014
                } //#12014
                $1.msgtmp = $a(); //#12014
            } //#12014
            $1.msg = $1.msgtmp; //#12016
            $1.msglen = $1.msg.length; //#12017
            $1.enc = function() {
                var _3x = $k[--$j]; //#12021
                $put($1.cws, $1.j, $get(_3x, $k[--$j])); //#12021
                $1.j = $1.j + 1; //#12022
            }; //#12023
            $1.cws = $a($1.msglen * 2); //#12025
            $1.i = 0; //#12026
            $1.j = 0; //#12026
            $1.cset = "set0"; //#12026
            for (;;) { //#12076
                if ($1.i == $1.msglen) { //#12028
                    break; //#12028
                } //#12028
                for (;;) { //#12074
                    $1.char1 = $get($1.msg, $1.i); //#12033
                    $k[$j++] = 'char2'; //#12034
                    if (($1.i + 1) < $1.msglen) { //#12034
                        $k[$j++] = $get($1.msg, $1.i + 1); //#12034
                    } else { //#12034
                        $k[$j++] = -99; //#12034
                    } //#12034
                    var _4F = $k[--$j]; //#12034
                    $1[$k[--$j]] = _4F; //#12034
                    var _4K = $get($1[$1.cset], $1.char1) !== undefined; //#12037
                    if (_4K) { //#12041
                        $k[$j++] = $1.char1; //#12038
                        $k[$j++] = $1[$1.cset]; //#12038
                        $1.enc(); //#12038
                        $1.i = $1.i + 1; //#12039
                        break; //#12040
                    } //#12040
                    var _4R = $get($1.set2, $1.char1) !== undefined; //#12044
                    if (_4R) { //#12049
                        $k[$j++] = $1.sf2; //#12045
                        $k[$j++] = $1[$1.cset]; //#12045
                        $1.enc(); //#12045
                        $k[$j++] = $1.char1; //#12046
                        $k[$j++] = $1.set2; //#12046
                        $1.enc(); //#12046
                        $1.i = $1.i + 1; //#12047
                        break; //#12048
                    } //#12048
                    var _4b = $get($1[$1.cset], $1.char2) !== undefined; //#12052
                    if (!_4b) { //#12070
                        if ($eq($1.cset, "set0")) { //#12058
                            $k[$j++] = $1.la1; //#12054
                            $k[$j++] = $1[$1.cset]; //#12054
                            $1.enc(); //#12054
                            $1.cset = "set1"; //#12055
                        } else { //#12058
                            $k[$j++] = $1.la0; //#12057
                            $k[$j++] = $1[$1.cset]; //#12057
                            $1.enc(); //#12057
                            $1.cset = "set0"; //#12058
                        } //#12058
                        break; //#12060
                    } else { //#12070
                        if ($eq($1.cset, "set0")) { //#12067
                            $k[$j++] = $1.sf1; //#12063
                            $k[$j++] = $1[$1.cset]; //#12063
                            $1.enc(); //#12063
                            $k[$j++] = $1.char1; //#12064
                            $k[$j++] = $1.set1; //#12064
                            $1.enc(); //#12064
                        } else { //#12067
                            $k[$j++] = $1.sf0; //#12066
                            $k[$j++] = $1[$1.cset]; //#12066
                            $1.enc(); //#12066
                            $k[$j++] = $1.char1; //#12067
                            $k[$j++] = $1.set0; //#12067
                            $1.enc(); //#12067
                        } //#12067
                        $1.i = $1.i + 1; //#12069
                        break; //#12070
                    } //#12070
                    break; //#12073
                } //#12073
            } //#12073
            $1.cws = $geti($1.cws, 0, $1.j); //#12077
        } //#12077
        var _52 = new Map([
            ["a", $a(["141112", "131212", "121312", "111412", "131113", "121213", "111313", "121114", "111214", "111115", "181111", "171211", "161311", "151411", "141511", "131611", "121711", "111811", "171112", "161212", "151312", "141412", "131512", "121612", "111712", "161113", "151213", "141313", "131413", "121513", "111613", "151114", "141214", "131314", "121414", "111514", "141115", "131215", "121315", "111415", "131116", "121216", "111316", "121117", "111217", "111118", "1<111112", "111111111;1"])],
            ["b", $a(["151213", "141313", "131413", "121513", "141214", "131314", "121414", "131215", "121315", "121216", "191212", "181312", "171412", "161512", "151612", "141712", "131812", "121912", "181213", "171313", "161413", "151513", "141613", "131713", "121813", "171214", "161314", "151414", "141514", "131614", "121714", "161215", "151315", "141415", "131515", "121615", "151216", "141316", "131416", "121516", "141217", "131317", "121417", "131218", "121318", "121219", "1<121312", "121212121<1"])],
            ["limiteda", $a(["111411", "111312", "111213", "111114", "121311", "121212", "121113", "141111", "131211", "131112", "171111", "161211", "151311", "141411", "131511", "121611", "111711", "161112", "151212", "141312", "131412", "121512", "111612", "151113", "141213", "131313", "121413", "111513", "141114", "131214", "121314", "111414", "131115", "121215", "111315", "121116", "111216", "111117", "151111", "1"])],
            ["limitedb", $a(["121512", "121413", "121314", "121215", "131412", "131313", "131214", "151212", "141312", "141213", "181212", "171312", "161412", "151512", "141612", "131712", "121812", "171213", "161313", "151413", "141513", "131613", "121713", "161214", "151314", "141414", "131514", "121614", "151215", "141315", "131415", "121515", "141216", "131316", "121416", "131217", "121317", "121218", "141212", "1"])]
        ]); //#12124
        $1.encs = $get(_52, $1.version); //#12127
        $1.c2w = $a([$a([495, 330, 210, 126, 70, 35, 15, 5]), $a([165, 120, 84, 56, 35, 20, 10, 4]), $a([45, 36, 28, 21, 15, 10, 6, 3]), $a([9, 8, 7, 6, 5, 4, 3, 2]), $a([1, 1, 1, 1, 1, 1, 1, 1])]); //#12135
        $1.v = 0; //#12136
        var _5B = $1.cws; //#12137
        for (var _5C = 0, _5D = _5B.length; _5C < _5D; _5C++) { //#12146
            $1.cw = $get(_5B, _5C); //#12138
            for (var _5F = 0, _5G = 6; _5F < _5G; _5F++) { //#12145
                if ((($1.cw ^ $1.v) & 1) != 0) { //#12142
                    $1.v = $1.v ^ 7682; //#12141
                } //#12141
                $1.v = $1.v >>> 1; //#12143
                $1.cw = $1.cw >>> 1; //#12144
            } //#12144
        } //#12144
        $1.v = $1.v + $1.checkoffset; //#12147
        if ($eq($1.version, "limiteda") || $eq($1.version, "limitedb")) { //#12155
            $1.v = $1.v & 1023; //#12150
            if (($1.v > 824) && ($1.v < 853)) { //#12153
                $1.v = $1.v + 292; //#12152
            } //#12152
        } else { //#12155
            $1.v = ($1.v & 1023) + 45; //#12155
        } //#12155
        $1.d = $a([2, 2, 2, 2, 2, 2]); //#12158
        $1.r = 0; //#12159
        $1.c = 0; //#12159
        $1.w = 0; //#12159
        $1.sum = 0; //#12159
        for (;;) { //#12178
            if ($1.sum == $1.v) { //#12161
                break; //#12161
            } //#12161
            $1.t = $f($1.sum + $get($get($1.c2w, $1.r), $1.c)); //#12162
            if ($1.t == $1.v) { //#12167
                $1.w = $1.w + 1; //#12164
                $put($1.d, $1.r, $1.w + 2); //#12165
                $1.sum = $1.t; //#12166
            } //#12166
            if ($1.t > $1.v) { //#12172
                $put($1.d, $1.r, $1.w + 2); //#12169
                $1.r = $1.r + 1; //#12170
                $1.w = 0; //#12171
            } //#12171
            if ($1.t < $1.v) { //#12177
                $1.c = $1.c + 1; //#12174
                $1.w = $1.w + 1; //#12175
                $1.sum = $1.t; //#12176
            } //#12176
        } //#12176
        $k[$j++] = 20; //#12179
        for (var _5w = 0; _5w <= 4; _5w += 1) { //#12179
            var _5z = $k[--$j]; //#12179
            $k[$j++] = $f(_5z - $get($1.d, _5w)); //#12179
        } //#12179
        $put($1.d, 5, $k[--$j]); //#12180
        if ($eq($1.version, "b") || $eq($1.version, "limitedb")) { //#12183
            $k[$j++] = Infinity; //#12182
            var _64 = $1.d; //#12182
            for (var _65 = 0, _66 = _64.length; _65 < _66; _65++) { //#12182
                $k[$j++] = $f($get(_64, _65) + 1); //#12182
            } //#12182
            $1.d = $a(); //#12182
        } //#12182
        $1.cbs = $strcpy($s(12), "111111111111"); //#12184
        for (var _6B = 5; _6B >= 0; _6B -= 1) { //#12188
            $1.i = _6B; //#12186
            $put($1.cbs, ((5 - $1.i) * 2) + 1, $f($get($1.d, $1.i) + 47)); //#12187
        } //#12187
        $1.sbs = $s(($1.cws.length * 6) + 31); //#12191
        var _6L = $get($1.encs, $1.encs.length - 2); //#12192
        $puti($1.sbs, 0, _6L); //#12193
        $1.j = _6L.length; //#12194
        for (var _6P = 0, _6O = $1.cws.length - 1; _6P <= _6O; _6P += 1) { //#12199
            $1.i = _6P; //#12196
            $puti($1.sbs, $1.j, $get($1.encs, $get($1.cws, $1.i))); //#12197
            $1.j = $1.j + 6; //#12198
        } //#12198
        $puti($1.sbs, $1.j, $1.cbs); //#12200
        $1.j = $1.j + 12; //#12201
        var _6e = $get($1.encs, $1.encs.length - 1); //#12202
        $puti($1.sbs, $1.j, _6e); //#12203
        $1.j = _6e.length + $1.j; //#12204
        $1.sbs = $geti($1.sbs, 0, $1.j); //#12205
        $k[$j++] = Infinity; //#12215
        $k[$j++] = Infinity; //#12210
        var _6l = $1.sbs; //#12210
        for (var _6m = 0, _6n = _6l.length; _6m < _6n; _6m++) { //#12210
            $k[$j++] = $f($get(_6l, _6m) - 48); //#12210
        } //#12210
        var _6p = $a(); //#12210
        $k[$j++] = Infinity; //#12211
        for (var _6r = 0, _6s = ~~(($1.sbs.length + 1) / 2); _6r < _6s; _6r++) { //#12211
            $k[$j++] = $1.height; //#12211
        } //#12211
        var _6u = $a(); //#12211
        $k[$j++] = Infinity; //#12212
        for (var _6w = 0, _6x = ~~(($1.sbs.length + 1) / 2); _6w < _6x; _6w++) { //#12212
            $k[$j++] = 0; //#12212
        } //#12212
        var _6y = $a(); //#12212
        $k[$j++] = 'ren'; //#12215
        $k[$j++] = $0.renlinear; //#12215
        $k[$j++] = 'sbs'; //#12215
        $k[$j++] = _6p; //#12215
        $k[$j++] = 'bhs'; //#12215
        $k[$j++] = _6u; //#12215
        $k[$j++] = 'bbs'; //#12215
        $k[$j++] = _6y; //#12215
        $k[$j++] = 'txt'; //#12215
        $k[$j++] = $a([$a([$1.text, $1.textxoffset, $1.textyoffset, $1.textfont, $1.textsize])]); //#12215
        $k[$j++] = 'textxalign'; //#12215
        $k[$j++] = "center"; //#12215
        $k[$j++] = 'opt'; //#12215
        $k[$j++] = $1.options; //#12215
        var _77 = $d(); //#12215
        $k[$j++] = _77; //#12218
        if (!$1.dontdraw) { //#12218
            $0.renlinear(); //#12218
        } //#12218
    };
    $0.codablockf = function() {
        var $1 = {}; //#12248
        $1.options = $k[--$j]; //#12250
        $1.barcode = $k[--$j]; //#12251
        $1.dontdraw = false; //#12253
        $1.rows = -1; //#12254
        $1.columns = 8; //#12255
        $1.rowheight = 10; //#12256
        $1.sepheight = 1; //#12257
        $1.parse = false; //#12258
        $1.parsefnc = false; //#12259
        $forall($1.options, function() { //#12270
            var _3 = $k[--$j]; //#12270
            $1[$k[--$j]] = _3; //#12270
        }); //#12270
        $1.rows = ~~$1.rows; //#12272
        $1.columns = ~~$1.columns; //#12273
        $1.rowheight = ~~$1.rowheight; //#12274
        $1.sepheight = ~~$1.sepheight; //#12275
        $k[$j++] = 'c'; //#12277
        if (($1.columns >= 4) && ($1.columns <= 62)) { //#12277
            $k[$j++] = $1.columns; //#12277
        } else { //#12277
            $k[$j++] = 8; //#12277
        } //#12277
        var _C = $k[--$j]; //#12277
        $1[$k[--$j]] = _C; //#12277
        $k[$j++] = 'rows'; //#12278
        if (($1.rows >= 2) && ($1.rows <= 44)) { //#12278
            $k[$j++] = $1.rows; //#12278
        } else { //#12278
            $k[$j++] = -1; //#12278
        } //#12278
        var _H = $k[--$j]; //#12278
        $1[$k[--$j]] = _H; //#12278
        $1.swa = -1; //#12281
        $1.swb = -2; //#12281
        $1.swc = -3; //#12281
        $1.sft = -4; //#12281
        $1.fn1 = -5; //#12282
        $1.fn2 = -6; //#12282
        $1.fn3 = -7; //#12282
        $1.fn4 = -8; //#12282
        $1.sta = -9; //#12283
        $1.stp = -10; //#12283
        var _N = new Map([
            ["parse", $1.parse],
            ["parsefnc", $1.parsefnc],
            ["FNC1", $1.fn1],
            ["FNC3", $1.fn3]
        ]); //#12291
        $1.fncvals = _N; //#12293
        $k[$j++] = 'msg'; //#12294
        $k[$j++] = $1.barcode; //#12294
        $k[$j++] = $1.fncvals; //#12294
        $0.parseinput(); //#12294
        var _Q = $k[--$j]; //#12294
        $1[$k[--$j]] = _Q; //#12294
        $1.msglen = $1.msg.length; //#12295
        $1.msgtmp = $a([]); //#12298
        var _U = $1.msg; //#12299
        for (var _V = 0, _W = _U.length; _V < _W; _V++) { //#12302
            $1.char = $get(_U, _V); //#12300
            $k[$j++] = Infinity; //#12301
            $aload($1.msgtmp); //#12301
            if ($1.char < 128) { //#12301
                $k[$j++] = $1.char; //#12301
            } else { //#12301
                $k[$j++] = $1.fn4; //#12301
                $k[$j++] = $1.char & 127; //#12301
            } //#12301
            $1.msgtmp = $a(); //#12301
        } //#12301
        $1.msg = $1.msgtmp; //#12303
        $1.msglen = $1.msg.length; //#12304
        $1.charmaps = $a([$a([32, 32, "00"]), $a(["!", "!", "01"]), $a(["\"", "\"", "02"]), $a(["#", "#", "03"]), $a(["$", "$", "04"]), $a(["%", "%", "05"]), $a(["&", "&", "06"]), $a(["'", "'", "07"]), $a([40, 40, "08"]), $a([41, 41, "09"]), $a(["*", "*", "10"]), $a(["+", "+", "11"]), $a([",", ",", "12"]), $a(["-", "-", "13"]), $a([".", ".", "14"]), $a(["/", "/", "15"]), $a(["0", "0", "16"]), $a(["1", "1", "17"]), $a(["2", "2", "18"]), $a(["3", "3", "19"]), $a(["4", "4", "20"]), $a(["5", "5", "21"]), $a(["6", "6", "22"]), $a(["7", "7", "23"]), $a(["8", "8", "24"]), $a(["9", "9", "25"]), $a([":", ":", "26"]), $a([";", ";", "27"]), $a(["<", "<", "28"]), $a(["=", "=", "29"]), $a([">", ">", "30"]), $a(["?", "?", "31"]), $a(["@", "@", "32"]), $a(["A", "A", "33"]), $a(["B", "B", "34"]), $a(["C", "C", "35"]), $a(["D", "D", "36"]), $a(["E", "E", "37"]), $a(["F", "F", "38"]), $a(["G", "G", "39"]), $a(["H", "H", "40"]), $a(["I", "I", "41"]), $a(["J", "J", "42"]), $a(["K", "K", "43"]), $a(["L", "L", "44"]), $a(["M", "M", "45"]), $a(["N", "N", "46"]), $a(["O", "O", "47"]), $a(["P", "P", "48"]), $a(["Q", "Q", "49"]), $a(["R", "R", "50"]), $a(["S", "S", "51"]), $a(["T", "T", "52"]), $a(["U", "U", "53"]), $a(["V", "V", "54"]), $a(["W", "W", "55"]), $a(["X", "X", "56"]), $a(["Y", "Y", "57"]), $a(["Z", "Z", "58"]), $a(["[", "[", "59"]), $a([92, 92, "60"]), $a(["]", "]", "61"]), $a(["^", "^", "62"]), $a(["_", "_", "63"]), $a([0, "`", "64"]), $a([1, "a", "65"]), $a([2, "b", "66"]), $a([3, "c", "67"]), $a([4, "d", "68"]), $a([5, "e", "69"]), $a([6, "f", "70"]), $a([7, "g", "71"]), $a([8, "h", "72"]), $a([9, "i", "73"]), $a([10, "j", "74"]), $a([11, "k", "75"]), $a([12, "l", "76"]), $a([13, "m", "77"]), $a([14, "n", "78"]), $a([15, "o", "79"]), $a([16, "p", "80"]), $a([17, "q", "81"]), $a([18, "r", "82"]), $a([19, "s", "83"]), $a([20, "t", "84"]), $a([21, "u", "85"]), $a([22, "v", "86"]), $a([23, "w", "87"]), $a([24, "x", "88"]), $a([25, "y", "89"]), $a([26, "z", "90"]), $a([27, "{", "91"]), $a([28, "|", "92"]), $a([29, "}", "93"]), $a([30, "~", "94"]), $a([31, 127, "95"]), $a([$1.fn3, $1.fn3, "96"]), $a([$1.fn2, $1.fn2, "97"]), $a([$1.sft, $1.sft, "98"]), $a([$1.swc, $1.swc, "99"]), $a([$1.swb, $1.fn4, $1.swb]), $a([$1.fn4, $1.swa, $1.swa]), $a([$1.fn1, $1.fn1, $1.fn1]), $a([$1.sta, $1.sta, $1.sta]), $a([$1.stp, $1.stp, $1.stp])]); //#12344
        $1.charvals = $a([new Map, new Map, new Map]); //#12347
        for (var _2o = 0, _2n = $1.charmaps.length - 1; _2o <= _2n; _2o += 1) { //#12356
            $1.i = _2o; //#12349
            $1.encs = $get($1.charmaps, $1.i); //#12350
            for (var _2s = 0; _2s <= 2; _2s += 1) { //#12355
                $1.j = _2s; //#12352
                var _2v = $get($1.encs, $1.j); //#12353
                $k[$j++] = _2v; //#12353
                if ($eq($type(_2v), 'stringtype')) { //#12353
                    var _2y = $get($k[--$j], 0); //#12353
                    $k[$j++] = _2y; //#12353
                } //#12353
                $put($get($1.charvals, $1.j), $k[--$j], $1.i); //#12354
            } //#12354
        } //#12354
        $1.seta = $get($1.charvals, 0); //#12357
        $1.setb = $get($1.charvals, 1); //#12358
        $1.setc = $get($1.charvals, 2); //#12359
        $1.numsscr = function() {
            $1.n = 0; //#12363
            $1.s = 0; //#12363
            $1.p = $k[--$j]; //#12364
            for (;;) { //#12375
                if ($1.p >= $1.msglen) { //#12365
                    break; //#12365
                } //#12365
                var _3F = $get($1.msg, $1.p); //#12366
                var _3H = $get($1.setc, _3F) !== undefined; //#12367
                $k[$j++] = _3F; //#12367
                if (!_3H) { //#12367
                    $j--; //#12367
                    break; //#12367
                } //#12367
                if ($k[--$j] == $1.fn1) { //#12371
                    if (($1.s % 2) == 0) { //#12370
                        $1.s = $1.s + 1; //#12370
                    } else { //#12370
                        break; //#12370
                    } //#12370
                } //#12370
                $1.n = $1.n + 1; //#12372
                $1.s = $1.s + 1; //#12373
                $1.p = $f($1.p + 1); //#12374
            } //#12374
            $k[$j++] = $1.n; //#12376
            $k[$j++] = $1.s; //#12376
        }; //#12377
        $1.enca = function() {
            $put($1.cws, $1.j, $get($1.seta, $k[--$j])); //#12381
            $1.j = $1.j + 1; //#12382
        }; //#12383
        $1.encb = function() {
            $put($1.cws, $1.j, $get($1.setb, $k[--$j])); //#12385
            $1.j = $1.j + 1; //#12386
        }; //#12387
        $1.encc = function() {
            var _3d = $k[--$j]; //#12389
            $k[$j++] = _3d; //#12393
            if ($ne($type(_3d), 'arraytype')) { //#12392
                var _3h = $get($1.setc, $k[--$j]); //#12390
                $k[$j++] = _3h; //#12390
            } else { //#12392
                $aload($k[--$j]); //#12392
                var _3j = $k[--$j]; //#12392
                var _3k = $k[--$j]; //#12392
                $k[$j++] = $f($f(_3j - 48) + ($f(_3k - 48) * 10)); //#12392
            } //#12392
            $put($1.cws, $1.j, $k[--$j]); //#12394
            $1.j = $1.j + 1; //#12395
        }; //#12396
        $1.anotb = function() {
            var _3p = $k[--$j]; //#12399
            var _3r = $get($1.seta, _3p) !== undefined; //#12399
            var _3t = $get($1.setb, _3p) !== undefined; //#12399
            $k[$j++] = _3r && (!_3t); //#12399
        }; //#12399
        $1.bnota = function() {
            var _3u = $k[--$j]; //#12400
            var _3w = $get($1.setb, _3u) !== undefined; //#12400
            var _3y = $get($1.seta, _3u) !== undefined; //#12400
            $k[$j++] = _3w && (!_3y); //#12400
        }; //#12400
        $k[$j++] = Infinity; //#12403
        for (var _40 = 0, _41 = $1.msg.length; _40 < _41; _40++) { //#12403
            $k[$j++] = 0; //#12403
        } //#12403
        $k[$j++] = 9999; //#12403
        $1.nextanotb = $a(); //#12403
        $k[$j++] = Infinity; //#12404
        for (var _44 = 0, _45 = $1.msg.length; _44 < _45; _44++) { //#12404
            $k[$j++] = 0; //#12404
        } //#12404
        $k[$j++] = 9999; //#12404
        $1.nextbnota = $a(); //#12404
        for (var _48 = $1.msg.length - 1; _48 >= 0; _48 -= 1) { //#12417
            $1.i = _48; //#12406
            $k[$j++] = $get($1.msg, $1.i); //#12407
            $1.anotb(); //#12407
            if ($k[--$j]) { //#12410
                $put($1.nextanotb, $1.i, 0); //#12408
            } else { //#12410
                $put($1.nextanotb, $1.i, $f($get($1.nextanotb, $1.i + 1) + 1)); //#12410
            } //#12410
            $k[$j++] = $get($1.msg, $1.i); //#12412
            $1.bnota(); //#12412
            if ($k[--$j]) { //#12415
                $put($1.nextbnota, $1.i, 0); //#12413
            } else { //#12415
                $put($1.nextbnota, $1.i, $f($get($1.nextbnota, $1.i + 1) + 1)); //#12415
            } //#12415
        } //#12415
        $1.abeforeb = function() {
            var _4V = $k[--$j]; //#12420
            $k[$j++] = $lt($get($1.nextanotb, _4V), $get($1.nextbnota, _4V)); //#12420
        }; //#12420
        $1.bbeforea = function() {
            var _4a = $k[--$j]; //#12421
            $k[$j++] = $lt($get($1.nextbnota, _4a), $get($1.nextanotb, _4a)); //#12421
        }; //#12421
        $1.padrow = function() {
            for (var _4g = 0, _4h = $k[--$j]; _4g < _4h; _4g++) { //#12431
                for (;;) { //#12430
                    if ($eq($1.cset, "seta")) { //#12427
                        $k[$j++] = $1.swc; //#12427
                        $1.enca(); //#12427
                        $1.cset = "setc"; //#12427
                        break; //#12427
                    } //#12427
                    if ($eq($1.cset, "setb")) { //#12428
                        $k[$j++] = $1.swc; //#12428
                        $1.encb(); //#12428
                        $1.cset = "setc"; //#12428
                        break; //#12428
                    } //#12428
                    if ($eq($1.cset, "setc")) { //#12429
                        $k[$j++] = $1.swb; //#12429
                        $1.encc(); //#12429
                        $1.cset = "setb"; //#12429
                        break; //#12429
                    } //#12429
                } //#12429
            } //#12429
        }; //#12432
        $1.cws = $a($f($1.c + 5) * 44); //#12435
        $1.i = 0; //#12437
        $1.j = 0; //#12437
        $1.r = 1; //#12437
        $1.lastrow = false; //#12438
        for (;;) { //#12607
            if ($1.lastrow) { //#12440
                break; //#12440
            } //#12440
            $k[$j++] = $1.sta; //#12443
            $1.enca(); //#12443
            if ($1.i < $1.msglen) { //#12444
                $k[$j++] = $1.i; //#12444
                $1.numsscr(); //#12444
            } else { //#12444
                $k[$j++] = -1; //#12444
                $k[$j++] = -1; //#12444
            } //#12444
            $1.nums = $k[--$j]; //#12444
            $1.nchars = $k[--$j]; //#12444
            for (;;) { //#12464
                if ($1.msglen == 0) { //#12450
                    $k[$j++] = $1.swb; //#12447
                    $1.enca(); //#12447
                    $1.cset = "setb"; //#12448
                    break; //#12449
                } //#12449
                if ($1.nums >= 2) { //#12455
                    $k[$j++] = $1.swc; //#12452
                    $1.enca(); //#12452
                    $1.cset = "setc"; //#12453
                    break; //#12454
                } //#12454
                $k[$j++] = $1.i; //#12456
                $1.abeforeb(); //#12456
                if ($k[--$j]) { //#12460
                    $k[$j++] = $1.sft; //#12457
                    $1.enca(); //#12457
                    $1.cset = "seta"; //#12458
                    break; //#12459
                } //#12459
                $k[$j++] = $1.swb; //#12461
                $1.enca(); //#12461
                $1.cset = "setb"; //#12462
                break; //#12463
            } //#12463
            $1.j = $1.j + 1; //#12465
            $1.endofrow = false; //#12468
            for (;;) { //#12592
                $1.rem = $f($f($1.c + 3) - ($1.j % $f($1.c + 5))); //#12469
                if (($1.i == $1.msglen) || $1.endofrow) { //#12471
                    break; //#12471
                } //#12471
                $k[$j++] = $1.i; //#12473
                $1.numsscr(); //#12473
                $1.nums = $k[--$j]; //#12473
                $1.nchars = $k[--$j]; //#12473
                $k[$j++] = 'remnums'; //#12474
                if ($1.nums > ($1.rem * 2)) { //#12474
                    $k[$j++] = $1.rem * 2; //#12474
                } else { //#12474
                    $k[$j++] = $1.nums; //#12474
                } //#12474
                var _5J = $k[--$j]; //#12474
                $1[$k[--$j]] = _5J; //#12474
                for (;;) { //#12590
                    if ((($eq($1.cset, "seta") || $eq($1.cset, "setb")) && ($1.remnums >= 4)) && ($get($1.msg, $1.i) != $1.fn1)) { //#12512
                        if ((($1.remnums % 2) == 0) && ($1.rem >= 3)) { //#12494
                            $k[$j++] = $1.swc; //#12482
                            if ($eq($1.cset, "seta")) { //#12482
                                $1.enca(); //#12482
                            } else { //#12482
                                $1.encb(); //#12482
                            } //#12482
                            $1.cset = "setc"; //#12483
                            for (var _5W = 0, _5X = 2; _5W < _5X; _5W++) { //#12492
                                if ($get($1.msg, $1.i) == $1.fn1) { //#12490
                                    $k[$j++] = $1.fn1; //#12486
                                    $1.encc(); //#12486
                                    $1.i = $1.i + 1; //#12487
                                } else { //#12490
                                    $k[$j++] = $geti($1.msg, $1.i, 2); //#12489
                                    $1.encc(); //#12489
                                    $1.i = $1.i + 2; //#12490
                                } //#12490
                            } //#12490
                            break; //#12493
                        } //#12493
                        if ((($1.remnums % 2) != 0) && ($1.rem >= 4)) { //#12511
                            $k[$j++] = $get($1.msg, $1.i); //#12497
                            if ($eq($1.cset, "seta")) { //#12497
                                $1.enca(); //#12497
                            } else { //#12497
                                $1.encb(); //#12497
                            } //#12497
                            $1.i = $1.i + 1; //#12498
                            $k[$j++] = $1.swc; //#12499
                            if ($eq($1.cset, "seta")) { //#12499
                                $1.enca(); //#12499
                            } else { //#12499
                                $1.encb(); //#12499
                            } //#12499
                            $1.cset = "setc"; //#12500
                            for (var _5r = 0, _5s = 2; _5r < _5s; _5r++) { //#12509
                                if ($get($1.msg, $1.i) == $1.fn1) { //#12507
                                    $k[$j++] = $1.fn1; //#12503
                                    $1.encc(); //#12503
                                    $1.i = $1.i + 1; //#12504
                                } else { //#12507
                                    $k[$j++] = $geti($1.msg, $1.i, 2); //#12506
                                    $1.encc(); //#12506
                                    $1.i = $1.i + 2; //#12507
                                } //#12507
                            } //#12507
                            break; //#12510
                        } //#12510
                    } //#12510
                    $k[$j++] = $eq($1.cset, "setb"); //#12513
                    $k[$j++] = $get($1.msg, $1.i); //#12513
                    $1.anotb(); //#12513
                    var _67 = $k[--$j]; //#12513
                    var _68 = $k[--$j]; //#12513
                    if ((_68 && _67) && ($1.rem >= 2)) { //#12528
                        if ($1.i < ($1.msglen - 1)) { //#12522
                            $k[$j++] = $1.i + 1; //#12516
                            $1.bbeforea(); //#12516
                            if ($k[--$j]) { //#12521
                                $k[$j++] = $1.sft; //#12517
                                $1.encb(); //#12517
                                $k[$j++] = $get($1.msg, $1.i); //#12518
                                $1.enca(); //#12518
                                $1.i = $1.i + 1; //#12519
                                break; //#12520
                            } //#12520
                        } //#12520
                        $k[$j++] = $1.swa; //#12523
                        $1.encb(); //#12523
                        $1.cset = "seta"; //#12524
                        $k[$j++] = $get($1.msg, $1.i); //#12525
                        $1.enca(); //#12525
                        $1.i = $1.i + 1; //#12526
                        break; //#12527
                    } //#12527
                    $k[$j++] = $eq($1.cset, "seta"); //#12529
                    $k[$j++] = $get($1.msg, $1.i); //#12529
                    $1.bnota(); //#12529
                    var _6S = $k[--$j]; //#12529
                    var _6T = $k[--$j]; //#12529
                    if ((_6T && _6S) && ($1.rem >= 2)) { //#12544
                        if ($1.i < ($1.msglen - 1)) { //#12538
                            $k[$j++] = $1.i + 1; //#12532
                            $1.abeforeb(); //#12532
                            if ($k[--$j]) { //#12537
                                $k[$j++] = $1.sft; //#12533
                                $1.enca(); //#12533
                                $k[$j++] = $get($1.msg, $1.i); //#12534
                                $1.encb(); //#12534
                                $1.i = $1.i + 1; //#12535
                                break; //#12536
                            } //#12536
                        } //#12536
                        $k[$j++] = $1.swb; //#12539
                        $1.enca(); //#12539
                        $1.cset = "setb"; //#12540
                        $k[$j++] = $get($1.msg, $1.i); //#12541
                        $1.encb(); //#12541
                        $1.i = $1.i + 1; //#12542
                        break; //#12543
                    } //#12543
                    if (($eq($1.cset, "setc") && ($1.remnums < 2)) && ($1.rem >= 2)) { //#12559
                        $k[$j++] = $1.i; //#12547
                        $1.abeforeb(); //#12547
                        if ($k[--$j]) { //#12553
                            $k[$j++] = $1.swa; //#12548
                            $1.encc(); //#12548
                            $1.cset = "seta"; //#12549
                            $k[$j++] = $get($1.msg, $1.i); //#12550
                            $1.enca(); //#12550
                            $1.i = $1.i + 1; //#12551
                            break; //#12552
                        } //#12552
                        $k[$j++] = $1.swb; //#12554
                        $1.encc(); //#12554
                        $1.cset = "setb"; //#12555
                        $k[$j++] = $get($1.msg, $1.i); //#12556
                        $1.encb(); //#12556
                        $1.i = $1.i + 1; //#12557
                        break; //#12558
                    } //#12558
                    var _73 = $get($1.seta, $get($1.msg, $1.i)) !== undefined; //#12562
                    if (($eq($1.cset, "seta") && _73) && ($1.rem >= 1)) { //#12567
                        $k[$j++] = $get($1.msg, $1.i); //#12564
                        $1.enca(); //#12564
                        $1.i = $1.i + 1; //#12565
                        break; //#12566
                    } //#12566
                    var _7E = $get($1.setb, $get($1.msg, $1.i)) !== undefined; //#12568
                    if (($eq($1.cset, "setb") && _7E) && ($1.rem >= 1)) { //#12573
                        $k[$j++] = $get($1.msg, $1.i); //#12570
                        $1.encb(); //#12570
                        $1.i = $1.i + 1; //#12571
                        break; //#12572
                    } //#12572
                    if (($eq($1.cset, "setc") && ($1.remnums >= 2)) && ($1.rem >= 1)) { //#12584
                        if ($get($1.msg, $1.i) == $1.fn1) { //#12581
                            $k[$j++] = $1.fn1; //#12577
                            $1.encc(); //#12577
                            $1.i = $1.i + 1; //#12578
                        } else { //#12581
                            $k[$j++] = $geti($1.msg, $1.i, 2); //#12580
                            $1.encc(); //#12580
                            $1.i = $1.i + 2; //#12581
                        } //#12581
                        break; //#12583
                    } //#12583
                    $1.endofrow = true; //#12587
                    break; //#12588
                } //#12588
            } //#12588
            if ((((($1.r >= $1.rows) || ($1.rows == -1)) && ($1.r > 1)) && ($1.i == $1.msglen)) && ($1.rem >= 2)) { //#12604
                $k[$j++] = $f($1.rem - 2); //#12596
                $1.padrow(); //#12596
                $1.j = $1.j + 3; //#12597
                $k[$j++] = $1.stp; //#12598
                $1.enca(); //#12598
                $1.lastrow = true; //#12599
            } else { //#12604
                $k[$j++] = $1.rem; //#12601
                $1.padrow(); //#12601
                $1.j = $1.j + 1; //#12602
                $k[$j++] = $1.stp; //#12603
                $1.enca(); //#12603
                $1.r = $1.r + 1; //#12604
            } //#12604
        } //#12604
        $1.cws = $geti($1.cws, 0, $1.j); //#12608
        $k[$j++] = Infinity; //#12614
        for (var _7o = 64; _7o <= 95; _7o += 1) { //#12612
            $k[$j++] = _7o; //#12612
        } //#12612
        for (var _7p = 0; _7p <= 15; _7p += 1) { //#12613
            $k[$j++] = _7p; //#12613
        } //#12613
        for (var _7q = 26; _7q <= 63; _7q += 1) { //#12614
            $k[$j++] = _7q; //#12614
        } //#12614
        $1.abmap = $a(); //#12614
        $k[$j++] = Infinity; //#12616
        for (var _7s = 0; _7s <= 85; _7s += 1) { //#12616
            $k[$j++] = _7s; //#12616
        } //#12616
        $1.cmap = $a(); //#12616
        $1.chkmsg = $a($1.msglen); //#12619
        $1.j = 0; //#12620
        for (var _7y = 0, _7x = $1.msglen - 1; _7y <= _7x; _7y += 1) { //#12626
            $1.i = _7y; //#12622
            $1.char = $get($1.msg, $1.i); //#12623
            if ($1.char >= 0) { //#12624
                $put($1.chkmsg, $1.j, $1.char); //#12624
                $1.j = $1.j + 1; //#12624
            } //#12624
            if (($1.char == $1.fn1) && ($1.i != 0)) { //#12625
                $put($1.chkmsg, $1.j, 29); //#12625
                $1.j = $1.j + 1; //#12625
            } //#12625
        } //#12625
        $1.t1 = 0; //#12627
        $1.t2 = 0; //#12627
        $1.k1 = 0; //#12627
        $1.k2 = 0; //#12627
        for (var _8F = 0, _8E = $1.j - 1; _8F <= _8E; _8F += 1) { //#12634
            $1.i = _8F; //#12629
            $1.t1 = ($get($1.chkmsg, $1.i) * $1.i) % 86; //#12630
            $1.t2 = $f($1.t1 + $get($1.chkmsg, $1.i)) % 86; //#12631
            $1.k1 = $f($1.k1 + $1.t2) % 86; //#12632
            $1.k2 = $f($1.k2 + $1.t1) % 86; //#12633
        } //#12633
        $k[$j++] = $1.cws; //#12635
        $k[$j++] = $1.cws.length - 4; //#12635
        if ($ne($1.cset, "setc")) { //#12635
            $k[$j++] = $1.abmap; //#12635
        } else { //#12635
            $k[$j++] = $1.cmap; //#12635
        } //#12635
        var _8Z = $get($k[--$j], $1.k1); //#12635
        var _8a = $k[--$j]; //#12635
        $put($k[--$j], _8a, _8Z); //#12635
        $k[$j++] = $1.cws; //#12636
        $k[$j++] = $1.cws.length - 3; //#12636
        if ($ne($1.cset, "setc")) { //#12636
            $k[$j++] = $1.abmap; //#12636
        } else { //#12636
            $k[$j++] = $1.cmap; //#12636
        } //#12636
        var _8j = $get($k[--$j], $1.k2); //#12636
        var _8k = $k[--$j]; //#12636
        $put($k[--$j], _8k, _8j); //#12636
        var _8m = $1.cws; //#12639
        $k[$j++] = _8m; //#12639
        $k[$j++] = 2; //#12639
        if ($get(_8m, 2 - 1) != 99) { //#12639
            $k[$j++] = $1.abmap; //#12639
        } else { //#12639
            $k[$j++] = $1.cmap; //#12639
        } //#12639
        var _8s = $get($k[--$j], $1.r - 2); //#12639
        var _8t = $k[--$j]; //#12639
        $put($k[--$j], _8t, _8s); //#12639
        for (var _8x = 1, _8w = $1.r - 1; _8x <= _8w; _8x += 1) { //#12643
            $1.i = _8x; //#12641
            var _8y = $1.cws; //#12642
            var _8z = $1.i; //#12642
            var _90 = $1.c; //#12642
            $k[$j++] = _8y; //#12642
            $k[$j++] = $f((_8z * $f(_90 + 5)) + 2); //#12642
            if ($get(_8y, $f(($f((_8z * $f(_90 + 5)) + 2)) - 1)) != 99) { //#12642
                $k[$j++] = $1.abmap; //#12642
            } else { //#12642
                $k[$j++] = $1.cmap; //#12642
            } //#12642
            var _96 = $get($k[--$j], $1.i + 42); //#12642
            var _97 = $k[--$j]; //#12642
            $put($k[--$j], _97, _96); //#12642
        } //#12642
        for (var _9B = 0, _9A = $1.r - 1; _9B <= _9A; _9B += 1) { //#12654
            $1.rcws = $geti($1.cws, _9B * $f($1.c + 5), $f($1.c + 4)); //#12647
            $1.csum = $get($1.rcws, 0); //#12648
            for (var _9K = 1, _9J = $1.rcws.length - 2; _9K <= _9J; _9K += 1) { //#12652
                $1.i = _9K; //#12650
                $1.csum = $f($1.csum + ($get($1.rcws, $1.i) * $1.i)); //#12651
            } //#12651
            $put($1.rcws, $1.rcws.length - 1, $1.csum % 103); //#12653
        } //#12653
        $1.encs = $a(["212222", "222122", "222221", "121223", "121322", "131222", "122213", "122312", "132212", "221213", "221312", "231212", "112232", "122132", "122231", "113222", "123122", "123221", "223211", "221132", "221231", "213212", "223112", "312131", "311222", "321122", "321221", "312212", "322112", "322211", "212123", "212321", "232121", "111323", "131123", "131321", "112313", "132113", "132311", "211313", "231113", "231311", "112133", "112331", "132131", "113123", "113321", "133121", "313121", "211331", "231131", "213113", "213311", "213131", "311123", "311321", "331121", "312113", "312311", "332111", "314111", "221411", "431111", "111224", "111422", "121124", "121421", "141122", "141221", "112214", "112412", "122114", "122411", "142112", "142211", "241211", "221114", "413111", "241112", "134111", "111242", "121142", "121241", "114212", "124112", "124211", "411212", "421112", "421211", "212141", "214121", "412121", "111143", "111341", "131141", "114113", "114311", "411113", "411311", "113141", "114131", "311141", "411131", "211412", "2331112"]); //#12673
        $1.rowbits = $a($1.r); //#12676
        for (var _9Y = 0, _9X = $1.r - 1; _9Y <= _9X; _9Y += 1) { //#12688
            $1.i = _9Y; //#12678
            $k[$j++] = Infinity; //#12681
            var _9d = $geti($1.cws, $1.i * $f($1.c + 5), $f($1.c + 5)); //#12680
            for (var _9e = 0, _9f = _9d.length; _9e < _9f; _9e++) { //#12682
                $forall($get($1.encs, $get(_9d, _9e)), function() { //#12681
                    var _9j = $k[--$j]; //#12681
                    $k[$j++] = $f(_9j - 48); //#12681
                }); //#12681
            } //#12681
            $1.sbs = $a(); //#12681
            $k[$j++] = Infinity; //#12684
            var _9l = $1.sbs; //#12685
            $k[$j++] = 0; //#12685
            for (var _9m = 0, _9n = _9l.length; _9m < _9n; _9m++) { //#12685
                var _9p = $k[--$j]; //#12685
                var _9q = (_9p == 1) ? 0 : 1; //#12685
                $k[$j++] = _9p; //#12685
                for (var _9r = 0, _9s = $get(_9l, _9m); _9r < _9s; _9r++) { //#12685
                    $k[$j++] = _9q //#12685
                } //#12685
            } //#12685
            $astore($a($counttomark() - 1)); //#12686
            var _9v = $k[--$j]; //#12686
            var _9w = $k[--$j]; //#12686
            $k[$j++] = _9v; //#12686
            $k[$j++] = _9w; //#12686
            $j--; //#12686
            var _9x = $k[--$j]; //#12686
            var _9y = $k[--$j]; //#12686
            $k[$j++] = _9x; //#12686
            $k[$j++] = _9y; //#12686
            $j--; //#12686
            $put($1.rowbits, $1.i, $k[--$j]); //#12687
        } //#12687
        $1.symwid = $f(($1.c * 11) + 57); //#12691
        $k[$j++] = Infinity; //#12704
        for (var _A5 = 0, _A6 = $1.symwid * $1.sepheight; _A5 < _A6; _A5++) { //#12693
            $k[$j++] = 1; //#12693
        } //#12693
        for (var _A9 = 0, _A8 = $1.r - 2; _A9 <= _A8; _A9 += 1) { //#12702
            $1.i = _A9; //#12695
            for (var _AB = 0, _AC = $1.rowheight; _AB < _AC; _AB++) { //#12696
                $aload($get($1.rowbits, $1.i)); //#12696
            } //#12696
            for (var _AH = 0, _AI = $1.sepheight; _AH < _AI; _AH++) { //#12701
                $k[$j++] = 1; //#12699
                $k[$j++] = 1; //#12699
                $k[$j++] = 0; //#12699
                $k[$j++] = 1; //#12699
                $k[$j++] = 0; //#12699
                $k[$j++] = 0; //#12699
                $k[$j++] = 0; //#12699
                $k[$j++] = 0; //#12699
                $k[$j++] = 1; //#12699
                $k[$j++] = 0; //#12699
                $k[$j++] = 0; //#12699
                for (var _AK = 0, _AL = $f($1.symwid - 24); _AK < _AL; _AK++) { //#12699
                    $k[$j++] = 1; //#12699
                } //#12699
                $k[$j++] = 1; //#12700
                $k[$j++] = 1; //#12700
                $k[$j++] = 0; //#12700
                $k[$j++] = 0; //#12700
                $k[$j++] = 0; //#12700
                $k[$j++] = 1; //#12700
                $k[$j++] = 1; //#12700
                $k[$j++] = 1; //#12700
                $k[$j++] = 0; //#12700
                $k[$j++] = 1; //#12700
                $k[$j++] = 0; //#12700
                $k[$j++] = 1; //#12700
                $k[$j++] = 1; //#12700
            } //#12700
        } //#12700
        for (var _AN = 0, _AO = $1.rowheight; _AN < _AO; _AN++) { //#12703
            $aload($get($1.rowbits, $1.r - 1)); //#12703
        } //#12703
        for (var _AU = 0, _AV = $1.symwid * $1.sepheight; _AU < _AV; _AU++) { //#12704
            $k[$j++] = 1; //#12704
        } //#12704
        $1.pixs = $a(); //#12704
        var _Af = new Map([
            ["ren", $0.renmatrix],
            ["pixs", $1.pixs],
            ["pixx", $1.symwid],
            ["pixy", ~~($1.pixs.length / $1.symwid)],
            ["height", (~~($1.pixs.length / $1.symwid)) / 72],
            ["width", $1.symwid / 72],
            ["opt", $1.options]
        ]); //#12715
        $k[$j++] = _Af; //#12718
        if (!$1.dontdraw) { //#12718
            $0.renmatrix(); //#12718
        } //#12718
    };
    $0.code16k = function() {
        var $1 = {}; //#12748
        $1.options = $k[--$j]; //#12750
        $1.barcode = $k[--$j]; //#12751
        $1.dontdraw = false; //#12753
        $1.mode = -1; //#12754
        $1.pos = -1; //#12755
        $1.rows = 0; //#12756
        $1.rowheight = 8; //#12757
        $1.sepheight = 1; //#12758
        $1.encoding = "auto"; //#12759
        $1.raw = false; //#12760
        $1.parse = false; //#12761
        $1.parsefnc = false; //#12762
        $forall($1.options, function() { //#12773
            var _3 = $k[--$j]; //#12773
            $1[$k[--$j]] = _3; //#12773
        }); //#12773
        $1.mode = ~~$1.mode; //#12775
        $1.pos = ~~$1.pos; //#12776
        $1.rows = ~~$1.rows; //#12777
        $1.rowheight = ~~$1.rowheight; //#12778
        $1.sepheight = ~~$1.sepheight; //#12779
        if ($1.pos != -1) { //#12781
            $1.rows = 16; //#12781
        } //#12781
        $1.swa = -1; //#12784
        $1.swb = -2; //#12784
        $1.swc = -3; //#12784
        $1.sa1 = -4; //#12785
        $1.sb1 = -5; //#12785
        $1.sc1 = -6; //#12785
        $1.sa2 = -7; //#12786
        $1.sb2 = -8; //#12786
        $1.sc2 = -9; //#12786
        $1.pad = -10; //#12787
        $1.sb3 = -11; //#12787
        $1.sc3 = -12; //#12787
        $1.fn1 = -13; //#12788
        $1.fn2 = -14; //#12788
        $1.fn3 = -15; //#12788
        $1.fn4 = -16; //#12788
        $1.charmaps = $a([$a([32, 32, "00"]), $a(["!", "!", "01"]), $a(["\"", "\"", "02"]), $a(["#", "#", "03"]), $a(["$", "$", "04"]), $a(["%", "%", "05"]), $a(["&", "&", "06"]), $a(["'", "'", "07"]), $a([40, 40, "08"]), $a([41, 41, "09"]), $a(["*", "*", "10"]), $a(["+", "+", "11"]), $a([",", ",", "12"]), $a(["-", "-", "13"]), $a([".", ".", "14"]), $a(["/", "/", "15"]), $a(["0", "0", "16"]), $a(["1", "1", "17"]), $a(["2", "2", "18"]), $a(["3", "3", "19"]), $a(["4", "4", "20"]), $a(["5", "5", "21"]), $a(["6", "6", "22"]), $a(["7", "7", "23"]), $a(["8", "8", "24"]), $a(["9", "9", "25"]), $a([":", ":", "26"]), $a([";", ";", "27"]), $a(["<", "<", "28"]), $a(["=", "=", "29"]), $a([">", ">", "30"]), $a(["?", "?", "31"]), $a(["@", "@", "32"]), $a(["A", "A", "33"]), $a(["B", "B", "34"]), $a(["C", "C", "35"]), $a(["D", "D", "36"]), $a(["E", "E", "37"]), $a(["F", "F", "38"]), $a(["G", "G", "39"]), $a(["H", "H", "40"]), $a(["I", "I", "41"]), $a(["J", "J", "42"]), $a(["K", "K", "43"]), $a(["L", "L", "44"]), $a(["M", "M", "45"]), $a(["N", "N", "46"]), $a(["O", "O", "47"]), $a(["P", "P", "48"]), $a(["Q", "Q", "49"]), $a(["R", "R", "50"]), $a(["S", "S", "51"]), $a(["T", "T", "52"]), $a(["U", "U", "53"]), $a(["V", "V", "54"]), $a(["W", "W", "55"]), $a(["X", "X", "56"]), $a(["Y", "Y", "57"]), $a(["Z", "Z", "58"]), $a(["[", "[", "59"]), $a([92, 92, "60"]), $a(["]", "]", "61"]), $a(["^", "^", "62"]), $a(["_", "_", "63"]), $a([0, "`", "64"]), $a([1, "a", "65"]), $a([2, "b", "66"]), $a([3, "c", "67"]), $a([4, "d", "68"]), $a([5, "e", "69"]), $a([6, "f", "70"]), $a([7, "g", "71"]), $a([8, "h", "72"]), $a([9, "i", "73"]), $a([10, "j", "74"]), $a([11, "k", "75"]), $a([12, "l", "76"]), $a([13, "m", "77"]), $a([14, "n", "78"]), $a([15, "o", "79"]), $a([16, "p", "80"]), $a([17, "q", "81"]), $a([18, "r", "82"]), $a([19, "s", "83"]), $a([20, "t", "84"]), $a([21, "u", "85"]), $a([22, "v", "86"]), $a([23, "w", "87"]), $a([24, "x", "88"]), $a([25, "y", "89"]), $a([26, "z", "90"]), $a([27, "{", "91"]), $a([28, "|", "92"]), $a([29, "}", "93"]), $a([30, "~", "94"]), $a([31, 127, "95"]), $a([$1.fn3, $1.fn3, "96"]), $a([$1.fn2, $1.fn2, "97"]), $a([$1.sb1, $1.sa1, "98"]), $a([$1.swc, $1.swc, "99"]), $a([$1.swb, $1.fn4, $1.swb]), $a([$1.fn4, $1.swa, $1.swa]), $a([$1.fn1, $1.fn1, $1.fn1]), $a([$1.pad, $1.pad, $1.pad]), $a([$1.sb2, $1.sa2, $1.sb1]), $a([$1.sc2, $1.sc2, $1.sb2]), $a([$1.sc3, $1.sc3, $1.sb3])]); //#12829
        $1.charvals = $a([new Map, new Map, new Map]); //#12832
        for (var _2R = 0, _2Q = $1.charmaps.length - 1; _2R <= _2Q; _2R += 1) { //#12841
            $1.i = _2R; //#12834
            $1.encs = $get($1.charmaps, $1.i); //#12835
            for (var _2V = 0; _2V <= 2; _2V += 1) { //#12840
                $1.j = _2V; //#12837
                var _2Y = $get($1.encs, $1.j); //#12838
                $k[$j++] = _2Y; //#12838
                if ($eq($type(_2Y), 'stringtype')) { //#12838
                    var _2b = $get($k[--$j], 0); //#12838
                    $k[$j++] = _2b; //#12838
                } //#12838
                $put($get($1.charvals, $1.j), $k[--$j], $1.i); //#12839
            } //#12839
        } //#12839
        $1.seta = $get($1.charvals, 0); //#12842
        $1.setb = $get($1.charvals, 1); //#12843
        $1.setc = $get($1.charvals, 2); //#12844
        if ($1.raw) { //#12846
            $1.encoding = "raw"; //#12846
        } //#12846
        if ($eq($1.encoding, "raw")) { //#12859
            $1.cws = $a($1.barcode.length); //#12849
            $1.i = 0; //#12850
            $1.j = 0; //#12850
            for (;;) { //#12857
                if ($1.i == $1.barcode.length) { //#12852
                    break; //#12852
                } //#12852
                $1.cw = ~~$z($geti($1.barcode, $1.i + 1, 3)); //#12853
                $put($1.cws, $1.j, $1.cw); //#12854
                $1.i = $1.i + 4; //#12855
                $1.j = $1.j + 1; //#12856
            } //#12856
            $1.cws = $geti($1.cws, 0, $1.j); //#12858
        } //#12858
        if ($eq($1.encoding, "auto")) { //#13310
            var _3A = new Map([
                ["parse", $1.parse],
                ["parsefnc", $1.parsefnc],
                ["eci", true],
                ["FNC1", $1.fn1],
                ["FNC2", $1.fn2],
                ["FNC3", $1.fn3]
            ]); //#12870
            $1.fncvals = _3A; //#12872
            $k[$j++] = 'msg'; //#12873
            $k[$j++] = $1.barcode; //#12873
            $k[$j++] = $1.fncvals; //#12873
            $0.parseinput(); //#12873
            var _3D = $k[--$j]; //#12873
            $1[$k[--$j]] = _3D; //#12873
            $1.msglen = $1.msg.length; //#12874
            $k[$j++] = Infinity; //#12877
            for (var _3H = 0, _3I = $1.msglen; _3H < _3I; _3H++) { //#12877
                $k[$j++] = 0; //#12877
            } //#12877
            $k[$j++] = 0; //#12877
            $1.numSA = $a(); //#12877
            $k[$j++] = Infinity; //#12878
            for (var _3L = 0, _3M = $1.msglen; _3L < _3M; _3L++) { //#12878
                $k[$j++] = 0; //#12878
            } //#12878
            $k[$j++] = 0; //#12878
            $1.numEA = $a(); //#12878
            for (var _3P = $1.msglen - 1; _3P >= 0; _3P -= 1) { //#12888
                $1.i = _3P; //#12880
                if ($get($1.msg, $1.i) >= 0) { //#12887
                    if ($get($1.msg, $1.i) >= 128) { //#12885
                        $put($1.numEA, $1.i, $f($get($1.numEA, $1.i + 1) + 1)); //#12883
                    } else { //#12885
                        $put($1.numSA, $1.i, $f($get($1.numSA, $1.i + 1) + 1)); //#12885
                    } //#12885
                } //#12885
            } //#12885
            $1.ea = false; //#12891
            $1.msgtmp = $a([]); //#12891
            for (var _3j = 0, _3i = $1.msglen - 1; _3j <= _3i; _3j += 1) { //#12906
                $1.i = _3j; //#12893
                $1.c = $get($1.msg, $1.i); //#12894
                if ((!$xo($1.ea, $1.c < 128)) && ($1.c >= 0)) { //#12904
                    if ($1.ea) { //#12896
                        $k[$j++] = $1.numSA; //#12896
                    } else { //#12896
                        $k[$j++] = $1.numEA; //#12896
                    } //#12896
                    var _3v = $get($k[--$j], $1.i); //#12896
                    var _3y = ($f(_3v + $1.i) == $1.msglen) ? 3 : 5; //#12897
                    if (_3v < _3y) { //#12902
                        $k[$j++] = Infinity; //#12899
                        $aload($1.msgtmp); //#12899
                        $k[$j++] = $1.fn4; //#12899
                        $1.msgtmp = $a(); //#12899
                    } else { //#12902
                        $k[$j++] = Infinity; //#12901
                        $aload($1.msgtmp); //#12901
                        $k[$j++] = $1.fn4; //#12901
                        $k[$j++] = $1.fn4; //#12901
                        $1.msgtmp = $a(); //#12901
                        $1.ea = !$1.ea; //#12902
                    } //#12902
                } //#12902
                $k[$j++] = Infinity; //#12905
                $aload($1.msgtmp); //#12905
                if ($1.c >= 0) { //#12905
                    $k[$j++] = $1.c & 127; //#12905
                } else { //#12905
                    $k[$j++] = $1.c; //#12905
                } //#12905
                $1.msgtmp = $a(); //#12905
            } //#12905
            $1.msg = $1.msgtmp; //#12907
            $1.msglen = $1.msg.length; //#12908
            $1.numsscr = function() {
                $1.n = 0; //#12912
                $1.s = 0; //#12912
                $1.p = $k[--$j]; //#12913
                for (;;) { //#12924
                    if ($1.p >= $1.msglen) { //#12914
                        break; //#12914
                    } //#12914
                    var _4J = $get($1.msg, $1.p); //#12915
                    var _4L = $get($1.setc, _4J) !== undefined; //#12916
                    $k[$j++] = _4J; //#12916
                    if (!_4L) { //#12916
                        $j--; //#12916
                        break; //#12916
                    } //#12916
                    if ($k[--$j] == $1.fn1) { //#12920
                        if (($1.s % 2) == 0) { //#12919
                            $1.s = $1.s + 1; //#12919
                        } else { //#12919
                            break; //#12919
                        } //#12919
                    } //#12919
                    $1.n = $1.n + 1; //#12921
                    $1.s = $1.s + 1; //#12922
                    $1.p = $f($1.p + 1); //#12923
                } //#12923
                $k[$j++] = $1.n; //#12925
                $k[$j++] = $1.s; //#12925
            }; //#12926
            $1.enca = function() {
                $put($1.cws, $1.j, $get($1.seta, $k[--$j])); //#12930
                $1.j = $1.j + 1; //#12931
            }; //#12932
            $1.encb = function() {
                $put($1.cws, $1.j, $get($1.setb, $k[--$j])); //#12934
                $1.j = $1.j + 1; //#12935
            }; //#12936
            $1.encc = function() {
                var _4h = $k[--$j]; //#12938
                $k[$j++] = _4h; //#12942
                if ($ne($type(_4h), 'arraytype')) { //#12941
                    var _4l = $get($1.setc, $k[--$j]); //#12939
                    $k[$j++] = _4l; //#12939
                } else { //#12941
                    $aload($k[--$j]); //#12941
                    var _4n = $k[--$j]; //#12941
                    var _4o = $k[--$j]; //#12941
                    $k[$j++] = $f($f(_4n - 48) + ($f(_4o - 48) * 10)); //#12941
                } //#12941
                $put($1.cws, $1.j, $k[--$j]); //#12943
                $1.j = $1.j + 1; //#12944
            }; //#12945
            $1.anotb = function() {
                var _4t = $k[--$j]; //#12948
                var _4v = $get($1.seta, _4t) !== undefined; //#12948
                var _4x = $get($1.setb, _4t) !== undefined; //#12948
                $k[$j++] = _4v && (!_4x); //#12948
            }; //#12948
            $1.bnota = function() {
                var _4y = $k[--$j]; //#12949
                var _50 = $get($1.setb, _4y) !== undefined; //#12949
                var _52 = $get($1.seta, _4y) !== undefined; //#12949
                $k[$j++] = _50 && (!_52); //#12949
            }; //#12949
            $k[$j++] = Infinity; //#12952
            for (var _54 = 0, _55 = $1.msg.length; _54 < _55; _54++) { //#12952
                $k[$j++] = 0; //#12952
            } //#12952
            $k[$j++] = 9999; //#12952
            $1.nextanotb = $a(); //#12952
            $k[$j++] = Infinity; //#12953
            for (var _58 = 0, _59 = $1.msg.length; _58 < _59; _58++) { //#12953
                $k[$j++] = 0; //#12953
            } //#12953
            $k[$j++] = 9999; //#12953
            $1.nextbnota = $a(); //#12953
            for (var _5C = $1.msg.length - 1; _5C >= 0; _5C -= 1) { //#12966
                $1.i = _5C; //#12955
                $k[$j++] = $get($1.msg, $1.i); //#12956
                $1.anotb(); //#12956
                if ($k[--$j]) { //#12959
                    $put($1.nextanotb, $1.i, 0); //#12957
                } else { //#12959
                    $put($1.nextanotb, $1.i, $f($get($1.nextanotb, $1.i + 1) + 1)); //#12959
                } //#12959
                $k[$j++] = $get($1.msg, $1.i); //#12961
                $1.bnota(); //#12961
                if ($k[--$j]) { //#12964
                    $put($1.nextbnota, $1.i, 0); //#12962
                } else { //#12964
                    $put($1.nextbnota, $1.i, $f($get($1.nextbnota, $1.i + 1) + 1)); //#12964
                } //#12964
            } //#12964
            $1.abeforeb = function() {
                var _5Z = $k[--$j]; //#12969
                $k[$j++] = $lt($get($1.nextanotb, _5Z), $get($1.nextbnota, _5Z)); //#12969
            }; //#12969
            $1.bbeforea = function() {
                var _5e = $k[--$j]; //#12970
                $k[$j++] = $lt($get($1.nextbnota, _5e), $get($1.nextanotb, _5e)); //#12970
            }; //#12970
            $1.cws = $a(($1.barcode.length * 2) + 3); //#12972
            $1.i = 0; //#12975
            $1.j = 0; //#12975
            for (;;) { //#13065
                if ($1.pos != -1) { //#12986
                    $put($1.cws, 0, (~~($1.pos / 10)) - 1); //#12980
                    $put($1.cws, 1, ($1.pos % 10) - 1); //#12981
                    $1.j = 2; //#12982
                    $1.cset = "setb"; //#12983
                    $1.mode = 7; //#12984
                    break; //#12985
                } //#12985
                if ($1.msglen == 0) { //#12992
                    $1.cset = "setb"; //#12989
                    $1.mode = 1; //#12990
                    break; //#12991
                } //#12991
                if ($1.msglen >= 2) { //#13001
                    $k[$j++] = $get($1.msg, 0) == $1.fn1; //#12995
                    $k[$j++] = 1; //#12995
                    $1.numsscr(); //#12995
                    var _5v = $k[--$j]; //#12995
                    var _5w = $k[--$j]; //#12995
                    $k[$j++] = _5v; //#12995
                    $k[$j++] = _5w; //#12995
                    $j--; //#12995
                    var _5x = $k[--$j]; //#12995
                    var _5y = $k[--$j]; //#12995
                    if (_5y && (_5x >= 2)) { //#13000
                        $1.cset = "setc"; //#12996
                        $1.mode = 4; //#12997
                        $1.i = 1; //#12998
                        break; //#12999
                    } //#12999
                } //#12999
                if ($get($1.msg, 0) == $1.fn1) { //#13008
                    $1.cset = "setb"; //#13004
                    $1.mode = 3; //#13005
                    $1.i = 1; //#13006
                    break; //#13007
                } //#13007
                if ($1.msglen >= 2) { //#13054
                    $k[$j++] = 0; //#13011
                    $1.numsscr(); //#13011
                    var _63 = $k[--$j]; //#13011
                    var _64 = $k[--$j]; //#13011
                    $k[$j++] = _63; //#13011
                    $k[$j++] = _64; //#13011
                    $j--; //#13011
                    var _65 = $k[--$j]; //#13011
                    if ((_65 >= 2) && ((_65 % 2) == 0)) { //#13015
                        $1.cset = "setc"; //#13012
                        $1.mode = 2; //#13013
                        break; //#13014
                    } //#13014
                    $k[$j++] = 0; //#13017
                    $1.numsscr(); //#13017
                    var _66 = $k[--$j]; //#13017
                    var _67 = $k[--$j]; //#13017
                    $k[$j++] = _66; //#13017
                    $k[$j++] = _67; //#13017
                    $j--; //#13017
                    var _68 = $k[--$j]; //#13017
                    if ((_68 >= 3) && ((_68 % 2) == 1)) { //#13023
                        $k[$j++] = $get($1.msg, 0); //#13018
                        $1.encb(); //#13018
                        $1.cset = "setc"; //#13019
                        $1.mode = 5; //#13020
                        $1.i = 1; //#13021
                        break; //#13022
                    } //#13022
                    var _6E = $get($1.setb, $get($1.msg, 0)) !== undefined; //#13025
                    $k[$j++] = _6E; //#13026
                    $k[$j++] = 1; //#13026
                    $1.numsscr(); //#13026
                    var _6F = $k[--$j]; //#13026
                    var _6G = $k[--$j]; //#13026
                    $k[$j++] = _6F; //#13026
                    $k[$j++] = _6G; //#13026
                    $j--; //#13026
                    var _6H = $k[--$j]; //#13026
                    var _6I = $k[--$j]; //#13026
                    if (_6I && ((_6H >= 2) && ((_6H % 2) == 0))) { //#13032
                        $k[$j++] = $get($1.msg, 0); //#13027
                        $1.encb(); //#13027
                        $1.cset = "setc"; //#13028
                        $1.mode = 5; //#13029
                        $1.i = 1; //#13030
                        break; //#13031
                    } //#13031
                    var _6O = $get($1.setb, $get($1.msg, 0)) !== undefined; //#13034
                    $k[$j++] = _6O; //#13035
                    $k[$j++] = 1; //#13035
                    $1.numsscr(); //#13035
                    var _6P = $k[--$j]; //#13035
                    var _6Q = $k[--$j]; //#13035
                    $k[$j++] = _6P; //#13035
                    $k[$j++] = _6Q; //#13035
                    $j--; //#13035
                    var _6R = $k[--$j]; //#13035
                    var _6S = $k[--$j]; //#13035
                    if (_6S && ((_6R >= 3) && ((_6R % 2) == 1))) { //#13042
                        $k[$j++] = $get($1.msg, 0); //#13036
                        $1.encb(); //#13036
                        $k[$j++] = $get($1.msg, 1); //#13037
                        $1.encb(); //#13037
                        $1.cset = "setc"; //#13038
                        $1.mode = 6; //#13039
                        $1.i = 2; //#13040
                        break; //#13041
                    } //#13041
                    var _6a = $get($1.setb, $get($1.msg, 0)) !== undefined; //#13044
                    var _6e = $get($1.setb, $get($1.msg, 1)) !== undefined; //#13045
                    $k[$j++] = _6a && _6e; //#13046
                    $k[$j++] = 2; //#13046
                    $1.numsscr(); //#13046
                    var _6f = $k[--$j]; //#13046
                    var _6g = $k[--$j]; //#13046
                    $k[$j++] = _6f; //#13046
                    $k[$j++] = _6g; //#13046
                    $j--; //#13046
                    var _6h = $k[--$j]; //#13046
                    var _6i = $k[--$j]; //#13046
                    if (_6i && ((_6h >= 2) && ((_6h % 2) == 0))) { //#13053
                        $k[$j++] = $get($1.msg, 0); //#13047
                        $1.encb(); //#13047
                        $k[$j++] = $get($1.msg, 1); //#13048
                        $1.encb(); //#13048
                        $1.cset = "setc"; //#13049
                        $1.mode = 6; //#13050
                        $1.i = 2; //#13051
                        break; //#13052
                    } //#13052
                } //#13052
                $k[$j++] = 0; //#13056
                $1.abeforeb(); //#13056
                if ($k[--$j]) { //#13060
                    $1.cset = "seta"; //#13057
                    $1.mode = 0; //#13058
                    break; //#13059
                } //#13059
                $1.cset = "setb"; //#13062
                $1.mode = 1; //#13063
                break; //#13064
            } //#13064
            for (;;) { //#13308
                if ($1.i == $1.msglen) { //#13068
                    break; //#13068
                } //#13068
                $k[$j++] = $1.i; //#13070
                $1.numsscr(); //#13070
                $1.nums = $k[--$j]; //#13070
                $1.nchars = $k[--$j]; //#13070
                for (;;) { //#13306
                    if ($eq($1.cset, "seta")) { //#13147
                        if ($1.i < ($1.msglen - 1)) { //#13084
                            $k[$j++] = $get($1.msg, $1.i); //#13077
                            $1.bnota(); //#13077
                            $k[$j++] = $1.i + 1; //#13078
                            $1.abeforeb(); //#13078
                            var _70 = $k[--$j]; //#13078
                            var _71 = $k[--$j]; //#13078
                            if ((_71 && _70)) { //#13083
                                $k[$j++] = $1.sb1; //#13079
                                $1.enca(); //#13079
                                $k[$j++] = $get($1.msg, $1.i); //#13080
                                $1.encb(); //#13080
                                $1.i = $1.i + 1; //#13081
                                break; //#13082
                            } //#13082
                        } //#13082
                        if ($1.i < ($1.msglen - 2)) { //#13096
                            $k[$j++] = $get($1.msg, $1.i); //#13087
                            $1.bnota(); //#13087
                            $k[$j++] = $get($1.msg, $1.i + 1); //#13088
                            $1.bnota(); //#13088
                            var _7F = $k[--$j]; //#13088
                            var _7G = $k[--$j]; //#13088
                            $k[$j++] = $an(_7G, _7F); //#13089
                            $k[$j++] = $1.i + 2; //#13089
                            $1.abeforeb(); //#13089
                            var _7I = $k[--$j]; //#13089
                            var _7J = $k[--$j]; //#13089
                            if ((_7J && _7I)) { //#13095
                                $k[$j++] = $1.sb2; //#13090
                                $1.enca(); //#13090
                                $k[$j++] = $get($1.msg, $1.i); //#13091
                                $1.encb(); //#13091
                                $k[$j++] = $get($1.msg, $1.i + 1); //#13092
                                $1.encb(); //#13092
                                $1.i = $1.i + 2; //#13093
                                break; //#13094
                            } //#13094
                        } //#13094
                        $k[$j++] = $get($1.msg, $1.i); //#13098
                        $1.bnota(); //#13098
                        if ($k[--$j]) { //#13102
                            $k[$j++] = $1.swb; //#13099
                            $1.enca(); //#13099
                            $1.cset = "setb"; //#13100
                            break; //#13101
                        } //#13101
                        if ($1.i < ($1.msglen - 4)) { //#13119
                            var _7e = $get($1.seta, $get($1.msg, $1.i + 4)) !== undefined; //#13106
                            if (($1.nums == 4) && _7e) { //#13118
                                $k[$j++] = $1.sc2; //#13107
                                $1.enca(); //#13107
                                for (var _7g = 0, _7h = 2; _7g < _7h; _7g++) { //#13116
                                    if ($get($1.msg, $1.i) == $1.fn1) { //#13114
                                        $k[$j++] = $1.fn1; //#13110
                                        $1.encc(); //#13110
                                        $1.i = $1.i + 1; //#13111
                                    } else { //#13114
                                        $k[$j++] = $geti($1.msg, $1.i, 2); //#13113
                                        $1.encc(); //#13113
                                        $1.i = $1.i + 2; //#13114
                                    } //#13114
                                } //#13114
                                break; //#13117
                            } //#13117
                        } //#13117
                        if ($1.i < ($1.msglen - 6)) { //#13136
                            var _7z = $get($1.seta, $get($1.msg, $1.i + 6)) !== undefined; //#13123
                            if (($1.nums == 6) && _7z) { //#13135
                                $k[$j++] = $1.sc3; //#13124
                                $1.enca(); //#13124
                                for (var _81 = 0, _82 = 3; _81 < _82; _81++) { //#13133
                                    if ($get($1.msg, $1.i) == $1.fn1) { //#13131
                                        $k[$j++] = $1.fn1; //#13127
                                        $1.encc(); //#13127
                                        $1.i = $1.i + 1; //#13128
                                    } else { //#13131
                                        $k[$j++] = $geti($1.msg, $1.i, 2); //#13130
                                        $1.encc(); //#13130
                                        $1.i = $1.i + 2; //#13131
                                    } //#13131
                                } //#13131
                                break; //#13134
                            } //#13134
                        } //#13134
                        if (($1.nums >= 4) && (($1.nums % 2) == 0)) { //#13142
                            $k[$j++] = $1.swc; //#13139
                            $1.enca(); //#13139
                            $1.cset = "setc"; //#13140
                            break; //#13141
                        } //#13141
                        $k[$j++] = $get($1.msg, $1.i); //#13144
                        $1.enca(); //#13144
                        $1.i = $1.i + 1; //#13145
                        break; //#13146
                    } //#13146
                    if ($eq($1.cset, "setb")) { //#13221
                        if ($1.i < ($1.msglen - 1)) { //#13158
                            $k[$j++] = $get($1.msg, $1.i); //#13151
                            $1.anotb(); //#13151
                            $k[$j++] = $1.i + 1; //#13152
                            $1.bbeforea(); //#13152
                            var _8R = $k[--$j]; //#13152
                            var _8S = $k[--$j]; //#13152
                            if ((_8S && _8R)) { //#13157
                                $k[$j++] = $1.sa1; //#13153
                                $1.encb(); //#13153
                                $k[$j++] = $get($1.msg, $1.i); //#13154
                                $1.enca(); //#13154
                                $1.i = $1.i + 1; //#13155
                                break; //#13156
                            } //#13156
                        } //#13156
                        if ($1.i < ($1.msglen - 2)) { //#13170
                            $k[$j++] = $get($1.msg, $1.i); //#13161
                            $1.anotb(); //#13161
                            $k[$j++] = $get($1.msg, $1.i + 1); //#13162
                            $1.anotb(); //#13162
                            var _8g = $k[--$j]; //#13162
                            var _8h = $k[--$j]; //#13162
                            $k[$j++] = $an(_8h, _8g); //#13163
                            $k[$j++] = $1.i + 2; //#13163
                            $1.bbeforea(); //#13163
                            var _8j = $k[--$j]; //#13163
                            var _8k = $k[--$j]; //#13163
                            if ((_8k && _8j)) { //#13169
                                $k[$j++] = $1.sa2; //#13164
                                $1.encb(); //#13164
                                $k[$j++] = $get($1.msg, $1.i); //#13165
                                $1.enca(); //#13165
                                $k[$j++] = $get($1.msg, $1.i + 1); //#13166
                                $1.enca(); //#13166
                                $1.i = $1.i + 2; //#13167
                                break; //#13168
                            } //#13168
                        } //#13168
                        $k[$j++] = $get($1.msg, $1.i); //#13172
                        $1.anotb(); //#13172
                        if ($k[--$j]) { //#13176
                            $k[$j++] = $1.swa; //#13173
                            $1.encb(); //#13173
                            $1.cset = "seta"; //#13174
                            break; //#13175
                        } //#13175
                        if ($1.i < ($1.msglen - 4)) { //#13193
                            var _95 = $get($1.setb, $get($1.msg, $1.i + 4)) !== undefined; //#13180
                            if (($1.nums == 4) && _95) { //#13192
                                $k[$j++] = $1.sc2; //#13181
                                $1.encb(); //#13181
                                for (var _97 = 0, _98 = 2; _97 < _98; _97++) { //#13190
                                    if ($get($1.msg, $1.i) == $1.fn1) { //#13188
                                        $k[$j++] = $1.fn1; //#13184
                                        $1.encc(); //#13184
                                        $1.i = $1.i + 1; //#13185
                                    } else { //#13188
                                        $k[$j++] = $geti($1.msg, $1.i, 2); //#13187
                                        $1.encc(); //#13187
                                        $1.i = $1.i + 2; //#13188
                                    } //#13188
                                } //#13188
                                break; //#13191
                            } //#13191
                        } //#13191
                        if ($1.i < ($1.msglen - 6)) { //#13210
                            var _9Q = $get($1.setb, $get($1.msg, $1.i + 6)) !== undefined; //#13197
                            if (($1.nums == 6) && _9Q) { //#13209
                                $k[$j++] = $1.sc3; //#13198
                                $1.encb(); //#13198
                                for (var _9S = 0, _9T = 3; _9S < _9T; _9S++) { //#13207
                                    if ($get($1.msg, $1.i) == $1.fn1) { //#13205
                                        $k[$j++] = $1.fn1; //#13201
                                        $1.encc(); //#13201
                                        $1.i = $1.i + 1; //#13202
                                    } else { //#13205
                                        $k[$j++] = $geti($1.msg, $1.i, 2); //#13204
                                        $1.encc(); //#13204
                                        $1.i = $1.i + 2; //#13205
                                    } //#13205
                                } //#13205
                                break; //#13208
                            } //#13208
                        } //#13208
                        if (($1.nums >= 4) && (($1.nums % 2) == 0)) { //#13216
                            $k[$j++] = $1.swc; //#13213
                            $1.encb(); //#13213
                            $1.cset = "setc"; //#13214
                            break; //#13215
                        } //#13215
                        $k[$j++] = $get($1.msg, $1.i); //#13218
                        $1.encb(); //#13218
                        $1.i = $1.i + 1; //#13219
                        break; //#13220
                    } //#13220
                    if ($eq($1.cset, "setc")) { //#13304
                        if ($1.nums >= 2) { //#13233
                            if ($get($1.msg, $1.i) == $1.fn1) { //#13230
                                $k[$j++] = $1.fn1; //#13226
                                $1.encc(); //#13226
                                $1.i = $1.i + 1; //#13227
                            } else { //#13230
                                $k[$j++] = $geti($1.msg, $1.i, 2); //#13229
                                $1.encc(); //#13229
                                $1.i = $1.i + 2; //#13230
                            } //#13230
                            break; //#13232
                        } //#13232
                        if ($1.i < ($1.msglen - 1)) { //#13243
                            var _A3 = $get($1.setb, $get($1.msg, $1.i)) !== undefined; //#13236
                            $k[$j++] = _A3; //#13237
                            $k[$j++] = $1.i + 1; //#13237
                            $1.numsscr(); //#13237
                            var _A5 = $k[--$j]; //#13237
                            var _A6 = $k[--$j]; //#13237
                            $k[$j++] = _A5; //#13237
                            $k[$j++] = _A6; //#13237
                            $j--; //#13237
                            var _A7 = $k[--$j]; //#13237
                            var _A8 = $k[--$j]; //#13237
                            if (_A8 && ((_A7 >= 2) && ((_A7 % 2) == 0))) { //#13242
                                $k[$j++] = $1.sb1; //#13238
                                $1.encc(); //#13238
                                $k[$j++] = $get($1.msg, $1.i); //#13239
                                $1.encb(); //#13239
                                $1.i = $1.i + 1; //#13240
                                break; //#13241
                            } //#13241
                        } //#13241
                        if ($1.i < ($1.msglen - 1)) { //#13254
                            var _AK = $get($1.setb, $get($1.msg, $1.i)) !== undefined; //#13246
                            $k[$j++] = _AK; //#13247
                            $k[$j++] = $1.i + 1; //#13247
                            $1.numsscr(); //#13247
                            var _AM = $k[--$j]; //#13247
                            var _AN = $k[--$j]; //#13247
                            $k[$j++] = _AM; //#13247
                            $k[$j++] = _AN; //#13247
                            $j--; //#13247
                            var _AO = $k[--$j]; //#13247
                            var _AP = $k[--$j]; //#13247
                            if (_AP && ((_AO >= 3) && ((_AO % 2) == 1))) { //#13253
                                $k[$j++] = $1.sb2; //#13248
                                $1.encc(); //#13248
                                $k[$j++] = $get($1.msg, $1.i); //#13249
                                $1.encb(); //#13249
                                $k[$j++] = $get($1.msg, $1.i + 1); //#13250
                                $1.encb(); //#13250
                                $1.i = $1.i + 2; //#13251
                                break; //#13252
                            } //#13252
                        } //#13252
                        if ($1.i < ($1.msglen - 2)) { //#13266
                            var _Ae = $get($1.setb, $get($1.msg, $1.i)) !== undefined; //#13257
                            var _Aj = $get($1.setb, $get($1.msg, $1.i + 1)) !== undefined; //#13258
                            $k[$j++] = _Ae && _Aj; //#13259
                            $k[$j++] = $1.i + 2; //#13259
                            $1.numsscr(); //#13259
                            var _Al = $k[--$j]; //#13259
                            var _Am = $k[--$j]; //#13259
                            $k[$j++] = _Al; //#13259
                            $k[$j++] = _Am; //#13259
                            $j--; //#13259
                            var _An = $k[--$j]; //#13259
                            var _Ao = $k[--$j]; //#13259
                            if (_Ao && ((_An >= 2) && ((_An % 2) == 0))) { //#13265
                                $k[$j++] = $1.sb2; //#13260
                                $1.encc(); //#13260
                                $k[$j++] = $get($1.msg, $1.i); //#13261
                                $1.encb(); //#13261
                                $k[$j++] = $get($1.msg, $1.i + 1); //#13262
                                $1.encb(); //#13262
                                $1.i = $1.i + 2; //#13263
                                break; //#13264
                            } //#13264
                        } //#13264
                        if ($1.i < ($1.msglen - 3)) { //#13279
                            var _B3 = $get($1.setb, $get($1.msg, $1.i)) !== undefined; //#13269
                            var _B8 = $get($1.setb, $get($1.msg, $1.i + 1)) !== undefined; //#13270
                            $k[$j++] = _B3 && _B8; //#13271
                            $k[$j++] = $1.i + 2; //#13271
                            $1.numsscr(); //#13271
                            var _BA = $k[--$j]; //#13271
                            var _BB = $k[--$j]; //#13271
                            $k[$j++] = _BA; //#13271
                            $k[$j++] = _BB; //#13271
                            $j--; //#13271
                            var _BC = $k[--$j]; //#13271
                            var _BD = $k[--$j]; //#13271
                            if (_BD && ((_BC >= 3) && ((_BC % 2) == 1))) { //#13278
                                $k[$j++] = $1.sb3; //#13272
                                $1.encc(); //#13272
                                $k[$j++] = $get($1.msg, $1.i); //#13273
                                $1.encb(); //#13273
                                $k[$j++] = $get($1.msg, $1.i + 1); //#13274
                                $1.encb(); //#13274
                                $k[$j++] = $get($1.msg, $1.i + 2); //#13275
                                $1.encb(); //#13275
                                $1.i = $1.i + 3; //#13276
                                break; //#13277
                            } //#13277
                        } //#13277
                        if ($1.i < ($1.msglen - 3)) { //#13293
                            var _BV = $get($1.setb, $get($1.msg, $1.i)) !== undefined; //#13282
                            var _Ba = $get($1.setb, $get($1.msg, $1.i + 1)) !== undefined; //#13283
                            var _Bf = $get($1.setb, $get($1.msg, $1.i + 2)) !== undefined; //#13284
                            $k[$j++] = (_BV && _Ba) && _Bf; //#13285
                            $k[$j++] = $1.i + 3; //#13285
                            $1.numsscr(); //#13285
                            var _Bh = $k[--$j]; //#13285
                            var _Bi = $k[--$j]; //#13285
                            $k[$j++] = _Bh; //#13285
                            $k[$j++] = _Bi; //#13285
                            $j--; //#13285
                            var _Bj = $k[--$j]; //#13285
                            var _Bk = $k[--$j]; //#13285
                            if (_Bk && ((_Bj >= 2) && ((_Bj % 2) == 0))) { //#13292
                                $k[$j++] = $1.sb3; //#13286
                                $1.encc(); //#13286
                                $k[$j++] = $get($1.msg, $1.i); //#13287
                                $1.encb(); //#13287
                                $k[$j++] = $get($1.msg, $1.i + 1); //#13288
                                $1.encb(); //#13288
                                $k[$j++] = $get($1.msg, $1.i + 2); //#13289
                                $1.encb(); //#13289
                                $1.i = $1.i + 3; //#13290
                                break; //#13291
                            } //#13291
                        } //#13291
                        $k[$j++] = $1.i; //#13295
                        $1.abeforeb(); //#13295
                        if ($k[--$j]) { //#13299
                            $k[$j++] = $1.swa; //#13296
                            $1.encc(); //#13296
                            $1.cset = "seta"; //#13297
                            break; //#13298
                        } //#13298
                        $k[$j++] = $1.swb; //#13301
                        $1.encc(); //#13301
                        $1.cset = "setb"; //#13302
                        break; //#13303
                    } //#13303
                    break; //#13305
                } //#13305
            } //#13305
            $1.cws = $geti($1.cws, 0, $1.j); //#13309
        } //#13309
        $1.metrics = $a([$a([2, 7]), $a([3, 12]), $a([4, 17]), $a([5, 22]), $a([6, 27]), $a([7, 32]), $a([8, 37]), $a([9, 42]), $a([10, 47]), $a([11, 52]), $a([12, 57]), $a([13, 62]), $a([14, 67]), $a([15, 72]), $a([16, 77])]); //#13330
        $1.urows = $1.rows; //#13333
        $1.i = 0; //#13334
        for (;;) { //#13344
            $1.m = $get($1.metrics, $1.i); //#13336
            $1.r = $get($1.m, 0); //#13337
            $1.dcws = $get($1.m, 1); //#13338
            $1.okay = true; //#13339
            if (($1.urows != 0) && ($1.urows != $1.r)) { //#13340
                $1.okay = false; //#13340
            } //#13340
            if ($1.cws.length > $1.dcws) { //#13341
                $1.okay = false; //#13341
            } //#13341
            if ($1.okay) { //#13342
                break; //#13342
            } //#13342
            $1.i = $1.i + 1; //#13343
        } //#13343
        $k[$j++] = Infinity; //#13347
        $aload($1.cws); //#13347
        for (var _Cb = 0, _Cc = $f($1.dcws - $1.cws.length); _Cb < _Cc; _Cb++) { //#13347
            $k[$j++] = 103; //#13347
        } //#13347
        $1.cws = $a(); //#13347
        $k[$j++] = Infinity; //#13350
        $k[$j++] = $f(($f($1.r - 2) * 7) + $1.mode); //#13350
        $aload($1.cws); //#13350
        $1.cws = $a(); //#13350
        $k[$j++] = 0; //#13353
        for (var _Ck = 0, _Cj = $1.dcws; _Ck <= _Cj; _Ck += 1) { //#13353
            var _Cn = $k[--$j]; //#13353
            $k[$j++] = $f(_Cn + ((_Ck + 2) * $get($1.cws, _Ck))); //#13353
        } //#13353
        $1.c1 = $k[--$j] % 107; //#13354
        $k[$j++] = 0; //#13355
        for (var _Cr = 0, _Cq = $1.dcws; _Cr <= _Cq; _Cr += 1) { //#13355
            var _Cu = $k[--$j]; //#13355
            $k[$j++] = $f(_Cu + ((_Cr + 1) * $get($1.cws, _Cr))); //#13355
        } //#13355
        $1.c2 = ($f($k[--$j] + ($1.c1 * $f($1.dcws + 2)))) % 107; //#13356
        $k[$j++] = Infinity; //#13357
        $aload($1.cws); //#13357
        $k[$j++] = $1.c1; //#13357
        $k[$j++] = $1.c2; //#13357
        $1.cws = $a(); //#13357
        $1.encs = $a(["212222", "222122", "222221", "121223", "121322", "131222", "122213", "122312", "132212", "221213", "221312", "231212", "112232", "122132", "122231", "113222", "123122", "123221", "223211", "221132", "221231", "213212", "223112", "312131", "311222", "321122", "321221", "312212", "322112", "322211", "212123", "212321", "232121", "111323", "131123", "131321", "112313", "132113", "132311", "211313", "231113", "231311", "112133", "112331", "132131", "113123", "113321", "133121", "313121", "211331", "231131", "213113", "213311", "213131", "311123", "311321", "331121", "312113", "312311", "332111", "314111", "221411", "431111", "111224", "111422", "121124", "121421", "141122", "141221", "112214", "112412", "122114", "122411", "142112", "142211", "241211", "221114", "413111", "241112", "134111", "111242", "121142", "121241", "114212", "124112", "124211", "411212", "421112", "421211", "212141", "214121", "412121", "111143", "111341", "131141", "114113", "114311", "411113", "411311", "113141", "114131", "311141", "411131", "211412", "211214", "211232", "211133"]); //#13377
        $1.startencs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "3112", "3211", "2221", "2122", "1411", "1132", "1231", "1114", "3112"]); //#13383
        $1.stopencsodd = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "3112", "1132", "1231", "1114", "3112", "3211", "2221", "2122", "1411"]); //#13387
        $1.stopencseven = $a(["2122", "1411", "1132", "1231", "1114", "3112", "1132", "1231", "1114", "3112", "3211", "2221", "2122", "1411", "3211", "2221"]); //#13391
        if (($1.pos == -1) || (((~~($1.pos / 10)) % 2) == 1)) { //#13395
            $1.stopencs = $1.stopencsodd; //#13393
        } else { //#13395
            $1.stopencs = $1.stopencseven; //#13395
        } //#13395
        $1.rowbits = $a($1.r); //#13399
        for (var _DE = 0, _DD = $f($1.r - 1); _DE <= _DD; _DE += 1) { //#13416
            $1.i = _DE; //#13401
            $k[$j++] = Infinity; //#13410
            $k[$j++] = 10; //#13404
            $forall($get($1.startencs, $1.i), function() { //#13404
                var _DI = $k[--$j]; //#13404
                $k[$j++] = $f(_DI - 48); //#13404
            }); //#13404
            var _DL = $geti($1.cws, $1.i * 5, 5); //#13406
            $k[$j++] = 1; //#13408
            for (var _DM = 0, _DN = _DL.length; _DM < _DN; _DM++) { //#13408
                $forall($get($1.encs, $get(_DL, _DM)), function() { //#13407
                    var _DR = $k[--$j]; //#13407
                    $k[$j++] = $f(_DR - 48); //#13407
                }); //#13407
            } //#13407
            $forall($get($1.stopencs, $1.i), function() { //#13409
                var _DV = $k[--$j]; //#13409
                $k[$j++] = $f(_DV - 48); //#13409
            }); //#13409
            $k[$j++] = 1; //#13410
            $1.sbs = $a(); //#13410
            $k[$j++] = Infinity; //#13412
            var _DX = $1.sbs; //#13413
            $k[$j++] = 1; //#13413
            for (var _DY = 0, _DZ = _DX.length; _DY < _DZ; _DY++) { //#13413
                var _Db = $k[--$j]; //#13413
                var _Dc = (_Db == 0) ? 1 : 0; //#13413
                $k[$j++] = _Db; //#13413
                for (var _Dd = 0, _De = $get(_DX, _DY); _Dd < _De; _Dd++) { //#13413
                    $k[$j++] = _Dc //#13413
                } //#13413
            } //#13413
            $astore($a($counttomark() - 1)); //#13414
            var _Dh = $k[--$j]; //#13414
            var _Di = $k[--$j]; //#13414
            $k[$j++] = _Dh; //#13414
            $k[$j++] = _Di; //#13414
            $j--; //#13414
            var _Dj = $k[--$j]; //#13414
            var _Dk = $k[--$j]; //#13414
            $k[$j++] = _Dj; //#13414
            $k[$j++] = _Dk; //#13414
            $j--; //#13414
            $put($1.rowbits, $1.i, $k[--$j]); //#13415
        } //#13415
        $k[$j++] = Infinity; //#13427
        for (var _Dp = 0, _Dq = 81 * $1.sepheight; _Dp < _Dq; _Dp++) { //#13420
            $k[$j++] = 1; //#13420
        } //#13420
        for (var _Dt = 0, _Ds = $f($1.r - 2); _Dt <= _Ds; _Dt += 1) { //#13425
            $1.i = _Dt; //#13422
            for (var _Dv = 0, _Dw = $1.rowheight; _Dv < _Dw; _Dv++) { //#13423
                $aload($get($1.rowbits, $1.i)); //#13423
            } //#13423
            for (var _E1 = 0, _E2 = $1.sepheight; _E1 < _E2; _E1++) { //#13424
                for (var _E3 = 0, _E4 = 10; _E3 < _E4; _E3++) { //#13424
                    $k[$j++] = 0; //#13424
                } //#13424
                for (var _E5 = 0, _E6 = 70; _E5 < _E6; _E5++) { //#13424
                    $k[$j++] = 1; //#13424
                } //#13424
                $k[$j++] = 0; //#13424
            } //#13424
        } //#13424
        for (var _E8 = 0, _E9 = $1.rowheight; _E8 < _E9; _E8++) { //#13426
            $aload($get($1.rowbits, $f($1.r - 1))); //#13426
        } //#13426
        for (var _EE = 0, _EF = 81 * $1.sepheight; _EE < _EF; _EE++) { //#13427
            $k[$j++] = 1; //#13427
        } //#13427
        $1.pixs = $a(); //#13427
        var _EL = new Map([
            ["ren", $0.renmatrix],
            ["pixs", $1.pixs],
            ["pixx", 81],
            ["pixy", ~~($1.pixs.length / 81)],
            ["height", (~~($1.pixs.length / 81)) / 72],
            ["width", 81 / 72],
            ["opt", $1.options]
        ]); //#13438
        $k[$j++] = _EL; //#13441
        if (!$1.dontdraw) { //#13441
            $0.renmatrix(); //#13441
        } //#13441
    };
    $0.code49 = function() {
        var $1 = {}; //#13471
        $1.options = $k[--$j]; //#13473
        $1.barcode = $k[--$j]; //#13474
        $1.dontdraw = false; //#13476
        $1.mode = -1; //#13477
        $1.pos = -1; //#13478
        $1.rows = 0; //#13479
        $1.rowheight = 8; //#13480
        $1.sepheight = 1; //#13481
        $1.parse = false; //#13482
        $1.parsefnc = false; //#13483
        $forall($1.options, function() { //#13494
            var _3 = $k[--$j]; //#13494
            $1[$k[--$j]] = _3; //#13494
        }); //#13494
        $1.mode = ~~$1.mode; //#13496
        $1.pos = ~~$1.pos; //#13497
        $1.rows = ~~$1.rows; //#13498
        $1.rowheight = ~~$1.rowheight; //#13499
        $1.sepheight = ~~$1.sepheight; //#13500
        $1.s1 = -1; //#13503
        $1.s2 = -2; //#13503
        $1.fn1 = -3; //#13503
        $1.fn2 = -4; //#13503
        $1.fn3 = -5; //#13503
        $1.ns = -6; //#13503
        var _F = new Map([
            ["parse", $1.parse],
            ["parsefnc", $1.parsefnc],
            ["FNC1", $1.fn1],
            ["FNC2", $1.fn2],
            ["FNC3", $1.fn3]
        ]); //#13509
        $1.fncvals = _F; //#13510
        $k[$j++] = 'msg'; //#13511
        $k[$j++] = $1.barcode; //#13511
        $k[$j++] = $1.fncvals; //#13511
        $0.parseinput(); //#13511
        var _I = $k[--$j]; //#13511
        $1[$k[--$j]] = _I; //#13511
        $1.msglen = $1.msg.length; //#13512
        $1.charmap = $a(["0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "-", ".", " ", "$", "/", "+", "%", $1.s1, $1.s2, $1.fn1, $1.fn2, $1.fn3, $1.ns]); //#13521
        $1.charvals = new Map; //#13524
        for (var _S = 0; _S <= 48; _S += 1) { //#13530
            $1.i = _S; //#13526
            var _V = $get($1.charmap, $1.i); //#13527
            $k[$j++] = _V; //#13528
            if ($eq($type(_V), 'stringtype')) { //#13528
                var _Y = $get($k[--$j], 0); //#13528
                $k[$j++] = _Y; //#13528
            } //#13528
            $put($1.charvals, $k[--$j], $1.i); //#13529
        } //#13529
        $1.combos = $a(["1 ", "1A", "1B", "1C", "1D", "1E", "1F", "1G", "1H", "1I", "1J", "1K", "1L", "1M", "1N", "1O", "1P", "1Q", "1R", "1S", "1T", "1U", "1V", "1W", "1X", "1Y", "1Z", "11", "12", "13", "14", "15", "  ", "16", "17", "18", " $", " %", "19", "10", "1-", "1.", "1$", " +", "1/", " -", " .", " /", " 0", " 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8", " 9", "1+", "21", "22", "23", "24", "25", "26", " A", " B", " C", " D", " E", " F", " G", " H", " I", " J", " K", " L", " M", " N", " O", " P", " Q", " R", " S", " T", " U", " V", " W", " X", " Y", " Z", "27", "28", "29", "20", "2-", "2.", "2A", "2B", "2C", "2D", "2E", "2F", "2G", "2H", "2I", "2J", "2K", "2L", "2M", "2N", "2O", "2P", "2Q", "2R", "2S", "2T", "2U", "2V", "2W", "2X", "2Y", "2Z", "2$", "2/", "2+", "2%", "2 "]); //#13542
        for (var _d = 0; _d <= 127; _d += 1) { //#13552
            $1.i = _d; //#13544
            var _g = $get($1.combos, $1.i); //#13545
            $1.c1 = $get(_g, 0); //#13545
            $1.c2 = $get(_g, 1); //#13545
            if ($1.c1 == 49) { //#13548
                $put($1.charvals, $1.i, $a([$get($1.charvals, $1.s1), $get($1.charvals, $1.c2)])); //#13547
            } //#13547
            if ($1.c1 == 50) { //#13551
                $put($1.charvals, $1.i, $a([$get($1.charvals, $1.s2), $get($1.charvals, $1.c2)])); //#13550
            } //#13550
        } //#13550
        $1.encodealpha = function() {
            var _15 = $get($1.charvals, $k[--$j]); //#13555
            $k[$j++] = _15; //#13556
            if ($ne($type(_15), 'arraytype')) { //#13556
                $astore($a(1)); //#13556
            } //#13556
            var _18 = $k[--$j]; //#13557
            $puti($1.cws, $1.j, _18); //#13557
            $1.j = _18.length + $1.j; //#13558
        }; //#13559
        $1.base48 = function() {
            var _1C = $k[--$j]; //#13562
            $k[$j++] = 0; //#13562
            $forall(_1C, function() { //#13562
                var _1D = $k[--$j]; //#13562
                var _1E = $k[--$j]; //#13562
                $k[$j++] = $f($f(_1D - 48) + (_1E * 10)); //#13562
            }); //#13562
            $k[$j++] = Infinity; //#13563
            var _1F = $k[--$j]; //#13563
            var _1G = $k[--$j]; //#13563
            var _1H = $k[--$j]; //#13563
            $k[$j++] = _1F; //#13563
            $k[$j++] = _1G; //#13563
            for (var _1I = 0, _1J = $f(_1H - 1); _1I < _1J; _1I++) { //#13563
                var _1K = $k[--$j]; //#13563
                $k[$j++] = _1K % 48; //#13563
                $k[$j++] = ~~(_1K / 48); //#13563
            } //#13563
            var _1L = $a(); //#13563
            $k[$j++] = _1L; //#13564
            $k[$j++] = Infinity; //#13564
            var _1M = $k[--$j]; //#13564
            var _1N = $k[--$j]; //#13564
            $k[$j++] = _1M; //#13564
            $k[$j++] = _1N; //#13564
            for (var _1O = _1N.length - 1; _1O >= 0; _1O -= 1) { //#13564
                var _1P = $k[--$j]; //#13564
                $k[$j++] = $get(_1P, _1O); //#13564
                $k[$j++] = _1P; //#13564
            } //#13564
            $j--; //#13564
            var _1R = $a(); //#13564
            $puti($1.cws, $1.j, _1R); //#13565
            $1.j = _1R.length + $1.j; //#13566
        }; //#13567
        $1.encodenumeric = function() {
            $1.nums = $k[--$j]; //#13570
            var _1X = $1.nums.length; //#13571
            var _1Y = _1X % 5; //#13571
            $k[$j++] = 'pre'; //#13571
            $k[$j++] = _1X; //#13571
            $k[$j++] = _1Y; //#13571
            if (_1Y != 2) { //#13571
                var _1Z = $k[--$j]; //#13571
                var _1a = $k[--$j]; //#13571
                $k[$j++] = $f(_1a - _1Z); //#13571
            } else { //#13571
                var _1b = $k[--$j]; //#13571
                var _1c = $k[--$j]; //#13571
                $k[$j++] = $f($f(_1c - _1b) - 5); //#13571
            } //#13571
            var _1d = $k[--$j]; //#13571
            $1[$k[--$j]] = _1d; //#13571
            for (var _1h = 0, _1g = $f($1.pre - 1); _1h <= _1g; _1h += 5) { //#13572
                $k[$j++] = 3; //#13572
                $k[$j++] = $geti($1.nums, _1h, 5); //#13572
                $1.base48(); //#13572
            } //#13572
            $1.nums = $geti($1.nums, $1.pre, $f($1.nums.length - $1.pre)); //#13573
            var _1q = $1.nums.length; //#13575
            $k[$j++] = _1q; //#13575
            if (_1q == 1) { //#13575
                $k[$j++] = $get($1.nums, $1.i); //#13575
                $1.encodealpha(); //#13575
            } //#13575
            var _1u = $k[--$j]; //#13576
            $k[$j++] = _1u; //#13576
            if (_1u == 3) { //#13576
                $k[$j++] = 2; //#13576
                $k[$j++] = $1.nums; //#13576
                $1.base48(); //#13576
            } //#13576
            var _1w = $k[--$j]; //#13577
            $k[$j++] = _1w; //#13577
            if (_1w == 4) { //#13577
                $k[$j++] = 3; //#13577
                $k[$j++] = Infinity; //#13577
                $k[$j++] = 49; //#13577
                $k[$j++] = 48; //#13577
                $aload($1.nums); //#13577
                var _1y = $a(); //#13577
                $k[$j++] = _1y; //#13577
                $1.base48(); //#13577
            } //#13577
            if ($k[--$j] == 7) { //#13581
                $k[$j++] = 3; //#13579
                $k[$j++] = Infinity; //#13579
                $k[$j++] = 49; //#13579
                $k[$j++] = 48; //#13579
                $aload($geti($1.nums, 0, 4)); //#13579
                var _22 = $a(); //#13579
                $k[$j++] = _22; //#13579
                $1.base48(); //#13579
                $k[$j++] = 2; //#13580
                $k[$j++] = $geti($1.nums, 4, 3); //#13580
                $1.base48(); //#13580
            } //#13580
        }; //#13582
        $k[$j++] = Infinity; //#13585
        for (var _26 = 0, _27 = $1.msglen; _26 < _27; _26++) { //#13585
            $k[$j++] = 0; //#13585
        } //#13585
        $k[$j++] = 0; //#13585
        $1.numericruns = $a(); //#13585
        for (var _2A = $1.msglen - 1; _2A >= 0; _2A -= 1) { //#13594
            $1.i = _2A; //#13587
            var _2D = $get($1.msg, $1.i); //#13588
            if ((_2D >= 48) && (_2D <= 57)) { //#13592
                $put($1.numericruns, $1.i, $f($get($1.numericruns, $1.i + 1) + 1)); //#13590
            } else { //#13592
                $put($1.numericruns, $1.i, 0); //#13592
            } //#13592
        } //#13592
        if ($1.mode == -1) { //#13606
            for (;;) { //#13605
                if ($1.pos != -1) { //#13599
                    $1.mode = 3; //#13599
                    break; //#13599
                } //#13599
                if ($get($1.numericruns, 0) >= 5) { //#13600
                    $1.mode = 2; //#13600
                    break; //#13600
                } //#13600
                var _2S = $get($1.charvals, $get($1.msg, 0)); //#13601
                $k[$j++] = _2S; //#13602
                if ($ne($type(_2S), 'arraytype')) { //#13602
                    $j--; //#13602
                    $1.mode = 0; //#13602
                    break; //#13602
                } //#13602
                var _2W = ($get($k[--$j], 0) == 43) ? 4 : 5; //#13603
                $1.mode = _2W; //#13603
                break; //#13604
            } //#13604
        } //#13604
        $1.cws = $a(($1.msglen * 2) + 1); //#13608
        $1.method = "alpha"; //#13609
        $1.i = -1; //#13609
        $1.j = -1; //#13609
        if (($1.mode == 0) || ($1.mode == 1)) { //#13612
            $1.method = "alpha"; //#13612
            $1.i = 0; //#13612
            $1.j = 0; //#13612
        } //#13612
        if ($1.mode == 2) { //#13613
            $1.method = "numeric"; //#13613
            $1.i = 0; //#13613
            $1.j = 0; //#13613
        } //#13613
        if ($1.mode == 3) { //#13627
            $1.posval = $a([12, 22, 13, 23, 33, 14, 24, 34, 44, 15, 25, 35, 45, 55, 16, 26, 36, 46, 56, 66, 17, 27, 37, 47, 57, 67, 77, 18, 28, 38, 48, 58, 68, 78, 88, 19, 29, 39, 49, 59, 69, 79, 89, 99]); //#13624
            $k[$j++] = $1.cws; //#13625
            $k[$j++] = 0; //#13625
            for (var _2f = 0; _2f <= 43; _2f += 1) { //#13625
                $k[$j++] = _2f; //#13625
                if ($get($1.posval, _2f) != $1.pos) { //#13625
                    $j--; //#13625
                } //#13625
            } //#13625
            var _2j = $k[--$j]; //#13625
            var _2k = $k[--$j]; //#13625
            $put($k[--$j], _2k, $f(_2j + 1)); //#13625
            $1.method = "alpha"; //#13626
            $1.i = 0; //#13626
            $1.j = 1; //#13626
        } //#13626
        if (($1.mode == 4) || ($1.mode == 5)) { //#13631
            $put($1.cws, 0, $get($get($1.charvals, $get($1.msg, 0)), 1)); //#13629
            $1.method = "alpha"; //#13630
            $1.i = 1; //#13630
            $1.j = 1; //#13630
        } //#13630
        for (;;) { //#13658
            if ($1.i == $1.msglen) { //#13635
                break; //#13635
            } //#13635
            for (;;) { //#13657
                if ($eq($1.method, "alpha")) { //#13646
                    if ($get($1.numericruns, $1.i) >= 5) { //#13642
                        $k[$j++] = $1.ns; //#13639
                        $1.encodealpha(); //#13639
                        $1.method = "numeric"; //#13640
                        break; //#13641
                    } //#13641
                    $k[$j++] = $get($1.msg, $1.i); //#13643
                    $1.encodealpha(); //#13643
                    $1.i = $1.i + 1; //#13644
                    break; //#13645
                } //#13645
                if ($eq($1.method, "numeric")) { //#13656
                    if ($get($1.numericruns, $1.i) < 5) { //#13652
                        $k[$j++] = $1.ns; //#13649
                        $1.encodealpha(); //#13649
                        $1.method = "alpha"; //#13650
                        break; //#13651
                    } //#13651
                    $k[$j++] = $geti($1.msg, $1.i, $get($1.numericruns, $1.i)); //#13653
                    $1.encodenumeric(); //#13653
                    $1.i = $f($1.i + $get($1.numericruns, $1.i)); //#13654
                    break; //#13655
                } //#13655
            } //#13655
        } //#13655
        $1.cws = $geti($1.cws, 0, $1.j); //#13659
        $1.metrics = $a([$a([2, 9]), $a([3, 16]), $a([4, 23]), $a([5, 30]), $a([6, 37]), $a([7, 42]), $a([8, 49])]); //#13671
        $1.urows = $1.rows; //#13674
        $1.i = 0; //#13675
        for (;;) { //#13685
            $1.m = $get($1.metrics, $1.i); //#13677
            $1.r = $get($1.m, 0); //#13678
            $1.dcws = $get($1.m, 1); //#13679
            $1.okay = true; //#13680
            if (($1.urows != 0) && ($1.urows != $1.r)) { //#13681
                $1.okay = false; //#13681
            } //#13681
            if ($1.cws.length > $1.dcws) { //#13682
                $1.okay = false; //#13682
            } //#13682
            if ($1.okay) { //#13683
                break; //#13683
            } //#13683
            $1.i = $1.i + 1; //#13684
        } //#13684
        $k[$j++] = Infinity; //#13688
        $aload($1.cws); //#13688
        for (var _3n = 0, _3o = $f($1.dcws - $1.cws.length); _3n < _3o; _3n++) { //#13688
            $k[$j++] = 48; //#13688
        } //#13688
        $1.cws = $a(); //#13688
        $1.ccs = $a($1.r * 8); //#13691
        $1.j = 0; //#13692
        for (var _3u = 0, _3t = $f($1.r - 2); _3u <= _3t; _3u += 1) { //#13699
            $1.i = _3u; //#13694
            $1.cc = $geti($1.cws, $1.j, 7); //#13695
            $puti($1.ccs, $1.i * 8, $1.cc); //#13696
            var _43 = $1.cc; //#13697
            $k[$j++] = $1.ccs; //#13697
            $k[$j++] = ($1.i * 8) + 7; //#13697
            $k[$j++] = 0; //#13697
            for (var _44 = 0, _45 = _43.length; _44 < _45; _44++) { //#13697
                var _47 = $k[--$j]; //#13697
                $k[$j++] = $f(_47 + $get(_43, _44)); //#13697
            } //#13697
            var _48 = $k[--$j]; //#13697
            var _49 = $k[--$j]; //#13697
            $put($k[--$j], _49, _48 % 49); //#13697
            $1.j = $1.j + 7; //#13698
        } //#13698
        if ($1.j < $1.dcws) { //#13702
            $puti($1.ccs, $1.ccs.length - 8, $geti($1.cws, $1.j, $f($1.dcws - $1.j))); //#13701
        } //#13701
        $1.cr7 = $f(($f($1.r - 2) * 7) + $1.mode); //#13705
        $put($1.ccs, $1.ccs.length - 2, $1.cr7); //#13706
        var _4Q = $a([1, 9, 31, 26, 2, 12, 17, 23, 37, 18, 22, 6, 27, 44, 15, 43, 39, 11, 13, 5, 41, 33, 36, 8, 4, 32, 3, 19, 40, 25, 29, 10, 24, 30]); //#13710
        $k[$j++] = _4Q; //#13711
        $k[$j++] = _4Q; //#13711
        $k[$j++] = Infinity; //#13711
        var _4R = $k[--$j]; //#13711
        var _4T = $geti($k[--$j], 0, 32); //#13711
        $k[$j++] = _4R; //#13711
        $k[$j++] = 20; //#13711
        $aload(_4T); //#13711
        $1.weightx = $a(); //#13711
        var _4V = $k[--$j]; //#13712
        $k[$j++] = _4V; //#13712
        $k[$j++] = _4V; //#13712
        $k[$j++] = Infinity; //#13712
        var _4W = $k[--$j]; //#13712
        var _4Y = $geti($k[--$j], 1, 32); //#13712
        $k[$j++] = _4W; //#13712
        $k[$j++] = 16; //#13712
        $aload(_4Y); //#13712
        $1.weighty = $a(); //#13712
        $k[$j++] = Infinity; //#13713
        var _4a = $k[--$j]; //#13713
        var _4c = $geti($k[--$j], 2, 32); //#13713
        $k[$j++] = _4a; //#13713
        $k[$j++] = 38; //#13713
        $aload(_4c); //#13713
        $1.weightz = $a(); //#13713
        $1.calccheck = function() {
            $1.weights = $k[--$j]; //#13715
            $1.score = 0; //#13716
            for (var _4h = 0, _4g = (~~(($f($1.r - 1) * 8) / 2)) - 1; _4h <= _4g; _4h += 1) { //#13721
                $1.i = _4h; //#13718
                $1.score = $f((($f(($get($1.ccs, $1.i * 2) * 49) + $get($1.ccs, ($1.i * 2) + 1))) * $get($1.weights, $1.i + 1)) + $1.score); //#13720
            } //#13720
            $k[$j++] = $1.score; //#13722
        }; //#13723
        $1.lastrow = $geti($1.ccs, $1.ccs.length - 8, 8); //#13724
        if ($1.r >= 7) { //#13731
            $k[$j++] = $1.cr7 * $get($1.weightz, 0); //#13728
            $k[$j++] = $1.weightz; //#13728
            $1.calccheck(); //#13728
            var _51 = $k[--$j]; //#13728
            var _53 = $f($k[--$j] + _51) % 2401; //#13729
            $k[$j++] = ~~(_53 / 49); //#13729
            $k[$j++] = _53 % 49; //#13729
            $astore($a(2)); //#13729
            $puti($1.lastrow, 0, $k[--$j]); //#13730
        } //#13730
        $1.wr1 = $f(($get($1.lastrow, 0) * 49) + $get($1.lastrow, 1)); //#13732
        $k[$j++] = $1.cr7 * $get($1.weighty, 0); //#13735
        $k[$j++] = $1.weighty; //#13735
        $1.calccheck(); //#13735
        var _5F = $k[--$j]; //#13735
        var _5L = ($f($f($k[--$j] + _5F) + ($1.wr1 * $get($1.weighty, $f(($1.r * 4) - 3))))) % 2401; //#13737
        $1.wr2 = _5L; //#13737
        $k[$j++] = ~~(_5L / 49); //#13738
        $k[$j++] = _5L % 49; //#13738
        $astore($a(2)); //#13738
        $puti($1.lastrow, 2, $k[--$j]); //#13739
        $k[$j++] = $1.cr7 * $get($1.weightx, 0); //#13742
        $k[$j++] = $1.weightx; //#13742
        $1.calccheck(); //#13742
        var _5T = $k[--$j]; //#13742
        var _5d = ($f(($f($f($k[--$j] + _5T) + ($1.wr1 * $get($1.weightx, $f(($1.r * 4) - 3))))) + ($1.wr2 * $get($1.weightx, $f(($1.r * 4) - 2))))) % 2401; //#13746
        $k[$j++] = ~~(_5d / 49); //#13746
        $k[$j++] = _5d % 49; //#13746
        $astore($a(2)); //#13746
        $puti($1.lastrow, 4, $k[--$j]); //#13747
        var _5j = $geti($1.ccs, $1.ccs.length - 8, 7); //#13750
        $k[$j++] = 0; //#13750
        for (var _5k = 0, _5l = _5j.length; _5k < _5l; _5k++) { //#13750
            var _5n = $k[--$j]; //#13750
            $k[$j++] = $f(_5n + $get(_5j, _5k)); //#13750
        } //#13750
        $put($1.ccs, $1.ccs.length - 1, $k[--$j] % 49); //#13751
        $1.patterns = $a([$a(["11521132", "25112131", "14212132", "25121221", "14221222", "12412132", "23321221", "12421222", "21521221", "15112222", "15121312", "13312222", "24221311", "13321312", "11512222", "22421311", "11521312", "25112311", "14212312", "23312311", "12412312", "21512311", "16121131", "14321131", "12521131", "15212131", "15221221", "13412131", "13421221", "11612131", "16112221", "16121311", "14312221", "14321311", "12512221", "12521311", "15212311", "13412311", "11612311", "11131135", "31131133", "51131131", "21122134", "41122132", "21131224", "41131222", "11113135", "31113133", "51113131", "11122225", "31122223", "51122221", "11131315", "31131313", "51131311", "21113224", "41113222", "21122314", "41122312", "11113315", "31113313", "51113311", "12131134", "32131132", "21231133", "41231131", "22122133", "42122131", "11222134", "22131223", "42131221", "11231224", "31231222", "12113134", "32113132", "12122224", "32122222", "12131314", "32131312", "21231313", "41231311", "22113223", "42113221", "11213224", "22122313", "42122311", "11222314", "31222312", "12113314", "32113312", "21213313", "41213311", "13131133", "33131131", "22231132", "11331133", "31331131", "23122132", "12222133", "23131222", "12231223", "32231221", "21331222", "13113133", "33113131", "13122223", "33122221", "11313133", "13131313", "33131311", "11322223", "22231312", "11331313", "31331311", "23113222", "12213223", "23122312", "12222313", "32222311", "21322312", "13113313", "33113311", "22213312", "11313313", "31313311", "14131132", "23231131", "12331132", "21431131", "24122131", "13222132", "24131221", "13231222", "11422132", "22331221", "11431222", "14113132", "14122222", "12313132", "14131312", "12322222", "23231311", "12331312", "21431311", "24113221", "13213222", "24122311", "13222312", "11413222", "22322311", "11422312", "14113312", "23213311", "12313312", "21413311", "15131131", "13331131", "14222131", "14231221", "12422131", "12431221", "15113131", "15122221", "13313131", "15131311", "13322221", "11513131", "13331311", "11522221", "14213221", "14222311", "12413221", "12422311", "15113311", "13313311", "11513311", "11141134", "31141132", "21132133", "41132131", "21141223", "41141221", "11123134", "31123132", "11132224", "31132222", "11141314", "31141312", "21114133", "41114131", "21123223", "41123221", "21132313", "41132311", "11114224", "31114222", "11123314", "31123312", "21114313", "41114311", "12141133", "32141131", "21241132", "22132132", "11232133", "22141222", "11241223", "31241221", "12123133", "32123131", "12132223", "32132221", "12141313", "32141311", "21241312", "22114132", "11214133", "22123222", "11223223", "22132312", "11232313", "31232311", "12114223", "32114221", "12123313", "32123311", "21223312", "22114312", "11214313", "31214311", "13141132", "22241131", "11341132", "23132131", "12232132", "23141221", "12241222", "21341221", "13123132", "13132222", "11323132", "13141312", "11332222", "22241311", "11341312", "23114131", "12214132", "23123221", "12223222", "23132311", "12232312", "21332311", "13114222", "13123312", "11314222", "22223311", "11323312", "23114311", "12214312", "21314311", "14141131", "12341131", "13232131", "13241221", "11432131", "14123131", "14132221", "12323131", "14141311", "12332221", "12341311", "13214131", "13223221", "11414131", "13232311", "11423221", "11432311", "14114221", "14123311", "12314221", "12323311", "13214311", "11414311", "11151133", "31151131", "21142132", "21151222", "11133133", "31133131", "11142223", "31142221", "11151313", "31151311", "21124132", "21133222", "21142312", "11115133", "31115131", "11124223", "31124221", "11133313", "31133311", "21115222", "21124312", "12151132", "21251131", "22142131", "11242132", "22151221", "11251222", "12133132", "12142222", "12151312", "21251311", "22124131", "11224132", "22133221", "11233222", "22142311", "11242312", "12115132", "12124222", "12133312", "21233311", "22115221", "11215222", "22124311", "11224312", "13151131", "12242131", "12251221", "13133131", "13142221", "11333131", "13151311", "11342221", "12224131", "12233221", "12242311", "13115131", "13124221", "11315131", "13133311", "11324221", "11333311", "12215221", "12224311", "11161132", "21152131", "21161221", "11143132", "11152222", "11161312", "21134131", "21143221", "21152311", "11125132", "11134222", "11143312", "21116131", "21125221", "21134311", "12161131", "11252131", "12143131", "12152221", "12161311", "11234131", "11243221", "11252311", "12125131", "12134221", "12143311", "11216131", "11225221", "11234311", "11111236", "31111234", "51111232", "21111325", "41111323", "61111321", "11111416", "31111414", "51111412", "31211143", "51211141", "12111235", "32111233", "52111231", "21211234", "41211232", "22111324", "42111322", "11211325", "31211323", "51211321", "12111415", "32111413", "52111411", "21211414", "41211412", "12211144", "32211142", "21311143", "41311141", "13111234", "33111232", "22211233", "42211231", "11311234", "31311232", "23111323", "43111321", "12211324", "32211322", "21311323", "41311321", "13111414", "33111412", "22211413", "42211411", "11311414", "31311412", "13211143", "33211141", "22311142", "11411143", "31411141", "14111233", "34111231", "23211232", "12311233", "32311231", "21411232", "24111322", "13211323", "33211321", "22311322", "11411323", "31411321", "14111413", "34111411", "23211412", "12311413", "32311411", "21411412", "14211142", "23311141", "12411142", "21511141", "15111232", "24211231", "13311232", "22411231", "11511232", "25111321", "14211322", "23311321", "12411322", "21511321", "15111412", "24211411", "13311412", "22411411", "11511412", "15211141", "13411141", "11611141", "16111231", "14311231", "12511231", "15211321", "13411321", "11611321", "16111411", "14311411", "12511411", "21121144", "41121142", "11112145", "31112143", "51112141", "11121235", "31121233", "51121231", "21112234", "41112232", "21121324", "41121322", "11112325", "31112323", "51112321", "11121415", "31121413", "51121411", "21112414", "41112412", "22121143", "42121141", "11221144", "31221142", "12112144", "32112142", "12121234", "32121232", "21221233", "41221231", "22112233", "42112231", "11212234", "22121323", "42121321", "11221324", "31221322", "12112324", "32112322", "12121414", "32121412", "21221413", "41221411", "22112413", "42112411", "11212414", "31212412", "23121142", "12221143", "32221141", "21321142", "13112143", "33112141", "13121233", "33121231", "11312143", "22221232", "11321233", "31321231", "23112232", "12212233", "23121322", "12221323", "32221321", "21321322", "13112323", "33112321", "13121413", "33121411", "11312323", "22221412", "11321413", "31321411", "23112412", "12212413", "32212411", "21312412", "24121141", "13221142", "22321141", "11421142", "14112142", "14121232", "12312142", "23221231", "12321232", "21421231", "24112231", "13212232", "24121321", "13221322", "11412232", "22321321", "11421322", "14112322", "14121412", "12312322", "23221411", "12321412", "21421411", "24112411", "13212412", "22312411", "11412412", "14221141", "12421141", "15112141", "15121231", "13312141", "13321231", "11512141", "11521231", "14212231", "14221321", "12412231", "12421321", "15112321", "15121411", "13312321", "13321411", "11512321", "11521411", "14212411", "12412411", "21131143", "41131141", "11122144", "31122142", "11131234", "31131232", "21113143", "41113141", "21122233", "41122231", "21131323", "41131321", "11113234", "31113232", "11122324", "31122322", "11131414", "31131412", "21113323", "41113321", "21122413", "41122411", "11113414", "31113412", "22131142", "11231143", "31231141", "12122143", "32122141", "12131233", "32131231", "21231232", "22113142", "11213143", "22122232", "11222233", "22131322", "11231323", "31231321", "12113233", "32113231", "12122323", "32122321", "12131413", "32131411", "21231412", "22113322", "11213323", "22122412", "11222413", "31222411", "12113413", "32113411", "21213412", "23131141", "12231142", "21331141", "13122142", "13131232", "11322142", "22231231", "11331232", "23113141", "12213142", "23122231", "12222232", "23131321", "12231322", "21331321", "13113232", "13122322", "11313232", "13131412", "11322322", "22231411", "11331412", "23113321", "12213322", "23122411", "12222412", "21322411", "13113412", "22213411", "11313412", "13231141", "11431141", "14122141", "14131231", "12322141", "12331231", "13213141", "13222231", "11413141", "13231321", "11422231", "11431321", "14113231", "14122321", "12313231", "14131411", "12322321", "12331411", "13213321", "13222411", "11413321", "11422411", "14113411", "12313411", "21141142", "11132143", "31132141", "11141233", "31141231", "21123142", "21132232", "21141322", "11114143", "31114141", "11123233", "31123231", "11132323", "31132321", "11141413", "31141411", "21114232", "21123322", "21132412", "11114323", "31114321", "11123413", "31123411", "22141141", "11241142", "12132142", "12141232", "21241231", "22123141", "11223142", "22132231", "11232232", "22141321", "11241322", "12114142", "12123232", "12132322", "12141412", "21241411", "22114231", "11214232", "22123321", "11223322", "22132411", "11232412", "12114322", "12123412", "21223411", "12241141", "13132141", "13141231", "11332141", "11341231", "12223141", "12232231", "12241321", "13114141", "13123231", "11314141", "13132321", "11323231", "13141411", "11332321", "11341411", "12214231", "12223321", "12232411", "13114321", "13123411", "11314321", "11323411", "21151141", "11142142", "11151232", "21133141", "21142231", "21151321", "11124142", "11133232", "11142322", "11151412", "21115141", "21124231", "21133321", "21142411", "11115232", "11124322", "11133412", "11251141", "12142141", "12151231", "11233141", "11242231", "11251321", "12124141", "12133231", "12142321", "12151411", "11215141", "11224231", "11233321", "11242411", "12115231", "12124321", "12133411", "11152141", "11161231", "11134141", "11143231", "11152321", "11161411", "11116141", "11125231", "11134321", "11143411", "21111244", "41111242", "11111335", "31111333", "51111331", "21111424", "41111422", "11111515", "31111513", "51111511", "21211153", "41211151", "22111243", "42111241", "11211244", "31211242", "12111334", "32111332", "21211333", "41211331", "22111423", "42111421", "11211424", "31211422", "12111514", "32111512", "21211513", "41211511", "22211152", "11311153", "31311151", "23111242", "12211243", "32211241", "21311242", "13111333", "33111331", "22211332", "11311333", "31311331", "23111422", "12211423", "32211421", "21311422", "13111513", "33111511", "22211512", "11311513", "31311511", "23211151", "12311152", "21411151", "24111241", "13211242", "22311241", "11411242", "14111332", "23211331", "12311332", "21411331", "24111421", "13211422", "22311421", "11411422", "14111512", "23211511", "12311512", "21411511", "13311151", "11511151", "14211241", "12411241", "15111331", "13311331", "11511331", "14211421", "12411421", "15111511", "13311511", "11511511", "31121152", "21112153", "41112151", "21121243", "41121241", "11112244", "31112242", "11121334", "31121332", "21112333", "41112331", "21121423", "41121421", "11112424", "31112422", "11121514", "31121512", "21112513", "41112511", "12121153", "32121151", "21221152", "22112152", "11212153", "22121242", "11221243", "31221241", "12112243", "32112241", "12121333", "32121331", "21221332", "22112332", "11212333", "22121422", "11221423", "31221421", "12112423", "32112421", "12121513", "32121511", "21221512", "22112512", "11212513", "31212511", "13121152", "22221151", "11321152", "23112151", "12212152", "23121241", "12221242", "21321241", "13112242", "13121332", "11312242", "22221331", "11321332", "23112331", "12212332", "23121421", "12221422", "21321421", "13112422", "13121512", "11312422", "22221511", "11321512", "23112511", "12212512", "21312511", "14121151", "12321151", "13212151", "13221241", "11412151", "11421241", "14112241", "14121331", "12312241", "12321331", "13212331", "13221421", "11412331", "11421421", "14112421", "14121511", "12312421", "12321511", "13212511", "11412511", "11131153", "31131151", "21122152", "21131242", "11113153", "31113151", "11122243", "31122241", "11131333", "31131331", "21113242", "21122332", "21131422", "11113333", "31113331", "11122423", "31122421", "11131513", "31131511", "21113422", "21122512", "12131152", "21231151", "22122151", "11222152", "22131241", "11231242", "12113152", "12122242", "12131332", "21231331", "22113241", "11213242", "22122331", "11222332", "22131421", "11231422", "12113332", "12122422", "12131512", "21231511", "22113421", "11213422", "22122511", "11222512", "13131151", "11331151", "12222151", "12231241", "13113151", "13122241", "11313151", "13131331", "11322241", "11331331", "12213241", "12222331", "12231421", "13113331", "13122421", "11313331", "13131511", "11322421", "11331511", "12213421", "12222511", "11141152", "21132151", "21141241", "11123152", "11132242", "11141332", "21114151", "21123241", "21132331", "21141421", "11114242", "11123332", "11132422", "11141512", "21114331", "21123421", "21132511", "12141151", "11232151", "11241241", "12123151", "12132241", "12141331", "11214151", "11223241", "11232331", "11241421", "12114241", "12123331", "12132421", "12141511", "11214331", "11223421", "11232511", "11151151", "11133151", "11142241", "11151331", "11115151", "11124241", "11133331", "11142421", "11151511", "11111254", "31111252", "21111343", "41111341", "11111434", "31111432", "21111523", "41111521", "11111614", "31111612", "31211161", "12111253", "32111251", "21211252", "22111342", "11211343", "31211341", "12111433", "32111431", "21211432", "22111522", "11211523", "31211521", "12111613", "32111611", "21211612", "12211162", "21311161", "13111252", "22211251", "11311252", "23111341", "12211342", "21311341", "13111432", "22211431", "11311432", "23111521", "12211522", "21311521", "13111612", "22211611", "11311612", "13211161", "11411161", "14111251", "12311251", "13211341", "11411341", "14111431", "12311431", "13211521", "11411521", "14111611", "12311611", "21121162", "11112163", "31112161", "11121253", "31121251", "21112252", "21121342", "11112343", "31112341", "11121433", "31121431", "21112432", "21121522", "11112523", "31112521", "11121613", "31121611", "22121161", "11221162", "12112162", "12121252", "21221251", "22112251", "11212252", "22121341", "11221342", "12112342", "12121432", "21221431", "22112431", "11212432", "22121521", "11221522", "12112522", "12121612", "21221611", "12221161", "13112161", "13121251", "11312161", "11321251", "32121115", "52121113", "21221116", "41221114", "61221112", "22112116", "42112114", "31212115", "51212113", "13121116", "33121114", "22221115", "42221113", "11321116", "31321114", "51321112", "23112115", "43112113", "12212116", "32212114", "52212112", "21312115", "41312113", "61312111", "14121115", "34121113", "23221114", "43221112", "12321115", "32321113", "52321111", "21421114", "41421112", "24112114", "13212115", "33212113", "22312114", "42312112", "11412115", "31412113", "51412111", "15121114", "24221113", "13321114", "33321112", "22421113", "42421111", "11521114", "31521112", "25112113", "14212114", "34212112", "23312113", "43312111", "12412114", "32412112", "21512113", "41512111", "16121113", "25221112", "14321113", "34321111", "23421112", "12521113", "32521111", "15212113", "24312112", "13412113", "33412111", "22512112", "11612113", "31612111", "31131115", "51131113", "21122116", "41122114", "61122112", "31113115", "51113113", "12131116", "32131114", "52131112", "21231115", "41231113", "61231111", "22122115", "42122113", "11222116", "31222114", "51222112", "12113116", "32113114", "52113112", "21213115", "41213113", "61213111", "13131115", "33131113", "22231114", "42231112", "11331115", "31331113", "51331111", "23122114", "43122112", "12222115", "32222113", "52222111", "21322114", "41322112", "13113115", "33113113", "22213114", "42213112", "11313115", "31313113", "51313111", "14131114", "34131112", "23231113", "43231111", "12331114", "32331112", "21431113", "41431111", "24122113", "13222114", "33222112", "22322113", "42322111", "11422114", "31422112", "14113114", "34113112", "23213113", "43213111", "12313114", "32313112", "21413113", "41413111", "15131113", "24231112", "13331113", "33331111", "22431112", "25122112", "14222113", "34222111", "23322112", "12422113", "32422111", "21522112", "15113113", "24213112", "13313113", "33313111", "22413112", "11513113", "31513111", "16131112", "25231111", "14331112", "23431111", "15222112", "24322111", "13422112", "22522111", "16113112", "25213111", "14313112", "23413111", "12513112", "21613111", "11141116", "31141114", "51141112", "21132115", "41132113", "61132111", "11123116", "31123114", "51123112", "21114115", "41114113", "61114111", "12141115", "32141113", "52141111", "21241114", "41241112", "22132114", "42132112", "11232115", "31232113", "51232111", "12123115", "32123113", "52123111", "21223114", "41223112", "22114114", "42114112", "11214115", "31214113", "51214111", "13141114", "33141112", "22241113", "42241111", "11341114", "31341112", "23132113", "43132111", "12232114", "32232112", "21332113", "41332111", "13123114", "33123112", "22223113", "42223111", "11323114", "31323112", "23114113", "43114111", "12214114", "32214112", "21314113", "41314111", "14141113", "34141111", "23241112", "12341113", "32341111", "24132112", "13232113", "33232111", "22332112", "11432113", "31432111", "14123113", "34123111", "23223112", "12323113", "32323111", "21423112", "24114112", "13214113", "33214111", "22314112", "11414113", "31414111", "15141112", "24241111", "13341112", "25132111", "14232112", "23332111", "12432112", "15123112", "24223111", "13323112", "22423111", "11523112", "25114111", "14214112", "23314111", "12414112", "21514111", "16141111", "14341111", "15232111", "13432111", "16123111", "14323111", "12523111", "15214111", "13414111", "11614111", "11151115", "31151113", "51151111", "21142114", "41142112", "11133115", "31133113", "51133111", "21124114", "41124112", "11115115", "31115113", "51115111", "12151114", "32151112", "21251113", "41251111", "22142113", "42142111", "11242114", "31242112", "12133114", "32133112", "21233113", "41233111", "22124113", "42124111", "11224114", "31224112", "12115114", "32115112", "21215113", "41215111", "13151113", "33151111", "22251112", "23142112", "12242113", "32242111", "21342112", "13133113", "33133111", "22233112", "11333113", "31333111", "23124112", "12224113", "32224111", "21324112", "13115113", "33115111", "22215112", "11315113", "31315111", "14151112", "23251111", "24142111", "13242112", "22342111", "14133112", "23233111", "12333112", "21433111", "24124111", "13224112", "22324111", "11424112", "14115112", "23215111", "12315112", "21415111", "15151111", "14242111", "15133111", "13333111", "14224111", "12424111", "15115111", "13315111", "11515111", "11161114", "31161112", "21152113", "41152111", "11143114", "31143112", "21134113", "41134111", "11125114", "31125112", "21116113", "41116111", "12161113", "32161111", "22152112", "11252113", "31252111", "12143113", "32143111", "21243112", "22134112", "11234113", "31234111", "12125113", "32125111", "21225112", "22116112", "11216113", "31216111", "13161112", "23152111", "12252112", "13143112", "22243111", "11343112", "23134111", "12234112", "21334111", "13125112", "22225111", "11325112", "23116111", "12216112", "21316111", "14161111", "13252111", "14143111", "12343111", "13234111", "11434111", "14125111", "12325111", "13216111", "11416111", "31111216", "51111214", "31211125", "51211123", "32111215", "52111213", "21211216", "41211214", "61211212", "12211126", "32211124", "52211122", "21311125", "41311123", "61311121", "13111216", "33111214", "22211215", "42211213", "11311216", "31311214", "51311212", "13211125", "33211123", "22311124", "42311122", "11411125", "31411123", "51411121", "14111215", "34111213", "23211214", "43211212", "12311215", "32311213", "52311211", "21411214", "41411212", "14211124", "34211122", "23311123", "43311121", "12411124", "32411122", "21511123", "41511121", "15111214", "24211213", "13311214", "33311212", "22411213", "42411211", "11511214", "31511212", "15211123", "24311122", "13411123", "33411121", "22511122", "11611123", "31611121", "16111213", "25211212", "14311213", "34311211", "23411212", "12511213", "32511211", "21611212", "21121126", "41121124", "61121122", "31112125", "51112123", "31121215", "51121213", "21112216", "41112214", "61112212", "22121125", "42121123", "11221126", "31221124", "51221122", "12112126", "32112124", "52112122", "12121216", "32121214", "52121212", "21221215", "41221213", "61221211", "22112215", "42112213", "11212216", "31212214", "51212212", "23121124", "43121122", "12221125", "32221123", "52221121", "21321124", "41321122", "13112125", "33112123", "13121215", "33121213", "11312125", "22221214", "42221212", "11321215", "31321213", "51321211", "23112214", "43112212", "12212215", "32212213", "52212211", "21312214", "41312212", "24121123", "13221124", "33221122", "22321123", "42321121", "11421124", "31421122", "14112124", "34112122", "14121214", "34121212", "12312124", "23221213", "43221211", "12321214", "32321212", "21421213", "41421211", "24112213", "13212214", "33212212", "22312213", "42312211", "11412214", "31412212", "25121122", "14221123", "34221121", "23321122", "12421123", "32421121", "21521122", "15112123", "15121213", "13312123", "24221212", "13321213", "33321211", "11512123", "22421212", "11521213", "31521211", "25112212", "14212213", "34212211", "23312212", "12412213", "32412211", "21512212", "15221122", "24321121", "13421122", "22521121", "16112122", "16121212", "14312122", "25221211", "14321212", "12512122", "23421211", "12521212", "15212212", "24312211", "13412212", "22512211", "11612212", "21131125", "41131123", "61131121", "11122126", "31122124", "51122122", "11131216", "31131214", "51131212", "21113125", "41113123", "61113121", "21122215", "41122213", "61122211", "11113216", "31113214", "51113212", "22131124", "42131122", "11231125", "31231123", "51231121", "12122125", "32122123", "52122121", "12131215", "32131213", "52131211", "21231214", "41231212", "22113124", "42113122", "11213125", "22122214", "42122212", "11222215", "31222213", "51222211", "12113215", "32113213", "52113211", "21213214", "41213212", "23131123", "43131121", "12231124", "32231122", "21331123", "41331121", "13122124", "33122122", "13131214", "33131212", "11322124", "22231213", "42231211", "11331214", "31331212", "23113123", "43113121", "12213124", "23122213", "43122211", "12222214", "32222212", "21322213", "41322211", "13113214", "33113212", "22213213", "42213211", "11313214", "31313212", "24131122", "13231123", "33231121", "22331122", "11431123", "31431121", "14122123", "34122121", "14131213", "34131211", "12322123", "23231212", "12331213", "32331211", "21431212", "24113122", "13213123", "24122212", "13222213", "33222211", "11413123", "22322212", "11422213", "31422211", "14113213", "34113211", "23213212", "12313213", "32313211", "21413212", "25131121", "14231122", "23331121", "12431122", "15122122", "15131212", "13322122", "24231211", "13331212", "11522122", "22431211", "25113121", "14213122", "25122211", "14222212", "12413122", "23322211", "12422212", "21522211", "15113212", "24213211", "13313212", "22413211", "11513212", "15231121", "13431121", "16122121", "16131211", "14322121", "14331211", "12522121", "15213121", "15222211", "13413121", "13422211", "11613121", "16113211", "14313211", "12513211", "21141124", "41141122", "11132125", "31132123", "51132121", "11141215", "31141213", "51141211", "21123124", "41123122", "21132214", "41132212", "11114125", "31114123", "51114121", "11123215", "31123213", "51123211", "21114214", "41114212", "22141123", "42141121", "11241124", "31241122", "12132124", "32132122", "12141214", "32141212", "21241213", "41241211", "22123123", "42123121", "11223124", "22132213", "42132211", "11232214", "31232212", "12114124", "32114122", "12123214", "32123212", "21223213", "41223211", "22114213", "42114211", "11214214", "31214212", "23141122", "12241123", "32241121", "21341122", "13132123", "33132121", "13141213", "33141211", "11332123", "22241212", "11341213", "31341211", "23123122", "12223123", "23132212", "12232213", "32232211", "21332212", "13114123", "33114121", "13123213", "33123211", "11314123", "22223212", "11323213", "31323211", "23114212", "12214213", "32214211", "21314212", "24141121", "13241122", "22341121", "14132122", "14141212", "12332122", "23241211", "12341212", "24123121", "13223122", "24132211", "13232212", "11423122", "22332211", "11432212", "14114122", "14123212", "12314122", "23223211", "12323212", "21423211", "24114211", "13214212", "22314211", "11414212", "14241121", "15132121", "15141211", "13332121", "13341211", "14223121", "14232211", "12423121", "12432211", "15114121", "15123211", "13314121", "13323211", "11514121", "11523211", "14214211", "12414211", "21151123", "41151121", "11142124", "31142122", "11151214", "31151212", "21133123", "41133121", "21142213", "41142211", "11124124", "31124122", "11133214", "31133212", "21115123", "41115121", "21124213", "41124211", "11115214", "31115212", "22151122", "11251123", "31251121", "12142123", "32142121", "12151213", "32151211", "21251212", "22133122", "11233123", "22142212", "11242213", "31242211", "12124123", "32124121", "12133213", "32133211", "21233212", "22115122", "11215123", "22124212", "11224213", "31224211", "12115213", "32115211", "21215212", "23151121", "12251122", "13142122", "13151212", "11342122", "22251211", "23133121", "12233122", "23142211", "12242212", "21342211", "13124122", "13133212", "11324122", "22233211", "11333212", "23115121", "12215122", "23124211", "12224212", "21324211", "13115212", "22215211", "11315212", "13251121", "14142121", "14151211", "12342121", "13233121", "13242211", "11433121", "14124121", "14133211", "12324121", "12333211", "13215121", "13224211", "11415121", "11424211", "14115211", "12315211", "21161122", "11152123", "31152121", "11161213", "31161211", "21143122", "21152212", "11134123", "31134121", "11143213", "31143211", "21125122", "21134212", "11116123", "31116121", "11125213", "31125211", "22161121", "12152122", "12161212", "22143121", "11243122", "22152211", "11252212", "12134122", "12143212", "21243211", "22125121", "11225122", "22134211", "11234212", "12116122", "12125212", "21225211", "13152121", "13161211", "12243121", "12252211", "13134121", "13143211", "11334121", "11343211", "12225121", "12234211", "13116121", "13125211", "11316121", "11325211", "21111226", "41111224", "61111222", "31111315", "51111313", "21211135", "41211133", "61211131", "22111225", "42111223", "11211226", "31211224", "51211222", "12111316", "32111314", "52111312", "21211315", "41211313", "61211311", "22211134", "42211132", "11311135", "31311133", "51311131", "23111224", "43111222", "12211225", "32211223", "52211221", "21311224", "41311222", "13111315", "33111313", "22211314", "42211312", "11311315", "31311313", "51311311", "23211133", "43211131", "12311134", "32311132", "21411133", "41411131", "24111223", "13211224", "33211222", "22311223", "42311221", "11411224", "31411222", "14111314", "34111312", "23211313", "43211311", "12311314", "32311312", "21411313", "41411311", "24211132", "13311133", "33311131", "22411132", "11511133", "31511131", "25111222", "14211223", "34211221", "23311222", "12411223", "32411221", "21511222", "15111313", "24211312", "13311313", "33311311", "22411312", "11511313", "31511311", "25211131", "14311132", "23411131", "12511132", "21611131", "15211222", "24311221", "13411222", "22511221", "11611222", "16111312", "25211311", "14311312", "23411311", "12511312", "21611311", "31121134", "51121132", "21112135", "41112133", "61112131", "21121225", "41121223", "61121221", "11112226", "31112224", "51112222", "11121316", "31121314", "51121312", "21112315", "41112313", "61112311", "12121135", "32121133", "52121131", "21221134", "41221132", "22112134", "42112132", "11212135", "22121224", "42121222", "11221225", "31221223", "51221221", "12112225", "32112223", "52112221", "12121315", "32121313", "52121311", "21221314", "41221312", "22112314", "42112312", "11212315", "31212313", "51212311", "13121134", "33121132", "22221133", "42221131", "11321134", "31321132", "23112133", "43112131", "12212134", "23121223", "43121221", "12221224", "32221222", "21321223", "41321221", "13112224", "33112222", "13121314", "33121312", "11312224", "22221313", "42221311", "11321314", "31321312", "23112313", "43112311", "12212314", "32212312", "21312313", "41312311", "14121133", "34121131", "23221132", "12321133", "32321131", "21421132", "24112132", "13212133", "24121222", "13221223", "33221221", "11412133", "22321222", "11421223", "31421221", "14112223", "34112221", "14121313", "34121311", "12312223", "23221312", "12321313", "32321311", "21421312", "24112312", "13212313", "33212311", "22312312", "11412313", "31412311", "15121132", "24221131", "13321132", "22421131"]), $a(["22121116", "42121114", "31221115", "51221113", "32112115", "52112113", "21212116", "41212114", "61212112", "23121115", "43121113", "12221116", "32221114", "52221112", "21321115", "41321113", "61321111", "13112116", "33112114", "22212115", "42212113", "11312116", "31312114", "51312112", "24121114", "13221115", "33221113", "22321114", "42321112", "11421115", "31421113", "51421111", "14112115", "34112113", "23212114", "43212112", "12312115", "32312113", "52312111", "21412114", "41412112", "25121113", "14221114", "34221112", "23321113", "43321111", "12421114", "32421112", "21521113", "41521111", "15112114", "24212113", "13312114", "33312112", "22412113", "42412111", "11512114", "31512112", "15221113", "24321112", "13421113", "33421111", "22521112", "16112113", "25212112", "14312113", "34312111", "23412112", "12512113", "32512111", "21612112", "21131116", "41131114", "61131112", "31122115", "51122113", "21113116", "41113114", "61113112", "22131115", "42131113", "11231116", "31231114", "51231112", "12122116", "32122114", "52122112", "21222115", "41222113", "61222111", "22113115", "42113113", "11213116", "31213114", "51213112", "23131114", "43131112", "12231115", "32231113", "52231111", "21331114", "41331112", "13122115", "33122113", "22222114", "42222112", "11322115", "31322113", "51322111", "23113114", "43113112", "12213115", "32213113", "52213111", "21313114", "41313112", "24131113", "13231114", "33231112", "22331113", "42331111", "11431114", "31431112", "14122114", "34122112", "23222113", "43222111", "12322114", "32322112", "21422113", "41422111", "24113113", "13213114", "33213112", "22313113", "42313111", "11413114", "31413112", "25131112", "14231113", "34231111", "23331112", "12431113", "32431111", "15122113", "24222112", "13322113", "33322111", "22422112", "11522113", "31522111", "25113112", "14213113", "34213111", "23313112", "12413113", "32413111", "21513112", "15231112", "24331111", "13431112", "16122112", "25222111", "14322112", "23422111", "12522112", "15213112", "24313111", "13413112", "22513111", "11613112", "21141115", "41141113", "61141111", "11132116", "31132114", "51132112", "21123115", "41123113", "61123111", "11114116", "31114114", "51114112", "22141114", "42141112", "11241115", "31241113", "51241111", "12132115", "32132113", "52132111", "21232114", "41232112", "22123114", "42123112", "11223115", "31223113", "51223111", "12114115", "32114113", "52114111", "21214114", "41214112", "23141113", "43141111", "12241114", "32241112", "21341113", "41341111", "13132114", "33132112", "22232113", "42232111", "11332114", "31332112", "23123113", "43123111", "12223114", "32223112", "21323113", "41323111", "13114114", "33114112", "22214113", "42214111", "11314114", "31314112", "24141112", "13241113", "33241111", "22341112", "14132113", "34132111", "23232112", "12332113", "32332111", "21432112", "24123112", "13223113", "33223111", "22323112", "11423113", "31423111", "14114113", "34114111", "23214112", "12314113", "32314111", "21414112", "25141111", "14241112", "23341111", "15132112", "24232111", "13332112", "22432111", "25123111", "14223112", "23323111", "12423112", "21523111", "15114112", "24214111", "13314112", "22414111", "11514112", "15241111", "16132111", "14332111", "15223111", "13423111", "16114111", "14314111", "12514111", "21151114", "41151112", "11142115", "31142113", "51142111", "21133114", "41133112", "11124115", "31124113", "51124111", "21115114", "41115112", "22151113", "42151111", "11251114", "31251112", "12142114", "32142112", "21242113", "41242111", "22133113", "42133111", "11233114", "31233112", "12124114", "32124112", "21224113", "41224111", "22115113", "42115111", "11215114", "31215112", "23151112", "12251113", "32251111", "13142113", "33142111", "22242112", "11342113", "31342111", "23133112", "12233113", "32233111", "21333112", "13124113", "33124111", "22224112", "11324113", "31324111", "23115112", "12215113", "32215111", "21315112", "24151111", "13251112", "14142112", "23242111", "12342112", "24133111", "13233112", "22333111", "11433112", "14124112", "23224111", "12324112", "21424111", "24115111", "13215112", "22315111", "11415112", "14251111", "15142111", "13342111", "14233111", "12433111", "15124111", "13324111", "11524111", "14215111", "12415111", "21161113", "41161111", "11152114", "31152112", "21143113", "41143111", "11134114", "31134112", "21125113", "41125111", "11116114", "31116112", "22161112", "12152113", "32152111", "21252112", "22143112", "11243113", "31243111", "12134113", "32134111", "21234112", "22125112", "11225113", "31225111", "12116113", "32116111", "21216112", "23161111", "13152112", "22252111", "23143111", "12243112", "21343111", "13134112", "22234111", "11334112", "23125111", "12225112", "21325111", "13116112", "22216111", "11316112", "14152111", "13243111", "14134111", "12334111", "13225111", "11425111", "14116111", "12316111", "41111215", "61111213", "21211126", "41211124", "61211122", "22111216", "42111214", "31211215", "51211213", "22211125", "42211123", "11311126", "31311124", "51311122", "23111215", "43111213", "12211216", "32211214", "52211212", "21311215", "41311213", "61311211", "23211124", "43211122", "12311125", "32311123", "52311121", "21411124", "41411122", "24111214", "13211215", "33211213", "22311214", "42311212", "11411215", "31411213", "51411211", "24211123", "13311124", "33311122", "22411123", "42411121", "11511124", "31511122", "25111213", "14211214", "34211212", "23311213", "43311211", "12411214", "32411212", "21511213", "41511211", "25211122", "14311123", "34311121", "23411122", "12511123", "32511121", "21611122", "15211213", "24311212", "13411213", "33411211", "22511212", "11611213", "31611211", "31121125", "51121123", "21112126", "41112124", "61112122", "21121216", "41121214", "61121212", "31112215", "51112213", "12121126", "32121124", "52121122", "21221125", "41221123", "61221121", "22112125", "42112123", "11212126", "22121215", "42121213", "11221216", "31221214", "51221212", "12112216", "32112214", "52112212", "21212215", "41212213", "61212211", "13121125", "33121123", "22221124", "42221122", "11321125", "31321123", "51321121", "23112124", "43112122", "12212125", "23121214", "43121212", "12221215", "32221213", "52221211", "21321214", "41321212", "13112215", "33112213", "22212214", "42212212", "11312215", "31312213", "51312211", "14121124", "34121122", "23221123", "43221121", "12321124", "32321122", "21421123", "41421121", "24112123", "13212124", "24121213", "13221214", "33221212", "11412124", "22321213", "42321211", "11421214", "31421212", "14112214", "34112212", "23212213", "43212211", "12312214", "32312212", "21412213", "41412211", "15121123", "24221122", "13321123", "33321121", "22421122", "11521123", "31521121", "25112122", "14212123", "25121212", "14221213", "34221211", "12412123", "23321212", "12421213", "32421211", "21521212", "15112213", "24212212", "13312213", "33312211", "22412212", "11512213", "31512211", "16121122", "25221121", "14321122", "23421121", "12521122", "15212122", "15221212", "13412122", "24321211", "13421212", "11612122", "22521211", "16112212", "25212211", "14312212", "23412211", "12512212", "21612211", "11131126", "31131124", "51131122", "21122125", "41122123", "61122121", "21131215", "41131213", "61131211", "11113126", "31113124", "51113122", "11122216", "31122214", "51122212", "21113215", "41113213", "61113211", "12131125", "32131123", "52131121", "21231124", "41231122", "22122124", "42122122", "11222125", "22131214", "42131212", "11231215", "31231213", "51231211", "12113125", "32113123", "52113121", "12122215", "32122213", "52122211", "21222214", "41222212", "22113214", "42113212", "11213215", "31213213", "51213211", "13131124", "33131122", "22231123", "42231121", "11331124", "31331122", "23122123", "43122121", "12222124", "23131213", "43131211", "12231214", "32231212", "21331213", "41331211", "13113124", "33113122", "13122214", "33122212", "11313124", "22222213", "42222211", "11322214", "31322212", "23113213", "43113211", "12213214", "32213212", "21313213", "41313211", "14131123", "34131121", "23231122", "12331123", "32331121", "21431122", "24122122", "13222123", "24131212", "13231213", "33231211", "11422123", "22331212", "11431213", "31431211", "14113123", "34113121", "14122213", "34122211", "12313123", "23222212", "12322213", "32322211", "21422212", "24113212", "13213213", "33213211", "22313212", "11413213", "31413211", "15131122", "24231121", "13331122", "22431121", "25122121", "14222122", "25131211", "14231212", "12422122", "23331211", "12431212", "15113122", "15122212", "13313122", "24222211", "13322212", "11513122", "22422211", "11522212", "25113211", "14213212", "23313211", "12413212", "21513211", "16131121", "14331121", "15222121", "15231211", "13422121", "13431211", "16113121", "16122211", "14313121", "14322211", "12513121", "12522211", "15213211", "13413211", "11613211", "11141125", "31141123", "51141121", "21132124", "41132122", "21141214", "41141212", "11123125", "31123123", "51123121", "11132215", "31132213", "51132211", "21114124", "41114122", "21123214", "41123212", "11114215", "31114213", "51114211", "12141124", "32141122", "21241123", "41241121", "22132123", "42132121", "11232124", "22141213", "42141211", "11241214", "31241212", "12123124", "32123122", "12132214", "32132212", "21232213", "41232211", "22114123", "42114121", "11214124", "22123213", "42123211", "11223214", "31223212", "12114214", "32114212", "21214213", "41214211", "13141123", "33141121", "22241122", "11341123", "31341121", "23132122", "12232123", "23141212", "12241213", "32241211", "21341212", "13123123", "33123121", "13132213", "33132211", "11323123", "22232212", "11332213", "31332211", "23114122", "12214123", "23123212", "12223213", "32223211", "21323212", "13114213", "33114211", "22214212", "11314213", "31314211", "14141122", "23241121", "12341122", "24132121", "13232122", "24141211", "13241212", "11432122", "22341211", "14123122", "14132212", "12323122", "23232211", "12332212", "21432211", "24114121", "13214122", "24123211", "13223212", "11414122", "22323211", "11423212", "14114212", "23214211", "12314212", "21414211", "15141121", "13341121", "14232121", "14241211", "12432121", "15123121", "15132211", "13323121", "13332211", "11523121", "14214121", "14223211", "12414121", "12423211", "15114211", "13314211", "11514211", "11151124", "31151122", "21142123", "41142121", "21151213", "41151211", "11133124", "31133122", "11142214", "31142212", "21124123", "41124121", "21133213", "41133211", "11115124", "31115122", "11124214", "31124212", "21115213", "41115211", "12151123", "32151121", "21251122", "22142122", "11242123", "22151212", "11251213", "31251211", "12133123", "32133121", "12142213", "32142211", "21242212", "22124122", "11224123", "22133212", "11233213", "31233211", "12115123", "32115121", "12124213", "32124211", "21224212", "22115212", "11215213", "31215211", "13151122", "22251121", "23142121", "12242122", "23151211", "12251212", "13133122", "13142212", "11333122", "22242211", "11342212", "23124121", "12224122", "23133211", "12233212", "21333211", "13115122", "13124212", "11315122", "22224211", "11324212", "23115211", "12215212", "21315211", "14151121", "13242121", "13251211", "14133121", "14142211", "12333121", "12342211", "13224121", "13233211", "11424121", "11433211", "14115121", "14124211", "12315121", "12324211", "13215211", "11415211", "11161123", "31161121", "21152122", "21161212", "11143123", "31143121", "11152213", "31152211", "21134122", "21143212", "11125123", "31125121", "11134213", "31134211", "21116122", "21125212", "12161122", "22152121", "11252122", "22161211", "12143122", "12152212", "21252211", "22134121", "11234122", "22143211", "11243212", "12125122", "12134212", "21234211", "22116121", "11216122", "22125211", "11225212", "13161121", "12252121", "13143121", "13152211", "11343121", "12234121", "12243211", "13125121", "13134211", "11325121", "11334211", "12216121", "12225211", "31111225", "51111223", "21111316", "41111314", "61111312", "31211134", "51211132", "12111226", "32111224", "52111222", "21211225", "41211223", "61211221", "22111315", "42111313", "11211316", "31211314", "51211312", "12211135", "32211133", "52211131", "21311134", "41311132", "13111225", "33111223", "22211224", "42211222", "11311225", "31311223", "51311221", "23111314", "43111312", "12211315", "32211313", "52211311", "21311314", "41311312", "13211134", "33211132", "22311133", "42311131", "11411134", "31411132", "14111224", "34111222", "23211223", "43211221", "12311224", "32311222", "21411223", "41411221", "24111313", "13211314", "33211312", "22311313", "42311311", "11411314", "31411312", "14211133", "34211131", "23311132", "12411133", "32411131", "21511132", "15111223", "24211222", "13311223", "33311221", "22411222", "11511223", "31511221", "25111312", "14211313", "34211311", "23311312", "12411313", "32411311", "21511312", "15211132", "24311131", "13411132", "22511131", "11611132", "16111222", "25211221", "14311222", "23411221", "12511222", "21611221", "15211312", "24311311", "13411312", "22511311", "11611312", "21121135", "41121133", "61121131", "11112136", "31112134", "51112132", "11121226", "31121224", "51121222", "21112225", "41112223", "61112221", "21121315", "41121313", "61121311", "11112316", "31112314", "51112312", "22121134", "42121132", "11221135", "31221133", "51221131", "12112135", "32112133", "52112131", "12121225", "32121223", "52121221", "21221224", "41221222", "22112224", "42112222", "11212225", "22121314", "42121312", "11221315", "31221313", "51221311", "12112315", "32112313", "52112311", "21212314", "41212312", "23121133", "43121131", "12221134", "32221132", "21321133", "41321131", "13112134", "33112132", "13121224", "33121222", "11312134", "22221223", "42221221", "11321224", "31321222", "23112223", "43112221", "12212224", "23121313", "43121311", "12221314", "32221312", "21321313", "41321311", "13112314", "33112312", "22212313", "42212311", "11312314", "31312312", "24121132", "13221133", "33221131", "22321132", "11421133", "31421131", "14112133", "34112131", "14121223", "34121221", "12312133", "23221222", "12321223", "32321221", "21421222", "24112222", "13212223", "24121312", "13221313", "33221311", "11412223", "22321312", "11421313", "31421311", "14112313", "34112311", "23212312", "12312313", "32312311", "21412312", "25121131", "14221132", "23321131", "12421132", "21521131", "15112132", "15121222", "13312132", "24221221", "13321222", "11512132", "22421221", "11521222", "25112221", "14212222", "25121311", "14221312", "12412222", "23321311", "12421312", "21521311", "15112312", "24212311", "13312312", "22412311", "11512312", "15221131", "13421131", "16112131", "16121221", "14312131", "14321221", "12512131", "12521221", "15212221", "15221311", "13412221", "13421311", "11612221", "16112311", "14312311", "12512311", "21131134", "41131132", "11122135", "31122133", "51122131", "11131225", "31131223", "51131221", "21113134", "41113132", "21122224", "41122222", "21131314", "41131312", "11113225", "31113223", "51113221", "11122315", "31122313", "51122311", "21113314", "41113312", "22131133", "42131131", "11231134", "31231132", "12122134", "32122132", "12131224", "32131222", "21231223", "41231221", "22113133", "42113131", "11213134", "22122223", "42122221", "11222224", "22131313", "42131311", "11231314", "31231312", "12113224", "32113222", "12122314", "32122312", "21222313", "41222311", "22113313", "42113311", "11213314", "31213312", "23131132", "12231133", "32231131", "21331132", "13122133", "33122131", "13131223", "33131221", "11322133", "22231222", "11331223", "31331221", "23113132", "12213133", "23122222", "12222223", "23131312", "12231313", "32231311", "21331312", "13113223", "33113221", "13122313", "33122311", "11313223", "22222312", "11322313", "31322311", "23113312", "12213313", "32213311", "21313312", "24131131", "13231132", "22331131", "11431132", "14122132", "14131222", "12322132", "23231221", "12331222", "21431221", "24113131", "13213132", "24122221", "13222222", "24131311", "11413132", "13231312", "11422222", "22331311", "11431312", "14113222", "14122312", "12313222", "23222311", "12322312", "21422311", "24113311", "13213312", "22313311", "11413312", "14231131", "12431131", "15122131", "15131221", "13322131", "13331221", "11522131", "14213131", "14222221", "12413131", "14231311", "12422221", "12431311", "15113221", "15122311", "13313221", "13322311", "11513221", "11522311", "14213311", "12413311", "21141133", "41141131", "11132134", "31132132", "11141224", "31141222", "21123133", "41123131", "21132223", "41132221", "21141313", "41141311", "11114134", "31114132", "11123224", "31123222", "11132314", "31132312", "21114223", "41114221", "21123313", "41123311", "11114314", "31114312", "22141132", "11241133", "31241131", "12132133", "32132131", "12141223", "32141221", "21241222", "22123132", "11223133", "22132222", "11232223", "22141312", "11241313", "31241311", "12114133", "32114131", "12123223", "32123221", "12132313", "32132311", "21232312", "22114222", "11214223", "22123312", "11223313", "31223311", "12114313", "32114311", "21214312", "23141131", "12241132", "21341131", "13132132", "13141222", "11332132", "22241221", "11341222", "23123131", "12223132", "23132221", "12232222", "23141311", "12241312", "21341311", "13114132", "13123222", "11314132", "13132312", "11323222", "22232311", "11332312", "23114221", "12214222", "23123311", "12223312", "21323311", "13114312", "22214311", "11314312", "13241131", "14132131", "14141221", "12332131", "12341221", "13223131", "13232221", "11423131", "13241311", "11432221", "14114131", "14123221", "12314131", "14132311", "12323221", "12332311", "13214221", "13223311", "11414221", "11423311", "14114311", "12314311", "21151132", "11142133", "31142131", "11151223", "31151221", "21133132", "21142222", "21151312", "11124133", "31124131", "11133223", "31133221", "11142313", "31142311", "21115132", "21124222", "21133312", "11115223", "31115221", "11124313", "31124311", "22151131", "11251132", "12142132", "12151222", "21251221", "22133131", "11233132", "22142221", "11242222", "22151311", "11251312", "12124132", "12133222", "12142312", "21242311", "22115131", "11215132", "22124221", "11224222", "22133311", "11233312", "12115222", "12124312", "21224311", "12251131", "13142131", "13151221", "11342131", "12233131", "12242221", "12251311", "13124131", "13133221", "11324131", "13142311", "11333221", "11342311", "12215131", "12224221", "12233311", "13115221", "13124311", "11315221", "11324311", "21161131", "11152132", "11161222", "21143131", "21152221", "21161311", "11134132", "11143222", "11152312", "21125131", "21134221", "21143311", "11116132", "11125222", "11134312", "12152131", "12161221", "11243131", "11252221", "12134131", "12143221", "12152311", "11225131", "11234221", "11243311", "12116131", "12125221", "12134311", "21111235", "41111233", "61111231", "11111326", "31111324", "51111322", "21111415", "41111413", "61111411", "21211144", "41211142", "22111234", "42111232", "11211235", "31211233", "51211231", "12111325", "32111323", "52111321", "21211324", "41211322", "22111414", "42111412", "11211415", "31211413", "51211411", "22211143", "42211141", "11311144", "31311142", "23111233", "43111231", "12211234", "32211232", "21311233", "41311231", "13111324", "33111322", "22211323", "42211321", "11311324", "31311322", "23111413", "43111411", "12211414", "32211412", "21311413", "41311411", "23211142", "12311143", "32311141", "21411142", "24111232", "13211233", "33211231", "22311232", "11411233", "31411231", "14111323", "34111321", "23211322", "12311323", "32311321", "21411322", "24111412", "13211413", "33211411", "22311412", "11411413", "31411411", "24211141", "13311142", "22411141", "11511142", "25111231", "14211232", "23311231", "12411232", "21511231", "15111322", "24211321", "13311322", "22411321", "11511322", "25111411", "14211412", "23311411", "12411412", "21511411", "14311141", "12511141", "15211231", "13411231", "11611231", "16111321", "14311321", "12511321", "15211411", "13411411", "11611411", "31121143", "51121141", "21112144", "41112142", "21121234", "41121232", "11112235", "31112233", "51112231", "11121325", "31121323", "51121321", "21112324", "41112322", "21121414", "41121412", "11112415", "31112413", "51112411", "12121144", "32121142", "21221143", "41221141", "22112143", "42112141", "11212144", "22121233", "42121231", "11221234", "31221232", "12112234", "32112232", "12121324", "32121322", "21221323", "41221321", "22112323", "42112321", "11212324", "22121413", "42121411", "11221414", "31221412", "12112414", "32112412", "21212413", "41212411", "13121143", "33121141", "22221142", "11321143", "31321141", "23112142", "12212143", "23121232", "12221233", "32221231", "21321232", "13112233", "33112231", "13121323", "33121321", "11312233", "22221322", "11321323", "31321321", "23112322", "12212323", "23121412", "12221413", "32221411", "21321412", "13112413", "33112411", "22212412", "11312413", "31312411", "14121142", "23221141", "12321142", "21421141", "24112141", "13212142", "24121231", "13221232", "11412142", "22321231", "11421232", "14112232", "14121322", "12312232", "23221321", "12321322", "21421321", "24112321", "13212322", "24121411", "13221412", "11412322", "22321411", "11421412", "14112412", "23212411", "12312412", "21412411", "15121141", "13321141", "11521141", "14212141", "14221231", "12412141", "12421231", "15112231", "15121321", "13312231", "13321321", "11512231", "11521321", "14212321", "14221411", "12412321", "12421411", "15112411", "13312411", "11512411", "11131144", "31131142", "21122143", "41122141", "21131233", "41131231", "11113144", "31113142", "11122234", "31122232", "11131324", "31131322", "21113233", "41113231", "21122323", "41122321", "21131413", "41131411", "11113324", "31113322", "11122414", "31122412", "21113413", "41113411", "12131143", "32131141", "21231142", "22122142", "11222143", "22131232", "11231233", "31231231", "12113143", "32113141", "12122233", "32122231", "12131323", "32131321", "21231322", "22113232", "11213233", "22122322", "11222323", "22131412", "11231413", "31231411", "12113323", "32113321", "12122413", "32122411", "21222412", "22113412", "11213413", "31213411", "13131142", "22231141", "11331142", "23122141", "12222142", "23131231", "12231232", "21331231", "13113142", "13122232", "11313142", "13131322", "11322232", "22231321", "11331322", "23113231", "12213232", "23122321", "12222322", "23131411", "12231412", "21331411", "13113322", "13122412", "11313322", "22222411", "11322412", "23113411", "12213412", "21313411", "14131141", "12331141", "13222141", "13231231", "11422141", "11431231", "14113141", "14122231", "12313141", "14131321", "12322231", "12331321", "13213231", "13222321", "11413231", "13231411", "11422321", "11431411", "14113321", "14122411", "12313321", "12322411", "13213411", "11413411", "11141143", "31141141", "21132142", "21141232", "11123143", "31123141", "11132233", "31132231", "11141323", "31141321", "21114142", "21123232", "21132322", "21141412", "11114233", "31114231", "11123323", "31123321", "11132413", "31132411", "21114322", "21123412", "12141142", "21241141", "22132141", "11232142", "22141231", "11241232", "12123142", "12132232", "12141322", "21241321", "22114141", "11214142", "22123231", "11223232", "22132321", "11232322", "22141411", "11241412", "12114232", "12123322", "12132412", "21232411", "22114321", "11214322", "22123411", "11223412", "13141141", "11341141", "12232141", "12241231", "13123141", "13132231", "11323141", "13141321", "11332231", "11341321", "12214141", "12223231", "12232321", "12241411", "13114231", "13123321", "11314231", "13132411", "11323321", "11332411", "12214321", "12223411", "11151142", "21142141", "21151231", "11133142", "11142232", "11151322", "21124141", "21133231", "21142321", "21151411", "11115142", "11124232", "11133322", "11142412", "21115231", "21124321", "21133411", "12151141", "11242141", "11251231", "12133141", "12142231", "12151321", "11224141", "11233231", "11242321", "11251411", "12115141", "12124231", "12133321", "12142411", "11215231", "11224321", "11233411", "11161141", "11143141", "11152231", "11161321", "11125141", "11134231", "11143321", "11152411", "11111245", "31111243", "51111241", "21111334", "41111332", "11111425", "31111423", "51111421", "21111514", "41111512", "31211152", "12111244", "32111242", "21211243", "41211241", "22111333", "42111331", "11211334", "31211332", "12111424", "32111422", "21211423", "41211421", "22111513", "42111511", "11211514", "31211512", "12211153", "32211151", "21311152", "13111243", "33111241", "22211242", "11311243", "31311241", "23111332", "12211333", "32211331", "21311332", "13111423", "33111421", "22211422", "11311423", "31311421", "23111512", "12211513", "32211511", "21311512", "13211152", "22311151", "11411152", "14111242", "23211241", "12311242", "21411241", "24111331", "13211332", "22311331", "11411332", "14111422", "23211421", "12311422", "21411421", "24111511", "13211512", "22311511", "11411512", "14211151", "12411151", "15111241", "13311241", "11511241", "14211331", "12411331", "15111421", "13311421", "11511421", "14211511", "12411511", "21121153", "41121151", "11112154", "31112152", "11121244", "31121242", "21112243", "41112241", "21121333", "41121331", "11112334", "31112332", "11121424", "31121422", "21112423", "41112421", "21121513", "41121511", "11112514", "31112512", "22121152", "11221153", "31221151", "12112153", "32112151", "12121243", "32121241", "21221242", "22112242", "11212243", "22121332", "11221333", "31221331", "12112333", "32112331", "12121423", "32121421", "21221422", "22112422", "11212423", "22121512", "11221513", "31221511", "12112513", "32112511", "21212512", "23121151", "12221152", "21321151", "13112152", "13121242", "11312152", "22221241", "11321242", "23112241", "12212242", "23121331", "12221332", "21321331", "13112332", "13121422", "11312332", "22221421", "11321422", "23112421", "12212422", "23121511", "12221512", "21321511", "13112512", "22212511", "11312512", "13221151", "11421151", "14112151", "14121241", "12312151", "12321241", "13212241", "13221331", "11412241", "11421331", "14112331", "14121421", "12312331", "12321421", "13212421", "13221511", "11412421", "11421511", "14112511", "12312511", "21131152", "11122153", "31122151", "11131243", "31131241", "21113152", "21122242", "21131332", "11113243", "31113241", "11122333", "31122331", "11131423", "31131421", "21113332", "21122422", "21131512", "11113423", "31113421", "11122513", "31122511", "22131151", "11231152", "12122152", "12131242", "21231241", "22113151", "11213152", "22122241", "11222242", "22131331", "11231332", "12113242", "12122332", "12131422", "21231421", "22113331", "11213332", "22122421", "11222422", "22131511", "11231512", "12113422", "12122512", "21222511", "12231151", "13122151", "13131241", "11322151", "11331241", "12213151", "12222241", "12231331", "13113241", "13122331", "11313241", "13131421", "11322331", "11331421", "12213331", "12222421", "12231511", "13113421", "13122511", "11313421", "11322511", "21141151", "11132152", "11141242", "21123151", "21132241", "21141331", "11114152", "11123242", "11132332", "11141422", "21114241", "21123331", "21132421", "21141511", "11114332", "11123422", "11132512", "11241151", "12132151", "12141241", "11223151", "11232241", "11241331", "12114151", "12123241", "12132331", "12141421", "11214241", "11223331", "11232421", "11241511", "12114331", "12123421", "12132511", "11142151", "11151241", "11124151", "11133241", "11142331", "11151421", "11115241", "11124331", "11133421", "11142511", "21111253", "41111251", "11111344", "31111342", "21111433", "41111431", "11111524", "31111522", "21111613", "41111611", "21211162", "22111252", "11211253", "31211251", "12111343", "32111341", "21211342", "22111432", "11211433", "31211431", "12111523", "32111521", "21211522", "22111612", "11211613", "31211611", "22211161", "11311162", "23111251", "12211252", "21311251", "13111342", "22211341", "11311342", "23111431", "12211432", "21311431", "13111522", "22211521", "11311522", "23111611", "12211612", "21311611", "12311161", "13211251", "11411251", "14111341", "12311341", "13211431", "11411431", "14111521", "12311521", "13211611", "11411611", "31121161", "21112162", "21121252", "11112253", "31112251", "11121343", "31121341", "21112342", "21121432", "11112433", "31112431", "11121523", "31121521", "21112522", "21121612", "12121162", "21221161", "22112161", "11212162", "22121251", "11221252", "12112252", "12121342", "21221341", "22112341", "11212342", "22121431", "11221432", "12112432", "12121522", "21221521", "22112521", "11212522", "22121611", "11221612", "13121161", "11321161", "12212161", "12221251", "13112251", "13121341", "11312251", "11321341", "12212341", "12221431", "13112431", "13121521", "11312431", "11321521", "12212521", "12221611", "11131162", "21122161", "21131251", "11113162"])]); //#14443
        $1.parity = $a(["1001", "0101", "1100", "0011", "1010", "0110", "1111", "0000"]); //#14447
        $1.rowbits = $a($1.r); //#14450
        for (var _5z = 0, _5y = $f($1.r - 1); _5z <= _5y; _5z += 1) { //#14470
            $1.i = _5z; //#14452
            $k[$j++] = 'p'; //#14453
            if ($1.i != $f($1.r - 1)) { //#14453
                $k[$j++] = $get($1.parity, $1.i); //#14453
            } else { //#14453
                $k[$j++] = "0000"; //#14453
            } //#14453
            var _65 = $k[--$j]; //#14453
            $1[$k[--$j]] = _65; //#14453
            $1.ccrow = $geti($1.ccs, $1.i * 8, 8); //#14454
            $k[$j++] = Infinity; //#14456
            for (var _6A = 0; _6A <= 7; _6A += 2) { //#14456
                $aload($geti($1.ccrow, _6A, 2)); //#14456
                var _6D = $k[--$j]; //#14456
                var _6E = $k[--$j]; //#14456
                $k[$j++] = $f(_6D + (_6E * 49)); //#14456
            } //#14456
            $1.scrow = $a(); //#14456
            $k[$j++] = Infinity; //#14464
            $k[$j++] = 10; //#14463
            $k[$j++] = 1; //#14463
            $k[$j++] = 1; //#14463
            for (var _6G = 0; _6G <= 3; _6G += 1) { //#14463
                $1.j = _6G; //#14461
                $forall($get($get($1.patterns, $f($get($1.p, $1.j) - 48)), $get($1.scrow, $1.j)), function() { //#14462
                    var _6Q = $k[--$j]; //#14462
                    $k[$j++] = $f(_6Q - 48); //#14462
                }); //#14462
            } //#14462
            $k[$j++] = 4; //#14464
            $k[$j++] = 1; //#14464
            $1.sbs = $a(); //#14464
            $k[$j++] = Infinity; //#14466
            var _6S = $1.sbs; //#14467
            $k[$j++] = 1; //#14467
            for (var _6T = 0, _6U = _6S.length; _6T < _6U; _6T++) { //#14467
                var _6W = $k[--$j]; //#14467
                var _6X = (_6W == 0) ? 1 : 0; //#14467
                $k[$j++] = _6W; //#14467
                for (var _6Y = 0, _6Z = $get(_6S, _6T); _6Y < _6Z; _6Y++) { //#14467
                    $k[$j++] = _6X //#14467
                } //#14467
            } //#14467
            $astore($a($counttomark() - 1)); //#14468
            var _6c = $k[--$j]; //#14468
            var _6d = $k[--$j]; //#14468
            $k[$j++] = _6c; //#14468
            $k[$j++] = _6d; //#14468
            $j--; //#14468
            var _6e = $k[--$j]; //#14468
            var _6f = $k[--$j]; //#14468
            $k[$j++] = _6e; //#14468
            $k[$j++] = _6f; //#14468
            $j--; //#14468
            $put($1.rowbits, $1.i, $k[--$j]); //#14469
        } //#14469
        $k[$j++] = Infinity; //#14481
        for (var _6k = 0, _6l = 81 * $1.sepheight; _6k < _6l; _6k++) { //#14474
            $k[$j++] = 1; //#14474
        } //#14474
        for (var _6o = 0, _6n = $f($1.r - 2); _6o <= _6n; _6o += 1) { //#14479
            $1.i = _6o; //#14476
            for (var _6q = 0, _6r = $1.rowheight; _6q < _6r; _6q++) { //#14477
                $aload($get($1.rowbits, $1.i)); //#14477
            } //#14477
            for (var _6w = 0, _6x = $1.sepheight; _6w < _6x; _6w++) { //#14478
                for (var _6y = 0, _6z = 10; _6y < _6z; _6y++) { //#14478
                    $k[$j++] = 0; //#14478
                } //#14478
                for (var _70 = 0, _71 = 70; _70 < _71; _70++) { //#14478
                    $k[$j++] = 1; //#14478
                } //#14478
                $k[$j++] = 0; //#14478
            } //#14478
        } //#14478
        for (var _73 = 0, _74 = $1.rowheight; _73 < _74; _73++) { //#14480
            $aload($get($1.rowbits, $f($1.r - 1))); //#14480
        } //#14480
        for (var _79 = 0, _7A = 81 * $1.sepheight; _79 < _7A; _79++) { //#14481
            $k[$j++] = 1; //#14481
        } //#14481
        $1.pixs = $a(); //#14481
        var _7G = new Map([
            ["ren", $0.renmatrix],
            ["pixs", $1.pixs],
            ["pixx", 81],
            ["pixy", ~~($1.pixs.length / 81)],
            ["height", (~~($1.pixs.length / 81)) / 72],
            ["width", 81 / 72],
            ["opt", $1.options]
        ]); //#14492
        $k[$j++] = _7G; //#14495
        if (!$1.dontdraw) { //#14495
            $0.renmatrix(); //#14495
        } //#14495
    };
    $0.flattermarken = function() {
        var $1 = {}; //#14786
        $1.options = $k[--$j]; //#14788
        $1.barcode = $k[--$j]; //#14789
        $1.dontdraw = false; //#14791
        $1.includetext = false; //#14792
        $1.textfont = "OCR-B"; //#14793
        $1.textsize = 10; //#14794
        $1.textyoffset = -8; //#14795
        $1.height = 0.3; //#14796
        $forall($1.barcode, function() { //#14803
            var _3 = $k[--$j]; //#14800
            if ((_3 < 48) || (_3 > 57)) { //#14802
                $k[$j++] = 'bwipp.flattermarkenBadCharacter'; //#14801
                $k[$j++] = "Flattermarken must contain only digits"; //#14801
                $0.raiseerror(); //#14801
            } //#14801
        }); //#14801
        $forall($1.options, function() { //#14814
            var _5 = $k[--$j]; //#14814
            $1[$k[--$j]] = _5; //#14814
        }); //#14814
        $1.textfont = "" + $1.textfont; //#14816
        $1.textsize = +$1.textsize; //#14817
        $1.textyoffset = +$1.textyoffset; //#14818
        $1.height = +$1.height; //#14819
        $forall($1.barcode, function() { //#14826
            var _C = $k[--$j]; //#14823
            if ((_C < 48) || (_C > 57)) { //#14825
                $k[$j++] = 'bwipp.flattermarkenBadCharacter'; //#14824
                $k[$j++] = "Flattermarken must contain only digits"; //#14824
                $0.raiseerror(); //#14824
            } //#14824
        }); //#14824
        $1.encs = $a(["0018", "0117", "0216", "0315", "0414", "0513", "0612", "0711", "0810", "0900"]); //#14832
        $1.barchars = "1234567890"; //#14835
        $1.barlen = $1.barcode.length; //#14837
        $1.sbs = $s($1.barlen * 4); //#14839
        $1.txt = $a($1.barlen); //#14840
        for (var _L = 0, _K = $1.barlen - 1; _L <= _K; _L += 1) { //#14852
            $1.i = _L; //#14843
            $search($1.barchars, $geti($1.barcode, $1.i, 1)); //#14845
            $j--; //#14846
            $1.indx = $k[--$j].length; //#14847
            $j -= 2; //#14848
            $1.enc = $get($1.encs, $1.indx); //#14849
            $puti($1.sbs, $1.i * 4, $1.enc); //#14850
            $put($1.txt, $1.i, $a([$geti($1.barcode, $1.i, 1), $1.i * 9, $1.textyoffset, $1.textfont, $1.textsize])); //#14851
        } //#14851
        $k[$j++] = Infinity; //#14861
        $k[$j++] = Infinity; //#14857
        var _h = $1.sbs; //#14857
        for (var _i = 0, _j = _h.length; _i < _j; _i++) { //#14857
            $k[$j++] = $get(_h, _i) - 48; //#14857
        } //#14857
        var _l = $a(); //#14857
        $k[$j++] = Infinity; //#14858
        for (var _n = 0, _o = ~~(($1.sbs.length + 1) / 2); _n < _o; _n++) { //#14858
            $k[$j++] = $1.height; //#14858
        } //#14858
        var _q = $a(); //#14858
        $k[$j++] = Infinity; //#14859
        for (var _s = 0, _t = ~~(($1.sbs.length + 1) / 2); _s < _t; _s++) { //#14859
            $k[$j++] = 0; //#14859
        } //#14859
        var _u = $a(); //#14859
        $k[$j++] = 'ren'; //#14861
        $k[$j++] = $0.renlinear; //#14861
        $k[$j++] = 'sbs'; //#14861
        $k[$j++] = _l; //#14861
        $k[$j++] = 'bhs'; //#14861
        $k[$j++] = _q; //#14861
        $k[$j++] = 'bbs'; //#14861
        $k[$j++] = _u; //#14861
        $k[$j++] = 'txt'; //#14861
        $k[$j++] = $1.txt; //#14861
        $k[$j++] = 'opt'; //#14861
        $k[$j++] = $1.options; //#14861
        var _x = $d(); //#14861
        $k[$j++] = _x; //#14864
        if (!$1.dontdraw) { //#14864
            $0.renlinear(); //#14864
        } //#14864
    };
    $0.raw = function() {
        var $1 = {}; //#14893
        $1.options = $k[--$j]; //#14895
        $1.barcode = $k[--$j]; //#14896
        $1.dontdraw = false; //#14898
        $1.height = 1; //#14899
        $forall($1.options, function() { //#14910
            var _3 = $k[--$j]; //#14910
            $1[$k[--$j]] = _3; //#14910
        }); //#14910
        $1.height = +$1.height; //#14912
        $forall($1.barcode, function() { //#14919
            var _7 = $k[--$j]; //#14916
            if ((_7 < 49) || (_7 > 57)) { //#14918
                $k[$j++] = 'bwipp.rawBadCharacter'; //#14917
                $k[$j++] = "Raw must contain only digits 1 to 9"; //#14917
                $0.raiseerror(); //#14917
            } //#14917
        }); //#14917
        $k[$j++] = Infinity; //#14927
        $k[$j++] = Infinity; //#14924
        $forall($1.barcode, function() { //#14924
            var _9 = $k[--$j]; //#14924
            $k[$j++] = $f(_9 - 48); //#14924
        }); //#14924
        var _A = $a(); //#14924
        $k[$j++] = Infinity; //#14925
        for (var _C = 0, _D = ~~(($1.barcode.length + 1) / 2); _C < _D; _C++) { //#14925
            $k[$j++] = $1.height; //#14925
        } //#14925
        var _F = $a(); //#14925
        $k[$j++] = Infinity; //#14926
        for (var _H = 0, _I = ~~(($1.barcode.length + 1) / 2); _H < _I; _H++) { //#14926
            $k[$j++] = 0; //#14926
        } //#14926
        var _J = $a(); //#14926
        $k[$j++] = 'ren'; //#14927
        $k[$j++] = $0.renlinear; //#14927
        $k[$j++] = 'sbs'; //#14927
        $k[$j++] = _A; //#14927
        $k[$j++] = 'bhs'; //#14927
        $k[$j++] = _F; //#14927
        $k[$j++] = 'bbs'; //#14927
        $k[$j++] = _J; //#14927
        $k[$j++] = 'opt'; //#14927
        $k[$j++] = $1.options; //#14927
        var _L = $d(); //#14927
        $k[$j++] = _L; //#14930
        if (!$1.dontdraw) { //#14930
            $0.renlinear(); //#14930
        } //#14930
    };
    $0.daft = function() {
        var $1 = {}; //#14959
        $1.options = $k[--$j]; //#14961
        $1.barcode = $k[--$j]; //#14962
        $1.dontdraw = false; //#14964
        $1.height = 0.175; //#14965
        $forall($1.options, function() { //#14976
            var _3 = $k[--$j]; //#14976
            $1[$k[--$j]] = _3; //#14976
        }); //#14976
        $1.height = +$1.height; //#14978
        $forall($1.barcode, function() { //#14984
            var _7 = $k[--$j]; //#14981
            if ((_7 != 68) && ((_7 != 65) && ((_7 != 70) && (_7 != 84)))) { //#14983
                $k[$j++] = 'bwipp.daftBadCharacter'; //#14982
                $k[$j++] = "DAFT must contain only characters D, A, F and T"; //#14982
                $0.raiseerror(); //#14982
            } //#14982
        }); //#14982
        $1.barlen = $1.barcode.length; //#14986
        $1.bbs = $a($1.barlen); //#14988
        $1.bhs = $a($1.barlen); //#14989
        for (var _F = 0, _E = $1.barlen - 1; _F <= _E; _F += 1) { //#15009
            $1.i = _F; //#14991
            $1.enc = $geti($1.barcode, $1.i, 1); //#14992
            if ($eq($1.enc, "D")) { //#14996
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#14994
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#14995
            } //#14995
            if ($eq($1.enc, "A")) { //#15000
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#14998
                $put($1.bhs, $1.i, (5 * $1.height) / 8); //#14999
            } //#14999
            if ($eq($1.enc, "F")) { //#15004
                $put($1.bbs, $1.i, (0 * $1.height) / 8); //#15002
                $put($1.bhs, $1.i, (8 * $1.height) / 8); //#15003
            } //#15003
            if ($eq($1.enc, "T")) { //#15008
                $put($1.bbs, $1.i, (3 * $1.height) / 8); //#15006
                $put($1.bhs, $1.i, (2 * $1.height) / 8); //#15007
            } //#15007
        } //#15007
        $k[$j++] = Infinity; //#15017
        var _l = $1.bbs; //#15014
        var _m = $1.bhs; //#15015
        $k[$j++] = Infinity; //#15016
        for (var _o = 0, _p = $1.bhs.length - 1; _o < _p; _o++) { //#15016
            $k[$j++] = 1.44; //#15016
            $k[$j++] = 1.872; //#15016
        } //#15016
        $k[$j++] = 1.44; //#15016
        var _q = $a(); //#15016
        $k[$j++] = 'ren'; //#15017
        $k[$j++] = $0.renlinear; //#15017
        $k[$j++] = 'bbs'; //#15017
        $k[$j++] = _l; //#15017
        $k[$j++] = 'bhs'; //#15017
        $k[$j++] = _m; //#15017
        $k[$j++] = 'sbs'; //#15017
        $k[$j++] = _q; //#15017
        $k[$j++] = 'opt'; //#15017
        $k[$j++] = $1.options; //#15017
        var _s = $d(); //#15017
        $k[$j++] = _s; //#15020
        if (!$1.dontdraw) { //#15020
            $0.renlinear(); //#15020
        } //#15020
    };
    $0.symbol = function() {
        var $1 = {}; //#15049
        $1.options = $k[--$j]; //#15051
        $1.barcode = $k[--$j]; //#15052
        $1.dontdraw = false; //#15054
        $forall($1.options, function() { //#15065
            var _3 = $k[--$j]; //#15065
            $1[$k[--$j]] = _3; //#15065
        }); //#15065
        var _H = new Map([
            ["fima", function() {
                $1.sbs = $a([2.25, 2.25, 2.25, 11.25, 2.25, 11.25, 2.25, 2.25, 2.25]); //#15069
                $1.bhs = $a([0.625, 0.625, 0.625, 0.625, 0.625]); //#15070
                $1.bbs = $a([0, 0, 0, 0, 0]); //#15071
            }],
            ["fimb", function() {
                $1.sbs = $a([2.25, 6.75, 2.25, 2.25, 2.25, 6.25, 2.25, 2.25, 2.25, 6.75, 2.25]); //#15074
                $1.bhs = $a([0.625, 0.625, 0.625, 0.625, 0.625, 0.625]); //#15075
                $1.bbs = $a([0, 0, 0, 0, 0, 0]); //#15076
            }],
            ["fimc", function() {
                $1.sbs = $a([2.25, 2.25, 2.25, 6.75, 2.25, 6.75, 2.25, 6.75, 2.25, 2.25, 2.25]); //#15079
                $1.bhs = $a([0.625, 0.625, 0.625, 0.625, 0.625, 0.625]); //#15080
                $1.bbs = $a([0, 0, 0, 0, 0, 0]); //#15081
            }],
            ["fimd", function() {
                $1.sbs = $a([2.25, 2.25, 2.25, 2.25, 2.25, 6.75, 2.25, 6.75, 2.25, 2.25, 2.25, 2.25, 2.25]); //#15084
                $1.bhs = $a([0.625, 0.625, 0.625, 0.625, 0.625, 0.625, 0.625]); //#15085
                $1.bbs = $a([0, 0, 0, 0, 0, 0, 0]); //#15086
            }]
        ]); //#15086
        $1.symbols = _H; //#15088
        var _K = $get($1.symbols, $1.barcode) !== undefined; //#15091
        if (!_K) { //#15093
            $k[$j++] = 'bwipp.symbolUnknownSymbol'; //#15092
            $k[$j++] = "Unknown symbol name provided"; //#15092
            $0.raiseerror(); //#15092
        } //#15092
        if ($get($1.symbols, $1.barcode)() === true) {
            return true;
        } //#15095
        var _S = new Map([
            ["ren", $0.renlinear],
            ["sbs", $1.sbs],
            ["bhs", $1.bhs],
            ["bbs", $1.bbs],
            ["opt", $1.options]
        ]); //#15103
        $k[$j++] = _S; //#15106
        if (!$1.dontdraw) { //#15106
            $0.renlinear(); //#15106
        } //#15106
    };
    $0.pdf417 = function() {
        var $1 = {}; //#15136
        $1.options = $k[--$j]; //#15138
        $1.barcode = $k[--$j]; //#15139
        $1.dontdraw = false; //#15141
        $1.compact = false; //#15142
        $1.eclevel = -1; //#15143
        $1.columns = 0; //#15144
        $1.rows = 0; //#15145
        $1.rowmult = 3; //#15146
        $1.encoding = "auto"; //#15147
        $1.ccc = false; //#15148
        $1.raw = false; //#15149
        $1.parse = false; //#15150
        $1.parsefnc = false; //#15151
        $forall($1.options, function() { //#15162
            var _3 = $k[--$j]; //#15162
            $1[$k[--$j]] = _3; //#15162
        }); //#15162
        $1.eclevel = ~~$1.eclevel; //#15164
        $1.columns = ~~$1.columns; //#15165
        $1.rows = ~~$1.rows; //#15166
        $1.rowmult = +$1.rowmult; //#15167
        if ($1.raw) { //#15169
            $1.encoding = "raw"; //#15169
        } //#15169
        if ($eq($1.encoding, "raw")) { //#15183
            $1.datcws = $a($1.barcode.length); //#15173
            $1.i = 0; //#15174
            $1.j = 0; //#15174
            for (;;) { //#15181
                if ($1.i == $1.barcode.length) { //#15176
                    break; //#15176
                } //#15176
                $1.cw = ~~$z($geti($1.barcode, $1.i + 1, 3)); //#15177
                $put($1.datcws, $1.j, $1.cw); //#15178
                $1.i = $1.i + 4; //#15179
                $1.j = $1.j + 1; //#15180
            } //#15180
            $1.datcws = $geti($1.datcws, 0, $1.j); //#15182
        } //#15182
        if ($1.ccc) { //#15185
            $1.encoding = "ccc"; //#15185
        } //#15185
        $1.encb = function() {
            $1.in = $k[--$j]; //#15188
            $1.inlen = $1.in.length; //#15189
            $1.out = $a(((~~($1.inlen / 6)) * 5) + ($1.inlen % 6)); //#15190
            for (var _Y = 0, _X = (~~($1.inlen / 6)) - 1; _Y <= _X; _Y += 1) { //#15213
                $1.k = _Y; //#15192
                $k[$j++] = Infinity; //#15193
                $aload($geti($1.in, $1.k * 6, 3)); //#15193
                $1.msbs = $a(); //#15193
                $k[$j++] = Infinity; //#15196
                $aload($1.msbs); //#15195
                var _e = $k[--$j]; //#15195
                var _f = $k[--$j]; //#15195
                var _g = $k[--$j]; //#15195
                $k[$j++] = $f(($f(_e + (_f * 256))) + (_g * 65536)); //#15196
                for (var _h = 0, _i = 3; _h < _i; _h++) { //#15196
                    var _j = $k[--$j]; //#15196
                    $k[$j++] = _j % 900; //#15196
                    $k[$j++] = ~~(_j / 900); //#15196
                } //#15196
                $1.mscs = $a(); //#15196
                $k[$j++] = Infinity; //#15198
                $aload($geti($1.in, ($1.k * 6) + 3, 3)); //#15198
                $1.lsbs = $a(); //#15198
                $k[$j++] = Infinity; //#15201
                $aload($1.lsbs); //#15200
                var _q = $k[--$j]; //#15200
                var _r = $k[--$j]; //#15200
                var _s = $k[--$j]; //#15200
                $k[$j++] = $f(($f(_q + (_r * 256))) + (_s * 65536)); //#15201
                for (var _t = 0, _u = 3; _t < _u; _t++) { //#15201
                    var _v = $k[--$j]; //#15201
                    $k[$j++] = _v % 900; //#15201
                    $k[$j++] = ~~(_v / 900); //#15201
                } //#15201
                $1.lscs = $a(); //#15201
                var _y = $get($1.lscs, 0); //#15203
                var _10 = $get($1.mscs, 0); //#15203
                $put($1.out, ($1.k * 5) + 4, ($f(_y + (_10 * 316))) % 900); //#15204
                var _14 = $get($1.lscs, 1); //#15205
                var _16 = $get($1.mscs, 0); //#15205
                var _18 = $get($1.mscs, 1); //#15205
                $put($1.out, ($1.k * 5) + 3, ($f(($f(($f((~~(($f(_y + (_10 * 316))) / 900)) + _14)) + (_16 * 641))) + (_18 * 316))) % 900); //#15206
                var _1C = $get($1.lscs, 2); //#15207
                var _1E = $get($1.mscs, 0); //#15207
                var _1G = $get($1.mscs, 1); //#15207
                var _1I = $get($1.mscs, 2); //#15207
                $put($1.out, ($1.k * 5) + 2, ($f(($f(($f(($f((~~(($f(($f(($f((~~(($f(_y + (_10 * 316))) / 900)) + _14)) + (_16 * 641))) + (_18 * 316))) / 900)) + _1C)) + (_1E * 20))) + (_1G * 641))) + (_1I * 316))) % 900); //#15208
                var _1M = $get($1.lscs, 3); //#15209
                var _1O = $get($1.mscs, 1); //#15209
                var _1Q = $get($1.mscs, 2); //#15209
                $put($1.out, ($1.k * 5) + 1, ($f(($f(($f((~~(($f(($f(($f(($f((~~(($f(($f(($f((~~(($f(_y + (_10 * 316))) / 900)) + _14)) + (_16 * 641))) + (_18 * 316))) / 900)) + _1C)) + (_1E * 20))) + (_1G * 641))) + (_1I * 316))) / 900)) + _1M)) + (_1O * 20))) + (_1Q * 641))) % 900); //#15210
                $put($1.out, $1.k * 5, ($f((~~(($f(($f(($f((~~(($f(($f(($f(($f((~~(($f(($f(($f((~~(($f(_y + (_10 * 316))) / 900)) + _14)) + (_16 * 641))) + (_18 * 316))) / 900)) + _1C)) + (_1E * 20))) + (_1G * 641))) + (_1I * 316))) / 900)) + _1M)) + (_1O * 20))) + (_1Q * 641))) / 900)) + ($get($1.mscs, 2) * 20))) % 900); //#15212
            } //#15212
            $1.rem = $1.inlen % 6; //#15214
            if ($1.rem != 0) { //#15219
                $k[$j++] = $1.out; //#15217
                $k[$j++] = $1.out.length - $1.rem; //#15217
                $k[$j++] = Infinity; //#15217
                $aload($geti($1.in, $1.inlen - $1.rem, $1.rem)); //#15217
                var _1h = $a(); //#15217
                var _1i = $k[--$j]; //#15218
                $puti($k[--$j], _1i, _1h); //#15218
            } //#15218
            $k[$j++] = $1.out; //#15220
        }; //#15221
        if ($eq($1.encoding, "byte") || $eq($1.encoding, "ccc")) { //#15229
            $1.barlen = $1.barcode.length; //#15224
            $1.datcws = $a((((~~($1.barlen / 6)) * 5) + ($1.barlen % 6)) + 1); //#15225
            var _1t = (($1.barlen % 6) == 0) ? 924 : 901; //#15226
            $put($1.datcws, 0, _1t); //#15226
            $k[$j++] = $1.datcws; //#15227
            $k[$j++] = 1; //#15227
            $k[$j++] = Infinity; //#15227
            $forall($1.barcode); //#15227
            var _1w = $a(); //#15227
            $k[$j++] = _1w; //#15227
            $1.encb(); //#15227
            var _1x = $k[--$j]; //#15227
            var _1y = $k[--$j]; //#15227
            $puti($k[--$j], _1y, _1x); //#15227
            if ($eq($1.encoding, "ccc")) { //#15228
                $k[$j++] = Infinity; //#15228
                $k[$j++] = 920; //#15228
                $aload($1.datcws); //#15228
                $1.datcws = $a(); //#15228
            } //#15228
        } //#15228
        if ($eq($1.encoding, "auto")) { //#15632
            var _26 = new Map([
                ["parse", $1.parse],
                ["parsefnc", $1.parsefnc],
                ["eci", true]
            ]); //#15237
            $1.fncvals = _26; //#15239
            $k[$j++] = 'msg'; //#15240
            $k[$j++] = $1.barcode; //#15240
            $k[$j++] = $1.fncvals; //#15240
            $0.parseinput(); //#15240
            var _29 = $k[--$j]; //#15240
            $1[$k[--$j]] = _29; //#15240
            $1.msglen = $1.msg.length; //#15241
            $1.T = 0; //#15244
            $1.N = 1; //#15244
            $1.B = 2; //#15244
            $1.A = 0; //#15245
            $1.L = 1; //#15245
            $1.M = 2; //#15245
            $1.P = 3; //#15245
            $1.tl = -1; //#15248
            $1.nl = -2; //#15248
            $1.bl = -3; //#15248
            $1.bl6 = -4; //#15248
            $1.bs = -5; //#15248
            $1.al = -6; //#15251
            $1.ll = -7; //#15251
            $1.ml = -8; //#15251
            $1.pl = -9; //#15251
            $1.as = -10; //#15251
            $1.ps = -11; //#15251
            $1.charmaps = $a([$a(["A", "a", "0", ";"]), $a(["B", "b", "1", "<"]), $a(["C", "c", "2", ">"]), $a(["D", "d", "3", "@"]), $a(["E", "e", "4", "["]), $a(["F", "f", "5", 92]), $a(["G", "g", "6", "]"]), $a(["H", "h", "7", "_"]), $a(["I", "i", "8", "`"]), $a(["J", "j", "9", "~"]), $a(["K", "k", "&", "!"]), $a(["L", "l", 13, 13]), $a(["M", "m", 9, 9]), $a(["N", "n", ",", ","]), $a(["O", "o", ":", ":"]), $a(["P", "p", "#", 10]), $a(["Q", "q", "-", "-"]), $a(["R", "r", ".", "."]), $a(["S", "s", "$", "$"]), $a(["T", "t", "/", "/"]), $a(["U", "u", "+", "\""]), $a(["V", "v", "%", "|"]), $a(["W", "w", "*", "*"]), $a(["X", "x", "=", 40]), $a(["Y", "y", "^", 41]), $a(["Z", "z", $1.pl, "?"]), $a([" ", " ", " ", "{"]), $a([$1.ll, $1.as, $1.ll, "}"]), $a([$1.ml, $1.ml, $1.al, "'"]), $a([$1.ps, $1.ps, $1.ps, $1.al])]); //#15286
            $1.charvals = $a([new Map, new Map, new Map, new Map]); //#15289
            $1.alltext = new Map; //#15290
            for (var _2v = 0, _2u = $1.charmaps.length - 1; _2v <= _2u; _2v += 1) { //#15300
                $1.i = _2v; //#15292
                $1.encs = $get($1.charmaps, $1.i); //#15293
                for (var _2z = 0; _2z <= 3; _2z += 1) { //#15299
                    $1.j = _2z; //#15295
                    var _32 = $get($1.encs, $1.j); //#15296
                    $k[$j++] = _32; //#15296
                    if ($eq($type(_32), 'stringtype')) { //#15296
                        var _35 = $get($k[--$j], 0); //#15296
                        $k[$j++] = _35; //#15296
                    } //#15296
                    var _36 = $k[--$j]; //#15297
                    $put($get($1.charvals, $1.j), _36, $1.i); //#15297
                    $put($1.alltext, _36, -1); //#15298
                } //#15298
            } //#15298
            $1.e = 10000; //#15302
            $1.latlen = $a([$a([0, 1, 1, 2]), $a([2, 0, 1, 2]), $a([1, 1, 0, 1]), $a([1, 2, 2, 0])]); //#15310
            $1.latseq = $a([$a([$a([]), $a([$1.ll]), $a([$1.ml]), $a([$1.ml, $1.pl])]), $a([$a([$1.ml, $1.al]), $a([]), $a([$1.ml]), $a([$1.ml, $1.pl])]), $a([$a([$1.al]), $a([$1.ll]), $a([]), $a([$1.pl])]), $a([$a([$1.al]), $a([$1.al, $1.ll]), $a([$1.al, $1.ml]), $a([])])]); //#15318
            $1.shftlen = $a([$a([$1.e, $1.e, $1.e, 1]), $a([1, $1.e, $1.e, 1]), $a([$1.e, $1.e, $1.e, 1]), $a([$1.e, $1.e, $1.e, $1.e])]); //#15326
            $k[$j++] = Infinity; //#15329
            for (var _4B = 0, _4C = $1.msglen; _4B < _4C; _4B++) { //#15329
                $k[$j++] = 0; //#15329
            } //#15329
            $k[$j++] = 0; //#15329
            $1.numdigits = $a(); //#15329
            $k[$j++] = Infinity; //#15330
            for (var _4F = 0, _4G = $1.msglen; _4F < _4G; _4F++) { //#15330
                $k[$j++] = 0; //#15330
            } //#15330
            $k[$j++] = 0; //#15330
            $1.numtext = $a(); //#15330
            $k[$j++] = Infinity; //#15331
            for (var _4J = 0, _4K = $1.msglen; _4J < _4K; _4J++) { //#15331
                $k[$j++] = 0; //#15331
            } //#15331
            $k[$j++] = 0; //#15331
            $1.numbytes = $a(); //#15331
            $k[$j++] = Infinity; //#15332
            for (var _4N = 0, _4O = $1.msglen; _4N < _4O; _4N++) { //#15332
                $k[$j++] = 0; //#15332
            } //#15332
            $k[$j++] = 0; //#15332
            $1.iseci = $a(); //#15332
            for (var _4R = $1.msglen - 1; _4R >= 0; _4R -= 1) { //#15345
                $1.i = _4R; //#15334
                var _4U = $get($1.msg, $1.i); //#15335
                if ((_4U >= 48) && (_4U <= 57)) { //#15337
                    $put($1.numdigits, $1.i, $f($get($1.numdigits, $1.i + 1) + 1)); //#15336
                } //#15336
                var _4e = $get($1.alltext, $get($1.msg, $1.i)) !== undefined; //#15338
                if (_4e && ($get($1.numdigits, $1.i) < 13)) { //#15340
                    $put($1.numtext, $1.i, $f($get($1.numtext, $1.i + 1) + 1)); //#15339
                } //#15339
                if ((($get($1.msg, $1.i) >= 0) && ($get($1.numtext, $1.i) < 5)) && ($get($1.numdigits, $1.i) < 13)) { //#15343
                    $put($1.numbytes, $1.i, $f($get($1.numbytes, $1.i + 1) + 1)); //#15342
                } //#15342
                $put($1.iseci, $1.i, $get($1.msg, $1.i) <= -1000000); //#15344
            } //#15344
            $1.numdigits = $geti($1.numdigits, 0, $1.msglen); //#15346
            $1.numtext = $geti($1.numtext, 0, $1.msglen); //#15347
            $1.numbytes = $geti($1.numbytes, 0, $1.msglen); //#15348
            $1.seq = $a([]); //#15350
            $1.seqlen = 0; //#15350
            $1.state = $1.T; //#15350
            $1.p = 0; //#15350
            for (;;) { //#15404
                if ($1.p == $1.msglen) { //#15351
                    break; //#15351
                } //#15351
                if ($get($1.iseci, $1.p)) { //#15402
                    $1.eci = $get($1.msg, $1.p); //#15353
                    $k[$j++] = Infinity; //#15356
                    $aload($1.seq); //#15355
                    $k[$j++] = $a([$1.eci]); //#15356
                    $1.seq = $a(); //#15356
                    $1.p = $1.p + 1; //#15358
                    $k[$j++] = 'seqlen'; //#15359
                    $k[$j++] = $1.seqlen; //#15359
                    if ($1.eci <= -1810900) { //#15359
                        $k[$j++] = 2; //#15359
                    } else { //#15359
                        var _5X = ($1.eci <= -1000900) ? 3 : 2; //#15359
                        $k[$j++] = _5X; //#15359
                    } //#15359
                    var _5Y = $k[--$j]; //#15359
                    var _5Z = $k[--$j]; //#15359
                    $1[$k[--$j]] = $f(_5Z + _5Y); //#15359
                } else { //#15402
                    $1.n = $get($1.numdigits, $1.p); //#15361
                    if ($1.n >= 13) { //#15402
                        $k[$j++] = Infinity; //#15366
                        $aload($1.seq); //#15364
                        $k[$j++] = $1.nl; //#15366
                        $k[$j++] = Infinity; //#15366
                        $aload($geti($1.msg, $1.p, $1.n)); //#15366
                        var _5l = $a(); //#15366
                        $k[$j++] = _5l; //#15366
                        $1.seq = $a(); //#15366
                        $1.state = $1.N; //#15368
                        $1.p = $f($1.p + $1.n); //#15369
                        $1.seqlen = $f(($1.seqlen + 1) + $1.n); //#15370
                    } else { //#15402
                        $1.t = $get($1.numtext, $1.p); //#15372
                        if ($1.t >= 5) { //#15402
                            $k[$j++] = Infinity; //#15379
                            $aload($1.seq); //#15375
                            if ($1.state != $1.T) { //#15378
                                $k[$j++] = $1.tl; //#15377
                            } //#15377
                            $k[$j++] = Infinity; //#15379
                            $aload($geti($1.msg, $1.p, $1.t)); //#15379
                            var _64 = $a(); //#15379
                            $k[$j++] = _64; //#15379
                            $1.seq = $a(); //#15379
                            $1.state = $1.T; //#15381
                            $1.p = $f($1.p + $1.t); //#15382
                            $1.seqlen = $f($f($1.seqlen + 1) + $1.t); //#15383
                        } else { //#15402
                            $1.b = $get($1.numbytes, $1.p); //#15385
                            if (($1.b == 1) && ($1.state == $1.T)) { //#15402
                                $k[$j++] = Infinity; //#15390
                                $aload($1.seq); //#15388
                                $k[$j++] = $1.bs; //#15390
                                $k[$j++] = $a([$get($1.msg, $1.p)]); //#15390
                                $1.seq = $a(); //#15390
                                $1.p = $f($1.p + $1.b); //#15392
                                $1.seqlen = $f($1.seqlen + 2); //#15393
                            } else { //#15402
                                $k[$j++] = Infinity; //#15398
                                $aload($1.seq); //#15396
                                var _6T = (($1.b % 6) != 0) ? $1.bl : $1.bl6; //#15397
                                $k[$j++] = _6T; //#15398
                                $k[$j++] = Infinity; //#15398
                                $aload($geti($1.msg, $1.p, $1.b)); //#15398
                                var _6Y = $a(); //#15398
                                $k[$j++] = _6Y; //#15398
                                $1.seq = $a(); //#15398
                                $1.state = $1.B; //#15400
                                $1.p = $f($1.p + $1.b); //#15401
                                $1.seqlen = $f($f($1.seqlen + 1) + $1.b); //#15402
                            } //#15402
                        } //#15402
                    } //#15402
                } //#15402
            } //#15402
            $k[$j++] = Infinity; //#15407
            $k[$j++] = $1.tl; //#15407
            $k[$j++] = 900; //#15407
            $k[$j++] = $1.bl; //#15407
            $k[$j++] = 901; //#15407
            $k[$j++] = $1.bl6; //#15407
            $k[$j++] = 924; //#15407
            $k[$j++] = $1.nl; //#15407
            $k[$j++] = 902; //#15407
            $k[$j++] = $1.bs; //#15407
            $k[$j++] = 913; //#15407
            $1.latchcws = $d(); //#15408
            $1.enca = function() {
                var _6p = $get($get($1.charvals, $1.A), $k[--$j]); //#15411
                $k[$j++] = _6p; //#15411
            }; //#15411
            $1.encl = function() {
                var _6u = $get($get($1.charvals, $1.L), $k[--$j]); //#15412
                $k[$j++] = _6u; //#15412
            }; //#15412
            $1.encm = function() {
                var _6z = $get($get($1.charvals, $1.M), $k[--$j]); //#15413
                $k[$j++] = _6z; //#15413
            }; //#15413
            $1.encp = function() {
                var _74 = $get($get($1.charvals, $1.P), $k[--$j]); //#15414
                $k[$j++] = _74; //#15414
            }; //#15414
            $1.textencfuncs = $a(['enca', 'encl', 'encm', 'encp']); //#15415
            $1.addtotext = function() {
                $put($1.text, $1.l, $k[--$j]); //#15418
                $1.l = $1.l + 1; //#15419
            }; //#15420
            $1.enct = function() {
                $1.in = $k[--$j]; //#15423
                $1.curlen = $a([$1.e, $1.e, $1.e, $1.e]); //#15425
                $put($1.curlen, $1.submode, 0); //#15426
                $1.curseq = $a([$a([]), $a([]), $a([]), $a([])]); //#15427
                $forall($1.in, function() { //#15497
                    $1.char = $k[--$j]; //#15432
                    for (;;) { //#15453
                        $1.imp = false; //#15436
                        var _7T = $a([$1.A, $1.L, $1.M, $1.P]); //#15437
                        for (var _7U = 0, _7V = _7T.length; _7U < _7V; _7U++) { //#15451
                            $1.x = $get(_7T, _7U); //#15438
                            var _7b = $a([$1.A, $1.L, $1.M, $1.P]); //#15439
                            for (var _7c = 0, _7d = _7b.length; _7c < _7d; _7c++) { //#15450
                                $1.y = $get(_7b, _7c); //#15440
                                $1.cost = $f($get($1.curlen, $1.x) + $get($get($1.latlen, $1.x), $1.y)); //#15441
                                if ($1.cost < $get($1.curlen, $1.y)) { //#15449
                                    $put($1.curlen, $1.y, $1.cost); //#15443
                                    $k[$j++] = $1.curseq; //#15446
                                    $k[$j++] = $1.y; //#15446
                                    $k[$j++] = Infinity; //#15446
                                    $aload($get($1.curseq, $1.x)); //#15445
                                    $aload($get($get($1.latseq, $1.x), $1.y)); //#15446
                                    var _84 = $a(); //#15446
                                    var _85 = $k[--$j]; //#15447
                                    $put($k[--$j], _85, _84); //#15447
                                    $1.imp = true; //#15448
                                } //#15448
                            } //#15448
                        } //#15448
                        if (!$1.imp) { //#15452
                            break; //#15452
                        } //#15452
                    } //#15452
                    $1.nxtlen = $a([$1.e, $1.e, $1.e, $1.e]); //#15456
                    $1.nxtseq = $a(4); //#15457
                    var _8I = $a([$1.A, $1.L, $1.M, $1.P]); //#15459
                    for (var _8J = 0, _8K = _8I.length; _8J < _8K; _8J++) { //#15492
                        $1.x = $get(_8I, _8J); //#15460
                        for (;;) { //#15490
                            var _8Q = $get($get($1.charvals, $1.x), $1.char) !== undefined; //#15464
                            if (!_8Q) { //#15464
                                break; //#15464
                            } //#15464
                            $1.cost = $f($get($1.curlen, $1.x) + 1); //#15467
                            if ($1.cost < $get($1.nxtlen, $1.x)) { //#15471
                                $put($1.nxtlen, $1.x, $1.cost); //#15469
                                $k[$j++] = $1.nxtseq; //#15470
                                $k[$j++] = $1.x; //#15470
                                $k[$j++] = Infinity; //#15470
                                $aload($get($1.curseq, $1.x)); //#15470
                                $k[$j++] = $1.char; //#15470
                                var _8h = $a(); //#15470
                                var _8i = $k[--$j]; //#15470
                                $put($k[--$j], _8i, _8h); //#15470
                            } //#15470
                            var _8o = $a([$1.A, $1.L, $1.M, $1.P]); //#15474
                            for (var _8p = 0, _8q = _8o.length; _8p < _8q; _8p++) { //#15487
                                $1.y = $get(_8o, _8p); //#15475
                                if ($ne($1.x, $1.y)) { //#15486
                                    $1.cost = $f($f($get($1.curlen, $1.y) + $get($get($1.shftlen, $1.y), $1.x)) + 1); //#15477
                                    if ($1.cost < $get($1.nxtlen, $1.y)) { //#15485
                                        $put($1.nxtlen, $1.y, $1.cost); //#15479
                                        $k[$j++] = $1.nxtseq; //#15483
                                        $k[$j++] = $1.y; //#15483
                                        $k[$j++] = Infinity; //#15483
                                        $aload($get($1.curseq, $1.y)); //#15481
                                        var _9G = ($1.x == $1.A) ? $1.as : $1.ps; //#15482
                                        $k[$j++] = _9G; //#15483
                                        $k[$j++] = $1.char; //#15483
                                        var _9I = $a(); //#15483
                                        var _9J = $k[--$j]; //#15484
                                        $put($k[--$j], _9J, _9I); //#15484
                                    } //#15484
                                } //#15484
                            } //#15484
                            break; //#15489
                        } //#15489
                    } //#15489
                    $1.curlen = $1.nxtlen; //#15494
                    $1.curseq = $1.nxtseq; //#15495
                }); //#15495
                $1.minseq = $1.e; //#15500
                var _9S = $a([$1.A, $1.L, $1.M, $1.P]); //#15501
                for (var _9T = 0, _9U = _9S.length; _9T < _9U; _9T++) { //#15507
                    $1.k = $get(_9S, _9T); //#15502
                    if ($get($1.curlen, $1.k) < $1.minseq) { //#15506
                        $1.minseq = $get($1.curlen, $1.k); //#15504
                        $1.txtseq = $get($1.curseq, $1.k); //#15505
                    } //#15505
                } //#15505
                $1.text = $a($1.minseq); //#15510
                $1.k = 0; //#15511
                $1.l = 0; //#15511
                for (;;) { //#15531
                    if ($1.k >= $1.txtseq.length) { //#15512
                        break; //#15512
                    } //#15512
                    $1.char = $get($1.txtseq, $1.k); //#15514
                    $k[$j++] = $1.char; //#15517
                    if ($1[$get($1.textencfuncs, $1.submode)]() === true) {
                        break;
                    } //#15517
                    $1.addtotext(); //#15517
                    $1.k = $1.k + 1; //#15518
                    if (($1.char == $1.as) || ($1.char == $1.ps)) { //#15524
                        $k[$j++] = $get($1.txtseq, $1.k); //#15522
                        if ($1.char == $1.as) { //#15522
                            $1.enca(); //#15522
                        } else { //#15522
                            $1.encp(); //#15522
                        } //#15522
                        $1.addtotext(); //#15522
                        $1.k = $1.k + 1; //#15523
                    } //#15523
                    if ($1.char == $1.al) { //#15527
                        $1.submode = $1.A; //#15527
                    } //#15527
                    if ($1.char == $1.ll) { //#15528
                        $1.submode = $1.L; //#15528
                    } //#15528
                    if ($1.char == $1.ml) { //#15529
                        $1.submode = $1.M; //#15529
                    } //#15529
                    if ($1.char == $1.pl) { //#15530
                        $1.submode = $1.P; //#15530
                    } //#15530
                } //#15530
                if (($1.text.length % 2) == 1) { //#15541
                    if ($1.submode == $1.P) { //#15538
                        $k[$j++] = 'pad'; //#15535
                        $k[$j++] = $1.al; //#15535
                        $1.encp(); //#15535
                        var _AJ = $k[--$j]; //#15535
                        $1[$k[--$j]] = _AJ; //#15535
                        $1.submode = $1.A; //#15536
                    } else { //#15538
                        $k[$j++] = 'pad'; //#15538
                        $k[$j++] = $1.ps; //#15538
                        if ($1[$get($1.textencfuncs, $1.submode)]() === true) {
                            return true;
                        } //#15538
                        var _AR = $k[--$j]; //#15538
                        $1[$k[--$j]] = _AR; //#15538
                    } //#15538
                    $k[$j++] = Infinity; //#15540
                    $aload($1.text); //#15540
                    $k[$j++] = $1.pad; //#15540
                    $1.text = $a(); //#15540
                } //#15540
                $1.out = $a(~~($1.text.length / 2)); //#15543
                for (var _Aa = 0, _AZ = $1.out.length - 1; _Aa <= _AZ; _Aa += 1) { //#15547
                    $1.k = _Aa; //#15545
                    $put($1.out, $1.k, $f(($get($1.text, $1.k * 2) * 30) + $get($1.text, ($1.k * 2) + 1))); //#15546
                } //#15546
                $k[$j++] = $1.out; //#15549
            }; //#15551
            $1.encn = function() {
                $1.in = $k[--$j]; //#15554
                $1.out = $a([]); //#15555
                for (var _Ao = 0, _An = $1.in.length - 1; _Ao <= _An; _Ao += 44) { //#15583
                    $1.k = _Ao; //#15557
                    $k[$j++] = Infinity; //#15560
                    var _At = $1.in.length - $1.k; //#15559
                    $k[$j++] = 1; //#15559
                    $k[$j++] = $1.in; //#15559
                    $k[$j++] = $1.k; //#15559
                    $k[$j++] = _At; //#15559
                    if (_At > 44) { //#15559
                        $j--; //#15559
                        $k[$j++] = 44; //#15559
                    } //#15559
                    var _Au = $k[--$j]; //#15559
                    var _Av = $k[--$j]; //#15559
                    var _Ax = $geti($k[--$j], _Av, _Au); //#15559
                    for (var _Ay = 0, _Az = _Ax.length; _Ay < _Az; _Ay++) { //#15560
                        $k[$j++] = $f($get(_Ax, _Ay) - 48); //#15560
                    } //#15560
                    $1.gmod = $a(); //#15560
                    $1.cwn = $a([]); //#15562
                    for (;;) { //#15581
                        $1.dv = 900; //#15563
                        $1.gmul = $a([]); //#15564
                        $1.val = 0; //#15564
                        for (;;) { //#15576
                            if ($1.gmod.length == 0) { //#15565
                                break; //#15565
                            } //#15565
                            $1.val = $f(($1.val * 10) + $get($1.gmod, 0)); //#15566
                            $1.gmod = $geti($1.gmod, 1, $1.gmod.length - 1); //#15567
                            if ($1.val < $1.dv) { //#15573
                                if ($1.gmul.length != 0) { //#15571
                                    $k[$j++] = Infinity; //#15570
                                    $aload($1.gmul); //#15570
                                    $k[$j++] = 0; //#15570
                                    $1.gmul = $a(); //#15570
                                } //#15570
                            } else { //#15573
                                $k[$j++] = Infinity; //#15573
                                $aload($1.gmul); //#15573
                                $k[$j++] = ~~($1.val / $1.dv); //#15573
                                $1.gmul = $a(); //#15573
                            } //#15573
                            $1.val = $1.val % $1.dv; //#15575
                        } //#15575
                        $1.dv = $1.val; //#15577
                        $k[$j++] = Infinity; //#15578
                        $k[$j++] = $1.dv; //#15578
                        $aload($1.cwn); //#15578
                        $1.cwn = $a(); //#15578
                        $1.gmod = $1.gmul; //#15579
                        if ($1.gmul.length == 0) { //#15580
                            break; //#15580
                        } //#15580
                    } //#15580
                    $k[$j++] = Infinity; //#15582
                    $aload($1.out); //#15582
                    $aload($1.cwn); //#15582
                    $1.out = $a(); //#15582
                } //#15582
                $k[$j++] = $1.out; //#15584
            }; //#15585
            $1.ence = function() {
                var _BY = $f((-$get($k[--$j], 0)) - 1000000); //#15589
                $k[$j++] = _BY; //#15597
                if (_BY <= 899) { //#15596
                    var _BZ = $k[--$j]; //#15590
                    $k[$j++] = 927; //#15590
                    $k[$j++] = _BZ; //#15590
                    $astore($a(2)); //#15590
                } else { //#15596
                    var _Bb = $k[--$j]; //#15591
                    $k[$j++] = _Bb; //#15597
                    if (_Bb <= 810899) { //#15596
                        var _Bc = $k[--$j]; //#15592
                        $k[$j++] = 926; //#15592
                        $k[$j++] = (~~(_Bc / 900)) - 1; //#15592
                        $k[$j++] = _Bc % 900; //#15592
                        $astore($a(3)); //#15592
                    } else { //#15596
                        var _Be = $k[--$j]; //#15593
                        $k[$j++] = _Be; //#15597
                        if (_Be <= 811799) { //#15596
                            var _Bf = $k[--$j]; //#15594
                            $k[$j++] = 925; //#15594
                            $k[$j++] = $f(_Bf - 810900); //#15594
                            $astore($a(2)); //#15594
                        } else { //#15596
                            $k[$j++] = 'bwipp.pdf417badECI'; //#15596
                            $k[$j++] = "PDF417 supports ECIs 000000 to 811799"; //#15596
                            $0.raiseerror(); //#15596
                        } //#15596
                    } //#15596
                } //#15596
            }; //#15598
            $1.encfuncs = $a(['enct', 'encn', 'encb']); //#15600
            $1.addtocws = function() {
                var _Bi = $k[--$j]; //#15603
                $puti($1.datcws, $1.j, _Bi); //#15603
                $1.j = _Bi.length + $1.j; //#15604
            }; //#15605
            $1.state = $1.T; //#15608
            $1.submode = $1.A; //#15608
            $1.datcws = $a($1.seqlen); //#15609
            $1.i = 0; //#15610
            $1.j = 0; //#15610
            for (;;) { //#15630
                if ($1.i >= $1.seq.length) { //#15611
                    break; //#15611
                } //#15611
                $1.chars = $get($1.seq, $1.i); //#15612
                if ($eq($type($1.chars), 'arraytype')) { //#15626
                    if ($get($1.chars, 0) <= -1000000) { //#15617
                        $k[$j++] = $1.chars; //#15615
                        $1.ence(); //#15615
                        $1.addtocws(); //#15615
                    } else { //#15617
                        $k[$j++] = $1.chars; //#15617
                        if ($1[$get($1.encfuncs, $1.state)]() === true) {
                            break;
                        } //#15617
                        $1.addtocws(); //#15617
                    } //#15617
                } else { //#15626
                    $k[$j++] = $a([$get($1.latchcws, $1.chars)]); //#15620
                    $1.addtocws(); //#15620
                    if ($1.chars == $1.tl) { //#15621
                        $1.state = $1.T; //#15621
                        $1.submode = $1.A; //#15621
                    } //#15621
                    if ($1.chars == $1.nl) { //#15622
                        $1.state = $1.N; //#15622
                    } //#15622
                    if (($1.chars == $1.bl) || ($1.chars == $1.bl6)) { //#15623
                        $1.state = $1.B; //#15623
                    } //#15623
                    if ($1.chars == $1.bs) { //#15627
                        $1.i = $1.i + 1; //#15625
                        $k[$j++] = $get($1.seq, $1.i); //#15626
                        $1.encb(); //#15626
                        $1.addtocws(); //#15626
                    } //#15626
                } //#15626
                $1.i = $1.i + 1; //#15629
            } //#15629
            $1.datcws = $geti($1.datcws, 0, $1.j); //#15631
        } //#15631
        $1.m = $1.datcws.length; //#15635
        if ($1.eclevel == -1) { //#15641
            if ($1.m <= 40) { //#15637
                $1.eclevel = 2; //#15637
            } //#15637
            if (($1.m >= 41) && ($1.m <= 160)) { //#15638
                $1.eclevel = 3; //#15638
            } //#15638
            if (($1.m >= 161) && ($1.m <= 320)) { //#15639
                $1.eclevel = 4; //#15639
            } //#15639
            if ($1.m >= 321) { //#15640
                $1.eclevel = 5; //#15640
            } //#15640
        } //#15640
        $1.maxeclevel = (~~((Math.log((928 - 1) - $1.m)) / Math.log(2))) - 1; //#15644
        if ($1.eclevel > $1.maxeclevel) { //#15645
            $1.eclevel = $1.maxeclevel; //#15645
        } //#15645
        $1.k = ~~Math.pow(2, $1.eclevel + 1); //#15646
        if ($1.columns == 0) { //#15649
            $1.columns = ~~(Math.round(Math.sqrt(($1.m + $1.k) / 3))); //#15649
        } //#15649
        $k[$j++] = 'c'; //#15650
        if ($1.columns >= 1) { //#15650
            $k[$j++] = $1.columns; //#15650
        } else { //#15650
            $k[$j++] = 1; //#15650
        } //#15650
        var _Cn = $k[--$j]; //#15650
        $1[$k[--$j]] = _Cn; //#15650
        $1.r = ~~(Math.ceil((($1.m + $1.k) + 1) / $1.columns)); //#15653
        if (($1.r < $1.rows) && ($1.rows <= 90)) { //#15654
            $1.r = $1.rows; //#15654
        } //#15654
        if ($1.r < 3) { //#15655
            $1.r = 3; //#15655
        } //#15655
        var _Cx = $1.c; //#15658
        var _Cy = $1.r; //#15658
        var _Cz = $1.m; //#15658
        var _D0 = 8; //#15658
        var _D1 = (~~((Math.log($f(($f((_Cx * _Cy) - 1)) - _Cz))) / Math.log(2))) - 1; //#15658
        if (((~~((Math.log($f(($f((_Cx * _Cy) - 1)) - _Cz))) / Math.log(2))) - 1) > 8) { //#15658
            var _ = _D0; //#15658
            _D0 = _D1; //#15658
            _D1 = _; //#15658
        } //#15658
        $1.maxeclevel = _D1; //#15658
        if ($1.maxeclevel > $1.eclevel) { //#15662
            $1.eclevel = $1.maxeclevel; //#15660
            $1.k = ~~Math.pow(2, $1.eclevel + 1); //#15661
        } //#15661
        $1.n = $f(($1.c * $1.r) - $1.k); //#15665
        $1.cws = $a($f(($1.c * $1.r) + 1)); //#15666
        $put($1.cws, 0, $1.n); //#15667
        $puti($1.cws, 1, $1.datcws); //#15668
        $k[$j++] = $1.cws; //#15669
        $k[$j++] = $1.m + 1; //#15669
        $k[$j++] = Infinity; //#15669
        for (var _DK = 0, _DL = $f($f($1.n - $1.m) - 1); _DK < _DL; _DK++) { //#15669
            $k[$j++] = 900; //#15669
        } //#15669
        var _DM = $a(); //#15669
        var _DN = $k[--$j]; //#15669
        $puti($k[--$j], _DN, _DM); //#15669
        $k[$j++] = $1.cws; //#15670
        $k[$j++] = $1.n; //#15670
        $k[$j++] = Infinity; //#15670
        for (var _DS = 0, _DT = $1.k; _DS < _DT; _DS++) { //#15670
            $k[$j++] = 0; //#15670
        } //#15670
        $k[$j++] = 0; //#15670
        var _DU = $a(); //#15670
        var _DV = $k[--$j]; //#15670
        $puti($k[--$j], _DV, _DU); //#15670
        $k[$j++] = Infinity; //#15673
        $k[$j++] = 1; //#15673
        for (var _DX = 0, _DY = 928; _DX < _DY; _DX++) { //#15673
            var _DZ = $k[--$j]; //#15673
            $k[$j++] = _DZ; //#15673
            $k[$j++] = (_DZ * 3) % 929; //#15673
        } //#15673
        $1.rsalog = $a(); //#15673
        $1.rslog = $a(929); //#15674
        for (var _Dc = 1; _Dc <= 928; _Dc += 1) { //#15675
            $put($1.rslog, $get($1.rsalog, _Dc), _Dc); //#15675
        } //#15675
        $1.rsprod = function() {
            var _Dg = $k[--$j]; //#15679
            var _Dh = $k[--$j]; //#15679
            $k[$j++] = _Dh; //#15683
            $k[$j++] = _Dg; //#15683
            if ((_Dg != 0) && (_Dh != 0)) { //#15682
                var _Dk = $get($1.rslog, $k[--$j]); //#15680
                var _Dp = $get($1.rsalog, $f(_Dk + $get($1.rslog, $k[--$j])) % 928); //#15680
                $k[$j++] = _Dp; //#15680
            } else { //#15682
                $j -= 2; //#15682
                $k[$j++] = 0; //#15682
            } //#15682
        }; //#15684
        $k[$j++] = Infinity; //#15687
        $k[$j++] = 1; //#15687
        for (var _Dr = 0, _Ds = $1.k; _Dr < _Ds; _Dr++) { //#15687
            $k[$j++] = 0; //#15687
        } //#15687
        $1.coeffs = $a(); //#15687
        for (var _Dw = 1, _Dv = $1.k; _Dw <= _Dv; _Dw += 1) { //#15696
            $1.i = _Dw; //#15689
            $put($1.coeffs, $1.i, $get($1.coeffs, $1.i - 1)); //#15690
            for (var _E3 = $1.i - 1; _E3 >= 1; _E3 -= 1) { //#15694
                $1.j = _E3; //#15692
                $k[$j++] = $1.coeffs; //#15693
                $k[$j++] = $1.j; //#15693
                $k[$j++] = $get($1.coeffs, $1.j - 1); //#15693
                $k[$j++] = $get($1.coeffs, $1.j); //#15693
                $k[$j++] = $get($1.rsalog, $1.i); //#15693
                $1.rsprod(); //#15693
                var _EF = $k[--$j]; //#15693
                var _EG = $k[--$j]; //#15693
                var _EH = $k[--$j]; //#15693
                $put($k[--$j], _EH, $f(_EG + _EF) % 929); //#15693
            } //#15693
            $k[$j++] = $1.coeffs; //#15695
            $k[$j++] = 0; //#15695
            $k[$j++] = $get($1.coeffs, 0); //#15695
            $k[$j++] = $get($1.rsalog, $1.i); //#15695
            $1.rsprod(); //#15695
            var _EP = $k[--$j]; //#15695
            var _EQ = $k[--$j]; //#15695
            $put($k[--$j], _EQ, _EP); //#15695
        } //#15695
        $1.coeffs = $geti($1.coeffs, 0, $1.coeffs.length - 1); //#15697
        for (var _EW = $1.coeffs.length - 1; _EW >= 0; _EW -= 2) { //#15698
            var _EX = $1.coeffs; //#15698
            $put(_EX, _EW, $f(929 - $get(_EX, _EW))); //#15698
        } //#15698
        for (var _Eb = 0, _Ea = $f($1.n - 1); _Eb <= _Ea; _Eb += 1) { //#15707
            $1.t = $f($get($1.cws, _Eb) + $get($1.cws, $1.n)) % 929; //#15702
            for (var _Ej = 0, _Ei = $1.k - 1; _Ej <= _Ei; _Ej += 1) { //#15706
                $1.j = _Ej; //#15704
                $put($1.cws, $f($1.n + $1.j), ($f($get($1.cws, $f($f($1.n + $1.j) + 1)) + ($f(929 - (($1.t * $get($1.coeffs, ($1.k - $1.j) - 1)) % 929))))) % 929); //#15705
            } //#15705
        } //#15705
        for (var _F0 = $1.n, _Ez = $f($1.n + $1.k); _F0 <= _Ez; _F0 += 1) { //#15708
            $put($1.cws, _F0, $f(929 - $get($1.cws, _F0)) % 929); //#15708
        } //#15708
        $1.cws = $geti($1.cws, 0, $1.cws.length - 1); //#15711
        $1.clusters = $a([$a([120256, 125680, 128380, 120032, 125560, 128318, 108736, 119920, 108640, 86080, 108592, 86048, 110016, 120560, 125820, 109792, 120440, 125758, 88256, 109680, 88160, 89536, 110320, 120700, 89312, 110200, 120638, 89200, 110140, 89840, 110460, 89720, 110398, 89980, 128506, 119520, 125304, 128190, 107712, 119408, 125244, 107616, 119352, 84032, 107568, 119324, 84000, 107544, 83984, 108256, 119672, 125374, 85184, 108144, 119612, 85088, 108088, 119582, 85040, 108060, 85728, 108408, 119742, 85616, 108348, 85560, 108318, 85880, 108478, 85820, 85790, 107200, 119152, 125116, 107104, 119096, 125086, 83008, 107056, 119068, 82976, 107032, 82960, 82952, 83648, 107376, 119228, 83552, 107320, 119198, 83504, 107292, 83480, 83468, 83824, 107452, 83768, 107422, 83740, 83900, 106848, 118968, 125022, 82496, 106800, 118940, 82464, 106776, 118926, 82448, 106764, 82440, 106758, 82784, 106936, 119006, 82736, 106908, 82712, 106894, 82700, 82694, 106974, 82830, 82240, 106672, 118876, 82208, 106648, 118862, 82192, 106636, 82184, 106630, 82180, 82352, 82328, 82316, 82080, 118830, 106572, 106566, 82050, 117472, 124280, 127678, 103616, 117360, 124220, 103520, 117304, 124190, 75840, 103472, 75808, 104160, 117624, 124350, 76992, 104048, 117564, 76896, 103992, 76848, 76824, 77536, 104312, 117694, 77424, 104252, 77368, 77340, 77688, 104382, 77628, 77758, 121536, 126320, 128700, 121440, 126264, 128670, 111680, 121392, 126236, 111648, 121368, 126222, 111632, 121356, 103104, 117104, 124092, 112320, 103008, 117048, 124062, 112224, 121656, 126366, 93248, 74784, 102936, 117006, 93216, 112152, 93200, 75456, 103280, 117180, 93888, 75360, 103224, 117150, 93792, 112440, 121758, 93744, 75288, 93720, 75632, 103356, 94064, 75576, 103326, 94008, 112542, 93980, 75708, 94140, 75678, 94110, 121184, 126136, 128606, 111168, 121136, 126108, 111136, 121112, 126094, 111120, 121100, 111112, 111108, 102752, 116920, 123998, 111456, 102704, 116892, 91712, 74272, 121244, 116878, 91680, 74256, 102668, 91664, 111372, 102662, 74244, 74592, 102840, 116958, 92000, 74544, 102812, 91952, 111516, 102798, 91928, 74508, 74502, 74680, 102878, 92088, 74652, 92060, 74638, 92046, 92126, 110912, 121008, 126044, 110880, 120984, 126030, 110864, 120972, 110856, 120966, 110852, 110850, 74048, 102576, 116828, 90944, 74016, 102552, 116814, 90912, 111000, 121038, 90896, 73992, 102534, 90888, 110982, 90884, 74160, 102620, 91056, 74136, 102606, 91032, 111054, 91020, 74118, 91014, 91100, 91086, 110752, 120920, 125998, 110736, 120908, 110728, 120902, 110724, 110722, 73888, 102488, 116782, 90528, 73872, 102476, 90512, 110796, 102470, 90504, 73860, 90500, 73858, 73944, 90584, 90572, 90566, 120876, 120870, 110658, 102444, 73800, 90312, 90308, 90306, 101056, 116080, 123580, 100960, 116024, 70720, 100912, 115996, 70688, 100888, 70672, 70664, 71360, 101232, 116156, 71264, 101176, 116126, 71216, 101148, 71192, 71180, 71536, 101308, 71480, 101278, 71452, 71612, 71582, 118112, 124600, 127838, 105024, 118064, 124572, 104992, 118040, 124558, 104976, 118028, 104968, 118022, 100704, 115896, 123486, 105312, 100656, 115868, 79424, 70176, 118172, 115854, 79392, 105240, 100620, 79376, 70152, 79368, 70496, 100792, 115934, 79712, 70448, 118238, 79664, 105372, 100750, 79640, 70412, 79628, 70584, 100830, 79800, 70556, 79772, 70542, 70622, 79838, 122176, 126640, 128860, 122144, 126616, 128846, 122128, 126604, 122120, 126598, 122116, 104768, 117936, 124508, 113472, 104736, 126684, 124494, 113440, 122264, 126670, 113424, 104712, 117894, 113416, 122246, 104706, 69952, 100528, 115804, 78656, 69920, 100504, 115790, 96064, 78624, 104856, 117966, 96032, 113560, 122318, 100486, 96016, 78600, 104838, 96008, 69890, 70064, 100572, 78768, 70040, 100558, 96176, 78744, 104910, 96152, 113614, 70022, 78726, 70108, 78812, 70094, 96220, 78798, 122016, 126552, 128814, 122000, 126540, 121992, 126534, 121988, 121986, 104608, 117848, 124462, 113056, 104592, 126574, 113040, 122060, 117830, 113032, 104580, 113028, 104578, 113026, 69792, 100440, 115758, 78240, 69776, 100428, 95136, 78224, 104652, 100422, 95120, 113100, 69764, 95112, 78212, 69762, 78210, 69848, 100462, 78296, 69836, 95192, 78284, 69830, 95180, 78278, 69870, 95214, 121936, 126508, 121928, 126502, 121924, 121922, 104528, 117804, 112848, 104520, 117798, 112840, 121958, 112836, 104514, 112834, 69712, 100396, 78032, 69704, 100390, 94672, 78024, 104550, 94664, 112870, 69698, 94660, 78018, 94658, 78060, 94700, 94694, 126486, 121890, 117782, 104484, 104482, 69672, 77928, 94440, 69666, 77922, 99680, 68160, 99632, 68128, 99608, 115342, 68112, 99596, 68104, 99590, 68448, 99768, 115422, 68400, 99740, 68376, 99726, 68364, 68358, 68536, 99806, 68508, 68494, 68574, 101696, 116400, 123740, 101664, 116376, 101648, 116364, 101640, 116358, 101636, 67904, 99504, 115292, 72512, 67872, 116444, 115278, 72480, 101784, 116430, 72464, 67848, 99462, 72456, 101766, 67842, 68016, 99548, 72624, 67992, 99534, 72600, 101838, 72588, 67974, 68060, 72668, 68046, 72654, 118432, 124760, 127918, 118416, 124748, 118408, 124742, 118404, 118402, 101536, 116312, 105888, 101520, 116300, 105872, 118476, 116294, 105864, 101508, 105860, 101506, 105858, 67744, 99416, 72096, 67728, 116334, 80800, 72080, 101580, 99398, 80784, 105932, 67716, 80776, 72068, 67714, 72066, 67800, 99438, 72152, 67788, 80856, 72140, 67782, 80844, 72134, 67822, 72174, 80878, 126800, 128940, 126792, 128934, 126788, 126786, 118352, 124716, 122576, 126828, 124710, 122568, 126822, 122564, 118338, 122562, 101456, 116268, 105680, 101448, 116262, 114128, 105672, 118374, 114120, 122598, 101442, 114116, 105666, 114114, 67664, 99372, 71888, 67656, 99366, 80336, 71880, 101478, 97232, 80328, 105702, 67650, 97224, 114150, 71874, 97220, 67692, 71916, 67686, 80364, 71910, 97260, 80358, 97254, 126760, 128918, 126756, 126754, 118312, 124694, 122472, 126774, 122468, 118306, 122466, 101416, 116246, 105576, 101412, 113896, 105572, 101410, 113892, 105570, 113890, 67624, 99350, 71784, 101430, 80104, 71780, 67618, 96744, 80100, 71778, 96740, 80098, 96738, 71798, 96758, 126738, 122420, 122418, 105524, 113780, 113778, 71732, 79988, 96500, 96498, 66880, 66848, 98968, 66832, 66824, 66820, 66992, 66968, 66956, 66950, 67036, 67022, 100000, 99984, 115532, 99976, 115526, 99972, 99970, 66720, 98904, 69024, 100056, 98892, 69008, 100044, 69000, 100038, 68996, 66690, 68994, 66776, 98926, 69080, 100078, 69068, 66758, 69062, 66798, 69102, 116560, 116552, 116548, 116546, 99920, 102096, 116588, 115494, 102088, 116582, 102084, 99906, 102082, 66640, 68816, 66632, 98854, 73168, 68808, 66628, 73160, 68804, 66626, 73156, 68802, 66668, 68844, 66662, 73196, 68838, 73190, 124840, 124836, 124834, 116520, 118632, 124854, 118628, 116514, 118626, 99880, 115478, 101992, 116534, 106216, 101988, 99874, 106212, 101986, 106210, 66600, 98838, 68712, 99894, 72936, 68708, 66594, 81384, 72932, 68706, 81380, 72930, 66614, 68726, 72950, 81398, 128980, 128978, 124820, 126900, 124818, 126898, 116500, 118580, 116498, 122740, 118578, 122738, 99860, 101940, 99858, 106100, 101938, 114420]), $a([128352, 129720, 125504, 128304, 129692, 125472, 128280, 129678, 125456, 128268, 125448, 128262, 125444, 125792, 128440, 129758, 120384, 125744, 128412, 120352, 125720, 128398, 120336, 125708, 120328, 125702, 120324, 120672, 125880, 128478, 110144, 120624, 125852, 110112, 120600, 125838, 110096, 120588, 110088, 120582, 110084, 110432, 120760, 125918, 89664, 110384, 120732, 89632, 110360, 120718, 89616, 110348, 89608, 110342, 89952, 110520, 120798, 89904, 110492, 89880, 110478, 89868, 90040, 110558, 90012, 89998, 125248, 128176, 129628, 125216, 128152, 129614, 125200, 128140, 125192, 128134, 125188, 125186, 119616, 125360, 128220, 119584, 125336, 128206, 119568, 125324, 119560, 125318, 119556, 119554, 108352, 119728, 125404, 108320, 119704, 125390, 108304, 119692, 108296, 119686, 108292, 108290, 85824, 108464, 119772, 85792, 108440, 119758, 85776, 108428, 85768, 108422, 85764, 85936, 108508, 85912, 108494, 85900, 85894, 85980, 85966, 125088, 128088, 129582, 125072, 128076, 125064, 128070, 125060, 125058, 119200, 125144, 128110, 119184, 125132, 119176, 125126, 119172, 119170, 107424, 119256, 125166, 107408, 119244, 107400, 119238, 107396, 107394, 83872, 107480, 119278, 83856, 107468, 83848, 107462, 83844, 83842, 83928, 107502, 83916, 83910, 83950, 125008, 128044, 125000, 128038, 124996, 124994, 118992, 125036, 118984, 125030, 118980, 118978, 106960, 119020, 106952, 119014, 106948, 106946, 82896, 106988, 82888, 106982, 82884, 82882, 82924, 82918, 124968, 128022, 124964, 124962, 118888, 124982, 118884, 118882, 106728, 118902, 106724, 106722, 82408, 106742, 82404, 82402, 124948, 124946, 118836, 118834, 106612, 106610, 124224, 127664, 129372, 124192, 127640, 129358, 124176, 127628, 124168, 127622, 124164, 124162, 117568, 124336, 127708, 117536, 124312, 127694, 117520, 124300, 117512, 124294, 117508, 117506, 104256, 117680, 124380, 104224, 117656, 124366, 104208, 117644, 104200, 117638, 104196, 104194, 77632, 104368, 117724, 77600, 104344, 117710, 77584, 104332, 77576, 104326, 77572, 77744, 104412, 77720, 104398, 77708, 77702, 77788, 77774, 128672, 129880, 93168, 128656, 129868, 92664, 128648, 129862, 92412, 128644, 128642, 124064, 127576, 129326, 126368, 124048, 129902, 126352, 128716, 127558, 126344, 124036, 126340, 124034, 126338, 117152, 124120, 127598, 121760, 117136, 124108, 121744, 126412, 124102, 121736, 117124, 121732, 117122, 121730, 103328, 117208, 124142, 112544, 103312, 117196, 112528, 121804, 117190, 112520, 103300, 112516, 103298, 112514, 75680, 103384, 117230, 94112, 75664, 103372, 94096, 112588, 103366, 94088, 75652, 94084, 75650, 75736, 103406, 94168, 75724, 94156, 75718, 94150, 75758, 128592, 129836, 91640, 128584, 129830, 91388, 128580, 91262, 128578, 123984, 127532, 126160, 123976, 127526, 126152, 128614, 126148, 123970, 126146, 116944, 124012, 121296, 116936, 124006, 121288, 126182, 121284, 116930, 121282, 102864, 116972, 111568, 102856, 116966, 111560, 121318, 111556, 102850, 111554, 74704, 102892, 92112, 74696, 102886, 92104, 111590, 92100, 74690, 92098, 74732, 92140, 74726, 92134, 128552, 129814, 90876, 128548, 90750, 128546, 123944, 127510, 126056, 128566, 126052, 123938, 126050, 116840, 123958, 121064, 116836, 121060, 116834, 121058, 102632, 116854, 111080, 121078, 111076, 102626, 111074, 74216, 102646, 91112, 74212, 91108, 74210, 91106, 74230, 91126, 128532, 90494, 128530, 123924, 126004, 123922, 126002, 116788, 120948, 116786, 120946, 102516, 110836, 102514, 110834, 73972, 90612, 73970, 90610, 128522, 123914, 125978, 116762, 120890, 102458, 110714, 123552, 127320, 129198, 123536, 127308, 123528, 127302, 123524, 123522, 116128, 123608, 127342, 116112, 123596, 116104, 123590, 116100, 116098, 101280, 116184, 123630, 101264, 116172, 101256, 116166, 101252, 101250, 71584, 101336, 116206, 71568, 101324, 71560, 101318, 71556, 71554, 71640, 101358, 71628, 71622, 71662, 127824, 129452, 79352, 127816, 129446, 79100, 127812, 78974, 127810, 123472, 127276, 124624, 123464, 127270, 124616, 127846, 124612, 123458, 124610, 115920, 123500, 118224, 115912, 123494, 118216, 124646, 118212, 115906, 118210, 100816, 115948, 105424, 100808, 115942, 105416, 118246, 105412, 100802, 105410, 70608, 100844, 79824, 70600, 100838, 79816, 105446, 79812, 70594, 79810, 70636, 79852, 70630, 79846, 129960, 95728, 113404, 129956, 95480, 113278, 129954, 95356, 95294, 127784, 129430, 78588, 128872, 129974, 95996, 78462, 128868, 127778, 95870, 128866, 123432, 127254, 124520, 123428, 126696, 128886, 123426, 126692, 124514, 126690, 115816, 123446, 117992, 115812, 122344, 117988, 115810, 122340, 117986, 122338, 100584, 115830, 104936, 100580, 113640, 104932, 100578, 113636, 104930, 113634, 70120, 100598, 78824, 70116, 96232, 78820, 70114, 96228, 78818, 96226, 70134, 78838, 129940, 94968, 113022, 129938, 94844, 94782, 127764, 78206, 128820, 127762, 95102, 128818, 123412, 124468, 123410, 126580, 124466, 126578, 115764, 117876, 115762, 122100, 117874, 122098, 100468, 104692, 100466, 113140, 104690, 113138, 69876, 78324, 69874, 95220, 78322, 95218, 129930, 94588, 94526, 127754, 128794, 123402, 124442, 126522, 115738, 117818, 121978, 100410, 104570, 112890, 69754, 78074, 94714, 94398, 123216, 127148, 123208, 127142, 123204, 123202, 115408, 123244, 115400, 123238, 115396, 115394, 99792, 115436, 99784, 115430, 99780, 99778, 68560, 99820, 68552, 99814, 68548, 68546, 68588, 68582, 127400, 129238, 72444, 127396, 72318, 127394, 123176, 127126, 123752, 123172, 123748, 123170, 123746, 115304, 123190, 116456, 115300, 116452, 115298, 116450, 99560, 115318, 101864, 99556, 101860, 99554, 101858, 68072, 99574, 72680, 68068, 72676, 68066, 72674, 68086, 72694, 129492, 80632, 105854, 129490, 80508, 80446, 127380, 72062, 127924, 127378, 80766, 127922, 123156, 123700, 123154, 124788, 123698, 124786, 115252, 116340, 115250, 118516, 116338, 118514, 99444, 101620, 99442, 105972, 101618, 105970, 67828, 72180, 67826, 80884, 72178, 80882, 97008, 114044, 96888, 113982, 96828, 96798, 129482, 80252, 130010, 97148, 80190, 97086, 127370, 127898, 128954, 123146, 123674, 124730, 126842, 115226, 116282, 118394, 122618, 99386, 101498, 105722, 114170, 67706, 71930, 80378, 96632, 113854, 96572, 96542, 80062, 96702, 96444, 96414, 96350, 123048, 123044, 123042, 115048, 123062, 115044, 115042, 99048, 115062, 99044, 99042, 67048, 99062, 67044, 67042, 67062, 127188, 68990, 127186, 123028, 123316, 123026, 123314, 114996, 115572, 114994, 115570, 98932, 100084, 98930, 100082, 66804, 69108, 66802, 69106, 129258, 73084, 73022, 127178, 127450, 123018, 123290, 123834, 114970, 115514, 116602, 98874, 99962, 102138, 66682, 68858, 73210, 81272, 106174, 81212, 81182, 72894, 81342, 97648, 114364, 97592, 114334, 97564, 97550, 81084, 97724, 81054, 97694, 97464, 114270, 97436, 97422, 80990, 97502, 97372, 97358, 97326, 114868, 114866, 98676, 98674, 66292, 66290, 123098, 114842, 115130, 98618, 99194, 66170, 67322, 69310, 73404, 73374, 81592, 106334, 81564, 81550, 73310, 81630, 97968, 114524, 97944, 114510, 97932, 97926, 81500, 98012, 81486, 97998, 97880, 114478, 97868, 97862, 81454, 97902, 97836, 97830, 69470, 73564, 73550, 81752, 106414, 81740, 81734, 73518, 81774, 81708, 81702]), $a([109536, 120312, 86976, 109040, 120060, 86496, 108792, 119934, 86256, 108668, 86136, 129744, 89056, 110072, 129736, 88560, 109820, 129732, 88312, 109694, 129730, 88188, 128464, 129772, 89592, 128456, 129766, 89340, 128452, 89214, 128450, 125904, 128492, 125896, 128486, 125892, 125890, 120784, 125932, 120776, 125926, 120772, 120770, 110544, 120812, 110536, 120806, 110532, 84928, 108016, 119548, 84448, 107768, 119422, 84208, 107644, 84088, 107582, 84028, 129640, 85488, 108284, 129636, 85240, 108158, 129634, 85116, 85054, 128232, 129654, 85756, 128228, 85630, 128226, 125416, 128246, 125412, 125410, 119784, 125430, 119780, 119778, 108520, 119798, 108516, 108514, 83424, 107256, 119166, 83184, 107132, 83064, 107070, 83004, 82974, 129588, 83704, 107390, 129586, 83580, 83518, 128116, 83838, 128114, 125172, 125170, 119284, 119282, 107508, 107506, 82672, 106876, 82552, 106814, 82492, 82462, 129562, 82812, 82750, 128058, 125050, 119034, 82296, 106686, 82236, 82206, 82366, 82108, 82078, 76736, 103920, 117500, 76256, 103672, 117374, 76016, 103548, 75896, 103486, 75836, 129384, 77296, 104188, 129380, 77048, 104062, 129378, 76924, 76862, 127720, 129398, 77564, 127716, 77438, 127714, 124392, 127734, 124388, 124386, 117736, 124406, 117732, 117730, 104424, 117750, 104420, 104418, 112096, 121592, 126334, 92608, 111856, 121468, 92384, 111736, 121406, 92272, 111676, 92216, 111646, 92188, 75232, 103160, 117118, 93664, 74992, 103036, 93424, 112252, 102974, 93304, 74812, 93244, 74782, 93214, 129332, 75512, 103294, 129908, 129330, 93944, 75388, 129906, 93820, 75326, 93758, 127604, 75646, 128756, 127602, 94078, 128754, 124148, 126452, 124146, 126450, 117236, 121844, 117234, 121842, 103412, 103410, 91584, 111344, 121212, 91360, 111224, 121150, 91248, 111164, 91192, 111134, 91164, 91150, 74480, 102780, 91888, 74360, 102718, 91768, 111422, 91708, 74270, 91678, 129306, 74620, 129850, 92028, 74558, 91966, 127546, 128634, 124026, 126202, 116986, 121338, 102906, 90848, 110968, 121022, 90736, 110908, 90680, 110878, 90652, 90638, 74104, 102590, 91000, 74044, 90940, 74014, 90910, 74174, 91070, 90480, 110780, 90424, 110750, 90396, 90382, 73916, 90556, 73886, 90526, 90296, 110686, 90268, 90254, 73822, 90334, 90204, 90190, 71136, 101112, 116094, 70896, 100988, 70776, 100926, 70716, 70686, 129204, 71416, 101246, 129202, 71292, 71230, 127348, 71550, 127346, 123636, 123634, 116212, 116210, 101364, 101362, 79296, 105200, 118140, 79072, 105080, 118078, 78960, 105020, 78904, 104990, 78876, 78862, 70384, 100732, 79600, 70264, 100670, 79480, 105278, 79420, 70174, 79390, 129178, 70524, 129466, 79740, 70462, 79678, 127290, 127866, 123514, 124666, 115962, 118266, 100858, 113376, 122232, 126654, 95424, 113264, 122172, 95328, 113208, 122142, 95280, 113180, 95256, 113166, 95244, 78560, 104824, 117950, 95968, 78448, 104764, 95856, 113468, 104734, 95800, 78364, 95772, 78350, 95758, 70008, 100542, 78712, 69948, 96120, 78652, 69918, 96060, 78622, 96030, 70078, 78782, 96190, 94912, 113008, 122044, 94816, 112952, 122014, 94768, 112924, 94744, 112910, 94732, 94726, 78192, 104636, 95088, 78136, 104606, 95032, 113054, 95004, 78094, 94990, 69820, 78268, 69790, 95164, 78238, 95134, 94560, 112824, 121950, 94512, 112796, 94488, 112782, 94476, 94470, 78008, 104542, 94648, 77980, 94620, 77966, 94606, 69726, 78046, 94686, 94384, 112732, 94360, 112718, 94348, 94342, 77916, 94428, 77902, 94414, 94296, 112686, 94284, 94278, 77870, 94318, 94252, 94246, 68336, 99708, 68216, 99646, 68156, 68126, 68476, 68414, 127162, 123258, 115450, 99834, 72416, 101752, 116414, 72304, 101692, 72248, 101662, 72220, 72206, 67960, 99518, 72568, 67900, 72508, 67870, 72478, 68030, 72638, 80576, 105840, 118460, 80480, 105784, 118430, 80432, 105756, 80408, 105742, 80396, 80390, 72048, 101564, 80752, 71992, 101534, 80696, 71964, 80668, 71950, 80654, 67772, 72124, 67742, 80828, 72094, 80798, 114016, 122552, 126814, 96832, 113968, 122524, 96800, 113944, 122510, 96784, 113932, 96776, 113926, 96772, 80224, 105656, 118366, 97120, 80176, 105628, 97072, 114076, 105614, 97048, 80140, 97036, 80134, 97030, 71864, 101470, 80312, 71836, 97208, 80284, 71822, 97180, 80270, 97166, 67678, 71902, 80350, 97246, 96576, 113840, 122460, 96544, 113816, 122446, 96528, 113804, 96520, 113798, 96516, 96514, 80048, 105564, 96688, 80024, 105550, 96664, 113870, 96652, 80006, 96646, 71772, 80092, 71758, 96732, 80078, 96718, 96416, 113752, 122414, 96400, 113740, 96392, 113734, 96388, 96386, 79960, 105518, 96472, 79948, 96460, 79942, 96454, 71726, 79982, 96494, 96336, 113708, 96328, 113702, 96324, 96322, 79916, 96364, 79910, 96358, 96296, 113686, 96292, 96290, 79894, 96310, 66936, 99006, 66876, 66846, 67006, 68976, 100028, 68920, 99998, 68892, 68878, 66748, 69052, 66718, 69022, 73056, 102072, 116574, 73008, 102044, 72984, 102030, 72972, 72966, 68792, 99934, 73144, 68764, 73116, 68750, 73102, 66654, 68830, 73182, 81216, 106160, 118620, 81184, 106136, 118606, 81168, 106124, 81160, 106118, 81156, 81154, 72880, 101980, 81328, 72856, 101966, 81304, 106190, 81292, 72838, 81286, 68700, 72924, 68686, 81372, 72910, 81358, 114336, 122712, 126894, 114320, 122700, 114312, 122694, 114308, 114306, 81056, 106072, 118574, 97696, 81040, 106060, 97680, 114380, 106054, 97672, 81028, 97668, 81026, 97666, 72792, 101934, 81112, 72780, 97752, 81100, 72774, 97740, 81094, 97734, 68654, 72814, 81134, 97774, 114256, 122668, 114248, 122662, 114244, 114242, 80976, 106028, 97488, 80968, 106022, 97480, 114278, 97476, 80962, 97474, 72748, 81004, 72742, 97516, 80998, 97510, 114216, 122646, 114212, 114210, 80936, 106006, 97384, 80932, 97380, 80930, 97378, 72726, 80950, 97398, 114196, 114194, 80916, 97332, 80914, 97330, 66236, 66206, 67256, 99166, 67228, 67214, 66142, 67294, 69296, 100188, 69272, 100174, 69260, 69254, 67164, 69340, 67150, 69326, 73376, 102232, 116654, 73360, 102220, 73352, 102214, 73348, 73346, 69208, 100142, 73432, 102254, 73420, 69190, 73414, 67118, 69230, 73454, 106320, 118700, 106312, 118694, 106308, 106306, 73296, 102188, 81616, 106348, 102182, 81608, 73284, 81604, 73282, 81602, 69164, 73324, 69158, 81644, 73318, 81638, 122792, 126934, 122788, 122786, 106280, 118678, 114536, 106276, 114532, 106274, 114530, 73256, 102166, 81512, 73252, 98024, 81508, 73250, 98020, 81506, 98018, 69142, 73270, 81526, 98038, 122772, 122770, 106260, 114484, 106258, 114482, 73236, 81460, 73234, 97908, 81458, 97906, 122762, 106250, 114458, 73226, 81434, 97850, 66396, 66382, 67416, 99246, 67404, 67398, 66350, 67438, 69456, 100268, 69448, 100262, 69444, 69442, 67372, 69484, 67366, 69478, 102312, 116694, 102308, 102306, 69416, 100246, 73576, 102326, 73572, 69410, 73570, 67350, 69430, 73590, 118740, 118738, 102292, 106420, 102290, 106418, 69396, 73524, 69394, 81780, 73522, 81778, 118730, 102282, 106394, 69386, 73498, 81722, 66476, 66470, 67496, 99286, 67492, 67490, 66454, 67510, 100308, 100306, 67476, 69556, 67474, 69554, 116714])]); //#15953
        $1.cwtobits = function() {
            var _FD = $get($1.clusters, $k[--$j]); //#15957
            $1.v = $get(_FD, $k[--$j]); //#15957
            $k[$j++] = Infinity; //#15958
            for (var _FG = 0, _FH = 17; _FG < _FH; _FG++) { //#15958
                $k[$j++] = 0; //#15958
            } //#15958
            var _FK = $cvrs($s(17), $1.v, 2); //#15958
            for (var _FL = 0, _FM = _FK.length; _FL < _FM; _FL++) { //#15958
                $k[$j++] = $get(_FK, _FL) - 48; //#15958
            } //#15958
            var _FO = $a(); //#15958
            $k[$j++] = $geti(_FO, _FO.length - 17, 17); //#15959
        }; //#15960
        if ($1.compact) { //#15966
            $1.rwid = $f(($f(($f((17 * $1.c) + 17)) + 17)) + 1); //#15964
        } else { //#15966
            $1.rwid = $f(($f(($f(($f((17 * $1.c) + 17)) + 17)) + 17)) + 18); //#15966
        } //#15966
        $1.pixs = $a($1.rwid * $1.r); //#15968
        for (var _FY = 0, _FX = $1.r - 1; _FY <= _FX; _FY += 1) { //#15998
            $1.i = _FY; //#15971
            if (($1.i % 3) == 0) { //#15976
                $1.lcw = ((~~($1.i / 3)) * 30) + (~~(($1.r - 1) / 3)); //#15974
                $1.rcw = $f(($f(((~~($1.i / 3)) * 30) + $1.c)) - 1); //#15975
            } //#15975
            if (($1.i % 3) == 1) { //#15980
                $1.lcw = (((~~($1.i / 3)) * 30) + ($1.eclevel * 3)) + (($1.r - 1) % 3); //#15978
                $1.rcw = ((~~($1.i / 3)) * 30) + (~~(($1.r - 1) / 3)); //#15979
            } //#15979
            if (($1.i % 3) == 2) { //#15984
                $1.lcw = $f(($f(((~~($1.i / 3)) * 30) + $1.c)) - 1); //#15982
                $1.rcw = (((~~($1.i / 3)) * 30) + ($1.eclevel * 3)) + (($1.r - 1) % 3); //#15983
            } //#15983
            $k[$j++] = $1.pixs; //#15994
            $k[$j++] = $1.rwid * $1.i; //#15994
            $k[$j++] = Infinity; //#15994
            $k[$j++] = 1; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 0; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 0; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 0; //#15988
            $k[$j++] = 1; //#15988
            $k[$j++] = 0; //#15988
            $k[$j++] = 0; //#15988
            $k[$j++] = 0; //#15988
            $k[$j++] = $1.lcw; //#15988
            $k[$j++] = $1.i % 3; //#15988
            $1.cwtobits(); //#15988
            $forall($k[--$j]); //#15988
            var _G0 = $geti($1.cws, $1.c * $1.i, $1.c); //#15989
            for (var _G1 = 0, _G2 = _G0.length; _G1 < _G2; _G1++) { //#15989
                $k[$j++] = $get(_G0, _G1); //#15989
                $k[$j++] = $1.i % 3; //#15989
                $1.cwtobits(); //#15989
                $forall($k[--$j]); //#15989
            } //#15989
            if ($1.compact) { //#15994
                $k[$j++] = 1; //#15991
            } else { //#15994
                $k[$j++] = $1.rcw; //#15993
                $k[$j++] = $1.i % 3; //#15993
                $1.cwtobits(); //#15993
                $forall($k[--$j]); //#15993
                $k[$j++] = 1; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 0; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 0; //#15994
                $k[$j++] = 0; //#15994
                $k[$j++] = 0; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 0; //#15994
                $k[$j++] = 1; //#15994
                $k[$j++] = 0; //#15994
                $k[$j++] = 0; //#15994
                $k[$j++] = 1; //#15994
            } //#15994
            var _GA = $a(); //#15994
            var _GB = $k[--$j]; //#15996
            $puti($k[--$j], _GB, _GA); //#15996
        } //#15996
        var _GK = new Map([
            ["ren", $0.renmatrix],
            ["pixs", $1.pixs],
            ["pixx", $1.rwid],
            ["pixy", $1.r],
            ["height", ($1.r / 72) * $1.rowmult],
            ["width", $1.rwid / 72],
            ["opt", $1.options]
        ]); //#16008
        $k[$j++] = _GK; //#16011
        if (!$1.dontdraw) { //#16011
            $0.renmatrix(); //#16011
        } //#16011
    };
    $0.pdf417compact = function() {
        var $1 = {}; //#16041
        $1.options = $k[--$j]; //#16043
        $1.barcode = $k[--$j]; //#16044
        $1.dontdraw = false; //#16046
        $forall($1.options, function() { //#16057
            var _3 = $k[--$j]; //#16057
            $1[$k[--$j]] = _3; //#16057
        }); //#16057
        $put($1.options, "dontdraw", true); //#16060
        $put($1.options, "compact", true); //#16061
        $k[$j++] = 'args'; //#16063
        $k[$j++] = $1.barcode; //#16063
        $k[$j++] = $1.options; //#16063
        $0.pdf417(); //#16063
        var _9 = $k[--$j]; //#16063
        $1[$k[--$j]] = _9; //#16063
        $put($1.args, "opt", $1.options); //#16065
        $k[$j++] = $1.args; //#16068
        if (!$1.dontdraw) { //#16068
            $0.renmatrix(); //#16068
        } //#16068
    };
    $0.micropdf417 = function() {
        var $1 = {}; //#16098
        $1.options = $k[--$j]; //#16100
        $1.barcode = $k[--$j]; //#16101
        $1.dontdraw = false; //#16103
        $1.version = "unset"; //#16104
        $1.columns = 0; //#16105
        $1.rows = 0; //#16106
        $1.rowmult = 2; //#16107
        $1.encoding = "auto"; //#16108
        $1.cca = false; //#16109
        $1.ccb = false; //#16110
        $1.raw = false; //#16111
        $1.parse = false; //#16112
        $1.parsefnc = false; //#16113
        $forall($1.options, function() { //#16124
            var _3 = $k[--$j]; //#16124
            $1[$k[--$j]] = _3; //#16124
        }); //#16124
        if ($ne($1.version, "unset")) { //#16130
            $search($1.version, "x"); //#16127
            $j--; //#16128
            $1.columns = $k[--$j]; //#16128
            $j--; //#16129
            $1.rows = $k[--$j]; //#16129
        } //#16129
        $1.columns = ~~$z($1.columns); //#16132
        $1.rows = ~~$z($1.rows); //#16133
        $1.rowmult = +$1.rowmult; //#16134
        if ($1.raw) { //#16136
            $1.encoding = "raw"; //#16136
        } //#16136
        if ($1.cca) { //#16137
            $1.encoding = "cca"; //#16137
        } //#16137
        if ($eq($1.encoding, "raw") || $eq($1.encoding, "cca")) { //#16151
            $1.datcws = $a($1.barcode.length); //#16141
            $1.i = 0; //#16142
            $1.j = 0; //#16142
            for (;;) { //#16149
                if ($1.i == $1.barcode.length) { //#16144
                    break; //#16144
                } //#16144
                $1.cw = ~~$z($geti($1.barcode, $1.i + 1, 3)); //#16145
                $put($1.datcws, $1.j, $1.cw); //#16146
                $1.i = $1.i + 4; //#16147
                $1.j = $1.j + 1; //#16148
            } //#16148
            $1.datcws = $geti($1.datcws, 0, $1.j); //#16150
        } //#16150
        if ($1.ccb) { //#16153
            $1.encoding = "ccb"; //#16153
        } //#16153
        $1.encb = function() {
            $1.in = $k[--$j]; //#16156
            $1.inlen = $1.in.length; //#16157
            $1.out = $a(((~~($1.inlen / 6)) * 5) + ($1.inlen % 6)); //#16158
            for (var _d = 0, _c = (~~($1.inlen / 6)) - 1; _d <= _c; _d += 1) { //#16181
                $1.k = _d; //#16160
                $k[$j++] = Infinity; //#16161
                $aload($geti($1.in, $1.k * 6, 3)); //#16161
                $1.msbs = $a(); //#16161
                $k[$j++] = Infinity; //#16164
                $aload($1.msbs); //#16163
                var _j = $k[--$j]; //#16163
                var _k = $k[--$j]; //#16163
                var _l = $k[--$j]; //#16163
                $k[$j++] = $f(($f(_j + (_k * 256))) + (_l * 65536)); //#16164
                for (var _m = 0, _n = 3; _m < _n; _m++) { //#16164
                    var _o = $k[--$j]; //#16164
                    $k[$j++] = _o % 900; //#16164
                    $k[$j++] = ~~(_o / 900); //#16164
                } //#16164
                $1.mscs = $a(); //#16164
                $k[$j++] = Infinity; //#16166
                $aload($geti($1.in, ($1.k * 6) + 3, 3)); //#16166
                $1.lsbs = $a(); //#16166
                $k[$j++] = Infinity; //#16169
                $aload($1.lsbs); //#16168
                var _v = $k[--$j]; //#16168
                var _w = $k[--$j]; //#16168
                var _x = $k[--$j]; //#16168
                $k[$j++] = $f(($f(_v + (_w * 256))) + (_x * 65536)); //#16169
                for (var _y = 0, _z = 3; _y < _z; _y++) { //#16169
                    var _10 = $k[--$j]; //#16169
                    $k[$j++] = _10 % 900; //#16169
                    $k[$j++] = ~~(_10 / 900); //#16169
                } //#16169
                $1.lscs = $a(); //#16169
                var _13 = $get($1.lscs, 0); //#16171
                var _15 = $get($1.mscs, 0); //#16171
                $put($1.out, ($1.k * 5) + 4, ($f(_13 + (_15 * 316))) % 900); //#16172
                var _19 = $get($1.lscs, 1); //#16173
                var _1B = $get($1.mscs, 0); //#16173
                var _1D = $get($1.mscs, 1); //#16173
                $put($1.out, ($1.k * 5) + 3, ($f(($f(($f((~~(($f(_13 + (_15 * 316))) / 900)) + _19)) + (_1B * 641))) + (_1D * 316))) % 900); //#16174
                var _1H = $get($1.lscs, 2); //#16175
                var _1J = $get($1.mscs, 0); //#16175
                var _1L = $get($1.mscs, 1); //#16175
                var _1N = $get($1.mscs, 2); //#16175
                $put($1.out, ($1.k * 5) + 2, ($f(($f(($f(($f((~~(($f(($f(($f((~~(($f(_13 + (_15 * 316))) / 900)) + _19)) + (_1B * 641))) + (_1D * 316))) / 900)) + _1H)) + (_1J * 20))) + (_1L * 641))) + (_1N * 316))) % 900); //#16176
                var _1R = $get($1.lscs, 3); //#16177
                var _1T = $get($1.mscs, 1); //#16177
                var _1V = $get($1.mscs, 2); //#16177
                $put($1.out, ($1.k * 5) + 1, ($f(($f(($f((~~(($f(($f(($f(($f((~~(($f(($f(($f((~~(($f(_13 + (_15 * 316))) / 900)) + _19)) + (_1B * 641))) + (_1D * 316))) / 900)) + _1H)) + (_1J * 20))) + (_1L * 641))) + (_1N * 316))) / 900)) + _1R)) + (_1T * 20))) + (_1V * 641))) % 900); //#16178
                $put($1.out, $1.k * 5, ($f((~~(($f(($f(($f((~~(($f(($f(($f(($f((~~(($f(($f(($f((~~(($f(_13 + (_15 * 316))) / 900)) + _19)) + (_1B * 641))) + (_1D * 316))) / 900)) + _1H)) + (_1J * 20))) + (_1L * 641))) + (_1N * 316))) / 900)) + _1R)) + (_1T * 20))) + (_1V * 641))) / 900)) + ($get($1.mscs, 2) * 20))) % 900); //#16180
            } //#16180
            $1.rem = $1.inlen % 6; //#16182
            if ($1.rem != 0) { //#16187
                $k[$j++] = $1.out; //#16185
                $k[$j++] = $1.out.length - $1.rem; //#16185
                $k[$j++] = Infinity; //#16185
                $aload($geti($1.in, $1.inlen - $1.rem, $1.rem)); //#16185
                var _1m = $a(); //#16185
                var _1n = $k[--$j]; //#16186
                $puti($k[--$j], _1n, _1m); //#16186
            } //#16186
            $k[$j++] = $1.out; //#16188
        }; //#16189
        if ($eq($1.encoding, "byte") || $eq($1.encoding, "ccb")) { //#16197
            $1.barlen = $1.barcode.length; //#16192
            $1.datcws = $a((((~~($1.barlen / 6)) * 5) + ($1.barlen % 6)) + 1); //#16193
            var _1y = (($1.barlen % 6) == 0) ? 924 : 901; //#16194
            $put($1.datcws, 0, _1y); //#16194
            $k[$j++] = $1.datcws; //#16195
            $k[$j++] = 1; //#16195
            $k[$j++] = Infinity; //#16195
            $forall($1.barcode); //#16195
            var _21 = $a(); //#16195
            $k[$j++] = _21; //#16195
            $1.encb(); //#16195
            var _22 = $k[--$j]; //#16195
            var _23 = $k[--$j]; //#16195
            $puti($k[--$j], _23, _22); //#16195
            if ($eq($1.encoding, "ccb")) { //#16196
                $k[$j++] = Infinity; //#16196
                $k[$j++] = 920; //#16196
                $aload($1.datcws); //#16196
                $1.datcws = $a(); //#16196
            } //#16196
        } //#16196
        if ($eq($1.encoding, "auto")) { //#16601
            var _2B = new Map([
                ["parse", $1.parse],
                ["parsefnc", $1.parsefnc],
                ["eci", true]
            ]); //#16205
            $1.fncvals = _2B; //#16207
            $k[$j++] = 'msg'; //#16208
            $k[$j++] = $1.barcode; //#16208
            $k[$j++] = $1.fncvals; //#16208
            $0.parseinput(); //#16208
            var _2E = $k[--$j]; //#16208
            $1[$k[--$j]] = _2E; //#16208
            $1.msglen = $1.msg.length; //#16209
            $1.T = 0; //#16212
            $1.N = 1; //#16212
            $1.B = 2; //#16212
            $1.A = 0; //#16213
            $1.L = 1; //#16213
            $1.M = 2; //#16213
            $1.P = 3; //#16213
            $1.tl = -1; //#16216
            $1.nl = -2; //#16216
            $1.bl = -3; //#16216
            $1.bl6 = -4; //#16216
            $1.bs = -5; //#16216
            $1.al = -6; //#16219
            $1.ll = -7; //#16219
            $1.ml = -8; //#16219
            $1.pl = -9; //#16219
            $1.as = -10; //#16219
            $1.ps = -11; //#16219
            $1.charmaps = $a([$a(["A", "a", "0", ";"]), $a(["B", "b", "1", "<"]), $a(["C", "c", "2", ">"]), $a(["D", "d", "3", "@"]), $a(["E", "e", "4", "["]), $a(["F", "f", "5", 92]), $a(["G", "g", "6", "]"]), $a(["H", "h", "7", "_"]), $a(["I", "i", "8", "`"]), $a(["J", "j", "9", "~"]), $a(["K", "k", "&", "!"]), $a(["L", "l", 13, 13]), $a(["M", "m", 9, 9]), $a(["N", "n", ",", ","]), $a(["O", "o", ":", ":"]), $a(["P", "p", "#", 10]), $a(["Q", "q", "-", "-"]), $a(["R", "r", ".", "."]), $a(["S", "s", "$", "$"]), $a(["T", "t", "/", "/"]), $a(["U", "u", "+", "\""]), $a(["V", "v", "%", "|"]), $a(["W", "w", "*", "*"]), $a(["X", "x", "=", 40]), $a(["Y", "y", "^", 41]), $a(["Z", "z", $1.pl, "?"]), $a([" ", " ", " ", "{"]), $a([$1.ll, $1.as, $1.ll, "}"]), $a([$1.ml, $1.ml, $1.al, "'"]), $a([$1.ps, $1.ps, $1.ps, $1.al])]); //#16254
            $1.charvals = $a([new Map, new Map, new Map, new Map]); //#16257
            $1.alltext = new Map; //#16258
            for (var _30 = 0, _2z = $1.charmaps.length - 1; _30 <= _2z; _30 += 1) { //#16268
                $1.i = _30; //#16260
                $1.encs = $get($1.charmaps, $1.i); //#16261
                for (var _34 = 0; _34 <= 3; _34 += 1) { //#16267
                    $1.j = _34; //#16263
                    var _37 = $get($1.encs, $1.j); //#16264
                    $k[$j++] = _37; //#16264
                    if ($eq($type(_37), 'stringtype')) { //#16264
                        var _3A = $get($k[--$j], 0); //#16264
                        $k[$j++] = _3A; //#16264
                    } //#16264
                    var _3B = $k[--$j]; //#16265
                    $put($get($1.charvals, $1.j), _3B, $1.i); //#16265
                    $put($1.alltext, _3B, -1); //#16266
                } //#16266
            } //#16266
            $1.e = 10000; //#16270
            $1.latlen = $a([$a([0, 1, 1, 2]), $a([2, 0, 1, 2]), $a([1, 1, 0, 1]), $a([1, 2, 2, 0])]); //#16278
            $1.latseq = $a([$a([$a([]), $a([$1.ll]), $a([$1.ml]), $a([$1.ml, $1.pl])]), $a([$a([$1.ml, $1.al]), $a([]), $a([$1.ml]), $a([$1.ml, $1.pl])]), $a([$a([$1.al]), $a([$1.ll]), $a([]), $a([$1.pl])]), $a([$a([$1.al]), $a([$1.al, $1.ll]), $a([$1.al, $1.ml]), $a([])])]); //#16286
            $1.shftlen = $a([$a([$1.e, $1.e, $1.e, 1]), $a([1, $1.e, $1.e, 1]), $a([$1.e, $1.e, $1.e, 1]), $a([$1.e, $1.e, $1.e, $1.e])]); //#16294
            $k[$j++] = Infinity; //#16297
            for (var _4G = 0, _4H = $1.msglen; _4G < _4H; _4G++) { //#16297
                $k[$j++] = 0; //#16297
            } //#16297
            $k[$j++] = 0; //#16297
            $1.numdigits = $a(); //#16297
            $k[$j++] = Infinity; //#16298
            for (var _4K = 0, _4L = $1.msglen; _4K < _4L; _4K++) { //#16298
                $k[$j++] = 0; //#16298
            } //#16298
            $k[$j++] = 0; //#16298
            $1.numtext = $a(); //#16298
            $k[$j++] = Infinity; //#16299
            for (var _4O = 0, _4P = $1.msglen; _4O < _4P; _4O++) { //#16299
                $k[$j++] = 0; //#16299
            } //#16299
            $k[$j++] = 0; //#16299
            $1.numbytes = $a(); //#16299
            $k[$j++] = Infinity; //#16300
            for (var _4S = 0, _4T = $1.msglen; _4S < _4T; _4S++) { //#16300
                $k[$j++] = 0; //#16300
            } //#16300
            $1.iseci = $a(); //#16300
            for (var _4W = $1.msglen - 1; _4W >= 0; _4W -= 1) { //#16313
                $1.i = _4W; //#16302
                var _4Z = $get($1.msg, $1.i); //#16303
                if ((_4Z >= 48) && (_4Z <= 57)) { //#16305
                    $put($1.numdigits, $1.i, $f($get($1.numdigits, $1.i + 1) + 1)); //#16304
                } //#16304
                var _4j = $get($1.alltext, $get($1.msg, $1.i)) !== undefined; //#16306
                if (_4j && ($get($1.numdigits, $1.i) < 13)) { //#16308
                    $put($1.numtext, $1.i, $f($get($1.numtext, $1.i + 1) + 1)); //#16307
                } //#16307
                if ((($get($1.msg, $1.i) >= 0) && ($get($1.numtext, $1.i) < 5)) && ($get($1.numdigits, $1.i) < 13)) { //#16311
                    $put($1.numbytes, $1.i, $f($get($1.numbytes, $1.i + 1) + 1)); //#16310
                } //#16310
                $put($1.iseci, $1.i, $get($1.msg, $1.i) <= -1000000); //#16312
            } //#16312
            $1.numdigits = $geti($1.numdigits, 0, $1.msglen); //#16314
            $1.numtext = $geti($1.numtext, 0, $1.msglen); //#16315
            $1.numbytes = $geti($1.numbytes, 0, $1.msglen); //#16316
            $1.seq = $a([]); //#16318
            $1.seqlen = 0; //#16318
            $1.state = $1.B; //#16318
            $1.p = 0; //#16318
            for (;;) { //#16372
                if ($1.p == $1.msglen) { //#16319
                    break; //#16319
                } //#16319
                if ($get($1.iseci, $1.p)) { //#16370
                    $1.eci = $get($1.msg, $1.p); //#16321
                    $k[$j++] = Infinity; //#16324
                    $aload($1.seq); //#16323
                    $k[$j++] = $a([$1.eci]); //#16324
                    $1.seq = $a(); //#16324
                    $1.p = $1.p + 1; //#16326
                    $k[$j++] = 'seqlen'; //#16327
                    $k[$j++] = $1.seqlen; //#16327
                    if ($1.eci <= -1810900) { //#16327
                        $k[$j++] = 2; //#16327
                    } else { //#16327
                        var _5c = ($1.eci <= -1000900) ? 3 : 2; //#16327
                        $k[$j++] = _5c; //#16327
                    } //#16327
                    var _5d = $k[--$j]; //#16327
                    var _5e = $k[--$j]; //#16327
                    $1[$k[--$j]] = $f(_5e + _5d); //#16327
                } else { //#16370
                    $1.n = $get($1.numdigits, $1.p); //#16329
                    if ($1.n >= 13) { //#16370
                        $k[$j++] = Infinity; //#16334
                        $aload($1.seq); //#16332
                        $k[$j++] = $1.nl; //#16334
                        $k[$j++] = Infinity; //#16334
                        $aload($geti($1.msg, $1.p, $1.n)); //#16334
                        var _5q = $a(); //#16334
                        $k[$j++] = _5q; //#16334
                        $1.seq = $a(); //#16334
                        $1.state = $1.N; //#16336
                        $1.p = $f($1.p + $1.n); //#16337
                        $1.seqlen = $f(($1.seqlen + 1) + $1.n); //#16338
                    } else { //#16370
                        $1.t = $get($1.numtext, $1.p); //#16340
                        if ($1.t >= 5) { //#16370
                            $k[$j++] = Infinity; //#16347
                            $aload($1.seq); //#16343
                            if ($1.state != $1.T) { //#16346
                                $k[$j++] = $1.tl; //#16345
                            } //#16345
                            $k[$j++] = Infinity; //#16347
                            $aload($geti($1.msg, $1.p, $1.t)); //#16347
                            var _69 = $a(); //#16347
                            $k[$j++] = _69; //#16347
                            $1.seq = $a(); //#16347
                            $1.state = $1.T; //#16349
                            $1.p = $f($1.p + $1.t); //#16350
                            $1.seqlen = $f($f($1.seqlen + 1) + $1.t); //#16351
                        } else { //#16370
                            $1.b = $get($1.numbytes, $1.p); //#16353
                            if (($1.b == 1) && ($1.state == $1.T)) { //#16370
                                $k[$j++] = Infinity; //#16358
                                $aload($1.seq); //#16356
                                $k[$j++] = $1.bs; //#16358
                                $k[$j++] = $a([$get($1.msg, $1.p)]); //#16358
                                $1.seq = $a(); //#16358
                                $1.p = $f($1.p + $1.b); //#16360
                                $1.seqlen = $f($1.seqlen + 2); //#16361
                            } else { //#16370
                                $k[$j++] = Infinity; //#16366
                                $aload($1.seq); //#16364
                                var _6Y = (($1.b % 6) != 0) ? $1.bl : $1.bl6; //#16365
                                $k[$j++] = _6Y; //#16366
                                $k[$j++] = Infinity; //#16366
                                $aload($geti($1.msg, $1.p, $1.b)); //#16366
                                var _6d = $a(); //#16366
                                $k[$j++] = _6d; //#16366
                                $1.seq = $a(); //#16366
                                $1.state = $1.B; //#16368
                                $1.p = $f($1.p + $1.b); //#16369
                                $1.seqlen = $f($f($1.seqlen + 1) + $1.b); //#16370
                            } //#16370
                        } //#16370
                    } //#16370
                } //#16370
            } //#16370
            $k[$j++] = Infinity; //#16375
            $k[$j++] = $1.tl; //#16375
            $k[$j++] = 900; //#16375
            $k[$j++] = $1.bl; //#16375
            $k[$j++] = 901; //#16375
            $k[$j++] = $1.bl6; //#16375
            $k[$j++] = 924; //#16375
            $k[$j++] = $1.nl; //#16375
            $k[$j++] = 902; //#16375
            $k[$j++] = $1.bs; //#16375
            $k[$j++] = 913; //#16375
            $1.latchcws = $d(); //#16376
            $1.enca = function() {
                var _6u = $get($get($1.charvals, $1.A), $k[--$j]); //#16379
                $k[$j++] = _6u; //#16379
            }; //#16379
            $1.encl = function() {
                var _6z = $get($get($1.charvals, $1.L), $k[--$j]); //#16380
                $k[$j++] = _6z; //#16380
            }; //#16380
            $1.encm = function() {
                var _74 = $get($get($1.charvals, $1.M), $k[--$j]); //#16381
                $k[$j++] = _74; //#16381
            }; //#16381
            $1.encp = function() {
                var _79 = $get($get($1.charvals, $1.P), $k[--$j]); //#16382
                $k[$j++] = _79; //#16382
            }; //#16382
            $1.textencfuncs = $a(['enca', 'encl', 'encm', 'encp']); //#16383
            $1.addtotext = function() {
                $put($1.text, $1.l, $k[--$j]); //#16386
                $1.l = $1.l + 1; //#16387
            }; //#16388
            $1.enct = function() {
                $1.in = $k[--$j]; //#16392
                $1.curlen = $a([$1.e, $1.e, $1.e, $1.e]); //#16394
                $put($1.curlen, $1.submode, 0); //#16395
                $1.curseq = $a([$a([]), $a([]), $a([]), $a([])]); //#16396
                $forall($1.in, function() { //#16466
                    $1.char = $k[--$j]; //#16401
                    for (;;) { //#16422
                        $1.imp = false; //#16405
                        var _7Y = $a([$1.A, $1.L, $1.M, $1.P]); //#16406
                        for (var _7Z = 0, _7a = _7Y.length; _7Z < _7a; _7Z++) { //#16420
                            $1.x = $get(_7Y, _7Z); //#16407
                            var _7g = $a([$1.A, $1.L, $1.M, $1.P]); //#16408
                            for (var _7h = 0, _7i = _7g.length; _7h < _7i; _7h++) { //#16419
                                $1.y = $get(_7g, _7h); //#16409
                                $1.cost = $f($get($1.curlen, $1.x) + $get($get($1.latlen, $1.x), $1.y)); //#16410
                                if ($1.cost < $get($1.curlen, $1.y)) { //#16418
                                    $put($1.curlen, $1.y, $1.cost); //#16412
                                    $k[$j++] = $1.curseq; //#16415
                                    $k[$j++] = $1.y; //#16415
                                    $k[$j++] = Infinity; //#16415
                                    $aload($get($1.curseq, $1.x)); //#16414
                                    $aload($get($get($1.latseq, $1.x), $1.y)); //#16415
                                    var _89 = $a(); //#16415
                                    var _8A = $k[--$j]; //#16416
                                    $put($k[--$j], _8A, _89); //#16416
                                    $1.imp = true; //#16417
                                } //#16417
                            } //#16417
                        } //#16417
                        if (!$1.imp) { //#16421
                            break; //#16421
                        } //#16421
                    } //#16421
                    $1.nxtlen = $a([$1.e, $1.e, $1.e, $1.e]); //#16425
                    $1.nxtseq = $a(4); //#16426
                    var _8N = $a([$1.A, $1.L, $1.M, $1.P]); //#16428
                    for (var _8O = 0, _8P = _8N.length; _8O < _8P; _8O++) { //#16461
                        $1.x = $get(_8N, _8O); //#16429
                        for (;;) { //#16459
                            var _8V = $get($get($1.charvals, $1.x), $1.char) !== undefined; //#16433
                            if (!_8V) { //#16433
                                break; //#16433
                            } //#16433
                            $1.cost = $f($get($1.curlen, $1.x) + 1); //#16436
                            if ($1.cost < $get($1.nxtlen, $1.x)) { //#16440
                                $put($1.nxtlen, $1.x, $1.cost); //#16438
                                $k[$j++] = $1.nxtseq; //#16439
                                $k[$j++] = $1.x; //#16439
                                $k[$j++] = Infinity; //#16439
                                $aload($get($1.curseq, $1.x)); //#16439
                                $k[$j++] = $1.char; //#16439
                                var _8m = $a(); //#16439
                                var _8n = $k[--$j]; //#16439
                                $put($k[--$j], _8n, _8m); //#16439
                            } //#16439
                            var _8t = $a([$1.A, $1.L, $1.M, $1.P]); //#16443
                            for (var _8u = 0, _8v = _8t.length; _8u < _8v; _8u++) { //#16456
                                $1.y = $get(_8t, _8u); //#16444
                                if ($ne($1.x, $1.y)) { //#16455
                                    $1.cost = $f($f($get($1.curlen, $1.y) + $get($get($1.shftlen, $1.y), $1.x)) + 1); //#16446
                                    if ($1.cost < $get($1.nxtlen, $1.y)) { //#16454
                                        $put($1.nxtlen, $1.y, $1.cost); //#16448
                                        $k[$j++] = $1.nxtseq; //#16452
                                        $k[$j++] = $1.y; //#16452
                                        $k[$j++] = Infinity; //#16452
                                        $aload($get($1.curseq, $1.y)); //#16450
                                        var _9L = ($1.x == $1.A) ? $1.as : $1.ps; //#16451
                                        $k[$j++] = _9L; //#16452
                                        $k[$j++] = $1.char; //#16452
                                        var _9N = $a(); //#16452
                                        var _9O = $k[--$j]; //#16453
                                        $put($k[--$j], _9O, _9N); //#16453
                                    } //#16453
                                } //#16453
                            } //#16453
                            break; //#16458
                        } //#16458
                    } //#16458
                    $1.curlen = $1.nxtlen; //#16463
                    $1.curseq = $1.nxtseq; //#16464
                }); //#16464
                $1.minseq = $1.e; //#16469
                var _9X = $a([$1.A, $1.L, $1.M, $1.P]); //#16470
                for (var _9Y = 0, _9Z = _9X.length; _9Y < _9Z; _9Y++) { //#16476
                    $1.k = $get(_9X, _9Y); //#16471
                    if ($get($1.curlen, $1.k) < $1.minseq) { //#16475
                        $1.minseq = $get($1.curlen, $1.k); //#16473
                        $1.txtseq = $get($1.curseq, $1.k); //#16474
                    } //#16474
                } //#16474
                $1.text = $a($1.minseq); //#16479
                $1.k = 0; //#16480
                $1.l = 0; //#16480
                for (;;) { //#16500
                    if ($1.k >= $1.txtseq.length) { //#16481
                        break; //#16481
                    } //#16481
                    $1.char = $get($1.txtseq, $1.k); //#16483
                    $k[$j++] = $1.char; //#16486
                    if ($1[$get($1.textencfuncs, $1.submode)]() === true) {
                        break;
                    } //#16486
                    $1.addtotext(); //#16486
                    $1.k = $1.k + 1; //#16487
                    if (($1.char == $1.as) || ($1.char == $1.ps)) { //#16493
                        $k[$j++] = $get($1.txtseq, $1.k); //#16491
                        if ($1.char == $1.as) { //#16491
                            $1.enca(); //#16491
                        } else { //#16491
                            $1.encp(); //#16491
                        } //#16491
                        $1.addtotext(); //#16491
                        $1.k = $1.k + 1; //#16492
                    } //#16492
                    if ($1.char == $1.al) { //#16496
                        $1.submode = $1.A; //#16496
                    } //#16496
                    if ($1.char == $1.ll) { //#16497
                        $1.submode = $1.L; //#16497
                    } //#16497
                    if ($1.char == $1.ml) { //#16498
                        $1.submode = $1.M; //#16498
                    } //#16498
                    if ($1.char == $1.pl) { //#16499
                        $1.submode = $1.P; //#16499
                    } //#16499
                } //#16499
                if (($1.text.length % 2) == 1) { //#16510
                    if ($1.submode == $1.P) { //#16507
                        $k[$j++] = 'pad'; //#16504
                        $k[$j++] = $1.al; //#16504
                        $1.encp(); //#16504
                        var _AO = $k[--$j]; //#16504
                        $1[$k[--$j]] = _AO; //#16504
                        $1.submode = $1.A; //#16505
                    } else { //#16507
                        $k[$j++] = 'pad'; //#16507
                        $k[$j++] = $1.ps; //#16507
                        if ($1[$get($1.textencfuncs, $1.submode)]() === true) {
                            return true;
                        } //#16507
                        var _AW = $k[--$j]; //#16507
                        $1[$k[--$j]] = _AW; //#16507
                    } //#16507
                    $k[$j++] = Infinity; //#16509
                    $aload($1.text); //#16509
                    $k[$j++] = $1.pad; //#16509
                    $1.text = $a(); //#16509
                } //#16509
                $1.out = $a(~~($1.text.length / 2)); //#16512
                for (var _Af = 0, _Ae = $1.out.length - 1; _Af <= _Ae; _Af += 1) { //#16516
                    $1.k = _Af; //#16514
                    $put($1.out, $1.k, $f(($get($1.text, $1.k * 2) * 30) + $get($1.text, ($1.k * 2) + 1))); //#16515
                } //#16515
                $k[$j++] = $1.out; //#16518
            }; //#16520
            $1.encn = function() {
                $1.in = $k[--$j]; //#16523
                $1.out = $a([]); //#16524
                for (var _At = 0, _As = $1.in.length - 1; _At <= _As; _At += 44) { //#16552
                    $1.k = _At; //#16526
                    $k[$j++] = Infinity; //#16529
                    var _Ay = $1.in.length - $1.k; //#16528
                    $k[$j++] = 1; //#16528
                    $k[$j++] = $1.in; //#16528
                    $k[$j++] = $1.k; //#16528
                    $k[$j++] = _Ay; //#16528
                    if (_Ay > 44) { //#16528
                        $j--; //#16528
                        $k[$j++] = 44; //#16528
                    } //#16528
                    var _Az = $k[--$j]; //#16528
                    var _B0 = $k[--$j]; //#16528
                    var _B2 = $geti($k[--$j], _B0, _Az); //#16528
                    for (var _B3 = 0, _B4 = _B2.length; _B3 < _B4; _B3++) { //#16529
                        $k[$j++] = $f($get(_B2, _B3) - 48); //#16529
                    } //#16529
                    $1.gmod = $a(); //#16529
                    $1.cwn = $a([]); //#16531
                    for (;;) { //#16550
                        $1.dv = 900; //#16532
                        $1.gmul = $a([]); //#16533
                        $1.val = 0; //#16533
                        for (;;) { //#16545
                            if ($1.gmod.length == 0) { //#16534
                                break; //#16534
                            } //#16534
                            $1.val = $f(($1.val * 10) + $get($1.gmod, 0)); //#16535
                            $1.gmod = $geti($1.gmod, 1, $1.gmod.length - 1); //#16536
                            if ($1.val < $1.dv) { //#16542
                                if ($1.gmul.length != 0) { //#16540
                                    $k[$j++] = Infinity; //#16539
                                    $aload($1.gmul); //#16539
                                    $k[$j++] = 0; //#16539
                                    $1.gmul = $a(); //#16539
                                } //#16539
                            } else { //#16542
                                $k[$j++] = Infinity; //#16542
                                $aload($1.gmul); //#16542
                                $k[$j++] = ~~($1.val / $1.dv); //#16542
                                $1.gmul = $a(); //#16542
                            } //#16542
                            $1.val = $1.val % $1.dv; //#16544
                        } //#16544
                        $1.dv = $1.val; //#16546
                        $k[$j++] = Infinity; //#16547
                        $k[$j++] = $1.dv; //#16547
                        $aload($1.cwn); //#16547
                        $1.cwn = $a(); //#16547
                        $1.gmod = $1.gmul; //#16548
                        if ($1.gmul.length == 0) { //#16549
                            break; //#16549
                        } //#16549
                    } //#16549
                    $k[$j++] = Infinity; //#16551
                    $aload($1.out); //#16551
                    $aload($1.cwn); //#16551
                    $1.out = $a(); //#16551
                } //#16551
                $k[$j++] = $1.out; //#16553
            }; //#16554
            $1.ence = function() {
                var _Bd = $f((-$get($k[--$j], 0)) - 1000000); //#16558
                $k[$j++] = _Bd; //#16566
                if (_Bd <= 899) { //#16565
                    var _Be = $k[--$j]; //#16559
                    $k[$j++] = 927; //#16559
                    $k[$j++] = _Be; //#16559
                    $astore($a(2)); //#16559
                } else { //#16565
                    var _Bg = $k[--$j]; //#16560
                    $k[$j++] = _Bg; //#16566
                    if (_Bg <= 810899) { //#16565
                        var _Bh = $k[--$j]; //#16561
                        $k[$j++] = 926; //#16561
                        $k[$j++] = (~~(_Bh / 900)) - 1; //#16561
                        $k[$j++] = _Bh % 900; //#16561
                        $astore($a(3)); //#16561
                    } else { //#16565
                        var _Bj = $k[--$j]; //#16562
                        $k[$j++] = _Bj; //#16566
                        if (_Bj <= 811799) { //#16565
                            var _Bk = $k[--$j]; //#16563
                            $k[$j++] = 925; //#16563
                            $k[$j++] = $f(_Bk - 810900); //#16563
                            $astore($a(2)); //#16563
                        } else { //#16565
                            $k[$j++] = 'bwipp.pdf417badECI'; //#16565
                            $k[$j++] = "PDF417 supports ECIs 000000 to 811799"; //#16565
                            $0.raiseerror(); //#16565
                        } //#16565
                    } //#16565
                } //#16565
            }; //#16567
            $1.encfuncs = $a(['enct', 'encn', 'encb']); //#16569
            $1.addtocws = function() {
                var _Bn = $k[--$j]; //#16572
                $puti($1.datcws, $1.j, _Bn); //#16572
                $1.j = _Bn.length + $1.j; //#16573
            }; //#16574
            $1.datcws = $a($1.seqlen); //#16578
            $1.i = 0; //#16579
            $1.j = 0; //#16579
            for (;;) { //#16599
                if ($1.i >= $1.seq.length) { //#16580
                    break; //#16580
                } //#16580
                $1.chars = $get($1.seq, $1.i); //#16581
                if ($eq($type($1.chars), 'arraytype')) { //#16595
                    if ($get($1.chars, 0) <= -1000000) { //#16586
                        $k[$j++] = $1.chars; //#16584
                        $1.ence(); //#16584
                        $1.addtocws(); //#16584
                    } else { //#16586
                        $k[$j++] = $1.chars; //#16586
                        if ($1[$get($1.encfuncs, $1.state)]() === true) {
                            break;
                        } //#16586
                        $1.addtocws(); //#16586
                    } //#16586
                } else { //#16595
                    $k[$j++] = $a([$get($1.latchcws, $1.chars)]); //#16589
                    $1.addtocws(); //#16589
                    if ($1.chars == $1.tl) { //#16590
                        $1.state = $1.T; //#16590
                        $1.submode = $1.A; //#16590
                    } //#16590
                    if ($1.chars == $1.nl) { //#16591
                        $1.state = $1.N; //#16591
                    } //#16591
                    if (($1.chars == $1.bl) || ($1.chars == $1.bl6)) { //#16592
                        $1.state = $1.B; //#16592
                    } //#16592
                    if ($1.chars == $1.bs) { //#16596
                        $1.i = $1.i + 1; //#16594
                        $k[$j++] = $get($1.seq, $1.i); //#16595
                        $1.encb(); //#16595
                        $1.addtocws(); //#16595
                    } //#16595
                } //#16595
                $1.i = $1.i + 1; //#16598
            } //#16598
            $1.datcws = $geti($1.datcws, 0, $1.j); //#16600
        } //#16600
        $1.metrics = $a([$a([1, 11, 7, 1, 0, 9]), $a([1, 14, 7, 8, 0, 8]), $a([1, 17, 7, 36, 0, 36]), $a([1, 20, 8, 19, 0, 19]), $a([1, 24, 8, 9, 0, 17]), $a([1, 28, 8, 25, 0, 33]), $a([2, 8, 8, 1, 0, 1]), $a([2, 11, 9, 1, 0, 9]), $a([2, 14, 9, 8, 0, 8]), $a([2, 17, 10, 36, 0, 36]), $a([2, 20, 11, 19, 0, 19]), $a([2, 23, 13, 9, 0, 17]), $a([2, 26, 15, 27, 0, 35]), $a([3, 6, 12, 1, 1, 1]), $a([3, 8, 14, 7, 7, 7]), $a([3, 10, 16, 15, 15, 15]), $a([3, 12, 18, 25, 25, 25]), $a([3, 15, 21, 37, 37, 37]), $a([3, 20, 26, 1, 17, 33]), $a([3, 26, 32, 1, 9, 17]), $a([3, 32, 38, 21, 29, 37]), $a([3, 38, 44, 15, 31, 47]), $a([3, 44, 50, 1, 25, 49]), $a([4, 4, 8, 47, 19, 43]), $a([4, 6, 12, 1, 1, 1]), $a([4, 8, 14, 7, 7, 7]), $a([4, 10, 16, 15, 15, 15]), $a([4, 12, 18, 25, 25, 25]), $a([4, 15, 21, 37, 37, 37]), $a([4, 20, 26, 1, 17, 33]), $a([4, 26, 32, 1, 9, 17]), $a([4, 32, 38, 21, 29, 37]), $a([4, 38, 44, 15, 31, 47]), $a([4, 44, 50, 1, 25, 49])]); //#16640
        $1.ccametrics = $a([$a([2, 5, 4, 39, 0, 19]), $a([2, 6, 4, 1, 0, 33]), $a([2, 7, 5, 32, 0, 12]), $a([2, 8, 5, 8, 0, 40]), $a([2, 9, 6, 14, 0, 46]), $a([2, 10, 6, 43, 0, 23]), $a([2, 12, 7, 20, 0, 52]), $a([3, 4, 4, 11, 43, 23]), $a([3, 5, 5, 1, 33, 13]), $a([3, 6, 6, 5, 37, 17]), $a([3, 7, 7, 15, 47, 27]), $a([3, 8, 7, 21, 1, 33]), $a([4, 3, 4, 40, 20, 52]), $a([4, 4, 5, 43, 23, 3]), $a([4, 5, 6, 46, 26, 6]), $a([4, 6, 7, 34, 14, 46]), $a([4, 7, 8, 29, 9, 41])]); //#16662
        if ($1.cca) { //#16663
            $1.metrics = $1.ccametrics; //#16663
        } //#16663
        $1.urows = $1.rows; //#16666
        $1.ucols = $1.columns; //#16667
        $1.i = 0; //#16668
        for (;;) { //#16684
            $1.m = $get($1.metrics, $1.i); //#16670
            $1.c = $get($1.m, 0); //#16671
            $1.r = $get($1.m, 1); //#16672
            $1.k = $get($1.m, 2); //#16673
            $1.rapl = $get($1.m, 3); //#16674
            $1.rapc = $get($1.m, 4); //#16675
            $1.rapr = $get($1.m, 5); //#16676
            $1.ncws = $f(($1.r * $1.c) - $1.k); //#16677
            $1.okay = true; //#16678
            if ($1.datcws.length > $1.ncws) { //#16679
                $1.okay = false; //#16679
            } //#16679
            if (($1.urows != 0) && ($1.urows != $1.r)) { //#16680
                $1.okay = false; //#16680
            } //#16680
            if (($1.ucols != 0) && ($1.ucols != $1.c)) { //#16681
                $1.okay = false; //#16681
            } //#16681
            if ($1.okay) { //#16682
                break; //#16682
            } //#16682
            $1.i = $1.i + 1; //#16683
        } //#16683
        $1.m = $1.datcws.length; //#16685
        $1.n = $f(($1.c * $1.r) - $1.k); //#16688
        $1.cws = $a($f(($1.c * $1.r) + 1)); //#16689
        $puti($1.cws, 0, $1.datcws); //#16690
        $k[$j++] = $1.cws; //#16691
        $k[$j++] = $1.m; //#16691
        $k[$j++] = Infinity; //#16691
        for (var _E8 = 0, _E9 = $f($1.n - $1.m); _E8 < _E9; _E8++) { //#16691
            $k[$j++] = 900; //#16691
        } //#16691
        var _EA = $a(); //#16691
        var _EB = $k[--$j]; //#16691
        $puti($k[--$j], _EB, _EA); //#16691
        $k[$j++] = $1.cws; //#16692
        $k[$j++] = $1.n; //#16692
        $k[$j++] = Infinity; //#16692
        for (var _EG = 0, _EH = $1.k; _EG < _EH; _EG++) { //#16692
            $k[$j++] = 0; //#16692
        } //#16692
        $k[$j++] = 0; //#16692
        var _EI = $a(); //#16692
        var _EJ = $k[--$j]; //#16692
        $puti($k[--$j], _EJ, _EI); //#16692
        $k[$j++] = Infinity; //#16695
        $k[$j++] = 1; //#16695
        for (var _EL = 0, _EM = 928; _EL < _EM; _EL++) { //#16695
            var _EN = $k[--$j]; //#16695
            $k[$j++] = _EN; //#16695
            $k[$j++] = (_EN * 3) % 929; //#16695
        } //#16695
        $1.rsalog = $a(); //#16695
        $1.rslog = $a(929); //#16696
        for (var _EQ = 1; _EQ <= 928; _EQ += 1) { //#16697
            $put($1.rslog, $get($1.rsalog, _EQ), _EQ); //#16697
        } //#16697
        $1.rsprod = function() {
            var _EU = $k[--$j]; //#16701
            var _EV = $k[--$j]; //#16701
            $k[$j++] = _EV; //#16705
            $k[$j++] = _EU; //#16705
            if ((_EU != 0) && (_EV != 0)) { //#16704
                var _EY = $get($1.rslog, $k[--$j]); //#16702
                var _Ed = $get($1.rsalog, $f(_EY + $get($1.rslog, $k[--$j])) % 928); //#16702
                $k[$j++] = _Ed; //#16702
            } else { //#16704
                $j -= 2; //#16704
                $k[$j++] = 0; //#16704
            } //#16704
        }; //#16706
        $k[$j++] = Infinity; //#16709
        $k[$j++] = 1; //#16709
        for (var _Ef = 0, _Eg = $1.k; _Ef < _Eg; _Ef++) { //#16709
            $k[$j++] = 0; //#16709
        } //#16709
        $1.coeffs = $a(); //#16709
        for (var _Ek = 1, _Ej = $1.k; _Ek <= _Ej; _Ek += 1) { //#16718
            $1.i = _Ek; //#16711
            $put($1.coeffs, $1.i, $get($1.coeffs, $1.i - 1)); //#16712
            for (var _Er = $1.i - 1; _Er >= 1; _Er -= 1) { //#16716
                $1.j = _Er; //#16714
                $k[$j++] = $1.coeffs; //#16715
                $k[$j++] = $1.j; //#16715
                $k[$j++] = $get($1.coeffs, $1.j - 1); //#16715
                $k[$j++] = $get($1.coeffs, $1.j); //#16715
                $k[$j++] = $get($1.rsalog, $1.i); //#16715
                $1.rsprod(); //#16715
                var _F3 = $k[--$j]; //#16715
                var _F4 = $k[--$j]; //#16715
                var _F5 = $k[--$j]; //#16715
                $put($k[--$j], _F5, $f(_F4 + _F3) % 929); //#16715
            } //#16715
            $k[$j++] = $1.coeffs; //#16717
            $k[$j++] = 0; //#16717
            $k[$j++] = $get($1.coeffs, 0); //#16717
            $k[$j++] = $get($1.rsalog, $1.i); //#16717
            $1.rsprod(); //#16717
            var _FD = $k[--$j]; //#16717
            var _FE = $k[--$j]; //#16717
            $put($k[--$j], _FE, _FD); //#16717
        } //#16717
        $1.coeffs = $geti($1.coeffs, 0, $1.coeffs.length - 1); //#16719
        for (var _FK = $1.coeffs.length - 1; _FK >= 0; _FK -= 2) { //#16720
            var _FL = $1.coeffs; //#16720
            $put(_FL, _FK, $f(929 - $get(_FL, _FK))); //#16720
        } //#16720
        for (var _FP = 0, _FO = $f($1.n - 1); _FP <= _FO; _FP += 1) { //#16729
            $1.t = $f($get($1.cws, _FP) + $get($1.cws, $1.n)) % 929; //#16724
            for (var _FX = 0, _FW = $f($1.k - 1); _FX <= _FW; _FX += 1) { //#16728
                $1.j = _FX; //#16726
                $put($1.cws, $f($1.n + $1.j), ($f($get($1.cws, $f($f($1.n + $1.j) + 1)) + ($f(929 - (($1.t * $get($1.coeffs, $f($f($1.k - $1.j) - 1))) % 929))))) % 929); //#16727
            } //#16727
        } //#16727
        for (var _Fo = $1.n, _Fn = $f($1.n + $1.k); _Fo <= _Fn; _Fo += 1) { //#16730
            $put($1.cws, _Fo, $f(929 - $get($1.cws, _Fo)) % 929); //#16730
        } //#16730
        $1.cws = $geti($1.cws, 0, $1.cws.length - 1); //#16733
        $1.clusters = $a([$a([120256, 125680, 128380, 120032, 125560, 128318, 108736, 119920, 108640, 86080, 108592, 86048, 110016, 120560, 125820, 109792, 120440, 125758, 88256, 109680, 88160, 89536, 110320, 120700, 89312, 110200, 120638, 89200, 110140, 89840, 110460, 89720, 110398, 89980, 128506, 119520, 125304, 128190, 107712, 119408, 125244, 107616, 119352, 84032, 107568, 119324, 84000, 107544, 83984, 108256, 119672, 125374, 85184, 108144, 119612, 85088, 108088, 119582, 85040, 108060, 85728, 108408, 119742, 85616, 108348, 85560, 108318, 85880, 108478, 85820, 85790, 107200, 119152, 125116, 107104, 119096, 125086, 83008, 107056, 119068, 82976, 107032, 82960, 82952, 83648, 107376, 119228, 83552, 107320, 119198, 83504, 107292, 83480, 83468, 83824, 107452, 83768, 107422, 83740, 83900, 106848, 118968, 125022, 82496, 106800, 118940, 82464, 106776, 118926, 82448, 106764, 82440, 106758, 82784, 106936, 119006, 82736, 106908, 82712, 106894, 82700, 82694, 106974, 82830, 82240, 106672, 118876, 82208, 106648, 118862, 82192, 106636, 82184, 106630, 82180, 82352, 82328, 82316, 82080, 118830, 106572, 106566, 82050, 117472, 124280, 127678, 103616, 117360, 124220, 103520, 117304, 124190, 75840, 103472, 75808, 104160, 117624, 124350, 76992, 104048, 117564, 76896, 103992, 76848, 76824, 77536, 104312, 117694, 77424, 104252, 77368, 77340, 77688, 104382, 77628, 77758, 121536, 126320, 128700, 121440, 126264, 128670, 111680, 121392, 126236, 111648, 121368, 126222, 111632, 121356, 103104, 117104, 124092, 112320, 103008, 117048, 124062, 112224, 121656, 126366, 93248, 74784, 102936, 117006, 93216, 112152, 93200, 75456, 103280, 117180, 93888, 75360, 103224, 117150, 93792, 112440, 121758, 93744, 75288, 93720, 75632, 103356, 94064, 75576, 103326, 94008, 112542, 93980, 75708, 94140, 75678, 94110, 121184, 126136, 128606, 111168, 121136, 126108, 111136, 121112, 126094, 111120, 121100, 111112, 111108, 102752, 116920, 123998, 111456, 102704, 116892, 91712, 74272, 121244, 116878, 91680, 74256, 102668, 91664, 111372, 102662, 74244, 74592, 102840, 116958, 92000, 74544, 102812, 91952, 111516, 102798, 91928, 74508, 74502, 74680, 102878, 92088, 74652, 92060, 74638, 92046, 92126, 110912, 121008, 126044, 110880, 120984, 126030, 110864, 120972, 110856, 120966, 110852, 110850, 74048, 102576, 116828, 90944, 74016, 102552, 116814, 90912, 111000, 121038, 90896, 73992, 102534, 90888, 110982, 90884, 74160, 102620, 91056, 74136, 102606, 91032, 111054, 91020, 74118, 91014, 91100, 91086, 110752, 120920, 125998, 110736, 120908, 110728, 120902, 110724, 110722, 73888, 102488, 116782, 90528, 73872, 102476, 90512, 110796, 102470, 90504, 73860, 90500, 73858, 73944, 90584, 90572, 90566, 120876, 120870, 110658, 102444, 73800, 90312, 90308, 90306, 101056, 116080, 123580, 100960, 116024, 70720, 100912, 115996, 70688, 100888, 70672, 70664, 71360, 101232, 116156, 71264, 101176, 116126, 71216, 101148, 71192, 71180, 71536, 101308, 71480, 101278, 71452, 71612, 71582, 118112, 124600, 127838, 105024, 118064, 124572, 104992, 118040, 124558, 104976, 118028, 104968, 118022, 100704, 115896, 123486, 105312, 100656, 115868, 79424, 70176, 118172, 115854, 79392, 105240, 100620, 79376, 70152, 79368, 70496, 100792, 115934, 79712, 70448, 118238, 79664, 105372, 100750, 79640, 70412, 79628, 70584, 100830, 79800, 70556, 79772, 70542, 70622, 79838, 122176, 126640, 128860, 122144, 126616, 128846, 122128, 126604, 122120, 126598, 122116, 104768, 117936, 124508, 113472, 104736, 126684, 124494, 113440, 122264, 126670, 113424, 104712, 117894, 113416, 122246, 104706, 69952, 100528, 115804, 78656, 69920, 100504, 115790, 96064, 78624, 104856, 117966, 96032, 113560, 122318, 100486, 96016, 78600, 104838, 96008, 69890, 70064, 100572, 78768, 70040, 100558, 96176, 78744, 104910, 96152, 113614, 70022, 78726, 70108, 78812, 70094, 96220, 78798, 122016, 126552, 128814, 122000, 126540, 121992, 126534, 121988, 121986, 104608, 117848, 124462, 113056, 104592, 126574, 113040, 122060, 117830, 113032, 104580, 113028, 104578, 113026, 69792, 100440, 115758, 78240, 69776, 100428, 95136, 78224, 104652, 100422, 95120, 113100, 69764, 95112, 78212, 69762, 78210, 69848, 100462, 78296, 69836, 95192, 78284, 69830, 95180, 78278, 69870, 95214, 121936, 126508, 121928, 126502, 121924, 121922, 104528, 117804, 112848, 104520, 117798, 112840, 121958, 112836, 104514, 112834, 69712, 100396, 78032, 69704, 100390, 94672, 78024, 104550, 94664, 112870, 69698, 94660, 78018, 94658, 78060, 94700, 94694, 126486, 121890, 117782, 104484, 104482, 69672, 77928, 94440, 69666, 77922, 99680, 68160, 99632, 68128, 99608, 115342, 68112, 99596, 68104, 99590, 68448, 99768, 115422, 68400, 99740, 68376, 99726, 68364, 68358, 68536, 99806, 68508, 68494, 68574, 101696, 116400, 123740, 101664, 116376, 101648, 116364, 101640, 116358, 101636, 67904, 99504, 115292, 72512, 67872, 116444, 115278, 72480, 101784, 116430, 72464, 67848, 99462, 72456, 101766, 67842, 68016, 99548, 72624, 67992, 99534, 72600, 101838, 72588, 67974, 68060, 72668, 68046, 72654, 118432, 124760, 127918, 118416, 124748, 118408, 124742, 118404, 118402, 101536, 116312, 105888, 101520, 116300, 105872, 118476, 116294, 105864, 101508, 105860, 101506, 105858, 67744, 99416, 72096, 67728, 116334, 80800, 72080, 101580, 99398, 80784, 105932, 67716, 80776, 72068, 67714, 72066, 67800, 99438, 72152, 67788, 80856, 72140, 67782, 80844, 72134, 67822, 72174, 80878, 126800, 128940, 126792, 128934, 126788, 126786, 118352, 124716, 122576, 126828, 124710, 122568, 126822, 122564, 118338, 122562, 101456, 116268, 105680, 101448, 116262, 114128, 105672, 118374, 114120, 122598, 101442, 114116, 105666, 114114, 67664, 99372, 71888, 67656, 99366, 80336, 71880, 101478, 97232, 80328, 105702, 67650, 97224, 114150, 71874, 97220, 67692, 71916, 67686, 80364, 71910, 97260, 80358, 97254, 126760, 128918, 126756, 126754, 118312, 124694, 122472, 126774, 122468, 118306, 122466, 101416, 116246, 105576, 101412, 113896, 105572, 101410, 113892, 105570, 113890, 67624, 99350, 71784, 101430, 80104, 71780, 67618, 96744, 80100, 71778, 96740, 80098, 96738, 71798, 96758, 126738, 122420, 122418, 105524, 113780, 113778, 71732, 79988, 96500, 96498, 66880, 66848, 98968, 66832, 66824, 66820, 66992, 66968, 66956, 66950, 67036, 67022, 100000, 99984, 115532, 99976, 115526, 99972, 99970, 66720, 98904, 69024, 100056, 98892, 69008, 100044, 69000, 100038, 68996, 66690, 68994, 66776, 98926, 69080, 100078, 69068, 66758, 69062, 66798, 69102, 116560, 116552, 116548, 116546, 99920, 102096, 116588, 115494, 102088, 116582, 102084, 99906, 102082, 66640, 68816, 66632, 98854, 73168, 68808, 66628, 73160, 68804, 66626, 73156, 68802, 66668, 68844, 66662, 73196, 68838, 73190, 124840, 124836, 124834, 116520, 118632, 124854, 118628, 116514, 118626, 99880, 115478, 101992, 116534, 106216, 101988, 99874, 106212, 101986, 106210, 66600, 98838, 68712, 99894, 72936, 68708, 66594, 81384, 72932, 68706, 81380, 72930, 66614, 68726, 72950, 81398, 128980, 128978, 124820, 126900, 124818, 126898, 116500, 118580, 116498, 122740, 118578, 122738, 99860, 101940, 99858, 106100, 101938, 114420]), $a([128352, 129720, 125504, 128304, 129692, 125472, 128280, 129678, 125456, 128268, 125448, 128262, 125444, 125792, 128440, 129758, 120384, 125744, 128412, 120352, 125720, 128398, 120336, 125708, 120328, 125702, 120324, 120672, 125880, 128478, 110144, 120624, 125852, 110112, 120600, 125838, 110096, 120588, 110088, 120582, 110084, 110432, 120760, 125918, 89664, 110384, 120732, 89632, 110360, 120718, 89616, 110348, 89608, 110342, 89952, 110520, 120798, 89904, 110492, 89880, 110478, 89868, 90040, 110558, 90012, 89998, 125248, 128176, 129628, 125216, 128152, 129614, 125200, 128140, 125192, 128134, 125188, 125186, 119616, 125360, 128220, 119584, 125336, 128206, 119568, 125324, 119560, 125318, 119556, 119554, 108352, 119728, 125404, 108320, 119704, 125390, 108304, 119692, 108296, 119686, 108292, 108290, 85824, 108464, 119772, 85792, 108440, 119758, 85776, 108428, 85768, 108422, 85764, 85936, 108508, 85912, 108494, 85900, 85894, 85980, 85966, 125088, 128088, 129582, 125072, 128076, 125064, 128070, 125060, 125058, 119200, 125144, 128110, 119184, 125132, 119176, 125126, 119172, 119170, 107424, 119256, 125166, 107408, 119244, 107400, 119238, 107396, 107394, 83872, 107480, 119278, 83856, 107468, 83848, 107462, 83844, 83842, 83928, 107502, 83916, 83910, 83950, 125008, 128044, 125000, 128038, 124996, 124994, 118992, 125036, 118984, 125030, 118980, 118978, 106960, 119020, 106952, 119014, 106948, 106946, 82896, 106988, 82888, 106982, 82884, 82882, 82924, 82918, 124968, 128022, 124964, 124962, 118888, 124982, 118884, 118882, 106728, 118902, 106724, 106722, 82408, 106742, 82404, 82402, 124948, 124946, 118836, 118834, 106612, 106610, 124224, 127664, 129372, 124192, 127640, 129358, 124176, 127628, 124168, 127622, 124164, 124162, 117568, 124336, 127708, 117536, 124312, 127694, 117520, 124300, 117512, 124294, 117508, 117506, 104256, 117680, 124380, 104224, 117656, 124366, 104208, 117644, 104200, 117638, 104196, 104194, 77632, 104368, 117724, 77600, 104344, 117710, 77584, 104332, 77576, 104326, 77572, 77744, 104412, 77720, 104398, 77708, 77702, 77788, 77774, 128672, 129880, 93168, 128656, 129868, 92664, 128648, 129862, 92412, 128644, 128642, 124064, 127576, 129326, 126368, 124048, 129902, 126352, 128716, 127558, 126344, 124036, 126340, 124034, 126338, 117152, 124120, 127598, 121760, 117136, 124108, 121744, 126412, 124102, 121736, 117124, 121732, 117122, 121730, 103328, 117208, 124142, 112544, 103312, 117196, 112528, 121804, 117190, 112520, 103300, 112516, 103298, 112514, 75680, 103384, 117230, 94112, 75664, 103372, 94096, 112588, 103366, 94088, 75652, 94084, 75650, 75736, 103406, 94168, 75724, 94156, 75718, 94150, 75758, 128592, 129836, 91640, 128584, 129830, 91388, 128580, 91262, 128578, 123984, 127532, 126160, 123976, 127526, 126152, 128614, 126148, 123970, 126146, 116944, 124012, 121296, 116936, 124006, 121288, 126182, 121284, 116930, 121282, 102864, 116972, 111568, 102856, 116966, 111560, 121318, 111556, 102850, 111554, 74704, 102892, 92112, 74696, 102886, 92104, 111590, 92100, 74690, 92098, 74732, 92140, 74726, 92134, 128552, 129814, 90876, 128548, 90750, 128546, 123944, 127510, 126056, 128566, 126052, 123938, 126050, 116840, 123958, 121064, 116836, 121060, 116834, 121058, 102632, 116854, 111080, 121078, 111076, 102626, 111074, 74216, 102646, 91112, 74212, 91108, 74210, 91106, 74230, 91126, 128532, 90494, 128530, 123924, 126004, 123922, 126002, 116788, 120948, 116786, 120946, 102516, 110836, 102514, 110834, 73972, 90612, 73970, 90610, 128522, 123914, 125978, 116762, 120890, 102458, 110714, 123552, 127320, 129198, 123536, 127308, 123528, 127302, 123524, 123522, 116128, 123608, 127342, 116112, 123596, 116104, 123590, 116100, 116098, 101280, 116184, 123630, 101264, 116172, 101256, 116166, 101252, 101250, 71584, 101336, 116206, 71568, 101324, 71560, 101318, 71556, 71554, 71640, 101358, 71628, 71622, 71662, 127824, 129452, 79352, 127816, 129446, 79100, 127812, 78974, 127810, 123472, 127276, 124624, 123464, 127270, 124616, 127846, 124612, 123458, 124610, 115920, 123500, 118224, 115912, 123494, 118216, 124646, 118212, 115906, 118210, 100816, 115948, 105424, 100808, 115942, 105416, 118246, 105412, 100802, 105410, 70608, 100844, 79824, 70600, 100838, 79816, 105446, 79812, 70594, 79810, 70636, 79852, 70630, 79846, 129960, 95728, 113404, 129956, 95480, 113278, 129954, 95356, 95294, 127784, 129430, 78588, 128872, 129974, 95996, 78462, 128868, 127778, 95870, 128866, 123432, 127254, 124520, 123428, 126696, 128886, 123426, 126692, 124514, 126690, 115816, 123446, 117992, 115812, 122344, 117988, 115810, 122340, 117986, 122338, 100584, 115830, 104936, 100580, 113640, 104932, 100578, 113636, 104930, 113634, 70120, 100598, 78824, 70116, 96232, 78820, 70114, 96228, 78818, 96226, 70134, 78838, 129940, 94968, 113022, 129938, 94844, 94782, 127764, 78206, 128820, 127762, 95102, 128818, 123412, 124468, 123410, 126580, 124466, 126578, 115764, 117876, 115762, 122100, 117874, 122098, 100468, 104692, 100466, 113140, 104690, 113138, 69876, 78324, 69874, 95220, 78322, 95218, 129930, 94588, 94526, 127754, 128794, 123402, 124442, 126522, 115738, 117818, 121978, 100410, 104570, 112890, 69754, 78074, 94714, 94398, 123216, 127148, 123208, 127142, 123204, 123202, 115408, 123244, 115400, 123238, 115396, 115394, 99792, 115436, 99784, 115430, 99780, 99778, 68560, 99820, 68552, 99814, 68548, 68546, 68588, 68582, 127400, 129238, 72444, 127396, 72318, 127394, 123176, 127126, 123752, 123172, 123748, 123170, 123746, 115304, 123190, 116456, 115300, 116452, 115298, 116450, 99560, 115318, 101864, 99556, 101860, 99554, 101858, 68072, 99574, 72680, 68068, 72676, 68066, 72674, 68086, 72694, 129492, 80632, 105854, 129490, 80508, 80446, 127380, 72062, 127924, 127378, 80766, 127922, 123156, 123700, 123154, 124788, 123698, 124786, 115252, 116340, 115250, 118516, 116338, 118514, 99444, 101620, 99442, 105972, 101618, 105970, 67828, 72180, 67826, 80884, 72178, 80882, 97008, 114044, 96888, 113982, 96828, 96798, 129482, 80252, 130010, 97148, 80190, 97086, 127370, 127898, 128954, 123146, 123674, 124730, 126842, 115226, 116282, 118394, 122618, 99386, 101498, 105722, 114170, 67706, 71930, 80378, 96632, 113854, 96572, 96542, 80062, 96702, 96444, 96414, 96350, 123048, 123044, 123042, 115048, 123062, 115044, 115042, 99048, 115062, 99044, 99042, 67048, 99062, 67044, 67042, 67062, 127188, 68990, 127186, 123028, 123316, 123026, 123314, 114996, 115572, 114994, 115570, 98932, 100084, 98930, 100082, 66804, 69108, 66802, 69106, 129258, 73084, 73022, 127178, 127450, 123018, 123290, 123834, 114970, 115514, 116602, 98874, 99962, 102138, 66682, 68858, 73210, 81272, 106174, 81212, 81182, 72894, 81342, 97648, 114364, 97592, 114334, 97564, 97550, 81084, 97724, 81054, 97694, 97464, 114270, 97436, 97422, 80990, 97502, 97372, 97358, 97326, 114868, 114866, 98676, 98674, 66292, 66290, 123098, 114842, 115130, 98618, 99194, 66170, 67322, 69310, 73404, 73374, 81592, 106334, 81564, 81550, 73310, 81630, 97968, 114524, 97944, 114510, 97932, 97926, 81500, 98012, 81486, 97998, 97880, 114478, 97868, 97862, 81454, 97902, 97836, 97830, 69470, 73564, 73550, 81752, 106414, 81740, 81734, 73518, 81774, 81708, 81702]), $a([109536, 120312, 86976, 109040, 120060, 86496, 108792, 119934, 86256, 108668, 86136, 129744, 89056, 110072, 129736, 88560, 109820, 129732, 88312, 109694, 129730, 88188, 128464, 129772, 89592, 128456, 129766, 89340, 128452, 89214, 128450, 125904, 128492, 125896, 128486, 125892, 125890, 120784, 125932, 120776, 125926, 120772, 120770, 110544, 120812, 110536, 120806, 110532, 84928, 108016, 119548, 84448, 107768, 119422, 84208, 107644, 84088, 107582, 84028, 129640, 85488, 108284, 129636, 85240, 108158, 129634, 85116, 85054, 128232, 129654, 85756, 128228, 85630, 128226, 125416, 128246, 125412, 125410, 119784, 125430, 119780, 119778, 108520, 119798, 108516, 108514, 83424, 107256, 119166, 83184, 107132, 83064, 107070, 83004, 82974, 129588, 83704, 107390, 129586, 83580, 83518, 128116, 83838, 128114, 125172, 125170, 119284, 119282, 107508, 107506, 82672, 106876, 82552, 106814, 82492, 82462, 129562, 82812, 82750, 128058, 125050, 119034, 82296, 106686, 82236, 82206, 82366, 82108, 82078, 76736, 103920, 117500, 76256, 103672, 117374, 76016, 103548, 75896, 103486, 75836, 129384, 77296, 104188, 129380, 77048, 104062, 129378, 76924, 76862, 127720, 129398, 77564, 127716, 77438, 127714, 124392, 127734, 124388, 124386, 117736, 124406, 117732, 117730, 104424, 117750, 104420, 104418, 112096, 121592, 126334, 92608, 111856, 121468, 92384, 111736, 121406, 92272, 111676, 92216, 111646, 92188, 75232, 103160, 117118, 93664, 74992, 103036, 93424, 112252, 102974, 93304, 74812, 93244, 74782, 93214, 129332, 75512, 103294, 129908, 129330, 93944, 75388, 129906, 93820, 75326, 93758, 127604, 75646, 128756, 127602, 94078, 128754, 124148, 126452, 124146, 126450, 117236, 121844, 117234, 121842, 103412, 103410, 91584, 111344, 121212, 91360, 111224, 121150, 91248, 111164, 91192, 111134, 91164, 91150, 74480, 102780, 91888, 74360, 102718, 91768, 111422, 91708, 74270, 91678, 129306, 74620, 129850, 92028, 74558, 91966, 127546, 128634, 124026, 126202, 116986, 121338, 102906, 90848, 110968, 121022, 90736, 110908, 90680, 110878, 90652, 90638, 74104, 102590, 91000, 74044, 90940, 74014, 90910, 74174, 91070, 90480, 110780, 90424, 110750, 90396, 90382, 73916, 90556, 73886, 90526, 90296, 110686, 90268, 90254, 73822, 90334, 90204, 90190, 71136, 101112, 116094, 70896, 100988, 70776, 100926, 70716, 70686, 129204, 71416, 101246, 129202, 71292, 71230, 127348, 71550, 127346, 123636, 123634, 116212, 116210, 101364, 101362, 79296, 105200, 118140, 79072, 105080, 118078, 78960, 105020, 78904, 104990, 78876, 78862, 70384, 100732, 79600, 70264, 100670, 79480, 105278, 79420, 70174, 79390, 129178, 70524, 129466, 79740, 70462, 79678, 127290, 127866, 123514, 124666, 115962, 118266, 100858, 113376, 122232, 126654, 95424, 113264, 122172, 95328, 113208, 122142, 95280, 113180, 95256, 113166, 95244, 78560, 104824, 117950, 95968, 78448, 104764, 95856, 113468, 104734, 95800, 78364, 95772, 78350, 95758, 70008, 100542, 78712, 69948, 96120, 78652, 69918, 96060, 78622, 96030, 70078, 78782, 96190, 94912, 113008, 122044, 94816, 112952, 122014, 94768, 112924, 94744, 112910, 94732, 94726, 78192, 104636, 95088, 78136, 104606, 95032, 113054, 95004, 78094, 94990, 69820, 78268, 69790, 95164, 78238, 95134, 94560, 112824, 121950, 94512, 112796, 94488, 112782, 94476, 94470, 78008, 104542, 94648, 77980, 94620, 77966, 94606, 69726, 78046, 94686, 94384, 112732, 94360, 112718, 94348, 94342, 77916, 94428, 77902, 94414, 94296, 112686, 94284, 94278, 77870, 94318, 94252, 94246, 68336, 99708, 68216, 99646, 68156, 68126, 68476, 68414, 127162, 123258, 115450, 99834, 72416, 101752, 116414, 72304, 101692, 72248, 101662, 72220, 72206, 67960, 99518, 72568, 67900, 72508, 67870, 72478, 68030, 72638, 80576, 105840, 118460, 80480, 105784, 118430, 80432, 105756, 80408, 105742, 80396, 80390, 72048, 101564, 80752, 71992, 101534, 80696, 71964, 80668, 71950, 80654, 67772, 72124, 67742, 80828, 72094, 80798, 114016, 122552, 126814, 96832, 113968, 122524, 96800, 113944, 122510, 96784, 113932, 96776, 113926, 96772, 80224, 105656, 118366, 97120, 80176, 105628, 97072, 114076, 105614, 97048, 80140, 97036, 80134, 97030, 71864, 101470, 80312, 71836, 97208, 80284, 71822, 97180, 80270, 97166, 67678, 71902, 80350, 97246, 96576, 113840, 122460, 96544, 113816, 122446, 96528, 113804, 96520, 113798, 96516, 96514, 80048, 105564, 96688, 80024, 105550, 96664, 113870, 96652, 80006, 96646, 71772, 80092, 71758, 96732, 80078, 96718, 96416, 113752, 122414, 96400, 113740, 96392, 113734, 96388, 96386, 79960, 105518, 96472, 79948, 96460, 79942, 96454, 71726, 79982, 96494, 96336, 113708, 96328, 113702, 96324, 96322, 79916, 96364, 79910, 96358, 96296, 113686, 96292, 96290, 79894, 96310, 66936, 99006, 66876, 66846, 67006, 68976, 100028, 68920, 99998, 68892, 68878, 66748, 69052, 66718, 69022, 73056, 102072, 116574, 73008, 102044, 72984, 102030, 72972, 72966, 68792, 99934, 73144, 68764, 73116, 68750, 73102, 66654, 68830, 73182, 81216, 106160, 118620, 81184, 106136, 118606, 81168, 106124, 81160, 106118, 81156, 81154, 72880, 101980, 81328, 72856, 101966, 81304, 106190, 81292, 72838, 81286, 68700, 72924, 68686, 81372, 72910, 81358, 114336, 122712, 126894, 114320, 122700, 114312, 122694, 114308, 114306, 81056, 106072, 118574, 97696, 81040, 106060, 97680, 114380, 106054, 97672, 81028, 97668, 81026, 97666, 72792, 101934, 81112, 72780, 97752, 81100, 72774, 97740, 81094, 97734, 68654, 72814, 81134, 97774, 114256, 122668, 114248, 122662, 114244, 114242, 80976, 106028, 97488, 80968, 106022, 97480, 114278, 97476, 80962, 97474, 72748, 81004, 72742, 97516, 80998, 97510, 114216, 122646, 114212, 114210, 80936, 106006, 97384, 80932, 97380, 80930, 97378, 72726, 80950, 97398, 114196, 114194, 80916, 97332, 80914, 97330, 66236, 66206, 67256, 99166, 67228, 67214, 66142, 67294, 69296, 100188, 69272, 100174, 69260, 69254, 67164, 69340, 67150, 69326, 73376, 102232, 116654, 73360, 102220, 73352, 102214, 73348, 73346, 69208, 100142, 73432, 102254, 73420, 69190, 73414, 67118, 69230, 73454, 106320, 118700, 106312, 118694, 106308, 106306, 73296, 102188, 81616, 106348, 102182, 81608, 73284, 81604, 73282, 81602, 69164, 73324, 69158, 81644, 73318, 81638, 122792, 126934, 122788, 122786, 106280, 118678, 114536, 106276, 114532, 106274, 114530, 73256, 102166, 81512, 73252, 98024, 81508, 73250, 98020, 81506, 98018, 69142, 73270, 81526, 98038, 122772, 122770, 106260, 114484, 106258, 114482, 73236, 81460, 73234, 97908, 81458, 97906, 122762, 106250, 114458, 73226, 81434, 97850, 66396, 66382, 67416, 99246, 67404, 67398, 66350, 67438, 69456, 100268, 69448, 100262, 69444, 69442, 67372, 69484, 67366, 69478, 102312, 116694, 102308, 102306, 69416, 100246, 73576, 102326, 73572, 69410, 73570, 67350, 69430, 73590, 118740, 118738, 102292, 106420, 102290, 106418, 69396, 73524, 69394, 81780, 73522, 81778, 118730, 102282, 106394, 69386, 73498, 81722, 66476, 66470, 67496, 99286, 67492, 67490, 66454, 67510, 100308, 100306, 67476, 69556, 67474, 69554, 116714])]); //#16975
        $1.raps = $a([$a([802, 930, 946, 818, 882, 890, 826, 954, 922, 986, 970, 906, 778, 794, 786, 914, 978, 982, 980, 916, 948, 932, 934, 942, 940, 936, 808, 812, 814, 806, 822, 950, 918, 790, 788, 820, 884, 868, 870, 878, 876, 872, 840, 856, 860, 862, 846, 844, 836, 838, 834, 866]), $a([718, 590, 622, 558, 550, 566, 534, 530, 538, 570, 562, 546, 610, 626, 634, 762, 754, 758, 630, 628, 612, 614, 582, 578, 706, 738, 742, 740, 748, 620, 556, 552, 616, 744, 712, 716, 708, 710, 646, 654, 652, 668, 664, 696, 688, 656, 720, 592, 600, 604, 732, 734])]); //#16990
        $1.cwtobits = function() {
            var _G4 = $get($1.clusters, $k[--$j]); //#16994
            $1.v = $get(_G4, $k[--$j]); //#16994
            $k[$j++] = Infinity; //#16995
            for (var _G7 = 0, _G8 = 17; _G7 < _G8; _G7++) { //#16995
                $k[$j++] = 0; //#16995
            } //#16995
            var _GB = $cvrs($s(17), $1.v, 2); //#16995
            for (var _GC = 0, _GD = _GB.length; _GC < _GD; _GC++) { //#16995
                $k[$j++] = $get(_GB, _GC) - 48; //#16995
            } //#16995
            var _GF = $a(); //#16995
            $k[$j++] = $geti(_GF, _GF.length - 17, 17); //#16996
        }; //#16997
        $1.raptobits = function() {
            var _GJ = $get($1.raps, $k[--$j]); //#17001
            $1.v = $get(_GJ, $k[--$j]); //#17001
            $k[$j++] = Infinity; //#17002
            for (var _GM = 0, _GN = 10; _GM < _GN; _GM++) { //#17002
                $k[$j++] = 0; //#17002
            } //#17002
            var _GQ = $cvrs($s(10), $1.v, 2); //#17002
            for (var _GR = 0, _GS = _GQ.length; _GR < _GS; _GR++) { //#17002
                $k[$j++] = $get(_GQ, _GR) - 48; //#17002
            } //#17002
            var _GU = $a(); //#17002
            $k[$j++] = $geti(_GU, _GU.length - 10, 10); //#17003
        }; //#17004
        $1.rwid = $get($a([38, 55, 82, 99]), $f($1.c - 1)); //#17007
        if (($1.c == 3) && $1.cca) { //#17008
            $1.rwid = 72; //#17008
        } //#17008
        $1.pixs = $a($1.rwid * $1.r); //#17009
        for (var _Gg = 0, _Gf = $f($1.r - 1); _Gg <= _Gf; _Gg += 1) { //#17047
            $1.i = _Gg; //#17012
            $1.clst = ($f($f($1.i + $1.rapl) - 1)) % 3; //#17013
            $k[$j++] = $1.pixs; //#17045
            $k[$j++] = $1.rwid * $1.i; //#17045
            $k[$j++] = Infinity; //#17045
            if ($1.c == 1) { //#17019
                $k[$j++] = ($f($f($1.i + $1.rapl) - 1)) % 52; //#17016
                $k[$j++] = 0; //#17016
                $1.raptobits(); //#17016
                $aload($k[--$j]); //#17016
                $k[$j++] = $get($1.cws, $1.i); //#17017
                $k[$j++] = $1.clst; //#17017
                $1.cwtobits(); //#17017
                $aload($k[--$j]); //#17017
                $k[$j++] = ($f($f($1.i + $1.rapr) - 1)) % 52; //#17018
                $k[$j++] = 0; //#17018
                $1.raptobits(); //#17018
                $aload($k[--$j]); //#17018
            } //#17018
            if ($1.c == 2) { //#17025
                $k[$j++] = ($f($f($1.i + $1.rapl) - 1)) % 52; //#17021
                $k[$j++] = 0; //#17021
                $1.raptobits(); //#17021
                $aload($k[--$j]); //#17021
                $k[$j++] = $get($1.cws, $1.i * 2); //#17022
                $k[$j++] = $1.clst; //#17022
                $1.cwtobits(); //#17022
                $aload($k[--$j]); //#17022
                $k[$j++] = $get($1.cws, ($1.i * 2) + 1); //#17023
                $k[$j++] = $1.clst; //#17023
                $1.cwtobits(); //#17023
                $aload($k[--$j]); //#17023
                $k[$j++] = ($f($f($1.i + $1.rapr) - 1)) % 52; //#17024
                $k[$j++] = 0; //#17024
                $1.raptobits(); //#17024
                $aload($k[--$j]); //#17024
            } //#17024
            if ($1.c == 3) { //#17035
                if (!$1.cca) { //#17029
                    $k[$j++] = ($f($f($1.i + $1.rapl) - 1)) % 52; //#17028
                    $k[$j++] = 0; //#17028
                    $1.raptobits(); //#17028
                    $aload($k[--$j]); //#17028
                } //#17028
                $k[$j++] = $get($1.cws, $1.i * 3); //#17030
                $k[$j++] = $1.clst; //#17030
                $1.cwtobits(); //#17030
                $aload($k[--$j]); //#17030
                $k[$j++] = ($f($f($1.i + $1.rapc) - 1)) % 52; //#17031
                $k[$j++] = 1; //#17031
                $1.raptobits(); //#17031
                $aload($k[--$j]); //#17031
                $k[$j++] = $get($1.cws, ($1.i * 3) + 1); //#17032
                $k[$j++] = $1.clst; //#17032
                $1.cwtobits(); //#17032
                $aload($k[--$j]); //#17032
                $k[$j++] = $get($1.cws, ($1.i * 3) + 2); //#17033
                $k[$j++] = $1.clst; //#17033
                $1.cwtobits(); //#17033
                $aload($k[--$j]); //#17033
                $k[$j++] = ($f($f($1.i + $1.rapr) - 1)) % 52; //#17034
                $k[$j++] = 0; //#17034
                $1.raptobits(); //#17034
                $aload($k[--$j]); //#17034
            } //#17034
            if ($1.c == 4) { //#17044
                $k[$j++] = ($f($f($1.i + $1.rapl) - 1)) % 52; //#17037
                $k[$j++] = 0; //#17037
                $1.raptobits(); //#17037
                $aload($k[--$j]); //#17037
                $k[$j++] = $get($1.cws, $1.i * 4); //#17038
                $k[$j++] = $1.clst; //#17038
                $1.cwtobits(); //#17038
                $aload($k[--$j]); //#17038
                $k[$j++] = $get($1.cws, ($1.i * 4) + 1); //#17039
                $k[$j++] = $1.clst; //#17039
                $1.cwtobits(); //#17039
                $aload($k[--$j]); //#17039
                $k[$j++] = ($f($f($1.i + $1.rapc) - 1)) % 52; //#17040
                $k[$j++] = 1; //#17040
                $1.raptobits(); //#17040
                $aload($k[--$j]); //#17040
                $k[$j++] = $get($1.cws, ($1.i * 4) + 2); //#17041
                $k[$j++] = $1.clst; //#17041
                $1.cwtobits(); //#17041
                $aload($k[--$j]); //#17041
                $k[$j++] = $get($1.cws, ($1.i * 4) + 3); //#17042
                $k[$j++] = $1.clst; //#17042
                $1.cwtobits(); //#17042
                $aload($k[--$j]); //#17042
                $k[$j++] = ($f($f($1.i + $1.rapr) - 1)) % 52; //#17043
                $k[$j++] = 0; //#17043
                $1.raptobits(); //#17043
                $aload($k[--$j]); //#17043
            } //#17043
            $k[$j++] = 1; //#17045
            var _I9 = $a(); //#17045
            var _IA = $k[--$j]; //#17046
            $puti($k[--$j], _IA, _I9); //#17046
        } //#17046
        var _IJ = new Map([
            ["ren", $0.renmatrix],
            ["pixs", $1.pixs],
            ["pixx", $1.rwid],
            ["pixy", $1.r],
            ["height", ($1.r / 72) * $1.rowmult],
            ["width", $1.rwid / 72],
            ["opt", $1.options]
        ]); //#17057
        $k[$j++] = _IJ; //#17060
        if (!$1.dontdraw) { //#17060
            $0.renmatrix(); //#17060
        } //#17060
    };
    $0.datamatrix = function() {
        var $1 = {}; //#17090
        $1.options = $k[--$j]; //#17092
        $1.barcode = $k[--$j]; //#17093
        $1.dontdraw = false; //#17095
        $1.columns = 0; //#17096
        $1.rows = 0; //#17097
        $1.format = "square"; //#17098
        $1.version = "unset"; //#17099
        $1.parse = false; //#17100
        $1.parsefnc = false; //#17101
        $1.encoding = "auto"; //#17102
        $1.mailmark = false; //#17103
        $1.raw = false; //#17104
        $1.dmre = false; //#17105
        $1.dindmre = false; //#17106
        $1.isodmre = false; //#17107
        $forall($1.options, function() { //#17118
            var _3 = $k[--$j]; //#17118
            $1[$k[--$j]] = _3; //#17118
        }); //#17118
        if ($ne($1.version, "unset")) { //#17124
            $search($1.version, "x"); //#17121
            $j--; //#17122
            $1.rows = $k[--$j]; //#17122
            $j--; //#17123
            $1.columns = $k[--$j]; //#17123
        } //#17123
        $1.columns = ~~$z($1.columns); //#17126
        $1.rows = ~~$z($1.rows); //#17127
        $1.isodmre = $1.dmre; //#17129
        $k[$j++] = Infinity; //#17186
        $k[$j++] = $a([10, 10, 1, 1, 5, 1]); //#17162
        $k[$j++] = $a([12, 12, 1, 1, 7, 1]); //#17162
        $k[$j++] = $a([14, 14, 1, 1, 10, 1]); //#17162
        $k[$j++] = $a([16, 16, 1, 1, 12, 1]); //#17162
        $k[$j++] = $a([18, 18, 1, 1, 14, 1]); //#17162
        $k[$j++] = $a([20, 20, 1, 1, 18, 1]); //#17162
        $k[$j++] = $a([22, 22, 1, 1, 20, 1]); //#17162
        $k[$j++] = $a([24, 24, 1, 1, 24, 1]); //#17162
        $k[$j++] = $a([26, 26, 1, 1, 28, 1]); //#17162
        $k[$j++] = $a([32, 32, 2, 2, 36, 1]); //#17162
        $k[$j++] = $a([36, 36, 2, 2, 42, 1]); //#17162
        $k[$j++] = $a([40, 40, 2, 2, 48, 1]); //#17162
        $k[$j++] = $a([44, 44, 2, 2, 56, 1]); //#17162
        $k[$j++] = $a([48, 48, 2, 2, 68, 1]); //#17162
        $k[$j++] = $a([52, 52, 2, 2, 84, 2]); //#17162
        $k[$j++] = $a([64, 64, 4, 4, 112, 2]); //#17162
        $k[$j++] = $a([72, 72, 4, 4, 144, 4]); //#17162
        $k[$j++] = $a([80, 80, 4, 4, 192, 4]); //#17162
        $k[$j++] = $a([88, 88, 4, 4, 224, 4]); //#17162
        $k[$j++] = $a([96, 96, 4, 4, 272, 4]); //#17162
        $k[$j++] = $a([104, 104, 4, 4, 336, 6]); //#17162
        $k[$j++] = $a([120, 120, 6, 6, 408, 6]); //#17162
        $k[$j++] = $a([132, 132, 6, 6, 496, 8]); //#17162
        $k[$j++] = $a([144, 144, 6, 6, 620, 10]); //#17162
        $k[$j++] = $a([8, 18, 1, 1, 7, 1]); //#17162
        $k[$j++] = $a([8, 32, 1, 2, 11, 1]); //#17162
        if ($1.dindmre || $1.isodmre) { //#17162
            $k[$j++] = $a([8, 48, 1, 2, 15, 1]); //#17162
        } //#17162
        if ($1.dindmre || $1.isodmre) { //#17163
            $k[$j++] = $a([8, 64, 1, 4, 18, 1]); //#17163
        } //#17163
        if ($1.isodmre) { //#17164
            $k[$j++] = $a([8, 80, 1, 4, 22, 1]); //#17164
        } //#17164
        if ($1.isodmre) { //#17165
            $k[$j++] = $a([8, 96, 1, 4, 28, 1]); //#17165
        } //#17165
        if ($1.isodmre) { //#17166
            $k[$j++] = $a([8, 120, 1, 6, 32, 1]); //#17166
        } //#17166
        if ($1.isodmre) { //#17167
            $k[$j++] = $a([8, 144, 1, 6, 36, 1]); //#17167
        } //#17167
        $k[$j++] = $a([12, 26, 1, 1, 14, 1]); //#17170
        $k[$j++] = $a([12, 36, 1, 2, 18, 1]); //#17170
        if ($1.dindmre || $1.isodmre) { //#17170
            $k[$j++] = $a([12, 64, 1, 4, 27, 1]); //#17170
        } //#17170
        if ($1.isodmre) { //#17171
            $k[$j++] = $a([12, 88, 1, 4, 36, 1]); //#17171
        } //#17171
        $k[$j++] = $a([16, 36, 1, 2, 24, 1]); //#17174
        $k[$j++] = $a([16, 48, 1, 2, 28, 1]); //#17174
        if ($1.dindmre || $1.isodmre) { //#17174
            $k[$j++] = $a([16, 64, 1, 4, 36, 1]); //#17174
        } //#17174
        if ($1.isodmre) { //#17175
            $k[$j++] = $a([20, 36, 1, 2, 28, 1]); //#17175
        } //#17175
        if ($1.isodmre) { //#17176
            $k[$j++] = $a([20, 44, 1, 2, 34, 1]); //#17176
        } //#17176
        if ($1.isodmre) { //#17177
            $k[$j++] = $a([20, 64, 1, 4, 42, 1]); //#17177
        } //#17177
        if ($1.isodmre) { //#17178
            $k[$j++] = $a([22, 48, 1, 2, 38, 1]); //#17178
        } //#17178
        if ($1.dindmre) { //#17179
            $k[$j++] = $a([24, 32, 1, 2, 28, 1]); //#17179
        } //#17179
        if ($1.dindmre) { //#17180
            $k[$j++] = $a([24, 36, 1, 2, 33, 1]); //#17180
        } //#17180
        if ($1.dindmre || $1.isodmre) { //#17181
            $k[$j++] = $a([24, 48, 1, 2, 41, 1]); //#17181
        } //#17181
        if ($1.dindmre || $1.isodmre) { //#17182
            $k[$j++] = $a([24, 64, 1, 4, 46, 1]); //#17182
        } //#17182
        if ($1.dindmre) { //#17183
            $k[$j++] = $a([26, 32, 1, 2, 32, 1]); //#17183
        } //#17183
        if ($1.dindmre || $1.isodmre) { //#17184
            $k[$j++] = $a([26, 40, 1, 2, 38, 1]); //#17184
        } //#17184
        if ($1.dindmre || $1.isodmre) { //#17185
            $k[$j++] = $a([26, 48, 1, 2, 42, 1]); //#17185
        } //#17185
        if ($1.dindmre || $1.isodmre) { //#17186
            $k[$j++] = $a([26, 64, 1, 4, 50, 1]); //#17186
        } //#17186
        $1.metrics = $a(); //#17186
        $1.urows = $1.rows; //#17190
        $1.ucols = $1.columns; //#17191
        $1.fullcws = $a([]); //#17192
        var _1Z = $1.metrics; //#17193
        for (var _1a = 0, _1b = _1Z.length; _1a < _1b; _1a++) { //#17210
            $1.m = $get(_1Z, _1a); //#17194
            $1.rows = $get($1.m, 0); //#17195
            $1.cols = $get($1.m, 1); //#17196
            $1.regh = $get($1.m, 2); //#17197
            $1.regv = $get($1.m, 3); //#17198
            $1.rscw = $get($1.m, 4); //#17199
            $1.rsbl = $get($1.m, 5); //#17200
            $1.mrows = $f($1.rows - (2 * $1.regh)); //#17201
            $1.mcols = $f($1.cols - (2 * $1.regv)); //#17202
            $1.ncws = $f((~~(($1.mrows * $1.mcols) / 8)) - $1.rscw); //#17203
            $1.okay = true; //#17204
            if (($1.urows != 0) && ($1.urows != $1.rows)) { //#17205
                $1.okay = false; //#17205
            } //#17205
            if (($1.ucols != 0) && ($1.ucols != $1.cols)) { //#17206
                $1.okay = false; //#17206
            } //#17206
            if ($eq($1.format, "square") && $ne($1.rows, $1.cols)) { //#17207
                $1.okay = false; //#17207
            } //#17207
            if ($eq($1.format, "rectangle") && $eq($1.rows, $1.cols)) { //#17208
                $1.okay = false; //#17208
            } //#17208
            if ($1.okay) { //#17209
                $k[$j++] = Infinity; //#17209
                $aload($1.fullcws); //#17209
                $k[$j++] = $1.ncws; //#17209
                $1.fullcws = $a(); //#17209
            } //#17209
        } //#17209
        $k[$j++] = Infinity; //#17211
        for (var _2C = 0, _2D = 1558; _2C < _2D; _2C++) { //#17211
            $k[$j++] = 10000; //#17211
        } //#17211
        $1.numremcws = $a(); //#17211
        var _2F = $1.fullcws; //#17212
        for (var _2G = 0, _2H = _2F.length; _2G < _2H; _2G++) { //#17212
            $put($1.numremcws, $f($get(_2F, _2G) - 1), 1); //#17212
        } //#17212
        for (var _2K = 1556; _2K >= 0; _2K -= 1) { //#17218
            $1.i = _2K; //#17214
            if ($get($1.numremcws, $1.i) != 1) { //#17217
                $put($1.numremcws, $1.i, $f($get($1.numremcws, $1.i + 1) + 1)); //#17216
            } //#17216
        } //#17216
        if ($1.raw) { //#17220
            $1.encoding = "raw"; //#17220
        } //#17220
        if ($eq($1.encoding, "raw")) { //#17233
            $1.cws = $a($1.barcode.length); //#17223
            $1.i = 0; //#17224
            $1.j = 0; //#17224
            for (;;) { //#17231
                if ($1.i == $1.barcode.length) { //#17226
                    break; //#17226
                } //#17226
                $1.cw = ~~$z($geti($1.barcode, $1.i + 1, 3)); //#17227
                $put($1.cws, $1.j, $1.cw); //#17228
                $1.i = $1.i + 4; //#17229
                $1.j = $1.j + 1; //#17230
            } //#17230
            $1.cws = $geti($1.cws, 0, $1.j); //#17232
        } //#17232
        if ($eq($1.encoding, "auto")) { //#17733
            $1.fnc1 = -1; //#17238
            $1.prog = -2; //#17238
            $1.m05 = -3; //#17238
            $1.m06 = -4; //#17238
            $1.lC = -5; //#17239
            $1.lB = -6; //#17239
            $1.lX = -7; //#17239
            $1.lT = -8; //#17239
            $1.lE = -9; //#17239
            $1.unl = -10; //#17239
            $1.sapp = -11; //#17240
            $1.usft = -12; //#17240
            $1.sft1 = -13; //#17240
            $1.sft2 = -14; //#17240
            $1.sft3 = -15; //#17240
            $1.eci = -16; //#17240
            $1.pad = -17; //#17240
            $1.unlcw = 254; //#17241
            var _2p = new Map([
                ["parse", $1.parse],
                ["parsefnc", $1.parsefnc],
                ["eci", true],
                ["FNC1", $1.fnc1],
                ["PROG", $1.prog]
            ]); //#17249
            $1.fncvals = _2p; //#17252
            $k[$j++] = 'msg'; //#17253
            $k[$j++] = $1.barcode; //#17253
            $k[$j++] = $1.fncvals; //#17253
            $0.parseinput(); //#17253
            var _2s = $k[--$j]; //#17253
            $1[$k[--$j]] = _2s; //#17253
            $1.msglen = $1.msg.length; //#17254
            if ($1.msglen >= 9) { //#17271
                $aload($geti($1.msg, 0, 7)); //#17258
                var _2y = $k[--$j]; //#17259
                var _2z = $k[--$j]; //#17259
                $k[$j++] = _2y == 29; //#17259
                $k[$j++] = _2z; //#17259
                $j--; //#17259
                var _30 = $k[--$j]; //#17259
                var _31 = $k[--$j]; //#17259
                var _32 = $k[--$j]; //#17259
                var _33 = $k[--$j]; //#17260
                var _34 = $k[--$j]; //#17260
                var _35 = $k[--$j]; //#17260
                if (((((_30 && (_31 == 48)) && (_32 == 30)) && (_33 == 62)) && (_34 == 41)) && (_35 == 91)) { //#17270
                    $aload($geti($1.msg, $1.msglen - 2, 2)); //#17261
                    var _39 = $k[--$j]; //#17262
                    var _3A = $k[--$j]; //#17262
                    if ((_39 == 4) && (_3A == 30)) { //#17269
                        if ($get($1.msg, 5) == 53) { //#17265
                            $k[$j++] = Infinity; //#17264
                            $k[$j++] = $1.m05; //#17264
                            $aload($geti($1.msg, 7, $1.msg.length - 9)); //#17264
                            $1.msg = $a(); //#17264
                        } //#17264
                        if ($get($1.msg, 5) == 54) { //#17268
                            $k[$j++] = Infinity; //#17267
                            $k[$j++] = $1.m06; //#17267
                            $aload($geti($1.msg, 7, $1.msg.length - 9)); //#17267
                            $1.msg = $a(); //#17267
                        } //#17267
                    } //#17267
                } //#17267
            } //#17267
            $1.msglen = $1.msg.length; //#17272
            $k[$j++] = Infinity; //#17281
            for (var _3Q = 0; _3Q <= 128; _3Q += 1) { //#17275
                $k[$j++] = _3Q; //#17275
                $k[$j++] = _3Q + 1; //#17275
            } //#17275
            $k[$j++] = $1.pad; //#17280
            $k[$j++] = 129; //#17280
            for (var _3S = 0; _3S <= 99; _3S += 1) { //#17280
                var _3U = $cvrs($s(2), _3S, 10); //#17278
                var _3W = $strcpy($s(2), "00"); //#17278
                $puti(_3W, 2 - _3U.length, _3U); //#17278
                $k[$j++] = _3W; //#17279
                $k[$j++] = _3S + 130; //#17279
            } //#17279
            var _3j = $a([$1.lC, $1.lB, $1.fnc1, $1.sapp, $1.prog, $1.usft, $1.m05, $1.m06, $1.lX, $1.lT, $1.lE, $1.eci]); //#17281
            $k[$j++] = 229; //#17281
            for (var _3k = 0, _3l = _3j.length; _3k < _3l; _3k++) { //#17281
                var _3o = $f($k[--$j] + 1); //#17281
                $k[$j++] = $get(_3j, _3k); //#17281
                $k[$j++] = _3o; //#17281
                $k[$j++] = _3o; //#17281
            } //#17281
            $j--; //#17281
            $1.Avals = $d(); //#17282
            $k[$j++] = Infinity; //#17284
            var _3q = $1.Avals; //#17284
            for (var _3v = _3q.size, _3u = _3q.keys(), _3t = 0; _3t < _3v; _3t++) { //#17284
                var _3r = _3u.next().value; //#17284
                $k[$j++] = _3r; //#17284
                $k[$j++] = _3q.get(_3r); //#17284
                $k[$j++] = Infinity; //#17284
                var _3w = $k[--$j]; //#17284
                var _3x = $k[--$j]; //#17284
                $k[$j++] = _3w; //#17284
                $k[$j++] = _3x; //#17284
                var _3y = $a(); //#17284
                $k[$j++] = _3y; //#17284
            } //#17284
            $1.Avals = $d(); //#17285
            $k[$j++] = Infinity; //#17293
            $k[$j++] = $1.sft1; //#17292
            $k[$j++] = 0; //#17292
            $k[$j++] = $1.sft2; //#17292
            $k[$j++] = 1; //#17292
            $k[$j++] = $1.sft3; //#17292
            $k[$j++] = 2; //#17292
            $k[$j++] = 32; //#17292
            $k[$j++] = 3; //#17292
            for (var _43 = 48; _43 <= 57; _43 += 1) { //#17292
                $k[$j++] = _43; //#17292
                $k[$j++] = _43 - 44; //#17292
            } //#17292
            for (var _44 = 65; _44 <= 90; _44 += 1) { //#17293
                $k[$j++] = _44; //#17293
                $k[$j++] = _44 - 51; //#17293
            } //#17293
            $1.CNvals = $d(); //#17294
            $k[$j++] = Infinity; //#17295
            for (var _46 = 0; _46 <= 31; _46 += 1) { //#17295
                $k[$j++] = _46; //#17295
                $k[$j++] = _46; //#17295
            } //#17295
            $1.C1vals = $d(); //#17295
            $k[$j++] = Infinity; //#17301
            for (var _48 = 33; _48 <= 47; _48 += 1) { //#17297
                $k[$j++] = _48; //#17297
                $k[$j++] = _48 - 33; //#17297
            } //#17297
            for (var _49 = 58; _49 <= 64; _49 += 1) { //#17298
                $k[$j++] = _49; //#17298
                $k[$j++] = _49 - 43; //#17298
            } //#17298
            for (var _4A = 91; _4A <= 95; _4A += 1) { //#17299
                $k[$j++] = _4A; //#17299
                $k[$j++] = _4A - 69; //#17299
            } //#17299
            $k[$j++] = $1.fnc1; //#17301
            $k[$j++] = 27; //#17301
            $k[$j++] = $1.usft; //#17301
            $k[$j++] = 30; //#17301
            $1.C2vals = $d(); //#17302
            $k[$j++] = Infinity; //#17303
            for (var _4E = 96; _4E <= 127; _4E += 1) { //#17303
                $k[$j++] = _4E; //#17303
                $k[$j++] = _4E - 96; //#17303
            } //#17303
            $1.C3vals = $d(); //#17303
            $k[$j++] = Infinity; //#17308
            var _4G = $1.CNvals; //#17305
            for (var _4L = _4G.size, _4K = _4G.keys(), _4J = 0; _4J < _4L; _4J++) { //#17305
                var _4H = _4K.next().value; //#17305
                $k[$j++] = _4H; //#17305
                $k[$j++] = _4G.get(_4H); //#17305
                $k[$j++] = Infinity; //#17305
                var _4M = $k[--$j]; //#17305
                var _4N = $k[--$j]; //#17305
                $k[$j++] = _4M; //#17305
                $k[$j++] = _4N; //#17305
                var _4O = $a(); //#17305
                $k[$j++] = _4O; //#17305
            } //#17305
            var _4P = $1.C1vals; //#17306
            for (var _4U = _4P.size, _4T = _4P.keys(), _4S = 0; _4S < _4U; _4S++) { //#17306
                var _4Q = _4T.next().value; //#17306
                $k[$j++] = _4Q; //#17306
                $k[$j++] = _4P.get(_4Q); //#17306
                $k[$j++] = Infinity; //#17306
                var _4V = $k[--$j]; //#17306
                var _4W = $k[--$j]; //#17306
                $k[$j++] = _4V; //#17306
                $k[$j++] = $get($1.CNvals, $1.sft1); //#17306
                $k[$j++] = _4W; //#17306
                var _4a = $a(); //#17306
                $k[$j++] = _4a; //#17306
            } //#17306
            var _4b = $1.C2vals; //#17307
            for (var _4g = _4b.size, _4f = _4b.keys(), _4e = 0; _4e < _4g; _4e++) { //#17307
                var _4c = _4f.next().value; //#17307
                $k[$j++] = _4c; //#17307
                $k[$j++] = _4b.get(_4c); //#17307
                $k[$j++] = Infinity; //#17307
                var _4h = $k[--$j]; //#17307
                var _4i = $k[--$j]; //#17307
                $k[$j++] = _4h; //#17307
                $k[$j++] = $get($1.CNvals, $1.sft2); //#17307
                $k[$j++] = _4i; //#17307
                var _4m = $a(); //#17307
                $k[$j++] = _4m; //#17307
            } //#17307
            var _4n = $1.C3vals; //#17308
            for (var _4s = _4n.size, _4r = _4n.keys(), _4q = 0; _4q < _4s; _4q++) { //#17308
                var _4o = _4r.next().value; //#17308
                $k[$j++] = _4o; //#17308
                $k[$j++] = _4n.get(_4o); //#17308
                $k[$j++] = Infinity; //#17308
                var _4t = $k[--$j]; //#17308
                var _4u = $k[--$j]; //#17308
                $k[$j++] = _4t; //#17308
                $k[$j++] = $get($1.CNvals, $1.sft3); //#17308
                $k[$j++] = _4u; //#17308
                var _4y = $a(); //#17308
                $k[$j++] = _4y; //#17308
            } //#17308
            $1.Cvals = $d(); //#17309
            $k[$j++] = Infinity; //#17317
            $k[$j++] = $1.sft1; //#17316
            $k[$j++] = 0; //#17316
            $k[$j++] = $1.sft2; //#17316
            $k[$j++] = 1; //#17316
            $k[$j++] = $1.sft3; //#17316
            $k[$j++] = 2; //#17316
            $k[$j++] = 32; //#17316
            $k[$j++] = 3; //#17316
            for (var _53 = 48; _53 <= 57; _53 += 1) { //#17316
                $k[$j++] = _53; //#17316
                $k[$j++] = _53 - 44; //#17316
            } //#17316
            for (var _54 = 97; _54 <= 122; _54 += 1) { //#17317
                $k[$j++] = _54; //#17317
                $k[$j++] = _54 - 83; //#17317
            } //#17317
            $1.TNvals = $d(); //#17318
            $k[$j++] = Infinity; //#17319
            for (var _56 = 0; _56 <= 31; _56 += 1) { //#17319
                $k[$j++] = _56; //#17319
                $k[$j++] = _56; //#17319
            } //#17319
            $1.T1vals = $d(); //#17319
            $k[$j++] = Infinity; //#17325
            for (var _58 = 33; _58 <= 47; _58 += 1) { //#17321
                $k[$j++] = _58; //#17321
                $k[$j++] = _58 - 33; //#17321
            } //#17321
            for (var _59 = 58; _59 <= 64; _59 += 1) { //#17322
                $k[$j++] = _59; //#17322
                $k[$j++] = _59 - 43; //#17322
            } //#17322
            for (var _5A = 91; _5A <= 95; _5A += 1) { //#17323
                $k[$j++] = _5A; //#17323
                $k[$j++] = _5A - 69; //#17323
            } //#17323
            $k[$j++] = $1.fnc1; //#17325
            $k[$j++] = 27; //#17325
            $k[$j++] = $1.usft; //#17325
            $k[$j++] = 30; //#17325
            $1.T2vals = $d(); //#17326
            $k[$j++] = Infinity; //#17330
            $k[$j++] = 96; //#17329
            $k[$j++] = 0; //#17329
            for (var _5E = 65; _5E <= 90; _5E += 1) { //#17329
                $k[$j++] = _5E; //#17329
                $k[$j++] = _5E - 64; //#17329
            } //#17329
            for (var _5F = 123; _5F <= 127; _5F += 1) { //#17330
                $k[$j++] = _5F; //#17330
                $k[$j++] = _5F - 96; //#17330
            } //#17330
            $1.T3vals = $d(); //#17331
            $k[$j++] = Infinity; //#17336
            var _5H = $1.TNvals; //#17333
            for (var _5M = _5H.size, _5L = _5H.keys(), _5K = 0; _5K < _5M; _5K++) { //#17333
                var _5I = _5L.next().value; //#17333
                $k[$j++] = _5I; //#17333
                $k[$j++] = _5H.get(_5I); //#17333
                $k[$j++] = Infinity; //#17333
                var _5N = $k[--$j]; //#17333
                var _5O = $k[--$j]; //#17333
                $k[$j++] = _5N; //#17333
                $k[$j++] = _5O; //#17333
                var _5P = $a(); //#17333
                $k[$j++] = _5P; //#17333
            } //#17333
            var _5Q = $1.T1vals; //#17334
            for (var _5V = _5Q.size, _5U = _5Q.keys(), _5T = 0; _5T < _5V; _5T++) { //#17334
                var _5R = _5U.next().value; //#17334
                $k[$j++] = _5R; //#17334
                $k[$j++] = _5Q.get(_5R); //#17334
                $k[$j++] = Infinity; //#17334
                var _5W = $k[--$j]; //#17334
                var _5X = $k[--$j]; //#17334
                $k[$j++] = _5W; //#17334
                $k[$j++] = $get($1.TNvals, $1.sft1); //#17334
                $k[$j++] = _5X; //#17334
                var _5b = $a(); //#17334
                $k[$j++] = _5b; //#17334
            } //#17334
            var _5c = $1.T2vals; //#17335
            for (var _5h = _5c.size, _5g = _5c.keys(), _5f = 0; _5f < _5h; _5f++) { //#17335
                var _5d = _5g.next().value; //#17335
                $k[$j++] = _5d; //#17335
                $k[$j++] = _5c.get(_5d); //#17335
                $k[$j++] = Infinity; //#17335
                var _5i = $k[--$j]; //#17335
                var _5j = $k[--$j]; //#17335
                $k[$j++] = _5i; //#17335
                $k[$j++] = $get($1.TNvals, $1.sft2); //#17335
                $k[$j++] = _5j; //#17335
                var _5n = $a(); //#17335
                $k[$j++] = _5n; //#17335
            } //#17335
            var _5o = $1.T3vals; //#17336
            for (var _5t = _5o.size, _5s = _5o.keys(), _5r = 0; _5r < _5t; _5r++) { //#17336
                var _5p = _5s.next().value; //#17336
                $k[$j++] = _5p; //#17336
                $k[$j++] = _5o.get(_5p); //#17336
                $k[$j++] = Infinity; //#17336
                var _5u = $k[--$j]; //#17336
                var _5v = $k[--$j]; //#17336
                $k[$j++] = _5u; //#17336
                $k[$j++] = $get($1.TNvals, $1.sft3); //#17336
                $k[$j++] = _5v; //#17336
                var _5z = $a(); //#17336
                $k[$j++] = _5z; //#17336
            } //#17336
            $1.Tvals = $d(); //#17337
            for (var _61 = 128; _61 <= 255; _61 += 1) { //#17345
                $1.i = _61; //#17341
                $k[$j++] = $1.Avals; //#17342
                $k[$j++] = $1.i; //#17342
                $k[$j++] = Infinity; //#17342
                $aload($get($1.Avals, $1.usft)); //#17342
                $aload($get($1.Avals, $1.i - 128)); //#17342
                var _6A = $a(); //#17342
                var _6B = $k[--$j]; //#17342
                $put($k[--$j], _6B, _6A); //#17342
                $k[$j++] = $1.Cvals; //#17343
                $k[$j++] = $1.i; //#17343
                $k[$j++] = Infinity; //#17343
                $aload($get($1.Cvals, $1.usft)); //#17343
                $aload($get($1.Cvals, $1.i - 128)); //#17343
                var _6L = $a(); //#17343
                var _6M = $k[--$j]; //#17343
                $put($k[--$j], _6M, _6L); //#17343
                $k[$j++] = $1.Tvals; //#17344
                $k[$j++] = $1.i; //#17344
                $k[$j++] = Infinity; //#17344
                $aload($get($1.Tvals, $1.usft)); //#17344
                $aload($get($1.Tvals, $1.i - 128)); //#17344
                var _6W = $a(); //#17344
                var _6X = $k[--$j]; //#17344
                $put($k[--$j], _6X, _6W); //#17344
            } //#17344
            $k[$j++] = Infinity; //#17353
            $k[$j++] = 13; //#17352
            $k[$j++] = 0; //#17352
            $k[$j++] = 42; //#17352
            $k[$j++] = 1; //#17352
            $k[$j++] = 62; //#17352
            $k[$j++] = 2; //#17352
            $k[$j++] = 32; //#17352
            $k[$j++] = 3; //#17352
            for (var _6Z = 48; _6Z <= 57; _6Z += 1) { //#17352
                $k[$j++] = _6Z; //#17352
                $k[$j++] = _6Z - 44; //#17352
            } //#17352
            for (var _6a = 65; _6a <= 90; _6a += 1) { //#17353
                $k[$j++] = _6a; //#17353
                $k[$j++] = _6a - 51; //#17353
            } //#17353
            $1.Xvals = $d(); //#17354
            $k[$j++] = Infinity; //#17356
            var _6c = $1.Xvals; //#17356
            for (var _6h = _6c.size, _6g = _6c.keys(), _6f = 0; _6f < _6h; _6f++) { //#17356
                var _6d = _6g.next().value; //#17356
                $k[$j++] = _6d; //#17356
                $k[$j++] = _6c.get(_6d); //#17356
                $k[$j++] = Infinity; //#17356
                var _6i = $k[--$j]; //#17356
                var _6j = $k[--$j]; //#17356
                $k[$j++] = _6i; //#17356
                $k[$j++] = _6j; //#17356
                var _6k = $a(); //#17356
                $k[$j++] = _6k; //#17356
            } //#17356
            $1.Xvals = $d(); //#17357
            $k[$j++] = Infinity; //#17362
            for (var _6m = 64; _6m <= 94; _6m += 1) { //#17360
                $k[$j++] = _6m; //#17360
                $k[$j++] = _6m - 64; //#17360
            } //#17360
            $k[$j++] = $1.unl; //#17362
            $k[$j++] = 31; //#17362
            for (var _6o = 32; _6o <= 63; _6o += 1) { //#17362
                $k[$j++] = _6o; //#17362
                $k[$j++] = _6o; //#17362
            } //#17362
            $1.Evals = $d(); //#17363
            $k[$j++] = Infinity; //#17365
            var _6q = $1.Evals; //#17365
            for (var _6v = _6q.size, _6u = _6q.keys(), _6t = 0; _6t < _6v; _6t++) { //#17365
                var _6r = _6u.next().value; //#17365
                $k[$j++] = _6r; //#17365
                $k[$j++] = _6q.get(_6r); //#17365
                $k[$j++] = Infinity; //#17365
                var _6w = $k[--$j]; //#17365
                var _6x = $k[--$j]; //#17365
                $k[$j++] = _6w; //#17365
                $k[$j++] = _6x; //#17365
                var _6y = $a(); //#17365
                $k[$j++] = _6y; //#17365
            } //#17365
            $1.Evals = $d(); //#17366
            $k[$j++] = Infinity; //#17369
            for (var _70 = 0; _70 <= 255; _70 += 1) { //#17369
                $k[$j++] = _70; //#17369
                $k[$j++] = _70; //#17369
            } //#17369
            $1.Bvals = $d(); //#17370
            $k[$j++] = Infinity; //#17372
            var _72 = $1.Bvals; //#17372
            for (var _77 = _72.size, _76 = _72.keys(), _75 = 0; _75 < _77; _75++) { //#17372
                var _73 = _76.next().value; //#17372
                $k[$j++] = _73; //#17372
                $k[$j++] = _72.get(_73); //#17372
                $k[$j++] = Infinity; //#17372
                var _78 = $k[--$j]; //#17372
                var _79 = $k[--$j]; //#17372
                $k[$j++] = _78; //#17372
                $k[$j++] = _79; //#17372
                var _7A = $a(); //#17372
                $k[$j++] = _7A; //#17372
            } //#17372
            $1.Bvals = $d(); //#17373
            $1.encvals = $a([$1.Avals, $1.Cvals, $1.Tvals, $1.Xvals, $1.Evals, $1.Bvals]); //#17375
            $k[$j++] = Infinity; //#17377
            for (var _7K = 0, _7L = $1.msglen; _7K < _7L; _7K++) { //#17377
                $k[$j++] = 0; //#17377
            } //#17377
            $k[$j++] = 0; //#17377
            $1.numD = $a(); //#17377
            $k[$j++] = Infinity; //#17378
            for (var _7O = 0, _7P = $1.msglen; _7O < _7P; _7O++) { //#17378
                $k[$j++] = 0; //#17378
            } //#17378
            $k[$j++] = 9999; //#17378
            $1.nextXterm = $a(); //#17378
            $k[$j++] = Infinity; //#17379
            for (var _7S = 0, _7T = $1.msglen; _7S < _7T; _7S++) { //#17379
                $k[$j++] = 0; //#17379
            } //#17379
            $k[$j++] = 9999; //#17379
            $1.nextNonX = $a(); //#17379
            $k[$j++] = Infinity; //#17380
            for (var _7W = 0, _7X = $1.msglen; _7W < _7X; _7W++) { //#17380
                $k[$j++] = false; //#17380
            } //#17380
            $k[$j++] = false; //#17380
            $1.isECI = $a(); //#17380
            for (var _7a = $1.msglen - 1; _7a >= 0; _7a -= 1) { //#17398
                $1.i = _7a; //#17382
                $1.barchar = $get($1.msg, $1.i); //#17383
                if (($1.barchar >= 48) && ($1.barchar <= 57)) { //#17386
                    $put($1.numD, $1.i, $f($get($1.numD, $1.i + 1) + 1)); //#17385
                } //#17385
                if ((($1.barchar == 13) || ($1.barchar == 42)) || ($1.barchar == 62)) { //#17390
                    $put($1.nextXterm, $1.i, 0); //#17388
                } else { //#17390
                    $put($1.nextXterm, $1.i, $f($get($1.nextXterm, $1.i + 1) + 1)); //#17390
                } //#17390
                var _7x = $get($1.Xvals, $1.barchar) !== undefined; //#17392
                if (!_7x) { //#17395
                    $put($1.nextNonX, $1.i, 0); //#17393
                } else { //#17395
                    $put($1.nextNonX, $1.i, $f($get($1.nextNonX, $1.i + 1) + 1)); //#17395
                } //#17395
                $put($1.isECI, $1.i, $1.barchar <= -1000000); //#17397
            } //#17397
            $k[$j++] = Infinity; //#17399
            var _88 = $1.nextXterm; //#17399
            for (var _89 = 0, _8A = _88.length; _89 < _8A; _89++) { //#17399
                var _8B = $get(_88, _89); //#17399
                $k[$j++] = _8B; //#17399
                if (_8B > 10000) { //#17399
                    $j--; //#17399
                    $k[$j++] = 10000; //#17399
                } //#17399
            } //#17399
            $1.nextXterm = $a(); //#17399
            $k[$j++] = Infinity; //#17400
            var _8D = $1.nextNonX; //#17400
            for (var _8E = 0, _8F = _8D.length; _8E < _8F; _8E++) { //#17400
                var _8G = $get(_8D, _8E); //#17400
                $k[$j++] = _8G; //#17400
                if (_8G > 10000) { //#17400
                    $j--; //#17400
                    $k[$j++] = 10000; //#17400
                } //#17400
            } //#17400
            $1.nextNonX = $a(); //#17400
            $1.isD = function() {
                $k[$j++] = ($1.char >= 48) && ($1.char <= 57); //#17402
            }; //#17402
            $1.isC = function() {
                var _8M = $get($1.CNvals, $1.char) !== undefined; //#17403
                $k[$j++] = _8M; //#17403
            }; //#17403
            $1.isT = function() {
                var _8P = $get($1.TNvals, $1.char) !== undefined; //#17404
                $k[$j++] = _8P; //#17404
            }; //#17404
            $1.isX = function() {
                var _8S = $get($1.Xvals, $1.char) !== undefined; //#17405
                $k[$j++] = _8S; //#17405
            }; //#17405
            $1.isE = function() {
                var _8V = $get($1.Evals, $1.char) !== undefined; //#17406
                $k[$j++] = _8V; //#17406
            }; //#17406
            $1.isEA = function() {
                $k[$j++] = $1.char > 127; //#17407
            }; //#17407
            $1.isFN = function() {
                $k[$j++] = $1.char < 0; //#17408
            }; //#17408
            $1.XtermFirst = function() {
                var _8Y = $k[--$j]; //#17409
                $k[$j++] = $lt($get($1.nextXterm, _8Y), $get($1.nextNonX, _8Y)); //#17409
            }; //#17409
            $1.A = 0; //#17411
            $1.C = 1; //#17411
            $1.T = 2; //#17411
            $1.X = 3; //#17411
            $1.E = 4; //#17411
            $1.B = 5; //#17411
            $1.lookup = function() {
                $1.ac = 1; //#17414
                $1.cc = 2; //#17414
                $1.tc = 2; //#17414
                $1.xc = 2; //#17414
                $1.ec = 2; //#17414
                $1.bc = 2.25; //#17414
                if ($1.mode == $1.A) { //#17415
                    $1.ac = 0; //#17415
                    $1.cc = 1; //#17415
                    $1.tc = 1; //#17415
                    $1.xc = 1; //#17415
                    $1.ec = 1; //#17415
                    $1.bc = 1.25; //#17415
                } //#17415
                if ($1.mode == $1.C) { //#17416
                    $1.cc = 0; //#17416
                } //#17416
                if ($1.mode == $1.T) { //#17417
                    $1.tc = 0; //#17417
                } //#17417
                if ($1.mode == $1.X) { //#17418
                    $1.xc = 0; //#17418
                } //#17418
                if ($1.mode == $1.E) { //#17419
                    $1.ec = 0; //#17419
                } //#17419
                if ($1.mode == $1.B) { //#17420
                    $1.bc = 0; //#17420
                } //#17420
                for (var _8p = 0, _8q = 1; _8p < _8q; _8p++) { //#17455
                    if ($get($1.isECI, $1.i)) { //#17422
                        $k[$j++] = $1.A; //#17422
                        break; //#17422
                    } //#17422
                    if ($1.mailmark && ($1.i < 45)) { //#17423
                        $k[$j++] = $1.C; //#17423
                        break; //#17423
                    } //#17423
                    $1.k = 0; //#17424
                    for (;;) { //#17454
                        if (($1.i + $1.k) == $1.msglen) { //#17433
                            var _91 = $a(['ac', 'cc', 'tc', 'xc', 'ec', 'bc']); //#17426
                            for (var _92 = 0, _93 = _91.length; _92 < _93; _92++) { //#17426
                                var _94 = $get(_91, _92); //#17426
                                $1[_94] = Math.ceil($1[_94]); //#17426
                            } //#17426
                            var _9B = $a([$1.cc, $1.tc, $1.xc, $1.ec, $1.bc]); //#17427
                            $k[$j++] = true; //#17427
                            for (var _9C = 0, _9D = _9B.length; _9C < _9D; _9C++) { //#17427
                                var _9G = $k[--$j]; //#17427
                                $k[$j++] = _9G && ($1.ac <= $get(_9B, _9C)); //#17427
                            } //#17427
                            if ($k[--$j]) { //#17427
                                $k[$j++] = $1.A; //#17427
                                break; //#17427
                            } //#17427
                            var _9O = $a([$1.ac, $1.cc, $1.tc, $1.xc, $1.ec]); //#17428
                            $k[$j++] = true; //#17428
                            for (var _9P = 0, _9Q = _9O.length; _9P < _9Q; _9P++) { //#17428
                                var _9T = $k[--$j]; //#17428
                                $k[$j++] = _9T && ($1.bc < $get(_9O, _9P)); //#17428
                            } //#17428
                            if ($k[--$j]) { //#17428
                                $k[$j++] = $1.B; //#17428
                                break; //#17428
                            } //#17428
                            var _9b = $a([$1.ac, $1.cc, $1.tc, $1.xc, $1.bc]); //#17429
                            $k[$j++] = true; //#17429
                            for (var _9c = 0, _9d = _9b.length; _9c < _9d; _9c++) { //#17429
                                var _9g = $k[--$j]; //#17429
                                $k[$j++] = _9g && ($1.ec < $get(_9b, _9c)); //#17429
                            } //#17429
                            if ($k[--$j]) { //#17429
                                $k[$j++] = $1.E; //#17429
                                break; //#17429
                            } //#17429
                            var _9o = $a([$1.ac, $1.cc, $1.xc, $1.ec, $1.bc]); //#17430
                            $k[$j++] = true; //#17430
                            for (var _9p = 0, _9q = _9o.length; _9p < _9q; _9p++) { //#17430
                                var _9t = $k[--$j]; //#17430
                                $k[$j++] = _9t && ($1.tc < $get(_9o, _9p)); //#17430
                            } //#17430
                            if ($k[--$j]) { //#17430
                                $k[$j++] = $1.T; //#17430
                                break; //#17430
                            } //#17430
                            var _A1 = $a([$1.ac, $1.cc, $1.tc, $1.ec, $1.bc]); //#17431
                            $k[$j++] = true; //#17431
                            for (var _A2 = 0, _A3 = _A1.length; _A2 < _A3; _A2++) { //#17431
                                var _A6 = $k[--$j]; //#17431
                                $k[$j++] = _A6 && ($1.xc < $get(_A1, _A2)); //#17431
                            } //#17431
                            if ($k[--$j]) { //#17431
                                $k[$j++] = $1.X; //#17431
                                break; //#17431
                            } //#17431
                            $k[$j++] = $1.C; //#17432
                            break; //#17432
                        } //#17432
                        $1.char = $get($1.msg, $1.i + $1.k); //#17434
                        $k[$j++] = 'ac'; //#17435
                        $k[$j++] = $1.ac; //#17435
                        $1.isD(); //#17435
                        if ($k[--$j]) { //#17435
                            var _AG = $k[--$j]; //#17435
                            $k[$j++] = $f(_AG + (1 / 2)); //#17435
                        } else { //#17435
                            $1.isEA(); //#17435
                            if ($k[--$j]) { //#17435
                                var _AI = $k[--$j]; //#17435
                                $k[$j++] = Math.ceil(_AI) + 2; //#17435
                            } else { //#17435
                                var _AJ = $k[--$j]; //#17435
                                $k[$j++] = Math.ceil(_AJ) + 1; //#17435
                            } //#17435
                        } //#17435
                        var _AK = $k[--$j]; //#17435
                        $1[$k[--$j]] = _AK; //#17435
                        $k[$j++] = 'cc'; //#17436
                        $k[$j++] = $1.cc; //#17436
                        $1.isC(); //#17436
                        if ($k[--$j]) { //#17436
                            var _AO = $k[--$j]; //#17436
                            $k[$j++] = $f(_AO + 0.66666667); //#17436
                        } else { //#17436
                            $1.isEA(); //#17436
                            if ($k[--$j]) { //#17436
                                var _AQ = $k[--$j]; //#17436
                                $k[$j++] = $f(_AQ + 2.66666667); //#17436
                            } else { //#17436
                                var _AR = $k[--$j]; //#17436
                                $k[$j++] = $f(_AR + 1.33333334); //#17436
                            } //#17436
                        } //#17436
                        var _AS = $k[--$j]; //#17436
                        $1[$k[--$j]] = _AS; //#17436
                        $k[$j++] = 'tc'; //#17437
                        $k[$j++] = $1.tc; //#17437
                        $1.isT(); //#17437
                        if ($k[--$j]) { //#17437
                            var _AW = $k[--$j]; //#17437
                            $k[$j++] = $f(_AW + 0.66666667); //#17437
                        } else { //#17437
                            $1.isEA(); //#17437
                            if ($k[--$j]) { //#17437
                                var _AY = $k[--$j]; //#17437
                                $k[$j++] = $f(_AY + 2.66666667); //#17437
                            } else { //#17437
                                var _AZ = $k[--$j]; //#17437
                                $k[$j++] = $f(_AZ + 1.33333334); //#17437
                            } //#17437
                        } //#17437
                        var _Aa = $k[--$j]; //#17437
                        $1[$k[--$j]] = _Aa; //#17437
                        $k[$j++] = 'xc'; //#17438
                        $k[$j++] = $1.xc; //#17438
                        $1.isX(); //#17438
                        if ($k[--$j]) { //#17438
                            var _Ae = $k[--$j]; //#17438
                            $k[$j++] = $f(_Ae + 0.66666667); //#17438
                        } else { //#17438
                            $1.isEA(); //#17438
                            if ($k[--$j]) { //#17438
                                var _Ag = $k[--$j]; //#17438
                                $k[$j++] = $f(_Ag + 4.33333334); //#17438
                            } else { //#17438
                                var _Ah = $k[--$j]; //#17438
                                $k[$j++] = $f(_Ah + 3.33333334); //#17438
                            } //#17438
                        } //#17438
                        var _Ai = $k[--$j]; //#17438
                        $1[$k[--$j]] = _Ai; //#17438
                        $k[$j++] = 'ec'; //#17439
                        $k[$j++] = $1.ec; //#17439
                        $1.isE(); //#17439
                        if ($k[--$j]) { //#17439
                            var _Am = $k[--$j]; //#17439
                            $k[$j++] = $f(_Am + (3 / 4)); //#17439
                        } else { //#17439
                            $1.isEA(); //#17439
                            if ($k[--$j]) { //#17439
                                var _Ao = $k[--$j]; //#17439
                                $k[$j++] = $f(_Ao + (17 / 4)); //#17439
                            } else { //#17439
                                var _Ap = $k[--$j]; //#17439
                                $k[$j++] = $f(_Ap + (13 / 4)); //#17439
                            } //#17439
                        } //#17439
                        var _Aq = $k[--$j]; //#17439
                        $1[$k[--$j]] = _Aq; //#17439
                        $k[$j++] = 'bc'; //#17440
                        $k[$j++] = $1.bc; //#17440
                        $1.isFN(); //#17440
                        if ($k[--$j]) { //#17440
                            var _Au = $k[--$j]; //#17440
                            $k[$j++] = $f(_Au + 4); //#17440
                        } else { //#17440
                            var _Av = $k[--$j]; //#17440
                            $k[$j++] = $f(_Av + 1); //#17440
                        } //#17440
                        var _Aw = $k[--$j]; //#17440
                        $1[$k[--$j]] = _Aw; //#17440
                        if ($1.k >= 4) { //#17452
                            var _B4 = $a([$1.cc, $1.tc, $1.xc, $1.ec, $1.bc]); //#17442
                            $k[$j++] = true; //#17442
                            for (var _B5 = 0, _B6 = _B4.length; _B5 < _B6; _B5++) { //#17442
                                var _B9 = $k[--$j]; //#17442
                                $k[$j++] = _B9 && (($1.ac + 1) <= $get(_B4, _B5)); //#17442
                            } //#17442
                            if ($k[--$j]) { //#17442
                                $k[$j++] = $1.A; //#17442
                                break; //#17442
                            } //#17442
                            if (($1.bc + 1) <= $1.ac) { //#17443
                                $k[$j++] = $1.B; //#17443
                                break; //#17443
                            } //#17443
                            var _BJ = $a([$1.cc, $1.tc, $1.xc, $1.ec]); //#17444
                            $k[$j++] = true; //#17444
                            for (var _BK = 0, _BL = _BJ.length; _BK < _BL; _BK++) { //#17444
                                var _BO = $k[--$j]; //#17444
                                $k[$j++] = _BO && (($1.bc + 1) < $get(_BJ, _BK)); //#17444
                            } //#17444
                            if ($k[--$j]) { //#17444
                                $k[$j++] = $1.B; //#17444
                                break; //#17444
                            } //#17444
                            var _BW = $a([$1.ac, $1.cc, $1.tc, $1.xc, $1.bc]); //#17445
                            $k[$j++] = true; //#17445
                            for (var _BX = 0, _BY = _BW.length; _BX < _BY; _BX++) { //#17445
                                var _Bb = $k[--$j]; //#17445
                                $k[$j++] = _Bb && (($1.ec + 1) < $get(_BW, _BX)); //#17445
                            } //#17445
                            if ($k[--$j]) { //#17445
                                $k[$j++] = $1.E; //#17445
                                break; //#17445
                            } //#17445
                            var _Bj = $a([$1.ac, $1.cc, $1.xc, $1.ec, $1.bc]); //#17446
                            $k[$j++] = true; //#17446
                            for (var _Bk = 0, _Bl = _Bj.length; _Bk < _Bl; _Bk++) { //#17446
                                var _Bo = $k[--$j]; //#17446
                                $k[$j++] = _Bo && (($1.tc + 1) < $get(_Bj, _Bk)); //#17446
                            } //#17446
                            if ($k[--$j]) { //#17446
                                $k[$j++] = $1.T; //#17446
                                break; //#17446
                            } //#17446
                            var _Bw = $a([$1.ac, $1.cc, $1.tc, $1.ec, $1.bc]); //#17447
                            $k[$j++] = true; //#17447
                            for (var _Bx = 0, _By = _Bw.length; _Bx < _By; _Bx++) { //#17447
                                var _C1 = $k[--$j]; //#17447
                                $k[$j++] = _C1 && (($1.xc + 1) < $get(_Bw, _Bx)); //#17447
                            } //#17447
                            if ($k[--$j]) { //#17447
                                $k[$j++] = $1.X; //#17447
                                break; //#17447
                            } //#17447
                            var _C8 = $a([$1.ac, $1.tc, $1.ec, $1.bc]); //#17448
                            $k[$j++] = true; //#17448
                            for (var _C9 = 0, _CA = _C8.length; _C9 < _CA; _C9++) { //#17448
                                var _CD = $k[--$j]; //#17448
                                $k[$j++] = _CD && (($1.cc + 1) < $get(_C8, _C9)); //#17448
                            } //#17448
                            if ($k[--$j]) { //#17451
                                if ($1.cc < $1.xc) { //#17449
                                    $k[$j++] = $1.C; //#17449
                                    break; //#17449
                                } //#17449
                                if ($1.cc == $1.xc) { //#17450
                                    $k[$j++] = ($1.i + $1.k) + 1; //#17450
                                    $1.XtermFirst(); //#17450
                                    if ($k[--$j]) { //#17450
                                        $k[$j++] = $1.X; //#17450
                                        break; //#17450
                                    } else { //#17450
                                        $k[$j++] = $1.C; //#17450
                                        break; //#17450
                                    } //#17450
                                } //#17450
                            } //#17450
                        } //#17450
                        $1.k = $1.k + 1; //#17453
                    } //#17453
                } //#17453
            }; //#17456
            $1.addtocws = function() {
                var _CQ = $k[--$j]; //#17459
                $puti($1.cws, $1.j, _CQ); //#17459
                $1.j = _CQ.length + $1.j; //#17460
            }; //#17461
            $1.ECItocws = function() {
                var _CV = $f((-$k[--$j]) - 1000000); //#17465
                $k[$j++] = _CV; //#17479
                if (_CV <= 126) { //#17478
                    var _CW = $k[--$j]; //#17466
                    $k[$j++] = $f(_CW + 1); //#17467
                    $astore($a(1)); //#17467
                } else { //#17478
                    var _CY = $k[--$j]; //#17468
                    $k[$j++] = _CY; //#17479
                    if (_CY <= 16382) { //#17478
                        var _Ca = $f($k[--$j] - 127); //#17470
                        $k[$j++] = (~~(_Ca / 254)) + 128; //#17472
                        $k[$j++] = $f((_Ca % 254) + 1); //#17472
                        $astore($a(2)); //#17472
                    } else { //#17478
                        var _Cd = $f($k[--$j] - 16383); //#17475
                        $k[$j++] = (~~(_Cd / 64516)) + 192; //#17478
                        $k[$j++] = ((~~(_Cd / 254)) % 254) + 1; //#17478
                        $k[$j++] = $f((_Cd % 254) + 1); //#17478
                        $astore($a(3)); //#17478
                    } //#17478
                } //#17478
            }; //#17480
            $1.encA = function() {
                for (var _Cf = 0, _Cg = 1; _Cf < _Cg; _Cf++) { //#17504
                    if ($get($1.isECI, $1.i)) { //#17489
                        $k[$j++] = $get($1.Avals, $1.eci); //#17485
                        $1.addtocws(); //#17485
                        $k[$j++] = $get($1.msg, $1.i); //#17486
                        $1.ECItocws(); //#17486
                        $1.addtocws(); //#17486
                        $1.i = $1.i + 1; //#17487
                        break; //#17488
                    } //#17488
                    if ($get($1.numD, $1.i) >= 2) { //#17494
                        var _Cu = $s(2); //#17491
                        $put(_Cu, 0, $get($1.msg, $1.i)); //#17491
                        $put(_Cu, 1, $get($1.msg, $1.i + 1)); //#17491
                        $k[$j++] = $get($1.Avals, _Cu); //#17491
                        $1.addtocws(); //#17491
                        $1.i = $1.i + 2; //#17492
                        break; //#17493
                    } //#17493
                    $k[$j++] = 'newmode'; //#17495
                    $1.lookup(); //#17495
                    var _D4 = $k[--$j]; //#17495
                    $1[$k[--$j]] = _D4; //#17495
                    if ($1.newmode != $1.mode) { //#17500
                        $k[$j++] = $get($1.Avals, $get($a([-1, $1.lC, $1.lT, $1.lX, $1.lE, $1.lB]), $1.newmode)); //#17497
                        $1.addtocws(); //#17497
                        $1.mode = $1.newmode; //#17498
                        break; //#17499
                    } //#17499
                    $k[$j++] = $get($1.Avals, $get($1.msg, $1.i)); //#17501
                    $1.addtocws(); //#17501
                    $1.i = $1.i + 1; //#17502
                    break; //#17503
                } //#17503
            }; //#17505
            $1.CTXvalstocws = function() {
                $1.in = $k[--$j]; //#17508
                $k[$j++] = Infinity; //#17509
                for (var _DS = 0, _DR = $1.in.length - 1; _DS <= _DR; _DS += 3) { //#17513
                    var _DU = $geti($1.in, _DS, 3); //#17511
                    $k[$j++] = 0; //#17511
                    for (var _DV = 0, _DW = _DU.length; _DV < _DW; _DV++) { //#17511
                        var _DY = $k[--$j]; //#17511
                        $k[$j++] = $f(_DY + $get(_DU, _DV)) * 40; //#17511
                    } //#17511
                    var _Da = (~~($k[--$j] / 40)) + 1; //#17512
                    $k[$j++] = ~~(_Da / 256); //#17512
                    $k[$j++] = _Da % 256; //#17512
                } //#17512
                $astore($a($counttomark())); //#17514
                var _Dd = $k[--$j]; //#17514
                var _De = $k[--$j]; //#17514
                $k[$j++] = _Dd; //#17514
                $k[$j++] = _De; //#17514
                $j--; //#17514
            }; //#17515
            $1.encCTX = function() {
                $1.p = 0; //#17518
                $1.ctxvals = $a(2500); //#17519
                $1.done = false; //#17520
                for (;;) { //#17597
                    if ($1.i == $1.msglen) { //#17524
                        break; //#17524
                    } //#17524
                    var _Do = $get($get($1.encvals, $1.mode), $get($1.msg, $1.i)) !== undefined; //#17525
                    if (!_Do) { //#17525
                        break; //#17525
                    } //#17525
                    if (($1.p % 3) == 0) { //#17592
                        $k[$j++] = 'newmode'; //#17527
                        $1.lookup(); //#17527
                        var _Dq = $k[--$j]; //#17527
                        $1[$k[--$j]] = _Dq; //#17527
                        if ($ne($1.newmode, $1.mode)) { //#17537
                            $k[$j++] = $geti($1.ctxvals, 0, $1.p); //#17529
                            $1.CTXvalstocws(); //#17529
                            $1.addtocws(); //#17529
                            $k[$j++] = $a([$1.unlcw]); //#17530
                            $1.addtocws(); //#17530
                            if ($1.newmode != $1.A) { //#17533
                                $k[$j++] = $get($1.Avals, $get($a([-1, $1.lC, $1.lT, $1.lX, $1.lE, $1.lB]), $1.newmode)); //#17532
                                $1.addtocws(); //#17532
                            } //#17532
                            $1.mode = $1.newmode; //#17534
                            $1.done = true; //#17535
                            break; //#17536
                        } //#17536
                        if (($1.msglen - $1.i) <= 3) { //#17591
                            $1.remcws = $get($1.numremcws, $1.j + ((~~($1.p / 3)) * 2)); //#17539
                            $k[$j++] = Infinity; //#17545
                