/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2022
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { forEach } from '../../globals/internal/collection-helpers';

/**
 * @param monthNumber The month number.
 * @param shorthand `true` to use shorthand month.
 * @param locale The Flatpickr locale data.
 * @returns The month string.
 */
const monthToStr = (monthNumber, shorthand, locale) => locale.months[shorthand ? 'shorthand' : 'longhand'][monthNumber];

/**
 * The configuration for the Flatpickr plugin to use text instead of `<select>` for month picker.
 */

/**
 * @param config Plugin configuration.
 * @returns A Flatpickr plugin to use text instead of `<select>` for month picker.
 */
export default (config => fp => {
  /**
   * Replaces `<select>` for month with text content.
   */
  const setupElements = () => {
    const {
      monthElements,
      yearElements,
      currentMonth,
      l10n,
      _createElement: createElement
    } = fp;
    if (!monthElements) {
      return;
    }
    const {
      shorthand,
      selectorFlatpickrMonthYearContainer,
      selectorFlatpickrYearContainer,
      classFlatpickrCurrentMonth
    } = config;
    monthElements.forEach(elem => {
      if (!elem.parentNode) return;
      elem.parentNode.removeChild(elem);
    });
    monthElements.splice(0, monthElements.length, ...monthElements.map(() => {
      const monthElement = createElement('span', classFlatpickrCurrentMonth);
      monthElement.textContent = monthToStr(currentMonth, shorthand === true, l10n);
      const monthYearContainer = yearElements[0].closest(selectorFlatpickrMonthYearContainer);
      if (monthYearContainer) {
        monthYearContainer.insertBefore(monthElement, yearElements[0].closest(selectorFlatpickrYearContainer));
      }
      return monthElement;
    }));
  };

  /**
   * Updates the text-based month UI with the latest selected date.
   */
  const updateCurrentMonth = () => {
    const {
      yearElements,
      currentMonth,
      l10n
    } = fp;
    const {
      shorthand,
      selectorFlatpickrMonthYearContainer,
      selectorFlatpickrCurrentMonth
    } = config;
    const monthStr = monthToStr(currentMonth, shorthand === true, l10n);
    yearElements.forEach(elem => {
      const currentMonthContainer = elem.closest(selectorFlatpickrMonthYearContainer);
      if (currentMonthContainer) {
        forEach(currentMonthContainer.querySelectorAll(selectorFlatpickrCurrentMonth), monthElement => {
          monthElement.textContent = monthStr;
        });
      }
    });
  };

  /**
   * Registers this Flatpickr plugin.
   */
  const register = () => {
    fp.loadedPlugins.push('carbonFlatpickrMonthSelectPlugin');
  };
  return {
    onMonthChange: updateCurrentMonth,
    onValueUpdate: updateCurrentMonth,
    onOpen: updateCurrentMonth,
    onReady: [setupElements, updateCurrentMonth, register]
  };
});
//# sourceMappingURL=month-select-plugin.js.map
