/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import ChevronRight16 from '../../icons/chevron--right/16.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import { ACCORDION_ITEM_BREAKPOINT } from './defs.js';
export { ACCORDION_ITEM_BREAKPOINT } from './defs.js';
import styles from './accordion.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Observes resize of the given element with the given resize observer.
 *
 * @param observer The resize observer.
 * @param elem The element to observe the resize.
 */
const observeResize = (observer, elem) => {
    if (!elem) {
        return null;
    }
    observer.observe(elem);
    return {
        release() {
            observer.unobserve(elem);
            return null;
        },
    };
};
/**
 * Accordion item.
 *
 * @element cds-accordion-item
 * @fires cds-accordion-item-beingtoggled
 *   The custom event fired before this accordion item is being toggled upon a user gesture.
 *   Cancellation of this event stops the user-initiated action of toggling this accordion item.
 * @fires cds-accordion-item-toggled - The custom event fired after this accordion item is toggled upon a user gesture.
 * @csspart expando The expando button.
 * @csspart expando-icon The expando icon.
 * @csspart title The title.
 * @csspart content The content.
 */
let CDSAccordionItem = class CDSAccordionItem extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * The handle for observing resize of the parent element of this element.
         */
        this._hObserveResize = null;
        /**
         * Handler for the `keydown` event on the expando button.
         */
        this._handleKeydownExpando = ({ key }) => {
            if (this.open && (key === 'Esc' || key === 'Escape')) {
                this._handleUserInitiatedToggle(false);
            }
        };
        /**
         * The `ResizeObserver` instance for observing element resizes for re-positioning floating menu position.
         */
        // TODO: Wait for `.d.ts` update to support `ResizeObserver`
        // @ts-ignore
        this._resizeObserver = new ResizeObserver((records) => {
            const { width } = records[records.length - 1].contentRect;
            const { _sizesBreakpoints: sizesBreakpoints } = this
                .constructor;
            this._currentBreakpoint = Object.keys(sizesBreakpoints)
                .sort((lhs, rhs) => sizesBreakpoints[rhs] - sizesBreakpoints[lhs])
                .find((size) => width >= sizesBreakpoints[size]);
            this.requestUpdate();
        });
        /**
         * `true` if the accordion item should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if the accordion item should be open.
         */
        this.open = false;
        /**
         * The title text.
         */
        this.title = '';
    }
    /**
     * Handles user-initiated toggle request of this accordion item.
     *
     * @param open The new open state.
     */
    _handleUserInitiatedToggle(open = !this.open) {
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                open,
            },
        };
        if (this.dispatchEvent(new CustomEvent(this.constructor.eventBeforeToggle, init))) {
            const { selectorAccordionContent } = this
                .constructor;
            !this.open
                ? this.setAttribute('expanding', '')
                : this.setAttribute('collapsing', '');
            this.shadowRoot.querySelector(selectorAccordionContent).addEventListener('animationend', () => {
                this.removeAttribute('expanding');
                this.removeAttribute('collapsing');
            });
            this.open = open;
            this.dispatchEvent(new CustomEvent(this.constructor.eventToggle, init));
        }
        const content = this.shadowRoot.querySelector(`.${prefix}--accordion__wrapper`);
        if (this.open) {
            // accordion opens
            content.style.maxBlockSize = content.scrollHeight + 15 + 'px';
        }
        else {
            // accordion closes
            content.style.maxBlockSize = '';
        }
    }
    /**
     * Handler for the `click` event on the expando button.
     */
    _handleClickExpando() {
        this._handleUserInitiatedToggle();
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'listitem');
        }
        super.connectedCallback();
        if (this._hObserveResize) {
            this._hObserveResize = this._hObserveResize.release();
        }
        this._hObserveResize = observeResize(this._resizeObserver, this);
    }
    disconnectedCallback() {
        if (this._hObserveResize) {
            this._hObserveResize = this._hObserveResize.release();
        }
    }
    render() {
        const { disabled, title, open, _currentBreakpoint: currentBreakpoint, _handleClickExpando: handleClickExpando, _handleKeydownExpando: handleKeydownExpando, } = this;
        const { _classesBreakpoints: classesBreakpoints } = this
            .constructor;
        const { [currentBreakpoint]: classBreakpoint } = classesBreakpoints;
        const contentClasses = classMap({
            [classBreakpoint]: classBreakpoint,
            [`${prefix}--accordion__content`]: true,
        });
        return html `
      <button
        ?disabled="${disabled}"
        type="button"
        part="expando"
        class="${prefix}--accordion__heading"
        aria-controls="content"
        aria-expanded="${open}"
        @click="${handleClickExpando}"
        @keydown="${handleKeydownExpando}">
        ${ChevronRight16({
            part: 'expando-icon',
            class: `${prefix}--accordion__arrow`,
        })}
        <div part="title" class="${prefix}--accordion__title">
          <slot name="title">${title}</slot>
        </div>
      </button>
      <div class="${prefix}--accordion__wrapper">
        <div id="content" part="content" class="${contentClasses}">
          <slot></slot>
        </div>
      </div>
    `;
    }
    /**
     * The CSS classes for breakpoints.
     *
     * @private
     */
    static get _classesBreakpoints() {
        return {
            [ACCORDION_ITEM_BREAKPOINT.SMALL]: `${prefix}-ce--accordion__content--${ACCORDION_ITEM_BREAKPOINT.SMALL}`,
            [ACCORDION_ITEM_BREAKPOINT.MEDIUM]: `${prefix}-ce--accordion__content--${ACCORDION_ITEM_BREAKPOINT.MEDIUM}`,
        };
    }
    /**
     * The breakpoints.
     *
     * @private
     */
    static get _sizesBreakpoints() {
        return {
            [ACCORDION_ITEM_BREAKPOINT.SMALL]: 480,
            [ACCORDION_ITEM_BREAKPOINT.MEDIUM]: 640,
        };
    }
    /**
     * The name of the custom event fired before this accordion item is being toggled upon a user gesture.
     * Cancellation of this event stops the user-initiated action of toggling this accordion item.
     */
    static get eventBeforeToggle() {
        return `${prefix}-accordion-item-beingtoggled`;
    }
    /**
     * The name of the custom event fired after this accordion item is toggled upon a user gesture.
     */
    static get eventToggle() {
        return `${prefix}-accordion-item-toggled`;
    }
    static get selectorAccordionContent() {
        return `.${prefix}--accordion__content`;
    }
};
CDSAccordionItem.styles = styles;
__decorate([
    property({ type: Boolean, reflect: true })
], CDSAccordionItem.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSAccordionItem.prototype, "open", void 0);
__decorate([
    property({ attribute: 'title' })
], CDSAccordionItem.prototype, "title", void 0);
CDSAccordionItem = __decorate([
    carbonElement(`${prefix}-accordion-item`)
], CDSAccordionItem);
var CDSAccordionItem$1 = CDSAccordionItem;

export { CDSAccordionItem$1 as default };
//# sourceMappingURL=accordion-item.js.map
