/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import './accordion.js';
import { forEach } from '../../globals/internal/collection-helpers.js';
import ChevronRight16 from '../../icons/chevron--right/16.js';
import './accordion-item-skeleton.js';
import '../skeleton-text/skeleton-text.js';
import styles from './accordion.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { ACCORDION_ALIGNMENT } from './defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Skeleton of code snippet.
 */
let CDSAccordionSkeleton = class CDSAccordionSkeleton extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * Specify the alignment of the accordion heading title and chevron
         */
        this.alignment = ACCORDION_ALIGNMENT.END;
        /**
         * Set number of items to render
         */
        this.count = 4;
        /**
         * Specify whether Accordion text should be flush, default is false, does not work with align="start"
         */
        this.isFlush = false;
        /**
         * `true` if the first accordion item should be open.
         */
        this.open = true;
    }
    updated(changedProperties) {
        if (changedProperties.has('alignment')) {
            // Propagate `alignment` attribute to descendants until `:host-context()` gets supported in all major browsers
            forEach(this.shadowRoot.querySelectorAll(this.constructor
                .selectorAccordionItemSkeletons), (elem) => {
                elem.setAttribute('alignment', this.alignment);
            });
        }
        if (changedProperties.has('isFlush') ||
            changedProperties.has('alignment')) {
            // Propagate `isFlush` attribute to descendants until `:host-context()` gets supported in all major browsers
            forEach(this.shadowRoot.querySelectorAll(this.constructor
                .selectorAccordionItemSkeletons), (elem) => {
                this.isFlush && this.alignment !== 'start'
                    ? elem.setAttribute('isFlush', '')
                    : elem.removeAttribute('isFlush');
            });
        }
    }
    render() {
        const classes = classMap({
            [`${prefix}--accordion__item`]: true,
            [`${prefix}--accordion__item--active`]: true,
            [`${prefix}--accordion--${this.alignment}`]: this.alignment,
            [`${prefix}--accordion--flush`]: this.isFlush && this.alignment !== 'start',
        });
        const numSkeletonItems = this.open ? this.count - 1 : this.count;
        return html `
      ${this.open
            ? html `
            <li class="${classes}">
              <span class="${prefix}--accordion__heading">
                ${ChevronRight16({
                part: 'expando-icon',
                class: `${prefix}--accordion__arrow`,
            })}
                <cds-skeleton-text
                  class="${prefix}--accordion__title"></cds-skeleton-text>
              </span>
              <div class="${prefix}--accordion__content">
                <cds-skeleton-text width="90%"></cds-skeleton-text>
                <cds-skeleton-text width="80%"></cds-skeleton-text>
                <cds-skeleton-text width="85%"></cds-skeleton-text>
              </div>
            </li>
          `
            : ``}
      ${Array.from(new Array(numSkeletonItems)).map((_, index) => html `
          <cds-accordion-item-skeleton
            key=${index}></cds-accordion-item-skeleton>
        `)}
    `;
    }
    static get selectorAccordionItemSkeletons() {
        return `${prefix}-accordion-item-skeleton`;
    }
};
CDSAccordionSkeleton.styles = styles;
__decorate([
    property({ reflect: true })
], CDSAccordionSkeleton.prototype, "alignment", void 0);
__decorate([
    property({ type: Number, attribute: 'count' })
], CDSAccordionSkeleton.prototype, "count", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSAccordionSkeleton.prototype, "isFlush", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSAccordionSkeleton.prototype, "open", void 0);
CDSAccordionSkeleton = __decorate([
    carbonElement(`${prefix}-accordion-skeleton`)
], CDSAccordionSkeleton);
var CDSAccordionSkeleton$1 = CDSAccordionSkeleton;

export { CDSAccordionSkeleton$1 as default };
//# sourceMappingURL=accordion-skeleton.js.map
