/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import CDSToggleTip from '../toggle-tip/toggletip.js';
import styles from './ai-label.scss.js';
import Undo16 from '../../icons/undo/16';
import { AI_LABEL_KIND, AI_LABEL_SIZE } from './defs.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Basic AI Label.
 *
 * @element cds-ai-label
 */
let CDSAILabel = class CDSAILabel extends CDSToggleTip {
    constructor() {
        super(...arguments);
        this.slot = 'ai-label';
        /**
         * Specify the correct translation of the AI text
         */
        this.aiText = 'AI';
        /**
         * Specify additional text to be rendered next to the AI label in the inline variant
         */
        this.aiTextLabel = '';
        /**
         * Specify the type of AI Label, from the following list of types: (default, inline)
         */
        this.kind = AI_LABEL_KIND.DEFAULT;
        /**
         * Specify whether the revert button should be visible
         */
        this.revertActive = false;
        /**
         * Specify whether the revert button should be visible
         */
        this.revertLabel = 'Revert to AI input';
        /**
         * AI Label size should be mini, 2xs, xs, sm, md, lg, xl.
         */
        this.size = AI_LABEL_SIZE.EXTRA_SMALL;
        /**
         * Specify the text that will be provided to the aria-label of the `AI Label` button
         */
        this.buttonLabel = 'Show information';
        this._handleClick = () => {
            if (this.revertActive) {
                this.revertActive = false;
                this.removeAttribute('revert-active');
            }
            else {
                this.open = !this.open;
            }
        };
        this._renderToggleTipLabel = () => {
            return html ``;
        };
        this._renderTooltipButton = () => {
            const { size, kind, aiText, aiTextLabel, buttonLabel } = this;
            const ariaLabel = `${aiText} - ${buttonLabel}`;
            const classes = classMap({
                [`${prefix}--toggletip-button`]: true,
                [`${prefix}--slug__button`]: true,
                [`${prefix}--slug__button--${size}`]: size,
                [`${prefix}--slug__button--${kind}`]: kind,
                [`${prefix}--slug__button--inline-with-content`]: kind === AI_LABEL_KIND.INLINE && aiTextLabel,
            });
            return html `
      <button
        aria-controls="${this.id}"
        @click="${this._handleClick}"
        class=${classes}
        aria-label="${ariaLabel}">
        <span class="${prefix}--slug__text">${aiText}</span>
        ${aiTextLabel && kind === AI_LABEL_KIND.INLINE
                ? html `
              <span class="${prefix}--slug__additional-text">
                ${aiTextLabel}
              </span>
            `
                : ``}
      </button>
    `;
        };
        this._renderInnerContent = () => {
            const { autoalign, revertActive, revertLabel } = this;
            return html `
      ${revertActive
                ? html `
            <cds-icon-button
              ?autoalign=${autoalign}
              kind="ghost"
              size="sm"
              @click="${this._handleClick}">
              <span slot="tooltip-content"> ${revertLabel} </span>
              ${Undo16({ slot: 'icon' })}
            </cds-icon-button>
          `
                : html `
            ${this._renderTooltipButton()} ${this._renderTooltipContent()}
          `}
    `;
        };
    }
    attributeChangedCallback(name, old, newValue) {
        var _a;
        super.attributeChangedCallback(name, old, newValue);
        //@ts-ignore typescript does not think requestUpdate() exists on parentElement
        name === 'revert-active' ? (_a = this.parentElement) === null || _a === void 0 ? void 0 : _a.requestUpdate() : ``;
    }
};
CDSAILabel.styles = styles;
__decorate([
    property({ reflect: true })
], CDSAILabel.prototype, "slot", void 0);
__decorate([
    property({ attribute: 'ai-text' })
], CDSAILabel.prototype, "aiText", void 0);
__decorate([
    property({ attribute: 'ai-text-label' })
], CDSAILabel.prototype, "aiTextLabel", void 0);
__decorate([
    property({ reflect: true })
], CDSAILabel.prototype, "kind", void 0);
__decorate([
    property({ type: Boolean, attribute: 'revert-active' })
], CDSAILabel.prototype, "revertActive", void 0);
__decorate([
    property({ attribute: 'revert-label' })
], CDSAILabel.prototype, "revertLabel", void 0);
__decorate([
    property({ reflect: true })
], CDSAILabel.prototype, "size", void 0);
__decorate([
    property({ attribute: 'button-label' })
], CDSAILabel.prototype, "buttonLabel", void 0);
__decorate([
    property()
], CDSAILabel.prototype, "previousValue", void 0);
CDSAILabel = __decorate([
    carbonElement(`${prefix}-ai-label`)
], CDSAILabel);
var CDSAILabel$1 = CDSAILabel;

export { CDSAILabel$1 as default };
//# sourceMappingURL=ai-label.js.map
