/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import '../button/button.js';
import { CHAT_BUTTON_KIND, CHAT_BUTTON_SIZE } from './defs.js';
export { CHAT_BUTTON_KIND, CHAT_BUTTON_SIZE } from './defs.js';
import styles from './chat-button.scss.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Icon Button
 *
 */
let CDSChatButton = class CDSChatButton extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an icon.
         */
        this._hasIcon = false;
        /**
         * `true` if the button should be disabled.
         */
        this.disabled = false;
        /**
         *  Specify whether the `ChatButton` should be disabled
         */
        this.kind = CHAT_BUTTON_KIND.PRIMARY;
        /**
         * Chat button size.
         */
        this.size = CHAT_BUTTON_SIZE.LARGE;
        /**
         * Specify whether the `ChatButton` should be rendered as a quick action button
         */
        this.isQuickAction = false;
        /**
         * Specify whether the quick action `ChatButton` should be rendered as selected. This disables the input
         */
        this.isSelected = false;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        this._hasIcon = target
            .assignedNodes()
            .some((node) => node.nodeType !== Node.TEXT_NODE || node.textContent.trim());
        this.requestUpdate();
    }
    render() {
        const allowedSizes = [
            CHAT_BUTTON_SIZE.SMALL,
            CHAT_BUTTON_SIZE.MEDIUM,
            CHAT_BUTTON_SIZE.LARGE,
        ];
        if (this.isQuickAction) {
            this.kind = CHAT_BUTTON_KIND.GHOST;
            this.size = CHAT_BUTTON_SIZE.SMALL;
        }
        else {
            // Do not allow size larger than `lg`
            this.size = allowedSizes.includes(this.size)
                ? this.size
                : CHAT_BUTTON_SIZE.LARGE;
        }
        let classes = `${prefix}--chat-btn`;
        classes += this._hasIcon ? ` ${prefix}--chat-btn--with-icon` : '';
        classes += this.isQuickAction ? ` ${prefix}--chat-btn--quick-action` : '';
        classes += this.isSelected
            ? ` ${prefix}--chat-btn--quick-action--selected`
            : '';
        return html `
      <cds-button
        button-class-name="${classes}"
        size="${this.size}"
        kind="${this.kind}"
        ?disabled="${this.disabled}">
        <slot></slot
        ><slot
          name="icon"
          slot="icon"
          @slotchange="${this._handleSlotChange}"></slot>
      </cds-button>
    `;
    }
};
CDSChatButton.styles = styles; // `styles` here is a `CSSResult` generated by custom WebPack loader
__decorate([
    property({ type: Boolean, reflect: true })
], CDSChatButton.prototype, "disabled", void 0);
__decorate([
    property({ reflect: true })
], CDSChatButton.prototype, "kind", void 0);
__decorate([
    property({ reflect: true })
], CDSChatButton.prototype, "size", void 0);
__decorate([
    property({ attribute: 'is-quick-action', type: Boolean })
], CDSChatButton.prototype, "isQuickAction", void 0);
__decorate([
    property({ attribute: 'is-selected', type: Boolean })
], CDSChatButton.prototype, "isSelected", void 0);
CDSChatButton = __decorate([
    carbonElement(`${prefix}-chat-button`)
], CDSChatButton);
var CDSChatButton$1 = CDSChatButton;

export { CDSChatButton$1 as default };
//# sourceMappingURL=chat-button.js.map
