/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import styles from './checkbox.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Check box.
 *
 * @element cds-checkbox
 * @fires cds-checkbox-changed - The custom event fired after this changebox changes its checked state.
 * @csspart input The checkbox.
 * @csspart label The label.
 */
let CDSCheckboxGroup = class CDSCheckboxGroup extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * Whether the CheckboxGroup should be read-only
         */
        this.readonly = false;
        /**
         * Specify whether the form group is currently in warning state
         */
        this.warn = false;
        /**
         * Provide the text that is displayed when the form group is in warning state
         */
        this.warnText = '';
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
    }
    /*
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove slug reference in v12
                elem.matches(this.constructor.slugItem)
            : false);
        this._hasAILabel = Boolean(hasContent);
        hasContent[0].setAttribute('size', 'mini');
        this.requestUpdate();
    }
    updated(changedProperties) {
        const { selectorCheckbox } = this.constructor;
        const checkboxes = this.querySelectorAll(selectorCheckbox);
        ['disabled', 'readonly'].forEach((name) => {
            if (changedProperties.has(name)) {
                const { [name]: value } = this;
                // Propagate the property to descendants until `:host-context()` gets supported in all major browsers
                checkboxes.forEach((elem) => {
                    elem[name] = value;
                });
            }
        });
        if (changedProperties.has('invalid')) {
            const { invalid } = this;
            checkboxes.forEach((elem) => {
                if (invalid) {
                    elem.setAttribute('invalid-group', '');
                }
                else {
                    elem.removeAttribute('invalid-group');
                }
            });
        }
    }
    render() {
        const { ariaLabelledBy, disabled, helperText, invalid, invalidText, legendId, legendText, readonly, warn, warnText, _hasAILabel: hasAILabel, _handleSlotChange: handleSlotChange, } = this;
        const showWarning = !readonly && !invalid && warn;
        const showHelper = !invalid && !warn;
        const checkboxGroupInstanceId = Math.random().toString(16).slice(2);
        const helperId = !helperText
            ? undefined
            : `checkbox-group-helper-text-${checkboxGroupInstanceId}`;
        const helper = helperText
            ? html ` <div id="${helperId}" class="${prefix}--form__helper-text">
          ${helperText}
        </div>`
            : null;
        const fieldsetClasses = classMap({
            [`${prefix}--checkbox-group`]: true,
            [`${prefix}--checkbox-group--readonly`]: readonly,
            [`${prefix}--checkbox-group--invalid`]: !readonly && invalid,
            [`${prefix}--checkbox-group--warning`]: showWarning,
            [`${prefix}--checkbox-group--slug`]: hasAILabel,
        });
        return html `
      <fieldset
        class="${fieldsetClasses}"
        ?data-invalid=${invalid}
        ?disabled=${disabled}
        aria-readonly=${readonly}
        ?aria-labelledby=${ariaLabelledBy || legendId}
        ?aria-describedby=${!invalid && !warn && helper ? helperId : undefined}>
        <legend class="${prefix}--label" id=${legendId || ariaLabelledBy}>
          ${legendText}
          <slot name="ai-label" @slotchange="${handleSlotChange}"></slot>
          <slot name="slug" @slotchange="${handleSlotChange}"></slot>
        </legend>
        <slot></slot>
        <div class="${prefix}--checkbox-group__validation-msg">
          ${!readonly && invalid
            ? html `
                ${WarningFilled16({
                class: `${prefix}--checkbox__invalid-icon`,
            })}
                <div class="${prefix}--form-requirement">${invalidText}</div>
              `
            : null}
          ${showWarning
            ? html `
                ${WarningAltFilled16({
                class: `${prefix}--checkbox__invalid-icon ${prefix}--checkbox__invalid-icon--warning`,
            })}
                <div class="${prefix}--form-requirement">${warnText}</div>
              `
            : null}
        </div>
        ${showHelper ? helper : null}
      </fieldset>
    `;
    }
    /**
     * A selector that will return the checkboxes.
     */
    static get selectorCheckbox() {
        return `${prefix}-checkbox`;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
};
CDSCheckboxGroup.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSCheckboxGroup.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    property({ type: String, reflect: true, attribute: 'aria-labelledby' })
], CDSCheckboxGroup.prototype, "ariaLabelledBy", void 0);
__decorate([
    property({ type: Boolean })
], CDSCheckboxGroup.prototype, "disabled", void 0);
__decorate([
    property({ type: String, reflect: true, attribute: 'helper-text' })
], CDSCheckboxGroup.prototype, "helperText", void 0);
__decorate([
    property({ type: Boolean, attribute: 'invalid' })
], CDSCheckboxGroup.prototype, "invalid", void 0);
__decorate([
    property({ type: String, reflect: true, attribute: 'invalid-text' })
], CDSCheckboxGroup.prototype, "invalidText", void 0);
__decorate([
    property({ type: String, reflect: true, attribute: 'legend-id' })
], CDSCheckboxGroup.prototype, "legendId", void 0);
__decorate([
    property({ type: String, reflect: true, attribute: 'legend-text' })
], CDSCheckboxGroup.prototype, "legendText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCheckboxGroup.prototype, "readonly", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCheckboxGroup.prototype, "warn", void 0);
__decorate([
    property({ type: String, reflect: true, attribute: 'warn-text' })
], CDSCheckboxGroup.prototype, "warnText", void 0);
CDSCheckboxGroup = __decorate([
    carbonElement(`${prefix}-checkbox-group`)
], CDSCheckboxGroup);
var CDSCheckboxGroup$1 = CDSCheckboxGroup;

export { CDSCheckboxGroup$1 as default };
//# sourceMappingURL=checkbox-group.js.map
