/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { LitElement, html } from 'lit';
import { query, property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import FormMixin from '../../globals/mixins/form.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import styles from './checkbox.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Check box.
 *
 * @element cds-checkbox
 * @fires cds-checkbox-changed - The custom event fired after this changebox changes its checked state.
 * @csspart input The checkbox.
 * @csspart label The label.
 */
let CDSCheckbox = class CDSCheckbox extends FocusMixin(FormMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * Specify whether the underlying input should be checked
         */
        this.checked = false;
        /**
         * Specify if checkbox is being used in a data table
         */
        this.dataTable = false;
        /**
         * Specify whether the Checkbox should be disabled
         */
        this.disabled = false;
        /**
         * Specify whether the checkbox should be present in the DOM,
         * but invisible and uninteractable. Used for data-table purposes.
         */
        this.hideCheckbox = false;
        /**
         * Specify whether the label should be hidden, or not
         */
        this.hideLabel = false;
        /**
         * Specify whether the Checkbox is in an indeterminate state
         */
        this.indeterminate = false;
        /**
         * Provide a label to provide a description of the Checkbox input that you are
         * exposing to the user
         */
        this.labelText = '';
        /**
         * Specify whether the Checkbox is read-only
         */
        this.readonly = false;
        /**
         * Specify whether the Checkbox is currently invalid
         */
        this.invalid = false;
        /**
         * Specify a title for the node for the Checkbox
         */
        this.title = '';
        /**
         * Specify whether the Checkbox is in a warn state
         */
        this.warn = false;
        /**
         * Provide the text that is displayed when the Checkbox is in a warn state
         */
        this.warnText = false;
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
    }
    /**
     * Handles `click` event on the `<input>` in the shadow DOM.
     */
    _handleChange() {
        const { checked, indeterminate } = this._checkboxNode;
        this.checked = checked;
        this.indeterminate = indeterminate;
        const { eventChange } = this.constructor;
        this.dispatchEvent(new CustomEvent(eventChange, {
            bubbles: true,
            composed: true,
            detail: {
                checked,
                indeterminate,
            },
        }));
    }
    /**
     * Prevent checkbox state from updating when readonly
     */
    _handleClick(event) {
        if (this.readonly) {
            event.preventDefault();
        }
    }
    _handleFormdata(event) {
        const { formData } = event; // TODO: Wait for `FormDataEvent` being available in `lib.dom.d.ts`
        const { checked, disabled, name, value = 'on' } = this;
        if (!disabled && checked) {
            formData.append(name, value);
        }
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove slug reference in v12
                elem.matches(this.constructor.slugItem)
            : false);
        this._hasAILabel = Boolean(hasContent);
        const type = hasContent[0].getAttribute('kind');
        hasContent[0].setAttribute('size', type === 'inline' ? 'md' : 'mini');
        this.requestUpdate();
    }
    updated() {
        const { _hasAILabel: hasAILabel } = this;
        hasAILabel
            ? this.setAttribute('ai-label', '')
            : this.removeAttribute('ai-label');
    }
    render() {
        const { checked, disabled, helperText, hideLabel, indeterminate, invalid, invalidText, labelText, name, readonly, title, value, warn, warnText, _handleChange: handleChange, _handleClick: handleClick, } = this;
        const showWarning = !readonly && !invalid && warn;
        const showHelper = !invalid && !warn;
        const helper = helperText
            ? html ` <div class="${prefix}--form__helper-text">${helperText}</div>`
            : null;
        const labelClasses = classMap({
            [`${prefix}--checkbox-label`]: true,
        });
        const labelTextClasses = classMap({
            [`${prefix}--checkbox-label-text`]: true,
            [`${prefix}--visually-hidden`]: hideLabel,
        });
        return html `
      <input
        id="checkbox"
        type="checkbox"
        part="input"
        class="${`${prefix}--checkbox`}"
        aria-checked="${indeterminate ? 'mixed' : String(Boolean(checked))}"
        aria-readonly="${String(Boolean(readonly))}"
        .checked="${checked}"
        ?disabled="${disabled}"
        .indeterminate="${indeterminate}"
        name="${ifDefined(name)}"
        value="${ifDefined(value)}"
        @change="${handleChange}"
        @click="${handleClick}" />
      <label
        for="checkbox"
        part="label"
        class="${labelClasses}"
        title="${ifDefined(title)}">
        <span class="${labelTextClasses}"
          >${labelText ? labelText : html `<slot></slot>`}</span
        >
      </label>
      <slot name="ai-label" @slotchange="${this._handleSlotChange}"></slot>
      <slot name="slug" @slotchange="${this._handleSlotChange}"></slot>
      <div class="${prefix}--checkbox__validation-msg">
        ${!readonly && invalid
            ? html `
              ${WarningFilled16({
                class: `${prefix}--checkbox__invalid-icon`,
            })}
              <div class="${prefix}--form-requirement">${invalidText}</div>
            `
            : null}
        ${showWarning
            ? html `
              ${WarningAltFilled16({
                class: `${prefix}--checkbox__invalid-icon ${prefix}--checkbox__invalid-icon--warning`,
            })}
              <div class="${prefix}--form-requirement">${warnText}</div>
            `
            : null}
      </div>
      ${showHelper ? helper : null}
    `;
    }
    /**
     * The name of the custom event fired after this changebox changes its checked state.
     */
    static get eventChange() {
        return `${prefix}-checkbox-changed`;
    }
    /**
     * A selector that will return the slug item.
     *
     * Remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the ai-label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
};
CDSCheckbox.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSCheckbox.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('input')
], CDSCheckbox.prototype, "_checkboxNode", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'checked' })
], CDSCheckbox.prototype, "checked", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'data-table' })
], CDSCheckbox.prototype, "dataTable", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCheckbox.prototype, "disabled", void 0);
__decorate([
    property({ type: String, reflect: true, attribute: 'helper-text' })
], CDSCheckbox.prototype, "helperText", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'hide-checkbox' })
], CDSCheckbox.prototype, "hideCheckbox", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'hide-label' })
], CDSCheckbox.prototype, "hideLabel", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCheckbox.prototype, "indeterminate", void 0);
__decorate([
    property({ attribute: 'label-text' })
], CDSCheckbox.prototype, "labelText", void 0);
__decorate([
    property()
], CDSCheckbox.prototype, "name", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCheckbox.prototype, "readonly", void 0);
__decorate([
    property({ type: Boolean })
], CDSCheckbox.prototype, "invalid", void 0);
__decorate([
    property({ type: String, attribute: 'invalid-text' })
], CDSCheckbox.prototype, "invalidText", void 0);
__decorate([
    property({ attribute: 'title' })
], CDSCheckbox.prototype, "title", void 0);
__decorate([
    property()
], CDSCheckbox.prototype, "value", void 0);
__decorate([
    property({ type: Boolean })
], CDSCheckbox.prototype, "warn", void 0);
__decorate([
    property({ type: String, attribute: 'warn-text' })
], CDSCheckbox.prototype, "warnText", void 0);
CDSCheckbox = __decorate([
    carbonElement(`${prefix}-checkbox`)
], CDSCheckbox);
var CDSCheckbox$1 = CDSCheckbox;

export { CDSCheckbox$1 as default };
//# sourceMappingURL=checkbox.js.map
