/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html } from 'lit';
import { query, property } from 'lit/decorators.js';
import Close16 from '../../icons/close/16.js';
import { prefix } from '../../globals/settings.js';
import { findIndex, forEach } from '../../globals/internal/collection-helpers.js';
import CDSDropdown from '../dropdown/dropdown.js';
import styles from './combo-box.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import ifNonEmpty from '../../globals/directives/if-non-empty.js';
import { DROPDOWN_KEYBOARD_ACTION } from '../dropdown/defs.js';
export { DROPDOWN_DIRECTION, DROPDOWN_SIZE } from '../dropdown/defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Combo box.
 *
 * @element cds-combo-box
 * @fires cds-combo-box-beingselected
 *   The custom event fired before a combo box item is selected upon a user gesture.
 *   Cancellation of this event stops changing the user-initiated selection.
 * @fires cds-combo-box-beingtoggled
 *   The custom event fired before the open state of this combo box is toggled upon a user gesture.
 *   Cancellation of this event stops the user-initiated toggling.
 * @fires cds-combo-box-selected - The custom event fired after a combo box item is selected upon a user gesture.
 * @fires cds-combo-box-toggled - The custom event fired after the open state of this combo box is toggled upon a user gesture.
 */
let CDSComboBox = class CDSComboBox extends CDSDropdown {
    constructor() {
        super(...arguments);
        /**
         * The text content that should be set to the `<input>` for filtering.
         */
        this._filterInputValue = '';
        this._shouldTriggerBeFocusable = false;
        /**
         * The `aria-label` attribute for the icon to clear selection.
         */
        this.clearSelectionLabel = 'Clear selection';
        /**
         * The `aria-label` attribute for the `<input>` for filtering.
         */
        this.inputLabel = '';
    }
    /**
     * @param item A combo box item.
     * @returns `true` if the given combo box item matches the query text user types.
     */
    _testItemWithQueryText(item) {
        return (this.itemMatches || this._defaultItemMatches)(item, this._filterInputNode.value);
    }
    /* eslint-disable class-methods-use-this */
    /**
     * The default item matching callback.
     *
     * @param item The combo box item.
     * @param queryText The query text user types.
     * @returns `true` if the given combo box item matches the given query text.
     */
    _defaultItemMatches(item, queryText) {
        return (item.textContent.toLowerCase().indexOf(queryText.toLowerCase()) >= 0);
    }
    /* eslint-enable class-methods-use-this */
    /**
     * Handles `input` event on the `<input>` for filtering.
     */
    _handleInput() {
        if (this._filterInputValue.length != 0) {
            this.setAttribute('isClosable', '');
        }
        else {
            this.removeAttribute('isClosable');
        }
        const items = this.querySelectorAll(this.constructor.selectorItem);
        const index = !this._filterInputNode.value
            ? -1
            : findIndex(items, this._testItemWithQueryText, this);
        forEach(items, (item, i) => {
            var _a;
            if (i === index) {
                const menuRect = (_a = this._itemMenu) === null || _a === void 0 ? void 0 : _a.getBoundingClientRect();
                const itemRect = item.getBoundingClientRect();
                if (menuRect && itemRect) {
                    const isViewable = menuRect.top <= (itemRect === null || itemRect === void 0 ? void 0 : itemRect.top) &&
                        (itemRect === null || itemRect === void 0 ? void 0 : itemRect.bottom) <= (menuRect === null || menuRect === void 0 ? void 0 : menuRect.top) + this._itemMenu.clientHeight;
                    if (!isViewable) {
                        const scrollTop = (itemRect === null || itemRect === void 0 ? void 0 : itemRect.top) - (menuRect === null || menuRect === void 0 ? void 0 : menuRect.top);
                        const scrollBot = (itemRect === null || itemRect === void 0 ? void 0 : itemRect.bottom) - (menuRect === null || menuRect === void 0 ? void 0 : menuRect.bottom);
                        if (Math.abs(scrollTop) < Math.abs(scrollBot)) {
                            this._itemMenu.scrollTop += scrollTop;
                        }
                        else {
                            this._itemMenu.scrollTop += scrollBot;
                        }
                    }
                }
            }
            item.highlighted = i === index;
        });
        const { _filterInputNode: filterInput } = this;
        this._filterInputValue = !filterInput ? '' : filterInput.value;
        this.open = true;
        this.requestUpdate(); // If the only change is to `_filterInputValue`, auto-update doesn't happen
    }
    _handleClickInner(event) {
        var _a;
        const { target } = event;
        if ((_a = this._selectionButtonNode) === null || _a === void 0 ? void 0 : _a.contains(target)) {
            this._handleUserInitiatedClearInput();
        }
        else {
            super._handleClickInner(event);
        }
    }
    _handleKeypressInner(event) {
        var _a;
        const { key } = event;
        const action = this.constructor.getAction(key);
        const { TRIGGERING } = DROPDOWN_KEYBOARD_ACTION;
        if (((_a = this._selectionButtonNode) === null || _a === void 0 ? void 0 : _a.contains(event.target)) &&
            // Space key should be handled by `<input>` unless "clear selection" button has focus
            (action === TRIGGERING || key === ' ')) {
            this._handleUserInitiatedClearInput();
        }
        else {
            super._handleKeypressInner(event);
        }
    }
    /**
     * Handles user-initiated clearing the `<input>` for filtering.
     */
    _handleUserInitiatedClearInput() {
        forEach(this.querySelectorAll(this.constructor.selectorItem), (item) => {
            item.highlighted = false;
        });
        this._filterInputValue = '';
        this._filterInputNode.focus();
        this._handleUserInitiatedSelectItem();
    }
    _handleUserInitiatedSelectItem(item) {
        if (item && !this._selectionShouldChange(item)) {
            // Escape hatch for `shouldUpdate()` logic that updates `._filterInputValue()` when selection changes,
            // given we want to update the `<input>` and close the dropdown even if selection doesn't update.
            // Use case:
            // 1. Select the 2nd item in combo box drop down
            // 2. Type some text in the `<input>`
            // 3. Re-select the 2nd item in combo box drop down,
            //    the `<input>` has to updated with the 2nd item and the dropdown should be closed,
            //    even if there is no change in the selected value
            this._filterInputValue = item.textContent || '';
            this.open = false;
            this.requestUpdate();
        }
        super._handleUserInitiatedSelectItem(item);
    }
    _selectionDidChange(itemToSelect) {
        this.value = !itemToSelect ? '' : itemToSelect.value;
        forEach(this.querySelectorAll(this.constructor.selectorItemSelected), (item) => {
            item.selected = false;
            item.setAttribute('aria-selected', 'false');
        });
        if (itemToSelect) {
            itemToSelect.selected = true;
            itemToSelect.setAttribute('aria-selected', 'true');
        }
        this._handleUserInitiatedToggle(false);
    }
    _renderLabel() {
        var _a;
        const { disabled, inputLabel, label, open, readOnly, value, _activeDescendant: activeDescendant, _filterInputValue: filterInputValue, _handleInput: handleInput, } = this;
        const inputClasses = classMap({
            [`${prefix}--text-input`]: true,
            [`${prefix}--text-input--empty`]: !value,
        });
        let activeDescendantFallback;
        if (open && !activeDescendant) {
            const constructor = this.constructor;
            const items = this.querySelectorAll(constructor.selectorItem);
            activeDescendantFallback = (_a = items[0]) === null || _a === void 0 ? void 0 : _a.id;
        }
        return html `
      <input
        id="trigger-button"
        class="${inputClasses}"
        ?disabled=${disabled}
        placeholder="${label}"
        .value=${filterInputValue}
        role="combobox"
        aria-label="${ifNonEmpty(inputLabel)}"
        aria-controls="menu-body"
        aria-haspopup="listbox"
        aria-autocomplete="list"
        aria-expanded="${String(open)}"
        aria-activedescendant="${ifDefined(open ? (activeDescendant !== null && activeDescendant !== void 0 ? activeDescendant : activeDescendantFallback) : '')}"
        ?readonly=${readOnly}
        @input=${handleInput} />
    `;
    }
    _renderFollowingLabel() {
        const { clearSelectionLabel, _filterInputValue: filterInputValue } = this;
        if (filterInputValue.length != 0) {
            this.setAttribute('isClosable', '');
        }
        else {
            this.removeAttribute('isClosable');
        }
        return filterInputValue.length === 0
            ? undefined
            : html `
          <div
            id="selection-button"
            role="button"
            class="${prefix}--list-box__selection"
            tabindex="0"
            title="${clearSelectionLabel}">
            ${Close16({ 'aria-label': clearSelectionLabel })}
          </div>
        `;
    }
    shouldUpdate(changedProperties) {
        super.shouldUpdate(changedProperties);
        const { _selectedItemContent: selectedItemContent } = this;
        if (selectedItemContent && changedProperties.has('value')) {
            this._filterInputValue = (selectedItemContent === null || selectedItemContent === void 0 ? void 0 : selectedItemContent.textContent) || '';
        }
        return true;
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const { _listBoxNode: listBoxNode } = this;
        if (listBoxNode) {
            listBoxNode.classList.add(`${prefix}--combo-box`);
        }
    }
    /**
     * A selector that will return highlighted items.
     */
    static get selectorItemHighlighted() {
        return `${prefix}-combo-box-item[highlighted]`;
    }
    /**
     * A selector that will return combo box items.
     */
    static get selectorItem() {
        return `${prefix}-combo-box-item`;
    }
    /**
     * A selector that will return selected items.
     */
    static get selectorItemSelected() {
        return `${prefix}-combo-box-item[selected]`;
    }
    /**
     * The name of the custom event fired before this combo box item is being toggled upon a user gesture.
     * Cancellation of this event stops the user-initiated action of toggling this combo box item.
     */
    static get eventBeforeToggle() {
        return `${prefix}-combo-box-beingtoggled`;
    }
    /**
     * The name of the custom event fired after this combo box item is toggled upon a user gesture.
     */
    static get eventToggle() {
        return `${prefix}-combo-box-toggled`;
    }
    /**
     * The name of the custom event fired before a combo box item is selected upon a user gesture.
     * Cancellation of this event stops changing the user-initiated selection.
     */
    static get eventBeforeSelect() {
        return `${prefix}-combo-box-beingselected`;
    }
    /**
     * The name of the custom event fired after a a combo box item is selected upon a user gesture.
     */
    static get eventSelect() {
        return `${prefix}-combo-box-selected`;
    }
};
// For combo box, open/selection with space key is disabled given the input box should take it over
CDSComboBox.TRIGGER_KEYS = new Set(['Enter']);
CDSComboBox.styles = styles;
__decorate([
    query('input')
], CDSComboBox.prototype, "_filterInputNode", void 0);
__decorate([
    query('#menu-body')
], CDSComboBox.prototype, "_itemMenu", void 0);
__decorate([
    query('#selection-button')
], CDSComboBox.prototype, "_selectionButtonNode", void 0);
__decorate([
    property({ attribute: 'clear-selection-label' })
], CDSComboBox.prototype, "clearSelectionLabel", void 0);
__decorate([
    property({ attribute: 'input-label' })
], CDSComboBox.prototype, "inputLabel", void 0);
__decorate([
    property({ attribute: false })
], CDSComboBox.prototype, "itemMatches", void 0);
CDSComboBox = __decorate([
    carbonElement(`${prefix}-combo-box`)
], CDSComboBox);
var CDSComboBox$1 = CDSComboBox;

export { CDSComboBox$1 as default };
//# sourceMappingURL=combo-box.js.map
