/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import styles from './content-switcher.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Content switcher button.
 *
 * @element cds-content-switcher-item
 */
let CDSContentSwitcherItem = class CDSContentSwitcherItem extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if this content switcher item should be disabled.
         */
        this.disabled = false;
        /**
         * `true` to hide the divider at the left.
         *
         * @private
         */
        this.hideDivider = false;
        /**
         * `true` if the content switcher button should be selected.
         *
         * @private
         */
        this.selected = false;
        /**
         * The `value` attribute that is set to the parent `<cds-content-switcher>`
         * when this content switcher item is selected.
         */
        this.value = '';
        /**
         * `true` if the content switcher button should be icon-only.
         */
        this.icon = false;
        /**
         * Specify how the trigger should align with the tooltip for icon-only
         * switcher item
         */
        this.align = 'top';
        /**
         * Determines whether the tooltip should close when inner content is
         * activated (click, Enter or Space)
         */
        this.closeOnActivation = true;
        /**
         * Specify the duration in milliseconds to delay before displaying the
         * tooltip for icon-only switcher item
         */
        this.enterDelayMs = 100;
        /**
         * Specify the duration in milliseconds to delay before hiding the tooltip
         * for icon-only switcher-item
         *
         * TODO: match upstream value once #10471 is resolved
         */
        this.leaveDelayMs = 100;
    }
    updated(changedProperties) {
        var _a, _b, _c, _d;
        if (changedProperties) {
            (_d = (_c = (_b = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(`${prefix}-tooltip`)) === null || _b === void 0 ? void 0 : _b.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector(`.${prefix}--tooltip`)) === null || _d === void 0 ? void 0 : _d.classList.add(`${prefix}--icon-tooltip`);
        }
    }
    // eslint-disable-next-line class-methods-use-this
    _renderTooltipContent() {
        return html `
      <cds-tooltip-content>
        <slot name="tooltip-content"></slot>
      </cds-tooltip-content>
    `;
    }
    shouldUpdate(changedProperties) {
        if (changedProperties.has('selected') || changedProperties.has('target')) {
            const { selected, target } = this;
            if (target) {
                const doc = this.getRootNode();
                // `doc` can be an element if such element is orphaned
                const targetNode = doc === null || doc === void 0 ? void 0 : doc.getElementById(target);
                targetNode === null || targetNode === void 0 ? void 0 : targetNode.toggleAttribute('hidden', !selected);
            }
        }
        return true;
    }
    render() {
        const { disabled, selected, target } = this;
        const className = classMap({
            [`${prefix}--content-switcher-btn`]: true,
            [`${prefix}--content-switcher--selected`]: selected,
        });
        const switcherItem = html `<button
      type="button"
      role="tab"
      class="${className}"
      ?disabled="${disabled}"
      tabindex="${selected ? '0' : '-1'}"
      aria-controls="${ifDefined(target)}"
      aria-selected="${Boolean(selected)}">
      <span class="${prefix}--content-switcher__label"><slot></slot></span>
    </button>`;
        if (this.icon) {
            const { align, closeOnActivation, enterDelayMs, leaveDelayMs } = this;
            return html `<cds-tooltip
        align=${align}
        close-on-activation="${closeOnActivation}"
        enter-delay-ms=${enterDelayMs}
        leave-delay-ms=${leaveDelayMs}>
        ${switcherItem} ${this._renderTooltipContent()}
      </cds-tooltip>`;
        }
        return switcherItem;
    }
};
CDSContentSwitcherItem.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSContentSwitcherItem.styles = styles;
__decorate([
    property({ type: Boolean, reflect: true })
], CDSContentSwitcherItem.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'hide-divider' })
], CDSContentSwitcherItem.prototype, "hideDivider", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSContentSwitcherItem.prototype, "selected", void 0);
__decorate([
    property()
], CDSContentSwitcherItem.prototype, "target", void 0);
__decorate([
    property()
], CDSContentSwitcherItem.prototype, "value", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSContentSwitcherItem.prototype, "icon", void 0);
__decorate([
    property({ reflect: true, type: String })
], CDSContentSwitcherItem.prototype, "align", void 0);
__decorate([
    property({ attribute: 'close-on-activation', reflect: true, type: Boolean })
], CDSContentSwitcherItem.prototype, "closeOnActivation", void 0);
CDSContentSwitcherItem = __decorate([
    carbonElement(`${prefix}-content-switcher-item`)
], CDSContentSwitcherItem);
var CDSContentSwitcherItem$1 = CDSContentSwitcherItem;

export { CDSContentSwitcherItem$1 as default };
//# sourceMappingURL=content-switcher-item.js.map
