/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import styles from './data-table.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Table batch actions.
 *
 * @element cds-table-batch-actions
 * @fires cds-table-batch-actions-cancel-clicked - The custom event fired after the Cancel button is clicked.
 */
let CDSTableBatchActions = class CDSTableBatchActions extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * `true` if this batch actions bar should be active.
         */
        this.active = false;
        /**
         * The formatter for selected items. Should be changed upon the locale the UI is rendered with.
         */
        this.formatSelectedItemsCount = ({ count }) => `${count} item${count <= 1 ? '' : 's'} selected`;
        /**
         * Numeric representation of the total number of items selected in a table.
         * This number is used to derive the selection message.
         */
        this.selectedRowsCount = 0;
    }
    /**
     * Handles `click` event on the Cancel button.
     */
    _handleCancel() {
        const { eventClickCancel } = this
            .constructor;
        this.dispatchEvent(new CustomEvent(eventClickCancel, { bubbles: true, composed: true }));
    }
    firstUpdated() {
        this.querySelectorAll(this.constructor.selectorButtons).forEach((e) => {
            e.setAttribute('batch-action', '');
        });
    }
    updated(changedProperties) {
        if (changedProperties.has('active')) {
            this.setAttribute('tabindex', `${this.active ? '' : '-1'}`);
        }
    }
    render() {
        const { formatSelectedItemsCount, selectedRowsCount, _handleCancel: handleCancel, } = this;
        return html `
      <div class="${prefix}--batch-summary">
        <p class="${prefix}--batch-summary__para">
          ${formatSelectedItemsCount({ count: selectedRowsCount })}
        </p>
      </div>
      <div class="${prefix}--action-list">
        <slot></slot>
        <button
          class="${prefix}--btn ${prefix}--btn--primary ${prefix}--batch-summary__cancel"
          @click=${handleCancel}>
          <slot name="cancel-button-content">Cancel</slot>
        </button>
      </div>
    `;
    }
    /**
     * The CSS selector to find the action buttons.
     */
    static get selectorButtons() {
        return `${prefix}-button`;
    }
    /**
     * The name of the custom event fired after the Cancel button is clicked.
     */
    static get eventClickCancel() {
        return `${prefix}-table-batch-actions-cancel-clicked`;
    }
};
CDSTableBatchActions.styles = styles;
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableBatchActions.prototype, "active", void 0);
__decorate([
    property({ attribute: false })
], CDSTableBatchActions.prototype, "formatSelectedItemsCount", void 0);
__decorate([
    property({ type: Number, attribute: 'selected-rows-count' })
], CDSTableBatchActions.prototype, "selectedRowsCount", void 0);
CDSTableBatchActions = __decorate([
    carbonElement(`${prefix}-table-batch-actions`)
], CDSTableBatchActions);
var CDSTableBatchActions$1 = CDSTableBatchActions;

export { CDSTableBatchActions$1 as default };
//# sourceMappingURL=table-batch-actions.js.map
