/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import ChevronRight16 from '../../icons/chevron--right/16.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import FocusMixin from '../../globals/mixins/focus.js';
import styles from './data-table.scss.js';
import '../checkbox/checkbox.js';
import '../checkbox/checkbox-group.js';
import '../checkbox/checkbox-skeleton.js';
import '../radio-button/radio-button.js';
import '../radio-button/radio-button-group.js';
import '../radio-button/radio-button-skeleton.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Data table row.
 *
 * @element cds-table-row
 * @csspart selection-container The container of the checkbox.
 * @csspart selection The checkbox.
 * @fires cds-table-row-change-selection
 *   The custom event fired before this row is selected/unselected upon a user gesture.
 *   Cancellation of this event stops the user-initiated change in selection.
 * @fires cds-radio-button-changed
 *   The name of the custom event fired after this radio button changes its checked state.
 * @fires cds-checkbox-changed
 *   The name of the custom event fired after this checkbox changes its checked state.
 * @fires cds-table-row-expando-beingtoggled
 *   The name of the custom event fired before the expanded state of this row is being toggled upon a user gesture.
 *   Cancellation of this event stops the user-initiated action of toggling the expanded state.
 * @fires cds-table-row-expando-toggled
 *   The name of the custom event fired after the expanded state of this row is toggled upon a user gesture.
 */
let CDSTableRow = class CDSTableRow extends HostListenerMixin(FocusMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * `true` if this table should support batch expansion
         */
        this.batchExpansion = false;
        /**
         * `true` if this table row should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if this table row is placed at an even position in parent `<cds-table-body>`.
         * `<cds-table-body>` sets this property, _only_ in zebra stripe mode.
         *
         * @private
         */
        this.even = false;
        /**
         * `true` if this table row can be expanded to show content underneath
         *
         * @private
         */
        this.expandable = false;
        /**
         * `true` when the table row expanded is showing
         *
         * @private
         */
        this.expanded = false;
        /**
         * `true` if this table row should be filtered out.
         */
        this.filtered = false;
        /**
         * Specify whether the checkbox should be present in the DOM,
         * but invisible and uninteractable.
         */
        this.hideCheckbox = false;
        /**
         * `true` if the table row should be highlighted.
         */
        this.highlighted = false;
        /**
         * `true` if this table row is placed at an odd position in parent `<cds-table-body>`.
         * `<cds-table-body>` sets this property, _only_ in zebra stripe mode.
         *
         * @private
         */
        this.odd = false;
        /**
         * Specify whether the overflow menu (if it exists) should be shown always, or only on hover
         */
        this.overflowMenuOnHover = false;
        /**
         * Specify whether the control should be a radio button or inline checkbox
         *
         * @private
         */
        this.radio = false;
        /**
         * `true` if this table row should be selected.
         */
        this.selected = false;
        /**
         * The `aria-label` attribute for the `<label>` for selection.
         */
        this.selectionLabel = '';
        /**
         * The `name` attribute for the `<input>` for selection.
         * If present, this table row will be a selectable one.
         */
        this.selectionName = '';
        /**
         * The `value` attribute for the `<input>` for selection.
         */
        this.selectionValue = '';
    }
    /**
     * Handles `click` event on the radio button.
     *
     * @param event The event.
     */
    _handleClickSelectionRadio(event) {
        var _a;
        const { detail } = event;
        const selected = detail.checked;
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                selected,
            },
        };
        const constructor = this.constructor;
        if (this.dispatchEvent(new CustomEvent(constructor.eventBeforeChangeSelection, init))) {
            this.selected = selected;
            const { selectorExpandedRow } = this.constructor;
            if ((_a = this.nextElementSibling) === null || _a === void 0 ? void 0 : _a.matches(selectorExpandedRow)) {
                this.nextElementSibling.selected = selected;
            }
        }
    }
    /**
     * Handles `click` event on the check box.
     *
     * @param event The event.
     */
    _handleClickSelectionCheckbox(event) {
        var _a;
        const { detail } = event;
        const selected = detail.checked;
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                selected,
            },
        };
        const constructor = this.constructor;
        if (this.dispatchEvent(new CustomEvent(constructor.eventBeforeChangeSelection, init))) {
            this.selected = selected;
            const { selectorExpandedRow } = this.constructor;
            if ((_a = this.nextElementSibling) === null || _a === void 0 ? void 0 : _a.matches(selectorExpandedRow)) {
                this.nextElementSibling.selected = selected;
            }
        }
    }
    /**
     * Handles `click` event on the expando button.
     */
    _handleClickExpando() {
        this._handleUserInitiatedToggleExpando();
    }
    /**
     * Handles `mouseover`/`mouseout` event handler on this element.
     *
     * @param event The event.
     */
    _handleMouseOverOut(event) {
        const { selectorExpandedRow, selectorTableCellOverflowMenu } = this
            .constructor;
        const { nextElementSibling } = this;
        if (nextElementSibling === null || nextElementSibling === void 0 ? void 0 : nextElementSibling.matches(selectorExpandedRow)) {
            nextElementSibling.highlighted =
                event.type === 'mouseover';
        }
        if (this.overflowMenuOnHover) {
            const overflowMenu = this.querySelector(selectorTableCellOverflowMenu);
            const parentCell = overflowMenu === null || overflowMenu === void 0 ? void 0 : overflowMenu.parentElement;
            if (event.type === 'mouseout') {
                parentCell.overflowMenuOnHover = true;
            }
            else {
                parentCell.overflowMenuOnHover = false;
            }
        }
    }
    /**
     * Handles user-initiated toggle request of the expando button in this table row.
     *
     * @param expanded The new expanded state.
     */
    _handleUserInitiatedToggleExpando(expanded = !this.expanded) {
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                expanded,
            },
        };
        if (this.dispatchEvent(new CustomEvent(this.constructor.eventBeforeExpandoToggle, init))) {
            this.expanded = expanded;
            this.dispatchEvent(new CustomEvent(this.constructor.eventExpandoToggle, init));
        }
    }
    _renderExpandButton() {
        const { _handleClickExpando: handleClickExpando } = this;
        return html `
      <div class="${prefix}--table-expand">
        <div>
          <slot name="ai-label" @slotchange="${this._handleSlotChange}"></slot>
          <slot name="slug" @slotchange="${this._handleSlotChange}"></slot>
          <button
            class="${prefix}--table-expand__button"
            @click="${handleClickExpando}">
            ${ChevronRight16({ class: `${prefix}--table-expand__svg` })}
          </button>
        </div>
      </div>
    `;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                elem.matches(this.constructor.slugItem)
            : false);
        if (hasContent.length > 0) {
            this._hasAILabel = Boolean(hasContent);
            hasContent[0].setAttribute('size', 'mini');
        }
        this.requestUpdate();
    }
    /**
     * @returns The first set of table cells.
     */
    _renderFirstCells() {
        const { disabled, hideCheckbox, radio, selected, selectionLabel, selectionName, selectionValue, } = this;
        return !selectionName
            ? undefined
            : html `
          <div class="${prefix}--table-column-checkbox">
            <div>
              <slot
                name="ai-label"
                @slotchange="${this._handleSlotChange}"></slot>
              <slot name="slug" @slotchange="${this._handleSlotChange}"></slot>
              ${radio
                ? html `<cds-radio-button data-table></cds-radio-button>`
                : html `<cds-checkbox
                    hide-label
                    ?hide-checkbox="${hideCheckbox}"
                    label-text="${selectionLabel}"
                    name=${selectionName}
                    data-table
                    ?disabled=${disabled}
                    ?checked=${selected}
                    value=${selectionValue}></cds-checkbox> `}
            </div>
          </div>
        `;
    }
    /**
     * TODO: Uncomment when Carbon fully implements sticky header
     * Specify whether the header should be sticky.
     * Still experimental: may not work with every combination of table props
     */
    // @property({ type: Boolean, reflect: true, attribute: 'sticky-header' })
    // stickyHeader = false;
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'row');
        }
        super.connectedCallback();
    }
    updated(changedProperties) {
        if (changedProperties.has('expanded')) {
            const { selectorExpandedRow } = this.constructor;
            const { expanded, nextElementSibling } = this;
            if (nextElementSibling === null || nextElementSibling === void 0 ? void 0 : nextElementSibling.matches(selectorExpandedRow)) {
                nextElementSibling.expanded = expanded;
            }
        }
        if (changedProperties.has('highlighted')) {
            const { selectorExpandedRow } = this.constructor;
            const { highlighted, nextElementSibling } = this;
            if (nextElementSibling === null || nextElementSibling === void 0 ? void 0 : nextElementSibling.matches(selectorExpandedRow)) {
                nextElementSibling.highlighted = highlighted;
            }
        }
        if (this._hasAILabel) {
            this.setAttribute('ai-label', '');
        }
        else {
            this.removeAttribute('ai-label');
        }
    }
    render() {
        var _a;
        if (this.selectionName) {
            (_a = this.closest(this.constructor.selectorTable)) === null || _a === void 0 ? void 0 : _a.setAttribute('is-selectable', '');
        }
        return html `
      ${this.expandable ? this._renderExpandButton() : ''}
      ${this._renderFirstCells()}
      <slot></slot>
    `;
    }
    /**
     * The name of the custom event fired after this radio button changes its checked state.
     */
    static get eventRadioChange() {
        return `${prefix}-radio-button-changed`;
    }
    /**
     * The name of the custom event fired after this radio button changes its checked state.
     */
    static get eventCheckboxChange() {
        return `${prefix}-checkbox-changed`;
    }
    /**
     * The name of the custom event fired before this row is selected/unselected upon a user gesture.
     * Cancellation of this event stops the user-initiated change in selection.
     */
    static get eventBeforeChangeSelection() {
        return `${prefix}-table-row-change-selection`;
    }
    /**
     * A selector that will return the parent table
     */
    static get selectorTable() {
        return `${prefix}-table`;
    }
    /**
     * The CSS selector to find the overflow menu on the table cell
     */
    static get selectorTableCellOverflowMenu() {
        return `${prefix}-table-cell ${prefix}-overflow-menu`;
    }
    /**
     * A selector that will return the corresponding expanded row.
     */
    static get selectorExpandedRow() {
        return `${prefix}-table-expanded-row`;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
    /**
     * The name of the custom event fired before the expanded state this row is being toggled upon a user gesture.
     * Cancellation of this event stops the user-initiated action of toggling the expanded state.
     */
    static get eventBeforeExpandoToggle() {
        return `${prefix}-table-row-expando-beingtoggled`;
    }
    /**
     * The name of the custom event fired after the expanded state this row is toggled upon a user gesture.
     */
    static get eventExpandoToggle() {
        return `${prefix}-table-row-expando-toggled`;
    }
};
CDSTableRow.styles = styles;
__decorate([
    HostListener('eventRadioChange')
    // @ts-ignore
], CDSTableRow.prototype, "_handleClickSelectionRadio", null);
__decorate([
    HostListener('eventCheckboxChange')
    // @ts-ignore
], CDSTableRow.prototype, "_handleClickSelectionCheckbox", null);
__decorate([
    HostListener('mouseover'),
    HostListener('mouseout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTableRow.prototype, "_handleMouseOverOut", null);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'batch-expansion' })
], CDSTableRow.prototype, "batchExpansion", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "even", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "expandable", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "expanded", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "filtered", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'hide-checkbox' })
], CDSTableRow.prototype, "hideCheckbox", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "highlighted", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "odd", void 0);
__decorate([
    property({
        type: Boolean,
        reflect: true,
        attribute: 'overflow-menu-on-hover',
    })
], CDSTableRow.prototype, "overflowMenuOnHover", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "radio", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableRow.prototype, "selected", void 0);
__decorate([
    property({ attribute: 'selection-label' })
], CDSTableRow.prototype, "selectionLabel", void 0);
__decorate([
    property({ attribute: 'selection-name' })
], CDSTableRow.prototype, "selectionName", void 0);
__decorate([
    property({ attribute: 'selection-value' })
], CDSTableRow.prototype, "selectionValue", void 0);
CDSTableRow = __decorate([
    carbonElement(`${prefix}-table-row`)
], CDSTableRow);
var CDSTableRow$1 = CDSTableRow;

export { CDSTableRow$1 as default };
//# sourceMappingURL=table-row.js.map
