/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { prefix } from '../../globals/settings.js';
import styles from './data-table.scss.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Data table skeleton
 *
 * @element cds-table-skeleton
 */
let CDSTableSkeleton = class CDSTableSkeleton extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * Optionally specify the displayed headers
         */
        this.headers = [];
        /**
         * Optionally specify whether you want the Skeleton to be rendered as a compact DataTable
         */
        this.compact = false;
        /**
         * Specify the number of columns that you want to render in the skeleton state
         */
        this.columnCount = 5;
        /**
         * Specify the number of rows that you want to render in the skeleton state
         */
        this.rowCount = 5;
        /**
         * Specify if the table header should be rendered as part of the skeleton.
         */
        this.showHeader = true;
        /**
         * Specify if the table toolbar should be rendered as part of the skeleton.
         */
        this.showToolbar = true;
        /**
         *  true to add useZebraStyles striping.
         */
        this.zebra = false;
    }
    /**
     * @returns The header
     */
    _renderHeader() {
        const { showHeader } = this;
        return !showHeader
            ? undefined
            : html `
          <div class="${prefix}--skeleton ${prefix}--data-table-container">
            <div class="${prefix}--data-table-header">
              <div class="${prefix}--data-table-header__title"></div>
              <div class="${prefix}--data-table-header__description"></div>
            </div>
          </div>
        `;
    }
    /**
     * @returns The header
     */
    _renderToolbar() {
        const { showToolbar } = this;
        return !showToolbar
            ? undefined
            : html `
          <section class="${prefix}--table-toolbar">
            <div class="${prefix}--toolbar-content">
              <span
                class="${prefix}--skeleton ${prefix}--btn ${prefix}--btn--sm"></span>
            </div>
          </section>
        `;
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'table');
        }
        super.connectedCallback();
    }
    updated() {
        if (this.headers.length) {
            this.columnCount = this.headers.length;
        }
        else {
            this.headers = Array(this.columnCount).fill('');
        }
    }
    render() {
        const { compact, columnCount, headers, rowCount, zebra } = this;
        const classes = classMap({
            [`${prefix}--skeleton`]: true,
            [`${prefix}--data-table`]: true,
            [`${prefix}--data-table--compact`]: compact,
            [`${prefix}--data-table--zebra`]: zebra,
        });
        return html `
      ${this._renderHeader()} ${this._renderToolbar()}

      <table class="${classes}">
        <thead>
          <tr>
            ${Array.from(new Array(columnCount)).map((_, index) => html `
                <th>
                  <div class="${prefix}--table-header-label">
                    ${headers[index]}
                  </div>
                </th>
              `)}
          </tr>
        </thead>
        <tbody>
          ${Array.from(new Array(rowCount)).map(() => html `
              <tr>
                ${Array.from(new Array(columnCount)).map(() => html `
                    <td>
                      <span></span>
                    </td>
                  `)}
              </tr>
            `)}
        </tbody>
      </table>
    `;
    }
};
CDSTableSkeleton.styles = styles;
__decorate([
    property()
], CDSTableSkeleton.prototype, "headers", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableSkeleton.prototype, "compact", void 0);
__decorate([
    property({ type: Number, reflect: true, attribute: 'column-count' })
], CDSTableSkeleton.prototype, "columnCount", void 0);
__decorate([
    property({ type: Number, reflect: true, attribute: 'row-count' })
], CDSTableSkeleton.prototype, "rowCount", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'show-header' })
], CDSTableSkeleton.prototype, "showHeader", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'show-toolbar' })
], CDSTableSkeleton.prototype, "showToolbar", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableSkeleton.prototype, "zebra", void 0);
CDSTableSkeleton = __decorate([
    carbonElement(`${prefix}-table-skeleton`)
], CDSTableSkeleton);
var CDSTableSkeleton$1 = CDSTableSkeleton;

export { CDSTableSkeleton$1 as default };
//# sourceMappingURL=table-skeleton.js.map
