/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import '../text-input/text-input.js';
import CDSSearch from '../search/search.js';
import styles from './data-table.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { INPUT_SIZE } from '../text-input/defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Table toolbar search.
 *
 * @element cds-table-toolbar-search
 * @fires cds-search-input - The custom event fired after the search content is changed upon a user gesture.
 */
let CDSTableToolbarSearch = class CDSTableToolbarSearch extends HostListenerMixin(CDSSearch) {
    constructor() {
        super(...arguments);
        /**
         * `true` if the search box should be expanded.
         */
        this.expanded = false;
        /**
         * `true` if the search box should be always be open.
         */
        this.persistent = false;
        /**
         * The search box size.
         */
        this.size = INPUT_SIZE.LARGE;
    }
    /**
     * Handles user-initiated gestures for expanding the search box.
     */
    async _handleUserInitiatedExpand() {
        this.expanded = true;
        await this.updateComplete;
        const { _inputNode: inputNode } = this;
        inputNode === null || inputNode === void 0 ? void 0 : inputNode.focus();
    }
    /**
     * Handles `focus` event handler on this element.
     */
    _handleFocusIn() {
        this._handleUserInitiatedExpand();
    }
    /**
     * Handles `blur` event handler on this element.
     *
     * @param event The event.
     */
    _handleFocusOut(event) {
        if (!this.contains(event.relatedTarget) &&
            !this.value &&
            !this.persistent) {
            this.expanded = false;
        }
    }
    /**
     * Handles `click` event handler on the search box.
     */
    _handleSearchClick() {
        this._handleUserInitiatedExpand();
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'search');
        }
        super.connectedCallback();
    }
    render() {
        const result = super.render();
        const { persistent, expanded, size, _handleSearchClick: handleSearchClick, } = this;
        const classes = classMap({
            [`${prefix}--search`]: true,
            [`${prefix}--search--${size}`]: size,
        });
        if (persistent) {
            this.expanded = true;
        }
        return html `
      <div
        class="${classes}"
        tabindex="${expanded ? '-1' : '0'}"
        @click="${handleSearchClick}">
        ${result}
      </div>
    `;
    }
    /**
     * The name of the custom event fired after the search content is changed upon a user gesture.
     */
    static get eventInput() {
        // The code uses on in `<cds-search>`, but definition is done also here for React event generation
        return `${prefix}-search-input`;
    }
};
CDSTableToolbarSearch.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSTableToolbarSearch.styles = styles;
__decorate([
    query('input')
], CDSTableToolbarSearch.prototype, "_inputNode", void 0);
__decorate([
    HostListener('focusin')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTableToolbarSearch.prototype, "_handleFocusIn", null);
__decorate([
    HostListener('focusout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTableToolbarSearch.prototype, "_handleFocusOut", null);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableToolbarSearch.prototype, "expanded", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTableToolbarSearch.prototype, "persistent", void 0);
__decorate([
    property({ reflect: true })
], CDSTableToolbarSearch.prototype, "size", void 0);
CDSTableToolbarSearch = __decorate([
    carbonElement(`${prefix}-table-toolbar-search`)
], CDSTableToolbarSearch);
var CDSTableToolbarSearch$1 = CDSTableToolbarSearch;

export { CDSTableToolbarSearch$1 as default };
//# sourceMappingURL=table-toolbar-search.js.map
