/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { state, property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import { forEach } from '../../globals/internal/collection-helpers.js';
import { TABLE_SORT_DIRECTION, TABLE_SIZE } from './defs.js';
export { TABLE_SIZE } from './defs.js';
import styles from './data-table.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Data table.
 *
 * @element cds-table
 * @fires cds-table-header-cell-sort
 *   The name of the custom event fired before a new sort direction is set upon a user gesture.
 *   Cancellation of this event stops the user-initiated change in sort direction.
 * @fires cds-search input
 *   The name of the custom event fired during search bar input
 * @fires cds-table-change-selection-all
 *   The name of the custom event fired before header row is selected/unselected upon a user gesture.
 * @fires cds-table-row-change-selection
 *   The name of the custom event fired before a row is selected/unselected upon a user gesture.
 * @fires cds-table-batch-actions-cancel-clicked
 *   The name of the custom event fired after the Cancel button is clicked.
 * @fires cds-table-row-expando-toggled
 *   The name of the custom event fired after the expanded state of a row is toggled upon a user gesture.
 * @fires cds-table-row-selected
 *   The name of the custom event fired after a row has been selected.
 * @fires cds-table-row-all-selected
 *   The name of the custom event fired after all rows have been selected.
 * @fires cds-table-sorted
 *   The name of the custom event fired after the table has been sorted.
 * @fires cds-table-filtered
 *   The name of the custom event fired after the table has been filtered containing remaining rows.
 */
let CDSTable = class CDSTable extends HostListenerMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * The map of how sorting direction affects sorting order.
         */
        this.collationFactors = {
            [TABLE_SORT_DIRECTION.ASCENDING]: 1,
            [TABLE_SORT_DIRECTION.DESCENDING]: -1,
        };
        /**
         * Current search value for filtering
         */
        this._searchValue = '';
        this._selectedRows = [];
        /**
         * `true` if this table should support batch expansion
         */
        this.batchExpansion = false;
        /**
         * Specify whether the rows should be able to be expandable
         */
        this.expandable = false;
        /**
         * The method used when filtering the table with the search bar.
         * Can be replaced with custom method.
         *
         * @param rowText A table row.
         * @param searchString A search string.
         * @returns `false` if the given table row matches the given search string.
         */
        this.filterRows = (rowText, searchString) => rowText.toLowerCase().indexOf(searchString.toLowerCase()) < 0;
        /**
         * The total headers
         */
        this.headerCount = 0;
        /**
         * `true` if this table contains selectable rows
         */
        this.isSelectable = false;
        /**
         * `true` if this table should support sorting.
         */
        this.isSortable = false;
        /**
         * The table size.
         */
        this.locale = 'en';
        /**
         * Specify whether the overflow menu (if it exists) should be shown always, or only on hover
         */
        this.overflowMenuOnHover = false;
        /**
         * Specify whether the control should be a radio button or inline checkbox
         */
        this.radio = false;
        /**
         * The table size.
         */
        this.size = TABLE_SIZE.LG;
        /**
         * TODO: Uncomment when Carbon fully implements sticky header
         * Specify whether the header should be sticky.
         * Still experimental: may not work with every combination of table props
         */
        // @property({ type: Boolean, attribute: 'sticky-header', reflect: true })
        // stickyHeader = false;
        /**
         *  If true, will use a width of 'auto' instead of 100%
         */
        this.useStaticWidth = false;
        /**
         *  true to add useZebraStyles striping.
         */
        this.useZebraStyles = false;
        /**
         *  true if AI Labels are added in the rows
         */
        this.withRowAILabels = false;
        /**
         *  true if slugs are added in the rows
         *
         * @deprecated remove in v12, use `with-row-ai-labels` instead
         */
        this.withRowSlugs = false;
        /**
         * Handles batch expansion
         */
        this._handleBatchExpansion = async (event) => {
            const { detail, target } = event;
            const { expanded } = detail;
            if (target === this._tableHeaderRow) {
                this._tableRows.forEach((e) => (e.expanded = expanded));
            }
        };
        /**
         * Handles sorting the table depending on the column selected
         */
        this._handleSort = async (event) => {
            const { detail, target } = event;
            const { sortDirection } = detail;
            if (!this.contains(target)) {
                return;
            }
            const columns = [...this._tableHeaderRow.children];
            const columnIndex = columns.indexOf(target);
            columns.forEach((e) => {
                if (e !== target && this.isSortable) {
                    e.setAttribute('sort-direction', 'none');
                }
                else if (e.hasAttribute('is-sortable')) {
                    e.setAttribute('sort-direction', 'none');
                }
            });
            this._handleSortAction(columnIndex, sortDirection);
            const init = {
                bubbles: true,
                cancelable: true,
                composed: true,
                detail: {
                    sortedHeader: columns[columnIndex],
                },
            };
            this.dispatchEvent(new CustomEvent(this.constructor.eventTableSorted, init));
            this._handleFilterRows();
        };
        /**
         * Handles search input within the toolbar actions
         */
        this._handleSearchInput = async (event) => {
            const { detail, target } = event;
            if (this.contains(target)) {
                const { value } = detail;
                this._searchValue = value;
                this._handleFilterRows();
            }
        };
        /**
         * Handles row selection
         */
        this._handleRowSelect = async (event) => {
            var _a, _b;
            const { detail, target } = event;
            const { selected } = detail;
            const { _tableBatchActions: tableBatchActions, _tableToolbarContent: tableToolbarContent, _tableHeaderRow: tableHeaderRow, _selectedRows: selectedRows, } = this;
            if (!this.contains(target)) {
                return;
            }
            if (this.radio) {
                this._tableRows.forEach((e) => {
                    if (e !== target) {
                        e.removeAttribute('selected');
                        e.shadowRoot.querySelector(`${prefix}-radio-button`).checked = false;
                    }
                });
                this._selectedRows.push(...[target]);
            }
            else {
                if (selectedRows.includes(target)) {
                    this._selectedRows = selectedRows.filter((e) => e !== target);
                }
                else {
                    selectedRows.push(target);
                }
                if (tableBatchActions) {
                    tableBatchActions.active = (_a = this._selectedRows) === null || _a === void 0 ? void 0 : _a.length;
                    tableBatchActions.selectedRowsCount += selected ? 1 : -1;
                }
                if (tableToolbarContent) {
                    tableToolbarContent.hasBatchActions = this._selectedRows.length;
                }
            }
            const totalRows = [...this._tableRows].filter((elem) => !elem.hasAttribute('filtered')).length;
            // selected header checkbox upon all rows being selected
            const headerCheckbox = (_b = tableHeaderRow.shadowRoot) === null || _b === void 0 ? void 0 : _b.querySelector(`${prefix}-checkbox`).shadowRoot.querySelector(`.${prefix}--checkbox`);
            const allRowsSelected = this._selectedRows.length === totalRows;
            headerCheckbox.checked = !this._selectedRows.length ? false : true;
            headerCheckbox.indeterminate =
                !allRowsSelected && this._selectedRows.length > 0;
            const init = {
                bubbles: true,
                cancelable: true,
                composed: true,
                detail: {
                    selectedRow: target,
                    selectedRows: selectedRows,
                },
            };
            this.dispatchEvent(new CustomEvent(this.constructor.eventTableRowSelect, init));
        };
        /**
         * Handles header row selection, selecting/unselecting all rows
         */
        this._handleAllRowsSelect = async (event) => {
            const { detail, target } = event;
            const { selected } = detail;
            const { _tableBatchActions: tableBatchActions, _tableToolbarContent: tableToolbarContent, _tableRows: tableRows, } = this;
            if (!this.contains(target)) {
                return;
            }
            let totalRows = 0;
            forEach(tableRows, (elem) => {
                if (!elem.filtered) {
                    elem.selected = selected;
                    this.radio
                        ? (elem.shadowRoot.querySelector(`${prefix}-radio-button`).checked = selected)
                        : null;
                    this._selectedRows.push(elem);
                    totalRows++;
                    const { selectorTableExpandedRows } = this
                        .constructor;
                    const { nextElementSibling } = elem;
                    // selecting the expanded row as well
                    if (nextElementSibling === null || nextElementSibling === void 0 ? void 0 : nextElementSibling.matches(selectorTableExpandedRows)) {
                        elem.nextElementSibling.selected = selected;
                    }
                }
            });
            if (!selected) {
                this._selectedRows = [];
            }
            if (tableBatchActions) {
                tableBatchActions.selectedRowsCount = selected ? totalRows : 0;
                tableBatchActions.active = selected;
            }
            if (tableToolbarContent) {
                tableToolbarContent.hasBatchActions = selected;
            }
            const init = {
                bubbles: true,
                cancelable: true,
                composed: true,
                detail: {
                    selectedRows: this._selectedRows,
                },
            };
            this.dispatchEvent(new CustomEvent(this.constructor.eventTableRowSelectAll, init));
        };
        /**
         * Handles cancel button within the toolbar actions
         */
        this._handleCancelSelection = async (event) => {
            var _a;
            const { target } = event;
            const { _tableHeaderRow: tableHeaderRow } = this;
            if (this.contains(target)) {
                (_a = tableHeaderRow.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(`${prefix}-checkbox`).shadowRoot.querySelector(`.${prefix}--checkbox`).click();
            }
        };
    }
    /**
     * @param lhs A value.
     * @param rhs Another value.
     * @param collator A custom collator.
     * @returns
     *   `0` if the given two values are equal
     *   A negative value to sort `lhs` to an index lower than `rhs`
     *   A positive value to sort `rhs` to an index lower than `lhs`
     */
    // eslint-disable-next-line class-methods-use-this
    customSortRow(lhs, rhs, collator) {
        if (typeof lhs === 'number' && typeof rhs === 'number') {
            return lhs - rhs;
        }
        return collator.compare(lhs, rhs);
    }
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .some((node) => node.nodeType !== Node.TEXT_NODE || node.textContent.trim());
        this.withHeader = hasContent;
    }
    _handleTableBodyChange() {
        // cds-table-head or cds-table-body may have changed
        this._tableBody = this.querySelector(this.constructor.selectorTableBody);
        this._tableExpandedRows = this.querySelectorAll(this.constructor.selectorTableExpandedRows);
        this._tableRows = this.querySelectorAll(this.constructor.selectorTableRow);
        // update any row dependant features
        this.updateExpandable();
    }
    _handleSortAction(columnIndex, sortDirection) {
        const rows = [...this._tableRows];
        // regular row sorting
        rows.sort((a, b) => {
            const cellA = a.querySelectorAll(this.constructor.selectorTableRowCells)[columnIndex].textContent;
            const cellB = b.querySelectorAll(this.constructor.selectorTableRowCells)[columnIndex].textContent;
            return (this.collationFactors[sortDirection] *
                this.customSortRow(cellA, cellB, this.collator));
        });
        // take into account the expanded rows, mapping each expandable row to its original for proper reinsertion
        if (this.expandable) {
            const originalRows = [...this._tableRows];
            const expandedRows = [...this._tableExpandedRows];
            const mapping = originalRows.reduce((acc, element, index) => {
                const sortId = element.getAttribute('sort-id');
                acc[sortId] = expandedRows[index];
                return acc;
            }, {});
            const sortedWithExpanded = [];
            rows.forEach((e) => {
                const sortId = e.getAttribute('sort-id');
                sortedWithExpanded.push(e);
                sortedWithExpanded.push(mapping[sortId]);
            });
            sortedWithExpanded.forEach((e) => {
                this._tableBody.insertBefore(e, null);
            });
        }
        else {
            rows.forEach((e) => {
                this._tableBody.insertBefore(e, null);
            });
        }
    }
    _handleFilterRows() {
        const unfilteredRows = [];
        forEach(this._tableRows, (elem) => {
            var _a, _b;
            let rowText = (_a = elem.textContent) === null || _a === void 0 ? void 0 : _a.trim();
            let filtered = this.filterRows(rowText, this._searchValue);
            elem.filtered = filtered;
            if (filtered && this.expandable) {
                rowText = (_b = elem.nextElementSibling.textContent) === null || _b === void 0 ? void 0 : _b.trim();
                filtered = this.filterRows(rowText, this._searchValue);
                elem.filtered = filtered;
            }
            if (!filtered) {
                unfilteredRows.push(elem);
            }
            if (this.expandable) {
                elem.nextElementSibling.filtered = filtered;
            }
        });
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                unfilteredRows,
            },
        };
        this.dispatchEvent(new CustomEvent(this.constructor.eventTableFiltered, init));
    }
    /**
     * Download manager for selected rows.
     */
    _handleDownload({ target }) {
        const data = [];
        const elementsToArray = (elements) => Array.from(elements, (element) => element.textContent);
        const headerCells = this.querySelectorAll(this.constructor.selectorHeaderCell);
        const rows = this._selectedRows;
        const headerTitleArray = elementsToArray(headerCells);
        rows.forEach((row) => {
            const rowData = {};
            const cells = elementsToArray(row.querySelectorAll(this.constructor.selectorTableRowCells));
            cells.forEach((cellText, index) => {
                const headerTitle = headerTitleArray[index];
                rowData[headerTitle] = cellText;
            });
            data.push(rowData);
        });
        const blob = new Blob([JSON.stringify(data)], { type: 'application/json' });
        target.href = URL.createObjectURL(blob);
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'table');
        }
        /**
         * If using `with-row-slugs`, set `with-row-ai-labels` attribute to true so
         * the styles are applied for slug as well
         *
         * remove in v12
         */
        if (this.withRowSlugs) {
            this.setAttribute('with-rows-ai-labels', '');
            this.withRowAILabels = true;
        }
        super.connectedCallback();
    }
    firstUpdated() {
        this._tableBatchActions = this.querySelector(this.constructor.selectorTableBatchActions);
        this._tableToolbar = this.querySelector(this.constructor.selectorTableToolbar);
        this._tableToolbarContent = this.querySelector(this.constructor.selectorTableToolbarContent);
        this._tableBody = this.querySelector(this.constructor.selectorTableBody);
        this._tableHeaderRow = this.querySelector(this.constructor.selectorRowsWithHeader);
        this._tableExpandedRows = this.querySelectorAll(this.constructor.selectorTableExpandedRows);
        this._tableRows = this.querySelectorAll(this.constructor.selectorTableRow);
        this._downloadButton = this.querySelector(this.constructor.selectorToolbarDownload);
        if (this._downloadButton) {
            this._downloadButton.onclick = this._handleDownload.bind(this);
        }
        this.headerCount = this._tableHeaderRow.children.length;
    }
    updateExpandable() {
        this._tableRows.forEach((e, index) => {
            e.expandable = this.expandable;
            e.setAttribute('sort-id', index);
        });
        this._tableHeaderRow.expandable = this.expandable;
        this._tableHeaderRow.batchExpansion = this.batchExpansion;
        this.headerCount += this.expandable ? 1 : -1;
    }
    updated(changedProperties) {
        var _a;
        if (changedProperties.has('expandable')) {
            this.updateExpandable();
        }
        if (changedProperties.has('headerCount')) {
            this._tableExpandedRows.forEach((e) => {
                e.setAttribute('colspan', this.headerCount);
            });
        }
        if (changedProperties.has('isSelectable')) {
            if (this.isSelectable) {
                this._tableHeaderRow.setAttribute('selection-name', 'header');
                this._tableRows.forEach((e, index) => {
                    if (!e.hasAttribute('selection-name')) {
                        e.setAttribute('selection-name', index);
                    }
                });
            }
            this.headerCount++;
        }
        if (changedProperties.has('locale')) {
            this.collator = new Intl.Collator(this.locale);
        }
        if (changedProperties.has('isSortable')) {
            if (this.isSortable) {
                this._enableSortAction();
            }
        }
        if (changedProperties.has('overflowMenuOnHover') ||
            changedProperties.has('size')) {
            forEach(this.querySelectorAll(this.constructor.selectorTableCellOverflowMenu), (elem) => {
                const cell = elem.parentNode;
                const row = cell.parentNode;
                cell.overflowMenuOnHover = this.overflowMenuOnHover;
                row.overflowMenuOnHover = this.overflowMenuOnHover;
                cell.setAttribute('size', this.size);
                elem.setAttribute('size', this.size);
                elem.setAttribute('data-table', '');
            });
        }
        if (changedProperties.has('radio')) {
            // Propagate `size` attribute to descendants until `:host-context()` gets supported in all major browsers
            forEach(this.querySelectorAll(this.constructor.selectorTableRow), (elem) => {
                elem.radio = this.radio;
            });
        }
        if (changedProperties.has('size')) {
            // Propagate `size` attribute to descendants until `:host-context()` gets supported in all major browsers
            forEach(this.querySelectorAll(this.constructor.selectorAllRows), (elem) => {
                elem.setAttribute('size', this.size);
            });
            (_a = this._tableToolbar) === null || _a === void 0 ? void 0 : _a.setAttribute('size', this.size);
        }
        // TODO: Uncomment when Carbon fully implements Sticky header feature
        // if (changedProperties.has('stickyHeader')) {
        //   const tableBody = this.querySelector(
        //     (this.constructor as typeof CDSTable).selectorTableBody
        //   );
        //   const tableHead = this.querySelector(
        //     (this.constructor as typeof CDSTable).selectorTableHead
        //   );
        //   (tableBody as any).stickyHeader = this.stickyHeader;
        //   (tableHead as any).stickyHeader = this.stickyHeader;
        //   forEach(
        //     this.querySelectorAll(
        //       (this.constructor as typeof CDSTable).selectorRowsWithHeader
        //     ),
        //     (elem) => {
        //       (elem as any).stickyHeader = this.stickyHeader;
        //     }
        //   );
        //   forEach(
        //     this.querySelectorAll(
        //       (this.constructor as typeof CDSTable).selectorTableCells
        //     ),
        //     (elem) => {
        //       (elem as any).stickyHeader = this.stickyHeader;
        //     }
        //   );
        // }
        if (changedProperties.has('useZebraStyles')) {
            const tableBody = this.querySelector(this.constructor.selectorTableBody);
            tableBody.useZebraStyles = this.useZebraStyles;
        }
        if (this.withRowAILabels) {
            this._tableHeaderRow.setAttribute('rows-with-ai-label', '');
            this._tableRows.forEach((row) => {
                row.setAttribute('rows-with-ai-label', '');
            });
        }
        else {
            this._tableHeaderRow.removeAttribute('rows-with-ai-label');
            this._tableRows.forEach((row) => {
                row.removeAttribute('rows-with-ai-label');
            });
        }
        // Gets table header info to add to the column cells for styles
        const headersWithAILabel = [];
        Array.prototype.slice
            .call(this._tableHeaderRow.children)
            .forEach((headerCell, index) => {
            if (headerCell.querySelector(`${prefix}-ai-label`) ||
                headerCell.querySelector(`${prefix}-slug`)) {
                headerCell.setAttribute('ai-label', '');
                headersWithAILabel.push(index);
            }
            else {
                headerCell.removeAttribute('ai-label');
            }
        });
        this._tableRows.forEach((row) => {
            Array.prototype.slice
                .call(row.children)
                .forEach((cell, index) => {
                headersWithAILabel.includes(index)
                    ? cell.setAttribute('ai-label-in-header', '')
                    : cell.removeAttribute('ai-label-in-header');
            });
        });
    }
    /* eslint-disable no-constant-condition */
    render() {
        return html `
      <div class="${prefix}--data-table-header-container">
        <div ?hidden="${!this.withHeader}" class="${prefix}--data-table-header">
          <slot @slotchange="${this._handleSlotChange}" name="title"></slot>
          <slot
            @slotchange="${this._handleSlotChange}"
            name="description"></slot>
        </div>
        <slot name="toolbar"></slot>
      </div>

      ${html `<slot
            @cds-table-body-content-change="${this
                ._handleTableBodyChange}"></slot>`}
    `;
    }
    /**
     * Adds isSortable value for table header cells.
     */
    _enableSortAction() {
        const headerCells = this.querySelectorAll(this.constructor.selectorHeaderCell);
        headerCells.forEach((e) => {
            e.isSortable = this.isSortable;
            e.isSelectable = this.isSelectable;
            e.isExpandable = this.expandable;
        });
        const columns = [...this._tableHeaderRow.children];
        let sortDirection;
        let columnIndex = 0;
        columns.forEach((column, index) => {
            if (column.hasAttribute('sort-direction') &&
                column.getAttribute('sort-direction') !== 'none') {
                sortDirection = column.getAttribute('sort-direction');
                columnIndex = index;
            }
        });
        columns.forEach((e, index) => {
            if (index !== columnIndex && this.isSortable) {
                e.setAttribute('sort-direction', 'none');
            }
            else if (e.hasAttribute('is-sortable')) {
                e.setAttribute('sort-direction', 'none');
            }
        });
        this._handleSortAction(columnIndex, sortDirection);
    }
    /* eslint-enable no-constant-condition */
    /**
     * The name of the custom event fired before a new sort direction is set upon a user gesture.
     * Cancellation of this event stops the user-initiated change in sort direction.
     */
    static get eventBeforeSort() {
        return `${prefix}-table-header-cell-sort`;
    }
    /**
     * The name of the custom event fired during search bar input
     */
    static get eventSearchInput() {
        return `${prefix}-search-input`;
    }
    /**
     * The name of the custom event fired before header row is selected/unselected upon a user gesture.
     */
    static get eventBeforeChangeSelectionAll() {
        return `${prefix}-table-change-selection-all`;
    }
    /**
     * The name of the custom event fired before a row is selected/unselected upon a user gesture.
     */
    static get eventBeforeChangeSelection() {
        return `${prefix}-table-row-change-selection`;
    }
    /**
     * The name of the custom event fired after the Cancel button is clicked.
     */
    static get eventClickCancel() {
        return `${prefix}-table-batch-actions-cancel-clicked`;
    }
    /**
     * The name of the custom event fired after the expanded state a row is toggled upon a user gesture.
     */
    static get eventExpandoToggle() {
        return `${prefix}-table-row-expando-toggled`;
    }
    /**
     * The name of the custom event fired after a row has been selected
     */
    static get eventTableRowSelect() {
        return `${prefix}-table-row-selected`;
    }
    /**
     * The name of the custom event fired after all rows have been selected
     */
    static get eventTableRowSelectAll() {
        return `${prefix}-table-row-all-selected`;
    }
    /**
     * The name of the custom event fired after the table has been sorted
     */
    static get eventTableSorted() {
        return `${prefix}-table-sorted`;
    }
    /**
     * The name of the custom event fired after the table has been filtered containing remaining rows.
     */
    static get eventTableFiltered() {
        return `${prefix}-table-filtered`;
    }
    /**
     * The CSS selector to find the overflow menu on the table cell
     */
    static get selectorTableCellOverflowMenu() {
        return `${prefix}-table-cell ${prefix}-overflow-menu`;
    }
    /**
     * The CSS selector to find the download button
     */
    static get selectorToolbarDownload() {
        return `${prefix}-button[download]`;
    }
    /**
     * The CSS selector to find the table batch actions
     */
    static get selectorTableBatchActions() {
        return `${prefix}-table-batch-actions`;
    }
    /**
     * The CSS selector to find the table toolbar
     */
    static get selectorTableToolbar() {
        return `${prefix}-table-toolbar`;
    }
    /**
     * The CSS selector to find the table toolbar content
     */
    static get selectorTableToolbarContent() {
        return `${prefix}-table-toolbar-content`;
    }
    /**
     * The CSS selector to find the table toolbar search
     */
    static get selectorTableToolbarSearch() {
        return `${prefix}-table-toolbar-search`;
    }
    /**
     * The CSS selector to find the table head
     */
    static get selectorTableHead() {
        return `${prefix}-table-head`;
    }
    /**
     * The CSS selector to find the table body
     */
    static get selectorTableBody() {
        return `${prefix}-table-body`;
    }
    /**
     * The CSS selector to find the table expanded rows
     */
    static get selectorTableExpandedRows() {
        return `${prefix}-table-expanded-row`;
    }
    /**
     * The CSS selector to find the table rows
     */
    static get selectorTableRow() {
        return `${prefix}-table-row`;
    }
    /**
     * The CSS selector to find the rows cells.
     */
    static get selectorTableRowCells() {
        return `${prefix}-table-cell`;
    }
    /**
     * The CSS selector to find the rows cells, including header cells.
     */
    static get selectorTableCells() {
        return `${prefix}-table-cell, ${prefix}-table-header-cell`;
    }
    /**
     * The CSS selector to find the header cell
     */
    static get selectorHeaderCell() {
        return `${prefix}-table-header-cell`;
    }
    /**
     * The CSS selector to find the rows, including header rows.
     */
    static get selectorRowsWithHeader() {
        return `${prefix}-table-header-row,${prefix}-table-row`;
    }
    /**
     * The CSS selector to find all rows
     */
    static get selectorAllRows() {
        return `${prefix}-table-header-row,${prefix}-table-row,${prefix}-table-expanded-row`;
    }
};
CDSTable.styles = styles;
__decorate([
    state()
], CDSTable.prototype, "_downloadButton", void 0);
__decorate([
    state()
], CDSTable.prototype, "_searchValue", void 0);
__decorate([
    state()
], CDSTable.prototype, "_tableHeaderRow", void 0);
__decorate([
    state()
], CDSTable.prototype, "_tableBody", void 0);
__decorate([
    state()
], CDSTable.prototype, "_tableExpandedRows", void 0);
__decorate([
    state()
], CDSTable.prototype, "_tableRows", void 0);
__decorate([
    state()
], CDSTable.prototype, "_tableBatchActions", void 0);
__decorate([
    state()
], CDSTable.prototype, "_tableToolbar", void 0);
__decorate([
    state()
], CDSTable.prototype, "_tableToolbarContent", void 0);
__decorate([
    state()
], CDSTable.prototype, "_selectedRows", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'batch-expansion' })
], CDSTable.prototype, "batchExpansion", void 0);
__decorate([
    property({ attribute: false })
], CDSTable.prototype, "collator", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTable.prototype, "expandable", void 0);
__decorate([
    property()
], CDSTable.prototype, "filterRows", void 0);
__decorate([
    property()
], CDSTable.prototype, "headerCount", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'is-selectable' })
], CDSTable.prototype, "isSelectable", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'is-sortable' })
], CDSTable.prototype, "isSortable", void 0);
__decorate([
    property({ reflect: true })
], CDSTable.prototype, "locale", void 0);
__decorate([
    property({
        type: Boolean,
        reflect: true,
        attribute: 'overflow-menu-on-hover',
    })
], CDSTable.prototype, "overflowMenuOnHover", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTable.prototype, "radio", void 0);
__decorate([
    property({ reflect: true })
], CDSTable.prototype, "size", void 0);
__decorate([
    property({ type: Boolean, attribute: 'use-static-width', reflect: true })
], CDSTable.prototype, "useStaticWidth", void 0);
__decorate([
    property({ type: Boolean, attribute: 'use-zebra-styles', reflect: true })
], CDSTable.prototype, "useZebraStyles", void 0);
__decorate([
    property({ type: Boolean, attribute: 'with-header', reflect: true })
], CDSTable.prototype, "withHeader", void 0);
__decorate([
    property({ type: Boolean, attribute: 'with-row-ai-labels' })
], CDSTable.prototype, "withRowAILabels", void 0);
__decorate([
    property({ type: Boolean, attribute: 'with-row-slugs' })
], CDSTable.prototype, "withRowSlugs", void 0);
__decorate([
    HostListener('eventExpandoToggle')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTable.prototype, "_handleBatchExpansion", void 0);
__decorate([
    HostListener('eventBeforeSort')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTable.prototype, "_handleSort", void 0);
__decorate([
    HostListener('eventSearchInput')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTable.prototype, "_handleSearchInput", void 0);
__decorate([
    HostListener('eventBeforeChangeSelection')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTable.prototype, "_handleRowSelect", void 0);
__decorate([
    HostListener('eventBeforeChangeSelectionAll')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTable.prototype, "_handleAllRowsSelect", void 0);
__decorate([
    HostListener('eventClickCancel')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTable.prototype, "_handleCancelSelection", void 0);
CDSTable = __decorate([
    carbonElement(`${prefix}-table`)
], CDSTable);
var CDSTable$1 = CDSTable;

export { CDSTable$1 as default };
//# sourceMappingURL=table.js.map
