/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { query, state, property } from 'lit/decorators.js';
import Calendar16 from '../../icons/calendar/16.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import '../text-input/text-input.js';
import { DATE_PICKER_INPUT_KIND } from './defs.js';
export { DATE_PICKER_INPUT_KIND } from './defs.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import styles from './date-picker.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { FORM_ELEMENT_COLOR_SCHEME } from '../../globals/shared-enums.js';
export { FORM_ELEMENT_COLOR_SCHEME as DATE_PICKER_INPUT_COLOR_SCHEME } from '../../globals/shared-enums.js';
import { INPUT_SIZE } from '../text-input/defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * The input box for date picker.
 *
 * @element cds-date-picker-input
 */
let CDSDatePickerInput = class CDSDatePickerInput extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * `true` if there is helper text content.
         */
        this._hasHelperText = false;
        /**
         * The color scheme.
         */
        this.colorScheme = FORM_ELEMENT_COLOR_SCHEME.REGULAR;
        /**
         * `true` if the check box should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if the label should be hidden.
         */
        this.hideLabel = false;
        /**
         * Specify whether the control is currently in warning state
         */
        this.warn = false;
        /**
         * Provide the text that is displayed when the control is in warning state
         */
        this.warnText = '';
        /**
         * Message which is displayed if the value is invalid.
         */
        this.invalidText = '';
        /**
         * Controls the invalid state and visibility of the `validityMessage`.
         */
        this.invalid = false;
        /**
         * Date picker input kind.
         */
        this.kind = DATE_PICKER_INPUT_KIND.SIMPLE;
        /**
         * Specify if the component should be read-only
         */
        this.readonly = false;
        /**
         * `true` if the value is required.
         */
        this.required = false;
        /**
         * Vertical size of this date picker input.
         */
        this.size = INPUT_SIZE.MEDIUM;
        /**
         * true to use the short version.
         */
        this.short = false;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleAILabelSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove reference to slug in v12
                elem.matches(this.constructor.slugItem)
            : false);
        this._hasAILabel = Boolean(hasContent);
        hasContent[0].setAttribute('size', 'mini');
        this.requestUpdate();
    }
    /**
     * Handles `click` event on the calendar icon.
     *
     * @param event The event.
     */
    _handleClickWrapper(event) {
        if (event.target === this._iconNode) {
            this.input.focus();
        }
    }
    /**
     * Handles `input` event on `<input>` in the shadow DOM.
     *
     * @param event The event.
     * @param event.target The event target.
     */
    _handleInput({ target }) {
        const { value } = target;
        this.value = value;
    }
    /**
     * @returns The template for the the calendar icon.
     */
    _renderIcon() {
        return this.kind === DATE_PICKER_INPUT_KIND.SIMPLE
            ? undefined
            : Calendar16({
                class: `${prefix}--date-picker__icon`,
                role: 'img',
                children: [html ` <title>Open calendar</title> `],
            });
    }
    /**
     * Handles `slotchange` event on the default `<slot>`.
     */
    _handleSlotChange({ target }) {
        if (!target.name) {
            const hasContent = target
                .assignedNodes()
                .some((node) => node.nodeType !== Node.TEXT_NODE || node.textContent.trim());
            this._hasHelperText = hasContent;
        }
    }
    render() {
        const constructor = this.constructor;
        const { disabled, _hasHelperText: hasHelperText, hideLabel, invalid, invalidText, labelText, pattern = constructor.defaultPattern, placeholder, readonly, size, type = constructor.defaultType, value, warn, warnText, _handleClickWrapper: handleClickWrapper, _handleInput: handleInput, _hasAILabel: hasAILabel, } = this;
        const invalidIcon = WarningFilled16({
            class: `${prefix}--date-picker__icon ${prefix}--date-picker__icon--invalid`,
        });
        const warnIcon = WarningAltFilled16({
            class: `${prefix}--date-picker__icon ${prefix}--date-picker__icon--warn`,
        });
        const normalizedProps = {
            disabled: !readonly && disabled,
            invalid: !readonly && invalid,
            warn: !readonly && !invalid && warn,
            'slot-name': '',
            'slot-text': '',
            icon: null,
        };
        if (normalizedProps.invalid) {
            normalizedProps.icon = invalidIcon;
            normalizedProps['slot-name'] = 'invalid-text';
            normalizedProps['slot-text'] = invalidText;
        }
        else if (normalizedProps.warn) {
            normalizedProps.icon = warnIcon;
            normalizedProps['slot-name'] = 'warn-text';
            normalizedProps['slot-text'] = warnText;
        }
        const labelClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--visually-hidden`]: hideLabel,
            [`${prefix}--label--disabled`]: disabled,
        });
        const inputClasses = classMap({
            [`${prefix}--date-picker__input`]: true,
            [`${prefix}--date-picker__input--invalid`]: normalizedProps.invalid,
            [`${prefix}--date-picker__input--warn`]: normalizedProps.warn,
            [`${prefix}--date-picker__input--${size}`]: size,
        });
        const inputWrapperClasses = classMap({
            [`${prefix}--date-picker-input__wrapper`]: true,
            [`${prefix}--date-picker-input__wrapper--invalid`]: normalizedProps.invalid,
            [`${prefix}--date-picker-input__wrapper--warn`]: normalizedProps.warn,
            [`${prefix}--date-picker-input__wrapper--slug`]: hasAILabel,
        });
        const helperTextClasses = classMap({
            [`${prefix}--form__helper-text`]: true,
            [`${prefix}--form__helper-text--disabled`]: disabled,
        });
        return html `
      <label for="input" class="${labelClasses}">
        <slot name="label-text">${labelText}</slot>
      </label>
      <div class="${inputWrapperClasses}" @click="${handleClickWrapper}">
        <span>
          <input
            id="input"
            type="${type}"
            class="${inputClasses}"
            ?disabled="${disabled}"
            pattern="${pattern}"
            placeholder="${ifDefined(placeholder)}"
            .value="${ifDefined(value)}"
            ?data-invalid="${invalid}"
            @input="${handleInput}"
            ?readonly="${readonly}" />
          ${normalizedProps.icon || this._renderIcon()}
          <slot
            name="ai-label"
            @slotchange="${this._handleAILabelSlotChange}"></slot>
          <slot
            name="slug"
            @slotchange="${this._handleAILabelSlotChange}"></slot>
        </span>
      </div>
      <div
        class="${prefix}--form-requirement"
        ?hidden="${!normalizedProps.invalid && !normalizedProps.warn}">
        <slot name="${normalizedProps['slot-name']}">
          ${normalizedProps['slot-text']}
        </slot>
      </div>
      <div ?hidden="${hasHelperText}" class="${helperTextClasses}">
        <slot name="helper-text" @slotchange="${this._handleSlotChange}"></slot>
      </div>
    `;
    }
    updated() {
        var _a, _b, _c, _d, _e;
        const label = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector("slot[name='ai-label']");
        if (label) {
            label === null || label === void 0 ? void 0 : label.classList.toggle(`${prefix}--slug--revert`, (_b = this.querySelector(`${prefix}-ai-label`)) === null || _b === void 0 ? void 0 : _b.hasAttribute('revert-active'));
        }
        else {
            (_d = (_c = this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector("slot[name='slug']")) === null || _d === void 0 ? void 0 : _d.classList.toggle(`${prefix}--slug--revert`, (_e = this.querySelector(`${prefix}-slug`)) === null || _e === void 0 ? void 0 : _e.hasAttribute('revert-active'));
        }
    }
    /**
     * A selector that will return the parent date picker.
     */
    static get selectorParent() {
        return `${prefix}-date-picker`;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
};
/**
 * The default value for `pattern` property.
 */
CDSDatePickerInput.defaultPattern = '\\d{1,2}\\/\\d{1,2}\\/\\d{4}';
/**
 * The default value for `type` property.
 */
CDSDatePickerInput.defaultType = 'text';
CDSDatePickerInput.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSDatePickerInput.styles = styles;
__decorate([
    query(`.${prefix}--date-picker__icon`)
], CDSDatePickerInput.prototype, "_iconNode", void 0);
__decorate([
    state()
], CDSDatePickerInput.prototype, "_hasHelperText", void 0);
__decorate([
    query('input')
], CDSDatePickerInput.prototype, "input", void 0);
__decorate([
    property({ attribute: 'color-scheme', reflect: true })
], CDSDatePickerInput.prototype, "colorScheme", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePickerInput.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'hide-label' })
], CDSDatePickerInput.prototype, "hideLabel", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePickerInput.prototype, "warn", void 0);
__decorate([
    property({ attribute: 'warn-text' })
], CDSDatePickerInput.prototype, "warnText", void 0);
__decorate([
    property({ attribute: 'invalid-text' })
], CDSDatePickerInput.prototype, "invalidText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePickerInput.prototype, "invalid", void 0);
__decorate([
    property({ reflect: true })
], CDSDatePickerInput.prototype, "kind", void 0);
__decorate([
    property({ attribute: 'label-text' })
], CDSDatePickerInput.prototype, "labelText", void 0);
__decorate([
    property()
], CDSDatePickerInput.prototype, "pattern", void 0);
__decorate([
    property()
], CDSDatePickerInput.prototype, "placeholder", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePickerInput.prototype, "readonly", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePickerInput.prototype, "required", void 0);
__decorate([
    property({ attribute: 'size', reflect: true })
], CDSDatePickerInput.prototype, "size", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePickerInput.prototype, "short", void 0);
__decorate([
    property()
], CDSDatePickerInput.prototype, "type", void 0);
__decorate([
    property()
], CDSDatePickerInput.prototype, "value", void 0);
CDSDatePickerInput = __decorate([
    carbonElement(`${prefix}-date-picker-input`)
], CDSDatePickerInput);
var CDSDatePickerInput$1 = CDSDatePickerInput;

export { CDSDatePickerInput$1 as default };
//# sourceMappingURL=date-picker-input.js.map
