/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import flatpickr from 'flatpickr';
import { prefix } from '../../globals/settings.js';
import FormMixin from '../../globals/mixins/form.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import { getISODateString, parseISODateString } from './iso-date.js';
import appendToPlugin from './append-to-plugin.js';
import cssClassPlugin from './css-class-plugin.js';
import fixEventsPlugin from './fix-events-plugin.js';
import focusPlugin from './focus-plugin.js';
import iconPlugin from './icon-plugin.js';
import monthSelectPlugin from './month-select-plugin.js';
import rangePlugin from './range-plugin.js';
import shadowDOMEventPlugin from './shadow-dom-events-plugin.js';
import stateHandshakePlugin from './state-handshake-plugin.js';
import styles from './date-picker.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Date picker modes.
 */
var DATE_PICKER_MODE;
(function (DATE_PICKER_MODE) {
    /**
     * Simple mode, without calendar dropdown.
     */
    DATE_PICKER_MODE["SIMPLE"] = "simple";
    /**
     * Single date mode.
     */
    DATE_PICKER_MODE["SINGLE"] = "single";
    /**
     * Range mode.
     */
    DATE_PICKER_MODE["RANGE"] = "range";
})(DATE_PICKER_MODE || (DATE_PICKER_MODE = {}));
// Weekdays shorthand for english locale
flatpickr.l10ns.en.weekdays.shorthand.forEach((_day, index) => {
    const currentDay = flatpickr.l10ns.en.weekdays.shorthand;
    if (currentDay[index] === 'Thu' || currentDay[index] === 'Th') {
        currentDay[index] = 'Th';
    }
    else {
        currentDay[index] = currentDay[index].charAt(0);
    }
});
/**
 * Date picker.
 *
 * @element cds-date-picker
 * @fires cds-date-picker-changed - The custom event fired on this element when Flatpickr updates its value.
 * @fires cds-date-picker-flatpickr-error
 *   The name of the custom event when Flatpickr throws an error.
 */
let CDSDatePicker = class CDSDatePicker extends HostListenerMixin(FormMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * The slotted `<cds-date-input kind="from">`.
         */
        this._dateInteractNode = null;
        /**
         * Handles `${prefix}-date-picker-changed` event on this element.
         */
        this._handleChange = ({ detail }) => {
            this._value = detail.selectedDates
                .map((date) => getISODateString(date))
                .join('/');
        };
        /**
         * Fires a custom event to notify an error in Flatpickr.
         */
        this._handleFlatpickrError = (error) => {
            this.dispatchEvent(new CustomEvent(this.constructor.eventFlatpickrError, {
                bubbles: true,
                cancelable: false,
                composed: true,
                detail: {
                    error,
                },
            }));
        };
        /**
         * The Flatpickr instance.
         */
        this.calendar = null;
        /**
         * flatpickr prop passthrough. Allows the user to enter a date directly into the input field
         */
        this.allowInput = true;
        /**
         * flatpickr prop passthrough. Controls whether the calendar dropdown closes upon selection.
         */
        this.closeOnSelect = true;
        /**
         * Controls the disabled state of the input
         */
        this.disabled = false;
        /**
         * Name for the input in the `FormData`
         */
        this.name = '';
        /**
         * `true` if the date picker should be open.
         */
        this.open = false;
        /**
         * Specify if the component should be read-only
         */
        this.readonly = false;
    }
    /**
     * @returns The effective date picker mode, determined by the child `<cds-date-picker-input>`.
     */
    get _mode() {
        const { selectorInputFrom, selectorInputTo } = this
            .constructor;
        if (this.querySelector(selectorInputTo)) {
            return DATE_PICKER_MODE.RANGE;
        }
        if (this.querySelector(selectorInputFrom)) {
            return DATE_PICKER_MODE.SINGLE;
        }
        return DATE_PICKER_MODE.SIMPLE;
    }
    /**
     * @returns The Flatpickr plugins to use.
     */
    get _datePickerPlugins() {
        const { classCalendarContainer, classMonth, classWeekdays, classDays, classWeekday, classDay, classNoBorder, selectorInputFrom, selectorInputTo, _selectorFlatpickrMonthYearContainer: selectorFlatpickrMonthYearContainer, _selectorFlatpickrYearContainer: selectorFlatpickrYearContainer, _selectorFlatpickrCurrentMonth: selectorFlatpickrCurrentMonth, _selectorFlatpickrMonth: selectorFlatpickrMonth, _selectorFlatpickrWeekdays: selectorFlatpickrWeekdays, _selectorFlatpickrDays: selectorFlatpickrDays, _selectorFlatpickrWeekday: selectorFlatpickrWeekday, _selectorFlatpickrDay: selectorFlatpickrDay, _classFlatpickrCurrentMonth: classFlatpickrCurrentMonth, _classFlatpickrToday: classFlatpickrToday, } = this.constructor;
        const { _floatingMenuContainerNode: floatingMenuContainerNode, _mode: mode, } = this;
        const inputFrom = this.querySelector(selectorInputFrom);
        const inputTo = this.querySelector(selectorInputTo);
        const plugins = [
            appendToPlugin({ appendTo: floatingMenuContainerNode }),
            cssClassPlugin({
                classCalendarContainer,
                classMonth,
                classWeekdays,
                classDays,
                classWeekday,
                classDay,
                classNoBorder,
                selectorFlatpickrMonth,
                selectorFlatpickrWeekdays,
                selectorFlatpickrDays,
                selectorFlatpickrWeekday,
                selectorFlatpickrDay,
                classFlatpickrToday,
            }),
            fixEventsPlugin({
                inputFrom: inputFrom,
                inputTo: inputTo,
            }),
            focusPlugin({
                inputFrom: inputFrom,
                inputTo: inputTo,
            }),
            iconPlugin(),
            monthSelectPlugin({
                selectorFlatpickrMonthYearContainer,
                selectorFlatpickrYearContainer,
                selectorFlatpickrCurrentMonth,
                classFlatpickrCurrentMonth,
            }),
            shadowDOMEventPlugin(),
            stateHandshakePlugin(this),
        ];
        if (mode === DATE_PICKER_MODE.RANGE) {
            // Flatpickr runs event handlers of last registered plugins first.
            // Ensures `onValueUpdate` of `rangePlugin` runs first
            // given Flatpickr puts `01/01/1970 to 01/02/1970` to from date
            // where `rangePlugin` overrides it to separate them to from/to dates.
            // We want to ensure our handler of `onValueUpdate` (notably one in `<cds-date-picker-input>`)
            // gets the `<input>` value set by `rangePlugin` instead of Flatpickr core.
            plugins.push(rangePlugin({ input: inputTo }));
        }
        return plugins;
    }
    /**
     * @returns The options to instantiate Flatpickr with.
     */
    get _datePickerOptions() {
        var _a;
        const { locale = this.constructor.defaultLocale, enabledRange, _dateInteractNode: dateInteractNode, _datePickerPlugins: plugins, _handleFlatpickrError: handleFlatpickrError, } = this;
        // We use `<cds-date-picker-input>` to communicate values/events with Flatpickr,
        // but want to use `<input>` in shadow DOM to base the calendar dropdown's position on
        const { input: positionElement } = dateInteractNode;
        const [minDate = undefined, maxDate = undefined] = !enabledRange
            ? []
            : enabledRange.split('/');
        return {
            allowInput: this.allowInput,
            closeOnSelect: this.closeOnSelect,
            dateFormat: (_a = this.dateFormat) !== null && _a !== void 0 ? _a : this.constructor.defaultDateFormat,
            disableMobile: true,
            errorHandler: handleFlatpickrError,
            locale,
            maxDate,
            minDate,
            positionElement,
            plugins,
        };
    }
    _handleFormdata(event) {
        const { formData } = event; // TODO: Wait for `FormDataEvent` being available in `lib.dom.d.ts`
        const { disabled, name, value } = this;
        if (!disabled) {
            formData.append(name, value);
        }
    }
    /**
     * Handles `slotchange` event in the `<slot>`.
     */
    _handleSlotChange({ target }) {
        const { _dateInteractNode: oldDateInteractNode } = this;
        const dateInteractNode = target
            .assignedNodes()
            .find((node) => node.nodeType === Node.ELEMENT_NODE &&
            node.matches(this.constructor.selectorInputFrom));
        if (oldDateInteractNode !== dateInteractNode) {
            this._dateInteractNode = dateInteractNode;
            this._instantiateDatePicker();
        }
    }
    /**
     * Sets calendar options
     * @param property property to set
     * @param calendar calendar object
     */
    _setCalendar(property, calendar) {
        const { disabled, dateFormat, open, readonly, minDate, maxDate, value } = this;
        const { selectorInputFrom, selectorInputTo } = this
            .constructor;
        const inputFrom = this.querySelector(selectorInputFrom);
        const inputTo = this.querySelector(selectorInputTo);
        if (property === 'dateFormat') {
            calendar.set({ dateFormat });
        }
        if (property === 'date') {
            if (!parseISODateString(minDate)) {
                // Allows empty start/end
                throw new Error(`Wrong date format found in \`minDate\` property: ${minDate}`);
            }
            if (!parseISODateString(maxDate)) {
                // Allows empty start/end
                throw new Error(`Wrong date format found in \`maxDate\` property: ${maxDate}`);
            }
            if (minDate && maxDate && minDate > maxDate) {
                throw new Error(`\`maxDate\` property, shouldn't be smaller than the \`minDate\` property. You have: minDate: ${minDate}, maxDate: ${maxDate}`);
            }
            calendar.set({ minDate, maxDate });
        }
        if (property === 'open') {
            if (open && !readonly) {
                calendar.open();
            }
            else {
                calendar.close();
            }
        }
        if (property === 'disabled') {
            [inputFrom, inputTo].forEach((input) => {
                if (input) {
                    input.disabled = disabled;
                    input.readonly = readonly;
                }
            });
        }
        if (property === 'value') {
            const dates = value
                .split('/')
                .filter(Boolean)
                .map((item) => parseISODateString(item));
            if (dates.some((item) => isNaN(Number(item)))) {
                throw new Error(`Wrong date format found in \`value\` property: ${value}`);
            }
            const [startDate, endDate] = dates;
            if (startDate && endDate && startDate > endDate) {
                throw new Error(`In \`value\` property, the end date shouldn't be smaller than the start date. You have: ${value}`);
            }
            if (calendar) {
                calendar.setDate(dates);
                [inputFrom, inputTo].forEach((input, i) => {
                    if (input) {
                        input.value = !dates[i]
                            ? ''
                            : calendar.formatDate(new Date(dates[i]), calendar.config.dateFormat);
                    }
                });
            }
        }
        return calendar;
    }
    /**
     * Instantiates Flatpickr.
     *
     * @returns The Flatpickr instance.
     */
    _instantiateDatePicker() {
        this._releaseDatePicker();
        const { _dateInteractNode: dateInteractNode } = this;
        // `this._dateInteractNode` won't be there unless there is a slotted `<cds-date-input type="from">`,
        // which means Flatpickr will never be instantiated in "simple" mode.
        if (dateInteractNode && dateInteractNode.input) {
            this.calendar = flatpickr(dateInteractNode.input, this._datePickerOptions);
        }
        const { calendar, disabled, dateFormat, open, readonly, minDate, maxDate, value, } = this;
        if (dateFormat) {
            this._setCalendar('dateFormat', calendar);
        }
        if (minDate || maxDate) {
            this._setCalendar('date', calendar);
        }
        if (open) {
            this._setCalendar('open', calendar);
        }
        if (disabled || readonly) {
            this._setCalendar('disabled', calendar);
        }
        if (value) {
            this._setCalendar('value', calendar);
        }
        return this.calendar;
    }
    /**
     * Releases Flatpickr instances.
     */
    _releaseDatePicker() {
        if (this.calendar) {
            this.calendar.destroy();
            this.calendar = null;
        }
        return this.calendar;
    }
    /**
     * The date(s) in ISO8601 format (date portion only), for range mode, '/' is used for separate start/end dates.
     */
    get value() {
        return this._value;
    }
    set value(value) {
        const { _value: oldValue } = this;
        this._value = value;
        this.requestUpdate('value', oldValue);
    }
    connectedCallback() {
        super.connectedCallback();
        this._instantiateDatePicker();
    }
    disconnectedCallback() {
        this._releaseDatePicker();
        super.disconnectedCallback();
    }
    updated(changedProperties) {
        const { calendar } = this;
        if (calendar && changedProperties.has('dateFormat')) {
            this._setCalendar('dateFormat', calendar);
        }
        if (changedProperties.has('minDate') || changedProperties.has('maxDate')) {
            this._setCalendar('date', calendar);
        }
        if (changedProperties.has('open') && calendar) {
            this._setCalendar('open', calendar);
        }
        if (changedProperties.has('disabled') ||
            changedProperties.has('readonly')) {
            this._setCalendar('disabled', calendar);
        }
        if (changedProperties.has('value')) {
            this._setCalendar('value', calendar);
        }
    }
    render() {
        const { _handleSlotChange: handleSlotChange } = this;
        return html `
      <a
        class="${prefix}--visually-hidden"
        href="javascript:void 0"
        role="navigation"></a>
      <slot @slotchange="${handleSlotChange}"></slot>
      <div id="floating-menu-container"></div>
      <a
        class="${prefix}--visually-hidden"
        href="javascript:void 0"
        role="navigation"></a>
    `;
    }
    /**
     * The CSS class for the calendar dropdown.
     */
    static get classCalendarContainer() {
        return `${prefix}--date-picker__calendar`;
    }
    /**
     * The CSS class for the month navigator.
     */
    static get classMonth() {
        return `${prefix}--date-picker__month`;
    }
    /**
     * The CSS class for the container of the weekdays.
     */
    static get classWeekdays() {
        return `${prefix}--date-picker__weekdays`;
    }
    /**
     * The CSS class for the container of the days.
     */
    static get classDays() {
        return `${prefix}--date-picker__days`;
    }
    /**
     * The CSS class applied to each weekdays.
     */
    static get classWeekday() {
        return `${prefix}--date-picker__weekday`;
    }
    /**
     * The CSS class applied to each days.
     */
    static get classDay() {
        return `${prefix}--date-picker__day`;
    }
    /**
     * A selector that will return the `<input>` to enter starting date.
     */
    static get selectorInputFrom() {
        return `${prefix}-date-picker-input,${prefix}-date-picker-input[kind="from"]`;
    }
    /**
     * A selector that will return the `<input>` to enter end date.
     */
    static get selectorInputTo() {
        return `${prefix}-date-picker-input[kind="to"]`;
    }
    /**
     * The name of the custom event when Flatpickr throws an error.
     */
    static get eventFlatpickrError() {
        return `${prefix}-date-picker-flatpickr-error`;
    }
    /**
     * The name of the custom event fired on this element when Flatpickr updates its value.
     */
    static get eventChange() {
        return `${prefix}-date-picker-changed`;
    }
};
/**
 * The CSS selector for Flatpickr's month/year portion.
 */
CDSDatePicker._selectorFlatpickrMonthYearContainer = '.flatpickr-current-month';
/**
 * The CSS selector for Flatpickr's year portion.
 */
CDSDatePicker._selectorFlatpickrYearContainer = '.numInputWrapper';
/**
 * The CSS selector for the inner element of Flatpickr's month portion.
 */
CDSDatePicker._selectorFlatpickrCurrentMonth = '.cur-month';
/**
 * The CSS selector for Flatpickr's month navigator.
 */
CDSDatePicker._selectorFlatpickrMonth = '.flatpickr-month';
/**
 * The CSS selector for Flatpickr's container of the weekdays.
 */
CDSDatePicker._selectorFlatpickrWeekdays = '.flatpickr-weekdays';
/**
 * The CSS selector for Flatpickr's container of the days.
 */
CDSDatePicker._selectorFlatpickrDays = '.flatpickr-days';
/**
 * The CSS selector applied to Flatpickr's each weekdays.
 */
CDSDatePicker._selectorFlatpickrWeekday = '.flatpickr-weekday';
/**
 * The CSS selector applied to Flatpickr's each days.
 */
CDSDatePicker._selectorFlatpickrDay = '.flatpickr-day';
/**
 * The CSS class for the inner element of Flatpickr's month portion.
 */
CDSDatePicker._classFlatpickrCurrentMonth = 'cur-month';
/**
 * The CSS class applied to Flatpickr's "today" highlight.
 */
CDSDatePicker._classFlatpickrToday = 'today';
/**
 * The CSS class applied to the "today" highlight if there are any dates selected.
 */
CDSDatePicker.classNoBorder = 'no-border';
/**
 * The default date format.
 */
CDSDatePicker.defaultDateFormat = 'm/d/Y';
/**
 * The default localization data.
 */
CDSDatePicker.defaultLocale = flatpickr.l10ns.default;
CDSDatePicker.styles = styles;
__decorate([
    query('#floating-menu-container')
], CDSDatePicker.prototype, "_floatingMenuContainerNode", void 0);
__decorate([
    HostListener('eventChange')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSDatePicker.prototype, "_handleChange", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'allow-input' })
], CDSDatePicker.prototype, "allowInput", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'close-on-select' })
], CDSDatePicker.prototype, "closeOnSelect", void 0);
__decorate([
    property({ attribute: 'date-format' })
], CDSDatePicker.prototype, "dateFormat", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePicker.prototype, "disabled", void 0);
__decorate([
    property({ attribute: false })
], CDSDatePicker.prototype, "locale", void 0);
__decorate([
    property({ attribute: 'enabled-range' })
], CDSDatePicker.prototype, "enabledRange", void 0);
__decorate([
    property({ attribute: 'min-date' })
], CDSDatePicker.prototype, "minDate", void 0);
__decorate([
    property({ attribute: 'max-date' })
], CDSDatePicker.prototype, "maxDate", void 0);
__decorate([
    property()
], CDSDatePicker.prototype, "name", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePicker.prototype, "open", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDatePicker.prototype, "readonly", void 0);
__decorate([
    property()
], CDSDatePicker.prototype, "value", null);
CDSDatePicker = __decorate([
    carbonElement(`${prefix}-date-picker`)
], CDSDatePicker);
var CDSDatePicker$1 = CDSDatePicker;

export { CDSDatePicker$1 as default };
//# sourceMappingURL=date-picker.js.map
