/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import Checkmark16 from '../../icons/checkmark/16.js';
import { prefix } from '../../globals/settings.js';
import './dropdown.js';
import styles from './dropdown.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { DROPDOWN_SIZE } from './defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Dropdown item.
 *
 * @element cds-dropdown-item
 * @csspart selected-icon The selected icon.
 */
let CDSDropdownItem = class CDSDropdownItem extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * `true` if this dropdown item should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if this dropdown item should be highlighted.
         * If `true`, parent `<dropdown>` selects/deselects this dropdown item upon keyboard interaction.
         *
         * @private
         */
        this.highlighted = false;
        /**
         * `true` if this dropdown item should be selected.
         *
         * @private
         */
        this.selected = false;
        /**
         * Dropdown size.
         */
        this.size = DROPDOWN_SIZE.MEDIUM;
        /**
         * The `value` attribute that is set to the parent `<cds-dropdown>` when this dropdown item is selected.
         */
        this.value = '';
    }
    connectedCallback() {
        super.connectedCallback();
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'option');
        }
        if (!this.hasAttribute('id')) {
            this.setAttribute('id', `${prefix}-dropdown-item-${this.constructor
                .id++}`);
        }
        this.setAttribute('aria-selected', String(this.selected));
    }
    render() {
        const { selected } = this;
        return html `
      <div class="${prefix}--list-box__menu-item__option">
        <slot></slot>
        ${!selected
            ? undefined
            : Checkmark16({
                part: 'selected-icon',
                class: `${prefix}--list-box__menu-item__selected-icon`,
            })}
      </div>
    `;
    }
};
/**
 * Store an identifier for use in composing this item's id.
 *
 * Auto-increments anytime a new dropdown-item appears.
 */
CDSDropdownItem.id = 0;
CDSDropdownItem.styles = styles;
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdownItem.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdownItem.prototype, "highlighted", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdownItem.prototype, "selected", void 0);
__decorate([
    property({ reflect: true })
], CDSDropdownItem.prototype, "size", void 0);
__decorate([
    property()
], CDSDropdownItem.prototype, "value", void 0);
CDSDropdownItem = __decorate([
    carbonElement(`${prefix}-dropdown-item`)
], CDSDropdownItem);
var CDSDropdownItem$1 = CDSDropdownItem;

export { CDSDropdownItem$1 as default };
//# sourceMappingURL=dropdown-item.js.map
