/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { html, LitElement } from 'lit';
import { state, query, property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import ChevronDown16 from '../../icons/chevron--down/16.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import FocusMixin from '../../globals/mixins/focus.js';
import FormMixin from '../../globals/mixins/form.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import ValidityMixin from '../../globals/mixins/validity.js';
import HostListener from '../../globals/decorators/host-listener.js';
import { forEach, indexOf, find } from '../../globals/internal/collection-helpers.js';
import { DROPDOWN_DIRECTION, DROPDOWN_SIZE, DROPDOWN_TYPE, DROPDOWN_KEYBOARD_ACTION, NAVIGATION_DIRECTION } from './defs.js';
export { DROPDOWN_DIRECTION, DROPDOWN_KEYBOARD_ACTION, DROPDOWN_SIZE, DROPDOWN_TYPE, NAVIGATION_DIRECTION } from './defs.js';
import styles from './dropdown.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Dropdown.
 *
 * @element cds-dropdown
 * @csspart label-text The label text.
 * @csspart helper-text The helper text.
 * @csspart trigger-button The trigger button.
 * @csspart menu-body The menu body.
 * @csspart validity-message The validity message.
 * @fires cds-dropdown-beingselected
 *   The custom event fired before a dropdown item is selected upon a user gesture.
 *   Cancellation of this event stops changing the user-initiated selection.
 * @fires cds-dropdown-beingtoggled
 *   The custom event fired before the open state of this dropdown is toggled upon a user gesture.
 *   Cancellation of this event stops the user-initiated toggling.
 * @fires cds-dropdown-selected - The custom event fired after a dropdown item is selected upon a user gesture.
 * @fires cds-dropdown-toggled - The custom event fired after the open state of this dropdown is toggled upon a user gesture.
 */
let CDSDropdown = class CDSDropdown extends ValidityMixin(HostListenerMixin(FormMixin(FocusMixin(LitElement)))) {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * The content of the selected item.
         */
        this._selectedItemContent = null;
        /**
         * `true` if the trigger button should be focusable.
         * Derived class can set `false` to this if the trigger button contains another primary focusable element (e.g. `<input>`).
         */
        this._shouldTriggerBeFocusable = true;
        /**
         * 'aria-label' of the ListBox component.
         * Specify a label to be read by screen readers on the container node
         */
        this.ariaLabel = '';
        /**
         * Specify the direction of the dropdown. Can be either top or bottom.
         */
        this.direction = DROPDOWN_DIRECTION.BOTTOM;
        /**
         * `true` if this dropdown should be disabled.
         */
        this.disabled = false;
        /**
         * The helper text.
         */
        this.helperText = '';
        /**
         * Specify whether the title text should be hidden or not
         */
        this.hideLabel = false;
        /**
         * `true` to show the UI of the invalid state.
         */
        this.invalid = false;
        /**
         * Message which is displayed if the value is invalid.
         */
        this.invalidText = '';
        /**
         * Provide the title text that will be read by a screen reader when visiting this control
         */
        this.titleText = '';
        /**
         * Name for the dropdown in the `FormData`
         */
        this.name = '';
        /**
         * `true` if this dropdown should be open.
         */
        this.open = false;
        /**
         * Whether or not the Dropdown is readonly
         */
        this.readOnly = false;
        /**
         * `true` if the value is required.
         */
        this.required = false;
        /**
         * The special validity message for `required`.
         */
        this.requiredValidityMessage = 'Please fill out this field.';
        /**
         * Dropdown size.
         */
        this.size = DROPDOWN_SIZE.MEDIUM;
        /**
         * The `aria-label` attribute for the UI indicating the closed state.
         */
        this.toggleLabelClosed = '';
        /**
         * The `aria-label` attribute for the UI indicating the open state.
         */
        this.toggleLabelOpen = '';
        /**
         * Generic label that will be used as the textual representation of what this field is for
         */
        this.label = '';
        /**
         * `true` if this dropdown should use the inline UI variant.
         */
        this.type = DROPDOWN_TYPE.DEFAULT;
        /**
         * The validity message.
         */
        this.validityMessage = '';
        /**
         * The value of the selected item.
         */
        this.value = '';
        /**
         * Specify whether the control is currently in warning state
         */
        this.warn = false;
        /**
         * Provide the text that is displayed when the control is in warning state
         */
        this.warnText = '';
    }
    /**
     * @param itemToSelect A dropdown item. Absense of this argument means clearing selection.
     * @returns `true` if the selection of this dropdown should change if the given item is selected upon user interaction.
     */
    _selectionShouldChange(itemToSelect) {
        return !itemToSelect || itemToSelect.value !== this.value;
    }
    /**
     * A callback that runs after change in dropdown selection upon user interaction is confirmed.
     *
     * @param itemToSelect
     *   A dropdown item.
     *   Absense of this argument means clearing selection, which may be handled by a derived class.
     */
    _selectionDidChange(itemToSelect) {
        if (itemToSelect) {
            this.value = itemToSelect.value;
            this._activeDescendant = itemToSelect.id;
            forEach(this.querySelectorAll(this.constructor.selectorItemSelected), (item) => {
                item.selected = false;
                item.setAttribute('aria-selected', 'false');
            });
            itemToSelect.selected = true;
            itemToSelect.setAttribute('aria-selected', 'true');
            this._handleUserInitiatedToggle(false);
        }
    }
    /**
     * Handles `click` event on the top-level element in the shadow DOM.
     *
     * @param event The event.
     */
    _handleClickInner(event) {
        if (this.readOnly) {
            return;
        }
        if (this.shadowRoot.contains(event.target)) {
            this._handleUserInitiatedToggle();
        }
        else {
            const item = event.target.closest(this.constructor.selectorItem);
            if (this.contains(item)) {
                this._handleUserInitiatedSelectItem(item);
            }
        }
    }
    /**
     * Handler for the `keydown` event on the top-level element in the shadow DOM.
     */
    _handleKeydownInner(event) {
        const { key } = event;
        const action = this.constructor.getAction(key);
        if (!this.open) {
            switch (action) {
                case DROPDOWN_KEYBOARD_ACTION.NAVIGATING:
                    this._handleUserInitiatedToggle(true);
                    // If this menu gets open with an arrow key, reset the highlight
                    this._clearHighlight();
                    break;
            }
        }
        else {
            switch (action) {
                case DROPDOWN_KEYBOARD_ACTION.CLOSING:
                    this._handleUserInitiatedToggle(false);
                    break;
                case DROPDOWN_KEYBOARD_ACTION.NAVIGATING:
                    this._navigate(NAVIGATION_DIRECTION[key]);
                    break;
            }
        }
    }
    /**
     * Handler for the `keypress` event on the top-level element in the shadow DOM.
     */
    _handleKeypressInner(event) {
        const { key } = event;
        const action = this.constructor.getAction(key);
        if (!this.open) {
            switch (action) {
                case DROPDOWN_KEYBOARD_ACTION.TRIGGERING:
                    this._handleUserInitiatedToggle(true);
                    break;
            }
        }
        else {
            switch (action) {
                case DROPDOWN_KEYBOARD_ACTION.TRIGGERING:
                    {
                        const constructor = this.constructor;
                        const highlightedItem = this.querySelector(constructor.selectorItemHighlighted);
                        if (highlightedItem) {
                            this._handleUserInitiatedSelectItem(highlightedItem);
                        }
                        else {
                            this._handleUserInitiatedToggle(false);
                        }
                    }
                    break;
            }
        }
    }
    /**
     * Handles `blur` event handler on the document this element is in.
     *
     * @param event The event.
     */
    _handleFocusOut(event) {
        if (!this.contains(event.relatedTarget)) {
            this._handleUserInitiatedToggle(false);
        }
    }
    /**
     * Handles `slotchange` event for the `<slot>` for helper text.
     */
    _handleSlotchangeHelperText() {
        this.requestUpdate();
    }
    /**
     * Handles `slotchange` event for the `<slot>` for label text.
     */
    _handleSlotchangeLabelText() {
        this.requestUpdate();
    }
    /**
     * Handles `slotchange` event.
     */
    _handleAILabelSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove reference to slug in v12
                elem.matches(this.constructor.slugItem)
            : false);
        this._hasAILabel = Boolean(hasContent);
        hasContent[0].setAttribute('size', 'mini');
        this.requestUpdate();
    }
    /**
     * Handles user-initiated selection of a dropdown item.
     *
     * @param [item] The dropdown item user wants to select. Absense of this argument means clearing selection.
     */
    _handleUserInitiatedSelectItem(item) {
        if (item === null || item === void 0 ? void 0 : item.hasAttribute('disabled')) {
            return;
        }
        if (this._selectionShouldChange(item)) {
            const init = {
                bubbles: true,
                composed: true,
                detail: {
                    item,
                },
            };
            const constructor = this.constructor;
            const beforeSelectEvent = new CustomEvent(constructor.eventBeforeSelect, Object.assign(Object.assign({}, init), { cancelable: true }));
            if (this.dispatchEvent(beforeSelectEvent)) {
                this._selectionDidChange(item);
                const afterSelectEvent = new CustomEvent(constructor.eventSelect, init);
                this.dispatchEvent(afterSelectEvent);
            }
        }
    }
    /**
     * Handles user-initiated toggling the open state.
     *
     * @param [force] If specified, forces the open state to the given one.
     */
    _handleUserInitiatedToggle(force = !this.open) {
        const { eventBeforeToggle, eventToggle } = this
            .constructor;
        const { disabled } = this;
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                open: force,
            },
        };
        if (!disabled) {
            if (this.dispatchEvent(new CustomEvent(eventBeforeToggle, init))) {
                this.open = force;
                if (!this.open) {
                    forEach(this.querySelectorAll(this.constructor.selectorItemHighlighted), (item) => {
                        item.highlighted = false;
                    });
                }
                this.requestUpdate();
                this.dispatchEvent(new CustomEvent(eventToggle, init));
            }
        }
    }
    /**
     * Clears the selection of dropdown items.
     */
    _clearHighlight() {
        forEach(this.querySelectorAll(this.constructor.selectorItem), (item) => {
            item.highlighted = false;
        });
    }
    /**
     * Navigate through dropdown items.
     *
     * @param direction `-1` to navigate backward, `1` to navigate forward.
     */
    _navigate(direction) {
        var _a;
        const constructor = this.constructor;
        const items = this.querySelectorAll(constructor.selectorItem);
        const highlightedItem = this.querySelector(constructor.selectorItemHighlighted);
        const highlightedIndex = indexOf(items, highlightedItem);
        let nextIndex = highlightedIndex + direction;
        if ((_a = items[nextIndex]) === null || _a === void 0 ? void 0 : _a.hasAttribute('disabled')) {
            nextIndex += direction;
        }
        if (nextIndex < 0) {
            nextIndex = items.length - 1;
        }
        if (nextIndex >= items.length) {
            nextIndex = 0;
        }
        forEach(items, (item, i) => {
            item.highlighted = i === nextIndex;
        });
        const nextItem = items[nextIndex];
        // Using `{ block: 'nearest' }` to prevent scrolling unless scrolling is absolutely necessary.
        // `scrollIntoViewOptions` seems to work in latest Safari despite of MDN/caniuse table.
        // IE falls back to the old behavior.
        nextItem.scrollIntoView({ block: 'nearest' });
        const nextItemId = nextItem.id;
        if (nextItemId) {
            this._activeDescendant = nextItemId;
        }
    }
    /* eslint-disable class-methods-use-this */
    /**
     * @returns The content preceding the trigger button.
     */
    _renderPrecedingLabel() {
        return undefined;
    }
    /* eslint-enable class-methods-use-this */
    /**
     * @returns The main content of the trigger button.
     */
    _renderLabel() {
        const { label, _selectedItemContent: selectedItemContent } = this;
        return html `
      <span id="trigger-label" class="${prefix}--list-box__label"
        >${selectedItemContent || label}</span
      >
    `;
    }
    /**
     * @returns The title label.
     */
    _renderTitleLabel() {
        const { disabled, hideLabel, titleText, _slotTitleTextNode: slotTitleTextNode, _handleSlotchangeLabelText: handleSlotchangeLabelText, } = this;
        const labelClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--label--disabled`]: disabled,
            [`${prefix}--visually-hidden`]: hideLabel,
        });
        const hasTitleText = titleText ||
            (slotTitleTextNode && slotTitleTextNode.assignedNodes().length > 0);
        return html `
      <label
        id="dropdown-label"
        part="title-text"
        class="${labelClasses}"
        for="trigger-button"
        ?hidden="${!hasTitleText}">
        <slot name="title-text" @slotchange="${handleSlotchangeLabelText}"
          >${titleText}</slot
        >
      </label>
    `;
    }
    /* eslint-disable class-methods-use-this */
    /**
     * @returns The content following the trigger button.
     */
    _renderFollowingLabel() {
        return undefined;
    }
    /* eslint-enable class-methods-use-this */
    /**
     * Handles event to include selected value on the parent form.
     *
     * @param event The event.
     */
    _handleFormdata(event) {
        const { formData } = event; // TODO: Wait for `FormDataEvent` being available in `lib.dom.d.ts`
        const { disabled, name, value } = this;
        if (!disabled) {
            formData.append(name, value);
        }
    }
    shouldUpdate(changedProperties) {
        const { selectorItem } = this.constructor;
        if (changedProperties.has('size')) {
            forEach(this.querySelectorAll(selectorItem), (elem) => {
                elem.size = this.size;
            });
        }
        if (changedProperties.has('disabled') && this.disabled) {
            const { disabled } = this;
            // Propagate `disabled` attribute to descendants until `:host-context()` gets supported in all major browsers
            forEach(this.querySelectorAll(selectorItem), (elem) => {
                if (disabled) {
                    elem.disabled = disabled;
                }
                else {
                    elem.removeAttribute('disabled');
                }
            });
        }
        if (changedProperties.has('value')) {
            // `<cds-multi-select>` updates selection beforehand
            // because our rendering logic for `<cds-multi-select>` looks for selected items via `qSA()`
            forEach(this.querySelectorAll(selectorItem), (elem) => {
                elem.selected =
                    elem.value === this.value;
            });
            const item = find(this.querySelectorAll(selectorItem), (elem) => elem.value === this.value);
            if (item) {
                const range = this.ownerDocument.createRange();
                range.selectNodeContents(item);
                this._selectedItemContent = range.cloneContents();
            }
            else {
                this._selectedItemContent = null;
            }
        }
        return true;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    updated(_changedProperties) {
        var _a, _b, _c, _d, _e;
        this._hasAILabel
            ? this.setAttribute('ai-label', '')
            : this.removeAttribute('ai-label');
        const label = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector("slot[name='ai-label']");
        if (label) {
            label === null || label === void 0 ? void 0 : label.classList.toggle(`${prefix}--slug--revert`, (_b = this.querySelector(`${prefix}-ai-label`)) === null || _b === void 0 ? void 0 : _b.hasAttribute('revert-active'));
        }
        else {
            (_d = (_c = this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector("slot[name='slug']")) === null || _d === void 0 ? void 0 : _d.classList.toggle(`${prefix}--slug--revert`, (_e = this.querySelector(`${prefix}-slug`)) === null || _e === void 0 ? void 0 : _e.hasAttribute('revert-active'));
        }
    }
    /**
     * The CSS class list for dropdown listbox
     */
    get _classes() {
        const { disabled, size, type, invalid, open, warn } = this;
        const inline = type === DROPDOWN_TYPE.INLINE;
        const selectedItemsCount = this.querySelectorAll(this.constructor.selectorItemSelected).length;
        return classMap({
            [`${prefix}--dropdown`]: true,
            [`${prefix}--list-box`]: true,
            [`${prefix}--list-box--disabled`]: disabled,
            [`${prefix}--list-box--inline`]: inline,
            [`${prefix}--list-box--expanded`]: open,
            [`${prefix}--list-box--${size}`]: size,
            [`${prefix}--dropdown--invalid`]: invalid,
            [`${prefix}--dropdown--warn`]: warn,
            [`${prefix}--dropdown--inline`]: inline,
            [`${prefix}--dropdown--selected`]: selectedItemsCount > 0,
        });
    }
    render() {
        var _a;
        const { ariaLabel, _classes: classes, disabled, helperText, invalid, invalidText, open, toggleLabelClosed, toggleLabelOpen, type, warn, warnText, _activeDescendant: activeDescendant, _shouldTriggerBeFocusable: shouldTriggerBeFocusable, _handleClickInner: handleClickInner, _handleKeydownInner: handleKeydownInner, _handleKeypressInner: handleKeypressInner, _handleSlotchangeHelperText: handleSlotchangeHelperText, _handleAILabelSlotChange: handleAILabelSlotChange, _slotHelperTextNode: slotHelperTextNode, } = this;
        const inline = type === DROPDOWN_TYPE.INLINE;
        let activeDescendantFallback;
        if (open && !activeDescendant) {
            const constructor = this.constructor;
            const items = this.querySelectorAll(constructor.selectorItem);
            activeDescendantFallback = (_a = items[0]) === null || _a === void 0 ? void 0 : _a.id;
        }
        const helperClasses = classMap({
            [`${prefix}--form__helper-text`]: true,
            [`${prefix}--form__helper-text--disabled`]: disabled,
        });
        const iconContainerClasses = classMap({
            [`${prefix}--list-box__menu-icon`]: true,
            [`${prefix}--list-box__menu-icon--open`]: open,
        });
        const toggleLabel = (open ? toggleLabelOpen : toggleLabelClosed) || undefined;
        const hasHelperText = helperText ||
            invalidText ||
            warnText ||
            (slotHelperTextNode && slotHelperTextNode.assignedNodes().length > 0);
        const validityIcon = !invalid
            ? undefined
            : WarningFilled16({
                class: `${prefix}--list-box__invalid-icon`,
                'aria-label': toggleLabel,
            });
        const warningIcon = !warn || (invalid && warn)
            ? undefined
            : WarningAltFilled16({
                class: `${prefix}--list-box__invalid-icon ${prefix}--list-box__invalid-icon--warning`,
                'aria-label': toggleLabel,
            });
        const helperMessage = invalid ? invalidText : warn ? warnText : helperText;
        const menuBody = html `
      <div
        aria-labelledby="${ifDefined(ariaLabel ? undefined : 'dropdown-label')}"
        aria-label="${ifDefined(ariaLabel ? ariaLabel : undefined)}"
        id="menu-body"
        part="menu-body"
        class="${prefix}--list-box__menu"
        role="listbox"
        tabindex="-1"
        ?hidden=${!open}>
        <slot></slot>
      </div>
    `;
        return html `
      ${this._renderTitleLabel()}
      <div
        class="${classes}"
        ?data-invalid=${invalid}
        @click=${handleClickInner}
        @keydown=${handleKeydownInner}
        @keypress=${handleKeypressInner}>
        <div
          id="${ifDefined(!shouldTriggerBeFocusable ? undefined : 'trigger-button')}"
          class="${prefix}--list-box__field"
          part="trigger-button"
          tabindex="${ifDefined(!shouldTriggerBeFocusable ? undefined : '0')}"
          role="${ifDefined(!shouldTriggerBeFocusable ? undefined : 'combobox')}"
          aria-labelledby="${ifDefined(!shouldTriggerBeFocusable ? undefined : 'dropdown-label')}"
          aria-expanded="${ifDefined(!shouldTriggerBeFocusable ? undefined : String(open))}"
          aria-haspopup="${ifDefined(!shouldTriggerBeFocusable ? undefined : 'listbox')}"
          aria-controls="${ifDefined(!shouldTriggerBeFocusable ? undefined : 'menu-body')}"
          aria-activedescendant="${ifDefined(!shouldTriggerBeFocusable
            ? undefined
            : open
                ? (activeDescendant !== null && activeDescendant !== void 0 ? activeDescendant : activeDescendantFallback)
                : '')}">
          ${this._renderPrecedingLabel()}${this._renderLabel()}${validityIcon}${warningIcon}${this._renderFollowingLabel()}
          <div id="trigger-caret" class="${iconContainerClasses}">
            ${ChevronDown16({ 'aria-label': toggleLabel })}
          </div>
        </div>
        <slot name="ai-label" @slotchange=${handleAILabelSlotChange}></slot>
        <slot name="slug" @slotchange=${handleAILabelSlotChange}></slot>
        ${menuBody}
      </div>
      <div
        part="helper-text"
        class="${helperClasses}"
        ?hidden="${(inline && !warn && !invalid) || !hasHelperText}">
        <slot name="helper-text" @slotchange="${handleSlotchangeHelperText}"
          >${helperMessage}</slot
        >
      </div>
    `;
    }
    /**
     * A selector that will return highlighted items.
     */
    static get selectorItemHighlighted() {
        return `${prefix}-dropdown-item[highlighted]`;
    }
    /**
     * A selector that will return dropdown items.
     */
    static get selectorItem() {
        return `${prefix}-dropdown-item`;
    }
    /**
     * A selector that will return selected items.
     */
    static get selectorItemSelected() {
        return `${prefix}-dropdown-item[selected]`;
    }
    /**
     * The name of the custom event fired before a dropdown item is selected upon a user gesture.
     * Cancellation of this event stops changing the user-initiated selection.
     */
    static get eventBeforeSelect() {
        return `${prefix}-dropdown-beingselected`;
    }
    /**
     * The name of the custom event fired after a a dropdown item is selected upon a user gesture.
     */
    static get eventSelect() {
        return `${prefix}-dropdown-selected`;
    }
    /**
     * The name of the custom event fired before this dropdown item is being toggled upon a user gesture.
     * Cancellation of this event stops the user-initiated action of toggling this dropdown item.
     */
    static get eventBeforeToggle() {
        return `${prefix}-dropdown-beingtoggled`;
    }
    /**
     * The name of the custom event fired after this dropdown item is toggled upon a user gesture.
     */
    static get eventToggle() {
        return `${prefix}-dropdown-toggled`;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
    /**
     * @returns A action for dropdown for the given key symbol.
     */
    static getAction(key) {
        if (key === 'Escape') {
            return DROPDOWN_KEYBOARD_ACTION.CLOSING;
        }
        if (key in NAVIGATION_DIRECTION) {
            return DROPDOWN_KEYBOARD_ACTION.NAVIGATING;
        }
        if (this.TRIGGER_KEYS.has(key)) {
            return DROPDOWN_KEYBOARD_ACTION.TRIGGERING;
        }
        return DROPDOWN_KEYBOARD_ACTION.NONE;
    }
};
/**
 * Symbols of keys that triggers opening/closing menu and selecting/deselecting menu item.
 */
CDSDropdown.TRIGGER_KEYS = new Set([' ', 'Enter']);
CDSDropdown.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSDropdown.styles = styles;
__decorate([
    state()
], CDSDropdown.prototype, "_activeDescendant", void 0);
__decorate([
    query(`.${prefix}--list-box`)
], CDSDropdown.prototype, "_listBoxNode", void 0);
__decorate([
    query('slot[name="helper-text"]')
], CDSDropdown.prototype, "_slotHelperTextNode", void 0);
__decorate([
    query('slot[name="title-text"]')
], CDSDropdown.prototype, "_slotTitleTextNode", void 0);
__decorate([
    HostListener('focusout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSDropdown.prototype, "_handleFocusOut", null);
__decorate([
    property({ type: String, reflect: true, attribute: 'aria-label' })
], CDSDropdown.prototype, "ariaLabel", void 0);
__decorate([
    property({ type: String, reflect: true })
], CDSDropdown.prototype, "direction", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdown.prototype, "disabled", void 0);
__decorate([
    property({ attribute: 'helper-text' })
], CDSDropdown.prototype, "helperText", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'hide-label' })
], CDSDropdown.prototype, "hideLabel", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdown.prototype, "invalid", void 0);
__decorate([
    property({ attribute: 'invalid-text' })
], CDSDropdown.prototype, "invalidText", void 0);
__decorate([
    property({ attribute: 'title-text' })
], CDSDropdown.prototype, "titleText", void 0);
__decorate([
    property()
], CDSDropdown.prototype, "name", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdown.prototype, "open", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'read-only' })
], CDSDropdown.prototype, "readOnly", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdown.prototype, "required", void 0);
__decorate([
    property({ attribute: 'required-validity-message' })
], CDSDropdown.prototype, "requiredValidityMessage", void 0);
__decorate([
    property({ reflect: true })
], CDSDropdown.prototype, "size", void 0);
__decorate([
    property({ attribute: 'toggle-label-closed' })
], CDSDropdown.prototype, "toggleLabelClosed", void 0);
__decorate([
    property({ attribute: 'toggle-label-open' })
], CDSDropdown.prototype, "toggleLabelOpen", void 0);
__decorate([
    property({ attribute: 'label' })
], CDSDropdown.prototype, "label", void 0);
__decorate([
    property({ reflect: true })
], CDSDropdown.prototype, "type", void 0);
__decorate([
    property({ attribute: 'validity-message' })
], CDSDropdown.prototype, "validityMessage", void 0);
__decorate([
    property({ reflect: true })
], CDSDropdown.prototype, "value", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSDropdown.prototype, "warn", void 0);
__decorate([
    property({ attribute: 'warn-text' })
], CDSDropdown.prototype, "warnText", void 0);
CDSDropdown = __decorate([
    carbonElement(`${prefix}-dropdown`)
], CDSDropdown);
var CDSDropdown$1 = CDSDropdown;

export { CDSDropdown$1 as default };
//# sourceMappingURL=dropdown.js.map
