/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { prefix } from '../../globals/settings.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import './file-uploader.js';
import './file-uploader-item.js';
import './file-uploader-drop-container.js';
import './file-uploader-button.js';
import './file-uploader-skeleton.js';
import '../button/button.js';
import { BUTTON_SIZE } from '../button/defs.js';
import { FILE_UPLOADER_ITEM_STATE } from './defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * A class to manage file uploading states, like sending file contents to server.
 * DEMONSTRATION-PURPOSE ONLY.
 * Data/state handling in file uploading tends to involve lots of application-specific logics
 * and thus abstracting everything in a library won't be a good return on investment
 * vs. letting users copy code here and implement features that fit their needs.
 */
let CDSCEDemoFileUploader = class CDSCEDemoFileUploader extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * The files being uploaded.
         */
        this._files = [];
        /**
         * The file types the file input should accept, separated by space.
         */
        this.accept = '';
        /**
         * `true` if the drop container should be disabled.
         */
        this.button = false;
        /**
         * Button kind.
         */
        this.buttonKind = 'primary';
        /**
         * Button label.
         */
        this.buttonLabel = 'Add file';
        /**
         * `true` if the drop container should be disabled.
         */
        this.disabled = false;
        /**
         * Icon description.
         */
        this.iconDescription = '';
        /**
         * The input name.
         */
        this.inputName = '';
        /**
         * The label description text.
         */
        this.labelDescription = '';
        /**
         * The label title.
         */
        this.labelTitle = '';
        /**
         * `true` if the drop container should accept more than one files at once.
         * Note that even with `false` set here, user _can_ select multiple files one by one.
         */
        this.multiple = false;
        /**
         * The size of the button item.
         */
        this.size = BUTTON_SIZE.MEDIUM;
        /**
         * The state of this file uploader item.
         */
        this.inputState = '';
    }
    /**
     * Handles `cds-drop-container-changed` on `<cds-file-drop-container>`.
     *
     * @param event The event.
     */
    _handleChange(event) {
        const { addedFiles } = event.detail;
        const newFiles = addedFiles.map((item) => ({
            id: Math.random().toString(36).slice(2),
            file: item,
            state: FILE_UPLOADER_ITEM_STATE.UPLOADING,
        }));
        const { multiple, _files: files, _simulateUpload: simulateUpload } = this;
        if (multiple) {
            this._files = files.concat(newFiles);
            this.requestUpdate();
            newFiles.forEach(simulateUpload, this);
        }
        else if (addedFiles.length > 0) {
            this._files = files.concat(newFiles[0]);
            this.requestUpdate();
            this._simulateUpload(newFiles[0]);
        }
    }
    /**
     * Handles `cds-file-uploader-item-deleted` on `<cds-file-uploader-item>`.
     *
     * @param event The event.
     */
    _handleDelete(event) {
        const { fileId: idToDelete } = event.target.dataset;
        this._files = this._files.filter(({ id }) => idToDelete !== id);
        this.requestUpdate();
    }
    /**
     * Simulates updating file.
     *
     * @param data The data of the file being uploaded.
     */
    async _simulateUpload(data) {
        const { id, file } = data;
        if (file.size > 524288) {
            this._files = this._files.map((item) => id !== item.id
                ? item
                : Object.assign(Object.assign({}, item), { state: FILE_UPLOADER_ITEM_STATE.EDIT, invalid: true, errorSubject: 'File size exceeds limit', errorBody: '500kb max file size. Select a new file and try again.' }));
            this.requestUpdate();
        }
        else {
            // Simulates network request time
            const rand = Math.random() * 1000;
            await new Promise((t) => setTimeout(t, rand));
            this._files = this._files.map((item) => id !== item.id
                ? item
                : Object.assign(Object.assign({}, item), { state: FILE_UPLOADER_ITEM_STATE.COMPLETE }));
            this.requestUpdate();
            // Shows x icon after 1 second
            await new Promise((t) => setTimeout(t, 1000));
            this._files = this._files.map((item) => id !== item.id
                ? item
                : Object.assign(Object.assign({}, item), { state: FILE_UPLOADER_ITEM_STATE.EDIT }));
            this.requestUpdate();
        }
    }
    render() {
        const { accept, button, buttonKind, buttonLabel, disabled, labelDescription, labelTitle, multiple, size, inputState, iconDescription, _files: files, _handleChange: handleChange, _handleDelete: handleDelete, } = this;
        return html `
      <cds-file-uploader
        label-description="${ifDefined(labelDescription)}"
        label-title="${ifDefined(labelTitle)}"
        ?disabled="${disabled}">
        ${!button
            ? html ` <cds-file-uploader-drop-container
              accept="${ifDefined(accept)}"
              ?multiple="${multiple}"
              name="${ifDefined(this.inputName)}"
              @cds-file-uploader-drop-container-changed="${handleChange}">
              Drag and drop files here or click to upload
            </cds-file-uploader-drop-container>`
            : html ` <cds-file-uploader-button
              size="${ifDefined(size)}"
              button-kind="${buttonKind}"
              accept="${ifDefined(accept)}"
              name="${ifDefined(this.inputName)}"
              ?multiple="${multiple}"
              @cds-file-uploader-button-changed="${handleChange}">
              ${buttonLabel}
            </cds-file-uploader-button>`}
        ${files.map(({ id, invalid, file, state, errorSubject, errorBody }) => html `
            <cds-file-uploader-item
              data-file-id="${id}"
              ?invalid="${invalid}"
              state="${inputState || ifDefined(state)}"
              icon-description="${ifDefined(iconDescription)}"
              error-subject="${ifDefined(errorSubject)}"
              error-body="${ifDefined(errorBody)}"
              @cds-file-uploader-item-deleted="${handleDelete}">
              ${file.name}
            </cds-file-uploader-item>
          `)}
      </cds-file-uploader>
    `;
    }
};
__decorate([
    property()
], CDSCEDemoFileUploader.prototype, "accept", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCEDemoFileUploader.prototype, "button", void 0);
__decorate([
    property({ attribute: 'button-kind' })
], CDSCEDemoFileUploader.prototype, "buttonKind", void 0);
__decorate([
    property({ attribute: 'button-label' })
], CDSCEDemoFileUploader.prototype, "buttonLabel", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCEDemoFileUploader.prototype, "disabled", void 0);
__decorate([
    property({ attribute: 'icon-description' })
], CDSCEDemoFileUploader.prototype, "iconDescription", void 0);
__decorate([
    property({ attribute: 'input-name' })
], CDSCEDemoFileUploader.prototype, "inputName", void 0);
__decorate([
    property({ attribute: 'label-description' })
], CDSCEDemoFileUploader.prototype, "labelDescription", void 0);
__decorate([
    property({ attribute: 'label-title' })
], CDSCEDemoFileUploader.prototype, "labelTitle", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSCEDemoFileUploader.prototype, "multiple", void 0);
__decorate([
    property({ reflect: true })
], CDSCEDemoFileUploader.prototype, "size", void 0);
__decorate([
    property({ reflect: true, attribute: 'input-state' })
], CDSCEDemoFileUploader.prototype, "inputState", void 0);
CDSCEDemoFileUploader = __decorate([
    carbonElement(`${prefix}-ce-demo-file-uploader`)
], CDSCEDemoFileUploader);
var CDSCEDemoFileUploader$1 = CDSCEDemoFileUploader;

export { CDSCEDemoFileUploader$1 as default };
//# sourceMappingURL=demo-file-uploader.js.map
