/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { prefix } from '../../globals/settings.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import ifNonEmpty from '../../globals/directives/if-non-empty.js';
import styles from './file-uploader.scss.js';
import { BUTTON_KIND, BUTTON_SIZE } from '../button/defs.js';
export { FORM_ELEMENT_COLOR_SCHEME as TILE_COLOR_SCHEME } from '../../globals/shared-enums.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2020, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * File uploader button .
 *
 * @element cds-file-uploader-container
 * @fires cds-file-uploader-button-changed The custom event fired when there is a user gesture to select files to upload.
 */
let CDSFileUploaderButton = class CDSFileUploaderButton extends HostListenerMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * Button kind.
         */
        this.buttonKind = BUTTON_KIND.PRIMARY;
        /**
         * Button size.
         */
        this.size = BUTTON_SIZE.MEDIUM;
        /**
         * The file types the file input should accept, separated by space.
         */
        this.accept = '';
        /**
         * `true` if this drop container should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if this drop container should accept more than one files at once.
         * Note that even with `false` set here, user _can_ select multiple files one by one.
         */
        this.multiple = false;
        /**
         * The name of the input.
         */
        this.name = '';
        /**
         * The shadow DOM slot to put this drop container in.
         */
        this.slot = 'drop-container';
    }
    /**
     * Handles `click` event on the button.
     */
    _handleClick(event) {
        var _a, _b, _c;
        event.target.value = null;
        const { selectorInput } = this.constructor;
        (_b = (_a = this === null || this === void 0 ? void 0 : this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(selectorInput)) === null || _b === void 0 ? void 0 : _b.setAttribute('value', '');
        ((_c = this === null || this === void 0 ? void 0 : this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector(selectorInput)).click();
    }
    /**
     * Handles `keydown` event on the button.
     */
    _handleKeyDown(event) {
        var _a, _b, _c;
        const { selectorInput } = this.constructor;
        if (event.key === 'Enter' || event.key === 'Space') {
            (_b = (_a = this === null || this === void 0 ? void 0 : this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(selectorInput)) === null || _b === void 0 ? void 0 : _b.setAttribute('value', '');
            ((_c = this === null || this === void 0 ? void 0 : this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector(selectorInput)).click();
        }
    }
    /**
     * Handles user gesture to select files to upload.
     *
     * @param event The event.
     */
    _handleChange(event) {
        var _a;
        const addedFiles = this._getFiles(event);
        const { eventChange, selectorInput } = this
            .constructor;
        this.dispatchEvent(new CustomEvent(eventChange, {
            bubbles: true,
            composed: true,
            detail: {
                addedFiles,
            },
        }));
        const fileInput = (_a = this === null || this === void 0 ? void 0 : this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(selectorInput);
        if (fileInput) {
            fileInput.value = '';
        }
    }
    /**
     * @param event The event.
     * @returns The list of files user chose to upload.
     */
    _getFiles(event) {
        var _a;
        const { files } = (_a = (event.type === 'drop'
            ? event.dataTransfer
            : event.target)) !== null && _a !== void 0 ? _a : {};
        const { accept } = this;
        if (!accept || !/^(change|drop)$/.test(event.type)) {
            return Array.from(files !== null && files !== void 0 ? files : []);
        }
        const acceptedTypes = new Set(accept.split(' '));
        return Array.prototype.filter.call(files, ({ name, type: mimeType = '' }) => {
            var _a;
            const fileExtensionRegExp = /\.[^.]+$/;
            const hasFileExtension = fileExtensionRegExp.test(name);
            const [fileExtension] = !hasFileExtension
                ? [undefined]
                : ((_a = fileExtensionRegExp.exec(name)) !== null && _a !== void 0 ? _a : []);
            return (acceptedTypes.has(mimeType) ||
                (fileExtension && acceptedTypes.has(fileExtension)));
        });
    }
    render() {
        const { accept, disabled, multiple, name, buttonKind, size, _handleChange: handleChange, } = this;
        const labelClasses = classMap({
            [`${prefix}--file-browse-btn`]: true,
            [`${prefix}--file-browse-btn--disabled`]: disabled,
        });
        const buttonClasses = classMap({
            [`${prefix}--btn`]: true,
            [`${prefix}--btn--${buttonKind}`]: buttonKind,
            [`${prefix}--layout--size-${size}`]: size,
            [`${prefix}--btn--disabled`]: disabled,
            [`${prefix}--btn--${size}`]: size,
        });
        return html `
      <button
        type="button"
        class="${buttonClasses}"
        @click="${this._handleClick}"
        @keydown="${this._handleKeyDown}">
        <slot></slot>
      </button>
      <label class="${labelClasses}" for="file"> </label>
      <input
        id="file"
        type="file"
        class="${prefix}--file-input"
        tabindex="-1"
        accept="${ifNonEmpty(accept)}"
        ?disabled="${disabled}"
        ?multiple="${multiple}"
        name="${ifNonEmpty(name)}"
        @change="${handleChange}" />
    `;
    }
    /**
     * The name of the custom event fired when there is a user gesture to select files to upload.
     */
    static get eventChange() {
        return `${prefix}-file-uploader-button-changed`;
    }
    /**
     * A selector that will return the file `<input>`.
     */
    static get selectorInput() {
        return `.${prefix}--file-input`;
    }
};
CDSFileUploaderButton.styles = styles;
__decorate([
    property({ reflect: true, attribute: 'button-kind' })
], CDSFileUploaderButton.prototype, "buttonKind", void 0);
__decorate([
    property({ reflect: true })
], CDSFileUploaderButton.prototype, "size", void 0);
__decorate([
    property()
], CDSFileUploaderButton.prototype, "accept", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSFileUploaderButton.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSFileUploaderButton.prototype, "multiple", void 0);
__decorate([
    property({ reflect: true })
], CDSFileUploaderButton.prototype, "name", void 0);
__decorate([
    property({ reflect: true })
], CDSFileUploaderButton.prototype, "slot", void 0);
CDSFileUploaderButton = __decorate([
    carbonElement(`${prefix}-file-uploader-button`)
], CDSFileUploaderButton);
var CDSFileUploaderButton$1 = CDSFileUploaderButton;

export { CDSFileUploaderButton$1 as default };
//# sourceMappingURL=file-uploader-button.js.map
