/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import ifNonEmpty from '../../globals/directives/if-non-empty.js';
import styles from './file-uploader.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
export { FORM_ELEMENT_COLOR_SCHEME as TILE_COLOR_SCHEME } from '../../globals/shared-enums.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2020, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * The value to set to `event.dataTransfer.dropEffect`, keyed by the event nane.
 */
const dropEffects = {
    dragover: 'copy',
    dragleave: 'move',
};
/**
 * File uploader drop container.
 *
 * @element cds-file-uploader-drop-container
 * @fires cds-file-uploader-drop-container-changed The custom event fired when there is a user gesture to select files to upload.
 */
let CDSFileUploaderDropContainer = class CDSFileUploaderDropContainer extends HostListenerMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` to show the active state of this UI.
         */
        this._active = false;
        /**
         * The file types the file input should accept, separated by space.
         */
        this.accept = '';
        /**
         * `true` if this drop container should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if this drop container should accept more than one files at once.
         * Note that even with `false` set here, user _can_ select multiple files one by one.
         */
        this.multiple = false;
        /**
         * The name of the input.
         */
        this.name = '';
        /**
         * The shadow DOM slot to put this drop container in.
         */
        this.slot = 'drop-container';
    }
    /**
     * Handles user gesture to select files to upload.
     *
     * @param event The event.
     */
    _handleChange(event) {
        var _a, _b;
        const { eventChange, selectorInput } = this
            .constructor;
        const { files } = (_a = (event.type === 'drop'
            ? event.dataTransfer
            : event.target)) !== null && _a !== void 0 ? _a : {};
        const addedFiles = this._getFiles(event, files);
        this.dispatchEvent(new CustomEvent(eventChange, {
            bubbles: true,
            composed: true,
            detail: {
                addedFiles,
            },
        }));
        const fileInput = (_b = this === null || this === void 0 ? void 0 : this.shadowRoot) === null || _b === void 0 ? void 0 : _b.querySelector(selectorInput);
        if (fileInput) {
            fileInput.value = ''; // carbon-web-components#904
        }
    }
    /**
     * Handles `dragover`, `dragleave` and `drop` events.
     *
     * @param event The event.
     */
    _handleDrag(event) {
        event.preventDefault(); // Prevents page navigation upon dropping
        if (this.disabled) {
            return;
        }
        const { dataTransfer, type } = event;
        const dropEffect = dropEffects[type];
        if (dataTransfer && dropEffect) {
            dataTransfer.dropEffect = dropEffect;
        }
        this._active = type === 'dragover';
        if (type === 'drop') {
            this._handleChange(event);
        }
        this.requestUpdate();
    }
    /**
     * @param event The event.
     * @returns The list of files user chose to upload.
     */
    _getFiles(event, files) {
        const { accept } = this;
        if (!accept || !/^(change|drop)$/.test(event.type)) {
            return Array.from(files !== null && files !== void 0 ? files : []);
        }
        const acceptedTypes = new Set(accept.split(' '));
        return Array.prototype.filter.call(files, ({ name, type: mimeType = '' }) => {
            var _a;
            const fileExtensionRegExp = /\.[^.]+$/;
            const hasFileExtension = fileExtensionRegExp.test(name);
            const [fileExtension] = !hasFileExtension
                ? [undefined]
                : ((_a = fileExtensionRegExp.exec(name)) !== null && _a !== void 0 ? _a : []);
            return (acceptedTypes.has(mimeType) ||
                (fileExtension && acceptedTypes.has(fileExtension)));
        });
    }
    render() {
        const { accept, disabled, multiple, name, _active: active, _handleChange: handleChange, } = this;
        const labelClasses = classMap({
            [`${prefix}--file-browse-btn`]: true,
            [`${prefix}--file-browse-btn--disabled`]: disabled,
        });
        const dropareaClasses = classMap({
            [`${prefix}--file__drop-container`]: true,
            [`${prefix}--file__drop-container--drag-over`]: active,
        });
        return html `
      <label class="${labelClasses}" for="file" tabindex="0">
        <div class="${dropareaClasses}" role="button">
          <slot></slot>
          <input
            id="file"
            type="file"
            name="${ifNonEmpty(name)}"
            class="${prefix}--file-input"
            tabindex="-1"
            accept="${ifNonEmpty(accept)}"
            ?disabled="${disabled}"
            ?multiple="${multiple}"
            @change="${handleChange}" />
        </div>
      </label>
    `;
    }
    /**
     * The name of the custom event fired when there is a user gesture to select files to upload.
     */
    static get eventChange() {
        return `${prefix}-file-uploader-drop-container-changed`;
    }
    /**
     * A selector that will return the file `<input>`.
     */
    static get selectorInput() {
        return `.${prefix}--file-input`;
    }
};
CDSFileUploaderDropContainer.styles = styles;
__decorate([
    HostListener('dragover'),
    HostListener('dragleave'),
    HostListener('drop')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSFileUploaderDropContainer.prototype, "_handleDrag", null);
__decorate([
    property()
], CDSFileUploaderDropContainer.prototype, "accept", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSFileUploaderDropContainer.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSFileUploaderDropContainer.prototype, "multiple", void 0);
__decorate([
    property({ reflect: true })
], CDSFileUploaderDropContainer.prototype, "name", void 0);
__decorate([
    property({ reflect: true })
], CDSFileUploaderDropContainer.prototype, "slot", void 0);
CDSFileUploaderDropContainer = __decorate([
    carbonElement(`${prefix}-file-uploader-drop-container`)
], CDSFileUploaderDropContainer);
var CDSFileUploaderDropContainer$1 = CDSFileUploaderDropContainer;

export { CDSFileUploaderDropContainer$1 as default };
//# sourceMappingURL=file-uploader-drop-container.js.map
