/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import CheckmarkFilled16 from '../../icons/checkmark--filled/16.js';
import Close16 from '../../icons/close/16.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import { prefix } from '../../globals/settings.js';
import '../loading/loading.js';
import { FILE_UPLOADER_ITEM_SIZE, FILE_UPLOADER_ITEM_STATE } from './defs.js';
export { FILE_UPLOADER_ITEM_SIZE, FILE_UPLOADER_ITEM_STATE } from './defs.js';
import styles from './file-uploader.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { LOADING_TYPE } from '../loading/defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2020, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * File uploader item.
 *
 * @element cds-file-uploader-item
 * @slot validity-message The validity message.
 * @slot validity-message-supplement The supplemental validity message.
 * @fires cds-file-uploader-item-beingdeleted
 *   The custom event fired before this file uploader item is being deleted upon a user gesture.
 *   Cancellation of this event stops the user-initiated action of deleting this file uploader item.
 * @fires cds-file-uploader-item-deleted - The custom event fired after this file uploader item is deleted upon a user gesture.
 */
let CDSFileUploaderItem = class CDSFileUploaderItem extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * The `aria-label` attribute for the icon to delete this file uploader item.
         */
        this.iconDescription = 'Delete this file';
        /**
         * Controls the invalid state and visibility of the `validityMessage`.
         */
        this.invalid = false;
        /**
         * The size of this file uploader item.
         */
        this.size = FILE_UPLOADER_ITEM_SIZE.MEDIUM;
        /**
         * The state of this file uploader item.
         */
        this.state = FILE_UPLOADER_ITEM_STATE.UPLOADING;
        /**
         * The error subject text.
         */
        this.errorSubject = '';
        /**
         * The error body text
         */
        this.errorBody = '';
    }
    /**
     * Handles `click` event on the delete button.
     */
    _handleClickDeleteButton() {
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
        };
        const { eventBeforeDelete, eventDelete } = this
            .constructor;
        if (this.dispatchEvent(new CustomEvent(eventBeforeDelete, init))) {
            this.dispatchEvent(new CustomEvent(eventDelete, init));
        }
    }
    /**
     * @returns The content showing the editing UI of this file uploader item.
     */
    _renderEditing() {
        const { iconDescription, invalid, _handleClickDeleteButton: handleClickDeleteButton, } = this;
        return html `
      ${!invalid
            ? undefined
            : WarningFilled16({ class: `${prefix}--file-invalid` })}
      <button
        type="button"
        aria-label="${iconDescription}"
        class="${prefix}--file-close"
        @click="${handleClickDeleteButton}">
        ${Close16()}
      </button>
    `;
    }
    /**
     * @returns The content showing this file uploader's file uploading status as in progress.
     */
    _renderUploading() {
        const { iconDescription } = this;
        return html `
      <cds-loading
        assistive-text="${iconDescription}"
        type="${LOADING_TYPE.SMALL}"></cds-loading>
    `;
    }
    /**
     * @returns The content showing this file uploader's file uploading status as complete.
     */
    _renderUploaded() {
        const { iconDescription } = this;
        return CheckmarkFilled16({
            class: `${prefix}--file-complete`,
            'aria-label': iconDescription,
        });
    }
    /**
     * @returns The content showing this file uploader's status.
     */
    _renderStatus() {
        const { state } = this;
        switch (state) {
            case FILE_UPLOADER_ITEM_STATE.EDIT:
                return this._renderEditing();
            case FILE_UPLOADER_ITEM_STATE.UPLOADING:
                return this._renderUploading();
            case FILE_UPLOADER_ITEM_STATE.COMPLETE:
                return this._renderUploaded();
            default:
                return undefined;
        }
    }
    render() {
        const { invalid, errorSubject, errorBody } = this;
        return html ` <p class="${prefix}--file-filename"><slot></slot></p>
      <span class="${prefix}--file__state-container"
        >${this._renderStatus()}</span
      >
      <div
        class="${prefix}--form-requirement"
        ?hidden="${!invalid && !errorSubject}">
        <div class="${prefix}--form-requirement__title">${errorSubject}</div>
        <p
          class="${prefix}--form-requirement__supplement"
          ?hidden="${!errorBody}">
          ${errorBody}
        </p>
      </div>`;
    }
    /**
     * The name of the custom event fired before this file uplodaer item is being deleted upon a user gesture.
     * Cancellation of this event stops the user-initiated action of deleting this file uploader item.
     */
    static get eventBeforeDelete() {
        return `${prefix}-file-uploader-item-beingdeleted`;
    }
    /**
     * The name of the custom event fired after this file uplodaer item is deleted upon a user gesture.
     */
    static get eventDelete() {
        return `${prefix}-file-uploader-item-deleted`;
    }
};
CDSFileUploaderItem.styles = styles;
__decorate([
    property({ attribute: 'icon-description' })
], CDSFileUploaderItem.prototype, "iconDescription", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSFileUploaderItem.prototype, "invalid", void 0);
__decorate([
    property({ reflect: true })
], CDSFileUploaderItem.prototype, "size", void 0);
__decorate([
    property({ reflect: true })
], CDSFileUploaderItem.prototype, "state", void 0);
__decorate([
    property({ attribute: 'error-subject' })
], CDSFileUploaderItem.prototype, "errorSubject", void 0);
__decorate([
    property({ attribute: 'error-body' })
], CDSFileUploaderItem.prototype, "errorBody", void 0);
CDSFileUploaderItem = __decorate([
    carbonElement(`${prefix}-file-uploader-item`)
], CDSFileUploaderItem);
var CDSFileUploaderItem$1 = CDSFileUploaderItem;

export { CDSFileUploaderItem$1 as default };
//# sourceMappingURL=file-uploader-item.js.map
