/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement } from 'lit';
import HostListener from '../../globals/decorators/host-listener.js';
import FocusMixin from '../../globals/mixins/focus.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import { FLOATING_MENU_POSITION_DIRECTION, FLOATING_MENU_DIRECTION } from './defs.js';
export { FLOATING_MENU_DIRECTION, FLOATING_MENU_POSITION_DIRECTION } from './defs.js';
import { prefix } from '../../globals/settings.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Observes resize of the given element with the given resize observer.
 *
 * @param observer The resize observer.
 * @param elem The element to observe the resize.
 */
const observeResize = (observer, elem) => {
    observer.observe(elem);
    return {
        release() {
            observer.unobserve(elem);
            return null;
        },
    };
};
/**
 * @param elem The starting element.
 * @param selector The CSS selector.
 * @returns {Element}
 *   The closest ancestor node of the given element that matches the given selector, crossing Shadow DOM boundary.
 */
const closestComposed = (elem, selector) => {
    const found = elem.closest(selector);
    if (found) {
        return found;
    }
    const { host } = elem.getRootNode();
    if (host) {
        return closestComposed(host, selector);
    }
    return null;
};
/**
 * Floating menu.
 */
class CDSFloatingMenu extends HostListenerMixin(FocusMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * The handle for observing resize of the element containing the trigger button.
         */
        this._hObserveResizeParent = null;
        /**
         * The handle for observing resize of the floating menu container.
         */
        this._hObserveResizeContainer = null;
        /**
         * The `ResizeObserver` instance for observing element resizes for re-positioning floating menu position.
         */
        // TODO: Wait for `.d.ts` update to support `ResizeObserver`
        // @ts-ignore
        this._resizeObserver = new ResizeObserver(() => {
            const { container, open, parent, position } = this;
            if (container && open && parent) {
                const { direction, start, top } = position;
                this.style[direction !== FLOATING_MENU_POSITION_DIRECTION.RTL ? 'left' : 'right'] = `${start}px`;
                this.style.top = `${top}px`;
            }
        });
        this._handleBlur = ({ relatedTarget }) => {
            if (!this.contains(relatedTarget)) {
                const { parent } = this;
                if (parent && parent !== relatedTarget) {
                    parent.open = false;
                    HTMLElement.prototype.focus.call(this.parent); // SVGElement in IE11 does not have `.focus()` method
                }
            }
        };
        this._click = () => {
            const { parent } = this;
            if (parent) {
                parent.open = false;
            }
        };
        this._handleKeyDown = (event) => {
            if (event.key === 'Tab') {
                if (event.shiftKey) {
                    const { parent } = this;
                    if (parent) {
                        parent.open = false;
                    }
                }
            }
        };
        /**
         * The DOM element, typically a custom element in this library, launching this floating menu.
         */
        this.parent = null;
    }
    /**
     * The DOM element to put this menu into.
     */
    get container() {
        return (closestComposed(this, this.constructor.selectorContainer) || this.ownerDocument.body);
    }
    /**
     * The position of this floating menu.
     */
    get position() {
        const { triggerPosition } = this.parent;
        if (!triggerPosition) {
            throw new TypeError('Missing information of trigger button position.');
        }
        const { container } = this;
        const { left: refLeft = 0, top: refTop = 0, right: refRight = 0, } = triggerPosition;
        let { bottom: refBottom = 0 } = triggerPosition;
        const { width, height } = this.getBoundingClientRect();
        const { left: containerLeft = 0, right: containerRight = 0, top: containerTop = 0, } = container.getBoundingClientRect();
        refBottom = refBottom - containerTop;
        const containerComputedStyle = container.ownerDocument.defaultView.getComputedStyle(container);
        const positionDirection = containerComputedStyle.getPropertyValue('direction');
        const isRtl = positionDirection === FLOATING_MENU_POSITION_DIRECTION.RTL;
        const containerStartFromViewport = !isRtl
            ? containerLeft
            : container.ownerDocument.defaultView.innerWidth - containerRight;
        const refStartFromContainer = !isRtl
            ? refLeft - containerLeft
            : containerRight - refRight;
        const refEndFromContainer = !isRtl
            ? refRight - containerLeft
            : containerRight - refLeft;
        const refTopFromContainer = refTop - containerTop;
        if ((container !== this.ownerDocument.body ||
            containerStartFromViewport !== 0 ||
            containerTop !== 0) &&
            containerComputedStyle.getPropertyValue('position') === 'static') {
            throw new Error('Floating menu container must not have `position:static`.');
        }
        const { flipped, direction } = this;
        if (Object.values(FLOATING_MENU_DIRECTION).indexOf(direction) < 0) {
            throw new Error(`Wrong menu position direction: ${direction}`);
        }
        const alignmentStart = flipped
            ? refEndFromContainer - width
            : refStartFromContainer;
        const { start, top } = {
            [FLOATING_MENU_DIRECTION.TOP]: () => ({
                start: alignmentStart,
                top: refTopFromContainer - height,
            }),
            [FLOATING_MENU_DIRECTION.BOTTOM]: () => ({
                start: alignmentStart,
                top: refBottom,
            }),
        }[direction]();
        return {
            direction: positionDirection,
            start,
            top,
        };
    }
    disconnectedCallback() {
        if (this._hObserveResizeContainer) {
            this._hObserveResizeContainer = this._hObserveResizeContainer.release();
        }
        if (this._hObserveResizeParent) {
            this._hObserveResizeParent = this._hObserveResizeParent.release();
        }
    }
    updated(changedProperties) {
        var _a;
        const { container, open, parent } = this;
        if ((changedProperties.has('flipped') ||
            changedProperties.has('direction') ||
            changedProperties.has('open')) &&
            open) {
            if (!parent) {
                this.parent = this.parentElement;
                container.appendChild(this);
            }
            // Note: `this.position` cannot be referenced until `this.parent` is set
            const { direction, start, top } = this.position;
            this.style[direction !== FLOATING_MENU_POSITION_DIRECTION.RTL ? 'left' : 'right'] = `${start}px`;
            this.style.top = `${top}px`;
        }
        if (changedProperties.has('open')) {
            if (this._hObserveResizeContainer) {
                this._hObserveResizeContainer = this._hObserveResizeContainer.release();
            }
            if (this._hObserveResizeParent) {
                this._hObserveResizeParent = this._hObserveResizeParent.release();
            }
            if (open) {
                const { parentElement } = (_a = this.parent) !== null && _a !== void 0 ? _a : {};
                this._hObserveResizeContainer = observeResize(this._resizeObserver, container);
                if (parentElement) {
                    this._hObserveResizeParent = observeResize(this._resizeObserver, parentElement);
                }
            }
        }
    }
    /**
     * The CSS selector to find the element to put this floating menu in.
     */
    static get selectorContainer() {
        return `[data-floating-menu-container],${prefix}-modal`;
    }
}
/**
 * A constant indicating that this class is a floating menu.
 */
CDSFloatingMenu.FLOATING_MENU = true;
CDSFloatingMenu.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
__decorate([
    HostListener('focusout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSFloatingMenu.prototype, "_handleBlur", void 0);
__decorate([
    HostListener('click')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSFloatingMenu.prototype, "_click", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSFloatingMenu.prototype, "_handleKeyDown", void 0);

export { CDSFloatingMenu as default };
//# sourceMappingURL=floating-menu.js.map
